# -*- coding: utf-8 -*-
# pylint: disable=line-too-long, unused-wildcard-import
"""
.. _`German Federal Institute of Hydrology (BfG)`: https://www.bafg.de/EN

Version 3 of H-Land combines `HBV96's` :cite:t:`ref-Lindstrom1997HBV96` and `PREVAH's`
:cite:t:`ref-Viviroli2009PREVAH` process equations.  We implemented it on behalf of the
`German Federal Institute of Hydrology (BfG)`_ as an alternative to |hland_v1| for
modelling large river basins in central Europe.  All processes "above the soil" (input
data correction, interception, snowmelt) and "inside the soil" (evaporation, generation
of effective precipitation), as well as the handling of water areas, are identical with
|hland_v1| (and so with HBV96). Most processes "below the soil" agree with PREVAH
(runoff generation and runoff concentration).

The following figure shows the general structure of |hland_v3|:

.. image:: HydPy-H-Land_Version-3.png

Comparing the above figure with the analogue figure of |hland_v1| reveals that
|hland_v3| models five instead of two runoff components, requiring a larger number of
vertically arranged storages.  The two upper storages (|SUZ|, related to the soil, and
|SG1|, related to dynamic groundwater aquifers) are zone-specific.  In comparison, the
upper zone layer (|UZ|) of |hland_v1| is subbasin-specific.  |hland_v3| models the two
lower storages (|SG2| and |SG3|), dealing with slow and slowest groundwater responses,
slightly different from PREVAH.  First, their outflows (|RG2| and |RG3|) are
individually accessible (PREVAH handles their outflow as one runoff component). Second,
it represents internal lakes as an "extension" of the groundwater (like in
|hland_v1|/HBV96).  Third, it allows for additional runoff concentration via a linear
storage cascade (|SC|), like implemented in |hland_v2|.  The latter is supposed to
improve flexibility when modelling the effects of sealed areas on runoff generation.

Integration tests
=================

.. how_to_understand_integration_tests::

We define the settings of the following test calculations as similar to the ones of
application model |hland_v1| as possible.  Hence, comparing the test results gives a
good impression of the functional differences of both models.

The following settings are identical:

>>> from hydpy import pub
>>> pub.timegrids = "01.01.2000", "05.01.2000", "1h"

>>> from hydpy.models.hland_v3 import *
>>> parameterstep("1h")
>>> from hydpy import Node, Element
>>> outlet = Node("outlet")
>>> land = Element("land", outlets=outlet)
>>> land.model = model
>>> nmbzones(1)
>>> sclass(1)
>>> area(1.0)
>>> zonearea(1.0)
>>> zonez(1.0)

>>> zrelp(2.0)
>>> zrelt(2.0)
>>> zrele(2.0)

>>> psi(1.0)

>>> from hydpy import IntegrationTest
>>> IntegrationTest.plotting_options.axis1 = (
...     inputs.p, fluxes.rs, fluxes.ri, fluxes.rg1, fluxes.rg2, fluxes.rg3, fluxes.rt)
>>> IntegrationTest.plotting_options.axis2 = inputs.t
>>> test = IntegrationTest(land)
>>> test.dateformat = '%d/%m %H:00'

.. _hland_v3_field:

field
_____

We assign identical values to all parameters, besides those that are specific to
|hland_v3| (|SGR|, |SG1Max|, |K0|, |K1|, |K2|, and |K3|, and |NmbStorages|):

>>> zonetype(FIELD)
>>> pcorr(1.2)
>>> pcalt(0.1)
>>> rfcf(1.1)
>>> sfcf(1.3)
>>> tcalt(0.6)
>>> ecorr(0.8)
>>> ecalt(-0.1)
>>> epf(0.1)
>>> etf(0.1)
>>> ered(0.5)
>>> icmax(2.0)
>>> sfdist(1.0)
>>> smax(inf)
>>> sred(0.0)
>>> tt(0.0)
>>> ttint(2.0)
>>> dttm(1.0)
>>> cfmax(0.5)
>>> cfvar(0.1)
>>> gmelt(1.0)
>>> gvar(0.2)
>>> cfr(0.1)
>>> whc(0.4)
>>> fc(200.0)
>>> lp(0.8)
>>> beta(2.0)
>>> percmax(2.0)
>>> sgr(20.0)
>>> sg1max(50.0)
>>> k0(2.0)
>>> k1(10.0)
>>> k2(20.0)
>>> k3(100.0)
>>> maxbaz(3.0)
>>> nmbstorages(5)
>>> recstep(100)

The same holds for the initial conditions.

>>> test.inits = ((states.ic, 0.0),
...               (states.sp, 0.0),
...               (states.wc, 0.0),
...               (states.sm, 100.0),
...               (states.suz, 0.0),
...               (states.sg1, 10.0),
...               (states.sg2, 10.0),
...               (states.sg3, 10.0),
...               (states.sc, 0.05))

All input time series are identical:

>>> inputs.p.series = (
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.2, 0.0, 0.0, 1.3, 5.6, 2.9, 4.9, 10.6, 0.1, 0.7, 3.0, 2.1, 10.4, 3.5, 3.4,
...     1.2, 0.1, 0.0, 0.0, 0.4, 0.1, 3.6, 5.9, 1.1, 20.7, 37.9, 8.2, 3.6, 7.5, 18.5,
...     15.4, 6.3, 1.9, 4.9, 2.7, 0.5, 0.2, 0.5, 2.4, 0.4, 0.2, 0.0, 0.0, 0.3, 2.6,
...     0.7, 0.3, 0.3, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     1.3, 0.0, 0.0, 0.0, 0.7, 0.4, 0.1, 0.4, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0)
>>> inputs.t.series = (
...     21.2, 19.4, 18.9, 18.3, 18.9, 22.5, 25.1, 28.3, 27.8, 31.4, 32.2, 35.2, 37.1,
...     31.2, 24.3, 25.4, 25.9, 23.7, 21.6, 21.2, 20.4, 19.8, 19.6, 19.2, 19.2, 19.2,
...     18.9, 18.7, 18.5, 18.3, 18.5, 18.8, 18.8, 19.0, 19.2, 19.3, 19.0, 18.8, 18.7,
...     17.8, 17.4, 17.3, 16.8, 16.5, 16.3, 16.2, 15.5, 14.6, 14.7, 14.6, 14.1, 14.3,
...     14.9, 15.7, 16.0, 16.7, 17.1, 16.2, 15.9, 16.3, 16.3, 16.4, 16.5, 18.4, 18.3,
...     18.1, 16.7, 15.2, 13.4, 12.4, 11.6, 11.0, 10.5, 11.7, 11.9, 11.2, 11.1, 11.9,
...     12.2, 11.8, 11.4, 11.6, 13.0, 17.1, 18.2, 22.4, 21.4, 21.8, 22.2, 20.1, 17.8,
...     15.2, 14.5, 12.4, 11.7, 11.9)
>>> inputs.tn.series = inputs.t.series-1.0
>>> inputs.epn.series = (
...     0.100707, 0.097801, 0.096981, 0.09599, 0.096981, 0.102761, 0.291908, 1.932875,
...     4.369536, 7.317556, 8.264362, 9.369867, 5.126178, 6.62503, 7.397619, 2.39151,
...     1.829834, 1.136569, 0.750986, 0.223895, 0.099425, 0.098454, 0.098128, 0.097474,
...     0.097474, 0.097474, 0.096981, 0.096652, 0.096321, 0.09599, 0.187298, 1.264612,
...     3.045538, 1.930758, 2.461001, 6.215945, 3.374783, 8.821555, 4.046025, 2.110757,
...     2.239257, 2.877848, 1.591452, 0.291604, 0.092622, 0.092451, 0.091248, 0.089683,
...     0.089858, 0.089683, 0.088805, 0.089157, 0.090207, 0.091593, 0.154861, 0.470369,
...     1.173726, 4.202296, 4.359715, 5.305753, 5.376027, 4.658915, 7.789594, 4.851567,
...     5.30692, 3.286036, 1.506216, 0.274762, 0.087565, 0.085771, 0.084317, 0.083215,
...     0.082289, 0.0845, 0.084864, 0.083584, 0.0834, 0.084864, 0.310229, 1.391958,
...     3.195876, 5.191651, 7.155036, 8.391432, 8.391286, 10.715238, 9.383394, 7.861915,
...     6.298329, 2.948416, 1.309232, 0.32955, 0.089508, 0.085771, 0.0845, 0.084864)

>>> test.reset_inits()
>>> conditions = sequences.conditions

|hland_v3| neither implements an approach analogue to HBV96's contributing area concept
nor a substep mechanism controlling the numerical accuracy of the runoff generation
module.  Hence, we provide only a single "field" example that is comparable both with
the :ref:`first <hland_v1_field>` and the :ref:`second <hland_v1_resparea>` example of
|hland_v1|:

.. integration-test::

    >>> test('hland_v3_field')
    |        date |    p |    t |   tn |       epn | tmean |   tc | fracrain | rfc | sfc |    cfact | swe | gact |      pc |        ep |      epc |       ei |        tf | spl | wcl | spg | wcg | glmelt | melt | refr |       in_ |         r |  sr |       ea |       dp |  el |       rs |       ri |      gr1 |      rg1 |      gr2 |      rg2 |      gr3 |      rg3 |     inuh |    outuh |       rt |       qt |       ic |  sp |  wc |         sm |       suz |       sg1 |       sg2 |       sg3 |                                               sc |   outlet |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | 21.2 | 20.2 |  0.100707 |  21.8 | 21.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |     0.0 |   0.11682 |  0.08411 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.052569 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.487706 |      0.0 | 0.099502 |      0.0 | 0.011105 | 0.598312 | 0.202814 | 0.202814 | 0.056337 |      0.0 | 0.0 | 0.0 |  99.947431 |       0.0 |  9.512294 |  9.900498 |  9.988895 | 0.169348  0.155472  0.132115  0.105643  0.082921 | 0.056337 |
    | 01/01 01:00 |  0.0 | 19.4 | 18.4 |  0.097801 |  20.0 | 20.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |     0.0 |  0.113449 | 0.081683 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.051025 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |  0.46392 |      0.0 | 0.098512 |      0.0 | 0.011093 | 0.573524 | 0.414293 | 0.414293 | 0.115081 |      0.0 | 0.0 | 0.0 |  99.896406 |       0.0 |  9.048374 |  9.801987 |  9.977802 | 0.166424  0.166296  0.164522  0.159031  0.148456 | 0.115081 |
    | 01/01 02:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |     0.0 |  0.112498 | 0.080999 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.050572 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.441294 |      0.0 | 0.097531 |      0.0 |  0.01108 | 0.549906 | 0.547167 | 0.547167 | 0.151991 |      0.0 | 0.0 | 0.0 |  99.845834 |       0.0 |   8.60708 |  9.704455 |  9.966722 | 0.159707  0.160484  0.161724  0.162805  0.162748 | 0.151991 |
    | 01/01 03:00 |  0.0 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |     0.0 |  0.111348 | 0.080171 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.05003 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.419772 |      0.0 | 0.096561 |      0.0 | 0.011068 | 0.527401 | 0.556325 | 0.556325 | 0.154535 |      0.0 | 0.0 | 0.0 |  99.795804 |       0.0 |  8.187308 |  9.607894 |  9.955654 | 0.153174   0.15396  0.155368  0.157139  0.158904 | 0.154535 |
    | 01/01 04:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |     0.0 |  0.112498 | 0.080999 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.050521 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.3993 |      0.0 |   0.0956 |      0.0 | 0.011056 | 0.505956 | 0.537334 | 0.537334 |  0.14926 |      0.0 | 0.0 | 0.0 |  99.745284 |       0.0 |  7.788008 |  9.512294 |  9.944598 | 0.146944  0.147695   0.14905  0.150798  0.152677 |  0.14926 |
    | 01/01 05:00 |  0.0 | 22.5 | 21.5 |  0.102761 |  23.1 | 23.1 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |     0.0 |  0.119203 | 0.085826 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.053505 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.379826 |      0.0 | 0.094649 |      0.0 | 0.011043 | 0.485518 | 0.515762 | 0.515762 | 0.143267 |      0.0 | 0.0 | 0.0 |  99.691779 |       0.0 |  7.408182 |  9.417645 |  9.933555 | 0.141007  0.141723  0.143015  0.144685  0.146492 | 0.143267 |
    | 01/01 06:00 |  0.0 | 25.1 | 24.1 |  0.291908 |  25.7 | 25.7 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |     0.0 |  0.338613 | 0.243802 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.151906 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.361301 |      0.0 | 0.093707 |      0.0 | 0.011031 |  0.46604 | 0.494888 | 0.494888 | 0.137469 |      0.0 | 0.0 | 0.0 |  99.539873 |       0.0 |  7.046881 |  9.323938 |  9.922524 | 0.135349  0.136031  0.137262  0.138854  0.140577 | 0.137469 |
    | 01/01 07:00 |  0.0 | 28.3 | 27.3 |  1.932875 |  28.9 | 28.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |     0.0 |  2.242135 | 1.614337 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.004318 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |  0.34368 |      0.0 | 0.092775 |      0.0 | 0.011019 | 0.447474 | 0.474971 | 0.474971 | 0.131936 |      0.0 | 0.0 | 0.0 |  98.535555 |       0.0 |    6.7032 |  9.231163 |  9.911505 | 0.129956  0.130606  0.131779  0.133296  0.134939 | 0.131936 |
    | 01/01 08:00 |  0.0 | 27.8 | 26.8 |  4.369536 |  28.4 | 28.4 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |     0.0 |  5.068662 | 3.649436 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.247495 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.326919 |      0.0 | 0.091852 |      0.0 | 0.011007 | 0.429777 | 0.455987 | 0.455987 | 0.126663 |      0.0 | 0.0 | 0.0 |  96.288059 |       0.0 |  6.376282 |  9.139312 |  9.900498 | 0.124815  0.125434  0.126553  0.127999  0.129565 | 0.126663 |
    | 01/01 09:00 |  0.0 | 31.4 | 30.4 |  7.317556 |  32.0 | 32.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |     0.0 |  8.488365 | 6.111623 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.677977 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.310975 |      0.0 | 0.090938 |      0.0 | 0.010994 | 0.412907 | 0.437892 | 0.437892 | 0.121637 |      0.0 | 0.0 | 0.0 |  92.610082 |       0.0 |  6.065307 |  9.048374 |  9.889504 | 0.119914  0.120505  0.121571   0.12295  0.124442 | 0.121637 |
    | 01/01 10:00 |  0.0 | 32.2 | 31.2 |  8.264362 |  32.8 | 32.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |     0.0 |   9.58666 | 6.902395 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.995196 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.295808 |      0.0 | 0.090033 |      0.0 | 0.010982 | 0.396824 | 0.420642 | 0.420642 | 0.116845 |      0.0 | 0.0 | 0.0 |  88.614886 |       0.0 |  5.769498 |  8.958341 |  9.878522 | 0.115242  0.115805  0.116822  0.118136  0.119559 | 0.116845 |
    | 01/01 11:00 |  0.0 | 35.2 | 34.2 |  9.369867 |  35.8 | 35.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |     0.0 | 10.869046 | 7.825713 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 4.334217 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.281382 |      0.0 | 0.089137 |      0.0 |  0.01097 | 0.381489 | 0.404198 | 0.404198 | 0.112277 |      0.0 | 0.0 | 0.0 |   84.28067 |       0.0 |  5.488116 |  8.869204 |  9.867552 | 0.110787  0.111324  0.112293  0.113546  0.114903 | 0.112277 |
    | 01/01 12:00 |  0.0 | 37.1 | 36.1 |  5.126178 |  37.7 | 37.7 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |     0.0 |  5.946366 | 4.281384 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.255237 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.267659 |      0.0 |  0.08825 |      0.0 | 0.010958 | 0.366867 | 0.388519 | 0.388519 | 0.107922 |      0.0 | 0.0 | 0.0 |  82.025433 |       0.0 |  5.220458 |  8.780954 |  9.856594 | 0.106539  0.107051  0.107976  0.109171  0.110464 | 0.107922 |
    | 01/01 13:00 |  0.0 | 31.2 | 30.2 |   6.62503 |  31.8 | 31.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |     0.0 |  7.685035 | 5.533225 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.836657 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.254605 |      0.0 | 0.087372 |      0.0 | 0.010946 | 0.352922 |  0.37357 |  0.37357 |  0.10377 |      0.0 | 0.0 | 0.0 |  79.188775 |       0.0 |  4.965853 |  8.693582 |  9.845648 | 0.102489  0.102977  0.103858  0.104998  0.106231 |  0.10377 |
    | 01/01 14:00 |  0.0 | 24.3 | 23.3 |  7.397619 |  24.9 | 24.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |     0.0 |  8.581238 | 6.178491 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  3.05792 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.242188 |      0.0 | 0.086503 |      0.0 | 0.010934 | 0.339624 | 0.359315 | 0.359315 |  0.09981 |      0.0 | 0.0 | 0.0 |  76.130856 |       0.0 |  4.723666 |   8.60708 |  9.834715 | 0.098626  0.099091  0.099932  0.101018  0.102195 |  0.09981 |
    | 01/01 15:00 |  0.2 | 25.4 | 24.4 |   2.39151 |  26.0 | 26.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |  0.2376 |  2.774152 | 1.950491 |   0.2376 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.928078 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.230376 |      0.0 | 0.085642 |      0.0 | 0.010921 | 0.326939 |  0.34572 |  0.34572 | 0.096033 |      0.0 | 0.0 | 0.0 |  75.202777 |       0.0 |   4.49329 |  8.521438 |  9.823793 | 0.094941  0.095385  0.096187  0.097223  0.098345 | 0.096033 |
    | 01/01 16:00 |  0.0 | 25.9 | 24.9 |  1.829834 |  26.5 | 26.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |     0.0 |  2.122607 | 1.528277 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.718317 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |  0.21914 |      0.0 |  0.08479 |      0.0 | 0.010909 | 0.314839 | 0.332754 | 0.332754 | 0.092432 |      0.0 | 0.0 | 0.0 |  74.484461 |       0.0 |  4.274149 |  8.436648 |  9.812884 | 0.091426   0.09185  0.092614  0.093603  0.094673 | 0.092432 |
    | 01/01 17:00 |  0.0 | 23.7 | 22.7 |  1.136569 |  24.3 | 24.3 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |     0.0 |   1.31842 | 0.949262 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.441908 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.208453 |      0.0 | 0.083946 |      0.0 | 0.010897 | 0.303296 | 0.320386 | 0.320386 | 0.088996 |      0.0 | 0.0 | 0.0 |  74.042552 |       0.0 |  4.065697 |  8.352702 |  9.801987 | 0.088073  0.088477  0.089207   0.09015  0.091171 | 0.088996 |
    | 01/01 18:00 |  1.3 | 21.6 | 20.6 |  0.750986 |  22.2 | 22.2 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |  1.5444 |  0.871144 | 0.537465 | 0.537465 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.12436 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.198286 |      0.0 | 0.083111 |      0.0 | 0.010885 | 0.292282 | 0.308587 | 0.308587 | 0.085719 | 1.006935 | 0.0 | 0.0 |  73.918192 |       0.0 |   3.86741 |  8.269591 |  9.791102 | 0.084873  0.085259  0.085955  0.086855  0.087829 | 0.085719 |
    | 01/01 19:00 |  5.6 | 21.2 | 20.2 |  0.223895 |  21.8 | 21.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |  6.6528 |  0.259718 | 0.096141 | 0.096141 |  5.659735 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.659735 |  0.773106 | 0.0 | 0.023676 | 0.773106 | 0.0 |      0.0 |      0.0 | 0.773106 | 0.207625 |      0.0 | 0.082284 |      0.0 | 0.010873 | 0.300782 | 0.298687 | 0.298687 | 0.082969 | 1.903859 | 0.0 | 0.0 |  78.781146 |       0.0 |   4.43289 |  8.187308 |  9.780229 | 0.087147  0.086896  0.086518  0.086195   0.08611 | 0.082969 |
    | 01/01 20:00 |  2.9 | 20.4 | 19.4 |  0.099425 |  21.0 | 21.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |  3.4452 |  0.115333 | 0.058839 | 0.058839 |  3.349059 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.349059 |  0.519646 | 0.0 | 0.015006 | 0.519646 | 0.0 |      0.0 |      0.0 | 0.519646 | 0.228972 |      0.0 | 0.081465 |      0.0 | 0.010861 | 0.321298 | 0.299645 | 0.299645 | 0.083235 | 1.941161 | 0.0 | 0.0 |  81.595554 |       0.0 |  4.723564 |  8.105842 |  9.769368 | 0.092973  0.092287  0.091106  0.089713   0.08844 | 0.083235 |
    | 01/01 21:00 |  4.9 | 19.8 | 18.8 |  0.098454 |  20.4 | 20.4 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |  5.8212 |  0.114207 | 0.045942 | 0.045942 |  5.762361 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.762361 |  0.959121 | 0.0 | 0.012404 | 0.959121 | 0.0 |      0.0 |      0.0 | 0.959121 | 0.253954 |      0.0 | 0.080654 |      0.0 | 0.010849 | 0.345458 | 0.313657 | 0.313657 | 0.087127 | 1.954058 | 0.0 | 0.0 |  86.386389 |       0.0 |  5.428731 |  8.025188 |  9.758519 |  0.09994  0.099107  0.097626  0.095772  0.093876 | 0.087127 |
    | 01/01 22:00 | 10.6 | 19.6 | 18.6 |  0.098128 |  20.2 | 20.2 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 | 12.5928 |  0.113828 | 0.023264 | 0.023264 | 12.546858 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.546858 |  2.340807 | 0.0 | 0.007022 |      2.0 | 0.0 |      0.0 | 0.032432 |      2.0 | 0.313939 |      0.0 | 0.079852 |      0.0 | 0.010837 |  0.43706 | 0.339886 | 0.339886 | 0.094413 | 1.976736 | 0.0 | 0.0 |  96.585418 |  0.308375 |  7.114792 |  7.945336 |  9.747682 | 0.125844  0.122804  0.117591  0.111459  0.105796 | 0.094413 |
    | 01/01 23:00 |  0.1 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |  0.1188 |   0.11307 | 0.080449 | 0.080449 |  0.095536 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.095536 |  0.022281 | 0.0 |   0.0243 | 0.330656 | 0.0 |      0.0 |      0.0 | 0.330656 | 0.355123 |      0.0 | 0.079057 |      0.0 | 0.010825 | 0.445005 | 0.396222 | 0.396222 | 0.110062 | 1.919551 | 0.0 | 0.0 |  96.634373 |       0.0 |  7.090325 |  7.866279 |  9.736857 | 0.128943  0.128481  0.127357  0.125302  0.122195 | 0.110062 |
    | 02/01 00:00 |  0.7 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  0.8316 |   0.11307 | 0.074914 | 0.074914 |  0.751151 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.751151 |   0.17536 | 0.0 | 0.022758 |  0.17536 | 0.0 |      0.0 |      0.0 |  0.17536 | 0.350111 |      0.0 | 0.078271 |      0.0 | 0.010813 | 0.439195 |  0.43335 |  0.43335 | 0.120375 | 1.925086 | 0.0 | 0.0 |  97.187406 |       0.0 |  6.915574 |  7.788008 |  9.726045 |  0.12742  0.127581  0.127789  0.127844  0.127487 | 0.120375 |
    | 02/01 01:00 |  3.0 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   3.564 |   0.11307 | 0.057003 | 0.057003 |  3.489086 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.489086 |  0.823895 | 0.0 | 0.017787 | 0.823895 | 0.0 |      0.0 |      0.0 | 0.823895 | 0.357535 |      0.0 | 0.077492 |      0.0 | 0.010801 | 0.445828 | 0.440585 | 0.440585 | 0.122385 | 1.942997 | 0.0 | 0.0 |  99.834811 |       0.0 |  7.381934 |  7.710516 |  9.715244 | 0.129227  0.129022  0.128694  0.128349  0.128073 | 0.122385 |
    | 02/01 02:00 |  2.1 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  2.4948 |  0.112498 | 0.063115 | 0.063115 |  2.437797 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.437797 |  0.607437 | 0.0 | 0.020052 | 0.607437 | 0.0 |      0.0 |      0.0 | 0.607437 | 0.374957 |      0.0 | 0.076721 |      0.0 | 0.010789 | 0.462467 | 0.444501 | 0.444501 | 0.123473 | 1.936885 | 0.0 | 0.0 | 101.645119 |       0.0 |  7.614414 |  7.633795 |  9.704455 | 0.133951  0.133395  0.132436  0.131301  0.130248 | 0.123473 |
    | 02/01 03:00 | 10.4 | 18.7 | 17.7 |  0.096652 |  19.3 | 19.3 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 | 12.3552 |  0.112116 | 0.023465 | 0.023465 | 12.292085 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.292085 |  3.174963 | 0.0 | 0.008122 |      2.0 | 0.0 |      0.0 | 0.111812 |      2.0 | 0.420536 |      0.0 | 0.075958 |      0.0 | 0.010777 | 0.619083 |  0.46602 |  0.46602 |  0.12945 | 1.976535 | 0.0 | 0.0 | 110.754119 |   1.06315 |  9.193878 |  7.557837 |  9.693679 | 0.177998  0.172858  0.164143  0.154122  0.145272 |  0.12945 |
    | 02/01 04:00 |  3.5 | 18.5 | 17.5 |  0.096321 |  19.1 | 19.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   4.158 |  0.111732 |  0.05308 |  0.05308 |  4.134535 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.134535 |  1.267904 | 0.0 | 0.018847 |      2.0 | 0.0 |      0.0 | 0.031504 |      2.0 | 0.497568 |      0.0 | 0.075202 |      0.0 | 0.010765 | 0.615038 | 0.550337 | 0.550337 | 0.152871 |  1.94692 | 0.0 | 0.0 | 113.601903 |   0.29955 |  10.69631 |  7.482636 |  9.682914 | 0.178349  0.178135  0.177195  0.174828  0.170588 | 0.152871 |
    | 02/01 05:00 |  3.4 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  4.0392 |  0.111348 |  0.05353 |  0.05353 |   3.98612 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   3.98612 |  1.286061 | 0.0 | 0.019455 | 1.585612 | 0.0 |      0.0 |      0.0 | 1.585612 | 0.560653 |      0.0 | 0.074453 |      0.0 | 0.010753 | 0.645859 | 0.605829 | 0.605829 | 0.168286 |  1.94647 | 0.0 | 0.0 | 116.282507 |       0.0 | 11.721268 |  7.408182 |  9.672161 | 0.186998  0.185985  0.184236  0.182092  0.179813 | 0.168286 |
    | 02/01 06:00 |  1.2 | 18.5 | 17.5 |  0.187298 |  19.1 | 19.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  1.4256 |  0.217266 | 0.135647 | 0.135647 |   1.37207 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   1.37207 |  0.463815 | 0.0 | 0.049677 | 0.463815 | 0.0 |      0.0 |      0.0 | 0.463815 | 0.583058 |      0.0 | 0.073713 |      0.0 | 0.010741 | 0.667511 | 0.633016 | 0.633016 | 0.175838 | 1.864353 | 0.0 | 0.0 | 117.141085 |       0.0 | 11.602026 |   7.33447 |   9.66142 | 0.193356  0.192583  0.191166  0.189295  0.187219 | 0.175838 |
    | 02/01 07:00 |  0.1 | 18.8 | 17.8 |  1.264612 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  0.1188 |   1.46695 |  1.04373 |  1.04373 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.382074 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.565837 |      0.0 | 0.072979 |      0.0 | 0.010729 | 0.649546 | 0.654315 | 0.654315 | 0.181754 | 0.939422 | 0.0 | 0.0 | 116.759011 |       0.0 | 11.036189 |   7.26149 |  9.650691 | 0.188536  0.189071  0.189879   0.19058  0.190784 | 0.181754 |
    | 02/01 08:00 |  0.0 | 18.8 | 17.8 |  3.045538 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |     0.0 |  3.532824 | 2.543633 | 0.939422 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.730206 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.538241 |      0.0 | 0.072253 |      0.0 | 0.010717 | 0.621211 | 0.653112 | 0.653112 |  0.18142 |      0.0 | 0.0 | 0.0 | 115.028805 |       0.0 | 10.497947 |  7.189237 |  9.639974 | 0.180434  0.181394  0.183067   0.18508  0.186975 |  0.18142 |
    | 02/01 09:00 |  0.0 | 19.0 | 18.0 |  1.930758 |  19.6 | 19.6 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |     0.0 |  2.239679 | 1.612569 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.159324 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.511991 |      0.0 | 0.071534 |      0.0 | 0.010705 |  0.59423 | 0.632661 | 0.632661 | 0.175739 |      0.0 | 0.0 | 0.0 | 113.869481 |       0.0 |  9.985957 |  7.117703 |  9.629269 |   0.1726  0.173543  0.175241  0.177415  0.179719 | 0.175739 |
    | 02/01 10:00 |  0.4 | 19.2 | 18.2 |  2.461001 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  0.4752 |  2.854761 | 1.960038 |   0.4752 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.394929 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.487021 |      0.0 | 0.070822 |      0.0 | 0.010693 | 0.568536 | 0.606443 | 0.606443 | 0.168456 |      0.0 | 0.0 | 0.0 | 112.474552 |       0.0 |  9.498936 |  7.046881 |  9.618576 | 0.165136  0.166035  0.167659  0.169757  0.172024 | 0.168456 |
    | 02/01 11:00 |  0.1 | 19.3 | 18.3 |  6.215945 |  19.9 | 19.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  0.1188 |  7.210496 | 5.130246 |   0.1188 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.606389 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.463269 |      0.0 | 0.070118 |      0.0 | 0.010681 | 0.544068 | 0.580309 | 0.580309 | 0.161197 |      0.0 | 0.0 | 0.0 | 108.868163 |       0.0 |  9.035667 |  6.976763 |  9.607894 | 0.158028  0.158884  0.160431  0.162431  0.164597 | 0.161197 |
    | 02/01 12:00 |  3.6 | 19.0 | 18.0 |  3.374783 |  19.6 | 19.6 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  4.2768 |  3.914748 | 1.837796 | 1.837796 |    2.2768 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    2.2768 |  0.674632 | 0.0 | 0.634444 | 0.674632 | 0.0 |      0.0 |      0.0 | 0.674632 | 0.457263 |      0.0 |  0.06942 |      0.0 |  0.01067 | 0.537352 | 0.556474 | 0.556474 | 0.154576 | 0.162204 | 0.0 | 0.0 | 109.835888 |       0.0 |  9.253036 |  6.907343 |  9.597225 | 0.155906  0.156182  0.156745  0.157619  0.158797 | 0.154576 |
    | 02/01 13:00 |  5.9 | 18.8 | 17.8 |  8.821555 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  7.0092 | 10.233004 | 3.655358 |      2.0 |  5.171404 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.171404 |  1.559685 | 0.0 |  2.12359 | 1.559685 | 0.0 |      0.0 |      0.0 | 1.559685 | 0.489626 |      0.0 | 0.068729 |      0.0 | 0.010658 | 0.569013 | 0.544787 | 0.544787 |  0.15133 |      0.0 | 0.0 | 0.0 | 111.324016 |       0.0 | 10.323095 |  6.838614 |  9.586567 | 0.164707  0.163693   0.16202  0.160213  0.158841 |  0.15133 |
    | 02/01 14:00 |  1.1 | 18.7 | 17.7 |  4.046025 |  19.3 | 19.3 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  1.3068 |  4.693389 | 2.965278 |   1.3068 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.860822 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.503463 |      0.0 | 0.068045 |      0.0 | 0.010646 | 0.582154 |  0.55722 |  0.55722 | 0.154783 |      0.0 | 0.0 | 0.0 | 109.463194 |       0.0 |  9.819632 |  6.770569 |  9.575921 | 0.168686  0.168189   0.16724  0.165917  0.164377 | 0.154783 |
    | 02/01 15:00 | 20.7 | 17.8 | 16.8 |  2.110757 |  18.4 | 18.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 | 24.5916 |  2.448478 |  0.15074 |  0.15074 |   22.5916 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   22.5916 |  6.767422 | 0.0 | 0.059018 |      2.0 | 0.0 |      0.0 |  0.45368 |      2.0 | 0.528086 |      0.0 | 0.067368 |      0.0 | 0.010634 | 1.059769 | 0.608088 | 0.608088 | 0.168913 |  1.84926 | 0.0 | 0.0 | 125.228354 |  4.313741 | 11.291546 |    6.7032 |  9.565287 |  0.30266  0.287067  0.260759  0.230798  0.204805 | 0.168913 |
    | 02/01 16:00 | 37.9 | 17.4 | 16.4 |  2.239257 |  18.0 | 18.0 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 | 45.0252 |  2.597538 | 0.020724 | 0.020724 |  44.87446 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  44.87446 |  17.59319 | 0.0 | 0.009877 |      2.0 | 0.0 |      0.0 | 1.894395 | 1.935423 | 0.598284 | 0.057402 | 0.066984 | 0.007175 | 0.010626 | 2.570289 |  0.96182 |  0.96182 | 0.267172 | 1.979276 | 0.0 | 0.0 | 152.499747 | 18.012536 | 12.628684 |  6.693618 |  9.561836 | 0.730463  0.680199  0.593836  0.492033  0.398028 | 0.267172 |
    | 02/01 17:00 |  8.2 | 17.3 | 16.3 |  2.877848 |  17.9 | 17.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  9.7416 |  3.338304 | 0.907373 | 0.907373 |  9.720876 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  9.720876 |  5.651759 | 0.0 | 0.443958 |      2.0 | 0.0 | 0.654849 |  2.06163 | 1.868566 | 0.661853 |  0.11683 | 0.067185 | 0.014604 | 0.010626 | 3.456144 | 1.947424 | 1.947424 | 0.540951 | 1.092627 | 0.0 | 0.0 | 156.124906 | 18.947816 | 13.835397 |  6.743264 |  9.565814 | 0.993121  0.960889  0.900927  0.820029  0.728313 | 0.540951 |
    | 02/01 18:00 |  3.6 | 16.8 | 15.8 |  1.591452 |  17.4 | 17.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  4.2768 |  1.846084 | 0.866652 | 0.866652 |  3.369427 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.369427 |  2.053243 | 0.0 | 0.426396 |      2.0 | 0.0 |      0.0 |  1.80819 |  1.80823 | 0.719222 | 0.170462 | 0.067946 | 0.021308 | 0.010635 | 2.605992 | 2.886616 | 2.886616 | 0.801838 | 1.133348 | 0.0 | 0.0 | 157.014694 | 17.192869 | 14.924405 |   6.84578 |  9.576487 | 0.763738  0.789321  0.828571  0.863891  0.877134 | 0.801838 |
    | 02/01 19:00 |  7.5 | 16.5 | 15.5 |  0.291604 |  17.1 | 17.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |    8.91 |  0.338261 | 0.099914 | 0.099914 |  8.043348 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  8.043348 |   4.95744 | 0.0 | 0.049957 |      2.0 | 0.0 | 0.059142 | 1.917555 |  1.75378 | 0.770995 | 0.218862 | 0.069207 | 0.027358 |  0.01065 | 2.827549 | 2.897153 | 2.897153 | 0.804765 | 1.900086 | 0.0 | 0.0 | 160.050645 | 18.173612 |  15.90719 |  6.995435 |  9.593195 | 0.818093  0.812636  0.806195  0.804577  0.811549 | 0.804765 |
    | 02/01 20:00 | 18.5 | 16.3 | 15.3 |  0.092622 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  21.978 |  0.107442 |  0.00859 |  0.00859 | 21.878086 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 21.878086 |  14.01084 | 0.0 | 0.004295 |      2.0 | 0.0 |  4.00727 |  2.87243 | 1.704641 | 0.817717 | 0.262542 | 0.070914 | 0.032818 | 0.010671 | 7.779003 | 3.146677 | 3.146677 | 0.874077 |  1.99141 | 0.0 | 0.0 | 167.913595 | 23.304752 | 16.794113 |  7.187063 |  9.615341 | 2.207451  2.045736  1.772995  1.463019  1.196175 | 0.874077 |
    | 02/01 21:00 | 15.4 | 16.2 | 15.2 |  0.092451 |  16.8 | 16.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 | 18.2952 |  0.107243 | 0.012392 | 0.012392 |  18.28661 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  18.28661 | 12.889763 | 0.0 | 0.006196 |      2.0 | 0.0 | 5.585106 | 3.254038 | 1.660294 | 0.859883 | 0.301961 | 0.073017 | 0.037745 | 0.010699 | 9.782743 | 5.792256 | 5.792256 |  1.60896 | 1.987608 | 0.0 | 0.0 | 173.304246 |  25.35537 | 17.594525 |  7.416006 |  9.642388 | 2.815778  2.739601  2.593186  2.386585  2.140714 |  1.60896 |
    | 02/01 22:00 |  6.3 | 15.5 | 14.5 |  0.091248 |  16.1 | 16.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  7.4844 |  0.105848 | 0.036055 | 0.036055 |  7.472008 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  7.472008 |  5.610425 | 0.0 | 0.018028 |      2.0 | 0.0 | 3.527765 |  2.75646 | 1.620274 | 0.897935 | 0.337534 | 0.075473 | 0.042192 | 0.010731 | 7.268364 | 8.334197 | 8.334197 | 2.315055 | 1.963945 | 0.0 | 0.0 | 175.147801 | 22.681569 | 18.316863 |  7.678068 |  9.673848 | 2.131686  2.208653  2.329044  2.442961  2.497686 | 2.315055 |
    | 02/01 23:00 |  1.9 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  2.2572 |  0.104032 | 0.059768 | 0.059768 |  2.221145 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.221145 |  1.703438 | 0.0 | 0.029884 |      2.0 | 0.0 | 0.938427 | 2.130215 | 1.584157 | 0.932276 | 0.369638 |  0.07824 | 0.046205 | 0.010768 | 4.089927 | 7.903256 | 7.903256 | 2.195349 | 1.940232 | 0.0 | 0.0 | 175.635624 | 19.316365 | 18.968744 |  7.969466 |  9.709285 | 1.217949  1.326777  1.518309    1.7529  1.980766 | 2.195349 |
    | 03/01 00:00 |  4.9 | 14.7 | 13.7 |  0.089858 |  15.3 | 15.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  5.8212 |  0.104235 | 0.041931 | 0.041931 |  5.761432 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.761432 |  4.443198 | 0.0 | 0.020966 |      2.0 | 0.0 | 0.692334 | 2.070696 | 1.551563 | 0.963267 | 0.398611 | 0.081284 | 0.049826 |  0.01081 | 3.818391 | 5.615246 | 5.615246 | 1.559791 | 1.958069 | 0.0 | 0.0 | 176.932892 | 18.996532 |  19.55704 |  8.286793 |  9.748301 | 1.111061  1.127156  1.166757   1.24047  1.354401 | 1.559791 |
    | 03/01 01:00 |  2.7 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  3.2076 |  0.104032 |  0.05435 |  0.05435 |  3.165669 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.165669 |  2.477551 | 0.0 | 0.027175 |      2.0 | 0.0 |      0.0 | 1.853204 | 1.522148 | 0.991235 | 0.424757 | 0.084572 | 0.053095 | 0.010855 | 2.939866 | 4.157546 | 4.157546 | 1.154874 |  1.94565 | 0.0 | 0.0 | 177.593835 |  17.62088 | 20.087952 |  8.626979 |  9.790541 | 0.861274  0.890857  0.942886  1.008267  1.078882 | 1.154874 |
    | 03/01 02:00 |  0.5 | 14.1 | 13.1 |  0.088805 |  14.7 | 14.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.594 |  0.103014 | 0.069893 | 0.069893 |   0.53965 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   0.53965 |  0.425509 | 0.0 | 0.034946 |      2.0 | 0.0 |      0.0 | 1.527016 | 1.495602 | 1.016476 | 0.448353 | 0.088074 | 0.056044 | 0.010903 | 2.642469 | 3.349923 | 3.349923 | 0.930534 | 1.930107 | 0.0 | 0.0 | 177.673031 | 14.519372 | 20.567079 |  8.987258 |  9.835682 | 0.769516  0.781181  0.804216  0.838212  0.881586 | 0.930534 |
    | 03/01 03:00 |  0.2 | 14.3 | 13.3 |  0.089157 |  14.9 | 14.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  0.2376 |  0.103422 | 0.072716 | 0.072716 |  0.167707 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.167707 |  0.132353 | 0.0 | 0.036358 |      2.0 | 0.0 |      0.0 | 1.203971 | 1.471646 | 1.039254 | 0.469648 | 0.091765 | 0.058706 | 0.010955 | 2.345945 | 2.826831 | 2.826831 | 0.785231 | 1.927284 | 0.0 | 0.0 | 177.672027 | 11.447755 | 20.999471 |  9.365141 |  9.883432 |  0.68333  0.693744  0.712698  0.737754    0.7663 | 0.785231 |
    | 03/01 04:00 |  0.5 | 14.9 | 13.9 |  0.090207 |  15.5 | 15.5 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.594 |   0.10464 | 0.070996 | 0.070996 |  0.521284 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.521284 |  0.411389 | 0.0 | 0.035498 |      2.0 | 0.0 |      0.0 | 0.938222 | 1.450026 |  1.05981 | 0.488865 | 0.095621 | 0.061108 | 0.011009 | 2.104662 | 2.489213 | 2.489213 | 0.691448 | 1.929004 | 0.0 | 0.0 | 177.746424 |  8.920923 | 21.389688 |  9.758385 |  9.933531 | 0.612812  0.621361  0.636971  0.657556  0.680575 | 0.691448 |
    | 03/01 05:00 |  2.4 | 15.7 | 14.7 |  0.091593 |  16.3 | 16.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  2.8512 |  0.106248 | 0.057521 | 0.057521 |  2.780204 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.780204 |   2.19593 | 0.0 | 0.028761 |      2.0 | 0.0 |      0.0 | 0.867583 | 1.430516 | 1.078362 | 0.506208 |  0.09962 | 0.063276 | 0.011066 | 2.056631 | 2.232712 | 2.232712 | 0.620198 | 1.942479 | 0.0 | 0.0 | 178.301938 |  8.249269 | 21.741842 | 10.164974 |  9.985741 | 0.596975  0.599105  0.603691  0.611268  0.622155 | 0.620198 |
    | 03/01 06:00 |  0.4 | 16.0 | 15.0 |  0.154861 |  16.6 | 16.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  0.4752 |  0.179639 | 0.123337 | 0.123337 |  0.417679 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.417679 |  0.331967 | 0.0 | 0.061669 |      2.0 | 0.0 |      0.0 | 0.626287 | 1.412908 | 1.095103 |  0.52186 | 0.103744 | 0.065232 | 0.011125 |  1.83626 |  2.08463 |  2.08463 | 0.579064 | 1.876663 | 0.0 | 0.0 | 178.325981 |  5.954949 | 22.059646 | 10.583089 | 10.039848 | 0.534688  0.542001  0.554585  0.569574  0.583975 | 0.579064 |
    | 03/01 07:00 |  0.2 | 16.7 | 15.7 |  0.470369 |  17.3 | 17.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  0.2376 |  0.545628 | 0.383628 | 0.383628 |  0.114263 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.114263 |  0.090839 | 0.0 | 0.191814 |      2.0 | 0.0 |      0.0 | 0.385008 | 1.397018 | 1.110212 | 0.535984 | 0.107975 | 0.066998 | 0.011186 | 1.614381 | 1.921319 | 1.921319 |   0.5337 | 1.616372 | 0.0 | 0.0 | 178.157591 |   3.66078 | 22.346452 | 11.011099 |  10.09566 | 0.470412  0.478132  0.491957  0.509515  0.527869 |   0.5337 |
    | 03/01 08:00 |  0.0 | 17.1 | 16.1 |  1.173726 |  17.7 | 17.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |     0.0 |  1.361522 | 0.980296 | 0.980296 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.490148 |      2.0 | 0.0 |      0.0 | 0.158044 | 1.382677 | 1.123847 | 0.548731 | 0.112297 | 0.068591 | 0.011249 | 1.405437 | 1.713359 | 1.713359 | 0.475933 | 0.636076 | 0.0 | 0.0 | 177.667443 |  1.502736 | 22.605282 | 11.447534 | 10.153002 | 0.409695  0.417014   0.43023  0.447303  0.465722 | 0.475933 |
    | 03/01 09:00 |  0.0 | 16.2 | 15.2 |  4.202296 |  16.8 | 16.8 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |     0.0 |  4.874663 | 3.509758 | 0.636076 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  3.19172 | 1.502736 | 0.0 |      0.0 |      0.0 | 1.369736 | 1.136152 | 0.118222 | 0.114494 | 0.014778 | 0.011283 | 1.261929 |  1.50518 |  1.50518 | 0.418106 |      0.0 | 0.0 | 0.0 | 174.475723 |       0.0 | 22.838865 | 11.451262 | 10.156497 | 0.367434  0.372594  0.382132   0.39495  0.409603 | 0.418106 |
    | 03/01 10:00 |  0.3 | 15.9 | 14.9 |  4.359715 |  16.5 | 16.5 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  0.3564 |  5.057269 | 3.513746 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.335546 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 1.113865 |      0.0 | 0.113942 |      0.0 | 0.011279 | 1.239085 | 1.341362 | 1.341362 | 0.372601 |      0.0 | 0.0 | 0.0 | 171.140177 |       0.0 | 21.725001 |  11.33732 | 10.145218 | 0.359608  0.360691  0.363124   0.36734  0.373673 | 0.372601 |
    | 03/01 11:00 |  2.6 | 16.3 | 15.3 |  5.305753 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  3.0888 |  6.154673 | 3.253813 |      2.0 |    1.0888 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    1.0888 |  0.797246 | 0.0 | 2.253813 | 0.797246 | 0.0 |      0.0 |      0.0 | 0.797246 | 1.079144 |      0.0 | 0.112808 |      0.0 | 0.011266 | 1.203218 | 1.260419 | 1.260419 | 0.350117 |      0.0 | 0.0 | 0.0 | 169.177919 |       0.0 | 21.443102 | 11.224512 | 10.133952 | 0.349293  0.350529  0.352751  0.355659  0.359003 | 0.350117 |
    | 03/01 12:00 |  0.7 | 16.3 | 15.3 |  5.376027 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  0.8316 |  6.236191 | 4.131769 |   0.8316 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.715969 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 1.045792 |      0.0 | 0.111686 |      0.0 | 0.011254 | 1.168732 | 1.219583 | 1.219583 | 0.338773 |      0.0 | 0.0 | 0.0 |  165.46195 |       0.0 |  20.39731 | 11.112826 | 10.122698 | 0.339281  0.340487   0.34266  0.345462  0.348493 | 0.338773 |
    | 03/01 13:00 |  0.3 | 16.4 | 15.4 |  4.658915 |  17.0 | 17.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  0.3564 |  5.404341 | 3.754888 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.576688 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.994789 |      0.0 | 0.110574 |      0.0 | 0.011241 | 1.116604 | 1.183098 | 1.183098 | 0.328638 |      0.0 | 0.0 | 0.0 | 161.885262 |       0.0 | 19.402521 | 11.002251 | 10.111457 | 0.324336  0.326115  0.329248   0.33313  0.337061 | 0.328638 |
    | 03/01 14:00 |  0.3 | 16.5 | 15.5 |  7.789594 |  17.1 | 17.1 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  0.3564 |  9.035929 | 6.278083 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 6.099883 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.946272 |      0.0 | 0.109474 |      0.0 | 0.011229 | 1.066975 | 1.138602 | 1.138602 | 0.316278 |      0.0 | 0.0 | 0.0 | 155.785379 |       0.0 | 18.456249 | 10.892777 | 10.100228 | 0.309925  0.311661  0.314787  0.318802  0.323088 | 0.316278 |
    | 03/01 15:00 |  0.0 | 18.4 | 17.4 |  4.851567 |  19.0 | 19.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |     0.0 |  5.627818 | 4.052029 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.945293 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.900122 |      0.0 | 0.108385 |      0.0 | 0.011216 | 1.019723 | 1.089525 | 1.089525 | 0.302646 |      0.0 | 0.0 | 0.0 | 151.840086 |       0.0 | 17.556127 | 10.784392 | 10.089012 | 0.296198  0.297853   0.30084  0.304699  0.308871 | 0.302646 |
    | 03/01 16:00 |  0.0 | 18.3 | 17.3 |   5.30692 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |     0.0 |  6.156027 |  4.43234 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 4.206293 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.856222 |      0.0 | 0.107306 |      0.0 | 0.011204 | 0.974733 |  1.04138 |  1.04138 | 0.289272 |      0.0 | 0.0 | 0.0 | 147.633793 |       0.0 | 16.699905 | 10.677086 | 10.077808 | 0.283128  0.284704  0.287548  0.291226  0.295208 | 0.289272 |
    | 03/01 17:00 |  0.0 | 18.1 | 17.1 |  3.286036 |  18.7 | 18.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |     0.0 |  3.811802 | 2.744497 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.532378 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.814464 |      0.0 | 0.106239 |      0.0 | 0.011191 | 0.931894 |  0.99537 |  0.99537 | 0.276492 |      0.0 | 0.0 | 0.0 | 145.101415 |       0.0 | 15.885441 | 10.570847 | 10.066617 | 0.270683  0.272183  0.274892  0.278394  0.282186 | 0.276492 |
    | 03/01 18:00 |  0.0 | 16.7 | 15.7 |  1.506216 |  17.3 | 17.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |     0.0 |  1.747211 | 1.257992 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.140852 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.774742 |      0.0 | 0.105182 |      0.0 | 0.011179 | 0.891103 | 0.951545 | 0.951545 | 0.264318 |      0.0 | 0.0 | 0.0 | 143.960563 |       0.0 | 15.110699 | 10.465665 | 10.055438 | 0.258833  0.260261   0.26284  0.266175  0.269786 | 0.264318 |
    | 03/01 19:00 |  0.0 | 15.2 | 14.2 |  0.274762 |  15.8 | 15.8 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |     0.0 |  0.318724 | 0.229481 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.206477 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.736957 |      0.0 | 0.104135 |      0.0 | 0.011167 | 0.852259 | 0.909814 | 0.909814 | 0.252726 |      0.0 | 0.0 | 0.0 | 143.754086 |       0.0 | 14.373741 |  10.36153 | 10.044271 | 0.247549  0.248909  0.251365   0.25454  0.257978 | 0.252726 |
    | 03/01 20:00 |  0.0 | 13.4 | 12.4 |  0.087565 |  14.0 | 14.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |     0.0 |  0.101575 | 0.073134 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.065708 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.701016 |      0.0 | 0.103099 |      0.0 | 0.011154 | 0.815269 | 0.870077 | 0.870077 | 0.241688 |      0.0 | 0.0 | 0.0 | 143.688378 |       0.0 | 13.672726 | 10.258431 | 10.033117 | 0.236803  0.238098  0.240437   0.24346  0.246735 | 0.241688 |
    | 03/01 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |     0.0 |  0.099494 | 0.071636 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.064333 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.666827 |      0.0 | 0.102073 |      0.0 | 0.011142 | 0.780042 | 0.832236 | 0.832236 | 0.231177 |      0.0 | 0.0 | 0.0 | 143.624045 |       0.0 | 13.005899 | 10.156358 | 10.021975 | 0.226569  0.227802   0.23003  0.232909  0.236027 | 0.231177 |
    | 03/01 22:00 |  0.0 | 11.6 | 10.6 |  0.084317 |  12.2 | 12.2 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |     0.0 |  0.097808 | 0.070422 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.063214 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.634305 |      0.0 | 0.101057 |      0.0 | 0.011129 | 0.746492 |   0.7962 |   0.7962 | 0.221167 |      0.0 | 0.0 | 0.0 | 143.560831 |       0.0 | 12.371594 | 10.055301 | 10.010846 | 0.216823  0.217997  0.220118  0.222861   0.22583 | 0.221167 |
    | 03/01 23:00 |  0.0 | 11.0 | 10.0 |  0.083215 |  11.6 | 11.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |     0.0 |  0.096529 | 0.069501 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.06236 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |  0.60337 |      0.0 | 0.100052 |      0.0 | 0.011117 | 0.714539 | 0.761881 | 0.761881 | 0.211633 |      0.0 | 0.0 | 0.0 | 143.498471 |       0.0 | 11.768224 |  9.955249 |  9.999729 |  0.20754  0.208659  0.210679  0.213291  0.216119 | 0.211633 |
    | 04/01 00:00 |  0.0 | 10.5 |  9.5 |  0.082289 |  11.1 | 11.1 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |     0.0 |  0.095455 | 0.068728 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.06164 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.573943 |      0.0 | 0.099056 |      0.0 | 0.011105 | 0.684104 | 0.729195 | 0.729195 | 0.202554 |      0.0 | 0.0 | 0.0 | 143.436831 |       0.0 | 11.194281 |  9.856192 |  9.988625 | 0.198699  0.199764  0.201688  0.204176   0.20687 | 0.202554 |
    | 04/01 01:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |     0.0 |   0.09802 | 0.070574 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.063269 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.545952 |      0.0 | 0.098071 |      0.0 | 0.011092 | 0.655115 | 0.698063 | 0.698063 | 0.193906 |      0.0 | 0.0 | 0.0 | 143.373563 |       0.0 |  10.64833 |  9.758122 |  9.977532 | 0.190277  0.191292  0.193125  0.195494   0.19806 | 0.193906 |
    | 04/01 02:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |     0.0 |  0.098442 | 0.070878 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.063513 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.519325 |      0.0 | 0.097095 |      0.0 |  0.01108 |   0.6275 | 0.668411 | 0.668411 |  0.18567 |      0.0 | 0.0 | 0.0 |  143.31005 |       0.0 | 10.129004 |  9.661027 |  9.966452 | 0.182255  0.183222  0.184968  0.187225  0.189669 |  0.18567 |
    | 04/01 03:00 |  1.3 | 11.2 | 10.2 |  0.083584 |  11.8 | 11.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |  1.5444 |  0.096957 | 0.059819 | 0.059819 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.02679 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.493997 |      0.0 | 0.096129 |      0.0 | 0.011068 | 0.601194 | 0.640165 | 0.640165 | 0.177824 | 1.484581 | 0.0 | 0.0 |  143.28326 |       0.0 |  9.635007 |  9.564898 |  9.955385 | 0.174613  0.175534  0.177197  0.179347  0.181675 | 0.177824 |
    | 04/01 04:00 |  0.0 | 11.1 | 10.1 |    0.0834 |  11.7 | 11.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |     0.0 |  0.096744 | 0.069656 | 0.069656 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.031189 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.469905 |      0.0 | 0.095172 |      0.0 | 0.011055 | 0.576133 | 0.613259 | 0.613259 |  0.17035 | 1.414925 | 0.0 | 0.0 | 143.252071 |       0.0 |  9.165102 |  9.469726 |  9.944329 | 0.167332   0.16821  0.169794  0.171843  0.174061 |  0.17035 |
    | 04/01 05:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |     0.0 |  0.098442 | 0.070878 | 0.070878 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.03173 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.446987 |      0.0 | 0.094225 |      0.0 | 0.011043 | 0.552256 | 0.587626 | 0.587626 |  0.16323 | 1.344047 | 0.0 | 0.0 | 143.220341 |       0.0 |  8.718115 |    9.3755 |  9.933286 | 0.160396  0.161232  0.162742  0.164693  0.166806 |  0.16323 |
    | 04/01 06:00 |  0.0 | 12.2 | 11.2 |  0.310229 |  12.8 | 12.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |     0.0 |  0.359866 | 0.259103 | 0.259103 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.115965 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.425187 |      0.0 | 0.093288 |      0.0 | 0.011031 | 0.529506 | 0.563206 | 0.563206 | 0.156446 | 1.084943 | 0.0 | 0.0 | 143.104376 |       0.0 |  8.292927 |  9.282212 |  9.922255 | 0.153787  0.154584  0.156022  0.157881  0.159895 | 0.156446 |
    | 04/01 07:00 |  0.7 | 11.8 | 10.8 |  1.391958 |  12.4 | 12.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |  0.8316 |  1.614671 | 1.069795 | 1.069795 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.478414 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.404451 |      0.0 |  0.09236 |      0.0 | 0.011019 | 0.507829 | 0.539939 | 0.539939 | 0.149983 | 0.846748 | 0.0 | 0.0 | 142.625962 |       0.0 |  7.888477 |  9.189853 |  9.911237 |  0.14749  0.148249  0.149619  0.151391  0.153309 | 0.149983 |
    | 04/01 08:00 |  0.4 | 11.4 | 10.4 |  3.195876 |  12.0 | 12.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |  0.4752 |  3.707216 | 2.545322 | 1.321948 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.746153 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.384726 |      0.0 | 0.091441 |      0.0 | 0.011006 | 0.487172 |  0.51777 |  0.51777 | 0.143825 |      0.0 | 0.0 | 0.0 |  140.87981 |       0.0 |  7.503751 |  9.098412 |   9.90023 | 0.141489  0.142213  0.143518  0.145207  0.147035 | 0.143825 |
    | 04/01 09:00 |  0.1 | 11.6 | 10.6 |  5.191651 |  12.2 | 12.2 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |  0.1188 |  6.022315 | 4.284859 |   0.1188 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.772813 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.365962 |      0.0 | 0.090531 |      0.0 | 0.010994 | 0.467487 | 0.496645 | 0.496645 | 0.137957 |      0.0 | 0.0 | 0.0 | 137.106996 |       0.0 |  7.137789 |  9.007882 |  9.889236 | 0.135771   0.13646  0.137704  0.139313  0.141055 | 0.137957 |
    | 04/01 10:00 |  0.4 | 13.0 | 12.0 |  7.155036 |  13.6 | 13.6 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |  0.4752 |  8.299842 | 5.698554 |   0.4752 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 4.883197 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.348114 |      0.0 |  0.08963 |      0.0 | 0.010982 | 0.448726 | 0.476514 | 0.476514 | 0.132365 |      0.0 | 0.0 | 0.0 | 132.223799 |       0.0 |  6.789675 |  8.918252 |  9.878254 | 0.130321  0.130977  0.132164  0.133697  0.135357 | 0.132365 |
    | 04/01 11:00 |  0.0 | 17.1 | 16.1 |  8.391432 |  17.7 | 17.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |     0.0 |  9.734061 | 7.008524 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.791835 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.331136 |      0.0 | 0.088738 |      0.0 |  0.01097 | 0.430844 | 0.457329 | 0.457329 | 0.127036 |      0.0 | 0.0 | 0.0 | 126.431964 |       0.0 |  6.458538 |  8.829514 |  9.867284 | 0.125126  0.125752  0.126882  0.128344  0.129926 | 0.127036 |
    | 04/01 12:00 |  0.0 | 18.2 | 17.2 |  8.391286 |  18.8 | 18.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |     0.0 |  9.733892 | 7.008402 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.538038 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.314987 |      0.0 | 0.087855 |      0.0 | 0.010958 | 0.413799 | 0.439044 | 0.439044 | 0.121957 |      0.0 | 0.0 | 0.0 | 120.893926 |       0.0 |  6.143552 |  8.741659 |  9.856327 | 0.120175  0.120771  0.121849  0.123242   0.12475 | 0.121957 |
    | 04/01 13:00 |  0.0 | 22.4 | 21.4 | 10.715238 |  23.0 | 23.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |     0.0 | 12.429676 | 8.949367 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 6.762026 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.299625 |      0.0 | 0.086981 |      0.0 | 0.010945 | 0.397551 | 0.421615 | 0.421615 | 0.117115 |      0.0 | 0.0 | 0.0 |   114.1319 |       0.0 |  5.843927 |  8.654678 |  9.845381 | 0.115454  0.116023   0.11705  0.118378  0.119816 | 0.117115 |
    | 04/01 14:00 |  0.0 | 21.4 | 20.4 |  9.383394 |  22.0 | 22.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |     0.0 | 10.884737 | 7.837011 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.590331 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.285012 |      0.0 | 0.086115 |      0.0 | 0.010933 |  0.38206 | 0.405001 | 0.405001 |   0.1125 |      0.0 | 0.0 | 0.0 |  108.54157 |       0.0 |  5.558915 |  8.568562 |  9.834448 | 0.110954  0.111497  0.112476  0.113742  0.115112 |   0.1125 |
    | 04/01 15:00 |  0.0 | 21.8 | 20.8 |  7.861915 |  22.4 | 22.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |     0.0 |  9.119821 | 6.566271 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 4.454459 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.271112 |      0.0 | 0.085259 |      0.0 | 0.010921 | 0.367291 | 0.389163 | 0.389163 | 0.108101 |      0.0 | 0.0 | 0.0 | 104.087111 |       0.0 |  5.287804 |  8.483303 |  9.823527 | 0.106664  0.107181  0.108115  0.109322  0.110628 | 0.108101 |
    | 04/01 16:00 |  0.0 | 22.2 | 21.2 |  6.298329 |  22.8 | 22.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |     0.0 |  7.306062 | 5.260364 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.422101 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.257889 |      0.0 |  0.08441 |      0.0 | 0.010909 | 0.353208 | 0.374063 | 0.374063 | 0.103906 |      0.0 | 0.0 | 0.0 |  100.66501 |       0.0 |  5.029915 |  8.398893 |  9.812618 | 0.102573  0.103066  0.103956  0.105107  0.106353 | 0.103906 |
    | 04/01 17:00 |  0.0 | 20.1 | 19.1 |  2.948416 |  20.7 | 20.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |     0.0 |  3.420163 | 2.462517 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.549308 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.245312 |      0.0 |  0.08357 |      0.0 | 0.010897 | 0.339779 | 0.359665 | 0.359665 | 0.099907 |      0.0 | 0.0 | 0.0 |  99.115702 |       0.0 |  4.784603 |  8.315323 |  9.801721 | 0.098672  0.099142  0.099991  0.101088  0.102276 | 0.099907 |
    | 04/01 18:00 |  0.0 | 17.8 | 16.8 |  1.309232 |  18.4 | 18.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |     0.0 |  1.518709 | 1.093471 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.677376 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.233348 |      0.0 | 0.082739 |      0.0 | 0.010885 | 0.326971 | 0.345936 | 0.345936 | 0.096093 |      0.0 | 0.0 | 0.0 |  98.438326 |       0.0 |  4.551255 |  8.232584 |  9.790836 | 0.094951    0.0954  0.096209  0.097256  0.098389 | 0.096093 |
    | 04/01 19:00 |  0.0 | 15.2 | 14.2 |   0.32955 |  15.8 | 15.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |     0.0 |  0.382278 |  0.27524 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.169339 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.221967 |      0.0 | 0.081916 |      0.0 | 0.010873 | 0.314756 | 0.332843 | 0.332843 | 0.092456 |      0.0 | 0.0 | 0.0 |  98.268988 |       0.0 |  4.329288 |  8.150668 |  9.779964 | 0.091403  0.091831  0.092603  0.093601  0.094681 | 0.092456 |
    | 04/01 20:00 |  0.0 | 14.5 | 13.5 |  0.089508 |  15.1 | 15.1 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |     0.0 |  0.103829 | 0.074757 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.045914 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.211142 |      0.0 | 0.081101 |      0.0 | 0.010861 | 0.303103 | 0.320355 | 0.320355 | 0.088988 |      0.0 | 0.0 | 0.0 |  98.223073 |       0.0 |  4.118146 |  8.069568 |  9.769103 | 0.088018  0.088426  0.089162  0.090114  0.091145 | 0.088988 |
    | 04/01 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |     0.0 |  0.099494 | 0.071636 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.043977 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.200844 |      0.0 | 0.080294 |      0.0 | 0.010849 | 0.291986 | 0.308444 | 0.308444 | 0.085679 |      0.0 | 0.0 | 0.0 |  98.179096 |       0.0 |  3.917302 |  7.989274 |  9.758255 | 0.084789  0.085178  0.085881  0.086789  0.087772 | 0.085679 |
    | 04/01 22:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |     0.0 |   0.09802 | 0.070574 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.043306 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.191049 |      0.0 | 0.079495 |      0.0 | 0.010836 |  0.28138 | 0.297082 | 0.297082 | 0.082523 |      0.0 | 0.0 | 0.0 |   98.13579 |       0.0 |  3.726252 |   7.90978 |  9.747418 | 0.081708  0.082079  0.082749  0.083616  0.084554 | 0.082523 |
    | 04/01 23:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |     0.0 |  0.098442 | 0.070878 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.043473 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.181731 |      0.0 | 0.078704 |      0.0 | 0.010824 |  0.27126 | 0.286242 | 0.286242 | 0.079512 |      0.0 | 0.0 | 0.0 |  98.092317 |       0.0 |  3.544521 |  7.831076 |  9.736594 | 0.078768  0.079122  0.079762  0.080589  0.081484 | 0.079512 |

There is no indication of an error in the water balance:

>>> from hydpy import round_
>>> round_(model.check_waterbalance(conditions))
0.0

Regarding numerical accuracy of the linear storage cascade, please see the
:ref:`hland_v2_low_accuracy` example of |hland_v2|, which also applies the storage
cascade instead of the triangular Unit Hydrograph.

.. _hland_v3_ilake:

internal lake
_____________

Like in the :ref:`hland_v1_ilake` example of |hland_v1|, we now focus on |ILAKE| zones.
We understand the combination of the slow-response groundwater reservoirs |SG2| and
|SG3| as the surrogate for HBV96's lower zone storage (|LZ|).  Hence, we add
precipitation falling on "internal lakes" to both storages and remove lake evaporation
from both storages.  As for percolation, parameter |FSG| determines how |hland_v3|
divides precipitation and evaporation between both storages:

.. integration-test::

    >>> zonetype(ILAKE)
    >>> ttice(13.0)
    >>> test('hland_v3_ilake')
    |        date |    p |    t |   tn |       epn | tmean |   tc | fracrain | rfc | sfc | cfact | swe | gact |      pc |        ep |      epc |  ei |      tf | spl | wcl | spg | wcg | glmelt | melt | refr |     in_ |       r |  sr |  ea |  dp |       el |  rs |  ri | gr1 | rg1 |     gr2 |      rg2 |    gr3 |      rg3 |     inuh |    outuh |       rt |       qt |  ic |  sp |  wc |  sm | suz | sg1 |        sg2 |       sg3 |                                               sc |   outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | 21.2 | 20.2 |  0.100707 |  21.8 | 21.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |   0.11682 |  0.08411 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.08411 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.099502 |    0.0 | 0.011105 | 0.110607 | 0.168002 | 0.168002 | 0.046667 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.825733 |  9.979549 |  0.03268  0.034694  0.038083  0.041925  0.045222 | 0.046667 |
    | 01/01 01:00 |  0.0 | 19.4 | 18.4 |  0.097801 |  20.0 | 20.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  0.113449 | 0.081683 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.081683 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.097768 |    0.0 | 0.011082 |  0.10885 |  0.13693 |  0.13693 | 0.038036 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.655358 |  9.959391 | 0.031604  0.031797  0.032348  0.033478  0.035298 | 0.038036 |
    | 01/01 02:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  0.112498 | 0.080999 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.080999 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.096072 |    0.0 |  0.01106 | 0.107132 | 0.114599 | 0.114599 | 0.031833 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.487287 |  9.939332 | 0.031086  0.031153  0.031295  0.031546  0.031978 | 0.031833 |
    | 01/01 03:00 |  0.0 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  0.111348 | 0.080171 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.080171 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 |   0.0944 |    0.0 | 0.011038 | 0.105438 | 0.108492 | 0.108492 | 0.030137 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.321624 |  9.919386 | 0.030594  0.030654  0.030762  0.030908  0.031086 | 0.030137 |
    | 01/01 04:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  0.112498 | 0.080999 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.080999 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.092752 |    0.0 | 0.011015 | 0.103767 | 0.106256 | 0.106256 | 0.029515 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.156874 |  9.899371 | 0.030109  0.030168  0.030273  0.030409  0.030557 | 0.029515 |
    | 01/01 05:00 |  0.0 | 22.5 | 21.5 |  0.102761 |  23.1 | 23.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  0.119203 | 0.085826 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.085826 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.091112 |    0.0 | 0.010993 | 0.102106 | 0.104526 | 0.104526 | 0.029035 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   8.989472 |  9.878841 | 0.029627  0.029685   0.02979  0.029924  0.030069 | 0.029035 |
    | 01/01 06:00 |  0.0 | 25.1 | 24.1 |  0.291908 |  25.7 | 25.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  0.338613 | 0.243802 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.243802 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.089447 |    0.0 |  0.01097 | 0.100417 | 0.102853 | 0.102853 |  0.02857 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   8.683312 |  9.840782 | 0.029138  0.029197  0.029303  0.029438  0.029584 |  0.02857 |
    | 01/01 07:00 |  0.0 | 28.3 | 27.3 |  1.932875 |  28.9 | 28.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  2.242135 | 1.614337 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 1.614337 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 |   0.0864 |    0.0 | 0.010928 | 0.097329 | 0.101075 | 0.101075 | 0.028076 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   7.161946 |  9.650483 | 0.028256  0.028361  0.028543  0.028767  0.028987 | 0.028076 |
    | 01/01 08:00 |  0.0 | 27.8 | 26.8 |  4.369536 |  28.4 | 28.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  5.068662 | 3.649436 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 3.649436 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.071263 |    0.0 | 0.010717 | 0.081979 | 0.097712 | 0.097712 | 0.027142 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   3.846739 |  9.234273 | 0.023925  0.024432  0.025298  0.026306   0.02722 | 0.027142 |
    | 01/01 09:00 |  0.0 | 31.4 | 30.4 |  7.317556 |  32.0 | 32.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  8.488365 | 6.111623 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 6.111623 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.038276 |    0.0 | 0.010255 |  0.04853 | 0.086434 | 0.086434 | 0.024009 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   -1.62409 |   8.54495 | 0.014406   0.01553  0.017479  0.019817  0.022046 | 0.024009 |
    | 01/01 10:00 |  0.0 | 32.2 | 31.2 |  8.264362 |  32.8 | 32.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |   9.58666 | 6.902395 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 6.902395 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 |      0.0 |    0.0 | 0.009489 | 0.009489 |  0.06136 |  0.06136 | 0.017044 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -7.759552 |  7.768528 | 0.003145  0.004492  0.006888  0.009896  0.012987 | 0.017044 |
    | 01/01 11:00 |  0.0 | 35.2 | 34.2 |  9.369867 |  35.8 | 35.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 | 10.869046 | 7.825713 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 7.825713 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 |      0.0 |    0.0 | 0.008627 | 0.008627 | 0.029056 | 0.029056 | 0.008071 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -14.715741 |  6.890377 | 0.002523  0.002641  0.002995  0.003758   0.00506 | 0.008071 |
    | 01/01 12:00 |  0.0 | 37.1 | 36.1 |  5.126178 |  37.7 | 37.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  5.946366 | 4.281384 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 4.281384 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 |      0.0 |    0.0 | 0.007652 | 0.007652 | 0.012498 | 0.012498 | 0.003472 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -18.521416 |  6.407016 | 0.002229  0.002267   0.00235  0.002503  0.002781 | 0.003472 |
    | 01/01 13:00 |  0.0 | 31.2 | 30.2 |   6.62503 |  31.8 | 31.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  7.685035 | 5.533225 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 5.533225 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 |      0.0 |    0.0 | 0.007115 | 0.007115 | 0.008511 | 0.008511 | 0.002364 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -23.439838 |  5.785098 | 0.002069  0.002089  0.002127  0.002185  0.002265 | 0.002364 |
    | 01/01 14:00 |  0.0 | 24.3 | 23.3 |  7.397619 |  24.9 | 24.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  8.581238 | 6.178491 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 6.178491 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 |      0.0 |    0.0 | 0.006424 | 0.006424 |  0.00742 |  0.00742 | 0.002061 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -28.931831 |  5.092175 |  0.00187  0.001894  0.001936   0.00199  0.002049 | 0.002061 |
    | 01/01 15:00 |  0.2 | 25.4 | 24.4 |   2.39151 |  26.0 | 26.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.2376 |  2.774152 | 1.950491 | 0.0 |  0.2376 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.2376 |  0.2376 | 0.0 | 0.0 | 0.0 | 1.950491 | 0.0 | 0.0 | 0.0 | 0.0 |  0.2112 |      0.0 | 0.0264 | 0.005669 | 0.005669 | 0.006723 | 0.006723 | 0.001868 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   -30.4544 |  4.896184 | 0.001652  0.001678  0.001725  0.001784  0.001847 | 0.001868 |
    | 01/01 16:00 |  0.0 | 25.9 | 24.9 |  1.829834 |  26.5 | 26.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  2.122607 | 1.528277 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 1.528277 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 |      0.0 |    0.0 | 0.005437 | 0.005437 | 0.006039 | 0.006039 | 0.001678 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -31.812869 |  4.720939 | 0.001579  0.001589  0.001607  0.001636  0.001673 | 0.001678 |
    | 01/01 17:00 |  0.0 | 23.7 | 22.7 |  1.136569 |  24.3 | 24.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |   1.31842 | 0.949262 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.949262 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 |      0.0 |    0.0 | 0.005243 | 0.005243 | 0.005592 | 0.005592 | 0.001553 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -32.656658 |  4.610222 | 0.001522  0.001529  0.001542   0.00156   0.00158 | 0.001553 |
    | 01/01 18:00 |  1.3 | 21.6 | 20.6 |  0.750986 |  22.2 | 22.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  1.5444 |  0.871144 | 0.537465 | 0.0 |  1.5444 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  1.5444 |  1.5444 | 0.0 | 0.0 | 0.0 | 0.537465 | 0.0 | 0.0 | 0.0 | 0.0 |  1.3728 |      0.0 | 0.1716 | 0.005215 | 0.005215 | 0.005351 | 0.005351 | 0.001486 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -31.761604 |  4.716889 | 0.001513  0.001514  0.001517  0.001523  0.001531 | 0.001486 |
    | 01/01 19:00 |  5.6 | 21.2 | 20.2 |  0.223895 |  21.8 | 21.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  6.6528 |  0.259718 | 0.096141 | 0.0 |  6.6528 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  6.6528 |  6.6528 | 0.0 | 0.0 | 0.0 | 0.096141 | 0.0 | 0.0 | 0.0 | 0.0 |  5.9136 |      0.0 | 0.7392 | 0.005649 | 0.005649 | 0.005277 | 0.005277 | 0.001466 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -25.933462 |  5.439758 | 0.001634   0.00162  0.001596   0.00157  0.001549 | 0.001466 |
    | 01/01 20:00 |  2.9 | 20.4 | 19.4 |  0.099425 |  21.0 | 21.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  3.4452 |  0.115333 | 0.058839 | 0.0 |  3.4452 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.4452 |  3.4452 | 0.0 | 0.0 | 0.0 | 0.058839 | 0.0 | 0.0 | 0.0 | 0.0 |  3.0624 |      0.0 | 0.3828 | 0.006253 | 0.006253 | 0.005511 | 0.005511 | 0.001531 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -22.923363 |  5.809767 | 0.001807  0.001787   0.00175  0.001706  0.001662 | 0.001531 |
    | 01/01 21:00 |  4.9 | 19.8 | 18.8 |  0.098454 |  20.4 | 20.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  5.8212 |  0.114207 | 0.045942 | 0.0 |  5.8212 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.8212 |  5.8212 | 0.0 | 0.0 | 0.0 | 0.045942 | 0.0 | 0.0 | 0.0 | 0.0 |  5.1744 |      0.0 | 0.6468 | 0.006811 | 0.006811 | 0.005999 | 0.005999 | 0.001666 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -17.789801 |  6.444652 |  0.00197   0.00195  0.001915  0.001869  0.001821 | 0.001666 |
    | 01/01 22:00 | 10.6 | 19.6 | 18.6 |  0.098128 |  20.2 | 20.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 | 12.5928 |  0.113828 | 0.023264 | 0.0 | 12.5928 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.5928 | 12.5928 | 0.0 | 0.0 | 0.0 | 0.023264 | 0.0 | 0.0 | 0.0 | 0.0 | 11.1936 |      0.0 | 1.3992 | 0.007934 | 0.007934 | 0.006598 | 0.006598 | 0.001833 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   -6.61688 |  7.833333 |  0.00229  0.002252  0.002186  0.002106  0.002027 | 0.001833 |
    | 01/01 23:00 |  0.1 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.1188 |   0.11307 | 0.080449 | 0.0 |  0.1188 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.1188 |  0.1188 | 0.0 | 0.0 | 0.0 | 0.080449 | 0.0 | 0.0 | 0.0 | 0.0 |  0.1056 |      0.0 | 0.0132 | 0.008706 | 0.008706 | 0.007456 | 0.007456 | 0.002071 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   -6.58279 |  7.828888 | 0.002517  0.002489  0.002438  0.002371  0.002295 | 0.002071 |
    | 02/01 00:00 |  0.7 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.8316 |   0.11307 | 0.074914 | 0.0 |  0.8316 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.8316 |  0.8316 | 0.0 | 0.0 | 0.0 | 0.074914 | 0.0 | 0.0 | 0.0 | 0.0 |  0.7392 |      0.0 | 0.0924 | 0.008745 | 0.008745 | 0.008279 | 0.008279 |   0.0023 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   -5.91018 |  7.904219 | 0.002535  0.002532  0.002524  0.002507  0.002477 |   0.0023 |
    | 02/01 01:00 |  3.0 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |   3.564 |   0.11307 | 0.057003 | 0.0 |   3.564 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   3.564 |   3.564 | 0.0 | 0.0 | 0.0 | 0.057003 | 0.0 | 0.0 | 0.0 | 0.0 |   3.168 |      0.0 |  0.396 | 0.008998 | 0.008998 | 0.008671 | 0.008671 | 0.002409 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   -2.79285 |  8.284888 | 0.002607  0.002598  0.002584  0.002566  0.002547 | 0.002409 |
    | 02/01 02:00 |  2.1 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  2.4948 |  0.112498 | 0.063115 | 0.0 |  2.4948 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.4948 |  2.4948 | 0.0 | 0.0 | 0.0 | 0.063115 | 0.0 | 0.0 | 0.0 | 0.0 |  2.2176 |      0.0 | 0.2772 | 0.009354 | 0.009354 | 0.008903 | 0.008903 | 0.002473 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -0.631351 |  8.545721 | 0.002709  0.002697  0.002676  0.002649  0.002622 | 0.002473 |
    | 02/01 03:00 | 10.4 | 18.7 | 17.7 |  0.096652 |  19.3 | 19.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 | 12.3552 |  0.112116 | 0.023465 | 0.0 | 12.3552 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.3552 | 12.3552 | 0.0 | 0.0 | 0.0 | 0.023465 | 0.0 | 0.0 | 0.0 | 0.0 | 10.9824 | 0.002975 | 1.3728 | 0.010252 | 0.013227 | 0.009456 | 0.009456 | 0.002627 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  10.327216 |  9.905661 | 0.003798  0.003671  0.003456  0.003209  0.002991 | 0.002627 |
    | 02/01 04:00 |  3.5 | 18.5 | 17.5 |  0.096321 |  19.1 | 19.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |   4.158 |  0.111732 |  0.05308 | 0.0 |   4.158 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   4.158 |   4.158 | 0.0 | 0.0 | 0.0 |  0.05308 | 0.0 | 0.0 | 0.0 | 0.0 |   3.696 | 0.121176 |  0.462 | 0.011257 | 0.132433 | 0.020047 | 0.020047 | 0.005569 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  13.854858 | 10.350507 | 0.037239  0.033347  0.026781  0.019307  0.012837 | 0.005569 |
    | 02/01 05:00 |  3.4 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  4.0392 |  0.111348 |  0.05353 | 0.0 |  4.0392 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.0392 |  4.0392 | 0.0 | 0.0 | 0.0 |  0.05353 | 0.0 | 0.0 | 0.0 | 0.0 |  3.5904 |  0.15575 | 0.4488 | 0.011743 | 0.167494 | 0.083601 | 0.083601 | 0.023222 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  17.241925 | 10.781615 | 0.048191  0.046787  0.043985   0.03983  0.034611 | 0.023222 |
    | 02/01 06:00 |  1.2 | 18.5 | 17.5 |  0.187298 |  19.1 | 19.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  1.4256 |  0.217266 | 0.135647 | 0.0 |  1.4256 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  1.4256 |  1.4256 | 0.0 | 0.0 | 0.0 | 0.135647 | 0.0 | 0.0 | 0.0 | 0.0 |  1.2672 | 0.177875 | 0.1584 | 0.012061 | 0.189936 |  0.14418 |  0.14418 |  0.04005 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  18.210675 | 10.912883 | 0.054849  0.054028  0.052473  0.050275  0.047534 |  0.04005 |
    | 02/01 07:00 |  0.1 | 18.8 | 17.8 |  1.264612 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.1188 |   1.46695 |  1.04373 | 0.0 |  0.1188 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.1188 |  0.1188 | 0.0 | 0.0 | 0.0 |  1.04373 | 0.0 | 0.0 | 0.0 | 0.0 |  0.1056 | 0.181725 | 0.0132 | 0.012126 | 0.193852 | 0.176547 | 0.176547 | 0.049041 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  17.206789 | 10.797987 | 0.056171  0.055989  0.055586  0.054889  0.053828 | 0.049041 |
    | 02/01 08:00 |  0.0 | 18.8 | 17.8 |  3.045538 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  3.532824 | 2.543633 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 2.543633 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 |  0.17121 |    0.0 | 0.011991 | 0.183202 | 0.189114 | 0.189114 | 0.052532 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  14.774571 |  10.50337 | 0.053231  0.053567  0.054108  0.054657  0.054987 | 0.052532 |
    | 02/01 09:00 |  0.0 | 19.0 | 18.0 |  1.930758 |  19.6 | 19.6 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  2.239679 | 1.612569 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 1.612569 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.147009 |    0.0 | 0.011664 | 0.158673 | 0.185428 | 0.185428 | 0.051508 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  13.194167 | 10.312531 | 0.046258   0.04708  0.048501   0.05019  0.051765 | 0.051508 |
    | 02/01 10:00 |  0.4 | 19.2 | 18.2 |  2.461001 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.4752 |  2.854761 | 1.960038 | 0.0 |  0.4752 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.4752 |  0.4752 | 0.0 | 0.0 | 0.0 | 1.960038 | 0.0 | 0.0 | 0.0 | 0.0 |  0.4224 | 0.133389 | 0.0528 | 0.011481 |  0.14487 | 0.169007 | 0.169007 | 0.046946 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  11.740921 | 10.136068 | 0.042156   0.04266  0.043603  0.044885  0.046354 | 0.046946 |
    | 02/01 11:00 |  0.1 | 19.3 | 18.3 |  6.215945 |  19.9 | 19.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.1188 |  7.210496 | 5.130246 | 0.0 |  0.1188 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.1188 |  0.1188 | 0.0 | 0.0 | 0.0 | 5.130246 | 0.0 | 0.0 | 0.0 | 0.0 |  0.1056 |  0.11735 | 0.0132 | 0.011263 | 0.128614 | 0.151747 | 0.151747 | 0.042152 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   7.168952 |  9.567977 | 0.037462  0.038024  0.039032  0.040317  0.041691 | 0.042152 |
    | 02/01 12:00 |  3.6 | 19.0 | 18.0 |  3.374783 |  19.6 | 19.6 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  4.2768 |  3.914748 | 1.837796 | 0.0 |  4.2768 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.2768 |  4.2768 | 0.0 | 0.0 | 0.0 | 1.837796 | 0.0 | 0.0 | 0.0 | 0.0 |  3.8016 | 0.090277 | 0.4752 | 0.010889 | 0.101166 | 0.135001 | 0.135001 |   0.0375 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.246678 |  9.828089 | 0.029612  0.030544  0.032177  0.034181  0.036177 |   0.0375 |
    | 02/01 13:00 |  5.9 | 18.8 | 17.8 |  8.821555 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  7.0092 | 10.233004 | 3.655358 | 0.0 |  7.0092 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  7.0092 |  7.0092 | 0.0 | 0.0 | 0.0 | 3.655358 | 0.0 | 0.0 | 0.0 | 0.0 |  6.2304 | 0.123054 | 0.7788 | 0.011347 | 0.134401 | 0.116786 | 0.116786 | 0.032441 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  12.104817 | 10.189391 | 0.038661   0.03764  0.036029  0.034449  0.033527 | 0.032441 |
    | 02/01 14:00 |  1.1 | 18.7 | 17.7 |  4.046025 |  19.3 | 19.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  1.3068 |  4.693389 | 2.965278 | 0.0 |  1.3068 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  1.3068 |  1.3068 | 0.0 | 0.0 | 0.0 | 2.965278 | 0.0 | 0.0 | 0.0 | 0.0 |  1.1616 | 0.126234 | 0.1452 | 0.011396 |  0.13763 | 0.122854 | 0.122854 | 0.034126 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   10.50438 |   9.99372 |  0.03987  0.039695   0.03929  0.038594  0.037632 | 0.034126 |
    | 02/01 15:00 | 20.7 | 17.8 | 16.8 |  2.110757 |  18.4 | 18.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 | 24.5916 |  2.448478 |  0.15074 | 0.0 | 24.5916 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 24.5916 | 24.5916 | 0.0 | 0.0 | 0.0 |  0.15074 | 0.0 | 0.0 | 0.0 | 0.0 | 21.8592 | 0.213453 | 2.7324 | 0.012615 | 0.226068 | 0.140089 | 0.140089 | 0.038914 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  32.016136 | 12.696755 | 0.064694  0.061802   0.05691  0.051302  0.046352 | 0.038914 |
    | 02/01 16:00 | 37.9 | 17.4 | 16.4 |  2.239257 |  18.0 | 18.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 | 45.0252 |  2.597538 | 0.020724 | 0.0 | 45.0252 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 45.0252 | 45.0252 | 0.0 | 0.0 | 0.0 | 0.020724 | 0.0 | 0.0 | 0.0 | 0.0 | 40.0224 | 0.518013 | 5.0028 | 0.016878 |  0.53489 | 0.209702 | 0.209702 |  0.05825 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  71.502102 | 17.680375 | 0.152071  0.141814  0.124224  0.103558  0.084581 |  0.05825 |
    | 02/01 17:00 |  8.2 | 17.3 | 16.3 |  2.877848 |  17.9 | 17.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  9.7416 |  3.338304 | 0.907373 | 0.0 |  9.7416 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  9.7416 |  9.7416 | 0.0 | 0.0 | 0.0 | 0.907373 | 0.0 | 0.0 | 0.0 | 0.0 |  8.6592 |  0.75461 | 1.0824 | 0.020235 | 0.774845 | 0.412484 | 0.412484 | 0.114579 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  78.600138 | 18.641721 | 0.222257  0.213751   0.19827   0.17809  0.156242 | 0.114579 |
    | 02/01 18:00 |  3.6 | 16.8 | 15.8 |  1.591452 |  17.4 | 17.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  4.2768 |  1.846084 | 0.866652 | 0.0 |  4.2768 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.2768 |  4.2768 | 0.0 | 0.0 | 0.0 | 0.866652 | 0.0 | 0.0 | 0.0 | 0.0 |  3.8016 | 0.801029 | 0.4752 | 0.020965 | 0.821995 | 0.649085 | 0.649085 | 0.180302 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  80.830351 |  18.99966 | 0.237835  0.235737  0.231211  0.223728  0.213008 | 0.180302 |
    | 02/01 19:00 |  7.5 | 16.5 | 15.5 |  0.291604 |  17.1 | 17.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    8.91 |  0.338261 | 0.099914 | 0.0 |    8.91 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    8.91 |    8.91 | 0.0 | 0.0 | 0.0 | 0.099914 | 0.0 | 0.0 | 0.0 | 0.0 |    7.92 | 0.843744 |   0.99 | 0.021649 | 0.865393 | 0.781966 | 0.781966 | 0.217213 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  87.817795 |  19.95691 | 0.250521  0.248976  0.246105  0.242133  0.237211 | 0.217213 |
    | 02/01 20:00 | 18.5 | 16.3 | 15.3 |  0.092622 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  21.978 |  0.107442 |  0.00859 | 0.0 |  21.978 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  21.978 |  21.978 | 0.0 | 0.0 | 0.0 |  0.00859 | 0.0 | 0.0 | 0.0 | 0.0 |  19.536 | 0.971157 |  2.442 | 0.023518 | 0.994675 | 0.849546 | 0.849546 | 0.235985 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 106.375002 | 22.374437 | 0.287177  0.282863  0.275419   0.26654  0.258075 | 0.235985 |
    | 02/01 21:00 | 15.4 | 16.2 | 15.2 |  0.092451 |  16.8 | 16.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 | 18.2952 |  0.107243 | 0.012392 | 0.0 | 18.2952 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 18.2952 | 18.2952 | 0.0 | 0.0 | 0.0 | 0.012392 | 0.0 | 0.0 | 0.0 | 0.0 | 16.2624 | 1.139491 | 2.0328 | 0.025976 | 1.165466 | 0.947267 | 0.947267 |  0.26313 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 121.486896 | 24.379885 | 0.336273  0.330419  0.320075  0.307234  0.294272 |  0.26313 |
    | 02/01 22:00 |  6.3 | 15.5 | 14.5 |  0.091248 |  16.1 | 16.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  7.4844 |  0.105848 | 0.036055 | 0.0 |  7.4844 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  7.4844 |  7.4844 | 0.0 | 0.0 | 0.0 | 0.036055 | 0.0 | 0.0 | 0.0 | 0.0 |  6.6528 | 1.241968 | 0.8316 | 0.027536 | 1.269504 | 1.088621 | 1.088621 | 0.302395 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 126.865679 | 25.179943 | 0.367082  0.363302  0.356267  0.346731  0.335775 | 0.302395 |
    | 02/01 23:00 |  1.9 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  2.2572 |  0.104032 | 0.059768 | 0.0 |  2.2572 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.2572 |  2.2572 | 0.0 | 0.0 | 0.0 | 0.059768 | 0.0 | 0.0 | 0.0 | 0.0 |  2.0064 | 1.272333 | 0.2508 | 0.028101 | 1.300435 | 1.211749 | 1.211749 | 0.336597 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 127.546618 | 25.396001 | 0.376788  0.375532   0.37297  0.368991  0.363562 | 0.336597 |
    | 03/01 00:00 |  4.9 | 14.7 | 13.7 |  0.089858 |  15.3 | 15.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  5.8212 |  0.104235 | 0.041931 | 0.0 |  5.8212 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.8212 |  5.8212 | 0.0 | 0.0 | 0.0 | 0.041931 | 0.0 | 0.0 | 0.0 | 0.0 |  5.1744 | 1.294896 | 0.6468 | 0.028561 | 1.323457 | 1.278453 | 1.278453 | 0.355126 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  131.38885 |  26.00958 | 0.383566  0.382736  0.381182  0.379019  0.376345 | 0.355126 |
    | 03/01 01:00 |  2.7 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  3.2076 |  0.104032 |  0.05435 | 0.0 |  3.2076 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.2076 |  3.2076 | 0.0 | 0.0 | 0.0 |  0.05435 | 0.0 | 0.0 | 0.0 | 0.0 |  2.8512 | 1.321549 | 0.3564 | 0.029081 | 1.350631 | 1.311819 | 1.311819 | 0.364394 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  132.87019 |  26.33086 | 0.391409   0.39047  0.388789  0.386645  0.384346 | 0.364394 |
    | 03/01 02:00 |  0.5 | 14.1 | 13.1 |  0.088805 |  14.7 | 14.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |   0.594 |  0.103014 | 0.069893 | 0.0 |   0.594 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   0.594 |   0.594 | 0.0 | 0.0 | 0.0 | 0.069893 | 0.0 | 0.0 | 0.0 | 0.0 |   0.528 | 1.324712 |  0.066 | 0.029277 | 1.353989 | 1.336935 | 1.336935 | 0.371371 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 132.011351 | 26.359817 | 0.392615  0.392443  0.392045  0.391339  0.390272 | 0.371371 |
    | 03/01 03:00 |  0.2 | 14.3 | 13.3 |  0.089157 |  14.9 | 14.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.2376 |  0.103422 | 0.072716 | 0.0 |  0.2376 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.2376 |  0.2376 | 0.0 | 0.0 | 0.0 | 0.072716 | 0.0 | 0.0 | 0.0 | 0.0 |  0.2112 | 1.314587 | 0.0264 | 0.029287 | 1.343874 | 1.349322 | 1.349322 | 0.374812 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 130.843328 | 26.348851 | 0.389821   0.39014  0.390654  0.391173  0.391477 | 0.374812 |
    | 03/01 04:00 |  0.5 | 14.9 | 13.9 |  0.090207 |  15.5 | 15.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |   0.594 |   0.10464 | 0.070996 | 0.0 |   0.594 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   0.594 |   0.594 | 0.0 | 0.0 | 0.0 | 0.070996 | 0.0 | 0.0 | 0.0 | 0.0 |   0.528 | 1.304544 |  0.066 | 0.029297 | 1.333841 | 1.346922 | 1.346922 | 0.374145 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 130.003676 | 26.377665 | 0.386915  0.387264  0.387886  0.388666  0.389453 | 0.374145 |
    | 03/01 05:00 |  2.4 | 15.7 | 14.7 |  0.091593 |  16.3 | 16.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  2.8512 |  0.106248 | 0.057521 | 0.0 |  2.8512 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.8512 |  2.8512 | 0.0 | 0.0 | 0.0 | 0.057521 | 0.0 | 0.0 | 0.0 | 0.0 |  2.5344 | 1.306188 | 0.3168 | 0.029468 | 1.335656 | 1.339044 | 1.339044 | 0.371957 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 131.180758 | 26.658606 | 0.387326   0.38729   0.38727  0.387341  0.387569 | 0.371957 |
    | 03/01 06:00 |  0.4 | 16.0 | 15.0 |  0.154861 |  16.6 | 16.6 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.4752 |  0.179639 | 0.123337 | 0.0 |  0.4752 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.4752 |  0.4752 | 0.0 | 0.0 | 0.0 | 0.123337 | 0.0 | 0.0 | 0.0 | 0.0 |  0.4224 | 1.307375 | 0.0528 | 0.029634 | 1.337009 | 1.335901 | 1.335901 | 0.371084 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  130.18615 | 26.668068 | 0.387719  0.387672   0.38759  0.387497  0.387426 | 0.371084 |
    | 03/01 07:00 |  0.2 | 16.7 | 15.7 |  0.470369 |  17.3 | 17.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.2376 |  0.545628 | 0.383628 | 0.0 |  0.2376 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.2376 |  0.2376 | 0.0 | 0.0 | 0.0 | 0.383628 | 0.0 | 0.0 | 0.0 | 0.0 |  0.2112 | 1.296426 | 0.0264 | 0.029629 | 1.326056 | 1.335604 | 1.335604 | 0.371001 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 128.759921 | 26.622214 | 0.384663  0.385016  0.385607  0.386263  0.386807 | 0.371001 |
    | 03/01 08:00 |  0.0 | 17.1 | 16.1 |  1.173726 |  17.7 | 17.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  1.361522 | 0.980296 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.980296 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 1.281183 |    0.0 | 0.029564 | 1.310746 |  1.32953 |  1.32953 | 0.369314 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 126.607364 | 26.483728 |  0.38027  0.380792  0.381712  0.382841  0.383957 | 0.369314 |
    | 03/01 09:00 |  0.0 | 16.2 | 15.2 |  4.202296 |  16.8 | 16.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  4.874663 | 3.509758 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 3.509758 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 1.259764 |    0.0 |  0.02941 | 1.289174 | 1.316651 | 1.316651 | 0.365736 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 122.227815 | 26.064345 | 0.374077  0.374814  0.376116   0.37773  0.379359 | 0.365736 |
    | 03/01 10:00 |  0.3 | 15.9 | 14.9 |  4.359715 |  16.5 | 16.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.3564 |  5.057269 | 3.513746 | 0.0 |  0.3564 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.3564 |  0.3564 | 0.0 | 0.0 | 0.0 | 3.513746 | 0.0 | 0.0 | 0.0 | 0.0 |  0.3168 | 1.217766 | 0.0396 | 0.028966 | 1.246732 | 1.296749 | 1.296749 | 0.360208 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  118.20352 | 25.684562 | 0.361974  0.363406  0.365902  0.368926  0.371871 | 0.360208 |
    | 03/01 11:00 |  2.6 | 16.3 | 15.3 |  5.305753 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  3.0888 |  6.154673 | 3.253813 | 0.0 |  3.0888 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.0888 |  3.0888 | 0.0 | 0.0 | 0.0 | 3.253813 | 0.0 | 0.0 | 0.0 | 0.0 |  2.7456 | 1.189827 | 0.3432 | 0.028713 |  1.21854 | 1.264826 | 1.264826 | 0.351341 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 116.867014 | 25.637514 | 0.353666  0.354681  0.356552  0.359047  0.361847 | 0.351341 |
    | 03/01 12:00 |  0.7 | 16.3 | 15.3 |  5.376027 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.8316 |  6.236191 | 4.131769 | 0.0 |  0.8316 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.8316 |  0.8316 | 0.0 | 0.0 | 0.0 | 4.131769 | 0.0 | 0.0 | 0.0 | 0.0 |  0.7392 |  1.16653 | 0.0924 | 0.028522 | 1.195052 | 1.232666 | 1.232666 | 0.342407 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 112.767001 | 25.242307 | 0.346804  0.347636  0.349156  0.351164  0.353417 | 0.342407 |
    | 03/01 13:00 |  0.3 | 16.4 | 15.4 |  4.658915 |  17.0 | 17.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.3564 |  5.404341 | 3.754888 | 0.0 |  0.3564 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.3564 |  0.3564 | 0.0 | 0.0 | 0.0 | 3.754888 | 0.0 | 0.0 | 0.0 | 0.0 |  0.3168 | 1.123629 | 0.0396 | 0.028053 | 1.151683 | 1.204774 | 1.204774 | 0.334659 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 108.622493 | 24.836644 |  0.33442  0.335888  0.338456  0.341599  0.344724 | 0.334659 |
    | 03/01 14:00 |  0.3 | 16.5 | 15.5 |  7.789594 |  17.1 | 17.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.3564 |  9.035929 | 6.278083 | 0.0 |  0.3564 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.3564 |  0.3564 | 0.0 | 0.0 | 0.0 | 6.278083 | 0.0 | 0.0 | 0.0 | 0.0 |  0.3168 | 1.082391 | 0.0396 | 0.027603 | 1.109994 | 1.169522 | 1.169522 | 0.324867 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 102.276385 | 24.151077 |  0.32232  0.323776  0.326396  0.329752  0.333314 | 0.324867 |
    | 03/01 15:00 |  0.0 | 18.4 | 17.4 |  4.851567 |  19.0 | 19.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  5.627818 | 4.052029 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 4.052029 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 1.017667 |    0.0 |  0.02682 | 1.044487 | 1.127004 | 1.127004 | 0.313057 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  97.656914 | 23.674031 | 0.303556  0.305786  0.309711  0.314555  0.319432 | 0.313057 |
    | 03/01 16:00 |  0.0 | 18.3 | 17.3 |   5.30692 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  6.156027 |  4.43234 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  4.43234 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.971703 |    0.0 |  0.02629 | 0.997992 | 1.073096 | 1.073096 | 0.298082 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  92.745354 | 23.155259 | 0.289894  0.291558  0.294614   0.29867    0.3032 | 0.298082 |
    | 03/01 17:00 |  0.0 | 18.1 | 17.1 |  3.286036 |  18.7 | 18.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  3.811802 | 2.744497 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 2.744497 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.922832 |    0.0 | 0.025714 | 0.948545 | 1.020228 | 1.020228 | 0.283397 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  89.382969 | 22.824602 | 0.275578  0.277298   0.28039   0.28436  0.288629 | 0.283397 |
    | 03/01 18:00 |  0.0 | 16.7 | 15.7 |  1.506216 |  17.3 | 17.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  1.747211 | 1.257992 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 1.257992 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.889375 |    0.0 | 0.025347 | 0.914722 | 0.971816 | 0.971816 | 0.269949 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  87.375379 | 22.659478 | 0.265617  0.266833   0.26908  0.272096  0.275535 | 0.269949 |
    | 03/01 19:00 |  0.0 | 15.2 | 14.2 |  0.274762 |  15.8 | 15.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  0.318724 | 0.229481 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.229481 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 |   0.8694 |    0.0 | 0.025163 | 0.894563 | 0.932363 | 0.932363 |  0.25899 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  86.301996 | 22.608817 | 0.259632  0.260369  0.261752  0.263664  0.265944 |  0.25899 |
    | 03/01 20:00 |  0.0 | 13.4 | 12.4 |  0.087565 |  14.0 | 14.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  0.101575 | 0.073134 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.073134 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.858719 |    0.0 | 0.025107 | 0.883826 | 0.905749 | 0.905749 | 0.251597 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  85.378269 | 22.575584 | 0.256422   0.25682  0.257577  0.258649  0.259971 | 0.251597 |
    | 03/01 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  0.099494 | 0.071636 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.849528 |    0.0 |  0.02507 | 0.874598 | 0.889805 | 0.889805 | 0.247168 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  84.528741 | 22.550514 | 0.253727  0.254054  0.254652  0.255446  0.256352 | 0.247168 |
    | 03/01 22:00 |  0.0 | 11.6 | 10.6 |  0.084317 |  12.2 | 12.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  0.097808 | 0.070422 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.841075 |    0.0 | 0.025042 | 0.866117 | 0.879059 | 0.879059 | 0.244183 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  83.687666 | 22.525472 |  0.25126  0.251558  0.252098  0.252801  0.253573 | 0.244183 |
    | 03/01 23:00 |  0.0 | 11.0 | 10.0 |  0.083215 |  11.6 | 11.6 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  0.096529 | 0.069501 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.832706 |    0.0 | 0.025014 | 0.857721 | 0.870035 | 0.870035 | 0.241676 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   82.85496 | 22.500457 | 0.248824  0.249117  0.249646  0.250327  0.251061 | 0.241676 |
    | 04/01 00:00 |  0.0 | 10.5 |  9.5 |  0.082289 |  11.1 | 11.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  0.095455 | 0.068728 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.824421 |    0.0 | 0.024987 | 0.849407 | 0.861523 | 0.861523 | 0.239312 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  82.030539 | 22.475471 | 0.246412  0.246702  0.247225  0.247898  0.248621 | 0.239312 |
    | 04/01 01:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |   0.09802 | 0.070574 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.816218 |    0.0 | 0.024959 | 0.841176 | 0.853164 | 0.853164 |  0.23699 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  81.214322 | 22.450512 | 0.244024  0.244312  0.244829  0.245495  0.246211 |  0.23699 |
    | 04/01 02:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  0.098442 | 0.070878 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.808096 |    0.0 | 0.024931 | 0.833027 | 0.844895 | 0.844895 | 0.234693 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  80.406226 | 22.425581 |  0.24166  0.241945  0.242457  0.243117  0.243825 | 0.234693 |
    | 04/01 03:00 |  1.3 | 11.2 | 10.2 |  0.083584 |  11.8 | 11.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  1.5444 |  0.096957 | 0.059819 | 0.0 |  1.5444 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  1.5444 |  1.5444 | 0.0 | 0.0 | 0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  1.3728 | 0.806896 | 0.1716 | 0.024999 | 0.831895 | 0.837204 | 0.837204 | 0.232557 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  80.972129 | 22.572182 | 0.241263  0.241319  0.241446  0.241668  0.241999 | 0.232557 |
    | 04/01 04:00 |  0.0 | 11.1 | 10.1 |    0.0834 |  11.7 | 11.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  0.096744 | 0.069656 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.805686 |    0.0 | 0.025066 | 0.830752 | 0.833061 | 0.833061 | 0.231406 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  80.166443 | 22.547116 |  0.24093  0.240971  0.241046  0.241153  0.241288 | 0.231406 |
    | 04/01 05:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  0.098442 | 0.070878 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.797669 |    0.0 | 0.025038 | 0.822708 | 0.830848 | 0.830848 | 0.230791 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  79.368774 | 22.522077 | 0.238664  0.238929   0.23938  0.239902  0.240373 | 0.230791 |
    | 04/01 06:00 |  0.0 | 12.2 | 11.2 |  0.310229 |  12.8 | 12.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  0.359866 | 0.259103 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.789732 |    0.0 | 0.025011 | 0.814743 | 0.825695 | 0.825695 |  0.22936 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  78.579041 | 22.497067 | 0.236356  0.236633   0.23713   0.23776  0.238416 |  0.22936 |
    | 04/01 07:00 |  0.7 | 11.8 | 10.8 |  1.391958 |  12.4 | 12.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.8316 |  1.614671 | 1.069795 | 0.0 |  0.8316 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.8316 |  0.8316 | 0.0 | 0.0 | 0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.7392 | 0.785558 | 0.0924 | 0.025034 | 0.810592 | 0.818534 | 0.818534 | 0.227371 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  78.532683 | 22.564432 | 0.235115  0.235269  0.235559  0.235964  0.236447 | 0.227371 |
    | 04/01 08:00 |  0.4 | 11.4 | 10.4 |  3.195876 |  12.0 | 12.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.4752 |  3.707216 | 2.545322 | 0.0 |  0.4752 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.4752 |  0.4752 | 0.0 | 0.0 | 0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.4224 | 0.783518 | 0.0528 | 0.025087 | 0.808605 | 0.812966 | 0.812966 | 0.225824 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  78.171565 | 22.592145 | 0.234516  0.234591  0.234735  0.234944  0.235207 | 0.225824 |
    | 04/01 09:00 |  0.1 | 11.6 | 10.6 |  5.191651 |  12.2 | 12.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.1188 |  6.022315 | 4.284859 | 0.0 |  0.1188 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.1188 |  0.1188 | 0.0 | 0.0 | 0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.1056 | 0.778346 | 0.0132 | 0.025096 | 0.803442 | 0.809515 | 0.809515 | 0.224865 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  77.498818 |  22.58025 | 0.233049  0.233222  0.233523  0.233886   0.23424 | 0.224865 |
    | 04/01 10:00 |  0.4 | 13.0 | 12.0 |  7.155036 |  13.6 | 13.6 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.4752 |  8.299842 | 5.698554 | 0.0 |  0.4752 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.4752 |  0.4752 | 0.0 | 0.0 | 0.0 | 5.698554 | 0.0 | 0.0 | 0.0 | 0.0 |  0.4224 | 0.773231 | 0.0528 | 0.025105 | 0.798336 | 0.805499 | 0.805499 |  0.22375 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  72.082606 | 21.974772 | 0.231569  0.231747  0.232066  0.232473  0.232902 |  0.22375 |
    | 04/01 11:00 |  0.0 | 17.1 | 16.1 |  8.391432 |  17.7 | 17.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  9.734061 | 7.008524 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 7.008524 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.717234 |    0.0 | 0.024403 | 0.741637 | 0.796925 | 0.796925 | 0.221368 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  65.135573 | 21.171645 | 0.215631   0.21749   0.22064  0.224258  0.227451 | 0.221368 |
    | 04/01 12:00 |  0.0 | 18.2 | 17.2 |  8.391286 |  18.8 | 18.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  9.733892 | 7.008402 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 7.008402 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 |  0.64811 |    0.0 | 0.023511 | 0.671621 | 0.761377 | 0.761377 | 0.211494 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  58.257773 | 20.369422 | 0.195473  0.197879  0.202139  0.207438  0.212784 | 0.211494 |
    | 04/01 13:00 |  0.0 | 22.4 | 21.4 | 10.715238 |  23.0 | 23.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 | 12.429676 | 8.949367 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 8.949367 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.579675 |    0.0 |  0.02262 | 0.602295 | 0.701384 | 0.701384 | 0.194829 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  49.723105 | 19.352428 | 0.175367  0.177785  0.182134  0.187706  0.193631 | 0.194829 |
    | 04/01 14:00 |  0.0 | 21.4 | 20.4 |  9.383394 |  22.0 | 22.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 | 10.884737 | 7.837011 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 7.837011 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.494753 |    0.0 | 0.021491 | 0.516244 | 0.632173 | 0.632173 | 0.175604 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  42.262121 | 18.460158 | 0.150575  0.153539  0.158809  0.165442  0.172328 | 0.175604 |
    | 04/01 15:00 |  0.0 | 21.8 | 20.8 |  7.861915 |  22.4 | 22.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  9.119821 | 6.566271 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 6.566271 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.420515 |    0.0 |   0.0205 | 0.441015 | 0.554168 | 0.554168 | 0.153935 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   36.00492 | 17.710073 | 0.128659  0.131307  0.136107  0.142344  0.149126 | 0.153935 |
    | 04/01 16:00 |  0.0 | 22.2 | 21.2 |  6.298329 |  22.8 | 22.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  7.306062 | 5.260364 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 5.260364 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.358255 |    0.0 | 0.019667 | 0.377922 | 0.476567 | 0.476567 |  0.13238 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  30.970785 | 17.105921 |  0.11024   0.11247  0.116535   0.12187  0.127781 |  0.13238 |
    | 04/01 17:00 |  0.0 | 20.1 | 19.1 |  2.948416 |  20.7 | 20.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  3.420163 | 2.462517 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 2.462517 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.308164 |    0.0 | 0.018996 |  0.32716 | 0.408428 | 0.408428 | 0.113452 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  28.473717 | 16.813312 | 0.095394  0.097195  0.100488   0.10484  0.109711 | 0.113452 |
    | 04/01 18:00 |  0.0 | 17.8 | 16.8 |  1.309232 |  18.4 | 18.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  1.518709 | 1.093471 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 1.093471 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.283318 |    0.0 | 0.018671 | 0.301989 |  0.35312 |  0.35312 | 0.098089 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  27.218425 | 16.673144 |  0.08784  0.088779  0.090571  0.093104  0.096204 | 0.098089 |
    | 04/01 19:00 |  0.0 | 15.2 | 14.2 |   0.32955 |  15.8 | 15.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  0.382278 |  0.27524 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.27524 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.270828 |    0.0 | 0.018515 | 0.289343 | 0.316745 | 0.316745 | 0.087985 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  26.702939 | 16.624046 | 0.084042  0.084515  0.085425  0.086731  0.088383 | 0.087985 |
    | 04/01 20:00 |  0.0 | 14.5 | 13.5 |  0.089508 |  15.1 | 15.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  0.103829 | 0.074757 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.074757 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.265699 |    0.0 | 0.018461 |  0.28416 | 0.297021 | 0.297021 | 0.082506 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   26.37079 | 16.597279 | 0.082461  0.082661  0.083054  0.083641  0.084417 | 0.082506 |
    | 04/01 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  0.099494 | 0.071636 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.262394 |    0.0 | 0.018431 | 0.280825 | 0.287479 | 0.287479 | 0.079855 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  26.108396 | 16.578848 | 0.081474  0.081595  0.081824  0.082145  0.082542 | 0.079855 |
    | 04/01 22:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |   0.09802 | 0.070574 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.259783 |    0.0 | 0.018411 | 0.278194 | 0.282641 | 0.282641 | 0.078511 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  25.848613 | 16.560437 | 0.080703  0.080797  0.080969  0.081199  0.081465 | 0.078511 |
    | 04/01 23:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |     0.0 |  0.098442 | 0.070878 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 |     0.0 | 0.257198 |    0.0 |  0.01839 | 0.275588 | 0.279473 | 0.279473 | 0.077631 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  25.591415 | 16.542047 | 0.079947  0.080038  0.080202  0.080415  0.080646 | 0.077631 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_v3_snow_classes:

snow classes
____________

Regarding snow simulation, all functionalities of |hland_v3| are identical with the
ones of |hland_v1|.  Hence, all snow data in the following table agrees with the
:ref:`hland_v1_snow_classes` example of |hland_v1|:

.. integration-test::

    >>> sclass(2)
    >>> sfdist(linear=0.2)
    >>> zonetype(FIELD)
    >>> t_series = inputs.t.series.copy()
    >>> tn_series = inputs.tn.series.copy()
    >>> inputs.t.series[:48] = -20.0
    >>> inputs.t.series[48:] = 20.0
    >>> inputs.tn.series = inputs.t.series
    >>> test("hland_v3_snow_classes")
    |        date |    p |     t |    tn |       epn | tmean |    tc | fracrain | rfc | sfc |    cfact |                    swe | gact |      pc |        ep |      epc |       ei |        tf | spl | wcl | spg | wcg | glmelt |               melt |      refr |       in_ |        r |  sr |       ea |       dp |  el |       rs |       ri |      gr1 |      rg1 |      gr2 |      rg2 |      gr3 |      rg3 |     inuh |    outuh |       rt |       qt |       ic |                     sp |                   wc |         sm |       suz |       sg1 |       sg2 |       sg3 |                                               sc |   outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | -20.0 | -20.0 |  0.100707 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |     0.0 |  0.106749 |  0.07686 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.048037 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.487706 |      0.0 | 0.099502 |      0.0 | 0.011105 | 0.598312 | 0.202814 | 0.202814 | 0.056337 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.951963 |       0.0 |  9.512294 |  9.900498 |  9.988895 | 0.169348  0.155472  0.132115  0.105643  0.082921 | 0.056337 |
    | 01/01 01:00 |  0.0 | -20.0 | -20.0 |  0.097801 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |     0.0 |  0.103669 | 0.074642 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.046629 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |  0.46392 |      0.0 | 0.098512 |      0.0 | 0.011093 | 0.573524 | 0.414293 | 0.414293 | 0.115081 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.905334 |       0.0 |  9.048374 |  9.801987 |  9.977802 | 0.166424  0.166296  0.164522  0.159031  0.148456 | 0.115081 |
    | 01/01 02:00 |  0.0 | -20.0 | -20.0 |  0.096981 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |     0.0 |    0.1028 | 0.074016 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.046216 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.441294 |      0.0 | 0.097531 |      0.0 |  0.01108 | 0.549906 | 0.547167 | 0.547167 | 0.151991 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.859118 |       0.0 |   8.60708 |  9.704455 |  9.966722 | 0.159707  0.160484  0.161724  0.162805  0.162748 | 0.151991 |
    | 01/01 03:00 |  0.0 | -20.0 | -20.0 |   0.09599 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |     0.0 |  0.101749 |  0.07326 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.045723 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.419772 |      0.0 | 0.096561 |      0.0 | 0.011068 | 0.527401 | 0.556325 | 0.556325 | 0.154535 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.813395 |       0.0 |  8.187308 |  9.607894 |  9.955654 | 0.153174   0.15396  0.155368  0.157139  0.158904 | 0.154535 |
    | 01/01 04:00 |  0.0 | -20.0 | -20.0 |  0.096981 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |     0.0 |    0.1028 | 0.074016 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.046174 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.3993 |      0.0 |   0.0956 |      0.0 | 0.011056 | 0.505956 | 0.537334 | 0.537334 |  0.14926 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.767222 |       0.0 |  7.788008 |  9.512294 |  9.944598 | 0.146944  0.147695   0.14905  0.150798  0.152677 |  0.14926 |
    | 01/01 05:00 |  0.0 | -20.0 | -20.0 |  0.102761 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |     0.0 |  0.108927 | 0.078427 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.048903 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.379826 |      0.0 | 0.094649 |      0.0 | 0.011043 | 0.485518 | 0.515762 | 0.515762 | 0.143267 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.718319 |       0.0 |  7.408182 |  9.417645 |  9.933555 | 0.141007  0.141723  0.143015  0.144685  0.146492 | 0.143267 |
    | 01/01 06:00 |  0.0 | -20.0 | -20.0 |  0.291908 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |     0.0 |  0.309422 | 0.222784 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.138848 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.361301 |      0.0 | 0.093707 |      0.0 | 0.011031 |  0.46604 | 0.494888 | 0.494888 | 0.137469 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.579471 |       0.0 |  7.046881 |  9.323938 |  9.922524 | 0.135349  0.136031  0.137262  0.138854  0.140577 | 0.137469 |
    | 01/01 07:00 |  0.0 | -20.0 | -20.0 |  1.932875 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |     0.0 |  2.048847 |  1.47517 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.918104 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |  0.34368 |      0.0 | 0.092775 |      0.0 | 0.011019 | 0.447474 | 0.474971 | 0.474971 | 0.131936 |      0.0 |        0.0         0.0 |       0.0        0.0 |  98.661367 |       0.0 |    6.7032 |  9.231163 |  9.911505 | 0.129956  0.130606  0.131779  0.133296  0.134939 | 0.131936 |
    | 01/01 08:00 |  0.0 | -20.0 | -20.0 |  4.369536 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |     0.0 |  4.631708 |  3.33483 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 2.056368 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.326919 |      0.0 | 0.091852 |      0.0 | 0.011007 | 0.429777 | 0.455987 | 0.455987 | 0.126663 |      0.0 |        0.0         0.0 |       0.0        0.0 |  96.604999 |       0.0 |  6.376282 |  9.139312 |  9.900498 | 0.124815  0.125434  0.126553  0.127999  0.129565 | 0.126663 |
    | 01/01 09:00 |  0.0 | -20.0 | -20.0 |  7.317556 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |     0.0 |  7.756609 | 5.584759 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 3.371973 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.310975 |      0.0 | 0.090938 |      0.0 | 0.010994 | 0.412907 | 0.437892 | 0.437892 | 0.121637 |      0.0 |        0.0         0.0 |       0.0        0.0 |  93.233026 |       0.0 |  6.065307 |  9.048374 |  9.889504 | 0.119914  0.120505  0.121571   0.12295  0.124442 | 0.121637 |
    | 01/01 10:00 |  0.0 | -20.0 | -20.0 |  8.264362 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |     0.0 |  8.760224 | 6.307361 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |  3.67534 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.295808 |      0.0 | 0.090033 |      0.0 | 0.010982 | 0.396824 | 0.420642 | 0.420642 | 0.116845 |      0.0 |        0.0         0.0 |       0.0        0.0 |  89.557686 |       0.0 |  5.769498 |  8.958341 |  9.878522 | 0.115242  0.115805  0.116822  0.118136  0.119559 | 0.116845 |
    | 01/01 11:00 |  0.0 | -20.0 | -20.0 |  9.369867 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |     0.0 |  9.932059 | 7.151082 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 4.002715 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.281382 |      0.0 | 0.089137 |      0.0 |  0.01097 | 0.381489 | 0.404198 | 0.404198 | 0.112277 |      0.0 |        0.0         0.0 |       0.0        0.0 |  85.554971 |       0.0 |  5.488116 |  8.869204 |  9.867552 | 0.110787  0.111324  0.112293  0.113546  0.114903 | 0.112277 |
    | 01/01 12:00 |  0.0 | -20.0 | -20.0 |  5.126178 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |     0.0 |  5.433749 | 3.912299 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 2.091979 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.267659 |      0.0 |  0.08825 |      0.0 | 0.010958 | 0.366867 | 0.388519 | 0.388519 | 0.107922 |      0.0 |        0.0         0.0 |       0.0        0.0 |  83.462992 |       0.0 |  5.220458 |  8.780954 |  9.856594 | 0.106539  0.107051  0.107976  0.109171  0.110464 | 0.107922 |
    | 01/01 13:00 |  0.0 | -20.0 | -20.0 |   6.62503 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |     0.0 |  7.022532 | 5.056223 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 2.637547 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.254605 |      0.0 | 0.087372 |      0.0 | 0.010946 | 0.352922 |  0.37357 |  0.37357 |  0.10377 |      0.0 |        0.0         0.0 |       0.0        0.0 |  80.825446 |       0.0 |  4.965853 |  8.693582 |  9.845648 | 0.102489  0.102977  0.103858  0.104998  0.106231 |  0.10377 |
    | 01/01 14:00 |  0.0 | -20.0 | -20.0 |  7.397619 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |     0.0 |  7.841476 | 5.645863 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 2.852059 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.242188 |      0.0 | 0.086503 |      0.0 | 0.010934 | 0.339624 | 0.359315 | 0.359315 |  0.09981 |      0.0 |        0.0         0.0 |       0.0        0.0 |  77.973387 |       0.0 |  4.723666 |   8.60708 |  9.834715 | 0.098626  0.099091  0.099932  0.101018  0.102195 |  0.09981 |
    | 01/01 15:00 |  0.2 | -20.0 | -20.0 |   2.39151 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |  0.2808 |  2.535001 | 1.774662 |   0.2808 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.864852 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.230376 |      0.0 | 0.085642 |      0.0 | 0.010921 | 0.326939 |  0.34572 |  0.34572 | 0.096033 |      0.0 |        0.0         0.0 |       0.0        0.0 |  77.108535 |       0.0 |   4.49329 |  8.521438 |  9.823793 | 0.094941  0.095385  0.096187  0.097223  0.098345 | 0.096033 |
    | 01/01 16:00 |  0.0 | -20.0 | -20.0 |  1.829834 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |     0.0 |  1.939624 | 1.396529 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.673027 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |  0.21914 |      0.0 |  0.08479 |      0.0 | 0.010909 | 0.314839 | 0.332754 | 0.332754 | 0.092432 |      0.0 |        0.0         0.0 |       0.0        0.0 |  76.435507 |       0.0 |  4.274149 |  8.436648 |  9.812884 | 0.091426   0.09185  0.092614  0.093603  0.094673 | 0.092432 |
    | 01/01 17:00 |  0.0 | -20.0 | -20.0 |  1.136569 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |     0.0 |  1.204763 | 0.867429 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |  0.41439 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.208453 |      0.0 | 0.083946 |      0.0 | 0.010897 | 0.303296 | 0.320386 | 0.320386 | 0.088996 |      0.0 |        0.0         0.0 |       0.0        0.0 |  76.021117 |       0.0 |  4.065697 |  8.352702 |  9.801987 | 0.088073  0.088477  0.089207   0.09015  0.091171 | 0.088996 |
    | 01/01 18:00 |  1.3 | -20.0 | -20.0 |  0.750986 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |  1.8252 |  0.796045 | 0.477532 | 0.477532 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.113445 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.198286 |      0.0 | 0.083111 |      0.0 | 0.010885 | 0.292282 | 0.308587 | 0.308587 | 0.085719 | 1.347668 |        0.0         0.0 |       0.0        0.0 |  75.907672 |       0.0 |   3.86741 |  8.269591 |  9.791102 | 0.084873  0.085259  0.085955  0.086855  0.087829 | 0.085719 |
    | 01/01 19:00 |  5.6 | -20.0 | -20.0 |  0.223895 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |   5.768054    8.652081 |  0.0 |  7.8624 |  0.237329 | 0.077844 | 0.077844 |  7.210068 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.188616 |      0.0 | 0.082284 |      0.0 | 0.010873 | 0.281773 |  0.29733 |  0.29733 | 0.082592 | 1.922156 |   5.768054    8.652081 |       0.0        0.0 |  75.907672 |       0.0 |  3.678794 |  8.187308 |  9.780229 | 0.081821  0.082188  0.082853  0.083711  0.084641 | 0.082592 |
    | 01/01 20:00 |  2.9 | -20.0 | -20.0 |  0.099425 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |   8.963059   13.444589 |  0.0 |  4.0716 |   0.10539 | 0.050502 | 0.050502 |  3.993756 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.179417 |      0.0 | 0.081465 |      0.0 | 0.010861 | 0.271743 | 0.286589 | 0.286589 | 0.079608 | 1.949498 |   8.963059   13.444589 |       0.0        0.0 |  75.907672 |       0.0 |  3.499377 |  8.105842 |  9.769368 | 0.078907  0.079258  0.079892  0.080711  0.081598 | 0.079608 |
    | 01/01 21:00 |  4.9 | -20.0 | -20.0 |  0.098454 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |  14.426338   21.639507 |  0.0 |  6.8796 |  0.104361 | 0.037765 | 0.037765 |  6.829098 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.170667 |      0.0 | 0.080654 |      0.0 | 0.010849 |  0.26217 |  0.27634 |  0.27634 | 0.076761 | 1.962235 |  14.426338   21.639507 |       0.0        0.0 |  75.907672 |       0.0 |  3.328711 |  8.025188 |  9.758519 | 0.076126  0.076461  0.077066  0.077848  0.078695 | 0.076761 |
    | 01/01 22:00 | 10.6 | -20.0 | -20.0 |  0.098128 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |  26.302045   39.453068 |  0.0 | 14.8824 |  0.104016 | 0.016908 | 0.016908 | 14.844635 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.162343 |      0.0 | 0.079852 |      0.0 | 0.010837 | 0.253032 | 0.266557 | 0.266557 | 0.074044 | 1.983092 |  26.302045   39.453068 |       0.0        0.0 |  75.907672 |       0.0 |  3.166368 |  7.945336 |  9.747682 | 0.073472  0.073792  0.074369  0.075116  0.075924 | 0.074044 |
    | 01/01 23:00 |  0.1 | -20.0 | -20.0 |  0.097474 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |  26.400839   39.601258 |  0.0 |  0.1404 |  0.103322 | 0.073355 | 0.073355 |  0.123492 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.154426 |      0.0 | 0.079057 |      0.0 | 0.010825 | 0.244308 |  0.25722 |  0.25722 |  0.07145 | 1.926645 |  26.400839   39.601258 |       0.0        0.0 |  75.907672 |       0.0 |  3.011942 |  7.866279 |  9.736857 | 0.070938  0.071243  0.071794  0.072507  0.073278 |  0.07145 |
    | 02/01 00:00 |  0.7 | -20.0 | -20.0 |  0.097474 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  27.128395   40.692592 |  0.0 |  0.9828 |  0.103322 | 0.067429 | 0.067429 |  0.909445 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.146894 |      0.0 | 0.078271 |      0.0 | 0.010813 | 0.235978 | 0.248305 | 0.248305 | 0.068974 | 1.932571 |  27.128395   40.692592 |       0.0        0.0 |  75.907672 |       0.0 |  2.865048 |  7.788008 |  9.726045 | 0.068518   0.06881  0.069336  0.070016  0.070753 | 0.068974 |
    | 02/01 01:00 |  3.0 | -20.0 | -20.0 |  0.097474 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  30.444052   45.666078 |  0.0 |   4.212 |  0.103322 | 0.048821 | 0.048821 |  4.144571 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |  0.13973 |      0.0 | 0.077492 |      0.0 | 0.010801 | 0.228023 | 0.239795 | 0.239795 |  0.06661 | 1.951179 |  30.444052   45.666078 |       0.0        0.0 |  75.907672 |       0.0 |  2.725318 |  7.710516 |  9.715244 | 0.066207  0.066486  0.066988  0.067638  0.068341 |  0.06661 |
    | 02/01 02:00 |  2.1 | -20.0 | -20.0 |  0.096981 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  32.763716   49.145573 |  0.0 |  2.9484 |    0.1028 | 0.055116 | 0.055116 |  2.899579 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.132915 |      0.0 | 0.076721 |      0.0 | 0.010789 | 0.220425 | 0.231668 | 0.231668 | 0.064352 | 1.944884 |  32.763716   49.145573 |       0.0        0.0 |  75.907672 |       0.0 |  2.592403 |  7.633795 |  9.704455 |    0.064  0.064266  0.064746  0.065367  0.066039 | 0.064352 |
    | 02/01 03:00 | 10.4 | -20.0 | -20.0 |  0.096652 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  44.400903   66.601354 |  0.0 | 14.6016 |  0.102451 | 0.017128 | 0.017128 | 14.546484 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.126433 |      0.0 | 0.075958 |      0.0 | 0.010777 | 0.213167 | 0.223906 | 0.223906 | 0.062196 | 1.982872 |  44.400903   66.601354 |       0.0        0.0 |  75.907672 |       0.0 |   2.46597 |  7.557837 |  9.693679 | 0.061892  0.062146  0.062605  0.063198  0.063839 | 0.062196 |
    | 02/01 04:00 |  3.5 | -20.0 | -20.0 |  0.096321 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    48.3184     72.4776 |  0.0 |   4.914 |    0.1021 | 0.044973 | 0.044973 |  4.896872 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.120267 |      0.0 | 0.075202 |      0.0 | 0.010765 | 0.206233 | 0.216492 | 0.216492 | 0.060137 | 1.955027 |    48.3184     72.4776 |       0.0        0.0 |  75.907672 |       0.0 |  2.345703 |  7.482636 |  9.682914 | 0.059878  0.060121  0.060559  0.061125  0.061738 | 0.060137 |
    | 02/01 05:00 |  3.4 | -20.0 | -20.0 |   0.09599 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  52.101302   78.151953 |  0.0 |  4.7736 |  0.101749 | 0.045452 | 0.045452 |  4.728627 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.114401 |      0.0 | 0.074453 |      0.0 | 0.010753 | 0.199608 |  0.20941 |  0.20941 | 0.058169 | 1.954548 |  52.101302   78.151953 |       0.0        0.0 |  75.907672 |       0.0 |  2.231302 |  7.408182 |  9.672161 | 0.057953  0.058185  0.058604  0.059145  0.059731 | 0.058169 |
    | 02/01 06:00 |  1.2 | -20.0 | -20.0 |  0.187298 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  53.412781   80.119171 |  0.0 |  1.6848 |  0.198536 | 0.120782 | 0.120782 |  1.639348 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.108822 |      0.0 | 0.073713 |      0.0 | 0.010741 | 0.193275 | 0.202643 | 0.202643 |  0.05629 | 1.879218 |  53.412781   80.119171 |       0.0        0.0 |  75.907672 |       0.0 |   2.12248 |   7.33447 |   9.66142 | 0.056114  0.056336  0.056736  0.057253  0.057813 |  0.05629 |
    | 02/01 07:00 |  0.1 | -20.0 | -20.0 |  1.264612 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  53.428476   80.142713 |  0.0 |  0.1404 |  1.340489 | 0.951696 | 0.951696 |  0.019618 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.103515 |      0.0 | 0.072979 |      0.0 | 0.010729 | 0.187223 | 0.196176 | 0.196176 | 0.054493 | 1.048304 |  53.428476   80.142713 |       0.0        0.0 |  75.907672 |       0.0 |  2.018965 |   7.26149 |  9.650691 | 0.054356  0.054568   0.05495  0.055445  0.055979 | 0.054493 |
    | 02/01 08:00 |  0.0 | -20.0 | -20.0 |  3.045538 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  53.428476   80.142713 |  0.0 |     0.0 |   3.22827 | 2.324355 | 1.048304 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.098466 |      0.0 | 0.072253 |      0.0 | 0.010717 | 0.181436 | 0.189995 | 0.189995 | 0.052776 |      0.0 |  53.428476   80.142713 |       0.0        0.0 |  75.907672 |       0.0 |  1.920499 |  7.189237 |  9.639974 | 0.052675  0.052878  0.053243  0.053716  0.054227 | 0.052776 |
    | 02/01 09:00 |  0.0 | -20.0 | -20.0 |  1.930758 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  53.428476   80.142713 |  0.0 |     0.0 |  2.046603 | 1.473555 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.093664 |      0.0 | 0.071534 |      0.0 | 0.010705 | 0.175903 | 0.184086 | 0.184086 | 0.051135 |      0.0 |  53.428476   80.142713 |       0.0        0.0 |  75.907672 |       0.0 |  1.826835 |  7.117703 |  9.629269 | 0.051068  0.051262  0.051611  0.052063  0.052552 | 0.051135 |
    | 02/01 10:00 |  0.4 | -20.0 | -20.0 |  2.461001 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  53.428476   80.142713 |  0.0 |  0.5616 |  2.608661 | 1.775661 |   0.5616 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.089096 |      0.0 | 0.070822 |      0.0 | 0.010693 | 0.170611 | 0.178437 | 0.178437 | 0.049566 |      0.0 |  53.428476   80.142713 |       0.0        0.0 |  75.907672 |       0.0 |  1.737739 |  7.046881 |  9.618576 | 0.049531  0.049716   0.05005  0.050483   0.05095 | 0.049566 |
    | 02/01 11:00 |  0.1 | -20.0 | -20.0 |  6.215945 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  53.428476   80.142713 |  0.0 |  0.1404 |  6.588902 | 4.677869 |   0.1404 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.084751 |      0.0 | 0.070118 |      0.0 | 0.010681 |  0.16555 | 0.173035 | 0.173035 | 0.048065 |      0.0 |  53.428476   80.142713 |       0.0        0.0 |  75.907672 |       0.0 |  1.652989 |  6.976763 |  9.607894 | 0.048061  0.048238  0.048558  0.048971  0.049418 | 0.048065 |
    | 02/01 12:00 |  3.6 | -20.0 | -20.0 |  3.374783 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  55.871996   83.807993 |  0.0 |  5.0544 |   3.57727 | 1.553726 | 1.553726 |    3.0544 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.080617 |      0.0 |  0.06942 |      0.0 |  0.01067 | 0.160707 | 0.167867 | 0.167867 |  0.04663 | 0.446274 |  55.871996   83.807993 |       0.0        0.0 |  75.907672 |       0.0 |  1.572372 |  6.907343 |  9.597225 | 0.046654  0.046824  0.047129  0.047525  0.047953 |  0.04663 |
    | 02/01 13:00 |  5.9 | -20.0 | -20.0 |  8.821555 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  61.255895   91.883842 |  0.0 |  8.2836 |  9.350848 | 2.940569 |      2.0 |  6.729874 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.076685 |      0.0 | 0.068729 |      0.0 | 0.010658 | 0.156072 | 0.162924 | 0.162924 | 0.045257 |      0.0 |  61.255895   91.883842 |       0.0        0.0 |  75.907672 |       0.0 |  1.495686 |  6.838614 |  9.586567 | 0.045308   0.04547  0.045763  0.046141  0.046551 | 0.045257 |
    | 02/01 14:00 |  1.1 | -20.0 | -20.0 |  4.046025 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  61.255895   91.883842 |  0.0 |  1.5444 |  4.288787 | 2.646028 |   1.5444 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.072945 |      0.0 | 0.068045 |      0.0 | 0.010646 | 0.151637 | 0.158194 | 0.158194 | 0.043943 |      0.0 |  61.255895   91.883842 |       0.0        0.0 |  75.907672 |       0.0 |  1.422741 |  6.770569 |  9.575921 |  0.04402  0.044175  0.044455  0.044817  0.045209 | 0.043943 |
    | 02/01 15:00 | 20.7 | -20.0 | -20.0 |  2.110757 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  82.906135  124.359202 |  0.0 | 29.0628 |  2.237402 | 0.088084 | 0.088084 |   27.0628 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.069388 |      0.0 | 0.067368 |      0.0 | 0.010634 |  0.14739 | 0.153667 | 0.153667 | 0.042685 | 1.911916 |  82.906135  124.359202 |       0.0        0.0 |  75.907672 |       0.0 |  1.353353 |    6.7032 |  9.565287 | 0.042786  0.042935  0.043203   0.04355  0.043925 | 0.042685 |
    | 02/01 16:00 | 37.9 | -20.0 | -20.0 |  2.239257 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 125.404948  188.107422 |  0.0 | 53.2116 |  2.373612 | 0.008352 | 0.008352 | 53.123516 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.066004 |      0.0 | 0.066698 |      0.0 | 0.010622 | 0.143324 | 0.149334 | 0.149334 | 0.041482 | 1.991648 | 125.404948  188.107422 |       0.0        0.0 |  75.907672 |       0.0 |  1.287349 |  6.636503 |  9.554665 | 0.041605  0.041747  0.042004  0.042336  0.042695 | 0.041482 |
    | 02/01 17:00 |  8.2 | -20.0 | -20.0 |  2.877848 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 134.608506  201.912759 |  0.0 | 11.5128 |  3.050519 | 0.694563 | 0.694563 | 11.504448 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.062785 |      0.0 | 0.066034 |      0.0 |  0.01061 | 0.139429 | 0.145185 | 0.145185 | 0.040329 | 1.305437 | 134.608506  201.912759 |       0.0        0.0 |  75.907672 |       0.0 |  1.224564 |  6.570468 |  9.544055 | 0.040474   0.04061  0.040856  0.041174  0.041518 | 0.040329 |
    | 02/01 18:00 |  3.6 | -20.0 | -20.0 |  1.591452 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 138.096376  207.144564 |  0.0 |  5.0544 |  1.686939 | 0.732693 | 0.732693 |  4.359837 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.059723 |      0.0 | 0.065377 |      0.0 | 0.010599 | 0.135699 | 0.141212 | 0.141212 | 0.039226 | 1.267307 | 138.096376  207.144564 |       0.0        0.0 |  75.907672 |       0.0 |  1.164842 |  6.505091 |  9.533456 |  0.03939  0.039521  0.039757  0.040061   0.04039 | 0.039226 |
    | 02/01 19:00 |  7.5 | -20.0 | -20.0 |  0.291604 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 145.934221  218.901332 |  0.0 |   10.53 |    0.3091 | 0.077646 | 0.077646 |  9.797307 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |  0.05681 |      0.0 | 0.064727 |      0.0 | 0.010587 | 0.132124 | 0.137406 | 0.137406 | 0.038168 | 1.922354 | 145.934221  218.901332 |       0.0        0.0 |  75.907672 |       0.0 |  1.108032 |  6.440364 |  9.522869 | 0.038352  0.038477  0.038703  0.038995   0.03931 | 0.038168 |
    | 02/01 20:00 | 18.5 | -20.0 | -20.0 |  0.092622 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 166.651304  249.976957 |  0.0 |  25.974 |  0.098179 | 0.005264 | 0.005264 | 25.896354 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.054039 |      0.0 | 0.064083 |      0.0 | 0.010575 | 0.128697 | 0.133759 | 0.133759 | 0.037155 | 1.994736 | 166.651304  249.976957 |       0.0        0.0 |  75.907672 |       0.0 |  1.053992 |  6.376282 |  9.512294 | 0.037357  0.037477  0.037693  0.037973  0.038275 | 0.037155 |
    | 02/01 21:00 | 15.4 | -20.0 | -20.0 |  0.092451 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 183.944373   275.91656 |  0.0 | 21.6216 |  0.097998 |  0.00812 |  0.00812 | 21.616336 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.051404 |      0.0 | 0.063445 |      0.0 | 0.010563 | 0.125412 | 0.130265 | 0.130265 | 0.036185 |  1.99188 | 183.944373   275.91656 |       0.0        0.0 |  75.907672 |       0.0 |  1.002588 |  6.312836 |  9.501731 | 0.036403  0.036518  0.036725  0.036993  0.037283 | 0.036185 |
    | 02/01 22:00 |  6.3 | -20.0 | -20.0 |  0.091248 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 191.014038  286.521056 |  0.0 |  8.8452 |  0.096723 | 0.028755 | 0.028755 |   8.83708 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.048897 |      0.0 | 0.062814 |      0.0 | 0.010552 | 0.122262 | 0.126915 | 0.126915 | 0.035254 | 1.971245 | 191.014038  286.521056 |       0.0        0.0 |  75.907672 |       0.0 |  0.953692 |  6.250023 |  9.491179 | 0.035488  0.035598  0.035797  0.036054  0.036332 | 0.035254 |
    | 02/01 23:00 |  1.9 | -20.0 | -20.0 |  0.089683 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 193.125113   289.68767 |  0.0 |  2.6676 |  0.095064 |  0.05242 |  0.05242 |  2.638845 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.046512 |      0.0 | 0.062189 |      0.0 |  0.01054 | 0.119241 | 0.123703 | 0.123703 | 0.034362 |  1.94758 | 193.125113   289.68767 |       0.0        0.0 |  75.907672 |       0.0 |   0.90718 |  6.187834 |  9.480639 |  0.03461  0.034716  0.034907  0.035153   0.03542 | 0.034362 |
    | 03/01 00:00 |  4.9 |  20.0 |  20.0 |  0.089858 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 197.740137  296.610206 |  0.0 |  5.8212 |  0.095249 | 0.038316 | 0.038316 |   5.76878 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.044244 |      0.0 |  0.06157 |      0.0 | 0.010528 | 0.116342 | 0.120623 | 0.120623 | 0.033506 | 1.961684 |  184.27879  280.841347 | 13.461347  15.768859 |  75.907672 |       0.0 |  0.862936 |  6.126264 |  9.470111 | 0.033768   0.03387  0.034053   0.03429  0.034545 | 0.033506 |
    | 03/01 01:00 |  2.7 |  20.0 |  20.0 |  0.089683 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 200.275564  300.413346 |  0.0 |  3.2076 |  0.095064 | 0.049664 | 0.049664 |  3.169284 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.042086 |      0.0 | 0.060957 |      0.0 | 0.010517 |  0.11356 | 0.117668 | 0.117668 | 0.032685 | 1.950336 | 175.432468  271.995024 | 24.843097  28.418322 |  75.907672 |       0.0 |   0.82085 |  6.065307 |  9.459595 | 0.032961  0.033058  0.033234  0.033461  0.033706 | 0.032685 |
    | 03/01 02:00 |  0.5 |  20.0 |  20.0 |  0.088805 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 200.711033  301.066549 |  0.0 |   0.594 |  0.094133 | 0.063867 | 0.063867 |  0.544336 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.040033 |      0.0 | 0.060351 |      0.0 | 0.010505 | 0.110889 | 0.114832 | 0.114832 | 0.031898 | 1.936133 | 166.586145  263.148701 | 34.124888  37.917848 |  75.907672 |       0.0 |  0.780817 |  6.004956 |   9.44909 | 0.032185  0.032278  0.032447  0.032665    0.0329 | 0.031898 |
    | 03/01 03:00 |  0.2 |  20.0 |  20.0 |  0.089157 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 200.850019  301.275028 |  0.0 |  0.2376 |  0.094506 | 0.066447 | 0.066447 |  0.173733 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.038081 |      0.0 |  0.05975 |      0.0 | 0.010493 | 0.108324 |  0.11211 |  0.11211 | 0.031142 | 1.933553 | 157.739822  254.302379 | 43.110197   46.97265 |  75.907672 |       0.0 |  0.742736 |  5.945205 |  9.438597 |  0.03144   0.03153  0.031692  0.031901  0.032127 | 0.031142 |
    | 03/01 04:00 |  0.5 |  20.0 |  20.0 |  0.090207 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 201.272061  301.908092 |  0.0 |   0.594 |  0.095619 | 0.064876 | 0.064876 |  0.527553 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.036224 |      0.0 | 0.059156 |      0.0 | 0.010482 | 0.105861 | 0.109497 | 0.109497 | 0.030416 | 1.935124 | 148.893499  245.456056 | 52.378562  56.452036 |  75.907672 |       0.0 |  0.706512 |   5.88605 |  9.428115 | 0.030725  0.030811  0.030966  0.031167  0.031384 | 0.030416 |
    | 03/01 05:00 |  2.4 |  20.0 |  20.0 |  0.091593 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 196.066047  305.251681 |  0.0 |  2.8512 |  0.097089 | 0.052562 | 0.052562 |  2.786324 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |  3.717537 | 0.535509 | 0.0 |      0.0 | 0.535509 | 0.0 |      0.0 |      0.0 | 0.535509 | 0.047624 |      0.0 | 0.058567 |      0.0 |  0.01047 | 0.116661 | 0.107927 | 0.107927 |  0.02998 | 1.947438 | 140.047176  236.609733 |  56.01887  68.641949 |    79.0897 |       0.0 |  1.194397 |  5.827483 |  9.417645 | 0.033727  0.033381  0.032808  0.032183  0.031689 |  0.02998 |
    | 03/01 06:00 |  0.4 |  20.0 |  20.0 |  0.154861 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 183.681195  305.758847 |  0.0 |  0.4752 |  0.164153 | 0.112705 | 0.112705 |  0.422638 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |  6.361481 | 0.994805 | 0.0 |      0.0 | 0.994805 | 0.0 |      0.0 |      0.0 | 0.994805 | 0.082712 |      0.0 | 0.057984 |      0.0 | 0.010458 | 0.151155 | 0.114733 | 0.114733 |  0.03187 | 1.887295 | 131.200853   227.76341 | 52.480341  77.995437 |  84.456376 |       0.0 |   2.10649 |  5.769498 |  9.407187 | 0.043494  0.042347  0.040377   0.03806  0.035931 |  0.03187 |
    | 03/01 07:00 |  0.2 |  20.0 |  20.0 |  0.470369 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 171.296343  305.908721 |  0.0 |  0.2376 |  0.498591 | 0.350557 | 0.350557 |  0.124895 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |  6.242384 | 1.113154 | 0.0 |      0.0 | 1.113154 | 0.0 |      0.0 |      0.0 | 1.113154 | 0.130106 |      0.0 | 0.057407 |      0.0 | 0.010447 |  0.19796 | 0.138867 | 0.138867 | 0.038574 | 1.649443 | 122.354531  218.917087 | 48.941812  86.991634 |  89.585606 |       0.0 |  3.089539 |  5.712091 |   9.39674 | 0.056939  0.055337  0.052511  0.049012  0.045502 | 0.038574 |
    | 03/01 08:00 |  0.0 |  20.0 |  20.0 |  1.173726 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 158.911491   294.09907 |  0.0 |     0.0 |   1.24415 | 0.895788 | 0.895788 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.097251 | 2.427187 | 0.0 |      0.0 |      2.0 | 0.0 |      0.0 | 0.040652 |      2.0 | 0.199856 |      0.0 | 0.056836 |      0.0 | 0.010435 | 0.307779 |  0.18281 |  0.18281 | 0.050781 | 0.753656 | 113.508208  210.070764 | 45.403283  84.028306 |   99.25567 |  0.386534 |  4.889683 |  5.655254 |  9.386305 | 0.088167  0.084482  0.078095  0.070431  0.063096 | 0.050781 |
    | 03/01 09:00 |  0.0 |  20.0 |  20.0 |  4.202296 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 146.526639  281.714218 |  0.0 |     0.0 |  4.454434 | 3.207192 | 0.753656 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 3.050292 | 0.0 |      0.0 |      2.0 | 0.0 |      0.0 | 0.136732 |      2.0 |  0.28765 |      0.0 | 0.056271 |      0.0 | 0.010423 | 0.491076 | 0.269648 | 0.269648 | 0.074902 |      0.0 | 104.661885  201.224442 | 41.864754  80.489777 |  108.59023 |  1.300095 |  6.602033 |  5.598984 |  9.375882 | 0.140584  0.134365  0.123478  0.110179  0.097094 | 0.074902 |
    | 03/01 10:00 |  0.3 |  20.0 |  20.0 |  4.359715 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 134.141787  269.329366 |  0.0 |  0.3564 |  4.621298 | 3.210837 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 3.651004 | 0.0 |      0.0 |      2.0 | 0.0 |      0.0 | 0.280834 |      2.0 | 0.371162 |      0.0 | 0.055711 |      0.0 | 0.010412 | 0.718119 | 0.419265 | 0.419265 | 0.116462 |      0.0 |  95.815562  192.378119 | 38.326225  76.951247 | 117.324078 |  2.670265 |  8.230871 |  5.543273 |   9.36547 | 0.205974  0.198162  0.184299  0.166948  0.149172 | 0.116462 |
    | 03/01 11:00 |  2.6 |  20.0 |  20.0 |  5.305753 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 121.756935  256.944514 |  0.0 |  3.0888 |  5.624098 | 2.973312 |      2.0 |    1.0888 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 13.473652 |   4.6366 | 0.0 |      0.0 |      2.0 | 0.0 |      0.0 | 0.505015 |      2.0 | 0.450601 |      0.0 | 0.055156 |      0.0 |   0.0104 | 1.021173 | 0.626304 | 0.626304 | 0.173973 |      0.0 |  86.969239  183.531796 | 34.787696  73.412718 |  126.16113 |   4.80185 |   9.78027 |  5.488116 |   9.35507 | 0.293101  0.282708  0.264312  0.241367  0.217935 | 0.173973 |
    | 03/01 12:00 |  0.7 |  20.0 |  20.0 |  5.376027 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 109.372083  244.559662 |  0.0 |  0.8316 |  5.698589 | 3.775582 |   0.8316 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 4.928128 | 0.0 |      0.0 |      2.0 | 0.0 |      0.0 | 0.735605 |      2.0 | 0.526166 |      0.0 | 0.054608 |      0.0 | 0.010389 | 1.326767 | 0.892621 | 0.892621 |  0.24795 |      0.0 |  78.122916  174.685473 | 31.249167  69.874189 | 133.617854 |  6.994373 | 11.254104 |  5.433509 |  9.344681 | 0.381673  0.371025  0.351903  0.327452  0.301515 |  0.24795 |
    | 03/01 13:00 |  0.3 |  20.0 |  20.0 |  4.658915 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  96.987231   232.17481 |  0.0 |  0.3564 |   4.93845 | 3.431191 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 5.527895 | 0.0 |      0.0 |      2.0 | 0.0 |      0.0 | 1.001326 | 1.937295 | 0.596504 | 0.055738 | 0.054342 | 0.006967 | 0.010381 | 1.662554 | 1.192496 | 1.192496 | 0.331249 |      0.0 |  69.276594   165.83915 | 27.710637   66.33566 |  140.47481 |  9.520942 | 12.594894 |  5.434905 |  9.341267 | 0.478754  0.467109  0.446268  0.419749  0.391747 | 0.331249 |
    | 03/01 14:00 |  0.3 |  20.0 |  20.0 |  7.789594 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  84.602379  219.789958 |  0.0 |  0.3564 |   8.25697 | 5.736869 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 |  6.10981 | 0.0 |      0.0 |      2.0 | 0.0 |      0.0 | 1.297138 | 1.870255 | 0.660247 | 0.115329 | 0.054653 | 0.014416 | 0.010381 | 2.022419 | 1.516091 | 1.516091 | 0.421136 |      0.0 |  60.430271  156.992827 | 24.172108  62.797131 | 146.749852 | 12.333615 | 13.804903 |   5.49558 |  9.345302 |  0.58287  0.570373  0.547986  0.519455  0.489273 | 0.421136 |
    | 03/01 15:00 |  0.0 |  20.0 |  20.0 |  4.851567 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  72.217527  207.405106 |  0.0 |     0.0 |  5.142661 | 3.702716 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 6.667855 | 0.0 |      0.0 |      2.0 | 0.0 |      0.0 | 1.617904 | 1.809755 | 0.717772 | 0.169107 | 0.055525 | 0.021138 |  0.01039 |  2.40159 | 1.864437 | 1.864437 | 0.517899 |      0.0 |  51.583948  148.146505 | 20.633579  59.258602 | 152.466848 | 15.383567 | 14.896885 |  5.609162 |  9.356051 | 0.692633   0.67945   0.65581  0.625624  0.593592 | 0.517899 |
    | 03/01 16:00 |  0.0 |  20.0 |  20.0 |   5.30692 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  59.832675  195.020254 |  0.0 |     0.0 |  5.625335 | 4.050241 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 |   7.1975 | 0.0 |      0.0 |      2.0 | 0.0 | 0.228632 | 1.958547 | 1.755156 | 0.769686 | 0.217639 | 0.056897 | 0.027205 | 0.010405 | 3.024167 | 2.250426 | 2.250426 | 0.625118 |      0.0 |  42.737625  139.300182 |  17.09505  55.720073 |   157.6542 | 18.393887 | 15.882355 |  5.769905 |  9.372851 | 0.870794  0.849636  0.812491  0.766807  0.721122 | 0.625118 |
    | 03/01 17:00 |  0.0 |  20.0 |  20.0 |  3.286036 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  47.447823  182.635402 |  0.0 |     0.0 |  3.483198 | 2.507903 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 |  7.69559 | 0.0 |      0.0 |      2.0 | 0.0 | 1.609084 | 2.292417 | 1.705882 | 0.816537 | 0.261438 | 0.058714 |  0.03268 | 0.010427 | 4.787178 |  2.84839 |  2.84839 | 0.791219 |      0.0 |  33.891302  130.453859 | 13.556521  52.181544 | 162.343462 | 20.187976 |   16.7717 |  5.972629 |  9.395104 | 1.370841  1.311991  1.210498  1.089871   0.97644 | 0.791219 |
    | 03/01 18:00 |  0.0 |  20.0 |  20.0 |  1.506216 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  35.062971   170.25055 |  0.0 |     0.0 |  1.596589 | 1.149544 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 8.160193 | 0.0 |      0.0 |      2.0 | 0.0 |  2.49781 |  2.50736 | 1.661415 | 0.858817 | 0.300964 | 0.060928 | 0.037621 | 0.010454 |  5.93537 |  4.05913 |  4.05913 | 1.127536 |      0.0 |  25.044979  121.607536 | 10.017992  48.643014 | 166.568121 |    21.343 | 17.574298 |  6.212665 |   9.42227 | 1.709447  1.668096  1.591795  1.490094  1.376447 | 1.127536 |
    | 03/01 19:00 |  0.0 |  20.0 |  20.0 |  0.274762 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  22.678119  157.865698 |  0.0 |     0.0 |  0.291248 | 0.209698 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 8.590424 | 0.0 |      0.0 |      2.0 | 0.0 | 3.121559 | 2.658217 | 1.621285 | 0.896974 | 0.336635 | 0.063495 | 0.042079 | 0.010487 | 6.750731 |  5.34981 |  5.34981 | 1.486058 |      0.0 |  16.198657  112.761213 |  6.479463  45.104485 | 170.362549 | 22.153648 |  18.29861 |  6.485806 |  9.453863 | 1.949345   1.92006  1.865892  1.792862  1.708642 | 1.486058 |
    | 03/01 20:00 |  0.0 |  20.0 |  20.0 |  0.087565 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  10.293267  145.480847 |  0.0 |     0.0 |  0.092819 |  0.06683 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 8.986262 | 0.0 |      0.0 |      2.0 | 0.0 | 3.596274 | 2.773029 |  1.58507 | 0.931408 | 0.368827 | 0.066373 | 0.046103 | 0.010524 | 7.377609 | 6.333789 | 6.333789 | 1.759386 |      0.0 |   7.352334   103.91489 |  2.940934  41.565956 | 173.761139 | 22.770607 | 18.952271 |   6.78826 |  9.489442 | 2.133097  2.110747  2.069669  2.014827   1.95228 | 1.759386 |
    | 03/01 21:00 |  0.0 |  20.0 |  20.0 |  0.085771 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |        0.0  133.095995 |  0.0 |     0.0 |  0.090917 |  0.06546 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 7.352334  8.846323 | 0.0   0.0 |  11.33906 | 8.558987 | 0.0 |  0.03273 |      2.0 | 0.0 | 3.670909 |  2.79108 | 1.552386 | 0.962484 | 0.397879 | 0.069527 | 0.049735 | 0.010566 | 7.504566 | 7.038188 | 7.038188 | 1.955052 |      0.0 |        0.0   95.068568 |       0.0  38.027427 | 176.508481 | 22.867605 | 19.542173 |  7.116611 |  9.528611 | 2.174867  2.169258  2.157187  2.137233  2.108452 | 1.955052 |
    | 03/01 22:00 |  0.0 |  20.0 |  20.0 |  0.084317 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |        0.0  120.711143 |  0.0 |     0.0 |  0.089376 | 0.064351 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.846323 | 0.0   0.0 |  6.192426 | 4.823164 | 0.0 | 0.032175 |      2.0 | 0.0 | 2.239143 |   2.4448 | 1.522891 | 0.990529 | 0.424097 | 0.072925 | 0.053012 | 0.010611 | 5.758007 | 7.264635 | 7.264635 | 2.017954 |      0.0 |        0.0   86.222245 |       0.0  34.488898 | 177.845568 | 21.006826 | 20.074536 |  7.467783 |  9.571013 | 1.686844  1.743394  1.837868  1.943024  2.029238 | 2.017954 |
    | 03/01 23:00 |  0.0 |  20.0 |  20.0 |  0.083215 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |        0.0  108.326291 |  0.0 |     0.0 |  0.088208 |  0.06351 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.846323 | 0.0   0.0 |  6.192426 | 4.896513 | 0.0 | 0.031755 |      2.0 | 0.0 | 1.535844 | 2.274703 | 1.496273 | 1.015838 | 0.447757 | 0.076537 |  0.05597 |  0.01066 | 4.913582 | 6.427742 | 6.427742 | 1.785484 |      0.0 |        0.0   77.375922 |       0.0  30.950369 | 179.109726 | 20.092791 | 20.554972 |  7.839003 |  9.616323 | 1.433766  1.465095  1.524163  1.605149  1.698036 | 1.785484 |
    | 04/01 00:00 |  0.0 |  20.0 |  20.0 |  0.082289 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0    95.93582 |  0.0 |     0.0 |  0.087226 | 0.062803 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 4.968624 | 0.0 | 0.031401 |      2.0 | 0.0 | 1.204573 | 2.194584 | 1.472251 | 1.038678 |  0.46911 | 0.080337 | 0.058639 | 0.010711 | 4.528884 | 5.390317 | 5.390317 |  1.49731 |      0.0 |        0.0   68.525586 |       0.0  27.410234 | 180.304936 | 19.662259 | 20.988545 |  8.227776 |   9.66425 | 1.317434  1.332015  1.360293  1.401386  1.453648 |  1.49731 |
    | 04/01 01:00 |  0.0 |  20.0 |  20.0 |    0.0845 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   83.545349 |  0.0 |     0.0 |   0.08957 |  0.06449 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.035157 | 0.0 | 0.032245 |      2.0 | 0.0 |  1.06135 | 2.159945 | 1.450573 | 1.059291 |  0.48838 | 0.084302 | 0.061047 | 0.010766 | 4.375653 | 4.769316 | 4.769316 |  1.32481 |      0.0 |        0.0   59.675249 |       0.0    23.8701 | 181.432769 | 19.476121 | 21.379827 |  8.631854 |  9.714531 | 1.270574  1.276513  1.288276  1.305989  1.329761 |  1.32481 |
    | 04/01 02:00 |  0.0 |  20.0 |  20.0 |  0.084864 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   71.154878 |  0.0 |     0.0 |  0.089956 | 0.064768 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.098346 | 0.0 | 0.032384 |      2.0 | 0.0 | 1.012974 | 2.148245 | 1.431009 | 1.077893 |  0.50577 | 0.088409 | 0.063221 | 0.010823 | 4.338343 | 4.480103 | 4.480103 | 1.244473 |      0.0 |        0.0   50.824913 |       0.0  20.329965 | 182.497274 | 19.413248 | 21.732943 |  9.049215 |   9.76693 | 1.258539  1.260139  1.263558  1.269269  1.277848 | 1.244473 |
    | 04/01 03:00 |  1.3 |  20.0 |  20.0 |  0.083584 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   58.764408 |  0.0 |  1.5444 |  0.088599 | 0.054662 | 0.054662 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.158347 | 0.0 | 0.013666 |      2.0 | 0.0 | 1.011844 | 2.147971 | 1.413353 |  1.09468 | 0.521464 |  0.09264 | 0.065183 | 0.010882 | 4.358018 | 4.372969 | 4.372969 | 1.214714 | 1.489738 |        0.0   41.974577 |       0.0  16.789831 | 183.520497 |  19.41178 | 22.051615 |   9.47804 |   9.82123 | 1.263647  1.263107  1.262402  1.262148  1.263096 | 1.214714 |
    | 04/01 04:00 |  0.0 |  20.0 |  20.0 |    0.0834 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   46.373937 |  0.0 |     0.0 |  0.088404 | 0.063651 | 0.063651 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.216353 | 0.0 | 0.015913 |      2.0 | 0.0 |  1.03409 | 2.153352 | 1.397419 |  1.10983 | 0.535627 | 0.096977 | 0.066953 | 0.010944 | 4.405193 | 4.357932 | 4.357932 | 1.210537 | 1.426087 |        0.0   33.124241 |       0.0  13.249696 | 184.483466 | 19.440692 | 22.339204 |   9.91669 |   9.87724 | 1.277039  1.275464  1.272758  1.269598  1.266803 | 1.210537 |
    | 04/01 05:00 |  0.0 |  20.0 |  20.0 |  0.084864 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   33.983466 |  0.0 |     0.0 |  0.089956 | 0.064768 | 0.064768 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.271239 | 0.0 | 0.016192 |      2.0 | 0.0 | 1.067062 | 2.161326 |  1.38304 | 1.123503 | 0.548409 | 0.101406 | 0.068551 | 0.011007 | 4.464303 | 4.388116 | 4.388116 | 1.218921 | 1.361318 |        0.0   24.273904 |       0.0   9.709562 | 185.391271 | 19.483543 | 22.598741 | 10.363693 |  9.934785 | 1.294054  1.292023  1.288425  1.283942  1.279404 | 1.218921 |
    | 04/01 06:00 |  0.0 |  20.0 |  20.0 |  0.310229 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   21.592995 |  0.0 |     0.0 |  0.328843 | 0.236767 | 0.236767 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.323244 | 0.0 | 0.059192 |      2.0 | 0.0 | 1.104385 | 2.170353 | 1.370063 | 1.135841 | 0.559944 | 0.105911 | 0.069993 | 0.011071 | 4.527561 | 4.439468 | 4.439468 | 1.233186 | 1.124552 |        0.0   15.423568 |       0.0   6.169427 |  186.20407 |  19.53205 | 22.832963 | 10.817726 |  9.993706 | 1.312354  1.310158  1.306227  1.301229  1.295972 | 1.233186 |
    | 04/01 07:00 |  0.7 |  20.0 |  20.0 |  1.391958 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0    9.202525 |  0.0 |  0.8316 |  1.475475 | 0.977572 | 0.977572 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.370023 | 0.0 | 0.244393 |      2.0 | 0.0 | 1.141877 |  2.17942 | 1.358352 | 1.146977 | 0.570354 |  0.11048 | 0.071294 | 0.011138 | 4.589892 | 4.499573 | 4.499573 | 1.249881 |  0.97858 |        0.0    6.573232 |       0.0   2.629293 |  186.78489 | 19.580776 | 23.044338 |   11.2776 | 10.053863 | 1.330438  1.328261  1.324343  1.319307  1.313911 | 1.249881 |
    | 04/01 08:00 |  0.4 |  20.0 |  20.0 |  3.195876 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |  0.4752 |  3.387629 | 2.325898 |  1.45378 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  6.573232 | 0.0   0.0 |  4.601262 | 4.013289 | 0.0 | 1.599008 |      2.0 | 0.0 | 0.627216 | 2.054947 | 1.347783 | 1.157026 | 0.579748 | 0.115103 | 0.072469 | 0.011205 | 3.965496 | 4.512795 | 4.512795 | 1.253554 |      0.0 |        0.0         0.0 |       0.0        0.0 | 185.773855 | 18.911902 | 23.235095 | 11.742245 | 10.115126 | 1.156076  1.176274  1.210022  1.247675  1.278915 | 1.253554 |
    | 04/01 09:00 |  0.1 |  20.0 |  20.0 |  5.191651 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |  0.1188 |   5.50315 | 3.915475 |   0.1188 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 3.856075 |      2.0 | 0.0 |      0.0 |  1.60938 | 1.338245 | 1.166094 | 0.588226 | 0.119769 | 0.073528 | 0.011274 | 2.906517 | 4.151267 | 4.151267 |  1.15313 |      0.0 |        0.0         0.0 |       0.0        0.0 |  181.91778 | 15.302522 | 23.407246 | 12.210703 | 10.177381 | 0.853445  0.889311  0.951945   1.02795   1.10156 |  1.15313 |
    | 04/01 10:00 |  0.4 |  20.0 |  20.0 |  7.155036 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |  0.4752 |  7.584338 | 5.207299 |   0.4752 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 4.969699 |      2.0 | 0.0 |      0.0 | 1.265902 | 1.329638 | 1.174279 | 0.595878 | 0.124468 | 0.074485 | 0.011343 | 2.575992 | 3.387139 | 3.387139 | 0.940872 |      0.0 |        0.0         0.0 |       0.0        0.0 | 176.948081 |  12.03662 | 23.562605 | 12.682112 | 10.240523 | 0.750624  0.763787  0.790051  0.829238  0.879365 | 0.940872 |
    | 04/01 11:00 |  0.0 |  20.0 |  20.0 |  8.391432 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |     0.0 |  8.894918 | 6.404341 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 6.404341 |      2.0 | 0.0 |      0.0 | 0.955111 |  1.32187 | 1.181665 | 0.602782 | 0.129193 | 0.075348 | 0.011414 | 2.277382 | 2.789838 | 2.789838 | 0.774955 |      0.0 |        0.0         0.0 |       0.0        0.0 |  170.54374 |  9.081509 |  23.70281 | 13.155702 | 10.304456 |  0.66348  0.674056  0.693474  0.719564  0.750035 | 0.774955 |
    | 04/01 12:00 |  0.0 |  20.0 |  20.0 |  8.391286 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |     0.0 |  8.894763 | 6.404229 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 6.404229 |      2.0 | 0.0 |      0.0 | 0.673895 | 1.314859 |  1.18833 | 0.609014 | 0.133936 | 0.076127 | 0.011485 | 2.007646 | 2.424097 | 2.424097 |  0.67336 |      0.0 |        0.0         0.0 |       0.0        0.0 | 164.139511 |  6.407614 |  23.82934 | 13.630779 | 10.369098 | 0.584956  0.594442  0.611653  0.634131  0.658977 |  0.67336 |
    | 04/01 13:00 |  0.0 |  20.0 |  20.0 | 10.715238 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |     0.0 | 11.358152 |  8.17787 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |  8.17787 |      2.0 | 0.0 |      0.0 |  0.41944 | 1.308533 | 1.194345 | 0.614637 | 0.138691 |  0.07683 | 0.011557 | 1.764034 | 2.134568 | 2.134568 | 0.592936 |      0.0 |        0.0         0.0 |       0.0        0.0 | 155.961641 |  3.988174 | 23.943527 | 14.106725 |  10.43437 | 0.514043  0.522608  0.538132  0.558346  0.580497 | 0.592936 |
    | 04/01 14:00 |  0.0 |  20.0 |  20.0 |  9.383394 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |     0.0 |  9.946398 | 7.161406 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 6.980654 |      2.0 | 0.0 |      0.0 |   0.1892 | 1.302824 | 1.199774 | 0.619712 | 0.143453 | 0.077464 |  0.01163 | 1.544057 | 1.878171 | 1.878171 | 0.521714 |      0.0 |        0.0         0.0 |       0.0        0.0 | 148.980987 |  1.798975 | 24.046577 | 14.582985 | 10.500204 |  0.45001  0.457743  0.471761  0.490008  0.509988 | 0.521714 |
    | 04/01 15:00 |  0.0 |  20.0 |  20.0 |  7.861915 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |     0.0 |   8.33363 | 6.000214 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 5.586986 | 1.798975 | 0.0 |      0.0 |      0.0 | 1.297671 | 1.204673 | 0.445603 | 0.147324 |   0.0557 | 0.011691 | 1.363688 | 1.648392 | 1.648392 | 0.457887 |      0.0 |        0.0         0.0 |       0.0        0.0 | 143.394001 |       0.0 | 24.139575 | 14.881264 | 10.544213 | 0.397308  0.403696  0.415351  0.430693  0.447758 | 0.457887 |
    | 04/01 16:00 |  0.0 |  20.0 |  20.0 |  6.298329 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |     0.0 |  6.676229 | 4.806885 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |  4.30799 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 1.177301 |      0.0 | 0.148071 |      0.0 | 0.011709 | 1.337081 | 1.459809 | 1.459809 | 0.405502 |      0.0 |        0.0         0.0 |       0.0        0.0 | 139.086011 |       0.0 | 22.962274 | 14.733193 | 10.532503 | 0.388076  0.389364  0.392282  0.397364  0.404994 | 0.405502 |
    | 04/01 17:00 |  0.0 |  20.0 |  20.0 |  2.948416 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |     0.0 |  3.125321 | 2.250231 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 1.956098 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 1.119883 |      0.0 | 0.146598 |      0.0 | 0.011696 | 1.278177 | 1.361564 | 1.361564 | 0.378212 |      0.0 |        0.0         0.0 |       0.0        0.0 | 137.129913 |       0.0 | 21.842391 | 14.586596 | 10.520807 | 0.371258  0.373257  0.376791  0.381264  0.386122 | 0.378212 |
    | 04/01 18:00 |  0.0 |  20.0 |  20.0 |  1.309232 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |     0.0 |  1.387786 | 0.999206 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.856381 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 1.065266 |      0.0 | 0.145139 |      0.0 | 0.011683 | 1.222088 | 1.303539 | 1.303539 | 0.362094 |      0.0 |        0.0         0.0 |       0.0        0.0 | 136.273531 |       0.0 | 20.777125 | 14.441457 | 10.509124 | 0.354974  0.356934  0.360465  0.365003  0.369866 | 0.362094 |
    | 04/01 19:00 |  0.0 |  20.0 |  20.0 |   0.32955 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |     0.0 |  0.349323 | 0.251513 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.214216 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 1.013312 |      0.0 | 0.143695 |      0.0 |  0.01167 | 1.168678 | 1.247584 | 1.247584 | 0.346551 |      0.0 |        0.0         0.0 |       0.0        0.0 | 136.059316 |       0.0 | 19.763812 | 14.297762 | 10.497454 | 0.339458  0.341328  0.344704  0.349065  0.353781 | 0.346551 |
    | 04/01 20:00 |  0.0 |  20.0 |  20.0 |  0.089508 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |     0.0 |  0.094878 | 0.068313 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.058091 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.963893 |      0.0 | 0.142265 |      0.0 | 0.011657 | 1.117815 | 1.193155 | 1.193155 | 0.331432 |      0.0 |        0.0         0.0 |       0.0        0.0 | 136.001225 |       0.0 |  18.79992 | 14.155497 | 10.485796 | 0.324682  0.326463  0.329678  0.333836  0.338337 | 0.331432 |
    | 04/01 21:00 |  0.0 |  20.0 |  20.0 |  0.085771 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |     0.0 |  0.090917 |  0.06546 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.055642 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.916883 |      0.0 |  0.14085 |      0.0 | 0.011644 | 1.069377 | 1.141144 | 1.141144 | 0.316984 |      0.0 |        0.0         0.0 |       0.0        0.0 | 135.945583 |       0.0 | 17.883037 | 14.014647 | 10.474152 |  0.31061  0.312306  0.315369  0.319328  0.323615 | 0.316984 |
    | 04/01 22:00 |  0.0 |  20.0 |  20.0 |    0.0845 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |     0.0 |   0.08957 |  0.06449 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.054795 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.872166 |      0.0 | 0.139448 |      0.0 | 0.011631 | 1.023246 | 1.091595 | 1.091595 | 0.303221 |      0.0 |        0.0         0.0 |       0.0        0.0 | 135.890788 |       0.0 | 17.010871 | 13.875199 |  10.46252 | 0.297209  0.298824   0.30174  0.305511  0.309595 | 0.303221 |
    | 04/01 23:00 |  0.0 |  20.0 |  20.0 |  0.084864 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |     0.0 |  0.089956 | 0.064768 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.055009 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |  0.82963 |      0.0 | 0.138061 |      0.0 | 0.011619 | 0.979309 | 1.044405 | 1.044405 | 0.290113 |      0.0 |        0.0         0.0 |       0.0        0.0 | 135.835779 |       0.0 | 16.181241 | 13.737138 | 10.450902 | 0.284445  0.285983  0.288761  0.292352  0.296241 | 0.290113 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_v3_glacier:

glacier
_______

Regarding glacier simulation, all functionalities of |hland_v3| are identical with the
ones of |hland_v1|.  Hence, all snow data in the following table agrees with the
:ref:`hland_v1_glacier` example of |hland_v1|:

.. integration-test::

    >>> zonetype(GLACIER)
    >>> test('hland_v3_glacier')
    |        date |    p |     t |    tn |       epn | tmean |    tc | fracrain | rfc | sfc |    cfact |                    swe |     gact |      pc |        ep |      epc |  ei |      tf | spl | wcl | spg | wcg |    glmelt |               melt |      refr |       in_ |         r |  sr |  ea |       dp |  el |        rs |       ri |      gr1 |      rg1 |      gr2 |      rg2 |      gr3 |      rg3 |      inuh |     outuh |        rt |       qt |  ic |                     sp |                   wc |  sm |       suz |       sg1 |       sg2 |       sg3 |                                               sc |   outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | -20.0 | -20.0 |  0.100707 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |     0.0 |  0.106749 |  0.07686 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.487706 |      0.0 | 0.099502 |      0.0 | 0.011105 |  0.598312 |  0.202814 |  0.202814 | 0.056337 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.512294 |  9.900498 |  9.988895 | 0.169348  0.155472  0.132115  0.105643  0.082921 | 0.056337 |
    | 01/01 01:00 |  0.0 | -20.0 | -20.0 |  0.097801 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |     0.0 |  0.103669 | 0.074642 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 |  0.46392 |      0.0 | 0.098512 |      0.0 | 0.011093 |  0.573524 |  0.414293 |  0.414293 | 0.115081 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.048374 |  9.801987 |  9.977802 | 0.166424  0.166296  0.164522  0.159031  0.148456 | 0.115081 |
    | 01/01 02:00 |  0.0 | -20.0 | -20.0 |  0.096981 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |     0.0 |    0.1028 | 0.074016 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.441294 |      0.0 | 0.097531 |      0.0 |  0.01108 |  0.549906 |  0.547167 |  0.547167 | 0.151991 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |   8.60708 |  9.704455 |  9.966722 | 0.159707  0.160484  0.161724  0.162805  0.162748 | 0.151991 |
    | 01/01 03:00 |  0.0 | -20.0 | -20.0 |   0.09599 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |     0.0 |  0.101749 |  0.07326 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.419772 |      0.0 | 0.096561 |      0.0 | 0.011068 |  0.527401 |  0.556325 |  0.556325 | 0.154535 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  8.187308 |  9.607894 |  9.955654 | 0.153174   0.15396  0.155368  0.157139  0.158904 | 0.154535 |
    | 01/01 04:00 |  0.0 | -20.0 | -20.0 |  0.096981 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |     0.0 |    0.1028 | 0.074016 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 |   0.3993 |      0.0 |   0.0956 |      0.0 | 0.011056 |  0.505956 |  0.537334 |  0.537334 |  0.14926 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  7.788008 |  9.512294 |  9.944598 | 0.146944  0.147695   0.14905  0.150798  0.152677 |  0.14926 |
    | 01/01 05:00 |  0.0 | -20.0 | -20.0 |  0.102761 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |     0.0 |  0.108927 | 0.078427 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.379826 |      0.0 | 0.094649 |      0.0 | 0.011043 |  0.485518 |  0.515762 |  0.515762 | 0.143267 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  7.408182 |  9.417645 |  9.933555 | 0.141007  0.141723  0.143015  0.144685  0.146492 | 0.143267 |
    | 01/01 06:00 |  0.0 | -20.0 | -20.0 |  0.291908 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |     0.0 |  0.309422 | 0.222784 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.361301 |      0.0 | 0.093707 |      0.0 | 0.011031 |   0.46604 |  0.494888 |  0.494888 | 0.137469 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  7.046881 |  9.323938 |  9.922524 | 0.135349  0.136031  0.137262  0.138854  0.140577 | 0.137469 |
    | 01/01 07:00 |  0.0 | -20.0 | -20.0 |  1.932875 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |     0.0 |  2.048847 |  1.47517 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 |  0.34368 |      0.0 | 0.092775 |      0.0 | 0.011019 |  0.447474 |  0.474971 |  0.474971 | 0.131936 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |    6.7032 |  9.231163 |  9.911505 | 0.129956  0.130606  0.131779  0.133296  0.134939 | 0.131936 |
    | 01/01 08:00 |  0.0 | -20.0 | -20.0 |  4.369536 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |     0.0 |  4.631708 |  3.33483 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.326919 |      0.0 | 0.091852 |      0.0 | 0.011007 |  0.429777 |  0.455987 |  0.455987 | 0.126663 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  6.376282 |  9.139312 |  9.900498 | 0.124815  0.125434  0.126553  0.127999  0.129565 | 0.126663 |
    | 01/01 09:00 |  0.0 | -20.0 | -20.0 |  7.317556 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |     0.0 |  7.756609 | 5.584759 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.310975 |      0.0 | 0.090938 |      0.0 | 0.010994 |  0.412907 |  0.437892 |  0.437892 | 0.121637 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  6.065307 |  9.048374 |  9.889504 | 0.119914  0.120505  0.121571   0.12295  0.124442 | 0.121637 |
    | 01/01 10:00 |  0.0 | -20.0 | -20.0 |  8.264362 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |     0.0 |  8.760224 | 6.307361 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.295808 |      0.0 | 0.090033 |      0.0 | 0.010982 |  0.396824 |  0.420642 |  0.420642 | 0.116845 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  5.769498 |  8.958341 |  9.878522 | 0.115242  0.115805  0.116822  0.118136  0.119559 | 0.116845 |
    | 01/01 11:00 |  0.0 | -20.0 | -20.0 |  9.369867 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |     0.0 |  9.932059 | 7.151082 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.281382 |      0.0 | 0.089137 |      0.0 |  0.01097 |  0.381489 |  0.404198 |  0.404198 | 0.112277 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  5.488116 |  8.869204 |  9.867552 | 0.110787  0.111324  0.112293  0.113546  0.114903 | 0.112277 |
    | 01/01 12:00 |  0.0 | -20.0 | -20.0 |  5.126178 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |     0.0 |  5.433749 | 3.912299 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.267659 |      0.0 |  0.08825 |      0.0 | 0.010958 |  0.366867 |  0.388519 |  0.388519 | 0.107922 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  5.220458 |  8.780954 |  9.856594 | 0.106539  0.107051  0.107976  0.109171  0.110464 | 0.107922 |
    | 01/01 13:00 |  0.0 | -20.0 | -20.0 |   6.62503 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |     0.0 |  7.022532 | 5.056223 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.254605 |      0.0 | 0.087372 |      0.0 | 0.010946 |  0.352922 |   0.37357 |   0.37357 |  0.10377 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  4.965853 |  8.693582 |  9.845648 | 0.102489  0.102977  0.103858  0.104998  0.106231 |  0.10377 |
    | 01/01 14:00 |  0.0 | -20.0 | -20.0 |  7.397619 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |     0.0 |  7.841476 | 5.645863 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.242188 |      0.0 | 0.086503 |      0.0 | 0.010934 |  0.339624 |  0.359315 |  0.359315 |  0.09981 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  4.723666 |   8.60708 |  9.834715 | 0.098626  0.099091  0.099932  0.101018  0.102195 |  0.09981 |
    | 01/01 15:00 |  0.2 | -20.0 | -20.0 |   2.39151 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |    0.22464     0.33696 | 0.901953 |  0.2808 |  2.535001 | 1.774662 | 0.0 |  0.2808 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.230376 |      0.0 | 0.085642 |      0.0 | 0.010921 |  0.326939 |   0.34572 |   0.34572 | 0.096033 | 0.0 |    0.22464     0.33696 |       0.0        0.0 | 0.0 |       0.0 |   4.49329 |  8.521438 |  9.823793 | 0.094941  0.095385  0.096187  0.097223  0.098345 | 0.096033 |
    | 01/01 16:00 |  0.0 | -20.0 | -20.0 |  1.829834 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |    0.22464     0.33696 | 0.901953 |     0.0 |  1.939624 | 1.396529 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 |  0.21914 |      0.0 |  0.08479 |      0.0 | 0.010909 |  0.314839 |  0.332754 |  0.332754 | 0.092432 | 0.0 |    0.22464     0.33696 |       0.0        0.0 | 0.0 |       0.0 |  4.274149 |  8.436648 |  9.812884 | 0.091426   0.09185  0.092614  0.093603  0.094673 | 0.092432 |
    | 01/01 17:00 |  0.0 | -20.0 | -20.0 |  1.136569 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |    0.22464     0.33696 | 0.901953 |     0.0 |  1.204763 | 0.867429 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.208453 |      0.0 | 0.083946 |      0.0 | 0.010897 |  0.303296 |  0.320386 |  0.320386 | 0.088996 | 0.0 |    0.22464     0.33696 |       0.0        0.0 | 0.0 |       0.0 |  4.065697 |  8.352702 |  9.801987 | 0.088073  0.088477  0.089207   0.09015  0.091171 | 0.088996 |
    | 01/01 18:00 |  1.3 | -20.0 | -20.0 |  0.750986 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |     1.6848      2.5272 | 0.901953 |  1.8252 |  0.796045 | 0.477532 | 0.0 |  1.8252 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.198286 |      0.0 | 0.083111 |      0.0 | 0.010885 |  0.292282 |  0.308587 |  0.308587 | 0.085719 | 0.0 |     1.6848      2.5272 |       0.0        0.0 | 0.0 |       0.0 |   3.86741 |  8.269591 |  9.791102 | 0.084873  0.085259  0.085955  0.086855  0.087829 | 0.085719 |
    | 01/01 19:00 |  5.6 | -20.0 | -20.0 |  0.223895 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |    7.97472    11.96208 | 0.901953 |  7.8624 |  0.237329 | 0.077844 | 0.0 |  7.8624 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.188616 |      0.0 | 0.082284 |      0.0 | 0.010873 |  0.281773 |   0.29733 |   0.29733 | 0.082592 | 0.0 |    7.97472    11.96208 |       0.0        0.0 | 0.0 |       0.0 |  3.678794 |  8.187308 |  9.780229 | 0.081821  0.082188  0.082853  0.083711  0.084641 | 0.082592 |
    | 01/01 20:00 |  2.9 | -20.0 | -20.0 |  0.099425 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |     11.232      16.848 | 0.901953 |  4.0716 |   0.10539 | 0.050502 | 0.0 |  4.0716 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.179417 |      0.0 | 0.081465 |      0.0 | 0.010861 |  0.271743 |  0.286589 |  0.286589 | 0.079608 | 0.0 |     11.232      16.848 |       0.0        0.0 | 0.0 |       0.0 |  3.499377 |  8.105842 |  9.769368 | 0.078907  0.079258  0.079892  0.080711  0.081598 | 0.079608 |
    | 01/01 21:00 |  4.9 | -20.0 | -20.0 |  0.098454 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |   16.73568    25.10352 | 0.901953 |  6.8796 |  0.104361 | 0.037765 | 0.0 |  6.8796 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.170667 |      0.0 | 0.080654 |      0.0 | 0.010849 |   0.26217 |   0.27634 |   0.27634 | 0.076761 | 0.0 |   16.73568    25.10352 |       0.0        0.0 | 0.0 |       0.0 |  3.328711 |  8.025188 |  9.758519 | 0.076126  0.076461  0.077066  0.077848  0.078695 | 0.076761 |
    | 01/01 22:00 | 10.6 | -20.0 | -20.0 |  0.098128 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |    28.6416     42.9624 | 0.901953 | 14.8824 |  0.104016 | 0.016908 | 0.0 | 14.8824 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.162343 |      0.0 | 0.079852 |      0.0 | 0.010837 |  0.253032 |  0.266557 |  0.266557 | 0.074044 | 0.0 |    28.6416     42.9624 |       0.0        0.0 | 0.0 |       0.0 |  3.166368 |  7.945336 |  9.747682 | 0.073472  0.073792  0.074369  0.075116  0.075924 | 0.074044 |
    | 01/01 23:00 |  0.1 | -20.0 | -20.0 |  0.097474 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |   28.75392    43.13088 | 0.901953 |  0.1404 |  0.103322 | 0.073355 | 0.0 |  0.1404 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.154426 |      0.0 | 0.079057 |      0.0 | 0.010825 |  0.244308 |   0.25722 |   0.25722 |  0.07145 | 0.0 |   28.75392    43.13088 |       0.0        0.0 | 0.0 |       0.0 |  3.011942 |  7.866279 |  9.736857 | 0.070938  0.071243  0.071794  0.072507  0.073278 |  0.07145 |
    | 02/01 00:00 |  0.7 | -20.0 | -20.0 |  0.097474 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   29.54016    44.31024 | 0.902305 |  0.9828 |  0.103322 | 0.067429 | 0.0 |  0.9828 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.146894 |      0.0 | 0.078271 |      0.0 | 0.010813 |  0.235978 |  0.248305 |  0.248305 | 0.068974 | 0.0 |   29.54016    44.31024 |       0.0        0.0 | 0.0 |       0.0 |  2.865048 |  7.788008 |  9.726045 | 0.068518   0.06881  0.069336  0.070016  0.070753 | 0.068974 |
    | 02/01 01:00 |  3.0 | -20.0 | -20.0 |  0.097474 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   32.90976    49.36464 | 0.902305 |   4.212 |  0.103322 | 0.048821 | 0.0 |   4.212 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 |  0.13973 |      0.0 | 0.077492 |      0.0 | 0.010801 |  0.228023 |  0.239795 |  0.239795 |  0.06661 | 0.0 |   32.90976    49.36464 |       0.0        0.0 | 0.0 |       0.0 |  2.725318 |  7.710516 |  9.715244 | 0.066207  0.066486  0.066988  0.067638  0.068341 |  0.06661 |
    | 02/01 02:00 |  2.1 | -20.0 | -20.0 |  0.096981 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   35.26848    52.90272 | 0.902305 |  2.9484 |    0.1028 | 0.055116 | 0.0 |  2.9484 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.132915 |      0.0 | 0.076721 |      0.0 | 0.010789 |  0.220425 |  0.231668 |  0.231668 | 0.064352 | 0.0 |   35.26848    52.90272 |       0.0        0.0 | 0.0 |       0.0 |  2.592403 |  7.633795 |  9.704455 |    0.064  0.064266  0.064746  0.065367  0.066039 | 0.064352 |
    | 02/01 03:00 | 10.4 | -20.0 | -20.0 |  0.096652 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   46.94976    70.42464 | 0.902305 | 14.6016 |  0.102451 | 0.017128 | 0.0 | 14.6016 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.126433 |      0.0 | 0.075958 |      0.0 | 0.010777 |  0.213167 |  0.223906 |  0.223906 | 0.062196 | 0.0 |   46.94976    70.42464 |       0.0        0.0 | 0.0 |       0.0 |   2.46597 |  7.557837 |  9.693679 | 0.061892  0.062146  0.062605  0.063198  0.063839 | 0.062196 |
    | 02/01 04:00 |  3.5 | -20.0 | -20.0 |  0.096321 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   50.88096    76.32144 | 0.902305 |   4.914 |    0.1021 | 0.044973 | 0.0 |   4.914 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.120267 |      0.0 | 0.075202 |      0.0 | 0.010765 |  0.206233 |  0.216492 |  0.216492 | 0.060137 | 0.0 |   50.88096    76.32144 |       0.0        0.0 | 0.0 |       0.0 |  2.345703 |  7.482636 |  9.682914 | 0.059878  0.060121  0.060559  0.061125  0.061738 | 0.060137 |
    | 02/01 05:00 |  3.4 | -20.0 | -20.0 |   0.09599 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   54.69984    82.04976 | 0.902305 |  4.7736 |  0.101749 | 0.045452 | 0.0 |  4.7736 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.114401 |      0.0 | 0.074453 |      0.0 | 0.010753 |  0.199608 |   0.20941 |   0.20941 | 0.058169 | 0.0 |   54.69984    82.04976 |       0.0        0.0 | 0.0 |       0.0 |  2.231302 |  7.408182 |  9.672161 | 0.057953  0.058185  0.058604  0.059145  0.059731 | 0.058169 |
    | 02/01 06:00 |  1.2 | -20.0 | -20.0 |  0.187298 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   56.04768    84.07152 | 0.902305 |  1.6848 |  0.198536 | 0.120782 | 0.0 |  1.6848 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.108822 |      0.0 | 0.073713 |      0.0 | 0.010741 |  0.193275 |  0.202643 |  0.202643 |  0.05629 | 0.0 |   56.04768    84.07152 |       0.0        0.0 | 0.0 |       0.0 |   2.12248 |   7.33447 |   9.66142 | 0.056114  0.056336  0.056736  0.057253  0.057813 |  0.05629 |
    | 02/01 07:00 |  0.1 | -20.0 | -20.0 |  1.264612 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |      56.16       84.24 | 0.902305 |  0.1404 |  1.340489 | 0.951696 | 0.0 |  0.1404 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.103515 |      0.0 | 0.072979 |      0.0 | 0.010729 |  0.187223 |  0.196176 |  0.196176 | 0.054493 | 0.0 |      56.16       84.24 |       0.0        0.0 | 0.0 |       0.0 |  2.018965 |   7.26149 |  9.650691 | 0.054356  0.054568   0.05495  0.055445  0.055979 | 0.054493 |
    | 02/01 08:00 |  0.0 | -20.0 | -20.0 |  3.045538 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |      56.16       84.24 | 0.902305 |     0.0 |   3.22827 | 2.324355 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.098466 |      0.0 | 0.072253 |      0.0 | 0.010717 |  0.181436 |  0.189995 |  0.189995 | 0.052776 | 0.0 |      56.16       84.24 |       0.0        0.0 | 0.0 |       0.0 |  1.920499 |  7.189237 |  9.639974 | 0.052675  0.052878  0.053243  0.053716  0.054227 | 0.052776 |
    | 02/01 09:00 |  0.0 | -20.0 | -20.0 |  1.930758 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |      56.16       84.24 | 0.902305 |     0.0 |  2.046603 | 1.473555 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.093664 |      0.0 | 0.071534 |      0.0 | 0.010705 |  0.175903 |  0.184086 |  0.184086 | 0.051135 | 0.0 |      56.16       84.24 |       0.0        0.0 | 0.0 |       0.0 |  1.826835 |  7.117703 |  9.629269 | 0.051068  0.051262  0.051611  0.052063  0.052552 | 0.051135 |
    | 02/01 10:00 |  0.4 | -20.0 | -20.0 |  2.461001 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   56.60928    84.91392 | 0.902305 |  0.5616 |  2.608661 | 1.775661 | 0.0 |  0.5616 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.089096 |      0.0 | 0.070822 |      0.0 | 0.010693 |  0.170611 |  0.178437 |  0.178437 | 0.049566 | 0.0 |   56.60928    84.91392 |       0.0        0.0 | 0.0 |       0.0 |  1.737739 |  7.046881 |  9.618576 | 0.049531  0.049716   0.05005  0.050483   0.05095 | 0.049566 |
    | 02/01 11:00 |  0.1 | -20.0 | -20.0 |  6.215945 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    56.7216     85.0824 | 0.902305 |  0.1404 |  6.588902 | 4.677869 | 0.0 |  0.1404 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.084751 |      0.0 | 0.070118 |      0.0 | 0.010681 |   0.16555 |  0.173035 |  0.173035 | 0.048065 | 0.0 |    56.7216     85.0824 |       0.0        0.0 | 0.0 |       0.0 |  1.652989 |  6.976763 |  9.607894 | 0.048061  0.048238  0.048558  0.048971  0.049418 | 0.048065 |
    | 02/01 12:00 |  3.6 | -20.0 | -20.0 |  3.374783 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   60.76512    91.14768 | 0.902305 |  5.0544 |   3.57727 | 1.553726 | 0.0 |  5.0544 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.080617 |      0.0 |  0.06942 |      0.0 |  0.01067 |  0.160707 |  0.167867 |  0.167867 |  0.04663 | 0.0 |   60.76512    91.14768 |       0.0        0.0 | 0.0 |       0.0 |  1.572372 |  6.907343 |  9.597225 | 0.046654  0.046824  0.047129  0.047525  0.047953 |  0.04663 |
    | 02/01 13:00 |  5.9 | -20.0 | -20.0 |  8.821555 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |     67.392     101.088 | 0.902305 |  8.2836 |  9.350848 | 2.940569 | 0.0 |  8.2836 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.076685 |      0.0 | 0.068729 |      0.0 | 0.010658 |  0.156072 |  0.162924 |  0.162924 | 0.045257 | 0.0 |     67.392     101.088 |       0.0        0.0 | 0.0 |       0.0 |  1.495686 |  6.838614 |  9.586567 | 0.045308   0.04547  0.045763  0.046141  0.046551 | 0.045257 |
    | 02/01 14:00 |  1.1 | -20.0 | -20.0 |  4.046025 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   68.62752   102.94128 | 0.902305 |  1.5444 |  4.288787 | 2.646028 | 0.0 |  1.5444 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.072945 |      0.0 | 0.068045 |      0.0 | 0.010646 |  0.151637 |  0.158194 |  0.158194 | 0.043943 | 0.0 |   68.62752   102.94128 |       0.0        0.0 | 0.0 |       0.0 |  1.422741 |  6.770569 |  9.575921 |  0.04402  0.044175  0.044455  0.044817  0.045209 | 0.043943 |
    | 02/01 15:00 | 20.7 | -20.0 | -20.0 |  2.110757 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   91.87776   137.81664 | 0.902305 | 29.0628 |  2.237402 | 0.088084 | 0.0 | 29.0628 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.069388 |      0.0 | 0.067368 |      0.0 | 0.010634 |   0.14739 |  0.153667 |  0.153667 | 0.042685 | 0.0 |   91.87776   137.81664 |       0.0        0.0 | 0.0 |       0.0 |  1.353353 |    6.7032 |  9.565287 | 0.042786  0.042935  0.043203   0.04355  0.043925 | 0.042685 |
    | 02/01 16:00 | 37.9 | -20.0 | -20.0 |  2.239257 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  134.44704   201.67056 | 0.902305 | 53.2116 |  2.373612 | 0.008352 | 0.0 | 53.2116 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.066004 |      0.0 | 0.066698 |      0.0 | 0.010622 |  0.143324 |  0.149334 |  0.149334 | 0.041482 | 0.0 |  134.44704   201.67056 |       0.0        0.0 | 0.0 |       0.0 |  1.287349 |  6.636503 |  9.554665 | 0.041605  0.041747  0.042004  0.042336  0.042695 | 0.041482 |
    | 02/01 17:00 |  8.2 | -20.0 | -20.0 |  2.877848 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  143.65728   215.48592 | 0.902305 | 11.5128 |  3.050519 | 0.694563 | 0.0 | 11.5128 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.062785 |      0.0 | 0.066034 |      0.0 |  0.01061 |  0.139429 |  0.145185 |  0.145185 | 0.040329 | 0.0 |  143.65728   215.48592 |       0.0        0.0 | 0.0 |       0.0 |  1.224564 |  6.570468 |  9.544055 | 0.040474   0.04061  0.040856  0.041174  0.041518 | 0.040329 |
    | 02/01 18:00 |  3.6 | -20.0 | -20.0 |  1.591452 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   147.7008    221.5512 | 0.902305 |  5.0544 |  1.686939 | 0.732693 | 0.0 |  5.0544 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.059723 |      0.0 | 0.065377 |      0.0 | 0.010599 |  0.135699 |  0.141212 |  0.141212 | 0.039226 | 0.0 |   147.7008    221.5512 |       0.0        0.0 | 0.0 |       0.0 |  1.164842 |  6.505091 |  9.533456 |  0.03939  0.039521  0.039757  0.040061   0.04039 | 0.039226 |
    | 02/01 19:00 |  7.5 | -20.0 | -20.0 |  0.291604 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   156.1248    234.1872 | 0.902305 |   10.53 |    0.3091 | 0.077646 | 0.0 |   10.53 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 |  0.05681 |      0.0 | 0.064727 |      0.0 | 0.010587 |  0.132124 |  0.137406 |  0.137406 | 0.038168 | 0.0 |   156.1248    234.1872 |       0.0        0.0 | 0.0 |       0.0 |  1.108032 |  6.440364 |  9.522869 | 0.038352  0.038477  0.038703  0.038995   0.03931 | 0.038168 |
    | 02/01 20:00 | 18.5 | -20.0 | -20.0 |  0.092622 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    176.904     265.356 | 0.902305 |  25.974 |  0.098179 | 0.005264 | 0.0 |  25.974 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.054039 |      0.0 | 0.064083 |      0.0 | 0.010575 |  0.128697 |  0.133759 |  0.133759 | 0.037155 | 0.0 |    176.904     265.356 |       0.0        0.0 | 0.0 |       0.0 |  1.053992 |  6.376282 |  9.512294 | 0.037357  0.037477  0.037693  0.037973  0.038275 | 0.037155 |
    | 02/01 21:00 | 15.4 | -20.0 | -20.0 |  0.092451 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  194.20128   291.30192 | 0.902305 | 21.6216 |  0.097998 |  0.00812 | 0.0 | 21.6216 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.051404 |      0.0 | 0.063445 |      0.0 | 0.010563 |  0.125412 |  0.130265 |  0.130265 | 0.036185 | 0.0 |  194.20128   291.30192 |       0.0        0.0 | 0.0 |       0.0 |  1.002588 |  6.312836 |  9.501731 | 0.036403  0.036518  0.036725  0.036993  0.037283 | 0.036185 |
    | 02/01 22:00 |  6.3 | -20.0 | -20.0 |  0.091248 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  201.27744   301.91616 | 0.902305 |  8.8452 |  0.096723 | 0.028755 | 0.0 |  8.8452 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.048897 |      0.0 | 0.062814 |      0.0 | 0.010552 |  0.122262 |  0.126915 |  0.126915 | 0.035254 | 0.0 |  201.27744   301.91616 |       0.0        0.0 | 0.0 |       0.0 |  0.953692 |  6.250023 |  9.491179 | 0.035488  0.035598  0.035797  0.036054  0.036332 | 0.035254 |
    | 02/01 23:00 |  1.9 | -20.0 | -20.0 |  0.089683 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  203.41152   305.11728 | 0.902305 |  2.6676 |  0.095064 |  0.05242 | 0.0 |  2.6676 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.046512 |      0.0 | 0.062189 |      0.0 |  0.01054 |  0.119241 |  0.123703 |  0.123703 | 0.034362 | 0.0 |  203.41152   305.11728 |       0.0        0.0 | 0.0 |       0.0 |   0.90718 |  6.187834 |  9.480639 |  0.03461  0.034716  0.034907  0.035153   0.03542 | 0.034362 |
    | 03/01 00:00 |  4.9 |  20.0 |  20.0 |  0.089858 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  208.06848   312.10272 | 0.902686 |  5.8212 |  0.095249 | 0.038316 | 0.0 |  5.8212 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.044244 |      0.0 |  0.06157 |      0.0 | 0.010528 |  0.116342 |  0.120623 |  0.120623 | 0.033506 | 0.0 | 194.565197  296.270957 | 13.503283  15.831763 | 0.0 |       0.0 |  0.862936 |  6.126264 |  9.470111 | 0.033768   0.03387  0.034053   0.03429  0.034545 | 0.033506 |
    | 03/01 01:00 |  2.7 |  20.0 |  20.0 |  0.089683 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  210.63456   315.95184 | 0.902686 |  3.2076 |  0.095064 | 0.049664 | 0.0 |  3.2076 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.042086 |      0.0 | 0.060957 |      0.0 | 0.010517 |   0.11356 |  0.117668 |  0.117668 | 0.032685 | 0.0 | 185.718874  287.424634 | 24.915686  28.527206 | 0.0 |       0.0 |   0.82085 |  6.065307 |  9.459595 | 0.032961  0.033058  0.033234  0.033461  0.033706 | 0.032685 |
    | 03/01 02:00 |  0.5 |  20.0 |  20.0 |  0.088805 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  211.10976   316.66464 | 0.902686 |   0.594 |  0.094133 | 0.063867 | 0.0 |   0.594 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.040033 |      0.0 | 0.060351 |      0.0 | 0.010505 |  0.110889 |  0.114832 |  0.114832 | 0.031898 | 0.0 | 176.872552  278.578312 | 34.237208  38.086328 | 0.0 |       0.0 |  0.780817 |  6.004956 |   9.44909 | 0.032185  0.032278  0.032447  0.032665    0.0329 | 0.031898 |
    | 03/01 03:00 |  0.2 |  20.0 |  20.0 |  0.089157 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  211.29984   316.94976 | 0.902686 |  0.2376 |  0.094506 | 0.066447 | 0.0 |  0.2376 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.038081 |      0.0 |  0.05975 |      0.0 | 0.010493 |  0.108324 |   0.11211 |   0.11211 | 0.031142 | 0.0 | 168.026229  269.731989 | 43.273611  47.217771 | 0.0 |       0.0 |  0.742736 |  5.945205 |  9.438597 |  0.03144   0.03153  0.031692  0.031901  0.032127 | 0.031142 |
    | 03/01 04:00 |  0.5 |  20.0 |  20.0 |  0.090207 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  211.77504   317.66256 | 0.902686 |   0.594 |  0.095619 | 0.064876 | 0.0 |   0.594 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.036224 |      0.0 | 0.059156 |      0.0 | 0.010482 |  0.105861 |  0.109497 |  0.109497 | 0.030416 | 0.0 | 159.179906  260.885666 | 52.595134  56.776894 | 0.0 |       0.0 |  0.706512 |   5.88605 |  9.428115 | 0.030725  0.030811  0.030966  0.031167  0.031384 | 0.030416 |
    | 03/01 05:00 |  2.4 |  20.0 |  20.0 |  0.091593 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 210.467016     321.084 | 0.902686 |  2.8512 |  0.097089 | 0.052562 | 0.0 |  2.8512 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |  1.794492 |  1.794492 | 0.0 | 0.0 | 1.794492 | 0.0 |       0.0 |      0.0 | 1.794492 | 0.078581 |      0.0 | 0.058567 |      0.0 |  0.01047 |  0.147618 |  0.110137 |  0.110137 | 0.030594 | 0.0 | 150.333583  252.039343 | 60.133433  69.044657 | 0.0 |       0.0 |  2.422423 |  5.827483 |  9.417645 | 0.042402  0.041047  0.038777  0.036227  0.034082 | 0.030594 |
    | 03/01 06:00 |  0.4 |  20.0 |  20.0 |  0.154861 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 198.082164   321.65424 | 0.902686 |  0.4752 |  0.164153 | 0.112705 | 0.0 |  0.4752 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |  6.382506 |  6.382506 | 0.0 | 0.0 |      2.0 | 0.0 |       0.0 | 0.417051 |      2.0 |  0.16732 |      0.0 | 0.057984 |      0.0 | 0.010458 |  0.652813 |  0.166041 |  0.166041 | 0.046123 | 0.0 |  141.48726   243.19302 | 56.594904   78.46122 | 0.0 |  3.965455 |  4.255103 |  5.769498 |  9.407187 | 0.184364  0.167814  0.139801  0.107716  0.079611 | 0.046123 |
    | 03/01 07:00 |  0.2 |  20.0 |  20.0 |  0.470369 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 185.697312   321.93936 | 0.902686 |  0.2376 |  0.498591 | 0.350557 | 0.0 |  0.2376 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |  6.287466 |  6.287466 | 0.0 | 0.0 |      2.0 | 0.0 |       0.0 | 0.785369 |      2.0 | 0.256701 |      0.0 | 0.057407 |      0.0 | 0.010447 |  1.109924 |  0.465826 |  0.465826 | 0.129396 | 0.0 | 132.640937  234.346697 | 53.056375  87.592663 | 0.0 |  7.467552 |  5.998402 |  5.712091 |   9.39674 | 0.317243  0.301237  0.272427  0.235596  0.196901 | 0.129396 |
    | 03/01 08:00 |  0.0 |  20.0 |  20.0 |  1.173726 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  173.31246  315.700524 | 0.902686 |     0.0 |   1.24415 | 0.895788 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |  9.311844 |  9.311844 | 0.0 | 0.0 |      2.0 | 0.0 |       0.0 | 1.406445 |      2.0 | 0.341723 |      0.0 | 0.056836 |      0.0 | 0.010435 |  1.815439 |  0.922427 |  0.922427 |  0.25623 | 0.0 | 123.794615  225.500375 | 49.517846   90.20015 | 0.0 |  13.37295 |   7.65668 |  5.655254 |  9.386305 | 0.519425   0.49538  0.453041  0.400691  0.347879 |  0.25623 |
    | 03/01 09:00 |  0.0 |  20.0 |  20.0 |  4.202296 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 160.927608  303.315672 | 0.902686 |     0.0 |  4.454434 | 3.207192 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 |      2.0 | 0.0 |   1.47858 | 2.260854 |      2.0 | 0.422598 |      0.0 | 0.056271 |      0.0 | 0.010423 |  4.228726 |  1.642992 |  1.642992 | 0.456387 | 0.0 | 114.948292  216.654052 | 45.979317  86.661621 | 0.0 | 20.018369 |  9.234082 |  5.598984 |  9.375882 | 1.202505  1.122278  0.984464  0.821885  0.671018 | 0.456387 |
    | 03/01 10:00 |  0.3 |  20.0 |  20.0 |  4.359715 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 148.542756   290.93082 | 0.902686 |  0.3564 |  4.621298 | 3.210837 | 0.0 |  0.3564 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.741252 | 12.741252 | 0.0 | 0.0 |      2.0 | 0.0 |  4.233581 | 2.927165 |      2.0 | 0.499528 |      0.0 | 0.055711 |      0.0 | 0.010412 |  7.726397 |  3.382386 |  3.382386 | 0.939552 | 0.0 | 106.101969  207.807729 | 42.440788  83.123092 | 0.0 | 23.598875 | 10.734554 |  5.543273 |   9.36547 | 2.205666  2.086331  1.876345  1.617705  1.360114 | 0.939552 |
    | 03/01 11:00 |  2.6 |  20.0 |  20.0 |  5.305753 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 136.157904  278.545968 | 0.902686 |  3.0888 |  5.624098 | 2.973312 | 0.0 |  3.0888 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 15.473652 | 15.473652 | 0.0 | 0.0 |      2.0 | 0.0 |  6.717516 | 3.527917 | 1.963272 | 0.571804 | 0.032647 | 0.055319 | 0.004081 | 0.010403 | 10.882959 |  6.247821 |  6.247821 | 1.735506 | 0.0 |  97.255646  198.961406 | 38.902258  79.584562 | 0.0 | 26.827094 | 12.126022 |  5.520601 |  9.359148 | 3.124027  3.013233  2.813017  2.554444  2.276578 | 1.735506 |
    | 03/01 12:00 |  0.7 |  20.0 |  20.0 |  5.376027 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 123.773052  266.161116 | 0.902686 |  0.8316 |  5.698589 | 3.775582 | 0.0 |  0.8316 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 13.216452 | 13.216452 | 0.0 | 0.0 |      2.0 | 0.0 |  7.099582 | 3.620322 | 1.893699 | 0.637956 |  0.09449 | 0.055402 | 0.011811 |   0.0104 | 11.423662 |  9.257018 |  9.257018 | 2.571394 | 0.0 |  88.409323  190.115083 | 35.363729  76.046033 | 0.0 | 27.323642 | 13.381764 |  5.559689 |   9.36056 | 3.306498  3.281549  3.226642  3.134009  2.999245 | 2.571394 |
    | 03/01 13:00 |  0.3 |  20.0 |  20.0 |  4.658915 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 111.388201  253.776265 | 0.902686 |  0.3564 |   4.93845 | 3.431191 | 0.0 |  0.3564 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.741252 | 12.741252 | 0.0 | 0.0 |      2.0 | 0.0 |  7.107982 | 3.622354 | 1.830912 | 0.697656 | 0.150301 | 0.056069 | 0.018788 | 0.010405 | 11.494465 | 10.899812 | 10.899812 | 3.027726 | 0.0 |     79.563   181.26876 |   31.8252  72.507504 | 0.0 | 27.334558 | 14.515021 |   5.65392 |  9.368942 | 3.332488  3.328626  3.318886  3.299112  3.263484 | 3.027726 |
    | 03/01 14:00 |  0.3 |  20.0 |  20.0 |  7.789594 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  99.003349  241.391413 | 0.902686 |  0.3564 |   8.25697 | 5.736869 | 0.0 |  0.3564 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.741252 | 12.741252 | 0.0 | 0.0 |      2.0 | 0.0 |  7.112277 | 3.623392 | 1.774249 | 0.751532 | 0.200668 | 0.057257 | 0.025083 | 0.010418 | 11.554877 | 11.389911 | 11.389911 | 3.163864 | 0.0 |  70.716678  172.422438 | 28.286671  68.968975 | 0.0 | 27.340141 | 15.537737 |  5.797331 |  9.383607 | 3.350293  3.348093  3.343846  3.337406  3.327924 | 3.163864 |
    | 03/01 15:00 |  0.0 |  20.0 |  20.0 |  4.851567 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  86.618497  229.006561 | 0.902686 |     0.0 |  5.142661 | 3.702716 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 |      2.0 | 0.0 |  6.974241 | 3.590008 | 1.723113 | 0.800153 | 0.246122 | 0.058911 | 0.030765 | 0.010438 |  11.43375 | 11.506344 | 11.506344 | 3.196207 | 0.0 |  61.870355  163.576115 | 24.748142  65.430446 | 0.0 | 27.160744 | 16.460697 |  5.984541 |  9.403935 | 3.316951  3.320753  3.326888  3.333173  3.337203 | 3.196207 |
    | 03/01 16:00 |  0.0 |  20.0 |  20.0 |   5.30692 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  74.233645  216.621709 | 0.902686 |     0.0 |  5.625335 | 4.050241 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 |      2.0 | 0.0 |  6.903654 | 3.572936 | 1.676965 | 0.844032 | 0.287142 | 0.060978 | 0.035893 | 0.010463 | 11.392063 | 11.476242 | 11.476242 | 3.187845 | 0.0 |  53.024032  154.729792 | 21.209613  61.891917 | 0.0 | 27.069006 | 17.293631 |  6.210705 |  9.429365 | 3.304145  3.305762  3.308902  3.313363  3.318618 | 3.187845 |
    | 03/01 17:00 |  0.0 |  20.0 |  20.0 |  3.286036 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  61.848793  204.236857 | 0.902686 |     0.0 |  3.483198 | 2.507903 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 |      2.0 | 0.0 |  6.867558 | 3.564206 | 1.635318 |  0.88363 | 0.324161 | 0.063413 |  0.04052 | 0.010494 | 11.389301 | 11.419044 | 11.419044 | 3.171957 | 0.0 |  44.177709  145.883469 | 17.671084  58.353388 | 0.0 | 27.022094 | 18.045319 |  6.471454 |  9.459391 | 3.302939  3.303134  3.303657  3.304734  3.306582 | 3.171957 |
    | 03/01 18:00 |  0.0 |  20.0 |  20.0 |  1.506216 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  49.463941  191.852005 | 0.902686 |     0.0 |  1.596589 | 1.149544 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 |      2.0 | 0.0 |    6.8491 | 3.559742 | 1.597734 | 0.919366 |  0.35757 | 0.066174 | 0.044696 | 0.010529 | 11.404911 | 11.396167 | 11.396167 | 3.165602 | 0.0 |  35.331386  137.037146 | 14.132554  54.814858 | 0.0 | 26.998105 | 18.723687 |  6.762849 |  9.493558 | 3.307273  3.306776  3.305967  3.305138  3.304637 | 3.165602 |
    | 03/01 19:00 |  0.0 |  20.0 |  20.0 |  0.274762 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  37.079089  179.467153 | 0.902686 |     0.0 |  0.291248 | 0.209698 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 |      2.0 | 0.0 |  6.839661 | 3.557459 | 1.563816 | 0.951617 | 0.387719 | 0.069224 | 0.048465 | 0.010569 | 11.428529 |  11.40065 |  11.40065 | 3.166847 | 0.0 |  26.485063  128.190823 | 10.594025  51.276329 | 0.0 | 26.985838 | 19.335885 |  7.081345 |  9.531454 | 3.314038  3.313234  3.311827  3.310114  3.308457 | 3.166847 |
    | 03/01 20:00 |  0.0 |  20.0 |  20.0 |  0.087565 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  24.694237  167.082301 | 0.902686 |     0.0 |  0.092819 |  0.06683 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 |      2.0 | 0.0 |  6.834834 | 3.556291 | 1.533206 | 0.980721 | 0.414928 | 0.072528 | 0.051866 | 0.010613 | 11.454988 | 11.419088 | 11.419088 | 3.171969 | 0.0 |  17.638741  119.344501 |  7.055496    47.7378 | 0.0 | 26.979564 |  19.88837 |  7.423745 |  9.572706 |  3.32168  3.320764  3.319131  3.317068  3.314927 | 3.171969 |
    | 03/01 21:00 |  0.0 |  20.0 |  20.0 |  0.085771 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  12.309385  154.697449 | 0.902686 |     0.0 |  0.090917 |  0.06546 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 |      2.0 | 0.0 |  6.832365 | 3.555694 | 1.505582 | 1.006987 | 0.439483 | 0.076058 | 0.054935 | 0.010661 | 11.481766 | 11.443475 | 11.443475 | 3.178743 | 0.0 |   8.792418  110.498178 |  3.516967  44.199271 | 0.0 | 26.976357 | 20.386964 |  7.787171 |  9.616981 | 3.329441  3.328508  3.326831  3.324684  3.322397 | 3.178743 |
    | 03/01 22:00 |  0.0 |  20.0 |  20.0 |  0.084317 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |        0.0  142.312597 | 0.902686 |     0.0 |  0.089376 | 0.064351 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.846323 | 8.792418  8.846323 | 0.0   0.0 | 21.193441 | 21.193441 | 0.0 | 0.0 |      2.0 | 0.0 | 10.297013 | 4.393637 | 1.480652 | 1.030691 | 0.461643 | 0.079784 | 0.057705 | 0.010712 | 15.811837 | 11.776953 | 11.776953 | 3.271376 | 0.0 |        0.0  101.651855 |       0.0  40.660742 | 0.0 | 31.479148 | 20.836925 |  8.169029 |  9.663974 | 4.543102  4.401966  4.164279  3.894647  3.662752 | 3.271376 |
    | 03/01 23:00 |  0.0 |  20.0 |  20.0 |  0.083215 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |        0.0  129.927745 | 0.902686 |     0.0 |  0.088208 |  0.06351 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.846323 |      0.0  8.846323 | 0.0   0.0 | 15.038749 | 15.038749 | 0.0 | 0.0 |      2.0 | 0.0 |  9.647041 | 4.236438 | 1.458154 | 1.052083 | 0.481641 | 0.083683 | 0.060205 | 0.010765 |  15.03001 | 13.900614 | 13.900614 | 3.861282 | 0.0 |        0.0   92.805532 |       0.0  37.122213 | 0.0 | 30.634418 | 21.242996 |  8.566987 |  9.713414 | 4.364918  4.380877  4.391883  4.368924   4.28954 | 3.861282 |
    | 04/01 00:00 |  0.0 |  20.0 |  20.0 |  0.082289 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0  117.537274 | 0.903096 |     0.0 |  0.087226 | 0.062803 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 |      2.0 | 0.0 |   9.31735 | 4.156701 |  1.43785 | 1.071388 | 0.499689 | 0.087733 | 0.062461 | 0.010821 | 14.643993 | 14.978058 | 14.978058 | 4.160572 | 0.0 |        0.0   83.955196 |       0.0  33.582078 | 0.0 | 30.205939 | 21.609458 |  8.978943 |  9.765054 |  4.25074  4.264553  4.289062  4.317868  4.339853 | 4.160572 |
    | 04/01 01:00 |  0.0 |  20.0 |  20.0 |    0.0845 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0  105.146803 | 0.903096 |     0.0 |   0.08957 |  0.06449 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 |      2.0 | 0.0 |  9.148757 | 4.115925 | 1.419527 |  1.08881 | 0.515976 | 0.091913 | 0.064497 |  0.01088 | 14.456285 |  14.81848 |  14.81848 | 4.116244 | 0.0 |        0.0    75.10486 |       0.0  30.041944 | 0.0 | 29.986829 | 21.940176 |  9.403005 |  9.818671 | 4.194291   4.20132  4.214745  4.233602  4.255922 | 4.116244 |
    | 04/01 02:00 |  0.0 |  20.0 |  20.0 |  0.084864 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   92.756333 | 0.903096 |     0.0 |  0.089956 | 0.064768 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 |      2.0 | 0.0 |  9.062543 | 4.095074 | 1.402991 | 1.104532 | 0.530674 | 0.096206 | 0.066334 |  0.01094 | 14.369297 | 14.566268 | 14.566268 | 4.046186 | 0.0 |        0.0   66.254523 |       0.0  26.501809 | 0.0 | 29.874783 | 22.238635 |  9.837474 |  9.874065 | 4.168013  4.171305   4.17769  4.186992  4.198911 | 4.046186 |
    | 04/01 03:00 |  1.3 |  20.0 |  20.0 |  0.083584 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   80.365862 | 0.903096 |  1.5444 |  0.088599 | 0.054662 | 0.0 |  1.5444 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 16.589972 | 16.589972 | 0.0 | 0.0 |      2.0 | 0.0 |  9.626131 | 4.231381 | 1.388068 | 1.118722 | 0.543939 | 0.100595 | 0.067992 | 0.011003 | 15.087831 | 14.477501 | 14.477501 | 4.021528 | 0.0 |        0.0   57.404187 |       0.0  22.961675 | 0.0 | 30.607243 | 22.507981 | 10.280818 |  9.931055 | 4.368479  4.345283  4.306635  4.263824  4.229019 | 4.021528 |
    | 04/01 04:00 |  0.0 |  20.0 |  20.0 |    0.0834 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   67.975391 | 0.903096 |     0.0 |  0.088404 | 0.063651 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 |      2.0 | 0.0 |  9.306657 | 4.154115 | 1.374601 | 1.131527 |  0.55591 | 0.105066 | 0.069489 | 0.011067 | 14.708431 | 14.762663 | 14.762663 |  4.10074 | 0.0 |        0.0   48.553851 |       0.0   19.42154 | 0.0 | 30.192043 | 22.751056 | 10.731662 |  9.989476 | 4.268918  4.279711  4.295288    4.3072  4.307891 |  4.10074 |
    | 04/01 05:00 |  0.0 |  20.0 |  20.0 |  0.084864 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0    55.58492 | 0.903096 |     0.0 |  0.089956 | 0.064768 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 |      2.0 | 0.0 |  9.143289 | 4.114603 | 1.362447 | 1.143083 | 0.566714 | 0.109606 | 0.070839 | 0.011133 | 14.521713 | 14.798968 | 14.798968 | 4.110825 | 0.0 |        0.0   39.703515 |       0.0  15.881406 | 0.0 | 29.979722 |  22.97042 |  11.18877 | 10.049183 | 4.213239  4.220076   4.23271  4.249253  4.266474 | 4.110825 |
    | 04/01 06:00 |  0.0 |  20.0 |  20.0 |  0.310229 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0    43.19445 | 0.903096 |     0.0 |  0.328843 | 0.236767 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 |      2.0 | 0.0 |  9.059747 | 4.094398 | 1.351479 | 1.153511 | 0.576463 | 0.114203 | 0.072058 |   0.0112 | 14.433059 | 14.619824 | 14.619824 | 4.061062 | 0.0 |        0.0   30.853178 |       0.0  12.341271 | 0.0 | 29.871149 | 23.168388 |  11.65103 | 10.110041 | 4.186519  4.189856  4.196279  4.205476  4.216856 | 4.061062 |
    | 04/01 07:00 |  0.7 |  20.0 |  20.0 |  1.391958 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   30.803979 | 0.903096 |  0.8316 |  1.475475 | 0.977572 | 0.0 |  0.8316 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.877172 | 15.877172 | 0.0 | 0.0 |      2.0 | 0.0 |  9.344236 | 4.163203 | 1.341581 | 1.162923 | 0.585262 | 0.118846 | 0.073158 | 0.011268 | 14.800476 | 14.515765 | 14.515765 | 4.032157 | 0.0 |        0.0   22.002842 |       0.0   8.801137 | 0.0 | 30.240881 | 23.347045 | 12.117445 | 10.171931 | 4.288578  4.276825  4.257442  4.236468  4.220384 | 4.032157 |
    | 04/01 08:00 |  0.4 |  20.0 |  20.0 |  3.195876 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   18.413508 | 0.903096 |  0.4752 |  3.387629 | 2.325898 | 0.0 |  0.4752 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.520772 | 15.520772 | 0.0 | 0.0 |      2.0 | 0.0 |  9.349482 | 4.164472 | 1.332648 | 1.171417 | 0.593202 | 0.123527 |  0.07415 | 0.011337 | 14.820234 | 14.654097 | 14.654097 | 4.070582 | 0.0 |        0.0   13.152506 |       0.0   5.261002 | 0.0 | 30.247699 | 23.508276 | 12.587121 | 10.234745 | 4.297555  4.296115  4.292385  4.285286  4.274494 | 4.070582 |
    | 04/01 09:00 |  0.1 |  20.0 |  20.0 |  5.191651 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0    6.023037 | 0.903096 |  0.1188 |   5.50315 | 3.915475 | 0.0 |  0.1188 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.164372 | 15.164372 | 0.0 | 0.0 |      2.0 | 0.0 |  9.211932 | 4.131205 | 1.324586 | 1.179082 | 0.600368 | 0.128236 | 0.075046 | 0.011407 | 14.661862 | 14.771081 | 14.771081 | 4.103078 | 0.0 |        0.0     4.30217 |       0.0   1.720868 | 0.0 | 30.068934 | 23.653781 | 13.059253 | 10.298383 | 4.253477  4.258553  4.266892  4.275738  4.281955 | 4.103078 |
    | 04/01 10:00 |  0.4 |  20.0 |  20.0 |  7.155036 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |  0.4752 |  7.584338 | 5.207299 | 0.0 |  0.4752 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0   4.30217 | 0.0   0.0 | 21.187391 | 21.187391 | 0.0 | 0.0 |      2.0 | 0.0 | 11.511467 | 4.687359 | 1.317311 | 1.185999 | 0.606835 | 0.132966 | 0.075854 | 0.011478 | 17.529269 | 14.929193 | 14.929193 | 4.146998 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 33.057499 | 23.785093 | 13.533122 | 10.362759 | 5.055514  4.962438  4.806322  4.630622  4.481797 | 4.146998 |
    | 04/01 11:00 |  0.0 |  20.0 |  20.0 |  8.391432 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |     0.0 |  8.894918 | 6.404341 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 |      2.0 | 0.0 | 11.315459 | 4.639954 | 1.310745 | 1.192242 | 0.612671 |  0.13771 | 0.076584 |  0.01155 | 17.296915 | 16.298285 | 16.298285 | 4.527301 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 32.802759 | 23.903596 | 14.008082 | 10.427793 | 5.017433  5.018724  5.010458  4.977842  4.910865 | 4.527301 |
    | 04/01 12:00 |  0.0 |  20.0 |  20.0 |  8.391286 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |     0.0 |  8.894763 | 6.404229 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 |      2.0 | 0.0 | 11.215227 | 4.615712 |  1.30482 | 1.197876 | 0.617938 | 0.142462 | 0.077242 | 0.011623 | 17.182899 | 17.152187 | 17.152187 | 4.764496 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 32.672493 | 24.010541 | 14.483558 | 10.493412 |   4.9842  4.988107  4.994478  5.000065  4.999184 | 4.764496 |
    | 04/01 13:00 |  0.0 |  20.0 |  20.0 | 10.715238 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |     0.0 | 11.358152 |  8.17787 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 |      2.0 | 0.0 | 11.163971 | 4.603315 | 1.299473 |  1.20296 | 0.622691 | 0.147217 | 0.077836 | 0.011696 | 17.129159 | 17.219396 | 17.219396 | 4.783165 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |  32.60588 | 24.107054 | 14.959032 | 10.559552 | 4.968021  4.970033  4.973853  4.979085  4.984806 | 4.783165 |
    | 04/01 14:00 |  0.0 |  20.0 |  20.0 |  9.383394 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |     0.0 |  9.946398 | 7.161406 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 |      2.0 | 0.0 |  11.13776 | 4.596976 | 1.294647 | 1.207548 |  0.62698 | 0.151969 | 0.078373 |  0.01177 | 17.106024 | 17.159767 | 17.159767 | 4.766602 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 32.571816 | 24.194153 | 15.434042 | 10.626155 | 4.960992  4.961877  4.963608  4.966153  4.969425 | 4.766602 |
    | 04/01 15:00 |  0.0 |  20.0 |  20.0 |  7.861915 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |     0.0 |   8.33363 | 6.000214 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 |      2.0 | 0.0 | 11.124357 | 4.593735 | 1.290292 | 1.211689 | 0.630851 | 0.156715 | 0.078856 | 0.011844 |  17.09834 | 17.120954 | 17.120954 |  4.75582 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 32.554396 | 24.272756 | 15.908179 | 10.693167 | 4.958602   4.95891  4.959535  4.960513  4.961881 |  4.75582 |
    | 04/01 16:00 |  0.0 |  20.0 |  20.0 |  6.298329 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |     0.0 |  6.676229 | 4.806885 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 |      2.0 | 0.0 | 11.117503 | 4.592077 | 1.286362 | 1.215426 | 0.634345 |  0.16145 | 0.079293 | 0.011919 | 17.098375 | 17.104169 | 17.104169 | 4.751158 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 32.545489 | 24.343692 | 16.381073 | 10.760542 | 4.958531   4.95855  4.958619  4.958796   4.95915 | 4.751158 |
    | 04/01 17:00 |  0.0 |  20.0 |  20.0 |  2.948416 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |     0.0 |  3.125321 | 2.250231 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 |      2.0 | 0.0 | 11.113998 | 4.591229 | 1.282815 | 1.218798 | 0.637497 | 0.166171 | 0.079687 | 0.011994 | 17.102191 | 17.099547 | 17.099547 | 4.749874 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 32.540933 | 24.407709 | 16.852399 | 10.828235 | 4.959598  4.959475   4.95927  4.959051  4.958897 | 4.749874 |
    | 04/01 18:00 |  0.0 |  20.0 |  20.0 |  1.309232 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |     0.0 |  1.387786 | 0.999206 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 |      2.0 | 0.0 | 11.112206 | 4.590796 | 1.279615 | 1.221842 | 0.640343 | 0.170875 | 0.080043 | 0.012069 | 17.107788 | 17.101053 | 17.101053 | 4.750292 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 32.538604 | 24.465482 | 17.321867 | 10.896209 | 4.961203  4.961012  4.960677  4.960268  4.959867 | 4.750292 |
    | 04/01 19:00 |  0.0 |  20.0 |  20.0 |   0.32955 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |     0.0 |  0.349323 | 0.251513 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 |      2.0 | 0.0 |  11.11129 | 4.590574 | 1.276726 | 1.224588 |  0.64291 | 0.175559 | 0.080364 | 0.012145 | 17.114156 | 17.105539 | 17.105539 | 4.751539 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 32.537413 | 24.517619 | 17.789218 | 10.964428 | 4.963041  4.962821  4.962428  4.961933  4.961419 | 4.751539 |
    | 04/01 20:00 |  0.0 |  20.0 |  20.0 |  0.089508 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |     0.0 |  0.094878 | 0.068313 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 |      2.0 | 0.0 | 11.110821 | 4.590461 | 1.274119 | 1.227067 | 0.645228 | 0.180221 | 0.080653 | 0.012221 |  17.12079 | 17.111405 | 17.111405 | 4.753168 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 32.536804 | 24.564671 | 18.254224 |  11.03286 | 4.964962  4.964731  4.964318  4.963789  4.963229 | 4.753168 |
    | 04/01 21:00 |  0.0 |  20.0 |  20.0 |  0.085771 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |     0.0 |  0.090917 |  0.06546 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 |      2.0 | 0.0 | 11.110581 | 4.590403 | 1.271766 | 1.229304 | 0.647319 | 0.184858 | 0.080915 | 0.012297 | 17.127443 | 17.117841 | 17.117841 | 4.754956 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 32.536492 | 24.607134 | 18.716684 | 11.101478 | 4.966891  4.966659  4.966242  4.965706  4.965133 | 4.754956 |
    | 04/01 22:00 |  0.0 |  20.0 |  20.0 |    0.0845 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |     0.0 |   0.08957 |  0.06449 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 |      2.0 | 0.0 | 11.110459 | 4.590373 | 1.269643 | 1.231323 | 0.649206 | 0.189469 | 0.081151 | 0.012373 | 17.133997 | 17.124448 | 17.124448 | 4.756791 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 32.536333 | 24.645454 | 19.176421 | 11.170256 | 4.968793  4.968564  4.968152  4.967621  4.967051 | 4.756791 |
    | 04/01 23:00 |  0.0 |  20.0 |  20.0 |  0.084864 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |     0.0 |  0.089956 | 0.064768 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 |      2.0 | 0.0 | 11.110396 | 4.590358 | 1.267727 | 1.233144 | 0.650909 | 0.194052 | 0.081364 |  0.01245 |   17.1404 | 17.131028 | 17.131028 | 4.758619 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 32.536252 | 24.680037 | 19.633278 |  11.23917 | 4.970651  4.970428  4.970024  4.969504  4.968945 | 4.758619 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_v3_multiple_zones:

multiple zones
______________

This example focuses on the spatial configuration of |hland_v3|.  The most marked
difference to |hland_v1| is the separate modelling of the faster runoff components
(|RS|, |RI|, |RG1|) for each zone.  Hence, there are five distinct values for each
runoff component in the following table.  In contrast, the table of the corresponding
:ref:`hland_v1_multiple_zones` example of |hland_v1| contains only a single value for
the only fast runoff component |hland_fluxes.Q0| of |hland_v1|:

.. integration-test::

    >>> ttice(0.0)
    >>> name2value = {par.name: par.value for par in control}
    >>> nmbzones(5)
    >>> sclass(1)
    >>> area(15.0)
    >>> zonearea(5.0, 4.0, 3.0, 2.0, 1.0)
    >>> zonetype(FIELD, FOREST, GLACIER, ILAKE, SEALED)
    >>> sfdist(1.0)
    >>> for name, value in name2value.items():
    ...     if name not in ("nmbzones", "sclass", "area", "zonearea", "zonetype", "sfdist"):
    ...         control[name].value = value
    >>> inputs.t.series = t_series
    >>> inputs.tn.series = tn_series
    >>> test("hland_v3_multiple_zones",
    ...      axis1=(inputs.p, fluxes.rt, fluxes.qt),
    ...      axis2=inputs.t)
    |        date |    p |    t |   tn |       epn | tmean |                           tc |                     fracrain |                     rfc |                     sfc |                                       cfact |                     swe |                          gact |                                          pc |                                                    ep |                                              epc |                                     ei |                                                tf |                     spl |                     wcl |                     spg |                     wcg |                           glmelt |                     melt |                     refr |                                                 in_ |                                                   r |                            sr |                                ea |                                dp |                           el |                                      rs |                                     ri |                                    gr1 |                                    rg1 |      gr2 |      rg2 |      gr3 |      rg3 |      inuh |     outuh |        rt |        qt |                                     ic |                      sp |                      wc |                                    sm |                                       suz |                                       sg1 |       sg2 |       sg3 |                                               sc |    outlet |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | 21.2 | 20.2 |  0.100707 |  21.8 | 21.8  21.8  21.8  21.8  21.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |   0.11682    0.11682    0.11682    0.11682    0.11682 |  0.08411   0.08411   0.08411   0.08411   0.08411 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  18.760624  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  18.760624      0.0        0.0 |       0.0        0.0  18.760624      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.052569  0.052569  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0   0.08411  0.0 |      0.0       0.0        0.0  0.0  0.0 |      0.0       0.0  1.594984  0.0  0.0 |      0.0       0.0       2.0  0.0  0.0 | 0.487706  0.487706  0.536883  0.0  0.0 |      0.0 | 0.099502 |      0.0 | 0.011105 |   0.82223 |  0.218798 |  0.218798 |  0.911656 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.947431   99.947431  0.0  0.0  0.0 |       0.0        0.0   15.16564  0.0  0.0 |  9.512294   9.512294  11.463117  0.0  0.0 |  9.889818 |   9.98756 | 0.232095  0.210924  0.175286  0.134897  0.100229 |  0.911656 |
    | 01/01 01:00 |  0.0 | 19.4 | 18.4 |  0.097801 |  20.0 | 20.0  20.0  20.0  20.0  20.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  0.113449   0.113449   0.113449   0.113449   0.113449 | 0.081683  0.081683  0.081683  0.081683  0.081683 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  17.137109  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  17.137109      0.0        0.0 |       0.0        0.0  17.137109      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.051025  0.051025  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.081683  0.0 |      0.0       0.0   4.053816  0.0  0.0 |      0.0       0.0  2.883688  0.0  0.0 |      0.0       0.0  1.926844  0.0  0.0 |  0.46392   0.46392  0.606441  0.0  0.0 | 0.013934 | 0.098475 | 0.001742 | 0.011092 |  1.889403 |  0.620336 |  0.620336 |  2.584732 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.896406   99.896406  0.0  0.0  0.0 |       0.0        0.0  23.365245  0.0  0.0 |  9.048374   9.048374   12.78352  0.0  0.0 |  9.794905 |  9.976913 | 0.537282  0.501087  0.437779  0.360718  0.285633 |  2.584732 |
    | 01/01 02:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5  19.5  19.5  19.5  19.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  0.112498   0.112498   0.112498   0.112498   0.112498 | 0.080999  0.080999  0.080999  0.080999  0.080999 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.686132  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.686132      0.0        0.0 |       0.0        0.0  16.686132      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.050572  0.050572  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.080999  0.0 |      0.0       0.0   7.102663  0.0  0.0 |      0.0       0.0  3.621067  0.0  0.0 |      0.0       0.0  1.860824  0.0  0.0 | 0.441294  0.441294  0.669214  0.0  0.0 |  0.02651 | 0.097593 | 0.003314 | 0.011081 |  2.644795 |  1.432833 |  1.432833 |  5.970136 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.845834   99.845834  0.0  0.0  0.0 |       0.0        0.0  27.327646  0.0  0.0 |   8.60708    8.60708   13.97513  0.0  0.0 |  9.713536 |   9.96786 | 0.759249  0.732222  0.682581  0.616793  0.543617 |  5.970136 |
    | 01/01 03:00 |  0.0 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9  18.9  18.9  18.9  18.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  0.111348   0.111348   0.111348   0.111348   0.111348 | 0.080171  0.080171  0.080171  0.080171  0.080171 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   16.14496  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   16.14496      0.0        0.0 |       0.0        0.0   16.14496      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 |  0.05003   0.05003  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.080171  0.0 |      0.0       0.0   8.448812  0.0  0.0 |      0.0       0.0   3.94664  0.0  0.0 |      0.0       0.0  1.801244  0.0  0.0 | 0.419772  0.419772  0.725865  0.0  0.0 | 0.037858 |  0.09684 | 0.004732 | 0.011072 |  2.976845 |  2.250545 |  2.250545 |  9.377269 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.795804   99.795804  0.0  0.0  0.0 |       0.0        0.0  29.077154  0.0  0.0 |  8.187308   8.187308  15.050508  0.0  0.0 |  9.644374 |  9.960248 | 0.859779   0.84718  0.822796  0.787592  0.743416 |  9.377269 |
    | 01/01 04:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5  19.5  19.5  19.5  19.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  0.112498   0.112498   0.112498   0.112498   0.112498 | 0.080999  0.080999  0.080999  0.080999  0.080999 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.686132  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.686132      0.0        0.0 |       0.0        0.0  16.686132      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.050521  0.050521  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.080999  0.0 |      0.0       0.0   9.350124  0.0  0.0 |      0.0       0.0  4.164627  0.0  0.0 |      0.0       0.0  1.747475  0.0  0.0 |   0.3993    0.3993   0.77699  0.0  0.0 |   0.0481 | 0.096203 | 0.006013 | 0.011064 |  3.198044 |  2.777058 |  2.777058 | 11.571074 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.745284   99.745284  0.0  0.0  0.0 |       0.0        0.0  30.248534  0.0  0.0 |  7.788008   7.788008  16.020993  0.0  0.0 |  9.585986 |  9.953911 | 0.925153  0.917127  0.902096  0.881287  0.856085 | 11.571074 |
    | 01/01 05:00 |  0.0 | 22.5 | 21.5 |  0.102761 |  23.1 | 23.1  23.1  23.1  23.1  23.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  0.119203   0.119203   0.119203   0.119203   0.119203 | 0.085826  0.085826  0.085826  0.085826  0.085826 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  19.933163  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  19.933163      0.0        0.0 |       0.0        0.0  19.933163      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.053505  0.053505  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.085826  0.0 |      0.0       0.0  11.088634  0.0  0.0 |      0.0       0.0  4.585095  0.0  0.0 |      0.0       0.0   1.69895  0.0  0.0 | 0.379826  0.379826  0.823128  0.0  0.0 | 0.057343 | 0.095668 | 0.007168 | 0.011058 |  3.626877 |   3.09821 |   3.09821 | 12.909207 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.691779   99.691779  0.0  0.0  0.0 |       0.0        0.0  32.507969  0.0  0.0 |  7.408182   7.408182  16.896816  0.0  0.0 |  9.536762 |  9.948658 | 1.047526  1.033028   1.00767  0.976619  0.945572 | 12.909207 |
    | 01/01 06:00 |  0.0 | 25.1 | 24.1 |  0.291908 |  25.7 | 25.7  25.7  25.7  25.7  25.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  0.338613   0.338613   0.338613   0.338613   0.338613 | 0.243802  0.243802  0.243802  0.243802  0.243802 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  22.278241  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  22.278241      0.0        0.0 |       0.0        0.0  22.278241      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.151906  0.151906  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.243802  0.0 |      0.0       0.0  12.900368  0.0  0.0 |      0.0       0.0  5.023272  0.0  0.0 |      0.0       0.0  1.655159  0.0  0.0 | 0.361301  0.361301  0.864765  0.0  0.0 | 0.065684 |  0.09522 |  0.00821 | 0.011052 |  4.073649 |  3.453002 |  3.453002 | 14.387508 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.539873   99.539873  0.0  0.0  0.0 |       0.0        0.0   34.86257  0.0  0.0 |  7.046881   7.046881   17.68721  0.0  0.0 |  9.476267 |  9.941946 | 1.176848  1.161324  1.133531   1.09821  1.061151 | 14.387508 |
    | 01/01 07:00 |  0.0 | 28.3 | 27.3 |  1.932875 |  28.9 | 28.9  28.9  28.9  28.9  28.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  2.242135   2.242135   2.242135   2.242135   2.242135 | 1.614337  1.614337  1.614337  1.614337  1.614337 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  25.164491  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  25.164491      0.0        0.0 |       0.0        0.0  25.164491      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.004318  1.004318  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  1.614337  0.0 |      0.0       0.0  14.962482  0.0  0.0 |      0.0       0.0  5.522005  0.0  0.0 |      0.0       0.0   1.61564  0.0  0.0 |  0.34368   0.34368  0.902342  0.0  0.0 | 0.073212 | 0.094655 | 0.009151 | 0.011046 |  4.582228 |  3.880765 |  3.880765 | 16.169856 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  98.535555   98.535555  0.0  0.0  0.0 |       0.0        0.0  37.542573  0.0  0.0 |    6.7032     6.7032  18.400508  0.0  0.0 |  9.249828 |  9.914428 | 1.323723  1.306124  1.274698  1.234872  1.193109 | 16.169856 |
    | 01/01 08:00 |  0.0 | 27.8 | 26.8 |  4.369536 |  28.4 | 28.4  28.4  28.4  28.4  28.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  5.068662   5.068662   5.068662   5.068662   5.068662 | 3.649436  3.649436  3.649436  3.649436  3.649436 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  24.713515  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  24.713515      0.0        0.0 |       0.0        0.0  24.713515      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 2.247495  2.247495  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  3.649436  0.0 |      0.0       0.0  15.839536  0.0  0.0 |      0.0       0.0  5.734125  0.0  0.0 |      0.0       0.0  1.579975  0.0  0.0 | 0.326919  0.326919  0.936253  0.0  0.0 | 0.080005 | 0.092436 | 0.010001 | 0.011015 |  4.794689 |   4.33838 |   4.33838 | 18.076582 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  96.288059   96.288059  0.0  0.0  0.0 |       0.0        0.0  38.682427  0.0  0.0 |  6.376282   6.376282   19.04423  0.0  0.0 |  8.773977 |  9.855486 | 1.388211   1.38012  1.364463  1.341958  1.314084 | 18.076582 |
    | 01/01 09:00 |  0.0 | 31.4 | 30.4 |  7.317556 |  32.0 | 32.0  32.0  32.0  32.0  32.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  8.488365   8.488365   8.488365   8.488365   8.488365 | 6.111623  6.111623  6.111623  6.111623  6.111623 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  27.960546  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  27.960546      0.0        0.0 |       0.0        0.0  27.960546      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.677977  3.677977  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  6.111623  0.0 |      0.0       0.0  17.565641  0.0  0.0 |      0.0       0.0  6.151592  0.0  0.0 |      0.0       0.0  1.547789  0.0  0.0 | 0.310975  0.310975  0.966856  0.0  0.0 | 0.086136 | 0.087732 | 0.010767 |  0.01095 |  5.215506 |  4.686965 |  4.686965 | 19.529022 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  92.610082   92.610082  0.0  0.0  0.0 |       0.0        0.0  40.925739  0.0  0.0 |  6.065307   6.065307  19.625163  0.0  0.0 |  7.996301 |  9.758292 | 1.508308  1.494072  1.469131  1.438446  1.407419 | 19.529022 |
    | 01/01 10:00 |  0.0 | 32.2 | 31.2 |  8.264362 |  32.8 | 32.8  32.8  32.8  32.8  32.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |   9.58666    9.58666    9.58666    9.58666    9.58666 | 6.902395  6.902395  6.902395  6.902395  6.902395 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  28.682108  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  28.682108      0.0        0.0 |       0.0        0.0  28.682108      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.995196  3.995196  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  6.902395  0.0 |      0.0       0.0  18.732228  0.0  0.0 |      0.0       0.0  6.433737  0.0  0.0 |      0.0       0.0  1.518742  0.0  0.0 | 0.295808  0.295808  0.994474  0.0  0.0 | 0.091668 | 0.080021 | 0.011459 | 0.010843 |   5.49438 |  5.032391 |  5.032391 | 20.968298 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  88.614886   88.614886  0.0  0.0  0.0 |       0.0        0.0  42.441882  0.0  0.0 |  5.769498   5.769498   20.14943  0.0  0.0 |  7.131454 |  9.649346 | 1.590503  1.580467  1.561941  1.537181  1.509272 | 20.968298 |
    | 01/01 11:00 |  0.0 | 35.2 | 34.2 |  9.369867 |  35.8 | 35.8  35.8  35.8  35.8  35.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 | 10.869046  10.869046  10.869046  10.869046  10.869046 | 7.825713  7.825713  7.825713  7.825713  7.825713 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  31.387967  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  31.387967      0.0        0.0 |       0.0        0.0  31.387967      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 4.334217  4.334217  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  7.825713  0.0 |      0.0       0.0  20.393457  0.0  0.0 |      0.0       0.0  6.835514  0.0  0.0 |      0.0       0.0  1.492528  0.0  0.0 | 0.281382  0.281382  1.019398  0.0  0.0 | 0.096661 | 0.071441 | 0.012083 | 0.010722 |  5.895188 |   5.36627 |   5.36627 |  22.35946 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   84.28067    84.28067  0.0  0.0  0.0 |       0.0        0.0  44.600879  0.0  0.0 |  5.488116   5.488116  20.622561  0.0  0.0 |  6.162933 |  9.526489 |  1.70559  1.691872  1.667596  1.637256  1.605968 |  22.35946 |
    | 01/01 12:00 |  0.0 | 37.1 | 36.1 |  5.126178 |  37.7 | 37.7  37.7  37.7  37.7  37.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  5.946366   5.946366   5.946366   5.946366   5.946366 | 4.281384  4.281384  4.281384  4.281384  4.281384 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  33.101678  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  33.101678      0.0        0.0 |       0.0        0.0  33.101678      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 2.255237  2.255237  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  4.281384  0.0 |      0.0       0.0  21.917248  0.0  0.0 |      0.0       0.0  7.204051  0.0  0.0 |      0.0       0.0  1.468872  0.0  0.0 | 0.267659  0.267659  1.041891  0.0  0.0 | 0.101167 | 0.061826 | 0.012646 | 0.010586 |  6.260818 |  5.722373 |  5.722373 | 23.843222 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  82.025433   82.025433  0.0  0.0  0.0 |       0.0        0.0  46.581258  0.0  0.0 |  5.220458   5.220458  21.049541  0.0  0.0 |  5.658606 |   9.46059 | 1.811928  1.799103  1.775932  1.746002  1.713762 | 23.843222 |
    | 01/01 13:00 |  0.0 | 31.2 | 30.2 |   6.62503 |  31.8 | 31.8  31.8  31.8  31.8  31.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  7.685035   7.685035   7.685035   7.685035   7.685035 | 5.533225  5.533225  5.533225  5.533225  5.533225 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  27.780155  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  27.780155      0.0        0.0 |       0.0        0.0  27.780155      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 2.836657  2.836657  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  5.533225  0.0 |      0.0       0.0  20.602611  0.0  0.0 |      0.0       0.0  6.886099  0.0  0.0 |      0.0       0.0  1.447523  0.0  0.0 | 0.254605  0.254605  1.062191  0.0  0.0 | 0.105234 | 0.056828 | 0.013154 | 0.010513 |  5.925795 |  6.044162 |  6.044162 |  25.18401 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  79.188775   79.188775  0.0  0.0  0.0 |       0.0        0.0  44.872703  0.0  0.0 |  4.965853   4.965853  21.434873  0.0  0.0 |  5.004379 |  9.375402 |  1.72163  1.731696  1.747147  1.761138  1.766749 |  25.18401 |
    | 01/01 14:00 |  0.0 | 24.3 | 23.3 |  7.397619 |  24.9 | 24.9  24.9  24.9  24.9  24.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  8.581238   8.581238   8.581238   8.581238   8.581238 | 6.178491  6.178491  6.178491  6.178491  6.178491 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  21.556679  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  21.556679      0.0        0.0 |       0.0        0.0  21.556679      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 |  3.05792   3.05792  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  6.178491  0.0 |      0.0       0.0    17.4816  0.0  0.0 |      0.0       0.0  6.131266  0.0  0.0 |      0.0       0.0  1.428256  0.0  0.0 | 0.242188  0.242188   1.08051  0.0  0.0 | 0.108904 | 0.050337 | 0.013613 | 0.010419 |  5.140693 |  5.978675 |  5.978675 | 24.911147 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  76.130856   76.130856  0.0  0.0  0.0 |       0.0        0.0  40.816516  0.0  0.0 |  4.723666   4.723666   21.78262  0.0  0.0 |  4.278375 |  9.280525 | 1.498581  1.524852  1.570157  1.623741  1.673046 | 24.911147 |
    | 01/01 15:00 |  0.2 | 25.4 | 24.4 |   2.39151 |  26.0 | 26.0  26.0  26.0  26.0  26.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |  0.2376   0.2376   0.2376   0.2376   0.2376 |  2.774152   2.774152   2.774152   2.774152   2.774152 | 1.950491  1.950491  1.950491  1.950491  1.950491 |   0.2376    0.2376  0.0  0.0    0.2376 |       0.0        0.0   0.2376   0.2376        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  22.548827  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  22.786427   0.2376        0.0 |       0.0        0.0  22.786427   0.2376        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.928078  0.928078  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  1.950491  0.0 |      0.0       0.0  16.369483  0.0  0.0 |      0.0       0.0  5.862295  0.0  0.0 |      0.0       0.0  1.410869  0.0  0.0 | 0.230376  0.230376  1.097042  0.0  0.0 | 0.142387 |  0.04328 | 0.017798 | 0.010316 |  4.854012 |  5.480031 |  5.480031 | 22.833461 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  75.202777   75.202777  0.0  0.0  0.0 |       0.0        0.0  39.371166  0.0  0.0 |   4.49329    4.49329  22.096447  0.0  0.0 |  4.129801 |  9.257047 | 1.410728  1.421827  1.443499   1.47485  1.513455 | 22.833461 |
    | 01/01 16:00 |  0.0 | 25.9 | 24.9 |  1.829834 |  26.5 | 26.5  26.5  26.5  26.5  26.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  2.122607   2.122607   2.122607   2.122607   2.122607 | 1.528277  1.528277  1.528277  1.528277  1.528277 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  22.999804  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  22.999804      0.0        0.0 |       0.0        0.0  22.999804      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.718317  0.718317  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  1.528277  0.0 |      0.0       0.0  15.884739  0.0  0.0 |      0.0       0.0  5.745057  0.0  0.0 |      0.0       0.0  1.395178  0.0  0.0 |  0.21914   0.21914  1.111962  0.0  0.0 | 0.115204 | 0.041666 | 0.014401 | 0.010288 |  4.728326 |  5.035664 |  5.035664 | 20.981932 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  74.484461   74.484461  0.0  0.0  0.0 |       0.0        0.0  38.741173  0.0  0.0 |  4.274149   4.274149  22.379663  0.0  0.0 |  4.009272 |  9.236901 | 1.372546   1.37736  1.386809  1.400869  1.419438 | 20.981932 |
    | 01/01 17:00 |  0.0 | 23.7 | 22.7 |  1.136569 |  24.3 | 24.3  24.3  24.3  24.3  24.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |   1.31842    1.31842    1.31842    1.31842    1.31842 | 0.949262  0.949262  0.949262  0.949262  0.949262 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  21.015507  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  21.015507      0.0        0.0 |       0.0        0.0  21.015507      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.441908  0.441908  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.949262  0.0 |      0.0       0.0  14.856096  0.0  0.0 |      0.0       0.0  5.496275  0.0  0.0 |      0.0       0.0  1.381017  0.0  0.0 | 0.208453  0.208453  1.125426  0.0  0.0 | 0.117902 |  0.04048 | 0.014738 | 0.010266 |  4.467994 |  4.795157 |  4.795157 |  19.97982 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  74.042552   74.042552  0.0  0.0  0.0 |       0.0        0.0  37.404309  0.0  0.0 |  4.065697   4.065697  22.635253  0.0  0.0 |  3.966152 |  9.226306 | 1.298308  1.307101   1.32249  1.341402   1.36056 |  19.97982 |
    | 01/01 18:00 |  1.3 | 21.6 | 20.6 |  0.750986 |  22.2 | 22.2  22.2  22.2  22.2  22.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |  1.5444   1.5444   1.5444   1.5444   1.5444 |  0.871144   0.871144   0.871144   0.871144   0.871144 | 0.537465  0.537465  0.537465  0.537465  0.537465 | 0.537465  0.537465  0.0  0.0  0.537465 |       0.0        0.0   1.5444   1.5444        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  19.121405  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  20.665805   1.5444        0.0 |       0.0        0.0  20.665805   1.5444        0.0 | 0.0  0.0  0.0  0.0        0.0 |  0.12436   0.12436  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.537465  0.0 |      0.0       0.0  14.192484  0.0  0.0 |      0.0       0.0  5.335777  0.0  0.0 |      0.0       0.0  1.368237  0.0  0.0 | 0.198286  0.198286  1.137577  0.0  0.0 |  0.31645 | 0.041041 | 0.039556 | 0.010268 |  4.300028 |  4.581289 |  4.581289 | 19.088704 | 1.006935  1.006935  0.0  0.0  1.006935 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  73.918192   73.918192  0.0  0.0  0.0 |       0.0        0.0  36.541854  0.0  0.0 |   3.86741    3.86741  22.865914  0.0  0.0 |  4.173311 |  9.247063 | 1.248737  1.254797  1.266005  1.281029  1.298031 | 19.088704 |
    | 01/01 19:00 |  5.6 | 21.2 | 20.2 |  0.223895 |  21.8 | 21.8  21.8  21.8  21.8  21.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |  6.6528   6.6528   6.6528   6.6528   6.6528 |  0.259718   0.259718   0.259718   0.259718   0.259718 | 0.096141  0.096141  0.096141  0.096141  0.096141 | 0.096141  0.096141  0.0  0.0  0.096141 |  5.659735   5.659735   6.6528   6.6528   5.659735 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  18.760624  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  5.659735   5.659735  25.413424   6.6528   5.659735 |  0.773106   0.773106  25.413424   6.6528   5.659735 | 0.0  0.0  0.0  0.0   5.659735 | 0.023676  0.023676  0.0  0.0  0.0 | 0.773106  0.773106  2.0  0.0  0.0 | 0.0  0.0  0.0  0.096141  0.0 |      0.0       0.0  15.721177  0.0  0.0 |      0.0       0.0  5.705499  0.0  0.0 | 0.773106  0.773106  1.356704  0.0  0.0 | 0.207625  0.207625  1.148543  0.0  0.0 | 0.967333 | 0.046346 | 0.120917 | 0.010336 |  5.069838 |  4.449932 |  4.449932 | 18.541382 | 1.903859  1.903859  0.0  0.0  1.903859 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  78.781146   78.781146  0.0  0.0  0.0 |       0.0        0.0  38.528602  0.0  0.0 |   4.43289    4.43289  23.074075  0.0  0.0 |   5.08209 |  9.356118 | 1.462787  1.438105  1.397272  1.352706  1.317634 | 18.541382 |
    | 01/01 20:00 |  2.9 | 20.4 | 19.4 |  0.099425 |  21.0 | 21.0  21.0  21.0  21.0  21.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |  3.4452   3.4452   3.4452   3.4452   3.4452 |  0.115333   0.115333   0.115333   0.115333   0.115333 | 0.058839  0.058839  0.058839  0.058839  0.058839 | 0.058839  0.058839  0.0  0.0  0.058839 |  3.349059   3.349059   3.4452   3.4452   3.349059 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  18.039062  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  3.349059   3.349059  21.484262   3.4452   3.349059 |  0.519646   0.519646  21.484262   3.4452   3.349059 | 0.0  0.0  0.0  0.0   3.349059 | 0.015006  0.015006  0.0  0.0  0.0 | 0.519646  0.519646  2.0  0.0  0.0 | 0.0  0.0  0.0  0.058839  0.0 |      0.0       0.0  14.956896  0.0  0.0 |      0.0       0.0  5.520654  0.0  0.0 | 0.519646  0.519646  1.346296  0.0  0.0 | 0.228972  0.228972  1.158439  0.0  0.0 | 0.562001 | 0.053368 |  0.07025 | 0.010429 |  4.747396 |  4.733974 |  4.733974 | 19.724892 | 1.941161  1.941161  0.0  0.0  1.941161 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  81.595554   81.595554  0.0  0.0  0.0 |       0.0        0.0  37.535313  0.0  0.0 |  4.723564   4.723564  23.261932  0.0  0.0 |  5.583251 |  9.415005 | 1.379645  1.388479  1.400602  1.408286  1.404914 | 19.724892 |
    | 01/01 21:00 |  4.9 | 19.8 | 18.8 |  0.098454 |  20.4 | 20.4  20.4  20.4  20.4  20.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |  5.8212   5.8212   5.8212   5.8212   5.8212 |  0.114207   0.114207   0.114207   0.114207   0.114207 | 0.045942  0.045942  0.045942  0.045942  0.045942 | 0.045942  0.045942  0.0  0.0  0.045942 |  5.762361   5.762361   5.8212   5.8212   5.762361 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   17.49789  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  5.762361   5.762361   23.31909   5.8212   5.762361 |  0.959121   0.959121   23.31909   5.8212   5.762361 | 0.0  0.0  0.0  0.0   5.762361 | 0.012404  0.012404  0.0  0.0  0.0 | 0.959121  0.959121  2.0  0.0  0.0 | 0.0  0.0  0.0  0.045942  0.0 |      0.0       0.0  15.288016  0.0  0.0 |      0.0       0.0  5.600737  0.0  0.0 | 0.959121  0.959121  1.336903  0.0  0.0 | 0.253954  0.253954   1.16737  0.0  0.0 | 0.865504 | 0.059867 | 0.108188 | 0.010515 |  5.013445 |  4.842462 |  4.842462 | 20.176923 | 1.954058  1.954058  0.0  0.0  1.954058 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  86.386389   86.386389  0.0  0.0  0.0 |       0.0        0.0   37.96565  0.0  0.0 |  5.428731   5.428731  23.431465  0.0  0.0 |  6.383053 |  9.511948 | 1.451397  1.443352  1.430711  1.418087  1.409363 | 20.176923 |
    | 01/01 22:00 | 10.6 | 19.6 | 18.6 |  0.098128 |  20.2 | 20.2  20.2  20.2  20.2  20.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 | 12.5928  12.5928  12.5928  12.5928  12.5928 |  0.113828   0.113828   0.113828   0.113828   0.113828 | 0.023264  0.023264  0.023264  0.023264  0.023264 | 0.023264  0.023264  0.0  0.0  0.023264 | 12.546858  12.546858  12.5928  12.5928  12.546858 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  17.317499  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 12.546858  12.546858  29.910299  12.5928  12.546858 |  2.340807   2.340807  29.910299  12.5928  12.546858 | 0.0  0.0  0.0  0.0  12.546858 | 0.007022  0.007022  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.023264  0.0 |      0.0       0.0  18.050779  0.0  0.0 | 0.032432  0.032432  6.268925  0.0  0.0 |      2.0       2.0  1.328427  0.0  0.0 | 0.313939  0.313939   1.17543  0.0  0.0 | 1.727004 | 0.072119 | 0.215876 | 0.010683 |  6.220589 |  5.005159 |  5.005159 |  20.85483 | 1.976736  1.976736  0.0  0.0  1.976736 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  96.585418   96.585418  0.0  0.0  0.0 |  0.308375   0.308375  41.556244  0.0  0.0 |  7.114792   7.114792  23.584462  0.0  0.0 |  8.034985 |  9.716772 | 1.792088  1.752207    1.6842  1.605266  1.534578 |  20.85483 |
    | 01/01 23:00 |  0.1 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8  19.8  19.8  19.8  19.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |  0.1188   0.1188   0.1188   0.1188   0.1188 |   0.11307    0.11307    0.11307    0.11307    0.11307 | 0.080449  0.080449  0.080449  0.080449  0.080449 | 0.080449  0.080449  0.0  0.0  0.080449 |  0.095536   0.095536   0.1188   0.1188   0.095536 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.956718  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.095536   0.095536  17.075518   0.1188   0.095536 |  0.022281   0.022281  17.075518   0.1188   0.095536 | 0.0  0.0  0.0  0.0   0.095536 |   0.0243    0.0243  0.0  0.0  0.0 | 0.330656  0.330656  2.0  0.0  0.0 | 0.0  0.0  0.0  0.080449  0.0 |      0.0       0.0  14.413475  0.0  0.0 |      0.0       0.0  5.389225  0.0  0.0 | 0.330656  0.330656  1.320777  0.0  0.0 | 0.355123  0.355123  1.182704  0.0  0.0 | 0.144462 | 0.080669 | 0.018058 |   0.0108 |  4.501895 |  5.561796 |  5.561796 |  23.17415 | 1.919551  1.919551  0.0  0.0  1.919551 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  96.634373   96.634373  0.0  0.0  0.0 |       0.0        0.0  36.829062  0.0  0.0 |  7.090325   7.090325  23.722535  0.0  0.0 |  8.088561 |  9.722752 | 1.321948  1.375264  1.460502  1.546939  1.603786 |  23.17415 |
    | 02/01 00:00 |  0.7 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8  19.8  19.8  19.8  19.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  0.8316   0.8316   0.8316   0.8316   0.8316 |   0.11307    0.11307    0.11307    0.11307    0.11307 | 0.074914  0.074914  0.074914  0.074914  0.074914 | 0.074914  0.074914  0.0  0.0  0.074914 |  0.751151   0.751151   0.8316   0.8316   0.751151 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.963337  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.751151   0.751151  17.794937   0.8316   0.751151 |   0.17536    0.17536  17.794937   0.8316   0.751151 | 0.0  0.0  0.0  0.0   0.751151 | 0.022758  0.022758  0.0  0.0  0.0 |  0.17536   0.17536  2.0  0.0  0.0 | 0.0  0.0  0.0  0.074914  0.0 |      0.0       0.0  12.836543  0.0  0.0 |      0.0       0.0  5.007836  0.0  0.0 |  0.17536   0.17536  1.313873  0.0  0.0 | 0.350111  0.350111  1.189268  0.0  0.0 | 0.236291 |  0.08166 | 0.029536 | 0.010813 |  4.153181 |   5.11012 |   5.11012 | 21.292165 | 1.925086  1.925086  0.0  0.0  1.925086 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  97.187406   97.187406  0.0  0.0  0.0 |       0.0        0.0   34.77962  0.0  0.0 |  6.915574   6.915574   23.84714  0.0  0.0 |  8.233679 |  9.740286 | 1.208384  1.223384  1.254472  1.302233  1.363027 | 21.292165 |
    | 02/01 01:00 |  3.0 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8  19.8  19.8  19.8  19.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |   3.564    3.564    3.564    3.564    3.564 |   0.11307    0.11307    0.11307    0.11307    0.11307 | 0.057003  0.057003  0.057003  0.057003  0.057003 | 0.057003  0.057003  0.0  0.0  0.057003 |  3.489086   3.489086    3.564    3.564   3.489086 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.963337  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  3.489086   3.489086  20.527337    3.564   3.489086 |  0.823895   0.823895  20.527337    3.564   3.489086 | 0.0  0.0  0.0  0.0   3.489086 | 0.017787  0.017787  0.0  0.0  0.0 | 0.823895  0.823895  2.0  0.0  0.0 | 0.0  0.0  0.0  0.057003  0.0 |      0.0       0.0  13.105266  0.0  0.0 |      0.0       0.0  5.072828  0.0  0.0 | 0.823895  0.823895  1.307643  0.0  0.0 | 0.357535  0.357535  1.195192  0.0  0.0 | 0.584449 | 0.084839 | 0.073056 | 0.010857 |    4.4111 |  4.451249 |  4.451249 | 18.546871 | 1.942997  1.942997  0.0  0.0  1.942997 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.834811   99.834811  0.0  0.0  0.0 |       0.0        0.0  35.128863  0.0  0.0 |  7.381934   7.381934  23.959592  0.0  0.0 |  8.726051 |   9.80158 | 1.276832  1.269379  1.258716  1.251473  1.254951 | 18.546871 |
    | 02/01 02:00 |  2.1 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5  19.5  19.5  19.5  19.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  2.4948   2.4948   2.4948   2.4948   2.4948 |  0.112498   0.112498   0.112498   0.112498   0.112498 | 0.063115  0.063115  0.063115  0.063115  0.063115 | 0.063115  0.063115  0.0  0.0  0.063115 |  2.437797   2.437797   2.4948   2.4948   2.437797 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.692645  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  2.437797   2.437797  19.187445   2.4948   2.437797 |  0.607437   0.607437  19.187445   2.4948   2.437797 | 0.0  0.0  0.0  0.0   2.437797 | 0.020052  0.020052  0.0  0.0  0.0 | 0.607437  0.607437  2.0  0.0  0.0 | 0.0  0.0  0.0  0.063115  0.0 |      0.0       0.0  12.715476  0.0  0.0 |      0.0       0.0  4.978555  0.0  0.0 | 0.607437  0.607437   1.30202  0.0  0.0 | 0.374957  0.374957  1.200538  0.0  0.0 | 0.449748 | 0.089067 | 0.056219 | 0.010916 |  4.259725 |  4.334451 |  4.334451 | 18.060212 | 1.936885  1.936885  0.0  0.0  1.936885 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 101.645119  101.645119  0.0  0.0  0.0 |       0.0        0.0  34.622277  0.0  0.0 |  7.614414   7.614414  24.061074  0.0  0.0 |  9.078718 |  9.845881 | 1.236719  1.241132  1.247786  1.253831  1.257158 | 18.060212 |
    | 02/01 03:00 | 10.4 | 18.7 | 17.7 |  0.096652 |  19.3 | 19.3  19.3  19.3  19.3  19.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 | 12.3552  12.3552  12.3552  12.3552  12.3552 |  0.112116   0.112116   0.112116   0.112116   0.112116 | 0.023465  0.023465  0.023465  0.023465  0.023465 | 0.023465  0.023465  0.0  0.0  0.023465 | 12.292085  12.292085  12.3552  12.3552  12.292085 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.512184  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 12.292085  12.292085  28.867384  12.3552  12.292085 |  3.174963   3.174963  28.867384  12.3552  12.292085 | 0.0  0.0  0.0  0.0  12.292085 | 0.008122  0.008122  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.023465  0.0 |      0.0       0.0   16.32491  0.0  0.0 | 0.111812  0.111812  5.851515  0.0  0.0 |      2.0       2.0  1.296946  0.0  0.0 | 0.420536  0.420536  1.205362  0.0  0.0 | 1.702829 | 0.098821 | 0.212854 | 0.011052 |  5.917787 |  4.426264 |  4.426264 | 18.442768 | 1.976535  1.976535  0.0  0.0  1.976535 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 110.754119  110.754119  0.0  0.0  0.0 |   1.06315    1.06315  39.313237  0.0  0.0 |  9.193878   9.193878  24.152658  0.0  0.0 | 10.679747 |  10.04731 | 1.699999  1.646286  1.556337  1.455365   1.37016 | 18.442768 |
    | 02/01 04:00 |  3.5 | 18.5 | 17.5 |  0.096321 |  19.1 | 19.1  19.1  19.1  19.1  19.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |   4.158    4.158    4.158    4.158    4.158 |  0.111732   0.111732   0.111732   0.111732   0.111732 |  0.05308   0.05308   0.05308   0.05308   0.05308 |  0.05308   0.05308  0.0  0.0   0.05308 |  4.134535   4.134535    4.158    4.158   4.134535 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.331723  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  4.134535   4.134535  20.489723    4.158   4.134535 |  1.267904   1.267904  20.489723    4.158   4.134535 | 0.0  0.0  0.0  0.0   4.134535 | 0.018847  0.018847  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0   0.05308  0.0 |      0.0       0.0  14.874306  0.0  0.0 | 0.031504  0.031504  5.500679  0.0  0.0 |      2.0       2.0  1.292367  0.0  0.0 | 0.497568  0.497568  1.209716  0.0  0.0 | 0.662787 | 0.109568 | 0.082848 | 0.011203 |  5.022739 |  5.154817 |  5.154817 | 21.478404 |  1.94692   1.94692  0.0  0.0   1.94692 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 113.601903  113.601903  0.0  0.0  0.0 |   0.29955    0.29955  37.427975  0.0  0.0 |  10.69631   10.69631  24.235309  0.0  0.0 | 11.226225 | 10.118112 | 1.464798  1.490333  1.527298  1.555801   1.55784 | 21.478404 |
    | 02/01 05:00 |  3.4 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9  18.9  18.9  18.9  18.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  4.0392   4.0392   4.0392   4.0392   4.0392 |  0.111348   0.111348   0.111348   0.111348   0.111348 |  0.05353   0.05353   0.05353   0.05353   0.05353 |  0.05353   0.05353  0.0  0.0   0.05353 |   3.98612    3.98612   4.0392   4.0392    3.98612 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.151262  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |   3.98612    3.98612  20.190462   4.0392    3.98612 |  1.286061   1.286061  20.190462   4.0392    3.98612 | 0.0  0.0  0.0  0.0    3.98612 | 0.019455  0.019455  0.0  0.0  0.0 | 1.585612  1.585612  2.0  0.0  0.0 | 0.0  0.0  0.0   0.05353  0.0 |      0.0       0.0  14.014763  0.0  0.0 |      0.0       0.0  5.292794  0.0  0.0 | 1.585612  1.585612  1.288235  0.0  0.0 | 0.560653  0.560653  1.213646  0.0  0.0 | 0.648489 | 0.114934 | 0.081061 | 0.011281 |  4.824175 |  5.255509 |  5.255509 | 21.897955 |  1.94647   1.94647  0.0  0.0   1.94647 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 116.282507  116.282507  0.0  0.0  0.0 |       0.0        0.0   36.31088  0.0  0.0 | 11.721268  11.721268  24.309898  0.0  0.0 | 11.752982 | 10.187043 | 1.401228  1.409479  1.426035  1.450078  1.477915 | 21.897955 |
    | 02/01 06:00 |  1.2 | 18.5 | 17.5 |  0.187298 |  19.1 | 19.1  19.1  19.1  19.1  19.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  1.4256   1.4256   1.4256   1.4256   1.4256 |  0.217266   0.217266   0.217266   0.217266   0.217266 | 0.135647  0.135647  0.135647  0.135647  0.135647 | 0.135647  0.135647  0.0  0.0  0.135647 |   1.37207    1.37207   1.4256   1.4256    1.37207 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.331723  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |   1.37207    1.37207  17.757323   1.4256    1.37207 |  0.463815   0.463815  17.757323   1.4256    1.37207 | 0.0  0.0  0.0  0.0    1.37207 | 0.049677  0.049677  0.0  0.0  0.0 | 0.463815  0.463815  2.0  0.0  0.0 | 0.0  0.0  0.0  0.135647  0.0 |      0.0       0.0  12.617855  0.0  0.0 |      0.0       0.0  4.954945  0.0  0.0 | 0.463815  0.463815  1.284505  0.0  0.0 | 0.583058  0.583058  1.217192  0.0  0.0 | 0.317313 | 0.118525 | 0.039664 | 0.011335 |  4.320507 |  4.931351 |  4.931351 | 20.547296 | 1.864353  1.864353  0.0  0.0  1.864353 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 117.141085  117.141085  0.0  0.0  0.0 |       0.0        0.0  34.495404  0.0  0.0 | 11.602026  11.602026  24.377211  0.0  0.0 | 11.934545 | 10.213219 | 1.257945  1.274881  1.304408   1.34042  1.376237 | 20.547296 |
    | 02/01 07:00 |  0.1 | 18.8 | 17.8 |  1.264612 |  19.4 | 19.4  19.4  19.4  19.4  19.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  0.1188   0.1188   0.1188   0.1188   0.1188 |   1.46695    1.46695    1.46695    1.46695    1.46695 |  1.04373   1.04373   1.04373   1.04373   1.04373 |  1.04373   1.04373  0.0  0.0   1.04373 |       0.0        0.0   0.1188   0.1188        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.602415  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.721215   0.1188        0.0 |       0.0        0.0  16.721215   0.1188        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.382074  0.382074  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0   1.04373  0.0 |      0.0       0.0  11.495844  0.0  0.0 |      0.0       0.0  4.683581  0.0  0.0 |      0.0       0.0  1.281139  0.0  0.0 | 0.565837  0.565837  1.220392  0.0  0.0 | 0.152012 | 0.119508 | 0.019001 | 0.011352 |  3.941602 |   4.53272 |   4.53272 | 18.886335 | 0.939422  0.939422  0.0  0.0  0.939422 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 116.759011  116.759011  0.0  0.0  0.0 |       0.0        0.0  33.037194  0.0  0.0 | 11.036189  11.036189  24.437959  0.0  0.0 | 11.834511 | 10.204301 | 1.146936  1.160413  1.185037  1.217393  1.252994 | 18.886335 |
    | 02/01 08:00 |  0.0 | 18.8 | 17.8 |  3.045538 |  19.4 | 19.4  19.4  19.4  19.4  19.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  3.532824   3.532824   3.532824   3.532824   3.532824 | 2.543633  2.543633  2.543633  2.543633  2.543633 | 0.939422  0.939422  0.0  0.0  0.939422 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.602415  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.602415      0.0        0.0 |       0.0        0.0  16.602415      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.730206  1.730206  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  2.543633  0.0 |      0.0       0.0  10.875339  0.0  0.0 |      0.0       0.0  4.533508  0.0  0.0 |      0.0       0.0  1.278102  0.0  0.0 | 0.538241  0.538241   1.22328  0.0  0.0 | 0.137504 | 0.118441 | 0.017188 | 0.011341 |    3.7705 |  4.135838 |  4.135838 | 17.232658 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 115.028805  115.028805  0.0  0.0  0.0 |       0.0        0.0  32.230762  0.0  0.0 | 10.497947  10.497947  24.492781  0.0  0.0 | 11.530574 | 10.169773 | 1.095248  1.101712  1.114171  1.132042  1.154262 | 17.232658 |
    | 02/01 09:00 |  0.0 | 19.0 | 18.0 |  1.930758 |  19.6 | 19.6  19.6  19.6  19.6  19.6 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  2.239679   2.239679   2.239679   2.239679   2.239679 | 1.612569  1.612569  1.612569  1.612569  1.612569 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.782876  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.782876      0.0        0.0 |       0.0        0.0  16.782876      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.159324  1.159324  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  1.612569  0.0 |      0.0       0.0  10.629038  0.0  0.0 |      0.0       0.0  4.473939  0.0  0.0 |      0.0       0.0  1.275361  0.0  0.0 | 0.511991  0.511991  1.225886  0.0  0.0 | 0.138026 | 0.115419 | 0.017253 | 0.011303 |  3.691241 |  3.874554 |  3.874554 | 16.143976 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 113.869481  113.869481  0.0  0.0  0.0 |       0.0        0.0   31.91066  0.0  0.0 |  9.985957   9.985957  24.542256  0.0  0.0 | 11.348411 | 10.150127 | 1.071295  1.074298  1.080132  1.088669  1.099727 | 16.143976 |
    | 02/01 10:00 |  0.4 | 19.2 | 18.2 |  2.461001 |  19.8 | 19.8  19.8  19.8  19.8  19.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  0.4752   0.4752   0.4752   0.4752   0.4752 |  2.854761   2.854761   2.854761   2.854761   2.854761 | 1.960038  1.960038  1.960038  1.960038  1.960038 |   0.4752    0.4752  0.0  0.0    0.4752 |       0.0        0.0   0.4752   0.4752        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.963337  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  17.438537   0.4752        0.0 |       0.0        0.0  17.438537   0.4752        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.394929  1.394929  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  1.960038  0.0 |      0.0       0.0  10.761071  0.0  0.0 |      0.0       0.0  4.505872  0.0  0.0 |      0.0       0.0  1.272887  0.0  0.0 | 0.487021  0.487021  1.228238  0.0  0.0 | 0.198841 | 0.113909 | 0.024855 | 0.011285 |  3.708097 |  3.744657 |  3.744657 | 15.602739 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 112.474552  112.474552  0.0  0.0  0.0 |       0.0        0.0  32.082255  0.0  0.0 |  9.498936   9.498936  24.586904  0.0  0.0 | 11.184448 | 10.132584 | 1.075212  1.074857  1.074625  1.075267    1.0776 | 15.602739 |
    | 02/01 11:00 |  0.1 | 19.3 | 18.3 |  6.215945 |  19.9 | 19.9  19.9  19.9  19.9  19.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  0.1188   0.1188   0.1188   0.1188   0.1188 |  7.210496   7.210496   7.210496   7.210496   7.210496 | 5.130246  5.130246  5.130246  5.130246  5.130246 |   0.1188    0.1188  0.0  0.0    0.1188 |       0.0        0.0   0.1188   0.1188        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  17.053567  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  17.172367   0.1188        0.0 |       0.0        0.0  17.172367   0.1188        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.606389  3.606389  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  5.130246  0.0 |      0.0       0.0  10.723858  0.0  0.0 |      0.0       0.0  4.496872  0.0  0.0 |      0.0       0.0  1.270655  0.0  0.0 | 0.463269  0.463269  1.230361  0.0  0.0 | 0.154009 | 0.112055 | 0.019251 | 0.011263 |  3.683276 |  3.707694 |  3.707694 | 15.448727 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 108.868163  108.868163  0.0  0.0  0.0 |       0.0        0.0  32.033892  0.0  0.0 |  9.035667   9.035667  24.627198  0.0  0.0 | 10.574942 |  10.05914 | 1.068388  1.069169  1.070457  1.071899  1.073229 | 15.448727 |
    | 02/01 12:00 |  3.6 | 19.0 | 18.0 |  3.374783 |  19.6 | 19.6  19.6  19.6  19.6  19.6 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  4.2768   4.2768   4.2768   4.2768   4.2768 |  3.914748   3.914748   3.914748   3.914748   3.914748 | 1.837796  1.837796  1.837796  1.837796  1.837796 | 1.837796  1.837796  0.0  0.0  1.837796 |    2.2768     2.2768   4.2768   4.2768     2.2768 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.782876  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |    2.2768     2.2768  21.059676   4.2768     2.2768 |  0.674632   0.674632  21.059676   4.2768     2.2768 | 0.0  0.0  0.0  0.0     2.2768 | 0.634444  0.634444  0.0  0.0  0.0 | 0.674632  0.674632  2.0  0.0  0.0 | 0.0  0.0  0.0  1.837796  0.0 |      0.0       0.0  12.234366  0.0  0.0 |      0.0       0.0  4.862196  0.0  0.0 | 0.674632  0.674632   1.26864  0.0  0.0 | 0.457263  0.457263  1.232277  0.0  0.0 | 0.682392 | 0.108623 | 0.085299 | 0.011218 |  4.203764 |  3.730595 |  3.730595 | 15.544144 | 0.162204  0.162204  0.0  0.0  0.162204 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 109.835888  109.835888  0.0  0.0  0.0 |       0.0        0.0  33.997006  0.0  0.0 |  9.253036   9.253036  24.663562  0.0  0.0 | 10.915341 |  10.10405 | 1.214012  1.197108  1.168739   1.13678  1.109673 | 15.544144 |
    | 02/01 13:00 |  5.9 | 18.8 | 17.8 |  8.821555 |  19.4 | 19.4  19.4  19.4  19.4  19.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  7.0092   7.0092   7.0092   7.0092   7.0092 | 10.233004  10.233004  10.233004  10.233004  10.233004 | 3.655358  3.655358  3.655358  3.655358  3.655358 |      2.0       2.0  0.0  0.0       2.0 |  5.171404   5.171404   7.0092   7.0092   5.171404 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.602415  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  5.171404   5.171404  23.611615   7.0092   5.171404 |  1.559685   1.559685  23.611615   7.0092   5.171404 | 0.0  0.0  0.0  0.0   5.171404 |  2.12359   2.12359  0.0  0.0  0.0 | 1.559685  1.559685  2.0  0.0  0.0 | 0.0  0.0  0.0  3.655358  0.0 |      0.0       0.0  14.010901  0.0  0.0 |      0.0       0.0   5.29186  0.0  0.0 | 1.559685  1.559685  1.266822  0.0  0.0 | 0.489626  0.489626  1.234005  0.0  0.0 |  1.02971 | 0.113741 | 0.128714 | 0.011292 |  4.862587 |  4.030036 |  4.030036 | 16.791816 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 111.324016  111.324016  0.0  0.0  0.0 |       0.0        0.0  36.305861  0.0  0.0 | 10.323095  10.323095  24.696378  0.0  0.0 | 11.367137 |  10.16345 |  1.40354  1.380935  1.340987  1.291488  1.241913 | 16.791816 |
    | 02/01 14:00 |  1.1 | 18.7 | 17.7 |  4.046025 |  19.3 | 19.3  19.3  19.3  19.3  19.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  1.3068   1.3068   1.3068   1.3068   1.3068 |  4.693389   4.693389   4.693389   4.693389   4.693389 | 2.965278  2.965278  2.965278  2.965278  2.965278 |   1.3068    1.3068  0.0  0.0    1.3068 |       0.0        0.0   1.3068   1.3068        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.512184  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  17.818984   1.3068        0.0 |       0.0        0.0  17.818984   1.3068        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.860822  1.860822  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  2.965278  0.0 |      0.0       0.0  12.640141  0.0  0.0 |      0.0       0.0  4.960335  0.0  0.0 |      0.0       0.0  1.265181  0.0  0.0 | 0.503463  0.503463  1.235565  0.0  0.0 | 0.305908 | 0.114629 | 0.038239 | 0.011308 |  4.186828 |  4.489547 |  4.489547 | 18.706445 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 109.463194  109.463194  0.0  0.0  0.0 |       0.0        0.0  34.524368  0.0  0.0 |  9.819632   9.819632  24.725994  0.0  0.0 | 11.181873 | 10.143313 | 1.220562  1.241074  1.272998  1.303153  1.318357 | 18.706445 |
    | 02/01 15:00 | 20.7 | 17.8 | 16.8 |  2.110757 |  18.4 | 18.4  18.4  18.4  18.4  18.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 | 24.5916  24.5916  24.5916  24.5916  24.5916 |  2.448478   2.448478   2.448478   2.448478   2.448478 |  0.15074   0.15074   0.15074   0.15074   0.15074 |  0.15074   0.15074  0.0  0.0   0.15074 |   22.5916    22.5916  24.5916  24.5916    22.5916 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   15.70011  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |   22.5916    22.5916   40.29171  24.5916    22.5916 |  6.767422   6.767422   40.29171  24.5916    22.5916 | 0.0  0.0  0.0  0.0    22.5916 | 0.059018  0.059018  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0   0.15074  0.0 |      0.0       0.0  20.781507  0.0  0.0 |  0.45368   0.45368  6.929366  0.0  0.0 |      2.0       2.0    1.2637  0.0  0.0 | 0.528086  0.528086  1.236973  0.0  0.0 |  3.26299 | 0.127522 | 0.407874 | 0.011491 |  8.014481 |  4.689611 |  4.689611 | 19.540044 |  1.84926   1.84926  0.0  0.0   1.84926 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 125.228354  125.228354  0.0  0.0  0.0 |  4.313741   4.313741  45.105205  0.0  0.0 | 11.291546  11.291546  24.752721  0.0  0.0 |   14.2982 | 10.537304 | 2.287003  2.163706  1.958374  1.730314  1.541617 | 19.540044 |
    | 02/01 16:00 | 37.9 | 17.4 | 16.4 |  2.239257 |  18.0 | 18.0  18.0  18.0  18.0  18.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 | 45.0252  45.0252  45.0252  45.0252  45.0252 |  2.597538   2.597538   2.597538   2.597538   2.597538 | 0.020724  0.020724  0.020724  0.020724  0.020724 | 0.020724  0.020724  0.0  0.0  0.020724 |  44.87446   44.87446  45.0252  45.0252   44.87446 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.339188  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  44.87446   44.87446  60.364388  45.0252   44.87446 |  17.59319   17.59319  60.364388  45.0252   44.87446 | 0.0  0.0  0.0  0.0   44.87446 | 0.009877  0.009877  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.020724  0.0 |      0.0       0.0  32.842725  0.0  0.0 | 1.894395  1.894395  9.846434  0.0  0.0 | 1.935423  1.935423  1.262364  0.0  0.0 | 0.598284  0.598284  1.238244  0.0  0.0 | 5.894889 | 0.171646 | 0.736861 | 0.012111 | 13.444225 |  6.813266 |  6.813266 | 28.388608 | 1.979276  1.979276  0.0  0.0  1.979276 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 152.499747  152.499747  0.0  0.0  0.0 | 18.012536  18.012536  60.780433  0.0  0.0 | 12.628684  12.628684  24.776841  0.0  0.0 | 20.018811 | 11.261725 | 3.844501  3.659266  3.333676  2.934144  2.540386 | 28.388608 |
    | 02/01 17:00 |  8.2 | 17.3 | 16.3 |  2.877848 |  17.9 | 17.9  17.9  17.9  17.9  17.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  9.7416   9.7416   9.7416   9.7416   9.7416 |  3.338304   3.338304   3.338304   3.338304   3.338304 | 0.907373  0.907373  0.907373  0.907373  0.907373 | 0.907373  0.907373  0.0  0.0  0.907373 |  9.720876   9.720876   9.7416   9.7416   9.720876 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.248957  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  9.720876   9.720876  24.990557   9.7416   9.720876 |  5.651759   5.651759  24.990557   9.7416   9.720876 | 0.0  0.0  0.0  0.0   9.720876 | 0.443958  0.443958  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.907373  0.0 | 0.654849  0.654849  25.091929  0.0  0.0 |  2.06163   2.06163  7.971864  0.0  0.0 | 1.868566  1.868566  1.261158  0.0  0.0 | 0.661853  0.661853  1.239391  0.0  0.0 | 1.452866 | 0.206431 | 0.181608 | 0.012607 |   9.74013 | 10.545245 | 10.545245 | 43.938521 | 1.092627  1.092627  0.0  0.0  1.092627 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 156.124906  156.124906  0.0  0.0  0.0 | 18.947816  18.947816  50.707197  0.0  0.0 | 13.835397  13.835397  24.798608  0.0  0.0 | 21.150024 | 11.416323 | 2.859011  2.967344   3.12843  3.261942  3.290131 | 43.938521 |
    | 02/01 18:00 |  3.6 | 16.8 | 15.8 |  1.591452 |  17.4 | 17.4  17.4  17.4  17.4  17.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  4.2768   4.2768   4.2768   4.2768   4.2768 |  1.846084   1.846084   1.846084   1.846084   1.846084 | 0.866652  0.866652  0.866652  0.866652  0.866652 | 0.866652  0.866652  0.0  0.0  0.866652 |  3.369427   3.369427   4.2768   4.2768   3.369427 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.797804  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  3.369427   3.369427  19.074604   4.2768   3.369427 |  2.053243   2.053243  19.074604   4.2768   3.369427 | 0.0  0.0  0.0  0.0   3.369427 | 0.426396  0.426396  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.866652  0.0 |      0.0       0.0  18.800674  0.0  0.0 |  1.80819   1.80819  6.450291  0.0  0.0 |  1.80823   1.80823   1.26007  0.0  0.0 | 0.719222  0.719222  1.240426  0.0  0.0 | 0.793608 | 0.214401 | 0.099201 | 0.012733 |  7.251345 | 10.661148 | 10.661148 | 44.421449 | 1.133348  1.133348  0.0  0.0  1.133348 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 157.014694  157.014694  0.0  0.0  0.0 | 17.192869  17.192869  42.530836  0.0  0.0 | 14.924405  14.924405  24.818252  0.0  0.0 |  21.61918 | 11.489035 | 2.128374  2.216971  2.377564  2.582707  2.791439 | 44.421449 |
    | 02/01 19:00 |  7.5 | 16.5 | 15.5 |  0.291604 |  17.1 | 17.1  17.1  17.1  17.1  17.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |    8.91     8.91     8.91     8.91     8.91 |  0.338261   0.338261   0.338261   0.338261   0.338261 | 0.099914  0.099914  0.099914  0.099914  0.099914 | 0.099914  0.099914  0.0  0.0  0.099914 |  8.043348   8.043348     8.91     8.91   8.043348 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.527113  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  8.043348   8.043348  23.437113     8.91   8.043348 |   4.95744    4.95744  23.437113     8.91   8.043348 | 0.0  0.0  0.0  0.0   8.043348 | 0.049957  0.049957  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.099914  0.0 | 0.059142  0.059142   17.30004  0.0  0.0 | 1.917555  1.917555  6.087355  0.0  0.0 |  1.75378   1.75378  1.259087  0.0  0.0 | 0.770995  0.770995  1.241359  0.0  0.0 | 1.413252 | 0.222157 | 0.176657 | 0.012857 |  7.329936 |  8.572839 |  8.572839 | 35.720163 | 1.900086  1.900086  0.0  0.0  1.900086 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 160.050645  160.050645  0.0  0.0  0.0 | 18.173612  18.173612  40.580554  0.0  0.0 |  15.90719   15.90719   24.83598  0.0  0.0 | 22.797587 | 11.651249 | 2.125772  2.129061  2.144936  2.187224   2.26716 | 35.720163 |
    | 02/01 20:00 | 18.5 | 16.3 | 15.3 |  0.092622 |  16.9 | 16.9  16.9  16.9  16.9  16.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  21.978   21.978   21.978   21.978   21.978 |  0.107442   0.107442   0.107442   0.107442   0.107442 |  0.00859   0.00859   0.00859   0.00859   0.00859 |  0.00859   0.00859  0.0  0.0   0.00859 | 21.878086  21.878086   21.978   21.978  21.878086 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.346652  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 21.878086  21.878086  36.324652   21.978  21.878086 |  14.01084   14.01084  36.324652   21.978  21.878086 | 0.0  0.0  0.0  0.0  21.878086 | 0.004295  0.004295  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0   0.00859  0.0 |  4.00727   4.00727  21.603515  0.0  0.0 |  2.87243   2.87243  7.128173  0.0  0.0 | 1.704641  1.704641  1.258201  0.0  0.0 | 0.817717  0.817717  1.242202  0.0  0.0 | 3.100929 | 0.242293 | 0.387616 | 0.013154 | 12.310184 |  7.889037 |  7.889037 | 32.870989 |  1.99141   1.99141  0.0  0.0   1.99141 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 167.913595  167.913595  0.0  0.0  0.0 | 23.304752  23.304752  46.173518  0.0  0.0 | 16.794113  16.794113  24.851979  0.0  0.0 | 25.655132 | 12.025575 |  3.52128  3.359144  3.086934  2.781235  2.526707 | 32.870989 |
    | 02/01 21:00 | 15.4 | 16.2 | 15.2 |  0.092451 |  16.8 | 16.8  16.8  16.8  16.8  16.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 | 18.2952  18.2952  18.2952  18.2952  18.2952 |  0.107243   0.107243   0.107243   0.107243   0.107243 | 0.012392  0.012392  0.012392  0.012392  0.012392 | 0.012392  0.012392  0.0  0.0  0.012392 |  18.28661   18.28661  18.2952  18.2952   18.28661 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.256421  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  18.28661   18.28661  32.551621  18.2952   18.28661 | 12.889763  12.889763  32.551621  18.2952   18.28661 | 0.0  0.0  0.0  0.0   18.28661 | 0.006196  0.006196  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.012392  0.0 | 5.585106  5.585106  22.319603  0.0  0.0 | 3.254038  3.254038  7.301362  0.0  0.0 | 1.660294  1.660294  1.257401  0.0  0.0 | 0.859883  0.859883  1.242963  0.0  0.0 | 2.658765 | 0.268522 | 0.332346 | 0.013539 | 13.474567 |  10.28261 |  10.28261 | 42.844207 | 1.987608  1.987608  0.0  0.0  1.987608 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 173.304246  173.304246  0.0  0.0  0.0 |  25.35537   25.35537  47.104174  0.0  0.0 | 17.594525  17.594525  24.866417  0.0  0.0 | 28.043801 | 12.344185 | 3.894603  3.845735  3.745282  3.590931  3.390705 | 42.844207 |
    | 02/01 22:00 |  6.3 | 15.5 | 14.5 |  0.091248 |  16.1 | 16.1  16.1  16.1  16.1  16.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  7.4844   7.4844   7.4844   7.4844   7.4844 |  0.105848   0.105848   0.105848   0.105848   0.105848 | 0.036055  0.036055  0.036055  0.036055  0.036055 | 0.036055  0.036055  0.0  0.0  0.036055 |  7.472008   7.472008   7.4844   7.4844   7.472008 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.624808  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  7.472008   7.472008  21.109208   7.4844   7.472008 |  5.610425   5.610425  21.109208   7.4844   7.472008 | 0.0  0.0  0.0  0.0   7.472008 | 0.018028  0.018028  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.036055  0.0 | 3.527765  3.527765  18.183549  0.0  0.0 |  2.75646   2.75646  6.301036  0.0  0.0 | 1.620274  1.620274  1.256679  0.0  0.0 | 0.897935  0.897935  1.243649  0.0  0.0 | 1.308971 | 0.285564 | 0.163621 | 0.013799 | 10.232482 | 12.333926 | 12.333926 | 51.391356 | 1.963945  1.963945  0.0  0.0  1.963945 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 175.147801  175.147801  0.0  0.0  0.0 | 22.681569  22.681569  41.728796  0.0  0.0 | 18.316863  18.316863  24.879447  0.0  0.0 |  29.06263 | 12.493435 | 2.998669  3.101186  3.267654  3.440647  3.557657 | 51.391356 |
    | 02/01 23:00 |  1.9 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2  15.2  15.2  15.2  15.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  2.2572   2.2572   2.2572   2.2572   2.2572 |  0.104032   0.104032   0.104032   0.104032   0.104032 | 0.059768  0.059768  0.059768  0.059768  0.059768 | 0.059768  0.059768  0.0  0.0  0.059768 |  2.221145   2.221145   2.2572   2.2572   2.221145 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  12.812733  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  2.221145   2.221145  15.069933   2.2572   2.221145 |  1.703438   1.703438  15.069933   2.2572   2.221145 | 0.0  0.0  0.0  0.0   2.221145 | 0.029884  0.029884  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.059768  0.0 | 0.938427  0.938427  13.692233  0.0  0.0 | 2.130215  2.130215  5.214789  0.0  0.0 | 1.584157  1.584157  1.256028  0.0  0.0 | 0.932276  0.932276  1.244269  0.0  0.0 | 0.665962 | 0.292497 | 0.083245 |  0.01392 |  6.864874 | 11.234759 | 11.234759 | 46.811496 | 1.940232  1.940232  0.0  0.0  1.940232 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 175.635624  175.635624  0.0  0.0  0.0 | 19.316365  19.316365  35.891708  0.0  0.0 | 18.968744  18.968744  24.891206  0.0  0.0 | 29.428506 | 12.561812 | 2.024782  2.141464  2.349191  2.609124  2.871367 | 46.811496 |
    | 03/01 00:00 |  4.9 | 14.7 | 13.7 |  0.089858 |  15.3 | 15.3  15.3  15.3  15.3  15.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  5.8212   5.8212   5.8212   5.8212   5.8212 |  0.104235   0.104235   0.104235   0.104235   0.104235 | 0.041931  0.041931  0.041931  0.041931  0.041931 | 0.041931  0.041931  0.0  0.0  0.041931 |  5.761432   5.761432   5.8212   5.8212   5.761432 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   12.90841  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  5.761432   5.761432   18.72961   5.8212   5.761432 |  4.443198   4.443198   18.72961   5.8212   5.761432 | 0.0  0.0  0.0  0.0   5.761432 | 0.020966  0.020966  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.041931  0.0 | 0.692334  0.692334  12.835488  0.0  0.0 | 2.070696  2.070696   5.00758  0.0  0.0 | 1.551563  1.551563   1.25544  0.0  0.0 | 0.963267  0.963267  1.244828  0.0  0.0 | 1.137271 | 0.298486 | 0.142159 | 0.014029 |  6.729133 |  8.549255 |  8.549255 | 35.621894 | 1.958069  1.958069  0.0  0.0  1.958069 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 176.932892  176.932892  0.0  0.0  0.0 | 18.996532  18.996532  34.778249  0.0  0.0 |  19.55704   19.55704  24.901817  0.0  0.0 | 30.261967 | 12.689276 |  1.95392  1.966091   2.00007  2.069951  2.185775 | 35.621894 |
    | 03/01 01:00 |  2.7 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2  15.2  15.2  15.2  15.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  3.2076   3.2076   3.2076   3.2076   3.2076 |  0.104032   0.104032   0.104032   0.104032   0.104032 |  0.05435   0.05435   0.05435   0.05435   0.05435 |  0.05435   0.05435  0.0  0.0   0.05435 |  3.165669   3.165669   3.2076   3.2076   3.165669 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  12.818141  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  3.165669   3.165669  16.025741   3.2076   3.165669 |  2.477551   2.477551  16.025741   3.2076   3.165669 | 0.0  0.0  0.0  0.0   3.165669 | 0.027175  0.027175  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0   0.05435  0.0 |      0.0       0.0  11.333487  0.0  0.0 | 1.853204  1.853204  4.644314  0.0  0.0 | 1.522148  1.522148  1.254909  0.0  0.0 | 0.991235  0.991235  1.245332  0.0  0.0 | 0.822295 | 0.305209 | 0.102787 | 0.014148 |  5.660402 |  7.020052 |  7.020052 | 29.250216 |  1.94565   1.94565  0.0  0.0   1.94565 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 177.593835  177.593835  0.0  0.0  0.0 |  17.62088   17.62088  32.826189  0.0  0.0 | 20.087952  20.087952  24.911394  0.0  0.0 |  30.77215 | 12.777052 | 1.652046  1.687553  1.749145  1.824577  1.902837 | 29.250216 |
    | 03/01 02:00 |  0.5 | 14.1 | 13.1 |  0.088805 |  14.7 | 14.7  14.7  14.7  14.7  14.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |   0.594    0.594    0.594    0.594    0.594 |  0.103014   0.103014   0.103014   0.103014   0.103014 | 0.069893  0.069893  0.069893  0.069893  0.069893 | 0.069893  0.069893  0.0  0.0  0.069893 |   0.53965    0.53965    0.594    0.594    0.53965 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  12.366798  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |   0.53965    0.53965  12.960798    0.594    0.53965 |  0.425509   0.425509  12.960798    0.594    0.53965 | 0.0  0.0  0.0  0.0    0.53965 | 0.034946  0.034946  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.069893  0.0 |      0.0       0.0    9.35945  0.0  0.0 | 1.527016  1.527016  4.166883  0.0  0.0 | 1.495602  1.495602   1.25443  0.0  0.0 | 1.016476  1.016476  1.245788  0.0  0.0 | 0.505669 | 0.308708 | 0.063209 | 0.014224 |  4.817899 |  6.104204 |  6.104204 | 25.434182 | 1.930107  1.930107  0.0  0.0  1.930107 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 177.673031  177.673031  0.0  0.0  0.0 | 14.519372  14.519372  30.260655  0.0  0.0 | 20.567079  20.567079  24.920037  0.0  0.0 | 30.960236 | 12.824927 |  1.40578  1.435608   1.48987   1.56067  1.637924 | 25.434182 |
    | 03/01 03:00 |  0.2 | 14.3 | 13.3 |  0.089157 |  14.9 | 14.9  14.9  14.9  14.9  14.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  0.2376   0.2376   0.2376   0.2376   0.2376 |  0.103422   0.103422   0.103422   0.103422   0.103422 | 0.072716  0.072716  0.072716  0.072716  0.072716 | 0.072716  0.072716  0.0  0.0  0.072716 |  0.167707   0.167707   0.2376   0.2376   0.167707 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  12.547335  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.167707   0.167707  12.784935   0.2376   0.167707 |  0.132353   0.132353  12.784935   0.2376   0.167707 | 0.0  0.0  0.0  0.0   0.167707 | 0.036358  0.036358  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.072716  0.0 |      0.0       0.0   8.280794  0.0  0.0 | 1.203971  1.203971  3.906004  0.0  0.0 | 1.471646  1.471646  1.253998  0.0  0.0 | 1.039254  1.039254  1.246198  0.0  0.0 | 0.474184 | 0.310423 | 0.059273 | 0.014275 |  4.346766 |  5.237707 |  5.237707 | 21.823777 | 1.927284  1.927284  0.0  0.0  1.927284 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 177.672027  177.672027  0.0  0.0  0.0 | 11.447755  11.447755  28.858791  0.0  0.0 | 20.999471  20.999471  24.927837  0.0  0.0 | 31.114763 | 12.868771 | 1.265456  1.282776  1.315419  1.360666  1.414594 | 21.823777 |
    | 03/01 04:00 |  0.5 | 14.9 | 13.9 |  0.090207 |  15.5 | 15.5  15.5  15.5  15.5  15.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |   0.594    0.594    0.594    0.594    0.594 |   0.10464    0.10464    0.10464    0.10464    0.10464 | 0.070996  0.070996  0.070996  0.070996  0.070996 | 0.070996  0.070996  0.0  0.0  0.070996 |  0.521284   0.521284    0.594    0.594   0.521284 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.088947  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.521284   0.521284  13.682947    0.594   0.521284 |  0.411389   0.411389  13.682947    0.594   0.521284 | 0.0  0.0  0.0  0.0   0.521284 | 0.035498  0.035498  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.070996  0.0 |      0.0       0.0   8.082544  0.0  0.0 | 0.938222  0.938222  3.858056  0.0  0.0 | 1.450026  1.450026  1.253608  0.0  0.0 |  1.05981   1.05981  1.246569  0.0  0.0 | 0.531869 | 0.312248 | 0.066484 | 0.014328 |  4.175809 |  4.616671 |  4.616671 | 19.236128 | 1.929004  1.929004  0.0  0.0  1.929004 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 177.746424  177.746424  0.0  0.0  0.0 |  8.920923   8.920923  28.601138  0.0  0.0 | 21.389688  21.389688  24.934876  0.0  0.0 |  31.32537 |   12.9198 |  1.21282  1.219524  1.232871  1.253014  1.279821 | 19.236128 |
    | 03/01 05:00 |  2.4 | 15.7 | 14.7 |  0.091593 |  16.3 | 16.3  16.3  16.3  16.3  16.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  2.8512   2.8512   2.8512   2.8512   2.8512 |  0.106248   0.106248   0.106248   0.106248   0.106248 | 0.057521  0.057521  0.057521  0.057521  0.057521 | 0.057521  0.057521  0.0  0.0  0.057521 |  2.780204   2.780204   2.8512   2.8512   2.780204 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.811096  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  2.780204   2.780204  16.662296   2.8512   2.780204 |   2.19593    2.19593  16.662296   2.8512   2.780204 | 0.0  0.0  0.0  0.0   2.780204 | 0.028761  0.028761  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.057521  0.0 |      0.0       0.0   9.153448  0.0  0.0 | 0.867583  0.867583   4.11706  0.0  0.0 | 1.430516  1.430516  1.253256  0.0  0.0 | 1.078362  1.078362  1.246904  0.0  0.0 | 0.829714 | 0.315827 | 0.103714 | 0.014405 |  4.564613 |  4.324856 |  4.324856 | 18.020234 | 1.942479  1.942479  0.0  0.0  1.942479 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 178.301938  178.301938  0.0  0.0  0.0 |  8.249269   8.249269  29.992926  0.0  0.0 | 21.741842  21.741842  24.941228  0.0  0.0 | 31.831952 | 13.008196 | 1.319999  1.307764  1.287991  1.267665  1.254387 | 18.020234 |
    | 03/01 06:00 |  0.4 | 16.0 | 15.0 |  0.154861 |  16.6 | 16.6  16.6  16.6  16.6  16.6 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  0.4752   0.4752   0.4752   0.4752   0.4752 |  0.179639   0.179639   0.179639   0.179639   0.179639 | 0.123337  0.123337  0.123337  0.123337  0.123337 | 0.123337  0.123337  0.0  0.0  0.123337 |  0.417679   0.417679   0.4752   0.4752   0.417679 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.081902  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.417679   0.417679  14.557102   0.4752   0.417679 |  0.331967   0.331967  14.557102   0.4752   0.417679 | 0.0  0.0  0.0  0.0   0.417679 | 0.061669  0.061669  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.123337  0.0 |      0.0       0.0   8.872744  0.0  0.0 | 0.626287  0.626287   4.04917  0.0  0.0 | 1.412908  1.412908  1.252939  0.0  0.0 | 1.095103  1.095103  1.247206  0.0  0.0 | 0.538121 | 0.319415 | 0.067265 | 0.014483 |  4.206142 |   4.39217 |   4.39217 | 18.300709 | 1.876663  1.876663  0.0  0.0  1.876663 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 178.325981  178.325981  0.0  0.0  0.0 |  5.954949   5.954949  29.628112  0.0  0.0 | 22.059646  22.059646  24.946961  0.0  0.0 | 32.034996 | 13.059021 | 1.223159  1.234005  1.250917  1.267177  1.276521 | 18.300709 |
    | 03/01 07:00 |  0.2 | 16.7 | 15.7 |  0.470369 |  17.3 | 17.3  17.3  17.3  17.3  17.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  0.2376   0.2376   0.2376   0.2376   0.2376 |  0.545628   0.545628   0.545628   0.545628   0.545628 | 0.383628  0.383628  0.383628  0.383628  0.383628 | 0.383628  0.383628  0.0  0.0  0.383628 |  0.114263   0.114263   0.2376   0.2376   0.114263 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.713782  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.114263   0.114263  14.951382   0.2376   0.114263 |  0.090839   0.090839  14.951382   0.2376   0.114263 | 0.0  0.0  0.0  0.0   0.114263 | 0.191814  0.191814  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.383628  0.0 |      0.0       0.0   8.884339  0.0  0.0 | 0.385008  0.385008  4.051975  0.0  0.0 | 1.397018  1.397018  1.252652  0.0  0.0 | 1.110212  1.110212  1.247478  0.0  0.0 | 0.517085 |  0.32133 | 0.064636 | 0.014538 |  4.054985 |  4.318662 |  4.318662 | 17.994424 | 1.616372  1.616372  0.0  0.0  1.616372 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 178.157591  178.157591  0.0  0.0  0.0 |   3.66078    3.66078  29.643181  0.0  0.0 | 22.346452  22.346452  24.952134  0.0  0.0 | 32.182036 | 13.103029 | 1.177537  1.183207  1.193924  1.208542  1.224892 | 17.994424 |
    | 03/01 08:00 |  0.0 | 17.1 | 16.1 |  1.173726 |  17.7 | 17.7  17.7  17.7  17.7  17.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  1.361522   1.361522   1.361522   1.361522   1.361522 | 0.980296  0.980296  0.980296  0.980296  0.980296 | 0.980296  0.980296  0.0  0.0  0.980296 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.074856  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  15.074856      0.0        0.0 |       0.0        0.0  15.074856      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.490148  0.490148  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.980296  0.0 |      0.0       0.0   8.938851  0.0  0.0 | 0.158044  0.158044  4.065159  0.0  0.0 | 1.382677  1.382677  1.252393  0.0  0.0 | 1.123847  1.123847  1.247724  0.0  0.0 | 0.495157 | 0.322684 | 0.061895 | 0.014585 |  3.934266 |  4.137149 |  4.137149 |  17.23812 | 0.636076  0.636076  0.0  0.0  0.636076 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 177.667443  177.667443  0.0  0.0  0.0 |  1.502736   1.502736  29.714027  0.0  0.0 | 22.605282  22.605282  24.956803  0.0  0.0 | 32.230027 | 13.134778 | 1.142171  1.146474  1.154393  1.165007  1.177175 |  17.23812 |
    | 03/01 09:00 |  0.0 | 16.2 | 15.2 |  4.202296 |  16.8 | 16.8  16.8  16.8  16.8  16.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  4.874663   4.874663   4.874663   4.874663   4.874663 | 3.509758  3.509758  3.509758  3.509758  3.509758 | 0.636076  0.636076  0.0  0.0  0.636076 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.262439  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  14.262439      0.0        0.0 |       0.0        0.0  14.262439      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 |  3.19172   3.19172  0.0  0.0  0.0 | 1.502736  1.502736  2.0  0.0  0.0 | 0.0  0.0  0.0  3.509758  0.0 |      0.0       0.0   8.647066  0.0  0.0 |      0.0       0.0  3.994589  0.0  0.0 | 1.369736  1.369736   1.25216  0.0  0.0 | 1.136152  1.136152  1.247946  0.0  0.0 | 0.218446 | 0.321783 | 0.027306 | 0.014601 |   3.77357 |  3.990135 |  3.990135 | 16.625564 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 174.475723  174.475723  0.0  0.0  0.0 |       0.0        0.0  29.334812  0.0  0.0 | 22.838865  22.838865  24.961016  0.0  0.0 | 31.681006 | 13.091772 | 1.095948  1.101467  1.111263  1.123577  1.136399 | 16.625564 |
    | 03/01 10:00 |  0.3 | 15.9 | 14.9 |  4.359715 |  16.5 | 16.5  16.5  16.5  16.5  16.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  0.3564   0.3564   0.3564   0.3564   0.3564 |  5.057269   5.057269   5.057269   5.057269   5.057269 | 3.513746  3.513746  3.513746  3.513746  3.513746 |   0.3564    0.3564  0.0  0.0    0.3564 |       0.0        0.0   0.3564   0.3564        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.991633  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  14.348033   0.3564        0.0 |       0.0        0.0  14.348033   0.3564        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.335546  3.335546  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  3.513746  0.0 |      0.0       0.0   8.531535  0.0  0.0 |      0.0       0.0  3.966647  0.0  0.0 |      0.0       0.0  1.251949  0.0  0.0 | 1.113865  1.113865  1.248147  0.0  0.0 | 0.187743 | 0.316167 | 0.023468 | 0.014551 |  3.726255 |  3.850707 |  3.850707 | 16.044612 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 171.140177  171.140177  0.0  0.0  0.0 |       0.0        0.0  29.184663  0.0  0.0 | 21.725001  21.725001  24.964819  0.0  0.0 | 31.106393 | 13.044915 | 1.081131  1.083039  1.086889  1.092735  1.100407 | 16.044612 |
    | 03/01 11:00 |  2.6 | 16.3 | 15.3 |  5.305753 |  16.9 | 16.9  16.9  16.9  16.9  16.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  3.0888   3.0888   3.0888   3.0888   3.0888 |  6.154673   6.154673   6.154673   6.154673   6.154673 | 3.253813  3.253813  3.253813  3.253813  3.253813 |      2.0       2.0  0.0  0.0       2.0 |    1.0888     1.0888   3.0888   3.0888     1.0888 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.352707  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |    1.0888     1.0888  17.441507   3.0888     1.0888 |  0.797246   0.797246  17.441507   3.0888     1.0888 | 0.0  0.0  0.0  0.0     1.0888 | 2.253813  2.253813  0.0  0.0  0.0 | 0.797246  0.797246  2.0  0.0  0.0 | 0.0  0.0  0.0  3.253813  0.0 |      0.0       0.0   9.689643  0.0  0.0 |      0.0       0.0  4.246742  0.0  0.0 | 0.797246  0.797246  1.251759  0.0  0.0 | 1.079144  1.079144  1.248327  0.0  0.0 | 0.534751 | 0.312179 | 0.066844 | 0.014523 |  4.061937 |  3.785096 |  3.785096 | 15.771231 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 169.177919  169.177919  0.0  0.0  0.0 |       0.0        0.0  30.689786  0.0  0.0 | 21.443102  21.443102  24.968251  0.0  0.0 | 30.915782 | 13.045588 | 1.174698  1.163884  1.145916  1.126153  1.110392 | 15.771231 |
    | 03/01 12:00 |  0.7 | 16.3 | 15.3 |  5.376027 |  16.9 | 16.9  16.9  16.9  16.9  16.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  0.8316   0.8316   0.8316   0.8316   0.8316 |  6.236191   6.236191   6.236191   6.236191   6.236191 | 4.131769  4.131769  4.131769  4.131769  4.131769 |   0.8316    0.8316  0.0  0.0    0.8316 |       0.0        0.0   0.8316   0.8316        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.352707  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  15.184307   0.8316        0.0 |       0.0        0.0  15.184307   0.8316        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.715969  3.715969  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  4.131769  0.0 |      0.0       0.0   9.393724  0.0  0.0 |      0.0       0.0  4.175172  0.0  0.0 |      0.0       0.0  1.251587  0.0  0.0 | 1.045792  1.045792  1.248491  0.0  0.0 | 0.248155 | 0.308854 | 0.031019 | 0.014504 |  3.892754 |  3.911809 |  3.911809 | 16.299204 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  165.46195   165.46195  0.0  0.0  0.0 |       0.0        0.0  30.305198  0.0  0.0 |  20.39731   20.39731  24.971347  0.0  0.0 | 30.330414 | 12.996519 | 1.130442  1.135223  1.142064   1.14715  1.147109 | 16.299204 |
    | 03/01 13:00 |  0.3 | 16.4 | 15.4 |  4.658915 |  17.0 | 17.0  17.0  17.0  17.0  17.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  0.3564   0.3564   0.3564   0.3564   0.3564 |  5.404341   5.404341   5.404341   5.404341   5.404341 | 3.754888  3.754888  3.754888  3.754888  3.754888 |   0.3564    0.3564  0.0  0.0    0.3564 |       0.0        0.0   0.3564   0.3564        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.442976  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  14.799376   0.3564        0.0 |       0.0        0.0  14.799376   0.3564        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.576688  3.576688  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  3.754888  0.0 |      0.0       0.0   9.090941  0.0  0.0 |      0.0       0.0  4.101943  0.0  0.0 |      0.0       0.0  1.251433  0.0  0.0 | 0.994789  0.994789  1.248638  0.0  0.0 | 0.187841 | 0.302729 |  0.02348 | 0.014446 |  3.781207 |  3.929987 |  3.929987 | 16.374948 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 161.885262  161.885262  0.0  0.0  0.0 |       0.0        0.0   29.91169  0.0  0.0 | 19.402521  19.402521  24.974142  0.0  0.0 | 29.738715 | 12.945952 | 1.097692  1.101661  1.108838  1.117946   1.12707 | 16.374948 |
    | 03/01 14:00 |  0.3 | 16.5 | 15.5 |  7.789594 |  17.1 | 17.1  17.1  17.1  17.1  17.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  0.3564   0.3564   0.3564   0.3564   0.3564 |  9.035929   9.035929   9.035929   9.035929   9.035929 | 6.278083  6.278083  6.278083  6.278083  6.278083 |   0.3564    0.3564  0.0  0.0    0.3564 |       0.0        0.0   0.3564   0.3564        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.533245  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  14.889645   0.3564        0.0 |       0.0        0.0  14.889645   0.3564        0.0 | 0.0  0.0  0.0  0.0        0.0 | 6.099883  6.099883  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  6.278083  0.0 |      0.0       0.0   8.971626  0.0  0.0 |      0.0       0.0  4.073085  0.0  0.0 |      0.0       0.0  1.251293  0.0  0.0 | 0.946272  0.946272  1.248771  0.0  0.0 | 0.187868 | 0.296841 | 0.023484 | 0.014389 |  3.716942 |  3.835159 |  3.835159 | 15.979829 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 155.785379  155.785379  0.0  0.0  0.0 |       0.0        0.0  29.756623  0.0  0.0 | 18.456249  18.456249  24.976664  0.0  0.0 | 28.832525 | 12.855394 |  1.07858  1.080936  1.085364  1.091467  1.098644 | 15.979829 |
    | 03/01 15:00 |  0.0 | 18.4 | 17.4 |  4.851567 |  19.0 | 19.0  19.0  19.0  19.0  19.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  5.627818   5.627818   5.627818   5.627818   5.627818 | 4.052029  4.052029  4.052029  4.052029  4.052029 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.248348  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.248348      0.0        0.0 |       0.0        0.0  16.248348      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.945293  3.945293  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  4.052029  0.0 |      0.0       0.0    9.44522  0.0  0.0 |      0.0       0.0  4.187627  0.0  0.0 |      0.0       0.0  1.251167  0.0  0.0 | 0.900122  0.900122  1.248891  0.0  0.0 | 0.142635 | 0.287599 | 0.017829 | 0.014286 |   3.79818 |   3.76084 |   3.76084 | 15.670168 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 151.840086  151.840086  0.0  0.0  0.0 |       0.0        0.0  30.372124  0.0  0.0 | 17.556127  17.556127  24.978941  0.0  0.0 | 28.173017 |  12.79462 | 1.100701  1.098208  1.094293  1.090535  1.088594 | 15.670168 |
    | 03/01 16:00 |  0.0 | 18.3 | 17.3 |   5.30692 |  18.9 | 18.9  18.9  18.9  18.9  18.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  6.156027   6.156027   6.156027   6.156027   6.156027 |  4.43234   4.43234   4.43234   4.43234   4.43234 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.158079  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.158079      0.0        0.0 |       0.0        0.0  16.158079      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 4.206293  4.206293  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0   4.43234  0.0 |      0.0       0.0   9.651883  0.0  0.0 |      0.0       0.0  4.237609  0.0  0.0 |      0.0       0.0  1.251053  0.0  0.0 | 0.856222  0.856222  1.248999  0.0  0.0 | 0.142657 | 0.281037 | 0.017832 | 0.014218 |  3.817003 |   3.77129 |   3.77129 | 15.713708 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 147.633793  147.633793  0.0  0.0  0.0 |       0.0        0.0  30.640711  0.0  0.0 | 16.699905  16.699905  24.980995  0.0  0.0 |   27.4718 | 12.727879 | 1.106721  1.105937  1.104347  1.101974  1.099065 | 15.713708 |
    | 03/01 17:00 |  0.0 | 18.1 | 17.1 |  3.286036 |  18.7 | 18.7  18.7  18.7  18.7  18.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  3.811802   3.811802   3.811802   3.811802   3.811802 | 2.744497  2.744497  2.744497  2.744497  2.744497 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.977542  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  15.977542      0.0        0.0 |       0.0        0.0  15.977542      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 2.532378  2.532378  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  2.744497  0.0 |      0.0       0.0   9.686528  0.0  0.0 |      0.0       0.0  4.245988  0.0  0.0 |      0.0       0.0   1.25095  0.0  0.0 | 0.814464  0.814464  1.249096  0.0  0.0 | 0.142676 |  0.27406 | 0.017835 | 0.014144 |  3.793991 |  3.801305 |  3.801305 |  15.83877 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 145.101415  145.101415  0.0  0.0  0.0 |       0.0        0.0  30.685737  0.0  0.0 | 15.885441  15.885441  24.982849  0.0  0.0 | 26.991909 | 12.688006 | 1.100475  1.101175  1.102256  1.103234   1.10359 |  15.83877 |
    | 03/01 18:00 |  0.0 | 16.7 | 15.7 |  1.506216 |  17.3 | 17.3  17.3  17.3  17.3  17.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  1.747211   1.747211   1.747211   1.747211   1.747211 | 1.257992  1.257992  1.257992  1.257992  1.257992 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.713782  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  14.713782      0.0        0.0 |       0.0        0.0  14.713782      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.140852  1.140852  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  1.257992  0.0 |      0.0       0.0   9.206993  0.0  0.0 |      0.0       0.0   4.13001  0.0  0.0 |      0.0       0.0  1.250858  0.0  0.0 | 0.774742  0.774742  1.249185  0.0  0.0 | 0.142694 | 0.269285 | 0.017837 |   0.0141 |  3.646576 |  3.791031 |  3.791031 | 15.795961 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 143.960563  143.960563  0.0  0.0  0.0 |       0.0        0.0  30.062516  0.0  0.0 | 15.110699  15.110699  24.984522  0.0  0.0 | 26.705572 | 12.671775 | 1.058955  1.063806  1.072047  1.081543  1.089924 | 15.795961 |
    | 03/01 19:00 |  0.0 | 15.2 | 14.2 |  0.274762 |  15.8 | 15.8  15.8  15.8  15.8  15.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  0.318724   0.318724   0.318724   0.318724   0.318724 | 0.229481  0.229481  0.229481  0.229481  0.229481 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.359753  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  13.359753      0.0        0.0 |       0.0        0.0  13.359753      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.206477  0.206477  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.229481  0.0 |      0.0       0.0   8.429006  0.0  0.0 |      0.0       0.0   3.94185  0.0  0.0 |      0.0       0.0  1.250774  0.0  0.0 | 0.736957  0.736957  1.249264  0.0  0.0 |  0.14271 | 0.266436 | 0.017839 | 0.014082 |  3.428015 |  3.695715 |  3.695715 | 15.398814 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 143.754086  143.754086  0.0  0.0  0.0 |       0.0        0.0  29.051413  0.0  0.0 | 14.373741  14.373741  24.986031  0.0  0.0 | 26.552706 | 12.671889 | 0.996309  1.003756  1.016839  1.032898  1.048772 | 15.398814 |
    | 03/01 20:00 |  0.0 | 13.4 | 12.4 |  0.087565 |  14.0 | 14.0  14.0  14.0  14.0  14.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  0.101575   0.101575   0.101575   0.101575   0.101575 | 0.073134  0.073134  0.073134  0.073134  0.073134 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  11.734918  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  11.734918      0.0        0.0 |       0.0        0.0  11.734918      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.065708  0.065708  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.073134  0.0 |      0.0       0.0   7.391845  0.0  0.0 |      0.0       0.0  3.691007  0.0  0.0 |      0.0       0.0  1.250698  0.0  0.0 | 0.701016  0.701016  1.249336  0.0  0.0 | 0.142724 | 0.264915 | 0.017841 | 0.014082 |  3.147444 |  3.513928 |  3.513928 | 14.641365 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 143.688378  143.688378  0.0  0.0  0.0 |       0.0        0.0  27.703478  0.0  0.0 | 13.672726  13.672726  24.987394  0.0  0.0 | 26.421228 | 12.674487 | 0.915575  0.925212   0.94229  0.963617  0.985397 | 14.641365 |
    | 03/01 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0  13.0  13.0  13.0  13.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  0.099494   0.099494   0.099494   0.099494   0.099494 | 0.071636  0.071636  0.071636  0.071636  0.071636 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.832232  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.832232      0.0        0.0 |       0.0        0.0  10.832232      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.064333  0.064333  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.071636  0.0 |      0.0       0.0   6.506295  0.0  0.0 |      0.0       0.0  3.476833  0.0  0.0 |      0.0       0.0   1.25063  0.0  0.0 | 0.666827  0.666827  1.249401  0.0  0.0 | 0.142737 | 0.263607 | 0.017842 | 0.014085 |  2.905781 |  3.269809 |  3.269809 | 13.624205 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 143.624045  143.624045  0.0  0.0  0.0 |       0.0        0.0  26.552583  0.0  0.0 | 13.005899  13.005899  24.988623  0.0  0.0 | 26.291261 | 12.677107 | 0.845133  0.853648  0.869092  0.889177  0.911013 | 13.624205 |
    | 03/01 22:00 |  0.0 | 11.6 | 10.6 |  0.084317 |  12.2 | 12.2  12.2  12.2  12.2  12.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  0.097808   0.097808   0.097808   0.097808   0.097808 | 0.070422  0.070422  0.070422  0.070422  0.070422 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.110083  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.110083      0.0        0.0 |       0.0        0.0  10.110083      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.063214  0.063214  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.070422  0.0 |      0.0       0.0   5.769309  0.0  0.0 |      0.0       0.0  3.298589  0.0  0.0 |      0.0       0.0  1.250569  0.0  0.0 | 0.634305  0.634305  1.249459  0.0  0.0 | 0.142749 | 0.262314 | 0.017844 | 0.014088 |  2.702029 |  3.020139 |  3.020139 | 12.583912 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 143.560831  143.560831  0.0  0.0  0.0 |       0.0        0.0  25.594768  0.0  0.0 | 12.371594  12.371594  24.989733  0.0  0.0 | 26.162754 | 12.679745 | 0.785663  0.792864   0.80598  0.823193  0.842253 | 12.583912 |
    | 03/01 23:00 |  0.0 | 11.0 | 10.0 |  0.083215 |  11.6 | 11.6  11.6  11.6  11.6  11.6 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  0.096529   0.096529   0.096529   0.096529   0.096529 | 0.069501  0.069501  0.069501  0.069501  0.069501 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   9.568472  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   9.568472      0.0        0.0 |       0.0        0.0   9.568472      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 |  0.06236   0.06236  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.069501  0.0 |      0.0       0.0   5.179331  0.0  0.0 |      0.0       0.0  3.155899  0.0  0.0 |      0.0       0.0  1.250513  0.0  0.0 |  0.60337   0.60337  1.249512  0.0  0.0 | 0.142759 | 0.261035 | 0.017845 | 0.014091 |  2.535754 |  2.800287 |  2.800287 | 11.667862 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 143.498471  143.498471  0.0  0.0  0.0 |       0.0        0.0  24.828009  0.0  0.0 | 11.768224  11.768224  24.990735  0.0  0.0 | 26.035653 | 12.682396 | 0.737064  0.742957  0.753719  0.767914  0.783767 | 11.667862 |
    | 04/01 00:00 |  0.0 | 10.5 |  9.5 |  0.082289 |  11.1 | 11.1  11.1  11.1  11.1  11.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  0.095455   0.095455   0.095455   0.095455   0.095455 | 0.068728  0.068728  0.068728  0.068728  0.068728 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   9.121265  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   9.121265      0.0        0.0 |       0.0        0.0   9.121265      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 |  0.06164   0.06164  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.068728  0.0 |      0.0       0.0   4.701673  0.0  0.0 |      0.0       0.0  3.040375  0.0  0.0 |      0.0       0.0  1.250463  0.0  0.0 | 0.573943  0.573943   1.24956  0.0  0.0 | 0.142769 | 0.259771 | 0.017846 | 0.014094 |  2.398294 |  2.616572 |  2.616572 | 10.902382 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 143.436831  143.436831  0.0  0.0  0.0 |       0.0        0.0  24.207225  0.0  0.0 | 11.194281  11.194281  24.991638  0.0  0.0 | 25.909924 | 12.685058 | 0.696906  0.701773  0.710658  0.722367  0.735439 | 10.902382 |
    | 04/01 01:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3  12.3  12.3  12.3  12.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |   0.09802    0.09802    0.09802    0.09802    0.09802 | 0.070574  0.070574  0.070574  0.070574  0.070574 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   10.20498  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   10.20498      0.0        0.0 |       0.0        0.0   10.20498      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.063269  0.063269  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.070574  0.0 |      0.0       0.0   4.883822  0.0  0.0 |      0.0       0.0  3.084429  0.0  0.0 |      0.0       0.0  1.250418  0.0  0.0 | 0.545952  0.545952  1.249602  0.0  0.0 | 0.142778 |  0.25852 | 0.017847 | 0.014097 |  2.425583 |  2.475039 |  2.475039 | 10.312665 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 143.373563  143.373563  0.0  0.0  0.0 |       0.0        0.0  24.443954  0.0  0.0 |  10.64833   10.64833  24.992454  0.0  0.0 |  25.78522 | 12.687688 |   0.7032  0.702632  0.702247  0.703164  0.706445 | 10.312665 |
    | 04/01 02:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5  12.5  12.5  12.5  12.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  0.098442   0.098442   0.098442   0.098442   0.098442 | 0.070878  0.070878  0.070878  0.070878  0.070878 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.385599  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.385599      0.0        0.0 |       0.0        0.0  10.385599      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.063513  0.063513  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.070878  0.0 |      0.0       0.0   5.048036  0.0  0.0 |      0.0       0.0  3.124145  0.0  0.0 |      0.0       0.0  1.250377  0.0  0.0 | 0.519325  0.519325  1.249641  0.0  0.0 | 0.142785 | 0.257279 | 0.017848 |   0.0141 |  2.449246 |  2.428785 |  2.428785 | 10.119939 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  143.31005   143.31005  0.0  0.0  0.0 |       0.0        0.0  24.657372  0.0  0.0 | 10.129004  10.129004   24.99319  0.0  0.0 | 25.661726 | 12.690312 | 0.710043  0.709228  0.707812  0.706174  0.704891 | 10.119939 |
    | 04/01 03:00 |  1.3 | 11.2 | 10.2 |  0.083584 |  11.8 | 11.8  11.8  11.8  11.8  11.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |  1.5444   1.5444   1.5444   1.5444   1.5444 |  0.096957   0.096957   0.096957   0.096957   0.096957 | 0.059819  0.059819  0.059819  0.059819  0.059819 | 0.059819  0.059819  0.0  0.0  0.059819 |       0.0        0.0   1.5444   1.5444        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   9.753432  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  11.297832   1.5444        0.0 |       0.0        0.0  11.297832   1.5444        0.0 | 0.0  0.0  0.0  0.0        0.0 |  0.02679   0.02679  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.059819  0.0 |      0.0       0.0   5.490945  0.0  0.0 |      0.0       0.0  3.231265  0.0  0.0 |      0.0       0.0   1.25034  0.0  0.0 | 0.493997  0.493997  1.249676  0.0  0.0 | 0.338907 | 0.257027 | 0.042363 | 0.014116 |  2.543843 |  2.445102 |  2.445102 | 10.187926 | 1.484581  1.484581  0.0  0.0  1.484581 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  143.28326   143.28326  0.0  0.0  0.0 |       0.0        0.0  25.232994  0.0  0.0 |  9.635007   9.635007  24.993854  0.0  0.0 | 25.736009 | 12.717609 | 0.736782  0.733646  0.728273  0.721974  0.716214 | 10.187926 |
    | 04/01 04:00 |  0.0 | 11.1 | 10.1 |    0.0834 |  11.7 | 11.7  11.7  11.7  11.7  11.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  0.096744   0.096744   0.096744   0.096744   0.096744 | 0.069656  0.069656  0.069656  0.069656  0.069656 | 0.069656  0.069656  0.0  0.0  0.069656 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   9.663122  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   9.663122      0.0        0.0 |       0.0        0.0   9.663122      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.031189  0.031189  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.069656  0.0 |      0.0       0.0   5.074226  0.0  0.0 |      0.0       0.0  3.130479  0.0  0.0 |      0.0       0.0  1.250307  0.0  0.0 | 0.469905  0.469905  1.249708  0.0  0.0 | 0.142799 | 0.256789 |  0.01785 | 0.014133 |  2.425686 |  2.492833 |  2.492833 | 10.386802 | 1.414925  1.414925  0.0  0.0  1.414925 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 143.252071  143.252071  0.0  0.0  0.0 |       0.0        0.0  24.691411  0.0  0.0 |  9.165102   9.165102  24.994454  0.0  0.0 | 25.613173 | 12.720221 | 0.704572  0.708211  0.713982  0.719717  0.723259 | 10.386802 |
    | 04/01 05:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5  12.5  12.5  12.5  12.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  0.098442   0.098442   0.098442   0.098442   0.098442 | 0.070878  0.070878  0.070878  0.070878  0.070878 | 0.070878  0.070878  0.0  0.0  0.070878 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.385599  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.385599      0.0        0.0 |       0.0        0.0  10.385599      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 |  0.03173   0.03173  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.070878  0.0 |      0.0       0.0   5.145402  0.0  0.0 |      0.0       0.0  3.147694  0.0  0.0 |      0.0       0.0  1.250277  0.0  0.0 | 0.446987  0.446987  1.249736  0.0  0.0 | 0.142804 | 0.255567 | 0.017851 | 0.014136 |  2.428481 |  2.468368 |  2.468368 | 10.284865 | 1.344047  1.344047  0.0  0.0  1.344047 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 143.220341  143.220341  0.0  0.0  0.0 |       0.0        0.0  24.783913  0.0  0.0 |  8.718115   8.718115  24.994995  0.0  0.0 |  25.49141 | 12.722811 |  0.70427  0.704428   0.70509  0.706687   0.70938 | 10.284865 |
    | 04/01 06:00 |  0.0 | 12.2 | 11.2 |  0.310229 |  12.8 | 12.8  12.8  12.8  12.8  12.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  0.359866   0.359866   0.359866   0.359866   0.359866 | 0.259103  0.259103  0.259103  0.259103  0.259103 | 0.259103  0.259103  0.0  0.0  0.259103 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.656527  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.656527      0.0        0.0 |       0.0        0.0  10.656527      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.115965  0.115965  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.259103  0.0 |      0.0       0.0   5.288401  0.0  0.0 |      0.0       0.0  3.182279  0.0  0.0 |      0.0       0.0   1.25025  0.0  0.0 | 0.425187  0.425187  1.249762  0.0  0.0 | 0.142809 | 0.254355 | 0.017851 | 0.014139 |  2.449795 |  2.437654 |  2.437654 | 10.156893 | 1.084943  1.084943  0.0  0.0  1.084943 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 143.104376  143.104376  0.0  0.0  0.0 |       0.0        0.0  24.969761  0.0  0.0 |  8.292927   8.292927  24.995483  0.0  0.0 | 25.346962 | 12.722411 | 0.710233  0.709545  0.708418  0.707253  0.706548 | 10.156893 |
    | 04/01 07:00 |  0.7 | 11.8 | 10.8 |  1.391958 |  12.4 | 12.4  12.4  12.4  12.4  12.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |  0.8316   0.8316   0.8316   0.8316   0.8316 |  1.614671   1.614671   1.614671   1.614671   1.614671 | 1.069795  1.069795  1.069795  1.069795  1.069795 | 1.069795  1.069795  0.0  0.0  1.069795 |       0.0        0.0   0.8316   0.8316        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.295289  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  11.126889   0.8316        0.0 |       0.0        0.0  11.126889   0.8316        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.478414  0.478414  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  1.069795  0.0 |      0.0       0.0     5.5466  0.0  0.0 |      0.0       0.0  3.244725  0.0  0.0 |      0.0       0.0  1.250226  0.0  0.0 | 0.404451  0.404451  1.249785  0.0  0.0 | 0.248414 | 0.253444 | 0.031052 | 0.014145 |  2.500643 |  2.445137 |  2.445137 | 10.188072 | 0.846748  0.846748  0.0  0.0  0.846748 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 142.625962  142.625962  0.0  0.0  0.0 |       0.0        0.0  25.305325  0.0  0.0 |  7.888477   7.888477  24.995924  0.0  0.0 | 25.206085 | 12.722336 | 0.724682  0.722979  0.720036  0.716535  0.713269 | 10.188072 |
    | 04/01 08:00 |  0.4 | 11.4 | 10.4 |  3.195876 |  12.0 | 12.0  12.0  12.0  12.0  12.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |  0.4752   0.4752   0.4752   0.4752   0.4752 |  3.707216   3.707216   3.707216   3.707216   3.707216 | 2.545322  2.545322  2.545322  2.545322  2.545322 | 1.321948  1.321948  0.0  0.0  1.321948 |       0.0        0.0   0.4752   0.4752        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   9.934051  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.409251   0.4752        0.0 |       0.0        0.0  10.409251   0.4752        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.746153  1.746153  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  2.545322  0.0 |      0.0       0.0   5.396265  0.0  0.0 |      0.0       0.0  3.208366  0.0  0.0 |      0.0       0.0  1.250204  0.0  0.0 | 0.384726  0.384726  1.249806  0.0  0.0 | 0.203161 | 0.251817 | 0.025395 | 0.014142 |  2.449952 |  2.473568 |  2.473568 | 10.306534 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  140.87981   140.87981  0.0  0.0  0.0 |       0.0        0.0  25.109945  0.0  0.0 |  7.503751   7.503751  24.996321  0.0  0.0 | 24.834213 | 12.693187 | 0.710964  0.712502  0.714896  0.717168  0.718355 | 10.306534 |
    | 04/01 09:00 |  0.1 | 11.6 | 10.6 |  5.191651 |  12.2 | 12.2  12.2  12.2  12.2  12.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |  0.1188   0.1188   0.1188   0.1188   0.1188 |  6.022315   6.022315   6.022315   6.022315   6.022315 | 4.284859  4.284859  4.284859  4.284859  4.284859 |   0.1188    0.1188  0.0  0.0    0.1188 |       0.0        0.0   0.1188   0.1188        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   10.11467  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   10.23347   0.1188        0.0 |       0.0        0.0   10.23347   0.1188        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.772813  3.772813  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  4.284859  0.0 |      0.0       0.0   5.250225  0.0  0.0 |      0.0       0.0  3.173045  0.0  0.0 |      0.0       0.0  1.250184  0.0  0.0 | 0.365962  0.365962  1.249825  0.0  0.0 | 0.157908 | 0.247891 | 0.019738 | 0.014107 |  2.398728 |  2.463573 |  2.463573 | 10.264887 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 137.106996  137.106996  0.0  0.0  0.0 |       0.0        0.0  24.920145  0.0  0.0 |  7.137789   7.137789   24.99668  0.0  0.0 | 24.200121 | 12.630805 | 0.696148  0.697922  0.701075  0.704998  0.708897 | 10.264887 |
    | 04/01 10:00 |  0.4 | 13.0 | 12.0 |  7.155036 |  13.6 | 13.6  13.6  13.6  13.6  13.6 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |  0.4752   0.4752   0.4752   0.4752   0.4752 |  8.299842   8.299842   8.299842   8.299842   8.299842 | 5.698554  5.698554  5.698554  5.698554  5.698554 |   0.4752    0.4752  0.0  0.0    0.4752 |       0.0        0.0   0.4752   0.4752        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  11.379004  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  11.854204   0.4752        0.0 |       0.0        0.0  11.854204   0.4752        0.0 | 0.0  0.0  0.0  0.0        0.0 | 4.883197  4.883197  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  5.698554  0.0 |      0.0       0.0   5.813253  0.0  0.0 |      0.0       0.0  3.309217  0.0  0.0 |      0.0       0.0  1.250166  0.0  0.0 | 0.348114  0.348114  1.249842  0.0  0.0 | 0.203168 | 0.241808 | 0.025396 | 0.014041 |  2.522123 |  2.433143 |  2.433143 | 10.138095 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 132.223799  132.223799  0.0  0.0  0.0 |       0.0        0.0  25.651879  0.0  0.0 |  6.789675   6.789675  24.997004  0.0  0.0 | 23.437855 | 12.551708 | 0.730227  0.726325  0.719961  0.713224  0.708284 | 10.138095 |
    | 04/01 11:00 |  0.0 | 17.1 | 16.1 |  8.391432 |  17.7 | 17.7  17.7  17.7  17.7  17.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  9.734061   9.734061   9.734061   9.734061   9.734061 | 7.008524  7.008524  7.008524  7.008524  7.008524 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.081695  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  15.081695      0.0        0.0 |       0.0        0.0  15.081695      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 5.791835  5.791835  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  7.008524  0.0 |      0.0       0.0   7.371087  0.0  0.0 |      0.0       0.0  3.685987  0.0  0.0 |      0.0       0.0   1.25015  0.0  0.0 | 0.331136  0.331136  1.249858  0.0  0.0 | 0.142829 | 0.233922 | 0.017854 | 0.013949 |  2.891414 |   2.50037 |   2.50037 | 10.418208 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 126.431964  126.431964  0.0  0.0  0.0 |       0.0        0.0    27.6765  0.0  0.0 |  6.458538   6.458538  24.997296  0.0  0.0 |  22.45679 | 12.444366 | 0.834861  0.822564  0.801433  0.776545  0.753661 | 10.418208 |
    | 04/01 12:00 |  0.0 | 18.2 | 17.2 |  8.391286 |  18.8 | 18.8  18.8  18.8  18.8  18.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  9.733892   9.733892   9.733892   9.733892   9.733892 | 7.008402  7.008402  7.008402  7.008402  7.008402 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.075101  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.075101      0.0        0.0 |       0.0        0.0  16.075101      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 5.538038  5.538038  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  7.008402  0.0 |      0.0       0.0   8.558588  0.0  0.0 |      0.0       0.0   3.97319  0.0  0.0 |      0.0       0.0  1.250135  0.0  0.0 | 0.314987  0.314987  1.249871  0.0  0.0 | 0.142831 | 0.224161 | 0.017854 | 0.013829 |  3.167446 |  2.743406 |  2.743406 |  11.43086 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 120.893926  120.893926  0.0  0.0  0.0 |       0.0        0.0  29.219823  0.0  0.0 |  6.143552   6.143552   24.99756  0.0  0.0 | 21.485505 | 12.337146 | 0.915738  0.905921  0.887999  0.864521  0.838925 |  11.43086 |
    | 04/01 13:00 |  0.0 | 22.4 | 21.4 | 10.715238 |  23.0 | 23.0  23.0  23.0  23.0  23.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 | 12.429676  12.429676  12.429676  12.429676  12.429676 | 8.949367  8.949367  8.949367  8.949367  8.949367 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  19.868102  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  19.868102      0.0        0.0 |       0.0        0.0  19.868102      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 6.762026  6.762026  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  8.949367  0.0 |      0.0       0.0  10.658268  0.0  0.0 |      0.0       0.0  4.481008  0.0  0.0 |      0.0       0.0  1.250122  0.0  0.0 | 0.299625  0.299625  1.249884  0.0  0.0 | 0.142834 | 0.214496 | 0.017854 |  0.01371 |  3.670599 |  3.053501 |  3.053501 |  12.72292 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   114.1319    114.1319  0.0  0.0  0.0 |       0.0        0.0  31.948648  0.0  0.0 |  5.843927   5.843927  24.997798  0.0  0.0 | 20.277416 | 12.199237 | 1.059461   1.04242  1.012585  0.976046   0.93969 |  12.72292 |
    | 04/01 14:00 |  0.0 | 21.4 | 20.4 |  9.383394 |  22.0 | 22.0  22.0  22.0  22.0  22.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 | 10.884737  10.884737  10.884737  10.884737  10.884737 | 7.837011  7.837011  7.837011  7.837011  7.837011 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  18.965006  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  18.965006      0.0        0.0 |       0.0        0.0  18.965006      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 5.590331  5.590331  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  7.837011  0.0 |      0.0       0.0  11.376637  0.0  0.0 |      0.0       0.0   4.65475  0.0  0.0 |      0.0       0.0   1.25011  0.0  0.0 | 0.285012  0.285012  1.249895  0.0  0.0 | 0.142836 | 0.202475 | 0.017855 | 0.013557 |  3.828894 |  3.440038 |  3.440038 | 14.333491 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  108.54157   108.54157  0.0  0.0  0.0 |       0.0        0.0  32.882268  0.0  0.0 |  5.558915   5.558915  24.998013  0.0  0.0 |   19.2226 | 12.079137 | 1.108663  1.102368  1.089819    1.0711  1.047108 | 14.333491 |
    | 04/01 15:00 |  0.0 | 21.8 | 20.8 |  7.861915 |  22.4 | 22.4  22.4  22.4  22.4  22.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  9.119821   9.119821   9.119821   9.119821   9.119821 | 6.566271  6.566271  6.566271  6.566271  6.566271 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  19.326244  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  19.326244      0.0        0.0 |       0.0        0.0  19.326244      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 4.454459  4.454459  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  6.566271  0.0 |      0.0       0.0  11.886124  0.0  0.0 |      0.0       0.0  4.777972  0.0  0.0 |      0.0       0.0  1.250099  0.0  0.0 | 0.271112  0.271112  1.249906  0.0  0.0 | 0.142838 |  0.19198 | 0.017855 | 0.013424 |  3.937177 |  3.724206 |  3.724206 | 15.517524 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 104.087111  104.087111  0.0  0.0  0.0 |       0.0        0.0  33.544417  0.0  0.0 |  5.287804   5.287804  24.998206  0.0  0.0 | 18.339646 | 11.979342 | 1.140665  1.136732  1.129339   1.11901  1.106283 | 15.517524 |
    | 04/01 16:00 |  0.0 | 22.2 | 21.2 |  6.298329 |  22.8 | 22.8  22.8  22.8  22.8  22.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  7.306062   7.306062   7.306062   7.306062   7.306062 | 5.260364  5.260364  5.260364  5.260364  5.260364 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  19.687483  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  19.687483      0.0        0.0 |       0.0        0.0  19.687483      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.422101  3.422101  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  5.260364  0.0 |      0.0       0.0  12.288795  0.0  0.0 |      0.0       0.0   4.87536  0.0  0.0 |      0.0       0.0   1.25009  0.0  0.0 | 0.257889  0.257889  1.249915  0.0  0.0 |  0.14284 | 0.183194 | 0.017855 | 0.013313 |  4.020954 |  3.878565 |  3.878565 | 16.160688 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  100.66501   100.66501  0.0  0.0  0.0 |       0.0        0.0  34.067745  0.0  0.0 |  5.029915   5.029915  24.998381  0.0  0.0 | 17.631309 | 11.900386 |  1.16522  1.162233  1.156736  1.149367  1.140862 | 16.160688 |
    | 04/01 17:00 |  0.0 | 20.1 | 19.1 |  2.948416 |  20.7 | 20.7  20.7  20.7  20.7  20.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  3.420163   3.420163   3.420163   3.420163   3.420163 | 2.462517  2.462517  2.462517  2.462517  2.462517 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  17.790982  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  17.790982      0.0        0.0 |       0.0        0.0  17.790982      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.549308  1.549308  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  2.462517  0.0 |      0.0       0.0  11.748494  0.0  0.0 |      0.0       0.0  4.744685  0.0  0.0 |      0.0       0.0  1.250081  0.0  0.0 | 0.245312  0.245312  1.249923  0.0  0.0 | 0.142842 | 0.176146 | 0.017855 | 0.013225 |  3.872554 |  3.963557 |  3.963557 | 16.514823 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.115702   99.115702  0.0  0.0  0.0 |       0.0        0.0  33.365548  0.0  0.0 |  4.784603   4.784603  24.998539  0.0  0.0 | 17.285304 | 11.865929 | 1.124462    1.1291  1.136556  1.144166  1.149131 | 16.514823 |
    | 04/01 18:00 |  0.0 | 17.8 | 16.8 |  1.309232 |  18.4 | 18.4  18.4  18.4  18.4  18.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  1.518709   1.518709   1.518709   1.518709   1.518709 | 1.093471  1.093471  1.093471  1.093471  1.093471 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.713862  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  15.713862      0.0        0.0 |       0.0        0.0  15.713862      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.677376  0.677376  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  1.093471  0.0 |      0.0       0.0  10.654918  0.0  0.0 |      0.0       0.0  4.480198  0.0  0.0 |      0.0       0.0  1.250073  0.0  0.0 | 0.233348  0.233348  1.249931  0.0  0.0 | 0.142843 | 0.172703 | 0.017855 | 0.013187 |  3.590516 |  3.907767 |  3.907767 | 16.282363 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  98.438326   98.438326  0.0  0.0  0.0 |       0.0        0.0  31.944295  0.0  0.0 |  4.551255   4.551255  24.998682  0.0  0.0 | 17.116591 |  11.85324 | 1.044054  1.053559  1.070068  1.089874  1.108609 | 16.282363 |
    | 04/01 19:00 |  0.0 | 15.2 | 14.2 |   0.32955 |  15.8 | 15.8  15.8  15.8  15.8  15.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  0.382278   0.382278   0.382278   0.382278   0.382278 |  0.27524   0.27524   0.27524   0.27524   0.27524 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.365814  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  13.365814      0.0        0.0 |       0.0        0.0  13.365814      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.169339  0.169339  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0   0.27524  0.0 |      0.0       0.0   9.171813  0.0  0.0 |      0.0       0.0  4.121502  0.0  0.0 |      0.0       0.0  1.250066  0.0  0.0 | 0.221967  0.221967  1.249937  0.0  0.0 | 0.142845 | 0.171025 | 0.017856 | 0.013173 |  3.213749 |  3.695687 |  3.695687 | 15.398694 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  98.268988   98.268988  0.0  0.0  0.0 |       0.0        0.0  30.016794  0.0  0.0 |  4.329288   4.329288   24.99881  0.0  0.0 | 17.053459 | 11.853554 | 0.935764  0.948675  0.971492  0.999831  1.028464 | 15.398694 |
    | 04/01 20:00 |  0.0 | 14.5 | 13.5 |  0.089508 |  15.1 | 15.1  15.1  15.1  15.1  15.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  0.103829   0.103829   0.103829   0.103829   0.103829 | 0.074757  0.074757  0.074757  0.074757  0.074757 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  12.733647  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  12.733647      0.0        0.0 |       0.0        0.0  12.733647      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.045914  0.045914  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.074757  0.0 |      0.0       0.0   8.164662  0.0  0.0 |      0.0       0.0  3.877917  0.0  0.0 |      0.0       0.0  1.250059  0.0  0.0 | 0.211142  0.211142  1.249943  0.0  0.0 | 0.142846 | 0.170397 | 0.017856 | 0.013173 |  2.956522 |  3.381337 |  3.381337 | 14.088906 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  98.223073   98.223073  0.0  0.0  0.0 |       0.0        0.0  28.707861  0.0  0.0 |  4.118146   4.118146  24.998926  0.0  0.0 | 17.016415 |  11.85705 | 0.860033  0.869273  0.886313  0.909071   0.93472 | 14.088906 |
    | 04/01 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0  13.0  13.0  13.0  13.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  0.099494   0.099494   0.099494   0.099494   0.099494 | 0.071636  0.071636  0.071636  0.071636  0.071636 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.837146  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.837146      0.0        0.0 |       0.0        0.0  10.837146      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.043977  0.043977  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.071636  0.0 |      0.0       0.0   6.903423  0.0  0.0 |      0.0       0.0   3.57288  0.0  0.0 |      0.0       0.0  1.250054  0.0  0.0 | 0.200844  0.200844  1.249949  0.0  0.0 | 0.142847 | 0.170028 | 0.017856 | 0.013177 |  2.636748 |  3.077836 |  3.077836 | 12.824317 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  98.179096   98.179096  0.0  0.0  0.0 |       0.0        0.0  27.068705  0.0  0.0 |  3.917302   3.917302  24.999031  0.0  0.0 | 16.980138 | 11.860592 | 0.767872  0.778898  0.798547  0.823418  0.849588 | 12.824317 |
    | 04/01 22:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3  12.3  12.3  12.3  12.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |   0.09802    0.09802    0.09802    0.09802    0.09802 | 0.070574  0.070574  0.070574  0.070574  0.070574 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   10.20498  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   10.20498      0.0        0.0 |       0.0        0.0   10.20498      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.043306  0.043306  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.070574  0.0 |      0.0       0.0   6.009727  0.0  0.0 |      0.0       0.0  3.356735  0.0  0.0 |      0.0       0.0  1.250048  0.0  0.0 | 0.191049  0.191049  1.249954  0.0  0.0 | 0.142848 | 0.169667 | 0.017856 | 0.013181 |  2.408571 |  2.782512 |  2.782512 |   11.5938 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   98.13579    98.13579  0.0  0.0  0.0 |       0.0        0.0  25.907223  0.0  0.0 |  3.726252   3.726252  24.999126  0.0  0.0 | 16.944357 | 11.864146 | 0.700824  0.708991  0.724014  0.744016  0.766536 |   11.5938 |
    | 04/01 23:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5  12.5  12.5  12.5  12.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |     0.0      0.0      0.0      0.0      0.0 |  0.098442   0.098442   0.098442   0.098442   0.098442 | 0.070878  0.070878  0.070878  0.070878  0.070878 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.385599  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.385599      0.0        0.0 |       0.0        0.0  10.385599      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.043473  0.043473  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.070878  0.0 |      0.0       0.0   5.623787  0.0  0.0 |      0.0       0.0  3.263393  0.0  0.0 |      0.0       0.0  1.250044  0.0  0.0 | 0.181731  0.181731  1.249958  0.0  0.0 | 0.142849 | 0.169311 | 0.017856 | 0.013185 |  2.306796 |  2.528272 |  2.528272 | 10.534467 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  98.092317   98.092317  0.0  0.0  0.0 |       0.0        0.0  25.405641  0.0  0.0 |  3.544521   3.544521  24.999211  0.0  0.0 | 16.908894 | 11.867692 | 0.670044  0.673898  0.681346  0.692079  0.705537 | 10.534467 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_v3_snow_redistribution:

snow redistribution
___________________

Regarding snow redistribution, all functionalities of |hland_v3| are identical with the
ones of |hland_v1|.  Hence, all snow data in the following table agrees with the
:ref:`hland_v1_snow_redistribution_1` example of |hland_v1|:

.. integration-test::

    >>> zonez(0.0, 10.0, 30.0, 0.0, 20.0)
    >>> smax(200.0)
    >>> sred(n_zones=1)
    >>> inputs.t.series = 5.0
    >>> test("hland_v3_snow_redistribution",
    ...      axis1=(states.sp, states.wc), axis2=(factors.tc, fluxes.pc))
    |        date |    p |   t |   tn |       epn | tmean |                         tc |                     fracrain |                      rfc |                      sfc |                                       cfact |                                                 swe |                          gact |                                             pc |                                               ep |                                              epc |                                     ei |                                              tf |                                                 spl |                                             wcl |                                                 spg |                                             wcg |                     glmelt |                          melt |                           refr |                                    in_ |                                     r |                      sr |                      ea |                                dp |                           el |                      rs |                                ri |                               gr1 |                                    rg1 |      gr2 |      rg2 |      gr3 |      rg3 |     inuh |    outuh |       rt |       qt |                                     ic |                                                  sp |                                                  wc |                                    sm |                               suz |                                     sg1 |       sg2 |       sg3 |                                               sc |   outlet |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | 5.0 | 20.2 |  0.100707 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.487706  0.487706  0.487706  0.0  0.0 |      0.0 | 0.099502 |      0.0 | 0.011105 | 0.493397 | 0.195326 | 0.195326 | 0.813856 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  9.512294  9.512294  9.512294  0.0  0.0 |  9.900498 |  9.988895 | 0.139948   0.12949  0.111887  0.091936  0.074811 | 0.813856 |
    | 01/01 01:00 |  0.0 | 5.0 | 18.4 |  0.097801 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  0.46392   0.46392   0.46392  0.0  0.0 |      0.0 | 0.098512 |      0.0 | 0.011093 | 0.473433 | 0.354617 | 0.354617 | 1.477572 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  9.048374  9.048374  9.048374  0.0  0.0 |  9.801987 |  9.977802 | 0.137385  0.137331  0.136064  0.132005  0.124104 | 1.477572 |
    | 01/01 02:00 |  0.0 | 5.0 | 17.9 |  0.096981 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.441294  0.441294  0.441294  0.0  0.0 |      0.0 | 0.097531 |      0.0 |  0.01108 | 0.454406 |  0.45403 |  0.45403 | 1.891792 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |   8.60708   8.60708   8.60708  0.0  0.0 |  9.704455 |  9.966722 | 0.131967  0.132595  0.133607  0.134519  0.134577 | 1.891792 |
    | 01/01 03:00 |  0.0 | 5.0 | 17.3 |   0.09599 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.419772  0.419772  0.419772  0.0  0.0 |      0.0 | 0.096561 |      0.0 | 0.011068 | 0.436271 | 0.459792 | 0.459792 | 1.915798 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  8.187308  8.187308  8.187308  0.0  0.0 |  9.607894 |  9.955654 | 0.126702  0.127336  0.128471  0.129901  0.131334 | 1.915798 |
    | 01/01 04:00 |  0.0 | 5.0 | 17.9 |  0.096981 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |   0.3993    0.3993    0.3993  0.0  0.0 |      0.0 |   0.0956 |      0.0 | 0.011056 | 0.418985 | 0.444294 | 0.444294 | 1.851225 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  7.788008  7.788008  7.788008  0.0  0.0 |  9.512294 |  9.944598 | 0.121681  0.122286  0.123378  0.124787  0.126303 | 1.851225 |
    | 01/01 05:00 |  0.0 | 5.0 | 21.5 |  0.102761 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.379826  0.379826  0.379826  0.0  0.0 |      0.0 | 0.094649 |      0.0 | 0.011043 | 0.402507 |  0.42689 |  0.42689 | 1.778708 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  7.408182  7.408182  7.408182  0.0  0.0 |  9.417645 |  9.933555 | 0.116894  0.117471  0.118512  0.119859  0.121316 | 1.778708 |
    | 01/01 06:00 |  0.0 | 5.0 | 24.1 |  0.291908 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.361301  0.361301  0.361301  0.0  0.0 |      0.0 | 0.093707 |      0.0 | 0.011031 | 0.386797 | 0.410061 | 0.410061 | 1.708587 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  7.046881  7.046881  7.046881  0.0  0.0 |  9.323938 |  9.922524 |  0.11233   0.11288  0.113873  0.115157  0.116547 | 1.708587 |
    | 01/01 07:00 |  0.0 | 5.0 | 27.3 |  1.932875 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  0.34368   0.34368   0.34368  0.0  0.0 |      0.0 | 0.092775 |      0.0 | 0.011019 | 0.371818 |    0.394 |    0.394 | 1.641665 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |    6.7032    6.7032    6.7032  0.0  0.0 |  9.231163 |  9.911505 | 0.107979  0.108504   0.10945  0.110674  0.111999 | 1.641665 |
    | 01/01 08:00 |  0.0 | 5.0 | 26.8 |  4.369536 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.326919  0.326919  0.326919  0.0  0.0 |      0.0 | 0.091852 |      0.0 | 0.011007 | 0.357536 | 0.378686 | 0.378686 | 1.577857 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  6.376282  6.376282  6.376282  0.0  0.0 |  9.139312 |  9.900498 |  0.10383   0.10433  0.105233    0.1064  0.107664 | 1.577857 |
    | 01/01 09:00 |  0.0 | 5.0 | 30.4 |  7.317556 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.310975  0.310975  0.310975  0.0  0.0 |      0.0 | 0.090938 |      0.0 | 0.010994 | 0.343917 | 0.364084 | 0.364084 | 1.517017 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  6.065307  6.065307  6.065307  0.0  0.0 |  9.048374 |  9.889504 | 0.099874  0.100351  0.101212  0.102324  0.103529 | 1.517017 |
    | 01/01 10:00 |  0.0 | 5.0 | 31.2 |  8.264362 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.295808  0.295808  0.295808  0.0  0.0 |      0.0 | 0.090033 |      0.0 | 0.010982 | 0.330928 |  0.35016 |  0.35016 | 1.459002 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  5.769498  5.769498  5.769498  0.0  0.0 |  8.958341 |  9.878522 | 0.096101  0.096555  0.097376  0.098438  0.099587 | 1.459002 |
    | 01/01 11:00 |  0.0 | 5.0 | 34.2 |  9.369867 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.281382  0.281382  0.281382  0.0  0.0 |      0.0 | 0.089137 |      0.0 |  0.01097 | 0.318539 | 0.336882 | 0.336882 | 1.403675 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  5.488116  5.488116  5.488116  0.0  0.0 |  8.869204 |  9.867552 | 0.092502  0.092935  0.093719  0.094731  0.095827 | 1.403675 |
    | 01/01 12:00 |  0.0 | 5.0 | 36.1 |  5.126178 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.267659  0.267659  0.267659  0.0  0.0 |      0.0 |  0.08825 |      0.0 | 0.010958 | 0.306721 | 0.324218 | 0.324218 | 1.350908 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  5.220458  5.220458  5.220458  0.0  0.0 |  8.780954 |  9.856594 | 0.089069  0.089483   0.09023  0.091195   0.09224 | 1.350908 |
    | 01/01 13:00 |  0.0 | 5.0 | 30.2 |   6.62503 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.254605  0.254605  0.254605  0.0  0.0 |      0.0 | 0.087372 |      0.0 | 0.010946 | 0.295447 | 0.312138 | 0.312138 | 1.300575 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  4.965853  4.965853  4.965853  0.0  0.0 |  8.693582 |  9.845648 | 0.085794  0.086189  0.086901  0.087822  0.088819 | 1.300575 |
    | 01/01 14:00 |  0.0 | 5.0 | 23.3 |  7.397619 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.242188  0.242188  0.242188  0.0  0.0 |      0.0 | 0.086503 |      0.0 | 0.010934 |  0.28469 | 0.300615 | 0.300615 | 1.252561 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  4.723666  4.723666  4.723666  0.0  0.0 |   8.60708 |  9.834715 | 0.082669  0.083046  0.083726  0.084604  0.085556 | 1.252561 |
    | 01/01 15:00 |  0.2 | 5.0 | 24.4 |   2.39151 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0      1.1856  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |  0.2112   0.50976    1.1856   0.2112    0.8736 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0    1.1856   0.2112       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0   0.2112  0.0 |      0.0       0.0  0.0   0.2112  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.230376  0.230376  0.230376  0.0  0.0 | 0.026819 | 0.085776 | 0.003352 | 0.010923 | 0.274553 |  0.28963 |  0.28963 | 1.206791 |   0.2112   0.50976  0.0  0.0    0.8736 |        0.0         0.0      1.1856  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |   4.49329   4.49329   4.49329  0.0  0.0 |  8.548123 |  9.827144 | 0.079723  0.080078   0.08072  0.081551  0.082452 | 1.206791 |
    | 01/01 16:00 |  0.0 | 5.0 | 24.9 |  1.829834 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0      1.1856  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  0.21914   0.21914   0.21914  0.0  0.0 |      0.0 | 0.085055 |      0.0 | 0.010913 | 0.264883 | 0.279212 | 0.279212 | 1.163383 |   0.2112   0.50976  0.0  0.0    0.8736 |        0.0         0.0      1.1856  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  4.274149  4.274149  4.274149  0.0  0.0 |  8.463068 |  9.816231 | 0.076914  0.077253  0.077864  0.078654   0.07951 | 1.163383 |
    | 01/01 17:00 |  0.0 | 5.0 | 22.7 |  1.136569 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0      1.1856  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.208453  0.208453  0.208453  0.0  0.0 |      0.0 | 0.084209 |      0.0 | 0.010901 | 0.255531 | 0.269308 | 0.269308 | 1.122116 |   0.2112   0.50976  0.0  0.0    0.8736 |        0.0         0.0      1.1856  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  4.065697  4.065697  4.065697  0.0  0.0 |  8.378859 |   9.80533 | 0.074199  0.074526  0.075116  0.075877    0.0767 | 1.122116 |
    | 01/01 18:00 |  1.3 | 5.0 | 20.6 |  0.750986 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0    1.180822       8.892  0.0       4.552 | 0.0  0.0  0.901953  0.0   0.0 |  1.3728   3.31344    7.7064   1.3728    5.6784 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0    1.8232    7.7064   1.3728     4.552 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |      0.0   0.642378  0.0   1.3728  0.0 |      0.0  0.160595  0.0   1.3728  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0  0.160595  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0  0.160595  0.0  0.0  0.0 | 0.198286  0.202235  0.198286  0.0  0.0 | 0.174324 |  0.08424 |  0.02179 | 0.010901 |  0.24848 | 0.259939 | 0.259939 | 1.083079 |    1.584       2.0  0.0  0.0       2.0 |        0.0    0.843444       8.892  0.0       4.552 |        0.0    0.337378         0.0  0.0         0.0 |      100.0  100.481784  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |   3.86741  4.024056   3.86741  0.0  0.0 |  8.468943 |  9.816219 | 0.072131  0.072383  0.072844  0.073456  0.074145 | 1.083079 |
    | 01/01 19:00 |  5.6 | 5.0 | 20.2 |  0.223895 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0    10.04271     42.0888  0.0     29.0128 | 0.0  0.0  0.901953  0.0   0.0 |  5.9136  14.27328   33.1968   5.9136   24.4608 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   5.4976  14.27328   33.1968   5.9136   24.4608 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |   5.4976   5.411392  0.0   5.9136  0.0 |   1.3744  1.365915  0.0   5.9136  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   1.3744  1.365915  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |   1.3744  1.365915  0.0  0.0  0.0 |  0.22241  0.229841  0.188616  0.0  0.0 | 0.750933 |  0.08801 | 0.093867 | 0.010953 | 0.265516 | 0.253591 | 0.253591 |  1.05663 |      2.0       2.0  0.0  0.0       2.0 |        0.0    7.173364     42.0888  0.0     29.0128 |        0.0    2.869346         0.0  0.0         0.0 |   104.1232  104.527261  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |    5.0194   5.16013  3.678794  0.0  0.0 |  9.131867 |  9.899133 | 0.076836  0.076297   0.07542  0.074497  0.073834 |  1.05663 |
    | 01/01 20:00 |  2.9 | 5.0 | 19.4 |  0.099425 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   14.658902       59.28  0.0       41.68 | 0.0  0.0  0.901953  0.0   0.0 |  3.0624   7.39152   17.1912   3.0624   12.6672 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   3.0624   7.39152   17.1912   3.0624   12.6672 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |   3.0624   2.775328  0.0   3.0624  0.0 | 0.830036  0.758077  0.0   3.0624  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.830036  0.758077  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.830036  0.758077  0.0  0.0  0.0 | 0.265208  0.270302  0.179417  0.0  0.0 | 0.388876 | 0.092802 |  0.04861 |  0.01102 | 0.293267 | 0.260891 | 0.260891 | 1.087044 |      2.0       2.0  0.0  0.0       2.0 |        0.0   10.470644       59.28  0.0       41.68 |        0.0    4.188258         0.0  0.0         0.0 | 106.355564  106.544511  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  5.584228  5.647905  3.499377  0.0  0.0 |  9.427942 |  9.936723 | 0.084771   0.08383  0.082187  0.080195  0.078277 | 1.087044 |
    | 01/01 21:00 |  4.9 | 5.0 | 18.8 |  0.098454 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   22.420054     88.3272  0.0     63.0832 | 0.0  0.0  0.901953  0.0   0.0 |  5.1744  12.48912   29.0472   5.1744   21.4032 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   5.1744  12.48912   29.0472   5.1744   21.4032 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |   5.1744   4.727968  0.0   5.1744  0.0 | 1.463256  1.341766  0.0   5.1744  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.463256  1.341766  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.463256  1.341766  0.0  0.0  0.0 | 0.308325  0.308444  0.170667  0.0  0.0 | 0.657067 | 0.097084 | 0.082133 |  0.01108 | 0.320113 | 0.281974 | 0.281974 | 1.174891 |      2.0       2.0  0.0  0.0       2.0 |        0.0   16.014324     88.3272  0.0     63.0832 |        0.0     6.40573         0.0  0.0         0.0 | 110.066708  109.930714  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  6.739159  6.681227  3.328711  0.0  0.0 |  9.987924 | 10.007776 | 0.092561  0.091624  0.089938   0.08778  0.085496 | 1.174891 |
    | 01/01 22:00 | 10.6 | 5.0 | 18.6 |  0.098128 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   39.144342     151.164  0.0     109.384 | 0.0  0.0  0.901953  0.0   0.0 | 11.1936  27.01728   62.8368  11.1936   46.3008 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |  11.1936  27.01728   62.8368  11.1936   46.3008 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |  11.1936  10.292992  0.0  11.1936  0.0 | 3.390172  3.109709  0.0  11.1936  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       2.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.132292  0.105603  0.0  0.0  0.0 |      2.0       2.0  0.0  0.0  0.0 |  0.37785  0.375024  0.162343  0.0  0.0 |  1.42141 | 0.106465 | 0.177676 | 0.011212 | 0.440515 | 0.314762 | 0.314762 | 1.311509 |      2.0       2.0  0.0  0.0       2.0 |        0.0   27.960244     151.164  0.0     109.384 |        0.0   11.184098         0.0  0.0         0.0 | 117.870135  117.113997  0.0  0.0  0.0 |  1.25788  1.004106  0.0  0.0  0.0 |  8.361309  8.306202  3.166368  0.0  0.0 | 11.302869 |  10.17424 | 0.126563  0.122579  0.115762  0.107781  0.100468 | 1.311509 |
    | 01/01 23:00 |  0.1 | 5.0 | 18.2 |  0.097474 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0    39.35759    151.7568  0.0    109.8208 | 0.0  0.0  0.901953  0.0   0.0 |  0.1056   0.25488    0.5928   0.1056    0.4368 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.1056   0.25488    0.5928   0.1056    0.4368 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |   0.1056   0.041632  0.0   0.1056  0.0 | 0.036678  0.014275  0.0   0.1056  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.294558  1.018381  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.294558  1.018381  0.0  0.0  0.0 | 0.439617  0.430139  0.154426  0.0  0.0 |  0.01341 | 0.112532 | 0.001676 | 0.011299 | 0.407704 | 0.384144 | 0.384144 | 1.600599 |      2.0       2.0  0.0  0.0       2.0 |        0.0   28.112564    151.7568  0.0    109.8208 |        0.0   11.245026         0.0  0.0         0.0 | 117.939057  117.141354  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |   9.21625  8.894445  3.011942  0.0  0.0 | 11.203746 | 10.164617 | 0.118515  0.119316  0.120214  0.120211  0.118457 | 1.600599 |
    | 02/01 00:00 |  0.7 | 5.0 | 18.2 |  0.097474 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   40.514326    155.9064  0.0    112.8784 | 0.0  0.0  0.902305  0.0   0.0 |  0.7392   1.78416    4.1496   0.7392    3.0576 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.7392   1.78416    4.1496   0.7392    3.0576 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |   0.7392   0.627424  0.0   0.7392  0.0 |  0.25705  0.215239  0.0   0.7392  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.25705  0.215239  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  0.25705  0.215239  0.0  0.0  0.0 | 0.455802   0.43908  0.146894  0.0  0.0 | 0.093867 | 0.111947 | 0.011733 | 0.011294 | 0.413426 | 0.411894 | 0.411894 | 1.716227 |      2.0       2.0  0.0  0.0       2.0 |        0.0   28.938804    155.9064  0.0    112.8784 |        0.0   11.575522         0.0  0.0         0.0 | 118.421207  117.553538  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  9.017498  8.670605  2.865048  0.0  0.0 | 11.185666 | 10.165056 | 0.119847  0.119719  0.119579  0.119536  0.119564 | 1.716227 |
    | 02/01 01:00 |  3.0 | 5.0 | 18.2 |  0.097474 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   45.287766    173.6904  0.0    125.9824 | 0.0  0.0  0.902305  0.0   0.0 |   3.168    7.6464    17.784    3.168    13.104 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |    3.168    7.6464    17.784    3.168    13.104 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |    3.168    2.87296  0.0    3.168  0.0 | 1.110668  0.992524  0.0    3.168  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.110668  0.992524  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.110668  0.992524  0.0  0.0  0.0 | 0.467098  0.447275   0.13973  0.0  0.0 | 0.402286 | 0.113304 | 0.050286 | 0.011316 | 0.419231 | 0.412916 | 0.412916 | 1.720484 |      2.0       2.0  0.0  0.0       2.0 |        0.0   32.348404    173.6904  0.0    125.9824 |        0.0   12.939362         0.0  0.0         0.0 | 120.478539  119.433974  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  9.661067  9.215854  2.725318  0.0  0.0 | 11.474648 | 10.204025 | 0.121519   0.12132  0.120974  0.120562  0.120185 | 1.720484 |
    | 02/01 02:00 |  2.1 | 5.0 | 17.9 |  0.096981 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   48.645974    186.1392  0.0    135.1552 | 0.0  0.0  0.902305  0.0   0.0 |  2.2176   5.35248   12.4488   2.2176    9.1728 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   2.2176   5.35248   12.4488   2.2176    9.1728 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |   2.2176   1.994272  0.0   2.2176  0.0 | 0.804716  0.711181  0.0   2.2176  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.804716  0.711181  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.804716  0.711181  0.0  0.0  0.0 | 0.490963  0.466949  0.132915  0.0  0.0 |   0.2816 | 0.115578 |   0.0352 | 0.011351 | 0.433224 | 0.417486 | 0.417486 | 1.739524 |      2.0       2.0  0.0  0.0       2.0 |        0.0   34.747124    186.1392  0.0    135.1552 |        0.0    13.89885         0.0  0.0         0.0 | 121.891424  120.717065  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  9.974821  9.460085  2.592403  0.0  0.0 |  11.64067 | 10.227874 | 0.125496  0.125027  0.124215  0.123242  0.122318 | 1.739524 |
    | 02/01 03:00 | 10.4 | 5.0 | 17.7 |  0.096652 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  106.141894       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 | 10.9824  26.50752   61.6512  10.9824   45.4272 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |  10.9824  26.50752   61.6512  10.9824   45.4272 |        0.0         0.0     47.7904  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0     30.9884         0.0  0.0     19.4176 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |  10.9824        0.0  0.0  10.9824  0.0 |  4.07928       0.0  0.0  10.9824  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.19787       0.0  0.0  0.0  0.0 |      2.0       0.0  0.0  0.0  0.0 | 0.535655  0.461374  0.126433  0.0  0.0 |  1.39459 | 0.122776 | 0.174324 | 0.011455 |  0.51811 | 0.432779 | 0.432779 | 1.803247 |      2.0       2.0  0.0  0.0       2.0 |        0.0   77.456804       200.0  0.0       200.0 |        0.0    28.68509         0.0  0.0         0.0 | 128.794543  120.717065  0.0  0.0  0.0 | 1.881411       0.0  0.0  0.0  0.0 | 11.439166  8.998712   2.46597  0.0  0.0 | 12.912484 | 10.390743 | 0.149418  0.146621  0.141859   0.13634  0.131391 | 1.803247 |
    | 02/01 04:00 |  3.5 | 5.0 | 17.5 |  0.096321 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  134.445694       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |   3.696    8.9208    20.748    3.696    15.288 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |    3.696    8.9208    20.748    3.696    15.288 |        0.0         0.0      20.748  0.0      15.288 |       0.0        0.0        0.0  0.0        0.0 |        0.0      19.383         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |    3.696        0.0  0.0    3.696  0.0 | 1.532734       0.0  0.0    3.696  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.134574       0.0  0.0  0.0  0.0 | 1.928042       0.0  0.0  0.0  0.0 | 0.605302  0.438872  0.120267  0.0  0.0 | 0.492177 | 0.130934 | 0.061522 | 0.011573 | 0.520718 | 0.480753 | 0.480753 | 2.003138 |      2.0       2.0  0.0  0.0       2.0 |        0.0  100.811004       200.0  0.0       200.0 |        0.0    33.63469         0.0  0.0         0.0 | 130.957809  120.717065  0.0  0.0  0.0 | 1.279571       0.0  0.0  0.0  0.0 | 12.761905  8.559839  2.345703  0.0  0.0 | 13.273727 | 10.440692 | 0.150955  0.150682  0.149906  0.148315  0.145736 | 2.003138 |
    | 02/01 05:00 |  3.4 | 5.0 | 17.3 |   0.09599 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  161.940814       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  3.5904   8.66592   20.1552   3.5904   14.8512 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   3.5904   8.66592   20.1552   3.5904   14.8512 |        0.0         0.0     20.1552  0.0     14.8512 |       0.0        0.0        0.0  0.0        0.0 |        0.0     18.8292         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |   3.5904        0.0  0.0   3.5904  0.0 | 1.539379       0.0  0.0   3.5904  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.077933       0.0  0.0  0.0  0.0 | 1.861905       0.0  0.0  0.0  0.0 | 0.668187  0.417468  0.114401  0.0  0.0 | 0.499764 | 0.134566 |  0.06247 | 0.011629 | 0.519361 | 0.512333 | 0.512333 |  2.13472 |      2.0       2.0  0.0  0.0       2.0 |        0.0  123.499084       200.0  0.0       200.0 |        0.0    38.44173         0.0  0.0         0.0 | 133.008829  120.717065  0.0  0.0  0.0 | 0.741017       0.0  0.0  0.0  0.0 | 13.955623  8.142371  2.231302  0.0  0.0 | 13.638924 | 10.491534 | 0.150626  0.150655  0.150663  0.150543  0.150135 |  2.13472 |
    | 02/01 06:00 |  1.2 | 5.0 | 17.5 |  0.187298 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  171.644974       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  1.2672   3.05856    7.1136   1.2672    5.2416 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   1.2672   3.05856    7.1136   1.2672    5.2416 |        0.0         0.0      7.1136  0.0      5.2416 |       0.0        0.0        0.0  0.0        0.0 |        0.0      6.6456         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |   1.2672        0.0  0.0   1.2672  0.0 | 0.560462       0.0  0.0   1.2672  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.301479       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.301479       0.0  0.0  0.0  0.0 | 0.712625  0.397108  0.108822  0.0  0.0 | 0.160914 | 0.136511 | 0.020114 | 0.011662 | 0.503497 | 0.517757 | 0.517757 |  2.15732 |      2.0       2.0  0.0  0.0       2.0 |        0.0  131.532524       200.0  0.0       200.0 |        0.0    40.11245         0.0  0.0         0.0 | 133.715567  120.717065  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 14.544477  7.745263   2.12248  0.0  0.0 | 13.663327 | 10.499986 |  0.14617  0.146689  0.147564  0.148555  0.149384 |  2.15732 |
    | 02/01 07:00 |  0.1 | 5.0 | 17.8 |  1.264612 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  172.453654       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  0.1056   0.25488    0.5928   0.1056    0.4368 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.1056   0.25488    0.5928   0.1056    0.4368 |        0.0         0.0      0.5928  0.0      0.4368 |       0.0        0.0        0.0  0.0        0.0 |        0.0      0.5538         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |   0.1056        0.0  0.0   0.1056  0.0 | 0.047203       0.0  0.0   0.1056  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.047203       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.047203       0.0  0.0  0.0  0.0 | 0.710503  0.377741  0.103515  0.0  0.0 |  0.01341 | 0.136019 | 0.001676 | 0.011661 | 0.496103 | 0.509771 | 0.509771 | 2.124047 |      2.0       2.0  0.0  0.0       2.0 |        0.0  132.238644       200.0  0.0       200.0 |        0.0    40.21501         0.0  0.0         0.0 | 133.773965  120.717065  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 13.881177  7.367522  2.018965  0.0  0.0 | 13.540717 | 10.490001 | 0.143947  0.144223  0.144746  0.145469  0.146308 | 2.124047 |
    | 02/01 08:00 |  0.0 | 5.0 | 17.8 |  3.045538 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  172.453654       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.676993  0.359318  0.098466  0.0  0.0 |      0.0 | 0.134732 |      0.0 | 0.011649 | 0.477798 |  0.49928 |  0.49928 | 2.080334 |      2.0       2.0  0.0  0.0       2.0 |        0.0  132.278644       200.0  0.0       200.0 |        0.0    40.17501         0.0  0.0         0.0 | 133.773965  120.717065  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 13.204184  7.008204  1.920499  0.0  0.0 | 13.405985 | 10.478352 | 0.138743  0.139357  0.140425  0.141714  0.142972 | 2.080334 |
    | 02/01 09:00 |  0.0 | 5.0 | 18.0 |  1.930758 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  172.453654       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.643976  0.341794  0.093664  0.0  0.0 |      0.0 | 0.133392 |      0.0 | 0.011636 | 0.459896 | 0.485121 | 0.485121 | 2.021336 |      2.0       2.0  0.0  0.0       2.0 |        0.0  132.318644       200.0  0.0       200.0 |        0.0    40.13501         0.0  0.0         0.0 | 133.773965  120.717065  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 12.560208   6.66641  1.826835  0.0  0.0 | 13.272593 | 10.466716 | 0.133551  0.134175  0.135296  0.136728  0.138237 | 2.021336 |
    | 02/01 10:00 |  0.4 | 5.0 | 18.2 |  2.461001 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  175.688374       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  0.4224   1.01952    2.3712   0.4224    1.7472 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.4224   1.01952    2.3712   0.4224    1.7472 |        0.0         0.0      2.3712  0.0      1.7472 |       0.0        0.0        0.0  0.0        0.0 |        0.0      2.2152         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |   0.4224        0.0  0.0   0.4224  0.0 | 0.188976       0.0  0.0   0.4224  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.188976       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.188976       0.0  0.0  0.0  0.0 | 0.617215  0.325125  0.089096  0.0  0.0 | 0.053638 | 0.132332 | 0.006705 | 0.011627 | 0.444619 | 0.468068 | 0.468068 | 1.950283 |      2.0       2.0  0.0  0.0       2.0 |        0.0  135.023124       200.0  0.0       200.0 |        0.0    40.66525         0.0  0.0         0.0 | 134.007388  120.717065  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 12.131969  6.341285  1.737739  0.0  0.0 | 13.193899 | 10.461794 | 0.129095  0.129634  0.130614  0.131895    0.1333 | 1.950283 |
    | 02/01 11:00 |  0.1 | 5.0 | 18.3 |  6.215945 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  176.497054       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  0.1056   0.25488    0.5928   0.1056    0.4368 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.1056   0.25488    0.5928   0.1056    0.4368 |        0.0         0.0      0.5928  0.0      0.4368 |       0.0        0.0        0.0  0.0        0.0 |        0.0      0.5538         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |   0.1056        0.0  0.0   0.1056  0.0 | 0.047409       0.0  0.0   0.1056  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.047409       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.047409       0.0  0.0  0.0  0.0 | 0.592849  0.309268  0.084751  0.0  0.0 |  0.01341 | 0.131348 | 0.001676 | 0.011619 | 0.430474 | 0.451814 | 0.451814 |  1.88256 |      2.0       2.0  0.0  0.0       2.0 |        0.0  135.729244       200.0  0.0       200.0 |        0.0    40.76781         0.0  0.0         0.0 | 134.065579  120.717065  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 11.586529  6.032017  1.652989  0.0  0.0 | 13.075961 | 10.451851 | 0.124981  0.125477  0.126376  0.127544  0.128819 |  1.88256 |
    | 02/01 12:00 |  3.6 | 5.0 | 18.0 |  3.374783 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |   2.998287       200.0       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  3.8016   9.17568   21.3408   3.8016   15.7248 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   3.8016   9.17568   21.3408   3.8016   15.7248 |        0.0         0.0     21.3408  0.0     15.7248 |       0.0        0.0        0.0  0.0        0.0 |   4.487627   14.327266         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |  5.29094        0.0  0.0   3.8016  0.0 | 2.377428       0.0  0.0   3.8016  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.035917       0.0  0.0  0.0  0.0 | 1.920674       0.0  0.0  0.0  0.0 | 0.612308  0.294185  0.080617  0.0  0.0 | 0.507926 | 0.132639 | 0.063491 | 0.011642 |  0.44531 | 0.439041 | 0.439041 | 1.829338 |      2.0       2.0  0.0  0.0       2.0 |   2.141634   154.14003       200.0  0.0       200.0 |   0.856653    45.85997         0.0  0.0         0.0 | 136.979091  120.717065  0.0  0.0  0.0 | 0.341511       0.0  0.0  0.0  0.0 | 12.894895  5.737832  1.572372  0.0  0.0 | 13.451247 |   10.5037 |    0.129  0.128549  0.127849  0.127192  0.126877 | 1.829338 |
    | 02/01 13:00 |  5.9 | 5.0 | 17.8 |  8.821555 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |  47.398383       200.0       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  6.2304  15.03792   34.9752   6.2304   25.7712 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   6.2304  15.03792   34.9752   6.2304   25.7712 |        0.0   11.242668     34.9752  0.0     25.7712 |       0.0   3.795252        0.0  0.0        0.0 |  35.133494         0.0         0.0  0.0         0.0 |  3.036202        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |      0.0        0.0  0.0   6.2304  0.0 |      0.0       0.0  0.0   6.2304  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.341511       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.341511       0.0  0.0  0.0  0.0 | 0.637289  0.279837  0.076685  0.0  0.0 | 0.791162 | 0.137785 | 0.098895 | 0.011719 | 0.441927 | 0.440056 | 0.440056 | 1.833568 |      2.0       2.0  0.0  0.0       2.0 |  34.929135  149.564242       200.0  0.0       200.0 |  12.469248   50.435758         0.0  0.0         0.0 | 136.979091  120.717065  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 12.599117  5.457995  1.495686  0.0  0.0 | 14.104624 | 10.590876 | 0.128187  0.128266  0.128351  0.128345  0.128187 | 1.833568 |
    | 02/01 14:00 |  1.1 | 5.0 | 17.7 |  4.046025 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |   54.77413       200.0       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  1.1616   2.80368    6.5208   1.1616    4.8048 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   1.1616   2.80368    6.5208   1.1616    4.8048 |        0.0    2.084747      6.5208  0.0      4.8048 |       0.0   0.718933        0.0  0.0        0.0 |   6.541237         0.0         0.0  0.0         0.0 |  0.575147        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 | 0.902237        0.0  0.0   1.1616  0.0 | 0.423223       0.0  0.0   1.1616  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.423223       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.423223       0.0  0.0  0.0  0.0 | 0.624873   0.26619  0.072945  0.0  0.0 | 0.147505 | 0.141078 | 0.018438 | 0.011771 | 0.436524 | 0.441912 | 0.441912 | 1.841299 |      2.0       2.0  0.0  0.0       2.0 |  39.124378  148.755016       200.0  0.0       200.0 |  15.649751   51.244984         0.0  0.0         0.0 | 137.458105  120.717065  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 12.397468  5.191805  1.422741  0.0  0.0 | 14.111051 | 10.597543 | 0.126646  0.126828  0.127141  0.127507  0.127828 | 1.841299 |
    | 02/01 15:00 | 20.7 | 5.0 | 16.8 |  2.110757 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 204.057976  195.707265  204.057976  0.0  204.057976 | 0.0  0.0  0.902305  0.0   0.0 | 21.8592  52.76016  122.7096  21.8592   90.4176 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |  21.8592  52.76016  122.7096  21.8592   90.4176 |        0.0   35.903699    122.7096  0.0     90.4176 |       0.0  16.856461        0.0  0.0        0.0 | 114.593284    3.649347    3.649347  0.0    3.649347 | 12.831363   0.408629   0.408629  0.0   0.408629 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04  0.408629  0.0  0.34 |      0.0   8.350711  0.0  21.8592  0.0 |      0.0  3.042291  0.0  21.8592  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       2.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0  0.099187  0.0  0.0  0.0 |      0.0       2.0  0.0  0.0  0.0 | 0.604632  0.302384  0.069388  0.0  0.0 | 2.775771 |  0.15424 | 0.346971 | 0.011961 | 0.477628 | 0.441919 | 0.441919 |  1.84133 |      2.0       2.0  0.0  0.0       2.0 | 151.371669  139.790904  204.057976  0.0  203.989347 |  52.686307   55.916362         0.0  0.0    0.068629 | 137.458105  126.025485  0.0  0.0  0.0 |      0.0  0.943104  0.0  0.0  0.0 | 11.792836  6.889421  1.353353  0.0  0.0 | 16.732582 | 10.932553 | 0.138112  0.136785  0.134572  0.132111  0.130078 |  1.84133 |
    | 02/01 16:00 | 37.9 | 5.0 | 16.4 |  2.239257 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 302.049358  306.424159  311.186287  0.0  311.186287 | 0.0  0.0  0.902305  0.0   0.0 | 40.0224  96.59952  224.6712  40.0224  165.5472 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |  40.0224  96.59952  224.6712  40.0224  165.5472 |  27.337389   57.587039  228.729176  0.0  169.573684 | 16.742987  34.719746        0.0  0.0   0.031492 |  94.061255   94.061255   94.061255  0.0   94.061255 | 17.125032  17.125032  17.125032  0.0  17.125032 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 9.136928   4.762128  0.0  40.0224  0.0 | 4.315995  1.890853  0.0  40.0224  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       2.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.220396  0.079362  0.0  0.0  0.0 | 1.910358       2.0  0.0  0.0  0.0 | 0.622116  0.385178  0.066004  0.0  0.0 | 5.110667 |  0.19196 | 0.638833 | 0.012495 | 0.608741 | 0.469866 | 0.469866 | 1.957776 |      2.0       2.0  0.0  0.0       2.0 | 215.749542  218.874399  294.701255  0.0  294.364119 |  86.299817    87.54976   16.485032  0.0   16.822168 | 142.279039  128.896759  0.0  0.0  0.0 | 2.095599  0.754596  0.0  0.0  0.0 | 13.081078  8.504243  1.287349  0.0  0.0 | 21.651289 | 11.558891 |  0.17524  0.170879  0.163389  0.154571  0.146455 | 1.957776 |
    | 02/01 17:00 |  8.2 | 5.0 | 16.3 |  2.877848 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 329.940893  329.940893  329.940893  0.0  329.940893 | 0.0  0.0  0.902305  0.0   0.0 |  8.6592  20.90016   48.6096   8.6592   35.8176 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   8.6592  20.90016   48.6096   8.6592   35.8176 |  76.873713   88.721551  152.474401  0.0  139.877384 | 33.834845  38.602768   7.321485  0.0   7.126503 | 102.811951  102.811951  102.811951  0.0  102.811951 | 27.128942  27.128942  27.128942  0.0  27.128942 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   8.6592  0.0 |      0.0       0.0  0.0   8.6592  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0  0.754596  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.009097       0.0  0.0  0.0  0.0 | 1.845946  0.754596  0.0  0.0  0.0 | 0.683361  0.433311  0.062785  0.0  0.0 | 1.148487 | 0.221157 | 0.143561 | 0.012916 | 0.577394 | 0.547047 | 0.547047 | 2.279364 |      2.0       2.0  0.0  0.0       2.0 | 239.341786  242.215039  294.288404  0.0  293.456285 |  90.599108   87.725855   35.652489  0.0   36.484608 | 142.279039  128.896759  0.0  0.0  0.0 | 0.086502       0.0  0.0  0.0  0.0 | 14.243663  8.825527  1.224564  0.0  0.0 | 22.578619 | 11.689536 | 0.167707  0.168436  0.169168  0.168875  0.166694 | 2.279364 |
    | 02/01 18:00 |  3.6 | 5.0 | 15.8 |  1.591452 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 340.360733  340.360733  340.360733  0.0  340.360733 | 0.0  0.0  0.902305  0.0   0.0 |  3.8016   9.17568   21.3408   3.8016   15.7248 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   3.8016   9.17568   21.3408   3.8016   15.7248 |   95.91277  101.023216  135.927722  0.0   130.29085 | 37.829723  38.093357  15.353971  0.0  15.374843 | 109.363903  109.363903  109.363903  0.0  109.363903 | 30.996831  30.996831  30.996831  0.0  30.996831 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   3.8016  0.0 |      0.0       0.0  0.0   3.8016  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.086502       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.086502       0.0  0.0  0.0  0.0 | 0.696799  0.430426  0.059723  0.0  0.0 | 0.482743 | 0.227067 | 0.060343 | 0.013015 | 0.583067 | 0.579958 | 0.579958 | 2.416492 |      2.0       2.0  0.0  0.0       2.0 | 250.446924  254.639245  289.705384  0.0  288.594137 |  89.913809   85.721488   50.655349  0.0   51.766596 | 142.279039  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 13.633366  8.395101  1.164842  0.0  0.0 | 22.834295 | 11.736864 | 0.169043  0.168912  0.168757  0.168671  0.168605 | 2.416492 |
    | 02/01 19:00 |  7.5 | 5.0 | 15.5 |  0.291604 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 362.068733  362.068733  362.068733  0.0  362.068733 | 0.0  0.0  0.902305  0.0   0.0 |    7.92    19.116     44.46     7.92     32.76 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |     7.92    19.116     44.46     7.92     32.76 |  106.62793  116.704262   160.49211  0.0  149.102044 | 41.652803  42.772471  24.328623  0.0  24.018689 | 125.425775  125.425775  125.425775  0.0  125.425775 | 36.642959  36.642959  36.642959  0.0  36.642959 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0     7.92  0.0 |      0.0       0.0  0.0     7.92  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.664907  0.409434   0.05681  0.0  0.0 | 1.005714 | 0.232217 | 0.125714 | 0.013104 | 0.571146 | 0.581078 | 0.581078 | 2.421156 |      2.0       2.0  0.0  0.0       2.0 | 266.898776  271.824758  299.739049  0.0  298.017868 |  95.169957   90.243976   62.329684  0.0   64.050865 | 142.279039  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 12.968459  7.985667  1.108032  0.0  0.0 | 23.607792 | 11.849475 | 0.165747  0.166126  0.166751  0.167437  0.167995 | 2.421156 |
    | 02/01 20:00 | 18.5 | 5.0 | 15.3 |  0.092622 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 415.615133  415.615133  415.615133  0.0  415.615133 | 0.0  0.0  0.902305  0.0   0.0 |  19.536   47.1528   109.668   19.536    80.808 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   19.536   47.1528   109.668   19.536    80.808 | 127.016457  149.598796  235.832672  0.0  207.750786 | 54.588277  59.622737  35.904062  0.0  35.125947 | 165.286636  165.286636  165.286636  0.0  165.286636 | 50.328497  50.328497  50.328497  0.0  50.328497 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   19.536  0.0 |      0.0       0.0  0.0   19.536  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.632479  0.389466  0.054039  0.0  0.0 | 2.480762 | 0.247264 | 0.310095 | 0.013331 | 0.568714 | 0.575794 | 0.575794 |  2.39914 |      2.0       2.0  0.0  0.0       2.0 | 302.822962  308.331798  339.501014  0.0  336.701718 | 112.792171  107.283336    76.11412  0.0   78.913415 | 142.279039  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  12.33598  7.596202  1.053992  0.0  0.0 |  25.84129 | 12.146239 | 0.164955   0.16506  0.165278  0.165619  0.166065 |  2.39914 |
    | 02/01 21:00 | 15.4 | 5.0 | 15.2 |  0.092451 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 460.188893  460.188893  460.188893  0.0  460.188893 | 0.0  0.0  0.902305  0.0   0.0 | 16.2624  39.25152   91.2912  16.2624   67.2672 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |  16.2624  39.25152   91.2912  16.2624   67.2672 | 162.587391  182.453457  260.823055  0.0  236.653243 | 69.290142  72.413197  46.083278  0.0   46.22909 | 197.067169  197.067169  197.067169  0.0  197.067169 | 63.121725  63.121725  63.121725  0.0  63.121725 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0  16.2624  0.0 |      0.0       0.0  0.0  16.2624  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.601633  0.370471  0.051404  0.0  0.0 | 2.065067 | 0.267416 | 0.258133 | 0.013632 | 0.571929 | 0.570935 | 0.570935 | 2.378895 |      2.0       2.0  0.0  0.0       2.0 | 334.956746   340.28279  367.676327  0.0  364.722844 | 125.232147  119.906104   92.512567  0.0   95.466049 | 142.279039  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 11.734347  7.225731  1.002588  0.0  0.0 | 27.638941 | 12.390741 | 0.165829  0.165731  0.165579  0.165441  0.165392 | 2.378895 |
    | 02/01 22:00 |  6.3 | 5.0 | 14.5 |  0.091248 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 478.423613  478.423613  478.423613  0.0  478.423613 | 0.0  0.0  0.902305  0.0   0.0 |  6.6528  16.05744   37.3464   6.6528   27.5184 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   6.6528  16.05744   37.3464   6.6528   27.5184 | 191.457675  201.485302   242.21107  0.0  231.390156 | 75.384018  74.761032  55.324223  0.0  56.317137 |  209.32715   209.32715   209.32715  0.0   209.32715 | 69.096463  69.096463  69.096463  0.0  69.096463 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   6.6528  0.0 |      0.0       0.0  0.0   6.6528  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.572291  0.352403  0.048897  0.0  0.0 |   0.8448 | 0.279222 |   0.1056 | 0.013818 | 0.568022 | 0.570588 | 0.570588 | 2.377449 |      2.0       2.0  0.0  0.0       2.0 | 350.480227  355.240798  372.778807  0.0  370.518238 | 127.943386  123.182816  105.644807  0.0  107.905375 | 142.279039  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 11.162056  6.873328  0.953692  0.0  0.0 | 28.204519 | 12.482522 | 0.164763  0.164884  0.165076  0.165272  0.165408 | 2.377449 |
    | 02/01 23:00 |  1.9 | 5.0 | 13.6 |  0.089683 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 483.922973  483.922973  483.922973  0.0  483.922973 | 0.0  0.0  0.902305  0.0   0.0 |  2.0064   4.84272   11.2632   2.0064    8.2992 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   2.0064   4.84272   11.2632   2.0064    8.2992 | 204.577508  209.475116  227.189914  0.0  223.156997 | 75.852505  73.791217  62.496899  0.0  63.565816 | 212.731903  212.731903  212.731903  0.0  212.731903 |  71.19107   71.19107   71.19107  0.0   71.19107 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   2.0064  0.0 |      0.0       0.0  0.0   2.0064  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  0.54438  0.335216  0.046512  0.0  0.0 | 0.254781 | 0.281909 | 0.031848 | 0.013879 | 0.556223 | 0.568619 | 0.568619 | 2.369244 |      2.0       2.0  0.0  0.0       2.0 | 356.288629  360.671665  370.223996  0.0  368.732344 | 127.634344  123.251308  113.698978  0.0  115.190629 | 142.279039  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 10.617676  6.538111   0.90718  0.0  0.0 |  28.17739 | 12.500491 | 0.161421  0.161814  0.162488   0.16328  0.164005 | 2.369244 |
    | 03/01 00:00 |  4.9 | 5.0 | 13.7 |  0.089858 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 498.105533  498.105533  498.105533  0.0  498.105533 | 0.0  0.0  0.902686  0.0   0.0 |  5.1744  12.48912   29.0472   5.1744   21.4032 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   5.1744  12.48912   29.0472   5.1744   21.4032 |  210.59632  218.646568   243.60086  0.0  235.726149 | 78.501053  77.765525  69.369313  0.0  69.600025 | 222.622815  222.622815  222.622815  0.0  222.622815 | 75.482718  75.482718  75.482718  0.0  75.482718 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   5.1744  0.0 |      0.0       0.0  0.0   5.1744  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  0.51783  0.318867  0.044244  0.0  0.0 | 0.657067 | 0.283644 | 0.082133 | 0.013927 | 0.544223 |  0.56071 |  0.56071 | 2.336294 |      2.0       2.0  0.0  0.0       2.0 | 365.968141  370.191592  378.933151  0.0  377.372211 | 132.137392  127.913941  119.172382  0.0  120.733323 | 142.279039  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 10.099846  6.219244  0.862936  0.0  0.0 | 28.550813 | 12.568697 | 0.157946  0.158363   0.15911  0.160058  0.161044 | 2.336294 |
    | 03/01 01:00 |  2.7 | 5.0 | 13.6 |  0.089683 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 505.920413  505.920413  505.920413  0.0  505.920413 | 0.0  0.0  0.902686  0.0   0.0 |  2.8512   6.88176   16.0056   2.8512   11.7936 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   2.8512   6.88176   16.0056   2.8512   11.7936 | 219.860457  225.408936  241.299304  0.0  236.521578 | 81.096276  79.578357  72.811829  0.0  73.377556 | 227.796732  227.796732  227.796732  0.0  227.796732 | 78.123681  78.123681  78.123681  0.0  78.123681 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   2.8512  0.0 |      0.0       0.0  0.0   2.8512  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.492575  0.303316  0.042086  0.0  0.0 | 0.362057 |  0.28589 | 0.045257 | 0.013983 | 0.533374 | 0.549597 | 0.549597 | 2.289986 |      2.0       2.0  0.0  0.0       2.0 | 371.557433  375.652029  382.076179  0.0  380.780965 |  134.36298  130.268385  123.844234  0.0  125.139448 | 142.279039  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  9.607271  5.915928   0.82085  0.0  0.0 |  28.62698 | 12.599971 | 0.154789   0.15517   0.15586  0.156755  0.157726 | 2.289986 |
    | 03/01 02:00 |  0.5 | 5.0 | 13.1 |  0.088805 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 507.367613  507.367613  507.367613  0.0  507.367613 | 0.0  0.0  0.902686  0.0   0.0 |   0.528    1.2744     2.964    0.528     2.184 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |    0.528    1.2744     2.964    0.528     2.184 | 224.826819  227.862889  233.713014  0.0   232.22234 | 81.621594  79.331924    75.1714  0.0  75.882073 | 228.380541  228.380541  228.380541  0.0  228.380541 | 78.987072  78.987072  78.987072  0.0  78.987072 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0    0.528  0.0 |      0.0       0.0  0.0    0.528  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.468552  0.288523  0.040033  0.0  0.0 | 0.067048 | 0.285177 | 0.008381 | 0.013997 | 0.520359 | 0.538199 | 0.538199 | 2.242497 |      2.0       2.0  0.0  0.0       2.0 | 372.764171  376.771281  380.347707  0.0  379.463166 | 134.603442  130.596333  127.019907  0.0  127.904447 | 142.279039  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  9.138719  5.627405  0.780817  0.0  0.0 | 28.408851 | 12.594355 | 0.151035  0.151484  0.152285  0.153297  0.154357 | 2.242497 |
    | 03/01 03:00 |  0.2 | 5.0 | 13.3 |  0.089157 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 507.946493  507.946493  507.946493  0.0  507.946493 | 0.0  0.0  0.902686  0.0   0.0 |  0.2112   0.50976    1.1856   0.2112    0.8736 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.2112   0.50976    1.1856   0.2112    0.8736 | 225.884845  228.536493  231.486744  0.0  230.668948 | 81.693969   79.34088  77.066469  0.0  77.572265 | 228.361491  228.361491  228.361491  0.0  228.361491 | 79.585003  79.585003  79.585003  0.0  79.585003 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   0.2112  0.0 |      0.0       0.0  0.0   0.2112  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.445701  0.274452  0.038081  0.0  0.0 | 0.026819 | 0.282806 | 0.003352 | 0.013988 | 0.506378 | 0.525956 | 0.525956 | 2.191484 |      2.0       2.0  0.0  0.0       2.0 | 372.893834  376.860918  379.048053  0.0  378.369308 |  135.05266  131.085575  128.898441  0.0  129.577185 | 142.279039  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  8.693018  5.352953  0.742736  0.0  0.0 | 28.152863 |  12.58372 | 0.146991  0.147476  0.148345  0.149451  0.150618 | 2.191484 |
    | 03/01 04:00 |  0.5 | 5.0 | 13.9 |  0.090207 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 509.393693  509.393693  509.393693  0.0  509.393693 | 0.0  0.0  0.902686  0.0   0.0 |   0.528    1.2744     2.964    0.528     2.184 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |    0.528    1.2744     2.964    0.528     2.184 | 226.222235  229.187234  232.470378  0.0  231.354892 | 82.252258  80.033659  78.440115  0.0  78.775601 | 228.971242  228.971242  228.971242  0.0  228.971242 | 80.422452  80.422452  80.422452  0.0  80.422452 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0    0.528  0.0 |      0.0       0.0  0.0    0.528  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.423964  0.261067  0.036224  0.0  0.0 | 0.067048 |  0.28046 | 0.008381 | 0.013979 | 0.492993 |   0.5126 |   0.5126 | 2.135833 |      2.0       2.0  0.0  0.0       2.0 | 373.295856  377.246525  379.152916  0.0  378.509658 | 136.097837  132.147168  130.240778  0.0  130.884036 | 142.279039  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  8.269055  5.091886  0.706512  0.0  0.0 | 27.939451 | 12.578122 | 0.143104  0.143572  0.144416  0.145505  0.146677 | 2.135833 |
    | 03/01 05:00 |  2.4 | 5.0 | 14.7 |  0.091593 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 516.340253  516.340253  516.340253  0.0  516.340253 | 0.0  0.0  0.902686  0.0   0.0 |  2.5344   6.11712   14.2272   2.5344   10.4832 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   2.5344   6.11712   14.2272   2.5344   10.4832 |  227.45668   232.53804  243.126327  0.0  239.344792 | 84.471413  82.972773  80.494566  0.0  80.532101 | 233.550718  233.550718  233.550718  0.0  233.550718 | 82.789535  82.789535  82.789535  0.0  82.789535 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   2.5344  0.0 |      0.0       0.0  0.0   2.5344  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.403287  0.248334  0.034457  0.0  0.0 | 0.321829 | 0.279606 | 0.040229 |  0.01399 | 0.481566 |   0.4992 |   0.4992 | 2.079998 |      2.0       2.0  0.0  0.0       2.0 |  377.04291  380.994883  384.444507  0.0  383.538784 | 139.297343   135.34537  131.895746  0.0  132.801469 | 142.279039  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  7.865768  4.843552  0.672055  0.0  0.0 | 27.981674 |  12.60436 |  0.13977  0.140174  0.140907  0.141867  0.142923 | 2.079998 |
    | 03/01 06:00 |  0.4 | 5.0 | 15.0 |  0.154861 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 517.498013  517.498013  517.498013  0.0  517.498013 | 0.0  0.0  0.902686  0.0   0.0 |  0.4224   1.01952    2.3712   0.4224    1.7472 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.4224   1.01952    2.3712   0.4224    1.7472 | 231.117926  233.986173  237.670858  0.0  236.552027 | 85.644727  83.373601  81.040595  0.0  81.535426 | 233.930686  233.930686  233.930686  0.0  233.930686 | 83.567327  83.567327  83.567327  0.0  83.567327 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   0.4224  0.0 |      0.0       0.0  0.0   0.4224  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.383618  0.236223  0.032777  0.0  0.0 | 0.053638 |  0.27869 | 0.006705 | 0.014001 | 0.470598 | 0.486929 | 0.486929 | 2.028872 |      2.0       2.0  0.0  0.0       2.0 | 377.508687  381.428677  383.715535  0.0  383.004643 | 139.989326  136.069336  133.782478  0.0   134.49337 | 142.279039  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |   7.48215  4.607329  0.639279  0.0  0.0 | 27.756622 | 12.597064 | 0.136585  0.136969  0.137662   0.13856  0.139535 | 2.028872 |
    | 03/01 07:00 |  0.2 | 5.0 | 15.7 |  0.470369 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 518.076893  518.076893  518.076893  0.0  518.076893 | 0.0  0.0  0.902686  0.0   0.0 |  0.2112   0.50976    1.1856   0.2112    0.8736 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.2112   0.50976    1.1856   0.2112    0.8736 | 231.670569  234.299035  236.486524  0.0  235.768959 | 86.038644  83.708738   82.19709  0.0  82.602655 | 233.905963  233.905963  233.905963  0.0  233.905963 | 84.170931  84.170931  84.170931  0.0  84.170931 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   0.2112  0.0 |      0.0       0.0  0.0   0.2112  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.364909  0.224702  0.031178  0.0  0.0 | 0.026819 | 0.276317 | 0.003352 | 0.013991 | 0.458746 | 0.475535 | 0.475535 | 1.981395 |      2.0       2.0  0.0  0.0       2.0 | 377.397097  381.300245  382.960574  0.0  382.355247 | 140.679796  136.776649  135.116319  0.0  135.721646 | 142.279039  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  7.117241  4.382627  0.608101  0.0  0.0 | 27.507125 | 12.586426 | 0.133156  0.133568  0.134305  0.135246  0.136246 | 1.981395 |
    | 03/01 08:00 |  0.0 | 5.0 | 16.1 |  1.173726 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 518.076893  518.076893  518.076893  0.0  518.076893 | 0.0  0.0  0.902686  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 231.705559  234.101923  235.121294  0.0  234.749649 | 86.371334   83.97497  82.955599  0.0  83.327244 | 233.465309  233.465309  233.465309  0.0  233.465309 | 84.611584  84.611584  84.611584  0.0  84.611584 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.347112  0.213743  0.029657  0.0  0.0 |      0.0 |   0.2737 |      0.0 | 0.013977 | 0.447133 | 0.464021 | 0.464021 |  1.93342 |      2.0       2.0  0.0  0.0       2.0 | 376.809864  380.703631  381.944589  0.0  381.410907 | 141.267029  137.373263  136.132304  0.0  136.665986 | 142.279039  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  6.770129  4.168884  0.578443  0.0  0.0 | 27.233424 | 12.572449 | 0.129786  0.130192  0.130922  0.131862  0.132871 |  1.93342 |
    | 03/01 09:00 |  0.0 | 5.0 | 15.2 |  4.202296 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 518.076893  518.076893  518.076893  0.0  518.076893 | 0.0  0.0  0.902686  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 231.345023  233.735628  234.497523  0.0  234.169865 |  86.73187  84.341265   83.57937  0.0  83.907028 |  233.02539   233.02539   233.02539  0.0   233.02539 | 85.051504  85.051504  85.051504  0.0  85.051504 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.330183  0.203319  0.028211  0.0  0.0 |      0.0 | 0.270977 |      0.0 | 0.013962 | 0.435864 | 0.452397 | 0.452397 | 1.884987 |      2.0       2.0  0.0  0.0       2.0 | 376.143247  380.033392  381.112456  0.0  380.606431 | 141.933647  138.043501  136.964438  0.0  137.470462 | 142.279039  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  6.439946  3.965565  0.550232  0.0  0.0 | 26.962447 | 12.558487 | 0.126515  0.126909  0.127619  0.128535  0.129523 | 1.884987 |
    | 03/01 10:00 |  0.3 | 5.0 | 14.9 |  4.359715 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 518.945213  518.945213  518.945213  0.0  518.945213 | 0.0  0.0  0.902686  0.0   0.0 |  0.3168   0.76464    1.7784   0.3168    1.3104 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.3168   0.76464    1.7784   0.3168    1.3104 | 231.024488  233.747413  235.584149  0.0  234.852459 | 87.369205   85.09412  84.271144  0.0  84.534834 |    233.209     233.209     233.209  0.0     233.209 | 85.736213  85.736213  85.736213  0.0  85.736213 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   0.3168  0.0 |      0.0       0.0  0.0   0.3168  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  0.31408  0.193403  0.026835  0.0  0.0 | 0.040229 | 0.268481 | 0.005029 | 0.013949 | 0.425236 | 0.441015 | 0.441015 | 1.837562 |      2.0       2.0  0.0  0.0       2.0 | 375.980775  379.871939  381.155706  0.0  380.613372 | 142.964438  139.073275  137.789507  0.0  138.331841 | 142.279039  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  6.125866  3.772162  0.523397  0.0  0.0 | 26.734194 | 12.549567 | 0.123426  0.123799  0.124471  0.125341  0.126284 | 1.837562 |
    | 03/01 11:00 |  2.6 | 5.0 | 15.3 |  5.305753 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 526.470653  526.470653  526.470653  0.0  526.470653 | 0.0  0.0  0.902686  0.0   0.0 |  2.7456   6.62688   15.4128   2.7456   11.3568 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   2.7456   6.62688   15.4128   2.7456   11.3568 | 231.841463   237.12537  248.140487  0.0  244.141138 |  89.84935  88.446724  86.217526  0.0  86.160875 | 238.174723  238.174723  238.174723  0.0  238.174723 | 88.295931  88.295931  88.295931  0.0  88.295931 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   2.7456  0.0 |      0.0       0.0  0.0   2.7456  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.298762  0.183971  0.025526  0.0  0.0 | 0.348648 | 0.267747 | 0.043581 |  0.01396 | 0.416678 |  0.43023 |  0.43023 | 1.792626 |      2.0       2.0  0.0  0.0       2.0 | 379.967051  383.881611  387.242741  0.0  386.343757 | 146.503602  142.589042  139.227912  0.0  140.126897 | 142.279039  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  5.827104  3.588192  0.497871  0.0  0.0 | 26.815095 | 12.579187 | 0.120924  0.121227  0.121782  0.122512  0.123325 | 1.792626 |
    | 03/01 12:00 |  0.7 | 5.0 | 15.3 |  5.376027 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 528.496733  528.496733  528.496733  0.0  528.496733 | 0.0  0.0  0.902686  0.0   0.0 |  0.7392   1.78416    4.1496   0.7392    3.0576 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.7392   1.78416    4.1496   0.7392    3.0576 |  235.82443  239.030617  243.869762  0.0  242.326539 | 91.385423  89.224197  86.750491  0.0  87.201714 | 239.167726  239.167726  239.167726  0.0  239.167726 | 89.329007  89.329007  89.329007  0.0  89.329007 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   0.7392  0.0 |      0.0       0.0  0.0   0.7392  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.284191  0.174998  0.024281  0.0  0.0 | 0.093867 | 0.267282 | 0.011733 | 0.013976 |  0.40876 | 0.420776 | 0.420776 | 1.753232 |      2.0       2.0  0.0  0.0       2.0 | 380.963364  384.844961  387.330305  0.0  386.582544 | 147.533369  143.651772  141.166428  0.0  141.914189 | 142.279039  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  5.542913  3.413193  0.473589  0.0  0.0 | 26.641679 | 12.576945 | 0.118621  0.118899  0.119402  0.120057  0.120775 | 1.753232 |
    | 03/01 13:00 |  0.3 | 5.0 | 15.4 |  4.658915 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 529.365053  529.365053  529.365053  0.0  529.365053 | 0.0  0.0  0.902686  0.0   0.0 |  0.3168   0.76464    1.7784   0.3168    1.3104 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.3168   0.76464    1.7784   0.3168    1.3104 | 236.881059  239.627403  242.351416  0.0  241.464963 | 91.932474   89.63397  87.923717  0.0   88.34217 | 239.341086  239.341086  239.341086  0.0  239.341086 | 90.023967  90.023967  90.023967  0.0  90.023967 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   0.3168  0.0 |      0.0       0.0  0.0   0.3168  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.270331  0.166463  0.023097  0.0  0.0 | 0.040229 |  0.26529 | 0.005029 | 0.013969 | 0.399762 | 0.412329 | 0.412329 | 1.718037 |      2.0       2.0  0.0  0.0       2.0 | 381.076407  384.935604  386.738375  0.0  386.109067 | 148.288646  144.429449  142.626678  0.0  143.255986 | 142.279039  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  5.272582   3.24673  0.450492  0.0  0.0 | 26.416618 | 12.568004 | 0.116022  0.116333   0.11689  0.117598  0.118345 | 1.718037 |
    | 03/01 14:00 |  0.3 | 5.0 | 15.5 |  7.789594 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 530.233373  530.233373  530.233373  0.0  530.233373 | 0.0  0.0  0.902686  0.0   0.0 |  0.3168   0.76464    1.7784   0.3168    1.3104 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.3168   0.76464    1.7784   0.3168    1.3104 | 237.187616  239.922259  242.222296  0.0  241.409523 | 92.494237  90.207435  88.921158  0.0  89.265931 | 239.515656  239.515656  239.515656  0.0  239.515656 | 90.717717  90.717717  90.717717  0.0  90.717717 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   0.3168  0.0 |      0.0       0.0  0.0   0.3168  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.257147  0.158345  0.021971  0.0  0.0 | 0.040229 |  0.26305 | 0.005029 | 0.013959 | 0.390877 | 0.403727 | 0.403727 | 1.682196 |      2.0       2.0  0.0  0.0       2.0 | 381.057463  384.905961  386.450135  0.0    385.8656 |  149.17591  145.327412  143.783238  0.0  144.367773 | 142.279039  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  5.015435  3.088385  0.428521  0.0  0.0 | 26.193796 | 12.559073 | 0.113444  0.113754  0.114313   0.11503  0.115797 | 1.682196 |
    | 03/01 15:00 |  0.0 | 5.0 | 17.4 |  4.851567 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 530.233373  530.233373  530.233373  0.0  530.233373 | 0.0  0.0  0.902686  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 237.325483  239.722357   240.68408  0.0  240.320027 |  92.90789  90.511016  89.549293  0.0  89.913346 | 239.068393  239.068393  239.068393  0.0  239.068393 |  91.16498   91.16498   91.16498  0.0   91.16498 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.244606  0.150622  0.020899  0.0  0.0 |      0.0 | 0.260633 |      0.0 | 0.013947 | 0.382155 | 0.394887 | 0.394887 | 1.645362 |      2.0       2.0  0.0  0.0       2.0 |  380.45339  384.291998  385.474448  0.0  384.953966 | 149.779984  145.941376  144.758925  0.0  145.279407 | 142.279039  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |   4.77083  2.937763  0.407622  0.0  0.0 | 25.933164 | 12.545126 | 0.110913  0.111218  0.111767  0.112474  0.113234 | 1.645362 |
    | 03/01 16:00 |  0.0 | 5.0 | 17.3 |   5.30692 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 530.233373  530.233373  530.233373  0.0  530.233373 | 0.0  0.0  0.902686  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 236.949262  239.339976  240.076415  0.0  239.752255 | 93.284112  90.893398  90.156958  0.0  90.481119 | 238.622132  238.622132  238.622132  0.0  238.622132 | 91.611242  91.611242  91.611242  0.0  91.611242 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.232676  0.143276   0.01988  0.0  0.0 |      0.0 | 0.258039 |      0.0 | 0.013931 | 0.373581 | 0.386106 | 0.386106 | 1.608776 |      2.0       2.0  0.0  0.0       2.0 | 379.779276  383.614153  384.660165  0.0  384.163843 | 150.454097   146.61922  145.573208  0.0   146.06953 | 142.279039  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  4.538153  2.794486  0.387742  0.0  0.0 | 25.675125 | 12.531195 | 0.108425  0.108725  0.109264  0.109959  0.110707 | 1.608776 |
    | 03/01 17:00 |  0.0 | 5.0 | 17.1 |  3.286036 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 530.233373  530.233373  530.233373  0.0  530.233373 | 0.0  0.0  0.902686  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 236.529419  238.917809  239.569273  0.0   239.26016 | 93.703955  91.315564    90.6641  0.0  90.973213 |  238.17587   238.17587   238.17587  0.0   238.17587 | 92.057503  92.057503  92.057503  0.0  92.057503 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.221328  0.136289   0.01891  0.0  0.0 |      0.0 | 0.255472 |      0.0 | 0.013916 |  0.36533 | 0.377478 | 0.377478 | 1.572825 |      2.0       2.0  0.0  0.0       2.0 | 379.078744  382.912214  383.906762  0.0  383.419553 | 151.154629  147.321159  146.326612  0.0   146.81382 | 142.279039  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  4.316825  2.658198  0.368832  0.0  0.0 | 25.419653 | 12.517279 | 0.106029  0.106318  0.106839  0.107511  0.108237 | 1.572825 |
    | 03/01 18:00 |  0.0 | 5.0 | 15.7 |  1.506216 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 529.715546  530.233373  530.233373  0.0  530.233373 | 0.0  0.0  0.902686  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 236.093122  238.480636  239.100048  0.0  238.796611 | 94.140252  91.752737  91.133325  0.0  91.436763 | 237.729608  237.729608  237.729608  0.0  237.729608 | 92.503765  92.503765  92.503765  0.0  92.503765 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 0.517828        0.0  0.0      0.0  0.0 | 0.262064       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.262064       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.262064       0.0  0.0  0.0  0.0 | 0.216978  0.129642  0.017988  0.0  0.0 |      0.0 |  0.25293 |      0.0 |   0.0139 | 0.359536 |  0.36925 |  0.36925 | 1.538543 |      2.0       2.0  0.0  0.0       2.0 | 378.368247  382.201186  383.176322  0.0  382.692551 | 151.347299  148.032187  147.057051  0.0  147.540823 | 142.534802  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  4.361911  2.528556  0.350844  0.0  0.0 | 25.166723 | 12.503379 | 0.104325  0.104533  0.104916   0.10543  0.106015 | 1.538543 |
    | 03/01 19:00 |  0.0 | 5.0 | 14.2 |  0.274762 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 528.835785  530.034209  530.034209  0.0  530.034209 | 0.0  0.0  0.902686  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 235.511104  238.037803  238.645125  0.0  238.343828 | 94.204442  92.195571  91.588249  0.0  91.889545 | 237.229687  237.229687  237.229687  0.0  237.229687 | 92.804522  92.804522  92.804522  0.0  92.804522 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 1.198424        0.0  0.0      0.0  0.0 | 0.608685       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.608685       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.608685       0.0  0.0  0.0  0.0 |   0.2277  0.123319  0.017111  0.0  0.0 |      0.0 | 0.250413 |      0.0 | 0.013885 | 0.358885 | 0.362744 | 0.362744 | 1.511433 |      2.0       2.0  0.0  0.0       2.0 | 377.739846  381.433071  382.400885  0.0   381.91841 | 151.095939  148.601138  147.633324  0.0  148.115799 | 143.124542  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  4.742896  2.405237  0.333733  0.0  0.0 |  24.91631 | 12.489494 | 0.104085   0.10412  0.104203  0.104356  0.104596 | 1.511433 |
    | 03/01 20:00 |  0.0 | 5.0 | 12.4 |  0.087565 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 |  528.03676  529.573277  529.573277  0.0  529.573277 | 0.0  0.0  0.902686  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 234.882703  237.505353  238.107977  0.0  237.807557 | 93.953081  92.528856  91.926231  0.0  92.226652 | 236.658955  236.658955  236.658955  0.0  236.658955 | 92.914321  92.914321  92.914321  0.0  92.914321 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 1.536516        0.0  0.0      0.0  0.0 | 0.786874       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.786874       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.786874       0.0  0.0  0.0  0.0 | 0.250662  0.117305  0.016276  0.0  0.0 |      0.0 | 0.247921 |      0.0 |  0.01387 | 0.362429 | 0.359985 | 0.359985 | 1.499936 |      2.0       2.0  0.0  0.0       2.0 | 377.169114  380.626673  381.591862  0.0  381.109808 | 150.867646  148.946603  147.981414  0.0  148.463468 | 143.874184  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  5.279109  2.287932  0.317456  0.0  0.0 | 24.668388 | 12.475624 |  0.10507  0.104957  0.104771  0.104573  0.104434 | 1.499936 |
    | 03/01 21:00 |  0.0 | 5.0 | 11.4 |  0.085771 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 527.188356  528.982309  528.982309  0.0  528.982309 | 0.0  0.0  0.902686  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 234.311972  236.878229  237.478902  0.0  237.178902 | 93.724789  92.695048  92.094374  0.0  92.394375 | 236.052952  236.052952  236.052952  0.0  236.052952 | 92.929356  92.929356  92.929356  0.0  92.929356 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 1.793952        0.0  0.0      0.0  0.0 |  0.92836       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.92836       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  0.92836       0.0  0.0  0.0  0.0 | 0.280292  0.111584  0.015483  0.0  0.0 |      0.0 | 0.245455 |      0.0 | 0.013854 | 0.368304 |  0.36145 |  0.36145 |  1.50604 |      2.0       2.0  0.0  0.0       2.0 | 376.563112  379.841397  380.805913  0.0  380.323859 | 150.625245  149.140912  148.176396  0.0   148.65845 | 144.739776  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  5.927177  2.176348  0.301974  0.0  0.0 | 24.422934 |  12.46177 |  0.10675  0.106551  0.106203  0.105781  0.105375 |  1.50604 |
    | 03/01 22:00 |  0.0 | 5.0 | 10.6 |  0.084317 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 526.302602  528.292327  528.292327  0.0  528.292327 | 0.0  0.0  0.902686  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 233.705969  236.229261  236.829108  0.0  236.529311 | 93.482388  92.753048  92.153201  0.0  92.452998 | 235.420271  235.420271  235.420271  0.0  235.420271 | 92.872056  92.872056  92.872056  0.0  92.872056 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 1.989725        0.0  0.0      0.0  0.0 | 1.042099       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.042099       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.042099       0.0  0.0  0.0  0.0 | 0.314695  0.106142  0.014727  0.0  0.0 |      0.0 | 0.243012 |      0.0 | 0.013839 | 0.375876 | 0.366049 | 0.366049 | 1.525203 |      2.0       2.0  0.0  0.0       2.0 |  375.93043  379.072407  380.037076  0.0  379.554818 | 150.372172   149.21992  148.255252  0.0  148.737509 | 145.687402  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |   6.65458  2.070206  0.287246  0.0  0.0 | 24.179922 | 12.447932 | 0.108928  0.108668  0.108208  0.107634  0.107049 | 1.525203 |
    | 03/01 23:00 |  0.0 | 5.0 | 10.0 |  0.083215 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 525.388479  527.527048  527.527048  0.0  527.527048 | 0.0  0.0  0.902686  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 | 0.001664  0.001664  0.001664  0.001664  0.001664 | 0.001065  0.002397  0.005059  0.001065  0.003728 | 0.001065  0.002397  0.0  0.0  0.003728 |      0.0       0.0       0.0      0.0       0.0 | 233.073287  235.563828  236.163294  0.0  235.863608 | 93.229315  92.728499  92.129034  0.0  92.428719 | 234.767326  234.767326  234.767326  0.0  234.767326 | 92.759722  92.759722  92.759722  0.0  92.759722 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.138569        0.0  0.0      0.0  0.0 | 1.134769       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.134769       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.001065  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.134769       0.0  0.0  0.0  0.0 |  0.35245  0.100965  0.014009  0.0  0.0 |      0.0 | 0.240594 |      0.0 | 0.013823 | 0.384666 | 0.372743 | 0.372743 | 1.553097 | 1.998935  1.997603  0.0  0.0  1.996272 | 375.277485  378.315905  379.281108  0.0  378.798536 | 150.110994  149.211143   148.24594  0.0  148.728512 | 146.691202  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  7.436899  1.969241  0.273237  0.0  0.0 | 23.939192 | 12.434091 | 0.111465  0.111161   0.11062  0.109937  0.109227 | 1.553097 |
    | 04/01 00:00 |  0.0 | 5.0 |  9.5 |  0.082289 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 524.451307  526.704522  526.704522  0.0  526.704522 | 0.0  0.0  0.903096  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |  0.00576   0.00576   0.00576   0.00576   0.00576 | 0.003687  0.008295  0.017511  0.003687  0.012903 | 0.003687  0.008295  0.0  0.0  0.012903 |      0.0       0.0       0.0      0.0       0.0 | 232.420342   234.88595  235.485218  0.0  235.185602 | 92.968137  92.641098  92.041831  0.0  92.341446 | 234.098982  234.098982  234.098982  0.0  234.098982 |  92.60554   92.60554   92.60554  0.0   92.60554 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.253215        0.0  0.0      0.0  0.0 | 1.212134       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.212134       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.003687  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.212134       0.0  0.0  0.0  0.0 | 0.392506  0.096041  0.013326  0.0  0.0 |      0.0 | 0.238199 |      0.0 | 0.013808 | 0.394318 | 0.380906 | 0.380906 | 1.587109 | 1.995248  1.989309  0.0  0.0  1.983369 | 374.608076  377.568937  378.534872  0.0  378.051916 | 149.843231  149.135584  148.169649  0.0  148.652606 | 147.732283  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  8.256527    1.8732  0.259911  0.0  0.0 | 23.700525 | 12.420225 | 0.114255   0.11392  0.113322  0.112562  0.111762 | 1.587109 |
    | 04/01 01:00 |  0.0 | 5.0 | 10.7 |    0.0845 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 523.498661  525.837901  525.837901  0.0  525.837901 | 0.0  0.0  0.903096  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 231.750934  234.198633  234.797784  0.0  234.498215 | 92.700373  92.505888  91.906738  0.0  92.206306 | 233.418521  233.418521  233.418521  0.0  233.418521 |  92.41938   92.41938   92.41938  0.0   92.41938 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |  2.33924        0.0  0.0      0.0  0.0 | 1.276338       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.276338       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.276338       0.0  0.0  0.0  0.0 | 0.434059  0.091357  0.012676  0.0  0.0 |      0.0 | 0.235824 |      0.0 | 0.013793 | 0.404559 | 0.390112 | 0.390112 | 1.625466 | 1.995248  1.989309  0.0  0.0  1.983369 | 373.927615  376.828825  377.795609  0.0  377.312221 | 149.571046  149.009076  148.042292  0.0   148.52568 | 148.795185  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  9.098805  1.781843  0.247235  0.0  0.0 | 23.464701 | 12.406432 | 0.117219  0.116863  0.116225  0.115412   0.11455 | 1.625466 |
    | 04/01 02:00 |  0.0 | 5.0 | 10.9 |  0.084864 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 522.533546  524.938193  524.938193  0.0  524.938193 | 0.0  0.0  0.903096  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 231.070472  233.503734  234.102806  0.0  233.803273 | 92.428189  92.334167  91.735094  0.0  92.034628 | 232.729153  232.729153  232.729153  0.0  232.729153 |  92.20904   92.20904   92.20904  0.0   92.20904 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.404647        0.0  0.0      0.0  0.0 | 1.330972       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.330972       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.330972       0.0  0.0  0.0  0.0 | 0.476481  0.086902  0.012058  0.0  0.0 |      0.0 | 0.233478 |      0.0 | 0.013777 | 0.415183 | 0.400043 | 0.400043 | 1.666847 | 1.995248  1.989309  0.0  0.0  1.983369 | 373.238247  376.094244  377.061956  0.0  376.578101 | 149.295299  148.843949  147.876237  0.0  148.360092 | 149.868859  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  9.953297  1.694942  0.235177  0.0  0.0 | 23.231223 | 12.392655 | 0.120296  0.119926  0.119262  0.118414   0.11751 | 1.666847 |
    | 04/01 03:00 |  1.3 | 5.0 | 10.2 |  0.083584 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 524.784249  527.769652  527.769652  0.0  527.769652 | 0.0  0.0  0.903096  0.0   0.0 |  1.3728   3.31344    7.7064   1.3728    5.6784 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 | 1.368048  3.302749    7.7064   1.3728  5.661769 | 230.754143  234.603676  240.293649  0.0  238.161507 | 93.147452  93.637266  92.350944  0.0  92.438455 | 234.709836  234.709836  234.709836  0.0  234.709836 | 93.059816  93.059816  93.059816  0.0  93.059816 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.985403        0.0  0.0   1.3728  0.0 | 1.676354       0.0  0.0   1.3728  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.676354       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.676354       0.0  0.0  0.0  0.0 | 0.526647  0.082663   0.01147  0.0  0.0 | 0.174324 | 0.232023 |  0.02179 | 0.013774 | 0.429297 | 0.410694 | 0.410694 | 1.711227 |      2.0       2.0  0.0  0.0       2.0 | 374.845892  377.695853  379.824543  0.0    379.1282 | 149.938357  150.073799   147.94511  0.0  148.641453 | 151.177908  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 11.103005  1.612278  0.223708  0.0  0.0 | 23.173524 | 12.400671 | 0.124355   0.12387  0.123012  0.121939  0.120835 | 1.711227 |
    | 04/01 04:00 |  0.0 | 5.0 | 10.1 |    0.0834 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 524.240982   526.62142   526.62142  0.0   526.62142 | 0.0  0.0  0.903096  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 | 0.000834  0.000834  0.000834  0.000834  0.000834 | 0.000534  0.001201  0.002535  0.000534  0.001868 | 0.000534  0.001201  0.0  0.0  0.001868 |      0.0       0.0       0.0      0.0       0.0 | 231.988749    234.5668  235.888816  0.0  235.456354 |   92.7955  93.202852  91.880837  0.0  92.313298 |  233.94875   233.94875   233.94875  0.0   233.94875 | 92.672671  92.672671  92.672671  0.0  92.672671 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.380438        0.0  0.0      0.0  0.0 | 1.360109       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.360109       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.000534  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.360109       0.0  0.0  0.0  0.0 | 0.574943  0.078632   0.01091  0.0  0.0 |      0.0 |  0.23058 |      0.0 | 0.013771 | 0.442859 | 0.423238 | 0.423238 |  1.76349 | 1.999466  1.998799  0.0  0.0  1.998132 | 374.457844  377.117803  378.524477  0.0  377.960595 | 149.783138  149.503617  148.096944  0.0  148.660825 | 152.198238  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  11.88817  1.533647  0.212797  0.0  0.0 | 22.942876 | 12.386892 | 0.128292  0.127818  0.126964  0.125866  0.124693 |  1.76349 |
    | 04/01 05:00 |  0.0 | 5.0 | 10.9 |  0.084864 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 523.269541  525.705867  525.705867  0.0  525.705867 | 0.0  0.0  0.903096  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 231.600701  233.896206  234.768654  0.0  234.418924 | 92.640281  92.725214  91.852766  0.0  92.202497 | 233.254863  233.254863  233.254863  0.0  233.254863 | 92.451004  92.451004  92.451004  0.0  92.451004 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.436327        0.0  0.0      0.0  0.0 | 1.410895       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.410895       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.410895       0.0  0.0  0.0  0.0 | 0.614485  0.074797  0.010378  0.0  0.0 |      0.0 | 0.228285 |      0.0 | 0.013756 | 0.452755 | 0.436481 | 0.436481 | 1.818669 | 1.999466  1.998799  0.0  0.0  1.998132 | 373.763958   376.51646  377.650685  0.0  377.136535 | 149.505583  149.189408  148.055182  0.0  148.569332 | 153.223669  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 12.684581   1.45885  0.202419  0.0  0.0 |  22.71459 | 12.373136 | 0.131197  0.130844  0.130193  0.129326  0.128347 | 1.818669 |
    | 04/01 06:00 |  0.0 | 5.0 | 11.2 |  0.310229 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 522.290279  524.768818  524.768818  0.0  524.768818 | 0.0  0.0  0.903096  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 230.906815  233.274208  233.976928  0.0  233.658382 | 92.362726  92.431659  91.728939  0.0  92.047486 |  232.55539   232.55539   232.55539  0.0   232.55539 | 92.213428  92.213428  92.213428  0.0  92.213428 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |  2.47854        0.0  0.0      0.0  0.0 | 1.454748       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.454748       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.454748       0.0  0.0  0.0  0.0 | 0.654404  0.071149  0.009872  0.0  0.0 |      0.0 | 0.226014 |      0.0 |  0.01374 | 0.462853 | 0.447971 | 0.447971 | 1.866546 | 1.999466  1.998799  0.0  0.0  1.998132 | 373.064485  375.837642  376.869147  0.0  376.373543 | 149.225794  148.931177  147.899671  0.0  148.395275 | 154.247462  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 13.484924  1.387701  0.192547  0.0  0.0 | 22.488576 | 12.359396 | 0.134125  0.133773  0.133138   0.13232  0.131433 | 1.866546 |
    | 04/01 07:00 |  0.7 | 5.0 | 10.8 |  1.391958 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 523.041495  525.840895  525.840895  0.0  525.840895 | 0.0  0.0  0.903096  0.0   0.0 |  0.7392   1.78416    4.1496   0.7392    3.0576 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 | 0.738666  1.782959    4.1496   0.7392  3.055732 | 230.409097   233.57062  236.944072  0.0  235.658291 | 92.619848  92.981158  91.974347  0.0  92.166259 | 233.293729  233.293729  233.293729  0.0  233.293729 | 92.547167  92.547167  92.547167  0.0  92.547167 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |   2.7994        0.0  0.0   0.7392  0.0 | 1.665103       0.0  0.0   0.7392  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.665103       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.665103       0.0  0.0  0.0  0.0 |  0.69861  0.067679  0.009391  0.0  0.0 | 0.093867 | 0.224233 | 0.011733 | 0.013732 | 0.474896 | 0.458427 | 0.458427 | 1.910113 |      2.0       2.0  0.0  0.0       2.0 | 373.601068  376.386461  378.008404  0.0  377.404713 | 149.440427  149.454435  147.832491  0.0  148.436183 | 155.381759  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 14.451417  1.320022  0.183156  0.0  0.0 |  22.35821 | 12.357398 | 0.137599  0.137183  0.136442  0.135506  0.134528 | 1.910113 |
    | 04/01 08:00 |  0.4 | 5.0 | 10.4 |  3.195876 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 523.281753  525.921963  525.921963  0.0  525.921963 | 0.0  0.0  0.903096  0.0   0.0 |  0.4224   1.01952    2.3712   0.4224    1.7472 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.4224   1.01952    2.3712   0.4224    1.7472 | 230.859201  233.786185  236.354275  0.0  235.421637 | 92.604694   93.07423   91.85782  0.0  92.166459 | 233.378862  233.378862  233.378862  0.0  233.378862 | 92.543101  92.543101  92.543101  0.0  92.543101 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |  2.64021        0.0  0.0   0.4224  0.0 | 1.593597       0.0  0.0   0.4224  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.593597       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.593597       0.0  0.0  0.0  0.0 | 0.743988  0.064378  0.008933  0.0  0.0 | 0.053638 | 0.222735 | 0.006705 | 0.013727 | 0.487648 | 0.469708 | 0.469708 | 1.957115 |      2.0       2.0  0.0  0.0       2.0 | 373.772681  376.468418  378.044191  0.0  377.449138 | 149.509072  149.453545  147.877772  0.0  148.472825 | 156.428372  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 15.301026  1.255644  0.174224  0.0  0.0 | 22.189113 | 12.350376 | 0.141289  0.140846  0.140052   0.13904   0.13797 | 1.957115 |
    | 04/01 09:00 |  0.1 | 5.0 | 10.6 |  5.191651 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 522.680667  525.195938  525.195938  0.0  525.195938 | 0.0  0.0  0.903096  0.0   0.0 |  0.1056   0.25488    0.5928   0.1056    0.4368 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.1056   0.25488    0.5928   0.1056    0.4368 | 230.944361  233.442269  234.809309  0.0  234.301005 | 92.442992  92.734574  91.705454  0.0  92.057758 | 232.863063  232.863063  232.863063  0.0  232.863063 | 92.332875  92.332875  92.332875  0.0  92.332875 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |  2.51527        0.0  0.0   0.1056  0.0 | 1.538706       0.0  0.0   0.1056  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.538706       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.538706       0.0  0.0  0.0  0.0 | 0.784074  0.061238  0.008497  0.0  0.0 |  0.01341 | 0.220852 | 0.001676 | 0.013716 | 0.498318 | 0.481848 | 0.481848 | 2.007699 |      2.0       2.0  0.0  0.0       2.0 | 373.343334  376.041531  377.330745  0.0  376.787995 | 149.337334  149.154406  147.865192  0.0  148.407942 | 157.404936  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 16.055658  1.194405  0.165727  0.0  0.0 |  21.98167 | 12.338336 | 0.144404  0.144027   0.14334  0.142443  0.141455 | 2.007699 |
    | 04/01 10:00 |  0.4 | 5.0 | 12.0 |  7.155036 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 |  522.72229  525.386286  525.386286  0.0  525.386286 | 0.0  0.0  0.903096  0.0   0.0 |  0.4224   1.01952    2.3712   0.4224    1.7472 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.4224   1.01952    2.3712   0.4224    1.7472 | 230.601546  233.397025  235.757443  0.0  234.863073 | 92.501521  92.818433  91.809695  0.0  92.080065 | 232.979326  232.979326  232.979326  0.0  232.979326 |  92.40696   92.40696   92.40696  0.0   92.40696 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.663996        0.0  0.0   0.4224  0.0 |   1.6501       0.0  0.0   0.4224  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   1.6501       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |   1.6501       0.0  0.0  0.0  0.0 | 0.823617  0.058252  0.008083  0.0  0.0 | 0.053638 | 0.218989 | 0.006705 | 0.013705 |  0.50887 | 0.493323 | 0.493323 | 2.055511 |      2.0       2.0  0.0  0.0       2.0 | 373.373065  376.113112  377.563828  0.0  376.991448 | 149.349226  149.273174  147.822458  0.0  148.394838 | 158.418832  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 16.882141  1.136153  0.157644  0.0  0.0 |  21.81632 | 12.331336 | 0.147466  0.147097  0.146432  0.145574  0.144647 | 2.055511 |
    | 04/01 11:00 |  0.0 | 5.0 | 16.1 |  8.391432 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 521.869966  524.361672  524.361672  0.0  524.361672 | 0.0  0.0  0.903096  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 230.515922  232.937273  233.835741  0.0  233.481251 | 92.206369  92.449013  91.550545  0.0  91.905035 | 232.255167  232.255167  232.255167  0.0  232.255167 | 92.106505  92.106505  92.106505  0.0  92.106505 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.491706        0.0  0.0      0.0  0.0 | 1.563329       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.563329       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.563329       0.0  0.0  0.0  0.0 | 0.861792  0.055411  0.007688  0.0  0.0 |      0.0 | 0.217076 |      0.0 | 0.013694 | 0.518963 | 0.504057 | 0.504057 | 2.100236 |      2.0       2.0  0.0  0.0       2.0 | 372.764262  375.471006  376.623254  0.0  376.105365 | 149.105705  148.890666  147.738418  0.0  148.256308 | 159.347209  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 17.583678  1.080743  0.149956  0.0  0.0 | 21.599244 | 12.317642 | 0.150397  0.150044  0.149406  0.148583  0.147692 | 2.100236 |
    | 04/01 12:00 |  0.0 | 5.0 | 17.2 |  8.391286 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 520.882597  523.403324  523.403324  0.0  523.403324 | 0.0  0.0  0.903096  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 229.907119  232.260308   232.97307  0.0  232.652712 | 91.962848  92.101364  91.388603  0.0   91.70896 | 231.549904  231.549904  231.549904  0.0  231.549904 |  91.85342   91.85342   91.85342  0.0   91.85342 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.520726        0.0  0.0      0.0  0.0 | 1.600128       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.600128       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.600128       0.0  0.0  0.0  0.0 | 0.896911  0.052708  0.007313  0.0  0.0 |      0.0 | 0.214916 |      0.0 | 0.013679 | 0.527844 | 0.514289 | 0.514289 |  2.14287 |      2.0       2.0  0.0  0.0       2.0 | 372.058998  374.800602  375.840088  0.0  375.342557 | 148.823599  148.602722  147.563236  0.0  148.060767 | 160.267807  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 18.286895  1.028034  0.142642  0.0  0.0 | 21.384328 | 12.303963 | 0.152984  0.152672  0.152104  0.151364  0.150553 |  2.14287 |
    | 04/01 13:00 |  0.0 | 5.0 | 21.4 | 10.715238 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 519.891085  522.433814  522.433814  0.0  522.433814 | 0.0  0.0  0.903096  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 229.201855  231.583857   232.22614  0.0  231.918723 | 91.680742  91.819467  91.177184  0.0  91.484601 | 230.841681  230.841681  230.841681  0.0  230.841681 | 91.592133  91.592133  91.592133  0.0  91.592133 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.542729        0.0  0.0      0.0  0.0 | 1.632799       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.632799       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.585655       0.0  0.0  0.0  0.0 | 0.930851  0.050138  0.006957  0.0  0.0 | 0.014966 | 0.212852 | 0.001871 | 0.013665 | 0.536461 |  0.52371 |  0.52371 | 2.182126 |      2.0       2.0  0.0  0.0       2.0 | 371.350775  374.098426  375.095628  0.0  374.605514 |  148.54031  148.335388  147.338185  0.0  147.828299 | 161.177737  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 18.941699  0.977896  0.135686  0.0  0.0 | 21.186442 |  12.29217 | 0.155486  0.155185  0.154641  0.153938  0.153177 | 2.182126 |
    | 04/01 14:00 |  0.0 | 5.0 | 20.4 |  9.383394 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 518.896368  521.455841  521.455841  0.0  521.455841 | 0.0  0.0  0.903096  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 228.493632  230.884715  231.500165  0.0  231.197678 | 91.397453  91.549099  90.933649  0.0  91.236136 | 230.131169  230.131169  230.131169  0.0  230.131169 | 91.324673  91.324673  91.324673  0.0  91.324673 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.559473        0.0  0.0      0.0  0.0 | 1.662266       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.662266       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.552915       0.0  0.0  0.0  0.0 | 0.961981  0.047693  0.006617  0.0  0.0 | 0.034715 | 0.210982 | 0.004339 | 0.013653 | 0.544361 | 0.532494 | 0.532494 | 2.218725 |      2.0       2.0  0.0  0.0       2.0 | 370.640263  373.384879  374.366632  0.0  373.879005 | 148.256105  148.070962  147.089209  0.0  147.576836 | 162.074944  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 19.532633  0.930204  0.129068  0.0  0.0 | 21.010175 | 12.282856 | 0.157784  0.157507  0.157005  0.156354  0.155644 | 2.218725 |
    | 04/01 15:00 |  0.0 | 5.0 | 20.8 |  7.861915 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 517.899208  520.471428  520.471428  0.0  520.471428 | 0.0  0.0  0.903096  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |  227.78312  230.176251  230.781461  0.0  230.480859 | 91.113248   91.27959  90.674381  0.0  90.974982 | 229.418911  229.418911  229.418911  0.0  229.418911 | 91.052517  91.052517  91.052517  0.0  91.052517 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.572221        0.0  0.0      0.0  0.0 | 1.689196       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.689196       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.523368       0.0  0.0  0.0  0.0 | 0.990075  0.045367  0.006295  0.0  0.0 | 0.052644 | 0.209317 |  0.00658 | 0.013644 | 0.551478 | 0.540679 | 0.540679 | 2.252829 |      2.0       2.0  0.0  0.0       2.0 | 369.928006   372.66754  373.644083  0.0  373.157057 | 147.971202  147.803889  146.827346  0.0  147.314371 | 162.957969  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 20.065926  0.884837  0.122773  0.0  0.0 | 20.853501 | 12.275793 | 0.159856  0.159606  0.159153  0.158562  0.157916 | 2.252829 |
    | 04/01 16:00 |  0.0 | 5.0 | 21.2 |  6.298329 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 516.900183  519.482113  519.482113  0.0  519.482113 | 0.0  0.0  0.903096  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 227.070863  229.463698  230.064988  0.0   229.76511 | 90.828345   91.00773  90.406441  0.0  90.706318 | 228.705322  228.705322  228.705322  0.0  228.705322 | 90.776791  90.776791  90.776791  0.0  90.776791 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |  2.58193        0.0  0.0      0.0  0.0 | 1.714098       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.714098       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.496704       0.0  0.0  0.0  0.0 | 1.015428  0.043154  0.005988  0.0  0.0 | 0.069014 |  0.20784 | 0.008627 | 0.013637 | 0.557893 | 0.548144 | 0.548144 | 2.283934 |      2.0       2.0  0.0  0.0       2.0 | 369.214416  371.949163  372.924416  0.0  372.437269 | 147.685766   147.53295  146.557696  0.0  147.044844 | 163.825801  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 20.547201  0.841683  0.116786  0.0  0.0 | 20.714676 | 12.270783 | 0.161724  0.161498  0.161089  0.160557  0.159974 | 2.283934 |
    | 04/01 17:00 |  0.0 | 5.0 | 19.1 |  2.948416 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 515.899733  518.489063  518.489063  0.0  518.489063 | 0.0  0.0  0.903096  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 226.357273  228.749175  229.348956  0.0   229.04936 | 90.542909  90.732938  90.133156  0.0  90.432753 | 227.990715  227.990715  227.990715  0.0  227.990715 | 90.498348  90.498348  90.498348  0.0  90.498348 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |  2.58933        0.0  0.0      0.0  0.0 | 1.737369       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.737369       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  1.47264       0.0  0.0  0.0  0.0 | 1.038309  0.041049  0.005696  0.0  0.0 | 0.084041 | 0.206533 | 0.010505 | 0.013632 | 0.563677 | 0.554887 | 0.554887 | 2.312028 |      2.0       2.0  0.0  0.0       2.0 | 368.499809  371.230703  372.206175  0.0  371.718624 | 147.399924  147.258359  146.282888  0.0  146.770439 | 164.677762  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 20.981532  0.800634   0.11109  0.0  0.0 | 20.592183 | 12.267655 | 0.163407  0.163204  0.162836  0.162356   0.16183 | 2.312028 |
    | 04/01 18:00 |  0.0 | 5.0 | 16.8 |  1.309232 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 514.898195  517.493167  517.493167  0.0  517.493167 | 0.0  0.0  0.903096  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 225.642667  228.033583   228.63278  0.0  228.333295 | 90.257067   90.45548  89.856283  0.0  90.155768 | 227.275331  227.275331  227.275331  0.0  227.275331 | 90.217836  90.217836  90.217836  0.0  90.217836 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.594971        0.0  0.0      0.0  0.0 | 1.759311       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.759311       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.450923       0.0  0.0  0.0  0.0 | 1.058957  0.039047  0.005418  0.0  0.0 | 0.097901 | 0.205384 | 0.012238 |  0.01363 | 0.568895 | 0.560966 | 0.560966 |  2.33736 |      2.0       2.0  0.0  0.0       2.0 | 367.784425  370.512451  371.488726  0.0  371.000659 |  147.11377  146.980716  146.004441  0.0  146.492507 | 165.513423  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 21.373498  0.761586  0.105672  0.0  0.0 |   20.4847 | 12.266263 | 0.164926  0.164743   0.16441  0.163977  0.163503 |  2.33736 |
    | 04/01 19:00 |  0.0 | 5.0 | 14.2 |   0.32955 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 513.895825  516.495101  516.495101  0.0  516.495101 | 0.0  0.0  0.903096  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 224.927282  227.317342  227.916308  0.0  227.616869 | 89.970913  90.175824  89.576859  0.0  89.876298 | 226.559352  226.559352  226.559352  0.0  226.559352 | 89.935749  89.935749  89.935749  0.0  89.935749 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.599276        0.0  0.0      0.0  0.0 | 1.780159       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.780159       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.431325       0.0  0.0  0.0  0.0 | 1.077592  0.037143  0.005154  0.0  0.0 | 0.110741 | 0.204378 | 0.013843 | 0.013629 | 0.573606 |  0.56645 |  0.56645 | 2.360209 |      2.0       2.0  0.0  0.0       2.0 | 367.068446   369.79446   370.77177  0.0  370.283143 | 146.827379   146.70064  145.723331  0.0  146.211958 | 166.332539  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 21.727231  0.724443  0.100518  0.0  0.0 | 20.391063 | 12.266476 | 0.166298  0.166132  0.165832  0.165441  0.165013 | 2.360209 |
    | 04/01 20:00 |  0.0 | 5.0 | 13.5 |  0.089508 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 512.892817  515.495379  515.495379  0.0  515.495379 | 0.0  0.0  0.903096  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 224.211303  226.600668  227.199539  0.0   226.90012 | 89.684521  89.894432  89.295562  0.0   89.59498 | 225.842917  225.842917  225.842917  0.0  225.842917 | 89.652462  89.652462  89.652462  0.0  89.652462 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.602563        0.0  0.0      0.0  0.0 | 1.800096       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.800096       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.413638       0.0  0.0  0.0  0.0 | 1.094409  0.035332  0.004902  0.0  0.0 | 0.122685 | 0.203506 | 0.015336 |  0.01363 | 0.577866 |   0.5714 |   0.5714 | 2.380833 |      2.0       2.0  0.0  0.0       2.0 | 366.352012   369.07671  370.055148  0.0   369.56594 | 146.540805   146.41867  145.440231  0.0   145.92944 | 167.135006  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 22.046461  0.689112  0.095616  0.0  0.0 | 20.310242 | 12.268181 | 0.167538  0.167388  0.167117  0.166763  0.166377 | 2.380833 |
    | 04/01 21:00 |  0.0 | 5.0 | 11.4 |  0.085771 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 511.889319  514.494394  514.494394  0.0  514.494394 | 0.0  0.0  0.903096  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 223.494869  225.883686  226.482514  0.0  226.183107 | 89.397948  89.611693  89.012865  0.0  89.312273 | 225.126134  225.126134  225.126134  0.0  225.126134 |  89.36826   89.36826   89.36826  0.0   89.36826 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.605074        0.0  0.0      0.0  0.0 | 1.819261       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.819261       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.397677       0.0  0.0  0.0  0.0 | 1.109585  0.033608  0.004663  0.0  0.0 | 0.133836 | 0.202757 |  0.01673 | 0.013633 | 0.581721 | 0.575872 | 0.575872 | 2.399466 |      2.0       2.0  0.0  0.0       2.0 | 365.635228  368.359157  369.338768  0.0  368.848967 | 146.254091  146.135237  145.155625  0.0  145.645427 |  167.92082  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 22.334553  0.655503  0.090953  0.0  0.0 | 20.241321 | 12.271278 |  0.16866  0.168524  0.168279  0.167959  0.167609 | 2.399466 |
    | 04/01 22:00 |  0.0 | 5.0 | 10.7 |    0.0845 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 510.885446  513.492442  513.492442  0.0  513.492442 | 0.0  0.0  0.903096  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 222.778085  225.166476  225.765282  0.0  225.465882 | 89.111234  89.327917  88.729111  0.0  89.028512 | 224.409082  224.409082  224.409082  0.0  224.409082 |  89.08336   89.08336   89.08336  0.0   89.08336 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.606996        0.0  0.0      0.0  0.0 | 1.837763       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.837763       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.383272       0.0  0.0  0.0  0.0 | 1.123282  0.031969  0.004436  0.0  0.0 | 0.144283 | 0.202124 | 0.018035 | 0.013637 | 0.585216 | 0.579917 | 0.579917 | 2.416322 |      2.0       2.0  0.0  0.0       2.0 | 364.918176  367.641762  368.622568  0.0  368.132167 |  145.96727   145.85068  144.869874  0.0  145.360275 | 168.690053  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 22.594543  0.623534  0.086517  0.0  0.0 |  20.18348 | 12.275676 | 0.169677  0.169554  0.169332  0.169042  0.168725 | 2.416322 |
    | 04/01 23:00 |  0.0 | 5.0 | 10.9 |  0.084864 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 509.881284  512.489751  512.489751  0.0  512.489751 | 0.0  0.0  0.903096  0.0   0.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 222.061033  224.449095  225.047887  0.0  224.748492 | 88.824413  89.043347  88.444555  0.0   88.74395 | 223.691823  223.691823  223.691823  0.0  223.691823 | 88.797929  88.797929  88.797929  0.0  88.797929 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.608467        0.0  0.0      0.0  0.0 | 1.855685       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.855685       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.370273       0.0  0.0  0.0  0.0 | 1.135642   0.03041  0.004219  0.0  0.0 | 0.154099 | 0.201597 | 0.019262 | 0.013643 | 0.588391 | 0.583582 | 0.583582 | 2.431592 |      2.0       2.0  0.0  0.0       2.0 | 364.200917   366.92449  367.906504  0.0  367.415498 | 145.680367  145.565261  144.583248  0.0  145.074254 | 169.442834  128.896759  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 22.829174  0.593124  0.082297  0.0  0.0 | 20.135983 | 12.281296 | 0.170601   0.17049  0.170287  0.170025  0.169737 | 2.431592 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0
"""
# import...
# ...from standard library
from typing import *

# ...from HydPy
from hydpy.exe.modelimports import *
from hydpy.core import masktools
from hydpy.core import modeltools
from hydpy.core.typingtools import *

# ...from hland
from hydpy.models.hland import hland_model
from hydpy.models.hland import hland_masks
from hydpy.models.hland.hland_constants import *


class Model(modeltools.AdHocModel):
    """HBV96-SC/PREVAH version of HydPy-H-Land (|hland_v3|)."""

    INLET_METHODS = ()
    RECEIVER_METHODS = ()
    RUN_METHODS = (
        hland_model.Calc_TC_V1,
        hland_model.Calc_TMean_V1,
        hland_model.Calc_FracRain_V1,
        hland_model.Calc_RFC_SFC_V1,
        hland_model.Calc_PC_V1,
        hland_model.Calc_EP_V1,
        hland_model.Calc_EPC_V1,
        hland_model.Calc_TF_Ic_V1,
        hland_model.Calc_EI_Ic_V1,
        hland_model.Calc_SP_WC_V1,
        hland_model.Calc_SPL_WCL_SP_WC_V1,
        hland_model.Calc_SPG_WCG_SP_WC_V1,
        hland_model.Calc_CFAct_V1,
        hland_model.Calc_Melt_SP_WC_V1,
        hland_model.Calc_Refr_SP_WC_V1,
        hland_model.Calc_In_WC_V1,
        hland_model.Calc_SWE_V1,
        hland_model.Calc_SR_V1,
        hland_model.Calc_GAct_V1,
        hland_model.Calc_GlMelt_In_V1,
        hland_model.Calc_R_SM_V1,
        hland_model.Calc_EA_SM_V1,
        hland_model.Calc_SUZ_V1,
        hland_model.Calc_DP_SUZ_V1,
        hland_model.Calc_RS_RI_SUZ_V1,
        hland_model.Calc_GR1_V1,
        hland_model.Calc_RG1_SG1_V1,
        hland_model.Calc_GR2_GR3_V1,
        hland_model.Calc_RG2_SG2_V1,
        hland_model.Calc_RG3_SG3_V1,
        hland_model.Calc_EL_SG2_SG3_V1,
        hland_model.Calc_InUH_V2,
        hland_model.Calc_OutUH_SC_V1,
        hland_model.Calc_RT_V1,
        hland_model.Calc_QT_V1,
    )
    ADD_METHODS = ()
    OUTLET_METHODS = (hland_model.Pass_Q_v1,)
    SENDER_METHODS = ()
    SUBMODELS = ()

    def check_waterbalance(
        self,
        initial_conditions: Dict[str, Dict[str, ArrayFloat]],
    ) -> float:
        r"""Determine the water balance error of the previous simulation run in mm.

        Method |Model.check_waterbalance| calculates the balance error as follows:

          .. math::
            \sum_{k=1}^{NmbZones} RelZoneAreas^k \cdot \left(
            \sum_{t=t0}^{t1} \big( PC_t^k + GLMelt_t^k - EI_t^k - EA_t^k - EL_t^k \big)
            + \big( IC_{t0}^k - IC_{t1}^k \big)
            + \frac{1}{SClass} \cdot \sum_{c=1}^{SClass}
            \Big( \big( SP_{t0}^{c,k} - SP_{t1}^{c,k} \big)
            + \big( WC_{t0}^{c,k} - WC_{t1}^{c,k} \big) \Big)
            + \big( SM_{t0}^k - SM_{t1}^k \big)
            + \big( SUZ_{t0}^k - SUZ_{t1}^k \big)
            + \big( SG1_{t0}^k - SG1_{t1}^k \big) \right)
            - \sum_{t=t0}^{t1} RT_t
            + RelLowerZoneArea \cdot \Big(
            \big( SG2_{t0} - SG2_{t1} \big)
            + \big( SG3_{t0} - SG3_{t1} \big) \Big)
            + \sum_{i=1}^{NmbStorages} \big( SC_{t0}^i - SC_{t1}^i \big)

        The returned error should always be in scale with numerical precision so
        that it does not affect the simulation results in any relevant manner.

        Pick the required initial conditions before starting the simulation run
        via property |Sequences.conditions|.  See the integration tests of the
        application model |hland_v3| for some examples.
        """
        derived = self.parameters.derived
        fluxes = self.sequences.fluxes
        last = self.sequences.states
        first = initial_conditions["states"]
        areas = derived.relzoneareas.value
        zonetypes = self.parameters.control.zonetype.values
        idxs_lake = zonetypes == ILAKE
        idxs_glac = zonetypes == GLACIER
        idxs_seal = zonetypes == SEALED
        idxs_land = ~idxs_lake
        idxs_upper = idxs_land * ~idxs_seal
        idxs_soil = idxs_upper * ~idxs_glac
        return (
            numpy.sum(fluxes.pc.series * areas)
            + numpy.sum((fluxes.glmelt.series * areas)[:, idxs_glac])
            - numpy.sum((fluxes.ei.series * areas)[:, idxs_land])
            - numpy.sum((fluxes.ea.series * areas)[:, idxs_soil])
            - numpy.sum((fluxes.el.series * areas)[:, idxs_lake])
            - numpy.sum(fluxes.rt.series)
            - numpy.sum(((last.ic - first["ic"]) * areas)[idxs_land])
            - numpy.sum((numpy.mean(last.wc - first["wc"], axis=0) * areas)[idxs_land])
            - numpy.sum((numpy.mean(last.sp - first["sp"], axis=0) * areas)[idxs_land])
            - numpy.sum(((last.sm - first["sm"]) * areas)[idxs_soil])
            - numpy.sum(((last.suz - first["suz"]) * areas)[idxs_upper])
            - numpy.sum(((last.sg1 - first["sg1"]) * areas)[idxs_upper])
            - (last.sg2 - first["sg2"]) * derived.rellowerzonearea
            - (last.sg3 - first["sg3"]) * derived.rellowerzonearea
            - numpy.sum(last.sc - first["sc"])
        )


class Masks(masktools.Masks):
    """Masks applicable to |hland_v3|."""

    CLASSES = hland_masks.Masks.CLASSES


tester = Tester()
cythonizer = Cythonizer()
