/*
 * Decompiled with CFR 0.152.
 */
package com.example;

import com.example.OptimizationRule;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.apache.calcite.plan.RelOptCost;
import org.apache.calcite.rel.RelNode;

public class OptimizationEngine {
    private static final ObjectMapper objectMapper = new ObjectMapper();
    private List<OptimizationRule> rules = new ArrayList<OptimizationRule>();
    private Map<String, Object> optimizationStats = new HashMap<String, Object>();

    public void loadRulesFromFile(String rulesFilePath) throws IOException {
        if (rulesFilePath == null || rulesFilePath.trim().isEmpty()) {
            return;
        }
        File file = new File(rulesFilePath);
        if (!file.exists()) {
            System.err.println("Warning: Optimization rules file not found: " + rulesFilePath);
            return;
        }
        try {
            TypeReference<List<OptimizationRule>> typeRef = new TypeReference<List<OptimizationRule>>(){};
            List loadedRules = (List)objectMapper.readValue(file, (TypeReference)typeRef);
            List activeRules = loadedRules.stream().filter(OptimizationRule::isEnabled).sorted(Comparator.comparingInt(OptimizationRule::getPriority).reversed()).collect(Collectors.toList());
            this.rules.addAll(activeRules);
            System.out.println("Loaded " + activeRules.size() + " optimization rules from " + rulesFilePath);
            for (OptimizationRule rule : activeRules) {
                System.out.println("  - " + rule.getName() + " (priority: " + rule.getPriority() + ")");
            }
        }
        catch (Exception e) {
            System.err.println("Error loading optimization rules from " + rulesFilePath + ": " + e.getMessage());
            throw e;
        }
    }

    public void loadRulesFromJson(String rulesJson) throws IOException {
        if (rulesJson == null || rulesJson.trim().isEmpty()) {
            return;
        }
        try {
            TypeReference<List<OptimizationRule>> typeRef = new TypeReference<List<OptimizationRule>>(){};
            List loadedRules = (List)objectMapper.readValue(rulesJson, (TypeReference)typeRef);
            List activeRules = loadedRules.stream().filter(OptimizationRule::isEnabled).sorted(Comparator.comparingInt(OptimizationRule::getPriority).reversed()).collect(Collectors.toList());
            this.rules.addAll(activeRules);
            System.out.println("Loaded " + activeRules.size() + " optimization rules from JSON");
        }
        catch (Exception e) {
            System.err.println("Error loading optimization rules from JSON: " + e.getMessage());
            throw e;
        }
    }

    public RelNode optimize(RelNode originalNode) {
        if (this.rules.isEmpty()) {
            System.out.println("No optimization rules loaded, returning original plan");
            return originalNode;
        }
        RelNode currentNode = originalNode;
        ArrayList<String> appliedRules = new ArrayList<String>();
        HashMap optimizationDetails = new HashMap();
        System.out.println("Applying " + this.rules.size() + " optimization rules...");
        for (OptimizationRule rule : this.rules) {
            try {
                RelNode optimizedNode;
                if (!this.shouldApplyRule(rule, currentNode) || (optimizedNode = this.applyRule(rule, currentNode)) == currentNode) continue;
                appliedRules.add(rule.getName());
                currentNode = optimizedNode;
                System.out.println("Applied rule: " + rule.getName() + " - " + rule.getDescription());
                HashMap<String, Object> ruleDetails = new HashMap<String, Object>();
                ruleDetails.put("description", rule.getDescription());
                ruleDetails.put("priority", rule.getPriority());
                ruleDetails.put("transformations", rule.getTransformations());
                optimizationDetails.put(rule.getName(), ruleDetails);
            }
            catch (Exception e) {
                System.err.println("Error applying rule " + rule.getName() + ": " + e.getMessage());
            }
        }
        this.optimizationStats.put("appliedRules", appliedRules);
        this.optimizationStats.put("totalRulesConsidered", this.rules.size());
        this.optimizationStats.put("optimizationDetails", optimizationDetails);
        if (appliedRules.isEmpty()) {
            System.out.println("No optimization rules were applied");
        } else {
            System.out.println("Applied " + appliedRules.size() + " optimization rules: " + String.join((CharSequence)", ", appliedRules));
        }
        return currentNode;
    }

    private boolean shouldApplyRule(OptimizationRule rule, RelNode node) {
        if (!rule.isEnabled() || rule.getConditions() == null) {
            return false;
        }
        for (OptimizationRule.Condition condition : rule.getConditions()) {
            if (this.evaluateCondition(condition, node)) continue;
            return false;
        }
        return true;
    }

    private boolean evaluateCondition(OptimizationRule.Condition condition, RelNode node) {
        boolean result = false;
        try {
            switch (condition.getType().toLowerCase()) {
                case "node_type": {
                    result = this.evaluateNodeTypeCondition(condition, node);
                    break;
                }
                case "table_name": {
                    result = this.evaluateTableNameCondition(condition, node);
                    break;
                }
                case "cost_threshold": {
                    result = this.evaluateCostCondition(condition, node);
                    break;
                }
                case "field_count": {
                    result = this.evaluateFieldCountCondition(condition, node);
                    break;
                }
                default: {
                    System.err.println("Unknown condition type: " + condition.getType());
                    return false;
                }
            }
        }
        catch (Exception e) {
            System.err.println("Error evaluating condition " + condition.getType() + ": " + e.getMessage());
            return false;
        }
        return condition.isNegated() ? !result : result;
    }

    private boolean evaluateNodeTypeCondition(OptimizationRule.Condition condition, RelNode node) {
        String nodeType = node.getRelTypeName();
        String expectedType = condition.getValue().toString();
        switch (condition.getOperator().toLowerCase()) {
            case "equals": {
                return nodeType.equals(expectedType);
            }
            case "contains": {
                return nodeType.toLowerCase().contains(expectedType.toLowerCase());
            }
        }
        return false;
    }

    private boolean evaluateTableNameCondition(OptimizationRule.Condition condition, RelNode node) {
        String explanation = node.toString();
        String expectedTable = condition.getValue().toString();
        switch (condition.getOperator().toLowerCase()) {
            case "equals": 
            case "contains": {
                return explanation.toLowerCase().contains(expectedTable.toLowerCase());
            }
        }
        return false;
    }

    private boolean evaluateCostCondition(OptimizationRule.Condition condition, RelNode node) {
        try {
            RelOptCost cost = node.getCluster().getMetadataQuery().getCumulativeCost(node);
            double actualCost = cost.getRows();
            double threshold = ((Number)condition.getValue()).doubleValue();
            switch (condition.getOperator().toLowerCase()) {
                case "greater_than": {
                    return actualCost > threshold;
                }
                case "less_than": {
                    return actualCost < threshold;
                }
                case "equals": {
                    return Math.abs(actualCost - threshold) < 0.001;
                }
            }
            return false;
        }
        catch (Exception e) {
            return false;
        }
    }

    private boolean evaluateFieldCountCondition(OptimizationRule.Condition condition, RelNode node) {
        int fieldCount = node.getRowType().getFieldCount();
        int expectedCount = ((Number)condition.getValue()).intValue();
        switch (condition.getOperator().toLowerCase()) {
            case "greater_than": {
                return fieldCount > expectedCount;
            }
            case "less_than": {
                return fieldCount < expectedCount;
            }
            case "equals": {
                return fieldCount == expectedCount;
            }
        }
        return false;
    }

    private RelNode applyRule(OptimizationRule rule, RelNode node) {
        if (rule.getTransformations() != null && !rule.getTransformations().isEmpty()) {
            for (OptimizationRule.Transformation transformation : rule.getTransformations()) {
                node = this.applyTransformation(transformation, node);
            }
        }
        return node;
    }

    private RelNode applyTransformation(OptimizationRule.Transformation transformation, RelNode node) {
        System.out.println("  Applied transformation: " + transformation.getAction() + " on " + transformation.getTarget());
        if (transformation.getCostReduction() > 0.0) {
            System.out.println("  Expected cost reduction: " + transformation.getCostReduction() + "%");
        }
        return node;
    }

    public Map<String, Object> getOptimizationStats() {
        return new HashMap<String, Object>(this.optimizationStats);
    }

    public List<OptimizationRule> getRules() {
        return new ArrayList<OptimizationRule>(this.rules);
    }

    public void clearRules() {
        this.rules.clear();
        this.optimizationStats.clear();
    }
}

