"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CheckOpenApiProps = CheckOpenApiProps;
exports.ObtainApiDefinition = ObtainApiDefinition;
/*
 *  The functions found here in the core library are for internal use and can be changed
 *  or removed outside of a major release. We recommend against calling them directly from client code.
 */
/*
 * This file is core openapi functionality and should ideally be in the core library. Since
 * that causes a circular reference with the resources library we have left it here for now
 * in the interest of getting these updates out faster
 */
const apigateway = require("aws-cdk-lib/aws-apigateway");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const resources = require("@aws-solutions-constructs/resources");
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function CheckOpenApiProps(props) {
    let errorMessages = '';
    let errorFound = false;
    if ((props.apiDefinitionBucket && !props.apiDefinitionKey) || (!props.apiDefinitionBucket && props.apiDefinitionKey)) {
        errorMessages += 'apiDefinitionBucket and apiDefinitionKey must be specified together.\n';
        errorFound = true;
    }
    const definitionCount = (props.apiDefinitionAsset ? 1 : 0) +
        (props.apiDefinitionBucket ? 1 : 0) +
        (props.apiDefinitionJson ? 1 : 0);
    if (definitionCount !== 1) {
        errorMessages += 'Exactly one of apiDefinitionAsset, apiDefinitionJson or (apiDefinitionBucket/apiDefinitionKey) must be provided\n';
        errorFound = true;
    }
    if (props.apiIntegrations === undefined || props.apiIntegrations.length < 1) {
        errorMessages += 'At least one ApiIntegration must be specified in the apiIntegrations property\n';
        errorFound = true;
    }
    if (errorFound) {
        throw new Error(errorMessages);
    }
}
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function ObtainApiDefinition(scope, props) {
    const apiRawInlineSpec = props.apiJsonDefinition;
    const meldedDefinitionBucket = props.apiDefinitionBucket ?? props.apiDefinitionAsset?.bucket;
    const meldedDefinitionKey = props.apiDefinitionKey ?? props.apiDefinitionAsset?.s3ObjectKey;
    // Map each id and lambda function pair to the required format for the template writer custom resource
    const apiIntegrationUris = props.tokenToFunctionMap.map(apiLambdaFunction => {
        // the placeholder string that will be replaced in the OpenAPI Definition
        const uriPlaceholderString = apiLambdaFunction.id;
        // the endpoint URI of the backing lambda function, as defined in the API Gateway extensions for OpenAPI here:
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-integration.html
        const uriResolvedValue = `arn:${aws_cdk_lib_1.Aws.PARTITION}:apigateway:${aws_cdk_lib_1.Aws.REGION}:lambda:path/2015-03-31/functions/${apiLambdaFunction.lambdaFunction.functionArn}/invocations`;
        return {
            id: uriPlaceholderString,
            value: uriResolvedValue
        };
    });
    let apiDefinitionWriter;
    let newApiDefinition;
    if (props.apiDefinitionAsset || props.apiDefinitionBucket) {
        // This custom resource will overwrite the string placeholders in the openapi definition with the resolved values of the lambda URIs
        apiDefinitionWriter = resources.createTemplateWriterCustomResource(scope, 'Api', {
            // CheckOpenapiProps() has confirmed the existence of these values
            templateBucket: meldedDefinitionBucket,
            templateKey: meldedDefinitionKey,
            templateValues: apiIntegrationUris,
            timeout: props.internalTransformTimeout ?? aws_cdk_lib_1.Duration.minutes(1),
            memorySize: props.internalTransformMemorySize ?? 1024
        });
        newApiDefinition = apigateway.ApiDefinition.fromBucket(apiDefinitionWriter.s3Bucket, apiDefinitionWriter.s3Key);
    }
    else if (apiRawInlineSpec) {
        newApiDefinition = InlineTemplateWriter(apiRawInlineSpec, apiIntegrationUris);
    }
    else {
        throw new Error("No definition provided (this code should be unreachable)");
    }
    return newApiDefinition;
}
function InlineTemplateWriter(rawInlineSpec, templateValues) {
    let template = JSON.stringify(rawInlineSpec);
    // This replicates logic in the template writer custom resource (resources/lib/template-writer-custom-resource/index.ts),
    // any logic changes should be made to both locations every time
    templateValues.forEach((templateValue) => {
        template = template?.replace(new RegExp(templateValue.id, 'g'), templateValue.value);
    });
    return new apigateway.InlineApiDefinition(JSON.parse(template));
}
//# sourceMappingURL=data:application/json;base64,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