"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
//
// Utilities to assert that supplied values match with expected values
Object.defineProperty(exports, "__esModule", { value: true });
exports.assertIsNotPromise = exports.assertStringArraysOverlap = exports.assertStringArrayContainsString = exports.assertStringEquals = void 0;
const error_1 = require("./error");
/**
 * Assert value is a non-empty string and equal to the expected value,
 * or throw an error otherwise
 *
 * @param name - Name for the value being checked
 * @param actual - The value to check
 * @param expected - The expected value
 * @param errorConstructor - Constructor for the concrete error to be thrown
 */
function assertStringEquals(name, actual, expected, errorConstructor = error_1.FailedAssertionError) {
    if (!actual) {
        throw new errorConstructor(`Missing ${name}. Expected: ${expected}`, actual, expected);
    }
    if (typeof actual !== 'string') {
        throw new errorConstructor(`${name} is not of type string`, actual, expected);
    }
    if (expected !== actual) {
        throw new errorConstructor(`${name} not allowed: ${actual}. Expected: ${expected}`, actual, expected);
    }
}
exports.assertStringEquals = assertStringEquals;
/**
 * Assert value is a non-empty string and is indeed one of the expected values,
 * or throw an error otherwise
 *
 * @param name - Name for the value being checked
 * @param actual - The value to check
 * @param expected - The array of expected values. For your convenience you can provide
 * @param errorConstructor - Constructor for the concrete error to be thrown
 * a string here as well, which will mean an array with just that string
 */
function assertStringArrayContainsString(name, actual, expected, errorConstructor = error_1.FailedAssertionError) {
    if (!actual) {
        throw new errorConstructor(`Missing ${name}. ${expectationMessage(expected)}`, actual, expected);
    }
    if (typeof actual !== 'string') {
        throw new errorConstructor(`${name} is not of type string`, actual, expected);
    }
    return assertStringArraysOverlap(name, actual, expected, errorConstructor);
}
exports.assertStringArrayContainsString = assertStringArrayContainsString;
/**
 * Assert value is an array of strings, where at least one of the strings is indeed one of the expected values,
 * or throw an error otherwise
 *
 * @param name - Name for the value being checked
 * @param actual - The value to check, must be an array of strings, or a single string (which will be treated
 * as an array with just that string)
 * @param expected - The array of expected values. For your convenience you can provide
 * a string here as well, which will mean an array with just that string
 * @param errorConstructor - Constructor for the concrete error to be thrown
 */
function assertStringArraysOverlap(name, actual, expected, errorConstructor = error_1.FailedAssertionError) {
    if (!actual) {
        throw new errorConstructor(`Missing ${name}. ${expectationMessage(expected)}`, actual, expected);
    }
    const expectedAsSet = new Set(Array.isArray(expected) ? expected : [expected]);
    if (typeof actual === 'string') {
        actual = [actual];
    }
    if (!Array.isArray(actual)) {
        throw new errorConstructor(`${name} is not an array`, actual, expected);
    }
    const overlaps = actual.some((actualItem) => {
        if (typeof actualItem !== 'string') {
            throw new errorConstructor(`${name} includes elements that are not of type string`, actual, expected);
        }
        return expectedAsSet.has(actualItem);
    });
    if (!overlaps) {
        throw new errorConstructor(`${name} not allowed: ${actual.join(', ')}. ${expectationMessage(expected)}`, actual, expected);
    }
}
exports.assertStringArraysOverlap = assertStringArraysOverlap;
/**
 * Get a nicely readable message regarding an expectation
 *
 * @param expected - The expected value.
 */
function expectationMessage(expected) {
    if (Array.isArray(expected)) {
        if (expected.length > 1) {
            return `Expected one of: ${expected.join(', ')}`;
        }
        return `Expected: ${expected[0]}`;
    }
    return `Expected: ${expected}`;
}
/**
 * Assert value is not a promise, or throw an error otherwise
 *
 * @param actual - The value to check
 * @param errorFactory - Function that returns the error to be thrown
 */
function assertIsNotPromise(actual, errorFactory) {
    if (actual && typeof actual.then === 'function') {
        throw errorFactory();
    }
}
exports.assertIsNotPromise = assertIsNotPromise;
//# sourceMappingURL=data:application/json;base64,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