"""Report generation module.

This module orchestrates report generation by combining analysis data
with visualizations from the viz package. It creates structured reports
in various formats without duplicating visualization logic.
"""

import json
from dataclasses import dataclass, field
from datetime import datetime
from pathlib import Path
from typing import Any, Dict, List, Optional, Union

from tenets.config import TenetsConfig
from tenets.utils.logger import get_logger

# Import visualizers from viz package
from tenets.viz import (
    ChartConfig,
    ComplexityVisualizer,
    ContributorVisualizer,
    CouplingVisualizer,
    DependencyVisualizer,
    HotspotVisualizer,
    MomentumVisualizer,
)


@dataclass
class ReportSection:
    """Represents a section in the report.

    A report section contains structured content including text,
    metrics, tables, and charts. Sections can be nested to create
    hierarchical report structures.

    Attributes:
        id: Unique section identifier
        title: Section title
        level: Heading level (1-6)
        order: Display order
        icon: Optional icon/emoji
        content: Section text content
        metrics: Key metrics dictionary
        tables: List of table data
        charts: List of chart configurations
        code_snippets: List of code examples
        subsections: Nested sections
        visible: Whether section is visible
        collapsed: Whether section starts collapsed
    """

    id: str
    title: str
    level: int = 1
    order: int = 0
    icon: Optional[str] = None
    content: Optional[Union[str, List[str], Dict[str, Any]]] = None
    metrics: Dict[str, Any] = field(default_factory=dict)
    tables: List[Dict[str, Any]] = field(default_factory=list)
    charts: List[Dict[str, Any]] = field(default_factory=list)
    code_snippets: List[Dict[str, Any]] = field(default_factory=list)
    subsections: List["ReportSection"] = field(default_factory=list)
    visible: bool = True
    collapsed: bool = False
    # Whether section header is collapsible (used by HTML reporter)
    collapsible: bool = False

    def add_metric(self, name: str, value: Any) -> None:
        """Add a metric to the section.

        Args:
            name: Metric name
            value: Metric value
        """
        self.metrics[name] = value

    def add_table(self, table_data: Dict[str, Any]) -> None:
        """Add a table to the section.

        Args:
            table_data: Table configuration with headers and rows
        """
        self.tables.append(table_data)

    def add_chart(self, chart_config: Dict[str, Any]) -> None:
        """Add a chart to the section.

        Args:
            chart_config: Chart configuration from viz modules
        """
        self.charts.append(chart_config)

    def add_subsection(self, subsection: "ReportSection") -> None:
        """Add a subsection.

        Args:
            subsection: Nested section
        """
        self.subsections.append(subsection)


@dataclass
class ReportConfig:
    """Configuration for report generation.

    Controls report generation options including format, content
    inclusion, and visualization settings.

    Attributes:
        title: Report title
        format: Output format (html, markdown, json)
        include_summary: Include executive summary
        include_toc: Include table of contents
        include_charts: Include visualizations
        include_code_snippets: Include code examples
        include_recommendations: Include recommendations
        max_items: Maximum items in lists
        theme: Visual theme (light, dark, auto)
        footer_text: Footer text
        custom_css: Custom CSS for HTML reports
        chart_config: Default chart configuration
    """

    title: str = "Code Analysis Report"
    format: str = "html"
    include_summary: bool = True
    include_toc: bool = True
    include_charts: bool = True
    include_code_snippets: bool = True
    include_recommendations: bool = True
    max_items: int = 20
    theme: str = "light"
    footer_text: str = "Generated by Tenets Code Analysis"
    custom_css: Optional[str] = None
    chart_config: Optional[ChartConfig] = None
    # Optional custom logo path for HTML reports
    custom_logo: Optional[Path] = None


class ReportGenerator:
    """Main report generator orchestrator.

    Coordinates report generation by combining analysis data with
    visualizations from the viz package. Creates structured reports
    without duplicating visualization logic.

    The generator follows a clear separation of concerns:
    - Core modules provide analysis data
    - Viz modules create visualizations
    - Generator orchestrates and structures the report

    Attributes:
        config: Configuration object
        logger: Logger instance
        sections: List of report sections
        metadata: Report metadata
    """

    def __init__(self, config: TenetsConfig):
        """Initialize report generator.

        Args:
            config: Configuration object
        """
        self.config = config
        self.logger = get_logger(__name__)
        self.sections: List[ReportSection] = []
        self.metadata: Dict[str, Any] = {}

    def generate(
        self, data: Dict[str, Any], output_path: Path, config: Optional[ReportConfig] = None
    ) -> Path:
        """Generate a report from analysis data.

        This is the main entry point for report generation. It takes
        analysis data, creates appropriate visualizations using viz
        modules, and outputs a formatted report.

        Args:
            data: Analysis data from core modules
            output_path: Path for output file
            config: Report configuration

        Returns:
            Path: Path to generated report

        Example:
            >>> generator = ReportGenerator(config)
            >>> report_path = generator.generate(
            ...     analysis_data,
            ...     Path("report.html"),
            ...     ReportConfig(include_charts=True)
            ... )
        """
        if config is None:
            config = ReportConfig()

        self.logger.info(f"Generating {config.format} report: {output_path}")

        # Clear previous sections
        self.sections = []

        # Build metadata
        self.metadata = self._build_metadata(data, config)

        # Create report sections using viz modules
        if config.include_summary:
            self.sections.append(self._create_summary_section(data))

        # Add file overview section
        if "metrics" in data:
            self.sections.append(self._create_file_overview_section(data, config))

        # Add excluded files section if available
        if data.get("excluded_files") or data.get("ignored_patterns"):
            self.sections.append(self._create_excluded_files_section(data, config))

        # Add README section if available
        readme_info = self._find_readme(data)
        if readme_info:
            self.sections.append(self._create_readme_section(readme_info))

        # Add analysis sections based on available data
        # Add defensive checks to prevent NoneType errors
        if "complexity" in data and data["complexity"] is not None:
            self.sections.append(self._create_complexity_section(data["complexity"], config))

        if "contributors" in data and data["contributors"] is not None:
            self.sections.append(self._create_contributors_section(data["contributors"], config))

        if "hotspots" in data and data["hotspots"] is not None:
            self.sections.append(self._create_hotspots_section(data["hotspots"], config))

        if "dependencies" in data and data["dependencies"] is not None:
            self.sections.append(self._create_dependencies_section(data["dependencies"], config))

        if "coupling" in data and data["coupling"] is not None:
            self.sections.append(self._create_coupling_section(data["coupling"], config))

        if "momentum" in data and data["momentum"] is not None:
            self.sections.append(self._create_momentum_section(data["momentum"], config))

        if config.include_recommendations:
            self.sections.append(self._create_recommendations_section(data))

        # Generate output based on format
        if config.format == "html":
            from .html_reporter import HTMLReporter

            reporter = HTMLReporter(self.config)
            return reporter.generate(self.sections, self.metadata, output_path, config)
        elif config.format == "markdown":
            from .markdown_reporter import MarkdownReporter

            reporter = MarkdownReporter(self.config)
            return reporter.generate(self.sections, self.metadata, output_path, config)
        elif config.format == "json":
            return self._generate_json_report(output_path)
        else:
            raise ValueError(f"Unsupported report format: {config.format}")

    def _build_metadata(self, data: Dict[str, Any], config: ReportConfig) -> Dict[str, Any]:
        """Build report metadata.

        Args:
            data: Analysis data
            config: Report configuration

        Returns:
            Dict[str, Any]: Report metadata
        """
        metadata = {
            "title": config.title,
            "generated_at": datetime.now().isoformat(),
            "generator_version": "1.0.0",
            "format": config.format,
            "analysis_summary": self._extract_summary_metrics(data),
        }

        # Add path information - this is the directory/file that was examined
        if "path" in data:
            metadata["examined_path"] = data["path"]

        # Add data source information
        if "source" in data:
            metadata["source"] = data["source"]

        # Add timing information if available
        if "timing" in data:
            metadata["timing"] = data["timing"]

        return metadata

    def _extract_summary_metrics(self, data: Dict[str, Any]) -> Dict[str, Any]:
        """Extract summary metrics from analysis data.

        Args:
            data: Analysis data

        Returns:
            Dict[str, Any]: Summary metrics
        """
        summary = {
            "total_files": data.get("total_files", 0),
            "total_lines": data.get("total_lines", 0),
            "languages": data.get("languages", []),
            "health_score": data.get("health_score", 0),
            "critical_issues": 0,
            "total_issues": 0,
        }

        # Count issues from various sources
        if "complexity" in data and data["complexity"] is not None:
            summary["critical_issues"] += data["complexity"].get("critical_count", 0)
            summary["total_issues"] += data["complexity"].get("complex_functions", 0)

        if "hotspots" in data and data["hotspots"] is not None:
            summary["critical_issues"] += data["hotspots"].get("critical_count", 0)
            summary["total_issues"] += data["hotspots"].get("total_hotspots", 0)

        if "dependencies" in data and data["dependencies"] is not None:
            summary["total_issues"] += data["dependencies"].get("circular_count", 0)

        return summary

    def _create_summary_section(self, data: Dict[str, Any]) -> ReportSection:
        """Create executive summary section.

        Args:
            data: Analysis data

        Returns:
            ReportSection: Summary section
        """
        section = ReportSection(
            id="summary", title="Executive Summary", level=1, order=1, icon="📊"
        )

        summary = self.metadata.get("analysis_summary", {}) if self.metadata else {}
        metrics = data.get("metrics", {})

        # Build summary content
        health_score = summary.get("health_score", 0)
        if health_score >= 80:
            health_status = "excellent"
            health_icon = "✅"
        elif health_score >= 60:
            health_status = "good"
            health_icon = "✔️"
        elif health_score >= 40:
            health_status = "fair"
            health_icon = "⚠️"
        else:
            health_status = "needs improvement"
            health_icon = "❌"

        content = []

        # Add root path information
        root_path = data.get("root_path", ".")
        if root_path and root_path != ".":
            from pathlib import Path

            path_obj = Path(root_path) if isinstance(root_path, str) else root_path
            content.append(f"📂 **Project Path:** `{path_obj.resolve()}`")
            content.append("")

        content.extend(
            [
                f"The codebase is in **{health_status}** condition {health_icon} with a health score of **{health_score:.1f}/100**.",
                "",
                "### Key Findings",
                "",
            ]
        )

        if summary.get("critical_issues", 0) > 0:
            content.append(
                f"- 🚨 **{summary['critical_issues']} critical issues** require immediate attention"
            )

        if summary.get("total_issues", 0) > 0:
            content.append(
                f"- ⚠️ Total of **{summary['total_issues']} issues** identified across all categories"
            )

        content.append(
            f"- 📊 Analyzed **{summary.get('total_files', 0)} files** with **{summary.get('total_lines', 0):,} lines** of code"
        )

        # Add timing information if available
        if "timing" in data:
            timing = data["timing"]
            if "formatted_duration" in timing:
                content.append(f"- ⏱️ Analysis completed in **{timing['formatted_duration']}**")

        # Add language distribution info
        if metrics.get("languages"):
            lang_dict = metrics["languages"]
            if isinstance(lang_dict, dict):
                top_langs = sorted(
                    lang_dict.items(), key=lambda x: x[1].get("lines", 0), reverse=True
                )[:3]
                if top_langs:
                    content.append("")
                    content.append("### Primary Languages")
                    for lang, info in top_langs:
                        lines = info.get("lines", 0)
                        files = info.get("files", 0)
                        content.append(
                            f"- **{lang.title()}**: {lines:,} lines across {files} files"
                        )

        # Add test coverage if available
        if metrics.get("test_coverage", 0) > 0:
            coverage = metrics["test_coverage"] * 100
            content.append("")
            content.append(f"### Test Coverage: {coverage:.1f}%")

        section.content = content

        # Enhanced summary metrics
        section.metrics = {
            "Health Score": f"{health_score:.1f}/100",
            "Files Analyzed": summary.get("total_files", 0),
            "Total Lines": f"{summary.get('total_lines', 0):,}",
            "Functions": metrics.get("total_functions", 0),
            "Classes": metrics.get("total_classes", 0),
            "Test Coverage": (
                f"{metrics.get('test_coverage', 0) * 100:.1f}%"
                if metrics.get("test_coverage", 0) > 0
                else "N/A"
            ),
            "Languages": len(data.get("languages", [])),
            "Avg File Size": f"{metrics.get('avg_file_size', 0):.0f} lines",
        }

        return section

    def _create_complexity_section(
        self, complexity_data: Dict[str, Any], config: ReportConfig
    ) -> ReportSection:
        """Create complexity analysis section using viz modules.

        Args:
            complexity_data: Complexity analysis data
            config: Report configuration

        Returns:
            ReportSection: Complexity section
        """
        section = ReportSection(
            id="complexity", title="Complexity Analysis", level=1, order=3, icon="🔍"
        )

        # Add explanation of complexity metrics
        section.content = [
            "### Understanding Code Complexity",
            "",
            "**Cyclomatic Complexity** measures the number of independent paths through code:",
            "- **1-5**: Simple, easy to understand",
            "- **6-10**: Moderate, should be reviewed",
            "- **11-20**: Complex, consider refactoring",
            "- **21+**: Very complex, high risk of bugs",
            "",
        ]

        # Use ComplexityVisualizer for charts
        viz = ComplexityVisualizer(chart_config=config.chart_config)

        # Add metrics
        section.metrics = {
            "Average Complexity": f"{complexity_data.get('avg_complexity', 0):.2f}",
            "Maximum Complexity": complexity_data.get("max_complexity", 0),
            "Complex Functions": complexity_data.get("complex_functions", 0),
            "Total Functions": complexity_data.get("total_functions", 0),
        }

        # Add distribution chart if requested
        if config.include_charts:
            # Ensure we have distribution data
            if (
                "complexity_distribution" not in complexity_data
                or not complexity_data["complexity_distribution"]
            ):
                # Create a default distribution from the data we have
                complexity_data["complexity_distribution"] = {
                    "simple (1-5)": complexity_data.get("total_functions", 0)
                    - complexity_data.get("complex_functions", 0),
                    "moderate (6-10)": 0,
                    "complex (11-20)": 0,
                    "very complex (21+)": complexity_data.get("complex_functions", 0),
                }

            chart = viz.create_distribution_chart(complexity_data)
            section.add_chart(chart)

            # Add top complex functions chart
            if "complex_items" in complexity_data:
                chart = viz.create_top_complex_chart(
                    complexity_data["complex_items"], limit=config.max_items
                )
                section.add_chart(chart)

            # Add trend chart if available
            if "trend_data" in complexity_data:
                chart = viz.create_trend_chart(complexity_data["trend_data"])
                section.add_chart(chart)

        # Add complex functions table
        if "complex_items" in complexity_data:
            table_data = self._create_complex_functions_table(
                complexity_data["complex_items"][: config.max_items]
            )
            section.add_table(table_data)

        # Add code snippets if requested
        if config.include_code_snippets and "complex_items" in complexity_data:
            for item in complexity_data["complex_items"][:3]:
                if "code" in item:
                    section.code_snippets.append(
                        {
                            "filename": item.get("file", ""),
                            "language": item.get("language", "python"),
                            "code": item["code"],
                            "highlight_lines": item.get("highlight_lines", []),
                        }
                    )

        return section

    def _create_contributors_section(
        self, contributor_data: Dict[str, Any], config: ReportConfig
    ) -> ReportSection:
        """Create contributors analysis section using viz modules.

        Args:
            contributor_data: Contributor analysis data
            config: Report configuration

        Returns:
            ReportSection: Contributors section
        """
        section = ReportSection(
            id="contributors", title="Contributor Analysis", level=1, order=3, icon="👥"
        )

        # Use ContributorVisualizer for charts
        viz = ContributorVisualizer(chart_config=config.chart_config)

        # Add metrics
        section.metrics = {
            "Total Contributors": contributor_data.get("total_contributors", 0),
            "Active Contributors": contributor_data.get("active_contributors", 0),
            "Bus Factor": contributor_data.get("bus_factor", 0),
            "Avg Commits/Contributor": f"{contributor_data.get('avg_commits_per_contributor', 0):.1f}",
        }

        # Add visualizations
        if config.include_charts:
            # Contribution chart
            if "contributors" in contributor_data:
                chart = viz.create_contribution_chart(
                    contributor_data["contributors"], metric="commits", limit=config.max_items
                )
                section.add_chart(chart)

                # Distribution pie chart
                chart = viz.create_distribution_pie(
                    contributor_data["contributors"], metric="commits", top_n=5
                )
                section.add_chart(chart)

            # Activity timeline
            if "activity_data" in contributor_data:
                chart = viz.create_activity_timeline(contributor_data["activity_data"])
                section.add_chart(chart)

            # Bus factor gauge
            chart = viz.create_bus_factor_gauge(
                contributor_data.get("bus_factor", 0), contributor_data.get("total_contributors", 1)
            )
            section.add_chart(chart)

        # Add contributors table
        if "contributors" in contributor_data:
            table_data = self._create_contributors_table(
                contributor_data["contributors"][: config.max_items]
            )
            section.add_table(table_data)

        return section

    def _create_hotspots_section(
        self, hotspot_data: Dict[str, Any], config: ReportConfig
    ) -> ReportSection:
        """Create hotspots analysis section using viz modules.

        Args:
            hotspot_data: Hotspot analysis data
            config: Report configuration

        Returns:
            ReportSection: Hotspots section
        """
        section = ReportSection(id="hotspots", title="Code Hotspots", level=1, order=4, icon="🔥")

        # Add explanation of what hotspots are
        section.content = [
            "### What are Code Hotspots?",
            "",
            "Code hotspots are files that require special attention due to a combination of factors:",
            "",
            "- **🔄 High Change Frequency**: Files that are modified often are more likely to introduce bugs",
            "- **🧩 High Complexity**: Complex code is harder to maintain and more error-prone",
            "- **📏 Large Size**: Large files are difficult to understand and navigate",
            "- **🐛 Bug History**: Files with a history of bugs are likely to have more issues",
            "- **👥 Multiple Contributors**: Files touched by many developers may lack consistency",
            "",
            "**Risk Levels:**",
            "- 🔴 **Critical**: Immediate refactoring needed (score > 80)",
            "- 🟠 **High**: Should be addressed soon (score 60-80)",
            "- 🟡 **Medium**: Monitor and plan for improvement (score 40-60)",
            "- 🟢 **Low**: Acceptable but could be improved (score < 40)",
            "",
        ]

        # Use HotspotVisualizer for charts
        viz = HotspotVisualizer(chart_config=config.chart_config)

        # Add metrics - handle both hotspot_files and hotspots keys
        hotspot_files = hotspot_data.get("hotspot_files", hotspot_data.get("hotspots", []))

        section.metrics = {
            "Total Hotspots": hotspot_data.get("total_hotspots", len(hotspot_files)),
            "Critical": hotspot_data.get("critical_count", 0),
            "High Risk": hotspot_data.get("high_count", 0),
            "Files Analyzed": hotspot_data.get(
                "files_analyzed", hotspot_data.get("total_files", 0)
            ),
        }

        # Add visualizations
        if config.include_charts and hotspot_files:
            # Hotspot bubble chart
            chart = viz.create_hotspot_bubble(hotspot_files, limit=50)
            section.add_chart(chart)

            # Risk matrix
            chart = viz.create_risk_matrix(hotspot_files)
            section.add_chart(chart)

            # Hotspot heatmap
            chart = viz.create_hotspot_heatmap(
                hotspot_files, metric_x="change_frequency", metric_y="complexity"
            )
            section.add_chart(chart)

        # Add hotspots table
        if hotspot_files:
            table_data = self._create_hotspots_table(hotspot_files[: config.max_items])
            section.add_table(table_data)

        return section

    def _create_dependencies_section(
        self, dependency_data: Dict[str, Any], config: ReportConfig
    ) -> ReportSection:
        """Create dependencies analysis section using viz modules.

        Args:
            dependency_data: Dependency analysis data
            config: Report configuration

        Returns:
            ReportSection: Dependencies section
        """
        section = ReportSection(
            id="dependencies", title="Dependency Analysis", level=1, order=5, icon="🔗"
        )

        # Use DependencyVisualizer for charts
        viz = DependencyVisualizer(chart_config=config.chart_config)

        # Add metrics
        section.metrics = {
            "Total Modules": dependency_data.get("total_modules", 0),
            "Total Dependencies": dependency_data.get("total_dependencies", 0),
            "External Dependencies": dependency_data.get("external_dependencies", 0),
            "Circular Dependencies": dependency_data.get("circular_count", 0),
        }

        # Add visualizations
        if config.include_charts:
            # Dependency graph
            if "dependencies" in dependency_data:
                deps = dependency_data["dependencies"]
                # Tests sometimes provide a list; convert to expected dict form
                if isinstance(deps, list):
                    deps = {str(i): v for i, v in enumerate(deps)}
                chart = viz.create_dependency_graph(deps, highlight_circular=True)
                section.add_chart(chart)

            # Circular dependencies
            if dependency_data.get("circular_count", 0) > 0:
                if "circular_chains" in dependency_data:
                    chart = viz.create_circular_dependencies_chart(
                        dependency_data["circular_chains"]
                    )
                    section.add_chart(chart)

        # Add tables
        if "most_dependent" in dependency_data:
            table_data = self._create_dependency_table(
                dependency_data["most_dependent"][: config.max_items]
            )
            section.add_table(table_data)

        return section

    def _create_coupling_section(
        self, coupling_data: Dict[str, Any], config: ReportConfig
    ) -> ReportSection:
        """Create coupling analysis section using viz modules.

        Args:
            coupling_data: Coupling analysis data
            config: Report configuration

        Returns:
            ReportSection: Coupling section
        """
        section = ReportSection(
            id="coupling", title="Coupling Analysis", level=1, order=6, icon="🔄"
        )

        # Use CouplingVisualizer for charts
        viz = CouplingVisualizer(chart_config=config.chart_config)

        # Add metrics
        section.metrics = {
            "Average Coupling": f"{coupling_data.get('avg_coupling', 0):.2f}",
            "Max Coupling": coupling_data.get("max_coupling", 0),
            "Highly Coupled": coupling_data.get("highly_coupled_count", 0),
            "Total Modules": coupling_data.get("total_modules", 0),
        }

        # Add visualizations
        if config.include_charts:
            # Coupling network
            if "coupling_data" in coupling_data:
                chart = viz.create_coupling_network(coupling_data["coupling_data"], min_coupling=2)
                section.add_chart(chart)

            # Instability chart
            if "instability_data" in coupling_data:
                chart = viz.create_instability_chart(coupling_data["instability_data"])
                section.add_chart(chart)

        return section

    def _create_momentum_section(
        self, momentum_data: Dict[str, Any], config: ReportConfig
    ) -> ReportSection:
        """Create momentum analysis section using viz modules.

        Args:
            momentum_data: Momentum analysis data
            config: Report configuration

        Returns:
            ReportSection: Momentum section
        """
        section = ReportSection(
            id="momentum", title="Development Momentum", level=1, order=7, icon="📈"
        )

        # Use MomentumVisualizer for charts
        viz = MomentumVisualizer(chart_config=config.chart_config)

        # Add current sprint metrics
        if "current_sprint" in momentum_data:
            sprint = momentum_data["current_sprint"]
            section.metrics = {
                "Current Velocity": sprint.get("velocity", 0),
                "Sprint Progress": f"{sprint.get('completed', 0)}/{sprint.get('planned', 0)}",
                "Days Remaining": sprint.get("days_remaining", 0),
                "On Track": "✅" if sprint.get("on_track", False) else "❌",
            }

        # Add visualizations
        if config.include_charts:
            # Velocity chart
            if "velocity_data" in momentum_data:
                chart = viz.create_velocity_chart(momentum_data["velocity_data"], show_trend=True)
                section.add_chart(chart)

            # Burndown chart
            if "burndown" in momentum_data:
                chart = viz.create_burndown_chart(momentum_data["burndown"])
                section.add_chart(chart)

            # Team metrics radar
            if "team_metrics" in momentum_data:
                chart = viz.create_team_velocity_radar(momentum_data["team_metrics"])
                section.add_chart(chart)

        return section

    def _create_file_overview_section(
        self, data: Dict[str, Any], config: ReportConfig
    ) -> ReportSection:
        """Create file overview section with detailed metrics.

        Args:
            data: Analysis data
            config: Report configuration

        Returns:
            ReportSection: File overview section
        """
        section = ReportSection(
            id="file_overview", title="File Analysis Overview", level=1, order=2, icon="📁"
        )

        metrics = data.get("metrics", {})

        # Language breakdown table
        if metrics.get("languages"):
            lang_dict = metrics["languages"]
            if isinstance(lang_dict, dict):
                headers = ["Language", "Files", "Lines", "Avg Size", "Functions", "Classes"]
                rows = []
                for lang, info in sorted(
                    lang_dict.items(), key=lambda x: x[1].get("lines", 0), reverse=True
                ):
                    rows.append(
                        [
                            lang.title(),
                            info.get("files", 0),
                            f"{info.get('lines', 0):,}",
                            f"{info.get('avg_file_size', 0):.0f}",
                            info.get("functions", 0),
                            info.get("classes", 0),
                        ]
                    )
                section.add_table({"headers": headers, "rows": rows})

        # File size distribution chart
        if config.include_charts and metrics.get("size_distribution"):
            from tenets.viz import ChartType, create_chart

            dist = metrics["size_distribution"]
            chart = create_chart(
                ChartType.BAR,
                {
                    "labels": list(dist.keys()),
                    "values": list(dist.values()),
                },
                title="File Size Distribution",
            )
            section.add_chart(chart)

        # Largest files table
        if metrics.get("largest_files"):
            headers = ["File", "Lines", "Language"]
            rows = []
            for file_info in metrics["largest_files"][:10]:
                rows.append(
                    [
                        file_info.get("name", "Unknown"),
                        f"{file_info.get('lines', 0):,}",
                        file_info.get("language", "Unknown"),
                    ]
                )
            section.add_table({"headers": headers, "rows": rows})

        return section

    def _create_excluded_files_section(
        self, data: Dict[str, Any], config: ReportConfig
    ) -> ReportSection:
        """Create excluded files section.

        Args:
            data: Analysis data
            config: Report configuration

        Returns:
            ReportSection: Excluded files section
        """
        section = ReportSection(
            id="excluded_files",
            title="Excluded Files",
            level=1,
            order=2.5,
            icon="🚫",
            collapsible=True,
            collapsed=True,
        )

        content = []

        # Add ignored patterns if available
        ignored_patterns = data.get("ignored_patterns", [])
        if ignored_patterns:
            content.append("### Ignored Patterns")
            content.append("")
            for pattern in ignored_patterns:
                content.append(f"- `{pattern}`")
            content.append("")

        # Add excluded files if available
        excluded_files = data.get("excluded_files", [])
        if excluded_files:
            content.append(f"### Excluded Files ({len(excluded_files)} files)")
            content.append("")

            # Group files by reason or extension
            from pathlib import Path

            by_extension = {}
            for file_path in excluded_files[:100]:  # Limit to first 100 for readability
                ext = Path(file_path).suffix or "no extension"
                if ext not in by_extension:
                    by_extension[ext] = []
                by_extension[ext].append(file_path)

            for ext, files in sorted(by_extension.items()):
                content.append(f"**{ext}** ({len(files)} files):")
                for file_path in files[:10]:  # Show first 10 of each type
                    content.append(f"  - {file_path}")
                if len(files) > 10:
                    content.append(f"  - ... and {len(files) - 10} more")
                content.append("")

            if len(excluded_files) > 100:
                content.append(f"*... and {len(excluded_files) - 100} more excluded files*")

        # Add summary metrics
        section.metrics = {
            "Excluded Files": len(excluded_files),
            "Ignored Patterns": len(ignored_patterns),
        }

        section.content = content
        return section

    def _find_readme(self, data: Dict[str, Any]) -> Optional[Dict[str, Any]]:
        """Find and read README content if available.

        Args:
            data: Analysis data

        Returns:
            Optional[Dict]: README info with content and metadata
        """
        from pathlib import Path

        root_path = data.get("root_path", ".")
        if isinstance(root_path, str):
            root_path = Path(root_path)

        # Look for README files in common formats
        readme_patterns = [
            "README.md",
            "README.MD",
            "readme.md",
            "README.rst",
            "README.txt",
            "README",
        ]

        for pattern in readme_patterns:
            readme_path = root_path / pattern
            if readme_path.exists() and readme_path.is_file():
                try:
                    content = readme_path.read_text(encoding="utf-8", errors="ignore")
                    original_length = len(content)

                    # Create a summary if the README is too long
                    max_length = 2000  # Maximum characters to show
                    if len(content) > max_length:
                        # Find a good break point
                        truncated = content[:max_length]
                        last_newline = truncated.rfind("\n")
                        if last_newline > max_length * 0.8:  # If we have a newline in the last 20%
                            truncated = truncated[:last_newline]
                        summary = truncated + "\n\n... [README truncated for brevity]"
                        condensed_ratio = (1 - max_length / original_length) * 100
                    else:
                        summary = content
                        condensed_ratio = 0

                    return {
                        "path": str(readme_path),
                        "name": pattern,
                        "content": summary,
                        "original_length": original_length,
                        "displayed_length": len(summary),
                        "condensed_by": condensed_ratio,
                        "lines": content.count("\n") + 1,
                    }
                except Exception as e:
                    self.logger.warning(f"Failed to read README at {readme_path}: {e}")

        return None

    def _create_readme_section(self, readme_info: Dict[str, Any]) -> ReportSection:
        """Create README section.

        Args:
            readme_info: README information including content and metadata

        Returns:
            ReportSection: README section
        """
        section = ReportSection(id="readme", title="Project README", level=1, order=1.5, icon="📖")

        # Add README metadata
        content = []

        # Show README stats
        if readme_info.get("condensed_by", 0) > 0:
            content.append(
                f"*📄 {readme_info['name']} - {readme_info['lines']} lines "
                f"({readme_info['original_length']:,} characters)*"
            )
            content.append(f"*📊 Condensed by {readme_info['condensed_by']:.1f}% for readability*")
            content.append("")
        else:
            content.append(f"*📄 {readme_info['name']} - {readme_info['lines']} lines*")
            content.append("")

        # Add the README content
        content.append(readme_info["content"])

        section.content = content
        section.collapsible = True
        section.collapsed = True  # Start collapsed by default

        return section

    def _create_recommendations_section(self, data: Dict[str, Any]) -> ReportSection:
        """Create recommendations section.

        Args:
            data: Complete analysis data

        Returns:
            ReportSection: Recommendations section
        """
        section = ReportSection(
            id="recommendations", title="Recommendations", level=1, order=99, icon="💡"
        )

        recommendations = []

        # Always provide recommendations based on the analysis
        metrics = data.get("metrics", {})

        # Complexity recommendations
        if "complexity" in data and data["complexity"] is not None:
            complexity = data["complexity"]
            avg_complexity = complexity.get("avg_complexity", 0)
            max_complexity = complexity.get("max_complexity", 0)

            if avg_complexity > 15:
                recommendations.append(
                    {
                        "priority": "high",
                        "category": "complexity",
                        "action": "Refactor complex functions to reduce cognitive load",
                        "impact": "Improved maintainability and reduced bug risk",
                    }
                )
            elif max_complexity > 20:
                recommendations.append(
                    {
                        "priority": "medium",
                        "category": "complexity",
                        "action": f"Simplify the most complex functions (complexity: {max_complexity})",
                        "impact": "Better code readability and easier testing",
                    }
                )
            elif complexity.get("complex_functions", 0) > 0:
                recommendations.append(
                    {
                        "priority": "low",
                        "category": "complexity",
                        "action": f"Review {complexity.get('complex_functions', 0)} complex functions for potential simplification",
                        "impact": "Continuous improvement in code quality",
                    }
                )

        # Hotspot recommendations
        if "hotspots" in data and data["hotspots"] is not None:
            hotspots = data["hotspots"]
            total_hotspots = hotspots.get("total_hotspots", 0)
            critical_count = hotspots.get("critical_count", 0)

            if critical_count > 0:
                recommendations.append(
                    {
                        "priority": "critical",
                        "category": "hotspots",
                        "action": f"Address {critical_count} critical hotspots immediately",
                        "impact": "Reduced technical debt and improved stability",
                    }
                )
            elif total_hotspots > 20:
                recommendations.append(
                    {
                        "priority": "high",
                        "category": "hotspots",
                        "action": f"Review and refactor {total_hotspots} identified hotspots",
                        "impact": "Prevention of future maintenance issues",
                    }
                )
            elif total_hotspots > 0:
                recommendations.append(
                    {
                        "priority": "medium",
                        "category": "hotspots",
                        "action": f"Monitor {total_hotspots} hotspots for deterioration",
                        "impact": "Proactive maintenance planning",
                    }
                )

        # Test coverage recommendations
        test_coverage = metrics.get("test_coverage", 0)
        if test_coverage > 0:
            if test_coverage < 0.5:
                recommendations.append(
                    {
                        "priority": "high",
                        "category": "testing",
                        "action": f"Increase test coverage from {test_coverage * 100:.1f}% to at least 70%",
                        "impact": "Reduced regression bugs and safer refactoring",
                    }
                )
            elif test_coverage < 0.8:
                recommendations.append(
                    {
                        "priority": "medium",
                        "category": "testing",
                        "action": f"Improve test coverage from {test_coverage * 100:.1f}% to 80% or higher",
                        "impact": "Better confidence in code changes",
                    }
                )

        # File size recommendations
        if metrics.get("largest_files"):
            largest = metrics["largest_files"][0] if metrics["largest_files"] else None
            if largest and largest.get("lines", 0) > 1000:
                recommendations.append(
                    {
                        "priority": "medium",
                        "category": "architecture",
                        "action": f"Split large files (e.g., {largest.get('name', 'unknown')} with {largest.get('lines', 0):,} lines)",
                        "impact": "Improved modularity and easier navigation",
                    }
                )

        # Documentation recommendations
        doc_ratio = metrics.get("documentation_ratio", 0)
        if doc_ratio < 0.1:
            recommendations.append(
                {
                    "priority": "low",
                    "category": "documentation",
                    "action": "Add documentation to critical modules and functions",
                    "impact": "Better onboarding and knowledge transfer",
                }
            )

        # Dependencies recommendations
        if "dependencies" in data:
            deps = data["dependencies"]
            if deps.get("circular_count", 0) > 0:
                recommendations.append(
                    {
                        "priority": "high",
                        "category": "architecture",
                        "action": f"Resolve {deps['circular_count']} circular dependencies",
                        "impact": "Better modularity and easier testing",
                    }
                )

        # Team recommendations
        if "contributors" in data:
            contributors = data["contributors"]
            bus_factor = contributors.get("bus_factor", 0)
            if bus_factor > 0 and bus_factor <= 2:
                recommendations.append(
                    {
                        "priority": "medium",
                        "category": "team",
                        "action": f"Improve knowledge sharing (current bus factor: {bus_factor})",
                        "impact": "Reduced key person risk",
                    }
                )

        # Only provide general recommendations if we have actually analyzed files
        # Don't show generic recommendations if health score is perfect due to no analysis
        if not recommendations:
            # Check if we actually analyzed files
            total_files = metrics.get("total_files", 0)
            health_score = data.get("overview", {}).get("health_score", 75)

            # Only add general recommendations if we analyzed files or health isn't perfect
            if total_files > 0 or health_score < 95:
                recommendations.append(
                    {
                        "priority": "low",
                        "category": "general",
                        "action": "Continue monitoring code quality metrics",
                        "impact": "Maintain current quality standards",
                    }
                )

            if metrics.get("total_files", 0) > 100:
                recommendations.append(
                    {
                        "priority": "low",
                        "category": "maintenance",
                        "action": "Consider establishing coding standards if not already in place",
                        "impact": "Consistent code quality across the team",
                    }
                )

        # Sort by priority
        priority_order = {"critical": 0, "high": 1, "medium": 2, "low": 3}
        recommendations.sort(key=lambda x: priority_order.get(x["priority"], 99))

        # Format as content
        content = []

        # Add introduction
        if recommendations:
            content.extend(
                [
                    "Based on the analysis of your codebase, here are actionable recommendations to improve code quality:",
                    "",
                ]
            )

            # Group by priority
            priority_groups = {}
            for rec in recommendations:
                priority = rec["priority"]
                if priority not in priority_groups:
                    priority_groups[priority] = []
                priority_groups[priority].append(rec)

            # Add recommendations by priority
            for priority in ["critical", "high", "medium", "low"]:
                if priority in priority_groups:
                    priority_title = {
                        "critical": "🚨 **Critical Issues**",
                        "high": "⚠️ **High Priority**",
                        "medium": "🔔 **Medium Priority**",
                        "low": "💡 **Suggestions**",
                    }[priority]

                    content.append(priority_title)
                    content.append("")

                    for rec in priority_groups[priority]:
                        content.append(f"• **{rec['action']}**")
                        content.append(f"  *Impact: {rec['impact']}*")
                        content.append("")
        else:
            content.extend(
                [
                    "✅ **Your codebase is in excellent condition!**",
                    "",
                    "No critical issues were identified. Continue with your current practices and consider:",
                    "- Regular code reviews",
                    "- Maintaining test coverage",
                    "- Monitoring metrics over time",
                ]
            )

        section.content = content

        return section

    def _create_complex_functions_table(
        self, complex_items: List[Dict[str, Any]]
    ) -> Dict[str, Any]:
        """Create table data for complex functions.

        Args:
            complex_items: List of complex function data

        Returns:
            Dict[str, Any]: Table configuration
        """
        return {
            "headers": ["Function", "File", "Complexity", "Lines", "Risk"],
            "rows": [
                [
                    item.get("name", "Unknown"),
                    item.get("file", "Unknown"),
                    item.get("complexity", 0),
                    item.get("lines", 0),
                    self._get_risk_level(item.get("complexity", 0)),
                ]
                for item in complex_items
            ],
        }

    def _create_contributors_table(self, contributors: List[Dict[str, Any]]) -> Dict[str, Any]:
        """Create table data for contributors.

        Args:
            contributors: List of contributor data

        Returns:
            Dict[str, Any]: Table configuration
        """
        return {
            "headers": ["Contributor", "Commits", "Lines Changed", "Files", "Last Active"],
            "rows": [
                [
                    contributor.get("name", "Unknown"),
                    contributor.get("commits", 0),
                    contributor.get("lines", 0),
                    contributor.get("files", 0),
                    self._format_days_ago(contributor.get("last_commit_days_ago", 999)),
                ]
                for contributor in contributors
            ],
        }

    def _create_hotspots_table(self, hotspots: List[Dict[str, Any]]) -> Dict[str, Any]:
        """Create table data for hotspots.

        Args:
            hotspots: List of hotspot data

        Returns:
            Dict[str, Any]: Table configuration
        """
        rows = []
        for hotspot in hotspots:
            # Determine reasons for being a hotspot
            reasons = []
            change_freq = hotspot.get("change_frequency", 0)
            complexity = hotspot.get("complexity", 0)
            score = hotspot.get("risk_score", 0)

            if change_freq > 20:
                reasons.append(f"High changes ({change_freq})")
            elif change_freq > 10:
                reasons.append(f"Frequent changes ({change_freq})")

            if complexity > 20:
                reasons.append(f"Very complex ({complexity})")
            elif complexity > 10:
                reasons.append(f"Complex ({complexity})")

            if score > 60:
                reasons.append("High risk score")
            elif score > 40:
                reasons.append("Elevated risk")

            # If no specific reasons, provide general one based on score
            if not reasons:
                if score > 30:
                    reasons.append("Combined risk factors")
                else:
                    reasons.append("Monitoring recommended")

            rows.append(
                [
                    hotspot.get("file", "Unknown"),
                    hotspot.get("risk_level", "low").upper(),
                    change_freq,
                    complexity,
                    f"{score:.1f}",
                    ", ".join(reasons),
                ]
            )

        return {
            "headers": ["File", "Risk Level", "Changes", "Complexity", "Score", "Reasons"],
            "rows": rows,
        }

    def _create_dependency_table(self, modules: List[Dict[str, Any]]) -> Dict[str, Any]:
        """Create table data for dependencies.

        Args:
            modules: List of module dependency data

        Returns:
            Dict[str, Any]: Table configuration
        """
        return {
            "headers": ["Module", "Dependencies", "Dependents", "Total Coupling"],
            "rows": [
                [
                    module.get("name", "Unknown"),
                    module.get("dependencies", 0),
                    module.get("dependents", 0),
                    module.get("dependencies", 0) + module.get("dependents", 0),
                ]
                for module in modules
            ],
        }

    def _get_risk_level(self, complexity: int) -> str:
        """Get risk level for complexity value.

        Args:
            complexity: Complexity value

        Returns:
            str: Risk level
        """
        if complexity > 20:
            return "Critical"
        elif complexity > 10:
            return "High"
        elif complexity > 5:
            return "Medium"
        else:
            return "Low"

    def _format_days_ago(self, days: int) -> str:
        """Format days ago as human-readable string.

        Args:
            days: Number of days

        Returns:
            str: Formatted string
        """
        if days == 0:
            return "Today"
        elif days == 1:
            return "Yesterday"
        elif days <= 7:
            return f"{days} days ago"
        elif days <= 30:
            weeks = days // 7
            return f"{weeks} week{'s' if weeks > 1 else ''} ago"
        elif days <= 365:
            months = days // 30
            return f"{months} month{'s' if months > 1 else ''} ago"
        else:
            years = days // 365
            return f"{years} year{'s' if years > 1 else ''} ago"

    def _generate_json_report(self, output_path: Path) -> Path:
        """Generate JSON format report.

        Args:
            output_path: Output file path

        Returns:
            Path: Path to generated report
        """
        report_data = {"metadata": self.metadata, "sections": self._sections_to_dict(self.sections)}

        with open(output_path, "w") as f:
            json.dump(report_data, f, indent=2, default=str)

        self.logger.info(f"JSON report generated: {output_path}")
        return output_path

    def _sections_to_dict(self, sections: List[ReportSection]) -> List[Dict[str, Any]]:
        """Convert sections to dictionary format.

        Args:
            sections: List of report sections

        Returns:
            List[Dict[str, Any]]: Sections as dictionaries
        """
        result = []
        for section in sections:
            section_dict = {
                "id": section.id,
                "title": section.title,
                "level": section.level,
                "order": section.order,
                "icon": section.icon,
                "content": section.content,
                "metrics": section.metrics,
                "tables": section.tables,
                "charts": section.charts,
                "code_snippets": section.code_snippets,
                "visible": section.visible,
                "collapsed": section.collapsed,
            }

            if section.subsections:
                section_dict["subsections"] = self._sections_to_dict(section.subsections)

            result.append(section_dict)

        return result
