"""Module containing a preprocessor that converts outputs in the notebook from
one format to another.
"""

# Copyright (c) Jupyter Development Team.
# Distributed under the terms of the Modified BSD License.

import base64
import io
import os
import sys
import subprocess

from ipython_genutils.py3compat import cast_unicode_py2
from testpath.tempdir import TemporaryDirectory
from traitlets import Unicode, default

from .convertfigures import ConvertFiguresPreprocessor

from shutil import which


# inkscape path for darwin (macOS)
INKSCAPE_APP = '/Applications/Inkscape.app/Contents/Resources/bin/inkscape'
# Recent versions of Inkscape (v1.0) moved the executable from 
# Resources/bin/inkscape to MacOS/inkscape
INKSCAPE_APP_v1 = '/Applications/Inkscape.app/Contents/MacOS/inkscape'

if sys.platform == "win32":
    try:
        import winreg
    except ImportError:
        import _winreg as winreg


class SVG2PDFPreprocessor(ConvertFiguresPreprocessor):
    """
    Converts all of the outputs in a notebook from SVG to PDF.
    """

    @default('from_format')
    def _from_format_default(self):
        return 'image/svg+xml'

    @default('to_format')
    def _to_format_default(self):
        return 'application/pdf'

    inkscape_version = Unicode(
        help="""The version of inkscape being used.

        This affects how the conversion command is run.
        """
    ).tag(config=True)

    @default('inkscape_version')
    def _inkscape_version_default(self):
        p = subprocess.Popen([self.inkscape, '--version'],
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE)
        output, _ = p.communicate()
        if p.returncode != 0:
            raise RuntimeError("Unable to find inkscape executable --version")
        return output.decode('utf-8').split(' ')[1]

    command = Unicode(
        help="""The command to use for converting SVG to PDF

        This string is a template, which will be formatted with the keys
        to_filename and from_filename.

        The conversion call must read the SVG from {from_filename},
        and write a PDF to {to_filename}.
        """).tag(config=True)

    @default('command')
    def _command_default(self):
        major_verison = self.inkscape_version.split('.')[0]
        export_option = ' --export-filename' if int(major_verison) > 0 else ' --export-pdf'
        gui_option = '' if int(major_verison) > 0 else ' --without-gui'

        return '{inkscape}{gui_option}{export_option}='.format(
            inkscape=self.inkscape, export_option=export_option, gui_option=gui_option
        ) + '"{to_filename}" "{from_filename}"'

    inkscape = Unicode(help="The path to Inkscape, if necessary").tag(config=True)
    @default('inkscape')
    def _inkscape_default(self):
        inkscape_path = which('inkscape')
        if inkscape_path is not None:
            return inkscape_path
        if sys.platform == "darwin":
            if os.path.isfile(INKSCAPE_APP_v1):
                return INKSCAPE_APP_v1
            # Order is important. If INKSCAPE_APP exists, prefer it over
            # the executable in the MacOS directory.
            if os.path.isfile(INKSCAPE_APP):
                return INKSCAPE_APP
        if sys.platform == "win32":
            wr_handle = winreg.ConnectRegistry(None, winreg.HKEY_LOCAL_MACHINE)
            try:
                rkey = winreg.OpenKey(wr_handle, "SOFTWARE\\Classes\\inkscape.svg\\DefaultIcon")
                inkscape = winreg.QueryValueEx(rkey, "")[0]
            except FileNotFoundError:
                raise FileNotFoundError("Inkscape executable not found")
            return inkscape
        return "inkscape"


    def convert_figure(self, data_format, data):
        """
        Convert a single SVG figure to PDF.  Returns converted data.
        """

        # Work in a temporary directory
        with TemporaryDirectory() as tmpdir:

            # Write fig to temp file
            input_filename = os.path.join(tmpdir, 'figure.svg')
            # SVG data is unicode text
            with io.open(input_filename, 'w', encoding='utf8') as f:
                f.write(cast_unicode_py2(data))

            # Call conversion application
            output_filename = os.path.join(tmpdir, 'figure.pdf')
            shell = self.command.format(from_filename=input_filename,
                                   to_filename=output_filename)
            subprocess.call(shell, shell=True) # Shell=True okay since input is trusted.

            # Read output from drive
            # return value expects a filename
            if os.path.isfile(output_filename):
                with open(output_filename, 'rb') as f:
                    # PDF is a nb supported binary, data type, so base64 encode.
                    return base64.encodebytes(f.read())
            else:
                raise TypeError("Inkscape svg to pdf conversion failed")
