import gdsfactory as gf
import kfactory as kf
from gdsfactory.component import Component
from gdsfactory.typings import ComponentSpec

from . import doroutes as _doroutes  # type: ignore
from . import util
from .types import (
    Dbu,
    DirectivePointsDbu,
    LayerLike,
    OrientationChar,
    OrientationTransition,
    PointsDbu,
    PortLike,
    StepDbu,
    validate_layer,
    validate_position,
    validate_position_with_orientation,
)


def add_route_astar(
    c: Component,
    start: PortLike,
    stop: PortLike,
    straight: ComponentSpec,
    bend: ComponentSpec,
    layer: LayerLike,
    grid_unit: Dbu,
) -> None:
    corners = find_route_astar(c, start, stop, straight, bend, layer, grid_unit)
    _start = validate_position_with_orientation(start)
    _stop = validate_position_with_orientation(stop)
    add_route_from_corners(
        c=c,
        start=(_start[0], _start[1]),
        stop=(_stop[0], _stop[1]),
        corners=corners,
        straight=straight,
        bend=bend,
    )


def add_route_from_steps(
    c: Component,
    start: PortLike,
    stop: PortLike,
    steps: list[StepDbu],
    straight: ComponentSpec,
    bend: ComponentSpec,
) -> None:
    _start = validate_position(start)
    _stop = validate_position(stop)
    corners = util.steps_to_corners(c, steps, _start)
    add_route_from_corners(c, _start, _stop, corners, straight, bend)


def add_route_from_corners(
    c: Component,
    start: PortLike,
    stop: PortLike,
    corners: PointsDbu,
    straight: ComponentSpec,
    bend: ComponentSpec,
) -> None:
    _start = validate_position(start)
    _stop = validate_position(stop)
    radius_dbu = util.extract_bend_radius(bend)
    directive_path = util.corners_to_directive_path(_start, _stop, corners, radius_dbu)
    _add_route_from_directive_path(c, directive_path, straight, bend)


def _add_route_from_directive_path(
    c: Component,
    directive_path: DirectivePointsDbu,
    straight: ComponentSpec,
    bend: ComponentSpec,
) -> None:
    straights, dirs, bends, transitions = util.directive_path_to_sdbt(directive_path)
    _add_route(c, straights, dirs, bends, transitions, straight, bend)


def _add_route(
    c: Component,
    straights: list[PointsDbu],
    directions: list[OrientationChar],
    bends: list[PointsDbu],
    transitions: list[OrientationTransition],
    straight: ComponentSpec,
    bend: ComponentSpec,
) -> None:
    inv_dbu = util.get_inv_dbu()
    for d, path in zip(directions, straights):
        if path[0] == path[-1]:
            continue
        s = gf.get_component(straight, length=util.straight_length(path))
        r = c << s
        util.orient_at_origin(r, d)
        r.dmove((float(path[0][0] / inv_dbu), float(path[0][1] / inv_dbu)))
    for tran, path in zip(transitions, bends):
        r = c << gf.get_component(bend)
        util.orient_as_transition_at_origin(r, tran)
        r.dmove((float(path[0][0] / inv_dbu), float(path[0][1] / inv_dbu)))


def find_route_astar(
    c: Component,
    start: PortLike,
    stop: PortLike,
    straight: ComponentSpec,
    bend: ComponentSpec,
    layer: LayerLike,
    grid_unit: Dbu,
) -> PointsDbu:
    _layer = validate_layer(layer)
    width_dbu = util.extract_waveguide_width(straight)
    radius_dbu = util.extract_bend_radius(bend)
    if grid_unit > 0.5 * radius_dbu:
        raise ValueError("bend radius should at least be twice the grid unit.")
    _grid_unit = int(radius_dbu / int(radius_dbu / grid_unit))
    bbox = c.bbox()
    straight_width = width_dbu // grid_unit + 1
    straight_width += (straight_width + 1) % 2
    _bend = util.discretize_bend(bend, _grid_unit, _layer)
    _start = validate_position_with_orientation(start)
    _stop = validate_position_with_orientation(
        stop, invert_orientation=isinstance(stop, gf.Port | kf.Port)
    )
    corners = _doroutes.show(
        polys=util.extract_polys(c, _layer),
        bbox=(bbox.top, bbox.right, bbox.bottom, bbox.left),
        start=_start,
        stop=_stop,
        grid_unit=_grid_unit,
        straight_width=straight_width,
        discretized_bend_east_to_north=_bend,
    )
    return corners
