"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const sqs = require("@aws-cdk/aws-sqs");
const sns = require("@aws-cdk/aws-sns");
const kms = require("@aws-cdk/aws-kms");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
// --------------------------------------------------------------
// Pattern deployment with new Topic, new Queue and
// default properties
// --------------------------------------------------------------
test('Pattern deployment w/ new Topic, new Queue and default props', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {};
    new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Assertion 2
    expect(stack).toHaveResource("AWS::SNS::Topic", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "EncryptionKey1B843E66",
                "Arn"
            ]
        }
    });
    // Assertion 3
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "EncryptionKey1B843E66",
                "Arn"
            ]
        }
    });
    // Assertion 4
    expect(stack).toHaveResource("AWS::SNS::Subscription", {
        Protocol: "sqs",
        TopicArn: {
            Ref: "testsnssqsSnsTopic2CD0065B"
        },
        Endpoint: {
            "Fn::GetAtt": [
                "testsnssqsqueueB02504BF",
                "Arn"
            ]
        }
    });
});
// --------------------------------------------------------------
// Pattern deployment with new Topic, new Queue, and
// overridden properties
// --------------------------------------------------------------
test('Pattern deployment w/ new topic, new queue, and overridden props', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        topicProps: {
            topicName: "new-topic",
        },
        queueProps: {
            queueName: "new-queue.fifo",
            fifo: true
        },
        enableEncryptionWithCustomerManagedKey: true,
        encryptionKeyProps: {
            enableKeyRotation: false
        },
        deployDeadLetterQueue: false,
        maxReceiveCount: 0
    };
    new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    // Assertion 1
    expect(stack).toHaveResource("AWS::SNS::Topic", {
        TopicName: "new-topic",
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "EncryptionKey1B843E66",
                "Arn"
            ]
        }
    });
    // Assertion 2
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        QueueName: "new-queue.fifo",
        FifoQueue: true
    });
    // Assertion 3
    expect(stack).toHaveResource("AWS::KMS::Key", {
        EnableKeyRotation: false
    });
});
// --------------------------------------------------------------
// Test the getter methods
// --------------------------------------------------------------
test('Test getter methods', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        enableEncryptionWithCustomerManagedKey: true,
        deployDeadLetterQueue: true,
        maxReceiveCount: 0
    };
    const app = new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    // Assertion 1
    expect(app.snsTopic !== null);
    // Assertion 2
    expect(app.encryptionKey !== null);
    // Assertion 3
    expect(app.sqsQueue !== null);
    // Assertion 4
    expect(app.deadLetterQueue !== null);
});
// --------------------------------------------------------------
// Test deployment with existing queue, and topic
// --------------------------------------------------------------
test('Test deployment w/ existing queue, and topic', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    const topic = new sns.Topic(stack, "existing-topic-obj", {
        topicName: 'existing-topic-obj'
    });
    const queue = new sqs.Queue(stack, 'existing-queue-obj', {
        queueName: 'existing-queue-obj'
    });
    const app = new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
        existingTopicObj: topic,
        existingQueueObj: queue
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Assertion 2
    expect(app.snsTopic !== null);
    // Assertion 3
    expect(stack).toHaveResource("AWS::SNS::Topic", {
        TopicName: "existing-topic-obj"
    });
    // Assertion 4
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        QueueName: "existing-queue-obj"
    });
});
// --------------------------------------------------------------
// Test deployment with imported encryption key
// --------------------------------------------------------------
test('Test deployment with imported encryption key', () => {
    // Stack
    const stack = new core_1.Stack();
    // Setup
    const kmsKey = new kms.Key(stack, 'imported-key', {
        enableKeyRotation: false
    });
    // Helper declaration
    new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
        enableEncryptionWithCustomerManagedKey: true,
        encryptionKey: kmsKey
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Assertion 2
    expect(stack).toHaveResource("AWS::KMS::Key", {
        EnableKeyRotation: false
    });
    // Assertion 3
    expect(stack).toHaveResource("AWS::SNS::Topic", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "importedkey38675D68",
                "Arn"
            ]
        }
    });
});
// --------------------------------------------------------------
// Test deployment with SNS managed KMS key
// --------------------------------------------------------------
test('Test deployment with SNS managed KMS key', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
        topicProps: {
            masterKey: kms.Alias.fromAliasName(stack, 'sns-managed-key', 'alias/aws/sns')
        },
        queueProps: {
            encryption: sqs.QueueEncryption.KMS
        },
        enableEncryptionWithCustomerManagedKey: false
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Assertion 2
    expect(stack).toHaveResource("AWS::SNS::Topic", {
        KmsMasterKeyId: {
            "Fn::Join": [
                "",
                [
                    "arn:",
                    {
                        Ref: "AWS::Partition"
                    },
                    ":kms:",
                    {
                        Ref: "AWS::Region"
                    },
                    ":",
                    {
                        Ref: "AWS::AccountId"
                    },
                    ":alias/aws/sns"
                ]
            ]
        }
    });
    // Assertion 3
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "snstosqsstackqueueKey743636E7",
                "Arn"
            ]
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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