# -*- coding: utf-8 -*-
"""
# ---------------------------------------------------------------------------------------------------------
# ProjectName:  qlv-helper
# FileName:     order_page.py
# Description:  订单页的HTTP响应处理模块
# Author:       ASUS
# CreateDate:   2025/11/28
# Copyright ©2011-2025. Hunan xxxxxxx Company limited. All rights reserved.
# ---------------------------------------------------------------------------------------------------------
"""
import aiohttp
from bs4 import BeautifulSoup
from typing import Dict, Any, Optional
from http_helper.client.async_proxy import HttpClientFactory
from qlv_helper.utils.datetime_utils import get_current_dtstr


async def get_order_page_html(
        order_id: int, domain: str, protocol: str = "http", retry: int = 1, timeout: int = 5, enable_log: bool = True,
        cookie_jar: Optional[aiohttp.CookieJar] = None, playwright_state: Dict[str, Any] = None
) -> Dict[str, Any]:
    order_http_client = HttpClientFactory(
        protocol=protocol if protocol == "http" else "https",
        domain=domain,
        timeout=timeout,
        retry=retry,
        enable_log=enable_log,
        cookie_jar=cookie_jar,
        playwright_state=playwright_state
    )
    return await order_http_client.request(
        method="get",
        url=f"/OrderProcessing/NewTicket_show/{order_id}?&r={get_current_dtstr()}",
        is_end=True
    )


def parser_order_info(html: str) -> Dict[str, Any]:
    soup = BeautifulSoup(html, "html.parser")
    # 找到目标 table
    table = soup.find("table", class_="table no_border")
    if not table:
        return {}

    # 所有 td
    tds = table.find_all("td")
    result = {}

    for td in tds:
        text = td.get_text(strip=True)
        # 如果 td 内没有冒号、也不是按钮，跳过
        if "：" not in text:
            continue
        # 按 "：" 进行分割
        try:
            key, value = text.split("：", 1)
        except (Exception, ValueError):
            continue
        # 去掉换行符、空格
        key = key.strip()
        value = value.strip()

        # 如果 value 为空，尝试取 <b> 或其他控件内文本
        if not value:
            b = td.find("b")
            if b:
                value = b.get_text(strip=True)

        # 去掉尾部不需要的空格
        value = value.replace("\u00a0", "").strip()
        result[key] = value

    return result
