from __future__ import annotations

from re import match
from typing import Callable, Mapping, TypeAlias, Union

from . import dry

DryType: TypeAlias = Union[
    bool,
    int,
    float,
    str,
    list[bool],
    list[int],
    list[float],
    list[str],
    list[bool | int],
    list[bool | float],
    list[bool | str],
    list[int | float],
    list[int | str],
    list[float | str],
    list[bool | int | float],
    list[bool | int | str],
    list[bool | float | str],
    list[int | float | str],
    list[bool | int | float | str],
    list['DryType'],
    Mapping[bool, 'DryType'],
    Mapping[int, 'DryType'],
    Mapping[str, 'DryType'],
    Mapping[bool | int, 'DryType'],
    Mapping[bool | str, 'DryType'],
    Mapping[int | str, 'DryType'],
]

DryFunction: TypeAlias = Callable[..., DryType]


class Webview:
    _title: str
    _min_size: tuple[int, int]
    _size: tuple[int, int]
    _icon_path: str | None = None
    _html: str | None = None
    _url: str | None = None
    _api: Mapping[str, DryFunction] | None = None
    _dev_tools: bool = False

    def __init__(
        self,
        title: str = 'My Dry Webview',
        min_size: tuple[int, int] = (1152, 720),
        size: tuple[int, int] = (1280, 800),
        icon_path: str | None = None,
        content: str = '<h1>Hello, World!</h1>',
        api: Mapping[str, DryFunction] | None = None,
        dev_tools: bool = False,
    ):
        '''
        Initialize the webview window.
        
        :param title: The title of the webview window.
        :param min_size: The minimum size of the webview window.
        :param size: The size of the webview window.
        :param icon_path: The path to the icon of the webview window (only .ico).
        :param content: The content of the webview window, either an HTML or a URL.
        :param api: The functions being passed down to the webview window.
        :param dev_tools: Whether the developer tools are enabled.

        :type title: str
        :type min_size: tuple[int, int]
        :type size: tuple[int, int]
        :type icon_path: str | None
        :type content: str
        :type api: Mapping[str, DryFunction] | None
        :type dev_tools: bool

        :return: Webview
        '''
        self.title = title
        self.min_size = min_size
        self.size = size
        self.content = content
        self.api = api
        self.dev_tools = dev_tools
        self.icon_path = icon_path

    @property
    def title(self) -> str:
        """
        Get the title of the webview window.
        """
        return self._title

    @title.setter
    def title(self, title: str) -> None:
        """
        Set the title of the webview window.
        """
        self._title = title

    @property
    def min_size(self) -> tuple[int, int]:
        """
        Get the minimum size of the webview window.
        """
        return self._min_size

    @min_size.setter
    def min_size(self, width_and_height: tuple[int, int]) -> None:
        """
        Set the minimum size of the webview window.
        """
        self._min_size = width_and_height

    @property
    def size(self) -> tuple[int, int]:
        """
        Get the size of the webview window.
        """
        return self._size

    @size.setter
    def size(self, width_and_height: tuple[int, int]) -> None:
        """
        Set the size of the webview window.
        """
        self._size = width_and_height

    @property
    def content(self) -> str | None:
        """
        Get the content of the webview window.
        """
        return self._html or self._url

    @content.setter
    def content(self, content: str) -> None:
        """
        Set the content of the webview window, either an HTML or a URL.
        """
        is_url = bool(match(r'https?://[a-z0-9.-]+', content))
        self._url, self._html = (content, None) if is_url else (None, content)

    @property
    def api(self) -> Mapping[str, DryFunction] | None:
        """
        Get the functions being passed down to the webview window.
        """
        return self._api

    @api.setter
    def api(self, api: Mapping[str, DryFunction] | None) -> None:
        """
        Set the functions being passed down to the webview window.
        """
        self._api = api

    @property
    def dev_tools(self) -> bool:
        """
        Get whether the developer tools are enabled.
        """
        return self._dev_tools

    @dev_tools.setter
    def dev_tools(self, dev_tools: bool) -> None:
        """
        Set whether the developer tools are enabled.
        """
        self._dev_tools = dev_tools
    
    @property
    def icon_path(self) -> str | None:
        """
        Get the path to the icon of the webview window.
        """
        return self._icon_path
    
    @icon_path.setter
    def icon_path(self, icon_path: str | None) -> None:
        """
        Set the path to the icon of the webview window (only .ico).
        """
        if self._icon_path and not self._icon_path.endswith('.ico'):
            raise ValueError('The icon file must be of type .ico')
        self._icon_path = icon_path

    def run(self):
        """
        Run the webview window, in a blocking loop.
        """
        dry.run(
            title=self.title,
            min_size=self.min_size,
            size=self.size,
            icon_path=self.icon_path,
            html=self._html,
            url=self._url,
            api=self.api,
            dev_tools=self.dev_tools,
        )
