# Copyright 2022 Cartesi Pte. Ltd.
#
# SPDX-License-Identifier: Apache-2.0
# Licensed under the Apache License, Version 2.0 (the "License"); you may not use
# this file except in compliance with the License. You may obtain a copy of the
# License at http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software distributed
# under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
# CONDITIONS OF ANY KIND, either express or implied. See the License for the
# specific language governing permissions and limitations under the License.

from cartesi_wallet.util import str_to_hex


class Output():
    """
    Base class representing a result generated by processing an input.

    It is resposible for converting its payload to the format expected by
    Cartesi rollups.

        Parameters:
            payload(str): the actual data generated after processing the input
    """

    def __init__(self, payload: str):
        if payload[:2] == "0x":
            self.payload = payload
        else:
            self.payload = str_to_hex(payload)


class Voucher(Output):
    """
    A Voucher is an `Output` representing a transaction
    that can be carried out on the base layer blockchain,
    such as a transfer of assets.

        Parameters:
            destination(str): destination of the contract who will execute the payload
            payload(bytes): an ABI encoded contract function call
    """

    def __init__(self, destination: str, payload: bytes):
        self.destination = destination
        hexpayload = "0x" + payload.hex()
        super().__init__(hexpayload)


class Notice(Output):
    """
    A Notice is an `Output` representing an informational statement
    that can be validated in the base layer blockchain.

        Parameters:
            payload(str): a string containing arbitrary data
    """

    def __init__(self, payload: str):
        super().__init__(payload)


class Log(Output):
    """
    A Log is an `Output` representing an application log.

        Parameters:
            payload(str): a string containing arbitrary data
    """

    def __init__(self, payload: str):
        super().__init__(payload)


class Error(Output):
    """
    An Error is an `Output` representing that an error has ocurred
    during the processing of an input.

    If something unexpected such as an exception or error occurs during
    the processing of an input, that input must be rejected.

        Parameters:
            payload(str): a string containing arbitrary data
    """

    def __init__(self, payload: str):
        super().__init__(payload)
