"""
HTTP client for sandbox contexts API.

This module provides HTTP client implementation to interact with sandbox contexts API,
following the same pattern as the Go implementation in code_interpreter_data.go
"""

import json
import logging
from typing import Dict, List, Optional, Any
from datetime import datetime
import httpx

logger = logging.getLogger(__name__)


class SandboxHTTPClient:
    """HTTP client for sandbox contexts operations."""
    
    def __init__(self, base_url: str, timeout: float = 30.0):
        """
        Initialize sandbox HTTP client.
        
        Args:
            base_url: Base URL of the sandbox service
            timeout: Request timeout in seconds
        """
        self.base_url = base_url.rstrip('/')
        self.timeout = timeout
        self.client = httpx.Client(timeout=timeout)
        logger.info(f"Initialized SandboxHTTPClient with base_url: {self.base_url}")
    
    def _make_request(
        self,
        method: str,
        endpoint: str,
        session_id: Optional[str] = None,
        json_data: Optional[Dict[str, Any]] = None
    ) -> Any:
        """
        Make HTTP request to sandbox service.
        
        Args:
            method: HTTP method (GET, POST, DELETE, etc.)
            endpoint: API endpoint path
            session_id: Optional session ID for request header
            json_data: Optional JSON request body
            
        Returns:
            Response data (parsed JSON or None)
            
        Raises:
            Exception: If request fails
        """
        url = f"{self.base_url}{endpoint}"
        headers = {
            "Content-Type": "application/json",
            "Accept": "application/json"
        }
        
        if session_id:
            headers["X-Session-ID"] = session_id
        
        logger.debug(f"Making {method} request to {url}")
        
        try:
            response = self.client.request(
                method=method,
                url=url,
                headers=headers,
                json=json_data if json_data is not None else None
            )
            
            logger.info(
                f"Response: status_code={response.status_code}, "
                f"body_length={len(response.content)}"
            )
            
            # Check status code
            if response.status_code < 200 or response.status_code >= 300:
                error_msg = f"HTTP request failed with status {response.status_code}: {response.text}"
                logger.error(error_msg)
                raise Exception(error_msg)
            
            # Parse response if not empty
            if len(response.content) > 0:
                try:
                    return response.json()
                except Exception as json_error:
                    # If JSON parsing fails, return the raw text
                    logger.warning(f"Failed to parse JSON response: {json_error}, returning raw text")
                    return response.text
            
            return None
            
        except httpx.HTTPError as e:
            error_msg = f"HTTP request failed: {str(e)}"
            logger.error(error_msg)
            raise Exception(error_msg) from e
    
    def create_context(
        self,
        language: str = "python",
        cwd: Optional[str] = None,
        session_id: Optional[str] = None
    ) -> Dict[str, Any]:
        """
        Create a new execution context.
        
        Args:
            language: Programming language (default: python)
            cwd: Optional working directory
            session_id: Optional session ID
            
        Returns:
            Context information dict with keys: id, language, cwd, created_at
            
        Example response:
            {
                "id": "context-uuid",
                "language": "python",
                "cwd": "/home/user",
                "created_at": "2025-10-25T03:00:00Z"
            }
            
        Raises:
            Exception: If context creation fails or response is invalid
        """
        logger.info(f"Creating context with language={language}, cwd={cwd}")
        
        request_body = {"language": language}
        if cwd:
            request_body["cwd"] = cwd
        
        response = self._make_request(
            method="POST",
            endpoint="/contexts",
            session_id=session_id,
            json_data=request_body
        )
        
        # Validate response
        if not isinstance(response, dict):
            error_msg = f"Invalid response type: expected dict, got {type(response).__name__}"
            logger.error(error_msg)
            raise Exception(error_msg)
        
        if "id" not in response:
            error_msg = f"Response missing 'id' field: {response}"
            logger.error(error_msg)
            raise Exception(error_msg)
        
        logger.info(f"Context created: {response.get('id')}")
        return response
    
    def list_contexts(self, session_id: Optional[str] = None) -> List[Dict[str, Any]]:
        """
        List all available contexts.
        
        Args:
            session_id: Optional session ID
            
        Returns:
            List of context information dicts
            
        Example response:
            [
                {
                    "id": "context-uuid-1",
                    "language": "python",
                    "cwd": "/home/user",
                    "created_at": "2025-10-25T03:00:00Z"
                },
                {
                    "id": "context-uuid-2",
                    "language": "python",
                    "cwd": "/tmp",
                    "created_at": "2025-10-25T03:01:00Z"
                }
            ]
        """
        logger.info("Listing contexts")
        
        response = self._make_request(
            method="GET",
            endpoint="/contexts",
            session_id=session_id
        )
        
        contexts = response if isinstance(response, list) else []
        logger.info(f"Found {len(contexts)} contexts")
        return contexts
    
    def remove_context(
        self,
        context_id: str,
        session_id: Optional[str] = None
    ) -> None:
        """
        Remove a specific context.
        
        Args:
            context_id: Context ID to remove
            session_id: Optional session ID
            
        Raises:
            Exception: If removal fails
        """
        logger.info(f"Removing context: {context_id}")
        
        self._make_request(
            method="DELETE",
            endpoint=f"/contexts/{context_id}",
            session_id=session_id
        )
        
        logger.info(f"Context removed: {context_id}")
    
    def restart_context(
        self,
        context_id: str,
        session_id: Optional[str] = None
    ) -> None:
        """
        Restart a specific context.
        
        Args:
            context_id: Context ID to restart
            session_id: Optional session ID
            
        Raises:
            Exception: If restart fails
        """
        logger.info(f"Restarting context: {context_id}")
        
        self._make_request(
            method="POST",
            endpoint=f"/contexts/{context_id}/restart",
            session_id=session_id
        )
        
        logger.info(f"Context restarted: {context_id}")
    
    def health_check(self) -> Dict[str, Any]:
        """
        Check sandbox service health.
        
        Returns:
            Health status information
            
        Example response:
            {
                "status": "OK",
                "timestamp": "2025-10-25T03:00:00Z"
            }
            
        Raises:
            Exception: If health check fails
        """
        logger.info("Checking sandbox health")
        
        response = self._make_request(
            method="GET",
            endpoint="/health"
        )
        
        # Handle different response formats
        if isinstance(response, dict):
            # Add timestamp if not present
            if "timestamp" not in response:
                response["timestamp"] = datetime.now().isoformat()
            return response
        elif isinstance(response, str):
            return {"status": response, "timestamp": datetime.now().isoformat()}
        else:
            return {"status": "OK", "timestamp": datetime.now().isoformat()}
    
    def close(self):
        """Close the HTTP client."""
        self.client.close()
        logger.info("SandboxHTTPClient closed")
    
    def __enter__(self):
        """Context manager entry."""
        return self
    
    def __exit__(self, exc_type, exc_val, exc_tb):
        """Context manager exit."""
        self.close()

