#!/usr/bin/env python3
"""
FastMCP-based MCP Server for Sandbox Code Interpreter

This server uses FastMCP framework to provide a unified interface
supporting stdio, SSE, and HTTP Streamable transports.
"""

import asyncio
import json
import logging
import os
import sys
from typing import Any, Dict, List, Optional
from datetime import datetime
import aiohttp

# FastMCP imports
from fastmcp import FastMCP
from fastmcp.server.http import create_sse_app
from starlette.middleware import Middleware
from starlette.middleware.cors import CORSMiddleware

# E2B imports
from e2b_on_fc import Sandbox as E2BSandbox
from e2b import TimeoutException

# Configure logging
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

# Global state
contexts: Dict[str, Dict[str, Any]] = {}
e2b_sandbox: Optional[E2BSandbox] = None
configured_sandbox_url: str = "http://localhost:5001"

# Initialize FastMCP server
mcp = FastMCP("Sandbox Code Interpreter")

async def initialize_e2b_sandbox(sandbox_url: str = 'http://localhost:5001'):
    """Initialize E2B sandbox connection."""
    global e2b_sandbox, configured_sandbox_url
    try:
        configured_sandbox_url = sandbox_url
        e2b_sandbox = E2BSandbox(envd_url=sandbox_url)
        logger.info(f"E2B sandbox initialized successfully with URL: {sandbox_url}")
    except Exception as e:
        logger.error(f"Failed to initialize E2B sandbox: {e}")
        raise

async def cleanup_e2b_sandbox():
    """Cleanup E2B sandbox connection."""
    global e2b_sandbox
    if e2b_sandbox:
        try:
            # E2B sandbox cleanup
            if hasattr(e2b_sandbox, 'close'):
                await e2b_sandbox.close()
            elif hasattr(e2b_sandbox, 'kill'):
                await e2b_sandbox.kill()
            logger.info("E2B sandbox closed successfully")
        except Exception as e:
            logger.error(f"Error closing E2B sandbox: {e}")
        finally:
            e2b_sandbox = None

async def async_health_check_task(sandbox_url: str):
    """
    异步健康检查任务：检查 sandbox_url/health 端点
    - 每 10 秒检查一次
    - 持续 2 分钟（12 次检查）
    - 超时时间 3 秒
    - 返回 200 状态码时提前结束
    """
    logger.info(f"Starting async health check for sandbox: {sandbox_url}")
    
    # 确保 URL 有 /health 端点
    health_url = sandbox_url
    if not health_url.endswith('/health'):
        if not health_url.endswith('/'):
            health_url += '/'
        health_url += 'health'
    
    max_attempts = 12  # 2 分钟 = 120 秒，每 10 秒一次 = 12 次
    check_interval = 10  # 10 秒
    timeout = 3  # 3 秒超时
    
    for attempt in range(1, max_attempts + 1):
        try:
            logger.info(f"Health check attempt {attempt}/{max_attempts} for {health_url}")
            
            async with aiohttp.ClientSession() as session:
                async with session.get(health_url, timeout=aiohttp.ClientTimeout(total=timeout)) as response:
                    status_code = response.status
                    
                    if status_code == 200:
                        logger.info(f"✅ Sandbox health check successful! Status: {status_code}")
                        logger.info(f"Sandbox is ready after {attempt} attempts")
                        return True
                    else:
                        logger.warning(f"⚠️ Sandbox health check returned status {status_code} (attempt {attempt}/{max_attempts})")
                        
        except aiohttp.ClientTimeout:
            logger.warning(f"⏰ Health check timeout for {health_url} (attempt {attempt}/{max_attempts})")
        except aiohttp.ClientError as e:
            logger.warning(f"🔌 Health check connection error for {health_url}: {str(e)} (attempt {attempt}/{max_attempts})")
        except Exception as e:
            logger.warning(f"❌ Health check failed for {health_url}: {str(e)} (attempt {attempt}/{max_attempts})")
        
        # 如果不是最后一次尝试，等待下次检查
        if attempt < max_attempts:
            logger.info(f"Waiting {check_interval} seconds before next health check...")
            await asyncio.sleep(check_interval)
    
    logger.warning(f"❌ Sandbox health check failed after {max_attempts} attempts over 2 minutes")
    logger.warning(f"Sandbox at {health_url} may not be ready")
    return False

@mcp.tool
async def run_code(code: str, context_id: Optional[str] = None, language: Optional[str] = None) -> str:
    """
    Execute code in the sandbox environment.
    
    Args:
        code: Code to execute
        context_id: Optional context ID for isolated execution
        language: Programming language (required if context_id is not provided)
        
    Returns:
        Execution result as string
    """
    try:
        # Validate parameters: if no context_id, language is required
        if not context_id and not language:
            return "Error: Either context_id or language parameter must be provided"
        
        if not e2b_sandbox:
            await initialize_e2b_sandbox()
        
        # Execute code using E2B sandbox with language parameter
        if language:
            execution = e2b_sandbox.run_code(code, language=language)
        else:
            execution = e2b_sandbox.run_code(code)
        
        # Format the result
        if execution.error:
            result = f"Error: {execution.error.name}: {execution.error.value}\n{execution.error.traceback}"
        else:
            # 收集所有输出
            output_parts = []
            
            # 添加 stdout 输出
            if execution.logs.stdout:
                stdout_lines = []
                for msg in execution.logs.stdout:
                    if hasattr(msg, 'line'):
                        stdout_lines.append(msg.line)
                    else:
                        stdout_lines.append(str(msg))
                stdout_output = "\n".join(stdout_lines)
                if stdout_output.strip():
                    output_parts.append(stdout_output)
            
            # 添加 stderr 输出
            if execution.logs.stderr:
                stderr_lines = []
                for msg in execution.logs.stderr:
                    if hasattr(msg, 'line'):
                        stderr_lines.append(msg.line)
                    else:
                        stderr_lines.append(str(msg))
                stderr_output = "\n".join(stderr_lines)
                if stderr_output.strip():
                    output_parts.append(f"stderr: {stderr_output}")
            
            # 添加执行结果
            if execution.text:
                output_parts.append(execution.text)
            
            # 组合所有输出
            if output_parts:
                result = "\n".join(output_parts)
            else:
                result = "Code executed successfully (no output)"
        
        # Update context if provided
        if context_id and context_id in contexts:
            contexts[context_id]["last_used"] = datetime.now().isoformat()
        
        return result
        
    except TimeoutException as e:
        error_msg = f"Code execution timeout: {str(e)}. Configured sandbox URL: {configured_sandbox_url}"
        logger.error(error_msg)
        return error_msg
    except Exception as e:
        error_msg = f"Unexpected error: {str(e)}. Configured sandbox URL: {configured_sandbox_url}"
        logger.error(error_msg)
        return error_msg

@mcp.tool
async def create_context(language: str = "python", description: str = "") -> str:
    """
    Create a new execution context in the sandbox.
    
    Args:
        language: Programming language (default: python)
        description: Optional description for the context
        
    Returns:
        Context ID
    """
    try:
        # 使用 HTTP 客户端创建上下文
        from e2b_on_fc.sandbox_http_client import SandboxHTTPClient
        
        with SandboxHTTPClient(configured_sandbox_url) as client:
            # 在沙盒中创建上下文
            sandbox_context = client.create_context(language=language)
            
            context_id = sandbox_context.get('id')
            if not context_id:
                error_msg = f"Failed to get context ID from sandbox response: {sandbox_context}"
                logger.error(error_msg)
                return error_msg
            
            # 在本地记录上下文信息
            contexts[context_id] = {
                "id": context_id,
                "language": language,
                "description": description,
                "created_at": sandbox_context.get('created_at', datetime.now().isoformat()),
                "last_used": datetime.now().isoformat(),
                "cwd": sandbox_context.get('cwd', '')
            }
            
            logger.info(f"Created context {context_id} with language {language} in sandbox")
            return json.dumps({
                "context_id": context_id,
                "language": language,
                "description": description,
                "created_at": contexts[context_id]["created_at"]
            }, indent=2)
        
    except Exception as e:
        error_msg = f"Failed to create context: {str(e)}. Configured sandbox URL: {configured_sandbox_url}"
        logger.error(error_msg)
        return error_msg

@mcp.tool
async def stop_context(context_id: str) -> str:
    """
    Stop and remove a context from the sandbox.
    
    Args:
        context_id: Context ID to stop
        
    Returns:
        Success message
    """
    try:
        # 使用 HTTP 客户端停止上下文
        from e2b_on_fc.sandbox_http_client import SandboxHTTPClient
        
        if context_id not in contexts:
            return json.dumps({
                "success": False,
                "message": f"Context {context_id} not found in local records"
            }, indent=2)
        
        with SandboxHTTPClient(configured_sandbox_url) as client:
            # 在沙盒中删除上下文
            try:
                client.remove_context(context_id)
                logger.info(f"Removed context {context_id} from sandbox")
            except Exception as e:
                logger.warning(f"Failed to remove context {context_id} from sandbox: {str(e)}")
        
        # 从本地记录中删除
        del contexts[context_id]
        logger.info(f"Removed context {context_id} from local records")
        
        return json.dumps({
            "success": True,
            "message": f"Context {context_id} stopped successfully"
        }, indent=2)
            
    except Exception as e:
        error_msg = f"Failed to stop context: {str(e)}. Configured sandbox URL: {configured_sandbox_url}"
        logger.error(error_msg)
        return error_msg

@mcp.tool
async def list_contexts() -> str:
    """
    List all available contexts from the sandbox.
    
    Returns:
        JSON string of contexts
    """
    try:
        # 使用 HTTP 客户端列出上下文
        from e2b_on_fc.sandbox_http_client import SandboxHTTPClient
        
        with SandboxHTTPClient(configured_sandbox_url) as client:
            # 从沙盒获取上下文列表
            sandbox_contexts = client.list_contexts()
            logger.info(f"Retrieved {len(sandbox_contexts)} contexts from sandbox")
            
            # 更新本地记录，同步沙盒中的上下文
            sandbox_context_ids = {ctx.get('id') for ctx in sandbox_contexts}
            
            # 添加沙盒中存在但本地不存在的上下文
            for ctx in sandbox_contexts:
                context_id = ctx.get('id')
                if context_id and context_id not in contexts:
                    contexts[context_id] = {
                        "id": context_id,
                        "language": ctx.get('language', 'unknown'),
                        "description": "",
                        "created_at": ctx.get('created_at', datetime.now().isoformat()),
                        "last_used": datetime.now().isoformat(),
                        "cwd": ctx.get('cwd', '')
                    }
                    logger.info(f"Added context {context_id} from sandbox to local records")
            
            # 标记本地存在但沙盒中不存在的上下文
            result = []
            for context_id, context_info in contexts.items():
                context_info_copy = context_info.copy()
                context_info_copy['sandbox_exists'] = context_id in sandbox_context_ids
                
                if context_id not in sandbox_context_ids:
                    context_info_copy['status'] = 'orphaned'
                    context_info_copy['warning'] = 'Context exists locally but not in sandbox'
                else:
                    context_info_copy['status'] = 'active'
                
                result.append(context_info_copy)
            
            return json.dumps(result, indent=2)
        
    except Exception as e:
        error_msg = f"Failed to list contexts: {str(e)}. Configured sandbox URL: {configured_sandbox_url}"
        logger.error(error_msg)
        return error_msg

@mcp.tool
async def health_check() -> str:
    """
c    Check the health status of the FastMCP service.
    
    Returns:
        Health status information
    """
    try:
        # 只检查 FastMCP 服务状态，不检查 sandbox_url 连通性
        result = {
            "status": "healthy",
            "service": "fc-code-interpreter-mcp-server",
            "version": "0.0.10",
            "timestamp": datetime.now().isoformat(),
            "sandbox_url": configured_sandbox_url,
            "sandbox_available": e2b_sandbox is not None,
            "contexts_count": len(contexts)
        }
        
        logger.info("FastMCP service health check passed")
        return json.dumps(result, indent=2)
        
    except Exception as e:
        error_msg = f"Health check failed: {str(e)}"
        logger.error(error_msg)
        return json.dumps({
            "status": "error",
            "service": "fc-code-interpreter-mcp-server",
            "error": error_msg,
            "timestamp": datetime.now().isoformat()
        }, indent=2)

async def initialize_server(sandbox_url: str):
    """Initialize server resources."""
    try:
        await initialize_e2b_sandbox(sandbox_url)
        logger.info("Server initialized successfully")
        
        # 启动异步健康检查任务（不等待完成）
        asyncio.create_task(async_health_check_task(sandbox_url))
        logger.info("Async health check task started")
        
    except Exception as e:
        logger.error(f"Failed to initialize server: {e}")
        raise

async def cleanup_server():
    """Cleanup server resources."""
    try:
        await cleanup_e2b_sandbox()
        logger.info("Server cleanup completed")
    except Exception as e:
        logger.error(f"Error during server cleanup: {e}")

def parse_args():
    """Parse command line arguments with environment variable support."""
    import argparse
    
    # 从环境变量读取默认值（命令行参数优先级更高）
    default_transport = os.getenv('MCP_TRANSPORT', 'stdio')
    default_host = os.getenv('MCP_HOST', '0.0.0.0')
    default_port = int(os.getenv('MCP_PORT', '3000'))
    default_path = os.getenv('MCP_PATH', '/mcp')
    default_sandbox_url = os.getenv('SANDBOX_URL', 'http://localhost:5001')
    
    parser = argparse.ArgumentParser(description="FastMCP Sandbox Code Interpreter Server")
    parser.add_argument("--transport", choices=["stdio", "sse", "http"], default=default_transport,
                       help=f"Transport protocol to use (default: {default_transport}, env: MCP_TRANSPORT)")
    parser.add_argument("--host", default=default_host, 
                       help=f"Host to bind to (default: {default_host}, env: MCP_HOST)")
    parser.add_argument("--port", type=int, default=default_port, 
                       help=f"Port to bind to (default: {default_port}, env: MCP_PORT)")
    parser.add_argument("--path", default=default_path, 
                       help=f"Path for HTTP transport (default: {default_path}, env: MCP_PATH)")
    parser.add_argument("--sandbox-url", default=default_sandbox_url, 
                       help=f"E2B sandbox URL (default: {default_sandbox_url}, env: SANDBOX_URL)")
    
    return parser.parse_args()

def main():
    """Main entry point."""
    try:
        # Parse arguments
        args = parse_args()
        
        # Get host and port from args or environment
        host = args.host or os.getenv('MCP_HOST', '0.0.0.0')
        port = args.port or int(os.getenv('MCP_PORT', '3000'))
        path = args.path or os.getenv('MCP_PATH', '/mcp')
        
        logger.info(f"Starting FastMCP server with {args.transport} transport")
        logger.info(f"Host: {host}, Port: {port}")
        logger.info(f"Sandbox URL: {args.sandbox_url}")
        
        # Initialize server with sandbox URL
        asyncio.run(initialize_server(args.sandbox_url))
        
        if args.transport == "stdio":
            logger.info("Using stdio transport")
            # FastMCP stdio transport
            mcp.run(transport="stdio")
            
        elif args.transport == "sse":
            logger.info(f"Using SSE transport on http://{host}:{port}{path}")
            # Create SSE app with CORS support
            from starlette.applications import Starlette
            from starlette.responses import JSONResponse
            from starlette.routing import Route, Mount
            from datetime import datetime
            
            # 创建健康检查端点
            async def health_check_endpoint(request):
                return JSONResponse({
                    "status": "healthy",
                    "service": "fc-code-interpreter-mcp-server",
                    "version": "0.0.10",
                    "timestamp": datetime.now().isoformat(),
                    "transport": "sse",
                    "sandbox_url": configured_sandbox_url,
                    "sandbox_available": e2b_sandbox is not None
                })
            
            # 创建 SSE 应用
            sse_app = create_sse_app(
                server=mcp,
                message_path=f"{path}/message",
                sse_path=path,
                middleware=[
                    Middleware(
                        CORSMiddleware,
                        allow_origins=["*"],
                        allow_credentials=True,
                        allow_methods=["GET", "POST", "OPTIONS"],
                        allow_headers=["*"],
                    )
                ]
            )
            
            # 包装应用添加健康检查端点
            wrapped_app = Starlette(
                routes=[
                    Route("/health_check", health_check_endpoint, methods=["GET"]),
                    Mount("/", app=sse_app),
                ]
            )
            
            # Run the SSE app
            import uvicorn
            uvicorn.run(wrapped_app, host=host, port=port)
            
        elif args.transport == "http":
            logger.info(f"Using HTTP transport on http://{host}:{port}{path}")
            # FastMCP HTTP transport with CORS support
            from fastmcp.server.http import create_streamable_http_app
            from starlette.applications import Starlette
            from starlette.responses import JSONResponse
            from starlette.routing import Route, Mount
            from datetime import datetime
            
            # 创建健康检查端点
            async def health_check_endpoint(request):
                return JSONResponse({
                    "status": "healthy",
                    "service": "fc-code-interpreter-mcp-server",
                    "version": "0.0.10",
                    "timestamp": datetime.now().isoformat(),
                    "transport": "http",
                    "sandbox_url": configured_sandbox_url,
                    "sandbox_available": e2b_sandbox is not None
                })
            
            # 创建 MCP HTTP 应用
            http_app = create_streamable_http_app(
                server=mcp,
                streamable_http_path=path,
                middleware=[
                    Middleware(
                        CORSMiddleware,
                        allow_origins=["*"],
                        allow_credentials=True,
                        allow_methods=["GET", "POST", "OPTIONS"],
                        allow_headers=["*"],
                    )
                ]
            )
            
            # 包装应用添加健康检查端点
            # 重要：必须传递 http_app 的 lifespan 以初始化 task group
            wrapped_app = Starlette(
                routes=[
                    Route("/health_check", health_check_endpoint, methods=["GET"]),
                    Mount("/", app=http_app),
                ],
                lifespan=http_app.lifespan  # 传递 FastMCP 的 lifespan
            )
            
            # Run the HTTP app
            import uvicorn
            uvicorn.run(wrapped_app, host=host, port=port)
            
    except KeyboardInterrupt:
        logger.info("Server shutdown requested")
    except Exception as e:
        logger.error(f"Server error: {e}")
        sys.exit(1)

if __name__ == "__main__":
    main()
