"use strict";
// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
Object.defineProperty(exports, "__esModule", { value: true });
exports.updateSession = exports.startSession = exports.getSessionModel = exports.shutdownSession = exports.getSessionUrl = exports.listRunning = exports.SESSION_SERVICE_URL = void 0;
const serverconnection_1 = require("../serverconnection");
const coreutils_1 = require("@jupyterlab/coreutils");
const validate_1 = require("./validate");
/**
 * The url for the session service.
 */
exports.SESSION_SERVICE_URL = 'api/sessions';
/**
 * List the running sessions.
 */
async function listRunning(settings = serverconnection_1.ServerConnection.makeSettings()) {
    const url = coreutils_1.URLExt.join(settings.baseUrl, exports.SESSION_SERVICE_URL);
    const response = await serverconnection_1.ServerConnection.makeRequest(url, {}, settings);
    if (response.status !== 200) {
        const err = await serverconnection_1.ServerConnection.ResponseError.create(response);
        throw err;
    }
    const data = await response.json();
    if (!Array.isArray(data)) {
        throw new Error('Invalid Session list');
    }
    data.forEach(m => {
        validate_1.updateLegacySessionModel(m);
        validate_1.validateModel(m);
    });
    return data;
}
exports.listRunning = listRunning;
/**
 * Get a session url.
 */
function getSessionUrl(baseUrl, id) {
    return coreutils_1.URLExt.join(baseUrl, exports.SESSION_SERVICE_URL, id);
}
exports.getSessionUrl = getSessionUrl;
/**
 * Shut down a session by id.
 */
async function shutdownSession(id, settings = serverconnection_1.ServerConnection.makeSettings()) {
    var _a;
    const url = getSessionUrl(settings.baseUrl, id);
    const init = { method: 'DELETE' };
    const response = await serverconnection_1.ServerConnection.makeRequest(url, init, settings);
    if (response.status === 404) {
        const data = await response.json();
        const msg = (_a = data.message) !== null && _a !== void 0 ? _a : `The session "${id}"" does not exist on the server`;
        console.warn(msg);
    }
    else if (response.status === 410) {
        throw new serverconnection_1.ServerConnection.ResponseError(response, 'The kernel was deleted but the session was not');
    }
    else if (response.status !== 204) {
        const err = await serverconnection_1.ServerConnection.ResponseError.create(response);
        throw err;
    }
}
exports.shutdownSession = shutdownSession;
/**
 * Get a full session model from the server by session id string.
 */
async function getSessionModel(id, settings = serverconnection_1.ServerConnection.makeSettings()) {
    const url = getSessionUrl(settings.baseUrl, id);
    const response = await serverconnection_1.ServerConnection.makeRequest(url, {}, settings);
    if (response.status !== 200) {
        const err = await serverconnection_1.ServerConnection.ResponseError.create(response);
        throw err;
    }
    const data = await response.json();
    validate_1.updateLegacySessionModel(data);
    validate_1.validateModel(data);
    return data;
}
exports.getSessionModel = getSessionModel;
/**
 * Create a new session, or return an existing session if the session path
 * already exists.
 */
async function startSession(options, settings = serverconnection_1.ServerConnection.makeSettings()) {
    const url = coreutils_1.URLExt.join(settings.baseUrl, exports.SESSION_SERVICE_URL);
    const init = {
        method: 'POST',
        body: JSON.stringify(options)
    };
    const response = await serverconnection_1.ServerConnection.makeRequest(url, init, settings);
    if (response.status !== 201) {
        const err = await serverconnection_1.ServerConnection.ResponseError.create(response);
        throw err;
    }
    const data = await response.json();
    validate_1.updateLegacySessionModel(data);
    validate_1.validateModel(data);
    return data;
}
exports.startSession = startSession;
/**
 * Send a PATCH to the server, updating the session path or the kernel.
 */
async function updateSession(model, settings = serverconnection_1.ServerConnection.makeSettings()) {
    const url = getSessionUrl(settings.baseUrl, model.id);
    const init = {
        method: 'PATCH',
        body: JSON.stringify(model)
    };
    const response = await serverconnection_1.ServerConnection.makeRequest(url, init, settings);
    if (response.status !== 200) {
        const err = await serverconnection_1.ServerConnection.ResponseError.create(response);
        throw err;
    }
    const data = await response.json();
    validate_1.updateLegacySessionModel(data);
    validate_1.validateModel(data);
    return data;
}
exports.updateSession = updateSession;
//# sourceMappingURL=restapi.js.map