import dataclasses
import io
import os
from decimal import Decimal
from typing import Any

from . import builder, util
from .builder import EndpointEmitType, EndpointSpecificPath
from .config import PythonConfig

INDENT = "    "
LINE_BREAK = "\n"
MODIFY_NOTICE = "# DO NOT MODIFY -- This file is generated by type_spec\n"
# Turn excess line length warning and turn off ruff formatting
LINT_HEADER = "# ruff: noqa: E402 Q003\n# fmt: off\n# isort: skip_file\n"
LINT_FOOTER = "# fmt: on\n"
ROUTE_NOTICE = """# Routes are generated from $endpoint specifications in the
# type_spec API YAML files. Refer to the section on endpoints in the type_spec/README"""
START_ALL_EXPORTS = """

__all__: list[str] = [
"""
END_ALL_EXPORTS = "]\n"

ASYNC_BATCH_TYPE_NAMESPACE = builder.SpecNamespace(name="async_batch")
ASYNC_BATCH_REQUEST_PATH_STYPE = builder.SpecTypeDefnStringEnum(
    namespace=ASYNC_BATCH_TYPE_NAMESPACE, name="AsyncBatchRequestPath"
)
ASYNC_BATCH_REQUEST_STYPE = builder.SpecTypeDefnObject(
    namespace=ASYNC_BATCH_TYPE_NAMESPACE, name="AsyncBatchRequest"
)
QUEUED_BATCH_REQUEST_STYPE = builder.SpecTypeDefnObject(
    namespace=ASYNC_BATCH_TYPE_NAMESPACE, name="QueuedAsyncBatchRequest"
)


@dataclasses.dataclass(kw_only=True)
class TrackingContext:
    namespace: builder.SpecNamespace | None = None
    namespaces: set[builder.SpecNamespace] = dataclasses.field(default_factory=set)
    names: set[str] = dataclasses.field(default_factory=set)

    use_enum: bool = False
    use_serial_string_enum: bool = False
    use_dataclass: bool = False
    use_serial_union: bool = False
    use_serial_alias: bool = False
    use_missing: bool = False
    use_opaque_key: bool = False


@dataclasses.dataclass(kw_only=True)
class Context(TrackingContext):
    out: io.StringIO
    namespace: builder.SpecNamespace
    builder: builder.SpecBuilder


def _resolve_namespace_name(namespace: builder.SpecNamespace) -> str:
    if len(namespace.path) > 1:
        return namespace.name
    return f"{namespace.name}_t"


def _resolve_namespace_ref(namespace: builder.SpecNamespace) -> str:
    return f"{namespace.name}_t"


def _resolve_enum_name(name: str, name_case: builder.NameCase) -> str:
    if name_case == builder.NameCase.preserve:
        return name
    return name.upper()


def _check_type_match(stype: builder.SpecType, value: Any) -> bool:
    literal = builder.unwrap_literal_type(stype)
    if literal is not None:
        return bool(literal.value == value)

    if stype.is_base_type(builder.BaseTypeName.s_string):
        return isinstance(value, str)
    elif stype.is_base_type(builder.BaseTypeName.s_integer):
        return isinstance(value, int)
    elif stype.is_base_type(builder.BaseTypeName.s_boolean):
        return isinstance(value, bool)
    elif stype.is_base_type(builder.BaseTypeName.s_decimal) or stype.is_base_type(
        builder.BaseTypeName.s_lossy_decimal
    ):
        return isinstance(value, (Decimal, int))
    elif isinstance(stype, builder.SpecTypeInstance):
        if stype.defn_type.is_base_type(builder.BaseTypeName.s_list):
            return isinstance(value, list) and all(
                _check_type_match(stype.parameters[0], x) for x in value
            )

        if stype.defn_type.is_base_type(builder.BaseTypeName.s_dict):
            key_type = stype.parameters[0]
            value_type = stype.parameters[1]
            return isinstance(value, dict) and all(
                _check_type_match(key_type, k) and _check_type_match(value_type, v)
                for k, v in value.items()
            )

        if stype.defn_type.is_base_type(builder.BaseTypeName.s_optional):
            return value is None or _check_type_match(stype.parameters[0], value)

        if stype.defn_type.is_base_type(builder.BaseTypeName.s_union):
            return any(
                _check_type_match(sub_type, value) for sub_type in stype.parameters
            )

    elif isinstance(stype, builder.SpecTypeDefnStringEnum):
        return value in stype.values

    raise Exception("invalid type", stype, value)


def _emit_value(ctx: TrackingContext, stype: builder.SpecType, value: Any) -> str:
    literal = builder.unwrap_literal_type(stype)
    if literal is not None:
        return _emit_value(ctx, literal.value_type, literal.value)

    if stype.is_base_type(builder.BaseTypeName.s_string):
        assert isinstance(value, str)
        return util.encode_common_string(value)
    elif stype.is_base_type(builder.BaseTypeName.s_integer):
        assert isinstance(value, int)
        return str(value)
    elif stype.is_base_type(builder.BaseTypeName.s_boolean):
        assert isinstance(value, bool)
        return "True" if value else "False"
    elif stype.is_base_type(builder.BaseTypeName.s_decimal) or stype.is_base_type(
        builder.BaseTypeName.s_lossy_decimal
    ):
        # Note that decimal requires the `!decimal 123.12` style notation in the YAML
        # file since PyYaml parses numbers as float, unfortuantely
        assert isinstance(value, (Decimal, int))
        if isinstance(value, int):
            # skip quotes for integers
            return f"Decimal({value})"
        return f'Decimal("{value}")'
    elif isinstance(stype, builder.SpecTypeInstance):
        if stype.defn_type.is_base_type(builder.BaseTypeName.s_list):
            # assume correct types here since builder checks them
            sub_type = stype.parameters[0]
            return "[" + ", ".join([_emit_value(ctx, sub_type, x) for x in value]) + "]"

        if stype.defn_type.is_base_type(builder.BaseTypeName.s_dict):
            key_type = stype.parameters[0]
            value_type = stype.parameters[1]
            return (
                "{\n    "
                + ",\n    ".join(
                    _emit_value(ctx, key_type, dkey)
                    + ": "
                    + _emit_value(ctx, value_type, dvalue)
                    for dkey, dvalue in value.items()
                )
                + "\n}"
            )

        if stype.defn_type.is_base_type(builder.BaseTypeName.s_optional):
            sub_type = stype.parameters[0]
            if value is None:
                return "None"
            return _emit_value(ctx, sub_type, value)

        if stype.defn_type.is_base_type(builder.BaseTypeName.s_union):
            matching_sub_types = [
                sub_type
                for sub_type in stype.parameters
                if _check_type_match(sub_type, value)
            ]
            if len(matching_sub_types) == 0:
                raise Exception("invalid union type for constant", value, stype)
            elif len(matching_sub_types) > 1:
                raise Exception("ambiguous union type for constant", value, stype)
            else:
                return _emit_value(ctx, matching_sub_types[0], value)

    elif isinstance(stype, builder.SpecTypeDefnStringEnum):
        return f"{refer_to(ctx, stype)}.{_resolve_enum_name(value, stype.name_case)}"
    elif isinstance(stype, builder.SpecTypeDefnAlias):
        return _emit_value(ctx, stype.alias, value)

    raise Exception("invalid constant type", value, stype)


def python_class_name(name: str) -> str:
    return "".join([s.capitalize() for s in util.split_any_name(name)])


# This list is incomplete. Items should be added as they cause a problem.
RESERVED_NAMES = ["for", "in", "object", "list", "if", "else", "elif"]


def python_field_name(name: str, name_case: builder.NameCase) -> str:
    if name_case != builder.NameCase.preserve:
        name = "_".join(util.split_any_name(name))

    if name in RESERVED_NAMES:
        return f"{name}_"
    return name


def emit_python(builder: builder.SpecBuilder, *, config: PythonConfig) -> None:
    _emit_types(builder=builder, config=config)
    _emit_routes(builder=builder, config=config)
    _emit_id_source(builder=builder, config=config)
    _emit_api_stubs(builder=builder, config=config)
    _emit_api_argument_lookup(builder=builder, config=config)
    _emit_client_class(spec_builder=builder, config=config)
    _emit_async_batch_processor(spec_builder=builder, config=config)


def _emit_types_imports(*, out: io.StringIO, ctx: Context) -> None:
    out.write("from __future__ import annotations\n")  # Will be default in Python 3.10
    out.write("import typing  # noqa: F401\n")
    out.write("import datetime  # noqa: F401\n")
    out.write("from decimal import Decimal  # noqa: F401\n")
    if ctx.use_enum:
        out.write("from enum import StrEnum\n")
    if ctx.use_dataclass:
        out.write("import dataclasses\n")
        out.write("from pkgs.serialization import serial_class\n")
    if ctx.use_serial_union:
        out.write("from pkgs.serialization import serial_union_annotation\n")
    if ctx.use_serial_alias:
        out.write("from pkgs.serialization import serial_alias_annotation\n")
    if ctx.use_serial_string_enum:
        out.write("from pkgs.serialization import serial_string_enum\n")
    if ctx.use_missing:
        out.write("from pkgs.serialization import MissingType, MISSING_SENTRY\n")
    if ctx.use_opaque_key:
        out.write("from pkgs.serialization import OpaqueKey\n")


def _emit_types(*, builder: builder.SpecBuilder, config: PythonConfig) -> None:
    index_out = io.StringIO()
    index_out.write(MODIFY_NOTICE)
    index_out.write(LINT_HEADER)

    exports_out = io.StringIO()
    exports_out.write(START_ALL_EXPORTS)

    all_dirs: set[str] = set()

    for namespace in sorted(
        builder.namespaces.values(),
        key=lambda ns: _resolve_namespace_name(ns),
    ):
        if (
            namespace.endpoint is not None
            and namespace.endpoint.is_sdk == EndpointEmitType.EMIT_NOTHING
            and config.sdk_endpoints_only is True
        ):
            continue

        ctx = Context(
            out=io.StringIO(),
            namespace=namespace,
            builder=builder,
        )

        _emit_namespace(ctx, namespace)

        full = io.StringIO()
        full.write(MODIFY_NOTICE)
        full.write(LINT_HEADER)
        _emit_types_imports(out=full, ctx=ctx)
        _emit_namespace_imports(
            out=full, namespaces=ctx.namespaces, from_namespace=namespace, config=config
        )

        # UNC: private types for now
        full.write(
            "\n__all__: list[str] = [\n"
            + "".join([f'{INDENT}"{name}",\n' for name in sorted(ctx.names)])
            + "]\n"
        )

        prepart = builder.preparts["python"].get(namespace.name)
        part = builder.parts["python"].get(namespace.name)

        if prepart:
            full.write(MODIFY_NOTICE)
            full.write(f"# === START section from {namespace.name}.prepart.py ===\n")
            full.write(prepart)
            full.write(f"# === END section from {namespace.name}.prepart.py ===\n")

        full.write(ctx.out.getvalue())
        full.write(MODIFY_NOTICE)

        if part:
            full.write(MODIFY_NOTICE)
            full.write(f"# === START section from {namespace.name}.part.py ===\n\n\n")
            full.write(part)
            full.write(f"# === END section from {namespace.name}.part.py ===\n")

        basename = "/".join(namespace.path)
        filename = f"{config.types_output}/{basename}{'' if len(namespace.path) > 1 else '_t'}.py"
        util.rewrite_file(filename, full.getvalue())

        # Deprecated SDK support
        if config.all_named_type_exports and len(namespace.path) == 1:
            compat_out = _create_sdk_compat_namespace(namespace)
            compat_filename = f"{config.types_output}/{basename}.py"
            util.rewrite_file(compat_filename, compat_out.getvalue())

        path_to = os.path.dirname(basename)
        while path_to != "":
            all_dirs.add(path_to)
            path_to = os.path.dirname(path_to)

        # Deprecated SDK support
        if config.all_named_type_exports:
            index_out.write("from ")
            if len(namespace.path) == 1:
                index_out.write(".")
            for path_part in namespace.path[:-1]:
                index_out.write(f".{path_part}")
            index_out.write(
                f" import {_resolve_namespace_name(namespace)} as {_resolve_namespace_ref(namespace)}\n"
            )  # noqa: E501

            exports_out.write(f'{INDENT}"{_resolve_namespace_ref(namespace)}",\n')

    exports_out.write(END_ALL_EXPORTS)

    if config.all_named_type_exports:
        index_out.write(exports_out.getvalue())

    util.rewrite_file(f"{config.types_output}/__init__.py", index_out.getvalue())

    # Emit empty init's for API folders
    for dir_name in all_dirs:
        if dir_name.startswith("api/") or dir_name == "api":
            util.rewrite_file(
                f"{config.types_output}/{dir_name}/__init__.py", MODIFY_NOTICE
            )


ENDPOINT_METHOD = "ENDPOINT_METHOD"
ENDPOINT_PATH = "ENDPOINT_PATH"
# will be removed in Q1 2025 when ENDPOINT_PATH is made api_endpoint-agnostic
# is used when the API call has multiple endpoints for the one endpoint that isn't equal to the top_namespace
ENDPOINT_PATH_ALTERNATE = "ENDPOINT_PATH_ALTERNATE"


def _get_epf_root(endpoint_specific_path: EndpointSpecificPath) -> str:
    return endpoint_specific_path.root


def _emit_namespace(ctx: Context, namespace: builder.SpecNamespace) -> None:
    endpoint = namespace.endpoint
    if endpoint is not None:
        ctx.out.write("\n")
        ctx.out.write(f'{ENDPOINT_METHOD} = "{endpoint.method.upper()}"\n')
        for endpoint_specific_path in sorted(
            endpoint.path_per_api_endpoint.values(), key=_get_epf_root
        ):
            endpoint_path_name = ENDPOINT_PATH

            if (
                len(endpoint.path_per_api_endpoint.keys()) > 1
                and endpoint_specific_path.root != ctx.builder.top_namespace
            ):
                endpoint_path_name = ENDPOINT_PATH_ALTERNATE
                ctx.names.add(ENDPOINT_PATH_ALTERNATE)
            ctx.out.write(
                f'{endpoint_path_name} = "{endpoint_specific_path.path_root}/{endpoint_specific_path.path_dirname}/{endpoint_specific_path.path_basename}"\n'
            )

        ctx.names.add(ENDPOINT_METHOD)
        ctx.names.add(ENDPOINT_PATH)

    for stype in namespace.types.values():
        _emit_type(ctx, stype)

    for sconst in namespace.constants.values():
        _emit_constant(ctx, sconst)


def _create_sdk_compat_namespace(namespace: builder.SpecNamespace) -> io.StringIO:
    compat_out = io.StringIO()
    compat_out.write(LINT_HEADER)
    compat_out.write(MODIFY_NOTICE)
    compat_out.write("# Kept only for SDK backwards compatibility\n")

    # This mostly an prepart import, thus has no high-level knowledge. Since this onl
    # needs backwards copmatibiltiy from when written we can just hardcode what was there
    # (only those in __all__ assuming that worked, which it might not)
    if namespace.name == "base":
        compat_out.write("""
from .base_t import JsonScalar as JsonScalar
from .base_t import JsonValue as JsonValue
from .base_t import ObjectId as ObjectId
""")
    else:
        for stype in namespace.types.values():
            compat_out.write(
                f"from .{namespace.path[-1]}_t import {stype.name} as {stype.name}\n"
            )

    compat_out.write(MODIFY_NOTICE)

    return compat_out


def _validate_supports_handler_generation(
    stype: builder.SpecTypeDefn, name: str, supports_inheritance: bool = False
) -> builder.SpecTypeDefnObject:
    assert isinstance(stype, builder.SpecTypeDefnObject), (
        f"External api {name} must be an object"
    )
    if not supports_inheritance:
        assert stype.base is None or stype.base.is_base, (
            f"Inheritance not supported in external api {name}"
        )
    return stype


def _emit_endpoint_invocation_docstring(
    ctx: Context,
    endpoint: builder.SpecEndpoint,
    properties: list[builder.SpecProperty],
) -> None:
    has_argument_desc = any(prop.desc is not None for prop in properties)
    has_endpoint_desc = endpoint.desc
    if not has_argument_desc and not has_endpoint_desc:
        return

    FULL_INDENT = INDENT * 2
    ctx.out.write(FULL_INDENT)
    ctx.out.write('"""')

    if endpoint.desc is not None and has_endpoint_desc:
        ctx.out.write(f"{endpoint.desc}\n")
        ctx.out.write("\n")

    if has_argument_desc:
        for prop in properties:
            if prop.desc:
                ctx.out.write(f"{FULL_INDENT}:param {prop.name}: {prop.desc}\n")

    ctx.out.write(f'{FULL_INDENT}"""\n')


def _emit_endpoint_invocation_function_signature(
    ctx: Context,
    endpoint: builder.SpecEndpoint,
    arguments_type: builder.SpecTypeDefnObject,
    data_type: builder.SpecTypeDefnObject,
    extra_params: list[builder.SpecProperty] | None = None,
) -> None:
    all_arguments = (
        list(arguments_type.properties.values())
        if arguments_type.properties is not None
        else []
    ) + (extra_params if extra_params is not None else [])

    # All endpoints share a function name
    function = endpoint.path_per_api_endpoint[endpoint.default_endpoint_key].function
    assert function is not None
    function_name = function.split(".")[-1]
    ctx.out.write(
        f"""
    def {function_name}(
        self,\n"""
    )
    if len(all_arguments) > 0:
        ctx.out.write(f"{INDENT}{INDENT}*,\n")
        _emit_properties(
            ctx=ctx,
            properties=all_arguments,
            num_indent=2,
            separator=",\n",
            class_out=ctx.out,
        )
    ctx.out.write(f"{INDENT}) -> {refer_to(ctx=ctx, stype=data_type)}:")
    ctx.out.write("\n")

    if len(all_arguments) > 0:
        _emit_endpoint_invocation_docstring(
            ctx=ctx, endpoint=endpoint, properties=all_arguments
        )


def _emit_instantiate_type_from_locals(
    ctx: Context, variable_name: str, variable_type: builder.SpecTypeDefnObject
) -> None:
    ctx.out.write(
        f"{INDENT}{INDENT}{variable_name} = {refer_to(ctx=ctx, stype=variable_type)}("
    )
    if variable_type.properties is not None and len(variable_type.properties) > 0:
        ctx.out.write("\n")
        for prop in variable_type.properties.values():
            ctx.out.write(f"{INDENT}{INDENT}{INDENT}{prop.name}={prop.name},")
            ctx.out.write("\n")
        ctx.out.write(f"{INDENT}{INDENT})")
    else:
        ctx.out.write(")")


def _emit_async_batch_invocation_function(
    ctx: Context, namespace: builder.SpecNamespace
) -> None:
    endpoint = namespace.endpoint
    if endpoint is None:
        return
    if (
        endpoint.async_batch_path is None
        or endpoint.is_sdk != EndpointEmitType.EMIT_ENDPOINT
    ):
        return

    ctx.out.write("\n")
    arguments_type = namespace.types["Arguments"]
    arguments_type = _validate_supports_handler_generation(arguments_type, "arguments")
    data_type = QUEUED_BATCH_REQUEST_STYPE

    list_str_params: list[builder.SpecType] = []
    list_str_params.append(
        builder.SpecTypeGenericParameter(
            name="str",
            spec_type_definition=builder.SpecTypeDefnObject(
                namespace=namespace, name=builder.BaseTypeName.s_string, is_base=True
            ),
        )
    )

    _emit_endpoint_invocation_function_signature(
        ctx=ctx,
        endpoint=endpoint,
        arguments_type=arguments_type,
        data_type=data_type,
        extra_params=[
            builder.SpecProperty(
                name="depends_on",
                extant=builder.PropertyExtant.optional,
                convert_value=builder.PropertyConvertValue.auto,
                name_case=builder.NameCase.convert,
                label="depends_on",
                desc="A list of batch reference keys to process before processing this request",
                spec_type=builder.SpecTypeInstance(
                    defn_type=builder.SpecTypeDefnObject(
                        name=builder.BaseTypeName.s_list,
                        is_base=True,
                        namespace=namespace,
                    ),
                    parameters=list_str_params,
                ),
            )
        ],
    )

    # Emit function body
    _emit_instantiate_type_from_locals(
        ctx=ctx, variable_name="args", variable_type=arguments_type
    )

    path = _emit_value(
        ctx=ctx,
        stype=ASYNC_BATCH_REQUEST_PATH_STYPE,
        value=endpoint.async_batch_path,
    )

    ctx.out.write(
        f"""
        json_data = serialize_for_api(args)

        batch_reference = str(uuid.uuid4())

        req = {refer_to(ctx=ctx, stype=ASYNC_BATCH_REQUEST_STYPE)}(
            path={path},
            data=json_data,
            depends_on=depends_on,
            batch_reference=batch_reference,
        )

        self._enqueue(req)

        return {refer_to(ctx=ctx, stype=data_type)}(
            path=req.path,
            batch_reference=req.batch_reference,
        )"""
    )


def _emit_endpoint_invocation_function(
    ctx: Context, namespace: builder.SpecNamespace
) -> None:
    endpoint = namespace.endpoint
    if endpoint is None:
        return
    if endpoint.is_sdk != EndpointEmitType.EMIT_ENDPOINT or endpoint.is_beta:
        return

    ctx.out.write("\n")
    arguments_type = namespace.types["Arguments"]
    data_type = namespace.types["Data"]
    arguments_type = _validate_supports_handler_generation(arguments_type, "arguments")
    data_type = _validate_supports_handler_generation(
        data_type, "response", supports_inheritance=True
    )

    _emit_endpoint_invocation_function_signature(
        ctx=ctx, endpoint=endpoint, arguments_type=arguments_type, data_type=data_type
    )

    endpoint_method_stype = builder.SpecTypeDefnObject(
        namespace=arguments_type.namespace, name=ENDPOINT_METHOD
    )
    endpoint_path_stype = builder.SpecTypeDefnObject(
        namespace=arguments_type.namespace, name=ENDPOINT_PATH
    )

    # Emit function body
    _emit_instantiate_type_from_locals(
        ctx=ctx, variable_name="args", variable_type=arguments_type
    )

    ctx.out.write(
        f"""
        api_request = APIRequest(
            method={refer_to(ctx=ctx, stype=endpoint_method_stype)},
            endpoint={refer_to(ctx=ctx, stype=endpoint_path_stype)},
            args=args,
        )
        return self.do_request(api_request=api_request, return_type={refer_to(ctx=ctx, stype=data_type)})"""
    )


def _emit_string_enum(ctx: Context, stype: builder.SpecTypeDefnStringEnum) -> None:
    ctx.use_enum = True

    # Detect if we need serial_string_enum. Ideally we can just stick this on all enums
    # we generate, but let's limit it for initial testing
    need_labels = any(entry.label is not None for entry in stype.values.values())
    need_deprecated = any(entry.deprecated for entry in stype.values.values())

    if need_labels or need_deprecated:
        ctx.use_serial_string_enum = True
        ctx.out.write("@serial_string_enum(\n")
        if need_labels:
            ctx.out.write(f"{INDENT}labels={{\n")
            for entry in stype.values.values():
                if entry.label is not None:
                    ctx.out.write(
                        f'{INDENT}{INDENT}"{entry.value}": "{entry.label}",\n'
                    )

            ctx.out.write(f"{INDENT}}},\n")
        if need_deprecated:
            ctx.out.write(f"{INDENT}deprecated={{\n")
            for entry in stype.values.values():
                if entry.deprecated is True:
                    ctx.out.write(f'{INDENT}{INDENT}"{entry.value}",\n')

            ctx.out.write(f"{INDENT}}}\n")
        ctx.out.write(")\n")

    ctx.out.write(f"class {python_class_name(stype.name)}(StrEnum):\n")

    # IMPROVE: Move this into the serial_string_enum wrapper
    if stype.sql_type_name is not None:
        ctx.out.write(
            "\n".join((
                f"{INDENT}@staticmethod",
                f"{INDENT}def _sql_type() -> str:",
                f'{INDENT}{INDENT}return "{stype.sql_type_name}"\n\n',
            ))
        )

    assert stype.values
    for name, entry in stype.values.items():
        ctx.out.write(
            f'{INDENT}{_resolve_enum_name(name, stype.name_case)} = "{entry.value}"\n'
        )


@dataclasses.dataclass
class EmittedPropertiesMetadata:
    unconverted_keys: set[str]
    unconverted_values: set[str]
    to_string_values: set[str]
    parse_require: set[str]


def _emit_type_properties(
    *,
    ctx: Context,
    class_out: io.StringIO,
    stype: builder.SpecTypeDefnObject,
    num_indent: int = 1,
    separator: str = "\n",
) -> EmittedPropertiesMetadata:
    return _emit_properties(
        ctx=ctx,
        class_out=class_out,
        properties=list((stype.properties or {}).values()),
        num_indent=num_indent,
        separator=separator,
    )


def _emit_properties(
    *,
    ctx: Context,
    class_out: io.StringIO,
    properties: list[builder.SpecProperty],
    num_indent: int = 1,
    separator: str = "\n",
) -> EmittedPropertiesMetadata:
    unconverted_keys: set[str] = set()
    unconverted_values: set[str] = set()
    to_string_values: set[str] = set()
    parse_require: set[str] = set()

    if len(properties) > 0:

        def write_field(prop: builder.SpecProperty) -> None:
            stype = prop.spec_type
            if prop.name_case == builder.NameCase.preserve:
                unconverted_keys.add(prop.name)
            py_name = python_field_name(prop.name, prop.name_case)

            if prop.convert_value == builder.PropertyConvertValue.no_convert:
                unconverted_values.add(py_name)
            elif not stype.is_value_converted():
                assert prop.convert_value == builder.PropertyConvertValue.auto
                unconverted_values.add(py_name)
            if stype.is_value_to_string():
                to_string_values.add(py_name)

            if prop.parse_require:
                parse_require.add(py_name)

            ref_type = refer_to(ctx, stype)
            default = None
            if prop.extant == builder.PropertyExtant.missing:
                ref_type = f"MissingType[{ref_type}]"
                default = "MISSING_SENTRY"
                ctx.use_missing = True
            elif prop.extant == builder.PropertyExtant.optional:
                if isinstance(
                    stype, builder.SpecTypeInstance
                ) and stype.defn_type.is_base_type(builder.BaseTypeName.s_optional):
                    pass  # base type already adds the None union
                elif ref_type == "None":
                    pass  # no need to add a None union to a none type
                else:
                    ref_type = f"{ref_type} | None"
                default = "None"
            elif prop.has_default:
                default = _emit_value(ctx, stype, prop.default)
                if (
                    isinstance(stype, builder.SpecTypeInstance)
                    and (stype.defn_type.is_base_type(builder.BaseTypeName.s_list))
                    and default == "[]"
                ):
                    default = "dataclasses.field(default_factory=list)"
            class_out.write(f"{INDENT * num_indent}{py_name}: {ref_type}")
            if default:
                class_out.write(f" = {default}")
            class_out.write(separator)

        for prop in properties:
            if prop.extant == builder.PropertyExtant.required:
                write_field(prop)
        for prop in properties:
            if prop.extant != builder.PropertyExtant.required:
                write_field(prop)
    else:
        class_out.write(f"{INDENT * num_indent}pass\n")

    return EmittedPropertiesMetadata(
        unconverted_keys=unconverted_keys,
        unconverted_values=unconverted_values,
        to_string_values=to_string_values,
        parse_require=parse_require,
    )


def _named_type_path(ctx: Context, stype: builder.SpecTypeDefn) -> str:
    parts = [] if stype.is_base else stype.namespace.path.copy()
    parts.append(stype.name)
    return f"{ctx.builder.top_namespace}.{'.'.join(parts)}"


def _emit_type(ctx: Context, stype: builder.SpecType) -> None:
    if not isinstance(stype, builder.SpecTypeDefn):
        return

    if stype.is_exported:
        ctx.names.add(stype.name)

    if stype.is_base or stype.is_predefined:
        return

    ctx.out.write("\n\n")
    # Repeat this often so that it's noticed
    ctx.out.write(MODIFY_NOTICE)

    if isinstance(stype, builder.SpecTypeDefnExternal):
        ctx.out.write(stype.external_map["py"])
        ctx.out.write("\n")
        return

    if isinstance(stype, builder.SpecTypeDefnAlias):
        ctx.use_serial_alias = True
        ctx.out.write(f"{stype.name} = typing.Annotated[\n")
        ctx.out.write(f"{INDENT}{refer_to(ctx, stype.alias)},\n")
        ctx.out.write(f"{INDENT}serial_alias_annotation(\n")
        ctx.out.write(
            f"{INDENT}named_type_path={util.encode_common_string(_named_type_path(ctx, stype))},\n"
        )
        if stype.is_dynamic_allowed():
            ctx.out.write(f"{INDENT}is_dynamic_allowed=True,\n")
        ctx.out.write(f"{INDENT}),\n")
        ctx.out.write("]\n")
        return

    if isinstance(stype, builder.SpecTypeDefnUnion):
        ctx.use_serial_union = True
        ctx.out.write(f"{stype.name} = typing.Annotated[\n")
        ctx.out.write(f"{INDENT}{refer_to(ctx, stype.get_backing_type())},\n")
        ctx.out.write(f"{INDENT}serial_union_annotation(\n")
        ctx.out.write(
            f"{INDENT}named_type_path={util.encode_common_string(_named_type_path(ctx, stype))},\n"
        )
        if stype.is_dynamic_allowed():
            ctx.out.write(f"{INDENT}is_dynamic_allowed=True,\n")
        if stype.discriminator is not None:
            ctx.out.write(
                f"{INDENT * 2}discriminator={util.encode_common_string(stype.discriminator)},\n"
            )
        if stype.discriminator_map is not None:
            ctx.out.write(f"{INDENT * 2}discriminator_map={{\n")
            for key, value in stype.discriminator_map.items():
                ctx.out.write(
                    f"{INDENT * 3}{util.encode_common_string(key)}: {refer_to(ctx, value)},\n"
                )
            ctx.out.write(f"{INDENT * 2}}},\n")
        ctx.out.write(f"{INDENT}),\n")
        ctx.out.write("]\n")
        return

    if isinstance(stype, builder.SpecTypeDefnStringEnum):
        return _emit_string_enum(ctx, stype)

    assert isinstance(stype, builder.SpecTypeDefnObject)
    assert stype.base is not None
    ctx.use_dataclass = True

    class_out = io.StringIO()
    base_class = ""
    generics = stype.get_generics()
    if not stype.base.is_base:
        base_class = f"({refer_to(ctx, stype.base)})"
    elif len(generics) > 0:
        base_class = f"[{', '.join(generics)}]"
    class_out.write(f"class {stype.name}{base_class}:\n")

    emitted_properties_metadata = _emit_type_properties(
        ctx=ctx, class_out=class_out, stype=stype
    )
    unconverted_keys = emitted_properties_metadata.unconverted_keys
    unconverted_values = emitted_properties_metadata.unconverted_values
    to_string_values = emitted_properties_metadata.to_string_values
    parse_require = emitted_properties_metadata.parse_require

    _emit_generics(ctx, generics)

    # Emit serial_class decorator
    ctx.out.write("@serial_class(\n")
    ctx.out.write(
        f"{INDENT}named_type_path={util.encode_common_string(_named_type_path(ctx, stype))},\n"
    )
    if stype.is_dynamic_allowed():
        ctx.out.write(f"{INDENT}is_dynamic_allowed=True,\n")

    def write_values(key: str, values: set[str]) -> None:
        if len(values) == 0:
            return
        value_str = ", ".join([f'"{name}"' for name in sorted(values)])
        ctx.out.write(f"{INDENT}{key}={{{value_str}}},\n")

    write_values("unconverted_keys", unconverted_keys)
    write_values("unconverted_values", unconverted_values)
    write_values("to_string_values", to_string_values)
    write_values("parse_require", parse_require)

    ctx.out.write(")\n")

    # Emit dataclass decorator
    dataclass = "@dataclasses.dataclass"
    dc_args = []
    if stype.is_kw_only():
        dc_args.append("kw_only=True")
    if stype.is_hashable:
        dc_args.extend(["frozen=True", "eq=True"])
    if len(dc_args) > 0:
        dataclass += f"({', '.join(dc_args)})"

    ctx.out.write(f"{dataclass}\n")
    ctx.out.write(class_out.getvalue())


def _emit_generics(ctx: Context, generics: list[str]) -> None:
    for generic in generics:
        ctx.out.write(f'{generic} = typing.TypeVar("{generic}")\n')
        ctx.out.write(f"{LINE_BREAK}{LINE_BREAK}")


def _emit_constant(ctx: Context, sconst: builder.SpecConstant) -> None:
    ctx.out.write("\n\n")
    ctx.out.write(MODIFY_NOTICE)
    value = _emit_value(ctx, sconst.value_type, sconst.value)

    const_name = sconst.name.upper()
    ctx.names.add(const_name)
    ctx.out.write(f"{const_name} = {value}\n")


base_name_map = {
    builder.BaseTypeName.s_boolean: "bool",
    builder.BaseTypeName.s_date: "datetime.date",
    builder.BaseTypeName.s_date_time: "datetime.datetime",
    builder.BaseTypeName.s_decimal: "Decimal",
    builder.BaseTypeName.s_dict: "dict",
    builder.BaseTypeName.s_integer: "int",
    builder.BaseTypeName.s_json_value: "JsonValue",
    builder.BaseTypeName.s_list: "list",
    builder.BaseTypeName.s_lossy_decimal: "Decimal",
    builder.BaseTypeName.s_none: "None",
    builder.BaseTypeName.s_optional: "typing.Optional",
    builder.BaseTypeName.s_opaque_key: "OpaqueKey",
    builder.BaseTypeName.s_string: "str",
    builder.BaseTypeName.s_tuple: "tuple",
    builder.BaseTypeName.s_readonly_array: "tuple",
    builder.BaseTypeName.s_union: "typing.Union",
    builder.BaseTypeName.s_literal: "typing.Literal",
}


def refer_to(ctx: TrackingContext, stype: builder.SpecType) -> str:
    if isinstance(stype, builder.SpecTypeInstance):
        params = [refer_to(ctx, p) for p in stype.parameters]

        if stype.defn_type.is_base_type(builder.BaseTypeName.s_union):
            if len(stype.parameters) == 1:
                return f"typing.Union[{params[0]}]"
            return " | ".join(params)

        if stype.defn_type.is_base_type(builder.BaseTypeName.s_readonly_array):
            assert len(params) == 1, "Read Only Array takes one parameter"
            return f"tuple[{params[0]}, ...]"

        if stype.defn_type.is_base_type(builder.BaseTypeName.s_optional):
            assert len(params) == 1, "Optional only takes one parameter"
            return f"{params[0]} | None"

        return f"{refer_to(ctx, stype.defn_type)}[{', '.join(params)}]"

    if isinstance(stype, builder.SpecTypeLiteralWrapper):
        return _emit_value(ctx, stype.value_type, stype.value)

    if isinstance(stype, builder.SpecTypeGenericParameter):
        return stype.name

    assert isinstance(stype, builder.SpecTypeDefn)
    if stype.is_base:  # assume correct namespace
        if stype.name == builder.BaseTypeName.s_opaque_key:
            ctx.use_opaque_key = True
        return base_name_map[builder.BaseTypeName(stype.name)]

    if stype.namespace == ctx.namespace:
        return stype.name

    ctx.namespaces.add(stype.namespace)
    return f"{_resolve_namespace_ref(stype.namespace)}.{stype.name}"


# MYPY: There's a strange bug, where using builder.SpecEndpoint
# below for _emit_routes/endpoints causes MyPy to not recognize it as a type.
SpecEndpoint = builder.SpecEndpoint


def _emit_routes(*, builder: builder.SpecBuilder, config: PythonConfig) -> None:
    for endpoint_root in builder.api_endpoints:
        endpoints: list[SpecEndpoint] = []
        output = config.routes_output.get(endpoint_root)
        if output is None:
            continue
        last_endpoint: SpecEndpoint | None = None
        for namespace in builder.namespaces.values():
            endpoint = namespace.endpoint
            last_endpoint = endpoint
            if endpoint is None:
                continue
            if endpoint_root not in endpoint.path_per_api_endpoint:
                continue
            if endpoint.path_per_api_endpoint[endpoint_root].function is None:
                continue

            endpoints.append(endpoint)
        static_out = io.StringIO()
        static_out.write(
            f"""{MODIFY_NOTICE}{LINT_HEADER}{ROUTE_NOTICE}

# Static routes for use locally and to ensure linting/mypy finds all modules

from main.site.framework.types import StaticRouteType
"""
        )

        def _route_identifier(endpoint: SpecEndpoint) -> tuple[str, str, str]:
            endpoint_specific_path = endpoint.path_per_api_endpoint[endpoint_root]
            return (
                endpoint_specific_path.path_dirname,
                endpoint_specific_path.path_basename,
                endpoint.method,
            )

        sorted_endpoints = sorted(endpoints, key=_route_identifier)

        assert len(endpoints) == len(set(map(_route_identifier, endpoints))), (
            "Endpoints are not unique"
        )

        path_set = set()
        for endpoint in sorted_endpoints:
            last_endpoint = endpoint
            endpoint_function_path = endpoint.path_per_api_endpoint[endpoint_root]
            assert endpoint_function_path.function
            func_bits = endpoint_function_path.function.split(".")
            path = ".".join(func_bits[:-1])
            if path in path_set:
                continue
            path_set.add(path)
            static_out.write(f"import {path}\n")

        assert last_endpoint is not None

        static_out.write(
            f"""
ROUTE_PREFIX = "/{last_endpoint.path_per_api_endpoint[endpoint_root].path_root}"

ROUTES: list[StaticRouteType] = [
"""
        )

        dynamic_out = io.StringIO()
        dynamic_out.write(
            f"""{MODIFY_NOTICE}{LINT_HEADER}{ROUTE_NOTICE}

# Dynamic loaded routes to provide faster startup times

from main.site.framework.types import DynamicRouteType

ROUTE_PREFIX = "/{last_endpoint.path_per_api_endpoint[endpoint_root].path_root}"

ROUTES: list[DynamicRouteType] = [
"""
        )

        for endpoint in sorted_endpoints:
            endpoint_function_path = endpoint.path_per_api_endpoint[endpoint_root]
            dynamic_out.write(
                f'{INDENT}("{endpoint_function_path.path_dirname}/{endpoint_function_path.path_basename}", "{endpoint_function_path.function}", ["{endpoint.method.upper()}"]),\n'
            )

            assert endpoint_function_path.function
            static_out.write(
                f'{INDENT}("{endpoint_function_path.path_dirname}/{endpoint_function_path.path_basename}", {endpoint_function_path.function}, ["{endpoint.method.upper()}"]),\n'
            )

        dynamic_out.write(f"{MODIFY_NOTICE}]\n")
        static_out.write(f"{MODIFY_NOTICE}]\n")

        util.rewrite_file(f"{output}/routes_static.py", static_out.getvalue())
        util.rewrite_file(f"{output}/routes_dynamic.py", dynamic_out.getvalue())

        util.rewrite_file(f"{output}/__init__.py", MODIFY_NOTICE)


def _emit_namespace_imports(
    *,
    out: io.StringIO,
    namespaces: set[builder.SpecNamespace],
    from_namespace: builder.SpecNamespace | None,
    config: PythonConfig,
    skip_non_sdk: bool = False,
) -> None:
    for ns in sorted(
        namespaces,
        key=lambda name: _resolve_namespace_name(name),
    ):
        if (
            skip_non_sdk
            and ns.endpoint is not None
            and ns.endpoint.is_sdk != EndpointEmitType.EMIT_ENDPOINT
        ):
            continue
        resolved = _resolve_namespace_name(ns)
        if ns.endpoint is not None:
            import_alias = "_".join(ns.path[2:]) + "_t"
            out.write(
                f"import {config.types_package}.{'.'.join(ns.path)} as {import_alias}\n"
            )
            continue
        elif from_namespace is not None:
            from_path = "." * len(from_namespace.path)
        else:
            from_path = config.types_package

        out.write(f"from {from_path} import {resolved}\n")


def _emit_id_source(*, builder: builder.SpecBuilder, config: PythonConfig) -> None:
    """
    Create a map of full-type-name to enums for use in id-source.
    """
    id_source_output = config.id_source_output
    if id_source_output is None:
        return None
    enum_out = io.StringIO()
    enum_out.write(f"{LINT_HEADER}{MODIFY_NOTICE}\n")
    enum_out.write("import typing\n")
    enum_out.write("from enum import StrEnum\n")

    ctx = TrackingContext()
    # In this context the propername
    named_enums = {
        builder.resolve_proper_name(string_enum): refer_to(ctx, string_enum)
        for string_enum in builder.emit_id_source_enums
    }

    _emit_namespace_imports(
        out=enum_out, namespaces=ctx.namespaces, from_namespace=None, config=config
    )

    known_keys = []
    enum_out.write("\nENUM_MAP: dict[str, type[StrEnum]] = {\n")
    for key in sorted(named_enums.keys()):
        enum_out.write(f'{INDENT}"{key}": {named_enums[key]},\n')
        known_keys.append(f'"{key}"')
    enum_out.write(f"}}\n{MODIFY_NOTICE}\n")

    enum_out.write(f"\nKnownEnumsType = typing.Literal[\n{INDENT}")
    enum_out.write(f",\n{INDENT}".join(known_keys))
    enum_out.write(f"\n]\n{MODIFY_NOTICE}\n")

    enum_out.write(f'\n{MODIFY_NOTICE}\n__all__ = ["ENUM_MAP"]\n')
    enum_out.write(f"{MODIFY_NOTICE}\n{LINT_FOOTER}")

    util.rewrite_file(id_source_output, enum_out.getvalue())


def _emit_api_stubs(*, builder: builder.SpecBuilder, config: PythonConfig) -> None:
    for endpoint_root in builder.api_endpoints:
        for namespace in builder.namespaces.values():
            endpoint = namespace.endpoint

            if endpoint is None:
                continue
            if endpoint_root not in endpoint.path_per_api_endpoint:
                continue

            endpoint_function = endpoint.path_per_api_endpoint[endpoint_root].function
            if endpoint_function is None:
                continue

            module_dir, file_name, func_name = endpoint_function.rsplit(".", 2)
            module_path = os.path.abspath(module_dir.replace(".", "/"))
            api_stub_file = f"{module_path}/{file_name}.py"
            if os.path.isfile(api_stub_file):
                continue
            _create_api_stub(
                api_stub_file=api_stub_file,
                file_name=file_name,
                endpoint=endpoint,
                config=config,
                endpoint_root=endpoint_root,
                top_namespace=builder.top_namespace,
            )


def _create_api_stub(
    *,
    api_stub_file: str,
    file_name: str,
    endpoint: SpecEndpoint,
    config: PythonConfig,
    endpoint_root: str,
    top_namespace: str,
) -> None:
    assert (
        endpoint.method == builder.RouteMethod.post
        or endpoint.method == builder.RouteMethod.get
        or endpoint.method == builder.RouteMethod.delete
        or endpoint.method == builder.RouteMethod.patch
    )
    api_out = _create_api_function(
        file_name=file_name,
        endpoint=endpoint,
        config=config,
        endpoint_root=endpoint_root,
        top_namespace=top_namespace,
    )
    util.rewrite_file(api_stub_file, api_out.getvalue())


WRAP_ARGS_START = "\n    "
WRAP_ARGS_END = "\n"


def _create_api_function(
    *,
    file_name: str,
    endpoint: SpecEndpoint,
    config: PythonConfig,
    endpoint_root: str,
    top_namespace: str,
) -> io.StringIO:
    endpoint_specific_path = endpoint.path_per_api_endpoint[endpoint_root]
    assert endpoint_specific_path is not None
    api_out = io.StringIO()
    python_api_type_root = f"{config.types_package}.api"
    dot_dirname = endpoint_specific_path.path_dirname.replace("/", ".")
    api_import = (
        f"{python_api_type_root}.{dot_dirname}.{file_name}"
        if dot_dirname != ""
        else f"{python_api_type_root}.{endpoint_specific_path.path_basename}"
    )

    if endpoint.method == builder.RouteMethod.post:
        validated_method = "validated_post"
    elif endpoint.method == builder.RouteMethod.get:
        validated_method = "validated_get"
    elif endpoint.method == builder.RouteMethod.delete:
        validated_method = "validated_delete"
    elif endpoint.method == builder.RouteMethod.patch:
        validated_method = "validated_patch"

    ruff_requires_wrap = len(endpoint_specific_path.path_basename) > 14

    account_type = (
        endpoint_specific_path.root
        if endpoint_specific_path.root not in ["external", "portal"]
        else "materials"
    )

    endpoint_path_name = (
        ENDPOINT_PATH_ALTERNATE
        if len(endpoint.path_per_api_endpoint.keys()) > 1
        and endpoint_specific_path.root != top_namespace
        else ENDPOINT_PATH
    )

    api_out.write(
        f"""import {api_import} as api
from main.db.session import Session, SessionMaker
from main.site.decorators import APIError, APIResponse, {validated_method}


@{validated_method}(api.{endpoint_path_name}, "{account_type}", api.Arguments)
def {endpoint_specific_path.path_basename}({WRAP_ARGS_START if ruff_requires_wrap else ""}args: api.Arguments, client_sm: SessionMaker{WRAP_ARGS_END if ruff_requires_wrap else ""}) -> APIResponse[api.Data]:
    with Session(client_sm) as session:
        # return APIResponse(data=api.Data())
        pass
    raise APIError.unexpected_unimplemented()
"""
    )
    return api_out


API_ARGUMENTS_NAME = "api_arguments_lookup"


def _emit_api_argument_lookup(
    *, builder: builder.SpecBuilder, config: PythonConfig
) -> None:
    if not config.emit_api_argument_lookup:
        return

    for endpoint_root in builder.api_endpoints:
        routes_output = config.routes_output[endpoint_root]

        imports = ["import dataclasses"]
        mappings = []
        for namespace in sorted(
            builder.namespaces.values(),
            key=lambda ns: ns.path,
        ):
            endpoint = namespace.endpoint

            if endpoint is None:
                continue
            if endpoint_root not in endpoint.path_per_api_endpoint:
                continue
            if endpoint.path_per_api_endpoint[endpoint_root].function is None:
                continue
            if "Arguments" not in namespace.types or "Data" not in namespace.types:
                continue

            import_alias = "_".join(namespace.path[1:])
            api_import = f"{config.types_package}.{'.'.join(namespace.path)}"
            imports.append(f"import {api_import} as {import_alias}")

            route_group = (
                f'"{endpoint.route_group}"'
                if endpoint.route_group is not None
                else "None"
            )
            account_type = (
                f'"{endpoint.account_type}"'
                if endpoint.account_type is not None
                else "None"
            )

            mapping = f"{INDENT}ApiEndpointKey(route={import_alias}.ENDPOINT_PATH, method={import_alias}.ENDPOINT_METHOD): ApiEndpointSpec(\n"
            mapping += f"{INDENT}{INDENT}arguments_type={import_alias}.Arguments,\n"
            mapping += f"{INDENT}{INDENT}data_type={import_alias}.Data,\n"
            mapping += f"{INDENT}{INDENT}route_group={route_group},\n"
            mapping += f"{INDENT}{INDENT}account_type={account_type},\n"
            mapping += f"{INDENT}{INDENT}route={import_alias}.ENDPOINT_PATH,\n"
            mapping += f'{INDENT}{INDENT}handler="{endpoint.path_per_api_endpoint[endpoint_root].function}",\n'
            mapping += f"{INDENT}{INDENT}method={import_alias}.ENDPOINT_METHOD,\n"
            mapping += f"{INDENT})"
            mappings.append(mapping)

        argument_lookup_out = io.StringIO()
        argument_lookup_out.write(MODIFY_NOTICE)
        argument_lookup_out.write(LINT_HEADER)
        argument_lookup_out.write(
            f"""{LINE_BREAK.join(imports)}


@dataclasses.dataclass(kw_only=True, frozen=True)
class ApiEndpointKey:
    method: str
    route: str


@dataclasses.dataclass(kw_only=True)
class ApiEndpointSpec[AT, DT]:
    route: str
    arguments_type: type[AT]
    data_type: type[DT]
    route_group: str | None
    account_type: str | None
    handler: str
    method: str


{API_ARGUMENTS_NAME}: dict[ApiEndpointKey, ApiEndpointSpec] = {{
{f",{LINE_BREAK}".join(mappings)},
}}

__all__ = ["{API_ARGUMENTS_NAME}"]
"""
        )
        argument_lookup_out.write(LINT_FOOTER)

        util.rewrite_file(
            f"{routes_output}/{API_ARGUMENTS_NAME}.py", argument_lookup_out.getvalue()
        )


CLIENT_CLASS_FILENAME = "client_base"
CLIENT_CLASS_IMPORTS = [
    "from abc import ABC, abstractmethod",
    "import dataclasses",
]
ASYNC_BATCH_PROCESSOR_FILENAME = "async_batch_processor"
ASYNC_BATCH_PROCESSOR_IMPORTS = [
    "import uuid",
    "from abc import ABC, abstractmethod",
    "from pkgs.serialization_util import serialize_for_api",
]


def _emit_async_batch_processor(
    *, spec_builder: builder.SpecBuilder, config: PythonConfig
) -> None:
    if not config.emit_async_batch_processor:
        return

    async_batch_processor_out = io.StringIO()
    ctx = Context(
        out=io.StringIO(),
        namespace=builder.SpecNamespace("async_batch_processor"),
        builder=spec_builder,
    )

    for namespace in sorted(
        spec_builder.namespaces.values(),
        key=lambda ns: _resolve_namespace_name(ns),
    ):
        _emit_async_batch_invocation_function(ctx=ctx, namespace=namespace)

    async_batch_processor_out.write(MODIFY_NOTICE)
    async_batch_processor_out.write(LINT_HEADER)
    async_batch_processor_out.write("# ruff: noqa: PLR0904\n")

    _emit_types_imports(out=async_batch_processor_out, ctx=ctx)
    _emit_namespace_imports(
        out=async_batch_processor_out,
        namespaces=ctx.namespaces,
        from_namespace=None,
        config=config,
    )

    async_batch_processor_out.write(
        f"""{LINE_BREAK.join(ASYNC_BATCH_PROCESSOR_IMPORTS)}


class AsyncBatchProcessorBase(ABC):
    @abstractmethod
    def _enqueue(self, req: {refer_to(ctx=ctx, stype=ASYNC_BATCH_REQUEST_STYPE)}) -> None:
        ...

    @abstractmethod
    def send(self) -> base_t.ObjectId:
        ..."""
    )
    async_batch_processor_out.write(ctx.out.getvalue())
    async_batch_processor_out.write("\n")

    util.rewrite_file(
        f"{config.types_output}/{ASYNC_BATCH_PROCESSOR_FILENAME}.py",
        async_batch_processor_out.getvalue(),
    )


def _emit_client_class(
    *, spec_builder: builder.SpecBuilder, config: PythonConfig
) -> None:
    if not config.emit_client_class:
        return

    client_base_out = io.StringIO()
    ctx = Context(
        out=io.StringIO(),
        builder=spec_builder,
        namespace=builder.SpecNamespace("client_base"),
    )
    for namespace in sorted(
        spec_builder.namespaces.values(),
        key=lambda ns: _resolve_namespace_name(ns),
    ):
        _emit_endpoint_invocation_function(ctx=ctx, namespace=namespace)

    client_base_out.write(MODIFY_NOTICE)
    client_base_out.write(LINT_HEADER)
    client_base_out.write("# ruff: noqa: PLR0904\n")

    _emit_types_imports(out=client_base_out, ctx=ctx)
    _emit_namespace_imports(
        out=client_base_out,
        namespaces=ctx.namespaces,
        from_namespace=None,
        config=config,
        skip_non_sdk=True,
    )

    client_base_out.write(
        f"""{LINE_BREAK.join(CLIENT_CLASS_IMPORTS)}

DT = typing.TypeVar("DT")


@dataclasses.dataclass(kw_only=True)
class APIRequest:
    method: str
    endpoint: str
    args: typing.Any


class ClientMethods(ABC):

    @abstractmethod
    def do_request(self, *, api_request: APIRequest, return_type: type[DT]) -> DT:
        ..."""
    )
    client_base_out.write(ctx.out.getvalue())
    client_base_out.write("\n")

    util.rewrite_file(
        f"{config.types_output}/{CLIENT_CLASS_FILENAME}.py", client_base_out.getvalue()
    )
