"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const lambda = require("@aws-cdk/aws-lambda");
const kinesis = require("@aws-cdk/aws-kinesis");
require("@aws-cdk/assert/jest");
// --------------------------------------------------------------
// Test properties
// --------------------------------------------------------------
test('Test properties', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        }
    };
    const app = new lib_1.KinesisStreamsToLambda(stack, 'test-kinesis-streams-lambda', props);
    // Assertion 1
    expect(app.lambdaFunction !== null);
    // Assertion 2
    expect(app.kinesisStream !== null);
    // Assertion 3
    expect(app.cloudwatchAlarms !== null);
});
// --------------------------------------------------------------
// Test existing resources
// --------------------------------------------------------------
test('Test existing resources', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const fn = new lambda.Function(stack, 'test-fn', {
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    });
    const stream = new kinesis.Stream(stack, 'test-stream', {
        streamName: 'existing-stream',
        shardCount: 5,
        retentionPeriod: core_1.Duration.hours(48),
        encryption: kinesis.StreamEncryption.UNENCRYPTED
    });
    new lib_1.KinesisStreamsToLambda(stack, 'test-kinesis-streams-lambda', {
        existingLambdaObj: fn,
        existingStreamObj: stream,
    });
    expect(stack).toHaveResource('AWS::Kinesis::Stream', {
        Name: 'existing-stream',
        ShardCount: 5,
        RetentionPeriodHours: 48,
    });
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Handler: 'index.handler',
        Runtime: 'nodejs14.x',
    });
});
// --------------------------------------------------------------
// Test sqsDlqQueueProps override
// --------------------------------------------------------------
test('test sqsDlqQueueProps override', () => {
    const stack = new core_1.Stack();
    const props = {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        },
        sqsDlqQueueProps: {
            queueName: 'hello-world',
            visibilityTimeout: core_1.Duration.seconds(50)
        }
    };
    new lib_1.KinesisStreamsToLambda(stack, 'test-kinesis-streams-lambda', props);
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        QueueName: "hello-world",
        VisibilityTimeout: 50
    });
});
// --------------------------------------------------------------
// Test properties with no CW Alarms
// --------------------------------------------------------------
test('Test properties with no CW Alarms', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        },
        createCloudWatchAlarms: false
    };
    const app = new lib_1.KinesisStreamsToLambda(stack, 'test-kinesis-streams-lambda', props);
    // Assertion 1
    expect(app.lambdaFunction !== null);
    // Assertion 2
    expect(app.kinesisStream !== null);
    // Assertion 3
    expect(app.cloudwatchAlarms === null);
});
//# sourceMappingURL=data:application/json;base64,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