"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const core_1 = require("@aws-solutions-constructs/core");
const lambda = require("aws-cdk-lib/aws-lambda");
const defaults = require("@aws-solutions-constructs/core");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
// Note: All integration tests for alb are for HTTP APIs, as certificates require
// validation through DNS and email. This validation is impossible during our integration
// tests and the stack will fail to launch with an unvalidated certificate.
// Setup
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, core_1.generateIntegStackName(__filename), {
    env: { account: aws_cdk_lib_1.Aws.ACCOUNT_ID, region: 'us-east-1' },
});
stack.templateOptions.description = 'Integration Test for private HTTP API with a existing function and ALB';
const myVpc = defaults.getTestVpc(stack);
const testSg = new aws_ec2_1.SecurityGroup(stack, 'lambda-sg', { vpc: myVpc, allowAllOutbound: false });
const lambdaFunction = defaults.buildLambdaFunction(stack, {
    lambdaFunctionProps: {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: 'index.handler',
        vpc: myVpc,
        securityGroups: [testSg]
    }
});
const loadBalancer = defaults.ObtainAlb(stack, 'existing-alb', myVpc, false, undefined, {
    internetFacing: false,
    vpc: myVpc,
}, true, {
    removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
    autoDeleteObjects: true
});
const props = {
    existingLambdaObj: lambdaFunction,
    existingLoadBalancerObj: loadBalancer,
    existingVpc: myVpc,
    listenerProps: {
        protocol: 'HTTP'
    },
    publicApi: false
};
const albToLambda = new lib_1.AlbToLambda(stack, 'test-one', props);
defaults.addCfnSuppressRules(albToLambda.listener, [
    { id: 'W56', reason: 'All integration tests must be HTTP because of certificate limitations.' },
]);
const newSecurityGroup = albToLambda.loadBalancer.connections.securityGroups[0].node.defaultChild;
defaults.addCfnSuppressRules(newSecurityGroup, [
    { id: 'W29', reason: 'CDK created rule that blocks all traffic.' },
    { id: 'W2', reason: 'Rule does not apply for ELB.' },
    { id: 'W9', reason: 'Rule does not apply for ELB.' }
]);
defaults.addCfnSuppressRules(testSg, [
    { id: 'W29', reason: 'CDK created rule that blocks all traffic.' },
]);
defaults.suppressAutoDeleteHandlerWarnings(stack);
// Synth
app.synth();
//# sourceMappingURL=data:application/json;base64,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