"""
Byte pair encoding utilities adapted from:
https://github.com/openai/gpt-2/blob/master/src/encoder.py
"""

import gzip
import json
import os
from functools import lru_cache
from typing import List, Tuple

import regex as re


@lru_cache()
def bytes_to_unicode():
    """Returns list of utf-8 byte and a corresponding list of unicode strings.
    The reversible bpe codes work on unicode strings.
    This means you need a large # of unicode characters in your vocab if you want to avoid UNKs.
    When you're at something like a 10B token dataset you end up needing around 5K for decent coverage.
    This is a signficant percentage of your normal, say, 32K bpe vocab.
    To avoid that, we want lookup tables between utf-8 bytes and unicode strings.
    And avoids mapping to whitespace/control characters the bpe code barfs on.

    Args:

    Returns:

    """
    bs = (
        list(range(ord("!"), ord("~") + 1))
        + list(range(ord("¡"), ord("¬") + 1))
        + list(range(ord("®"), ord("ÿ") + 1))
    )
    cs = bs[:]
    n = 0
    for b in range(2 ** 8):
        if b not in bs:
            bs.append(b)
            cs.append(2 ** 8 + n)
            n += 1
    cs = [chr(n) for n in cs]
    return dict(zip(bs, cs))


def get_pairs(word):
    """

    Args:
      word:

    Returns:
      Word is represented as tuple of symbols (symbols being variable-length strings).

    """
    pairs = set()
    prev_char = word[0]
    for char in word[1:]:
        pairs.add((prev_char, char))
        prev_char = char
    return pairs


class Encoder:
    """ """

    def __init__(self, encoder, bpe_merges, errors="replace"):
        self.encoder = encoder
        self.decoder = {v: k for k, v in self.encoder.items()}
        self.errors = errors  # how to handle errors in decoding
        self.byte_encoder = bytes_to_unicode()
        self.byte_decoder = {v: k for k, v in self.byte_encoder.items()}
        self.bpe_ranks = dict(zip(bpe_merges, range(len(bpe_merges))))
        self.cache = {}

        # Should haved added re.IGNORECASE so BPE merges can happen for capitalized versions of contractions
        self.pat = re.compile(
            r"""'s|'t|'re|'ve|'m|'ll|'d| ?\p{L}+| ?\p{N}+| ?[^\s\p{L}\p{N}]+|\s+(?!\S)|\s+""",
        )

    @property
    def n_vocab(self) -> int:
        """ """
        return len(self.encoder)

    @property
    def end_token(self) -> int:
        """ """
        return self.n_vocab - 1

    def padded_tokens_and_mask(
        self,
        tokens: List[int],
        text_ctx: int,
    ) -> Tuple[List[int], List[bool]]:
        """

        Args:
          tokens: List[int]:
          text_ctx: int:

        Returns:

        """
        tokens = tokens[:text_ctx]
        padding = text_ctx - len(tokens)
        padded_tokens = tokens + [self.end_token] * padding
        mask = [True] * len(tokens) + [False] * padding
        return padded_tokens, mask

    def bpe(self, token):
        """

        Args:
          token:

        Returns:

        """
        if token in self.cache:
            return self.cache[token]
        word = tuple(token)
        pairs = get_pairs(word)

        if not pairs:
            return token

        while True:
            bigram = min(pairs, key=lambda pair: self.bpe_ranks.get(pair, float("inf")))
            if bigram not in self.bpe_ranks:
                break
            first, second = bigram
            new_word = []
            i = 0
            while i < len(word):
                try:
                    j = word.index(first, i)
                    new_word.extend(word[i:j])
                    i = j
                except:  # pylint: disable=bare-except
                    new_word.extend(word[i:])
                    break

                if word[i] == first and i < len(word) - 1 and word[i + 1] == second:
                    new_word.append(first + second)
                    i += 2
                else:
                    new_word.append(word[i])
                    i += 1
            new_word = tuple(new_word)
            word = new_word
            if len(word) == 1:
                break
            else:
                pairs = get_pairs(word)
        word = " ".join(word)
        self.cache[token] = word
        return word

    def encode(self, text):
        """

        Args:
          text:

        Returns:

        """
        text = text.lower()
        bpe_tokens = []
        for token in re.findall(self.pat, text):
            token = "".join(self.byte_encoder[b] for b in token.encode("utf-8"))
            bpe_tokens.extend(
                self.encoder[bpe_token] for bpe_token in self.bpe(token).split(" ")
            )
        return bpe_tokens

    def decode(self, tokens):
        """

        Args:
          tokens:

        Returns:

        """
        text = "".join([self.decoder[token] for token in tokens])
        text = bytearray(self.byte_decoder[c] for c in text).decode(
            "utf-8",
            errors=self.errors,
        )
        return text


def get_encoder():
    """ """
    root_dir = os.path.dirname(os.path.abspath(__file__))
    with gzip.open(os.path.join(root_dir, "encoder.json.gz"), "r") as f:
        encoder = json.load(f)
    with gzip.open(os.path.join(root_dir, "vocab.bpe.gz"), "r") as f:
        bpe_data = str(f.read(), "utf-8")
    bpe_merges = [tuple(merge_str.split()) for merge_str in bpe_data.split("\n")[1:-1]]
    return Encoder(
        encoder=encoder,
        bpe_merges=bpe_merges,
    )
