"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Resource = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cfn_resource_1 = require("./cfn-resource");
const construct_compat_1 = require("./construct-compat");
const lazy_1 = require("./lazy");
const physical_name_generator_1 = require("./private/physical-name-generator");
const reference_1 = require("./reference");
const stack_1 = require("./stack");
const token_1 = require("./token");
const RESOURCE_SYMBOL = Symbol.for('@aws-cdk/core.Resource');
/**
 * A construct which represents an AWS resource.
 *
 * @stability stable
 */
class Resource extends construct_compat_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _b, _c, _d, _e;
        super(scope, id);
        if ((props.account !== undefined || props.region !== undefined) && props.environmentFromArn !== undefined) {
            throw new Error(`Supply at most one of 'account'/'region' (${props.account}/${props.region}) and 'environmentFromArn' (${props.environmentFromArn})`);
        }
        Object.defineProperty(this, RESOURCE_SYMBOL, { value: true });
        this.stack = stack_1.Stack.of(this);
        const parsedArn = props.environmentFromArn ? this.stack.parseArn(props.environmentFromArn) : undefined;
        this.env = {
            account: (_c = (_b = props.account) !== null && _b !== void 0 ? _b : parsedArn === null || parsedArn === void 0 ? void 0 : parsedArn.account) !== null && _c !== void 0 ? _c : this.stack.account,
            region: (_e = (_d = props.region) !== null && _d !== void 0 ? _d : parsedArn === null || parsedArn === void 0 ? void 0 : parsedArn.region) !== null && _e !== void 0 ? _e : this.stack.region,
        };
        let physicalName = props.physicalName;
        if (props.physicalName && physical_name_generator_1.isGeneratedWhenNeededMarker(props.physicalName)) {
            // auto-generate only if cross-env is required
            this._physicalName = undefined;
            this._allowCrossEnvironment = true;
            physicalName = lazy_1.Lazy.string({ produce: () => this._physicalName });
        }
        else if (props.physicalName && !token_1.Token.isUnresolved(props.physicalName)) {
            // concrete value specified by the user
            this._physicalName = props.physicalName;
            this._allowCrossEnvironment = true;
        }
        else {
            // either undefined (deploy-time) or has tokens, which means we can't use for cross-env
            this._physicalName = props.physicalName;
            this._allowCrossEnvironment = false;
        }
        if (physicalName === undefined) {
            physicalName = token_1.Token.asString(undefined);
        }
        this.physicalName = physicalName;
    }
    /**
     * Check whether the given construct is a Resource.
     *
     * @stability stable
     */
    static isResource(construct) {
        return construct !== null && typeof (construct) === 'object' && RESOURCE_SYMBOL in construct;
    }
    /**
     * Called when this resource is referenced across environments
     * (account/region) to order to request that a physical name will be generated
     * for this resource during synthesis, so the resource can be referenced
     * through it's absolute name/arn.
     *
     * @internal
     */
    _enableCrossEnvironment() {
        if (!this._allowCrossEnvironment) {
            // error out - a deploy-time name cannot be used across environments
            throw new Error(`Cannot use resource '${this.node.path}' in a cross-environment fashion, ` +
                "the resource's physical name must be explicit set or use `PhysicalName.GENERATE_IF_NEEDED`");
        }
        if (!this._physicalName) {
            this._physicalName = this.generatePhysicalName();
        }
    }
    /**
     * Apply the given removal policy to this resource.
     *
     * The Removal Policy controls what happens to this resource when it stops
     * being managed by CloudFormation, either because you've removed it from the
     * CDK application or because you've made a change that requires the resource
     * to be replaced.
     *
     * The resource can be deleted (`RemovalPolicy.DELETE`), or left in your AWS
     * account for data recovery and cleanup later (`RemovalPolicy.RETAIN`).
     *
     * @stability stable
     */
    applyRemovalPolicy(policy) {
        const child = this.node.defaultChild;
        if (!child || !cfn_resource_1.CfnResource.isCfnResource(child)) {
            throw new Error('Cannot apply RemovalPolicy: no child or not a CfnResource. Apply the removal policy on the CfnResource directly.');
        }
        child.applyRemovalPolicy(policy);
    }
    /**
     * @stability stable
     */
    generatePhysicalName() {
        return physical_name_generator_1.generatePhysicalName(this);
    }
    /**
     * (experimental) Returns an environment-sensitive token that should be used for the resource's "name" attribute (e.g. `bucket.bucketName`).
     *
     * Normally, this token will resolve to `nameAttr`, but if the resource is
     * referenced across environments, it will be resolved to `this.physicalName`,
     * which will be a concrete name.
     *
     * @param nameAttr The CFN attribute which resolves to the resource's name.
     * @experimental
     */
    getResourceNameAttribute(nameAttr) {
        return mimicReference(nameAttr, {
            produce: (context) => {
                const consumingStack = stack_1.Stack.of(context.scope);
                if (this.stack.environment !== consumingStack.environment) {
                    this._enableCrossEnvironment();
                    return this.physicalName;
                }
                else {
                    return nameAttr;
                }
            },
        });
    }
    /**
     * (experimental) Returns an environment-sensitive token that should be used for the resource's "ARN" attribute (e.g. `bucket.bucketArn`).
     *
     * Normally, this token will resolve to `arnAttr`, but if the resource is
     * referenced across environments, `arnComponents` will be used to synthesize
     * a concrete ARN with the resource's physical name. Make sure to reference
     * `this.physicalName` in `arnComponents`.
     *
     * @param arnAttr The CFN attribute which resolves to the ARN of the resource.
     * @param arnComponents The format of the ARN of this resource.
     * @experimental
     */
    getResourceArnAttribute(arnAttr, arnComponents) {
        return mimicReference(arnAttr, {
            produce: (context) => {
                const consumingStack = stack_1.Stack.of(context.scope);
                if (this.stack.environment !== consumingStack.environment) {
                    this._enableCrossEnvironment();
                    return this.stack.formatArn(arnComponents);
                }
                else {
                    return arnAttr;
                }
            },
        });
    }
}
exports.Resource = Resource;
_a = JSII_RTTI_SYMBOL_1;
Resource[_a] = { fqn: "@aws-cdk/core.Resource", version: "1.96.0" };
/**
 * Produce a Lazy that is also a Reference (if the base value is a Reference).
 *
 * If the given value is a Reference (or resolves to a Reference), return a new
 * Reference that mimics the same target and display name, but resolves using
 * the logic of the passed lazy.
 *
 * If the given value is NOT a Reference, just return a simple Lazy.
 */
function mimicReference(refSource, producer) {
    const reference = token_1.Tokenization.reverse(refSource, {
        // If this is an ARN concatenation, just fail to extract a reference.
        failConcat: false,
    });
    if (!reference_1.Reference.isReference(reference)) {
        return lazy_1.Lazy.uncachedString(producer);
    }
    return token_1.Token.asString(new class extends reference_1.Reference {
        resolve(context) {
            return producer.produce(context);
        }
    }(reference, reference.target, reference.displayName));
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVzb3VyY2UuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJyZXNvdXJjZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUNBLGlEQUE2QztBQUM3Qyx5REFBNEU7QUFDNUUsaUNBQStDO0FBQy9DLCtFQUFzRztBQUN0RywyQ0FBd0M7QUFHeEMsbUNBQWdDO0FBQ2hDLG1DQUE4QztBQU05QyxNQUFNLGVBQWUsR0FBRyxNQUFNLENBQUMsR0FBRyxDQUFDLHdCQUF3QixDQUFDLENBQUM7Ozs7OztBQThGN0QsTUFBc0IsUUFBUyxTQUFRLDRCQUFhOzs7O0lBNEJsRCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLFFBQXVCLEVBQUU7O1FBQ2pFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLEtBQUssU0FBUyxJQUFJLEtBQUssQ0FBQyxNQUFNLEtBQUssU0FBUyxDQUFDLElBQUksS0FBSyxDQUFDLGtCQUFrQixLQUFLLFNBQVMsRUFBRTtZQUN6RyxNQUFNLElBQUksS0FBSyxDQUFDLDZDQUE2QyxLQUFLLENBQUMsT0FBTyxJQUFJLEtBQUssQ0FBQyxNQUFNLCtCQUErQixLQUFLLENBQUMsa0JBQWtCLEdBQUcsQ0FBQyxDQUFDO1NBQ3ZKO1FBRUQsTUFBTSxDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsZUFBZSxFQUFFLEVBQUUsS0FBSyxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7UUFFOUQsSUFBSSxDQUFDLEtBQUssR0FBRyxhQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRTVCLE1BQU0sU0FBUyxHQUFHLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLGtCQUFrQixDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQztRQUN2RyxJQUFJLENBQUMsR0FBRyxHQUFHO1lBQ1QsT0FBTyxjQUFFLEtBQUssQ0FBQyxPQUFPLG1DQUFJLFNBQVMsYUFBVCxTQUFTLHVCQUFULFNBQVMsQ0FBRSxPQUFPLG1DQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTztZQUNsRSxNQUFNLGNBQUUsS0FBSyxDQUFDLE1BQU0sbUNBQUksU0FBUyxhQUFULFNBQVMsdUJBQVQsU0FBUyxDQUFFLE1BQU0sbUNBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNO1NBQy9ELENBQUM7UUFFRixJQUFJLFlBQVksR0FBRyxLQUFLLENBQUMsWUFBWSxDQUFDO1FBRXRDLElBQUksS0FBSyxDQUFDLFlBQVksSUFBSSxxREFBMkIsQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLEVBQUU7WUFDekUsOENBQThDO1lBQzlDLElBQUksQ0FBQyxhQUFhLEdBQUcsU0FBUyxDQUFDO1lBQy9CLElBQUksQ0FBQyxzQkFBc0IsR0FBRyxJQUFJLENBQUM7WUFDbkMsWUFBWSxHQUFHLFdBQUksQ0FBQyxNQUFNLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLGFBQWEsRUFBRSxDQUFDLENBQUM7U0FDbkU7YUFBTSxJQUFJLEtBQUssQ0FBQyxZQUFZLElBQUksQ0FBQyxhQUFLLENBQUMsWUFBWSxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsRUFBRTtZQUN4RSx1Q0FBdUM7WUFDdkMsSUFBSSxDQUFDLGFBQWEsR0FBRyxLQUFLLENBQUMsWUFBWSxDQUFDO1lBQ3hDLElBQUksQ0FBQyxzQkFBc0IsR0FBRyxJQUFJLENBQUM7U0FDcEM7YUFBTTtZQUNMLHVGQUF1RjtZQUN2RixJQUFJLENBQUMsYUFBYSxHQUFHLEtBQUssQ0FBQyxZQUFZLENBQUM7WUFDeEMsSUFBSSxDQUFDLHNCQUFzQixHQUFHLEtBQUssQ0FBQztTQUNyQztRQUVELElBQUksWUFBWSxLQUFLLFNBQVMsRUFBRTtZQUM5QixZQUFZLEdBQUcsYUFBSyxDQUFDLFFBQVEsQ0FBQyxTQUFTLENBQUMsQ0FBQztTQUMxQztRQUVELElBQUksQ0FBQyxZQUFZLEdBQUcsWUFBWSxDQUFDO0lBQ25DLENBQUM7Ozs7OztJQS9ETSxNQUFNLENBQUMsVUFBVSxDQUFDLFNBQXFCO1FBQzVDLE9BQU8sU0FBUyxLQUFLLElBQUksSUFBSSxPQUFNLENBQUMsU0FBUyxDQUFDLEtBQUssUUFBUSxJQUFJLGVBQWUsSUFBSSxTQUFTLENBQUM7SUFDOUYsQ0FBQztJQStERDs7Ozs7OztPQU9HO0lBQ0ksdUJBQXVCO1FBQzVCLElBQUksQ0FBQyxJQUFJLENBQUMsc0JBQXNCLEVBQUU7WUFDaEMsb0VBQW9FO1lBQ3BFLE1BQU0sSUFBSSxLQUFLLENBQUMsd0JBQXdCLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxvQ0FBb0M7Z0JBQ3hGLDRGQUE0RixDQUFDLENBQUM7U0FDakc7UUFFRCxJQUFJLENBQUMsSUFBSSxDQUFDLGFBQWEsRUFBRTtZQUN2QixJQUFJLENBQUMsYUFBYSxHQUFHLElBQUksQ0FBQyxvQkFBb0IsRUFBRSxDQUFDO1NBQ2xEO0lBQ0gsQ0FBQzs7Ozs7Ozs7Ozs7Ozs7SUFhTSxrQkFBa0IsQ0FBQyxNQUFxQjtRQUM3QyxNQUFNLEtBQUssR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQztRQUNyQyxJQUFJLENBQUMsS0FBSyxJQUFJLENBQUMsMEJBQVcsQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLEVBQUU7WUFDL0MsTUFBTSxJQUFJLEtBQUssQ0FBQyxrSEFBa0gsQ0FBQyxDQUFDO1NBQ3JJO1FBQ0QsS0FBSyxDQUFDLGtCQUFrQixDQUFDLE1BQU0sQ0FBQyxDQUFDO0lBQ25DLENBQUM7Ozs7SUFFUyxvQkFBb0I7UUFDNUIsT0FBTyw4Q0FBb0IsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUNwQyxDQUFDOzs7Ozs7Ozs7OztJQWNTLHdCQUF3QixDQUFDLFFBQWdCO1FBQ2pELE9BQU8sY0FBYyxDQUFDLFFBQVEsRUFBRTtZQUM5QixPQUFPLEVBQUUsQ0FBQyxPQUF3QixFQUFFLEVBQUU7Z0JBQ3BDLE1BQU0sY0FBYyxHQUFHLGFBQUssQ0FBQyxFQUFFLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxDQUFDO2dCQUUvQyxJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsV0FBVyxLQUFLLGNBQWMsQ0FBQyxXQUFXLEVBQUU7b0JBQ3pELElBQUksQ0FBQyx1QkFBdUIsRUFBRSxDQUFDO29CQUMvQixPQUFPLElBQUksQ0FBQyxZQUFZLENBQUM7aUJBQzFCO3FCQUFNO29CQUNMLE9BQU8sUUFBUSxDQUFDO2lCQUNqQjtZQUNILENBQUM7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDOzs7Ozs7Ozs7Ozs7O0lBb0JTLHVCQUF1QixDQUFDLE9BQWUsRUFBRSxhQUE0QjtRQUM3RSxPQUFPLGNBQWMsQ0FBQyxPQUFPLEVBQUU7WUFDN0IsT0FBTyxFQUFFLENBQUMsT0FBd0IsRUFBRSxFQUFFO2dCQUNwQyxNQUFNLGNBQWMsR0FBRyxhQUFLLENBQUMsRUFBRSxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQztnQkFDL0MsSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLFdBQVcsS0FBSyxjQUFjLENBQUMsV0FBVyxFQUFFO29CQUN6RCxJQUFJLENBQUMsdUJBQXVCLEVBQUUsQ0FBQztvQkFDL0IsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxhQUFhLENBQUMsQ0FBQztpQkFDNUM7cUJBQU07b0JBQ0wsT0FBTyxPQUFPLENBQUM7aUJBQ2hCO1lBQ0gsQ0FBQztTQUNGLENBQUMsQ0FBQztJQUNMLENBQUM7O0FBektILDRCQTBLQzs7O0FBRUQ7Ozs7Ozs7O0dBUUc7QUFDSCxTQUFTLGNBQWMsQ0FBQyxTQUFjLEVBQUUsUUFBeUI7SUFDL0QsTUFBTSxTQUFTLEdBQUcsb0JBQVksQ0FBQyxPQUFPLENBQUMsU0FBUyxFQUFFO1FBQ2hELHFFQUFxRTtRQUNyRSxVQUFVLEVBQUUsS0FBSztLQUNsQixDQUFDLENBQUM7SUFDSCxJQUFJLENBQUMscUJBQVMsQ0FBQyxXQUFXLENBQUMsU0FBUyxDQUFDLEVBQUU7UUFDckMsT0FBTyxXQUFJLENBQUMsY0FBYyxDQUFDLFFBQVEsQ0FBQyxDQUFDO0tBQ3RDO0lBRUQsT0FBTyxhQUFLLENBQUMsUUFBUSxDQUFDLElBQUksS0FBTSxTQUFRLHFCQUFTO1FBQy9DLE9BQU8sQ0FBQyxPQUF3QjtZQUM5QixPQUFPLFFBQVEsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDbkMsQ0FBQztLQUNGLENBQUMsU0FBUyxFQUFFLFNBQVMsQ0FBQyxNQUFNLEVBQUUsU0FBUyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUM7QUFDekQsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEFybkNvbXBvbmVudHMgfSBmcm9tICcuL2Fybic7XG5pbXBvcnQgeyBDZm5SZXNvdXJjZSB9IGZyb20gJy4vY2ZuLXJlc291cmNlJztcbmltcG9ydCB7IElDb25zdHJ1Y3QsIENvbnN0cnVjdCBhcyBDb3JlQ29uc3RydWN0IH0gZnJvbSAnLi9jb25zdHJ1Y3QtY29tcGF0JztcbmltcG9ydCB7IElTdHJpbmdQcm9kdWNlciwgTGF6eSB9IGZyb20gJy4vbGF6eSc7XG5pbXBvcnQgeyBnZW5lcmF0ZVBoeXNpY2FsTmFtZSwgaXNHZW5lcmF0ZWRXaGVuTmVlZGVkTWFya2VyIH0gZnJvbSAnLi9wcml2YXRlL3BoeXNpY2FsLW5hbWUtZ2VuZXJhdG9yJztcbmltcG9ydCB7IFJlZmVyZW5jZSB9IGZyb20gJy4vcmVmZXJlbmNlJztcbmltcG9ydCB7IFJlbW92YWxQb2xpY3kgfSBmcm9tICcuL3JlbW92YWwtcG9saWN5JztcbmltcG9ydCB7IElSZXNvbHZlQ29udGV4dCB9IGZyb20gJy4vcmVzb2x2YWJsZSc7XG5pbXBvcnQgeyBTdGFjayB9IGZyb20gJy4vc3RhY2snO1xuaW1wb3J0IHsgVG9rZW4sIFRva2VuaXphdGlvbiB9IGZyb20gJy4vdG9rZW4nO1xuXG4vLyB2MiAtIGxlYXZlIHRoaXMgYXMgYSBzZXBhcmF0ZSBzZWN0aW9uIHNvIGl0IHJlZHVjZXMgbWVyZ2UgY29uZmxpY3RzIHdoZW4gY29tcGF0IGlzIHJlbW92ZWRcbi8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBpbXBvcnQvb3JkZXJcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuXG5jb25zdCBSRVNPVVJDRV9TWU1CT0wgPSBTeW1ib2wuZm9yKCdAYXdzLWNkay9jb3JlLlJlc291cmNlJyk7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBSZXNvdXJjZUVudmlyb25tZW50IHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGFjY291bnQ6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHJlZ2lvbjogc3RyaW5nO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgSVJlc291cmNlIGV4dGVuZHMgSUNvbnN0cnVjdCB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHN0YWNrOiBTdGFjaztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGVudjogUmVzb3VyY2VFbnZpcm9ubWVudDtcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgUmVzb3VyY2VQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHBoeXNpY2FsTmFtZT86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBhY2NvdW50Pzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSByZWdpb24/OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGVudmlyb25tZW50RnJvbUFybj86IHN0cmluZztcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBhYnN0cmFjdCBjbGFzcyBSZXNvdXJjZSBleHRlbmRzIENvcmVDb25zdHJ1Y3QgaW1wbGVtZW50cyBJUmVzb3VyY2Uge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGlzUmVzb3VyY2UoY29uc3RydWN0OiBJQ29uc3RydWN0KTogY29uc3RydWN0IGlzIENmblJlc291cmNlIHtcbiAgICByZXR1cm4gY29uc3RydWN0ICE9PSBudWxsICYmIHR5cGVvZihjb25zdHJ1Y3QpID09PSAnb2JqZWN0JyAmJiBSRVNPVVJDRV9TWU1CT0wgaW4gY29uc3RydWN0O1xuICB9XG5cbiAgcHVibGljIHJlYWRvbmx5IHN0YWNrOiBTdGFjaztcbiAgcHVibGljIHJlYWRvbmx5IGVudjogUmVzb3VyY2VFbnZpcm9ubWVudDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwcm90ZWN0ZWQgcmVhZG9ubHkgcGh5c2ljYWxOYW1lOiBzdHJpbmc7XG5cbiAgcHJpdmF0ZSBfcGh5c2ljYWxOYW1lOiBzdHJpbmcgfCB1bmRlZmluZWQ7XG4gIHByaXZhdGUgcmVhZG9ubHkgX2FsbG93Q3Jvc3NFbnZpcm9ubWVudDogYm9vbGVhbjtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogUmVzb3VyY2VQcm9wcyA9IHt9KSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGlmICgocHJvcHMuYWNjb3VudCAhPT0gdW5kZWZpbmVkIHx8IHByb3BzLnJlZ2lvbiAhPT0gdW5kZWZpbmVkKSAmJiBwcm9wcy5lbnZpcm9ubWVudEZyb21Bcm4gIT09IHVuZGVmaW5lZCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBTdXBwbHkgYXQgbW9zdCBvbmUgb2YgJ2FjY291bnQnLydyZWdpb24nICgke3Byb3BzLmFjY291bnR9LyR7cHJvcHMucmVnaW9ufSkgYW5kICdlbnZpcm9ubWVudEZyb21Bcm4nICgke3Byb3BzLmVudmlyb25tZW50RnJvbUFybn0pYCk7XG4gICAgfVxuXG4gICAgT2JqZWN0LmRlZmluZVByb3BlcnR5KHRoaXMsIFJFU09VUkNFX1NZTUJPTCwgeyB2YWx1ZTogdHJ1ZSB9KTtcblxuICAgIHRoaXMuc3RhY2sgPSBTdGFjay5vZih0aGlzKTtcblxuICAgIGNvbnN0IHBhcnNlZEFybiA9IHByb3BzLmVudmlyb25tZW50RnJvbUFybiA/IHRoaXMuc3RhY2sucGFyc2VBcm4ocHJvcHMuZW52aXJvbm1lbnRGcm9tQXJuKSA6IHVuZGVmaW5lZDtcbiAgICB0aGlzLmVudiA9IHtcbiAgICAgIGFjY291bnQ6IHByb3BzLmFjY291bnQgPz8gcGFyc2VkQXJuPy5hY2NvdW50ID8/IHRoaXMuc3RhY2suYWNjb3VudCxcbiAgICAgIHJlZ2lvbjogcHJvcHMucmVnaW9uID8/IHBhcnNlZEFybj8ucmVnaW9uID8/IHRoaXMuc3RhY2sucmVnaW9uLFxuICAgIH07XG5cbiAgICBsZXQgcGh5c2ljYWxOYW1lID0gcHJvcHMucGh5c2ljYWxOYW1lO1xuXG4gICAgaWYgKHByb3BzLnBoeXNpY2FsTmFtZSAmJiBpc0dlbmVyYXRlZFdoZW5OZWVkZWRNYXJrZXIocHJvcHMucGh5c2ljYWxOYW1lKSkge1xuICAgICAgLy8gYXV0by1nZW5lcmF0ZSBvbmx5IGlmIGNyb3NzLWVudiBpcyByZXF1aXJlZFxuICAgICAgdGhpcy5fcGh5c2ljYWxOYW1lID0gdW5kZWZpbmVkO1xuICAgICAgdGhpcy5fYWxsb3dDcm9zc0Vudmlyb25tZW50ID0gdHJ1ZTtcbiAgICAgIHBoeXNpY2FsTmFtZSA9IExhenkuc3RyaW5nKHsgcHJvZHVjZTogKCkgPT4gdGhpcy5fcGh5c2ljYWxOYW1lIH0pO1xuICAgIH0gZWxzZSBpZiAocHJvcHMucGh5c2ljYWxOYW1lICYmICFUb2tlbi5pc1VucmVzb2x2ZWQocHJvcHMucGh5c2ljYWxOYW1lKSkge1xuICAgICAgLy8gY29uY3JldGUgdmFsdWUgc3BlY2lmaWVkIGJ5IHRoZSB1c2VyXG4gICAgICB0aGlzLl9waHlzaWNhbE5hbWUgPSBwcm9wcy5waHlzaWNhbE5hbWU7XG4gICAgICB0aGlzLl9hbGxvd0Nyb3NzRW52aXJvbm1lbnQgPSB0cnVlO1xuICAgIH0gZWxzZSB7XG4gICAgICAvLyBlaXRoZXIgdW5kZWZpbmVkIChkZXBsb3ktdGltZSkgb3IgaGFzIHRva2Vucywgd2hpY2ggbWVhbnMgd2UgY2FuJ3QgdXNlIGZvciBjcm9zcy1lbnZcbiAgICAgIHRoaXMuX3BoeXNpY2FsTmFtZSA9IHByb3BzLnBoeXNpY2FsTmFtZTtcbiAgICAgIHRoaXMuX2FsbG93Q3Jvc3NFbnZpcm9ubWVudCA9IGZhbHNlO1xuICAgIH1cblxuICAgIGlmIChwaHlzaWNhbE5hbWUgPT09IHVuZGVmaW5lZCkge1xuICAgICAgcGh5c2ljYWxOYW1lID0gVG9rZW4uYXNTdHJpbmcodW5kZWZpbmVkKTtcbiAgICB9XG5cbiAgICB0aGlzLnBoeXNpY2FsTmFtZSA9IHBoeXNpY2FsTmFtZTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDYWxsZWQgd2hlbiB0aGlzIHJlc291cmNlIGlzIHJlZmVyZW5jZWQgYWNyb3NzIGVudmlyb25tZW50c1xuICAgKiAoYWNjb3VudC9yZWdpb24pIHRvIG9yZGVyIHRvIHJlcXVlc3QgdGhhdCBhIHBoeXNpY2FsIG5hbWUgd2lsbCBiZSBnZW5lcmF0ZWRcbiAgICogZm9yIHRoaXMgcmVzb3VyY2UgZHVyaW5nIHN5bnRoZXNpcywgc28gdGhlIHJlc291cmNlIGNhbiBiZSByZWZlcmVuY2VkXG4gICAqIHRocm91Z2ggaXQncyBhYnNvbHV0ZSBuYW1lL2Fybi5cbiAgICpcbiAgICogQGludGVybmFsXG4gICAqL1xuICBwdWJsaWMgX2VuYWJsZUNyb3NzRW52aXJvbm1lbnQoKTogdm9pZCB7XG4gICAgaWYgKCF0aGlzLl9hbGxvd0Nyb3NzRW52aXJvbm1lbnQpIHtcbiAgICAgIC8vIGVycm9yIG91dCAtIGEgZGVwbG95LXRpbWUgbmFtZSBjYW5ub3QgYmUgdXNlZCBhY3Jvc3MgZW52aXJvbm1lbnRzXG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYENhbm5vdCB1c2UgcmVzb3VyY2UgJyR7dGhpcy5ub2RlLnBhdGh9JyBpbiBhIGNyb3NzLWVudmlyb25tZW50IGZhc2hpb24sIGAgK1xuICAgICAgICBcInRoZSByZXNvdXJjZSdzIHBoeXNpY2FsIG5hbWUgbXVzdCBiZSBleHBsaWNpdCBzZXQgb3IgdXNlIGBQaHlzaWNhbE5hbWUuR0VORVJBVEVfSUZfTkVFREVEYFwiKTtcbiAgICB9XG5cbiAgICBpZiAoIXRoaXMuX3BoeXNpY2FsTmFtZSkge1xuICAgICAgdGhpcy5fcGh5c2ljYWxOYW1lID0gdGhpcy5nZW5lcmF0ZVBoeXNpY2FsTmFtZSgpO1xuICAgIH1cbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGFwcGx5UmVtb3ZhbFBvbGljeShwb2xpY3k6IFJlbW92YWxQb2xpY3kpIHtcbiAgICBjb25zdCBjaGlsZCA9IHRoaXMubm9kZS5kZWZhdWx0Q2hpbGQ7XG4gICAgaWYgKCFjaGlsZCB8fCAhQ2ZuUmVzb3VyY2UuaXNDZm5SZXNvdXJjZShjaGlsZCkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQ2Fubm90IGFwcGx5IFJlbW92YWxQb2xpY3k6IG5vIGNoaWxkIG9yIG5vdCBhIENmblJlc291cmNlLiBBcHBseSB0aGUgcmVtb3ZhbCBwb2xpY3kgb24gdGhlIENmblJlc291cmNlIGRpcmVjdGx5LicpO1xuICAgIH1cbiAgICBjaGlsZC5hcHBseVJlbW92YWxQb2xpY3kocG9saWN5KTtcbiAgfVxuXG4gIHByb3RlY3RlZCBnZW5lcmF0ZVBoeXNpY2FsTmFtZSgpOiBzdHJpbmcge1xuICAgIHJldHVybiBnZW5lcmF0ZVBoeXNpY2FsTmFtZSh0aGlzKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHJvdGVjdGVkIGdldFJlc291cmNlTmFtZUF0dHJpYnV0ZShuYW1lQXR0cjogc3RyaW5nKSB7XG4gICAgcmV0dXJuIG1pbWljUmVmZXJlbmNlKG5hbWVBdHRyLCB7XG4gICAgICBwcm9kdWNlOiAoY29udGV4dDogSVJlc29sdmVDb250ZXh0KSA9PiB7XG4gICAgICAgIGNvbnN0IGNvbnN1bWluZ1N0YWNrID0gU3RhY2sub2YoY29udGV4dC5zY29wZSk7XG5cbiAgICAgICAgaWYgKHRoaXMuc3RhY2suZW52aXJvbm1lbnQgIT09IGNvbnN1bWluZ1N0YWNrLmVudmlyb25tZW50KSB7XG4gICAgICAgICAgdGhpcy5fZW5hYmxlQ3Jvc3NFbnZpcm9ubWVudCgpO1xuICAgICAgICAgIHJldHVybiB0aGlzLnBoeXNpY2FsTmFtZTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICByZXR1cm4gbmFtZUF0dHI7XG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgfSk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHJvdGVjdGVkIGdldFJlc291cmNlQXJuQXR0cmlidXRlKGFybkF0dHI6IHN0cmluZywgYXJuQ29tcG9uZW50czogQXJuQ29tcG9uZW50cykge1xuICAgIHJldHVybiBtaW1pY1JlZmVyZW5jZShhcm5BdHRyLCB7XG4gICAgICBwcm9kdWNlOiAoY29udGV4dDogSVJlc29sdmVDb250ZXh0KSA9PiB7XG4gICAgICAgIGNvbnN0IGNvbnN1bWluZ1N0YWNrID0gU3RhY2sub2YoY29udGV4dC5zY29wZSk7XG4gICAgICAgIGlmICh0aGlzLnN0YWNrLmVudmlyb25tZW50ICE9PSBjb25zdW1pbmdTdGFjay5lbnZpcm9ubWVudCkge1xuICAgICAgICAgIHRoaXMuX2VuYWJsZUNyb3NzRW52aXJvbm1lbnQoKTtcbiAgICAgICAgICByZXR1cm4gdGhpcy5zdGFjay5mb3JtYXRBcm4oYXJuQ29tcG9uZW50cyk7XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgcmV0dXJuIGFybkF0dHI7XG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgfSk7XG4gIH1cbn1cblxuLyoqXG4gKiBQcm9kdWNlIGEgTGF6eSB0aGF0IGlzIGFsc28gYSBSZWZlcmVuY2UgKGlmIHRoZSBiYXNlIHZhbHVlIGlzIGEgUmVmZXJlbmNlKS5cbiAqXG4gKiBJZiB0aGUgZ2l2ZW4gdmFsdWUgaXMgYSBSZWZlcmVuY2UgKG9yIHJlc29sdmVzIHRvIGEgUmVmZXJlbmNlKSwgcmV0dXJuIGEgbmV3XG4gKiBSZWZlcmVuY2UgdGhhdCBtaW1pY3MgdGhlIHNhbWUgdGFyZ2V0IGFuZCBkaXNwbGF5IG5hbWUsIGJ1dCByZXNvbHZlcyB1c2luZ1xuICogdGhlIGxvZ2ljIG9mIHRoZSBwYXNzZWQgbGF6eS5cbiAqXG4gKiBJZiB0aGUgZ2l2ZW4gdmFsdWUgaXMgTk9UIGEgUmVmZXJlbmNlLCBqdXN0IHJldHVybiBhIHNpbXBsZSBMYXp5LlxuICovXG5mdW5jdGlvbiBtaW1pY1JlZmVyZW5jZShyZWZTb3VyY2U6IGFueSwgcHJvZHVjZXI6IElTdHJpbmdQcm9kdWNlcik6IHN0cmluZyB7XG4gIGNvbnN0IHJlZmVyZW5jZSA9IFRva2VuaXphdGlvbi5yZXZlcnNlKHJlZlNvdXJjZSwge1xuICAgIC8vIElmIHRoaXMgaXMgYW4gQVJOIGNvbmNhdGVuYXRpb24sIGp1c3QgZmFpbCB0byBleHRyYWN0IGEgcmVmZXJlbmNlLlxuICAgIGZhaWxDb25jYXQ6IGZhbHNlLFxuICB9KTtcbiAgaWYgKCFSZWZlcmVuY2UuaXNSZWZlcmVuY2UocmVmZXJlbmNlKSkge1xuICAgIHJldHVybiBMYXp5LnVuY2FjaGVkU3RyaW5nKHByb2R1Y2VyKTtcbiAgfVxuXG4gIHJldHVybiBUb2tlbi5hc1N0cmluZyhuZXcgY2xhc3MgZXh0ZW5kcyBSZWZlcmVuY2Uge1xuICAgIHJlc29sdmUoY29udGV4dDogSVJlc29sdmVDb250ZXh0KSB7XG4gICAgICByZXR1cm4gcHJvZHVjZXIucHJvZHVjZShjb250ZXh0KTtcbiAgICB9XG4gIH0ocmVmZXJlbmNlLCByZWZlcmVuY2UudGFyZ2V0LCByZWZlcmVuY2UuZGlzcGxheU5hbWUpKTtcbn0iXX0=