"""
This module provides functionality for working with Hebrew dates.
It enables the representation of Hebrew dates, conversions between Hebrew and Gregorian calendars, 
and various date arithmetic operations.
"""
#  Copyright (c) 2025 Isaac Dovolsky

from __future__ import annotations
import warnings
import datetime as dt
from .holidays import get_holiday
from .hebrewyear import HebrewYear

EPOCH_H_DATE = (14, 4, 5512)  # Hebrew date of Gregorian epoch (14 Nissan 5512)
EPOCH_G_DATE = (1752, 1, 1)   # Corresponding Gregorian epoch
G_BOUNDARY = (19, 4, 3671)    # Hebrew date of proleptic Gregorian epoch

# Hebrew Days and Weekdays
HEBREW_DAYS = (
    "א", "ב", "ג", "ד", "ה", "ו", "ז", "ח", "ט", "י", "יא", "יב", "יג", "יד",
    "טו", "טז", "יז", "יח", "יט", "כ", "כא", "כב", "כג", "כד", "כה", "כו", "כז",
    "כח", "כט", "ל"
)
WEEKDAYS = ("שבת", "ראשון", "שני", "שלישי", "רביעי", "חמישי", "שישי")

def _validate_month(month, year: HebrewYear) -> str:
    if isinstance(month, int):
        if month < 1 or month > (13 if year.is_leap else 12):
            raise ValueError(f"bad month value '{month}'")
        month = year.months[month - 1]
    elif isinstance(month, str):
        if month not in year.months:
            raise ValueError(f"bad month value '{month}'")
    else:
        raise TypeError("Invalid month type")
    return month

def _validate_day(day, month: str, year: HebrewYear) -> str:
    month = year.months.index(month)
    if isinstance(day, int):
        if day < 1 or day > year.days[month]:
            raise ValueError(f"bad day value '{day}' for {year.days[month]}-day month")
        day = HEBREW_DAYS[day - 1]
    elif isinstance(day, str):
        day = day.replace('"', '')
        if day not in HEBREW_DAYS:
            raise ValueError(f"bad day value '{day}'")
    else:
        raise TypeError("Invalid day type")
    return day


class HebrewDate:
    """
    Represents a Hebrew date, supporting conversions, arithmetic.

    Parameters:
    -----------
    **day**: ``int`` | ``str``, optional (default: 1)
        The day of the Hebrew date, represented either as an integer (1-30) or as a string (see notes below).
    **month**: ``int`` | ``str``, optional (default: 1)
        The month of the Hebrew date, represented either as an integer (1-12/13) or as a string.
    **year**: ``int`` | ``str``, optional (default: current year)
        The year of the Hebrew date, represented either as an integer (1-9999) or as a string (see notes below).
    **include_festive_days**: ``bool``, optional (default: False)
        Specifies whether to include festive days in the calculation of holidays.
    **include_fasts**: ``bool``, optional (default: False)
        Specifies whether to include fasts in the calculation of holidays.

    Attributes:
    -----------
    **year**: ``HebrewYear``
        ``HebrewYear`` object representing the year of the Hebrew date.
    **year_numeric**: ``int``
        Numeric value of the Hebrew year.
    **month**: ``str``
        Name of the Hebrew month.
    **month_numeric**: ``int``
        Numeric value of the Hebrew month (1-12/13).
    **day**: ``str``
        Day of the Hebrew date as a Hebrew string representation.
    **day_numeric**: ``int``
        Numeric day of the month (1-30).
    **weekday**: ``str``
        Hebrew name of the weekday.
    **weekday_numeric**: ``int``
        Numeric representation of the Hebrew weekday (1-7, where 1 is Sunday).
    **genesis**: ``int``
        The number of parts since the Hebrew epoch up to the start of the current date.
    **include_festive_days**: ``bool``
        Can be changed to include festive days in the calculation of holidays.
    **include_fasts**: ``bool``
        Can be changed to include fasts in the calculation of holidays.

    Properties:
    -----------
    **is_holiday**: ``bool``
        Specifies whether the current date is a Hebrew holiday.
    **holiday**: ``str``
        The holiday name if the current date is a Hebrew holiday, otherwise ''.

    Notes:
    ------
    **Important:** If both day and month are provided, then year must also be provided.

    The string input for the day parameter can be both with quotation marks in the middle or without.

    The valid input formats for the year parameter as a string are as follows:

    - A geresh after the thousands part, e.g. ה'תשפה
    - A space after the thousands part, e.g. ה תשפה
    - Quotation marks before the units part, e.g. ה'תשפ"ה or ה תשפ"ה
    """

    def __init__(self, day: int | str = None, month: int | str = None, year: int | str = None,
                 include_festive_days: bool = False, include_fasts: bool = False):
        if year is None:
            if day is not None and month is not None:
                raise ValueError("If both `day` and `month` are provided, then `year` must also be provided.")
            _d, _m, year = self.today()
            if day is None and month is None:
                day, month = _d, _m

        self.year = HebrewYear(year)
        self.month = _validate_month(1 if month is None else month, self.year)
        self.day = _validate_day(1 if day is None else day, self.month, self.year)

        self.year_numeric = self.year.year
        self.month_numeric = self.year.months.index(self.month) + 1
        self.day_numeric = HEBREW_DAYS.index(self.day) + 1

        weekday = (sum(i for i in self.year.days[:self.month_numeric - 1]) +
                   self.year.first_weekday + self.day_numeric - 1) % 7
        self.weekday = WEEKDAYS[weekday]
        self.weekday_numeric = (7 if weekday == 0 else weekday)

        self.genesis = self.year.first_new_moon() + self.days_before() * 1080 * 24
        self.include_festive_days = include_festive_days
        self.include_fasts = include_fasts
        self._is_holiday, self._holiday = get_holiday(self, include_festive_days, include_fasts)

    def __repr__(self) -> str:
        return f"HebrewDate({self.__str__()})"

    def __str__(self) -> str:
        return f"יום {self.weekday} {self.day} {self.month} {self.year}"

    def __int__(self) -> int:
        return self.genesis

    def __eq__(self, other: int | float | HebrewDate) -> bool:
        return int(self) == int(other)

    def __ne__(self, other: int | float | HebrewDate) -> bool:
        return int(self) != int(other)

    def __lt__(self, other: int | float | HebrewDate) -> bool:
        return int(self) < int(other)

    def __gt__(self, other: int | float | HebrewDate) -> bool:
        return int(self) > int(other)

    def __le__(self, other: int | float | HebrewDate) -> bool:
        return int(self) <= int(other)

    def __ge__(self, other: int | float | HebrewDate) -> bool:
        return int(self) >= int(other)

    def __add__(self, other) -> HebrewDate:
        if isinstance(other, (int, float)):
            return self.delta(days=int(other))
        raise ValueError(f"Unsupported operand type(s) for +: 'HebrewDate' and {type(other).__name__}")

    def __sub__(self, other) -> int | HebrewDate:
        if isinstance(other, (int, float)):
            return self.delta(days=-int(other))
        if isinstance(other, HebrewDate):
            return (int(self) - int(other)) // 1080 // 24
        raise TypeError(f"Unsupported operand type(s) for +: 'HebrewDate' and {type(other).__name__}")

    def __iter__(self): return iter((self.day_numeric, self.month_numeric, self.year_numeric))

    @property
    def is_holiday(self) -> bool:
        """Returns whether the current date is a Hebrew holiday."""
        return get_holiday(self, self.include_festive_days, self.include_fasts)[0]

    @property
    def holiday(self) -> str:
        """Returns the name of the Hebrew holiday if the current date is a Hebrew holiday, otherwise ''."""
        return get_holiday(self, self.include_festive_days, self.include_fasts)[1]

    def get_month_tuple(self) -> tuple[int, str]:
        """Get the month number and name as a tuple."""
        return self.month_numeric, self.month

    def get_day_tuple(self) -> tuple[int, str]:
        """Get the day number and name as a tuple."""
        return self.day_numeric, self.day

    def get_weekday_tuple(self) -> tuple[int, str]:
        """Get the weekday number and name as a tuple."""
        return self.weekday_numeric, self.weekday

    def days_before(self) -> int:
        """Calculates the number of days from the start of the year to the current date."""
        if self.month_numeric == 1:
            return self.day_numeric - 1
        return sum(self.year.days[:self.month_numeric - 1]) + self.day_numeric - 1

    def days_after(self) -> int:
        """Calculates the number of days from the current date till the end of the year."""
        return sum(self.year.days[self.month_numeric:]) + self.year.days[self.month_numeric - 1] - self.day_numeric

    # noinspection PyUnresolvedReferences
    def delta(self, days: int = 0, months: int = 0, years: int = 0) -> HebrewDate:
        """
        Computes a new HebrewDate instance by an offset of the given days, months, and years.
        """
        # Adjust the Year
        new_year = HebrewYear(self.year_numeric + years)

        # Adjust the Month
        new_month = self.month_numeric - 1 + months  # Convert to 0-based index
        while new_month < 0:  # Handle underflow
            new_year -= 1
            new_month += new_year.month_count
        while new_month >= new_year.month_count:  # Handle overflow
            new_month -= new_year.month_count
            new_year += 1

        # Adjust the Day
        new_day = self.day_numeric + days
        while new_day < 1:  # Handle day underflow
            new_month -= 1
            if new_month < 0:
                new_year -= 1
                new_month = new_year.month_count - 1
            new_day += new_year.days[new_month]
        while new_day > new_year.days[new_month]:  # Handle day overflow
            new_day -= new_year.days[new_month]
            new_month += 1
            if new_month >= new_year.month_count:
                new_year += 1
                new_month = 0
        return HebrewDate(day=new_day, month=new_month + 1, year=new_year.year)

    @classmethod
    def from_gregorian(cls, day: int = None, month: int = None, year: int = None, date: dt.date = None) -> HebrewDate:
        """
        Creates a HebrewDate object from a Gregorian date.

        Parameters:
        -----------
        **day**: ``int``, optional
            The day of the Gregorian date.
        **month**: ``int``, optional
            The month of the Gregorian date.
        **year**: ``int``, optional
            The year of the Gregorian date.
        **date**: ``datetime.date``, optional
            A datetime.date object representing the Gregorian date.

        Returns:
        --------
        ``HebrewDate``
            A corresponding HebrewDate object.

        Raises:
        -------
        ``TypeError``
            If `date` is provided but is not a datetime.date object.
        ``ValueError``
            If both `date` and the `day`, `month`, and `year` arguments are missing.
        """
        if day and month and year:
            date = dt.date(year, month, day)
        elif date is None:
            raise ValueError("Provide either a valid `date` or `day`, `month`, and `year` arguments.")
        if date.year < 1752:
            warnings.warn("Hebrew dates may be inaccurate for years earlier than 1752.", RuntimeWarning, 2)
        return cls(*EPOCH_H_DATE) + (date - dt.date(*EPOCH_G_DATE)).days

    def to_gregorian(self) -> dt.date | None:
        """
        Converts the current HebrewDate object to a Gregorian date.

        Returns:
        --------
        ``datetime.date`` | ``None``
            The corresponding Gregorian date, or None if the conversion is not possible.
        """
        try:
            date = dt.date(*EPOCH_G_DATE) + dt.timedelta(days=(self - HebrewDate(*EPOCH_H_DATE)) + 1)
            if date.year < 1752:
                warnings.warn(
                    "Hebrew dates may be inaccurate for years earlier than 1752.", RuntimeWarning, 2)
            return date
        except OverflowError:
            warnings.warn(
                "The Hebrew date is too far in the past to convert to a Gregorian date.", RuntimeWarning, 2)
            return None

    @classmethod
    def today(cls) -> HebrewDate:
        """ Returns the current Hebrew date. """
        return cls.from_gregorian(date=dt.date.today())
