import numpy as np

def traveltime_jma(delta, depth):

    """
    JMA2001構造による走時計算
    
    Parameters
    ----------
    delta: Float
           震央距離 (km)
    depth: Float
           震源深さ (km)
    
    Returns
    -------
    tp: Float
        P波到達時間 (s)
    ts: Float
        S波到達時間 (s)
    theta: Float
        震源射出角 (deg)
    """

    m0 = _nearest_grid(_dist_grd, delta)
    n0 = _nearest_grid(_dep_grd, depth)

    mat = np.zeros([9,9])
    vtp = np.zeros(9)
    vts = np.zeros(9)
    vtheta = np.zeros(9)

    for n in range(3):
        for m in range(3):

            k = 3*n + m
            mat[k,0] = _dist_grd[m0+m]**2 * _dep_grd[n0+n]**2
            mat[k,1] = _dist_grd[m0+m]**2 * _dep_grd[n0+n]
            mat[k,2] = _dist_grd[m0+m]    * _dep_grd[n0+n]**2
            mat[k,3] = _dist_grd[m0+m]**2 
            mat[k,4] =                      _dep_grd[n0+n]**2
            mat[k,5] = _dist_grd[m0+m]    * _dep_grd[n0+n]
            mat[k,6] = _dist_grd[m0+m]
            mat[k,7] =                      _dep_grd[n0+n]
            mat[k,8] = 1.0

            vtp[k] = _tp_tbl[m0+m, n0+n]
            vts[k] = _ts_tbl[m0+m, n0+n]
            vtheta[k] = _toff_tbl[m0+m, n0+n]

    vtp2 = np.linalg.solve(mat, vtp)
    vts2 = np.linalg.solve(mat, vts)
    vtheta2 = np.linalg.solve(mat, vtheta)

    tp = _interp(vtp2, delta, depth)
    ts = _interp(vts2, delta, depth)
    theta = _interp(vtheta2, delta, depth)

    return tp, ts, 180-theta

def _interp(a, dist, dep):
    
    v = a[0] * dist * dist * dep * dep \
      + a[1] * dist * dist * dep \
      + a[2] * dist * dep * dep \
      + a[3] * dist * dist \
      + a[4] * dep  * dep \
      + a[5] * dist * dep \
      + a[6] * dist \
      + a[7] * dep \
      + a[8]
    
    return v

def _nearest_grid(grd, v):
    
    n = len(grd)
    # edge
    if v < (grd[1] + grd[2]) / 2: 
        return 0
    elif (grd[n-2] + grd[n-1])/2 <= v:
        return n - 3
    else:
        for i in range(0, n-1):
            if (grd[i] + grd[i+1]) / 2 <= v < (grd[i+1] + grd[i+2]) / 2:
                return i

_tp_tbl = np.array([
    [   0.000,    0.394,    0.755,    1.099,    1.438,    1.773,    2.103,    2.429,    2.750,    3.067, 
        3.380,    3.687,    3.991,    4.289,    4.583,    4.872,    5.156,    5.436,    5.713,    5.986, 
        6.256,    6.523,    6.787,    7.049,    7.310,    7.570,    8.215,    8.857,    9.495,   10.132, 
      10.767,   11.401,   12.035,   12.666,   13.297,   13.927,   14.555,   15.183,   15.809,   16.434, 
      17.057,   17.679,   18.300,   18.920,   19.538,   20.155,   20.771,   21.385,   21.999,   22.612, 
      23.224,   23.834,   24.443,   25.051,   25.658,   26.263,   27.470,   28.673,   29.871,   31.065, 
      32.255,   33.441,   34.622,   35.798,   36.970,   38.137,   39.301,   40.459,   41.613,   42.762, 
      43.907,   45.047,   46.182,   47.312,   48.440,   49.563,   50.681,   51.793,   52.896,   53.992, 
      55.080,   56.159,   57.230,   58.293,   59.347,   60.392,   61.431,   62.461,   63.484,   64.499, 
      65.509,   66.512,   67.509,   68.499,   69.483,   70.461,   71.435,   72.403,   73.367,   74.326, 
      75.281,   76.232,   77.179,   78.122,   79.060,   79.996, ],
    [   0.416,    0.557,    0.844,    1.159,    1.482,    1.808,    2.132,    2.453,    2.771,    3.086, 
        3.396,    3.702,    4.004,    4.302,    4.594,    4.882,    5.166,    5.446,    5.722,    5.994, 
        6.263,    6.530,    6.794,    7.056,    7.316,    7.576,    8.220,    8.861,    9.500,   10.136, 
      10.771,   11.405,   12.038,   12.669,   13.300,   13.929,   14.558,   15.185,   15.811,   16.436, 
      17.059,   17.681,   18.302,   18.921,   19.540,   20.157,   20.772,   21.387,   22.001,   22.614, 
      23.225,   23.836,   24.445,   25.052,   25.659,   26.264,   27.471,   28.674,   29.872,   31.066, 
      32.256,   33.442,   34.623,   35.798,   36.970,   38.138,   39.301,   40.460,   41.614,   42.763, 
      43.908,   45.048,   46.182,   47.313,   48.440,   49.564,   50.682,   51.793,   52.897,   53.992, 
      55.080,   56.160,   57.231,   58.293,   59.347,   60.393,   61.431,   62.462,   63.484,   64.500, 
      65.509,   66.512,   67.509,   68.500,   69.484,   70.462,   71.435,   72.404,   73.367,   74.327, 
      75.282,   76.232,   77.179,   78.122,   79.061,   79.996, ],
    [   0.831,    0.879,    1.066,    1.320,    1.607,    1.909,    2.216,    2.525,    2.834,    3.142, 
        3.446,    3.747,    4.045,    4.339,    4.629,    4.914,    5.196,    5.473,    5.748,    6.019, 
        6.286,    6.552,    6.815,    7.075,    7.335,    7.593,    8.236,    8.876,    9.513,   10.148, 
      10.782,   11.415,   12.048,   12.679,   13.308,   13.937,   14.565,   15.192,   15.818,   16.443, 
      17.066,   17.687,   18.308,   18.927,   19.545,   20.162,   20.777,   21.392,   22.006,   22.618, 
      23.230,   23.840,   24.449,   25.056,   25.663,   26.268,   27.475,   28.677,   29.875,   31.069, 
      32.259,   33.445,   34.625,   35.801,   36.973,   38.141,   39.304,   40.462,   41.616,   42.765, 
      43.910,   45.050,   46.184,   47.315,   48.442,   49.566,   50.684,   51.795,   52.898,   53.994, 
      55.082,   56.161,   57.232,   58.295,   59.349,   60.394,   61.432,   62.463,   63.486,   64.501, 
      65.510,   66.513,   67.510,   68.501,   69.485,   70.463,   71.436,   72.405,   73.368,   74.328, 
      75.283,   76.233,   77.180,   78.123,   79.062,   79.997, ],
    [   1.243,    1.241,    1.357,    1.552,    1.796,    2.066,    2.350,    2.641,    2.936,    3.232, 
        3.527,    3.821,    4.112,    4.401,    4.686,    4.967,    5.245,    5.519,    5.791,    6.059, 
        6.325,    6.588,    6.849,    7.108,    7.366,    7.623,    8.263,    8.900,    9.535,   10.168, 
      10.801,   11.433,   12.064,   12.694,   13.323,   13.951,   14.578,   15.204,   15.830,   16.454, 
      17.076,   17.698,   18.318,   18.936,   19.554,   20.170,   20.786,   21.400,   22.013,   22.626, 
      23.237,   23.847,   24.456,   25.063,   25.669,   26.274,   27.481,   28.683,   29.881,   31.074, 
      32.264,   33.449,   34.630,   35.805,   36.977,   38.145,   39.308,   40.466,   41.619,   42.768, 
      43.913,   45.053,   46.187,   47.318,   48.445,   49.569,   50.686,   51.798,   52.901,   53.997, 
      55.084,   56.164,   57.235,   58.297,   59.351,   60.396,   61.434,   62.465,   63.488,   64.503, 
      65.512,   66.515,   67.512,   68.502,   69.486,   70.465,   71.438,   72.406,   73.370,   74.329, 
      75.284,   76.235,   77.182,   78.124,   79.063,   79.998, ],
    [   1.650,    1.613,    1.680,    1.826,    2.030,    2.267,    2.525,    2.795,    3.073,    3.354, 
        3.638,    3.922,    4.204,    4.486,    4.764,    5.040,    5.313,    5.583,    5.850,    6.115, 
        6.378,    6.638,    6.896,    7.153,    7.409,    7.664,    8.300,    8.933,    9.565,   10.196, 
      10.827,   11.457,   12.086,   12.715,   13.343,   13.970,   14.596,   15.221,   15.846,   16.469, 
      17.091,   17.712,   18.331,   18.949,   19.566,   20.182,   20.797,   21.411,   22.024,   22.636, 
      23.247,   23.857,   24.465,   25.072,   25.678,   26.283,   27.489,   28.691,   29.888,   31.081, 
      32.271,   33.456,   34.636,   35.811,   36.983,   38.150,   39.313,   40.471,   41.624,   42.773, 
      43.918,   45.058,   46.192,   47.322,   48.449,   49.573,   50.690,   51.801,   52.905,   54.000, 
      55.088,   56.167,   57.238,   58.300,   59.354,   60.399,   61.437,   62.468,   63.490,   64.506, 
      65.515,   66.518,   67.515,   68.505,   69.489,   70.467,   71.440,   72.409,   73.372,   74.331, 
      75.286,   76.237,   77.184,   78.126,   79.065,   80.000, ],
    [   2.052,    1.987,    2.017,    2.127,    2.295,    2.502,    2.733,    2.981,    3.240,    3.505, 
        3.775,    4.047,    4.320,    4.592,    4.863,    5.132,    5.399,    5.663,    5.926,    6.187, 
        6.445,    6.702,    6.957,    7.211,    7.464,    7.717,    8.347,    8.976,    9.604,   10.232, 
      10.860,   11.488,   12.115,   12.742,   13.368,   13.994,   14.619,   15.243,   15.867,   16.489, 
      17.110,   17.730,   18.349,   18.966,   19.583,   20.198,   20.812,   21.426,   22.038,   22.650, 
      23.260,   23.870,   24.478,   25.084,   25.690,   26.294,   27.500,   28.701,   29.898,   31.090, 
      32.280,   33.464,   34.644,   35.819,   36.990,   38.157,   39.320,   40.478,   41.631,   42.779, 
      43.924,   45.063,   46.197,   47.327,   48.454,   49.578,   50.695,   51.806,   52.909,   54.005, 
      55.092,   56.171,   57.242,   58.305,   59.358,   60.403,   61.441,   62.472,   63.494,   64.509, 
      65.518,   66.521,   67.518,   68.508,   69.492,   70.470,   71.443,   72.412,   73.375,   74.334, 
      75.289,   76.240,   77.186,   78.129,   79.068,   80.003, ],
    [   2.448,    2.361,    2.363,    2.442,    2.582,    2.761,    2.967,    3.193,    3.432,    3.681, 
        3.937,    4.196,    4.457,    4.719,    4.981,    5.242,    5.502,    5.760,    6.017,    6.273, 
        6.527,    6.779,    7.030,    7.281,    7.531,    7.780,    8.404,    9.028,    9.652,   10.276, 
      10.900,   11.526,   12.151,   12.775,   13.400,   14.024,   14.647,   15.270,   15.892,   16.513, 
      17.133,   17.752,   18.370,   18.987,   19.602,   20.217,   20.831,   21.443,   22.055,   22.666, 
      23.276,   23.885,   24.493,   25.099,   25.704,   26.308,   27.513,   28.713,   29.910,   31.102, 
      32.290,   33.475,   34.654,   35.829,   36.999,   38.166,   39.328,   40.486,   41.639,   42.787, 
      43.931,   45.070,   46.204,   47.334,   48.461,   49.584,   50.702,   51.812,   52.915,   54.010, 
      55.098,   56.177,   57.247,   58.310,   59.363,   60.408,   61.446,   62.476,   63.499,   64.514, 
      65.523,   66.526,   67.522,   68.512,   69.496,   70.474,   71.447,   72.415,   73.379,   74.338, 
      75.293,   76.243,   77.190,   78.132,   79.071,   80.006, ],
    [   2.838,    2.732,    2.711,    2.767,    2.883,    3.038,    3.222,    3.426,    3.647,    3.879, 
        4.119,    4.364,    4.614,    4.865,    5.118,    5.370,    5.622,    5.873,    6.124,    6.373, 
        6.622,    6.869,    7.116,    7.362,    7.609,    7.855,    8.471,    9.089,    9.708,   10.327, 
      10.948,   11.570,   12.192,   12.814,   13.436,   14.058,   14.680,   15.301,   15.922,   16.542, 
      17.160,   17.778,   18.395,   19.011,   19.625,   20.239,   20.852,   21.464,   22.075,   22.686, 
      23.295,   23.904,   24.511,   25.116,   25.721,   26.324,   27.528,   28.728,   29.924,   31.115, 
      32.303,   33.487,   34.666,   35.840,   37.010,   38.176,   39.338,   40.495,   41.648,   42.796, 
      43.940,   45.079,   46.213,   47.342,   48.469,   49.592,   50.709,   51.819,   52.922,   54.017, 
      55.104,   56.183,   57.254,   58.316,   59.369,   60.414,   61.452,   62.482,   63.504,   64.519, 
      65.528,   66.531,   67.527,   68.517,   69.501,   70.478,   71.451,   72.420,   73.383,   74.342, 
      75.297,   76.247,   77.194,   78.136,   79.075,   80.010, ],
    [   3.221,    3.100,    3.060,    3.098,    3.195,    3.329,    3.492,    3.677,    3.880,    4.095, 
        4.320,    4.551,    4.788,    5.028,    5.270,    5.513,    5.757,    6.000,    6.244,    6.487, 
        6.730,    6.972,    7.214,    7.456,    7.698,    7.940,    8.548,    9.159,    9.772,   10.386, 
      11.003,   11.621,   12.240,   12.859,   13.479,   14.098,   14.718,   15.337,   15.956,   16.575, 
      17.192,   17.808,   18.424,   19.039,   19.652,   20.265,   20.877,   21.488,   22.099,   22.708, 
      23.317,   23.925,   24.531,   25.136,   25.740,   26.343,   27.546,   28.745,   29.940,   31.130, 
      32.318,   33.501,   34.679,   35.853,   37.023,   38.188,   39.350,   40.507,   41.659,   42.806, 
      43.950,   45.089,   46.222,   47.351,   48.478,   49.600,   50.717,   51.827,   52.930,   54.025, 
      55.112,   56.190,   57.261,   58.323,   59.376,   60.421,   61.458,   62.488,   63.510,   64.525, 
      65.534,   66.536,   67.533,   68.522,   69.506,   70.484,   71.457,   72.425,   73.388,   74.347, 
      75.301,   76.252,   77.198,   78.141,   79.079,   80.014, ],
    [   3.600,    3.465,    3.410,    3.433,    3.513,    3.630,    3.774,    3.942,    4.127,    4.326, 
        4.536,    4.754,    4.978,    5.206,    5.438,    5.671,    5.906,    6.141,    6.377,    6.613, 
        6.850,    7.086,    7.323,    7.560,    7.797,    8.035,    8.634,    9.237,    9.844,   10.453, 
      11.065,   11.679,   12.294,   12.910,   13.527,   14.144,   14.761,   15.378,   15.995,   16.612, 
      17.228,   17.842,   18.457,   19.070,   19.683,   20.294,   20.905,   21.515,   22.125,   22.734, 
      23.342,   23.949,   24.554,   25.159,   25.762,   26.364,   27.566,   28.764,   29.958,   31.148, 
      32.334,   33.517,   34.695,   35.868,   37.037,   38.202,   39.363,   40.519,   41.671,   42.818, 
      43.961,   45.100,   46.233,   47.361,   48.488,   49.610,   50.727,   51.837,   52.939,   54.034, 
      55.120,   56.199,   57.269,   58.330,   59.383,   60.428,   61.465,   62.495,   63.517,   64.532, 
      65.540,   66.543,   67.539,   68.529,   69.512,   70.490,   71.462,   72.430,   73.394,   74.352, 
      75.307,   76.257,   77.203,   78.146,   79.084,   80.019, ],
    [   3.972,    3.826,    3.759,    3.770,    3.836,    3.937,    4.066,    4.217,    4.387,    4.570, 
        4.766,    4.970,    5.182,    5.399,    5.619,    5.842,    6.068,    6.294,    6.523,    6.752, 
        6.981,    7.212,    7.443,    7.674,    7.907,    8.140,    8.729,    9.324,    9.923,   10.527, 
      11.133,   11.743,   12.354,   12.967,   13.580,   14.194,   14.809,   15.424,   16.039,   16.653, 
      17.267,   17.880,   18.493,   19.105,   19.716,   20.327,   20.937,   21.546,   22.154,   22.762, 
      23.369,   23.975,   24.580,   25.184,   25.787,   26.388,   27.589,   28.786,   29.978,   31.167, 
      32.353,   33.535,   34.712,   35.884,   37.052,   38.217,   39.377,   40.533,   41.684,   42.831, 
      43.974,   45.112,   46.245,   47.373,   48.499,   49.621,   50.737,   51.847,   52.949,   54.043, 
      55.130,   56.208,   57.278,   58.339,   59.392,   60.436,   61.473,   62.503,   63.525,   64.539, 
      65.548,   66.550,   67.546,   68.536,   69.519,   70.496,   71.469,   72.437,   73.400,   74.358, 
      75.313,   76.263,   77.209,   78.151,   79.089,   80.024, ],
    [   4.340,    4.184,    4.107,    4.109,    4.163,    4.251,    4.365,    4.501,    4.656,    4.826, 
        5.007,    5.198,    5.398,    5.603,    5.813,    6.026,    6.242,    6.460,    6.680,    6.901, 
        7.124,    7.348,    7.573,    7.799,    8.026,    8.255,    8.833,    9.419,   10.011,   10.608, 
      11.209,   11.813,   12.420,   13.029,   13.639,   14.250,   14.862,   15.474,   16.087,   16.699, 
      17.311,   17.922,   18.533,   19.144,   19.754,   20.363,   20.971,   21.579,   22.187,   22.794, 
      23.400,   24.005,   24.609,   25.212,   25.814,   26.414,   27.614,   28.809,   30.001,   31.189, 
      32.374,   33.554,   34.730,   35.902,   37.070,   38.234,   39.394,   40.549,   41.700,   42.846, 
      43.988,   45.126,   46.258,   47.386,   48.511,   49.633,   50.749,   51.858,   52.960,   54.054, 
      55.140,   56.218,   57.288,   58.349,   59.401,   60.446,   61.482,   62.512,   63.533,   64.548, 
      65.556,   66.558,   67.554,   68.543,   69.526,   70.504,   71.476,   72.444,   73.407,   74.365, 
      75.319,   76.269,   77.215,   78.157,   79.095,   80.030, ],
    [   4.703,    4.539,    4.454,    4.448,    4.493,    4.569,    4.670,    4.793,    4.934,    5.090, 
        5.259,    5.437,    5.625,    5.818,    6.017,    6.220,    6.426,    6.636,    6.847,    7.061, 
        7.277,    7.494,    7.712,    7.933,    8.155,    8.379,    8.946,    9.522,   10.106,   10.695, 
      11.290,   11.890,   12.492,   13.096,   13.703,   14.310,   14.919,   15.529,   16.139,   16.749, 
      17.359,   17.968,   18.577,   19.186,   19.794,   20.402,   21.009,   21.616,   22.222,   22.828, 
      23.433,   24.037,   24.640,   25.242,   25.843,   26.443,   27.641,   28.835,   30.025,   31.212, 
      32.396,   33.576,   34.751,   35.922,   37.089,   38.252,   39.411,   40.566,   41.716,   42.862, 
      44.003,   45.140,   46.272,   47.400,   48.525,   49.646,   50.762,   51.871,   52.972,   54.066, 
      55.152,   56.229,   57.299,   58.359,   59.412,   60.456,   61.492,   62.521,   63.543,   64.557, 
      65.565,   66.567,   67.562,   68.552,   69.534,   70.512,   71.484,   72.451,   73.414,   74.372, 
      75.326,   76.276,   77.222,   78.164,   79.102,   80.037, ],
    [   5.062,    4.892,    4.800,    4.789,    4.824,    4.890,    4.979,    5.090,    5.219,    5.363, 
        5.519,    5.685,    5.861,    6.044,    6.232,    6.425,    6.621,    6.822,    7.025,    7.231, 
        7.439,    7.649,    7.862,    8.076,    8.292,    8.511,    9.066,    9.633,   10.208,   10.790, 
      11.379,   11.972,   12.569,   13.169,   13.772,   14.376,   14.981,   15.588,   16.195,   16.803, 
      17.410,   18.018,   18.625,   19.232,   19.838,   20.445,   21.050,   21.656,   22.261,   22.865, 
      23.469,   24.072,   24.674,   25.275,   25.875,   26.474,   27.670,   28.863,   30.052,   31.238, 
      32.420,   33.599,   34.773,   35.943,   37.110,   38.272,   39.430,   40.584,   41.734,   42.879, 
      44.020,   45.157,   46.288,   47.415,   48.540,   49.660,   50.776,   51.884,   52.985,   54.079, 
      55.164,   56.241,   57.310,   58.371,   59.423,   60.467,   61.503,   62.532,   63.553,   64.567, 
      65.574,   66.576,   67.572,   68.561,   69.543,   70.520,   71.492,   72.460,   73.422,   74.380, 
      75.334,   76.284,   77.230,   78.171,   79.109,   80.044, ],
    [   5.418,    5.241,    5.145,    5.129,    5.157,    5.213,    5.293,    5.392,    5.509,    5.641, 
        5.786,    5.941,    6.105,    6.277,    6.455,    6.638,    6.825,    7.017,    7.212,    7.410, 
        7.610,    7.814,    8.019,    8.227,    8.438,    8.651,    9.195,    9.751,   10.317,   10.891, 
      11.473,   12.061,   12.653,   13.248,   13.846,   14.446,   15.048,   15.652,   16.256,   16.861, 
      17.466,   18.071,   18.676,   19.281,   19.886,   20.491,   21.095,   21.698,   22.302,   22.905, 
      23.508,   24.110,   24.711,   25.311,   25.910,   26.508,   27.702,   28.893,   30.081,   31.265, 
      32.447,   33.625,   34.798,   35.967,   37.132,   38.294,   39.451,   40.604,   41.753,   42.898, 
      44.038,   45.174,   46.305,   47.431,   48.555,   49.676,   50.791,   51.899,   52.999,   54.092, 
      55.178,   56.255,   57.323,   58.383,   59.435,   60.478,   61.515,   62.543,   63.564,   64.577, 
      65.585,   66.586,   67.582,   68.570,   69.553,   70.530,   71.502,   72.469,   73.431,   74.389, 
      75.343,   76.292,   77.238,   78.179,   79.117,   80.051, ],
    [   5.771,    5.590,    5.490,    5.470,    5.491,    5.539,    5.609,    5.698,    5.805,    5.926, 
        6.059,    6.204,    6.357,    6.519,    6.687,    6.860,    7.038,    7.220,    7.407,    7.597, 
        7.790,    7.986,    8.185,    8.387,    8.591,    8.799,    9.330,    9.876,   10.433,   10.999, 
      11.574,   12.155,   12.741,   13.332,   13.925,   14.521,   15.120,   15.720,   16.321,   16.923, 
      17.526,   18.129,   18.731,   19.334,   19.937,   20.540,   21.142,   21.744,   22.346,   22.948, 
      23.550,   24.150,   24.750,   25.349,   25.947,   26.544,   27.737,   28.926,   30.112,   31.295, 
      32.475,   33.652,   34.824,   35.991,   37.156,   38.317,   39.473,   40.626,   41.774,   42.918, 
      44.057,   45.193,   46.323,   47.449,   48.573,   49.692,   50.807,   51.914,   53.015,   54.107, 
      55.192,   56.269,   57.337,   58.397,   59.448,   60.491,   61.527,   62.555,   63.576,   64.589, 
      65.596,   66.597,   67.592,   68.581,   69.563,   70.540,   71.511,   72.478,   73.441,   74.398, 
      75.352,   76.301,   77.247,   78.188,   79.126,   80.060, ],
    [   6.121,    5.937,    5.835,    5.811,    5.826,    5.866,    5.927,    6.007,    6.104,    6.215, 
        6.338,    6.472,    6.616,    6.767,    6.925,    7.089,    7.257,    7.431,    7.609,    7.791, 
        7.977,    8.166,    8.358,    8.554,    8.752,    8.955,    9.473,   10.007,   10.555,   11.113, 
      11.680,   12.255,   12.836,   13.421,   14.009,   14.601,   15.196,   15.792,   16.390,   16.990, 
      17.589,   18.190,   18.790,   19.391,   19.991,   20.592,   21.193,   21.793,   22.394,   22.994, 
      23.594,   24.193,   24.792,   25.390,   25.986,   26.583,   27.773,   28.961,   30.145,   31.327, 
      32.505,   33.680,   34.851,   36.018,   37.181,   38.341,   39.497,   40.649,   41.796,   42.939, 
      44.078,   45.213,   46.342,   47.468,   48.591,   49.710,   50.824,   51.931,   53.031,   54.123, 
      55.207,   56.284,   57.352,   58.411,   59.462,   60.505,   61.540,   62.568,   63.588,   64.601, 
      65.608,   66.609,   67.604,   68.592,   69.574,   70.550,   71.522,   72.489,   73.451,   74.408, 
      75.361,   76.311,   77.256,   78.197,   79.134,   80.068, ],
    [   6.469,    6.283,    6.179,    6.152,    6.162,    6.194,    6.247,    6.319,    6.407,    6.508, 
        6.622,    6.745,    6.879,    7.021,    7.170,    7.324,    7.484,    7.649,    7.819,    7.993, 
        8.171,    8.353,    8.539,    8.728,    8.920,    9.117,    9.622,   10.146,   10.683,   11.233, 
      11.793,   12.361,   12.935,   13.515,   14.099,   14.686,   15.276,   15.869,   16.464,   17.060, 
      17.657,   18.254,   18.852,   19.451,   20.049,   20.648,   21.247,   21.845,   22.444,   23.043, 
      23.641,   24.239,   24.836,   25.433,   26.029,   26.624,   27.812,   28.998,   30.180,   31.360, 
      32.537,   33.711,   34.881,   36.046,   37.209,   38.368,   39.522,   40.673,   41.820,   42.962, 
      44.100,   45.234,   46.363,   47.488,   48.610,   49.729,   50.842,   51.949,   53.048,   54.140, 
      55.224,   56.300,   57.367,   58.426,   59.477,   60.519,   61.554,   62.582,   63.602,   64.614, 
      65.621,   66.622,   67.616,   68.604,   69.586,   70.562,   71.533,   72.500,   73.461,   74.419, 
      75.372,   76.321,   77.266,   78.207,   79.144,   80.078, ],
    [   6.816,    6.628,    6.522,    6.493,    6.497,    6.524,    6.569,    6.633,    6.712,    6.805, 
        6.909,    7.024,    7.148,    7.280,    7.420,    7.565,    7.717,    7.873,    8.035,    8.202, 
        8.372,    8.547,    8.726,    8.908,    9.095,    9.286,    9.778,   10.290,   10.818,   11.359, 
      11.911,   12.472,   13.040,   13.614,   14.192,   14.775,   15.361,   15.950,   16.541,   17.134, 
      17.728,   18.323,   18.918,   19.514,   20.110,   20.707,   21.303,   21.900,   22.497,   23.095, 
      23.691,   24.288,   24.884,   25.479,   26.073,   26.667,   27.853,   29.037,   30.218,   31.396, 
      32.571,   33.744,   34.912,   36.076,   37.238,   38.395,   39.549,   40.699,   41.844,   42.986, 
      44.123,   45.257,   46.385,   47.509,   48.631,   49.749,   50.862,   51.968,   53.067,   54.158, 
      55.241,   56.317,   57.384,   58.442,   59.493,   60.535,   61.569,   62.597,   63.616,   64.628, 
      65.635,   66.635,   67.629,   68.617,   69.598,   70.574,   71.545,   72.511,   73.473,   74.430, 
      75.383,   76.332,   77.276,   78.217,   79.154,   80.088, ],
    [   7.162,    6.973,    6.865,    6.834,    6.833,    6.854,    6.893,    6.949,    7.020,    7.104, 
        7.200,    7.305,    7.421,    7.545,    7.675,    7.812,    7.955,    8.103,    8.257,    8.416, 
        8.579,    8.747,    8.919,    9.095,    9.276,    9.461,    9.940,   10.441,   10.958,   11.490, 
      12.034,   12.588,   13.149,   13.717,   14.291,   14.869,   15.451,   16.036,   16.623,   17.212, 
      17.803,   18.395,   18.987,   19.581,   20.174,   20.769,   21.363,   21.958,   22.554,   23.149, 
      23.744,   24.339,   24.933,   25.527,   26.120,   26.713,   27.897,   29.078,   30.257,   31.433, 
      32.607,   33.778,   34.945,   36.108,   37.268,   38.425,   39.577,   40.726,   41.871,   43.011, 
      44.148,   45.281,   46.408,   47.532,   48.653,   49.770,   50.882,   51.988,   53.086,   54.177, 
      55.260,   56.335,   57.401,   58.460,   59.509,   60.551,   61.585,   62.612,   63.631,   64.643, 
      65.649,   66.649,   67.643,   68.630,   69.612,   70.587,   71.558,   72.524,   73.485,   74.442, 
      75.394,   76.343,   77.288,   78.228,   79.165,   80.098, ],
    [   7.507,    7.317,    7.208,    7.174,    7.170,    7.184,    7.217,    7.266,    7.330,    7.406, 
        7.493,    7.591,    7.698,    7.813,    7.935,    8.063,    8.198,    8.338,    8.484,    8.635, 
        8.792,    8.953,    9.118,    9.288,    9.462,    9.642,   10.108,   10.597,   11.104,   11.627, 
      12.162,   12.709,   13.264,   13.826,   14.394,   14.967,   15.544,   16.125,   16.708,   17.294, 
      17.881,   18.470,   19.060,   19.651,   20.242,   20.834,   21.426,   22.019,   22.613,   23.206, 
      23.800,   24.393,   24.986,   25.578,   26.170,   26.761,   27.942,   29.122,   30.298,   31.473, 
      32.645,   33.814,   34.980,   36.141,   37.300,   38.456,   39.607,   40.755,   41.899,   43.038, 
      44.174,   45.306,   46.432,   47.555,   48.676,   49.793,   50.904,   52.009,   53.107,   54.197, 
      55.279,   56.354,   57.420,   58.477,   59.527,   60.568,   61.602,   62.628,   63.647,   64.659, 
      65.664,   66.664,   67.657,   68.645,   69.625,   70.601,   71.571,   72.537,   73.498,   74.454, 
      75.407,   76.355,   77.299,   78.240,   79.176,   80.109, ],
    [   7.852,    7.661,    7.551,    7.515,    7.506,    7.515,    7.542,    7.584,    7.641,    7.710, 
        7.790,    7.879,    7.978,    8.085,    8.198,    8.319,    8.445,    8.578,    8.716,    8.860, 
        9.009,    9.163,    9.322,    9.486,    9.654,    9.828,   10.281,   10.758,   11.255,   11.769, 
      12.296,   12.835,   13.383,   13.939,   14.501,   15.069,   15.642,   16.218,   16.798,   17.380, 
      17.964,   18.549,   19.136,   19.724,   20.313,   20.902,   21.492,   22.083,   22.675,   23.266, 
      23.858,   24.449,   25.040,   25.631,   26.221,   26.811,   27.990,   29.167,   30.342,   31.514, 
      32.685,   33.852,   35.016,   36.176,   37.334,   38.488,   39.639,   40.785,   41.928,   43.066, 
      44.201,   45.332,   46.458,   47.580,   48.700,   49.816,   50.927,   52.031,   53.128,   54.218, 
      55.300,   56.373,   57.439,   58.496,   59.545,   60.586,   61.619,   62.645,   63.664,   64.675, 
      65.680,   66.680,   67.673,   68.659,   69.640,   70.615,   71.585,   72.550,   73.511,   74.467, 
      75.419,   76.367,   77.312,   78.252,   79.188,   80.121, ],
    [   8.196,    8.004,    7.894,    7.855,    7.842,    7.847,    7.868,    7.904,    7.954,    8.016, 
        8.088,    8.170,    8.261,    8.360,    8.466,    8.578,    8.697,    8.822,    8.953,    9.089, 
        9.232,    9.379,    9.531,    9.689,    9.851,   10.019,   10.459,   10.925,   11.412,   11.915, 
      12.434,   12.966,   13.507,   14.057,   14.613,   15.176,   15.744,   16.315,   16.891,   17.469, 
      18.049,   18.632,   19.215,   19.800,   20.387,   20.974,   21.562,   22.150,   22.739,   23.329, 
      23.919,   24.508,   25.098,   25.687,   26.276,   26.864,   28.040,   29.215,   30.387,   31.558, 
      32.726,   33.892,   35.054,   36.213,   37.369,   38.522,   39.671,   40.817,   41.958,   43.096, 
      44.230,   45.360,   46.485,   47.606,   48.725,   49.841,   50.951,   52.054,   53.151,   54.240, 
      55.321,   56.394,   57.459,   58.516,   59.564,   60.605,   61.638,   62.663,   63.681,   64.692, 
      65.697,   66.696,   67.689,   68.675,   69.655,   70.630,   71.599,   72.565,   73.525,   74.481, 
      75.433,   76.381,   77.324,   78.264,   79.200,   80.133, ],
    [   8.539,    8.347,    8.237,    8.195,    8.178,    8.178,    8.194,    8.224,    8.268,    8.323, 
        8.388,    8.463,    8.546,    8.638,    8.736,    8.841,    8.952,    9.069,    9.193,    9.323, 
        9.458,    9.599,    9.745,    9.896,   10.053,   10.215,   10.643,   11.097,   11.573,   12.067, 
      12.577,   13.101,   13.635,   14.179,   14.729,   15.287,   15.849,   16.417,   16.988,   17.562, 
      18.139,   18.718,   19.298,   19.880,   20.464,   21.048,   21.634,   22.220,   22.807,   23.394, 
      23.982,   24.570,   25.158,   25.745,   26.333,   26.919,   28.093,   29.265,   30.435,   31.603, 
      32.770,   33.934,   35.094,   36.252,   37.406,   38.558,   39.706,   40.850,   41.990,   43.127, 
      44.260,   45.389,   46.513,   47.633,   48.751,   49.866,   50.976,   52.079,   53.174,   54.263, 
      55.343,   56.416,   57.481,   58.537,   59.585,   60.624,   61.657,   62.682,   63.700,   64.710, 
      65.714,   66.713,   67.705,   68.691,   69.671,   70.645,   71.615,   72.580,   73.540,   74.495, 
      75.447,   76.394,   77.338,   78.278,   79.213,   80.146, ],
    [   8.883,    8.690,    8.579,    8.535,    8.514,    8.510,    8.521,    8.545,    8.583,    8.632, 
        8.690,    8.757,    8.834,    8.918,    9.009,    9.106,    9.210,    9.320,    9.437,    9.560, 
        9.689,    9.823,    9.963,   10.108,   10.259,   10.416,   10.830,   11.273,   11.738,   12.223, 
      12.725,   13.241,   13.768,   14.305,   14.850,   15.401,   15.959,   16.522,   17.088,   17.659, 
      18.232,   18.807,   19.384,   19.963,   20.544,   21.126,   21.708,   22.292,   22.877,   23.463, 
      24.048,   24.634,   25.220,   25.806,   26.392,   26.977,   28.147,   29.317,   30.484,   31.651, 
      32.815,   33.977,   35.136,   36.292,   37.445,   38.595,   39.741,   40.884,   42.024,   43.159, 
      44.291,   45.419,   46.542,   47.662,   48.779,   49.893,   51.002,   52.104,   53.199,   54.287, 
      55.367,   56.439,   57.503,   58.558,   59.606,   60.645,   61.677,   62.702,   63.719,   64.729, 
      65.733,   66.731,   67.723,   68.708,   69.688,   70.662,   71.631,   72.595,   73.555,   74.510, 
      75.462,   76.409,   77.352,   78.291,   79.227,   80.159, ],
    [   9.226,    9.033,    8.921,    8.875,    8.850,    8.842,    8.847,    8.867,    8.899,    8.942, 
        8.994,    9.054,    9.124,    9.201,    9.285,    9.375,    9.471,    9.575,    9.685,    9.801, 
        9.923,   10.051,   10.184,   10.324,   10.469,   10.620,   11.022,   11.453,   11.908,   12.384, 
      12.877,   13.385,   13.905,   14.435,   14.974,   15.520,   16.072,   16.630,   17.193,   17.759, 
      18.328,   18.900,   19.474,   20.050,   20.627,   21.206,   21.786,   22.368,   22.950,   23.533, 
      24.117,   24.701,   25.285,   25.869,   26.453,   27.037,   28.204,   29.371,   30.536,   31.700, 
      32.862,   34.023,   35.180,   36.333,   37.485,   38.633,   39.779,   40.920,   42.058,   43.193, 
      44.324,   45.451,   46.573,   47.691,   48.808,   49.921,   51.029,   52.130,   53.225,   54.312, 
      55.391,   56.462,   57.526,   58.581,   59.628,   60.666,   61.698,   62.722,   63.739,   64.748, 
      65.752,   66.749,   67.741,   68.726,   69.705,   70.679,   71.647,   72.612,   73.571,   74.526, 
      75.477,   76.424,   77.367,   78.306,   79.241,   80.173, ],
    [  10.083,    9.889,    9.775,    9.722,    9.689,    9.671,    9.665,    9.673,    9.692,    9.720, 
        9.757,    9.801,    9.855,    9.916,    9.983,   10.056,   10.136,   10.222,   10.316,   10.416, 
      10.522,   10.635,   10.754,   10.879,   11.011,   11.149,   11.519,   11.922,   12.352,   12.804, 
      13.275,   13.763,   14.265,   14.778,   15.301,   15.833,   16.372,   16.918,   17.469,   18.024, 
      18.583,   19.146,   19.711,   20.278,   20.848,   21.420,   21.993,   22.568,   23.145,   23.722, 
      24.300,   24.879,   25.458,   26.038,   26.617,   27.197,   28.356,   29.515,   30.674,   31.831, 
      32.988,   34.143,   35.296,   36.445,   37.592,   38.737,   39.878,   41.017,   42.151,   43.283, 
      44.411,   45.535,   46.655,   47.771,   48.885,   49.996,   51.102,   52.201,   53.294,   54.379, 
      55.456,   56.526,   57.588,   58.641,   59.686,   60.724,   61.754,   62.777,   63.792,   64.800, 
      65.803,   66.799,   67.790,   68.774,   69.752,   70.724,   71.692,   72.655,   73.614,   74.568, 
      75.518,   76.464,   77.406,   78.344,   79.279,   80.210, ],
    [  10.939,   10.744,   10.627,   10.568,   10.527,   10.499,   10.483,   10.480,   10.487,   10.502, 
      10.525,   10.555,   10.594,   10.640,   10.691,   10.748,   10.812,   10.883,   10.962,   11.047, 
      11.139,   11.237,   11.342,   11.453,   11.572,   11.697,   12.038,   12.414,   12.818,   13.247, 
      13.697,   14.165,   14.648,   15.144,   15.652,   16.169,   16.694,   17.227,   17.766,   18.310, 
      18.859,   19.411,   19.967,   20.526,   21.088,   21.652,   22.218,   22.786,   23.356,   23.927, 
      24.499,   25.073,   25.647,   26.221,   26.795,   27.370,   28.521,   29.673,   30.824,   31.975, 
      33.126,   34.275,   35.422,   36.567,   37.709,   38.850,   39.987,   41.122,   42.253,   43.381, 
      44.506,   45.627,   46.744,   47.857,   48.969,   50.077,   51.181,   52.278,   53.369,   54.452, 
      55.527,   56.595,   57.655,   58.707,   59.751,   60.786,   61.815,   62.837,   63.850,   64.857, 
      65.858,   66.854,   67.843,   68.826,   69.803,   70.774,   71.741,   72.703,   73.660,   74.614, 
      75.563,   76.508,   77.449,   78.387,   79.320,   80.251, ],
    [  11.792,   11.596,   11.475,   11.411,   11.362,   11.326,   11.301,   11.287,   11.283,   11.286, 
      11.296,   11.313,   11.338,   11.369,   11.406,   11.449,   11.498,   11.555,   11.619,   11.690, 
      11.768,   11.853,   11.945,   12.043,   12.150,   12.263,   12.576,   12.925,   13.305,   13.711, 
      14.140,   14.588,   15.053,   15.532,   16.023,   16.525,   17.037,   17.556,   18.083,   18.616, 
      19.153,   19.696,   20.242,   20.792,   21.345,   21.901,   22.459,   23.020,   23.583,   24.148, 
      24.714,   25.281,   25.849,   26.418,   26.988,   27.558,   28.700,   29.843,   30.986,   32.130, 
      33.275,   34.418,   35.559,   36.698,   37.836,   38.972,   40.105,   41.236,   42.363,   43.487, 
      44.609,   45.727,   46.841,   47.951,   49.060,   50.166,   51.267,   52.362,   53.450,   54.531, 
      55.605,   56.671,   57.729,   58.779,   59.821,   60.855,   61.882,   62.901,   63.914,   64.919, 
      65.919,   66.913,   67.901,   68.882,   69.858,   70.828,   71.794,   72.755,   73.711,   74.663, 
      75.612,   76.556,   77.496,   78.433,   79.366,   80.295, ],
    [  12.644,   12.445,   12.322,   12.251,   12.195,   12.150,   12.117,   12.094,   12.079,   12.071, 
      12.068,   12.073,   12.086,   12.103,   12.126,   12.155,   12.191,   12.234,   12.285,   12.343, 
      12.408,   12.480,   12.560,   12.646,   12.741,   12.843,   13.130,   13.454,   13.810,   14.194, 
      14.602,   15.031,   15.478,   15.940,   16.415,   16.902,   17.399,   17.905,   18.419,   18.940, 
      19.466,   19.998,   20.534,   21.075,   21.619,   22.166,   22.717,   23.270,   23.826,   24.384, 
      24.943,   25.504,   26.067,   26.630,   27.194,   27.759,   28.891,   30.025,   31.161,   32.297, 
      33.434,   34.571,   35.707,   36.840,   37.973,   39.104,   40.232,   41.358,   42.482,   43.602, 
      44.720,   45.835,   46.945,   48.053,   49.159,   50.262,   51.360,   52.453,   53.538,   54.617, 
      55.688,   56.752,   57.808,   58.856,   59.896,   60.928,   61.953,   62.971,   63.982,   64.986, 
      65.984,   66.977,   67.963,   68.943,   69.918,   70.887,   71.851,   72.811,   73.766,   74.717, 
      75.664,   76.607,   77.547,   78.482,   79.414,   80.343, ],
    [  13.492,   13.292,   13.164,   13.087,   13.025,   12.973,   12.931,   12.899,   12.874,   12.855, 
      12.841,   12.834,   12.834,   12.840,   12.850,   12.865,   12.888,   12.919,   12.958,   13.003, 
      13.056,   13.117,   13.184,   13.260,   13.344,   13.435,   13.697,   13.997,   14.331,   14.694, 
      15.082,   15.492,   15.920,   16.365,   16.824,   17.296,   17.779,   18.272,   18.773,   19.282, 
      19.797,   20.318,   20.844,   21.374,   21.909,   22.448,   22.990,   23.535,   24.084,   24.634, 
      25.187,   25.742,   26.298,   26.855,   27.414,   27.973,   29.095,   30.220,   31.347,   32.475, 
      33.605,   34.735,   35.864,   36.992,   38.119,   39.244,   40.368,   41.490,   42.609,   43.725, 
      44.839,   45.950,   47.057,   48.161,   49.264,   50.364,   51.460,   52.549,   53.633,   54.709, 
      55.778,   56.839,   57.893,   58.939,   59.977,   61.007,   62.030,   63.047,   64.055,   65.058, 
      66.054,   67.045,   68.030,   69.009,   69.982,   70.949,   71.912,   72.871,   73.825,   74.775, 
      75.721,   76.663,   77.601,   78.535,   79.466,   80.394, ],
    [  14.338,   14.135,   14.004,   13.921,   13.852,   13.792,   13.743,   13.703,   13.667,   13.638, 
      13.613,   13.595,   13.584,   13.577,   13.574,   13.578,   13.589,   13.607,   13.635,   13.669, 
      13.711,   13.760,   13.817,   13.882,   13.956,   14.038,   14.275,   14.553,   14.866,   15.208, 
      15.576,   15.968,   16.379,   16.807,   17.251,   17.707,   18.176,   18.655,   19.143,   19.640, 
      20.144,   20.653,   21.169,   21.690,   22.215,   22.745,   23.279,   23.816,   24.356,   24.900, 
      25.446,   25.993,   26.543,   27.094,   27.647,   28.201,   29.312,   30.427,   31.545,   32.665, 
      33.787,   34.910,   36.032,   37.153,   38.274,   39.394,   40.513,   41.630,   42.744,   43.856, 
      44.966,   46.073,   47.176,   48.277,   49.376,   50.474,   51.566,   52.653,   53.733,   54.807, 
      55.873,   56.932,   57.984,   59.027,   60.063,   61.091,   62.112,   63.127,   64.134,   65.134, 
      66.129,   67.118,   68.101,   69.079,   70.050,   71.016,   71.978,   72.935,   73.888,   74.836, 
      75.781,   76.722,   77.659,   78.592,   79.522,   80.449, ],
    [  15.180,   14.974,   14.840,   14.751,   14.675,   14.609,   14.552,   14.503,   14.458,   14.418, 
      14.382,   14.355,   14.332,   14.313,   14.299,   14.290,   14.291,   14.298,   14.315,   14.338, 
      14.370,   14.409,   14.456,   14.511,   14.575,   14.648,   14.863,   15.120,   15.412,   15.735, 
      16.085,   16.459,   16.853,   17.265,   17.692,   18.134,   18.589,   19.054,   19.530,   20.014, 
      20.506,   21.005,   21.510,   22.021,   22.536,   23.057,   23.582,   24.111,   24.643,   25.179, 
      25.717,   26.258,   26.801,   27.346,   27.892,   28.440,   29.541,   30.646,   31.754,   32.865, 
      33.979,   35.095,   36.209,   37.324,   38.439,   39.553,   40.667,   41.778,   42.888,   43.995, 
      45.101,   46.204,   47.303,   48.400,   49.496,   50.590,   51.679,   52.763,   53.840,   54.911, 
      55.974,   57.031,   58.080,   59.121,   60.155,   61.180,   62.200,   63.212,   64.217,   65.215, 
      66.208,   67.196,   68.177,   69.153,   70.123,   71.087,   72.047,   73.003,   73.955,   74.902, 
      75.845,   76.785,   77.721,   78.653,   79.581,   80.507, ],
    [  16.018,   15.810,   15.672,   15.577,   15.495,   15.422,   15.358,   15.300,   15.246,   15.195, 
      15.148,   15.113,   15.079,   15.048,   15.022,   15.003,   14.993,   14.990,   14.997,   15.010, 
      15.032,   15.061,   15.099,   15.145,   15.201,   15.265,   15.460,   15.697,   15.970,   16.274, 
      16.606,   16.963,   17.340,   17.736,   18.148,   18.575,   19.016,   19.468,   19.931,   20.403, 
      20.883,   21.371,   21.865,   22.366,   22.872,   23.383,   23.899,   24.419,   24.943,   25.471, 
      26.002,   26.536,   27.072,   27.610,   28.151,   28.692,   29.781,   30.876,   31.974,   33.076, 
      34.182,   35.289,   36.397,   37.505,   38.613,   39.721,   40.829,   41.935,   43.039,   44.142, 
      45.243,   46.342,   47.437,   48.530,   49.622,   50.712,   51.798,   52.879,   53.953,   55.021, 
      56.082,   57.136,   58.182,   59.221,   60.252,   61.275,   62.292,   63.302,   64.305,   65.302, 
      66.293,   67.278,   68.258,   69.232,   70.200,   71.163,   72.121,   73.076,   74.025,   74.971, 
      75.913,   76.851,   77.786,   78.717,   79.644,   80.568, ],
    [  16.853,   16.642,   16.500,   16.400,   16.311,   16.232,   16.160,   16.093,   16.029,   15.968, 
      15.911,   15.867,   15.822,   15.780,   15.744,   15.714,   15.694,   15.681,   15.680,   15.684, 
      15.696,   15.717,   15.746,   15.784,   15.831,   15.887,   16.063,   16.282,   16.537,   16.823, 
      17.138,   17.478,   17.840,   18.220,   18.617,   19.030,   19.457,   19.896,   20.346,   20.806, 
      21.274,   21.751,   22.234,   22.724,   23.221,   23.722,   24.229,   24.741,   25.257,   25.777, 
      26.300,   26.827,   27.356,   27.887,   28.421,   28.956,   30.033,   31.117,   32.205,   33.298, 
      34.395,   35.494,   36.594,   37.695,   38.796,   39.898,   41.000,   42.100,   43.199,   44.297, 
      45.393,   46.487,   47.578,   48.667,   49.755,   50.842,   51.924,   53.001,   54.073,   55.137, 
      56.195,   57.246,   58.290,   59.326,   60.354,   61.375,   62.390,   63.398,   64.398,   65.393, 
      66.382,   67.365,   68.343,   69.315,   70.281,   71.242,   72.199,   73.152,   74.100,   75.044, 
      75.985,   76.922,   77.855,   78.784,   79.711,   80.634, ],
    [  17.683,   17.470,   17.323,   17.218,   17.123,   17.037,   16.957,   16.881,   16.807,   16.736, 
      16.674,   16.617,   16.561,   16.509,   16.462,   16.424,   16.395,   16.373,   16.363,   16.358, 
      16.362,   16.375,   16.395,   16.425,   16.465,   16.514,   16.672,   16.874,   17.111,   17.381, 
      17.680,   18.004,   18.350,   18.716,   19.099,   19.497,   19.910,   20.336,   20.774,   21.221, 
      21.678,   22.143,   22.616,   23.096,   23.582,   24.075,   24.572,   25.075,   25.583,   26.095, 
      26.611,   27.130,   27.651,   28.176,   28.703,   29.232,   30.297,   31.369,   32.447,   33.530, 
      34.618,   35.708,   36.800,   37.894,   38.988,   40.084,   41.179,   42.273,   43.367,   44.459, 
      45.551,   46.640,   47.726,   48.811,   49.895,   50.978,   52.057,   53.130,   54.198,   55.259, 
      56.314,   57.362,   58.403,   59.436,   60.462,   61.480,   62.492,   63.498,   64.496,   65.488, 
      66.475,   67.457,   68.432,   69.403,   70.367,   71.326,   72.281,   73.232,   74.179,   75.121, 
      76.060,   76.996,   77.928,   78.856,   79.780,   80.702, ],
    [  18.509,   18.293,   18.143,   18.031,   17.931,   17.838,   17.749,   17.663,   17.580,   17.499, 
      17.430,   17.363,   17.296,   17.233,   17.177,   17.131,   17.093,   17.064,   17.045,   17.033, 
      17.029,   17.033,   17.047,   17.070,   17.103,   17.145,   17.287,   17.472,   17.693,   17.947, 
      18.230,   18.540,   18.871,   19.222,   19.591,   19.976,   20.375,   20.789,   21.214,   21.649, 
      22.095,   22.549,   23.011,   23.480,   23.957,   24.439,   24.928,   25.422,   25.921,   26.425, 
      26.933,   27.444,   27.959,   28.476,   28.996,   29.519,   30.571,   31.632,   32.699,   33.772, 
      34.851,   35.933,   37.016,   38.102,   39.189,   40.278,   41.366,   42.455,   43.543,   44.629, 
      45.716,   46.800,   47.882,   48.962,   50.042,   51.121,   52.196,   53.266,   54.330,   55.388, 
      56.439,   57.484,   58.522,   59.552,   60.575,   61.591,   62.600,   63.603,   64.599,   65.589, 
      66.573,   67.553,   68.526,   69.494,   70.457,   71.414,   72.367,   73.317,   74.262,   75.202, 
      76.140,   77.074,   78.004,   78.931,   79.854,   80.774, ],
    [  19.331,   19.112,   18.957,   18.841,   18.734,   18.632,   18.535,   18.439,   18.345,   18.262, 
      18.180,   18.101,   18.025,   17.953,   17.889,   17.835,   17.789,   17.755,   17.726,   17.706, 
      17.695,   17.693,   17.699,   17.716,   17.742,   17.779,   17.905,   18.075,   18.281,   18.520, 
      18.789,   19.084,   19.401,   19.737,   20.093,   20.464,   20.851,   21.252,   21.665,   22.089, 
      22.523,   22.966,   23.417,   23.876,   24.343,   24.816,   25.295,   25.780,   26.271,   26.766, 
      27.266,   27.770,   28.277,   28.787,   29.301,   29.817,   30.856,   31.905,   32.962,   34.024, 
      35.093,   36.166,   37.241,   38.319,   39.398,   40.480,   41.562,   42.644,   43.726,   44.807, 
      45.888,   46.967,   48.044,   49.120,   50.196,   51.270,   52.341,   53.407,   54.467,   55.522, 
      56.570,   57.611,   58.646,   59.673,   60.693,   61.706,   62.713,   63.713,   64.707,   65.694, 
      66.676,   67.653,   68.625,   69.591,   70.551,   71.506,   72.458,   73.405,   74.348,   75.287, 
      76.223,   77.155,   78.084,   79.009,   79.931,   80.850, ],
    [  20.147,   19.926,   19.767,   19.645,   19.530,   19.420,   19.313,   19.207,   19.108,   19.015, 
      18.923,   18.833,   18.748,   18.668,   18.597,   18.535,   18.481,   18.441,   18.406,   18.379, 
      18.361,   18.353,   18.353,   18.363,   18.384,   18.414,   18.527,   18.682,   18.875,   19.100, 
      19.354,   19.635,   19.938,   20.262,   20.604,   20.963,   21.337,   21.725,   22.126,   22.539, 
      22.961,   23.394,   23.834,   24.283,   24.740,   25.203,   25.674,   26.150,   26.632,   27.119, 
      27.611,   28.107,   28.606,   29.109,   29.616,   30.125,   31.152,   32.189,   33.234,   34.286, 
      35.345,   36.409,   37.475,   38.544,   39.616,   40.691,   41.766,   42.841,   43.917,   44.992, 
      46.068,   47.142,   48.214,   49.284,   50.356,   51.426,   52.492,   53.554,   54.611,   55.662, 
      56.706,   57.744,   58.776,   59.800,   60.817,   61.827,   62.831,   63.828,   64.819,   65.804, 
      66.784,   67.758,   68.728,   69.691,   70.649,   71.603,   72.552,   73.497,   74.439,   75.376, 
      76.310,   77.240,   78.167,   79.091,   80.011,   80.928, ],
    [  20.959,   20.735,   20.572,   20.442,   20.319,   20.199,   20.082,   19.970,   19.865,   19.760, 
      19.657,   19.559,   19.465,   19.378,   19.301,   19.232,   19.173,   19.126,   19.083,   19.050, 
      19.027,   19.012,   19.006,   19.011,   19.027,   19.052,   19.151,   19.294,   19.473,   19.685, 
      19.926,   20.194,   20.484,   20.795,   21.124,   21.470,   21.832,   22.209,   22.598,   22.999, 
      23.410,   23.832,   24.262,   24.701,   25.148,   25.602,   26.063,   26.530,   27.003,   27.482, 
      27.966,   28.454,   28.946,   29.442,   29.941,   30.443,   31.457,   32.482,   33.516,   34.557, 
      35.606,   36.660,   37.718,   38.779,   39.843,   40.909,   41.978,   43.047,   44.116,   45.185, 
      46.255,   47.323,   48.390,   49.456,   50.522,   51.588,   52.650,   53.708,   54.761,   55.808, 
      56.849,   57.883,   58.911,   59.932,   60.946,   61.953,   62.954,   63.948,   64.936,   65.918, 
      66.896,   67.868,   68.835,   69.796,   70.752,   71.703,   72.650,   73.594,   74.533,   75.468, 
      76.401,   77.329,   78.255,   79.176,   80.095,   81.011, ],
    [  21.766,   21.538,   21.368,   21.231,   21.099,   20.969,   20.845,   20.728,   20.612,   20.496, 
      20.384,   20.277,   20.177,   20.084,   20.000,   19.924,   19.863,   19.807,   19.759,   19.720, 
      19.691,   19.671,   19.660,   19.660,   19.671,   19.691,   19.779,   19.909,   20.075,   20.274, 
      20.503,   20.758,   21.036,   21.334,   21.651,   21.986,   22.336,   22.701,   23.078,   23.468, 
      23.869,   24.280,   24.700,   25.129,   25.566,   26.010,   26.462,   26.920,   27.385,   27.855, 
      28.331,   28.811,   29.296,   29.784,   30.276,   30.772,   31.773,   32.785,   33.807,   34.838, 
      35.876,   36.921,   37.969,   39.021,   40.077,   41.136,   42.197,   43.259,   44.322,   45.385, 
      46.449,   47.512,   48.573,   49.634,   50.695,   51.756,   52.814,   53.868,   54.916,   55.959, 
      56.996,   58.027,   59.052,   60.069,   61.080,   62.083,   63.081,   64.073,   65.058,   66.038, 
      67.012,   67.982,   68.946,   69.905,   70.859,   71.808,   72.753,   73.694,   74.631,   75.565, 
      76.495,   77.422,   78.345,   79.265,   80.182,   81.096, ],
    [  22.566,   22.333,   22.157,   22.010,   21.868,   21.732,   21.604,   21.476,   21.348,   21.223, 
      21.103,   20.990,   20.884,   20.785,   20.695,   20.616,   20.548,   20.486,   20.432,   20.388, 
      20.354,   20.329,   20.313,   20.308,   20.315,   20.331,   20.408,   20.527,   20.681,   20.869, 
      21.085,   21.329,   21.595,   21.881,   22.186,   22.509,   22.847,   23.201,   23.568,   23.946, 
      24.337,   24.737,   25.147,   25.566,   25.993,   26.428,   26.871,   27.320,   27.776,   28.238, 
      28.706,   29.178,   29.655,   30.136,   30.621,   31.110,   32.097,   33.097,   34.107,   35.127, 
      36.155,   37.190,   38.229,   39.272,   40.320,   41.371,   42.425,   43.479,   44.535,   45.592, 
      46.650,   47.707,   48.763,   49.818,   50.875,   51.931,   52.984,   54.033,   55.078,   56.117, 
      57.150,   58.177,   59.198,   60.212,   61.219,   62.219,   63.214,   64.203,   65.185,   66.162, 
      67.133,   68.100,   69.062,   70.019,   70.970,   71.916,   72.859,   73.798,   74.733,   75.665, 
      76.593,   77.518,   78.440,   79.358,   80.273,   81.186, ],
    [  23.357,   23.118,   22.934,   22.778,   22.631,   22.490,   22.350,   22.211,   22.074,   21.942, 
      21.816,   21.698,   21.586,   21.481,   21.386,   21.304,   21.229,   21.162,   21.103,   21.055, 
      21.016,   20.986,   20.966,   20.957,   20.960,   20.973,   21.039,   21.147,   21.291,   21.467, 
      21.672,   21.904,   22.159,   22.433,   22.727,   23.039,   23.366,   23.709,   24.065,   24.433, 
      24.812,   25.203,   25.602,   26.012,   26.429,   26.855,   27.289,   27.729,   28.177,   28.631, 
      29.090,   29.555,   30.024,   30.498,   30.975,   31.457,   32.431,   33.418,   34.417,   35.425, 
      36.442,   37.467,   38.497,   39.531,   40.571,   41.614,   42.659,   43.707,   44.756,   45.806, 
      46.858,   47.909,   48.959,   50.009,   51.060,   52.112,   53.160,   54.205,   55.245,   56.280, 
      57.309,   58.332,   59.349,   60.359,   61.363,   62.360,   63.352,   64.337,   65.317,   66.290, 
      67.259,   68.223,   69.182,   70.136,   71.085,   72.029,   72.970,   73.907,   74.839,   75.769, 
      76.695,   77.618,   78.538,   79.454,   80.367,   81.278, ],
    [  24.137,   23.892,   23.700,   23.541,   23.387,   23.235,   23.084,   22.936,   22.792,   22.654, 
      22.523,   22.399,   22.282,   22.173,   22.077,   21.987,   21.906,   21.835,   21.772,   21.719, 
      21.676,   21.642,   21.618,   21.606,   21.605,   21.614,   21.671,   21.769,   21.903,   22.068, 
      22.263,   22.484,   22.728,   22.992,   23.274,   23.575,   23.892,   24.224,   24.569,   24.927, 
      25.296,   25.676,   26.066,   26.466,   26.874,   27.291,   27.716,   28.147,   28.586,   29.032, 
      29.483,   29.940,   30.402,   30.868,   31.338,   31.813,   32.774,   33.748,   34.735,   35.732, 
      36.738,   37.753,   38.773,   39.798,   40.829,   41.864,   42.902,   43.942,   44.984,   46.028, 
      47.072,   48.118,   49.162,   50.206,   51.252,   52.299,   53.342,   54.382,   55.418,   56.448, 
      57.473,   58.492,   59.506,   60.512,   61.512,   62.506,   63.494,   64.477,   65.453,   66.423, 
      67.389,   68.350,   69.307,   70.258,   71.204,   72.146,   73.084,   74.019,   74.949,   75.876, 
      76.800,   77.721,   78.639,   79.554,   80.465,   81.374, ],
    [  24.904,   24.655,   24.463,   24.294,   24.129,   23.966,   23.807,   23.652,   23.503,   23.361, 
      23.225,   23.095,   22.974,   22.864,   22.761,   22.666,   22.581,   22.505,   22.439,   22.382, 
      22.335,   22.297,   22.270,   22.255,   22.251,   22.256,   22.304,   22.393,   22.517,   22.672, 
      22.857,   23.068,   23.301,   23.555,   23.827,   24.117,   24.424,   24.745,   25.080,   25.428, 
      25.788,   26.158,   26.538,   26.928,   27.327,   27.735,   28.151,   28.574,   29.005,   29.442, 
      29.885,   30.334,   30.788,   31.247,   31.710,   32.178,   33.125,   34.087,   35.061,   36.046, 
      37.042,   38.046,   39.057,   40.073,   41.095,   42.121,   43.151,   44.184,   45.219,   46.256, 
      47.294,   48.333,   49.371,   50.410,   51.451,   52.492,   53.530,   54.566,   55.597,   56.623, 
      57.643,   58.658,   59.667,   60.670,   61.667,   62.657,   63.641,   64.620,   65.593,   66.561, 
      67.524,   68.482,   69.436,   70.384,   71.328,   72.267,   73.202,   74.135,   75.063,   75.988, 
      76.910,   77.828,   78.744,   79.657,   80.566,   81.473, ],
    [  25.667,   25.414,   25.210,   25.031,   24.857,   24.687,   24.521,   24.362,   24.209,   24.061, 
      23.920,   23.787,   23.665,   23.548,   23.440,   23.342,   23.253,   23.174,   23.103,   23.043, 
      22.992,   22.951,   22.921,   22.903,   22.896,   22.899,   22.939,   23.019,   23.133,   23.279, 
      23.455,   23.656,   23.879,   24.123,   24.385,   24.665,   24.961,   25.273,   25.598,   25.936, 
      26.286,   26.646,   27.017,   27.398,   27.788,   28.187,   28.594,   29.008,   29.431,   29.860, 
      30.295,   30.737,   31.183,   31.634,   32.090,   32.551,   33.484,   34.433,   35.395,   36.369, 
      37.354,   38.348,   39.349,   40.355,   41.368,   42.386,   43.408,   44.433,   45.461,   46.491, 
      47.522,   48.555,   49.587,   50.620,   51.655,   52.691,   53.724,   54.755,   55.781,   56.802, 
      57.819,   58.830,   59.835,   60.833,   61.826,   62.812,   63.793,   64.769,   65.739,   66.703, 
      67.663,   68.618,   69.569,   70.514,   71.455,   72.392,   73.325,   74.254,   75.180,   76.103, 
      77.022,   77.939,   78.853,   79.763,   80.671,   81.576, ],
    [  26.416,   26.155,   25.942,   25.755,   25.574,   25.399,   25.230,   25.066,   24.908,   24.755, 
      24.612,   24.478,   24.349,   24.228,   24.116,   24.014,   23.922,   23.840,   23.766,   23.702, 
      23.648,   23.604,   23.572,   23.551,   23.542,   23.541,   23.574,   23.646,   23.752,   23.889, 
      24.055,   24.247,   24.461,   24.695,   24.947,   25.218,   25.504,   25.806,   26.122,   26.450, 
      26.790,   27.142,   27.503,   27.875,   28.256,   28.646,   29.044,   29.451,   29.865,   30.286, 
      30.713,   31.147,   31.586,   32.029,   32.478,   32.932,   33.852,   34.788,   35.738,   36.700, 
      37.674,   38.657,   39.648,   40.645,   41.649,   42.658,   43.672,   44.689,   45.710,   46.732, 
      47.757,   48.783,   49.809,   50.836,   51.866,   52.896,   53.924,   54.949,   55.971,   56.988, 
      57.999,   59.006,   60.007,   61.002,   61.990,   62.973,   63.950,   64.922,   65.888,   66.849, 
      67.806,   68.758,   69.706,   70.649,   71.587,   72.521,   73.451,   74.378,   75.301,   76.222, 
      77.139,   78.053,   78.965,   79.873,   80.778,   81.681, ],
    [  27.148,   26.881,   26.662,   26.469,   26.284,   26.105,   25.931,   25.763,   25.600,   25.447, 
      25.301,   25.161,   25.028,   24.904,   24.789,   24.684,   24.589,   24.503,   24.427,   24.360, 
      24.303,   24.256,   24.222,   24.199,   24.187,   24.184,   24.209,   24.274,   24.372,   24.501, 
      24.658,   24.841,   25.046,   25.271,   25.514,   25.775,   26.052,   26.345,   26.651,   26.970, 
      27.301,   27.643,   27.996,   28.359,   28.731,   29.112,   29.502,   29.900,   30.306,   30.719, 
      31.139,   31.564,   31.996,   32.432,   32.874,   33.320,   34.227,   35.150,   36.088,   37.038, 
      38.001,   38.974,   39.954,   40.942,   41.937,   42.938,   43.943,   44.952,   45.965,   46.980, 
      47.998,   49.018,   50.037,   51.058,   52.082,   53.107,   54.130,   55.150,   56.166,   57.178, 
      58.185,   59.188,   60.184,   61.175,   62.159,   63.138,   64.112,   65.080,   66.043,   67.000, 
      67.954,   68.903,   69.847,   70.787,   71.722,   72.653,   73.581,   74.505,   75.426,   76.344, 
      77.259,   78.171,   79.080,   79.986,   80.890,   81.791, ],
    [  27.868,   27.596,   27.374,   27.177,   26.988,   26.804,   26.626,   26.455,   26.292,   26.134, 
      25.983,   25.839,   25.704,   25.576,   25.458,   25.351,   25.252,   25.164,   25.085,   25.015, 
      24.956,   24.908,   24.871,   24.846,   24.832,   24.827,   24.846,   24.903,   24.993,   25.114, 
      25.263,   25.438,   25.634,   25.850,   26.085,   26.337,   26.605,   26.888,   27.186,   27.496, 
      27.818,   28.151,   28.495,   28.849,   29.212,   29.585,   29.967,   30.356,   30.754,   31.159, 
      31.571,   31.989,   32.413,   32.843,   33.277,   33.717,   34.610,   35.520,   36.445,   37.384, 
      38.336,   39.298,   40.268,   41.246,   42.232,   43.224,   44.221,   45.222,   46.227,   47.235, 
      48.246,   49.259,   50.272,   51.287,   52.304,   53.323,   54.341,   55.356,   56.367,   57.374, 
      58.377,   59.374,   60.366,   61.353,   62.333,   63.308,   64.278,   65.243,   66.202,   67.156, 
      68.106,   69.052,   69.993,   70.930,   71.862,   72.790,   73.715,   74.637,   75.555,   76.470, 
      77.383,   78.292,   79.199,   80.103,   81.004,   81.903, ],
    [  28.579,   28.305,   28.079,   27.878,   27.684,   27.496,   27.317,   27.145,   26.977,   26.815, 
      26.661,   26.514,   26.375,   26.246,   26.125,   26.015,   25.914,   25.824,   25.741,   25.670, 
      25.609,   25.558,   25.520,   25.494,   25.477,   25.470,   25.482,   25.533,   25.616,   25.729, 
      25.871,   26.038,   26.226,   26.433,   26.659,   26.902,   27.162,   27.436,   27.725,   28.026, 
      28.340,   28.664,   28.999,   29.345,   29.700,   30.064,   30.438,   30.819,   31.209,   31.606, 
      32.011,   32.421,   32.838,   33.260,   33.687,   34.120,   34.999,   35.897,   36.810,   37.737, 
      38.677,   39.629,   40.589,   41.557,   42.533,   43.516,   44.505,   45.498,   46.495,   47.496, 
      48.500,   49.506,   50.512,   51.521,   52.532,   53.545,   54.558,   55.567,   56.573,   57.576, 
      58.573,   59.566,   60.554,   61.536,   62.512,   63.483,   64.449,   65.410,   66.365,   67.315, 
      68.262,   69.205,   70.143,   71.077,   72.006,   72.931,   73.853,   74.772,   75.687,   76.600, 
      77.510,   78.417,   79.322,   80.223,   81.122,   82.019, ],
    [  29.283,   29.006,   28.776,   28.571,   28.376,   28.188,   28.005,   27.827,   27.656,   27.491, 
      27.335,   27.185,   27.044,   26.912,   26.789,   26.677,   26.574,   26.480,   26.397,   26.323, 
      26.260,   26.208,   26.168,   26.140,   26.122,   26.112,   26.119,   26.163,   26.240,   26.346, 
      26.480,   26.640,   26.820,   27.019,   27.237,   27.472,   27.723,   27.989,   28.269,   28.562, 
      28.866,   29.182,   29.509,   29.846,   30.193,   30.549,   30.915,   31.288,   31.670,   32.060, 
      32.457,   32.860,   33.269,   33.684,   34.104,   34.530,   35.396,   36.281,   37.181,   38.097, 
      39.026,   39.967,   40.917,   41.875,   42.842,   43.816,   44.796,   45.781,   46.770,   47.763, 
      48.759,   49.758,   50.759,   51.761,   52.766,   53.773,   54.780,   55.784,   56.785,   57.782, 
      58.775,   59.763,   60.746,   61.724,   62.696,   63.662,   64.624,   65.581,   66.533,   67.480, 
      68.423,   69.362,   70.297,   71.227,   72.153,   73.075,   73.995,   74.911,   75.824,   76.733, 
      77.641,   78.545,   79.447,   80.347,   81.243,   82.138, ],
    [  29.979,   29.699,   29.467,   29.263,   29.065,   28.872,   28.685,   28.505,   28.331,   28.164, 
      28.005,   27.854,   27.711,   27.576,   27.451,   27.337,   27.231,   27.136,   27.050,   26.975, 
      26.911,   26.858,   26.816,   26.786,   26.766,   26.755,   26.756,   26.795,   26.865,   26.964, 
      27.091,   27.244,   27.416,   27.608,   27.818,   28.044,   28.287,   28.545,   28.817,   29.102, 
      29.398,   29.706,   30.024,   30.353,   30.692,   31.040,   31.398,   31.763,   32.138,   32.520, 
      32.909,   33.305,   33.707,   34.115,   34.528,   34.947,   35.800,   36.671,   37.560,   38.463, 
      39.381,   40.311,   41.251,   42.199,   43.157,   44.122,   45.093,   46.070,   47.051,   48.036, 
      49.025,   50.017,   51.011,   52.006,   53.006,   54.007,   55.008,   56.006,   57.002,   57.994, 
      58.982,   59.965,   60.943,   61.917,   62.884,   63.847,   64.804,   65.757,   66.705,   67.648, 
      68.588,   69.523,   70.455,   71.382,   72.305,   73.224,   74.140,   75.053,   75.963,   76.870, 
      77.775,   78.677,   79.577,   80.474,   81.368,   82.260, ],
    [  30.671,   30.391,   30.157,   29.948,   29.746,   29.550,   29.361,   29.179,   29.002,   28.833, 
      28.672,   28.519,   28.374,   28.237,   28.111,   27.994,   27.887,   27.790,   27.703,   27.626, 
      27.561,   27.506,   27.464,   27.432,   27.410,   27.397,   27.394,   27.426,   27.490,   27.583, 
      27.704,   27.849,   28.015,   28.199,   28.401,   28.620,   28.855,   29.105,   29.369,   29.646, 
      29.934,   30.234,   30.544,   30.865,   31.196,   31.537,   31.886,   32.244,   32.611,   32.986, 
      33.367,   33.756,   34.151,   34.552,   34.958,   35.370,   36.210,   37.069,   37.945,   38.837, 
      39.743,   40.662,   41.592,   42.530,   43.478,   44.434,   45.396,   46.365,   47.338,   48.315, 
      49.297,   50.282,   51.269,   52.258,   53.251,   54.246,   55.241,   56.234,   57.224,   58.211, 
      59.194,   60.172,   61.146,   62.114,   63.077,   64.035,   64.989,   65.938,   66.882,   67.821, 
      68.757,   69.689,   70.617,   71.541,   72.460,   73.376,   74.289,   75.199,   76.107,   77.011, 
      77.913,   78.813,   79.709,   80.604,   81.496,   82.386, ],
    [  31.360,   31.076,   30.839,   30.627,   30.422,   30.225,   30.033,   29.849,   29.671,   29.500, 
      29.337,   29.182,   29.035,   28.897,   28.768,   28.649,   28.541,   28.443,   28.354,   28.277, 
      28.210,   28.154,   28.110,   28.078,   28.054,   28.040,   28.031,   28.059,   28.117,   28.204, 
      28.318,   28.457,   28.615,   28.792,   28.987,   29.199,   29.426,   29.669,   29.925,   30.194, 
      30.474,   30.766,   31.069,   31.382,   31.705,   32.038,   32.380,   32.730,   33.090,   33.457, 
      33.832,   34.213,   34.601,   34.994,   35.394,   35.799,   36.626,   37.472,   38.336,   39.216, 
      40.112,   41.020,   41.939,   42.867,   43.805,   44.752,   45.706,   46.666,   47.631,   48.601, 
      49.575,   50.553,   51.532,   52.515,   53.501,   54.490,   55.479,   56.467,   57.451,   58.433, 
      59.410,   60.384,   61.353,   62.316,   63.275,   64.229,   65.178,   66.123,   67.063,   67.998, 
      68.930,   69.858,   70.783,   71.703,   72.620,   73.532,   74.442,   75.349,   76.254,   77.155, 
      78.054,   78.951,   79.846,   80.737,   81.627,   82.514, ],
    [  32.040,   31.755,   31.516,   31.301,   31.095,   30.895,   30.702,   30.516,   30.336,   30.163, 
      29.999,   29.842,   29.693,   29.553,   29.423,   29.303,   29.194,   29.095,   29.005,   28.927, 
      28.859,   28.802,   28.757,   28.723,   28.698,   28.682,   28.668,   28.691,   28.744,   28.825, 
      28.933,   29.066,   29.218,   29.388,   29.576,   29.780,   30.001,   30.236,   30.484,   30.745, 
      31.018,   31.303,   31.598,   31.903,   32.219,   32.544,   32.878,   33.222,   33.574,   33.934, 
      34.301,   34.675,   35.056,   35.443,   35.836,   36.234,   37.048,   37.882,   38.734,   39.602, 
      40.486,   41.384,   42.292,   43.210,   44.139,   45.077,   46.022,   46.973,   47.930,   48.892, 
      49.858,   50.829,   51.801,   52.777,   53.757,   54.740,   55.723,   56.705,   57.684,   58.660, 
      59.632,   60.600,   61.564,   62.523,   63.477,   64.426,   65.371,   66.312,   67.248,   68.179, 
      69.107,   70.032,   70.953,   71.870,   72.783,   73.692,   74.599,   75.503,   76.404,   77.303, 
      78.199,   79.093,   79.985,   80.874,   81.761,   82.646, ],
    [  32.717,   32.429,   32.188,   31.972,   31.764,   31.563,   31.368,   31.180,   30.999,   30.825, 
      30.658,   30.500,   30.350,   30.209,   30.077,   29.956,   29.846,   29.745,   29.655,   29.575, 
      29.507,   29.449,   29.403,   29.368,   29.341,   29.324,   29.306,   29.324,   29.372,   29.447, 
      29.550,   29.676,   29.822,   29.986,   30.167,   30.364,   30.578,   30.805,   31.047,   31.301, 
      31.566,   31.843,   32.131,   32.429,   32.737,   33.055,   33.382,   33.718,   34.063,   34.415, 
      34.776,   35.143,   35.517,   35.897,   36.283,   36.675,   37.476,   38.297,   39.137,   39.994, 
      40.867,   41.753,   42.651,   43.560,   44.479,   45.407,   46.343,   47.286,   48.234,   49.188, 
      50.148,   51.111,   52.076,   53.045,   54.019,   54.995,   55.972,   56.948,   57.921,   58.892, 
      59.859,   60.822,   61.781,   62.735,   63.684,   64.629,   65.569,   66.506,   67.437,   68.364, 
      69.289,   70.210,   71.127,   72.040,   72.950,   73.856,   74.759,   75.660,   76.558,   77.454, 
      78.348,   79.239,   80.128,   81.015,   81.899,   82.782, ],
    [  34.057,   33.767,   33.524,   33.304,   33.093,   32.889,   32.692,   32.501,   32.316,   32.140, 
      31.971,   31.810,   31.658,   31.515,   31.382,   31.259,   31.147,   31.045,   30.953,   30.872, 
      30.801,   30.742,   30.693,   30.656,   30.627,   30.607,   30.581,   30.590,   30.628,   30.694, 
      30.786,   30.901,   31.035,   31.186,   31.355,   31.539,   31.739,   31.953,   32.181,   32.421, 
      32.672,   32.935,   33.208,   33.492,   33.786,   34.089,   34.402,   34.724,   35.055,   35.394, 
      35.740,   36.094,   36.455,   36.821,   37.194,   37.573,   38.348,   39.145,   39.961,   40.794, 
      41.645,   42.510,   43.387,   44.276,   45.175,   46.085,   47.003,   47.928,   48.860,   49.798, 
      50.742,   51.690,   52.642,   53.597,   54.557,   55.521,   56.485,   57.449,   58.411,   59.370, 
      60.326,   61.279,   62.227,   63.172,   64.111,   65.046,   65.978,   66.906,   67.829,   68.748, 
      69.664,   70.577,   71.487,   72.393,   73.295,   74.194,   75.091,   75.986,   76.877,   77.767, 
      78.655,   79.540,   80.423,   81.305,   82.184,   83.061, ],
    [  35.385,   35.093,   34.847,   34.626,   34.412,   34.205,   34.005,   33.812,   33.626,   33.448, 
      33.277,   33.115,   32.961,   32.817,   32.682,   32.558,   32.445,   32.342,   32.248,   32.166, 
      32.094,   32.033,   31.982,   31.943,   31.912,   31.890,   31.856,   31.857,   31.887,   31.943, 
      32.025,   32.130,   32.253,   32.393,   32.550,   32.722,   32.909,   33.111,   33.325,   33.552, 
      33.790,   34.039,   34.299,   34.569,   34.850,   35.139,   35.439,   35.747,   36.064,   36.390, 
      36.723,   37.064,   37.411,   37.765,   38.125,   38.491,   39.241,   40.014,   40.806,   41.617, 
      42.445,   43.289,   44.145,   45.014,   45.894,   46.785,   47.685,   48.593,   49.508,   50.430, 
      51.358,   52.291,   53.228,   54.169,   55.116,   56.066,   57.018,   57.970,   58.920,   59.867, 
      60.812,   61.754,   62.692,   63.626,   64.556,   65.482,   66.404,   67.322,   68.237,   69.147, 
      70.055,   70.960,   71.862,   72.761,   73.656,   74.547,   75.437,   76.325,   77.211,   78.094, 
      78.975,   79.855,   80.732,   81.608,   82.481,   83.354, ],
    [  36.703,   36.409,   36.161,   35.938,   35.722,   35.514,   35.313,   35.118,   34.931,   34.751, 
      34.579,   34.416,   34.261,   34.116,   33.980,   33.855,   33.740,   33.636,   33.542,   33.458, 
      33.384,   33.322,   33.270,   33.228,   33.196,   33.171,   33.131,   33.125,   33.146,   33.194, 
      33.267,   33.363,   33.476,   33.606,   33.751,   33.912,   34.088,   34.277,   34.480,   34.694, 
      34.919,   35.156,   35.403,   35.660,   35.927,   36.204,   36.490,   36.785,   37.089,   37.402, 
      37.723,   38.051,   38.385,   38.726,   39.074,   39.428,   40.154,   40.902,   41.671,   42.459, 
      43.266,   44.089,   44.924,   45.773,   46.634,   47.506,   48.388,   49.278,   50.177,   51.082, 
      51.994,   52.912,   53.834,   54.761,   55.694,   56.631,   57.571,   58.510,   59.447,   60.383, 
      61.317,   62.248,   63.175,   64.099,   65.018,   65.934,   66.846,   67.756,   68.661,   69.562, 
      70.462,   71.359,   72.253,   73.143,   74.031,   74.915,   75.798,   76.679,   77.557,   78.434, 
      79.309,   80.182,   81.054,   81.924,   82.792,   83.658, ],
    [  38.012,   37.717,   37.468,   37.244,   37.027,   36.818,   36.615,   36.420,   36.232,   36.051, 
      35.878,   35.714,   35.558,   35.412,   35.275,   35.149,   35.034,   34.928,   34.833,   34.748, 
      34.673,   34.609,   34.556,   34.513,   34.479,   34.451,   34.405,   34.392,   34.407,   34.447, 
      34.512,   34.599,   34.703,   34.822,   34.958,   35.108,   35.273,   35.451,   35.642,   35.845, 
      36.058,   36.282,   36.517,   36.762,   37.016,   37.281,   37.554,   37.837,   38.129,   38.429, 
      38.738,   39.053,   39.376,   39.704,   40.040,   40.382,   41.084,   41.809,   42.555,   43.321, 
      44.106,   44.908,   45.723,   46.552,   47.393,   48.247,   49.111,   49.984,   50.865,   51.754, 
      52.651,   53.553,   54.460,   55.372,   56.292,   57.215,   58.141,   59.068,   59.993,   60.917, 
      61.839,   62.759,   63.675,   64.588,   65.497,   66.403,   67.305,   68.205,   69.101,   69.994, 
      70.884,   71.773,   72.658,   73.541,   74.420,   75.297,   76.173,   77.046,   77.918,   78.788, 
      79.656,   80.523,   81.389,   82.252,   83.114,   83.975, ],
    [  39.316,   39.021,   38.771,   38.545,   38.328,   38.118,   37.914,   37.718,   37.529,   37.348, 
      37.174,   37.009,   36.853,   36.706,   36.568,   36.442,   36.325,   36.219,   36.122,   36.036, 
      35.960,   35.895,   35.841,   35.797,   35.760,   35.729,   35.679,   35.660,   35.668,   35.700, 
      35.758,   35.838,   35.933,   36.043,   36.169,   36.309,   36.464,   36.632,   36.812,   37.003, 
      37.206,   37.418,   37.641,   37.874,   38.117,   38.369,   38.631,   38.902,   39.182,   39.470, 
      39.767,   40.070,   40.381,   40.698,   41.021,   41.351,   42.030,   42.733,   43.457,   44.201, 
      44.965,   45.746,   46.541,   47.349,   48.172,   49.007,   49.853,   50.708,   51.573,   52.445, 
      53.326,   54.213,   55.105,   56.002,   56.907,   57.818,   58.731,   59.644,   60.557,   61.469, 
      62.379,   63.287,   64.192,   65.094,   65.992,   66.888,   67.780,   68.670,   69.557,   70.440, 
      71.322,   72.201,   73.078,   73.953,   74.824,   75.693,   76.561,   77.427,   78.292,   79.155, 
      80.016,   80.877,   81.736,   82.593,   83.449,   84.304, ],
    [  40.617,   40.321,   40.070,   39.844,   39.626,   39.415,   39.211,   39.014,   38.824,   38.642, 
      38.468,   38.303,   38.145,   37.997,   37.859,   37.732,   37.614,   37.507,   37.409,   37.322, 
      37.246,   37.180,   37.124,   37.079,   37.039,   37.007,   36.952,   36.927,   36.929,   36.955, 
      37.006,   37.079,   37.165,   37.267,   37.384,   37.515,   37.660,   37.818,   37.988,   38.169, 
      38.361,   38.562,   38.774,   38.996,   39.228,   39.469,   39.719,   39.978,   40.247,   40.524, 
      40.808,   41.100,   41.399,   41.705,   42.017,   42.336,   42.992,   43.673,   44.375,   45.098, 
      45.841,   46.601,   47.376,   48.165,   48.969,   49.785,   50.613,   51.451,   52.299,   53.155, 
      54.019,   54.890,   55.767,   56.650,   57.541,   58.438,   59.337,   60.238,   61.138,   62.037, 
      62.935,   63.832,   64.725,   65.616,   66.504,   67.389,   68.271,   69.151,   70.027,   70.901, 
      71.774,   72.644,   73.513,   74.379,   75.242,   76.103,   76.963,   77.822,   78.679,   79.535, 
      80.389,   81.243,   82.095,   82.946,   83.796,   84.645, ],
    [  41.914,   41.618,   41.366,   41.140,   40.921,   40.710,   40.505,   40.308,   40.117,   39.935, 
      39.760,   39.594,   39.436,   39.287,   39.148,   39.020,   38.902,   38.794,   38.696,   38.608, 
      38.530,   38.463,   38.407,   38.358,   38.318,   38.284,   38.225,   38.195,   38.190,   38.211, 
      38.256,   38.321,   38.401,   38.495,   38.603,   38.725,   38.861,   39.010,   39.170,   39.341, 
      39.523,   39.714,   39.915,   40.127,   40.347,   40.577,   40.817,   41.065,   41.322,   41.588, 
      41.862,   42.143,   42.431,   42.725,   43.026,   43.334,   43.969,   44.627,   45.308,   46.010, 
      46.732,   47.473,   48.228,   48.998,   49.783,   50.581,   51.391,   52.211,   53.042,   53.881, 
      54.730,   55.586,   56.447,   57.316,   58.192,   59.075,   59.961,   60.848,   61.736,   62.622, 
      63.508,   64.393,   65.275,   66.154,   67.031,   67.905,   68.777,   69.647,   70.513,   71.378, 
      72.241,   73.102,   73.962,   74.819,   75.674,   76.527,   77.379,   78.229,   79.079,   79.927, 
      80.775,   81.622,   82.467,   83.312,   84.155,   84.998, ],
    [  43.209,   42.912,   42.661,   42.433,   42.214,   42.002,   41.797,   41.599,   41.408,   41.225, 
      41.049,   40.883,   40.724,   40.575,   40.436,   40.307,   40.188,   40.079,   39.980,   39.892, 
      39.814,   39.746,   39.686,   39.637,   39.595,   39.560,   39.498,   39.462,   39.452,   39.467, 
      39.506,   39.565,   39.638,   39.724,   39.825,   39.939,   40.066,   40.206,   40.357,   40.519, 
      40.691,   40.872,   41.064,   41.265,   41.475,   41.695,   41.923,   42.161,   42.408,   42.663, 
      42.926,   43.197,   43.474,   43.757,   44.048,   44.345,   44.958,   45.596,   46.256,   46.937, 
      47.639,   48.360,   49.096,   49.846,   50.613,   51.393,   52.185,   52.988,   53.802,   54.625, 
      55.457,   56.297,   57.144,   57.997,   58.860,   59.728,   60.601,   61.475,   62.349,   63.224, 
      64.097,   64.969,   65.840,   66.708,   67.574,   68.437,   69.298,   70.157,   71.014,   71.868, 
      72.722,   73.574,   74.424,   75.273,   76.119,   76.963,   77.807,   78.650,   79.492,   80.333, 
      81.173,   82.013,   82.851,   83.689,   84.526,   85.362, ],
    [  44.502,   44.205,   43.952,   43.725,   43.505,   43.293,   43.087,   42.889,   42.697,   42.513, 
      42.337,   42.170,   42.011,   41.861,   41.722,   41.592,   41.473,   41.364,   41.264,   41.175, 
      41.095,   41.025,   40.965,   40.915,   40.872,   40.835,   40.769,   40.728,   40.714,   40.724, 
      40.758,   40.811,   40.877,   40.956,   41.049,   41.155,   41.275,   41.406,   41.549,   41.702, 
      41.865,   42.037,   42.218,   42.410,   42.610,   42.820,   43.039,   43.266,   43.503,   43.748, 
      44.000,   44.260,   44.527,   44.801,   45.081,   45.367,   45.960,   46.577,   47.217,   47.878, 
      48.560,   49.261,   49.978,   50.710,   51.458,   52.220,   52.995,   53.781,   54.578,   55.384, 
      56.201,   57.025,   57.856,   58.695,   59.543,   60.398,   61.257,   62.117,   62.979,   63.840, 
      64.702,   65.562,   66.420,   67.277,   68.131,   68.983,   69.833,   70.682,   71.528,   72.373, 
      73.216,   74.059,   74.900,   75.740,   76.577,   77.413,   78.249,   79.083,   79.917,   80.751, 
      81.583,   82.416,   83.247,   84.078,   84.908,   85.738, ],
    [  45.792,   45.495,   45.242,   45.014,   44.794,   44.581,   44.375,   44.176,   43.984,   43.800, 
      43.624,   43.456,   43.297,   43.147,   43.006,   42.876,   42.757,   42.647,   42.547,   42.455, 
      42.375,   42.304,   42.243,   42.191,   42.147,   42.110,   42.040,   41.995,   41.975,   41.981, 
      42.010,   42.058,   42.117,   42.190,   42.276,   42.375,   42.487,   42.610,   42.745,   42.889, 
      43.043,   43.206,   43.379,   43.561,   43.752,   43.952,   44.161,   44.379,   44.606,   44.841, 
      45.084,   45.334,   45.591,   45.854,   46.124,   46.400,   46.973,   47.570,   48.190,   48.832, 
      49.495,   50.177,   50.875,   51.588,   52.318,   53.062,   53.820,   54.589,   55.369,   56.159, 
      56.959,   57.769,   58.585,   59.409,   60.242,   61.083,   61.928,   62.775,   63.624,   64.472, 
      65.321,   66.169,   67.015,   67.860,   68.703,   69.543,   70.383,   71.221,   72.057,   72.891, 
      73.725,   74.558,   75.390,   76.220,   77.048,   77.876,   78.703,   79.529,   80.355,   81.181, 
      82.006,   82.831,   83.655,   84.479,   85.302,   86.125, ],
    [  47.081,   46.783,   46.530,   46.302,   46.081,   45.868,   45.662,   45.462,   45.270,   45.085, 
      44.909,   44.741,   44.581,   44.431,   44.290,   44.160,   44.039,   43.928,   43.826,   43.734, 
      43.653,   43.581,   43.520,   43.467,   43.422,   43.384,   43.310,   43.261,   43.237,   43.238, 
      43.263,   43.305,   43.359,   43.425,   43.505,   43.597,   43.701,   43.818,   43.944,   44.081, 
      44.226,   44.381,   44.545,   44.718,   44.900,   45.091,   45.291,   45.499,   45.716,   45.942, 
      46.175,   46.416,   46.663,   46.917,   47.177,   47.444,   47.997,   48.575,   49.175,   49.798, 
      50.442,   51.105,   51.785,   52.480,   53.192,   53.919,   54.659,   55.411,   56.174,   56.948, 
      57.733,   58.527,   59.328,   60.137,   60.956,   61.783,   62.614,   63.448,   64.283,   65.119, 
      65.955,   66.790,   67.625,   68.458,   69.289,   70.118,   70.947,   71.774,   72.599,   73.423, 
      74.247,   75.070,   75.892,   76.713,   77.532,   78.351,   79.169,   79.987,   80.805,   81.622, 
      82.440,   83.257,   84.074,   84.891,   85.707,   86.523, ],
    [  48.368,   48.070,   47.817,   47.588,   47.367,   47.154,   46.947,   46.747,   46.555,   46.370, 
      46.193,   46.024,   45.864,   45.714,   45.572,   45.440,   45.318,   45.206,   45.104,   45.012, 
      44.930,   44.858,   44.795,   44.742,   44.696,   44.657,   44.579,   44.527,   44.499,   44.495, 
      44.516,   44.554,   44.602,   44.662,   44.736,   44.821,   44.919,   45.028,   45.147,   45.276, 
      45.413,   45.560,   45.715,   45.880,   46.053,   46.235,   46.426,   46.626,   46.834,   47.051, 
      47.275,   47.506,   47.744,   47.988,   48.239,   48.496,   49.030,   49.589,   50.171,   50.775, 
      51.401,   52.046,   52.707,   53.385,   54.079,   54.788,   55.511,   56.247,   56.994,   57.752, 
      58.521,   59.299,   60.085,   60.879,   61.684,   62.497,   63.314,   64.135,   64.957,   65.780, 
      66.603,   67.426,   68.248,   69.069,   69.889,   70.706,   71.524,   72.340,   73.155,   73.968, 
      74.782,   75.595,   76.407,   77.219,   78.029,   78.838,   79.648,   80.458,   81.267,   82.076, 
      82.886,   83.695,   84.504,   85.314,   86.123,   86.932, ],
    [  49.654,   49.355,   49.102,   48.873,   48.652,   48.438,   48.231,   48.031,   47.838,   47.653, 
      47.476,   47.306,   47.145,   46.993,   46.851,   46.719,   46.597,   46.485,   46.382,   46.289, 
      46.206,   46.133,   46.070,   46.016,   45.969,   45.929,   45.847,   45.792,   45.760,   45.753, 
      45.769,   45.803,   45.846,   45.901,   45.968,   46.047,   46.138,   46.241,   46.353,   46.474, 
      46.604,   46.743,   46.890,   47.046,   47.211,   47.385,   47.568,   47.759,   47.958,   48.166, 
      48.381,   48.604,   48.832,   49.067,   49.309,   49.557,   50.073,   50.614,   51.177,   51.763, 
      52.371,   52.998,   53.641,   54.301,   54.978,   55.671,   56.377,   57.096,   57.827,   58.569, 
      59.322,   60.085,   60.856,   61.635,   62.426,   63.224,   64.028,   64.835,   65.644,   66.454, 
      67.265,   68.075,   68.885,   69.694,   70.502,   71.308,   72.114,   72.920,   73.723,   74.526, 
      75.329,   76.132,   76.935,   77.737,   78.538,   79.338,   80.139,   80.940,   81.740,   82.541, 
      83.343,   84.144,   84.946,   85.748,   86.550,   87.352, ],
    [  50.938,   50.640,   50.386,   50.156,   49.935,   49.721,   49.514,   49.313,   49.120,   48.933, 
      48.755,   48.585,   48.424,   48.272,   48.129,   47.997,   47.874,   47.762,   47.658,   47.565, 
      47.482,   47.408,   47.344,   47.289,   47.242,   47.200,   47.114,   47.057,   47.021,   47.010, 
      47.023,   47.053,   47.090,   47.140,   47.201,   47.275,   47.360,   47.456,   47.561,   47.675, 
      47.798,   47.929,   48.069,   48.217,   48.374,   48.540,   48.714,   48.897,   49.088,   49.288, 
      49.494,   49.708,   49.928,   50.154,   50.387,   50.626,   51.124,   51.647,   52.193,   52.761, 
      53.351,   53.960,   54.587,   55.229,   55.889,   56.565,   57.255,   57.957,   58.672,   59.398, 
      60.136,   60.884,   61.640,   62.405,   63.181,   63.966,   64.756,   65.549,   66.345,   67.142, 
      67.939,   68.737,   69.535,   70.332,   71.128,   71.923,   72.717,   73.511,   74.304,   75.096, 
      75.889,   76.682,   77.475,   78.267,   79.058,   79.849,   80.641,   81.433,   82.225,   83.018, 
      83.811,   84.605,   85.399,   86.193,   86.987,   87.782, ],
    [  52.221,   51.923,   51.669,   51.439,   51.217,   51.002,   50.794,   50.593,   50.399,   50.212, 
      50.034,   49.864,   49.702,   49.549,   49.406,   49.274,   49.151,   49.037,   48.934,   48.840, 
      48.756,   48.682,   48.617,   48.562,   48.513,   48.470,   48.381,   48.321,   48.282,   48.268, 
      48.278,   48.303,   48.336,   48.380,   48.436,   48.504,   48.583,   48.673,   48.772,   48.879, 
      48.995,   49.119,   49.251,   49.392,   49.542,   49.699,   49.866,   50.040,   50.224,   50.415, 
      50.613,   50.819,   51.030,   51.248,   51.472,   51.703,   52.184,   52.689,   53.218,   53.768, 
      54.341,   54.933,   55.543,   56.168,   56.812,   57.471,   58.144,   58.831,   59.529,   60.240, 
      60.963,   61.695,   62.436,   63.187,   63.949,   64.719,   65.496,   66.276,   67.058,   67.842, 
      68.627,   69.413,   70.198,   70.983,   71.766,   72.550,   73.333,   74.116,   74.898,   75.679, 
      76.461,   77.244,   78.027,   78.809,   79.591,   80.372,   81.155,   81.938,   82.721,   83.505, 
      84.290,   85.076,   85.862,   86.649,   87.435,   88.223, ],
    [  53.502,   53.203,   52.948,   52.718,   52.496,   52.281,   52.072,   51.871,   51.677,   51.490, 
      51.311,   51.141,   50.979,   50.826,   50.683,   50.549,   50.426,   50.312,   50.208,   50.114, 
      50.030,   49.955,   49.890,   49.833,   49.783,   49.739,   49.648,   49.585,   49.543,   49.525, 
      49.532,   49.553,   49.582,   49.621,   49.672,   49.734,   49.808,   49.892,   49.985,   50.086, 
      50.195,   50.312,   50.437,   50.571,   50.713,   50.863,   51.022,   51.189,   51.364,   51.547, 
      51.738,   51.935,   52.138,   52.348,   52.564,   52.786,   53.250,   53.739,   54.250,   54.784, 
      55.340,   55.916,   56.508,   57.118,   57.744,   58.387,   59.044,   59.715,   60.398,   61.094, 
      61.801,   62.519,   63.245,   63.981,   64.729,   65.486,   66.248,   67.015,   67.784,   68.555, 
      69.327,   70.100,   70.873,   71.646,   72.417,   73.189,   73.961,   74.732,   75.503,   76.273, 
      77.045,   77.818,   78.590,   79.362,   80.134,   80.906,   81.680,   82.454,   83.229,   84.004, 
      84.781,   85.558,   86.336,   87.115,   87.894,   88.674, ],
    [  54.781,   54.482,   54.227,   53.996,   53.774,   53.559,   53.350,   53.149,   52.954,   52.767, 
      52.588,   52.417,   52.255,   52.101,   51.958,   51.824,   51.700,   51.587,   51.482,   51.388, 
      51.303,   51.227,   51.161,   51.103,   51.052,   51.006,   50.914,   50.848,   50.804,   50.783, 
      50.787,   50.804,   50.828,   50.863,   50.909,   50.966,   51.034,   51.112,   51.199,   51.294, 
      51.397,   51.507,   51.626,   51.752,   51.887,   52.030,   52.182,   52.341,   52.509,   52.685, 
      52.867,   53.057,   53.252,   53.454,   53.662,   53.876,   54.324,   54.796,   55.291,   55.809, 
      56.348,   56.907,   57.483,   58.076,   58.687,   59.314,   59.955,   60.610,   61.278,   61.958, 
      62.651,   63.354,   64.065,   64.787,   65.521,   66.264,   67.013,   67.766,   68.522,   69.280, 
      70.040,   70.800,   71.560,   72.321,   73.080,   73.840,   74.600,   75.360,   76.120,   76.880, 
      77.640,   78.402,   79.165,   79.927,   80.689,   81.452,   82.216,   82.981,   83.746,   84.513, 
      85.281,   86.050,   86.820,   87.591,   88.362,   89.135, ],
    [  56.059,   55.760,   55.505,   55.274,   55.051,   54.836,   54.627,   54.425,   54.230,   54.043, 
      53.863,   53.692,   53.530,   53.376,   53.232,   53.098,   52.974,   52.860,   52.755,   52.660, 
      52.574,   52.498,   52.431,   52.373,   52.320,   52.274,   52.180,   52.110,   52.064,   52.040, 
      52.041,   52.056,   52.075,   52.105,   52.146,   52.198,   52.261,   52.334,   52.416,   52.505, 
      52.601,   52.705,   52.817,   52.937,   53.065,   53.201,   53.345,   53.497,   53.658,   53.826, 
      54.002,   54.184,   54.372,   54.566,   54.766,   54.972,   55.404,   55.860,   56.339,   56.841, 
      57.364,   57.907,   58.467,   59.044,   59.639,   60.250,   60.876,   61.516,   62.169,   62.834, 
      63.511,   64.199,   64.896,   65.604,   66.324,   67.053,   67.789,   68.529,   69.272,   70.017, 
      70.763,   71.511,   72.259,   73.007,   73.755,   74.503,   75.251,   76.000,   76.748,   77.497, 
      78.247,   78.999,   79.751,   80.503,   81.255,   82.008,   82.762,   83.518,   84.275,   85.033, 
      85.792,   86.553,   87.315,   88.077,   88.841,   89.606, ],
    [  57.336,   57.036,   56.781,   56.550,   56.327,   56.111,   55.902,   55.700,   55.505,   55.317, 
      55.138,   54.967,   54.804,   54.650,   54.506,   54.371,   54.247,   54.132,   54.027,   53.931, 
      53.845,   53.768,   53.700,   53.640,   53.588,   53.541,   53.445,   53.372,   53.324,   53.298, 
      53.296,   53.307,   53.322,   53.348,   53.385,   53.432,   53.490,   53.558,   53.634,   53.717, 
      53.807,   53.905,   54.010,   54.124,   54.245,   54.375,   54.512,   54.658,   54.811,   54.972, 
      55.141,   55.315,   55.496,   55.682,   55.875,   56.074,   56.490,   56.931,   57.394,   57.880, 
      58.388,   58.915,   59.460,   60.021,   60.600,   61.196,   61.807,   62.431,   63.069,   63.719, 
      64.382,   65.056,   65.738,   66.432,   67.138,   67.854,   68.576,   69.303,   70.032,   70.764, 
      71.498,   72.233,   72.969,   73.705,   74.441,   75.176,   75.913,   76.651,   77.388,   78.125, 
      78.865,   79.605,   80.347,   81.089,   81.831,   82.575,   83.319,   84.066,   84.814,   85.563, 
      86.313,   87.066,   87.819,   88.574,   89.329,   90.086, ],
    [  58.612,   58.312,   58.056,   57.825,   57.602,   57.386,   57.177,   56.975,   56.779,   56.592, 
      56.412,   56.240,   56.077,   55.923,   55.778,   55.644,   55.519,   55.404,   55.298,   55.202, 
      55.115,   55.037,   54.968,   54.908,   54.855,   54.807,   54.709,   54.633,   54.584,   54.555, 
      54.551,   54.559,   54.569,   54.591,   54.624,   54.666,   54.719,   54.782,   54.853,   54.931, 
      55.015,   55.107,   55.206,   55.314,   55.429,   55.551,   55.682,   55.821,   55.968,   56.122, 
      56.283,   56.451,   56.624,   56.804,   56.989,   57.180,   57.582,   58.007,   58.456,   58.926, 
      59.419,   59.931,   60.460,   61.006,   61.570,   62.151,   62.746,   63.356,   63.979,   64.614, 
      65.262,   65.922,   66.591,   67.270,   67.962,   68.665,   69.374,   70.087,   70.804,   71.523, 
      72.244,   72.967,   73.690,   74.414,   75.137,   75.861,   76.586,   77.312,   78.038,   78.764, 
      79.493,   80.223,   80.954,   81.686,   82.418,   83.152,   83.887,   84.624,   85.362,   86.103, 
      86.844,   87.588,   88.333,   89.080,   89.827,   90.577, ],
    [  59.887,   59.587,   59.331,   59.100,   58.876,   58.660,   58.451,   58.248,   58.053,   57.865, 
      57.685,   57.513,   57.350,   57.195,   57.050,   56.915,   56.790,   56.674,   56.568,   56.471, 
      56.383,   56.304,   56.235,   56.175,   56.121,   56.073,   55.972,   55.895,   55.843,   55.812, 
      55.806,   55.810,   55.817,   55.835,   55.863,   55.901,   55.950,   56.008,   56.073,   56.146, 
      56.225,   56.311,   56.404,   56.506,   56.614,   56.731,   56.855,   56.987,   57.128,   57.276, 
      57.430,   57.591,   57.757,   57.929,   58.107,   58.292,   58.679,   59.090,   59.523,   59.978, 
      60.456,   60.954,   61.468,   61.999,   62.548,   63.114,   63.694,   64.289,   64.897,   65.519, 
      66.152,   66.798,   67.452,   68.118,   68.797,   69.486,   70.182,   70.882,   71.586,   72.292, 
      73.000,   73.710,   74.421,   75.133,   75.844,   76.556,   77.270,   77.984,   78.698,   79.414, 
      80.131,   80.851,   81.572,   82.293,   83.015,   83.739,   84.464,   85.192,   85.921,   86.652, 
      87.385,   88.120,   88.857,   89.595,   90.335,   91.076, ],
    [  61.161,   60.861,   60.605,   60.373,   60.150,   59.934,   59.724,   59.521,   59.326,   59.137, 
      58.957,   58.785,   58.621,   58.466,   58.321,   58.185,   58.059,   57.943,   57.836,   57.738, 
      57.650,   57.572,   57.502,   57.441,   57.387,   57.338,   57.235,   57.156,   57.102,   57.069, 
      57.061,   57.062,   57.065,   57.079,   57.103,   57.137,   57.181,   57.234,   57.295,   57.362, 
      57.436,   57.516,   57.604,   57.699,   57.802,   57.913,   58.031,   58.157,   58.291,   58.432, 
      58.580,   58.734,   58.894,   59.059,   59.230,   59.407,   59.780,   60.177,   60.596,   61.037, 
      61.500,   61.983,   62.482,   62.999,   63.533,   64.084,   64.651,   65.231,   65.825,   66.432, 
      67.052,   67.683,   68.324,   68.976,   69.641,   70.317,   70.999,   71.687,   72.378,   73.071, 
      73.767,   74.464,   75.163,   75.862,   76.561,   77.261,   77.963,   78.666,   79.369,   80.073, 
      80.780,   81.489,   82.199,   82.910,   83.622,   84.336,   85.051,   85.770,   86.490,   87.211, 
      87.936,   88.662,   89.390,   90.120,   90.851,   91.585, ],
    [  62.434,   62.134,   61.878,   61.646,   61.423,   61.206,   60.996,   60.793,   60.597,   60.408, 
      60.228,   60.055,   59.891,   59.736,   59.590,   59.454,   59.328,   59.211,   59.103,   59.006, 
      58.917,   58.839,   58.769,   58.707,   58.652,   58.602,   58.497,   58.417,   58.361,   58.326, 
      58.316,   58.313,   58.313,   58.323,   58.343,   58.373,   58.413,   58.462,   58.518,   58.580, 
      58.649,   58.723,   58.806,   58.895,   58.992,   59.097,   59.209,   59.329,   59.457,   59.592, 
      59.733,   59.881,   60.034,   60.193,   60.357,   60.528,   60.887,   61.270,   61.675,   62.101, 
      62.550,   63.019,   63.504,   64.006,   64.526,   65.063,   65.615,   66.181,   66.761,   67.353, 
      67.959,   68.577,   69.204,   69.842,   70.494,   71.157,   71.827,   72.501,   73.179,   73.860, 
      74.543,   75.228,   75.914,   76.601,   77.289,   77.977,   78.667,   79.358,   80.050,   80.743, 
      81.438,   82.136,   82.836,   83.537,   84.239,   84.942,   85.648,   86.357,   87.067,   87.780, 
      88.495,   89.213,   89.932,   90.654,   91.377,   92.102, ],
    [  63.707,   63.406,   63.150,   62.918,   62.694,   62.477,   62.267,   62.064,   61.867,   61.678, 
      61.497,   61.325,   61.160,   61.004,   60.858,   60.722,   60.595,   60.478,   60.371,   60.273, 
      60.184,   60.105,   60.034,   59.972,   59.916,   59.866,   59.758,   59.677,   59.619,   59.583, 
      59.570,   59.565,   59.561,   59.567,   59.584,   59.609,   59.645,   59.690,   59.741,   59.799, 
      59.862,   59.932,   60.009,   60.093,   60.184,   60.283,   60.389,   60.503,   60.625,   60.754, 
      60.890,   61.031,   61.178,   61.330,   61.488,   61.652,   61.997,   62.367,   62.758,   63.171, 
      63.606,   64.061,   64.531,   65.019,   65.525,   66.048,   66.586,   67.138,   67.704,   68.283, 
      68.875,   69.479,   70.093,   70.718,   71.357,   72.006,   72.663,   73.325,   73.990,   74.658, 
      75.329,   76.001,   76.675,   77.350,   78.025,   78.702,   79.380,   80.060,   80.740,   81.422, 
      82.106,   82.794,   83.483,   84.173,   84.865,   85.558,   86.254,   86.953,   87.654,   88.358, 
      89.064,   89.773,   90.484,   91.197,   91.911,   92.629, ],
    [  64.978,   64.677,   64.421,   64.189,   63.964,   63.747,   63.537,   63.333,   63.136,   62.947, 
      62.766,   62.593,   62.428,   62.272,   62.125,   61.989,   61.862,   61.745,   61.637,   61.539, 
      61.450,   61.370,   61.299,   61.236,   61.180,   61.129,   61.020,   60.938,   60.877,   60.840, 
      60.825,   60.817,   60.809,   60.812,   60.824,   60.846,   60.878,   60.919,   60.966,   61.019, 
      61.077,   61.141,   61.213,   61.292,   61.377,   61.471,   61.571,   61.680,   61.796,   61.919, 
      62.049,   62.184,   62.325,   62.470,   62.622,   62.779,   63.112,   63.468,   63.846,   64.245, 
      64.667,   65.108,   65.565,   66.039,   66.531,   67.040,   67.565,   68.103,   68.655,   69.220, 
      69.799,   70.390,   70.990,   71.602,   72.227,   72.864,   73.508,   74.157,   74.810,   75.466, 
      76.124,   76.784,   77.446,   78.108,   78.772,   79.436,   80.102,   80.771,   81.439,   82.110, 
      82.784,   83.460,   84.138,   84.818,   85.500,   86.183,   86.869,   87.559,   88.251,   88.945, 
      89.642,   90.342,   91.044,   91.748,   92.455,   93.164, ],
    [  66.248,   65.947,   65.691,   65.458,   65.233,   65.016,   64.805,   64.601,   64.404,   64.215, 
      64.033,   63.860,   63.695,   63.539,   63.392,   63.256,   63.129,   63.012,   62.903,   62.805, 
      62.715,   62.635,   62.563,   62.500,   62.443,   62.390,   62.281,   62.197,   62.135,   62.097, 
      62.080,   62.068,   62.057,   62.056,   62.065,   62.084,   62.112,   62.148,   62.191,   62.239, 
      62.293,   62.352,   62.418,   62.492,   62.573,   62.660,   62.756,   62.858,   62.969,   63.086, 
      63.210,   63.340,   63.474,   63.614,   63.759,   63.910,   64.230,   64.574,   64.938,   65.325, 
      65.733,   66.161,   66.604,   67.065,   67.543,   68.039,   68.550,   69.075,   69.613,   70.165, 
      70.731,   71.308,   71.895,   72.494,   73.106,   73.730,   74.362,   74.998,   75.638,   76.282, 
      76.927,   77.575,   78.225,   78.875,   79.527,   80.179,   80.834,   81.491,   82.148,   82.807, 
      83.470,   84.136,   84.803,   85.473,   86.144,   86.817,   87.494,   88.173,   88.855,   89.540, 
      90.228,   90.919,   91.613,   92.308,   93.006,   93.707, ],
    [  67.517,   67.216,   66.959,   66.726,   66.501,   66.284,   66.073,   65.869,   65.672,   65.482, 
      65.300,   65.127,   64.962,   64.806,   64.659,   64.522,   64.395,   64.277,   64.169,   64.069, 
      63.980,   63.899,   63.827,   63.762,   63.704,   63.652,   63.542,   63.457,   63.392,   63.353, 
      63.335,   63.319,   63.305,   63.301,   63.306,   63.321,   63.346,   63.378,   63.417,   63.460, 
      63.509,   63.564,   63.625,   63.694,   63.769,   63.852,   63.942,   64.039,   64.144,   64.256, 
      64.374,   64.498,   64.627,   64.760,   64.899,   65.044,   65.352,   65.683,   66.035,   66.408, 
      66.804,   67.218,   67.649,   68.096,   68.562,   69.044,   69.541,   70.053,   70.578,   71.117, 
      71.669,   72.233,   72.807,   73.393,   73.993,   74.605,   75.223,   75.847,   76.475,   77.106, 
      77.740,   78.375,   79.013,   79.651,   80.290,   80.931,   81.574,   82.219,   82.865,   83.514, 
      84.165,   84.820,   85.477,   86.136,   86.797,   87.460,   88.126,   88.796,   89.469,   90.144, 
      90.823,   91.505,   92.190,   92.877,   93.567,   94.259, ],
    [  68.785,   68.483,   68.226,   67.993,   67.769,   67.551,   67.340,   67.136,   66.939,   66.749, 
      66.567,   66.394,   66.228,   66.072,   65.925,   65.788,   65.660,   65.542,   65.433,   65.334, 
      65.243,   65.162,   65.089,   65.024,   64.966,   64.913,   64.802,   64.716,   64.650,   64.610, 
      64.589,   64.570,   64.552,   64.545,   64.548,   64.559,   64.580,   64.609,   64.643,   64.682, 
      64.727,   64.776,   64.833,   64.896,   64.967,   65.044,   65.129,   65.221,   65.321,   65.428, 
      65.540,   65.658,   65.781,   65.909,   66.043,   66.182,   66.477,   66.796,   67.135,   67.496, 
      67.879,   68.281,   68.699,   69.133,   69.585,   70.054,   70.539,   71.037,   71.550,   72.075, 
      72.615,   73.166,   73.727,   74.300,   74.888,   75.487,   76.093,   76.705,   77.320,   77.939, 
      78.560,   79.184,   79.809,   80.435,   81.063,   81.692,   82.323,   82.957,   83.591,   84.228, 
      84.869,   85.513,   86.159,   86.807,   87.458,   88.111,   88.767,   89.428,   90.091,   90.757, 
      91.427,   92.099,   92.775,   93.454,   94.135,   94.819, ],
    [  70.052,   69.751,   69.494,   69.260,   69.036,   68.818,   68.607,   68.402,   68.205,   68.015, 
      67.833,   67.659,   67.494,   67.337,   67.190,   67.052,   66.924,   66.806,   66.697,   66.597, 
      66.506,   66.424,   66.351,   66.286,   66.227,   66.175,   66.062,   65.974,   65.907,   65.866, 
      65.844,   65.820,   65.800,   65.790,   65.789,   65.797,   65.814,   65.839,   65.870,   65.905, 
      65.945,   65.990,   66.042,   66.100,   66.166,   66.238,   66.318,   66.405,   66.500,   66.601, 
      66.709,   66.821,   66.939,   67.061,   67.188,   67.322,   67.605,   67.912,   68.240,   68.588, 
      68.959,   69.348,   69.753,   70.174,   70.614,   71.071,   71.542,   72.028,   72.527,   73.040, 
      73.567,   74.105,   74.654,   75.214,   75.789,   76.376,   76.970,   77.570,   78.173,   78.779, 
      79.388,   80.000,   80.613,   81.228,   81.843,   82.460,   83.080,   83.702,   84.325,   84.951, 
      85.581,   86.214,   86.849,   87.487,   88.127,   88.770,   89.417,   90.067,   90.721,   91.377, 
      92.038,   92.702,   93.369,   94.038,   94.711,   95.387, ],
    [  71.319,   71.017,   70.760,   70.527,   70.302,   70.084,   69.873,   69.668,   69.471,   69.281, 
      69.098,   68.924,   68.758,   68.601,   68.454,   68.316,   68.188,   68.069,   67.960,   67.859, 
      67.768,   67.686,   67.612,   67.547,   67.489,   67.436,   67.322,   67.232,   67.165,   67.123, 
      67.098,   67.070,   67.047,   67.035,   67.030,   67.035,   67.049,   67.071,   67.097,   67.128, 
      67.163,   67.204,   67.251,   67.305,   67.366,   67.434,   67.508,   67.590,   67.680,   67.777, 
      67.879,   67.986,   68.098,   68.215,   68.337,   68.464,   68.737,   69.031,   69.347,   69.684, 
      70.042,   70.419,   70.812,   71.221,   71.648,   72.092,   72.551,   73.024,   73.511,   74.011, 
      74.525,   75.051,   75.587,   76.135,   76.698,   77.272,   77.854,   78.442,   79.033,   79.627, 
      80.224,   80.824,   81.425,   82.028,   82.631,   83.237,   83.845,   84.456,   85.068,   85.682, 
      86.301,   86.923,   87.548,   88.175,   88.805,   89.438,   90.074,   90.715,   91.359,   92.006, 
      92.657,   93.312,   93.970,   94.631,   95.295,   95.963, ],
    [  72.585,   72.283,   72.026,   71.793,   71.568,   71.349,   71.138,   70.933,   70.736,   70.545, 
      70.363,   70.189,   70.022,   69.865,   69.717,   69.579,   69.451,   69.332,   69.221,   69.121, 
      69.029,   68.947,   68.874,   68.808,   68.749,   68.696,   68.581,   68.490,   68.422,   68.379, 
      68.352,   68.321,   68.295,   68.279,   68.271,   68.273,   68.284,   68.302,   68.324,   68.351, 
      68.382,   68.419,   68.461,   68.511,   68.567,   68.630,   68.700,   68.777,   68.862,   68.953, 
      69.051,   69.153,   69.259,   69.371,   69.487,   69.609,   69.870,   70.154,   70.458,   70.783, 
      71.129,   71.495,   71.875,   72.272,   72.687,   73.118,   73.565,   74.026,   74.500,   74.988, 
      75.489,   76.003,   76.527,   77.063,   77.613,   78.176,   78.746,   79.321,   79.900,   80.482, 
      81.068,   81.655,   82.245,   82.835,   83.427,   84.021,   84.618,   85.217,   85.818,   86.421, 
      87.028,   87.640,   88.254,   88.871,   89.490,   90.113,   90.740,   91.370,   92.005,   92.642, 
      93.284,   93.930,   94.579,   95.231,   95.887,   96.546, ],
    [  73.851,   73.549,   73.291,   73.058,   72.832,   72.614,   72.402,   72.198,   72.000,   71.809, 
      71.626,   71.452,   71.285,   71.128,   70.980,   70.841,   70.712,   70.593,   70.483,   70.382, 
      70.291,   70.208,   70.135,   70.069,   70.010,   69.956,   69.840,   69.747,   69.679,   69.634, 
      69.606,   69.571,   69.542,   69.523,   69.513,   69.511,   69.519,   69.534,   69.552,   69.575, 
      69.602,   69.634,   69.672,   69.717,   69.769,   69.827,   69.893,   69.965,   70.045,   70.132, 
      70.224,   70.321,   70.423,   70.529,   70.640,   70.756,   71.006,   71.279,   71.572,   71.885, 
      72.220,   72.574,   72.942,   73.327,   73.730,   74.149,   74.584,   75.032,   75.495,   75.970, 
      76.460,   76.961,   77.473,   77.996,   78.535,   79.086,   79.644,   80.207,   80.774,   81.345, 
      81.918,   82.494,   83.071,   83.650,   84.231,   84.813,   85.398,   85.986,   86.575,   87.167, 
      87.764,   88.364,   88.968,   89.574,   90.183,   90.796,   91.412,   92.033,   92.658,   93.286, 
      93.919,   94.555,   95.196,   95.839,   96.486,   97.137, ],
    [  75.115,   74.813,   74.556,   74.322,   74.097,   73.878,   73.666,   73.461,   73.263,   73.072, 
      72.889,   72.714,   72.548,   72.390,   72.241,   72.103,   71.974,   71.855,   71.744,   71.644, 
      71.552,   71.469,   71.396,   71.330,   71.270,   71.216,   71.098,   71.005,   70.936,   70.890, 
      70.860,   70.820,   70.789,   70.767,   70.754,   70.749,   70.754,   70.765,   70.780,   70.799, 
      70.822,   70.850,   70.884,   70.924,   70.971,   71.025,   71.086,   71.154,   71.230,   71.312, 
      71.399,   71.491,   71.588,   71.688,   71.794,   71.905,   72.145,   72.407,   72.688,   72.990, 
      73.314,   73.656,   74.013,   74.386,   74.777,   75.185,   75.607,   76.044,   76.494,   76.958, 
      77.435,   77.925,   78.424,   78.936,   79.463,   80.002,   80.548,   81.100,   81.655,   82.214, 
      82.775,   83.339,   83.905,   84.473,   85.041,   85.612,   86.185,   86.762,   87.340,   87.921, 
      88.506,   89.096,   89.689,   90.285,   90.884,   91.486,   92.093,   92.704,   93.319,   93.938, 
      94.561,   95.188,   95.820,   96.454,   97.093,   97.735, ],
    [  76.379,   76.077,   75.819,   75.586,   75.360,   75.141,   74.929,   74.723,   74.525,   74.334, 
      74.151,   73.976,   73.809,   73.651,   73.503,   73.364,   73.235,   73.116,   73.006,   72.905, 
      72.813,   72.730,   72.656,   72.590,   72.530,   72.475,   72.356,   72.262,   72.192,   72.146, 
      72.114,   72.070,   72.036,   72.011,   71.995,   71.988,   71.990,   71.997,   72.009,   72.024, 
      72.042,   72.066,   72.095,   72.132,   72.175,   72.224,   72.281,   72.344,   72.415,   72.493, 
      72.575,   72.663,   72.754,   72.850,   72.951,   73.057,   73.286,   73.537,   73.808,   74.099, 
      74.412,   74.742,   75.088,   75.449,   75.829,   76.225,   76.636,   77.060,   77.499,   77.951, 
      78.416,   78.894,   79.382,   79.881,   80.397,   80.924,   81.459,   81.999,   82.543,   83.090, 
      83.639,   84.192,   84.746,   85.302,   85.858,   86.417,   86.980,   87.545,   88.112,   88.682, 
      89.256,   89.835,   90.418,   91.003,   91.591,   92.183,   92.780,   93.381,   93.987,   94.596, 
      95.210,   95.828,   96.451,   97.077,   97.706,   98.340, ],
    [  77.642,   77.340,   77.082,   76.848,   76.622,   76.403,   76.191,   75.985,   75.787,   75.596, 
      75.412,   75.238,   75.071,   74.913,   74.764,   74.626,   74.497,   74.377,   74.266,   74.165, 
      74.073,   73.990,   73.916,   73.849,   73.789,   73.733,   73.614,   73.519,   73.449,   73.401, 
      73.366,   73.319,   73.282,   73.255,   73.236,   73.226,   73.225,   73.229,   73.237,   73.248, 
      73.263,   73.282,   73.308,   73.340,   73.379,   73.424,   73.476,   73.535,   73.602,   73.675, 
      73.753,   73.836,   73.922,   74.013,   74.109,   74.210,   74.429,   74.669,   74.929,   75.210, 
      75.512,   75.832,   76.166,   76.516,   76.884,   77.269,   77.668,   78.081,   78.508,   78.948, 
      79.402,   79.868,   80.344,   80.832,   81.336,   81.852,   82.375,   82.904,   83.436,   83.971, 
      84.510,   85.050,   85.593,   86.137,   86.683,   87.230,   87.781,   88.335,   88.891,   89.449, 
      90.013,   90.581,   91.153,   91.728,   92.306,   92.888,   93.475,   94.066,   94.662,   95.262, 
      95.866,   96.476,   97.089,   97.706,   98.327,   98.952, ],
    [  78.904,   78.602,   78.344,   78.110,   77.884,   77.664,   77.452,   77.247,   77.048,   76.857, 
      76.674,   76.499,   76.332,   76.174,   76.025,   75.887,   75.757,   75.638,   75.527,   75.426, 
      75.333,   75.250,   75.175,   75.108,   75.047,   74.992,   74.871,   74.776,   74.706,   74.656, 
      74.617,   74.568,   74.529,   74.499,   74.477,   74.464,   74.461,   74.461,   74.466,   74.473, 
      74.484,   74.499,   74.521,   74.549,   74.583,   74.624,   74.672,   74.727,   74.790,   74.858, 
      74.932,   75.010,   75.092,   75.178,   75.268,   75.365,   75.573,   75.804,   76.054,   76.324, 
      76.615,   76.924,   77.247,   77.586,   77.943,   78.316,   78.704,   79.106,   79.522,   79.950, 
      80.393,   80.848,   81.312,   81.789,   82.281,   82.785,   83.297,   83.815,   84.335,   84.859, 
      85.386,   85.915,   86.446,   86.979,   87.513,   88.049,   88.589,   89.131,   89.676,   90.224, 
      90.777,   91.334,   91.895,   92.460,   93.028,   93.599,   94.176,   94.758,   95.344,   95.934, 
      96.529,   97.129,   97.734,   98.342,   98.954,   99.571, ],
    [  80.166,   79.864,   79.605,   79.371,   79.145,   78.926,   78.714,   78.508,   78.310,   78.118, 
      77.935,   77.760,   77.593,   77.435,   77.286,   77.147,   77.018,   76.898,   76.787,   76.685, 
      76.593,   76.509,   76.434,   76.367,   76.305,   76.249,   76.128,   76.033,   75.962,   75.911, 
      75.867,   75.816,   75.775,   75.742,   75.718,   75.702,   75.696,   75.694,   75.694,   75.698, 
      75.705,   75.716,   75.734,   75.758,   75.788,   75.825,   75.869,   75.920,   75.978,   76.043, 
      76.112,   76.185,   76.263,   76.344,   76.430,   76.521,   76.720,   76.940,   77.180,   77.440, 
      77.721,   78.019,   78.332,   78.660,   79.005,   79.368,   79.745,   80.136,   80.540,   80.957, 
      81.388,   81.832,   82.285,   82.750,   83.231,   83.724,   84.225,   84.731,   85.241,   85.753, 
      86.269,   86.786,   87.306,   87.827,   88.350,   88.875,   89.403,   89.934,   90.468,   91.005, 
      91.547,   92.093,   92.644,   93.198,   93.756,   94.317,   94.884,   95.456,   96.032,   96.613, 
      97.199,   97.790,   98.385,   98.985,   99.588,  100.197, ],
    [  81.427,   81.125,   80.867,   80.633,   80.406,   80.187,   79.975,   79.769,   79.571,   79.379, 
      79.196,   79.021,   78.854,   78.695,   78.546,   78.407,   78.278,   78.158,   78.047,   77.945, 
      77.852,   77.768,   77.692,   77.625,   77.563,   77.507,   77.386,   77.290,   77.218,   77.166, 
      77.118,   77.064,   77.020,   76.985,   76.959,   76.940,   76.931,   76.926,   76.923,   76.923, 
      76.926,   76.933,   76.947,   76.967,   76.994,   77.027,   77.067,   77.113,   77.167,   77.228, 
      77.293,   77.362,   77.435,   77.511,   77.592,   77.679,   77.868,   78.079,   78.308,   78.558, 
      78.829,   79.117,   79.419,   79.736,   80.071,   80.423,   80.789,   81.169,   81.562,   81.968, 
      82.388,   82.820,   83.262,   83.716,   84.186,   84.668,   85.158,   85.653,   86.151,   86.653, 
      87.157,   87.663,   88.172,   88.681,   89.193,   89.706,   90.223,   90.744,   91.266,   91.792, 
      92.323,   92.859,   93.399,   93.943,   94.490,   95.041,   95.598,   96.160,   96.727,   97.299, 
      97.875,   98.457,   99.043,   99.634,  100.229,  100.829, ],
    [  82.689,   82.386,   82.128,   81.894,   81.667,   81.448,   81.236,   81.030,   80.831,   80.640, 
      80.456,   80.281,   80.114,   79.955,   79.806,   79.667,   79.537,   79.417,   79.306,   79.203, 
      79.110,   79.026,   78.950,   78.882,   78.820,   78.764,   78.643,   78.547,   78.474,   78.421, 
      78.368,   78.312,   78.266,   78.229,   78.199,   78.177,   78.166,   78.158,   78.152,   78.148, 
      78.147,   78.150,   78.161,   78.177,   78.200,   78.229,   78.265,   78.307,   78.358,   78.414, 
      78.475,   78.539,   78.608,   78.679,   78.756,   78.838,   79.018,   79.219,   79.439,   79.679, 
      79.939,   80.218,   80.509,   80.816,   81.140,   81.481,   81.836,   82.205,   82.587,   82.983, 
      83.392,   83.813,   84.244,   84.687,   85.146,   85.617,   86.096,   86.580,   87.067,   87.558, 
      88.051,   88.546,   89.043,   89.541,   90.041,   90.543,   91.049,   91.559,   92.070,   92.585, 
      93.105,   93.631,   94.161,   94.694,   95.231,   95.772,   96.319,   96.871,   97.428,   97.991, 
      98.558,   99.131,   99.708,  100.290,  100.876,  101.467, ],
    [  83.950,   83.647,   83.389,   83.154,   82.928,   82.709,   82.496,   82.290,   82.091,   81.900, 
      81.716,   81.541,   81.373,   81.215,   81.066,   80.926,   80.796,   80.676,   80.564,   80.462, 
      80.368,   80.284,   80.208,   80.140,   80.078,   80.021,   79.900,   79.803,   79.729,   79.675, 
      79.617,   79.559,   79.511,   79.472,   79.440,   79.415,   79.401,   79.390,   79.380,   79.373, 
      79.368,   79.368,   79.374,   79.387,   79.406,   79.431,   79.463,   79.502,   79.548,   79.601, 
      79.657,   79.718,   79.782,   79.849,   79.921,   79.998,   80.169,   80.361,   80.571,   80.801, 
      81.052,   81.320,   81.602,   81.898,   82.212,   82.543,   82.887,   83.246,   83.617,   84.001, 
      84.400,   84.810,   85.230,   85.662,   86.111,   86.571,   87.039,   87.512,   87.989,   88.468, 
      88.950,   89.433,   89.919,   90.407,   90.895,   91.387,   91.881,   92.380,   92.880,   93.384, 
      93.894,   94.409,   94.928,   95.451,   95.978,   96.509,   97.046,   97.588,   98.136,   98.689, 
      99.247,   99.810,  100.379,  100.952,  101.529,  102.112, ],
    [  85.210,   84.908,   84.649,   84.415,   84.188,   83.969,   83.756,   83.550,   83.351,   83.160, 
      82.976,   82.800,   82.633,   82.474,   82.325,   82.185,   82.055,   81.934,   81.822,   81.719, 
      81.626,   81.541,   81.465,   81.397,   81.335,   81.279,   81.157,   81.060,   80.985,   80.929, 
      80.866,   80.806,   80.756,   80.714,   80.679,   80.653,   80.636,   80.622,   80.609,   80.598, 
      80.589,   80.585,   80.588,   80.597,   80.612,   80.634,   80.662,   80.697,   80.740,   80.788, 
      80.841,   80.897,   80.957,   81.020,   81.087,   81.160,   81.322,   81.504,   81.705,   81.926, 
      82.167,   82.426,   82.697,   82.983,   83.287,   83.607,   83.941,   84.289,   84.650,   85.024, 
      85.412,   85.811,   86.220,   86.642,   87.080,   87.530,   87.987,   88.449,   88.915,   89.383, 
      89.854,   90.327,   90.801,   91.278,   91.755,   92.235,   92.719,   93.206,   93.696,   94.189, 
      94.688,   95.192,   95.701,   96.214,   96.730,   97.251,   97.779,   98.311,   98.849,   99.393, 
      99.942,  100.496,  101.055,  101.619,  102.188,  102.762, ],
    [  86.470,   86.168,   85.909,   85.675,   85.448,   85.228,   85.016,   84.810,   84.611,   84.419, 
      84.235,   84.059,   83.891,   83.732,   83.583,   83.443,   83.313,   83.192,   83.080,   82.977, 
      82.883,   82.799,   82.723,   82.654,   82.592,   82.536,   82.414,   82.316,   82.240,   82.180, 
      82.115,   82.053,   82.000,   81.956,   81.919,   81.890,   81.871,   81.854,   81.837,   81.823, 
      81.810,   81.803,   81.802,   81.808,   81.819,   81.837,   81.862,   81.893,   81.932,   81.976, 
      82.025,   82.077,   82.133,   82.191,   82.254,   82.323,   82.476,   82.649,   82.841,   83.052, 
      83.284,   83.533,   83.794,   84.070,   84.364,   84.675,   84.999,   85.336,   85.686,   86.050, 
      86.427,   86.816,   87.215,   87.626,   88.053,   88.492,   88.939,   89.391,   89.846,   90.303, 
      90.763,   91.225,   91.688,   92.154,   92.620,   93.089,   93.562,   94.038,   94.517,   94.999, 
      95.488,   95.982,   96.480,   96.982,   97.489,   98.000,   98.517,   99.040,   99.569,  100.103, 
      100.642,  101.187,  101.738,  102.293,  102.853,  103.419, ],
    [  87.730,   87.427,   87.169,   86.934,   86.707,   86.488,   86.275,   86.069,   85.869,   85.677, 
      85.493,   85.317,   85.150,   84.990,   84.841,   84.701,   84.570,   84.449,   84.337,   84.234, 
      84.141,   84.056,   83.980,   83.911,   83.849,   83.793,   83.670,   83.572,   83.495,   83.430, 
      83.363,   83.299,   83.244,   83.198,   83.158,   83.128,   83.106,   83.085,   83.065,   83.048, 
      83.031,   83.020,   83.016,   83.018,   83.026,   83.040,   83.061,   83.089,   83.124,   83.165, 
      83.210,   83.258,   83.309,   83.363,   83.422,   83.486,   83.631,   83.795,   83.978,   84.180, 
      84.403,   84.642,   84.894,   85.160,   85.444,   85.745,   86.059,   86.386,   86.726,   87.079, 
      87.446,   87.825,   88.213,   88.613,   89.030,   89.459,   89.896,   90.337,   90.781,   91.227, 
      91.677,   92.128,   92.580,   93.035,   93.490,   93.948,   94.410,   94.875,   95.343,   95.815, 
      96.293,   96.776,   97.264,   97.757,   98.253,   98.754,   99.261,   99.775,  100.294,  100.818, 
      101.349,  101.885,  102.426,  102.973,  103.524,  104.081, ],
    [  88.989,   88.686,   88.428,   88.193,   87.966,   87.746,   87.533,   87.327,   87.128,   86.935, 
      86.751,   86.575,   86.407,   86.248,   86.098,   85.958,   85.828,   85.707,   85.595,   85.492, 
      85.398,   85.313,   85.237,   85.169,   85.106,   85.050,   84.927,   84.828,   84.750,   84.681, 
      84.611,   84.545,   84.488,   84.439,   84.398,   84.365,   84.341,   84.316,   84.293,   84.272, 
      84.252,   84.238,   84.230,   84.229,   84.233,   84.243,   84.261,   84.285,   84.317,   84.354, 
      84.395,   84.440,   84.487,   84.537,   84.591,   84.651,   84.787,   84.943,   85.117,   85.310, 
      85.523,   85.754,   85.996,   86.252,   86.527,   86.817,   87.121,   87.439,   87.769,   88.112, 
      88.468,   88.837,   89.214,   89.605,   90.011,   90.430,   90.857,   91.287,   91.721,   92.157, 
      92.595,   93.035,   93.477,   93.921,   94.365,   94.812,   95.263,   95.718,   96.175,   96.636, 
      97.103,   97.576,   98.054,   98.536,   99.022,   99.513,  100.011,  100.515,  101.024,  101.539, 
      102.060,  102.588,  103.120,  103.658,  104.201,  104.749, ],
    [  90.248,   89.945,   89.686,   89.451,   89.224,   89.004,   88.791,   88.585,   88.385,   88.193, 
      88.009,   87.833,   87.665,   87.505,   87.356,   87.216,   87.085,   86.964,   86.852,   86.749, 
      86.655,   86.570,   86.494,   86.426,   86.363,   86.306,   86.183,   86.083,   86.004,   85.930, 
      85.858,   85.791,   85.732,   85.681,   85.637,   85.602,   85.575,   85.547,   85.520,   85.496, 
      85.473,   85.455,   85.444,   85.439,   85.440,   85.447,   85.461,   85.482,   85.510,   85.544, 
      85.581,   85.622,   85.665,   85.710,   85.761,   85.816,   85.944,   86.091,   86.256,   86.441, 
      86.646,   86.867,   87.099,   87.347,   87.611,   87.892,   88.187,   88.494,   88.814,   89.147, 
      89.494,   89.852,   90.220,   90.600,   90.996,   91.405,   91.821,   92.242,   92.665,   93.090, 
      93.518,   93.947,   94.379,   94.811,   95.245,   95.681,   96.121,   96.565,   97.011,   97.462, 
      97.919,   98.381,   98.849,   99.321,   99.797,  100.278,  100.766,  101.260,  101.760,  102.266, 
      102.778,  103.296,  103.819,  104.348,  104.883,  105.423, ],
    [  91.506,   91.203,   90.944,   90.709,   90.482,   90.262,   90.049,   89.842,   89.643,   89.450, 
      89.266,   89.090,   88.922,   88.763,   88.613,   88.473,   88.342,   88.221,   88.109,   88.006, 
      87.912,   87.827,   87.751,   87.682,   87.620,   87.563,   87.439,   87.338,   87.258,   87.179, 
      87.105,   87.036,   86.975,   86.922,   86.875,   86.839,   86.809,   86.778,   86.748,   86.720, 
      86.693,   86.672,   86.658,   86.650,   86.647,   86.651,   86.662,   86.679,   86.704,   86.734, 
      86.768,   86.804,   86.843,   86.885,   86.931,   86.983,   87.102,   87.241,   87.398,   87.573, 
      87.769,   87.982,   88.205,   88.443,   88.698,   88.970,   89.255,   89.552,   89.863,   90.186, 
      90.522,   90.871,   91.228,   91.598,   91.985,   92.384,   92.790,   93.200,   93.613,   94.028, 
      94.445,   94.864,   95.284,   95.706,   96.129,   96.554,   96.984,   97.417,   97.853,   98.292, 
      98.739,   99.191,   99.648,  100.110,  100.576,  101.048,  101.526,  102.010,  102.501,  102.997, 
      103.500,  104.009,  104.524,  105.044,  105.570,  106.102, ],
    [  92.763,   92.460,   92.201,   91.966,   91.739,   91.519,   91.306,   91.100,   90.900,   90.708, 
      90.524,   90.347,   90.179,   90.020,   89.870,   89.730,   89.600,   89.479,   89.366,   89.263, 
      89.169,   89.084,   89.008,   88.939,   88.877,   88.819,   88.695,   88.593,   88.509,   88.427, 
      88.352,   88.281,   88.218,   88.162,   88.114,   88.076,   88.043,   88.009,   87.975,   87.944, 
      87.913,   87.889,   87.871,   87.860,   87.854,   87.855,   87.862,   87.876,   87.898,   87.925, 
      87.955,   87.987,   88.022,   88.060,   88.102,   88.150,   88.261,   88.392,   88.540,   88.707, 
      88.895,   89.098,   89.312,   89.541,   89.787,   90.049,   90.325,   90.613,   90.914,   91.227, 
      91.554,   91.893,   92.240,   92.600,   92.977,   93.366,   93.762,   94.162,   94.565,   94.969, 
      95.376,   95.784,   96.194,   96.605,   97.017,   97.432,   97.851,   98.273,   98.698,   99.128, 
      99.564,  100.006,  100.453,  100.905,  101.361,  101.822,  102.291,  102.766,  103.247,  103.734, 
      104.228,  104.728,  105.234,  105.745,  106.262,  106.786, ],
    [  94.021,   93.718,   93.459,   93.224,   92.997,   92.777,   92.563,   92.357,   92.157,   91.965, 
      91.781,   91.605,   91.437,   91.277,   91.128,   90.987,   90.857,   90.736,   90.623,   90.520, 
      90.426,   90.341,   90.264,   90.196,   90.133,   90.075,   89.950,   89.848,   89.759,   89.675, 
      89.598,   89.525,   89.460,   89.402,   89.352,   89.313,   89.276,   89.239,   89.202,   89.167, 
      89.132,   89.105,   89.084,   89.071,   89.061,   89.058,   89.063,   89.074,   89.092,   89.115, 
      89.142,   89.171,   89.202,   89.236,   89.274,   89.317,   89.420,   89.543,   89.683,   89.842, 
      90.021,   90.216,   90.422,   90.641,   90.878,   91.131,   91.397,   91.676,   91.967,   92.271, 
      92.588,   92.917,   93.255,   93.605,   93.972,   94.352,   94.738,   95.128,   95.521,   95.915, 
      96.311,   96.709,   97.108,   97.509,   97.910,   98.314,   98.722,   99.134,   99.549,   99.968, 
      100.393,  100.825,  101.262,  101.704,  102.150,  102.602,  103.060,  103.526,  103.998,  104.476, 
      104.960,  105.452,  105.949,  106.451,  106.960,  107.475, ],
    [  95.278,   94.975,   94.716,   94.481,   94.254,   94.034,   93.821,   93.614,   93.415,   93.223, 
      93.038,   92.862,   92.694,   92.535,   92.385,   92.244,   92.114,   91.992,   91.880,   91.777, 
      91.683,   91.598,   91.521,   91.452,   91.389,   91.331,   91.206,   91.103,   91.008,   90.923, 
      90.844,   90.769,   90.702,   90.642,   90.590,   90.549,   90.509,   90.469,   90.429,   90.389, 
      90.352,   90.321,   90.298,   90.281,   90.268,   90.262,   90.263,   90.271,   90.286,   90.307, 
      90.329,   90.355,   90.382,   90.412,   90.446,   90.485,   90.581,   90.696,   90.828,   90.978, 
      91.149,   91.335,   91.532,   91.743,   91.971,   92.215,   92.472,   92.741,   93.023,   93.317, 
      93.625,   93.945,   94.273,   94.613,   94.971,   95.341,   95.717,   96.098,   96.480,   96.864, 
      97.250,   97.638,   98.027,   98.416,   98.807,   99.200,   99.598,   99.999,  100.403,  100.812, 
      101.227,  101.649,  102.076,  102.508,  102.944,  103.386,  103.835,  104.291,  104.753,  105.222, 
      105.698,  106.180,  106.668,  107.162,  107.662,  108.169, ],
    [  96.536,   96.233,   95.974,   95.739,   95.511,   95.291,   95.078,   94.872,   94.672,   94.480, 
      94.295,   94.119,   93.951,   93.792,   93.642,   93.501,   93.371,   93.249,   93.137,   93.033, 
      92.939,   92.854,   92.777,   92.708,   92.645,   92.587,   92.461,   92.355,   92.257,   92.170, 
      92.089,   92.012,   91.943,   91.882,   91.828,   91.784,   91.742,   91.698,   91.655,   91.611, 
      91.571,   91.537,   91.511,   91.491,   91.474,   91.466,   91.464,   91.469,   91.481,   91.498, 
      91.517,   91.539,   91.562,   91.588,   91.618,   91.654,   91.742,   91.849,   91.973,   92.115, 
      92.278,   92.456,   92.644,   92.846,   93.065,   93.300,   93.548,   93.809,   94.081,   94.366, 
      94.665,   94.975,   95.293,   95.624,   95.972,   96.333,   96.700,   97.070,   97.443,   97.817, 
      98.193,   98.570,   98.948,   99.328,   99.708,  100.091,  100.478,  100.869,  101.262,  101.660, 
      102.065,  102.477,  102.894,  103.316,  103.742,  104.174,  104.614,  105.061,  105.514,  105.973, 
      106.440,  106.913,  107.393,  107.878,  108.370,  108.868, ],
    [  97.793,   97.490,   97.231,   96.996,   96.769,   96.548,   96.335,   96.129,   95.929,   95.737, 
      95.552,   95.376,   95.208,   95.048,   94.898,   94.758,   94.627,   94.506,   94.393,   94.290, 
      94.196,   94.110,   94.033,   93.964,   93.900,   93.842,   93.715,   93.605,   93.505,   93.417, 
      93.334,   93.255,   93.184,   93.121,   93.065,   93.020,   92.974,   92.927,   92.880,   92.833, 
      92.790,   92.754,   92.724,   92.700,   92.681,   92.670,   92.665,   92.666,   92.676,   92.689, 
      92.705,   92.723,   92.743,   92.765,   92.791,   92.823,   92.904,   93.003,   93.119,   93.253, 
      93.408,   93.578,   93.758,   93.951,   94.162,   94.388,   94.627,   94.878,   95.142,   95.417, 
      95.707,   96.008,   96.316,   96.638,   96.977,   97.328,   97.686,   98.046,   98.409,   98.773, 
      99.139,   99.506,   99.874,  100.243,  100.613,  100.985,  101.361,  101.742,  102.125,  102.513, 
      102.908,  103.309,  103.716,  104.128,  104.545,  104.967,  105.397,  105.835,  106.278,  106.729, 
      107.186,  107.651,  108.122,  108.599,  109.082,  109.571, ],
    [  99.050,   98.747,   98.488,   98.253,   98.026,   97.805,   97.592,   97.386,   97.186,   96.994, 
      96.809,   96.633,   96.464,   96.305,   96.155,   96.015,   95.884,   95.762,   95.649,   95.546, 
      95.452,   95.366,   95.289,   95.219,   95.156,   95.098,   94.970,   94.855,   94.753,   94.663, 
      94.579,   94.498,   94.425,   94.360,   94.302,   94.254,   94.206,   94.155,   94.105,   94.054, 
      94.008,   93.969,   93.936,   93.910,   93.888,   93.873,   93.865,   93.864,   93.870,   93.881, 
      93.894,   93.908,   93.923,   93.942,   93.965,   93.993,   94.066,   94.158,   94.266,   94.392, 
      94.540,   94.701,   94.873,   95.057,   95.259,   95.477,   95.707,   95.950,   96.204,   96.471, 
      96.751,   97.043,   97.342,   97.654,   97.984,   98.326,   98.674,   99.026,   99.379,   99.733, 
      100.088,  100.445,  100.803,  101.162,  101.521,  101.883,  102.249,  102.619,  102.992,  103.369, 
      103.754,  104.146,  104.543,  104.945,  105.351,  105.764,  106.185,  106.613,  107.047,  107.489, 
      107.938,  108.393,  108.855,  109.324,  109.798,  110.279, ],
    [ 100.307,  100.004,   99.745,   99.510,   99.283,   99.062,   98.849,   98.642,   98.443,   98.250, 
      98.066,   97.889,   97.721,   97.561,   97.411,   97.271,   97.140,   97.018,   96.906,   96.802, 
      96.708,   96.622,   96.544,   96.475,   96.411,   96.353,   96.222,   96.104,   96.001,   95.909, 
      95.822,   95.740,   95.666,   95.598,   95.540,   95.489,   95.437,   95.383,   95.330,   95.276, 
      95.226,   95.184,   95.149,   95.119,   95.094,   95.077,   95.066,   95.062,   95.065,   95.073, 
      95.082,   95.093,   95.105,   95.119,   95.138,   95.163,   95.228,   95.313,   95.414,   95.532, 
      95.672,   95.826,   95.989,   96.165,   96.358,   96.568,   96.789,   97.023,   97.269,   97.526, 
      97.798,   98.080,   98.370,   98.673,   98.994,   99.327,   99.666,  100.008,  100.351,  100.696, 
      101.041,  101.388,  101.736,  102.084,  102.433,  102.784,  103.140,  103.500,  103.862,  104.230, 
      104.604,  104.986,  105.373,  105.765,  106.162,  106.566,  106.977,  107.395,  107.821,  108.253, 
      108.693,  109.140,  109.593,  110.053,  110.519,  110.992, ],
    [ 101.564,  101.261,  101.002,  100.766,  100.539,  100.319,  100.106,   99.899,   99.699,   99.507, 
      99.322,   99.146,   98.977,   98.818,   98.668,   98.527,   98.396,   98.274,   98.161,   98.058, 
      97.963,   97.877,   97.800,   97.730,   97.666,   97.607,   97.472,   97.352,   97.248,   97.154, 
      97.066,   96.982,   96.905,   96.836,   96.777,   96.723,   96.667,   96.610,   96.554,   96.496, 
      96.444,   96.399,   96.361,   96.329,   96.301,   96.280,   96.267,   96.260,   96.260,   96.265, 
      96.271,   96.278,   96.286,   96.297,   96.312,   96.333,   96.392,   96.469,   96.562,   96.673, 
      96.805,   96.951,   97.106,   97.274,   97.459,   97.660,   97.873,   98.098,   98.335,   98.584, 
      98.846,   99.120,   99.401,   99.695,  100.007,  100.331,  100.660,  100.993,  101.327,  101.661, 
      101.997,  102.334,  102.671,  103.009,  103.348,  103.689,  104.035,  104.384,  104.736,  105.093, 
      105.458,  105.830,  106.207,  106.590,  106.977,  107.371,  107.772,  108.182,  108.598,  109.021, 
      109.452,  109.891,  110.335,  110.787,  111.244,  111.709, ],
    [ 102.821,  102.517,  102.258,  102.023,  101.796,  101.576,  101.362,  101.155,  100.956,  100.763, 
      100.578,  100.402,  100.233,  100.074,   99.924,   99.783,   99.652,   99.530,   99.417,   99.313, 
      99.219,   99.132,   99.055,   98.985,   98.921,   98.861,   98.722,   98.600,   98.494,   98.399, 
      98.309,   98.223,   98.145,   98.074,   98.013,   97.956,   97.898,   97.837,   97.776,   97.716, 
      97.662,   97.614,   97.573,   97.537,   97.507,   97.484,   97.467,   97.457,   97.455,   97.457, 
      97.459,   97.463,   97.467,   97.474,   97.486,   97.504,   97.555,   97.625,   97.711,   97.815, 
      97.939,   98.078,   98.224,   98.384,   98.561,   98.754,   98.958,   99.175,   99.403,   99.643, 
      99.897,  100.162,  100.434,  100.719,  101.022,  101.337,  101.657,  101.981,  102.305,  102.630, 
      102.956,  103.283,  103.610,  103.938,  104.266,  104.597,  104.932,  105.272,  105.614,  105.961, 
      106.316,  106.677,  107.045,  107.418,  107.795,  108.180,  108.572,  108.972,  109.380,  109.794, 
      110.216,  110.645,  111.082,  111.524,  111.974,  112.430, ],
    [ 104.077,  103.774,  103.515,  103.279,  103.052,  102.832,  102.618,  102.412,  102.212,  102.019, 
      101.834,  101.658,  101.489,  101.330,  101.179,  101.039,  100.907,  100.785,  100.672,  100.569, 
      100.474,  100.387,  100.310,  100.239,  100.174,  100.111,   99.970,   99.848,   99.740,   99.643, 
      99.552,   99.464,   99.384,   99.311,   99.249,   99.189,   99.127,   99.063,   98.998,   98.935, 
      98.878,   98.828,   98.784,   98.746,   98.712,   98.687,   98.668,   98.655,   98.651,   98.648, 
      98.648,   98.648,   98.648,   98.652,   98.661,   98.675,   98.719,   98.782,   98.860,   98.957, 
      99.074,   99.205,   99.343,   99.495,   99.664,   99.849,  100.045,  100.253,  100.473,  100.704, 
      100.950,  101.206,  101.469,  101.745,  102.039,  102.345,  102.657,  102.971,  103.286,  103.602, 
      103.918,  104.235,  104.552,  104.870,  105.188,  105.508,  105.834,  106.163,  106.495,  106.832, 
      107.176,  107.528,  107.886,  108.249,  108.618,  108.992,  109.375,  109.767,  110.165,  110.570, 
      110.983,  111.404,  111.832,  112.266,  112.707,  113.156, ],
    [ 105.333,  105.030,  104.771,  104.536,  104.308,  104.088,  103.874,  103.668,  103.468,  103.275, 
      103.090,  102.914,  102.745,  102.585,  102.435,  102.294,  102.163,  102.041,  101.928,  101.824, 
      101.729,  101.642,  101.564,  101.491,  101.424,  101.361,  101.218,  101.095,  100.986,  100.886, 
      100.794,  100.704,  100.622,  100.549,  100.484,  100.421,  100.356,  100.289,  100.220,  100.154, 
      100.094,  100.042,   99.995,   99.954,   99.918,   99.889,   99.868,   99.852,   99.846,   99.840, 
      99.836,   99.833,   99.830,   99.830,   99.835,   99.846,   99.883,   99.939,  100.010,  100.100, 
      100.210,  100.333,  100.464,  100.607,  100.769,  100.945,  101.133,  101.333,  101.544,  101.767, 
      102.004,  102.251,  102.506,  102.773,  103.058,  103.356,  103.659,  103.964,  104.270,  104.576, 
      104.882,  105.189,  105.497,  105.805,  106.113,  106.423,  106.738,  107.057,  107.379,  107.706, 
      108.041,  108.383,  108.731,  109.085,  109.443,  109.809,  110.182,  110.564,  110.954,  111.350, 
      111.755,  112.167,  112.586,  113.012,  113.445,  113.885, ],
    [ 106.590,  106.286,  106.027,  105.792,  105.564,  105.344,  105.130,  104.923,  104.723,  104.531, 
      104.346,  104.169,  104.000,  103.841,  103.690,  103.549,  103.418,  103.296,  103.182,  103.078, 
      102.983,  102.896,  102.815,  102.741,  102.673,  102.610,  102.466,  102.341,  102.230,  102.130, 
      102.035,  101.944,  101.861,  101.786,  101.719,  101.652,  101.583,  101.514,  101.441,  101.373, 
      101.310,  101.255,  101.206,  101.162,  101.123,  101.092,  101.068,  101.050,  101.040,  101.032, 
      101.025,  101.018,  101.011,  101.008,  101.009,  101.017,  101.048,  101.097,  101.161,  101.243, 
      101.346,  101.462,  101.585,  101.720,  101.874,  102.043,  102.223,  102.414,  102.617,  102.832, 
      103.060,  103.299,  103.545,  103.803,  104.080,  104.369,  104.664,  104.960,  105.256,  105.553, 
      105.850,  106.147,  106.444,  106.742,  107.040,  107.340,  107.645,  107.954,  108.266,  108.583, 
      108.908,  109.240,  109.579,  109.923,  110.272,  110.628,  110.993,  111.366,  111.746,  112.134, 
      112.530,  112.933,  113.344,  113.762,  114.186,  114.618, ],
    [ 107.845,  107.542,  107.283,  107.047,  106.820,  106.599,  106.386,  106.179,  105.979,  105.786, 
      105.601,  105.424,  105.256,  105.096,  104.945,  104.804,  104.673,  104.550,  104.437,  104.332, 
      104.235,  104.146,  104.065,  103.991,  103.922,  103.858,  103.714,  103.587,  103.475,  103.373, 
      103.276,  103.184,  103.098,  103.023,  102.954,  102.883,  102.811,  102.737,  102.662,  102.590, 
      102.526,  102.468,  102.417,  102.369,  102.329,  102.295,  102.268,  102.247,  102.235,  102.224, 
      102.213,  102.203,  102.192,  102.185,  102.184,  102.188,  102.212,  102.255,  102.312,  102.387, 
      102.483,  102.592,  102.707,  102.834,  102.980,  103.141,  103.314,  103.497,  103.692,  103.898, 
      104.118,  104.348,  104.586,  104.836,  105.104,  105.385,  105.670,  105.958,  106.245,  106.532, 
      106.820,  107.107,  107.395,  107.682,  107.970,  108.260,  108.555,  108.854,  109.156,  109.463, 
      109.778,  110.101,  110.430,  110.765,  111.105,  111.451,  111.807,  112.171,  112.542,  112.921, 
      113.308,  113.703,  114.106,  114.515,  114.931,  115.355, ],
    [ 109.101,  108.798,  108.538,  108.303,  108.075,  107.855,  107.641,  107.434,  107.234,  107.041, 
      106.856,  106.679,  106.511,  106.351,  106.200,  106.059,  105.927,  105.804,  105.689,  105.583, 
      105.485,  105.395,  105.314,  105.239,  105.170,  105.106,  104.960,  104.832,  104.718,  104.615, 
      104.517,  104.423,  104.336,  104.260,  104.187,  104.113,  104.038,  103.959,  103.881,  103.808, 
      103.741,  103.680,  103.626,  103.576,  103.533,  103.498,  103.468,  103.444,  103.430,  103.416, 
      103.402,  103.388,  103.373,  103.363,  103.358,  103.359,  103.377,  103.413,  103.463,  103.531, 
      103.621,  103.722,  103.829,  103.949,  104.088,  104.241,  104.405,  104.581,  104.767,  104.966, 
      105.177,  105.400,  105.628,  105.870,  106.130,  106.402,  106.679,  106.958,  107.236,  107.514, 
      107.792,  108.070,  108.348,  108.625,  108.903,  109.183,  109.468,  109.757,  110.049,  110.346, 
      110.652,  110.965,  111.285,  111.610,  111.940,  112.278,  112.624,  112.979,  113.341,  113.712, 
      114.090,  114.477,  114.871,  115.272,  115.680,  116.096, ],
    [ 110.356,  110.053,  109.794,  109.558,  109.331,  109.110,  108.896,  108.689,  108.489,  108.296, 
      108.111,  107.934,  107.765,  107.605,  107.454,  107.312,  107.178,  107.054,  106.939,  106.832, 
      106.734,  106.644,  106.562,  106.487,  106.418,  106.353,  106.207,  106.077,  105.962,  105.857, 
      105.757,  105.661,  105.574,  105.496,  105.420,  105.342,  105.263,  105.181,  105.100,  105.025, 
      104.955,  104.892,  104.836,  104.783,  104.738,  104.700,  104.667,  104.641,  104.624,  104.608, 
      104.590,  104.572,  104.554,  104.541,  104.533,  104.530,  104.542,  104.572,  104.615,  104.676, 
      104.759,  104.853,  104.953,  105.065,  105.196,  105.342,  105.499,  105.666,  105.845,  106.035, 
      106.238,  106.452,  106.672,  106.906,  107.158,  107.422,  107.690,  107.960,  108.229,  108.498, 
      108.766,  109.035,  109.303,  109.571,  109.839,  110.109,  110.384,  110.663,  110.945,  111.232, 
      111.528,  111.832,  112.142,  112.458,  112.779,  113.107,  113.444,  113.790,  114.144,  114.505, 
      114.876,  115.254,  115.639,  116.032,  116.432,  116.840, ],
    [ 111.612,  111.308,  111.049,  110.813,  110.586,  110.365,  110.151,  109.944,  109.744,  109.551, 
      109.366,  109.188,  109.018,  108.857,  108.704,  108.562,  108.428,  108.304,  108.188,  108.081, 
      107.982,  107.892,  107.810,  107.735,  107.665,  107.600,  107.452,  107.321,  107.205,  107.098, 
      106.996,  106.899,  106.811,  106.731,  106.651,  106.570,  106.488,  106.402,  106.320,  106.241, 
      106.169,  106.104,  106.045,  105.989,  105.942,  105.901,  105.866,  105.838,  105.819,  105.799, 
      105.777,  105.756,  105.735,  105.719,  105.707,  105.702,  105.707,  105.730,  105.766,  105.821, 
      105.898,  105.985,  106.077,  106.182,  106.305,  106.444,  106.593,  106.752,  106.923,  107.105, 
      107.301,  107.506,  107.718,  107.943,  108.187,  108.443,  108.703,  108.965,  109.225,  109.484, 
      109.743,  110.002,  110.260,  110.519,  110.776,  111.036,  111.302,  111.571,  111.843,  112.121, 
      112.407,  112.701,  113.002,  113.308,  113.620,  113.939,  114.268,  114.605,  114.950,  115.303, 
      115.664,  116.034,  116.411,  116.796,  117.188,  117.588, ],
    [ 112.867,  112.563,  112.304,  112.068,  111.840,  111.620,  111.406,  111.198,  110.997,  110.803, 
      110.617,  110.439,  110.268,  110.106,  109.954,  109.811,  109.677,  109.553,  109.436,  109.329, 
      109.231,  109.140,  109.058,  108.982,  108.912,  108.846,  108.697,  108.565,  108.447,  108.339, 
      108.236,  108.137,  108.048,  107.965,  107.882,  107.798,  107.712,  107.623,  107.537,  107.457, 
      107.383,  107.315,  107.253,  107.195,  107.146,  107.103,  107.065,  107.035,  107.013,  106.990, 
      106.965,  106.941,  106.916,  106.896,  106.881,  106.873,  106.872,  106.889,  106.919,  106.967, 
      107.037,  107.117,  107.201,  107.299,  107.415,  107.546,  107.688,  107.840,  108.002,  108.177, 
      108.364,  108.562,  108.766,  108.982,  109.218,  109.466,  109.718,  109.971,  110.222,  110.472, 
      110.722,  110.972,  111.220,  111.469,  111.717,  111.967,  112.222,  112.482,  112.744,  113.012, 
      113.289,  113.573,  113.865,  114.162,  114.465,  114.775,  115.094,  115.422,  115.758,  116.103, 
      116.456,  116.817,  117.186,  117.563,  117.947,  118.339, ],
    [ 114.121,  113.818,  113.558,  113.322,  113.093,  112.872,  112.657,  112.449,  112.247,  112.053, 
      111.866,  111.688,  111.517,  111.356,  111.203,  111.060,  110.926,  110.801,  110.684,  110.577, 
      110.478,  110.387,  110.305,  110.229,  110.158,  110.092,  109.941,  109.808,  109.688,  109.579, 
      109.474,  109.375,  109.285,  109.199,  109.112,  109.024,  108.934,  108.842,  108.755,  108.672, 
      108.596,  108.526,  108.461,  108.401,  108.349,  108.304,  108.264,  108.232,  108.208,  108.180, 
      108.153,  108.125,  108.096,  108.074,  108.056,  108.044,  108.038,  108.048,  108.071,  108.113, 
      108.176,  108.250,  108.327,  108.417,  108.526,  108.650,  108.784,  108.928,  109.083,  109.249, 
      109.429,  109.619,  109.815,  110.023,  110.251,  110.491,  110.735,  110.979,  111.222,  111.463, 
      111.703,  111.943,  112.182,  112.421,  112.659,  112.899,  113.145,  113.395,  113.647,  113.905, 
      114.173,  114.448,  114.730,  115.018,  115.312,  115.613,  115.923,  116.243,  116.570,  116.906, 
      117.251,  117.604,  117.965,  118.333,  118.709,  119.093, ],
    [ 115.373,  115.069,  114.809,  114.572,  114.344,  114.122,  113.906,  113.698,  113.497,  113.302, 
      113.116,  112.937,  112.766,  112.604,  112.451,  112.308,  112.174,  112.048,  111.932,  111.824, 
      111.725,  111.634,  111.551,  111.475,  111.404,  111.337,  111.185,  111.051,  110.930,  110.818, 
      110.713,  110.612,  110.520,  110.431,  110.341,  110.250,  110.155,  110.062,  109.972,  109.887, 
      109.808,  109.736,  109.668,  109.607,  109.552,  109.504,  109.463,  109.429,  109.401,  109.371, 
      109.340,  109.309,  109.277,  109.251,  109.230,  109.216,  109.203,  109.207,  109.223,  109.259, 
      109.316,  109.383,  109.452,  109.535,  109.638,  109.754,  109.880,  110.017,  110.165,  110.323, 
      110.495,  110.677,  110.865,  111.065,  111.286,  111.518,  111.754,  111.989,  112.223,  112.455, 
      112.686,  112.917,  113.147,  113.375,  113.604,  113.834,  114.070,  114.310,  114.553,  114.801, 
      115.059,  115.325,  115.598,  115.877,  116.161,  116.453,  116.755,  117.066,  117.385,  117.712, 
      118.048,  118.393,  118.746,  119.106,  119.474,  119.850, ],
    [ 116.623,  116.319,  116.058,  115.822,  115.593,  115.371,  115.156,  114.947,  114.745,  114.551, 
      114.364,  114.185,  114.014,  113.852,  113.699,  113.555,  113.421,  113.296,  113.179,  113.071, 
      112.972,  112.880,  112.797,  112.720,  112.649,  112.582,  112.429,  112.293,  112.171,  112.057, 
      111.950,  111.849,  111.755,  111.662,  111.569,  111.475,  111.377,  111.280,  111.188,  111.101, 
      111.020,  110.945,  110.875,  110.812,  110.755,  110.705,  110.661,  110.625,  110.595,  110.561, 
      110.527,  110.492,  110.457,  110.427,  110.404,  110.387,  110.368,  110.366,  110.376,  110.405, 
      110.456,  110.516,  110.579,  110.654,  110.750,  110.859,  110.978,  111.107,  111.247,  111.398, 
      111.563,  111.737,  111.916,  112.109,  112.322,  112.546,  112.774,  113.001,  113.226,  113.449, 
      113.671,  113.892,  114.113,  114.332,  114.550,  114.771,  114.997,  115.228,  115.461,  115.700, 
      115.948,  116.205,  116.469,  116.738,  117.013,  117.296,  117.589,  117.892,  118.202,  118.521, 
      118.849,  119.186,  119.530,  119.883,  120.243,  120.611, ],
    [ 117.873,  117.568,  117.308,  117.071,  116.841,  116.619,  116.404,  116.195,  115.993,  115.799, 
      115.612,  115.433,  115.262,  115.100,  114.946,  114.803,  114.668,  114.543,  114.425,  114.317, 
      114.217,  114.126,  114.042,  113.965,  113.893,  113.825,  113.672,  113.534,  113.410,  113.296, 
      113.188,  113.086,  112.989,  112.893,  112.796,  112.697,  112.596,  112.498,  112.403,  112.314, 
      112.231,  112.154,  112.082,  112.016,  111.957,  111.905,  111.859,  111.822,  111.788,  111.752, 
      111.714,  111.675,  111.636,  111.604,  111.578,  111.557,  111.533,  111.525,  111.529,  111.552, 
      111.597,  111.650,  111.705,  111.774,  111.862,  111.964,  112.077,  112.198,  112.331,  112.474, 
      112.631,  112.798,  112.969,  113.154,  113.359,  113.576,  113.796,  114.014,  114.231,  114.445, 
      114.658,  114.870,  115.080,  115.290,  115.499,  115.710,  115.926,  116.147,  116.371,  116.600, 
      116.839,  117.087,  117.341,  117.602,  117.868,  118.142,  118.426,  118.720,  119.022,  119.332, 
      119.652,  119.981,  120.317,  120.662,  121.014,  121.375, ],
    [ 119.121,  118.816,  118.556,  118.319,  118.090,  117.867,  117.652,  117.443,  117.241,  117.046, 
      116.859,  116.680,  116.509,  116.346,  116.193,  116.049,  115.914,  115.789,  115.671,  115.563, 
      115.463,  115.371,  115.286,  115.209,  115.137,  115.069,  114.914,  114.775,  114.650,  114.534, 
      114.425,  114.322,  114.222,  114.122,  114.022,  113.918,  113.815,  113.715,  113.619,  113.527, 
      113.443,  113.362,  113.288,  113.220,  113.160,  113.105,  113.057,  113.018,  112.981,  112.941, 
      112.900,  112.857,  112.815,  112.780,  112.751,  112.728,  112.698,  112.684,  112.682,  112.699, 
      112.738,  112.784,  112.832,  112.894,  112.975,  113.071,  113.176,  113.290,  113.415,  113.551, 
      113.701,  113.860,  114.023,  114.200,  114.398,  114.607,  114.819,  115.029,  115.237,  115.443, 
      115.647,  115.849,  116.050,  116.250,  116.449,  116.650,  116.857,  117.069,  117.282,  117.502, 
      117.732,  117.971,  118.216,  118.468,  118.725,  118.990,  119.266,  119.551,  119.844,  120.147, 
      120.458,  120.779,  121.107,  121.444,  121.788,  122.141, ],
    [ 120.369,  120.064,  119.804,  119.567,  119.337,  119.115,  118.899,  118.690,  118.488,  118.293, 
      118.106,  117.927,  117.756,  117.593,  117.439,  117.295,  117.160,  117.034,  116.917,  116.808, 
      116.707,  116.615,  116.530,  116.452,  116.380,  116.312,  116.155,  116.016,  115.889,  115.772, 
      115.663,  115.557,  115.454,  115.350,  115.247,  115.140,  115.034,  114.931,  114.833,  114.739, 
      114.653,  114.570,  114.494,  114.424,  114.361,  114.304,  114.254,  114.213,  114.173,  114.130, 
      114.086,  114.039,  113.994,  113.957,  113.925,  113.899,  113.863,  113.844,  113.834,  113.845, 
      113.879,  113.918,  113.959,  114.014,  114.089,  114.178,  114.275,  114.383,  114.500,  114.629, 
      114.771,  114.922,  115.078,  115.248,  115.438,  115.640,  115.844,  116.046,  116.245,  116.442, 
      116.637,  116.830,  117.022,  117.212,  117.401,  117.593,  117.790,  117.992,  118.196,  118.407, 
      118.627,  118.857,  119.093,  119.336,  119.584,  119.841,  120.108,  120.384,  120.669,  120.963, 
      121.267,  121.579,  121.900,  122.229,  122.565,  122.910, ],
    [ 121.617,  121.312,  121.051,  120.814,  120.584,  120.362,  120.146,  119.937,  119.735,  119.540, 
      119.352,  119.173,  119.002,  118.839,  118.685,  118.541,  118.405,  118.279,  118.161,  118.052, 
      117.951,  117.859,  117.774,  117.696,  117.623,  117.554,  117.397,  117.255,  117.128,  117.010, 
      116.900,  116.792,  116.685,  116.577,  116.469,  116.359,  116.252,  116.147,  116.047,  115.951, 
      115.862,  115.777,  115.699,  115.627,  115.562,  115.503,  115.451,  115.409,  115.365,  115.319, 
      115.270,  115.220,  115.173,  115.132,  115.097,  115.069,  115.028,  115.003,  114.987,  114.992, 
      115.020,  115.053,  115.087,  115.135,  115.203,  115.285,  115.376,  115.476,  115.586,  115.707, 
      115.842,  115.986,  116.134,  116.296,  116.479,  116.673,  116.870,  117.064,  117.255,  117.443, 
      117.628,  117.812,  117.995,  118.175,  118.355,  118.537,  118.725,  118.917,  119.112,  119.313, 
      119.524,  119.745,  119.972,  120.206,  120.445,  120.693,  120.952,  121.220,  121.497,  121.782, 
      122.078,  122.382,  122.695,  123.016,  123.345,  123.682, ],
    [ 122.864,  122.559,  122.298,  122.061,  121.831,  121.608,  121.393,  121.183,  120.981,  120.786, 
      120.598,  120.419,  120.247,  120.084,  119.930,  119.785,  119.650,  119.523,  119.405,  119.296, 
      119.195,  119.102,  119.017,  118.938,  118.865,  118.796,  118.637,  118.495,  118.366,  118.247, 
      118.136,  118.025,  117.914,  117.803,  117.690,  117.579,  117.469,  117.362,  117.260,  117.163, 
      117.070,  116.984,  116.904,  116.830,  116.763,  116.702,  116.648,  116.603,  116.557,  116.506, 
      116.455,  116.400,  116.351,  116.307,  116.270,  116.240,  116.193,  116.161,  116.140,  116.139, 
      116.161,  116.188,  116.214,  116.256,  116.318,  116.393,  116.477,  116.570,  116.673,  116.787, 
      116.914,  117.051,  117.191,  117.346,  117.521,  117.708,  117.897,  118.083,  118.266,  118.445, 
      118.622,  118.796,  118.969,  119.140,  119.310,  119.482,  119.661,  119.844,  120.029,  120.221, 
      120.423,  120.635,  120.853,  121.078,  121.309,  121.548,  121.798,  122.058,  122.326,  122.604, 
      122.891,  123.188,  123.493,  123.806,  124.127,  124.457, ],
    [ 124.110,  123.805,  123.544,  123.307,  123.077,  122.854,  122.638,  122.429,  122.226,  122.031, 
      121.843,  121.664,  121.492,  121.328,  121.174,  121.029,  120.893,  120.767,  120.649,  120.539, 
      120.438,  120.345,  120.259,  120.180,  120.106,  120.037,  119.877,  119.734,  119.603,  119.484, 
      119.371,  119.256,  119.142,  119.028,  118.912,  118.797,  118.685,  118.577,  118.472,  118.374, 
      118.278,  118.190,  118.108,  118.032,  117.963,  117.900,  117.845,  117.798,  117.748,  117.694, 
      117.639,  117.581,  117.529,  117.483,  117.443,  117.410,  117.357,  117.320,  117.293,  117.287, 
      117.303,  117.323,  117.342,  117.377,  117.433,  117.501,  117.578,  117.664,  117.760,  117.867, 
      117.987,  118.117,  118.249,  118.396,  118.565,  118.745,  118.926,  119.104,  119.278,  119.448, 
      119.616,  119.781,  119.945,  120.107,  120.267,  120.429,  120.598,  120.772,  120.948,  121.130, 
      121.324,  121.526,  121.736,  121.952,  122.174,  122.405,  122.646,  122.898,  123.158,  123.427, 
      123.707,  123.995,  124.293,  124.598,  124.912,  125.234, ],
    [ 125.356,  125.051,  124.790,  124.552,  124.322,  124.100,  123.883,  123.674,  123.471,  123.276, 
      123.088,  122.908,  122.736,  122.572,  122.418,  122.273,  122.137,  122.010,  121.892,  121.782, 
      121.680,  121.587,  121.501,  121.421,  121.347,  121.277,  121.116,  120.972,  120.841,  120.721, 
      120.605,  120.487,  120.369,  120.250,  120.131,  120.015,  119.901,  119.791,  119.684,  119.583, 
      119.486,  119.396,  119.312,  119.235,  119.163,  119.099,  119.042,  118.992,  118.938,  118.881, 
      118.822,  118.761,  118.706,  118.657,  118.615,  118.579,  118.521,  118.478,  118.445,  118.434, 
      118.444,  118.458,  118.470,  118.499,  118.548,  118.610,  118.680,  118.759,  118.848,  118.948, 
      119.061,  119.183,  119.308,  119.448,  119.609,  119.782,  119.956,  120.126,  120.291,  120.453, 
      120.612,  120.768,  120.922,  121.075,  121.225,  121.378,  121.538,  121.702,  121.868,  122.041, 
      122.226,  122.420,  122.621,  122.828,  123.041,  123.263,  123.496,  123.740,  123.992,  124.253, 
      124.525,  124.806,  125.095,  125.393,  125.699,  126.014, ],
    [ 126.601,  126.296,  126.035,  125.797,  125.567,  125.344,  125.128,  124.918,  124.715,  124.519, 
      124.331,  124.151,  123.979,  123.816,  123.661,  123.516,  123.380,  123.253,  123.134,  123.024, 
      122.922,  122.828,  122.742,  122.662,  122.588,  122.517,  122.355,  122.210,  122.078,  121.957, 
      121.838,  121.716,  121.595,  121.472,  121.350,  121.232,  121.116,  121.004,  120.896,  120.792, 
      120.693,  120.601,  120.515,  120.436,  120.363,  120.296,  120.238,  120.185,  120.128,  120.068, 
      120.004,  119.940,  119.883,  119.832,  119.787,  119.749,  119.686,  119.636,  119.598,  119.581, 
      119.586,  119.593,  119.598,  119.620,  119.663,  119.719,  119.782,  119.855,  119.936,  120.029, 
      120.136,  120.251,  120.368,  120.500,  120.655,  120.821,  120.987,  121.149,  121.306,  121.459, 
      121.609,  121.756,  121.901,  122.044,  122.185,  122.328,  122.478,  122.633,  122.790,  122.954, 
      123.130,  123.315,  123.507,  123.706,  123.910,  124.123,  124.348,  124.584,  124.828,  125.081, 
      125.345,  125.618,  125.900,  126.190,  126.488,  126.796, ],
    [ 127.846,  127.541,  127.279,  127.041,  126.811,  126.588,  126.372,  126.162,  125.959,  125.763, 
      125.575,  125.395,  125.223,  125.059,  124.904,  124.759,  124.622,  124.495,  124.376,  124.265, 
      124.163,  124.069,  123.983,  123.902,  123.827,  123.757,  123.594,  123.447,  123.315,  123.192, 
      123.069,  122.943,  122.818,  122.692,  122.569,  122.448,  122.331,  122.216,  122.107,  122.000, 
      121.899,  121.806,  121.718,  121.637,  121.562,  121.493,  121.434,  121.378,  121.317,  121.253, 
      121.185,  121.119,  121.059,  121.006,  120.958,  120.918,  120.849,  120.794,  120.750,  120.728, 
      120.727,  120.728,  120.726,  120.742,  120.779,  120.828,  120.885,  120.950,  121.025,  121.111, 
      121.211,  121.319,  121.429,  121.554,  121.701,  121.860,  122.019,  122.173,  122.322,  122.466, 
      122.607,  122.745,  122.881,  123.014,  123.145,  123.279,  123.420,  123.566,  123.713,  123.868, 
      124.035,  124.211,  124.395,  124.585,  124.781,  124.986,  125.202,  125.429,  125.665,  125.911, 
      126.167,  126.432,  126.706,  126.989,  127.280,  127.580, ],
    [ 129.090,  128.784,  128.523,  128.285,  128.055,  127.832,  127.615,  127.405,  127.202,  127.006, 
      126.818,  126.638,  126.465,  126.301,  126.146,  126.000,  125.864,  125.736,  125.617,  125.506, 
      125.404,  125.309,  125.222,  125.142,  125.067,  124.995,  124.831,  124.685,  124.552,  124.425, 
      124.299,  124.170,  124.040,  123.912,  123.786,  123.664,  123.544,  123.428,  123.316,  123.207, 
      123.105,  123.010,  122.921,  122.837,  122.761,  122.691,  122.630,  122.570,  122.505,  122.437, 
      122.366,  122.298,  122.235,  122.179,  122.130,  122.087,  122.013,  121.952,  121.903,  121.875, 
      121.869,  121.863,  121.855,  121.864,  121.895,  121.938,  121.988,  122.047,  122.115,  122.193, 
      122.286,  122.387,  122.490,  122.608,  122.749,  122.901,  123.052,  123.198,  123.339,  123.474, 
      123.606,  123.735,  123.861,  123.985,  124.107,  124.231,  124.363,  124.500,  124.638,  124.784, 
      124.942,  125.109,  125.284,  125.465,  125.653,  125.849,  126.058,  126.277,  126.505,  126.743, 
      126.991,  127.249,  127.515,  127.790,  128.074,  128.367, ],
    [ 130.333,  130.028,  129.766,  129.528,  129.298,  129.075,  128.858,  128.648,  128.445,  128.249, 
      128.060,  127.879,  127.707,  127.542,  127.387,  127.242,  127.105,  126.977,  126.858,  126.746, 
      126.644,  126.549,  126.462,  126.381,  126.305,  126.234,  126.069,  125.922,  125.787,  125.657, 
      125.527,  125.395,  125.261,  125.131,  125.003,  124.879,  124.758,  124.640,  124.525,  124.414, 
      124.311,  124.213,  124.123,  124.038,  123.959,  123.888,  123.825,  123.761,  123.693,  123.621, 
      123.546,  123.476,  123.411,  123.352,  123.301,  123.255,  123.176,  123.110,  123.055,  123.022, 
      123.011,  122.999,  122.983,  122.986,  123.012,  123.048,  123.091,  123.143,  123.204,  123.276, 
      123.363,  123.456,  123.552,  123.663,  123.797,  123.942,  124.086,  124.225,  124.357,  124.484, 
      124.607,  124.726,  124.843,  124.958,  125.070,  125.185,  125.307,  125.435,  125.563,  125.700, 
      125.849,  126.009,  126.175,  126.348,  126.527,  126.715,  126.915,  127.126,  127.346,  127.576, 
      127.817,  128.067,  128.326,  128.594,  128.870,  129.156, ],
    [ 131.577,  131.271,  131.009,  130.771,  130.541,  130.317,  130.100,  129.890,  129.686,  129.490, 
      129.301,  129.121,  128.948,  128.784,  128.628,  128.482,  128.345,  128.217,  128.097,  127.986, 
      127.883,  127.788,  127.700,  127.620,  127.543,  127.471,  127.307,  127.159,  127.021,  126.888, 
      126.754,  126.617,  126.481,  126.348,  126.219,  126.093,  125.970,  125.850,  125.733,  125.621, 
      125.516,  125.416,  125.324,  125.237,  125.157,  125.084,  125.019,  124.951,  124.880,  124.804, 
      124.726,  124.653,  124.586,  124.525,  124.471,  124.423,  124.339,  124.267,  124.206,  124.169, 
      124.153,  124.133,  124.111,  124.108,  124.128,  124.158,  124.195,  124.240,  124.295,  124.360, 
      124.439,  124.526,  124.614,  124.718,  124.846,  124.984,  125.121,  125.252,  125.376,  125.494, 
      125.608,  125.719,  125.826,  125.931,  126.033,  126.139,  126.252,  126.371,  126.490,  126.618, 
      126.759,  126.909,  127.067,  127.231,  127.402,  127.582,  127.774,  127.977,  128.189,  128.412, 
      128.644,  128.887,  129.139,  129.399,  129.668,  129.947, ],
    [ 132.819,  132.513,  132.251,  132.013,  131.782,  131.559,  131.342,  131.131,  130.928,  130.731, 
      130.543,  130.362,  130.189,  130.024,  129.869,  129.722,  129.585,  129.457,  129.337,  129.225, 
      129.122,  129.027,  128.939,  128.857,  128.781,  128.709,  128.544,  128.394,  128.254,  128.117, 
      127.980,  127.838,  127.700,  127.565,  127.434,  127.306,  127.181,  127.060,  126.940,  126.827, 
      126.720,  126.619,  126.525,  126.436,  126.354,  126.281,  126.212,  126.141,  126.066,  125.985, 
      125.905,  125.830,  125.760,  125.698,  125.642,  125.591,  125.502,  125.424,  125.358,  125.316, 
      125.294,  125.268,  125.239,  125.231,  125.244,  125.268,  125.299,  125.337,  125.385,  125.444, 
      125.516,  125.597,  125.677,  125.774,  125.896,  126.027,  126.157,  126.280,  126.395,  126.505, 
      126.610,  126.712,  126.810,  126.905,  126.998,  127.094,  127.198,  127.308,  127.418,  127.537, 
      127.669,  127.811,  127.960,  128.116,  128.278,  128.450,  128.634,  128.830,  129.034,  129.249, 
      129.474,  129.709,  129.953,  130.207,  130.468,  130.740, ],
    [ 134.060,  133.755,  133.493,  133.254,  133.024,  132.800,  132.583,  132.372,  132.169,  131.972, 
      131.783,  131.602,  131.428,  131.264,  131.108,  130.962,  130.824,  130.696,  130.576,  130.464, 
      130.361,  130.265,  130.177,  130.095,  130.019,  129.947,  129.781,  129.629,  129.485,  129.344, 
      129.202,  129.058,  128.918,  128.782,  128.649,  128.519,  128.393,  128.268,  128.147,  128.032, 
      127.923,  127.821,  127.725,  127.635,  127.551,  127.477,  127.405,  127.330,  127.250,  127.166, 
      127.084,  127.006,  126.935,  126.870,  126.811,  126.759,  126.664,  126.580,  126.510,  126.463, 
      126.435,  126.403,  126.367,  126.353,  126.361,  126.379,  126.403,  126.435,  126.476,  126.528, 
      126.594,  126.667,  126.741,  126.831,  126.946,  127.071,  127.193,  127.309,  127.416,  127.517, 
      127.613,  127.705,  127.794,  127.880,  127.964,  128.050,  128.145,  128.245,  128.347,  128.457, 
      128.580,  128.714,  128.855,  129.003,  129.156,  129.320,  129.496,  129.684,  129.880,  130.087, 
      130.305,  130.533,  130.770,  131.016,  131.270,  131.535, ],
    [ 135.302,  134.996,  134.734,  134.495,  134.264,  134.040,  133.823,  133.612,  133.408,  133.212, 
      133.022,  132.841,  132.668,  132.503,  132.347,  132.201,  132.063,  131.935,  131.814,  131.702, 
      131.598,  131.503,  131.414,  131.333,  131.256,  131.184,  131.017,  130.862,  130.714,  130.570, 
      130.424,  130.278,  130.136,  129.997,  129.862,  129.731,  129.603,  129.476,  129.354,  129.237, 
      129.126,  129.023,  128.925,  128.833,  128.748,  128.672,  128.597,  128.518,  128.435,  128.347, 
      128.262,  128.182,  128.108,  128.041,  127.981,  127.926,  127.826,  127.736,  127.662,  127.609, 
      127.576,  127.537,  127.495,  127.475,  127.478,  127.489,  127.507,  127.532,  127.567,  127.612, 
      127.672,  127.739,  127.805,  127.888,  127.997,  128.116,  128.231,  128.338,  128.437,  128.529, 
      128.617,  128.700,  128.780,  128.856,  128.930,  129.007,  129.093,  129.184,  129.276,  129.377, 
      129.492,  129.618,  129.751,  129.890,  130.035,  130.191,  130.359,  130.539,  130.728,  130.927, 
      131.138,  131.358,  131.588,  131.827,  132.074,  132.331, ],
    [ 136.542,  136.236,  135.974,  135.735,  135.504,  135.280,  135.063,  134.852,  134.648,  134.451, 
      134.262,  134.081,  133.907,  133.742,  133.586,  133.439,  133.301,  133.172,  133.052,  132.940, 
      132.836,  132.740,  132.652,  132.570,  132.493,  132.421,  132.251,  132.093,  131.942,  131.793, 
      131.644,  131.496,  131.352,  131.212,  131.075,  130.942,  130.811,  130.683,  130.560,  130.441, 
      130.329,  130.224,  130.124,  130.031,  129.945,  129.867,  129.787,  129.705,  129.617,  129.526, 
      129.440,  129.358,  129.281,  129.212,  129.150,  129.092,  128.988,  128.892,  128.813,  128.756, 
      128.717,  128.671,  128.622,  128.597,  128.594,  128.600,  128.611,  128.630,  128.658,  128.697, 
      128.751,  128.810,  128.869,  128.946,  129.049,  129.161,  129.269,  129.369,  129.459,  129.542, 
      129.621,  129.695,  129.765,  129.832,  129.896,  129.964,  130.042,  130.124,  130.207,  130.299, 
      130.406,  130.523,  130.647,  130.778,  130.916,  131.063,  131.224,  131.396,  131.577,  131.769, 
      131.972,  132.185,  132.408,  132.639,  132.880,  133.130, ],
    [ 137.782,  137.476,  137.214,  136.975,  136.744,  136.520,  136.302,  136.091,  135.887,  135.690, 
      135.501,  135.319,  135.145,  134.980,  134.824,  134.677,  134.539,  134.410,  134.289,  134.177, 
      134.073,  133.977,  133.889,  133.807,  133.730,  133.656,  133.484,  133.323,  133.168,  133.015, 
      132.863,  132.714,  132.568,  132.426,  132.287,  132.153,  132.020,  131.890,  131.765,  131.645, 
      131.532,  131.424,  131.323,  131.228,  131.142,  131.061,  130.977,  130.890,  130.798,  130.705, 
      130.616,  130.532,  130.454,  130.383,  130.318,  130.258,  130.149,  130.047,  129.963,  129.902, 
      129.858,  129.805,  129.750,  129.719,  129.711,  129.711,  129.715,  129.728,  129.750,  129.782, 
      129.829,  129.882,  129.934,  130.005,  130.101,  130.207,  130.308,  130.400,  130.482,  130.556, 
      130.626,  130.691,  130.752,  130.809,  130.864,  130.922,  130.991,  131.064,  131.138,  131.221, 
      131.320,  131.429,  131.545,  131.668,  131.797,  131.936,  132.089,  132.254,  132.428,  132.612, 
      132.808,  133.014,  133.229,  133.454,  133.687,  133.930, ],
    [ 139.021,  138.715,  138.453,  138.214,  137.983,  137.758,  137.541,  137.330,  137.126,  136.928, 
      136.739,  136.557,  136.383,  136.218,  136.062,  135.915,  135.777,  135.648,  135.527,  135.415, 
      135.311,  135.215,  135.126,  135.043,  134.965,  134.891,  134.716,  134.551,  134.392,  134.236, 
      134.082,  133.930,  133.783,  133.639,  133.499,  133.362,  133.227,  133.096,  132.969,  132.848, 
      132.733,  132.624,  132.521,  132.425,  132.338,  132.253,  132.166,  132.075,  131.978,  131.884, 
      131.793,  131.706,  131.626,  131.554,  131.486,  131.424,  131.310,  131.202,  131.114,  131.048, 
      130.999,  130.938,  130.877,  130.841,  130.828,  130.822,  130.820,  130.826,  130.841,  130.867, 
      130.908,  130.955,  130.999,  131.063,  131.154,  131.253,  131.347,  131.431,  131.505,  131.571, 
      131.632,  131.687,  131.739,  131.787,  131.831,  131.881,  131.940,  132.005,  132.069,  132.144, 
      132.235,  132.336,  132.444,  132.558,  132.679,  132.811,  132.956,  133.113,  133.280,  133.457, 
      133.645,  133.844,  134.052,  134.269,  134.496,  134.732, ],
    [ 140.260,  139.954,  139.691,  139.452,  139.221,  138.997,  138.779,  138.568,  138.363,  138.166, 
      137.976,  137.795,  137.621,  137.456,  137.299,  137.152,  137.014,  136.885,  136.764,  136.652, 
      136.548,  136.451,  136.361,  136.278,  136.199,  136.123,  135.945,  135.777,  135.613,  135.455, 
      135.300,  135.146,  134.997,  134.851,  134.710,  134.571,  134.434,  134.301,  134.173,  134.051, 
      133.934,  133.823,  133.719,  133.622,  133.533,  133.445,  133.353,  133.258,  133.159,  133.062, 
      132.968,  132.880,  132.798,  132.723,  132.654,  132.589,  132.470,  132.357,  132.264,  132.194, 
      132.139,  132.071,  132.004,  131.963,  131.944,  131.932,  131.924,  131.924,  131.933,  131.952, 
      131.987,  132.027,  132.064,  132.122,  132.208,  132.300,  132.387,  132.463,  132.528,  132.586, 
      132.637,  132.684,  132.726,  132.764,  132.799,  132.840,  132.891,  132.946,  133.001,  133.068, 
      133.150,  133.243,  133.343,  133.450,  133.563,  133.686,  133.824,  133.973,  134.133,  134.302, 
      134.484,  134.675,  134.876,  135.087,  135.306,  135.536, ],
    [ 141.498,  141.192,  140.930,  140.690,  140.459,  140.234,  140.017,  139.805,  139.601,  139.404, 
      139.214,  139.032,  138.858,  138.693,  138.537,  138.389,  138.251,  138.122,  138.001,  137.888, 
      137.783,  137.686,  137.596,  137.511,  137.431,  137.354,  137.172,  137.000,  136.834,  136.673, 
      136.516,  136.361,  136.210,  136.063,  135.920,  135.778,  135.640,  135.506,  135.377,  135.253, 
      135.134,  135.022,  134.917,  134.819,  134.728,  134.635,  134.540,  134.439,  134.337,  134.239, 
      134.143,  134.053,  133.970,  133.893,  133.821,  133.754,  133.630,  133.512,  133.414,  133.340, 
      133.279,  133.204,  133.131,  133.085,  133.061,  133.043,  133.028,  133.022,  133.025,  133.038, 
      133.067,  133.100,  133.130,  133.182,  133.261,  133.348,  133.427,  133.496,  133.553,  133.601, 
      133.644,  133.681,  133.714,  133.742,  133.768,  133.799,  133.841,  133.888,  133.934,  133.992, 
      134.066,  134.151,  134.243,  134.342,  134.447,  134.562,  134.693,  134.835,  134.987,  135.149, 
      135.323,  135.508,  135.702,  135.906,  136.118,  136.341, ],
    [ 142.736,  142.430,  142.167,  141.928,  141.697,  141.472,  141.254,  141.043,  140.838,  140.641, 
      140.451,  140.270,  140.096,  139.930,  139.774,  139.626,  139.488,  139.358,  139.236,  139.123, 
      139.018,  138.920,  138.828,  138.742,  138.661,  138.583,  138.398,  138.221,  138.053,  137.891, 
      137.732,  137.575,  137.422,  137.274,  137.128,  136.985,  136.846,  136.710,  136.579,  136.454, 
      136.334,  136.221,  136.114,  136.016,  135.921,  135.824,  135.724,  135.619,  135.516,  135.415, 
      135.318,  135.226,  135.140,  135.061,  134.988,  134.918,  134.789,  134.666,  134.563,  134.486, 
      134.419,  134.336,  134.258,  134.207,  134.177,  134.154,  134.133,  134.120,  134.116,  134.123, 
      134.146,  134.173,  134.196,  134.242,  134.315,  134.396,  134.468,  134.529,  134.577,  134.617, 
      134.650,  134.678,  134.702,  134.721,  134.736,  134.759,  134.792,  134.830,  134.867,  134.917, 
      134.983,  135.060,  135.144,  135.235,  135.332,  135.440,  135.562,  135.697,  135.842,  135.997, 
      136.164,  136.342,  136.529,  136.726,  136.932,  137.148, ],
    [ 143.974,  143.667,  143.405,  143.165,  142.934,  142.709,  142.492,  142.280,  142.076,  141.878, 
      141.689,  141.507,  141.333,  141.167,  141.010,  140.862,  140.723,  140.593,  140.471,  140.357, 
      140.250,  140.151,  140.059,  139.972,  139.889,  139.810,  139.621,  139.442,  139.272,  139.107, 
      138.947,  138.788,  138.634,  138.484,  138.336,  138.192,  138.051,  137.914,  137.782,  137.655, 
      137.533,  137.419,  137.311,  137.211,  137.113,  137.012,  136.908,  136.800,  136.694,  136.591, 
      136.492,  136.398,  136.311,  136.230,  136.154,  136.082,  135.948,  135.820,  135.713,  135.632, 
      135.558,  135.467,  135.384,  135.328,  135.294,  135.264,  135.237,  135.218,  135.208,  135.209, 
      135.226,  135.246,  135.262,  135.302,  135.370,  135.444,  135.510,  135.562,  135.602,  135.633, 
      135.657,  135.676,  135.690,  135.699,  135.705,  135.718,  135.743,  135.772,  135.801,  135.842, 
      135.901,  135.970,  136.046,  136.128,  136.217,  136.318,  136.433,  136.561,  136.698,  136.846, 
      137.006,  137.177,  137.358,  137.547,  137.746,  137.956, ],
    [ 145.211,  144.905,  144.642,  144.403,  144.171,  143.947,  143.729,  143.517,  143.313,  143.115, 
      142.925,  142.743,  142.568,  142.402,  142.245,  142.097,  141.957,  141.826,  141.703,  141.588, 
      141.481,  141.381,  141.287,  141.199,  141.116,  141.035,  140.842,  140.661,  140.489,  140.323, 
      140.161,  140.001,  139.846,  139.693,  139.543,  139.398,  139.256,  139.117,  138.984,  138.855, 
      138.732,  138.616,  138.508,  138.406,  138.304,  138.199,  138.089,  137.979,  137.871,  137.766, 
      137.665,  137.570,  137.481,  137.397,  137.319,  137.245,  137.105,  136.973,  136.862,  136.777, 
      136.696,  136.598,  136.510,  136.450,  136.410,  136.374,  136.341,  136.316,  136.300,  136.295, 
      136.306,  136.319,  136.328,  136.362,  136.425,  136.493,  136.551,  136.596,  136.627,  136.649, 
      136.664,  136.673,  136.678,  136.678,  136.674,  136.678,  136.695,  136.715,  136.734,  136.767, 
      136.818,  136.880,  136.948,  137.023,  137.104,  137.196,  137.305,  137.425,  137.555,  137.696, 
      137.850,  138.014,  138.187,  138.370,  138.563,  138.765, ],
    [ 146.448,  146.142,  145.879,  145.640,  145.408,  145.183,  144.965,  144.754,  144.549,  144.351, 
      144.160,  143.978,  143.803,  143.636,  143.478,  143.329,  143.189,  143.058,  142.934,  142.818, 
      142.709,  142.608,  142.514,  142.425,  142.339,  142.257,  142.062,  141.879,  141.705,  141.538, 
      141.374,  141.213,  141.055,  140.900,  140.750,  140.603,  140.459,  140.320,  140.184,  140.054, 
      139.930,  139.813,  139.704,  139.599,  139.493,  139.383,  139.269,  139.158,  139.048,  138.941, 
      138.838,  138.741,  138.650,  138.565,  138.484,  138.408,  138.262,  138.125,  138.010,  137.922, 
      137.835,  137.729,  137.636,  137.571,  137.526,  137.484,  137.445,  137.414,  137.391,  137.380, 
      137.386,  137.393,  137.394,  137.423,  137.480,  137.542,  137.593,  137.630,  137.652,  137.665, 
      137.671,  137.671,  137.666,  137.656,  137.643,  137.638,  137.647,  137.658,  137.668,  137.693, 
      137.737,  137.790,  137.851,  137.917,  137.991,  138.076,  138.177,  138.290,  138.413,  138.548, 
      138.694,  138.851,  139.018,  139.194,  139.380,  139.576, ],
    [ 147.685,  147.379,  147.116,  146.876,  146.644,  146.419,  146.200,  145.989,  145.783,  145.585, 
      145.394,  145.211,  145.035,  144.868,  144.710,  144.560,  144.419,  144.287,  144.162,  144.045, 
      143.936,  143.834,  143.738,  143.647,  143.561,  143.478,  143.281,  143.097,  142.921,  142.752, 
      142.587,  142.424,  142.264,  142.108,  141.956,  141.807,  141.662,  141.522,  141.385,  141.253, 
      141.128,  141.010,  140.900,  140.791,  140.680,  140.566,  140.449,  140.335,  140.224,  140.115, 
      140.010,  139.911,  139.818,  139.731,  139.648,  139.570,  139.418,  139.278,  139.158,  139.066, 
      138.972,  138.859,  138.762,  138.692,  138.642,  138.594,  138.549,  138.512,  138.483,  138.466, 
      138.466,  138.466,  138.461,  138.483,  138.536,  138.592,  138.636,  138.664,  138.678,  138.681, 
      138.678,  138.669,  138.654,  138.635,  138.612,  138.598,  138.598,  138.601,  138.602,  138.619, 
      138.655,  138.701,  138.754,  138.813,  138.878,  138.956,  139.050,  139.156,  139.272,  139.400, 
      139.539,  139.690,  139.850,  140.020,  140.199,  140.388, ],
    [ 148.921,  148.614,  148.351,  148.111,  147.879,  147.653,  147.434,  147.222,  147.016,  146.817, 
      146.625,  146.442,  146.266,  146.098,  145.939,  145.789,  145.647,  145.514,  145.389,  145.271, 
      145.160,  145.056,  144.959,  144.868,  144.781,  144.697,  144.499,  144.313,  144.136,  143.966, 
      143.798,  143.634,  143.472,  143.314,  143.161,  143.011,  142.865,  142.723,  142.585,  142.452, 
      142.325,  142.207,  142.095,  141.982,  141.867,  141.747,  141.628,  141.512,  141.399,  141.288, 
      141.182,  141.082,  140.987,  140.897,  140.812,  140.731,  140.574,  140.430,  140.306,  140.210, 
      140.109,  139.989,  139.887,  139.813,  139.758,  139.703,  139.652,  139.609,  139.574,  139.552, 
      139.546,  139.540,  139.527,  139.544,  139.592,  139.642,  139.678,  139.699,  139.703,  139.698, 
      139.686,  139.666,  139.642,  139.613,  139.580,  139.558,  139.550,  139.544,  139.536,  139.545, 
      139.574,  139.612,  139.657,  139.709,  139.766,  139.837,  139.923,  140.023,  140.132,  140.252, 
      140.385,  140.529,  140.683,  140.846,  141.018,  141.202, ],
    [ 150.155,  149.848,  149.584,  149.344,  149.111,  148.885,  148.666,  148.453,  148.247,  148.047, 
      147.855,  147.671,  147.494,  147.326,  147.166,  147.016,  146.873,  146.739,  146.612,  146.493, 
      146.382,  146.278,  146.180,  146.087,  146.000,  145.916,  145.716,  145.528,  145.350,  145.178, 
      145.010,  144.842,  144.679,  144.520,  144.365,  144.214,  144.067,  143.923,  143.784,  143.650, 
      143.523,  143.404,  143.288,  143.171,  143.051,  142.927,  142.807,  142.689,  142.573,  142.461, 
      142.353,  142.251,  142.154,  142.063,  141.975,  141.892,  141.730,  141.581,  141.454,  141.354, 
      141.246,  141.118,  141.012,  140.934,  140.873,  140.813,  140.755,  140.707,  140.666,  140.637, 
      140.626,  140.613,  140.593,  140.605,  140.648,  140.692,  140.721,  140.733,  140.728,  140.714, 
      140.693,  140.664,  140.630,  140.590,  140.549,  140.518,  140.502,  140.487,  140.470,  140.472, 
      140.493,  140.524,  140.561,  140.605,  140.655,  140.718,  140.798,  140.890,  140.993,  141.106, 
      141.232,  141.370,  141.517,  141.673,  141.839,  142.016, ],
    [ 151.387,  151.079,  150.815,  150.575,  150.342,  150.115,  149.895,  149.682,  149.475,  149.275, 
      149.083,  148.898,  148.721,  148.552,  148.392,  148.239,  148.096,  147.960,  147.833,  147.714, 
      147.602,  147.497,  147.399,  147.306,  147.218,  147.133,  146.932,  146.743,  146.563,  146.389, 
      146.219,  146.050,  145.886,  145.726,  145.569,  145.417,  145.268,  145.123,  144.983,  144.847, 
      144.720,  144.599,  144.480,  144.358,  144.233,  144.107,  143.984,  143.864,  143.747,  143.633, 
      143.524,  143.420,  143.321,  143.227,  143.138,  143.053,  142.885,  142.733,  142.601,  142.497, 
      142.382,  142.247,  142.137,  142.054,  141.988,  141.922,  141.859,  141.803,  141.757,  141.723, 
      141.706,  141.686,  141.660,  141.667,  141.704,  141.742,  141.764,  141.768,  141.754,  141.730, 
      141.699,  141.661,  141.617,  141.568,  141.517,  141.477,  141.453,  141.429,  141.404,  141.398, 
      141.412,  141.435,  141.465,  141.501,  141.544,  141.599,  141.672,  141.758,  141.854,  141.961, 
      142.080,  142.211,  142.351,  142.502,  142.661,  142.832, ],
    [ 152.617,  152.309,  152.045,  151.803,  151.570,  151.343,  151.123,  150.909,  150.702,  150.502, 
      150.309,  150.123,  149.944,  149.774,  149.613,  149.461,  149.317,  149.181,  149.053,  148.933, 
      148.821,  148.716,  148.617,  148.524,  148.435,  148.349,  148.147,  147.957,  147.775,  147.600, 
      147.428,  147.257,  147.091,  146.930,  146.772,  146.619,  146.469,  146.322,  146.181,  146.045, 
      145.916,  145.793,  145.670,  145.543,  145.413,  145.286,  145.161,  145.039,  144.920,  144.805, 
      144.694,  144.588,  144.487,  144.391,  144.300,  144.213,  144.039,  143.883,  143.749,  143.640, 
      143.516,  143.376,  143.261,  143.174,  143.103,  143.030,  142.961,  142.900,  142.848,  142.808, 
      142.785,  142.759,  142.726,  142.728,  142.760,  142.793,  142.807,  142.802,  142.779,  142.746, 
      142.706,  142.657,  142.604,  142.545,  142.485,  142.437,  142.404,  142.371,  142.338,  142.325, 
      142.331,  142.347,  142.370,  142.398,  142.433,  142.482,  142.548,  142.627,  142.716,  142.816, 
      142.929,  143.053,  143.187,  143.331,  143.484,  143.649, ],
    [ 153.845,  153.537,  153.272,  153.030,  152.796,  152.569,  152.349,  152.134,  151.925,  151.724, 
      151.530,  151.344,  151.166,  150.996,  150.834,  150.681,  150.537,  150.401,  150.273,  150.152, 
      150.039,  149.934,  149.834,  149.740,  149.651,  149.565,  149.361,  149.169,  148.987,  148.810, 
      148.635,  148.464,  148.297,  148.134,  147.975,  147.820,  147.669,  147.521,  147.378,  147.242, 
      147.112,  146.986,  146.858,  146.726,  146.593,  146.464,  146.337,  146.213,  146.093,  145.976, 
      145.863,  145.756,  145.653,  145.555,  145.461,  145.371,  145.193,  145.033,  144.896,  144.782, 
      144.649,  144.504,  144.385,  144.294,  144.217,  144.139,  144.064,  143.997,  143.939,  143.893, 
      143.865,  143.831,  143.792,  143.789,  143.817,  143.843,  143.850,  143.836,  143.804,  143.762, 
      143.712,  143.654,  143.591,  143.521,  143.452,  143.396,  143.356,  143.313,  143.271,  143.251, 
      143.251,  143.259,  143.274,  143.295,  143.322,  143.364,  143.424,  143.496,  143.578,  143.672, 
      143.778,  143.896,  144.024,  144.161,  144.308,  144.466, ],
    [ 155.071,  154.763,  154.497,  154.254,  154.020,  153.792,  153.570,  153.355,  153.147,  152.946, 
      152.752,  152.565,  152.386,  152.215,  152.053,  151.900,  151.756,  151.619,  151.490,  151.370, 
      151.256,  151.150,  151.050,  150.956,  150.866,  150.780,  150.575,  150.381,  150.197,  150.018, 
      149.842,  149.669,  149.501,  149.337,  149.178,  149.021,  148.868,  148.719,  148.575,  148.438, 
      148.307,  148.177,  148.044,  147.907,  147.772,  147.641,  147.512,  147.387,  147.265,  147.147, 
      147.032,  146.922,  146.818,  146.718,  146.622,  146.528,  146.347,  146.183,  146.042,  145.922, 
      145.781,  145.631,  145.508,  145.414,  145.331,  145.247,  145.166,  145.094,  145.029,  144.979, 
      144.944,  144.903,  144.858,  144.851,  144.874,  144.894,  144.893,  144.870,  144.829,  144.777, 
      144.718,  144.650,  144.577,  144.497,  144.419,  144.355,  144.307,  144.255,  144.205,  144.178, 
      144.170,  144.172,  144.179,  144.192,  144.212,  144.247,  144.300,  144.366,  144.441,  144.528, 
      144.628,  144.740,  144.861,  144.992,  145.133,  145.285, ],
    [ 156.293,  155.984,  155.719,  155.476,  155.241,  155.013,  154.791,  154.576,  154.367,  154.165, 
      153.971,  153.784,  153.605,  153.434,  153.272,  153.118,  152.973,  152.836,  152.707,  152.586, 
      152.473,  152.366,  152.266,  152.171,  152.081,  151.994,  151.787,  151.593,  151.406,  151.225, 
      151.049,  150.875,  150.705,  150.540,  150.379,  150.221,  150.067,  149.917,  149.773,  149.635, 
      149.501,  149.365,  149.228,  149.088,  148.951,  148.817,  148.687,  148.560,  148.436,  148.316, 
      148.200,  148.089,  147.982,  147.880,  147.782,  147.684,  147.500,  147.332,  147.188,  147.063, 
      146.912,  146.759,  146.631,  146.533,  146.444,  146.354,  146.268,  146.189,  146.119,  146.064, 
      146.023,  145.975,  145.924,  145.912,  145.931,  145.945,  145.935,  145.904,  145.853,  145.792, 
      145.723,  145.645,  145.562,  145.472,  145.385,  145.314,  145.257,  145.196,  145.138,  145.104, 
      145.090,  145.084,  145.083,  145.089,  145.102,  145.130,  145.176,  145.236,  145.304,  145.385, 
      145.479,  145.584,  145.699,  145.824,  145.959,  146.105, ],
    [ 157.515,  157.205,  156.939,  156.696,  156.461,  156.233,  156.011,  155.795,  155.586,  155.384, 
      155.189,  155.003,  154.823,  154.652,  154.489,  154.335,  154.190,  154.053,  153.923,  153.802, 
      153.688,  153.581,  153.480,  153.385,  153.294,  153.206,  152.999,  152.802,  152.614,  152.432, 
      152.254,  152.079,  151.908,  151.742,  151.579,  151.421,  151.266,  151.114,  150.969,  150.830, 
      150.692,  150.553,  150.409,  150.267,  150.128,  149.993,  149.861,  149.732,  149.607,  149.486, 
      149.368,  149.254,  149.145,  149.041,  148.941,  148.840,  148.652,  148.481,  148.334,  148.202, 
      148.042,  147.885,  147.754,  147.652,  147.558,  147.461,  147.369,  147.284,  147.210,  147.149, 
      147.102,  147.046,  146.990,  146.974,  146.988,  146.995,  146.978,  146.938,  146.877,  146.806, 
      146.727,  146.640,  146.546,  146.446,  146.351,  146.272,  146.207,  146.137,  146.071,  146.030, 
      146.009,  145.996,  145.988,  145.987,  145.992,  146.013,  146.053,  146.106,  146.168,  146.243, 
      146.330,  146.429,  146.538,  146.657,  146.785,  146.925, ],
    [ 158.734,  158.425,  158.159,  157.915,  157.680,  157.451,  157.229,  157.013,  156.804,  156.602, 
      156.407,  156.220,  156.040,  155.868,  155.705,  155.551,  155.406,  155.268,  155.139,  155.017, 
      154.902,  154.795,  154.693,  154.598,  154.506,  154.419,  154.210,  154.011,  153.821,  153.638, 
      153.459,  153.283,  153.111,  152.943,  152.780,  152.620,  152.463,  152.312,  152.166,  152.024, 
      151.881,  151.737,  151.589,  151.445,  151.305,  151.167,  151.034,  150.904,  150.778,  150.654, 
      150.535,  150.419,  150.308,  150.202,  150.098,  149.996,  149.804,  149.629,  149.480,  149.340, 
      149.172,  149.011,  148.876,  148.771,  148.671,  148.568,  148.470,  148.380,  148.299,  148.234, 
      148.181,  148.116,  148.056,  148.035,  148.045,  148.046,  148.020,  147.971,  147.900,  147.820, 
      147.731,  147.634,  147.530,  147.419,  147.316,  147.230,  147.156,  147.077,  147.004,  146.956, 
      146.929,  146.908,  146.893,  146.884,  146.882,  146.896,  146.930,  146.977,  147.033,  147.101, 
      147.182,  147.275,  147.378,  147.490,  147.613,  147.746, ],
    [ 159.953,  159.643,  159.377,  159.134,  158.898,  158.669,  158.447,  158.231,  158.021,  157.819, 
      157.623,  157.436,  157.256,  157.084,  156.921,  156.767,  156.620,  156.483,  156.353,  156.230, 
      156.115,  156.007,  155.906,  155.810,  155.718,  155.630,  155.419,  155.219,  155.028,  154.844, 
      154.663,  154.486,  154.313,  154.144,  153.979,  153.818,  153.660,  153.508,  153.362,  153.216, 
      153.069,  152.918,  152.769,  152.623,  152.480,  152.342,  152.206,  152.075,  151.947,  151.822, 
      151.701,  151.583,  151.470,  151.362,  151.255,  151.151,  150.955,  150.777,  150.625,  150.478, 
      150.302,  150.137,  149.999,  149.889,  149.783,  149.674,  149.571,  149.475,  149.389,  149.318, 
      149.259,  149.187,  149.122,  149.097,  149.102,  149.097,  149.062,  149.003,  148.923,  148.833, 
      148.734,  148.627,  148.513,  148.393,  148.281,  148.188,  148.105,  148.016,  147.936,  147.882, 
      147.848,  147.820,  147.798,  147.781,  147.772,  147.780,  147.808,  147.848,  147.897,  147.959, 
      148.035,  148.121,  148.218,  148.324,  148.441,  148.569, ],
    [ 161.170,  160.861,  160.594,  160.351,  160.115,  159.886,  159.663,  159.447,  159.237,  159.034, 
      158.839,  158.651,  158.471,  158.299,  158.135,  157.981,  157.834,  157.696,  157.566,  157.443, 
      157.328,  157.220,  157.118,  157.021,  156.929,  156.840,  156.627,  156.426,  156.234,  156.049, 
      155.867,  155.688,  155.514,  155.344,  155.178,  155.015,  154.858,  154.705,  154.556,  154.406, 
      154.253,  154.098,  153.947,  153.799,  153.655,  153.515,  153.378,  153.246,  153.116,  152.989, 
      152.866,  152.747,  152.632,  152.521,  152.411,  152.305,  152.106,  151.925,  151.770,  151.614, 
      151.431,  151.263,  151.120,  151.006,  150.895,  150.780,  150.671,  150.569,  150.478,  150.402, 
      150.338,  150.257,  150.187,  150.158,  150.159,  150.147,  150.104,  150.035,  149.945,  149.845, 
      149.736,  149.619,  149.495,  149.364,  149.244,  149.145,  149.053,  148.955,  148.867,  148.807, 
      148.767,  148.731,  148.702,  148.679,  148.662,  148.664,  148.685,  148.719,  148.762,  148.818, 
      148.887,  148.968,  149.059,  149.159,  149.269,  149.392, ],
    [ 162.387,  162.077,  161.811,  161.567,  161.331,  161.101,  160.879,  160.662,  160.452,  160.249, 
      160.054,  159.866,  159.685,  159.513,  159.349,  159.194,  159.047,  158.909,  158.778,  158.655, 
      158.540,  158.431,  158.329,  158.232,  158.138,  158.048,  157.834,  157.632,  157.439,  157.253, 
      157.070,  156.890,  156.714,  156.543,  156.376,  156.213,  156.055,  155.901,  155.748,  155.593, 
      155.435,  155.278,  155.125,  154.975,  154.830,  154.688,  154.550,  154.416,  154.284,  154.156, 
      154.030,  153.909,  153.793,  153.678,  153.566,  153.459,  153.256,  153.072,  152.914,  152.748, 
      152.560,  152.388,  152.242,  152.123,  152.006,  151.885,  151.770,  151.663,  151.566,  151.487, 
      151.416,  151.326,  151.252,  151.219,  151.216,  151.197,  151.146,  151.066,  150.966,  150.857, 
      150.737,  150.609,  150.475,  150.335,  150.207,  150.101,  150.000,  149.892,  149.799,  149.732, 
      149.685,  149.643,  149.606,  149.576,  149.552,  149.547,  149.563,  149.591,  149.628,  149.677, 
      149.741,  149.816,  149.900,  149.995,  150.099,  150.215, ],
    [ 163.603,  163.293,  163.026,  162.782,  162.546,  162.316,  162.093,  161.877,  161.667,  161.463, 
      161.267,  161.079,  160.898,  160.726,  160.562,  160.406,  160.259,  160.121,  159.990,  159.867, 
      159.751,  159.641,  159.538,  159.440,  159.347,  159.256,  159.041,  158.838,  158.644,  158.456, 
      158.272,  158.091,  157.914,  157.742,  157.573,  157.410,  157.251,  157.095,  156.938,  156.778, 
      156.615,  156.457,  156.302,  156.150,  156.003,  155.860,  155.721,  155.585,  155.452,  155.321, 
      155.194,  155.071,  154.953,  154.835,  154.721,  154.612,  154.406,  154.219,  154.057,  153.880, 
      153.687,  153.512,  153.363,  153.240,  153.116,  152.990,  152.869,  152.757,  152.654,  152.570, 
      152.493,  152.395,  152.317,  152.281,  152.273,  152.248,  152.187,  152.096,  151.986,  151.867, 
      151.736,  151.599,  151.455,  151.305,  151.170,  151.056,  150.947,  150.830,  150.730,  150.658, 
      150.604,  150.554,  150.510,  150.472,  150.442,  150.431,  150.441,  150.463,  150.493,  150.537, 
      150.594,  150.663,  150.742,  150.830,  150.929,  151.039, ],
    [ 164.817,  164.507,  164.240,  163.996,  163.760,  163.530,  163.307,  163.090,  162.880,  162.676, 
      162.480,  162.292,  162.111,  161.938,  161.773,  161.618,  161.471,  161.332,  161.200,  161.076, 
      160.960,  160.850,  160.746,  160.648,  160.554,  160.463,  160.247,  160.043,  159.847,  159.658, 
      159.474,  159.291,  159.113,  158.940,  158.771,  158.607,  158.447,  158.287,  158.125,  157.959, 
      157.794,  157.634,  157.478,  157.325,  157.176,  157.031,  156.891,  156.753,  156.618,  156.486, 
      156.357,  156.233,  156.111,  155.990,  155.875,  155.764,  155.555,  155.366,  155.198,  155.011, 
      154.815,  154.636,  154.484,  154.356,  154.226,  154.094,  153.968,  153.849,  153.742,  153.654, 
      153.570,  153.464,  153.382,  153.342,  153.330,  153.297,  153.227,  153.126,  153.006,  152.876, 
      152.735,  152.588,  152.432,  152.273,  152.131,  152.011,  151.893,  151.766,  151.660,  151.582, 
      151.522,  151.465,  151.414,  151.369,  151.332,  151.315,  151.319,  151.335,  151.359,  151.396, 
      151.448,  151.511,  151.584,  151.667,  151.759,  151.864, ],
    [ 166.031,  165.721,  165.454,  165.210,  164.973,  164.743,  164.519,  164.302,  164.092,  163.888, 
      163.692,  163.503,  163.322,  163.149,  162.985,  162.829,  162.681,  162.541,  162.409,  162.285, 
      162.168,  162.057,  161.953,  161.855,  161.761,  161.669,  161.452,  161.247,  161.050,  160.861, 
      160.674,  160.491,  160.312,  160.137,  159.968,  159.803,  159.641,  159.477,  159.309,  159.139, 
      158.973,  158.811,  158.653,  158.498,  158.348,  158.202,  158.060,  157.921,  157.784,  157.650, 
      157.520,  157.393,  157.267,  157.146,  157.029,  156.916,  156.704,  156.512,  156.339,  156.141, 
      155.942,  155.759,  155.605,  155.471,  155.335,  155.197,  155.066,  154.942,  154.829,  154.737, 
      154.646,  154.532,  154.447,  154.403,  154.387,  154.347,  154.266,  154.154,  154.024,  153.884, 
      153.733,  153.575,  153.408,  153.241,  153.092,  152.965,  152.837,  152.702,  152.590,  152.507, 
      152.440,  152.376,  152.318,  152.265,  152.222,  152.199,  152.197,  152.207,  152.225,  152.256, 
      152.303,  152.360,  152.427,  152.504,  152.591,  152.690, ],
    [ 167.244,  166.934,  166.666,  166.422,  166.185,  165.955,  165.731,  165.514,  165.303,  165.100, 
      164.903,  164.714,  164.533,  164.359,  164.194,  164.038,  163.890,  163.750,  163.617,  163.492, 
      163.375,  163.264,  163.160,  163.061,  162.967,  162.875,  162.657,  162.450,  162.253,  162.062, 
      161.875,  161.690,  161.510,  161.335,  161.165,  160.999,  160.833,  160.664,  160.490,  160.318, 
      160.151,  159.987,  159.827,  159.671,  159.520,  159.372,  159.229,  159.088,  158.949,  158.814, 
      158.681,  158.552,  158.423,  158.300,  158.182,  158.068,  157.852,  157.658,  157.477,  157.271, 
      157.068,  156.882,  156.725,  156.586,  156.444,  156.300,  156.163,  156.034,  155.916,  155.820, 
      155.721,  155.600,  155.512,  155.464,  155.444,  155.395,  155.305,  155.181,  155.041,  154.891, 
      154.729,  154.560,  154.382,  154.208,  154.052,  153.917,  153.781,  153.638,  153.520,  153.431, 
      153.357,  153.287,  153.221,  153.161,  153.111,  153.082,  153.075,  153.079,  153.091,  153.117, 
      153.157,  153.209,  153.270,  153.341,  153.422,  153.516, ],
    [ 168.456,  168.146,  167.878,  167.633,  167.397,  167.166,  166.943,  166.725,  166.514,  166.310, 
      166.113,  165.924,  165.742,  165.568,  165.402,  165.246,  165.097,  164.957,  164.824,  164.699, 
      164.582,  164.471,  164.366,  164.267,  164.172,  164.080,  163.861,  163.653,  163.455,  163.262, 
      163.074,  162.888,  162.707,  162.532,  162.361,  162.193,  162.022,  161.847,  161.670,  161.497, 
      161.327,  161.162,  161.001,  160.843,  160.691,  160.542,  160.396,  160.254,  160.114,  159.976, 
      159.842,  159.709,  159.579,  159.454,  159.334,  159.219,  159.000,  158.804,  158.615,  158.400, 
      158.194,  158.005,  157.844,  157.700,  157.551,  157.402,  157.260,  157.125,  157.003,  156.902, 
      156.796,  156.669,  156.577,  156.525,  156.501,  156.443,  156.343,  156.208,  156.057,  155.896, 
      155.724,  155.544,  155.356,  155.174,  155.011,  154.869,  154.723,  154.572,  154.449,  154.355, 
      154.274,  154.196,  154.124,  154.056,  154.000,  153.966,  153.953,  153.951,  153.957,  153.977, 
      154.012,  154.058,  154.113,  154.179,  154.254,  154.343, ],
    [ 169.668,  169.357,  169.089,  168.844,  168.607,  168.376,  168.152,  167.934,  167.723,  167.518, 
      167.321,  167.132,  166.949,  166.775,  166.610,  166.453,  166.304,  166.164,  166.031,  165.906, 
      165.788,  165.676,  165.572,  165.472,  165.376,  165.284,  165.064,  164.856,  164.656,  164.462, 
      164.273,  164.086,  163.904,  163.728,  163.557,  163.384,  163.208,  163.027,  162.849,  162.674, 
      162.503,  162.336,  162.173,  162.015,  161.861,  161.711,  161.563,  161.419,  161.277,  161.138, 
      161.002,  160.865,  160.734,  160.608,  160.486,  160.369,  160.147,  159.949,  159.750,  159.529, 
      159.319,  159.127,  158.964,  158.813,  158.658,  158.504,  158.356,  158.216,  158.089,  157.984, 
      157.871,  157.736,  157.641,  157.586,  157.557,  157.491,  157.380,  157.232,  157.071,  156.899, 
      156.717,  156.526,  156.328,  156.139,  155.970,  155.819,  155.663,  155.506,  155.377,  155.278, 
      155.191,  155.106,  155.026,  154.951,  154.889,  154.849,  154.831,  154.823,  154.823,  154.838, 
      154.867,  154.907,  154.957,  155.017,  155.087,  155.170, ],
    [ 170.877,  170.566,  170.298,  170.053,  169.816,  169.585,  169.360,  169.142,  168.931,  168.726, 
      168.529,  168.339,  168.156,  167.982,  167.816,  167.659,  167.510,  167.370,  167.237,  167.111, 
      166.993,  166.881,  166.776,  166.676,  166.580,  166.487,  166.266,  166.057,  165.856,  165.661, 
      165.471,  165.283,  165.102,  164.925,  164.750,  164.573,  164.390,  164.207,  164.027,  163.851, 
      163.678,  163.510,  163.346,  163.186,  163.030,  162.878,  162.730,  162.584,  162.440,  162.299, 
      162.159,  162.021,  161.888,  161.760,  161.637,  161.518,  161.294,  161.093,  160.882,  160.657, 
      160.444,  160.249,  160.082,  159.926,  159.765,  159.605,  159.451,  159.307,  159.175,  159.064, 
      158.942,  158.803,  158.705,  158.648,  158.614,  158.538,  158.415,  158.256,  158.084,  157.901, 
      157.708,  157.507,  157.298,  157.102,  156.928,  156.768,  156.602,  156.439,  156.305,  156.201, 
      156.106,  156.015,  155.927,  155.846,  155.778,  155.733,  155.709,  155.696,  155.690,  155.698, 
      155.722,  155.757,  155.801,  155.856,  155.920,  155.997, ],
    [ 172.086,  171.775,  171.506,  171.261,  171.023,  170.792,  170.568,  170.350,  170.138,  169.933, 
      169.735,  169.546,  169.363,  169.188,  169.023,  168.865,  168.716,  168.575,  168.442,  168.316, 
      168.197,  168.085,  167.980,  167.879,  167.783,  167.690,  167.468,  167.257,  167.055,  166.860, 
      166.668,  166.481,  166.298,  166.120,  165.941,  165.758,  165.571,  165.386,  165.204,  165.026, 
      164.852,  164.682,  164.517,  164.356,  164.199,  164.046,  163.895,  163.747,  163.602,  163.459, 
      163.316,  163.176,  163.042,  162.912,  162.788,  162.667,  162.441,  162.236,  162.012,  161.784, 
      161.568,  161.371,  161.200,  161.038,  160.870,  160.705,  160.546,  160.397,  160.260,  160.145, 
      160.012,  159.870,  159.769,  159.709,  159.669,  159.584,  159.448,  159.278,  159.096,  158.901, 
      158.697,  158.485,  158.268,  158.065,  157.885,  157.716,  157.539,  157.372,  157.233,  157.122, 
      157.021,  156.923,  156.829,  156.740,  156.667,  156.616,  156.587,  156.568,  156.556,  156.559, 
      156.578,  156.607,  156.646,  156.694,  156.753,  156.825, ],
    [ 173.293,  172.982,  172.714,  172.468,  172.230,  171.999,  171.774,  171.556,  171.344,  171.139, 
      170.942,  170.752,  170.569,  170.394,  170.228,  170.070,  169.921,  169.780,  169.646,  169.520, 
      169.401,  169.289,  169.183,  169.082,  168.986,  168.892,  168.669,  168.457,  168.254,  168.058, 
      167.866,  167.677,  167.495,  167.313,  167.128,  166.939,  166.750,  166.564,  166.381,  166.201, 
      166.025,  165.854,  165.688,  165.526,  165.367,  165.212,  165.060,  164.910,  164.763,  164.617, 
      164.472,  164.331,  164.195,  164.064,  163.938,  163.816,  163.588,  163.376,  163.142,  162.911, 
      162.692,  162.492,  162.317,  162.148,  161.975,  161.805,  161.640,  161.486,  161.345,  161.224, 
      161.082,  160.937,  160.832,  160.770,  160.725,  160.628,  160.479,  160.298,  160.105,  159.899, 
      159.685,  159.460,  159.236,  159.026,  158.840,  158.663,  158.476,  158.303,  158.160,  158.044, 
      157.936,  157.830,  157.729,  157.633,  157.554,  157.499,  157.465,  157.440,  157.422,  157.420, 
      157.433,  157.457,  157.490,  157.534,  157.587,  157.654, ],
    [ 174.500,  174.189,  173.920,  173.675,  173.437,  173.205,  172.981,  172.762,  172.550,  172.345, 
      172.147,  171.957,  171.774,  171.599,  171.432,  171.275,  171.125,  170.984,  170.849,  170.723, 
      170.604,  170.491,  170.385,  170.284,  170.187,  170.093,  169.869,  169.656,  169.453,  169.255, 
      169.063,  168.874,  168.689,  168.502,  168.311,  168.119,  167.929,  167.741,  167.556,  167.375, 
      167.198,  167.026,  166.858,  166.694,  166.534,  166.377,  166.223,  166.072,  165.923,  165.773, 
      165.627,  165.485,  165.347,  165.215,  165.087,  164.964,  164.734,  164.515,  164.272,  164.037, 
      163.815,  163.612,  163.433,  163.258,  163.079,  162.903,  162.734,  162.575,  162.430,  162.302, 
      162.151,  162.003,  161.896,  161.831,  161.780,  161.672,  161.509,  161.317,  161.112,  160.896, 
      160.669,  160.434,  160.202,  159.986,  159.795,  159.608,  159.412,  159.235,  159.086,  158.964, 
      158.850,  158.737,  158.629,  158.527,  158.442,  158.382,  158.342,  158.312,  158.288,  158.281, 
      158.289,  158.307,  158.335,  158.373,  158.421,  158.482, ],
    [ 175.706,  175.395,  175.126,  174.881,  174.643,  174.411,  174.186,  173.967,  173.755,  173.550, 
      173.352,  173.161,  172.978,  172.803,  172.636,  172.478,  172.328,  172.186,  172.052,  171.926, 
      171.806,  171.694,  171.587,  171.485,  171.388,  171.294,  171.069,  170.855,  170.650,  170.452, 
      170.260,  170.070,  169.881,  169.689,  169.492,  169.298,  169.106,  168.917,  168.730,  168.548, 
      168.370,  168.197,  168.028,  167.862,  167.700,  167.542,  167.386,  167.233,  167.081,  166.929, 
      166.782,  166.638,  166.499,  166.365,  166.236,  166.111,  165.879,  165.652,  165.401,  165.163, 
      164.937,  164.733,  164.548,  164.367,  164.182,  164.002,  163.827,  163.663,  163.515,  163.381, 
      163.220,  163.069,  162.959,  162.892,  162.835,  162.714,  162.536,  162.333,  162.118,  161.890, 
      161.653,  161.406,  161.168,  160.946,  160.747,  160.551,  160.348,  160.165,  160.011,  159.885, 
      159.763,  159.644,  159.529,  159.419,  159.330,  159.265,  159.220,  159.183,  159.154,  159.142, 
      159.145,  159.158,  159.180,  159.212,  159.255,  159.311, ],
    [ 176.912,  176.600,  176.332,  176.086,  175.847,  175.616,  175.391,  175.172,  174.959,  174.754, 
      174.555,  174.365,  174.181,  174.006,  173.839,  173.681,  173.531,  173.389,  173.255,  173.128, 
      173.008,  172.895,  172.788,  172.686,  172.588,  172.494,  172.268,  172.053,  171.848,  171.649, 
      171.456,  171.264,  171.069,  170.870,  170.672,  170.476,  170.283,  170.091,  169.904,  169.720, 
      169.541,  169.367,  169.196,  169.029,  168.866,  168.706,  168.548,  168.394,  168.238,  168.085, 
      167.935,  167.790,  167.650,  167.515,  167.384,  167.258,  167.025,  166.784,  166.529,  166.288, 
      166.060,  165.852,  165.662,  165.475,  165.285,  165.098,  164.919,  164.750,  164.599,  164.458, 
      164.288,  164.135,  164.022,  163.952,  163.889,  163.753,  163.562,  163.348,  163.121,  162.882, 
      162.633,  162.377,  162.132,  161.905,  161.699,  161.491,  161.282,  161.094,  160.937,  160.804, 
      160.675,  160.549,  160.427,  160.312,  160.217,  160.148,  160.097,  160.055,  160.020,  160.003, 
      160.000,  160.008,  160.025,  160.052,  160.089,  160.141, ],
    [ 178.116,  177.805,  177.536,  177.290,  177.052,  176.820,  176.594,  176.375,  176.163,  175.957, 
      175.758,  175.568,  175.384,  175.209,  175.042,  174.883,  174.733,  174.591,  174.456,  174.329, 
      174.208,  174.095,  173.988,  173.886,  173.788,  173.693,  173.466,  173.250,  173.045,  172.846, 
      172.651,  172.454,  172.253,  172.050,  171.850,  171.653,  171.458,  171.266,  171.077,  170.892, 
      170.712,  170.536,  170.364,  170.195,  170.030,  169.868,  169.709,  169.552,  169.394,  169.239, 
      169.089,  168.942,  168.800,  168.663,  168.531,  168.406,  168.168,  167.914,  167.656,  167.412, 
      167.182,  166.972,  166.776,  166.582,  166.387,  166.195,  166.010,  165.837,  165.683,  165.534, 
      165.356,  165.200,  165.085,  165.012,  164.942,  164.792,  164.586,  164.361,  164.121,  163.871, 
      163.611,  163.346,  163.095,  162.863,  162.648,  162.429,  162.215,  162.023,  161.861,  161.722, 
      161.586,  161.454,  161.326,  161.204,  161.104,  161.030,  160.974,  160.926,  160.886,  160.863, 
      160.856,  160.859,  160.870,  160.892,  160.924,  160.970, ],
    [ 179.320,  179.009,  178.740,  178.493,  178.255,  178.023,  177.797,  177.578,  177.365,  177.160, 
      176.961,  176.770,  176.587,  176.411,  176.243,  176.085,  175.934,  175.792,  175.657,  175.529, 
      175.409,  175.295,  175.187,  175.085,  174.986,  174.891,  174.663,  174.448,  174.242,  174.042, 
      173.843,  173.640,  173.433,  173.229,  173.028,  172.829,  172.633,  172.439,  172.249,  172.063, 
      171.882,  171.704,  171.531,  171.360,  171.194,  171.030,  170.870,  170.708,  170.549,  170.393, 
      170.241,  170.093,  169.950,  169.812,  169.679,  169.552,  169.310,  169.044,  168.783,  168.536, 
      168.303,  168.090,  167.888,  167.688,  167.487,  167.290,  167.101,  166.923,  166.766,  166.608, 
      166.424,  166.265,  166.147,  166.073,  165.993,  165.827,  165.608,  165.371,  165.120,  164.858, 
      164.585,  164.314,  164.056,  163.819,  163.595,  163.367,  163.148,  162.952,  162.785,  162.640, 
      162.497,  162.357,  162.223,  162.095,  161.990,  161.912,  161.851,  161.797,  161.752,  161.724, 
      161.712,  161.709,  161.715,  161.732,  161.759,  161.800, ],
    [ 180.523,  180.211,  179.942,  179.696,  179.457,  179.225,  179.000,  178.781,  178.568,  178.362, 
      178.163,  177.972,  177.788,  177.612,  177.444,  177.285,  177.135,  176.992,  176.856,  176.729, 
      176.608,  176.494,  176.386,  176.283,  176.185,  176.089,  175.861,  175.645,  175.438,  175.236, 
      175.032,  174.822,  174.613,  174.408,  174.205,  174.004,  173.807,  173.612,  173.421,  173.233, 
      173.050,  172.872,  172.696,  172.525,  172.356,  172.191,  172.027,  171.864,  171.704,  171.547, 
      171.393,  171.244,  171.099,  170.959,  170.826,  170.698,  170.449,  170.174,  169.909,  169.659, 
      169.424,  169.207,  169.000,  168.793,  168.587,  168.385,  168.191,  168.009,  167.848,  167.678, 
      167.491,  167.330,  167.210,  167.133,  167.043,  166.859,  166.627,  166.379,  166.116,  165.842, 
      165.558,  165.280,  165.016,  164.775,  164.541,  164.303,  164.079,  163.879,  163.709,  163.557, 
      163.407,  163.261,  163.118,  162.986,  162.876,  162.793,  162.727,  162.668,  162.617,  162.585, 
      162.568,  162.560,  162.561,  162.572,  162.594,  162.631, ],
    [ 181.726,  181.414,  181.145,  180.899,  180.660,  180.427,  180.202,  179.982,  179.769,  179.563, 
      179.364,  179.172,  178.988,  178.812,  178.644,  178.485,  178.334,  178.191,  178.056,  177.928, 
      177.807,  177.692,  177.584,  177.481,  177.382,  177.286,  177.058,  176.841,  176.633,  176.427, 
      176.216,  176.002,  175.792,  175.585,  175.380,  175.179,  174.979,  174.784,  174.591,  174.403, 
      174.219,  174.038,  173.862,  173.688,  173.518,  173.351,  173.184,  173.019,  172.858,  172.699, 
      172.544,  172.393,  172.247,  172.107,  171.973,  171.844,  171.584,  171.302,  171.035,  170.782, 
      170.545,  170.324,  170.110,  169.897,  169.687,  169.479,  169.281,  169.094,  168.929,  168.748, 
      168.558,  168.395,  168.272,  168.193,  168.090,  167.888,  167.644,  167.384,  167.109,  166.824, 
      166.529,  166.244,  165.975,  165.728,  165.484,  165.238,  165.010,  164.806,  164.631,  164.473, 
      164.316,  164.163,  164.014,  163.876,  163.762,  163.675,  163.603,  163.538,  163.483,  163.445, 
      163.423,  163.410,  163.406,  163.413,  163.429,  163.461, ],
    [ 182.928,  182.616,  182.346,  182.100,  181.861,  181.628,  181.402,  181.183,  180.969,  180.763, 
      180.564,  180.372,  180.188,  180.012,  179.844,  179.684,  179.533,  179.390,  179.254,  179.126, 
      179.005,  178.890,  178.782,  178.678,  178.579,  178.483,  178.255,  178.037,  177.826,  177.613, 
      177.396,  177.181,  176.970,  176.761,  176.555,  176.352,  176.152,  175.955,  175.761,  175.572, 
      175.386,  175.204,  175.026,  174.851,  174.679,  174.509,  174.339,  174.174,  174.011,  173.851, 
      173.695,  173.543,  173.395,  173.254,  173.119,  172.989,  172.715,  172.430,  172.160,  171.904, 
      171.665,  171.439,  171.219,  171.001,  170.785,  170.572,  170.369,  170.180,  170.008,  169.817, 
      169.625,  169.459,  169.334,  169.253,  169.135,  168.915,  168.659,  168.386,  168.099,  167.801, 
      167.498,  167.208,  166.934,  166.679,  166.424,  166.173,  165.940,  165.732,  165.553,  165.388, 
      165.224,  165.064,  164.908,  164.766,  164.647,  164.556,  164.479,  164.408,  164.348,  164.305, 
      164.279,  164.261,  164.251,  164.253,  164.264,  164.291, ],
    [ 184.129,  183.816,  183.547,  183.300,  183.061,  182.829,  182.603,  182.383,  182.169,  181.963, 
      181.763,  181.571,  181.387,  181.211,  181.042,  180.883,  180.731,  180.588,  180.452,  180.324, 
      180.202,  180.087,  179.979,  179.876,  179.776,  179.680,  179.451,  179.231,  179.014,  178.794, 
      178.576,  178.359,  178.146,  177.936,  177.729,  177.525,  177.323,  177.125,  176.931,  176.739, 
      176.552,  176.369,  176.189,  176.012,  175.839,  175.665,  175.495,  175.328,  175.163,  175.002, 
      174.844,  174.691,  174.543,  174.401,  174.265,  174.133,  173.845,  173.557,  173.284,  173.027, 
      172.785,  172.553,  172.327,  172.103,  171.882,  171.665,  171.457,  171.265,  171.087,  170.885, 
      170.691,  170.523,  170.396,  170.312,  170.176,  169.939,  169.671,  169.384,  169.086,  168.775, 
      168.466,  168.169,  167.892,  167.628,  167.362,  167.106,  166.870,  166.657,  166.473,  166.301, 
      166.131,  165.964,  165.801,  165.655,  165.532,  165.436,  165.354,  165.278,  165.213,  165.166, 
      165.134,  165.111,  165.097,  165.093,  165.100,  165.122, ],
    [ 185.329,  185.017,  184.747,  184.500,  184.261,  184.028,  183.802,  183.582,  183.368,  183.162, 
      182.962,  182.770,  182.585,  182.409,  182.241,  182.081,  181.929,  181.786,  181.649,  181.521, 
      181.399,  181.284,  181.176,  181.072,  180.973,  180.877,  180.646,  180.421,  180.196,  179.974, 
      179.754,  179.536,  179.321,  179.110,  178.902,  178.696,  178.494,  178.295,  178.099,  177.906, 
      177.717,  177.533,  177.351,  177.173,  176.996,  176.821,  176.649,  176.481,  176.315,  176.152, 
      175.994,  175.839,  175.690,  175.547,  175.410,  175.274,  174.975,  174.684,  174.408,  174.148, 
      173.904,  173.667,  173.434,  173.205,  172.978,  172.756,  172.544,  172.349,  172.165,  171.953, 
      171.757,  171.587,  171.457,  171.371,  171.215,  170.959,  170.679,  170.381,  170.070,  169.748, 
      169.432,  169.130,  168.848,  168.574,  168.298,  168.038,  167.798,  167.583,  167.393,  167.214, 
      167.037,  166.863,  166.694,  166.543,  166.416,  166.316,  166.229,  166.148,  166.078,  166.026, 
      165.989,  165.961,  165.942,  165.934,  165.935,  165.953, ],
    [ 186.528,  186.216,  185.946,  185.699,  185.460,  185.227,  185.001,  184.780,  184.567,  184.360, 
      184.160,  183.968,  183.783,  183.606,  183.438,  183.278,  183.126,  182.983,  182.847,  182.718, 
      182.596,  182.481,  182.373,  182.269,  182.170,  182.072,  181.837,  181.606,  181.377,  181.153, 
      180.931,  180.712,  180.496,  180.283,  180.074,  179.868,  179.664,  179.464,  179.266,  179.072, 
      178.882,  178.695,  178.513,  178.332,  178.152,  177.976,  177.803,  177.633,  177.466,  177.302, 
      177.142,  176.986,  176.837,  176.693,  176.555,  176.410,  176.104,  175.810,  175.531,  175.269, 
      175.021,  174.779,  174.539,  174.305,  174.073,  173.847,  173.631,  173.433,  173.240,  173.021, 
      172.822,  172.650,  172.519,  172.430,  172.248,  171.978,  171.685,  171.374,  171.051,  170.718, 
      170.396,  170.089,  169.801,  169.518,  169.234,  168.970,  168.726,  168.507,  168.311,  168.125, 
      167.942,  167.762,  167.587,  167.431,  167.300,  167.196,  167.104,  167.017,  166.942,  166.886, 
      166.845,  166.811,  166.787,  166.774,  166.771,  166.784, ],
    [ 187.727,  187.415,  187.145,  186.898,  186.658,  186.425,  186.199,  185.979,  185.765,  185.558, 
      185.358,  185.166,  184.981,  184.804,  184.635,  184.475,  184.324,  184.180,  184.044,  183.915, 
      183.793,  183.678,  183.569,  183.465,  183.364,  183.265,  183.024,  182.786,  182.556,  182.331, 
      182.108,  181.887,  181.670,  181.456,  181.246,  181.038,  180.834,  180.632,  180.433,  180.237, 
      180.045,  179.857,  179.673,  179.488,  179.307,  179.130,  178.956,  178.784,  178.616,  178.451, 
      178.290,  178.134,  177.983,  177.839,  177.696,  177.541,  177.231,  176.935,  176.654,  176.390, 
      176.138,  175.889,  175.644,  175.405,  175.168,  174.937,  174.716,  174.516,  174.310,  174.089, 
      173.887,  173.714,  173.580,  173.487,  173.276,  172.993,  172.686,  172.363,  172.026,  171.687, 
      171.359,  171.048,  170.753,  170.458,  170.169,  169.900,  169.653,  169.431,  169.229,  169.036, 
      168.846,  168.659,  168.478,  168.318,  168.183,  168.075,  167.978,  167.885,  167.806,  167.746, 
      167.700,  167.661,  167.632,  167.615,  167.606,  167.615, ],
    [ 188.926,  188.613,  188.343,  188.096,  187.856,  187.623,  187.396,  187.176,  186.962,  186.755, 
      186.555,  186.363,  186.178,  186.001,  185.832,  185.672,  185.520,  185.377,  185.240,  185.111, 
      184.989,  184.873,  184.763,  184.657,  184.554,  184.453,  184.205,  183.966,  183.734,  183.507, 
      183.283,  183.061,  182.843,  182.628,  182.416,  182.208,  182.002,  181.799,  181.598,  181.401, 
      181.208,  181.018,  180.831,  180.644,  180.462,  180.283,  180.107,  179.936,  179.766,  179.599, 
      179.437,  179.281,  179.129,  178.983,  178.833,  178.671,  178.359,  178.060,  177.776,  177.510, 
      177.253,  176.999,  176.748,  176.503,  176.261,  176.026,  175.802,  175.598,  175.379,  175.156, 
      174.952,  174.777,  174.641,  174.542,  174.301,  174.005,  173.684,  173.348,  172.999,  172.653, 
      172.320,  172.005,  171.702,  171.396,  171.102,  170.830,  170.579,  170.353,  170.145,  169.945, 
      169.748,  169.554,  169.369,  169.205,  169.067,  168.954,  168.852,  168.754,  168.670,  168.605, 
      168.554,  168.511,  168.477,  168.455,  168.442,  168.446, ],
    [ 190.123,  189.810,  189.540,  189.293,  189.053,  188.820,  188.593,  188.373,  188.159,  187.952, 
      187.752,  187.560,  187.375,  187.198,  187.029,  186.869,  186.717,  186.573,  186.436,  186.306, 
      186.183,  186.065,  185.953,  185.845,  185.739,  185.635,  185.385,  185.144,  184.911,  184.683, 
      184.457,  184.234,  184.015,  183.799,  183.586,  183.377,  183.169,  182.965,  182.763,  182.564, 
      182.370,  182.178,  181.987,  181.800,  181.616,  181.436,  181.259,  181.086,  180.915,  180.747, 
      180.585,  180.427,  180.275,  180.125,  179.964,  179.801,  179.486,  179.183,  178.898,  178.630, 
      178.368,  178.106,  177.850,  177.601,  177.354,  177.115,  176.887,  176.678,  176.448,  176.222, 
      176.017,  175.839,  175.702,  175.591,  175.323,  175.013,  174.679,  174.330,  173.970,  173.619, 
      173.280,  172.961,  172.648,  172.332,  172.035,  171.759,  171.505,  171.274,  171.060,  170.853, 
      170.649,  170.449,  170.260,  170.091,  169.950,  169.833,  169.725,  169.622,  169.534,  169.465, 
      169.409,  169.361,  169.322,  169.295,  169.277,  169.277, ],
    [ 191.320,  191.008,  190.738,  190.490,  190.250,  190.017,  189.790,  189.570,  189.356,  189.149, 
      188.949,  188.757,  188.571,  188.394,  188.225,  188.065,  187.912,  187.767,  187.629,  187.497, 
      187.372,  187.252,  187.137,  187.026,  186.919,  186.815,  186.564,  186.322,  186.087,  185.857, 
      185.631,  185.407,  185.186,  184.969,  184.755,  184.544,  184.336,  184.130,  183.926,  183.726, 
      183.530,  183.335,  183.143,  182.954,  182.769,  182.588,  182.410,  182.235,  182.063,  181.895, 
      181.732,  181.573,  181.420,  181.261,  181.094,  180.930,  180.611,  180.307,  180.019,  179.748, 
      179.481,  179.213,  178.952,  178.697,  178.446,  178.202,  177.973,  177.757,  177.516,  177.288, 
      177.081,  176.902,  176.763,  176.630,  176.341,  176.017,  175.670,  175.307,  174.939,  174.582, 
      174.239,  173.914,  173.591,  173.268,  172.966,  172.687,  172.430,  172.194,  171.973,  171.760, 
      171.549,  171.342,  171.149,  170.977,  170.832,  170.710,  170.598,  170.489,  170.397,  170.323, 
      170.263,  170.211,  170.167,  170.135,  170.113,  170.108, ],
    [ 192.517,  192.205,  191.935,  191.687,  191.447,  191.214,  190.987,  190.767,  190.553,  190.345, 
      190.145,  189.952,  189.767,  189.589,  189.419,  189.257,  189.103,  188.957,  188.817,  188.683, 
      188.555,  188.433,  188.317,  188.206,  188.099,  187.994,  187.741,  187.498,  187.262,  187.031, 
      186.803,  186.578,  186.357,  186.139,  185.924,  185.711,  185.501,  185.294,  185.089,  184.888, 
      184.688,  184.491,  184.298,  184.108,  183.921,  183.739,  183.560,  183.384,  183.211,  183.042, 
      182.878,  182.719,  182.562,  182.392,  182.223,  182.057,  181.737,  181.429,  181.140,  180.866, 
      180.593,  180.319,  180.052,  179.792,  179.537,  179.289,  179.057,  178.834,  178.584,  178.354, 
      178.145,  177.964,  177.824,  177.661,  177.356,  177.016,  176.656,  176.280,  175.906,  175.544, 
      175.197,  174.864,  174.529,  174.202,  173.897,  173.614,  173.354,  173.113,  172.885,  172.665, 
      172.448,  172.235,  172.038,  171.862,  171.714,  171.588,  171.470,  171.356,  171.260,  171.182, 
      171.118,  171.060,  171.012,  170.975,  170.948,  170.939, ],
    [ 193.714,  193.401,  193.131,  192.884,  192.644,  192.410,  192.183,  191.962,  191.748,  191.540, 
      191.339,  191.146,  190.959,  190.780,  190.609,  190.445,  190.289,  190.140,  189.998,  189.863, 
      189.735,  189.613,  189.497,  189.385,  189.277,  189.172,  188.918,  188.673,  188.436,  188.204, 
      187.975,  187.749,  187.527,  187.307,  187.091,  186.877,  186.666,  186.457,  186.251,  186.047, 
      185.845,  185.646,  185.452,  185.261,  185.073,  184.890,  184.709,  184.532,  184.358,  184.189, 
      184.024,  183.863,  183.696,  183.522,  183.352,  183.185,  182.861,  182.552,  182.261,  181.982, 
      181.703,  181.423,  181.152,  180.886,  180.627,  180.375,  180.141,  179.905,  179.652,  179.419, 
      179.209,  179.026,  178.885,  178.687,  178.366,  178.012,  177.639,  177.251,  176.871,  176.504, 
      176.154,  175.811,  175.466,  175.135,  174.826,  174.541,  174.277,  174.030,  173.796,  173.569, 
      173.346,  173.127,  172.926,  172.747,  172.596,  172.464,  172.341,  172.223,  172.122,  172.041, 
      171.971,  171.909,  171.856,  171.814,  171.783,  171.771, ],
    [ 194.911,  194.598,  194.327,  194.079,  193.839,  193.605,  193.377,  193.155,  192.940,  192.731, 
      192.529,  192.334,  192.146,  191.964,  191.791,  191.626,  191.470,  191.320,  191.178,  191.043, 
      190.914,  190.792,  190.675,  190.563,  190.454,  190.349,  190.093,  189.847,  189.609,  189.376, 
      189.147,  188.919,  188.696,  188.475,  188.257,  188.042,  187.829,  187.619,  187.411,  187.204, 
      187.000,  186.801,  186.605,  186.413,  186.225,  186.039,  185.858,  185.680,  185.506,  185.335, 
      185.170,  185.004,  184.826,  184.651,  184.479,  184.311,  183.985,  183.674,  183.380,  183.097, 
      182.811,  182.526,  182.250,  181.980,  181.716,  181.460,  181.224,  180.974,  180.719,  180.484, 
      180.272,  180.088,  179.946,  179.708,  179.372,  179.004,  178.615,  178.220,  177.835,  177.463, 
      177.108,  176.756,  176.402,  176.067,  175.755,  175.466,  175.199,  174.946,  174.705,  174.472, 
      174.242,  174.019,  173.814,  173.631,  173.477,  173.340,  173.212,  173.089,  172.985,  172.899, 
      172.825,  172.758,  172.700,  172.654,  172.619,  172.602, ],
    [ 196.105,  195.791,  195.520,  195.271,  195.030,  194.795,  194.566,  194.344,  194.126,  193.916, 
      193.712,  193.516,  193.326,  193.145,  192.972,  192.806,  192.649,  192.499,  192.357,  192.221, 
      192.092,  191.969,  191.852,  191.739,  191.630,  191.524,  191.267,  191.021,  190.781,  190.548, 
      190.317,  190.089,  189.864,  189.642,  189.423,  189.206,  188.992,  188.780,  188.569,  188.360, 
      188.155,  187.955,  187.757,  187.564,  187.374,  187.188,  187.006,  186.827,  186.652,  186.481, 
      186.314,  186.137,  185.956,  185.779,  185.606,  185.437,  185.108,  184.795,  184.500,  184.210, 
      183.918,  183.628,  183.347,  183.072,  182.804,  182.546,  182.304,  182.042,  181.785,  181.549, 
      181.336,  181.149,  181.006,  180.725,  180.372,  179.990,  179.588,  179.187,  178.796,  178.421, 
      178.059,  177.694,  177.336,  176.998,  176.683,  176.391,  176.120,  175.861,  175.613,  175.373, 
      175.136,  174.909,  174.700,  174.515,  174.357,  174.216,  174.082,  173.955,  173.846,  173.757, 
      173.678,  173.606,  173.544,  173.493,  173.454,  173.433, ],
    [ 197.295,  196.980,  196.708,  196.458,  196.215,  195.979,  195.749,  195.525,  195.307,  195.096, 
      194.892,  194.696,  194.506,  194.324,  194.150,  193.985,  193.827,  193.678,  193.534,  193.398, 
      193.269,  193.146,  193.028,  192.915,  192.805,  192.699,  192.441,  192.193,  191.953,  191.718, 
      191.487,  191.257,  191.031,  190.807,  190.587,  190.369,  190.153,  189.939,  189.726,  189.516, 
      189.310,  189.108,  188.909,  188.715,  188.524,  188.337,  188.153,  187.974,  187.798,  187.627, 
      187.453,  187.267,  187.085,  186.907,  186.733,  186.562,  186.231,  185.916,  185.617,  185.323, 
      185.024,  184.729,  184.442,  184.164,  183.891,  183.631,  183.383,  183.110,  182.851,  182.614, 
      182.398,  182.211,  182.064,  181.737,  181.368,  180.972,  180.559,  180.152,  179.756,  179.378, 
      179.006,  178.631,  178.270,  177.928,  177.610,  177.316,  177.039,  176.773,  176.519,  176.273, 
      176.029,  175.799,  175.587,  175.398,  175.236,  175.091,  174.951,  174.820,  174.708,  174.614, 
      174.531,  174.454,  174.388,  174.332,  174.289,  174.264, ],
    [ 198.478,  198.162,  197.889,  197.639,  197.396,  197.159,  196.929,  196.705,  196.487,  196.275, 
      196.071,  195.874,  195.685,  195.503,  195.329,  195.163,  195.005,  194.854,  194.711,  194.574, 
      194.445,  194.321,  194.203,  194.089,  193.980,  193.873,  193.614,  193.365,  193.124,  192.888, 
      192.655,  192.424,  192.197,  191.972,  191.750,  191.531,  191.314,  191.096,  190.881,  190.670, 
      190.463,  190.260,  190.061,  189.865,  189.673,  189.484,  189.301,  189.120,  188.944,  188.770, 
      188.584,  188.397,  188.213,  188.034,  187.858,  187.686,  187.353,  187.037,  186.734,  186.433, 
      186.129,  185.829,  185.537,  185.254,  184.977,  184.715,  184.459,  184.178,  183.917,  183.678, 
      183.461,  183.272,  183.089,  182.744,  182.359,  181.947,  181.527,  181.115,  180.716,  180.332, 
      179.951,  179.567,  179.202,  178.858,  178.536,  178.239,  177.957,  177.685,  177.424,  177.171, 
      176.922,  176.688,  176.473,  176.281,  176.115,  175.965,  175.820,  175.685,  175.569,  175.472, 
      175.383,  175.302,  175.231,  175.171,  175.124,  175.095, ],
    [ 199.658,  199.343,  199.069,  198.818,  198.575,  198.338,  198.108,  197.883,  197.665,  197.454, 
      197.249,  197.052,  196.862,  196.680,  196.505,  196.339,  196.181,  196.030,  195.886,  195.749, 
      195.619,  195.495,  195.377,  195.263,  195.153,  195.045,  194.786,  194.536,  194.294,  194.057, 
      193.823,  193.591,  193.362,  193.136,  192.913,  192.692,  192.471,  192.252,  192.036,  191.824, 
      191.615,  191.411,  191.211,  191.014,  190.821,  190.632,  190.448,  190.267,  190.089,  189.906, 
      189.714,  189.525,  189.340,  189.160,  188.983,  188.810,  188.475,  188.157,  187.849,  187.541, 
      187.232,  186.927,  186.631,  186.344,  186.063,  185.799,  185.528,  185.246,  184.983,  184.742, 
      184.523,  184.333,  184.108,  183.744,  183.345,  182.919,  182.493,  182.076,  181.674,  181.283, 
      180.889,  180.502,  180.133,  179.786,  179.462,  179.161,  178.873,  178.595,  178.327,  178.068, 
      177.814,  177.576,  177.358,  177.164,  176.994,  176.839,  176.688,  176.550,  176.430,  176.328, 
      176.235,  176.150,  176.074,  176.010,  175.959,  175.926, ],
    [ 200.837,  200.521,  200.248,  199.997,  199.754,  199.516,  199.286,  199.061,  198.843,  198.631, 
      198.426,  198.229,  198.038,  197.855,  197.681,  197.514,  197.356,  197.205,  197.061,  196.924, 
      196.793,  196.669,  196.550,  196.436,  196.325,  196.217,  195.957,  195.707,  195.463,  195.225, 
      194.990,  194.756,  194.526,  194.298,  194.074,  193.850,  193.627,  193.407,  193.190,  192.977, 
      192.767,  192.562,  192.360,  192.162,  191.968,  191.779,  191.594,  191.412,  191.232,  191.036, 
      190.843,  190.653,  190.467,  190.285,  190.107,  189.933,  189.597,  189.276,  188.962,  188.648, 
      188.334,  188.024,  187.723,  187.432,  187.149,  186.882,  186.596,  186.312,  186.048,  185.805, 
      185.585,  185.394,  185.122,  184.740,  184.324,  183.889,  183.458,  183.036,  182.630,  182.230, 
      181.826,  181.435,  181.064,  180.714,  180.387,  180.081,  179.787,  179.503,  179.229,  178.963, 
      178.706,  178.464,  178.242,  178.046,  177.872,  177.712,  177.556,  177.414,  177.290,  177.185, 
      177.087,  176.997,  176.917,  176.848,  176.794,  176.757, ],
    [ 202.015,  201.699,  201.426,  201.174,  200.931,  200.693,  200.462,  200.237,  200.019,  199.807, 
      199.602,  199.404,  199.213,  199.030,  198.855,  198.689,  198.530,  198.379,  198.234,  198.097, 
      197.966,  197.841,  197.722,  197.608,  197.497,  197.389,  197.127,  196.876,  196.631,  196.392, 
      196.155,  195.920,  195.689,  195.460,  195.233,  195.007,  194.783,  194.562,  194.344,  194.129, 
      193.919,  193.712,  193.509,  193.310,  193.116,  192.926,  192.740,  192.557,  192.363,  192.166, 
      191.971,  191.780,  191.593,  191.410,  191.231,  191.055,  190.718,  190.393,  190.074,  189.754, 
      189.435,  189.120,  188.815,  188.519,  188.234,  187.961,  187.665,  187.379,  187.113,  186.869, 
      186.647,  186.455,  186.130,  185.730,  185.298,  184.856,  184.420,  183.995,  183.584,  183.173, 
      182.762,  182.367,  181.993,  181.640,  181.311,  181.000,  180.700,  180.409,  180.129,  179.857, 
      179.596,  179.351,  179.126,  178.927,  178.749,  178.583,  178.423,  178.277,  178.150,  178.041, 
      177.938,  177.844,  177.760,  177.686,  177.628,  177.588, ],
    [ 203.192,  202.876,  202.602,  202.351,  202.107,  201.869,  201.638,  201.413,  201.194,  200.981, 
      200.776,  200.578,  200.388,  200.204,  200.029,  199.862,  199.703,  199.552,  199.407,  199.269, 
      199.138,  199.013,  198.894,  198.779,  198.668,  198.559,  198.297,  198.044,  197.799,  197.558, 
      197.320,  197.084,  196.851,  196.621,  196.390,  196.163,  195.938,  195.715,  195.496,  195.281, 
      195.069,  194.861,  194.657,  194.458,  194.263,  194.072,  193.885,  193.695,  193.493,  193.294, 
      193.098,  192.906,  192.718,  192.534,  192.353,  192.178,  191.838,  191.510,  191.184,  190.858, 
      190.534,  190.215,  189.905,  189.606,  189.319,  189.039,  188.733,  188.445,  188.177,  187.932, 
      187.709,  187.510,  187.130,  186.714,  186.269,  185.822,  185.381,  184.953,  184.534,  184.110, 
      183.696,  183.299,  182.921,  182.565,  182.234,  181.918,  181.611,  181.314,  181.028,  180.749, 
      180.485,  180.237,  180.010,  179.807,  179.625,  179.454,  179.290,  179.141,  179.010,  178.896, 
      178.789,  178.691,  178.602,  178.525,  178.462,  178.419, ],
    [ 204.367,  204.051,  203.777,  203.526,  203.281,  203.044,  202.812,  202.587,  202.368,  202.156, 
      201.950,  201.752,  201.561,  201.377,  201.202,  201.035,  200.875,  200.724,  200.579,  200.441, 
      200.310,  200.184,  200.064,  199.949,  199.838,  199.729,  199.465,  199.212,  198.965,  198.723, 
      198.483,  198.246,  198.012,  197.778,  197.546,  197.318,  197.092,  196.868,  196.648,  196.431, 
      196.218,  196.009,  195.805,  195.605,  195.409,  195.218,  195.029,  194.825,  194.622,  194.422, 
      194.225,  194.032,  193.842,  193.657,  193.476,  193.299,  192.957,  192.624,  192.292,  191.961, 
      191.633,  191.308,  190.995,  190.692,  190.403,  190.108,  189.800,  189.510,  189.241,  188.994, 
      188.770,  188.525,  188.125,  187.690,  187.238,  186.785,  186.340,  185.909,  185.480,  185.047, 
      184.629,  184.229,  183.849,  183.491,  183.155,  182.833,  182.521,  182.217,  181.924,  181.642, 
      181.373,  181.123,  180.893,  180.687,  180.500,  180.324,  180.156,  180.003,  179.869,  179.751, 
      179.640,  179.537,  179.444,  179.362,  179.296,  179.249, ],
    [ 205.542,  205.226,  204.952,  204.700,  204.456,  204.218,  203.986,  203.760,  203.541,  203.328, 
      203.123,  202.925,  202.733,  202.550,  202.374,  202.207,  202.047,  201.895,  201.750,  201.612, 
      201.480,  201.355,  201.234,  201.119,  201.007,  200.898,  200.633,  200.378,  200.130,  199.886, 
      199.646,  199.407,  199.170,  198.934,  198.701,  198.472,  198.244,  198.020,  197.799,  197.581, 
      197.367,  197.157,  196.952,  196.752,  196.555,  196.363,  196.161,  195.955,  195.750,  195.549, 
      195.351,  195.157,  194.966,  194.779,  194.598,  194.420,  194.076,  193.738,  193.399,  193.063, 
      192.729,  192.401,  192.084,  191.777,  191.487,  191.176,  190.867,  190.575,  190.305,  190.057, 
      189.832,  189.533,  189.112,  188.662,  188.204,  187.747,  187.299,  186.861,  186.419,  185.982, 
      185.561,  185.158,  184.775,  184.415,  184.074,  183.747,  183.428,  183.118,  182.819,  182.533, 
      182.261,  182.008,  181.776,  181.566,  181.375,  181.194,  181.022,  180.865,  180.728,  180.606, 
      180.490,  180.382,  180.286,  180.200,  180.130,  180.080, ],
    [ 206.716,  206.399,  206.125,  205.873,  205.629,  205.391,  205.159,  204.933,  204.714,  204.501, 
      204.295,  204.097,  203.905,  203.721,  203.546,  203.378,  203.218,  203.066,  202.920,  202.782, 
      202.650,  202.524,  202.403,  202.287,  202.175,  202.065,  201.800,  201.543,  201.294,  201.049, 
      200.807,  200.566,  200.326,  200.090,  199.856,  199.625,  199.396,  199.171,  198.949,  198.730, 
      198.515,  198.305,  198.099,  197.898,  197.701,  197.501,  197.291,  197.083,  196.878,  196.675, 
      196.476,  196.281,  196.089,  195.902,  195.719,  195.541,  195.193,  194.849,  194.504,  194.163, 
      193.825,  193.492,  193.171,  192.863,  192.567,  192.244,  191.933,  191.640,  191.369,  191.119, 
      190.893,  190.532,  190.093,  189.632,  189.168,  188.706,  188.256,  187.809,  187.356,  186.916, 
      186.492,  186.086,  185.701,  185.338,  184.992,  184.658,  184.334,  184.018,  183.713,  183.423, 
      183.148,  182.892,  182.658,  182.445,  182.248,  182.062,  181.887,  181.727,  181.586,  181.460, 
      181.340,  181.228,  181.127,  181.037,  180.964,  180.910, ],
    [ 207.889,  207.572,  207.298,  207.046,  206.801,  206.563,  206.331,  206.105,  205.885,  205.672, 
      205.466,  205.268,  205.076,  204.892,  204.716,  204.548,  204.388,  204.235,  204.090,  203.951, 
      203.819,  203.692,  203.571,  203.455,  203.342,  203.232,  202.965,  202.707,  202.457,  202.211, 
      201.967,  201.723,  201.482,  201.244,  201.009,  200.777,  200.548,  200.321,  200.098,  199.878, 
      199.663,  199.452,  199.246,  199.044,  198.845,  198.631,  198.420,  198.211,  198.005,  197.801, 
      197.601,  197.404,  197.211,  197.024,  196.840,  196.661,  196.309,  195.958,  195.608,  195.262, 
      194.919,  194.582,  194.257,  193.948,  193.642,  193.313,  192.999,  192.705,  192.432,  192.181, 
      191.951,  191.524,  191.066,  190.599,  190.130,  189.665,  189.210,  188.751,  188.292,  187.848, 
      187.421,  187.014,  186.626,  186.259,  185.909,  185.569,  185.238,  184.916,  184.605,  184.312, 
      184.035,  183.775,  183.539,  183.322,  183.122,  182.930,  182.751,  182.588,  182.444,  182.313, 
      182.189,  182.073,  181.968,  181.874,  181.797,  181.741, ],
    [ 209.061,  208.744,  208.470,  208.217,  207.973,  207.734,  207.502,  207.276,  207.056,  206.843, 
      206.637,  206.438,  206.246,  206.062,  205.886,  205.718,  205.557,  205.404,  205.258,  205.119, 
      204.986,  204.859,  204.738,  204.621,  204.508,  204.397,  204.129,  203.871,  203.619,  203.371, 
      203.124,  202.878,  202.637,  202.398,  202.162,  201.929,  201.699,  201.471,  201.246,  201.026, 
      200.810,  200.599,  200.392,  200.189,  199.976,  199.761,  199.548,  199.338,  199.131,  198.926, 
      198.724,  198.526,  198.333,  198.145,  197.961,  197.780,  197.423,  197.065,  196.710,  196.360, 
      196.012,  195.672,  195.343,  195.032,  194.711,  194.380,  194.065,  193.769,  193.495,  193.243, 
      192.956,  192.508,  192.036,  191.564,  191.091,  190.623,  190.159,  189.689,  189.226,  188.780, 
      188.350,  187.940,  187.550,  187.179,  186.823,  186.477,  186.140,  185.811,  185.497,  185.201, 
      184.921,  184.659,  184.420,  184.199,  183.993,  183.797,  183.615,  183.449,  183.301,  183.166, 
      183.037,  182.917,  182.809,  182.710,  182.631,  182.571, ],
    [ 210.232,  209.915,  209.641,  209.388,  209.143,  208.905,  208.672,  208.446,  208.226,  208.013, 
      207.806,  207.607,  207.415,  207.231,  207.054,  206.886,  206.725,  206.572,  206.425,  206.286, 
      206.153,  206.026,  205.904,  205.787,  205.673,  205.562,  205.293,  205.033,  204.779,  204.529, 
      204.280,  204.034,  203.791,  203.551,  203.314,  203.080,  202.848,  202.620,  202.394,  202.173, 
      201.957,  201.745,  201.538,  201.325,  201.106,  200.889,  200.675,  200.464,  200.256,  200.050, 
      199.847,  199.649,  199.455,  199.266,  199.081,  198.898,  198.535,  198.171,  197.811,  197.456, 
      197.104,  196.760,  196.429,  196.115,  195.780,  195.447,  195.130,  194.834,  194.557,  194.304, 
      193.952,  193.483,  193.004,  192.527,  192.050,  191.579,  191.104,  190.625,  190.159,  189.710, 
      189.278,  188.865,  188.473,  188.097,  187.734,  187.383,  187.040,  186.705,  186.388,  186.088, 
      185.806,  185.542,  185.300,  185.075,  184.864,  184.664,  184.479,  184.309,  184.158,  184.018, 
      183.885,  183.761,  183.649,  183.546,  183.464,  183.401, ],
    [ 211.403,  211.086,  210.811,  210.558,  210.313,  210.074,  209.842,  209.615,  209.395,  209.182, 
      208.975,  208.776,  208.583,  208.399,  208.222,  208.053,  207.892,  207.739,  207.592,  207.452, 
      207.318,  207.191,  207.069,  206.951,  206.837,  206.725,  206.455,  206.194,  205.937,  205.685, 
      205.435,  205.188,  204.944,  204.703,  204.465,  204.230,  203.998,  203.768,  203.542,  203.321, 
      203.103,  202.891,  202.679,  202.455,  202.235,  202.016,  201.802,  201.590,  201.380,  201.173, 
      200.970,  200.771,  200.576,  200.386,  200.199,  200.014,  199.645,  199.275,  198.911,  198.550, 
      198.195,  197.847,  197.514,  197.195,  196.848,  196.513,  196.195,  195.897,  195.620,  195.366, 
      194.940,  194.455,  193.970,  193.488,  193.008,  192.530,  192.042,  191.560,  191.091,  190.639, 
      190.205,  189.790,  189.395,  189.014,  188.645,  188.287,  187.939,  187.598,  187.278,  186.975, 
      186.690,  186.424,  186.179,  185.949,  185.734,  185.530,  185.342,  185.169,  185.014,  184.870, 
      184.733,  184.605,  184.488,  184.382,  184.297,  184.231, ],
    [ 212.572,  212.255,  211.980,  211.727,  211.482,  211.243,  211.010,  210.784,  210.563,  210.349, 
      210.143,  209.943,  209.750,  209.565,  209.388,  209.219,  209.058,  208.904,  208.757,  208.617, 
      208.483,  208.355,  208.232,  208.114,  208.000,  207.888,  207.616,  207.352,  207.093,  206.840, 
      206.590,  206.341,  206.096,  205.854,  205.615,  205.379,  205.146,  204.915,  204.689,  204.467, 
      204.249,  204.036,  203.809,  203.585,  203.362,  203.144,  202.928,  202.714,  202.504,  202.296, 
      202.092,  201.892,  201.697,  201.506,  201.316,  201.128,  200.752,  200.378,  200.009,  199.644, 
      199.285,  198.933,  198.599,  198.266,  197.915,  197.579,  197.260,  196.960,  196.682,  196.394, 
      195.917,  195.423,  194.933,  194.447,  193.965,  193.476,  192.979,  192.493,  192.022,  191.568, 
      191.131,  190.714,  190.314,  189.927,  189.553,  189.189,  188.834,  188.491,  188.167,  187.862, 
      187.574,  187.306,  187.057,  186.823,  186.603,  186.396,  186.204,  186.029,  185.870,  185.721, 
      185.580,  185.448,  185.327,  185.218,  185.129,  185.061, ],
    [ 213.741,  213.424,  213.148,  212.895,  212.650,  212.411,  212.178,  211.951,  211.730,  211.516, 
      211.309,  211.109,  210.916,  210.731,  210.554,  210.385,  210.223,  210.069,  209.921,  209.781, 
      209.646,  209.518,  209.395,  209.277,  209.161,  209.049,  208.775,  208.508,  208.249,  207.994, 
      207.743,  207.493,  207.247,  207.004,  206.765,  206.528,  206.293,  206.063,  205.836,  205.613, 
      205.395,  205.168,  204.939,  204.713,  204.490,  204.270,  204.053,  203.838,  203.626,  203.418, 
      203.214,  203.013,  202.817,  202.624,  202.432,  202.241,  201.858,  201.479,  201.105,  200.736, 
      200.373,  200.019,  199.683,  199.335,  198.982,  198.644,  198.324,  198.024,  197.744,  197.386, 
      196.890,  196.390,  195.895,  195.406,  194.917,  194.415,  193.914,  193.425,  192.952,  192.495, 
      192.056,  191.637,  191.232,  190.839,  190.459,  190.090,  189.728,  189.382,  189.055,  188.747, 
      188.457,  188.187,  187.934,  187.696,  187.471,  187.261,  187.066,  186.887,  186.725,  186.572, 
      186.427,  186.291,  186.166,  186.053,  185.961,  185.890, ],
    [ 214.908,  214.591,  214.316,  214.062,  213.817,  213.577,  213.344,  213.117,  212.896,  212.682, 
      212.475,  212.275,  212.081,  211.896,  211.718,  211.549,  211.387,  211.232,  211.084,  210.943, 
      210.809,  210.680,  210.557,  210.438,  210.322,  210.208,  209.931,  209.664,  209.404,  209.148, 
      208.895,  208.645,  208.398,  208.154,  207.913,  207.675,  207.441,  207.210,  206.982,  206.759, 
      206.532,  206.298,  206.068,  205.840,  205.616,  205.395,  205.177,  204.961,  204.749,  204.540, 
      204.335,  204.133,  203.936,  203.740,  203.545,  203.351,  202.963,  202.579,  202.200,  201.828, 
      201.460,  201.104,  200.765,  200.403,  200.049,  199.709,  199.388,  199.086,  198.805,  198.366, 
      197.859,  197.353,  196.854,  196.363,  195.864,  195.352,  194.847,  194.356,  193.881,  193.422, 
      192.981,  192.558,  192.148,  191.749,  191.363,  190.988,  190.621,  190.272,  189.942,  189.632, 
      189.340,  189.067,  188.810,  188.568,  188.338,  188.125,  187.927,  187.746,  187.579,  187.422, 
      187.273,  187.133,  187.004,  186.889,  186.793,  186.720, ],
    [ 216.075,  215.757,  215.482,  215.228,  214.982,  214.743,  214.509,  214.282,  214.061,  213.847, 
      213.639,  213.439,  213.245,  213.059,  212.882,  212.712,  212.549,  212.394,  212.246,  212.105, 
      211.970,  211.841,  211.716,  211.596,  211.479,  211.364,  211.087,  210.819,  210.557,  210.301, 
      210.047,  209.796,  209.548,  209.303,  209.061,  208.823,  208.588,  208.356,  208.128,  207.901, 
      207.662,  207.427,  207.195,  206.967,  206.741,  206.519,  206.300,  206.084,  205.871,  205.661, 
      205.455,  205.253,  205.053,  204.854,  204.657,  204.459,  204.066,  203.677,  203.294,  202.918, 
      202.547,  202.189,  201.840,  201.471,  201.115,  200.774,  200.452,  200.148,  199.845,  199.338, 
      198.825,  198.315,  197.813,  197.316,  196.803,  196.287,  195.780,  195.286,  194.808,  194.347, 
      193.905,  193.477,  193.061,  192.657,  192.265,  191.884,  191.514,  191.161,  190.829,  190.516, 
      190.223,  189.946,  189.686,  189.438,  189.205,  188.989,  188.788,  188.604,  188.433,  188.272, 
      188.118,  187.975,  187.842,  187.724,  187.625,  187.549, ],
    [ 217.240,  216.923,  216.647,  216.393,  216.147,  215.907,  215.674,  215.446,  215.225,  215.010, 
      214.802,  214.602,  214.408,  214.222,  214.044,  213.873,  213.711,  213.556,  213.407,  213.265, 
      213.129,  212.998,  212.873,  212.752,  212.635,  212.520,  212.242,  211.973,  211.710,  211.452, 
      211.198,  210.946,  210.697,  210.451,  210.209,  209.970,  209.735,  209.502,  209.273,  209.031, 
      208.791,  208.555,  208.322,  208.092,  207.866,  207.643,  207.423,  207.206,  206.993,  206.782, 
      206.575,  206.371,  206.168,  205.967,  205.766,  205.565,  205.167,  204.774,  204.387,  204.007, 
      203.633,  203.274,  202.909,  202.538,  202.181,  201.839,  201.515,  201.210,  200.827,  200.308, 
      199.789,  199.275,  198.771,  198.264,  197.740,  197.221,  196.711,  196.215,  195.735,  195.272, 
      194.826,  194.394,  193.972,  193.563,  193.165,  192.778,  192.405,  192.049,  191.714,  191.400, 
      191.104,  190.825,  190.560,  190.308,  190.071,  189.852,  189.648,  189.461,  189.286,  189.120, 
      188.963,  188.816,  188.680,  188.559,  188.457,  188.378, ],
    [ 218.404,  218.087,  217.811,  217.557,  217.311,  217.071,  216.837,  216.609,  216.388,  216.173, 
      215.965,  215.764,  215.570,  215.383,  215.205,  215.034,  214.871,  214.714,  214.565,  214.422, 
      214.285,  214.154,  214.029,  213.908,  213.790,  213.675,  213.396,  213.125,  212.862,  212.604, 
      212.348,  212.095,  211.845,  211.599,  211.356,  211.117,  210.881,  210.648,  210.404,  210.160, 
      209.919,  209.682,  209.448,  209.218,  208.990,  208.766,  208.545,  208.328,  208.114,  207.902, 
      207.694,  207.487,  207.282,  207.077,  206.873,  206.670,  206.267,  205.869,  205.478,  205.094, 
      204.718,  204.357,  203.977,  203.605,  203.246,  202.903,  202.578,  202.272,  201.800,  201.275, 
      200.751,  200.234,  199.724,  199.204,  198.676,  198.153,  197.641,  197.143,  196.661,  196.196, 
      195.746,  195.308,  194.881,  194.466,  194.063,  193.671,  193.295,  192.937,  192.600,  192.283, 
      191.985,  191.702,  191.433,  191.176,  190.937,  190.715,  190.508,  190.317,  190.138,  189.969, 
      189.808,  189.657,  189.517,  189.393,  189.288,  189.206, ],
    [ 219.568,  219.250,  218.974,  218.720,  218.473,  218.233,  217.999,  217.771,  217.549,  217.334, 
      217.125,  216.924,  216.730,  216.543,  216.363,  216.192,  216.027,  215.871,  215.721,  215.578, 
      215.441,  215.310,  215.184,  215.063,  214.945,  214.829,  214.549,  214.277,  214.014,  213.754, 
      213.498,  213.243,  212.993,  212.746,  212.503,  212.263,  212.026,  211.782,  211.534,  211.288, 
      211.046,  210.808,  210.573,  210.342,  210.113,  209.888,  209.667,  209.449,  209.234,  209.022, 
      208.811,  208.601,  208.393,  208.185,  207.978,  207.772,  207.365,  206.963,  206.568,  206.181, 
      205.803,  205.434,  205.045,  204.671,  204.311,  203.967,  203.640,  203.302,  202.770,  202.239, 
      201.711,  201.191,  200.673,  200.141,  199.609,  199.085,  198.570,  198.070,  197.586,  197.118, 
      196.664,  196.220,  195.788,  195.367,  194.958,  194.563,  194.184,  193.823,  193.484,  193.166, 
      192.865,  192.578,  192.305,  192.044,  191.802,  191.577,  191.368,  191.173,  190.990,  190.816, 
      190.651,  190.498,  190.353,  190.227,  190.119,  190.034, ],
    [ 220.730,  220.412,  220.135,  219.881,  219.634,  219.394,  219.160,  218.931,  218.709,  218.493, 
      218.284,  218.082,  217.887,  217.699,  217.519,  217.348,  217.183,  217.026,  216.876,  216.733, 
      216.596,  216.464,  216.338,  216.216,  216.098,  215.982,  215.701,  215.429,  215.164,  214.903, 
      214.646,  214.391,  214.140,  213.893,  213.649,  213.409,  213.164,  212.912,  212.662,  212.415, 
      212.173,  211.934,  211.698,  211.465,  211.236,  211.010,  210.789,  210.570,  210.354,  210.139, 
      209.926,  209.714,  209.502,  209.291,  209.082,  208.874,  208.462,  208.055,  207.657,  207.266, 
      206.888,  206.503,  206.112,  205.737,  205.376,  205.030,  204.703,  204.275,  203.737,  203.201, 
      202.670,  202.146,  201.613,  201.076,  200.542,  200.015,  199.499,  198.996,  198.510,  198.039, 
      197.580,  197.131,  196.693,  196.267,  195.851,  195.454,  195.072,  194.709,  194.368,  194.048, 
      193.744,  193.454,  193.176,  192.911,  192.666,  192.438,  192.226,  192.028,  191.841,  191.663, 
      191.495,  191.337,  191.189,  191.060,  190.950,  190.862, ],
    [ 221.891,  221.572,  221.296,  221.041,  220.794,  220.552,  220.317,  220.088,  219.866,  219.649, 
      219.440,  219.238,  219.043,  218.855,  218.675,  218.503,  218.338,  218.181,  218.031,  217.887, 
      217.750,  217.618,  217.491,  217.369,  217.250,  217.134,  216.852,  216.579,  216.313,  216.052, 
      215.794,  215.538,  215.287,  215.039,  214.795,  214.549,  214.294,  214.040,  213.789,  213.542, 
      213.298,  213.058,  212.821,  212.587,  212.358,  212.132,  211.910,  211.690,  211.472,  211.255, 
      211.039,  210.824,  210.608,  210.395,  210.184,  209.974,  209.557,  209.147,  208.745,  208.352, 
      207.972,  207.571,  207.179,  206.802,  206.440,  206.093,  205.765,  205.245,  204.701,  204.161, 
      203.628,  203.094,  202.550,  202.010,  201.474,  200.944,  200.426,  199.921,  199.433,  198.958, 
      198.492,  198.039,  197.595,  197.163,  196.744,  196.344,  195.960,  195.594,  195.251,  194.929, 
      194.622,  194.328,  194.045,  193.778,  193.529,  193.299,  193.084,  192.882,  192.691,  192.510, 
      192.338,  192.176,  192.025,  191.893,  191.781,  191.690, ],
    [ 223.049,  222.730,  222.453,  222.198,  221.950,  221.709,  221.474,  221.244,  221.021,  220.805, 
      220.596,  220.393,  220.198,  220.010,  219.830,  219.657,  219.493,  219.335,  219.184,  219.040, 
      218.902,  218.770,  218.643,  218.521,  218.402,  218.286,  218.003,  217.729,  217.462,  217.200, 
      216.941,  216.686,  216.434,  216.185,  215.938,  215.679,  215.422,  215.168,  214.916,  214.668, 
      214.423,  214.182,  213.944,  213.710,  213.480,  213.253,  213.030,  212.809,  212.589,  212.369, 
      212.150,  211.931,  211.713,  211.498,  211.284,  211.072,  210.651,  210.237,  209.831,  209.437, 
      209.049,  208.639,  208.245,  207.867,  207.504,  207.156,  206.764,  206.212,  205.663,  205.120, 
      204.582,  204.034,  203.485,  202.942,  202.404,  201.872,  201.352,  200.846,  200.354,  199.874, 
      199.404,  198.944,  198.495,  198.057,  197.636,  197.232,  196.846,  196.479,  196.134,  195.810, 
      195.499,  195.201,  194.914,  194.643,  194.392,  194.159,  193.942,  193.736,  193.541,  193.355, 
      193.180,  193.014,  192.861,  192.727,  192.612,  192.517, ],
    [ 224.205,  223.886,  223.609,  223.353,  223.106,  222.864,  222.629,  222.400,  222.177,  221.960, 
      221.750,  221.548,  221.352,  221.164,  220.983,  220.811,  220.646,  220.488,  220.337,  220.192, 
      220.054,  219.922,  219.795,  219.673,  219.553,  219.436,  219.152,  218.878,  218.610,  218.347, 
      218.089,  217.832,  217.580,  217.331,  217.068,  216.808,  216.550,  216.295,  216.042,  215.793, 
      215.547,  215.305,  215.066,  214.832,  214.601,  214.374,  214.149,  213.926,  213.703,  213.481, 
      213.258,  213.036,  212.816,  212.599,  212.383,  212.168,  211.743,  211.326,  210.917,  210.522, 
      210.118,  209.706,  209.311,  208.932,  208.567,  208.219,  207.733,  207.176,  206.623,  206.078, 
      205.531,  204.971,  204.419,  203.874,  203.333,  202.799,  202.277,  201.769,  201.273,  200.787, 
      200.312,  199.847,  199.393,  198.950,  198.527,  198.120,  197.732,  197.362,  197.016,  196.689, 
      196.375,  196.073,  195.782,  195.509,  195.255,  195.019,  194.799,  194.589,  194.389,  194.200, 
      194.021,  193.852,  193.696,  193.559,  193.442,  193.344, ],
    [ 225.360,  225.041,  224.764,  224.509,  224.261,  224.019,  223.783,  223.554,  223.331,  223.114, 
      222.904,  222.701,  222.505,  222.317,  222.136,  221.963,  221.798,  221.640,  221.489,  221.344, 
      221.206,  221.073,  220.946,  220.823,  220.703,  220.586,  220.301,  220.026,  219.757,  219.495, 
      219.235,  218.978,  218.725,  218.461,  218.197,  217.936,  217.677,  217.421,  217.167,  216.917, 
      216.670,  216.427,  216.188,  215.953,  215.722,  215.493,  215.267,  215.041,  214.816,  214.590, 
      214.364,  214.140,  213.918,  213.698,  213.479,  213.262,  212.834,  212.413,  212.002,  211.606, 
      211.186,  210.773,  210.376,  209.996,  209.631,  209.265,  208.700,  208.138,  207.582,  207.032, 
      206.470,  205.906,  205.351,  204.804,  204.261,  203.725,  203.201,  202.690,  202.190,  201.699, 
      201.218,  200.748,  200.288,  199.842,  199.416,  199.007,  198.617,  198.246,  197.898,  197.567, 
      197.250,  196.943,  196.649,  196.373,  196.116,  195.878,  195.655,  195.440,  195.237,  195.045, 
      194.863,  194.690,  194.532,  194.391,  194.273,  194.170, ],
    [ 226.515,  226.196,  225.918,  225.663,  225.415,  225.173,  224.937,  224.707,  224.484,  224.267, 
      224.057,  223.853,  223.657,  223.469,  223.288,  223.115,  222.950,  222.792,  222.640,  222.495, 
      222.356,  222.223,  222.096,  221.972,  221.852,  221.735,  221.449,  221.173,  220.905,  220.641, 
      220.381,  220.124,  219.857,  219.590,  219.325,  219.063,  218.803,  218.546,  218.291,  218.040, 
      217.793,  217.549,  217.310,  217.074,  216.842,  216.612,  216.383,  216.155,  215.926,  215.696, 
      215.468,  215.242,  215.018,  214.795,  214.575,  214.356,  213.924,  213.499,  213.088,  212.681, 
      212.254,  211.839,  211.441,  211.060,  210.694,  210.234,  209.664,  209.098,  208.540,  207.980, 
      207.406,  206.840,  206.283,  205.733,  205.188,  204.650,  204.125,  203.609,  203.104,  202.608, 
      202.122,  201.647,  201.181,  200.734,  200.304,  199.894,  199.501,  199.129,  198.778,  198.445, 
      198.123,  197.812,  197.515,  197.237,  196.978,  196.737,  196.509,  196.292,  196.085,  195.889, 
      195.704,  195.527,  195.366,  195.223,  195.103,  194.996, ],
    [ 227.668,  227.349,  227.072,  226.816,  226.567,  226.325,  226.089,  225.860,  225.636,  225.419, 
      225.208,  225.005,  224.809,  224.620,  224.439,  224.266,  224.100,  223.942,  223.790,  223.645, 
      223.506,  223.373,  223.245,  223.121,  223.001,  222.883,  222.597,  222.321,  222.051,  221.787, 
      221.527,  221.258,  220.987,  220.718,  220.452,  220.189,  219.929,  219.670,  219.415,  219.163, 
      218.915,  218.671,  218.431,  218.194,  217.961,  217.728,  217.497,  217.265,  217.033,  216.801, 
      216.571,  216.342,  216.116,  215.891,  215.668,  215.448,  215.012,  214.585,  214.172,  213.750, 
      213.321,  212.905,  212.506,  212.124,  211.756,  211.201,  210.626,  210.057,  209.494,  208.918, 
      208.341,  207.772,  207.213,  206.661,  206.114,  205.575,  205.046,  204.526,  204.015,  203.515, 
      203.023,  202.542,  202.074,  201.623,  201.192,  200.779,  200.385,  200.011,  199.658,  199.321, 
      198.996,  198.680,  198.381,  198.100,  197.838,  197.595,  197.364,  197.142,  196.932,  196.732, 
      196.543,  196.363,  196.201,  196.055,  195.932,  195.822, ],
    [ 228.821,  228.502,  228.224,  227.968,  227.719,  227.477,  227.241,  227.011,  226.788,  226.570, 
      226.360,  226.156,  225.960,  225.771,  225.589,  225.416,  225.250,  225.091,  224.939,  224.794, 
      224.655,  224.521,  224.393,  224.269,  224.148,  224.030,  223.744,  223.467,  223.198,  222.933, 
      222.662,  222.387,  222.115,  221.845,  221.579,  221.315,  221.053,  220.794,  220.537,  220.285, 
      220.037,  219.792,  219.551,  219.314,  219.078,  218.843,  218.608,  218.373,  218.138,  217.904, 
      217.672,  217.441,  217.212,  216.985,  216.761,  216.538,  216.099,  215.671,  215.256,  214.818, 
      214.387,  213.970,  213.571,  213.187,  212.748,  212.164,  211.585,  211.014,  210.440,  209.854, 
      209.275,  208.703,  208.142,  207.588,  207.039,  206.499,  205.965,  205.441,  204.925,  204.419, 
      203.922,  203.435,  202.965,  202.512,  202.078,  201.664,  201.268,  200.893,  200.536,  200.196, 
      199.866,  199.548,  199.246,  198.962,  198.698,  198.452,  198.217,  197.992,  197.778,  197.575, 
      197.382,  197.199,  197.034,  196.886,  196.762,  196.647, ],
    [ 229.973,  229.653,  229.375,  229.120,  228.871,  228.629,  228.392,  228.162,  227.938,  227.721, 
      227.510,  227.306,  227.110,  226.920,  226.739,  226.565,  226.399,  226.240,  226.088,  225.942, 
      225.803,  225.669,  225.540,  225.416,  225.296,  225.178,  224.891,  224.614,  224.344,  224.070, 
      223.792,  223.515,  223.242,  222.972,  222.704,  222.439,  222.176,  221.916,  221.660,  221.407, 
      221.158,  220.913,  220.671,  220.431,  220.192,  219.955,  219.716,  219.478,  219.241,  219.005, 
      218.771,  218.538,  218.307,  218.078,  217.852,  217.628,  217.186,  216.756,  216.328,  215.885, 
      215.453,  215.035,  214.635,  214.250,  213.714,  213.126,  212.544,  211.967,  211.377,  210.789, 
      210.207,  209.633,  209.070,  208.514,  207.964,  207.420,  206.882,  206.352,  205.832,  205.320, 
      204.818,  204.328,  203.855,  203.400,  202.965,  202.548,  202.151,  201.773,  201.414,  201.070, 
      200.736,  200.414,  200.110,  199.824,  199.558,  199.309,  199.070,  198.841,  198.623,  198.417, 
      198.220,  198.035,  197.868,  197.718,  197.591,  197.472, ],
    [ 231.124,  230.804,  230.526,  230.270,  230.021,  229.779,  229.542,  229.312,  229.088,  228.870, 
      228.659,  228.456,  228.259,  228.069,  227.887,  227.713,  227.547,  227.388,  227.236,  227.090, 
      226.950,  226.816,  226.688,  226.564,  226.443,  226.325,  226.037,  225.760,  225.481,  225.199, 
      224.920,  224.643,  224.368,  224.097,  223.829,  223.563,  223.299,  223.039,  222.782,  222.528, 
      222.278,  222.032,  221.789,  221.546,  221.305,  221.064,  220.822,  220.582,  220.343,  220.105, 
      219.868,  219.633,  219.400,  219.170,  218.942,  218.716,  218.271,  217.840,  217.396,  216.952, 
      216.519,  216.100,  215.698,  215.272,  214.677,  214.085,  213.501,  212.911,  212.313,  211.722, 
      211.138,  210.562,  209.997,  209.439,  208.887,  208.339,  207.797,  207.262,  206.736,  206.219, 
      205.712,  205.220,  204.744,  204.287,  203.850,  203.432,  203.033,  202.653,  202.290,  201.943, 
      201.604,  201.280,  200.973,  200.685,  200.416,  200.164,  199.921,  199.689,  199.468,  199.258, 
      199.058,  198.871,  198.701,  198.549,  198.419,  198.296, ],
    [ 232.274,  231.954,  231.676,  231.420,  231.171,  230.928,  230.692,  230.461,  230.237,  230.019, 
      229.808,  229.604,  229.407,  229.217,  229.035,  228.861,  228.695,  228.536,  228.383,  228.237, 
      228.097,  227.964,  227.835,  227.710,  227.589,  227.471,  227.183,  226.898,  226.611,  226.328, 
      226.047,  225.769,  225.494,  225.222,  224.952,  224.686,  224.422,  224.161,  223.903,  223.649, 
      223.398,  223.151,  222.904,  222.660,  222.415,  222.170,  221.927,  221.684,  221.443,  221.203, 
      220.963,  220.726,  220.492,  220.260,  220.030,  219.803,  219.357,  218.922,  218.464,  218.019, 
      217.584,  217.164,  216.762,  216.238,  215.638,  215.044,  214.453,  213.848,  213.247,  212.654, 
      212.067,  211.490,  210.922,  210.363,  209.808,  209.256,  208.708,  208.169,  207.638,  207.116, 
      206.605,  206.110,  205.632,  205.174,  204.734,  204.315,  203.915,  203.531,  203.165,  202.813, 
      202.472,  202.145,  201.836,  201.545,  201.274,  201.019,  200.772,  200.536,  200.312,  200.100, 
      199.896,  199.706,  199.533,  199.379,  199.247,  199.119, ],
    [ 233.424,  233.104,  232.825,  232.569,  232.320,  232.077,  231.840,  231.609,  231.385,  231.167, 
      230.956,  230.752,  230.554,  230.365,  230.183,  230.009,  229.842,  229.683,  229.530,  229.384, 
      229.244,  229.110,  228.981,  228.857,  228.735,  228.617,  228.322,  228.027,  227.739,  227.455, 
      227.174,  226.895,  226.619,  226.346,  226.075,  225.808,  225.544,  225.282,  225.024,  224.769, 
      224.517,  224.267,  224.018,  223.770,  223.521,  223.275,  223.029,  222.785,  222.541,  222.298, 
      222.057,  221.818,  221.582,  221.348,  221.117,  220.888,  220.442,  219.990,  219.532,  219.085, 
      218.649,  218.229,  217.807,  217.200,  216.597,  216.000,  215.394,  214.783,  214.180,  213.584, 
      212.996,  212.416,  211.847,  211.287,  210.727,  210.170,  209.618,  209.073,  208.537,  208.009, 
      207.497,  207.000,  206.520,  206.059,  205.618,  205.198,  204.795,  204.409,  204.039,  203.683, 
      203.339,  203.010,  202.698,  202.405,  202.132,  201.872,  201.623,  201.383,  201.155,  200.939, 
      200.733,  200.541,  200.366,  200.210,  200.074,  199.942, ],
    [ 234.572,  234.252,  233.973,  233.717,  233.468,  233.225,  232.988,  232.757,  232.532,  232.314, 
      232.103,  231.899,  231.702,  231.512,  231.330,  231.156,  230.989,  230.830,  230.677,  230.530, 
      230.390,  230.256,  230.127,  230.003,  229.880,  229.755,  229.451,  229.156,  228.866,  228.582, 
      228.300,  228.020,  227.743,  227.469,  227.198,  226.930,  226.665,  226.403,  226.144,  225.888, 
      225.634,  225.381,  225.129,  224.877,  224.626,  224.377,  224.130,  223.883,  223.637,  223.392, 
      223.150,  222.909,  222.671,  222.436,  222.203,  221.974,  221.526,  221.059,  220.598,  220.150, 
      219.714,  219.292,  218.772,  218.160,  217.555,  216.950,  216.330,  215.717,  215.111,  214.514, 
      213.924,  213.342,  212.772,  212.207,  211.644,  211.081,  210.524,  209.974,  209.433,  208.902, 
      208.387,  207.888,  207.406,  206.944,  206.501,  206.079,  205.674,  205.285,  204.912,  204.551, 
      204.205,  203.874,  203.560,  203.265,  202.988,  202.725,  202.472,  202.229,  201.998,  201.778, 
      201.569,  201.375,  201.198,  201.040,  200.901,  200.764, ],
    [ 235.720,  235.400,  235.121,  234.864,  234.615,  234.372,  234.135,  233.905,  233.680,  233.462, 
      233.250,  233.046,  232.849,  232.659,  232.477,  232.302,  232.135,  231.976,  231.823,  231.677, 
      231.536,  231.402,  231.267,  231.137,  231.009,  230.884,  230.579,  230.283,  229.993,  229.707, 
      229.425,  229.144,  228.866,  228.591,  228.320,  228.052,  227.786,  227.523,  227.263,  227.005, 
      226.748,  226.492,  226.237,  225.982,  225.729,  225.478,  225.228,  224.979,  224.731,  224.485, 
      224.240,  223.999,  223.759,  223.522,  223.289,  223.059,  222.601,  222.126,  221.664,  221.215, 
      220.778,  220.354,  219.734,  219.120,  218.510,  217.888,  217.265,  216.649,  216.042,  215.442, 
      214.850,  214.267,  213.694,  213.126,  212.557,  211.990,  211.428,  210.873,  210.326,  209.794, 
      209.277,  208.775,  208.292,  207.827,  207.384,  206.960,  206.552,  206.160,  205.782,  205.419, 
      205.070,  204.737,  204.420,  204.123,  203.844,  203.577,  203.320,  203.074,  202.840,  202.617, 
      202.405,  202.209,  202.029,  201.870,  201.727,  201.586, ],
    [ 236.867,  236.547,  236.268,  236.012,  235.762,  235.519,  235.283,  235.052,  234.827,  234.609, 
      234.397,  234.193,  233.995,  233.805,  233.623,  233.448,  233.281,  233.122,  232.968,  232.817, 
      232.671,  232.532,  232.397,  232.266,  232.138,  232.012,  231.707,  231.410,  231.119,  230.832, 
      230.548,  230.266,  229.988,  229.714,  229.442,  229.173,  228.907,  228.643,  228.380,  228.119, 
      227.860,  227.600,  227.342,  227.085,  226.831,  226.577,  226.325,  226.074,  225.824,  225.576, 
      225.330,  225.086,  224.845,  224.608,  224.374,  224.144,  223.669,  223.193,  222.730,  222.280, 
      221.842,  221.318,  220.694,  220.077,  219.456,  218.824,  218.198,  217.581,  216.971,  216.370, 
      215.776,  215.191,  214.615,  214.042,  213.468,  212.896,  212.329,  211.769,  211.219,  210.685, 
      210.165,  209.662,  209.177,  208.711,  208.266,  207.840,  207.429,  207.033,  206.652,  206.286, 
      205.934,  205.599,  205.281,  204.981,  204.698,  204.428,  204.168,  203.918,  203.681,  203.455, 
      203.240,  203.042,  202.861,  202.700,  202.553,  202.406, ],
    [ 238.014,  237.694,  237.416,  237.159,  236.909,  236.666,  236.429,  236.198,  235.973,  235.755, 
      235.543,  235.339,  235.141,  234.951,  234.769,  234.591,  234.420,  234.256,  234.098,  233.946, 
      233.800,  233.660,  233.525,  233.394,  233.265,  233.140,  232.833,  232.535,  232.244,  231.956, 
      231.671,  231.389,  231.111,  230.835,  230.563,  230.293,  230.026,  229.760,  229.495,  229.232, 
      228.968,  228.706,  228.445,  228.187,  227.930,  227.674,  227.420,  227.167,  226.916,  226.666, 
      226.419,  226.173,  225.931,  225.693,  225.459,  225.226,  224.737,  224.260,  223.796,  223.345, 
      222.906,  222.279,  221.652,  221.030,  220.393,  219.758,  219.130,  218.510,  217.899,  217.296, 
      216.700,  216.113,  215.533,  214.955,  214.377,  213.799,  213.228,  212.662,  212.111,  211.574, 
      211.052,  210.548,  210.061,  209.594,  209.148,  208.719,  208.305,  207.905,  207.520,  207.152, 
      206.798,  206.460,  206.141,  205.838,  205.552,  205.278,  205.014,  204.762,  204.522,  204.292, 
      204.076,  203.876,  203.692,  203.530,  203.378,  203.227, ],
    [ 239.161,  238.841,  238.562,  238.305,  238.055,  237.812,  237.575,  237.344,  237.119,  236.901, 
      236.689,  236.481,  236.280,  236.086,  235.900,  235.721,  235.549,  235.385,  235.226,  235.074, 
      234.928,  234.788,  234.652,  234.521,  234.392,  234.266,  233.959,  233.660,  233.367,  233.079, 
      232.794,  232.511,  232.232,  231.956,  231.683,  231.413,  231.144,  230.875,  230.608,  230.340, 
      230.074,  229.809,  229.547,  229.286,  229.027,  228.769,  228.513,  228.259,  228.006,  227.755, 
      227.505,  227.259,  227.016,  226.778,  226.543,  226.294,  225.804,  225.326,  224.861,  224.409, 
      223.876,  223.238,  222.609,  221.971,  221.328,  220.691,  220.061,  219.439,  218.826,  218.221, 
      217.624,  217.033,  216.449,  215.866,  215.282,  214.700,  214.123,  213.555,  213.001,  212.462, 
      211.939,  211.433,  210.944,  210.477,  210.028,  209.596,  209.179,  208.775,  208.388,  208.017, 
      207.661,  207.321,  206.999,  206.695,  206.405,  206.127,  205.860,  205.604,  205.361,  205.129, 
      204.911,  204.708,  204.523,  204.358,  204.203,  204.047, ],
    [ 240.307,  239.987,  239.708,  239.451,  239.201,  238.958,  238.718,  238.484,  238.256,  238.034, 
      237.819,  237.611,  237.409,  237.215,  237.028,  236.849,  236.677,  236.512,  236.354,  236.202, 
      236.055,  235.915,  235.779,  235.647,  235.518,  235.392,  235.084,  234.784,  234.490,  234.202, 
      233.916,  233.633,  233.353,  233.077,  232.803,  232.530,  232.259,  231.988,  231.716,  231.446, 
      231.177,  230.911,  230.647,  230.384,  230.122,  229.863,  229.605,  229.349,  229.095,  228.842, 
      228.591,  228.344,  228.102,  227.863,  227.612,  227.362,  226.871,  226.391,  225.925,  225.473, 
      224.836,  224.197,  223.559,  222.908,  222.262,  221.622,  220.991,  220.367,  219.752,  219.146, 
      218.546,  217.951,  217.361,  216.773,  216.185,  215.598,  215.016,  214.446,  213.891,  213.350, 
      212.825,  212.317,  211.828,  211.358,  210.907,  210.472,  210.052,  209.644,  209.254,  208.881, 
      208.524,  208.182,  207.858,  207.550,  207.256,  206.976,  206.706,  206.447,  206.200,  205.965, 
      205.745,  205.540,  205.353,  205.186,  205.027,  204.867, ],
    [ 241.451,  241.128,  240.847,  240.587,  240.334,  240.088,  239.848,  239.613,  239.385,  239.163, 
      238.947,  238.739,  238.537,  238.343,  238.156,  237.976,  237.804,  237.639,  237.481,  237.328, 
      237.182,  237.041,  236.904,  236.773,  236.644,  236.517,  236.208,  235.907,  235.613,  235.324, 
      235.038,  234.754,  234.474,  234.196,  233.921,  233.646,  233.372,  233.097,  232.822,  232.550, 
      232.279,  232.011,  231.744,  231.479,  231.216,  230.955,  230.695,  230.438,  230.182,  229.928, 
      229.677,  229.430,  229.186,  228.934,  228.680,  228.430,  227.937,  227.456,  226.990,  226.445, 
      225.795,  225.152,  224.496,  223.842,  223.194,  222.553,  221.919,  221.294,  220.677,  220.069, 
      219.465,  218.866,  218.271,  217.678,  217.085,  216.492,  215.908,  215.337,  214.778,  214.236, 
      213.710,  213.200,  212.710,  212.239,  211.785,  211.348,  210.923,  210.512,  210.120,  209.745, 
      209.385,  209.042,  208.715,  208.404,  208.107,  207.823,  207.550,  207.288,  207.038,  206.801, 
      206.579,  206.372,  206.184,  206.014,  205.850,  205.687, ],
    [ 242.580,  242.258,  241.976,  241.716,  241.463,  241.217,  240.976,  240.741,  240.513,  240.290, 
      240.075,  239.866,  239.664,  239.470,  239.283,  239.103,  238.931,  238.765,  238.607,  238.454, 
      238.307,  238.166,  238.029,  237.897,  237.767,  237.640,  237.331,  237.030,  236.735,  236.446, 
      236.159,  235.875,  235.593,  235.314,  235.036,  234.759,  234.481,  234.203,  233.926,  233.652, 
      233.379,  233.109,  232.840,  232.573,  232.308,  232.045,  231.785,  231.525,  231.268,  231.013, 
      230.762,  230.514,  230.260,  230.002,  229.748,  229.496,  229.002,  228.521,  228.054,  227.404, 
      226.753,  226.095,  225.432,  224.775,  224.125,  223.482,  222.847,  222.219,  221.601,  220.990, 
      220.382,  219.777,  219.178,  218.580,  217.982,  217.385,  216.800,  216.226,  215.666,  215.122, 
      214.594,  214.084,  213.592,  213.118,  212.662,  212.221,  211.793,  211.380,  210.985,  210.608, 
      210.246,  209.901,  209.572,  209.257,  208.957,  208.670,  208.394,  208.129,  207.876,  207.636, 
      207.412,  207.203,  207.014,  206.841,  206.671,  206.505, ],
    [ 243.709,  243.386,  243.104,  242.844,  242.591,  242.344,  242.103,  241.869,  241.640,  241.418, 
      241.202,  240.993,  240.791,  240.596,  240.409,  240.229,  240.056,  239.891,  239.732,  239.579, 
      239.432,  239.290,  239.153,  239.020,  238.891,  238.763,  238.453,  238.152,  237.857,  237.567, 
      237.279,  236.994,  236.711,  236.430,  236.149,  235.868,  235.587,  235.307,  235.029,  234.752, 
      234.477,  234.204,  233.934,  233.665,  233.399,  233.135,  232.872,  232.611,  232.354,  232.099, 
      231.847,  231.590,  231.328,  231.070,  230.815,  230.563,  230.068,  229.586,  229.022,  228.363, 
      227.705,  227.032,  226.366,  225.707,  225.055,  224.410,  223.773,  223.144,  222.524,  221.909, 
      221.296,  220.687,  220.082,  219.479,  218.875,  218.278,  217.690,  217.114,  216.553,  216.007, 
      215.477,  214.966,  214.473,  213.996,  213.537,  213.093,  212.661,  212.246,  211.849,  211.471, 
      211.107,  210.759,  210.427,  210.109,  209.806,  209.516,  209.236,  208.969,  208.713,  208.472, 
      208.246,  208.035,  207.844,  207.666,  207.493,  207.324, ],
    [ 244.836,  244.513,  244.231,  243.971,  243.718,  243.471,  243.230,  242.995,  242.766,  242.544, 
      242.328,  242.119,  241.916,  241.721,  241.534,  241.354,  241.181,  241.015,  240.856,  240.702, 
      240.555,  240.413,  240.276,  240.143,  240.013,  239.886,  239.575,  239.274,  238.978,  238.687, 
      238.399,  238.112,  237.827,  237.543,  237.258,  236.974,  236.691,  236.409,  236.129,  235.850, 
      235.573,  235.299,  235.026,  234.756,  234.488,  234.222,  233.958,  233.697,  233.439,  233.184, 
      232.925,  232.658,  232.396,  232.137,  231.881,  231.628,  231.133,  230.649,  229.981,  229.319, 
      228.643,  227.967,  227.298,  226.638,  225.984,  225.337,  224.698,  224.068,  223.444,  222.825, 
      222.207,  221.593,  220.983,  220.375,  219.768,  219.169,  218.578,  218.001,  217.438,  216.891, 
      216.361,  215.848,  215.352,  214.874,  214.411,  213.963,  213.529,  213.111,  212.713,  212.332, 
      211.967,  211.617,  211.282,  210.961,  210.654,  210.361,  210.079,  209.808,  209.549,  209.307, 
      209.078,  208.866,  208.673,  208.492,  208.313,  208.143, ],
    [ 245.963,  245.640,  245.358,  245.097,  244.844,  244.597,  244.356,  244.121,  243.892,  243.669, 
      243.453,  243.244,  243.041,  242.846,  242.658,  242.478,  242.305,  242.139,  241.979,  241.825, 
      241.678,  241.536,  241.399,  241.266,  241.136,  241.008,  240.697,  240.395,  240.098,  239.807, 
      239.517,  239.228,  238.940,  238.652,  238.365,  238.079,  237.793,  237.509,  237.227,  236.946, 
      236.667,  236.391,  236.117,  235.846,  235.576,  235.309,  235.044,  234.783,  234.524,  234.263, 
      233.993,  233.726,  233.463,  233.203,  232.947,  232.694,  232.197,  231.609,  230.939,  230.263, 
      229.579,  228.900,  228.230,  227.567,  226.911,  226.262,  225.622,  224.990,  224.362,  223.737, 
      223.115,  222.495,  221.881,  221.268,  220.660,  220.058,  219.466,  218.887,  218.323,  217.774, 
      217.243,  216.729,  216.231,  215.749,  215.284,  214.832,  214.396,  213.976,  213.576,  213.193, 
      212.826,  212.474,  212.135,  211.811,  211.501,  211.205,  210.920,  210.646,  210.385,  210.141, 
      209.910,  209.697,  209.502,  209.317,  209.133,  208.961, ],
    [ 247.089,  246.765,  246.483,  246.223,  245.969,  245.722,  245.481,  245.245,  245.016,  244.793, 
      244.577,  244.367,  244.165,  243.969,  243.781,  243.601,  243.427,  243.261,  243.101,  242.948, 
      242.800,  242.658,  242.521,  242.388,  242.258,  242.130,  241.818,  241.515,  241.218,  240.925, 
      240.633,  240.341,  240.049,  239.759,  239.469,  239.181,  238.894,  238.607,  238.323,  238.040, 
      237.760,  237.482,  237.207,  236.933,  236.662,  236.394,  236.129,  235.868,  235.604,  235.331, 
      235.061,  234.793,  234.529,  234.269,  234.012,  233.759,  233.247,  232.568,  231.892,  231.200, 
      230.512,  229.832,  229.159,  228.495,  227.838,  227.188,  226.546,  225.909,  225.277,  224.647, 
      224.020,  223.396,  222.775,  222.161,  221.551,  220.946,  220.353,  219.773,  219.207,  218.658, 
      218.125,  217.609,  217.108,  216.624,  216.154,  215.700,  215.262,  214.840,  214.438,  214.053, 
      213.684,  213.329,  212.988,  212.660,  212.347,  212.048,  211.761,  211.484,  211.221,  210.974, 
      210.742,  210.527,  210.330,  210.141,  209.953,  209.779, ],
    [ 248.214,  247.890,  247.608,  247.347,  247.093,  246.846,  246.604,  246.369,  246.140,  245.916, 
      245.700,  245.490,  245.287,  245.092,  244.904,  244.723,  244.550,  244.384,  244.224,  244.070, 
      243.922,  243.780,  243.642,  243.509,  243.379,  243.251,  242.938,  242.635,  242.336,  242.041, 
      241.746,  241.450,  241.156,  240.863,  240.571,  240.281,  239.992,  239.703,  239.417,  239.132, 
      238.851,  238.572,  238.294,  238.020,  237.748,  237.480,  237.215,  236.949,  236.672,  236.399, 
      236.128,  235.860,  235.595,  235.335,  235.077,  234.824,  234.206,  233.524,  232.829,  232.135, 
      231.445,  230.762,  230.088,  229.422,  228.763,  228.112,  227.466,  226.826,  226.188,  225.554, 
      224.922,  224.292,  223.668,  223.052,  222.440,  221.834,  221.239,  220.657,  220.091,  219.540, 
      219.006,  218.487,  217.984,  217.497,  217.024,  216.567,  216.127,  215.704,  215.299,  214.913, 
      214.542,  214.184,  213.839,  213.508,  213.193,  212.891,  212.600,  212.321,  212.056,  211.808, 
      211.574,  211.358,  211.157,  210.964,  210.773,  210.596, ],
    [ 249.337,  249.014,  248.731,  248.470,  248.217,  247.969,  247.727,  247.492,  247.262,  247.039, 
      246.822,  246.613,  246.410,  246.214,  246.026,  245.845,  245.672,  245.506,  245.346,  245.192, 
      245.044,  244.901,  244.763,  244.630,  244.499,  244.371,  244.058,  243.753,  243.452,  243.154, 
      242.855,  242.557,  242.260,  241.965,  241.671,  241.379,  241.087,  240.797,  240.509,  240.223, 
      239.940,  239.659,  239.381,  239.106,  238.833,  238.565,  238.297,  238.017,  237.740,  237.466, 
      237.194,  236.926,  236.661,  236.400,  236.142,  235.855,  235.164,  234.468,  233.765,  233.068, 
      232.376,  231.692,  231.016,  230.348,  229.688,  229.034,  228.385,  227.739,  227.097,  226.458, 
      225.821,  225.186,  224.561,  223.942,  223.328,  222.721,  222.125,  221.541,  220.974,  220.422, 
      219.886,  219.365,  218.859,  218.367,  217.892,  217.433,  216.991,  216.566,  216.160,  215.772, 
      215.398,  215.037,  214.689,  214.356,  214.037,  213.733,  213.439,  213.157,  212.891,  212.641, 
      212.405,  212.187,  211.984,  211.785,  211.592,  211.413, ],
    [ 250.460,  250.137,  249.854,  249.593,  249.339,  249.092,  248.850,  248.614,  248.385,  248.162, 
      247.945,  247.735,  247.532,  247.336,  247.148,  246.967,  246.794,  246.627,  246.467,  246.313, 
      246.165,  246.022,  245.884,  245.750,  245.619,  245.491,  245.176,  244.868,  244.565,  244.262, 
      243.961,  243.661,  243.362,  243.065,  242.769,  242.475,  242.182,  241.890,  241.600,  241.313, 
      241.028,  240.746,  240.467,  240.191,  239.919,  239.647,  239.365,  239.085,  238.807,  238.532, 
      238.260,  237.991,  237.726,  237.465,  237.162,  236.813,  236.116,  235.404,  234.699,  234.000, 
      233.306,  232.620,  231.942,  231.273,  230.611,  229.954,  229.300,  228.650,  228.003,  227.358, 
      226.716,  226.079,  225.452,  224.831,  224.215,  223.607,  223.009,  222.425,  221.856,  221.303, 
      220.764,  220.240,  219.732,  219.237,  218.759,  218.299,  217.855,  217.428,  217.020,  216.630, 
      216.254,  215.889,  215.539,  215.202,  214.881,  214.574,  214.277,  213.993,  213.725,  213.473, 
      213.236,  213.017,  212.809,  212.606,  212.410,  212.230, ],
    [ 251.583,  251.259,  250.977,  250.716,  250.462,  250.214,  249.972,  249.737,  249.507,  249.283, 
      249.067,  248.857,  248.654,  248.458,  248.269,  248.088,  247.915,  247.748,  247.587,  247.433, 
      247.285,  247.142,  247.003,  246.869,  246.737,  246.608,  246.291,  245.980,  245.673,  245.368, 
      245.065,  244.762,  244.462,  244.163,  243.865,  243.569,  243.274,  242.981,  242.690,  242.401, 
      242.115,  241.832,  241.552,  241.276,  241.002,  240.716,  240.433,  240.152,  239.874,  239.598, 
      239.326,  239.057,  238.791,  238.472,  238.121,  237.770,  237.053,  236.339,  235.631,  234.930, 
      234.235,  233.547,  232.868,  232.197,  231.532,  230.871,  230.212,  229.557,  228.905,  228.256, 
      227.609,  226.971,  226.341,  225.719,  225.102,  224.492,  223.893,  223.308,  222.738,  222.182, 
      221.641,  221.115,  220.603,  220.105,  219.625,  219.163,  218.717,  218.289,  217.879,  217.487, 
      217.108,  216.741,  216.387,  216.048,  215.724,  215.414,  215.115,  214.828,  214.559,  214.305, 
      214.067,  213.846,  213.635,  213.426,  213.229,  213.047, ],
    [ 252.705,  252.381,  252.099,  251.837,  251.584,  251.336,  251.094,  250.858,  250.628,  250.405, 
      250.188,  249.978,  249.774,  249.578,  249.390,  249.209,  249.035,  248.868,  248.707,  248.552, 
      248.404,  248.260,  248.121,  247.986,  247.853,  247.723,  247.403,  247.089,  246.779,  246.472, 
      246.167,  245.862,  245.560,  245.258,  244.959,  244.661,  244.365,  244.070,  243.778,  243.487, 
      243.201,  242.917,  242.637,  242.359,  242.070,  241.784,  241.500,  241.219,  240.940,  240.664, 
      240.391,  240.121,  239.785,  239.430,  239.075,  238.712,  237.989,  237.272,  236.562,  235.859, 
      235.162,  234.473,  233.792,  233.120,  232.451,  231.784,  231.121,  230.461,  229.804,  229.150, 
      228.501,  227.861,  227.229,  226.606,  225.987,  225.376,  224.776,  224.190,  223.618,  223.061, 
      222.517,  221.988,  221.472,  220.972,  220.491,  220.027,  219.579,  219.149,  218.738,  218.344, 
      217.961,  217.591,  217.235,  216.893,  216.566,  216.253,  215.951,  215.664,  215.392,  215.136, 
      214.897,  214.674,  214.459,  214.246,  214.047,  213.863, ]])

_ts_tbl = np.array([
    [   0.000,    0.666,    1.277,    1.861,    2.437,    3.007,    3.570,    4.127,    4.677,    5.221, 
        5.757,    6.286,    6.808,    7.323,    7.830,    8.330,    8.822,    9.309,    9.789,   10.264, 
      10.735,   11.200,   11.663,   12.122,   12.579,   13.035,   14.171,   15.304,   16.436,   17.567, 
      18.699,   19.830,   20.960,   22.089,   23.216,   24.341,   25.465,   26.588,   27.708,   28.827, 
      29.943,   31.057,   32.169,   33.279,   34.387,   35.493,   36.598,   37.700,   38.802,   39.902, 
      41.000,   42.096,   43.190,   44.282,   45.372,   46.459,   48.629,   50.792,   52.947,   55.095, 
      57.236,   59.371,   61.497,   63.615,   65.726,   67.831,   69.927,   72.016,   74.097,   76.170, 
      78.236,   80.294,   82.342,   84.382,   86.418,   88.446,   90.464,   92.470,   94.462,   96.439, 
      98.403,  100.351,  102.285,  104.203,  106.105,  107.992,  109.866,  111.727,  113.572,  115.405, 
      117.227,  119.038,  120.837,  122.625,  124.401,  126.167,  127.924,  129.672,  131.412,  133.143, 
      134.867,  136.583,  138.292,  139.994,  141.688,  143.377, ],
    [   0.703,    0.941,    1.427,    1.961,    2.512,    3.066,    3.619,    4.169,    4.713,    5.253, 
        5.785,    6.312,    6.832,    7.344,    7.850,    8.348,    8.839,    9.324,    9.804,   10.278, 
      10.748,   11.213,   11.674,   12.133,   12.590,   13.045,   14.180,   15.312,   16.443,   17.574, 
      18.705,   19.836,   20.966,   22.094,   23.221,   24.346,   25.470,   26.592,   27.712,   28.831, 
      29.947,   31.061,   32.173,   33.282,   34.390,   35.496,   36.601,   37.703,   38.805,   39.904, 
      41.002,   42.099,   43.193,   44.284,   45.374,   46.462,   48.631,   50.794,   52.949,   55.096, 
      57.238,   59.373,   61.499,   63.616,   65.728,   67.832,   69.929,   72.017,   74.098,   76.171, 
      78.237,   80.295,   82.343,   84.383,   86.419,   88.447,   90.465,   92.471,   94.463,   96.440, 
      98.404,  100.352,  102.286,  104.204,  106.106,  107.993,  109.867,  111.727,  113.573,  115.406, 
      117.228,  119.039,  120.838,  122.626,  124.402,  126.167,  127.924,  129.673,  131.412,  133.144, 
      134.867,  136.584,  138.293,  139.994,  141.689,  143.377, ],
    [   1.403,    1.486,    1.803,    2.235,    2.724,    3.238,    3.762,    4.291,    4.820,    5.347, 
        5.870,    6.388,    6.901,    7.408,    7.909,    8.403,    8.890,    9.372,    9.849,   10.320, 
      10.787,   11.250,   11.710,   12.167,   12.622,   13.076,   14.207,   15.337,   16.466,   17.595, 
      18.725,   19.854,   20.983,   22.110,   23.236,   24.360,   25.483,   26.605,   27.725,   28.842, 
      29.958,   31.071,   32.183,   33.292,   34.400,   35.505,   36.609,   37.712,   38.813,   39.912, 
      41.010,   42.106,   43.200,   44.291,   45.381,   46.468,   48.638,   50.800,   52.954,   55.102, 
      57.243,   59.378,   61.504,   63.621,   65.732,   67.836,   69.933,   72.021,   74.102,   76.175, 
      78.241,   80.299,   82.346,   84.387,   86.422,   88.450,   90.468,   92.473,   94.465,   96.443, 
      98.406,  100.355,  102.288,  104.206,  106.108,  107.995,  109.869,  111.730,  113.575,  115.408, 
      117.230,  119.041,  120.840,  122.628,  124.404,  126.169,  127.926,  129.675,  131.414,  133.145, 
      134.869,  136.585,  138.294,  139.996,  141.690,  143.379, ],
    [   2.099,    2.096,    2.295,    2.627,    3.043,    3.504,    3.989,    4.488,    4.993,    5.501, 
        6.009,    6.514,    7.016,    7.514,    8.006,    8.493,    8.974,    9.451,    9.923,   10.390, 
      10.853,   11.312,   11.769,   12.223,   12.675,   13.126,   14.253,   15.379,   16.504,   17.630, 
      18.757,   19.884,   21.011,   22.137,   23.261,   24.384,   25.506,   26.626,   27.745,   28.862, 
      29.977,   31.089,   32.200,   33.309,   34.415,   35.521,   36.624,   37.726,   38.827,   39.926, 
      41.023,   42.119,   43.212,   44.303,   45.392,   46.479,   48.648,   50.810,   52.964,   55.111, 
      57.252,   59.386,   61.512,   63.629,   65.739,   67.843,   69.940,   72.028,   74.108,   76.181, 
      78.247,   80.304,   82.352,   84.392,   86.427,   88.455,   90.473,   92.478,   94.470,   96.448, 
      98.411,  100.359,  102.293,  104.210,  106.112,  107.999,  109.873,  111.733,  113.579,  115.412, 
      117.233,  119.044,  120.843,  122.631,  124.407,  126.172,  127.929,  129.678,  131.417,  133.148, 
      134.872,  136.588,  138.297,  139.998,  141.693,  143.381, ],
    [   2.786,    2.726,    2.841,    3.092,    3.440,    3.846,    4.286,    4.749,    5.226,    5.710, 
        6.197,    6.686,    7.173,    7.659,    8.140,    8.618,    9.091,    9.560,   10.025,   10.486, 
      10.944,   11.399,   11.851,   12.301,   12.749,   13.197,   14.317,   15.437,   16.557,   17.679, 
      18.802,   19.926,   21.051,   22.174,   23.296,   24.417,   25.537,   26.656,   27.773,   28.889, 
      30.003,   31.114,   32.224,   33.332,   34.438,   35.542,   36.645,   37.746,   38.846,   39.944, 
      41.041,   42.136,   43.229,   44.320,   45.408,   46.495,   48.663,   50.824,   52.977,   55.124, 
      57.264,   59.398,   61.523,   63.640,   65.750,   67.853,   69.949,   72.037,   74.117,   76.190, 
      78.255,   80.313,   82.360,   84.400,   86.435,   88.462,   90.480,   92.485,   94.477,   96.454, 
      98.417,  100.365,  102.298,  104.216,  106.118,  108.005,  109.879,  111.739,  113.584,  115.417, 
      117.238,  119.049,  120.848,  122.636,  124.411,  126.177,  127.934,  129.682,  131.421,  133.152, 
      134.876,  136.592,  138.301,  140.002,  141.697,  143.385, ],
    [   3.465,    3.359,    3.413,    3.601,    3.890,    4.244,    4.640,    5.065,    5.510,    5.967, 
        6.431,    6.900,    7.371,    7.841,    8.310,    8.775,    9.238,    9.698,   10.155,   10.609, 
      11.060,   11.509,   11.955,   12.400,   12.844,   13.288,   14.398,   15.511,   16.625,   17.742, 
      18.860,   19.981,   21.101,   22.222,   23.341,   24.460,   25.578,   26.695,   27.810,   28.924, 
      30.036,   31.146,   32.255,   33.361,   34.466,   35.569,   36.671,   37.771,   38.871,   39.968, 
      41.064,   42.159,   43.251,   44.341,   45.429,   46.515,   48.682,   50.842,   52.995,   55.140, 
      57.280,   59.413,   61.537,   63.653,   65.763,   67.866,   69.962,   72.049,   74.129,   76.201, 
      78.266,   80.323,   82.370,   84.410,   86.444,   88.472,   90.489,   92.494,   94.485,   96.462, 
      98.425,  100.373,  102.306,  104.224,  106.125,  108.012,  109.886,  111.745,  113.591,  115.423, 
      117.245,  119.055,  120.854,  122.641,  124.417,  126.182,  127.939,  129.687,  131.426,  133.157, 
      134.881,  136.597,  138.305,  140.007,  141.701,  143.389, ],
    [   4.135,    3.991,    3.998,    4.136,    4.377,    4.684,    5.038,    5.426,    5.838,    6.267, 
        6.706,    7.153,    7.605,    8.058,    8.512,    8.964,    9.415,    9.864,   10.312,   10.757, 
      11.200,   11.642,   12.082,   12.520,   12.959,   13.398,   14.497,   15.601,   16.708,   17.818, 
      18.931,   20.046,   21.163,   22.280,   23.396,   24.512,   25.627,   26.741,   27.855,   28.967, 
      30.077,   31.185,   32.292,   33.397,   34.501,   35.603,   36.703,   37.803,   38.901,   39.998, 
      41.093,   42.186,   43.278,   44.367,   45.454,   46.540,   48.705,   50.864,   53.016,   55.160, 
      57.299,   59.431,   61.555,   63.670,   65.779,   67.882,   69.977,   72.064,   74.143,   76.214, 
      78.279,   80.336,   82.382,   84.422,   86.456,   88.483,   90.500,   92.505,   94.496,   96.473, 
      98.435,  100.383,  102.315,  104.233,  106.134,  108.021,  109.894,  111.754,  113.599,  115.431, 
      117.252,  119.063,  120.861,  122.649,  124.424,  126.189,  127.946,  129.694,  131.433,  133.164, 
      134.887,  136.603,  138.311,  140.013,  141.707,  143.395, ],
    [   4.794,    4.619,    4.588,    4.687,    4.888,    5.155,    5.471,    5.824,    6.203,    6.603, 
        7.017,    7.441,    7.872,    8.308,    8.745,    9.183,    9.620,   10.057,   10.494,   10.929, 
      11.364,   11.797,   12.229,   12.661,   13.094,   13.527,   14.614,   15.706,   16.805,   17.907, 
      19.014,   20.124,   21.236,   22.348,   23.460,   24.573,   25.685,   26.796,   27.907,   29.017, 
      30.125,   31.231,   32.336,   33.440,   34.542,   35.642,   36.742,   37.840,   38.936,   40.032, 
      41.126,   42.219,   43.309,   44.398,   45.484,   46.569,   48.733,   50.890,   53.040,   55.184, 
      57.322,   59.453,   61.576,   63.690,   65.799,   67.900,   69.995,   72.081,   74.160,   76.231, 
      78.295,   80.351,   82.397,   84.436,   86.469,   88.496,   90.513,   92.517,   94.508,   96.484, 
      98.447,  100.394,  102.326,  104.243,  106.145,  108.031,  109.904,  111.763,  113.608,  115.440, 
      117.261,  119.071,  120.870,  122.657,  124.432,  126.197,  127.954,  129.702,  131.440,  133.171, 
      134.894,  136.610,  138.318,  140.020,  141.714,  143.402, ],
    [   5.444,    5.242,    5.180,    5.248,    5.416,    5.649,    5.930,    6.250,    6.599,    6.971, 
        7.359,    7.760,    8.170,    8.586,    9.006,    9.428,    9.851,   10.275,   10.700,   11.125, 
      11.549,   11.973,   12.397,   12.822,   13.247,   13.674,   14.746,   15.828,   16.916,   18.010, 
      19.110,   20.213,   21.319,   22.427,   23.535,   24.643,   25.752,   26.860,   27.968,   29.075, 
      30.180,   31.285,   32.387,   33.489,   34.589,   35.688,   36.786,   37.882,   38.978,   40.072, 
      41.165,   42.256,   43.345,   44.433,   45.518,   46.602,   48.764,   50.920,   53.069,   55.211, 
      57.348,   59.478,   61.600,   63.713,   65.821,   67.922,   70.015,   72.101,   74.179,   76.249, 
      78.313,   80.368,   82.414,   84.452,   86.485,   88.512,   90.528,   92.532,   94.522,   96.498, 
      98.460,  100.407,  102.339,  104.256,  106.157,  108.043,  109.915,  111.775,  113.619,  115.451, 
      117.272,  119.082,  120.880,  122.667,  124.442,  126.207,  127.963,  129.711,  131.449,  133.180, 
      134.903,  136.618,  138.327,  140.028,  141.721,  143.409, ],
    [   6.084,    5.860,    5.772,    5.815,    5.956,    6.159,    6.410,    6.700,    7.020,    7.365, 
        7.728,    8.106,    8.494,    8.891,    9.293,    9.698,   10.107,   10.517,   10.929,   11.342, 
      11.756,   12.170,   12.585,   13.001,   13.419,   13.838,   14.895,   15.964,   17.041,   18.126, 
      19.217,   20.314,   21.414,   22.516,   23.619,   24.723,   25.827,   26.932,   28.036,   29.140, 
      30.243,   31.345,   32.445,   33.544,   34.643,   35.740,   36.835,   37.930,   39.024,   40.117, 
      41.208,   42.298,   43.387,   44.473,   45.557,   46.640,   48.800,   50.954,   53.102,   55.242, 
      57.378,   59.507,   61.627,   63.740,   65.846,   67.946,   70.038,   72.123,   74.201,   76.270, 
      78.333,   80.388,   82.433,   84.471,   86.503,   88.529,   90.545,   92.548,   94.538,   96.514, 
      98.475,  100.422,  102.354,  104.270,  106.170,  108.056,  109.929,  111.787,  113.632,  115.463, 
      117.284,  119.093,  120.891,  122.678,  124.453,  126.217,  127.973,  129.721,  131.459,  133.189, 
      134.912,  136.628,  138.336,  140.037,  141.730,  143.418, ],
    [   6.715,    6.472,    6.364,    6.387,    6.505,    6.682,    6.906,    7.169,    7.462,    7.782, 
        8.120,    8.475,    8.842,    9.219,    9.603,    9.992,   10.384,   10.780,   11.179,   11.580, 
      11.982,   12.386,   12.791,   13.198,   13.608,   14.020,   15.060,   16.114,   17.180,   18.254, 
      19.337,   20.426,   21.519,   22.614,   23.712,   24.811,   25.911,   27.012,   28.113,   29.213, 
      30.313,   31.412,   32.510,   33.606,   34.702,   35.797,   36.891,   37.984,   39.076,   40.167, 
      41.257,   42.346,   43.432,   44.517,   45.601,   46.682,   48.840,   50.992,   53.138,   55.277, 
      57.411,   59.538,   61.657,   63.769,   65.874,   67.973,   70.065,   72.149,   74.225,   76.294, 
      78.356,   80.410,   82.454,   84.491,   86.523,   88.549,   90.564,   92.567,   94.556,   96.531, 
      98.492,  100.439,  102.370,  104.286,  106.186,  108.071,  109.943,  111.802,  113.646,  115.477, 
      117.297,  119.106,  120.904,  122.690,  124.465,  126.229,  127.985,  129.732,  131.470,  133.200, 
      134.923,  136.638,  138.346,  140.047,  141.740,  143.428, ],
    [   7.338,    7.080,    6.954,    6.961,    7.060,    7.214,    7.415,    7.652,    7.922,    8.217, 
        8.532,    8.865,    9.212,    9.569,    9.934,   10.306,   10.683,   11.064,   11.449,   11.837, 
      12.227,   12.620,   13.015,   13.413,   13.814,   14.218,   15.240,   16.279,   17.332,   18.395, 
      19.468,   20.548,   21.634,   22.723,   23.815,   24.908,   26.004,   27.100,   28.197,   29.294, 
      30.390,   31.486,   32.581,   33.675,   34.768,   35.861,   36.952,   38.043,   39.133,   40.223, 
      41.311,   42.398,   43.483,   44.567,   45.649,   46.729,   48.884,   51.034,   53.178,   55.315, 
      57.447,   59.573,   61.691,   63.801,   65.905,   68.003,   70.093,   72.176,   74.252,   76.320, 
      78.381,   80.434,   82.478,   84.514,   86.546,   88.570,   90.585,   92.587,   94.576,   96.551, 
      98.511,  100.457,  102.388,  104.303,  106.203,  108.088,  109.959,  111.817,  113.661,  115.492, 
      117.312,  119.121,  120.918,  122.704,  124.478,  126.242,  127.998,  129.745,  131.483,  133.212, 
      134.935,  136.650,  138.357,  140.058,  141.751,  143.438, ],
    [   7.954,    7.682,    7.543,    7.538,    7.619,    7.754,    7.933,    8.148,    8.395,    8.668, 
        8.961,    9.273,    9.599,    9.937,   10.284,   10.639,   11.000,   11.366,   11.737,   12.112, 
      12.490,   12.872,   13.256,   13.644,   14.036,   14.431,   15.435,   16.458,   17.496,   18.548, 
      19.611,   20.682,   21.760,   22.842,   23.927,   25.015,   26.105,   27.196,   28.289,   29.382, 
      30.474,   31.566,   32.658,   33.749,   34.840,   35.930,   37.019,   38.108,   39.196,   40.284, 
      41.370,   42.455,   43.539,   44.621,   45.701,   46.780,   48.933,   51.080,   53.222,   55.357, 
      57.487,   59.612,   61.728,   63.836,   65.939,   68.036,   70.125,   72.207,   74.281,   76.348, 
      78.408,   80.461,   82.504,   84.539,   86.570,   88.594,   90.608,   92.609,   94.598,   96.572, 
      98.532,  100.477,  102.407,  104.322,  106.221,  108.106,  109.977,  111.835,  113.678,  115.508, 
      117.328,  119.136,  120.934,  122.719,  124.493,  126.257,  128.012,  129.758,  131.496,  133.226, 
      134.948,  136.662,  138.370,  140.070,  141.763,  143.450, ],
    [   8.564,    8.281,    8.130,    8.115,    8.182,    8.300,    8.459,    8.654,    8.880,    9.132, 
        9.405,    9.696,   10.003,   10.322,   10.652,   10.989,   11.334,   11.685,   12.042,   12.404, 
      12.770,   13.140,   13.513,   13.891,   14.273,   14.659,   15.643,   16.649,   17.674,   18.713, 
      19.765,   20.826,   21.895,   22.970,   24.048,   25.130,   26.214,   27.301,   28.388,   29.477, 
      30.566,   31.654,   32.742,   33.830,   34.918,   36.005,   37.092,   38.178,   39.264,   40.350, 
      41.434,   42.517,   43.599,   44.679,   45.758,   46.835,   48.985,   51.130,   53.269,   55.402, 
      57.531,   59.653,   61.768,   63.875,   65.976,   68.071,   70.159,   72.240,   74.313,   76.379, 
      78.438,   80.490,   82.532,   84.566,   86.596,   88.619,   90.633,   92.634,   94.621,   96.595, 
      98.554,  100.499,  102.428,  104.343,  106.241,  108.125,  109.996,  111.853,  113.696,  115.526, 
      117.345,  119.153,  120.950,  122.735,  124.509,  126.272,  128.027,  129.773,  131.511,  133.240, 
      134.962,  136.676,  138.383,  140.083,  141.776,  143.463, ],
    [   9.167,    8.874,    8.716,    8.693,    8.748,    8.850,    8.992,    9.169,    9.375,    9.608, 
        9.861,   10.133,   10.421,   10.722,   11.034,   11.355,   11.684,   12.020,   12.363,   12.711, 
      13.064,   13.422,   13.785,   14.152,   14.525,   14.902,   15.865,   16.854,   17.863,   18.889, 
      19.930,   20.981,   22.041,   23.107,   24.178,   25.253,   26.332,   27.413,   28.495,   29.579, 
      30.664,   31.748,   32.833,   33.918,   35.002,   36.086,   37.170,   38.254,   39.338,   40.421, 
      41.503,   42.584,   43.664,   44.742,   45.819,   46.895,   49.042,   51.184,   53.321,   55.451, 
      57.578,   59.698,   61.811,   63.916,   66.016,   68.109,   70.196,   72.275,   74.347,   76.412, 
      78.470,   80.521,   82.562,   84.596,   86.625,   88.647,   90.659,   92.660,   94.647,   96.619, 
      98.578,  100.522,  102.451,  104.365,  106.263,  108.147,  110.017,  111.874,  113.716,  115.545, 
      117.364,  119.172,  120.968,  122.753,  124.526,  126.289,  128.044,  129.790,  131.527,  133.256, 
      134.977,  136.691,  138.398,  140.098,  141.790,  143.476, ],
    [   9.766,    9.466,    9.301,    9.272,    9.315,    9.404,    9.530,    9.690,    9.879,   10.093, 
      10.328,   10.581,   10.852,   11.136,   11.430,   11.735,   12.048,   12.369,   12.698,   13.033, 
      13.373,   13.719,   14.071,   14.427,   14.790,   15.158,   16.100,   17.071,   18.064,   19.077, 
      20.105,   21.146,   22.196,   23.254,   24.317,   25.385,   26.457,   27.532,   28.610,   29.689, 
      30.769,   31.849,   32.930,   34.011,   35.092,   36.173,   37.254,   38.335,   39.416,   40.497, 
      41.577,   42.656,   43.733,   44.810,   45.885,   46.959,   49.103,   51.242,   53.376,   55.504, 
      57.628,   59.746,   61.857,   63.960,   66.058,   68.151,   70.236,   72.314,   74.384,   76.448, 
      78.505,   80.555,   82.594,   84.627,   86.655,   88.677,   90.688,   92.688,   94.674,   96.646, 
      98.604,  100.547,  102.476,  104.389,  106.287,  108.169,  110.039,  111.895,  113.737,  115.566, 
      117.384,  119.192,  120.988,  122.772,  124.545,  126.307,  128.061,  129.807,  131.544,  133.272, 
      134.993,  136.707,  138.414,  140.113,  141.805,  143.491, ],
    [  10.360,   10.055,    9.886,    9.851,    9.884,    9.960,   10.072,   10.216,   10.389,   10.586, 
      10.804,   11.040,   11.293,   11.560,   11.839,   12.127,   12.425,   12.731,   13.046,   13.367, 
      13.696,   14.030,   14.370,   14.716,   15.068,   15.426,   16.347,   17.299,   18.277,   19.275, 
      20.291,   21.320,   22.361,   23.410,   24.465,   25.526,   26.591,   27.660,   28.732,   29.806, 
      30.881,   31.957,   33.034,   34.111,   35.188,   36.266,   37.344,   38.422,   39.500,   40.578, 
      41.656,   42.732,   43.808,   44.882,   45.955,   47.027,   49.168,   51.304,   53.434,   55.560, 
      57.682,   59.798,   61.906,   64.008,   66.104,   68.194,   70.278,   72.355,   74.424,   76.486, 
      78.542,   80.590,   82.629,   84.661,   86.688,   88.709,   90.719,   92.718,   94.703,   96.674, 
      98.632,  100.574,  102.502,  104.415,  106.312,  108.194,  110.063,  111.919,  113.760,  115.588, 
      117.406,  119.213,  121.008,  122.792,  124.565,  126.327,  128.080,  129.825,  131.562,  133.290, 
      135.011,  136.724,  138.430,  140.129,  141.821,  143.507, ],
    [  10.951,   10.642,   10.469,   10.430,   10.454,   10.519,   10.618,   10.747,   10.905,   11.086, 
      11.288,   11.508,   11.745,   11.995,   12.258,   12.531,   12.814,   13.106,   13.406,   13.715, 
      14.030,   14.352,   14.681,   15.016,   15.358,   15.707,   16.606,   17.539,   18.500,   19.484, 
      20.487,   21.505,   22.535,   23.574,   24.621,   25.674,   26.732,   27.795,   28.861,   29.929, 
      31.000,   32.071,   33.143,   34.216,   35.290,   36.365,   37.439,   38.514,   39.589,   40.664, 
      41.739,   42.814,   43.887,   44.959,   46.030,   47.100,   49.237,   51.369,   53.497,   55.620, 
      57.739,   59.852,   61.959,   64.058,   66.153,   68.241,   70.323,   72.398,   74.466,   76.527, 
      78.581,   80.629,   82.666,   84.697,   86.723,   88.742,   90.752,   92.750,   94.734,   96.705, 
      98.661,  100.603,  102.530,  104.442,  106.338,  108.220,  110.088,  111.943,  113.784,  115.612, 
      117.429,  119.235,  121.030,  122.814,  124.586,  126.347,  128.101,  129.845,  131.581,  133.309, 
      135.029,  136.742,  138.448,  140.147,  141.838,  143.524, ],
    [  11.540,   11.228,   11.052,   11.009,   11.025,   11.079,   11.166,   11.282,   11.426,   11.592, 
      11.778,   11.983,   12.204,   12.439,   12.687,   12.945,   13.213,   13.491,   13.778,   14.073, 
      14.376,   14.686,   15.003,   15.328,   15.660,   15.999,   16.876,   17.790,   18.734,   19.703, 
      20.692,   21.698,   22.718,   23.748,   24.786,   25.831,   26.882,   27.938,   28.997,   30.060, 
      31.125,   32.191,   33.259,   34.328,   35.398,   36.469,   37.540,   38.611,   39.683,   40.756, 
      41.828,   42.900,   43.970,   45.040,   46.109,   47.177,   49.310,   51.439,   53.563,   55.683, 
      57.799,   59.910,   62.014,   64.112,   66.204,   68.291,   70.371,   72.444,   74.511,   76.570, 
      78.623,   80.669,   82.705,   84.735,   86.760,   88.778,   90.787,   92.784,   94.767,   96.737, 
      98.692,  100.634,  102.560,  104.471,  106.366,  108.247,  110.115,  111.969,  113.810,  115.637, 
      117.453,  119.259,  121.054,  122.837,  124.608,  126.369,  128.122,  129.866,  131.602,  133.329, 
      135.049,  136.762,  138.467,  140.165,  141.857,  143.542, ],
    [  12.127,   11.813,   11.634,   11.588,   11.596,   11.641,   11.716,   11.821,   11.951,   12.103, 
      12.275,   12.465,   12.671,   12.892,   13.124,   13.368,   13.622,   13.886,   14.159,   14.442, 
      14.732,   15.031,   15.337,   15.650,   15.972,   16.301,   17.157,   18.051,   18.978,   19.932, 
      20.907,   21.901,   22.910,   23.929,   24.958,   25.995,   27.039,   28.088,   29.141,   30.198, 
      31.257,   32.318,   33.381,   34.446,   35.512,   36.578,   37.646,   38.714,   39.783,   40.852, 
      41.921,   42.990,   44.059,   45.126,   46.193,   47.258,   49.387,   51.512,   53.633,   55.750, 
      57.863,   59.972,   62.073,   64.168,   66.258,   68.343,   70.421,   72.493,   74.558,   76.615, 
      78.667,   80.712,   82.747,   84.775,   86.799,   88.816,   90.824,   92.819,   94.802,   96.771, 
      98.725,  100.666,  102.591,  104.501,  106.396,  108.276,  110.143,  111.997,  113.837,  115.663, 
      117.479,  119.284,  121.078,  122.861,  124.632,  126.392,  128.145,  129.888,  131.623,  133.350, 
      135.070,  136.782,  138.487,  140.185,  141.876,  143.561, ],
    [  12.713,   12.397,   12.217,   12.166,   12.168,   12.203,   12.269,   12.361,   12.479,   12.619, 
      12.777,   12.953,   13.145,   13.351,   13.569,   13.799,   14.039,   14.289,   14.550,   14.820, 
      15.098,   15.385,   15.680,   15.983,   16.294,   16.614,   17.447,   18.322,   19.232,   20.170, 
      21.132,   22.113,   23.110,   24.119,   25.139,   26.167,   27.203,   28.245,   29.291,   30.342, 
      31.396,   32.452,   33.510,   34.570,   35.631,   36.694,   37.757,   38.822,   39.887,   40.953, 
      42.020,   43.086,   44.151,   45.216,   46.280,   47.343,   49.468,   51.589,   53.707,   55.820, 
      57.931,   60.036,   62.135,   64.228,   66.316,   68.398,   70.475,   72.544,   74.607,   76.663, 
      78.714,   80.757,   82.790,   84.817,   86.840,   88.856,   90.863,   92.857,   94.839,   96.806, 
      98.760,  100.700,  102.624,  104.534,  106.427,  108.307,  110.173,  112.026,  113.865,  115.691, 
      117.506,  119.311,  121.104,  122.886,  124.656,  126.417,  128.168,  129.912,  131.646,  133.373, 
      135.092,  136.804,  138.508,  140.206,  141.896,  143.580, ],
    [  13.298,   12.980,   12.799,   12.745,   12.740,   12.767,   12.823,   12.905,   13.011,   13.138, 
      13.283,   13.446,   13.624,   13.817,   14.022,   14.237,   14.464,   14.701,   14.949,   15.207, 
      15.473,   15.748,   16.032,   16.325,   16.626,   16.936,   17.747,   18.603,   19.495,   20.417, 
      21.365,   22.333,   23.319,   24.318,   25.328,   26.347,   27.374,   28.409,   29.449,   30.493, 
      31.540,   32.591,   33.644,   34.699,   35.756,   36.814,   37.874,   38.935,   39.997,   41.060, 
      42.123,   43.186,   44.249,   45.311,   46.372,   47.433,   49.553,   51.671,   53.784,   55.894, 
      58.001,   60.104,   62.200,   64.290,   66.376,   68.456,   70.530,   72.598,   74.659,   76.714, 
      78.762,   80.804,   82.836,   84.862,   86.883,   88.898,   90.903,   92.897,   94.877,   96.844, 
      98.797,  100.735,  102.659,  104.567,  106.460,  108.339,  110.205,  112.057,  113.895,  115.720, 
      117.535,  119.339,  121.132,  122.913,  124.683,  126.442,  128.193,  129.936,  131.670,  133.396, 
      135.115,  136.826,  138.530,  140.227,  141.917,  143.601, ],
    [  13.882,   13.562,   13.381,   13.324,   13.312,   13.331,   13.378,   13.450,   13.545,   13.660, 
      13.793,   13.943,   14.109,   14.288,   14.480,   14.682,   14.896,   15.121,   15.356,   15.602, 
      15.856,   16.120,   16.393,   16.675,   16.967,   17.267,   18.056,   18.892,   19.766,   20.673, 
      21.606,   22.562,   23.536,   24.524,   25.524,   26.534,   27.553,   28.580,   29.613,   30.650, 
      31.692,   32.737,   33.784,   34.834,   35.887,   36.941,   37.996,   39.053,   40.111,   41.171, 
      42.231,   43.291,   44.350,   45.410,   46.469,   47.527,   49.642,   51.755,   53.865,   55.972, 
      58.075,   60.175,   62.268,   64.356,   66.439,   68.517,   70.589,   72.655,   74.714,   76.767, 
      78.813,   80.853,   82.884,   84.908,   86.928,   88.942,   90.946,   92.938,   94.918,   96.883, 
      98.835,  100.772,  102.695,  104.603,  106.495,  108.373,  110.238,  112.089,  113.926,  115.751, 
      117.565,  119.368,  121.160,  122.941,  124.710,  126.469,  128.220,  129.962,  131.695,  133.421, 
      135.139,  136.850,  138.554,  140.250,  141.940,  143.623, ],
    [  14.465,   14.145,   13.963,   13.902,   13.884,   13.896,   13.934,   13.996,   14.081,   14.185, 
      14.307,   14.444,   14.598,   14.765,   14.944,   15.133,   15.335,   15.547,   15.770,   16.004, 
      16.247,   16.500,   16.762,   17.034,   17.315,   17.607,   18.374,   19.190,   20.047,   20.937, 
      21.856,   22.798,   23.760,   24.737,   25.727,   26.728,   27.739,   28.758,   29.783,   30.814, 
      31.849,   32.888,   33.930,   34.975,   36.023,   37.072,   38.123,   39.176,   40.231,   41.287, 
      42.343,   43.400,   44.457,   45.513,   46.569,   47.625,   49.735,   51.844,   53.950,   56.052, 
      58.153,   60.249,   62.339,   64.424,   66.505,   68.580,   70.650,   72.714,   74.771,   76.822, 
      78.867,   80.905,   82.934,   84.957,   86.975,   88.988,   90.991,   92.982,   94.960,   96.924, 
      98.875,  100.812,  102.733,  104.640,  106.531,  108.408,  110.272,  112.123,  113.959,  115.783, 
      117.596,  119.399,  121.190,  122.970,  124.739,  126.497,  128.247,  129.989,  131.722,  133.447, 
      135.164,  136.875,  138.578,  140.274,  141.963,  143.646, ],
    [  15.047,   14.727,   14.544,   14.480,   14.456,   14.461,   14.491,   14.544,   14.619,   14.713, 
      14.823,   14.949,   15.091,   15.246,   15.412,   15.590,   15.779,   15.979,   16.190,   16.412, 
      16.645,   16.887,   17.138,   17.400,   17.672,   17.954,   18.699,   19.496,   20.335,   21.209, 
      22.114,   23.043,   23.993,   24.958,   25.938,   26.930,   27.932,   28.942,   29.960,   30.984, 
      32.013,   33.046,   34.082,   35.122,   36.164,   37.209,   38.256,   39.305,   40.355,   41.408, 
      42.460,   43.514,   44.568,   45.621,   46.674,   47.727,   49.833,   51.936,   54.038,   56.137, 
      58.233,   60.326,   62.414,   64.496,   66.573,   68.647,   70.714,   72.776,   74.831,   76.880, 
      78.923,   80.959,   82.987,   85.008,   87.025,   89.036,   91.037,   93.027,   95.004,   96.967, 
      98.917,  100.852,  102.773,  104.679,  106.569,  108.445,  110.308,  112.158,  113.994,  115.817, 
      117.629,  119.431,  121.221,  123.001,  124.769,  126.526,  128.276,  130.017,  131.749,  133.474, 
      135.191,  136.901,  138.603,  140.299,  141.987,  143.670, ],
    [  15.630,   15.309,   15.125,   15.057,   15.029,   15.027,   15.048,   15.093,   15.159,   15.243, 
      15.342,   15.457,   15.588,   15.731,   15.885,   16.051,   16.228,   16.416,   16.617,   16.827, 
      17.049,   17.280,   17.521,   17.773,   18.035,   18.308,   19.032,   19.810,   20.631,   21.489, 
      22.379,   23.295,   24.232,   25.187,   26.156,   27.138,   28.131,   29.134,   30.144,   31.161, 
      32.183,   33.209,   34.240,   35.274,   36.311,   37.351,   38.394,   39.438,   40.485,   41.533, 
      42.582,   43.632,   44.683,   45.733,   46.783,   47.833,   49.934,   52.033,   54.130,   56.224, 
      58.317,   60.407,   62.491,   64.570,   66.645,   68.716,   70.781,   72.840,   74.893,   76.940, 
      78.981,   81.016,   83.041,   85.061,   87.076,   89.086,   91.086,   93.074,   95.050,   97.012, 
      98.961,  100.895,  102.814,  104.719,  106.608,  108.483,  110.346,  112.195,  114.029,  115.852, 
      117.663,  119.464,  121.254,  123.033,  124.800,  126.557,  128.306,  130.046,  131.778,  133.502, 
      135.218,  136.927,  138.630,  140.325,  142.013,  143.695, ],
    [  17.085,   16.763,   16.576,   16.500,   16.458,   16.440,   16.444,   16.469,   16.513,   16.574, 
      16.649,   16.737,   16.842,   16.957,   17.084,   17.222,   17.371,   17.531,   17.704,   17.888, 
      18.083,   18.289,   18.506,   18.733,   18.972,   19.223,   19.895,   20.625,   21.402,   22.221, 
      23.074,   23.956,   24.862,   25.788,   26.731,   27.688,   28.658,   29.639,   30.630,   31.628, 
      32.633,   33.643,   34.659,   35.678,   36.702,   37.730,   38.760,   39.793,   40.829,   41.867, 
      42.907,   43.948,   44.990,   46.032,   47.074,   48.117,   50.203,   52.289,   54.375,   56.459, 
      58.542,   60.622,   62.698,   64.769,   66.836,   68.900,   70.958,   73.012,   75.059,   77.100, 
      79.136,   81.166,   83.188,   85.203,   87.214,   89.220,   91.216,   93.201,   95.173,   97.132, 
      99.077,  101.009,  102.925,  104.827,  106.714,  108.586,  110.446,  112.293,  114.125,  115.945, 
      117.755,  119.554,  121.341,  123.118,  124.883,  126.639,  128.386,  130.124,  131.855,  133.577, 
      135.292,  137.000,  138.700,  140.394,  142.081,  143.762, ],
    [  18.538,   18.215,   18.025,   17.939,   17.886,   17.854,   17.841,   17.848,   17.873,   17.912, 
      17.965,   18.030,   18.110,   18.200,   18.301,   18.413,   18.536,   18.671,   18.818,   18.977, 
      19.147,   19.329,   19.522,   19.727,   19.944,   20.174,   20.796,   21.480,   22.215,   22.994, 
      23.811,   24.659,   25.534,   26.430,   27.346,   28.278,   29.224,   30.183,   31.153,   32.132, 
      33.119,   34.112,   35.111,   36.116,   37.126,   38.139,   39.157,   40.178,   41.203,   42.230, 
      43.260,   44.291,   45.323,   46.357,   47.391,   48.425,   50.497,   52.569,   54.642,   56.714, 
      58.787,   60.857,   62.923,   64.986,   67.045,   69.101,   71.153,   73.199,   75.240,   77.276, 
      79.306,   81.331,   83.347,   85.358,   87.364,   89.366,   91.358,   93.339,   95.308,   97.263, 
      99.205,  101.133,  103.046,  104.945,  106.829,  108.699,  110.556,  112.400,  114.230,  116.047, 
      117.855,  119.651,  121.437,  123.212,  124.975,  126.728,  128.473,  130.210,  131.939,  133.659, 
      135.372,  137.079,  138.778,  140.470,  142.155,  143.835, ],
    [  19.990,   19.664,   19.470,   19.376,   19.311,   19.266,   19.238,   19.228,   19.236,   19.255, 
      19.287,   19.330,   19.388,   19.455,   19.532,   19.619,   19.718,   19.829,   19.953,   20.088, 
      20.236,   20.395,   20.566,   20.749,   20.946,   21.155,   21.730,   22.369,   23.063,   23.804, 
      24.585,   25.400,   26.243,   27.110,   27.999,   28.905,   29.827,   30.763,   31.712,   32.670, 
      33.638,   34.614,   35.596,   36.585,   37.580,   38.580,   39.584,   40.593,   41.605,   42.621, 
      43.640,   44.661,   45.683,   46.707,   47.732,   48.758,   50.814,   52.872,   54.931,   56.991, 
      59.051,   61.111,   63.167,   65.220,   67.271,   69.319,   71.363,   73.403,   75.437,   77.466, 
      79.491,   81.510,   83.521,   85.526,   87.528,   89.525,   91.512,   93.489,   95.454,   97.405, 
      99.344,  101.268,  103.178,  105.074,  106.954,  108.821,  110.675,  112.516,  114.344,  116.159, 
      117.963,  119.757,  121.541,  123.313,  125.074,  126.825,  128.568,  130.303,  132.030,  133.748, 
      135.460,  137.165,  138.862,  140.553,  142.236,  143.914, ],
    [  21.438,   21.111,   20.913,   20.809,   20.733,   20.675,   20.633,   20.609,   20.599,   20.601, 
      20.613,   20.635,   20.672,   20.718,   20.772,   20.837,   20.913,   21.001,   21.103,   21.217, 
      21.343,   21.481,   21.631,   21.795,   21.972,   22.162,   22.692,   23.289,   23.944,   24.647, 
      25.393,   26.175,   26.987,   27.825,   28.686,   29.566,   30.464,   31.377,   32.304,   33.242, 
      34.190,   35.147,   36.113,   37.085,   38.065,   39.050,   40.040,   41.035,   42.035,   43.039, 
      44.047,   45.057,   46.069,   47.083,   48.098,   49.115,   51.154,   53.196,   55.241,   57.288, 
      59.336,   61.384,   63.430,   65.473,   67.515,   69.554,   71.590,   73.622,   75.649,   77.671, 
      79.689,   81.702,   83.707,   85.707,   87.704,   89.696,   91.679,   93.651,   95.611,   97.559, 
      99.493,  101.414,  103.320,  105.212,  107.089,  108.953,  110.804,  112.642,  114.466,  116.278, 
      118.080,  119.872,  121.653,  123.423,  125.181,  126.930,  128.671,  130.404,  132.128,  133.845, 
      135.555,  137.257,  138.953,  140.642,  142.324,  144.000, ],
    [  22.884,   22.554,   22.351,   22.239,   22.152,   22.081,   22.026,   21.988,   21.963,   21.947, 
      21.940,   21.944,   21.961,   21.986,   22.019,   22.062,   22.117,   22.184,   22.266,   22.359, 
      22.465,   22.583,   22.714,   22.859,   23.018,   23.190,   23.678,   24.236,   24.852,   25.520, 
      26.232,   26.981,   27.763,   28.572,   29.405,   30.259,   31.132,   32.022,   32.927,   33.844, 
      34.773,   35.712,   36.659,   37.615,   38.578,   39.548,   40.524,   41.505,   42.492,   43.484, 
      44.479,   45.478,   46.479,   47.483,   48.489,   49.496,   51.516,   53.542,   55.572,   57.605, 
      59.640,   61.677,   63.711,   65.744,   67.775,   69.806,   71.833,   73.856,   75.875,   77.891, 
      79.902,   81.908,   83.907,   85.901,   87.892,   89.879,   91.857,   93.825,   95.780,   97.723, 
      99.653,  101.570,  103.472,  105.361,  107.234,  109.094,  110.942,  112.777,  114.598,  116.407, 
      118.206,  119.995,  121.773,  123.540,  125.296,  127.043,  128.781,  130.512,  132.234,  133.948, 
      135.656,  137.357,  139.050,  140.737,  142.418,  144.092, ],
    [  24.325,   23.992,   23.785,   23.664,   23.567,   23.485,   23.417,   23.366,   23.324,   23.292, 
      23.267,   23.253,   23.252,   23.257,   23.270,   23.293,   23.328,   23.375,   23.438,   23.512, 
      23.599,   23.699,   23.812,   23.939,   24.081,   24.237,   24.685,   25.205,   25.785,   26.419, 
      27.098,   27.816,   28.568,   29.348,   30.154,   30.983,   31.831,   32.698,   33.580,   34.476, 
      35.385,   36.304,   37.234,   38.172,   39.119,   40.073,   41.034,   42.002,   42.975,   43.954, 
      44.937,   45.924,   46.914,   47.907,   48.902,   49.900,   51.901,   53.910,   55.924,   57.942, 
      59.964,   61.987,   64.010,   66.031,   68.053,   70.073,   72.091,   74.106,   76.117,   78.125, 
      80.129,   82.128,   84.121,   86.108,   88.094,   90.075,   92.047,   94.010,   95.961,   97.899, 
      99.824,  101.736,  103.635,  105.519,  107.389,  109.245,  111.089,  112.920,  114.738,  116.544, 
      118.340,  120.126,  121.901,  123.666,  125.419,  127.163,  128.899,  130.627,  132.347,  134.059, 
      135.764,  137.463,  139.155,  140.839,  142.518,  144.190, ],
    [  25.763,   25.427,   25.214,   25.085,   24.978,   24.884,   24.805,   24.741,   24.683,   24.634, 
      24.593,   24.564,   24.543,   24.529,   24.523,   24.526,   24.543,   24.572,   24.616,   24.673, 
      24.742,   24.825,   24.921,   25.032,   25.158,   25.298,   25.709,   26.194,   26.740,   27.341, 
      27.988,   28.676,   29.399,   30.152,   30.931,   31.733,   32.558,   33.401,   34.261,   35.136, 
      36.025,   36.925,   37.836,   38.757,   39.687,   40.625,   41.571,   42.523,   43.483,   44.448, 
      45.419,   46.393,   47.372,   48.353,   49.338,   50.325,   52.308,   54.299,   56.296,   58.298, 
      60.306,   62.316,   64.326,   66.336,   68.347,   70.357,   72.365,   74.371,   76.374,   78.373, 
      80.369,   82.362,   84.347,   86.328,   88.307,   90.282,   92.249,   94.206,   96.152,   98.085, 
      100.006,  101.913,  103.808,  105.688,  107.553,  109.405,  111.246,  113.073,  114.888,  116.690, 
      118.483,  120.266,  122.038,  123.799,  125.550,  127.291,  129.024,  130.750,  132.467,  134.177, 
      135.880,  137.576,  139.265,  140.948,  142.624,  144.295, ],
    [  27.196,   26.857,   26.639,   26.502,   26.383,   26.279,   26.188,   26.111,   26.039,   25.973, 
      25.914,   25.871,   25.832,   25.800,   25.775,   25.760,   25.760,   25.771,   25.800,   25.839, 
      25.892,   25.959,   26.040,   26.135,   26.246,   26.372,   26.749,   27.200,   27.714,   28.283, 
      28.900,   29.559,   30.254,   30.979,   31.732,   32.510,   33.310,   34.130,   34.968,   35.822, 
      36.690,   37.571,   38.464,   39.367,   40.280,   41.202,   42.132,   43.070,   44.015,   44.967, 
      45.924,   46.886,   47.853,   48.823,   49.796,   50.773,   52.735,   54.708,   56.688,   58.674, 
      60.667,   62.663,   64.660,   66.658,   68.657,   70.657,   72.655,   74.651,   76.645,   78.635, 
      80.624,   82.608,   84.587,   86.561,   88.533,   90.502,   92.463,   94.415,   96.355,   98.283, 
      100.198,  102.101,  103.990,  105.866,  107.727,  109.575,  111.411,  113.235,  115.046,  116.845, 
      118.634,  120.413,  122.182,  123.940,  125.688,  127.426,  129.157,  130.879,  132.594,  134.301, 
      136.002,  137.696,  139.383,  141.063,  142.737,  144.406, ],
    [  28.625,   28.282,   28.059,   27.913,   27.784,   27.669,   27.567,   27.476,   27.390,   27.308, 
      27.231,   27.175,   27.119,   27.069,   27.026,   26.994,   26.977,   26.972,   26.986,   27.010, 
      27.048,   27.099,   27.165,   27.246,   27.344,   27.457,   27.801,   28.221,   28.705,   29.244, 
      29.833,   30.463,   31.131,   31.830,   32.557,   33.310,   34.087,   34.884,   35.700,   36.533, 
      37.381,   38.242,   39.116,   40.001,   40.897,   41.802,   42.717,   43.640,   44.570,   45.508, 
      46.452,   47.402,   48.356,   49.314,   50.276,   51.242,   53.184,   55.137,   57.099,   59.069, 
      61.046,   63.028,   65.012,   66.997,   68.984,   70.972,   72.960,   74.946,   76.930,   78.912, 
      80.892,   82.868,   84.839,   86.806,   88.772,   90.734,   92.689,   94.634,   96.568,   98.491, 
      100.401,  102.299,  104.183,  106.054,  107.911,  109.755,  111.586,  113.406,  115.213,  117.008, 
      118.794,  120.569,  122.335,  124.090,  125.834,  127.569,  129.297,  131.017,  132.728,  134.433, 
      136.131,  137.822,  139.507,  141.185,  142.857,  144.523, ],
    [  30.048,   29.702,   29.473,   29.317,   29.179,   29.054,   28.942,   28.835,   28.733,   28.637, 
      28.545,   28.474,   28.402,   28.333,   28.274,   28.225,   28.194,   28.173,   28.173,   28.183, 
      28.206,   28.244,   28.296,   28.364,   28.449,   28.550,   28.864,   29.255,   29.710,   30.222, 
      30.782,   31.386,   32.028,   32.701,   33.404,   34.132,   34.885,   35.660,   36.455,   37.267, 
      38.094,   38.936,   39.791,   40.658,   41.537,   42.426,   43.325,   44.232,   45.148,   46.072, 
      47.002,   47.939,   48.880,   49.826,   50.777,   51.731,   53.652,   55.585,   57.529,   59.482, 
      61.443,   63.410,   65.380,   67.351,   69.326,   71.303,   73.279,   75.255,   77.230,   79.202, 
      81.173,   83.142,   85.104,   87.064,   89.022,   90.978,   92.926,   94.865,   96.793,   98.710, 
      100.615,  102.507,  104.386,  106.252,  108.104,  109.943,  111.771,  113.586,  115.389,  117.180, 
      118.961,  120.734,  122.495,  124.247,  125.988,  127.720,  129.444,  131.161,  132.870,  134.571, 
      136.267,  137.955,  139.637,  141.313,  142.982,  144.646, ],
    [  31.467,   31.117,   30.880,   30.716,   30.569,   30.434,   30.309,   30.188,   30.071,   29.957, 
      29.857,   29.767,   29.677,   29.592,   29.516,   29.454,   29.408,   29.374,   29.361,   29.357, 
      29.367,   29.392,   29.432,   29.488,   29.561,   29.650,   29.936,   30.300,   30.728,   31.213, 
      31.748,   32.326,   32.942,   33.591,   34.269,   34.975,   35.705,   36.458,   37.231,   38.022, 
      38.830,   39.652,   40.489,   41.338,   42.199,   43.072,   43.954,   44.846,   45.747,   46.657, 
      47.573,   48.496,   49.425,   50.359,   51.298,   52.241,   54.140,   56.053,   57.978,   59.913, 
      61.858,   63.809,   65.764,   67.722,   69.684,   71.649,   73.614,   75.579,   77.543,   79.506, 
      81.468,   83.428,   85.382,   87.334,   89.285,   91.233,   93.174,   95.107,   97.029,   98.939, 
      100.838,  102.725,  104.599,  106.460,  108.307,  110.141,  111.964,  113.775,  115.573,  117.360, 
      119.138,  120.906,  122.664,  124.412,  126.149,  127.878,  129.599,  131.312,  133.018,  134.717, 
      136.409,  138.095,  139.774,  141.447,  143.114,  144.776, ],
    [  32.879,   32.524,   32.282,   32.108,   31.952,   31.807,   31.667,   31.532,   31.400,   31.271, 
      31.161,   31.053,   30.946,   30.845,   30.754,   30.680,   30.620,   30.575,   30.548,   30.531, 
      30.529,   30.542,   30.570,   30.615,   30.677,   30.756,   31.017,   31.355,   31.758,   32.218, 
      32.728,   33.281,   33.873,   34.498,   35.153,   35.836,   36.544,   37.275,   38.027,   38.798, 
      39.586,   40.389,   41.207,   42.038,   42.882,   43.738,   44.604,   45.481,   46.367,   47.262, 
      48.165,   49.075,   49.990,   50.912,   51.838,   52.770,   54.646,   56.539,   58.445,   60.362, 
      62.290,   64.225,   66.165,   68.109,   70.058,   72.010,   73.964,   75.918,   77.871,   79.824, 
      81.776,   83.727,   85.673,   87.616,   89.559,   91.500,   93.434,   95.360,   97.275,   99.180, 
      101.073,  102.954,  104.822,  106.677,  108.519,  110.348,  112.166,  113.973,  115.766,  117.549, 
      119.322,  121.086,  122.841,  124.585,  126.319,  128.043,  129.761,  131.471,  133.174,  134.869, 
      136.559,  138.241,  139.918,  141.588,  143.252,  144.911, ],
    [  34.283,   33.925,   33.676,   33.495,   33.330,   33.170,   33.018,   32.867,   32.718,   32.583, 
      32.455,   32.329,   32.206,   32.091,   31.987,   31.901,   31.828,   31.775,   31.733,   31.705, 
      31.692,   31.694,   31.711,   31.745,   31.798,   31.867,   32.104,   32.418,   32.798,   33.234, 
      33.720,   34.250,   34.819,   35.421,   36.054,   36.714,   37.400,   38.110,   38.842,   39.592, 
      40.361,   41.145,   41.945,   42.759,   43.585,   44.424,   45.275,   46.136,   47.007,   47.887, 
      48.776,   49.672,   50.575,   51.484,   52.398,   53.317,   55.171,   57.043,   58.929,   60.828, 
      62.738,   64.657,   66.582,   68.512,   70.447,   72.386,   74.327,   76.270,   78.213,   80.155, 
      82.098,   84.039,   85.976,   87.910,   89.846,   91.779,   93.706,   95.624,   97.533,   99.431, 
      101.317,  103.192,  105.055,  106.904,  108.741,  110.565,  112.378,  114.179,  115.968,  117.746, 
      119.515,  121.275,  123.025,  124.765,  126.495,  128.216,  129.930,  131.637,  133.336,  135.029, 
      136.715,  138.395,  140.068,  141.735,  143.397,  145.053, ],
    [  35.682,   35.318,   35.064,   34.874,   34.697,   34.525,   34.356,   34.190,   34.031,   33.885, 
      33.739,   33.596,   33.458,   33.329,   33.215,   33.117,   33.031,   32.970,   32.916,   32.878, 
      32.854,   32.846,   32.853,   32.878,   32.921,   32.982,   33.197,   33.489,   33.846,   34.260, 
      34.724,   35.232,   35.779,   36.359,   36.969,   37.608,   38.273,   38.963,   39.674,   40.405, 
      41.154,   41.920,   42.701,   43.497,   44.307,   45.129,   45.964,   46.809,   47.665,   48.531, 
      49.406,   50.288,   51.178,   52.074,   52.976,   53.883,   55.714,   57.565,   59.430,   61.310, 
      63.203,   65.106,   67.015,   68.930,   70.851,   72.777,   74.705,   76.636,   78.567,   80.499, 
      82.432,   84.363,   86.291,   88.217,   90.144,   92.069,   93.988,   95.899,   97.801,   99.692, 
      101.572,  103.441,  105.297,  107.141,  108.972,  110.790,  112.598,  114.394,  116.178,  117.952, 
      119.716,  121.472,  123.217,  124.953,  126.680,  128.397,  130.107,  131.810,  133.506,  135.195, 
      136.878,  138.554,  140.225,  141.889,  143.548,  145.201, ],
    [  37.073,   36.706,   36.444,   36.245,   36.054,   35.867,   35.683,   35.504,   35.339,   35.173, 
      35.011,   34.852,   34.700,   34.561,   34.437,   34.327,   34.233,   34.162,   34.097,   34.048, 
      34.015,   33.998,   33.996,   34.012,   34.048,   34.100,   34.295,   34.566,   34.903,   35.295, 
      35.738,   36.225,   36.750,   37.309,   37.899,   38.517,   39.162,   39.831,   40.523,   41.234, 
      41.965,   42.712,   43.476,   44.254,   45.047,   45.853,   46.671,   47.501,   48.342,   49.193, 
      50.054,   50.923,   51.799,   52.682,   53.571,   54.467,   56.275,   58.103,   59.949,   61.809, 
      63.684,   65.570,   67.463,   69.362,   71.269,   73.182,   75.098,   77.016,   78.936,   80.857, 
      82.779,   84.700,   86.618,   88.535,   90.453,   92.370,   94.282,   96.185,   98.079,   99.963, 
      101.837,  103.699,  105.549,  107.387,  109.212,  111.025,  112.827,  114.618,  116.397,  118.166, 
      119.925,  121.676,  123.418,  125.149,  126.871,  128.585,  130.291,  131.990,  133.683,  135.368, 
      137.047,  138.721,  140.388,  142.049,  143.705,  145.355, ],
    [  38.457,   38.085,   37.816,   37.603,   37.398,   37.197,   36.998,   36.815,   36.631,   36.448, 
      36.269,   36.098,   35.935,   35.787,   35.652,   35.532,   35.433,   35.349,   35.275,   35.217, 
      35.175,   35.150,   35.139,   35.148,   35.176,   35.221,   35.396,   35.649,   35.966,   36.339, 
      36.761,   37.228,   37.733,   38.272,   38.841,   39.440,   40.065,   40.714,   41.387,   42.079, 
      42.791,   43.521,   44.266,   45.028,   45.804,   46.593,   47.396,   48.210,   49.036,   49.873, 
      50.719,   51.574,   52.437,   53.307,   54.184,   55.067,   56.852,   58.659,   60.483,   62.325, 
      64.181,   66.049,   67.926,   69.810,   71.702,   73.601,   75.503,   77.409,   79.317,   81.227, 
      83.138,   85.049,   86.958,   88.865,   90.775,   92.683,   94.586,   96.482,   98.369,  100.245, 
      102.112,  103.967,  105.811,  107.642,  109.461,  111.268,  113.065,  114.851,  116.625,  118.388, 
      120.143,  121.889,  123.626,  125.353,  127.071,  128.780,  130.482,  132.178,  133.866,  135.548, 
      137.224,  138.893,  140.557,  142.215,  143.868,  145.515, ],
    [  39.834,   39.454,   39.175,   38.948,   38.729,   38.512,   38.311,   38.109,   37.908,   37.709, 
      37.517,   37.335,   37.164,   37.005,   36.861,   36.733,   36.628,   36.532,   36.450,   36.384, 
      36.334,   36.300,   36.282,   36.284,   36.305,   36.343,   36.502,   36.737,   37.035,   37.389, 
      37.793,   38.241,   38.726,   39.246,   39.796,   40.374,   40.980,   41.611,   42.265,   42.939, 
      43.633,   44.344,   45.073,   45.817,   46.576,   47.350,   48.136,   48.936,   49.747,   50.569, 
      51.401,   52.242,   53.092,   53.949,   54.813,   55.684,   57.446,   59.230,   61.034,   62.855, 
      64.693,   66.544,   68.404,   70.272,   72.149,   74.033,   75.923,   77.816,   79.712,   81.610, 
      83.510,   85.411,   87.309,   89.207,   91.107,   93.007,   94.901,   96.789,   98.668,  100.538, 
      102.397,  104.245,  106.082,  107.907,  109.720,  111.521,  113.312,  115.092,  116.860,  118.619, 
      120.368,  122.110,  123.842,  125.564,  127.277,  128.982,  130.680,  132.372,  134.056,  135.734, 
      137.406,  139.073,  140.733,  142.388,  144.037,  145.681, ],
    [  41.196,   40.811,   40.520,   40.278,   40.044,   39.825,   39.605,   39.386,   39.169,   38.958, 
      38.755,   38.564,   38.384,   38.217,   38.063,   37.933,   37.815,   37.711,   37.622,   37.548, 
      37.490,   37.450,   37.425,   37.420,   37.435,   37.467,   37.610,   37.829,   38.110,   38.447, 
      38.832,   39.262,   39.729,   40.229,   40.761,   41.321,   41.908,   42.521,   43.156,   43.813, 
      44.489,   45.183,   45.894,   46.622,   47.365,   48.122,   48.893,   49.677,   50.473,   51.281, 
      52.099,   52.927,   53.763,   54.607,   55.459,   56.317,   58.055,   59.817,   61.600,   63.401, 
      65.220,   67.053,   68.896,   70.748,   72.610,   74.480,   76.355,   78.235,   80.119,   82.005, 
      83.894,   85.784,   87.672,   89.560,   91.451,   93.341,   95.228,   97.107,   98.978,  100.840, 
      102.692,  104.533,  106.363,  108.181,  109.988,  111.783,  113.567,  115.342,  117.105,  118.857, 
      120.602,  122.338,  124.065,  125.783,  127.492,  129.192,  130.886,  132.573,  134.254,  135.928, 
      137.596,  139.259,  140.916,  142.567,  144.213,  145.854, ],
    [  42.546,   42.151,   41.847,   41.592,   41.356,   41.118,   40.881,   40.646,   40.417,   40.195, 
      39.984,   39.785,   39.597,   39.420,   39.264,   39.125,   38.998,   38.887,   38.790,   38.710, 
      38.645,   38.598,   38.567,   38.557,   38.567,   38.593,   38.721,   38.924,   39.189,   39.509, 
      39.878,   40.290,   40.740,   41.223,   41.736,   42.278,   42.848,   43.442,   44.060,   44.699, 
      45.358,   46.035,   46.729,   47.440,   48.167,   48.909,   49.664,   50.433,   51.215,   52.008, 
      52.812,   53.626,   54.449,   55.280,   56.119,   56.966,   58.680,   60.419,   62.181,   63.962, 
      65.762,   67.576,   69.402,   71.238,   73.085,   74.940,   76.801,   78.668,   80.539,   82.413, 
      84.290,   86.169,   88.046,   89.925,   91.806,   93.687,   95.564,   97.435,   99.298,  101.152, 
      102.996,  104.830,  106.653,  108.465,  110.264,  112.053,  113.832,  115.600,  117.357,  119.104, 
      120.843,  122.574,  124.296,  126.009,  127.713,  129.409,  131.098,  132.782,  134.458,  136.128, 
      137.792,  139.451,  141.104,  142.752,  144.394,  146.032, ],
    [  43.879,   43.472,   43.160,   42.903,   42.647,   42.392,   42.139,   41.892,   41.653,   41.424, 
      41.205,   40.998,   40.801,   40.621,   40.461,   40.311,   40.177,   40.059,   39.956,   39.869, 
      39.799,   39.745,   39.709,   39.694,   39.698,   39.720,   39.834,   40.023,   40.273,   40.577, 
      40.930,   41.326,   41.759,   42.224,   42.720,   43.245,   43.797,   44.375,   44.975,   45.597, 
      46.239,   46.899,   47.578,   48.273,   48.984,   49.710,   50.450,   51.204,   51.971,   52.750, 
      53.540,   54.341,   55.150,   55.968,   56.795,   57.629,   59.319,   61.036,   62.776,   64.537, 
      66.317,   68.114,   69.922,   71.742,   73.572,   75.413,   77.260,   79.113,   80.971,   82.832, 
      84.698,   86.566,   88.432,   90.300,   92.171,   94.043,   95.912,   97.774,   99.628,  101.474, 
      103.310,  105.137,  106.953,  108.757,  110.550,  112.332,  114.104,  115.867,  117.618,  119.359, 
      121.093,  122.818,  124.535,  126.243,  127.942,  129.633,  131.318,  132.997,  134.669,  136.334, 
      137.995,  139.650,  141.299,  142.943,  144.582,  146.216, ],
    [  45.193,   44.785,   44.467,   44.190,   43.916,   43.647,   43.383,   43.127,   42.881,   42.644, 
      42.418,   42.202,   42.002,   41.819,   41.647,   41.491,   41.352,   41.228,   41.119,   41.026, 
      40.951,   40.891,   40.850,   40.830,   40.830,   40.847,   40.949,   41.124,   41.360,   41.650, 
      41.988,   42.368,   42.784,   43.234,   43.713,   44.221,   44.757,   45.317,   45.901,   46.507, 
      47.132,   47.776,   48.438,   49.118,   49.813,   50.524,   51.249,   51.988,   52.741,   53.506, 
      54.282,   55.069,   55.865,   56.671,   57.484,   58.306,   59.973,   61.667,   63.385,   65.126, 
      66.887,   68.665,   70.456,   72.259,   74.073,   75.898,   77.731,   79.570,   81.415,   83.264, 
      85.118,   86.974,   88.830,   90.687,   92.548,   94.410,   96.269,   98.123,   99.968,  101.806, 
      103.634,  105.453,  107.261,  109.059,  110.845,  112.620,  114.386,  116.142,  117.887,  119.622, 
      121.350,  123.070,  124.782,  126.484,  128.178,  129.865,  131.545,  133.219,  134.886,  136.548, 
      138.204,  139.855,  141.501,  143.141,  144.776,  146.406, ],
    [  46.503,   46.083,   45.748,   45.453,   45.167,   44.887,   44.615,   44.353,   44.100,   43.855, 
      43.621,   43.403,   43.201,   43.007,   42.829,   42.667,   42.522,   42.393,   42.278,   42.180, 
      42.099,   42.035,   41.991,   41.967,   41.963,   41.975,   42.066,   42.228,   42.451,   42.727, 
      43.050,   43.415,   43.817,   44.250,   44.714,   45.206,   45.725,   46.270,   46.837,   47.427, 
      48.036,   48.664,   49.311,   49.975,   50.655,   51.350,   52.061,   52.785,   53.524,   54.275, 
      55.037,   55.811,   56.594,   57.387,   58.188,   58.997,   60.640,   62.312,   64.009,   65.729, 
      67.470,   69.230,   71.003,   72.788,   74.587,   76.397,   78.215,   80.040,   81.871,   83.708, 
      85.549,   87.394,   89.238,   91.085,   92.935,   94.788,   96.637,   98.482,  100.319,  102.148, 
      103.968,  105.779,  107.579,  109.369,  111.148,  112.916,  114.675,  116.425,  118.164,  119.893, 
      121.615,  123.330,  125.036,  126.733,  128.422,  130.103,  131.778,  133.448,  135.111,  136.768, 
      138.420,  140.067,  141.708,  143.345,  144.976,  146.603, ],
    [  47.786,   47.354,   47.005,   46.699,   46.404,   46.117,   45.839,   45.570,   45.308,   45.057, 
      44.822,   44.601,   44.388,   44.189,   44.006,   43.839,   43.688,   43.554,   43.434,   43.333, 
      43.246,   43.178,   43.131,   43.103,   43.095,   43.104,   43.183,   43.335,   43.545,   43.808, 
      44.117,   44.468,   44.855,   45.274,   45.722,   46.199,   46.702,   47.231,   47.783,   48.357, 
      48.951,   49.563,   50.194,   50.843,   51.508,   52.189,   52.885,   53.595,   54.319,   55.057, 
      55.806,   56.566,   57.336,   58.116,   58.904,   59.702,   61.321,   62.970,   64.645,   66.345, 
      68.066,   69.807,   71.562,   73.331,   75.113,   76.907,   78.711,   80.522,   82.339,   84.163, 
      85.992,   87.824,   89.657,   91.493,   93.333,   95.176,   97.015,   98.850,  100.678,  102.499, 
      104.311,  106.113,  107.906,  109.689,  111.460,  113.221,  114.974,  116.717,  118.449,  120.172, 
      121.888,  123.597,  125.297,  126.989,  128.673,  130.349,  132.019,  133.684,  135.342,  136.995, 
      138.642,  140.285,  141.922,  143.554,  145.182,  146.805, ],
    [  49.045,   48.604,   48.246,   47.933,   47.631,   47.338,   47.053,   46.776,   46.510,   46.258, 
      46.019,   45.788,   45.570,   45.366,   45.178,   45.006,   44.852,   44.712,   44.589,   44.483, 
      44.393,   44.321,   44.270,   44.239,   44.228,   44.232,   44.302,   44.443,   44.641,   44.892, 
      45.188,   45.526,   45.899,   46.303,   46.737,   47.199,   47.687,   48.201,   48.738,   49.296, 
      49.875,   50.472,   51.088,   51.722,   52.372,   53.039,   53.720,   54.416,   55.127,   55.850, 
      56.586,   57.333,   58.091,   58.858,   59.634,   60.419,   62.014,   63.641,   65.295,   66.973, 
      68.675,   70.397,   72.134,   73.886,   75.652,   77.430,   79.218,   81.015,   82.819,   84.629, 
      86.445,   88.266,   90.087,   91.912,   93.742,   95.574,   97.404,   99.229,  101.048,  102.860, 
      104.663,  106.457,  108.242,  110.017,  111.781,  113.535,  115.280,  117.016,  118.742,  120.459, 
      122.169,  123.871,  125.566,  127.252,  128.931,  130.601,  132.267,  133.926,  135.580,  137.228, 
      138.871,  140.509,  142.142,  143.770,  145.394,  147.013, ],
    [  50.286,   49.839,   49.477,   49.157,   48.849,   48.548,   48.257,   47.977,   47.711,   47.453, 
      47.205,   46.969,   46.747,   46.539,   46.346,   46.170,   46.011,   45.867,   45.741,   45.630, 
      45.536,   45.462,   45.408,   45.374,   45.360,   45.361,   45.422,   45.552,   45.740,   45.979, 
      46.263,   46.588,   46.948,   47.338,   47.758,   48.205,   48.679,   49.178,   49.700,   50.244, 
      50.808,   51.391,   51.992,   52.611,   53.247,   53.899,   54.567,   55.249,   55.946,   56.656, 
      57.379,   58.112,   58.857,   59.611,   60.375,   61.148,   62.720,   64.324,   65.956,   67.614, 
      69.297,   71.000,   72.719,   74.453,   76.202,   77.965,   79.738,   81.520,   83.310,   85.107, 
      86.910,   88.718,   90.528,   92.341,   94.160,   95.982,   97.802,   99.618,  101.427,  103.230, 
      105.024,  106.810,  108.587,  110.354,  112.110,  113.857,  115.595,  117.324,  119.043,  120.753, 
      122.457,  124.154,  125.842,  127.523,  129.196,  130.861,  132.521,  134.176,  135.824,  137.467, 
      139.106,  140.740,  142.369,  143.992,  145.612,  147.227, ],
    [  51.516,   51.065,   50.697,   50.371,   50.055,   49.750,   49.458,   49.177,   48.902,   48.637, 
      48.385,   48.145,   47.919,   47.706,   47.510,   47.330,   47.167,   47.021,   46.890,   46.776, 
      46.679,   46.603,   46.545,   46.510,   46.492,   46.491,   46.543,   46.664,   46.841,   47.068, 
      47.341,   47.654,   48.001,   48.378,   48.785,   49.218,   49.678,   50.163,   50.671,   51.200, 
      51.750,   52.318,   52.905,   53.510,   54.132,   54.770,   55.424,   56.092,   56.776,   57.473, 
      58.182,   58.903,   59.635,   60.377,   61.129,   61.889,   63.438,   65.020,   66.630,   68.268, 
      69.930,   71.614,   73.315,   75.032,   76.764,   78.511,   80.268,   82.036,   83.812,   85.595, 
      87.386,   89.182,   90.979,   92.781,   94.589,   96.400,   98.210,  100.016,  101.816,  103.609, 
      105.395,  107.172,  108.941,  110.700,  112.448,  114.187,  115.918,  117.640,  119.352,  121.056, 
      122.753,  124.443,  126.126,  127.801,  129.468,  131.128,  132.782,  134.432,  136.075,  137.714, 
      139.347,  140.977,  142.601,  144.221,  145.836,  147.447, ],
    [  52.736,   52.280,   51.905,   51.574,   51.256,   50.951,   50.654,   50.364,   50.084,   49.816, 
      49.560,   49.316,   49.086,   48.870,   48.670,   48.487,   48.321,   48.171,   48.036,   47.920, 
      47.821,   47.742,   47.682,   47.645,   47.624,   47.620,   47.664,   47.776,   47.943,   48.161, 
      48.422,   48.723,   49.059,   49.423,   49.817,   50.237,   50.683,   51.154,   51.648,   52.164, 
      52.700,   53.254,   53.827,   54.418,   55.026,   55.650,   56.291,   56.946,   57.616,   58.300, 
      58.996,   59.705,   60.424,   61.154,   61.893,   62.642,   64.168,   65.727,   67.316,   68.933, 
      70.575,   72.240,   73.923,   75.622,   77.338,   79.068,   80.810,   82.563,   84.325,   86.094, 
      87.872,   89.655,   91.441,   93.231,   95.028,   96.828,   98.627,  100.423,  102.214,  103.998, 
      105.774,  107.543,  109.303,  111.054,  112.794,  114.526,  116.249,  117.964,  119.669,  121.366, 
      123.056,  124.740,  126.417,  128.086,  129.747,  131.401,  133.050,  134.694,  136.333,  137.966, 
      139.595,  141.220,  142.839,  144.455,  146.065,  147.672, ],
    [  53.944,   53.484,   53.107,   52.775,   52.456,   52.142,   51.838,   51.545,   51.262,   50.990, 
      50.729,   50.483,   50.249,   50.030,   49.827,   49.641,   49.472,   49.318,   49.182,   49.062, 
      48.962,   48.880,   48.819,   48.779,   48.756,   48.749,   48.786,   48.890,   49.048,   49.255, 
      49.506,   49.796,   50.120,   50.473,   50.853,   51.261,   51.694,   52.152,   52.633,   53.135, 
      53.657,   54.198,   54.757,   55.335,   55.929,   56.540,   57.167,   57.809,   58.466,   59.137, 
      59.821,   60.517,   61.224,   61.941,   62.669,   63.406,   64.908,   66.445,   68.013,   69.609, 
      71.232,   72.878,   74.542,   76.223,   77.922,   79.637,   81.363,   83.101,   84.848,   86.604, 
      88.369,   90.139,   91.913,   93.691,   95.476,   97.266,   99.055,  100.840,  102.621,  104.395, 
      106.163,  107.923,  109.674,  111.416,  113.149,  114.872,  116.588,  118.296,  119.994,  121.684, 
      123.367,  125.045,  126.715,  128.378,  130.033,  131.682,  133.325,  134.964,  136.597,  138.225, 
      139.849,  141.469,  143.084,  144.695,  146.301,  147.904, ],
    [  55.145,   54.685,   54.307,   53.969,   53.642,   53.324,   53.017,   52.721,   52.434,   52.158, 
      51.895,   51.646,   51.408,   51.186,   50.981,   50.792,   50.619,   50.465,   50.325,   50.204, 
      50.102,   50.018,   49.955,   49.913,   49.888,   49.878,   49.908,   50.004,   50.154,   50.352, 
      50.593,   50.872,   51.185,   51.526,   51.895,   52.290,   52.711,   53.156,   53.624,   54.113, 
      54.622,   55.149,   55.695,   56.259,   56.841,   57.439,   58.052,   58.681,   59.326,   59.984, 
      60.656,   61.339,   62.034,   62.739,   63.455,   64.180,   65.660,   67.175,   68.721,   70.296, 
      71.900,   73.526,   75.172,   76.836,   78.518,   80.216,   81.927,   83.650,   85.383,   87.125, 
      88.876,   90.633,   92.394,   94.160,   95.934,   97.713,   99.491,  101.267,  103.037,  104.802, 
      106.560,  108.311,  110.054,  111.788,  113.512,  115.227,  116.935,  118.635,  120.326,  122.009, 
      123.686,  125.357,  127.020,  128.677,  130.326,  131.969,  133.607,  135.240,  136.868,  138.491, 
      140.110,  141.724,  143.335,  144.941,  146.543,  148.141, ],
    [  56.345,   55.880,   55.495,   55.152,   54.822,   54.502,   54.191,   53.891,   53.601,   53.323, 
      53.057,   52.804,   52.564,   52.340,   52.132,   51.940,   51.766,   51.608,   51.467,   51.345, 
      51.241,   51.155,   51.091,   51.046,   51.020,   51.007,   51.031,   51.120,   51.261,   51.450, 
      51.682,   51.951,   52.253,   52.583,   52.941,   53.324,   53.733,   54.165,   54.621,   55.097, 
      55.593,   56.108,   56.641,   57.192,   57.760,   58.345,   58.946,   59.563,   60.194,   60.840, 
      61.499,   62.171,   62.853,   63.547,   64.251,   64.965,   66.422,   67.915,   69.440,   70.995, 
      72.578,   74.186,   75.813,   77.459,   79.124,   80.806,   82.501,   84.209,   85.927,   87.655, 
      89.393,   91.137,   92.886,   94.640,   96.402,   98.169,   99.937,  101.702,  103.463,  105.218, 
      106.966,  108.708,  110.442,  112.167,  113.883,  115.590,  117.290,  118.983,  120.666,  122.342, 
      124.012,  125.676,  127.333,  128.983,  130.626,  132.263,  133.895,  135.522,  137.145,  138.762, 
      140.376,  141.986,  143.591,  145.193,  146.790,  148.384, ],
    [  58.712,   58.242,   57.851,   57.502,   57.166,   56.840,   56.524,   56.218,   55.923,   55.639, 
      55.369,   55.113,   54.868,   54.639,   54.426,   54.231,   54.053,   53.893,   53.749,   53.623, 
      53.515,   53.428,   53.360,   53.312,   53.282,   53.266,   53.277,   53.353,   53.479,   53.652, 
      53.866,   54.117,   54.399,   54.708,   55.044,   55.404,   55.790,   56.199,   56.631,   57.082, 
      57.554,   58.044,   58.552,   59.078,   59.622,   60.182,   60.758,   61.350,   61.957,   62.579, 
      63.214,   63.861,   64.521,   65.191,   65.872,   66.562,   67.975,   69.424,   70.907,   72.422, 
      73.966,   75.535,   77.126,   78.737,   80.368,   82.017,   83.680,   85.357,   87.046,   88.746, 
      90.456,   92.174,   93.898,   95.627,   97.366,   99.111,  100.856,  102.600,  104.340,  106.075, 
      107.804,  109.527,  111.243,  112.950,  114.649,  116.340,  118.024,  119.700,  121.368,  123.029, 
      124.685,  126.335,  127.979,  129.616,  131.246,  132.871,  134.491,  136.107,  137.718,  139.325, 
      140.928,  142.527,  144.123,  145.714,  147.302,  148.887, ],
    [  61.056,   60.582,   60.186,   59.833,   59.491,   59.160,   58.840,   58.530,   58.232,   57.944, 
      57.669,   57.408,   57.161,   56.929,   56.713,   56.516,   56.335,   56.172,   56.026,   55.897, 
      55.788,   55.698,   55.628,   55.577,   55.544,   55.523,   55.525,   55.588,   55.701,   55.859, 
      56.057,   56.291,   56.555,   56.845,   57.160,   57.500,   57.864,   58.252,   58.660,   59.089, 
      59.538,   60.004,   60.489,   60.991,   61.511,   62.047,   62.600,   63.168,   63.751,   64.350, 
      64.961,   65.586,   66.222,   66.870,   67.528,   68.197,   69.565,   70.972,   72.414,   73.888, 
      75.393,   76.925,   78.479,   80.055,   81.651,   83.267,   84.899,   86.545,   88.204,   89.875, 
      91.558,   93.249,   94.947,   96.652,   98.366,  100.088,  101.811,  103.533,  105.252,  106.966, 
      108.676,  110.380,  112.076,  113.766,  115.447,  117.120,  118.788,  120.448,  122.101,  123.746, 
      125.387,  127.023,  128.653,  130.276,  131.893,  133.505,  135.113,  136.717,  138.316,  139.912, 
      141.504,  143.093,  144.678,  146.259,  147.837,  149.412, ],
    [  63.381,   62.903,   62.504,   62.146,   61.802,   61.468,   61.144,   60.830,   60.528,   60.238, 
      59.960,   59.697,   59.447,   59.213,   58.996,   58.796,   58.613,   58.448,   58.299,   58.170, 
      58.059,   57.966,   57.894,   57.841,   57.805,   57.781,   57.773,   57.826,   57.926,   58.071, 
      58.255,   58.473,   58.721,   58.993,   59.289,   59.609,   59.953,   60.320,   60.708,   61.115, 
      61.541,   61.986,   62.448,   62.928,   63.425,   63.938,   64.468,   65.013,   65.574,   66.150, 
      66.739,   67.341,   67.955,   68.581,   69.217,   69.864,   71.190,   72.555,   73.956,   75.390, 
      76.857,   78.352,   79.869,   81.410,   82.972,   84.555,   86.155,   87.770,   89.400,   91.042, 
      92.696,   94.361,   96.032,   97.712,   99.402,  101.100,  102.800,  104.500,  106.197,  107.891, 
      109.581,  111.264,  112.942,  114.613,  116.276,  117.932,  119.582,  121.226,  122.862,  124.492, 
      126.118,  127.739,  129.354,  130.964,  132.567,  134.166,  135.761,  137.352,  138.939,  140.523, 
      142.104,  143.682,  145.256,  146.827,  148.395,  149.960, ],
    [  65.691,   65.211,   64.809,   64.448,   64.100,   63.763,   63.437,   63.121,   62.816,   62.524, 
      62.245,   61.980,   61.729,   61.493,   61.273,   61.071,   60.887,   60.721,   60.571,   60.440, 
      60.327,   60.233,   60.159,   60.103,   60.064,   60.038,   60.021,   60.065,   60.154,   60.287, 
      60.457,   60.661,   60.893,   61.149,   61.428,   61.730,   62.055,   62.403,   62.770,   63.157, 
      63.563,   63.986,   64.427,   64.885,   65.361,   65.852,   66.360,   66.884,   67.423,   67.977, 
      68.545,   69.125,   69.718,   70.322,   70.937,   71.563,   72.847,   74.171,   75.532,   76.927, 
      78.356,   79.813,   81.295,   82.800,   84.329,   85.879,   87.447,   89.031,   90.631,   92.244, 
      93.870,   95.507,   97.152,   98.806,  100.472,  102.146,  103.823,  105.500,  107.176,  108.848, 
      110.517,  112.181,  113.839,  115.490,  117.135,  118.773,  120.405,  122.032,  123.652,  125.266, 
      126.876,  128.482,  130.082,  131.678,  133.267,  134.852,  136.434,  138.012,  139.587,  141.159, 
      142.728,  144.294,  145.857,  147.417,  148.975,  150.530, ],
    [  67.989,   67.507,   67.102,   66.740,   66.389,   66.050,   65.722,   65.405,   65.099,   64.806, 
      64.525,   64.258,   64.005,   63.768,   63.547,   63.344,   63.159,   62.992,   62.841,   62.708, 
      62.593,   62.498,   62.422,   62.364,   62.323,   62.295,   62.270,   62.305,   62.384,   62.506, 
      62.664,   62.855,   63.073,   63.313,   63.576,   63.861,   64.169,   64.498,   64.847,   65.215, 
      65.601,   66.004,   66.424,   66.862,   67.317,   67.788,   68.275,   68.777,   69.296,   69.829, 
      70.376,   70.935,   71.507,   72.090,   72.685,   73.290,   74.533,   75.818,   77.140,   78.496, 
      79.888,   81.309,   82.755,   84.225,   85.720,   87.237,   88.773,   90.327,   91.896,   93.480, 
      95.078,   96.688,   98.306,   99.934,  101.575,  103.225,  104.878,  106.533,  108.186,  109.837, 
      111.485,  113.128,  114.766,  116.398,  118.023,  119.643,  121.258,  122.867,  124.470,  126.067, 
      127.661,  129.251,  130.837,  132.417,  133.992,  135.563,  137.131,  138.697,  140.259,  141.818, 
      143.375,  144.929,  146.481,  148.030,  149.577,  151.121, ],
    [  70.277,   69.794,   69.388,   69.024,   68.673,   68.333,   68.004,   67.685,   67.377,   67.082, 
      66.800,   66.533,   66.279,   66.041,   65.819,   65.616,   65.429,   65.260,   65.108,   64.974, 
      64.858,   64.762,   64.683,   64.625,   64.580,   64.552,   64.518,   64.546,   64.616,   64.728, 
      64.875,   65.054,   65.259,   65.485,   65.732,   66.001,   66.292,   66.604,   66.935,   67.285, 
      67.653,   68.037,   68.438,   68.856,   69.291,   69.742,   70.209,   70.692,   71.190,   71.703, 
      72.230,   72.769,   73.321,   73.884,   74.459,   75.044,   76.248,   77.493,   78.777,   80.096, 
      81.450,   82.835,   84.246,   85.682,   87.143,   88.628,   90.132,   91.655,   93.194,   94.748, 
      96.318,   97.901,   99.492,  101.094,  102.710,  104.335,  105.965,  107.597,  109.227,  110.856, 
      112.482,  114.105,  115.723,  117.335,  118.941,  120.542,  122.138,  123.730,  125.315,  126.895, 
      128.473,  130.047,  131.617,  133.183,  134.743,  136.300,  137.854,  139.406,  140.955,  142.501, 
      144.045,  145.588,  147.128,  148.665,  150.201,  151.734, ],
    [  72.561,   72.077,   71.670,   71.305,   70.952,   70.610,   70.280,   69.960,   69.652,   69.356, 
      69.073,   68.805,   68.551,   68.312,   68.089,   67.884,   67.696,   67.527,   67.374,   67.239, 
      67.122,   67.023,   66.944,   66.883,   66.838,   66.808,   66.767,   66.788,   66.849,   66.952, 
      67.089,   67.257,   67.449,   67.662,   67.895,   68.149,   68.425,   68.721,   69.035,   69.368, 
      69.717,   70.083,   70.467,   70.866,   71.282,   71.714,   72.162,   72.625,   73.104,   73.598, 
      74.105,   74.625,   75.158,   75.701,   76.256,   76.822,   77.988,   79.195,   80.441,   81.724, 
      83.042,   84.391,   85.767,   87.169,   88.597,   90.049,   91.522,   93.014,   94.523,   96.049, 
      97.590,   99.145,  100.709,  102.285,  103.876,  105.476,  107.082,  108.691,  110.299,  111.905, 
      113.510,  115.111,  116.708,  118.300,  119.887,  121.468,  123.046,  124.620,  126.187,  127.750, 
      129.311,  130.869,  132.423,  133.973,  135.518,  137.060,  138.600,  140.138,  141.674,  143.207, 
      144.738,  146.268,  147.796,  149.322,  150.846,  152.368, ],
    [  74.839,   74.354,   73.946,   73.580,   73.227,   72.885,   72.553,   72.233,   71.924,   71.628, 
      71.345,   71.075,   70.820,   70.580,   70.356,   70.151,   69.962,   69.792,   69.638,   69.502, 
      69.383,   69.284,   69.204,   69.141,   69.095,   69.062,   69.015,   69.030,   69.084,   69.178, 
      69.306,   69.463,   69.644,   69.844,   70.065,   70.305,   70.565,   70.846,   71.145,   71.461, 
      71.794,   72.143,   72.508,   72.890,   73.288,   73.702,   74.131,   74.576,   75.037,   75.512, 
      76.000,   76.502,   77.015,   77.540,   78.077,   78.624,   79.752,   80.922,   82.131,   83.378, 
      84.660,   85.975,   87.317,   88.685,   90.081,   91.501,   92.942,   94.403,   95.882,   97.379, 
      98.892,  100.419,  101.957,  103.507,  105.071,  106.647,  108.229,  109.814,  111.399,  112.983, 
      114.566,  116.145,  117.722,  119.294,  120.860,  122.422,  123.981,  125.536,  127.085,  128.631, 
      130.175,  131.716,  133.254,  134.788,  136.318,  137.845,  139.370,  140.894,  142.416,  143.935, 
      145.454,  146.971,  148.487,  150.000,  151.512,  153.023, ],
    [  77.114,   76.628,   76.220,   75.853,   75.500,   75.157,   74.825,   74.504,   74.194,   73.897, 
      73.613,   73.343,   73.087,   72.846,   72.622,   72.416,   72.227,   72.055,   71.900,   71.763, 
      71.645,   71.544,   71.462,   71.398,   71.352,   71.314,   71.264,   71.273,   71.320,   71.406, 
      71.525,   71.673,   71.843,   72.032,   72.239,   72.466,   72.713,   72.979,   73.263,   73.564, 
      73.881,   74.213,   74.562,   74.927,   75.308,   75.704,   76.116,   76.543,   76.986,   77.443, 
      77.914,   78.397,   78.892,   79.399,   79.917,   80.446,   81.538,   82.673,   83.846,   85.057, 
      86.305,   87.585,   88.893,   90.229,   91.592,   92.980,   94.390,   95.821,   97.270,   98.738, 
      100.223,  101.722,  103.233,  104.756,  106.296,  107.847,  109.405,  110.965,  112.527,  114.089, 
      115.649,  117.208,  118.763,  120.314,  121.860,  123.402,  124.942,  126.478,  128.009,  129.537, 
      131.063,  132.588,  134.109,  135.627,  137.141,  138.653,  140.163,  141.673,  143.180,  144.686, 
      146.191,  147.695,  149.198,  150.700,  152.200,  153.699, ],
    [  79.387,   78.900,   78.492,   78.124,   77.770,   77.426,   77.094,   76.772,   76.462,   76.164, 
      75.879,   75.609,   75.352,   75.111,   74.886,   74.679,   74.489,   74.317,   74.161,   74.024, 
      73.904,   73.802,   73.719,   73.655,   73.607,   73.566,   73.512,   73.516,   73.556,   73.635, 
      73.746,   73.885,   74.046,   74.223,   74.419,   74.633,   74.867,   75.120,   75.389,   75.675, 
      75.977,   76.294,   76.627,   76.975,   77.340,   77.719,   78.114,   78.525,   78.950,   79.390, 
      79.844,   80.309,   80.787,   81.276,   81.777,   82.288,   83.345,   84.444,   85.583,   86.760, 
      87.973,   89.220,   90.495,   91.798,   93.129,   94.485,   95.865,   97.266,   98.686,  100.124, 
      101.581,  103.053,  104.537,  106.034,  107.548,  109.074,  110.607,  112.145,  113.683,  115.222, 
      116.760,  118.296,  119.830,  121.360,  122.886,  124.409,  125.929,  127.446,  128.959,  130.468, 
      131.977,  133.484,  134.988,  136.490,  137.988,  139.484,  140.979,  142.474,  143.967,  145.459, 
      146.950,  148.441,  149.931,  151.420,  152.907,  154.395, ],
    [  81.657,   81.170,   80.761,   80.393,   80.038,   79.693,   79.360,   79.038,   78.728,   78.429, 
      78.144,   77.873,   77.616,   77.374,   77.149,   76.941,   76.751,   76.578,   76.422,   76.283, 
      76.162,   76.060,   75.977,   75.912,   75.860,   75.818,   75.761,   75.759,   75.793,   75.866, 
      75.969,   76.100,   76.251,   76.418,   76.603,   76.805,   77.027,   77.266,   77.522,   77.795, 
      78.082,   78.384,   78.702,   79.035,   79.383,   79.747,   80.126,   80.519,   80.928,   81.352, 
      81.789,   82.238,   82.699,   83.171,   83.654,   84.149,   85.171,   86.236,   87.341,   88.484, 
      89.664,   90.878,   92.121,   93.391,   94.691,   96.016,   97.366,   98.736,  100.127,  101.537, 
      102.966,  104.411,  105.867,  107.338,  108.827,  110.328,  111.837,  113.350,  114.865,  116.381, 
      117.897,  119.411,  120.924,  122.433,  123.938,  125.440,  126.940,  128.438,  129.932,  131.423, 
      132.914,  134.403,  135.891,  137.376,  138.858,  140.338,  141.818,  143.297,  144.776,  146.253, 
      147.731,  149.208,  150.685,  152.160,  153.636,  155.111, ],
    [  83.924,   83.437,   83.027,   82.659,   82.304,   81.959,   81.626,   81.303,   80.992,   80.694, 
      80.408,   80.136,   79.878,   79.636,   79.410,   79.202,   79.012,   78.838,   78.680,   78.541, 
      78.420,   78.317,   78.234,   78.167,   78.112,   78.070,   78.009,   78.003,   78.031,   78.097, 
      78.194,   78.317,   78.459,   78.617,   78.791,   78.982,   79.191,   79.419,   79.662,   79.921, 
      80.194,   80.482,   80.785,   81.103,   81.437,   81.785,   82.148,   82.527,   82.920,   83.327, 
      83.748,   84.181,   84.625,   85.081,   85.548,   86.026,   87.015,   88.047,   89.119,   90.229, 
      91.377,   92.559,   93.769,   95.008,   96.276,   97.571,   98.891,  100.232,  101.594,  102.975, 
      104.376,  105.794,  107.224,  108.669,  110.131,  111.608,  113.092,  114.582,  116.073,  117.566, 
      119.059,  120.551,  122.042,  123.530,  125.014,  126.496,  127.976,  129.455,  130.929,  132.402, 
      133.874,  135.346,  136.816,  138.284,  139.750,  141.214,  142.678,  144.142,  145.605,  147.069, 
      148.532,  149.995,  151.459,  152.921,  154.384,  155.846, ],
    [  86.190,   85.703,   85.293,   84.924,   84.569,   84.224,   83.890,   83.567,   83.255,   82.956, 
      82.669,   82.398,   82.139,   81.897,   81.671,   81.462,   81.270,   81.096,   80.938,   80.798, 
      80.677,   80.574,   80.490,   80.419,   80.364,   80.322,   80.257,   80.246,   80.269,   80.330, 
      80.420,   80.535,   80.670,   80.818,   80.982,   81.162,   81.361,   81.576,   81.807,   82.054, 
      82.314,   82.588,   82.877,   83.181,   83.500,   83.834,   84.182,   84.545,   84.923,   85.315, 
      85.720,   86.138,   86.566,   87.006,   87.457,   87.919,   88.876,   89.876,   90.915,   91.993, 
      93.109,   94.259,   95.439,   96.647,   97.884,   99.149,  100.439,  101.751,  103.084,  104.437, 
      105.810,  107.201,  108.604,  110.023,  111.461,  112.912,  114.372,  115.838,  117.306,  118.776, 
      120.246,  121.716,  123.185,  124.651,  126.115,  127.576,  129.036,  130.495,  131.950,  133.404, 
      134.858,  136.311,  137.764,  139.215,  140.664,  142.111,  143.559,  145.008,  146.456,  147.905, 
      149.354,  150.803,  152.253,  153.702,  155.151,  156.601, ],
    [  88.455,   87.967,   87.557,   87.188,   86.831,   86.486,   86.151,   85.828,   85.516,   85.217, 
      84.931,   84.658,   84.399,   84.156,   83.929,   83.720,   83.528,   83.353,   83.195,   83.055, 
      82.934,   82.831,   82.742,   82.672,   82.616,   82.573,   82.506,   82.490,   82.508,   82.563, 
      82.647,   82.756,   82.882,   83.022,   83.176,   83.347,   83.534,   83.739,   83.958,   84.192, 
      84.440,   84.701,   84.977,   85.267,   85.572,   85.891,   86.225,   86.574,   86.937,   87.315, 
      87.705,   88.107,   88.520,   88.945,   89.380,   89.827,   90.752,   91.721,   92.729,   93.775, 
      94.861,   95.980,   97.128,   98.306,   99.513,  100.749,  102.009,  103.292,  104.597,  105.923, 
      107.268,  108.632,  110.009,  111.402,  112.814,  114.240,  115.676,  117.118,  118.562,  120.009, 
      121.457,  122.904,  124.351,  125.796,  127.238,  128.679,  130.119,  131.557,  132.993,  134.428, 
      135.863,  137.299,  138.734,  140.167,  141.599,  143.030,  144.462,  145.895,  147.328,  148.761, 
      150.196,  151.631,  153.066,  154.502,  155.938,  157.375, ],
    [  90.717,   90.229,   89.819,   89.449,   89.093,   88.747,   88.413,   88.089,   87.777,   87.477, 
      87.190,   86.917,   86.658,   86.414,   86.187,   85.977,   85.785,   85.610,   85.452,   85.313, 
      85.190,   85.083,   84.995,   84.924,   84.867,   84.822,   84.754,   84.733,   84.746,   84.797, 
      84.875,   84.978,   85.097,   85.228,   85.373,   85.534,   85.712,   85.905,   86.114,   86.336, 
      86.572,   86.820,   87.083,   87.360,   87.652,   87.957,   88.278,   88.612,   88.961,   89.324, 
      89.700,   90.088,   90.487,   90.896,   91.317,   91.748,   92.643,   93.581,   94.559,   95.575, 
      96.629,   97.719,   98.837,   99.985,  101.163,  102.369,  103.600,  104.855,  106.132,  107.430, 
      108.748,  110.085,  111.436,  112.803,  114.190,  115.592,  117.003,  118.421,  119.842,  121.266, 
      122.691,  124.116,  125.541,  126.964,  128.385,  129.804,  131.224,  132.643,  134.059,  135.474, 
      136.891,  138.308,  139.725,  141.141,  142.555,  143.970,  145.386,  146.802,  148.220,  149.638, 
      151.057,  152.478,  153.899,  155.321,  156.744,  158.167, ],
    [  92.979,   92.491,   92.080,   91.710,   91.354,   91.008,   90.672,   90.348,   90.036,   89.735, 
      89.448,   89.175,   88.915,   88.671,   88.444,   88.235,   88.042,   87.868,   87.709,   87.568, 
      87.442,   87.335,   87.247,   87.175,   87.119,   87.071,   87.001,   86.977,   86.986,   87.031, 
      87.104,   87.201,   87.313,   87.436,   87.573,   87.724,   87.892,   88.076,   88.273,   88.484, 
      88.708,   88.945,   89.196,   89.460,   89.739,   90.031,   90.338,   90.659,   90.995,   91.344, 
      91.706,   92.079,   92.464,   92.859,   93.265,   93.682,   94.548,   95.456,   96.404,   97.390, 
      98.415,   99.475,  100.564,  101.682,  102.831,  104.009,  105.212,  106.438,  107.688,  108.958, 
      110.250,  111.560,  112.885,  114.226,  115.588,  116.965,  118.353,  119.747,  121.144,  122.545, 
      123.947,  125.350,  126.752,  128.154,  129.553,  130.952,  132.351,  133.749,  135.146,  136.542, 
      137.940,  139.338,  140.737,  142.135,  143.532,  144.930,  146.329,  147.730,  149.131,  150.534, 
      151.939,  153.344,  154.751,  156.159,  157.568,  158.979, ],
    [  95.240,   94.751,   94.340,   93.969,   93.612,   93.266,   92.931,   92.606,   92.293,   91.993, 
      91.705,   91.432,   91.172,   90.929,   90.701,   90.492,   90.299,   90.124,   89.963,   89.820, 
      89.695,   89.588,   89.499,   89.427,   89.368,   89.320,   89.248,   89.221,   89.225,   89.266, 
      89.334,   89.425,   89.531,   89.646,   89.775,   89.917,   90.076,   90.250,   90.437,   90.637, 
      90.850,   91.075,   91.314,   91.566,   91.832,   92.112,   92.406,   92.714,   93.037,   93.373, 
      93.721,   94.081,   94.452,   94.833,   95.225,   95.627,   96.465,   97.344,   98.263,   99.220, 
      100.216,  101.247,  102.307,  103.397,  104.517,  105.667,  106.842,  108.041,  109.263,  110.506, 
      111.771,  113.056,  114.354,  115.671,  117.007,  118.360,  119.724,  121.094,  122.468,  123.845, 
      125.224,  126.605,  127.985,  129.365,  130.743,  132.121,  133.499,  134.877,  136.254,  137.631, 
      139.009,  140.389,  141.770,  143.150,  144.530,  145.910,  147.292,  148.677,  150.062,  151.449, 
      152.839,  154.230,  155.622,  157.016,  158.411,  159.808, ],
    [  97.498,   97.009,   96.598,   96.228,   95.870,   95.524,   95.188,   94.864,   94.551,   94.250, 
      93.962,   93.689,   93.430,   93.186,   92.958,   92.748,   92.553,   92.376,   92.215,   92.072, 
      91.947,   91.839,   91.750,   91.678,   91.617,   91.569,   91.495,   91.464,   91.464,   91.502, 
      91.565,   91.650,   91.750,   91.858,   91.979,   92.113,   92.262,   92.427,   92.604,   92.794, 
      92.996,   93.210,   93.437,   93.678,   93.932,   94.200,   94.482,   94.777,   95.087,   95.410, 
      95.745,   96.092,   96.449,   96.817,   97.196,   97.584,   98.394,   99.245,  100.136,  101.065, 
      102.033,  103.035,  104.067,  105.128,  106.221,  107.343,  108.490,  109.662,  110.857,  112.074, 
      113.313,  114.571,  115.844,  117.135,  118.447,  119.776,  121.115,  122.461,  123.812,  125.166, 
      126.523,  127.881,  129.239,  130.597,  131.954,  133.311,  134.668,  136.026,  137.383,  138.739, 
      140.099,  141.460,  142.822,  144.184,  145.546,  146.909,  148.275,  149.643,  151.012,  152.384, 
      153.757,  155.133,  156.511,  157.891,  159.272,  160.655, ],
    [  99.756,   99.267,   98.855,   98.485,   98.127,   97.781,   97.445,   97.121,   96.808,   96.507, 
      96.220,   95.946,   95.686,   95.441,   95.212,   95.000,   94.805,   94.628,   94.467,   94.324, 
      94.198,   94.091,   94.001,   93.927,   93.866,   93.817,   93.741,   93.707,   93.704,   93.738, 
      93.796,   93.877,   93.970,   94.071,   94.184,   94.310,   94.451,   94.606,   94.774,   94.955, 
      95.146,   95.349,   95.566,   95.795,   96.038,   96.294,   96.563,   96.847,   97.144,   97.455, 
      97.778,   98.112,   98.456,   98.811,   99.176,   99.551,  100.334,  101.158,  102.021,  102.922, 
      103.863,  104.838,  105.842,  106.876,  107.941,  109.035,  110.156,  111.301,  112.469,  113.660, 
      114.873,  116.105,  117.353,  118.619,  119.907,  121.211,  122.527,  123.849,  125.177,  126.508, 
      127.842,  129.178,  130.514,  131.850,  133.185,  134.520,  135.857,  137.195,  138.531,  139.868, 
      141.208,  142.551,  143.894,  145.238,  146.582,  147.928,  149.276,  150.628,  151.981,  153.336, 
      154.695,  156.055,  157.418,  158.783,  160.150,  161.519, ],
    [ 102.013,  101.524,  101.113,  100.742,  100.385,  100.038,   99.702,   99.378,   99.065,   98.764, 
      98.475,   98.200,   97.939,   97.693,   97.464,   97.252,   97.057,   96.880,   96.719,   96.576, 
      96.450,   96.342,   96.251,   96.176,   96.115,   96.066,   95.988,   95.950,   95.944,   95.974, 
      96.028,   96.104,   96.191,   96.286,   96.392,   96.509,   96.642,   96.789,   96.948,   97.118, 
      97.300,   97.493,   97.698,   97.917,   98.148,   98.393,   98.651,   98.923,   99.208,   99.507, 
      99.818,  100.139,  100.471,  100.813,  101.165,  101.527,  102.284,  103.082,  103.918,  104.793, 
      105.706,  106.654,  107.631,  108.638,  109.676,  110.744,  111.838,  112.956,  114.099,  115.263, 
      116.450,  117.658,  118.881,  120.122,  121.385,  122.665,  123.957,  125.257,  126.561,  127.869, 
      129.180,  130.494,  131.808,  133.122,  134.436,  135.750,  137.066,  138.383,  139.699,  141.016, 
      142.337,  143.660,  144.985,  146.311,  147.637,  148.965,  150.296,  151.631,  152.968,  154.307, 
      155.649,  156.995,  158.343,  159.693,  161.045,  162.401, ],
    [ 104.270,  103.781,  103.370,  102.999,  102.642,  102.295,  101.959,  101.633,  101.318,  101.016, 
      100.727,  100.452,  100.191,   99.945,   99.716,   99.504,   99.309,   99.132,   98.971,   98.827, 
      98.701,   98.591,   98.500,   98.424,   98.363,   98.314,   98.235,   98.193,   98.183,   98.210, 
      98.260,   98.332,   98.414,   98.502,   98.600,   98.710,   98.835,   98.974,   99.124,   99.285, 
      99.457,   99.640,   99.835,  100.043,  100.264,  100.497,  100.744,  101.005,  101.279,  101.566, 
      101.865,  102.174,  102.494,  102.824,  103.163,  103.513,  104.244,  105.016,  105.826,  106.674, 
      107.562,  108.483,  109.434,  110.414,  111.426,  112.467,  113.535,  114.628,  115.744,  116.883, 
      118.045,  119.228,  120.426,  121.642,  122.881,  124.138,  125.406,  126.682,  127.964,  129.249, 
      130.538,  131.829,  133.121,  134.413,  135.705,  136.998,  138.293,  139.589,  140.885,  142.183, 
      143.484,  144.788,  146.094,  147.402,  148.710,  150.020,  151.334,  152.652,  153.972,  155.295, 
      156.622,  157.952,  159.285,  160.620,  161.958,  163.299, ],
    [ 106.527,  106.038,  105.626,  105.254,  104.896,  104.548,  104.211,  103.885,  103.571,  103.268, 
      102.979,  102.704,  102.443,  102.197,  101.968,  101.756,  101.561,  101.383,  101.222,  101.078, 
      100.950,  100.840,  100.748,  100.673,  100.612,  100.563,  100.481,  100.435,  100.423,  100.447, 
      100.493,  100.560,  100.637,  100.719,  100.810,  100.913,  101.030,  101.161,  101.302,  101.455, 
      101.618,  101.790,  101.976,  102.173,  102.383,  102.606,  102.843,  103.092,  103.355,  103.631, 
      103.918,  104.216,  104.524,  104.842,  105.170,  105.507,  106.213,  106.960,  107.744,  108.567, 
      109.429,  110.325,  111.249,  112.204,  113.190,  114.205,  115.248,  116.315,  117.406,  118.519, 
      119.657,  120.814,  121.987,  123.180,  124.395,  125.628,  126.873,  128.126,  129.385,  130.647, 
      131.914,  133.182,  134.452,  135.723,  136.993,  138.265,  139.539,  140.815,  142.090,  143.368, 
      144.649,  145.934,  147.222,  148.511,  149.801,  151.093,  152.390,  153.690,  154.994,  156.301, 
      157.612,  158.926,  160.244,  161.564,  162.887,  164.214, ],
    [ 108.781,  108.291,  107.878,  107.507,  107.148,  106.800,  106.463,  106.137,  105.823,  105.521, 
      105.231,  104.956,  104.695,  104.449,  104.220,  104.007,  103.812,  103.634,  103.472,  103.327, 
      103.199,  103.089,  102.997,  102.921,  102.860,  102.811,  102.726,  102.678,  102.663,  102.684, 
      102.727,  102.789,  102.861,  102.937,  103.022,  103.117,  103.227,  103.350,  103.483,  103.627, 
      103.781,  103.944,  104.120,  104.307,  104.507,  104.720,  104.946,  105.184,  105.437,  105.702, 
      105.978,  106.265,  106.561,  106.867,  107.183,  107.508,  108.190,  108.913,  109.673,  110.471, 
      111.307,  112.178,  113.077,  114.006,  114.967,  115.957,  116.974,  118.016,  119.082,  120.171, 
      121.283,  122.417,  123.566,  124.734,  125.926,  127.136,  128.357,  129.587,  130.823,  132.063, 
      133.307,  134.553,  135.801,  137.050,  138.299,  139.550,  140.803,  142.058,  143.313,  144.571, 
      145.833,  147.098,  148.367,  149.637,  150.909,  152.184,  153.463,  154.746,  156.033,  157.323, 
      158.618,  159.917,  161.219,  162.524,  163.833,  165.145, ],
    [ 111.033,  110.543,  110.131,  109.759,  109.400,  109.052,  108.715,  108.389,  108.075,  107.772, 
      107.483,  107.208,  106.947,  106.700,  106.471,  106.258,  106.062,  105.883,  105.721,  105.576, 
      105.448,  105.338,  105.245,  105.170,  105.109,  105.058,  104.971,  104.921,  104.903,  104.921, 
      104.960,  105.019,  105.086,  105.155,  105.234,  105.323,  105.425,  105.540,  105.666,  105.801, 
      105.946,  106.100,  106.267,  106.445,  106.635,  106.838,  107.053,  107.282,  107.523,  107.778, 
      108.043,  108.319,  108.605,  108.899,  109.204,  109.517,  110.176,  110.874,  111.610,  112.384, 
      113.196,  114.042,  114.916,  115.820,  116.756,  117.722,  118.714,  119.731,  120.772,  121.837, 
      122.925,  124.034,  125.159,  126.304,  127.472,  128.659,  129.858,  131.065,  132.278,  133.496, 
      134.717,  135.942,  137.168,  138.395,  139.623,  140.852,  142.084,  143.318,  144.553,  145.791, 
      147.033,  148.279,  149.529,  150.781,  152.034,  153.291,  154.552,  155.819,  157.088,  158.362, 
      159.641,  160.924,  162.210,  163.501,  164.794,  166.092, ],
    [ 113.285,  112.796,  112.383,  112.011,  111.652,  111.304,  110.967,  110.641,  110.326,  110.024, 
      109.735,  109.459,  109.198,  108.951,  108.721,  108.507,  108.311,  108.132,  107.969,  107.824, 
      107.696,  107.586,  107.494,  107.418,  107.357,  107.305,  107.217,  107.163,  107.143,  107.159, 
      107.194,  107.249,  107.311,  107.375,  107.447,  107.529,  107.625,  107.733,  107.850,  107.978, 
      108.114,  108.259,  108.417,  108.585,  108.766,  108.959,  109.165,  109.383,  109.615,  109.859, 
      110.114,  110.379,  110.654,  110.938,  111.231,  111.533,  112.169,  112.844,  113.556,  114.306, 
      115.094,  115.916,  116.766,  117.646,  118.557,  119.499,  120.467,  121.460,  122.477,  123.517, 
      124.581,  125.667,  126.768,  127.889,  129.034,  130.198,  131.374,  132.559,  133.750,  134.945, 
      136.144,  137.346,  138.551,  139.756,  140.963,  142.170,  143.382,  144.595,  145.810,  147.028, 
      148.250,  149.477,  150.708,  151.941,  153.176,  154.415,  155.658,  156.907,  158.160,  159.418, 
      160.680,  161.947,  163.218,  164.493,  165.772,  167.055, ],
    [ 115.537,  115.047,  114.634,  114.263,  113.904,  113.556,  113.218,  112.892,  112.578,  112.275, 
      111.985,  111.709,  111.447,  111.200,  110.969,  110.756,  110.560,  110.381,  110.218,  110.073, 
      109.945,  109.835,  109.742,  109.667,  109.604,  109.552,  109.461,  109.405,  109.383,  109.396, 
      109.428,  109.480,  109.537,  109.595,  109.661,  109.737,  109.826,  109.926,  110.037,  110.156, 
      110.284,  110.421,  110.569,  110.729,  110.900,  111.084,  111.280,  111.488,  111.710,  111.945, 
      112.190,  112.445,  112.709,  112.982,  113.264,  113.555,  114.169,  114.821,  115.511,  116.237, 
      117.002,  117.800,  118.627,  119.482,  120.370,  121.287,  122.231,  123.200,  124.194,  125.210, 
      126.251,  127.313,  128.391,  129.489,  130.611,  131.752,  132.906,  134.068,  135.237,  136.410, 
      137.587,  138.767,  139.950,  141.134,  142.319,  143.505,  144.696,  145.889,  147.083,  148.281, 
      149.484,  150.691,  151.903,  153.117,  154.334,  155.554,  156.781,  158.012,  159.248,  160.489, 
      161.735,  162.985,  164.241,  165.500,  166.764,  168.032, ],
    [ 117.789,  117.299,  116.886,  116.514,  116.155,  115.807,  115.469,  115.143,  114.827,  114.524, 
      114.234,  113.958,  113.696,  113.449,  113.218,  113.005,  112.808,  112.629,  112.466,  112.321, 
      112.193,  112.083,  111.991,  111.914,  111.851,  111.799,  111.705,  111.647,  111.622,  111.634, 
      111.663,  111.711,  111.763,  111.816,  111.876,  111.946,  112.028,  112.122,  112.224,  112.336, 
      112.456,  112.584,  112.724,  112.875,  113.037,  113.212,  113.398,  113.598,  113.810,  114.035, 
      114.270,  114.515,  114.769,  115.031,  115.303,  115.583,  116.175,  116.806,  117.473,  118.177, 
      118.918,  119.694,  120.497,  121.329,  122.193,  123.087,  124.008,  124.953,  125.923,  126.917, 
      127.934,  128.973,  130.028,  131.103,  132.202,  133.321,  134.453,  135.593,  136.739,  137.890, 
      139.045,  140.204,  141.365,  142.527,  143.691,  144.856,  146.026,  147.198,  148.372,  149.550, 
      150.733,  151.921,  153.114,  154.309,  155.507,  156.710,  157.918,  159.132,  160.351,  161.575, 
      162.805,  164.039,  165.279,  166.523,  167.772,  169.025, ],
    [ 120.040,  119.550,  119.137,  118.765,  118.405,  118.056,  117.719,  117.392,  117.076,  116.773, 
      116.483,  116.207,  115.944,  115.697,  115.467,  115.253,  115.057,  114.878,  114.715,  114.570, 
      114.442,  114.332,  114.239,  114.161,  114.098,  114.046,  113.949,  113.890,  113.862,  113.871, 
      113.898,  113.942,  113.990,  114.038,  114.092,  114.155,  114.231,  114.318,  114.414,  114.518, 
      114.630,  114.750,  114.881,  115.023,  115.177,  115.343,  115.520,  115.710,  115.914,  116.129, 
      116.354,  116.590,  116.833,  117.086,  117.347,  117.617,  118.188,  118.797,  119.442,  120.124, 
      120.843,  121.596,  122.376,  123.186,  124.027,  124.898,  125.795,  126.718,  127.664,  128.635, 
      129.629,  130.645,  131.677,  132.730,  133.807,  134.904,  136.013,  137.131,  138.256,  139.385, 
      140.518,  141.655,  142.795,  143.936,  145.078,  146.222,  147.371,  148.523,  149.677,  150.834, 
      151.998,  153.167,  154.340,  155.516,  156.696,  157.881,  159.071,  160.268,  161.470,  162.677, 
      163.890,  165.108,  166.332,  167.561,  168.794,  170.033, ],
    [ 122.290,  121.800,  121.386,  121.014,  120.654,  120.305,  119.967,  119.641,  119.325,  119.022, 
      118.732,  118.455,  118.193,  117.946,  117.715,  117.502,  117.305,  117.126,  116.963,  116.818, 
      116.690,  116.579,  116.486,  116.408,  116.345,  116.291,  116.193,  116.132,  116.102,  116.109, 
      116.132,  116.174,  116.218,  116.260,  116.308,  116.365,  116.435,  116.515,  116.604,  116.701, 
      116.806,  116.917,  117.040,  117.174,  117.319,  117.476,  117.645,  117.826,  118.020,  118.227, 
      118.443,  118.669,  118.903,  119.145,  119.396,  119.656,  120.206,  120.794,  121.418,  122.078, 
      122.776,  123.507,  124.265,  125.051,  125.870,  126.718,  127.593,  128.493,  129.417,  130.365, 
      131.337,  132.330,  133.340,  134.370,  135.425,  136.500,  137.588,  138.684,  139.786,  140.894, 
      142.006,  143.121,  144.239,  145.359,  146.480,  147.603,  148.731,  149.863,  150.996,  152.134, 
      153.277,  154.427,  155.581,  156.739,  157.900,  159.066,  160.239,  161.418,  162.603,  163.793, 
      164.989,  166.192,  167.400,  168.613,  169.831,  171.054, ],
    [ 124.539,  124.048,  123.635,  123.262,  122.903,  122.554,  122.216,  121.889,  121.574,  121.270, 
      120.980,  120.704,  120.441,  120.194,  119.964,  119.750,  119.554,  119.375,  119.212,  119.067, 
      118.938,  118.826,  118.732,  118.655,  118.591,  118.537,  118.437,  118.374,  118.342,  118.347, 
      118.367,  118.406,  118.445,  118.482,  118.525,  118.576,  118.640,  118.714,  118.796,  118.886, 
      118.982,  119.087,  119.202,  119.327,  119.464,  119.612,  119.773,  119.945,  120.131,  120.328, 
      120.535,  120.751,  120.976,  121.209,  121.450,  121.700,  122.230,  122.797,  123.400,  124.039, 
      124.716,  125.425,  126.161,  126.926,  127.722,  128.548,  129.401,  130.279,  131.180,  132.106, 
      133.055,  134.027,  135.014,  136.022,  137.056,  138.109,  139.175,  140.250,  141.331,  142.417, 
      143.507,  144.601,  145.698,  146.796,  147.896,  148.999,  150.106,  151.217,  152.330,  153.447, 
      154.571,  155.702,  156.837,  157.976,  159.119,  160.266,  161.421,  162.583,  163.750,  164.923, 
      166.103,  167.290,  168.482,  169.679,  170.882,  172.090, ],
    [ 126.788,  126.297,  125.883,  125.511,  125.151,  124.802,  124.464,  124.138,  123.822,  123.519, 
      123.229,  122.952,  122.690,  122.443,  122.212,  121.999,  121.802,  121.623,  121.460,  121.314, 
      121.185,  121.073,  120.979,  120.902,  120.836,  120.782,  120.680,  120.616,  120.582,  120.585, 
      120.603,  120.638,  120.673,  120.705,  120.742,  120.788,  120.845,  120.913,  120.988,  121.071, 
      121.161,  121.257,  121.364,  121.482,  121.611,  121.751,  121.903,  122.067,  122.244,  122.433, 
      122.631,  122.838,  123.053,  123.277,  123.509,  123.749,  124.259,  124.806,  125.389,  126.007, 
      126.663,  127.351,  128.066,  128.809,  129.583,  130.388,  131.219,  132.074,  132.954,  133.857, 
      134.785,  135.735,  136.700,  137.687,  138.698,  139.730,  140.775,  141.828,  142.888,  143.953, 
      145.022,  146.094,  147.170,  148.247,  149.326,  150.408,  151.495,  152.585,  153.678,  154.775, 
      155.880,  156.991,  158.106,  159.227,  160.351,  161.481,  162.618,  163.762,  164.912,  166.068, 
      167.231,  168.401,  169.577,  170.759,  171.946,  173.140, ],
    [ 129.036,  128.546,  128.132,  127.759,  127.400,  127.051,  126.713,  126.386,  126.071,  125.767, 
      125.477,  125.201,  124.938,  124.691,  124.461,  124.247,  124.051,  123.871,  123.707,  123.561, 
      123.432,  123.320,  123.226,  123.147,  123.082,  123.027,  122.923,  122.858,  122.822,  122.823, 
      122.838,  122.870,  122.901,  122.928,  122.960,  122.999,  123.052,  123.113,  123.182,  123.258, 
      123.340,  123.429,  123.529,  123.638,  123.759,  123.891,  124.035,  124.191,  124.360,  124.540, 
      124.730,  124.928,  125.134,  125.349,  125.571,  125.802,  126.293,  126.820,  127.383,  127.981, 
      128.616,  129.284,  129.977,  130.699,  131.453,  132.236,  133.045,  133.879,  134.737,  135.619, 
      136.525,  137.453,  138.397,  139.362,  140.353,  141.364,  142.387,  143.419,  144.458,  145.501, 
      146.549,  147.601,  148.655,  149.712,  150.770,  151.831,  152.897,  153.967,  155.040,  156.117, 
      157.202,  158.293,  159.390,  160.492,  161.597,  162.709,  163.828,  164.954,  166.087,  167.226, 
      168.373,  169.527,  170.687,  171.853,  173.024,  174.203, ],
    [ 131.285,  130.794,  130.380,  130.008,  129.648,  129.299,  128.961,  128.635,  128.319,  128.016, 
      127.726,  127.449,  127.187,  126.940,  126.709,  126.495,  126.298,  126.118,  125.954,  125.808, 
      125.679,  125.567,  125.472,  125.393,  125.327,  125.271,  125.166,  125.099,  125.061,  125.061, 
      125.073,  125.103,  125.130,  125.152,  125.178,  125.212,  125.258,  125.314,  125.377,  125.446, 
      125.521,  125.602,  125.694,  125.797,  125.910,  126.034,  126.170,  126.318,  126.478,  126.650, 
      126.832,  127.021,  127.219,  127.424,  127.638,  127.859,  128.331,  128.840,  129.382,  129.961, 
      130.576,  131.223,  131.896,  132.597,  133.330,  134.092,  134.880,  135.693,  136.530,  137.391, 
      138.276,  139.182,  140.105,  141.049,  142.018,  143.008,  144.011,  145.022,  146.040,  147.062, 
      148.089,  149.120,  150.153,  151.189,  152.226,  153.267,  154.312,  155.362,  156.415,  157.472, 
      158.537,  159.609,  160.687,  161.770,  162.857,  163.950,  165.051,  166.160,  167.276,  168.398, 
      169.528,  170.665,  171.809,  172.959,  174.116,  175.279, ],
    [ 133.533,  133.043,  132.629,  132.256,  131.897,  131.548,  131.210,  130.883,  130.568,  130.264, 
      129.974,  129.698,  129.435,  129.187,  128.956,  128.742,  128.544,  128.365,  128.201,  128.055, 
      127.925,  127.813,  127.717,  127.638,  127.571,  127.514,  127.408,  127.339,  127.301,  127.299, 
      127.309,  127.336,  127.358,  127.375,  127.396,  127.425,  127.466,  127.516,  127.572,  127.634, 
      127.703,  127.777,  127.862,  127.956,  128.062,  128.178,  128.307,  128.446,  128.599,  128.763, 
      128.936,  129.118,  129.307,  129.503,  129.708,  129.920,  130.373,  130.863,  131.387,  131.946, 
      132.542,  133.169,  133.822,  134.503,  135.215,  135.956,  136.724,  137.516,  138.332,  139.172, 
      140.036,  140.922,  141.823,  142.746,  143.695,  144.664,  145.646,  146.637,  147.633,  148.635, 
      149.641,  150.651,  151.664,  152.679,  153.695,  154.715,  155.740,  156.770,  157.802,  158.840, 
      159.885,  160.938,  161.997,  163.061,  164.130,  165.205,  166.288,  167.379,  168.477,  169.583, 
      170.696,  171.817,  172.945,  174.079,  175.220,  176.368, ],
    [ 135.782,  135.291,  134.877,  134.505,  134.145,  133.796,  133.458,  133.132,  132.816,  132.512, 
      132.221,  131.945,  131.682,  131.434,  131.203,  130.988,  130.791,  130.612,  130.448,  130.301, 
      130.171,  130.058,  129.963,  129.883,  129.815,  129.758,  129.650,  129.580,  129.541,  129.537, 
      129.544,  129.569,  129.587,  129.599,  129.615,  129.638,  129.674,  129.718,  129.768,  129.824, 
      129.885,  129.952,  130.030,  130.117,  130.215,  130.324,  130.445,  130.577,  130.722,  130.879, 
      131.044,  131.217,  131.397,  131.585,  131.781,  131.984,  132.420,  132.891,  133.396,  133.936, 
      134.513,  135.121,  135.754,  136.415,  137.107,  137.828,  138.575,  139.347,  140.142,  140.961, 
      141.805,  142.670,  143.551,  144.453,  145.381,  146.330,  147.292,  148.262,  149.238,  150.219, 
      151.205,  152.194,  153.186,  154.180,  155.176,  156.175,  157.180,  158.190,  159.202,  160.220, 
      161.246,  162.280,  163.320,  164.365,  165.415,  166.472,  167.537,  168.611,  169.692,  170.780, 
      171.877,  172.981,  174.093,  175.211,  176.337,  177.469, ],
    [ 138.030,  137.540,  137.126,  136.753,  136.393,  136.044,  135.706,  135.379,  135.063,  134.759, 
      134.468,  134.192,  133.928,  133.681,  133.450,  133.235,  133.038,  132.858,  132.693,  132.546, 
      132.416,  132.303,  132.208,  132.127,  132.059,  132.002,  131.893,  131.820,  131.781,  131.775, 
      131.780,  131.802,  131.816,  131.823,  131.833,  131.851,  131.882,  131.920,  131.964,  132.014, 
      132.069,  132.128,  132.199,  132.279,  132.370,  132.472,  132.585,  132.710,  132.847,  132.996, 
      133.153,  133.319,  133.491,  133.670,  133.857,  134.052,  134.470,  134.923,  135.410,  135.931, 
      136.489,  137.079,  137.692,  138.333,  139.006,  139.707,  140.434,  141.186,  141.961,  142.760, 
      143.583,  144.428,  145.288,  146.170,  147.078,  148.007,  148.948,  149.898,  150.854,  151.815, 
      152.780,  153.748,  154.720,  155.694,  156.669,  157.648,  158.632,  159.622,  160.615,  161.613, 
      162.619,  163.634,  164.655,  165.681,  166.713,  167.751,  168.799,  169.855,  170.919,  171.990, 
      173.070,  174.158,  175.253,  176.356,  177.466,  178.583, ],
    [ 140.278,  139.787,  139.373,  139.000,  138.640,  138.291,  137.953,  137.626,  137.310,  137.006, 
      136.715,  136.438,  136.175,  135.928,  135.696,  135.482,  135.284,  135.103,  134.939,  134.792, 
      134.662,  134.548,  134.451,  134.371,  134.303,  134.245,  134.135,  134.060,  134.020,  134.013, 
      134.016,  134.035,  134.045,  134.047,  134.052,  134.065,  134.090,  134.124,  134.161,  134.205, 
      134.253,  134.306,  134.369,  134.442,  134.526,  134.621,  134.727,  134.844,  134.974,  135.116, 
      135.265,  135.423,  135.587,  135.758,  135.937,  136.123,  136.523,  136.960,  137.428,  137.931, 
      138.471,  139.041,  139.636,  140.258,  140.911,  141.593,  142.300,  143.032,  143.787,  144.566, 
      145.369,  146.194,  147.034,  147.896,  148.784,  149.693,  150.615,  151.545,  152.480,  153.421, 
      154.365,  155.314,  156.265,  157.218,  158.173,  159.132,  160.096,  161.066,  162.038,  163.017, 
      164.004,  164.999,  166.001,  167.009,  168.022,  169.043,  170.072,  171.111,  172.157,  173.212, 
      174.275,  175.347,  176.426,  177.513,  178.607,  179.709, ],
    [ 142.525,  142.034,  141.620,  141.247,  140.887,  140.538,  140.200,  139.873,  139.557,  139.253, 
      138.962,  138.685,  138.422,  138.174,  137.942,  137.727,  137.530,  137.349,  137.184,  137.037, 
      136.906,  136.792,  136.695,  136.614,  136.546,  136.488,  136.377,  136.300,  136.260,  136.251, 
      136.251,  136.269,  136.273,  136.271,  136.271,  136.279,  136.299,  136.327,  136.359,  136.396, 
      136.437,  136.483,  136.540,  136.606,  136.683,  136.771,  136.870,  136.980,  137.103,  137.237, 
      137.379,  137.529,  137.685,  137.849,  138.019,  138.197,  138.580,  138.999,  139.450,  139.936, 
      140.457,  141.010,  141.586,  142.188,  142.822,  143.485,  144.173,  144.885,  145.621,  146.380, 
      147.164,  147.969,  148.789,  149.631,  150.499,  151.389,  152.291,  153.201,  154.116,  155.037, 
      155.961,  156.889,  157.820,  158.753,  159.688,  160.627,  161.571,  162.521,  163.474,  164.432, 
      165.400,  166.376,  167.360,  168.349,  169.344,  170.346,  171.358,  172.379,  173.408,  174.445, 
      175.492,  176.547,  177.610,  178.681,  179.760,  180.847, ],
    [ 144.772,  144.281,  143.867,  143.494,  143.134,  142.785,  142.447,  142.119,  141.803,  141.499, 
      141.208,  140.931,  140.668,  140.419,  140.187,  139.973,  139.775,  139.594,  139.429,  139.281, 
      139.150,  139.036,  138.939,  138.858,  138.789,  138.731,  138.619,  138.540,  138.499,  138.489, 
      138.487,  138.502,  138.501,  138.495,  138.489,  138.492,  138.508,  138.531,  138.556,  138.587, 
      138.622,  138.662,  138.712,  138.772,  138.842,  138.922,  139.014,  139.118,  139.234,  139.361, 
      139.496,  139.638,  139.786,  139.941,  140.104,  140.273,  140.640,  141.042,  141.476,  141.944, 
      142.448,  142.982,  143.540,  144.124,  144.739,  145.383,  146.052,  146.745,  147.462,  148.202, 
      148.966,  149.751,  150.552,  151.374,  152.223,  153.093,  153.976,  154.866,  155.762,  156.663, 
      157.567,  158.475,  159.386,  160.299,  161.214,  162.132,  163.057,  163.986,  164.920,  165.859, 
      166.808,  167.765,  168.729,  169.700,  170.676,  171.661,  172.655,  173.658,  174.670,  175.690, 
      176.720,  177.759,  178.806,  179.861,  180.924,  181.996, ],
    [ 147.019,  146.528,  146.114,  145.741,  145.381,  145.032,  144.693,  144.366,  144.049,  143.745, 
      143.454,  143.177,  142.913,  142.665,  142.433,  142.218,  142.020,  141.839,  141.673,  141.525, 
      141.394,  141.280,  141.183,  141.101,  141.032,  140.973,  140.861,  140.780,  140.739,  140.728, 
      140.723,  140.736,  140.729,  140.718,  140.708,  140.706,  140.717,  140.734,  140.755,  140.779, 
      140.808,  140.841,  140.884,  140.937,  141.001,  141.075,  141.160,  141.257,  141.366,  141.486, 
      141.613,  141.748,  141.889,  142.037,  142.191,  142.352,  142.703,  143.089,  143.506,  143.956, 
      144.443,  144.960,  145.499,  146.065,  146.662,  147.287,  147.938,  148.612,  149.310,  150.030, 
      150.775,  151.542,  152.323,  153.126,  153.956,  154.807,  155.670,  156.541,  157.418,  158.298, 
      159.183,  160.071,  160.962,  161.855,  162.750,  163.648,  164.553,  165.463,  166.376,  167.296, 
      168.226,  169.164,  170.110,  171.062,  172.020,  172.986,  173.962,  174.948,  175.943,  176.946, 
      177.960,  178.982,  180.013,  181.053,  182.100,  183.156, ],
    [ 149.266,  148.775,  148.361,  147.987,  147.627,  147.277,  146.939,  146.611,  146.295,  145.990, 
      145.699,  145.422,  145.158,  144.910,  144.678,  144.462,  144.264,  144.082,  143.917,  143.769, 
      143.638,  143.524,  143.426,  143.344,  143.275,  143.215,  143.102,  143.021,  142.979,  142.966, 
      142.959,  142.970,  142.957,  142.941,  142.927,  142.920,  142.927,  142.939,  142.953,  142.972, 
      142.994,  143.020,  143.057,  143.104,  143.161,  143.228,  143.307,  143.397,  143.499,  143.612, 
      143.733,  143.861,  143.994,  144.134,  144.280,  144.434,  144.769,  145.138,  145.539,  145.972, 
      146.442,  146.942,  147.463,  148.011,  148.590,  149.197,  149.829,  150.485,  151.164,  151.866, 
      152.592,  153.339,  154.102,  154.886,  155.697,  156.529,  157.373,  158.225,  159.082,  159.943, 
      160.808,  161.677,  162.548,  163.421,  164.295,  165.174,  166.059,  166.949,  167.843,  168.744, 
      169.654,  170.574,  171.501,  172.434,  173.374,  174.322,  175.281,  176.249,  177.227,  178.213, 
      179.210,  180.216,  181.231,  182.255,  183.287,  184.327, ],
    [ 151.511,  151.020,  150.606,  150.233,  149.872,  149.523,  149.184,  148.856,  148.540,  148.236, 
      147.945,  147.667,  147.403,  147.154,  146.922,  146.706,  146.508,  146.326,  146.161,  146.013, 
      145.881,  145.766,  145.669,  145.587,  145.517,  145.457,  145.342,  145.261,  145.218,  145.204, 
      145.194,  145.203,  145.185,  145.165,  145.145,  145.134,  145.136,  145.143,  145.151,  145.164, 
      145.180,  145.200,  145.231,  145.271,  145.321,  145.382,  145.454,  145.538,  145.634,  145.740, 
      145.854,  145.975,  146.101,  146.233,  146.372,  146.518,  146.837,  147.190,  147.574,  147.992, 
      148.445,  148.927,  149.432,  149.962,  150.523,  151.112,  151.726,  152.364,  153.024,  153.708, 
      154.415,  155.144,  155.888,  156.653,  157.445,  158.258,  159.084,  159.917,  160.755,  161.597, 
      162.442,  163.291,  164.142,  164.996,  165.851,  166.709,  167.575,  168.446,  169.320,  170.202, 
      171.093,  171.993,  172.902,  173.817,  174.739,  175.669,  176.610,  177.561,  178.521,  179.491, 
      180.471,  181.461,  182.460,  183.468,  184.484,  185.509, ],
    [ 153.757,  153.266,  152.852,  152.478,  152.118,  151.768,  151.429,  151.102,  150.785,  150.480, 
      150.189,  149.911,  149.647,  149.398,  149.165,  148.950,  148.752,  148.570,  148.405,  148.256, 
      148.124,  148.009,  147.912,  147.829,  147.759,  147.699,  147.583,  147.501,  147.457,  147.442, 
      147.430,  147.437,  147.414,  147.388,  147.364,  147.347,  147.346,  147.347,  147.350,  147.357, 
      147.366,  147.380,  147.405,  147.438,  147.483,  147.537,  147.603,  147.680,  147.770,  147.870, 
      147.977,  148.090,  148.209,  148.334,  148.466,  148.604,  148.907,  149.245,  149.613,  150.014, 
      150.451,  150.917,  151.405,  151.917,  152.461,  153.033,  153.629,  154.248,  154.891,  155.556, 
      156.245,  156.956,  157.681,  158.427,  159.201,  159.996,  160.803,  161.617,  162.436,  163.259, 
      164.085,  164.914,  165.746,  166.580,  167.415,  168.254,  169.100,  169.952,  170.807,  171.669, 
      172.541,  173.423,  174.313,  175.210,  176.113,  177.026,  177.949,  178.883,  179.826,  180.779, 
      181.742,  182.716,  183.699,  184.691,  185.691,  186.702, ],
    [ 156.002,  155.511,  155.097,  154.724,  154.363,  154.013,  153.674,  153.346,  153.029,  152.724, 
      152.432,  152.155,  151.891,  151.642,  151.409,  151.194,  150.995,  150.814,  150.648,  150.499, 
      150.367,  150.252,  150.154,  150.071,  150.001,  149.941,  149.823,  149.741,  149.697,  149.680, 
      149.666,  149.671,  149.642,  149.611,  149.583,  149.561,  149.555,  149.552,  149.549,  149.550, 
      149.552,  149.560,  149.579,  149.606,  149.644,  149.693,  149.752,  149.823,  149.906,  150.000, 
      150.101,  150.207,  150.319,  150.437,  150.561,  150.691,  150.980,  151.303,  151.655,  152.040, 
      152.461,  152.911,  153.381,  153.877,  154.403,  154.958,  155.536,  156.138,  156.763,  157.410, 
      158.081,  158.774,  159.480,  160.208,  160.964,  161.741,  162.529,  163.325,  164.125,  164.929, 
      165.736,  166.546,  167.359,  168.173,  168.989,  169.808,  170.635,  171.467,  172.303,  173.146, 
      173.999,  174.862,  175.734,  176.612,  177.498,  178.392,  179.298,  180.214,  181.141,  182.077, 
      183.024,  183.981,  184.948,  185.924,  186.909,  187.904, ],
    [ 158.247,  157.756,  157.341,  156.968,  156.607,  156.257,  155.918,  155.589,  155.273,  154.968, 
      154.676,  154.399,  154.134,  153.886,  153.653,  153.437,  153.238,  153.056,  152.891,  152.742, 
      152.610,  152.494,  152.396,  152.313,  152.243,  152.183,  152.063,  151.981,  151.936,  151.918, 
      151.902,  151.905,  151.870,  151.834,  151.801,  151.774,  151.765,  151.756,  151.748,  151.743, 
      151.739,  151.740,  151.753,  151.775,  151.807,  151.849,  151.903,  151.967,  152.044,  152.132, 
      152.226,  152.326,  152.431,  152.541,  152.658,  152.781,  153.055,  153.362,  153.699,  154.069, 
      154.474,  154.907,  155.362,  155.840,  156.350,  156.887,  157.449,  158.033,  158.640,  159.270, 
      159.923,  160.598,  161.286,  161.996,  162.734,  163.493,  164.263,  165.041,  165.822,  166.607, 
      167.395,  168.186,  168.980,  169.775,  170.571,  171.371,  172.178,  172.991,  173.808,  174.632, 
      175.466,  176.311,  177.164,  178.024,  178.892,  179.768,  180.656,  181.556,  182.465,  183.384, 
      184.315,  185.256,  186.207,  187.167,  188.137,  189.117, ],
    [ 160.491,  160.000,  159.585,  159.211,  158.851,  158.501,  158.161,  157.833,  157.516,  157.212, 
      156.920,  156.642,  156.378,  156.129,  155.896,  155.680,  155.481,  155.299,  155.133,  154.984, 
      154.852,  154.736,  154.638,  154.555,  154.485,  154.424,  154.303,  154.221,  154.175,  154.156, 
      154.138,  154.139,  154.097,  154.058,  154.018,  153.987,  153.975,  153.960,  153.946,  153.936, 
      153.925,  153.921,  153.928,  153.943,  153.969,  154.006,  154.053,  154.112,  154.183,  154.265, 
      154.352,  154.446,  154.543,  154.647,  154.757,  154.872,  155.132,  155.424,  155.746,  156.100, 
      156.490,  156.908,  157.345,  157.808,  158.301,  158.821,  159.366,  159.933,  160.523,  161.135, 
      161.771,  162.428,  163.099,  163.791,  164.511,  165.252,  166.004,  166.764,  167.527,  168.293, 
      169.062,  169.834,  170.608,  171.384,  172.161,  172.942,  173.730,  174.524,  175.321,  176.126, 
      176.942,  177.768,  178.603,  179.445,  180.295,  181.154,  182.024,  182.906,  183.799,  184.701, 
      185.615,  186.540,  187.475,  188.420,  189.374,  190.339, ],
    [ 162.735,  162.244,  161.829,  161.455,  161.094,  160.744,  160.405,  160.077,  159.760,  159.455, 
      159.163,  158.885,  158.621,  158.372,  158.139,  157.923,  157.724,  157.542,  157.375,  157.226, 
      157.094,  156.978,  156.880,  156.797,  156.727,  156.665,  156.543,  156.461,  156.415,  156.394, 
      156.373,  156.373,  156.324,  156.280,  156.235,  156.200,  156.184,  156.164,  156.145,  156.129, 
      156.111,  156.102,  156.102,  156.112,  156.132,  156.163,  156.205,  156.257,  156.323,  156.398, 
      156.479,  156.566,  156.658,  156.754,  156.857,  156.965,  157.210,  157.488,  157.795,  158.134, 
      158.509,  158.911,  159.333,  159.779,  160.255,  160.760,  161.288,  161.838,  162.410,  163.005, 
      163.624,  164.264,  164.917,  165.591,  166.294,  167.017,  167.752,  168.493,  169.238,  169.986, 
      170.737,  171.490,  172.245,  173.002,  173.760,  174.521,  175.290,  176.065,  176.843,  177.629, 
      178.427,  179.234,  180.051,  180.875,  181.707,  182.548,  183.401,  184.266,  185.142,  186.028, 
      186.925,  187.834,  188.753,  189.682,  190.621,  191.570, ],
    [ 164.979,  164.487,  164.073,  163.699,  163.338,  162.988,  162.649,  162.320,  162.003,  161.698, 
      161.406,  161.128,  160.864,  160.615,  160.382,  160.166,  159.966,  159.784,  159.618,  159.468, 
      159.336,  159.221,  159.122,  159.039,  158.967,  158.905,  158.783,  158.700,  158.654,  158.631, 
      158.609,  158.607,  158.551,  158.503,  158.452,  158.414,  158.392,  158.368,  158.343,  158.322, 
      158.297,  158.282,  158.277,  158.281,  158.295,  158.320,  158.356,  158.403,  158.463,  158.533, 
      158.608,  158.688,  158.773,  158.862,  158.958,  159.059,  159.290,  159.554,  159.847,  160.171, 
      160.530,  160.917,  161.323,  161.753,  162.214,  162.702,  163.213,  163.747,  164.303,  164.880, 
      165.482,  166.105,  166.741,  167.398,  168.083,  168.789,  169.507,  170.230,  170.957,  171.686, 
      172.419,  173.153,  173.889,  174.627,  175.366,  176.108,  176.858,  177.614,  178.373,  179.141, 
      179.919,  180.709,  181.507,  182.313,  183.127,  183.951,  184.787,  185.635,  186.493,  187.363, 
      188.244,  189.137,  190.040,  190.954,  191.877,  192.811, ],
    [ 167.222,  166.731,  166.316,  165.942,  165.581,  165.231,  164.892,  164.563,  164.246,  163.941, 
      163.649,  163.371,  163.106,  162.857,  162.624,  162.408,  162.208,  162.026,  161.860,  161.710, 
      161.578,  161.463,  161.364,  161.280,  161.208,  161.145,  161.023,  160.940,  160.893,  160.869, 
      160.845,  160.839,  160.778,  160.725,  160.669,  160.627,  160.600,  160.571,  160.540,  160.514, 
      160.483,  160.463,  160.451,  160.451,  160.459,  160.478,  160.509,  160.550,  160.604,  160.668, 
      160.737,  160.811,  160.889,  160.972,  161.061,  161.155,  161.372,  161.622,  161.900,  162.210, 
      162.555,  162.927,  163.317,  163.731,  164.176,  164.648,  165.143,  165.661,  166.200,  166.761, 
      167.346,  167.951,  168.570,  169.210,  169.878,  170.567,  171.267,  171.973,  172.682,  173.393, 
      174.107,  174.823,  175.541,  176.260,  176.979,  177.703,  178.433,  179.170,  179.912,  180.660, 
      181.420,  182.192,  182.972,  183.760,  184.556,  185.362,  186.181,  187.012,  187.854,  188.707, 
      189.572,  190.448,  191.336,  192.234,  193.142,  194.061, ],
    [ 169.465,  168.974,  168.559,  168.185,  167.824,  167.474,  167.135,  166.806,  166.489,  166.184, 
      165.891,  165.613,  165.348,  165.099,  164.866,  164.650,  164.450,  164.268,  164.102,  163.952, 
      163.820,  163.705,  163.605,  163.520,  163.448,  163.385,  163.263,  163.180,  163.132,  163.106, 
      163.081,  163.070,  163.004,  162.946,  162.886,  162.839,  162.809,  162.773,  162.738,  162.707, 
      162.668,  162.644,  162.626,  162.620,  162.622,  162.636,  162.661,  162.697,  162.745,  162.804, 
      162.867,  162.935,  163.006,  163.082,  163.164,  163.252,  163.456,  163.691,  163.955,  164.250, 
      164.581,  164.938,  165.313,  165.712,  166.141,  166.598,  167.077,  167.578,  168.101,  168.645, 
      169.214,  169.803,  170.404,  171.027,  171.678,  172.351,  173.034,  173.722,  174.414,  175.107, 
      175.803,  176.500,  177.199,  177.899,  178.600,  179.305,  180.017,  180.735,  181.457,  182.187, 
      182.929,  183.682,  184.444,  185.215,  185.993,  186.782,  187.583,  188.397,  189.222,  190.059, 
      190.908,  191.769,  192.640,  193.523,  194.415,  195.319, ],
    [ 171.708,  171.217,  170.802,  170.428,  170.067,  169.717,  169.377,  169.048,  168.731,  168.426, 
      168.134,  167.855,  167.591,  167.341,  167.108,  166.892,  166.693,  166.510,  166.344,  166.194, 
      166.062,  165.945,  165.845,  165.760,  165.688,  165.625,  165.503,  165.420,  165.371,  165.342, 
      165.316,  165.299,  165.231,  165.167,  165.103,  165.052,  165.017,  164.976,  164.935,  164.898, 
      164.853,  164.824,  164.800,  164.790,  164.786,  164.795,  164.814,  164.844,  164.888,  164.941, 
      164.997,  165.059,  165.124,  165.194,  165.269,  165.350,  165.540,  165.762,  166.012,  166.293, 
      166.610,  166.953,  167.313,  167.696,  168.110,  168.551,  169.014,  169.499,  170.006,  170.534, 
      171.086,  171.659,  172.244,  172.850,  173.485,  174.140,  174.806,  175.478,  176.152,  176.827, 
      177.505,  178.184,  178.864,  179.546,  180.228,  180.914,  181.607,  182.306,  183.010,  183.722, 
      184.445,  185.180,  185.924,  186.677,  187.438,  188.209,  188.994,  189.791,  190.599,  191.419, 
      192.252,  193.097,  193.953,  194.820,  195.697,  196.587, ],
    [ 173.951,  173.459,  173.044,  172.670,  172.309,  171.959,  171.619,  171.290,  170.973,  170.668, 
      170.376,  170.097,  169.833,  169.583,  169.350,  169.134,  168.935,  168.752,  168.586,  168.436, 
      168.302,  168.186,  168.085,  168.001,  167.928,  167.865,  167.743,  167.659,  167.610,  167.579, 
      167.552,  167.527,  167.456,  167.386,  167.319,  167.264,  167.225,  167.178,  167.132,  167.089, 
      167.038,  167.004,  166.974,  166.959,  166.949,  166.953,  166.967,  166.992,  167.030,  167.078, 
      167.129,  167.184,  167.243,  167.306,  167.375,  167.449,  167.626,  167.835,  168.071,  168.338, 
      168.641,  168.969,  169.314,  169.683,  170.081,  170.507,  170.955,  171.424,  171.915,  172.427, 
      172.964,  173.520,  174.088,  174.677,  175.296,  175.935,  176.585,  177.239,  177.895,  178.553, 
      179.213,  179.874,  180.536,  181.199,  181.862,  182.529,  183.204,  183.885,  184.570,  185.263, 
      185.969,  186.686,  187.412,  188.147,  188.891,  189.645,  190.412,  191.192,  191.984,  192.788, 
      193.605,  194.434,  195.274,  196.125,  196.988,  197.862, ],
    [ 176.193,  175.702,  175.287,  174.912,  174.551,  174.201,  173.861,  173.533,  173.215,  172.910, 
      172.618,  172.339,  172.075,  171.825,  171.592,  171.376,  171.177,  170.994,  170.827,  170.676, 
      170.543,  170.426,  170.326,  170.241,  170.168,  170.105,  169.983,  169.899,  169.849,  169.816, 
      169.788,  169.755,  169.679,  169.605,  169.535,  169.476,  169.432,  169.381,  169.330,  169.280, 
      169.222,  169.183,  169.148,  169.128,  169.112,  169.111,  169.120,  169.140,  169.173,  169.215, 
      169.261,  169.310,  169.363,  169.419,  169.481,  169.549,  169.713,  169.909,  170.131,  170.385, 
      170.674,  170.988,  171.319,  171.672,  172.056,  172.466,  172.899,  173.353,  173.828,  174.324, 
      174.845,  175.385,  175.937,  176.510,  177.112,  177.735,  178.368,  179.006,  179.645,  180.285, 
      180.927,  181.570,  182.214,  182.858,  183.503,  184.152,  184.808,  185.470,  186.137,  186.812, 
      187.499,  188.198,  188.907,  189.625,  190.350,  191.087,  191.838,  192.601,  193.377,  194.164, 
      194.965,  195.778,  196.603,  197.439,  198.286,  199.145, ],
    [ 178.435,  177.944,  177.529,  177.155,  176.793,  176.443,  176.103,  175.775,  175.457,  175.152, 
      174.860,  174.582,  174.317,  174.067,  173.834,  173.618,  173.418,  173.234,  173.067,  172.916, 
      172.783,  172.666,  172.566,  172.481,  172.408,  172.346,  172.223,  172.139,  172.088,  172.052, 
      172.023,  171.983,  171.903,  171.823,  171.750,  171.689,  171.638,  171.583,  171.527,  171.469, 
      171.405,  171.362,  171.321,  171.298,  171.275,  171.270,  171.273,  171.288,  171.316,  171.353, 
      171.393,  171.437,  171.483,  171.533,  171.589,  171.649,  171.801,  171.984,  172.193,  172.433, 
      172.709,  173.009,  173.325,  173.664,  174.033,  174.429,  174.846,  175.285,  175.745,  176.225, 
      176.730,  177.254,  177.790,  178.347,  178.933,  179.540,  180.157,  180.778,  181.400,  182.023, 
      182.647,  183.272,  183.898,  184.524,  185.150,  185.780,  186.418,  187.062,  187.710,  188.367, 
      189.036,  189.718,  190.409,  191.109,  191.817,  192.537,  193.271,  194.018,  194.776,  195.548, 
      196.333,  197.130,  197.939,  198.760,  199.592,  200.437, ],
    [ 180.677,  180.186,  179.771,  179.397,  179.035,  178.685,  178.345,  178.017,  177.699,  177.394, 
      177.102,  176.824,  176.559,  176.309,  176.075,  175.858,  175.658,  175.475,  175.307,  175.157, 
      175.023,  174.906,  174.806,  174.721,  174.648,  174.586,  174.463,  174.378,  174.326,  174.289, 
      174.259,  174.211,  174.126,  174.042,  173.964,  173.901,  173.844,  173.783,  173.722,  173.658, 
      173.588,  173.541,  173.494,  173.467,  173.438,  173.428,  173.427,  173.437,  173.460,  173.492, 
      173.526,  173.564,  173.603,  173.647,  173.697,  173.751,  173.890,  174.060,  174.256,  174.483, 
      174.745,  175.032,  175.334,  175.658,  176.012,  176.394,  176.797,  177.220,  177.664,  178.130, 
      178.619,  179.128,  179.647,  180.188,  180.759,  181.350,  181.950,  182.555,  183.160,  183.766, 
      184.372,  184.980,  185.587,  186.195,  186.803,  187.415,  188.034,  188.660,  189.290,  189.929, 
      190.580,  191.244,  191.917,  192.600,  193.291,  193.994,  194.711,  195.441,  196.184,  196.939, 
      197.708,  198.489,  199.283,  200.089,  200.906,  201.735, ],
    [ 182.919,  182.428,  182.013,  181.639,  181.277,  180.927,  180.587,  180.259,  179.941,  179.636, 
      179.343,  179.065,  178.799,  178.549,  178.315,  178.098,  177.898,  177.715,  177.547,  177.397, 
      177.263,  177.146,  177.046,  176.961,  176.888,  176.826,  176.703,  176.618,  176.565,  176.525, 
      176.494,  176.438,  176.349,  176.259,  176.178,  176.113,  176.050,  175.983,  175.916,  175.846, 
      175.771,  175.718,  175.668,  175.635,  175.601,  175.586,  175.580,  175.585,  175.604,  175.631, 
      175.659,  175.691,  175.724,  175.762,  175.805,  175.853,  175.980,  176.137,  176.320,  176.534, 
      176.783,  177.057,  177.344,  177.654,  177.995,  178.362,  178.750,  179.159,  179.588,  180.038, 
      180.512,  181.005,  181.509,  182.034,  182.589,  183.164,  183.749,  184.337,  184.925,  185.514, 
      186.103,  186.693,  187.283,  187.872,  188.462,  189.055,  189.656,  190.264,  190.876,  191.496, 
      192.130,  192.776,  193.432,  194.098,  194.772,  195.457,  196.157,  196.871,  197.598,  198.337, 
      199.090,  199.856,  200.635,  201.425,  202.227,  203.042, ],
    [ 185.162,  184.670,  184.255,  183.881,  183.520,  183.169,  182.829,  182.500,  182.182,  181.876, 
      181.584,  181.305,  181.039,  180.789,  180.555,  180.338,  180.138,  179.955,  179.787,  179.637, 
      179.503,  179.386,  179.286,  179.201,  179.128,  179.066,  178.943,  178.857,  178.804,  178.761, 
      178.730,  178.665,  178.571,  178.476,  178.391,  178.326,  178.256,  178.182,  178.109,  178.033, 
      177.954,  177.895,  177.841,  177.803,  177.764,  177.744,  177.733,  177.734,  177.748,  177.770, 
      177.792,  177.818,  177.846,  177.878,  177.914,  177.956,  178.071,  178.215,  178.386,  178.587, 
      178.823,  179.083,  179.357,  179.653,  179.979,  180.332,  180.706,  181.100,  181.514,  181.949, 
      182.408,  182.886,  183.374,  183.883,  184.423,  184.983,  185.552,  186.124,  186.695,  187.267, 
      187.839,  188.411,  188.983,  189.555,  190.127,  190.702,  191.284,  191.874,  192.468,  193.070, 
      193.686,  194.315,  194.954,  195.602,  196.259,  196.927,  197.611,  198.308,  199.018,  199.742, 
      200.479,  201.230,  201.993,  202.768,  203.555,  204.355, ],
    [ 187.404,  186.912,  186.497,  186.122,  185.760,  185.409,  185.069,  184.740,  184.422,  184.116, 
      183.824,  183.545,  183.280,  183.029,  182.795,  182.579,  182.378,  182.195,  182.028,  181.877, 
      181.743,  181.627,  181.526,  181.441,  181.368,  181.306,  181.182,  181.096,  181.042,  180.998, 
      180.965,  180.891,  180.792,  180.693,  180.604,  180.536,  180.462,  180.382,  180.303,  180.220, 
      180.137,  180.072,  180.014,  179.970,  179.927,  179.901,  179.887,  179.883,  179.892,  179.909, 
      179.926,  179.946,  179.968,  179.993,  180.024,  180.059,  180.162,  180.294,  180.452,  180.640, 
      180.864,  181.112,  181.371,  181.653,  181.966,  182.304,  182.664,  183.044,  183.443,  183.863, 
      184.307,  184.770,  185.243,  185.737,  186.261,  186.806,  187.359,  187.915,  188.470,  189.025, 
      189.580,  190.134,  190.689,  191.243,  191.796,  192.353,  192.918,  193.490,  194.065,  194.650, 
      195.248,  195.859,  196.481,  197.112,  197.752,  198.404,  199.070,  199.752,  200.446,  201.153, 
      201.875,  202.610,  203.358,  204.118,  204.891,  205.676, ],
    [ 189.644,  189.152,  188.737,  188.362,  188.001,  187.650,  187.310,  186.980,  186.662,  186.357, 
      186.064,  185.785,  185.520,  185.270,  185.036,  184.819,  184.619,  184.435,  184.268,  184.117, 
      183.984,  183.867,  183.766,  183.681,  183.608,  183.545,  183.422,  183.335,  183.281,  183.234, 
      183.200,  183.116,  183.011,  182.910,  182.817,  182.745,  182.666,  182.581,  182.496,  182.405, 
      182.319,  182.248,  182.186,  182.137,  182.090,  182.059,  182.040,  182.032,  182.037,  182.048, 
      182.060,  182.074,  182.090,  182.109,  182.134,  182.163,  182.254,  182.374,  182.520,  182.695, 
      182.907,  183.141,  183.388,  183.655,  183.954,  184.279,  184.625,  184.990,  185.375,  185.781, 
      186.210,  186.658,  187.116,  187.594,  188.103,  188.633,  189.171,  189.711,  190.250,  190.788, 
      191.326,  191.863,  192.399,  192.936,  193.471,  194.010,  194.557,  195.111,  195.668,  196.235, 
      196.816,  197.410,  198.014,  198.628,  199.251,  199.886,  200.536,  201.201,  201.880,  202.571, 
      203.277,  203.997,  204.730,  205.475,  206.233,  207.004, ],
    [ 191.885,  191.393,  190.977,  190.603,  190.241,  189.890,  189.550,  189.221,  188.903,  188.597, 
      188.304,  188.025,  187.760,  187.510,  187.276,  187.059,  186.859,  186.675,  186.508,  186.357, 
      186.224,  186.107,  186.006,  185.921,  185.848,  185.785,  185.662,  185.575,  185.519,  185.470, 
      185.435,  185.339,  185.229,  185.125,  185.030,  184.953,  184.868,  184.778,  184.689,  184.589, 
      184.500,  184.425,  184.357,  184.303,  184.252,  184.217,  184.193,  184.180,  184.181,  184.188, 
      184.194,  184.203,  184.212,  184.225,  184.244,  184.267,  184.347,  184.455,  184.588,  184.751, 
      184.951,  185.172,  185.405,  185.660,  185.945,  186.256,  186.588,  186.939,  187.310,  187.701, 
      188.115,  188.549,  188.991,  189.455,  189.949,  190.464,  190.987,  191.511,  192.033,  192.555, 
      193.076,  193.596,  194.115,  194.633,  195.151,  195.672,  196.201,  196.737,  197.277,  197.825, 
      198.389,  198.965,  199.553,  200.150,  200.756,  201.374,  202.008,  202.657,  203.319,  203.995, 
      204.686,  205.391,  206.108,  206.839,  207.582,  208.338, ],
    [ 194.125,  193.633,  193.217,  192.843,  192.481,  192.130,  191.790,  191.461,  191.143,  190.837, 
      190.544,  190.265,  190.000,  189.750,  189.516,  189.299,  189.099,  188.915,  188.748,  188.597, 
      188.464,  188.347,  188.246,  188.161,  188.088,  188.025,  187.901,  187.814,  187.757,  187.707, 
      187.668,  187.562,  187.447,  187.340,  187.242,  187.161,  187.071,  186.976,  186.879,  186.772, 
      186.681,  186.602,  186.527,  186.468,  186.415,  186.375,  186.347,  186.328,  186.326,  186.328, 
      186.328,  186.331,  186.334,  186.342,  186.354,  186.371,  186.440,  186.536,  186.657,  186.809, 
      186.996,  187.205,  187.425,  187.665,  187.938,  188.236,  188.553,  188.890,  189.247,  189.624, 
      190.024,  190.443,  190.870,  191.319,  191.799,  192.299,  192.806,  193.315,  193.821,  194.326, 
      194.830,  195.333,  195.835,  196.336,  196.835,  197.338,  197.849,  198.368,  198.890,  199.421, 
      199.967,  200.526,  201.097,  201.677,  202.266,  202.868,  203.486,  204.119,  204.765,  205.425, 
      206.101,  206.790,  207.493,  208.209,  208.937,  209.679, ],
    [ 196.365,  195.873,  195.457,  195.083,  194.721,  194.370,  194.030,  193.701,  193.383,  193.077, 
      192.784,  192.506,  192.240,  191.990,  191.756,  191.539,  191.339,  191.155,  190.988,  190.837, 
      190.704,  190.587,  190.486,  190.401,  190.328,  190.265,  190.141,  190.053,  189.996,  189.943, 
      189.895,  189.785,  189.664,  189.554,  189.454,  189.367,  189.271,  189.172,  189.068,  188.955, 
      188.862,  188.776,  188.698,  188.634,  188.577,  188.532,  188.500,  188.476,  188.470,  188.467, 
      188.462,  188.460,  188.457,  188.458,  188.465,  188.476,  188.533,  188.618,  188.727,  188.866, 
      189.042,  189.239,  189.445,  189.673,  189.932,  190.216,  190.521,  190.844,  191.187,  191.549, 
      191.935,  192.340,  192.753,  193.186,  193.652,  194.137,  194.629,  195.122,  195.613,  196.102, 
      196.589,  197.075,  197.559,  198.042,  198.524,  199.009,  199.503,  200.003,  200.508,  201.021, 
      201.550,  202.093,  202.646,  203.209,  203.782,  204.368,  204.969,  205.586,  206.217,  206.861, 
      207.521,  208.196,  208.884,  209.584,  210.298,  211.026, ],
    [ 198.605,  198.113,  197.698,  197.323,  196.961,  196.610,  196.270,  195.941,  195.623,  195.317, 
      195.024,  194.746,  194.480,  194.230,  193.996,  193.779,  193.579,  193.396,  193.228,  193.077, 
      192.944,  192.827,  192.726,  192.641,  192.568,  192.505,  192.380,  192.292,  192.233,  192.179, 
      192.122,  192.006,  191.881,  191.767,  191.667,  191.573,  191.471,  191.365,  191.255,  191.138, 
      191.041,  190.949,  190.868,  190.799,  190.739,  190.689,  190.652,  190.625,  190.615,  190.607, 
      190.596,  190.588,  190.579,  190.575,  190.575,  190.580,  190.627,  190.700,  190.798,  190.925, 
      191.089,  191.274,  191.467,  191.681,  191.928,  192.199,  192.490,  192.800,  193.128,  193.477, 
      193.849,  194.240,  194.638,  195.057,  195.508,  195.979,  196.456,  196.934,  197.409,  197.881, 
      198.352,  198.820,  199.288,  199.753,  200.218,  200.685,  201.161,  201.644,  202.131,  202.627, 
      203.138,  203.664,  204.200,  204.747,  205.303,  205.872,  206.458,  207.059,  207.674,  208.303, 
      208.948,  209.607,  210.280,  210.966,  211.666,  212.379, ],
    [ 200.845,  200.353,  199.938,  199.563,  199.201,  198.850,  198.510,  198.181,  197.863,  197.557, 
      197.265,  196.986,  196.720,  196.470,  196.236,  196.019,  195.819,  195.635,  195.468,  195.317, 
      195.183,  195.066,  194.966,  194.881,  194.808,  194.744,  194.620,  194.531,  194.470,  194.415, 
      194.347,  194.225,  194.098,  193.980,  193.879,  193.779,  193.670,  193.558,  193.440,  193.322, 
      193.218,  193.123,  193.039,  192.965,  192.900,  192.845,  192.804,  192.773,  192.760,  192.746, 
      192.730,  192.716,  192.701,  192.691,  192.686,  192.685,  192.722,  192.783,  192.869,  192.985, 
      193.138,  193.310,  193.490,  193.691,  193.925,  194.184,  194.461,  194.758,  195.073,  195.407, 
      195.766,  196.142,  196.525,  196.930,  197.367,  197.824,  198.287,  198.749,  199.208,  199.664, 
      200.118,  200.570,  201.020,  201.468,  201.915,  202.364,  202.823,  203.288,  203.758,  204.237, 
      204.731,  205.240,  205.760,  206.290,  206.829,  207.382,  207.952,  208.537,  209.136,  209.750, 
      210.380,  211.024,  211.682,  212.354,  213.039,  213.738, ],
    [ 203.085,  202.593,  202.178,  201.803,  201.442,  201.091,  200.750,  200.421,  200.103,  199.797, 
      199.505,  199.226,  198.960,  198.710,  198.476,  198.259,  198.059,  197.875,  197.708,  197.557, 
      197.423,  197.306,  197.206,  197.120,  197.047,  196.984,  196.859,  196.769,  196.707,  196.651, 
      196.571,  196.443,  196.314,  196.193,  196.090,  195.983,  195.869,  195.752,  195.624,  195.504, 
      195.395,  195.296,  195.209,  195.128,  195.060,  195.002,  194.956,  194.921,  194.904,  194.885, 
      194.864,  194.844,  194.823,  194.808,  194.796,  194.790,  194.816,  194.866,  194.941,  195.045, 
      195.187,  195.347,  195.514,  195.703,  195.924,  196.170,  196.434,  196.717,  197.019,  197.340, 
      197.684,  198.047,  198.416,  198.806,  199.229,  199.672,  200.120,  200.568,  201.011,  201.451, 
      201.889,  202.324,  202.757,  203.187,  203.616,  204.048,  204.489,  204.937,  205.389,  205.851, 
      206.328,  206.820,  207.323,  207.837,  208.360,  208.897,  209.450,  210.020,  210.604,  211.202, 
      211.817,  212.447,  213.090,  213.747,  214.418,  215.103, ],
    [ 205.325,  204.833,  204.418,  204.043,  203.682,  203.331,  202.990,  202.661,  202.343,  202.037, 
      201.745,  201.466,  201.200,  200.950,  200.716,  200.499,  200.299,  200.115,  199.948,  199.797, 
      199.663,  199.546,  199.445,  199.360,  199.287,  199.223,  199.098,  199.008,  198.943,  198.887, 
      198.794,  198.661,  198.529,  198.406,  198.299,  198.186,  198.067,  197.943,  197.808,  197.685, 
      197.571,  197.469,  197.379,  197.291,  197.220,  197.158,  197.108,  197.069,  197.047,  197.024, 
      196.997,  196.971,  196.945,  196.924,  196.907,  196.895,  196.911,  196.950,  197.013,  197.106, 
      197.236,  197.385,  197.539,  197.715,  197.924,  198.157,  198.409,  198.679,  198.967,  199.274, 
      199.605,  199.954,  200.309,  200.685,  201.094,  201.523,  201.957,  202.389,  202.817,  203.241, 
      203.663,  204.081,  204.497,  204.910,  205.322,  205.736,  206.159,  206.590,  207.025,  207.469, 
      207.930,  208.405,  208.892,  209.389,  209.896,  210.416,  210.954,  211.508,  212.077,  212.660, 
      213.260,  213.874,  214.503,  215.146,  215.803,  216.474, ],
    [ 207.565,  207.073,  206.658,  206.283,  205.922,  205.571,  205.230,  204.901,  204.583,  204.277, 
      203.985,  203.706,  203.440,  203.190,  202.956,  202.739,  202.539,  202.355,  202.188,  202.037, 
      201.903,  201.786,  201.685,  201.600,  201.526,  201.463,  201.337,  201.247,  201.179,  201.123, 
      201.016,  200.878,  200.743,  200.618,  200.505,  200.387,  200.263,  200.131,  199.991,  199.866, 
      199.748,  199.640,  199.546,  199.454,  199.380,  199.314,  199.260,  199.217,  199.190,  199.162, 
      199.130,  199.098,  199.066,  199.040,  199.017,  198.999,  199.006,  199.034,  199.086,  199.167, 
      199.287,  199.423,  199.566,  199.729,  199.925,  200.146,  200.385,  200.642,  200.917,  201.211, 
      201.529,  201.864,  202.204,  202.567,  202.962,  203.377,  203.797,  204.214,  204.627,  205.035, 
      205.440,  205.841,  206.240,  206.637,  207.030,  207.427,  207.833,  208.247,  208.664,  209.091, 
      209.535,  209.994,  210.464,  210.945,  211.436,  211.940,  212.462,  213.001,  213.554,  214.122, 
      214.707,  215.307,  215.922,  216.550,  217.192,  217.849, ],
    [ 209.805,  209.313,  208.898,  208.523,  208.162,  207.811,  207.470,  207.141,  206.823,  206.517, 
      206.225,  205.946,  205.680,  205.430,  205.196,  204.979,  204.778,  204.595,  204.427,  204.277, 
      204.143,  204.025,  203.925,  203.839,  203.766,  203.702,  203.576,  203.485,  203.416,  203.359, 
      203.236,  203.095,  202.956,  202.830,  202.711,  202.586,  202.456,  202.317,  202.174,  202.046, 
      201.922,  201.810,  201.711,  201.617,  201.538,  201.470,  201.410,  201.364,  201.334,  201.300, 
      201.263,  201.225,  201.187,  201.157,  201.127,  201.103,  201.101,  201.118,  201.159,  201.229, 
      201.338,  201.463,  201.593,  201.743,  201.928,  202.137,  202.363,  202.607,  202.868,  203.149, 
      203.454,  203.776,  204.102,  204.451,  204.833,  205.234,  205.640,  206.042,  206.440,  206.832, 
      207.220,  207.605,  207.987,  208.366,  208.743,  209.122,  209.511,  209.907,  210.307,  210.717, 
      211.144,  211.587,  212.041,  212.505,  212.980,  213.468,  213.975,  214.498,  215.036,  215.589, 
      216.159,  216.745,  217.345,  217.959,  218.588,  219.231, ],
    [ 212.045,  211.553,  211.138,  210.763,  210.402,  210.051,  209.710,  209.381,  209.063,  208.757, 
      208.464,  208.185,  207.920,  207.670,  207.436,  207.219,  207.018,  206.835,  206.667,  206.516, 
      206.382,  206.265,  206.164,  206.079,  206.005,  205.942,  205.815,  205.724,  205.652,  205.588, 
      205.454,  205.311,  205.169,  205.043,  204.918,  204.786,  204.649,  204.501,  204.357,  204.223, 
      204.095,  203.981,  203.877,  203.780,  203.696,  203.625,  203.561,  203.512,  203.477,  203.437, 
      203.396,  203.352,  203.308,  203.272,  203.236,  203.208,  203.196,  203.202,  203.232,  203.291, 
      203.390,  203.503,  203.620,  203.759,  203.931,  204.128,  204.342,  204.573,  204.822,  205.089, 
      205.381,  205.689,  206.002,  206.337,  206.705,  207.094,  207.485,  207.873,  208.255,  208.631, 
      209.004,  209.372,  209.737,  210.099,  210.458,  210.820,  211.192,  211.571,  211.954,  212.347, 
      212.758,  213.183,  213.621,  214.070,  214.528,  215.001,  215.492,  216.000,  216.523,  217.061, 
      217.616,  218.188,  218.774,  219.374,  219.988,  220.617, ],
    [ 214.285,  213.793,  213.378,  213.003,  212.642,  212.290,  211.950,  211.621,  211.303,  210.997, 
      210.704,  210.425,  210.160,  209.909,  209.675,  209.458,  209.258,  209.075,  208.907,  208.756, 
      208.622,  208.505,  208.404,  208.318,  208.245,  208.181,  208.054,  207.962,  207.889,  207.812, 
      207.672,  207.526,  207.382,  207.255,  207.121,  206.984,  206.842,  206.684,  206.538,  206.399, 
      206.269,  206.151,  206.042,  205.942,  205.854,  205.779,  205.711,  205.659,  205.620,  205.575, 
      205.529,  205.479,  205.429,  205.386,  205.346,  205.312,  205.290,  205.286,  205.305,  205.354, 
      205.442,  205.544,  205.649,  205.775,  205.936,  206.121,  206.322,  206.540,  206.776,  207.031, 
      207.310,  207.605,  207.904,  208.225,  208.581,  208.956,  209.333,  209.707,  210.073,  210.434, 
      210.790,  211.142,  211.490,  211.835,  212.177,  212.521,  212.876,  213.238,  213.604,  213.980, 
      214.374,  214.784,  215.205,  215.638,  216.080,  216.537,  217.013,  217.506,  218.014,  218.537, 
      219.078,  219.635,  220.207,  220.793,  221.393,  222.008, ],
    [ 216.525,  216.033,  215.618,  215.243,  214.881,  214.530,  214.190,  213.861,  213.543,  213.237, 
      212.944,  212.665,  212.400,  212.149,  211.915,  211.698,  211.498,  211.314,  211.146,  210.996, 
      210.861,  210.744,  210.643,  210.558,  210.484,  210.420,  210.293,  210.200,  210.125,  210.035, 
      209.889,  209.740,  209.594,  209.462,  209.323,  209.181,  209.029,  208.867,  208.719,  208.576, 
      208.442,  208.322,  208.206,  208.104,  208.012,  207.932,  207.862,  207.806,  207.764,  207.713, 
      207.662,  207.606,  207.550,  207.500,  207.455,  207.416,  207.385,  207.370,  207.379,  207.417, 
      207.495,  207.585,  207.678,  207.792,  207.942,  208.114,  208.304,  208.509,  208.733,  208.975, 
      209.241,  209.523,  209.808,  210.116,  210.458,  210.820,  211.184,  211.543,  211.894,  212.239, 
      212.579,  212.915,  213.246,  213.574,  213.898,  214.226,  214.563,  214.908,  215.257,  215.616, 
      215.994,  216.388,  216.793,  217.210,  217.636,  218.078,  218.538,  219.016,  219.509,  220.018, 
      220.544,  221.087,  221.644,  222.216,  222.802,  223.405, ],
    [ 218.765,  218.273,  217.858,  217.483,  217.121,  216.770,  216.430,  216.101,  215.783,  215.477, 
      215.184,  214.905,  214.639,  214.389,  214.155,  213.938,  213.737,  213.554,  213.386,  213.235, 
      213.101,  212.984,  212.883,  212.797,  212.723,  212.659,  212.532,  212.438,  212.361,  212.255, 
      212.106,  211.953,  211.806,  211.668,  211.523,  211.374,  211.215,  211.050,  210.900,  210.752, 
      210.612,  210.490,  210.369,  210.265,  210.168,  210.085,  210.012,  209.953,  209.907,  209.850, 
      209.794,  209.732,  209.671,  209.614,  209.564,  209.520,  209.480,  209.454,  209.453,  209.480, 
      209.549,  209.627,  209.708,  209.810,  209.948,  210.109,  210.286,  210.480,  210.690,  210.920, 
      211.173,  211.442,  211.714,  212.008,  212.338,  212.687,  213.037,  213.381,  213.718,  214.047, 
      214.371,  214.690,  215.005,  215.316,  215.623,  215.933,  216.253,  216.582,  216.913,  217.256, 
      217.617,  217.995,  218.385,  218.785,  219.196,  219.622,  220.067,  220.530,  221.008,  221.502, 
      222.014,  222.542,  223.086,  223.644,  224.217,  224.805, ],
    [ 221.005,  220.513,  220.098,  219.723,  219.361,  219.010,  218.670,  218.340,  218.022,  217.716, 
      217.424,  217.145,  216.879,  216.629,  216.395,  216.177,  215.977,  215.793,  215.626,  215.475, 
      215.340,  215.223,  215.122,  215.036,  214.962,  214.898,  214.770,  214.674,  214.597,  214.473, 
      214.322,  214.166,  214.019,  213.874,  213.723,  213.567,  213.398,  213.234,  213.076,  212.925, 
      212.783,  212.656,  212.532,  212.424,  212.324,  212.238,  212.161,  212.099,  212.049,  211.985, 
      211.924,  211.856,  211.792,  211.728,  211.671,  211.624,  211.574,  211.539,  211.526,  211.543, 
      211.602,  211.670,  211.738,  211.828,  211.955,  212.105,  212.270,  212.451,  212.649,  212.866, 
      213.107,  213.363,  213.622,  213.903,  214.220,  214.555,  214.892,  215.223,  215.544,  215.858, 
      216.166,  216.468,  216.766,  217.060,  217.350,  217.642,  217.946,  218.258,  218.572,  218.898, 
      219.244,  219.605,  219.979,  220.364,  220.759,  221.169,  221.599,  222.047,  222.511,  222.991, 
      223.488,  224.003,  224.532,  225.077,  225.636,  226.211, ],
    [ 223.245,  222.753,  222.337,  221.963,  221.601,  221.250,  220.909,  220.580,  220.262,  219.956, 
      219.663,  219.384,  219.119,  218.868,  218.634,  218.417,  218.216,  218.033,  217.865,  217.714, 
      217.580,  217.462,  217.361,  217.275,  217.201,  217.137,  217.009,  216.911,  216.833,  216.690, 
      216.537,  216.379,  216.231,  216.078,  215.920,  215.758,  215.581,  215.415,  215.253,  215.098, 
      214.954,  214.821,  214.695,  214.584,  214.481,  214.390,  214.310,  214.245,  214.190,  214.120, 
      214.053,  213.980,  213.910,  213.842,  213.778,  213.728,  213.669,  213.623,  213.600,  213.606, 
      213.656,  213.712,  213.768,  213.847,  213.963,  214.101,  214.255,  214.423,  214.609,  214.814, 
      215.042,  215.286,  215.531,  215.799,  216.103,  216.426,  216.750,  217.066,  217.372,  217.671, 
      217.963,  218.249,  218.530,  218.807,  219.079,  219.355,  219.642,  219.936,  220.234,  220.544, 
      220.873,  221.219,  221.577,  221.946,  222.325,  222.720,  223.135,  223.569,  224.018,  224.483, 
      224.966,  225.467,  225.982,  226.513,  227.059,  227.620, ],
    [ 225.485,  224.993,  224.577,  224.202,  223.840,  223.489,  223.149,  222.820,  222.502,  222.196, 
      221.903,  221.624,  221.358,  221.108,  220.874,  220.656,  220.456,  220.272,  220.104,  219.953, 
      219.819,  219.702,  219.600,  219.514,  219.440,  219.376,  219.247,  219.147,  219.062,  218.908, 
      218.750,  218.591,  218.439,  218.280,  218.116,  217.944,  217.764,  217.596,  217.430,  217.271, 
      217.124,  216.987,  216.857,  216.741,  216.637,  216.542,  216.458,  216.392,  216.330,  216.255, 
      216.182,  216.102,  216.027,  215.956,  215.886,  215.832,  215.763,  215.707,  215.674,  215.670, 
      215.710,  215.755,  215.799,  215.867,  215.972,  216.099,  216.240,  216.397,  216.571,  216.762, 
      216.979,  217.210,  217.442,  217.697,  217.989,  218.299,  218.609,  218.911,  219.203,  219.486, 
      219.762,  220.031,  220.296,  220.556,  220.811,  221.069,  221.340,  221.618,  221.898,  222.191, 
      222.505,  222.835,  223.177,  223.531,  223.895,  224.275,  224.675,  225.093,  225.528,  225.979, 
      226.448,  226.935,  227.437,  227.954,  228.486,  229.034, ],
    [ 227.724,  227.232,  226.817,  226.442,  226.080,  225.729,  225.389,  225.059,  224.741,  224.435, 
      224.142,  223.863,  223.598,  223.347,  223.113,  222.896,  222.695,  222.512,  222.344,  222.193, 
      222.058,  221.941,  221.839,  221.753,  221.679,  221.615,  221.486,  221.383,  221.281,  221.125, 
      220.963,  220.803,  220.645,  220.479,  220.309,  220.128,  219.947,  219.774,  219.604,  219.442, 
      219.294,  219.150,  219.020,  218.899,  218.792,  218.692,  218.606,  218.538,  218.471,  218.390, 
      218.309,  218.224,  218.144,  218.068,  217.993,  217.935,  217.857,  217.790,  217.747,  217.734, 
      217.765,  217.798,  217.831,  217.886,  217.981,  218.096,  218.226,  218.371,  218.533,  218.712, 
      218.917,  219.136,  219.354,  219.596,  219.876,  220.174,  220.471,  220.759,  221.035,  221.303, 
      221.563,  221.816,  222.064,  222.307,  222.545,  222.786,  223.040,  223.301,  223.565,  223.842, 
      224.139,  224.454,  224.781,  225.119,  225.467,  225.832,  226.217,  226.621,  227.041,  227.478, 
      227.934,  228.406,  228.895,  229.398,  229.917,  230.452, ],
    [ 229.964,  229.472,  229.056,  228.682,  228.320,  227.969,  227.628,  227.299,  226.981,  226.675, 
      226.382,  226.103,  225.837,  225.587,  225.352,  225.135,  224.935,  224.751,  224.583,  224.432, 
      224.297,  224.180,  224.078,  223.992,  223.918,  223.853,  223.724,  223.620,  223.499,  223.340, 
      223.176,  223.016,  222.851,  222.678,  222.502,  222.311,  222.129,  221.951,  221.778,  221.612, 
      221.459,  221.312,  221.180,  221.056,  220.945,  220.843,  220.754,  220.684,  220.608,  220.524, 
      220.436,  220.345,  220.260,  220.179,  220.101,  220.039,  219.952,  219.874,  219.821,  219.797, 
      219.820,  219.842,  219.862,  219.907,  219.991,  220.095,  220.214,  220.346,  220.496,  220.663, 
      220.856,  221.062,  221.268,  221.497,  221.765,  222.051,  222.334,  222.608,  222.870,  223.122, 
      223.366,  223.603,  223.834,  224.060,  224.281,  224.505,  224.742,  224.987,  225.234,  225.495, 
      225.776,  226.075,  226.387,  226.709,  227.043,  227.392,  227.763,  228.152,  228.558,  228.981, 
      229.423,  229.882,  230.357,  230.847,  231.352,  231.874, ],
    [ 232.203,  231.711,  231.296,  230.921,  230.559,  230.208,  229.868,  229.538,  229.220,  228.914, 
      228.621,  228.342,  228.076,  227.826,  227.592,  227.374,  227.174,  226.990,  226.822,  226.671, 
      226.536,  226.419,  226.317,  226.231,  226.157,  226.092,  225.961,  225.856,  225.717,  225.554, 
      225.388,  225.227,  225.053,  224.875,  224.688,  224.494,  224.310,  224.128,  223.951,  223.783, 
      223.625,  223.475,  223.340,  223.213,  223.098,  222.993,  222.902,  222.827,  222.746,  222.657, 
      222.563,  222.466,  222.376,  222.288,  222.206,  222.141,  222.046,  221.956,  221.894,  221.861, 
      221.874,  221.885,  221.894,  221.927,  222.001,  222.094,  222.201,  222.322,  222.460,  222.615, 
      222.796,  222.990,  223.183,  223.400,  223.656,  223.929,  224.199,  224.459,  224.706,  224.943, 
      225.171,  225.392,  225.606,  225.815,  226.018,  226.226,  226.446,  226.675,  226.905,  227.149, 
      227.416,  227.699,  227.995,  228.303,  228.621,  228.955,  229.311,  229.687,  230.078,  230.487, 
      230.915,  231.360,  231.822,  232.299,  232.791,  233.300, ],
    [ 234.443,  233.951,  233.535,  233.161,  232.799,  232.447,  232.107,  231.778,  231.460,  231.154, 
      230.861,  230.581,  230.316,  230.065,  229.831,  229.614,  229.413,  229.229,  229.061,  228.910, 
      228.775,  228.658,  228.556,  228.470,  228.395,  228.331,  228.197,  228.092,  227.934,  227.767, 
      227.601,  227.433,  227.253,  227.068,  226.871,  226.677,  226.489,  226.303,  226.122,  225.953, 
      225.790,  225.638,  225.498,  225.369,  225.251,  225.143,  225.049,  224.970,  224.884,  224.790, 
      224.690,  224.587,  224.490,  224.397,  224.310,  224.243,  224.138,  224.038,  223.967,  223.925, 
      223.929,  223.928,  223.925,  223.948,  224.012,  224.094,  224.190,  224.299,  224.425,  224.568, 
      224.737,  224.920,  225.099,  225.304,  225.548,  225.809,  226.066,  226.312,  226.545,  226.765, 
      226.978,  227.182,  227.380,  227.572,  227.758,  227.948,  228.152,  228.364,  228.579,  228.806, 
      229.057,  229.325,  229.606,  229.898,  230.201,  230.521,  230.862,  231.224,  231.601,  231.996, 
      232.411,  232.842,  233.290,  233.754,  234.233,  234.729, ],
    [ 236.682,  236.190,  235.775,  235.400,  235.038,  234.687,  234.346,  234.017,  233.699,  233.393, 
      233.100,  232.821,  232.555,  232.304,  232.070,  231.853,  231.652,  231.468,  231.300,  231.149, 
      231.014,  230.897,  230.795,  230.709,  230.634,  230.569,  230.434,  230.317,  230.150,  229.980, 
      229.813,  229.639,  229.452,  229.260,  229.054,  228.859,  228.666,  228.476,  228.293,  228.121, 
      227.953,  227.801,  227.656,  227.525,  227.403,  227.293,  227.196,  227.114,  227.019,  226.920, 
      226.814,  226.708,  226.604,  226.506,  226.414,  226.344,  226.229,  226.119,  226.040,  225.988, 
      225.984,  225.972,  225.957,  225.968,  226.023,  226.094,  226.178,  226.276,  226.390,  226.522, 
      226.680,  226.850,  227.016,  227.209,  227.441,  227.690,  227.935,  228.167,  228.384,  228.590, 
      228.786,  228.974,  229.155,  229.330,  229.499,  229.673,  229.860,  230.056,  230.253,  230.465, 
      230.700,  230.954,  231.219,  231.496,  231.784,  232.089,  232.416,  232.764,  233.127,  233.508, 
      233.909,  234.328,  234.762,  235.213,  235.679,  236.163, ],
    [ 238.922,  238.430,  238.014,  237.639,  237.277,  236.926,  236.586,  236.256,  235.938,  235.632, 
      235.339,  235.060,  234.794,  234.543,  234.309,  234.092,  233.891,  233.707,  233.539,  233.388, 
      233.253,  233.135,  233.034,  232.947,  232.872,  232.807,  232.670,  232.535,  232.365,  232.193, 
      232.025,  231.842,  231.649,  231.446,  231.237,  231.040,  230.843,  230.649,  230.464,  230.287, 
      230.116,  229.961,  229.814,  229.680,  229.554,  229.441,  229.342,  229.256,  229.154,  229.049, 
      228.936,  228.826,  228.718,  228.613,  228.518,  228.445,  228.320,  228.200,  228.113,  228.052, 
      228.039,  228.015,  227.989,  227.989,  228.034,  228.094,  228.168,  228.254,  228.357,  228.477, 
      228.623,  228.781,  228.934,  229.115,  229.336,  229.573,  229.805,  230.023,  230.226,  230.415, 
      230.596,  230.768,  230.932,  231.090,  231.241,  231.398,  231.570,  231.749,  231.930,  232.126, 
      232.346,  232.584,  232.835,  233.097,  233.370,  233.660,  233.973,  234.306,  234.656,  235.023, 
      235.411,  235.816,  236.238,  236.675,  237.128,  237.599, ],
    [ 241.161,  240.669,  240.253,  239.879,  239.517,  239.165,  238.825,  238.496,  238.177,  237.871, 
      237.578,  237.299,  237.033,  236.783,  236.548,  236.331,  236.130,  235.946,  235.778,  235.627, 
      235.492,  235.374,  235.272,  235.186,  235.111,  235.045,  234.907,  234.752,  234.578,  234.405, 
      234.232,  234.042,  233.842,  233.629,  233.420,  233.219,  233.018,  232.821,  232.634,  232.453, 
      232.279,  232.121,  231.970,  231.833,  231.704,  231.589,  231.489,  231.397,  231.289,  231.176, 
      231.057,  230.942,  230.831,  230.720,  230.622,  230.546,  230.411,  230.280,  230.187,  230.115, 
      230.094,  230.058,  230.021,  230.011,  230.046,  230.095,  230.158,  230.232,  230.324,  230.432, 
      230.567,  230.713,  230.853,  231.022,  231.232,  231.458,  231.676,  231.881,  232.068,  232.243, 
      232.407,  232.562,  232.710,  232.851,  232.985,  233.125,  233.280,  233.444,  233.608,  233.788, 
      233.993,  234.216,  234.452,  234.699,  234.957,  235.233,  235.532,  235.851,  236.187,  236.541, 
      236.915,  237.307,  237.716,  238.140,  238.581,  239.039, ],
    [ 243.400,  242.908,  242.492,  242.118,  241.756,  241.404,  241.064,  240.735,  240.416,  240.110, 
      239.817,  239.538,  239.272,  239.021,  238.787,  238.570,  238.369,  238.185,  238.017,  237.865, 
      237.731,  237.613,  237.511,  237.424,  237.349,  237.281,  237.143,  236.969,  236.791,  236.617, 
      236.438,  236.241,  236.033,  235.812,  235.602,  235.395,  235.191,  234.992,  234.803,  234.616, 
      234.442,  234.279,  234.127,  233.986,  233.855,  233.737,  233.635,  233.535,  233.423,  233.303, 
      233.179,  233.059,  232.941,  232.828,  232.727,  232.645,  232.502,  232.361,  232.259,  232.178, 
      232.149,  232.101,  232.053,  232.032,  232.058,  232.096,  232.148,  232.211,  232.291,  232.388, 
      232.511,  232.646,  232.773,  232.930,  233.129,  233.343,  233.549,  233.739,  233.912,  234.071, 
      234.220,  234.358,  234.489,  234.613,  234.730,  234.853,  234.993,  235.139,  235.287,  235.452, 
      235.642,  235.850,  236.071,  236.303,  236.547,  236.808,  237.093,  237.398,  237.721,  238.061, 
      238.422,  238.801,  239.197,  239.609,  240.036,  240.482, ],
    [ 245.639,  245.147,  244.732,  244.357,  243.995,  243.644,  243.303,  242.974,  242.655,  242.349, 
      242.056,  241.777,  241.511,  241.260,  241.026,  240.808,  240.608,  240.424,  240.255,  240.104, 
      239.969,  239.851,  239.749,  239.662,  239.585,  239.518,  239.367,  239.184,  239.004,  238.830, 
      238.641,  238.437,  238.217,  237.995,  237.783,  237.572,  237.365,  237.162,  236.969,  236.779, 
      236.604,  236.437,  236.283,  236.138,  236.005,  235.884,  235.780,  235.673,  235.556,  235.430, 
      235.300,  235.173,  235.051,  234.934,  234.831,  234.744,  234.593,  234.442,  234.331,  234.241, 
      234.204,  234.144,  234.084,  234.053,  234.070,  234.098,  234.138,  234.190,  234.259,  234.344, 
      234.457,  234.580,  234.694,  234.839,  235.027,  235.230,  235.423,  235.600,  235.757,  235.901, 
      236.033,  236.155,  236.270,  236.376,  236.476,  236.583,  236.706,  236.837,  236.968,  237.117, 
      237.292,  237.485,  237.692,  237.909,  238.138,  238.385,  238.656,  238.948,  239.257,  239.584, 
      239.932,  240.298,  240.681,  241.080,  241.495,  241.928, ],
    [ 247.879,  247.386,  246.971,  246.596,  246.234,  245.882,  245.542,  245.213,  244.894,  244.588, 
      244.295,  244.016,  243.750,  243.499,  243.265,  243.047,  242.846,  242.662,  242.494,  242.342, 
      242.208,  242.089,  241.987,  241.899,  241.822,  241.754,  241.584,  241.397,  241.216,  241.038, 
      240.841,  240.630,  240.400,  240.178,  239.961,  239.746,  239.536,  239.333,  239.134,  238.942, 
      238.763,  238.594,  238.438,  238.290,  238.154,  238.031,  237.924,  237.809,  237.687,  237.553, 
      237.421,  237.287,  237.159,  237.038,  236.934,  236.843,  236.681,  236.522,  236.402,  236.304, 
      236.259,  236.186,  236.115,  236.074,  236.082,  236.099,  236.129,  236.169,  236.227,  236.301, 
      236.403,  236.514,  236.616,  236.749,  236.926,  237.118,  237.298,  237.461,  237.604,  237.731, 
      237.848,  237.953,  238.051,  238.140,  238.223,  238.313,  238.420,  238.535,  238.650,  238.783, 
      238.944,  239.122,  239.314,  239.517,  239.731,  239.964,  240.221,  240.500,  240.795,  241.109, 
      241.444,  241.797,  242.167,  242.554,  242.956,  243.377, ],
    [ 250.118,  249.625,  249.210,  248.835,  248.473,  248.121,  247.781,  247.451,  247.133,  246.827, 
      246.534,  246.254,  245.989,  245.738,  245.503,  245.286,  245.085,  244.901,  244.732,  244.581, 
      244.446,  244.328,  244.224,  244.135,  244.058,  243.990,  243.802,  243.610,  243.428,  243.244, 
      243.040,  242.817,  242.583,  242.359,  242.138,  241.920,  241.706,  241.501,  241.297,  241.104, 
      240.922,  240.751,  240.591,  240.440,  240.302,  240.178,  240.067,  239.944,  239.814,  239.675, 
      239.537,  239.401,  239.266,  239.142,  239.036,  238.942,  238.769,  238.600,  238.472,  238.367, 
      238.312,  238.228,  238.146,  238.095,  238.094,  238.100,  238.120,  238.149,  238.196,  238.258, 
      238.350,  238.449,  238.538,  238.660,  238.827,  239.006,  239.174,  239.323,  239.451,  239.563, 
      239.663,  239.752,  239.833,  239.905,  239.970,  240.044,  240.135,  240.234,  240.333,  240.451, 
      240.596,  240.761,  240.938,  241.126,  241.326,  241.544,  241.788,  242.053,  242.336,  242.636, 
      242.958,  243.299,  243.656,  244.030,  244.421,  244.829, ],
    [ 252.356,  251.864,  251.449,  251.074,  250.712,  250.360,  250.020,  249.690,  249.372,  249.066, 
      248.772,  248.493,  248.227,  247.976,  247.742,  247.524,  247.323,  247.139,  246.971,  246.819, 
      246.684,  246.564,  246.461,  246.372,  246.295,  246.221,  246.017,  245.823,  245.641,  245.446, 
      245.234,  245.000,  244.766,  244.539,  244.314,  244.093,  243.877,  243.667,  243.460,  243.267, 
      243.080,  242.907,  242.744,  242.591,  242.450,  242.324,  242.208,  242.079,  241.941,  241.796, 
      241.654,  241.513,  241.374,  241.246,  241.137,  241.041,  240.857,  240.678,  240.543,  240.430, 
      240.366,  240.269,  240.177,  240.116,  240.106,  240.102,  240.111,  240.128,  240.165,  240.216, 
      240.297,  240.384,  240.461,  240.572,  240.728,  240.896,  241.052,  241.186,  241.299,  241.395, 
      241.479,  241.552,  241.615,  241.670,  241.718,  241.775,  241.851,  241.934,  242.016,  242.119, 
      242.250,  242.401,  242.563,  242.737,  242.922,  243.127,  243.357,  243.609,  243.878,  244.166, 
      244.475,  244.803,  245.148,  245.510,  245.888,  246.284, ],
    [ 254.595,  254.103,  253.687,  253.312,  252.950,  252.599,  252.258,  251.929,  251.611,  251.304, 
      251.011,  250.732,  250.466,  250.215,  249.980,  249.763,  249.562,  249.378,  249.209,  249.056, 
      248.920,  248.801,  248.697,  248.608,  248.528,  248.438,  248.230,  248.035,  247.849,  247.646, 
      247.425,  247.183,  246.948,  246.716,  246.487,  246.263,  246.047,  245.832,  245.623,  245.427, 
      245.238,  245.063,  244.896,  244.741,  244.598,  244.471,  244.346,  244.212,  244.068,  243.917, 
      243.768,  243.622,  243.481,  243.350,  243.238,  243.138,  242.945,  242.756,  242.613,  242.492, 
      242.418,  242.310,  242.207,  242.137,  242.118,  242.103,  242.102,  242.108,  242.134,  242.174, 
      242.244,  242.320,  242.384,  242.484,  242.630,  242.787,  242.930,  243.050,  243.148,  243.228, 
      243.296,  243.352,  243.398,  243.436,  243.467,  243.507,  243.568,  243.634,  243.701,  243.788, 
      243.905,  244.041,  244.190,  244.349,  244.520,  244.711,  244.927,  245.166,  245.422,  245.697, 
      245.994,  246.309,  246.642,  246.991,  247.357,  247.741, ],
    [ 256.834,  256.342,  255.926,  255.551,  255.189,  254.838,  254.497,  254.167,  253.849,  253.543, 
      253.250,  252.970,  252.704,  252.453,  252.219,  252.001,  251.800,  251.615,  251.446,  251.293, 
      251.157,  251.037,  250.933,  250.842,  250.745,  250.655,  250.443,  250.248,  250.055,  249.843, 
      249.609,  249.366,  249.129,  248.893,  248.660,  248.434,  248.214,  247.995,  247.786,  247.586, 
      247.394,  247.217,  247.047,  246.890,  246.745,  246.616,  246.483,  246.343,  246.192,  246.038, 
      245.882,  245.730,  245.585,  245.454,  245.337,  245.235,  245.030,  244.834,  244.681,  244.554, 
      244.471,  244.350,  244.236,  244.157,  244.131,  244.105,  244.093,  244.088,  244.103,  244.132, 
      244.192,  244.257,  244.307,  244.396,  244.532,  244.679,  244.809,  244.915,  244.997,  245.061, 
      245.113,  245.152,  245.182,  245.202,  245.215,  245.240,  245.285,  245.335,  245.385,  245.458, 
      245.561,  245.683,  245.817,  245.963,  246.119,  246.296,  246.500,  246.725,  246.969,  247.231, 
      247.515,  247.818,  248.138,  248.475,  248.829,  249.201, ],
    [ 259.073,  258.580,  258.165,  257.790,  257.428,  257.076,  256.736,  256.406,  256.088,  255.781, 
      255.488,  255.209,  254.943,  254.692,  254.457,  254.238,  254.037,  253.851,  253.682,  253.529, 
      253.393,  253.273,  253.166,  253.060,  252.962,  252.869,  252.656,  252.460,  252.256,  252.036, 
      251.792,  251.549,  251.307,  251.067,  250.832,  250.604,  250.380,  250.158,  249.948,  249.743, 
      249.550,  249.370,  249.197,  249.038,  248.892,  248.759,  248.618,  248.471,  248.313,  248.154, 
      247.996,  247.838,  247.689,  247.558,  247.436,  247.331,  247.115,  246.912,  246.750,  246.617, 
      246.522,  246.389,  246.266,  246.177,  246.143,  246.106,  246.084,  246.068,  246.073,  246.090, 
      246.141,  246.194,  246.231,  246.310,  246.436,  246.571,  246.688,  246.781,  246.847,  246.895, 
      246.931,  246.953,  246.965,  246.969,  246.964,  246.972,  247.002,  247.037,  247.071,  247.129, 
      247.218,  247.326,  247.446,  247.577,  247.719,  247.882,  248.073,  248.286,  248.516,  248.766, 
      249.038,  249.328,  249.637,  249.961,  250.303,  250.664, ],
    [ 261.311,  260.819,  260.403,  260.028,  259.666,  259.315,  258.974,  258.644,  258.326,  258.020, 
      257.726,  257.447,  257.180,  256.929,  256.693,  256.475,  256.273,  256.088,  255.918,  255.766, 
      255.630,  255.503,  255.383,  255.276,  255.176,  255.082,  254.868,  254.668,  254.455,  254.222, 
      253.975,  253.730,  253.483,  253.241,  253.003,  252.774,  252.544,  252.320,  252.108,  251.901, 
      251.707,  251.522,  251.348,  251.186,  251.038,  250.902,  250.753,  250.598,  250.434,  250.270, 
      250.107,  249.945,  249.793,  249.659,  249.535,  249.425,  249.199,  248.990,  248.818,  248.678, 
      248.573,  248.428,  248.295,  248.197,  248.155,  248.107,  248.075,  248.048,  248.042,  248.049, 
      248.089,  248.131,  248.156,  248.224,  248.340,  248.464,  248.569,  248.647,  248.698,  248.730, 
      248.749,  248.754,  248.749,  248.735,  248.712,  248.705,  248.720,  248.739,  248.756,  248.800, 
      248.875,  248.969,  249.075,  249.192,  249.320,  249.470,  249.648,  249.848,  250.066,  250.303, 
      250.562,  250.841,  251.137,  251.450,  251.779,  252.128, ],
    [ 263.550,  263.057,  262.642,  262.267,  261.904,  261.553,  261.212,  260.883,  260.564,  260.257, 
      259.963,  259.683,  259.417,  259.165,  258.930,  258.711,  258.509,  258.324,  258.155,  258.002, 
      257.853,  257.720,  257.600,  257.490,  257.389,  257.296,  257.080,  256.873,  256.650,  256.405, 
      256.158,  255.909,  255.659,  255.413,  255.173,  254.939,  254.707,  254.483,  254.266,  254.057, 
      253.861,  253.673,  253.498,  253.334,  253.185,  253.040,  252.886,  252.724,  252.555,  252.384, 
      252.216,  252.052,  251.898,  251.761,  251.635,  251.519,  251.284,  251.067,  250.885,  250.740, 
      250.624,  250.466,  250.324,  250.217,  250.167,  250.107,  250.066,  250.028,  250.012,  250.007, 
      250.038,  250.068,  250.080,  250.138,  250.244,  250.358,  250.450,  250.513,  250.548,  250.564, 
      250.567,  250.555,  250.532,  250.501,  250.461,  250.438,  250.438,  250.440,  250.442,  250.472, 
      250.533,  250.614,  250.706,  250.809,  250.923,  251.059,  251.224,  251.412,  251.617,  251.841, 
      252.089,  252.355,  252.639,  252.940,  253.258,  253.595, ],
    [ 265.788,  265.296,  264.880,  264.505,  264.143,  263.791,  263.450,  263.119,  262.801,  262.494, 
      262.200,  261.920,  261.653,  261.402,  261.166,  260.948,  260.746,  260.561,  260.383,  260.219, 
      260.070,  259.935,  259.813,  259.703,  259.603,  259.508,  259.293,  259.073,  258.839,  258.588, 
      258.339,  258.086,  257.832,  257.584,  257.344,  257.104,  256.869,  256.644,  256.424,  256.214, 
      256.014,  255.824,  255.646,  255.481,  255.331,  255.177,  255.016,  254.845,  254.673,  254.498, 
      254.324,  254.156,  254.002,  253.861,  253.733,  253.610,  253.364,  253.142,  252.951,  252.802, 
      252.673,  252.503,  252.351,  252.237,  252.178,  252.108,  252.056,  252.007,  251.982,  251.966, 
      251.987,  252.005,  252.005,  252.052,  252.149,  252.252,  252.331,  252.380,  252.399,  252.399, 
      252.385,  252.355,  252.316,  252.266,  252.209,  252.171,  252.156,  252.142,  252.128,  252.143, 
      252.192,  252.259,  252.337,  252.426,  252.526,  252.649,  252.802,  252.977,  253.169,  253.381, 
      253.617,  253.871,  254.143,  254.433,  254.739,  255.064, ],
    [ 268.026,  267.533,  267.117,  266.742,  266.379,  266.027,  265.686,  265.356,  265.037,  264.730, 
      264.436,  264.156,  263.889,  263.638,  263.403,  263.184,  262.973,  262.779,  262.600,  262.435, 
      262.285,  262.149,  262.026,  261.916,  261.815,  261.720,  261.499,  261.270,  261.022,  260.771, 
      260.519,  260.262,  260.006,  259.754,  259.510,  259.267,  259.032,  258.804,  258.581,  258.370, 
      258.166,  257.974,  257.794,  257.628,  257.474,  257.312,  257.143,  256.966,  256.790,  256.610, 
      256.431,  256.260,  256.105,  255.960,  255.829,  255.701,  255.445,  255.217,  255.017,  254.862, 
      254.722,  254.539,  254.379,  254.257,  254.189,  254.108,  254.046,  253.987,  253.951,  253.925, 
      253.936,  253.943,  253.929,  253.967,  254.055,  254.147,  254.213,  254.247,  254.250,  254.234, 
      254.203,  254.156,  254.099,  254.030,  253.958,  253.903,  253.873,  253.844,  253.814,  253.816, 
      253.851,  253.904,  253.968,  254.043,  254.130,  254.240,  254.380,  254.543,  254.723,  254.923, 
      255.146,  255.389,  255.649,  255.927,  256.221,  256.536, ],
    [ 270.263,  269.770,  269.354,  268.978,  268.616,  268.264,  267.923,  267.592,  267.273,  266.967, 
      266.673,  266.393,  266.126,  265.870,  265.627,  265.401,  265.190,  264.995,  264.815,  264.649, 
      264.498,  264.362,  264.239,  264.129,  264.027,  263.933,  263.701,  263.462,  263.205,  262.953, 
      262.696,  262.435,  262.176,  261.925,  261.676,  261.430,  261.194,  260.962,  260.737,  260.523, 
      260.317,  260.124,  259.942,  259.774,  259.615,  259.446,  259.270,  259.087,  258.904,  258.720, 
      258.539,  258.364,  258.207,  258.059,  257.925,  257.791,  257.526,  257.291,  257.083,  256.922, 
      256.770,  256.575,  256.407,  256.277,  256.200,  256.107,  256.036,  255.966,  255.921,  255.883, 
      255.886,  255.881,  255.854,  255.882,  255.961,  256.042,  256.095,  256.114,  256.101,  256.068, 
      256.020,  255.956,  255.881,  255.795,  255.705,  255.635,  255.591,  255.546,  255.500,  255.488, 
      255.510,  255.550,  255.600,  255.662,  255.734,  255.831,  255.959,  256.110,  256.278,  256.466, 
      256.677,  256.908,  257.157,  257.423,  257.706,  258.009, ],
    [ 272.499,  272.006,  271.590,  271.215,  270.852,  270.500,  270.159,  269.829,  269.510,  269.203, 
      268.904,  268.618,  268.345,  268.087,  267.844,  267.617,  267.405,  267.209,  267.028,  266.862, 
      266.711,  266.575,  266.452,  266.341,  266.240,  266.143,  265.901,  265.645,  265.388,  265.133, 
      264.872,  264.609,  264.347,  264.092,  263.839,  263.593,  263.354,  263.119,  262.894,  262.676, 
      262.468,  262.273,  262.089,  261.921,  261.753,  261.578,  261.392,  261.207,  261.018,  260.827, 
      260.643,  260.468,  260.308,  260.158,  260.019,  259.879,  259.606,  259.364,  259.147,  258.982, 
      258.818,  258.609,  258.433,  258.296,  258.209,  258.107,  258.026,  257.944,  257.890,  257.842, 
      257.835,  257.818,  257.778,  257.798,  257.868,  257.938,  257.977,  257.982,  257.952,  257.902, 
      257.837,  257.755,  257.663,  257.559,  257.451,  257.367,  257.308,  257.247,  257.185,  257.160, 
      257.170,  257.196,  257.233,  257.280,  257.339,  257.424,  257.539,  257.678,  257.834,  258.010, 
      258.210,  258.429,  258.666,  258.921,  259.192,  259.484, ],
    [ 274.735,  274.243,  273.827,  273.451,  273.088,  272.736,  272.393,  272.058,  271.734,  271.421, 
      271.121,  270.835,  270.561,  270.302,  270.058,  269.830,  269.618,  269.422,  269.241,  269.075, 
      268.924,  268.787,  268.664,  268.553,  268.449,  268.347,  268.094,  267.828,  267.571,  267.310, 
      267.046,  266.780,  266.517,  266.258,  266.002,  265.756,  265.513,  265.276,  265.049,  264.828, 
      264.618,  264.421,  264.236,  264.065,  263.888,  263.705,  263.513,  263.323,  263.130,  262.935, 
      262.747,  262.572,  262.408,  262.257,  262.110,  261.966,  261.684,  261.437,  261.210,  261.041, 
      260.863,  260.642,  260.458,  260.315,  260.218,  260.107,  260.015,  259.922,  259.859,  259.800, 
      259.784,  259.755,  259.703,  259.713,  259.775,  259.834,  259.860,  259.849,  259.802,  259.736, 
      259.653,  259.554,  259.444,  259.321,  259.197,  259.098,  259.025,  258.947,  258.870,  258.832, 
      258.829,  258.842,  258.865,  258.899,  258.945,  259.017,  259.121,  259.247,  259.391,  259.555, 
      259.743,  259.951,  260.177,  260.420,  260.680,  260.961, ],
    [ 276.972,  276.479,  276.061,  275.681,  275.313,  274.957,  274.610,  274.275,  273.950,  273.637, 
      273.337,  273.049,  272.775,  272.515,  272.271,  272.044,  271.832,  271.635,  271.453,  271.287, 
      271.136,  270.999,  270.876,  270.762,  270.653,  270.546,  270.278,  270.011,  269.751,  269.487, 
      269.219,  268.950,  268.686,  268.422,  268.164,  267.916,  267.670,  267.432,  267.202,  266.979, 
      266.768,  266.569,  266.383,  266.208,  266.023,  265.831,  265.634,  265.437,  265.239,  265.042, 
      264.851,  264.675,  264.507,  264.353,  264.202,  264.052,  263.762,  263.508,  263.274,  263.101, 
      262.906,  262.675,  262.482,  262.334,  262.227,  262.107,  262.004,  261.900,  261.828,  261.758, 
      261.734,  261.691,  261.627,  261.629,  261.682,  261.730,  261.743,  261.716,  261.651,  261.569, 
      261.469,  261.352,  261.224,  261.082,  260.943,  260.828,  260.742,  260.646,  260.555,  260.504, 
      260.489,  260.488,  260.498,  260.519,  260.551,  260.610,  260.702,  260.817,  260.949,  261.101, 
      261.278,  261.475,  261.689,  261.921,  262.170,  262.439, ],
    [ 279.194,  278.698,  278.278,  277.898,  277.530,  277.173,  276.826,  276.490,  276.164,  275.851, 
      275.550,  275.263,  274.988,  274.729,  274.485,  274.256,  274.044,  273.848,  273.666,  273.500, 
      273.348,  273.211,  273.084,  272.964,  272.852,  272.741,  272.461,  272.194,  271.928,  271.660, 
      271.390,  271.121,  270.851,  270.585,  270.327,  270.075,  269.827,  269.588,  269.354,  269.129, 
      268.916,  268.716,  268.529,  268.346,  268.154,  267.953,  267.753,  267.551,  267.347,  267.147, 
      266.955,  266.777,  266.607,  266.449,  266.291,  266.137,  265.839,  265.579,  265.338,  265.159, 
      264.949,  264.707,  264.506,  264.352,  264.236,  264.106,  263.991,  263.878,  263.796,  263.716, 
      263.683,  263.627,  263.552,  263.545,  263.589,  263.627,  263.625,  263.583,  263.500,  263.401, 
      263.284,  263.148,  263.002,  262.842,  262.687,  262.559,  262.457,  262.345,  262.239,  262.176, 
      262.149,  262.135,  262.131,  262.138,  262.157,  262.204,  262.285,  262.388,  262.508,  262.649, 
      262.814,  262.999,  263.202,  263.423,  263.661,  263.919, ],
    [ 281.411,  280.914,  280.494,  280.113,  279.744,  279.387,  279.039,  278.703,  278.378,  278.064, 
      277.763,  277.476,  277.201,  276.941,  276.697,  276.469,  276.256,  276.060,  275.878,  275.712, 
      275.558,  275.416,  275.284,  275.163,  275.044,  274.925,  274.644,  274.375,  274.105,  273.834, 
      273.561,  273.290,  273.015,  272.747,  272.488,  272.233,  271.984,  271.742,  271.506,  271.280, 
      271.064,  270.863,  270.675,  270.481,  270.281,  270.074,  269.869,  269.662,  269.454,  269.251, 
      269.059,  268.877,  268.706,  268.542,  268.379,  268.221,  267.917,  267.647,  267.402,  267.216, 
      266.990,  266.740,  266.530,  266.370,  266.245,  266.103,  265.979,  265.855,  265.763,  265.673, 
      265.632,  265.561,  265.476,  265.461,  265.497,  265.524,  265.507,  265.448,  265.349,  265.231, 
      265.097,  264.944,  264.779,  264.601,  264.430,  264.289,  264.172,  264.043,  263.923,  263.847, 
      263.808,  263.781,  263.764,  263.758,  263.764,  263.799,  263.868,  263.959,  264.067,  264.197, 
      264.351,  264.525,  264.717,  264.927,  265.153,  265.401, ],
    [ 283.625,  283.128,  282.707,  282.326,  281.958,  281.600,  281.253,  280.916,  280.591,  280.277, 
      279.976,  279.688,  279.414,  279.154,  278.909,  278.681,  278.469,  278.272,  278.088,  277.919, 
      277.760,  277.616,  277.481,  277.352,  277.227,  277.108,  276.827,  276.552,  276.279,  276.005, 
      275.731,  275.455,  275.178,  274.910,  274.648,  274.390,  274.140,  273.895,  273.656,  273.429, 
      273.212,  273.010,  272.817,  272.616,  272.407,  272.195,  271.983,  271.771,  271.561,  271.355, 
      271.163,  270.976,  270.803,  270.633,  270.467,  270.302,  269.995,  269.716,  269.465,  269.272, 
      269.029,  268.773,  268.553,  268.387,  268.252,  268.100,  267.966,  267.833,  267.730,  267.631, 
      267.579,  267.495,  267.400,  267.377,  267.405,  267.420,  267.389,  267.313,  267.198,  267.062, 
      266.909,  266.739,  266.555,  266.358,  266.172,  266.018,  265.884,  265.740,  265.606,  265.519, 
      265.467,  265.427,  265.397,  265.378,  265.370,  265.393,  265.451,  265.531,  265.628,  265.746, 
      265.889,  266.052,  266.233,  266.432,  266.647,  266.884, ],
    [ 285.838,  285.341,  284.920,  284.539,  284.171,  283.813,  283.465,  283.129,  282.803,  282.489, 
      282.188,  281.901,  281.626,  281.366,  281.122,  280.893,  280.679,  280.479,  280.292,  280.119, 
      279.959,  279.810,  279.667,  279.535,  279.410,  279.291,  279.008,  278.729,  278.452,  278.175, 
      277.900,  277.619,  277.341,  277.072,  276.805,  276.546,  276.294,  276.046,  275.807,  275.577, 
      275.359,  275.156,  274.955,  274.746,  274.528,  274.313,  274.097,  273.879,  273.665,  273.459, 
      273.264,  273.075,  272.899,  272.724,  272.551,  272.383,  272.071,  271.783,  271.528,  271.326, 
      271.068,  270.803,  270.575,  270.403,  270.258,  270.097,  269.952,  269.811,  269.697,  269.588, 
      269.526,  269.427,  269.324,  269.293,  269.313,  269.317,  269.270,  269.178,  269.044,  268.891, 
      268.720,  268.532,  268.329,  268.115,  267.913,  267.745,  267.596,  267.435,  267.288,  267.190, 
      267.126,  267.073,  267.030,  266.997,  266.977,  266.988,  267.035,  267.104,  267.189,  267.295, 
      267.428,  267.580,  267.750,  267.938,  268.143,  268.369, ],
    [ 288.051,  287.554,  287.133,  286.752,  286.383,  286.025,  285.678,  285.341,  285.016,  284.702, 
      284.401,  284.113,  283.838,  283.577,  283.330,  283.099,  282.881,  282.679,  282.491,  282.314, 
      282.147,  281.993,  281.850,  281.718,  281.593,  281.474,  281.185,  280.903,  280.624,  280.346, 
      280.066,  279.782,  279.504,  279.231,  278.963,  278.703,  278.447,  278.197,  277.957,  277.725, 
      277.506,  277.301,  277.090,  276.872,  276.649,  276.428,  276.207,  275.986,  275.769,  275.563, 
      275.364,  275.175,  274.994,  274.812,  274.636,  274.464,  274.146,  273.849,  273.590,  273.377, 
      273.104,  272.832,  272.596,  272.420,  272.263,  272.093,  271.937,  271.788,  271.663,  271.546, 
      271.473,  271.360,  271.248,  271.209,  271.221,  271.214,  271.152,  271.042,  270.889,  270.719, 
      270.529,  270.322,  270.103,  269.869,  269.653,  269.473,  269.308,  269.129,  268.970,  268.861, 
      268.784,  268.718,  268.663,  268.617,  268.583,  268.583,  268.619,  268.676,  268.750,  268.846, 
      268.967,  269.109,  269.268,  269.445,  269.639,  269.855, ],
    [ 290.264,  289.767,  289.345,  288.964,  288.596,  288.238,  287.890,  287.554,  287.228,  286.914, 
      286.611,  286.322,  286.045,  285.781,  285.532,  285.299,  285.080,  284.875,  284.679,  284.498, 
      284.330,  284.176,  284.033,  283.901,  283.776,  283.655,  283.362,  283.076,  282.794,  282.515, 
      282.230,  281.945,  281.665,  281.389,  281.119,  280.858,  280.599,  280.348,  280.105,  279.873, 
      279.653,  279.442,  279.224,  278.996,  278.769,  278.542,  278.314,  278.091,  277.873,  277.666, 
      277.463,  277.272,  277.085,  276.900,  276.718,  276.542,  276.219,  275.914,  275.653,  275.427, 
      275.140,  274.861,  274.617,  274.437,  274.268,  274.090,  273.922,  273.763,  273.628,  273.503, 
      273.420,  273.292,  273.172,  273.124,  273.129,  273.109,  273.032,  272.903,  272.733,  272.545, 
      272.336,  272.111,  271.872,  271.622,  271.391,  271.200,  271.018,  270.823,  270.652,  270.531, 
      270.442,  270.364,  270.294,  270.236,  270.190,  270.178,  270.204,  270.250,  270.312,  270.397, 
      270.508,  270.638,  270.787,  270.953,  271.137,  271.342, ],
    [ 292.476,  291.979,  291.558,  291.176,  290.808,  290.450,  290.102,  289.763,  289.436,  289.120, 
      288.815,  288.523,  288.245,  287.980,  287.729,  287.491,  287.267,  287.058,  286.862,  286.681, 
      286.513,  286.359,  286.216,  286.083,  285.955,  285.832,  285.535,  285.247,  284.965,  284.681, 
      284.393,  284.108,  283.825,  283.547,  283.275,  283.011,  282.750,  282.498,  282.253,  282.020, 
      281.799,  281.578,  281.351,  281.117,  280.885,  280.654,  280.422,  280.196,  279.977,  279.767, 
      279.563,  279.369,  279.176,  278.985,  278.799,  278.620,  278.292,  277.978,  277.714,  277.475, 
      277.174,  276.888,  276.639,  276.453,  276.273,  276.085,  275.906,  275.736,  275.592,  275.461, 
      275.367,  275.223,  275.096,  275.040,  275.037,  275.005,  274.911,  274.763,  274.576,  274.369, 
      274.141,  273.898,  273.641,  273.372,  273.128,  272.924,  272.725,  272.514,  272.332,  272.201, 
      272.101,  272.009,  271.926,  271.854,  271.796,  271.773,  271.788,  271.824,  271.875,  271.949, 
      272.049,  272.169,  272.307,  272.462,  272.635,  272.831, ],
    [ 294.688,  294.191,  293.769,  293.386,  293.016,  292.656,  292.306,  291.966,  291.637,  291.319, 
      291.014,  290.720,  290.438,  290.168,  289.914,  289.674,  289.450,  289.241,  289.045,  288.864, 
      288.696,  288.541,  288.397,  288.261,  288.132,  288.009,  287.709,  287.418,  287.133,  286.844, 
      286.556,  286.269,  285.983,  285.703,  285.431,  285.163,  284.901,  284.647,  284.401,  284.166, 
      283.943,  283.713,  283.476,  283.238,  282.999,  282.762,  282.529,  282.300,  282.081,  281.867, 
      281.662,  281.463,  281.263,  281.069,  280.879,  280.698,  280.363,  280.042,  279.776,  279.520, 
      279.206,  278.915,  278.659,  278.469,  278.277,  278.078,  277.889,  277.710,  277.556,  277.418, 
      277.313,  277.153,  277.019,  276.955,  276.945,  276.900,  276.789,  276.623,  276.417,  276.191, 
      275.945,  275.682,  275.405,  275.121,  274.864,  274.647,  274.433,  274.204,  274.011,  273.869, 
      273.757,  273.654,  273.557,  273.472,  273.403,  273.368,  273.373,  273.397,  273.437,  273.501, 
      273.591,  273.700,  273.827,  273.972,  274.135,  274.320, ],
    [ 296.896,  296.397,  295.973,  295.589,  295.217,  294.856,  294.505,  294.164,  293.833,  293.511, 
      293.202,  292.905,  292.621,  292.352,  292.097,  291.858,  291.633,  291.424,  291.228,  291.046, 
      290.878,  290.721,  290.574,  290.438,  290.308,  290.182,  289.880,  289.588,  289.299,  289.007, 
      288.718,  288.429,  288.140,  287.859,  287.584,  287.314,  287.051,  286.795,  286.548,  286.313, 
      286.082,  285.843,  285.598,  285.355,  285.113,  284.870,  284.633,  284.404,  284.183,  283.966, 
      283.759,  283.554,  283.351,  283.152,  282.958,  282.776,  282.433,  282.106,  281.836,  281.563, 
      281.239,  280.940,  280.679,  280.484,  280.278,  280.072,  279.871,  279.682,  279.519,  279.375, 
      279.258,  279.082,  278.941,  278.871,  278.853,  278.795,  278.667,  278.480,  278.257,  278.011, 
      277.746,  277.465,  277.167,  276.867,  276.599,  276.368,  276.138,  275.894,  275.689,  275.537, 
      275.414,  275.297,  275.188,  275.089,  275.009,  274.963,  274.958,  274.972,  275.000,  275.053, 
      275.133,  275.232,  275.349,  275.483,  275.636,  275.811, ],
    [ 299.096,  298.596,  298.172,  297.787,  297.414,  297.049,  296.694,  296.350,  296.016,  295.694, 
      295.385,  295.088,  294.804,  294.535,  294.280,  294.041,  293.816,  293.606,  293.410,  293.226, 
      293.056,  292.897,  292.751,  292.613,  292.481,  292.355,  292.050,  291.757,  291.463,  291.170, 
      290.880,  290.586,  290.296,  290.014,  289.736,  289.464,  289.200,  288.943,  288.695,  288.459, 
      288.217,  287.970,  287.719,  287.469,  287.222,  286.977,  286.737,  286.508,  286.284,  286.065, 
      285.855,  285.644,  285.436,  285.232,  285.036,  284.854,  284.501,  284.170,  283.895,  283.603, 
      283.272,  282.964,  282.699,  282.496,  282.278,  282.065,  281.854,  281.655,  281.482,  281.330, 
      281.202,  281.011,  280.862,  280.786,  280.760,  280.689,  280.543,  280.336,  280.092,  279.831, 
      279.544,  279.243,  278.926,  278.612,  278.331,  278.088,  277.841,  277.583,  277.367,  277.205, 
      277.070,  276.941,  276.818,  276.705,  276.614,  276.558,  276.543,  276.546,  276.564,  276.606, 
      276.676,  276.764,  276.871,  276.995,  277.137,  277.302, ],
    [ 301.290,  300.788,  300.360,  299.973,  299.597,  299.232,  298.877,  298.533,  298.200,  297.878, 
      297.568,  297.271,  296.987,  296.718,  296.463,  296.222,  295.997,  295.786,  295.588,  295.403, 
      295.232,  295.073,  294.924,  294.786,  294.654,  294.526,  294.221,  293.922,  293.625,  293.333, 
      293.039,  292.744,  292.453,  292.168,  291.887,  291.615,  291.348,  291.090,  290.841,  290.599, 
      290.349,  290.091,  289.837,  289.583,  289.329,  289.082,  288.841,  288.611,  288.383,  288.164, 
      287.948,  287.732,  287.521,  287.313,  287.114,  286.929,  286.568,  286.233,  285.955,  285.642, 
      285.304,  284.988,  284.719,  284.509,  284.277,  284.058,  283.834,  283.628,  283.443,  283.285, 
      283.145,  282.940,  282.784,  282.701,  282.668,  282.582,  282.418,  282.189,  281.927,  281.646, 
      281.341,  281.020,  280.683,  280.355,  280.062,  279.807,  279.540,  279.270,  279.044,  278.873, 
      278.724,  278.583,  278.447,  278.322,  278.218,  278.153,  278.128,  278.120,  278.127,  278.159, 
      278.219,  278.297,  278.394,  278.508,  278.640,  278.795, ],
    [ 303.473,  302.971,  302.543,  302.156,  301.780,  301.415,  301.060,  300.716,  300.383,  300.061, 
      299.751,  299.454,  299.169,  298.899,  298.643,  298.401,  298.175,  297.963,  297.764,  297.579, 
      297.406,  297.246,  297.098,  296.958,  296.824,  296.696,  296.389,  296.086,  295.788,  295.494, 
      295.197,  294.900,  294.608,  294.320,  294.038,  293.764,  293.496,  293.237,  292.988,  292.735, 
      292.476,  292.212,  291.952,  291.694,  291.437,  291.186,  290.945,  290.712,  290.483,  290.261, 
      290.039,  289.819,  289.602,  289.391,  289.192,  289.003,  288.634,  288.296,  288.012,  287.679, 
      287.334,  287.011,  286.738,  286.518,  286.277,  286.047,  285.813,  285.601,  285.403,  285.240, 
      285.087,  284.868,  284.706,  284.617,  284.575,  284.474,  284.291,  284.040,  283.760,  283.458, 
      283.135,  282.794,  282.439,  282.097,  281.792,  281.523,  281.237,  280.954,  280.720,  280.539, 
      280.379,  280.224,  280.076,  279.938,  279.823,  279.748,  279.713,  279.694,  279.691,  279.713, 
      279.762,  279.831,  279.917,  280.021,  280.143,  280.288, ],
    [ 305.656,  305.154,  304.727,  304.339,  303.963,  303.598,  303.243,  302.899,  302.565,  302.242, 
      301.932,  301.634,  301.348,  301.077,  300.820,  300.578,  300.351,  300.138,  299.938,  299.752, 
      299.580,  299.418,  299.268,  299.128,  298.995,  298.867,  298.554,  298.248,  297.951,  297.653, 
      297.354,  297.056,  296.761,  296.471,  296.188,  295.912,  295.644,  295.383,  295.130,  294.868, 
      294.598,  294.332,  294.066,  293.801,  293.542,  293.290,  293.049,  292.812,  292.582,  292.356, 
      292.128,  291.903,  291.682,  291.469,  291.269,  291.076,  290.699,  290.359,  290.065,  289.714, 
      289.363,  289.032,  288.756,  288.527,  288.277,  288.035,  287.791,  287.572,  287.364,  287.195, 
      287.029,  286.795,  286.627,  286.532,  286.482,  286.366,  286.161,  285.889,  285.589,  285.267, 
      284.924,  284.564,  284.191,  283.836,  283.521,  283.235,  282.932,  282.638,  282.396,  282.204, 
      282.032,  281.864,  281.704,  281.553,  281.427,  281.342,  281.297,  281.269,  281.254,  281.266, 
      281.306,  281.364,  281.441,  281.535,  281.647,  281.782, ],
    [ 307.839,  307.337,  306.910,  306.521,  306.146,  305.780,  305.425,  305.080,  304.745,  304.421, 
      304.109,  303.811,  303.525,  303.253,  302.996,  302.753,  302.525,  302.312,  302.112,  301.925, 
      301.751,  301.589,  301.439,  301.299,  301.165,  301.034,  300.717,  300.411,  300.112,  299.810, 
      299.510,  299.211,  298.913,  298.621,  298.338,  298.060,  297.791,  297.530,  297.266,  296.995, 
      296.719,  296.447,  296.177,  295.909,  295.647,  295.394,  295.151,  294.911,  294.680,  294.447, 
      294.215,  293.986,  293.762,  293.547,  293.344,  293.148,  292.763,  292.421,  292.113,  291.747, 
      291.391,  291.053,  290.774,  290.534,  290.274,  290.022,  289.769,  289.543,  289.323,  289.149, 
      288.968,  288.722,  288.549,  288.447,  288.389,  288.255,  288.029,  287.735,  287.417,  287.073, 
      286.710,  286.330,  285.940,  285.574,  285.249,  284.945,  284.625,  284.321,  284.069,  283.868, 
      283.683,  283.502,  283.329,  283.166,  283.030,  282.937,  282.882,  282.843,  282.818,  282.820, 
      282.850,  282.899,  282.965,  283.049,  283.151,  283.277, ],
    [ 310.021,  309.519,  309.091,  308.702,  308.325,  307.959,  307.602,  307.257,  306.921,  306.598, 
      306.286,  305.987,  305.700,  305.427,  305.169,  304.926,  304.698,  304.484,  304.283,  304.095, 
      303.921,  303.760,  303.609,  303.467,  303.331,  303.198,  302.880,  302.574,  302.270,  301.967, 
      301.666,  301.364,  301.064,  300.772,  300.486,  300.208,  299.937,  299.673,  299.400,  299.117, 
      298.838,  298.561,  298.284,  298.014,  297.751,  297.498,  297.251,  297.010,  296.775,  296.537, 
      296.301,  296.067,  295.840,  295.625,  295.418,  295.217,  294.827,  294.483,  294.158,  293.779, 
      293.418,  293.075,  292.791,  292.539,  292.270,  292.008,  291.746,  291.514,  291.282,  291.102, 
      290.900,  290.648,  290.469,  290.363,  290.294,  290.142,  289.893,  289.579,  289.240,  288.876, 
      288.493,  288.091,  287.687,  287.310,  286.974,  286.653,  286.316,  286.003,  285.742,  285.530, 
      285.334,  285.140,  284.955,  284.778,  284.633,  284.531,  284.466,  284.417,  284.381,  284.374, 
      284.395,  284.433,  284.490,  284.564,  284.656,  284.772, ],
    [ 312.200,  311.697,  311.268,  310.879,  310.502,  310.135,  309.779,  309.433,  309.097,  308.772, 
      308.460,  308.160,  307.873,  307.600,  307.342,  307.098,  306.869,  306.654,  306.453,  306.266, 
      306.092,  305.929,  305.776,  305.633,  305.494,  305.361,  305.043,  304.734,  304.428,  304.124, 
      303.821,  303.516,  303.215,  302.921,  302.634,  302.355,  302.084,  301.809,  301.527,  301.238, 
      300.953,  300.672,  300.392,  300.119,  299.855,  299.600,  299.350,  299.109,  298.868,  298.625, 
      298.385,  298.147,  297.918,  297.702,  297.492,  297.286,  296.891,  296.545,  296.200,  295.812, 
      295.442,  295.095,  294.807,  294.545,  294.267,  293.993,  293.724,  293.481,  293.241,  293.055, 
      292.832,  292.573,  292.390,  292.278,  292.199,  292.027,  291.754,  291.420,  291.060,  290.676, 
      290.271,  289.849,  289.432,  289.044,  288.696,  288.359,  288.006,  287.683,  287.414,  287.192, 
      286.982,  286.777,  286.580,  286.390,  286.236,  286.124,  286.050,  285.991,  285.944,  285.927, 
      285.939,  285.968,  286.015,  286.079,  286.161,  286.269, ],
    [ 314.377,  313.874,  313.445,  313.055,  312.677,  312.310,  311.953,  311.606,  311.270,  310.945, 
      310.633,  310.333,  310.045,  309.771,  309.512,  309.268,  309.039,  308.825,  308.624,  308.435, 
      308.259,  308.094,  307.940,  307.795,  307.657,  307.524,  307.205,  306.892,  306.585,  306.280, 
      305.974,  305.667,  305.365,  305.069,  304.781,  304.501,  304.228,  303.943,  303.649,  303.358, 
      303.067,  302.779,  302.498,  302.223,  301.959,  301.701,  301.450,  301.206,  300.959,  300.711, 
      300.466,  300.226,  299.996,  299.777,  299.563,  299.352,  298.955,  298.605,  298.238,  297.845, 
      297.466,  297.115,  296.824,  296.549,  296.262,  295.976,  295.699,  295.446,  295.199,  295.005, 
      294.762,  294.497,  294.310,  294.193,  294.104,  293.911,  293.611,  293.256,  292.877,  292.472, 
      292.047,  291.605,  291.174,  290.775,  290.416,  290.060,  289.694,  289.361,  289.085,  288.852, 
      288.631,  288.413,  288.202,  288.002,  287.838,  287.717,  287.633,  287.563,  287.507,  287.481, 
      287.484,  287.503,  287.540,  287.595,  287.667,  287.765, ],
    [ 316.552,  316.048,  315.618,  315.228,  314.851,  314.483,  314.126,  313.779,  313.442,  313.117, 
      312.804,  312.503,  312.216,  311.942,  311.683,  311.439,  311.210,  310.994,  310.790,  310.601, 
      310.423,  310.258,  310.103,  309.958,  309.820,  309.687,  309.365,  309.050,  308.741,  308.434, 
      308.125,  307.817,  307.514,  307.217,  306.928,  306.648,  306.363,  306.070,  305.770,  305.472, 
      305.178,  304.887,  304.602,  304.327,  304.061,  303.801,  303.549,  303.299,  303.047,  302.796, 
      302.546,  302.304,  302.074,  301.850,  301.632,  301.418,  301.018,  300.664,  300.275,  299.875, 
      299.490,  299.135,  298.841,  298.549,  298.255,  297.959,  297.672,  297.410,  297.156,  296.955, 
      296.692,  296.421,  296.229,  296.108,  296.009,  295.790,  295.465,  295.090,  294.689,  294.263, 
      293.818,  293.358,  292.914,  292.505,  292.135,  291.757,  291.380,  291.039,  290.754,  290.511, 
      290.277,  290.046,  289.824,  289.612,  289.439,  289.310,  289.216,  289.136,  289.070,  289.035, 
      289.029,  289.038,  289.065,  289.111,  289.174,  289.263, ],
    [ 318.725,  318.221,  317.792,  317.401,  317.023,  316.655,  316.297,  315.949,  315.613,  315.287, 
      314.974,  314.674,  314.386,  314.112,  313.852,  313.607,  313.376,  313.159,  312.955,  312.764, 
      312.586,  312.421,  312.266,  312.121,  311.983,  311.848,  311.522,  311.206,  310.897,  310.586, 
      310.276,  309.968,  309.662,  309.364,  309.075,  308.792,  308.496,  308.191,  307.888,  307.586, 
      307.285,  306.992,  306.706,  306.431,  306.162,  305.900,  305.646,  305.390,  305.133,  304.877, 
      304.625,  304.382,  304.149,  303.923,  303.700,  303.483,  303.081,  302.722,  302.309,  301.904, 
      301.511,  301.155,  300.855,  300.549,  300.249,  299.941,  299.644,  299.373,  299.114,  298.903, 
      298.621,  298.345,  298.147,  298.023,  297.911,  297.667,  297.315,  296.919,  296.496,  296.050, 
      295.582,  295.107,  294.652,  294.234,  293.847,  293.451,  293.064,  292.717,  292.423,  292.169, 
      291.922,  291.678,  291.445,  291.222,  291.039,  290.902,  290.799,  290.708,  290.633,  290.588, 
      290.573,  290.573,  290.591,  290.627,  290.681,  290.760, ],
    [ 320.897,  320.392,  319.962,  319.572,  319.193,  318.825,  318.467,  318.120,  317.783,  317.458, 
      317.145,  316.843,  316.554,  316.279,  316.018,  315.772,  315.540,  315.322,  315.118,  314.927, 
      314.749,  314.583,  314.429,  314.283,  314.143,  314.007,  313.680,  313.363,  313.050,  312.738, 
      312.426,  312.116,  311.810,  311.511,  311.221,  310.927,  310.622,  310.312,  310.003,  309.696, 
      309.393,  309.097,  308.810,  308.534,  308.262,  307.999,  307.740,  307.478,  307.218,  306.958, 
      306.703,  306.460,  306.223,  305.993,  305.766,  305.547,  305.143,  304.771,  304.342,  303.931, 
      303.532,  303.174,  302.866,  302.548,  302.239,  301.920,  301.617,  301.335,  301.072,  300.850, 
      300.549,  300.269,  300.066,  299.937,  299.812,  299.540,  299.162,  298.745,  298.299,  297.832, 
      297.342,  296.853,  296.389,  295.962,  295.556,  295.143,  294.747,  294.392,  294.091,  293.825, 
      293.566,  293.310,  293.063,  292.830,  292.640,  292.494,  292.382,  292.280,  292.195,  292.141, 
      292.118,  292.108,  292.117,  292.143,  292.188,  292.259, ],
    [ 323.067,  322.563,  322.133,  321.742,  321.364,  320.996,  320.638,  320.290,  319.952,  319.625, 
      319.311,  319.009,  318.719,  318.443,  318.181,  317.935,  317.703,  317.485,  317.281,  317.090, 
      316.912,  316.746,  316.590,  316.443,  316.302,  316.165,  315.836,  315.518,  315.203,  314.888, 
      314.577,  314.264,  313.957,  313.658,  313.364,  313.058,  312.743,  312.429,  312.116,  311.803, 
      311.498,  311.201,  310.914,  310.635,  310.361,  310.096,  309.831,  309.565,  309.300,  309.037, 
      308.781,  308.536,  308.297,  308.061,  307.831,  307.611,  307.205,  306.814,  306.374,  305.958, 
      305.554,  305.192,  304.874,  304.547,  304.226,  303.897,  303.590,  303.295,  303.029,  302.793, 
      302.477,  302.193,  301.984,  301.851,  301.711,  301.408,  301.004,  300.565,  300.098,  299.609, 
      299.099,  298.596,  298.123,  297.685,  297.263,  296.833,  296.428,  296.066,  295.758,  295.480, 
      295.209,  294.940,  294.679,  294.438,  294.238,  294.085,  293.963,  293.853,  293.758,  293.695, 
      293.662,  293.643,  293.643,  293.660,  293.695,  293.757, ],
    [ 325.238,  324.733,  324.303,  323.912,  323.532,  323.163,  322.804,  322.455,  322.117,  321.790, 
      321.475,  321.172,  320.882,  320.606,  320.344,  320.098,  319.866,  319.648,  319.443,  319.253, 
      319.073,  318.906,  318.749,  318.601,  318.459,  318.322,  317.992,  317.671,  317.354,  317.039, 
      316.725,  316.412,  316.104,  315.804,  315.499,  315.182,  314.864,  314.543,  314.225,  313.911, 
      313.602,  313.305,  313.017,  312.734,  312.460,  312.191,  311.919,  311.650,  311.381,  311.115, 
      310.859,  310.611,  310.368,  310.128,  309.895,  309.675,  309.267,  308.853,  308.407,  307.982, 
      307.575,  307.210,  306.880,  306.544,  306.213,  305.875,  305.562,  305.256,  304.986,  304.733, 
      304.404,  304.115,  303.902,  303.764,  303.605,  303.269,  302.842,  302.383,  301.893,  301.382, 
      300.853,  300.338,  299.853,  299.406,  298.965,  298.522,  298.108,  297.739,  297.422,  297.133, 
      296.848,  296.568,  296.295,  296.044,  295.836,  295.676,  295.544,  295.423,  295.320,  295.248, 
      295.207,  295.178,  295.169,  295.176,  295.202,  295.256, ],
    [ 327.405,  326.900,  326.469,  326.077,  325.698,  325.328,  324.968,  324.619,  324.280,  323.953, 
      323.638,  323.335,  323.045,  322.769,  322.507,  322.260,  322.028,  321.811,  321.605,  321.413, 
      321.232,  321.064,  320.907,  320.758,  320.616,  320.478,  320.147,  319.823,  319.504,  319.188, 
      318.873,  318.559,  318.251,  317.945,  317.628,  317.304,  316.980,  316.656,  316.332,  316.015, 
      315.706,  315.409,  315.118,  314.833,  314.558,  314.282,  314.007,  313.732,  313.460,  313.193, 
      312.936,  312.684,  312.436,  312.194,  311.959,  311.738,  311.327,  310.890,  310.438,  310.005, 
      309.595,  309.226,  308.886,  308.541,  308.197,  307.853,  307.532,  307.215,  306.941,  306.665, 
      306.330,  306.037,  305.819,  305.678,  305.496,  305.125,  304.676,  304.192,  303.681,  303.146, 
      302.602,  302.077,  301.583,  301.124,  300.661,  300.208,  299.786,  299.410,  299.084,  298.784, 
      298.487,  298.194,  297.909,  297.649,  297.433,  297.267,  297.124,  296.993,  296.882,  296.800, 
      296.751,  296.713,  296.695,  296.693,  296.710,  296.756, ],
    [ 329.570,  329.065,  328.633,  328.241,  327.861,  327.491,  327.131,  326.782,  326.443,  326.116, 
      325.800,  325.498,  325.208,  324.931,  324.670,  324.422,  324.189,  323.970,  323.764,  323.570, 
      323.390,  323.222,  323.063,  322.914,  322.772,  322.635,  322.300,  321.974,  321.655,  321.337, 
      321.020,  320.706,  320.397,  320.080,  319.751,  319.424,  319.094,  318.764,  318.439,  318.119, 
      317.810,  317.511,  317.217,  316.932,  316.652,  316.371,  316.091,  315.813,  315.538,  315.271, 
      315.011,  314.756,  314.503,  314.258,  314.023,  313.801,  313.386,  312.924,  312.468,  312.028, 
      311.615,  311.243,  310.889,  310.535,  310.181,  309.829,  309.500,  309.174,  308.896,  308.595, 
      308.254,  307.958,  307.737,  307.592,  307.381,  306.976,  306.505,  305.997,  305.463,  304.906, 
      304.348,  303.813,  303.311,  302.835,  302.354,  301.891,  301.464,  301.081,  300.745,  300.432, 
      300.123,  299.819,  299.521,  299.254,  299.031,  298.856,  298.704,  298.563,  298.443,  298.353, 
      298.295,  298.249,  298.220,  298.210,  298.218,  298.255, ],
    [ 331.733,  331.228,  330.796,  330.404,  330.024,  329.654,  329.294,  328.945,  328.606,  328.278, 
      327.963,  327.661,  327.370,  327.093,  326.830,  326.582,  326.348,  326.128,  325.921,  325.728, 
      325.547,  325.378,  325.220,  325.071,  324.928,  324.788,  324.452,  324.125,  323.804,  323.485, 
      323.168,  322.853,  322.536,  322.207,  321.872,  321.538,  321.204,  320.871,  320.543,  320.224, 
      319.914,  319.612,  319.316,  319.030,  318.743,  318.458,  318.173,  317.893,  317.616,  317.349, 
      317.085,  316.825,  316.569,  316.322,  316.086,  315.863,  315.440,  314.956,  314.495,  314.049, 
      313.635,  313.259,  312.889,  312.529,  312.163,  311.802,  311.464,  311.132,  310.849,  310.525, 
      310.178,  309.880,  309.654,  309.504,  309.260,  308.823,  308.327,  307.796,  307.240,  306.662, 
      306.091,  305.548,  305.037,  304.543,  304.044,  303.574,  303.140,  302.750,  302.405,  302.079, 
      301.758,  301.441,  301.133,  300.857,  300.626,  300.444,  300.283,  300.133,  300.004,  299.906, 
      299.838,  299.783,  299.746,  299.727,  299.726,  299.755, ],
    [ 333.896,  333.390,  332.959,  332.566,  332.186,  331.816,  331.457,  331.107,  330.768,  330.440, 
      330.124,  329.821,  329.529,  329.251,  328.988,  328.739,  328.505,  328.285,  328.078,  327.884, 
      327.703,  327.534,  327.376,  327.225,  327.081,  326.941,  326.602,  326.275,  325.952,  325.632, 
      325.314,  324.998,  324.668,  324.328,  323.990,  323.652,  323.312,  322.977,  322.647,  322.328, 
      322.016,  321.711,  321.415,  321.124,  320.832,  320.542,  320.254,  319.971,  319.694,  319.424, 
      319.157,  318.892,  318.634,  318.386,  318.150,  317.926,  317.482,  316.989,  316.521,  316.071, 
      315.654,  315.273,  314.888,  314.522,  314.143,  313.774,  313.427,  313.089,  312.801,  312.453, 
      312.102,  311.801,  311.570,  311.415,  311.134,  310.664,  310.146,  309.590,  309.011,  308.414, 
      307.832,  307.278,  306.759,  306.247,  305.734,  305.255,  304.815,  304.418,  304.062,  303.725, 
      303.390,  303.062,  302.743,  302.459,  302.221,  302.031,  301.862,  301.701,  301.564,  301.458, 
      301.382,  301.317,  301.271,  301.244,  301.234,  301.255, ],
    [ 336.059,  335.553,  335.122,  334.729,  334.349,  333.978,  333.618,  333.268,  332.928,  332.599, 
      332.282,  331.978,  331.687,  331.409,  331.145,  330.896,  330.662,  330.442,  330.234,  330.041, 
      329.859,  329.689,  329.529,  329.378,  329.233,  329.092,  328.753,  328.423,  328.100,  327.779, 
      327.461,  327.134,  326.792,  326.449,  326.104,  325.760,  325.419,  325.081,  324.752,  324.432, 
      324.117,  323.810,  323.513,  323.215,  322.919,  322.624,  322.334,  322.049,  321.772,  321.497, 
      321.226,  320.958,  320.698,  320.449,  320.212,  319.987,  319.520,  319.022,  318.545,  318.092, 
      317.672,  317.282,  316.888,  316.510,  316.121,  315.747,  315.388,  315.047,  314.751,  314.381, 
      314.027,  313.721,  313.486,  313.326,  312.996,  312.499,  311.955,  311.377,  310.773,  310.161, 
      309.570,  309.008,  308.477,  307.943,  307.421,  306.934,  306.488,  306.085,  305.718,  305.368, 
      305.021,  304.679,  304.352,  304.061,  303.815,  303.618,  303.439,  303.269,  303.123,  303.010, 
      302.925,  302.851,  302.796,  302.761,  302.742,  302.755, ],
    [ 338.221,  337.714,  337.282,  336.889,  336.508,  336.137,  335.776,  335.425,  335.085,  334.757, 
      334.440,  334.136,  333.844,  333.565,  333.302,  333.053,  332.818,  332.598,  332.390,  332.195, 
      332.013,  331.842,  331.681,  331.529,  331.384,  331.242,  330.903,  330.572,  330.247,  329.926, 
      329.604,  329.264,  328.913,  328.564,  328.216,  327.868,  327.524,  327.185,  326.856,  326.533, 
      326.216,  325.910,  325.606,  325.303,  325.003,  324.705,  324.412,  324.126,  323.847,  323.569, 
      323.294,  323.023,  322.762,  322.513,  322.275,  322.048,  321.556,  321.052,  320.569,  320.112, 
      319.689,  319.288,  318.885,  318.496,  318.099,  317.720,  317.349,  317.004,  316.696,  316.308, 
      315.951,  315.640,  315.401,  315.235,  314.850,  314.328,  313.759,  313.157,  312.530,  311.906, 
      311.306,  310.736,  310.189,  309.636,  309.105,  308.612,  308.160,  307.749,  307.371,  307.008, 
      306.650,  306.296,  305.961,  305.662,  305.410,  305.205,  305.016,  304.836,  304.683,  304.562, 
      304.467,  304.385,  304.321,  304.277,  304.250,  304.255, ],
    [ 340.380,  339.873,  339.441,  339.047,  338.666,  338.295,  337.934,  337.583,  337.243,  336.913, 
      336.596,  336.292,  336.000,  335.722,  335.458,  335.209,  334.974,  334.752,  334.544,  334.348, 
      334.165,  333.993,  333.832,  333.680,  333.534,  333.393,  333.051,  332.719,  332.394,  332.072, 
      331.738,  331.386,  331.033,  330.678,  330.324,  329.974,  329.628,  329.289,  328.959,  328.633, 
      328.315,  328.006,  327.697,  327.389,  327.084,  326.784,  326.490,  326.204,  325.920,  325.638, 
      325.360,  325.087,  324.826,  324.576,  324.337,  324.108,  323.588,  323.081,  322.592,  322.132, 
      321.705,  321.293,  320.881,  320.480,  320.076,  319.691,  319.308,  318.962,  318.629,  318.234, 
      317.873,  317.559,  317.317,  317.140,  316.698,  316.150,  315.556,  314.931,  314.284,  313.648, 
      313.039,  312.461,  311.896,  311.326,  310.787,  310.289,  309.830,  309.411,  309.022,  308.646, 
      308.275,  307.910,  307.567,  307.262,  307.003,  306.790,  306.591,  306.402,  306.241,  306.112, 
      306.010,  305.919,  305.846,  305.794,  305.758,  305.755, ],
    [ 342.537,  342.031,  341.598,  341.205,  340.823,  340.452,  340.090,  339.739,  339.399,  339.070, 
      338.753,  338.448,  338.156,  337.877,  337.613,  337.363,  337.127,  336.905,  336.696,  336.500, 
      336.316,  336.144,  335.982,  335.830,  335.684,  335.542,  335.199,  334.866,  334.540,  334.211, 
      333.863,  333.507,  333.147,  332.787,  332.431,  332.078,  331.732,  331.393,  331.060,  330.732, 
      330.414,  330.099,  329.784,  329.474,  329.165,  328.862,  328.567,  328.279,  327.992,  327.706, 
      327.425,  327.151,  326.889,  326.638,  326.398,  326.157,  325.621,  325.109,  324.613,  324.152, 
      323.722,  323.294,  322.876,  322.463,  322.052,  321.659,  321.267,  320.918,  320.559,  320.159, 
      319.794,  319.477,  319.232,  319.034,  318.539,  317.966,  317.346,  316.694,  316.032,  315.387, 
      314.769,  314.182,  313.597,  313.015,  312.469,  311.965,  311.499,  311.071,  310.670,  310.282, 
      309.900,  309.522,  309.172,  308.860,  308.596,  308.374,  308.167,  307.968,  307.799,  307.662, 
      307.552,  307.453,  307.371,  307.310,  307.266,  307.255, ],
    [ 344.694,  344.188,  343.755,  343.361,  342.979,  342.608,  342.247,  341.896,  341.555,  341.226, 
      340.908,  340.603,  340.310,  340.031,  339.765,  339.515,  339.279,  339.057,  338.847,  338.650, 
      338.466,  338.294,  338.133,  337.980,  337.832,  337.690,  337.346,  337.013,  336.687,  336.343, 
      335.984,  335.623,  335.259,  334.895,  334.536,  334.183,  333.836,  333.496,  333.159,  332.831, 
      332.510,  332.188,  331.870,  331.554,  331.244,  330.940,  330.645,  330.352,  330.061,  329.772, 
      329.488,  329.215,  328.952,  328.700,  328.458,  328.196,  327.654,  327.133,  326.634,  326.171, 
      325.738,  325.294,  324.870,  324.445,  324.025,  323.623,  323.226,  322.873,  322.488,  322.083, 
      321.716,  321.395,  321.147,  320.908,  320.374,  319.772,  319.127,  318.452,  317.777,  317.124, 
      316.498,  315.897,  315.291,  314.701,  314.148,  313.638,  313.167,  312.729,  312.315,  311.915, 
      311.521,  311.134,  310.776,  310.457,  310.187,  309.957,  309.740,  309.533,  309.356,  309.212, 
      309.093,  308.986,  308.896,  308.826,  308.774,  308.756, ],
    [ 346.851,  346.344,  345.911,  345.517,  345.135,  344.764,  344.402,  344.050,  343.709,  343.379, 
      343.061,  342.755,  342.462,  342.182,  341.917,  341.666,  341.430,  341.207,  340.997,  340.801, 
      340.617,  340.444,  340.281,  340.128,  339.981,  339.838,  339.493,  339.160,  338.822,  338.465, 
      338.104,  337.737,  337.366,  337.002,  336.640,  336.287,  335.940,  335.596,  335.258,  334.929, 
      334.601,  334.276,  333.953,  333.635,  333.322,  333.018,  332.719,  332.424,  332.128,  331.837, 
      331.552,  331.278,  331.015,  330.762,  330.510,  330.231,  329.685,  329.157,  328.655,  328.189, 
      327.748,  327.294,  326.860,  326.423,  325.997,  325.586,  325.183,  324.827,  324.416,  324.007, 
      323.638,  323.313,  323.063,  322.765,  322.201,  321.572,  320.903,  320.206,  319.519,  318.857, 
      318.225,  317.606,  316.982,  316.384,  315.826,  315.310,  314.833,  314.385,  313.960,  313.547, 
      313.140,  312.744,  312.378,  312.054,  311.778,  311.539,  311.312,  311.096,  310.912,  310.762, 
      310.634,  310.518,  310.420,  310.341,  310.281,  310.256, ],
    [ 349.007,  348.500,  348.066,  347.672,  347.289,  346.917,  346.555,  346.203,  345.862,  345.531, 
      345.213,  344.907,  344.613,  344.333,  344.067,  343.817,  343.580,  343.357,  343.148,  342.950, 
      342.765,  342.592,  342.429,  342.276,  342.128,  341.985,  341.640,  341.302,  340.948,  340.586, 
      340.218,  339.846,  339.474,  339.106,  338.744,  338.391,  338.041,  337.695,  337.358,  337.024, 
      336.690,  336.361,  336.034,  335.713,  335.400,  335.094,  334.793,  334.493,  334.194,  333.901, 
      333.616,  333.341,  333.077,  332.822,  332.549,  332.264,  331.714,  331.180,  330.675,  330.206, 
      329.754,  329.291,  328.846,  328.401,  327.970,  327.546,  327.141,  326.779,  326.344,  325.931, 
      325.558,  325.230,  324.978,  324.613,  324.021,  323.363,  322.666,  321.954,  321.259,  320.588, 
      319.947,  319.310,  318.672,  318.066,  317.503,  316.981,  316.496,  316.038,  315.600,  315.176, 
      314.756,  314.352,  313.981,  313.649,  313.369,  313.120,  312.883,  312.659,  312.468,  312.311, 
      312.175,  312.050,  311.943,  311.857,  311.789,  311.756, ],
    [ 351.160,  350.653,  350.219,  349.824,  349.442,  349.069,  348.707,  348.355,  348.013,  347.682, 
      347.364,  347.058,  346.764,  346.484,  346.218,  345.967,  345.730,  345.507,  345.296,  345.099, 
      344.913,  344.740,  344.577,  344.423,  344.275,  344.132,  343.786,  343.434,  343.069,  342.702, 
      342.330,  341.953,  341.579,  341.210,  340.848,  340.493,  340.141,  339.794,  339.455,  339.114, 
      338.777,  338.443,  338.114,  337.791,  337.478,  337.169,  336.864,  336.560,  336.259,  335.965, 
      335.679,  335.404,  335.139,  334.875,  334.584,  334.297,  333.742,  333.202,  332.695,  332.223, 
      331.759,  331.288,  330.831,  330.378,  329.942,  329.507,  329.099,  328.719,  328.270,  327.856, 
      327.479,  327.147,  326.893,  326.452,  325.829,  325.145,  324.423,  323.699,  322.995,  322.318, 
      321.664,  321.004,  320.358,  319.746,  319.178,  318.650,  318.157,  317.687,  317.238,  316.802, 
      316.371,  315.960,  315.581,  315.244,  314.956,  314.699,  314.453,  314.222,  314.023,  313.859, 
      313.714,  313.582,  313.466,  313.372,  313.296,  313.256, ],
    [ 353.312,  352.805,  352.371,  351.976,  351.593,  351.220,  350.858,  350.505,  350.163,  349.833, 
      349.514,  349.208,  348.914,  348.634,  348.367,  348.116,  347.878,  347.655,  347.444,  347.247, 
      347.061,  346.887,  346.724,  346.570,  346.422,  346.278,  345.924,  345.556,  345.188,  344.815, 
      344.437,  344.060,  343.683,  343.314,  342.952,  342.594,  342.240,  341.893,  341.547,  341.202, 
      340.861,  340.523,  340.192,  339.869,  339.554,  339.242,  338.932,  338.625,  338.323,  338.028, 
      337.743,  337.466,  337.199,  336.914,  336.617,  336.329,  335.767,  335.223,  334.715,  334.240, 
      333.759,  333.282,  332.814,  332.354,  331.912,  331.466,  331.056,  330.649,  330.195,  329.777, 
      329.398,  329.064,  328.806,  328.284,  327.629,  326.919,  326.176,  325.441,  324.729,  324.045, 
      323.374,  322.695,  322.042,  321.425,  320.851,  320.318,  319.816,  319.334,  318.873,  318.425, 
      317.983,  317.565,  317.181,  316.838,  316.544,  316.279,  316.023,  315.784,  315.577,  315.407, 
      315.253,  315.113,  314.989,  314.887,  314.804,  314.756, ],
    [ 355.463,  354.956,  354.522,  354.126,  353.743,  353.371,  353.008,  352.656,  352.314,  351.983, 
      351.664,  351.357,  351.063,  350.782,  350.516,  350.264,  350.026,  349.803,  349.592,  349.394, 
      349.208,  349.035,  348.871,  348.716,  348.568,  348.422,  348.050,  347.678,  347.302,  346.924, 
      346.544,  346.164,  345.787,  345.418,  345.054,  344.693,  344.339,  343.988,  343.636,  343.287, 
      342.941,  342.602,  342.270,  341.947,  341.627,  341.312,  340.999,  340.690,  340.387,  340.092, 
      339.805,  339.528,  339.252,  338.948,  338.649,  338.360,  337.791,  337.244,  336.734,  336.255, 
      335.759,  335.276,  334.795,  334.327,  333.876,  333.424,  333.012,  332.578,  332.119,  331.699, 
      331.316,  330.979,  330.712,  330.106,  329.420,  328.680,  327.923,  327.180,  326.459,  325.767, 
      325.077,  324.385,  323.725,  323.102,  322.523,  321.984,  321.472,  320.979,  320.505,  320.047, 
      319.594,  319.169,  318.779,  318.432,  318.131,  317.856,  317.591,  317.345,  317.131,  316.954, 
      316.791,  316.643,  316.511,  316.401,  316.311,  316.256, ],
    [ 357.614,  357.106,  356.672,  356.277,  355.894,  355.521,  355.158,  354.805,  354.463,  354.131, 
      353.812,  353.505,  353.211,  352.930,  352.664,  352.412,  352.174,  351.950,  351.739,  351.541, 
      351.355,  351.181,  351.017,  350.862,  350.708,  350.554,  350.171,  349.793,  349.413,  349.031, 
      348.649,  348.268,  347.891,  347.521,  347.153,  346.792,  346.437,  346.079,  345.723,  345.370, 
      345.022,  344.680,  344.348,  344.022,  343.700,  343.380,  343.064,  342.754,  342.451,  342.155, 
      341.867,  341.588,  341.289,  340.981,  340.682,  340.389,  339.814,  339.264,  338.752,  338.264, 
      337.758,  337.265,  336.773,  336.300,  335.838,  335.382,  334.967,  334.506,  334.044,  333.621, 
      333.235,  332.894,  332.557,  331.916,  331.200,  330.436,  329.667,  328.916,  328.188,  327.484, 
      326.771,  326.072,  325.405,  324.778,  324.193,  323.647,  323.126,  322.621,  322.136,  321.663, 
      321.204,  320.772,  320.377,  320.024,  319.716,  319.432,  319.158,  318.906,  318.684,  318.501, 
      318.328,  318.173,  318.033,  317.914,  317.817,  317.756, ],
    [ 359.764,  359.256,  358.822,  358.426,  358.043,  357.669,  357.306,  356.953,  356.611,  356.279, 
      355.960,  355.653,  355.359,  355.078,  354.811,  354.559,  354.321,  354.097,  353.886,  353.687, 
      353.502,  353.327,  353.158,  352.995,  352.834,  352.676,  352.290,  351.906,  351.521,  351.137, 
      350.753,  350.372,  349.995,  349.621,  349.253,  348.891,  348.529,  348.167,  347.808,  347.451, 
      347.100,  346.758,  346.425,  346.095,  345.769,  345.446,  345.128,  344.817,  344.514,  344.217, 
      343.929,  343.638,  343.323,  343.013,  342.713,  342.417,  341.835,  341.284,  340.768,  340.269, 
      339.755,  339.250,  338.750,  338.273,  337.799,  337.340,  336.921,  336.433,  335.968,  335.542, 
      335.152,  334.810,  334.393,  333.716,  332.971,  332.187,  331.408,  330.648,  329.915,  329.192, 
      328.461,  327.756,  327.084,  326.452,  325.861,  325.308,  324.775,  324.259,  323.762,  323.279, 
      322.812,  322.374,  321.973,  321.615,  321.299,  321.007,  320.724,  320.465,  320.237,  320.046, 
      319.866,  319.702,  319.554,  319.427,  319.324,  319.256, ],
    [ 361.913,  361.404,  360.970,  360.574,  360.191,  359.818,  359.454,  359.101,  358.759,  358.427, 
      358.108,  357.801,  357.506,  357.225,  356.958,  356.706,  356.467,  356.243,  356.032,  355.834, 
      355.642,  355.461,  355.286,  355.117,  354.955,  354.796,  354.404,  354.014,  353.628,  353.241, 
      352.857,  352.476,  352.096,  351.721,  351.352,  350.987,  350.619,  350.253,  349.889,  349.530, 
      349.178,  348.836,  348.499,  348.167,  347.837,  347.511,  347.192,  346.881,  346.577,  346.279, 
      345.989,  345.675,  345.355,  345.046,  344.741,  344.441,  343.856,  343.304,  342.785,  342.270, 
      341.751,  341.234,  340.728,  340.244,  339.759,  339.297,  338.859,  338.359,  337.892,  337.463, 
      337.069,  336.725,  336.216,  335.506,  334.729,  333.933,  333.145,  332.378,  331.636,  330.892, 
      330.151,  329.438,  328.762,  328.124,  327.530,  326.966,  326.422,  325.895,  325.386,  324.891, 
      324.419,  323.975,  323.567,  323.207,  322.882,  322.579,  322.289,  322.023,  321.790,  321.591, 
      321.402,  321.231,  321.075,  320.940,  320.830,  320.755, ],
    [ 364.061,  363.553,  363.118,  362.722,  362.339,  361.965,  361.602,  361.249,  360.906,  360.574, 
      360.255,  359.948,  359.653,  359.371,  359.104,  358.852,  358.614,  358.388,  358.172,  357.966, 
      357.769,  357.583,  357.407,  357.238,  357.073,  356.911,  356.513,  356.121,  355.732,  355.345, 
      354.961,  354.578,  354.196,  353.820,  353.449,  353.078,  352.706,  352.336,  351.969,  351.608, 
      351.256,  350.912,  350.572,  350.236,  349.903,  349.575,  349.256,  348.944,  348.639,  348.341, 
      348.035,  347.708,  347.388,  347.076,  346.768,  346.465,  345.877,  345.323,  344.801,  344.270, 
      343.745,  343.216,  342.702,  342.212,  341.718,  341.255,  340.788,  340.284,  339.814,  339.382, 
      338.987,  338.641,  338.027,  337.283,  336.483,  335.675,  334.881,  334.107,  333.351,  332.585, 
      331.837,  331.119,  330.438,  329.795,  329.194,  328.622,  328.067,  327.526,  327.008,  326.502, 
      326.023,  325.575,  325.162,  324.795,  324.463,  324.151,  323.852,  323.581,  323.342,  323.135, 
      322.938,  322.758,  322.596,  322.453,  322.336,  322.254, ],
    [ 366.208,  365.700,  365.265,  364.870,  364.486,  364.112,  363.749,  363.396,  363.053,  362.721, 
      362.401,  362.094,  361.799,  361.518,  361.250,  360.993,  360.750,  360.518,  360.297,  360.088, 
      359.890,  359.704,  359.525,  359.353,  359.187,  359.023,  358.621,  358.227,  357.836,  357.450, 
      357.065,  356.678,  356.295,  355.919,  355.542,  355.165,  354.790,  354.416,  354.047,  353.686, 
      353.334,  352.986,  352.642,  352.302,  351.967,  351.639,  351.319,  351.007,  350.701,  350.401, 
      350.070,  349.740,  349.421,  349.104,  348.792,  348.487,  347.898,  347.341,  346.812,  346.270, 
      345.735,  345.194,  344.675,  344.175,  343.676,  343.211,  342.717,  342.208,  341.735,  341.301, 
      340.904,  340.535,  339.826,  339.046,  338.231,  337.414,  336.611,  335.832,  335.058,  334.276, 
      333.520,  332.797,  332.112,  331.464,  330.856,  330.274,  329.707,  329.157,  328.624,  328.112, 
      327.626,  327.172,  326.756,  326.382,  326.042,  325.721,  325.415,  325.137,  324.892,  324.677, 
      324.473,  324.285,  324.115,  323.965,  323.842,  323.753, ],
    [ 368.356,  367.847,  367.412,  367.017,  366.633,  366.259,  365.895,  365.542,  365.199,  364.867, 
      364.548,  364.238,  363.940,  363.654,  363.381,  363.120,  362.873,  362.639,  362.418,  362.208, 
      362.008,  361.818,  361.638,  361.465,  361.296,  361.130,  360.727,  360.331,  359.940,  359.554, 
      359.166,  358.778,  358.394,  358.013,  357.631,  357.251,  356.871,  356.495,  356.125,  355.764, 
      355.408,  355.058,  354.711,  354.368,  354.031,  353.703,  353.383,  353.070,  352.763,  352.440, 
      352.103,  351.773,  351.450,  351.130,  350.816,  350.508,  349.918,  349.358,  348.818,  348.266, 
      347.721,  347.172,  346.647,  346.136,  345.634,  345.166,  344.644,  344.132,  343.657,  343.220, 
      342.819,  342.360,  341.611,  340.803,  339.974,  339.151,  338.342,  337.552,  336.753,  335.964, 
      335.202,  334.475,  333.784,  333.132,  332.515,  331.922,  331.344,  330.783,  330.239,  329.720, 
      329.229,  328.770,  328.349,  327.969,  327.621,  327.290,  326.977,  326.692,  326.442,  326.220, 
      326.007,  325.812,  325.634,  325.476,  325.347,  325.252, ],
    [ 370.502,  369.994,  369.559,  369.163,  368.779,  368.405,  368.042,  367.687,  367.341,  367.007, 
      366.682,  366.368,  366.066,  365.777,  365.502,  365.241,  364.994,  364.758,  364.534,  364.322, 
      364.122,  363.930,  363.747,  363.572,  363.403,  363.238,  362.831,  362.435,  362.045,  361.655, 
      361.265,  360.877,  360.491,  360.104,  359.718,  359.333,  358.951,  358.573,  358.203,  357.840, 
      357.482,  357.127,  356.776,  356.432,  356.095,  355.766,  355.445,  355.131,  354.818,  354.474, 
      354.135,  353.805,  353.478,  353.154,  352.838,  352.530,  351.938,  351.375,  350.819,  350.263, 
      349.704,  349.150,  348.620,  348.096,  347.591,  347.115,  346.571,  346.056,  345.578,  345.137, 
      344.735,  344.168,  343.383,  342.553,  341.715,  340.883,  340.069,  339.262,  338.444,  337.648, 
      336.882,  336.150,  335.455,  334.798,  334.172,  333.567,  332.978,  332.405,  331.851,  331.327, 
      330.830,  330.365,  329.941,  329.555,  329.197,  328.858,  328.539,  328.247,  327.991,  327.761, 
      327.540,  327.337,  327.153,  326.987,  326.852,  326.752, ],
    [ 372.649,  372.140,  371.705,  371.307,  370.921,  370.544,  370.176,  369.817,  369.468,  369.130, 
      368.803,  368.489,  368.187,  367.898,  367.621,  367.358,  367.108,  366.872,  366.648,  366.433, 
      366.230,  366.038,  365.855,  365.680,  365.509,  365.342,  364.935,  364.539,  364.147,  363.755, 
      363.364,  362.975,  362.583,  362.192,  361.801,  361.414,  361.029,  360.651,  360.281,  359.914, 
      359.552,  359.194,  358.841,  358.496,  358.159,  357.829,  357.508,  357.193,  356.854,  356.507, 
      356.168,  355.833,  355.502,  355.178,  354.859,  354.550,  353.957,  353.391,  352.819,  352.256, 
      351.687,  351.124,  350.589,  350.055,  349.549,  349.044,  348.495,  347.979,  347.499,  347.055, 
      346.650,  345.963,  345.140,  344.298,  343.453,  342.614,  341.791,  340.962,  340.134,  339.331, 
      338.560,  337.824,  337.123,  336.460,  335.825,  335.210,  334.609,  334.025,  333.462,  332.931, 
      332.430,  331.960,  331.532,  331.138,  330.772,  330.424,  330.099,  329.801,  329.539,  329.300, 
      329.073,  328.862,  328.670,  328.496,  328.356,  328.251, ],
    [ 374.788,  374.276,  373.838,  373.436,  373.047,  372.667,  372.298,  371.939,  371.589,  371.251, 
      370.924,  370.608,  370.303,  370.012,  369.735,  369.472,  369.221,  368.982,  368.755,  368.541, 
      368.338,  368.145,  367.961,  367.784,  367.613,  367.446,  367.040,  366.643,  366.248,  365.854, 
      365.464,  365.069,  364.672,  364.276,  363.882,  363.492,  363.107,  362.729,  362.356,  361.987, 
      361.620,  361.260,  360.905,  360.560,  360.222,  359.892,  359.569,  359.239,  358.887,  358.539, 
      358.198,  357.860,  357.526,  357.199,  356.881,  356.571,  355.974,  355.402,  354.818,  354.246, 
      353.665,  353.097,  352.552,  352.014,  351.506,  350.973,  350.419,  349.901,  349.418,  348.972, 
      348.541,  347.742,  346.893,  346.040,  345.187,  344.342,  343.505,  342.654,  341.820,  341.012, 
      340.237,  339.496,  338.792,  338.119,  337.473,  336.847,  336.236,  335.641,  335.071,  334.534, 
      334.028,  333.554,  333.121,  332.720,  332.344,  331.990,  331.659,  331.354,  331.087,  330.839, 
      330.604,  330.386,  330.187,  330.005,  329.860,  329.750, ],
    [ 376.911,  376.399,  375.959,  375.557,  375.168,  374.788,  374.418,  374.057,  373.706,  373.366, 
      373.038,  372.722,  372.417,  372.124,  371.845,  371.580,  371.328,  371.089,  370.863,  370.648, 
      370.443,  370.249,  370.065,  369.888,  369.717,  369.550,  369.144,  368.744,  368.347,  367.954, 
      367.559,  367.158,  366.757,  366.358,  365.962,  365.570,  365.185,  364.805,  364.429,  364.055, 
      363.686,  363.324,  362.969,  362.623,  362.285,  361.954,  361.630,  361.273,  360.919,  360.571, 
      360.226,  359.884,  359.549,  359.220,  358.901,  358.591,  357.991,  357.407,  356.815,  356.231, 
      355.642,  355.070,  354.513,  353.972,  353.462,  352.900,  352.344,  351.822,  351.337,  350.889, 
      350.344,  349.503,  348.640,  347.778,  346.918,  346.067,  345.209,  344.344,  343.503,  342.691, 
      341.912,  341.167,  340.457,  339.776,  339.119,  338.482,  337.860,  337.253,  336.679,  336.137, 
      335.625,  335.148,  334.708,  334.299,  333.915,  333.553,  333.217,  332.907,  332.633,  332.376, 
      332.135,  331.909,  331.703,  331.514,  331.364,  331.248, ],
    [ 379.032,  378.519,  378.078,  377.675,  377.284,  376.903,  376.532,  376.171,  375.820,  375.479, 
      375.149,  374.831,  374.525,  374.232,  373.952,  373.687,  373.435,  373.196,  372.968,  372.752, 
      372.547,  372.354,  372.169,  371.992,  371.821,  371.654,  371.246,  370.843,  370.446,  370.051, 
      369.650,  369.245,  368.840,  368.438,  368.040,  367.648,  367.263,  366.879,  366.499,  366.122, 
      365.751,  365.388,  365.033,  364.686,  364.347,  364.016,  363.667,  363.306,  362.952,  362.600, 
      362.252,  361.908,  361.570,  361.242,  360.921,  360.610,  360.007,  359.407,  358.811,  358.214, 
      357.620,  357.042,  356.474,  355.929,  355.412,  354.826,  354.268,  353.744,  353.256,  352.805, 
      352.130,  351.258,  350.383,  349.514,  348.647,  347.784,  346.902,  346.032,  345.185,  344.369, 
      343.585,  342.835,  342.119,  341.429,  340.762,  340.113,  339.481,  338.865,  338.284,  337.737, 
      337.220,  336.740,  336.295,  335.878,  335.485,  335.115,  334.773,  334.459,  334.179,  333.914, 
      333.665,  333.431,  333.218,  333.023,  332.866,  332.745, ],
    [ 381.147,  380.633,  380.192,  379.789,  379.398,  379.016,  378.643,  378.280,  377.928,  377.586, 
      377.256,  376.938,  376.632,  376.339,  376.059,  375.792,  375.540,  375.300,  375.072,  374.856, 
      374.652,  374.458,  374.273,  374.096,  373.925,  373.757,  373.345,  372.942,  372.545,  372.143, 
      371.738,  371.329,  370.921,  370.516,  370.118,  369.726,  369.337,  368.951,  368.567,  368.188, 
      367.815,  367.452,  367.096,  366.749,  366.409,  366.067,  365.699,  365.339,  364.982,  364.628, 
      364.276,  363.930,  363.592,  363.262,  362.941,  362.629,  362.024,  361.407,  360.804,  360.195, 
      359.593,  359.011,  358.433,  357.887,  357.341,  356.750,  356.191,  355.665,  355.173,  354.720, 
      353.897,  353.006,  352.122,  351.245,  350.373,  349.491,  348.593,  347.715,  346.864,  346.045, 
      345.256,  344.503,  343.778,  343.078,  342.400,  341.741,  341.097,  340.475,  339.888,  339.337, 
      338.815,  338.331,  337.879,  337.454,  337.053,  336.678,  336.329,  336.010,  335.722,  335.450, 
      335.194,  334.953,  334.733,  334.531,  334.368,  334.242, ],
    [ 383.260,  382.745,  382.302,  381.898,  381.506,  381.123,  380.750,  380.388,  380.035,  379.694, 
      379.363,  379.044,  378.737,  378.443,  378.163,  377.897,  377.644,  377.404,  377.176,  376.960, 
      376.756,  376.562,  376.377,  376.199,  376.027,  375.857,  375.445,  375.042,  374.639,  374.232, 
      373.822,  373.410,  372.999,  372.594,  372.196,  371.802,  371.410,  371.020,  370.633,  370.252, 
      369.879,  369.515,  369.159,  368.811,  368.470,  368.100,  367.732,  367.371,  367.010,  366.652, 
      366.299,  365.951,  365.612,  365.282,  364.961,  364.646,  364.031,  363.405,  362.791,  362.173, 
      361.566,  360.974,  360.391,  359.844,  359.269,  358.675,  358.113,  357.585,  357.091,  356.542, 
      355.653,  354.749,  353.858,  352.975,  352.093,  351.185,  350.281,  349.398,  348.542,  347.719, 
      346.926,  346.167,  345.434,  344.723,  344.034,  343.365,  342.710,  342.083,  341.491,  340.935, 
      340.410,  339.921,  339.462,  339.030,  338.620,  338.239,  337.884,  337.560,  337.265,  336.985, 
      336.721,  336.474,  336.246,  336.039,  335.870,  335.739, ],
    [ 385.367,  384.852,  384.410,  384.005,  383.613,  383.230,  382.857,  382.494,  382.140,  381.798, 
      381.467,  381.148,  380.841,  380.547,  380.267,  380.001,  379.748,  379.508,  379.280,  379.064, 
      378.860,  378.664,  378.479,  378.300,  378.126,  377.957,  377.544,  377.138,  376.728,  376.317, 
      375.903,  375.489,  375.077,  374.672,  374.272,  373.875,  373.479,  373.086,  372.697,  372.316, 
      371.943,  371.578,  371.221,  370.873,  370.509,  370.133,  369.765,  369.400,  369.036,  368.676, 
      368.320,  367.972,  367.632,  367.302,  366.979,  366.663,  366.034,  365.402,  364.776,  364.151, 
      363.539,  362.934,  362.349,  361.800,  361.196,  360.599,  360.034,  359.504,  359.008,  358.315, 
      357.402,  356.490,  355.590,  354.702,  353.802,  352.876,  351.965,  351.078,  350.219,  349.390, 
      348.594,  347.828,  347.086,  346.365,  345.666,  344.985,  344.321,  343.689,  343.092,  342.531, 
      342.003,  341.508,  341.043,  340.602,  340.186,  339.799,  339.438,  339.109,  338.807,  338.519, 
      338.248,  337.994,  337.759,  337.547,  337.372,  337.236, ],
    [ 387.474,  386.959,  386.516,  386.111,  385.718,  385.335,  384.961,  384.598,  384.244,  383.902, 
      383.571,  383.252,  382.945,  382.651,  382.371,  382.105,  381.852,  381.612,  381.383,  381.166, 
      380.961,  380.765,  380.578,  380.399,  380.226,  380.056,  379.641,  379.228,  378.815,  378.400, 
      377.984,  377.567,  377.155,  376.750,  376.346,  375.946,  375.547,  375.151,  374.761,  374.380, 
      374.006,  373.641,  373.284,  372.924,  372.541,  372.166,  371.795,  371.427,  371.060,  370.697, 
      370.341,  369.993,  369.652,  369.321,  368.995,  368.679,  368.033,  367.396,  366.758,  366.126, 
      365.511,  364.894,  364.307,  363.744,  363.121,  362.523,  361.956,  361.422,  360.925,  360.072, 
      359.146,  358.226,  357.320,  356.423,  355.497,  354.565,  353.648,  352.756,  351.894,  351.061, 
      350.260,  349.486,  348.733,  348.002,  347.293,  346.601,  345.931,  345.293,  344.693,  344.126, 
      343.595,  343.095,  342.622,  342.172,  341.749,  341.358,  340.991,  340.657,  340.347,  340.052, 
      339.774,  339.513,  339.271,  339.054,  338.874,  338.733, ],
    [ 389.579,  389.063,  388.620,  388.215,  387.822,  387.439,  387.065,  386.702,  386.349,  386.006, 
      385.675,  385.356,  385.049,  384.755,  384.475,  384.208,  383.954,  383.714,  383.484,  383.267, 
      383.060,  382.864,  382.677,  382.498,  382.325,  382.154,  381.733,  381.316,  380.898,  380.480, 
      380.062,  379.645,  379.233,  378.824,  378.418,  378.014,  377.612,  377.215,  376.825,  376.443, 
      376.069,  375.703,  375.345,  374.957,  374.574,  374.197,  373.823,  373.451,  373.083,  372.719, 
      372.362,  372.013,  371.672,  371.338,  371.012,  370.691,  370.033,  369.386,  368.736,  368.099, 
      367.476,  366.853,  366.264,  365.672,  365.045,  364.446,  363.877,  363.340,  362.755,  361.821, 
      360.887,  359.959,  359.048,  358.133,  357.187,  356.249,  355.328,  354.433,  353.566,  352.729, 
      351.923,  351.141,  350.378,  349.636,  348.916,  348.214,  347.540,  346.896,  346.291,  345.721, 
      345.186,  344.680,  344.200,  343.742,  343.312,  342.915,  342.544,  342.205,  341.885,  341.584, 
      341.299,  341.032,  340.782,  340.560,  340.376,  340.231, ],
    [ 391.683,  391.167,  390.724,  390.319,  389.926,  389.543,  389.169,  388.806,  388.453,  388.110, 
      387.779,  387.460,  387.153,  386.858,  386.577,  386.309,  386.055,  385.813,  385.583,  385.366, 
      385.159,  384.963,  384.776,  384.597,  384.421,  384.248,  383.821,  383.400,  382.979,  382.559, 
      382.140,  381.723,  381.309,  380.897,  380.487,  380.080,  379.677,  379.279,  378.889,  378.506, 
      378.131,  377.765,  377.381,  376.990,  376.606,  376.226,  375.848,  375.475,  375.104,  374.740, 
      374.382,  374.033,  373.691,  373.355,  373.029,  372.695,  372.029,  371.371,  370.714,  370.072, 
      369.437,  368.811,  368.222,  367.600,  366.969,  366.367,  365.797,  365.258,  364.512,  363.566, 
      362.624,  361.689,  360.770,  359.830,  358.877,  357.932,  357.007,  356.108,  355.237,  354.397, 
      353.582,  352.790,  352.018,  351.266,  350.535,  349.825,  349.145,  348.498,  347.888,  347.315, 
      346.776,  346.263,  345.775,  345.309,  344.874,  344.471,  344.096,  343.750,  343.423,  343.115, 
      342.822,  342.549,  342.292,  342.066,  341.876,  341.726, ],
    [ 393.787,  393.271,  392.828,  392.423,  392.030,  391.647,  391.273,  390.910,  390.556,  390.213, 
      389.881,  389.562,  389.253,  388.958,  388.676,  388.408,  388.154,  387.912,  387.683,  387.465, 
      387.258,  387.061,  386.873,  386.690,  386.512,  386.336,  385.907,  385.481,  385.058,  384.637, 
      384.218,  383.800,  383.382,  382.967,  382.553,  382.144,  381.740,  381.343,  380.952,  380.569, 
      380.193,  379.812,  379.413,  379.022,  378.636,  378.253,  377.872,  377.496,  377.126,  376.760, 
      376.402,  376.052,  375.708,  375.372,  375.036,  374.695,  374.025,  373.354,  372.691,  372.045, 
      371.398,  370.769,  370.177,  369.526,  368.893,  368.289,  367.716,  367.175,  366.263,  365.307, 
      364.357,  363.418,  362.482,  361.520,  360.561,  359.612,  358.684,  357.781,  356.906,  356.061, 
      355.239,  354.436,  353.654,  352.892,  352.150,  351.435,  350.749,  350.098,  349.483,  348.908, 
      348.363,  347.844,  347.348,  346.875,  346.435,  346.026,  345.647,  345.295,  344.959,  344.646, 
      344.345,  344.065,  343.801,  343.571,  343.376,  343.221, ],
    [ 395.891,  395.375,  394.932,  394.527,  394.133,  393.749,  393.375,  393.011,  392.657,  392.313, 
      391.981,  391.661,  391.353,  391.057,  390.776,  390.508,  390.253,  390.012,  389.782,  389.563, 
      389.354,  389.154,  388.963,  388.779,  388.599,  388.422,  387.988,  387.560,  387.136,  386.715, 
      386.295,  385.873,  385.453,  385.034,  384.619,  384.208,  383.804,  383.406,  383.015,  382.631, 
      382.251,  381.845,  381.446,  381.054,  380.664,  380.277,  379.895,  379.518,  379.146,  378.780, 
      378.422,  378.070,  377.724,  377.386,  377.038,  376.695,  376.017,  375.333,  374.664,  374.013, 
      373.356,  372.726,  372.108,  371.450,  370.817,  370.211,  369.634,  368.976,  368.007,  367.044, 
      366.087,  365.140,  364.178,  363.209,  362.244,  361.291,  360.360,  359.453,  358.574,  357.721, 
      356.890,  356.079,  355.285,  354.514,  353.762,  353.043,  352.352,  351.697,  351.078,  350.500, 
      349.950,  349.423,  348.918,  348.439,  347.995,  347.580,  347.197,  346.837,  346.495,  346.174, 
      345.867,  345.580,  345.310,  345.076,  344.875,  344.715, ],
    [ 397.993,  397.477,  397.034,  396.628,  396.234,  395.850,  395.475,  395.111,  394.756,  394.412, 
      394.080,  393.760,  393.452,  393.157,  392.875,  392.607,  392.351,  392.108,  391.875,  391.655, 
      391.443,  391.242,  391.050,  390.864,  390.683,  390.505,  390.069,  389.638,  389.214,  388.792, 
      388.369,  387.945,  387.520,  387.099,  386.682,  386.272,  385.868,  385.469,  385.077,  384.693, 
      384.284,  383.878,  383.478,  383.082,  382.688,  382.300,  381.916,  381.539,  381.166,  380.800, 
      380.441,  380.086,  379.741,  379.391,  379.038,  378.691,  378.002,  377.311,  376.637,  375.975, 
      375.315,  374.684,  374.036,  373.374,  372.739,  372.131,  371.552,  370.727,  369.748,  368.777, 
      367.816,  366.852,  365.868,  364.893,  363.924,  362.968,  362.033,  361.122,  360.239,  359.379, 
      358.537,  357.716,  356.913,  356.130,  355.373,  354.648,  353.954,  353.294,  352.673,  352.091, 
      351.534,  351.000,  350.488,  350.002,  349.553,  349.133,  348.745,  348.379,  348.029,  347.701, 
      347.388,  347.094,  346.819,  346.580,  346.374,  346.208, ],
    [ 400.094,  399.577,  399.133,  398.727,  398.333,  397.949,  397.574,  397.210,  396.855,  396.512, 
      396.179,  395.859,  395.551,  395.254,  394.971,  394.701,  394.444,  394.198,  393.965,  393.742, 
      393.530,  393.328,  393.134,  392.946,  392.764,  392.585,  392.147,  391.716,  391.292,  390.867, 
      390.442,  390.014,  389.586,  389.163,  388.746,  388.336,  387.931,  387.532,  387.139,  386.732, 
      386.317,  385.910,  385.507,  385.108,  384.712,  384.322,  383.938,  383.559,  383.186,  382.820, 
      382.458,  382.103,  381.750,  381.390,  381.036,  380.687,  379.985,  379.288,  378.610,  377.935, 
      377.273,  376.641,  375.962,  375.298,  374.661,  374.051,  373.463,  372.471,  371.486,  370.508, 
      369.538,  368.548,  367.558,  366.576,  365.603,  364.644,  363.704,  362.790,  361.900,  361.032, 
      360.182,  359.349,  358.537,  357.743,  356.982,  356.251,  355.554,  354.889,  354.266,  353.678, 
      353.117,  352.576,  352.055,  351.565,  351.110,  350.686,  350.293,  349.918,  349.563,  349.227, 
      348.908,  348.607,  348.327,  348.083,  347.873,  347.701, ],
    [ 402.193,  401.676,  401.233,  400.827,  400.432,  400.048,  399.674,  399.309,  398.954,  398.610, 
      398.277,  397.955,  397.644,  397.346,  397.061,  396.790,  396.532,  396.286,  396.051,  395.827, 
      395.614,  395.410,  395.215,  395.027,  394.844,  394.664,  394.225,  393.794,  393.368,  392.940, 
      392.511,  392.079,  391.651,  391.227,  390.810,  390.399,  389.994,  389.594,  389.188,  388.765, 
      388.350,  387.940,  387.534,  387.131,  386.734,  386.343,  385.958,  385.579,  385.206,  384.838, 
      384.474,  384.118,  383.752,  383.390,  383.033,  382.680,  381.964,  381.262,  380.580,  379.894, 
      379.230,  378.580,  377.887,  377.222,  376.583,  375.969,  375.213,  374.212,  373.219,  372.236, 
      371.250,  370.238,  369.242,  368.257,  367.280,  366.317,  365.374,  364.456,  363.558,  362.680, 
      361.820,  360.979,  360.155,  359.355,  358.589,  357.854,  357.152,  356.484,  355.858,  355.265, 
      354.696,  354.147,  353.622,  353.126,  352.665,  352.238,  351.839,  351.456,  351.095,  350.752, 
      350.427,  350.120,  349.835,  349.585,  349.372,  349.193, ],
    [ 404.292,  403.776,  403.332,  402.926,  402.531,  402.146,  401.771,  401.404,  401.048,  400.702, 
      400.367,  400.044,  399.733,  399.434,  399.149,  398.876,  398.617,  398.369,  398.133,  397.908, 
      397.694,  397.490,  397.294,  397.105,  396.922,  396.742,  396.303,  395.871,  395.441,  395.010, 
      394.577,  394.144,  393.714,  393.291,  392.873,  392.462,  392.056,  391.651,  391.220,  390.797, 
      390.381,  389.968,  389.558,  389.154,  388.756,  388.364,  387.978,  387.599,  387.225,  386.854, 
      386.491,  386.122,  385.752,  385.387,  385.027,  384.665,  383.942,  383.235,  382.542,  381.853, 
      381.188,  380.507,  379.811,  379.145,  378.504,  377.888,  376.958,  375.949,  374.949,  373.959, 
      372.945,  371.927,  370.924,  369.935,  368.956,  367.989,  367.042,  366.117,  365.213,  364.325, 
      363.454,  362.603,  361.770,  360.965,  360.193,  359.455,  358.750,  358.079,  357.449,  356.851, 
      356.275,  355.718,  355.185,  354.686,  354.220,  353.788,  353.383,  352.993,  352.625,  352.276, 
      351.945,  351.631,  351.342,  351.087,  350.871,  350.684, ],
    [ 406.390,  405.873,  405.428,  405.020,  404.624,  404.238,  403.861,  403.494,  403.137,  402.790, 
      402.455,  402.131,  401.819,  401.519,  401.232,  400.959,  400.698,  400.450,  400.214,  399.988, 
      399.773,  399.568,  399.372,  399.183,  399.000,  398.820,  398.380,  397.944,  397.511,  397.077, 
      396.642,  396.208,  395.778,  395.354,  394.937,  394.524,  394.118,  393.683,  393.253,  392.829, 
      392.409,  391.992,  391.582,  391.176,  390.777,  390.384,  389.998,  389.618,  389.242,  388.871, 
      388.500,  388.122,  387.750,  387.383,  387.016,  386.648,  385.920,  385.208,  384.502,  383.811, 
      383.146,  382.433,  381.735,  381.067,  380.424,  379.722,  378.698,  377.682,  376.678,  375.669, 
      374.635,  373.611,  372.605,  371.613,  370.629,  369.658,  368.708,  367.775,  366.861,  365.965, 
      365.085,  364.223,  363.382,  362.573,  361.796,  361.055,  360.345,  359.672,  359.037,  358.434, 
      357.850,  357.286,  356.748,  356.245,  355.773,  355.338,  354.926,  354.529,  354.155,  353.799, 
      353.462,  353.141,  352.848,  352.589,  352.368,  352.174, ],
    [ 408.482,  407.964,  407.518,  407.109,  406.713,  406.326,  405.949,  405.581,  405.223,  404.875, 
      404.539,  404.214,  403.901,  403.600,  403.313,  403.039,  402.778,  402.529,  402.292,  402.066, 
      401.851,  401.646,  401.450,  401.261,  401.078,  400.897,  400.454,  400.016,  399.579,  399.142, 
      398.706,  398.272,  397.842,  397.418,  396.999,  396.586,  396.153,  395.716,  395.286,  394.858, 
      394.434,  394.016,  393.603,  393.197,  392.797,  392.404,  392.018,  391.636,  391.258,  390.884, 
      390.501,  390.122,  389.747,  389.375,  389.000,  388.627,  387.893,  387.178,  386.462,  385.769, 
      385.087,  384.358,  383.659,  382.988,  382.343,  381.466,  380.435,  379.412,  378.400,  377.362, 
      376.323,  375.293,  374.283,  373.288,  372.300,  371.326,  370.369,  369.430,  368.507,  367.600, 
      366.711,  365.839,  364.992,  364.178,  363.398,  362.653,  361.940,  361.264,  360.624,  360.014, 
      359.424,  358.853,  358.311,  357.801,  357.326,  356.886,  356.467,  356.064,  355.683,  355.320, 
      354.977,  354.650,  354.354,  354.091,  353.866,  353.664, ],
    [ 410.570,  410.051,  409.605,  409.196,  408.798,  408.411,  408.032,  407.664,  407.305,  406.957, 
      406.620,  406.295,  405.981,  405.680,  405.392,  405.118,  404.856,  404.607,  404.370,  404.144, 
      403.929,  403.724,  403.528,  403.339,  403.154,  402.972,  402.526,  402.084,  401.645,  401.206, 
      400.770,  400.336,  399.905,  399.480,  399.061,  398.629,  398.186,  397.749,  397.315,  396.884, 
      396.458,  396.038,  395.624,  395.218,  394.817,  394.424,  394.036,  393.652,  393.275,  392.888, 
      392.501,  392.118,  391.741,  391.360,  390.981,  390.605,  389.866,  389.140,  388.420,  387.726, 
      387.015,  386.282,  385.582,  384.910,  384.251,  383.206,  382.167,  381.141,  380.106,  379.052, 
      378.006,  376.973,  375.960,  374.961,  373.970,  372.992,  372.028,  371.079,  370.147,  369.231, 
      368.332,  367.451,  366.601,  365.781,  364.999,  364.249,  363.534,  362.855,  362.210,  361.593, 
      360.994,  360.418,  359.872,  359.357,  358.879,  358.434,  358.006,  357.597,  357.210,  356.841, 
      356.491,  356.158,  355.859,  355.591,  355.363,  355.153, ],
    [ 412.655,  412.135,  411.688,  411.278,  410.880,  410.492,  410.113,  409.744,  409.385,  409.037, 
      408.699,  408.374,  408.060,  407.758,  407.470,  407.196,  406.934,  406.685,  406.448,  406.222, 
      406.007,  405.801,  405.604,  405.413,  405.227,  405.045,  404.595,  404.150,  403.709,  403.270, 
      402.834,  402.399,  401.968,  401.543,  401.110,  400.662,  400.219,  399.779,  399.342,  398.908, 
      398.480,  398.059,  397.645,  397.238,  396.837,  396.443,  396.053,  395.669,  395.282,  394.888, 
      394.499,  394.114,  393.728,  393.342,  392.958,  392.582,  391.839,  391.100,  390.378,  389.684, 
      388.941,  388.206,  387.503,  386.830,  385.995,  384.942,  383.897,  382.861,  381.798,  380.739, 
      379.688,  378.652,  377.635,  376.632,  375.638,  374.654,  373.683,  372.725,  371.782,  370.857, 
      369.948,  369.062,  368.207,  367.384,  366.598,  365.844,  365.127,  364.443,  363.793,  363.169, 
      362.564,  361.981,  361.431,  360.912,  360.430,  359.980,  359.544,  359.129,  358.735,  358.361, 
      358.004,  357.667,  357.364,  357.091,  356.860,  356.641, ],
    [ 414.735,  414.216,  413.769,  413.359,  412.961,  412.572,  412.192,  411.823,  411.463,  411.115, 
      410.777,  410.452,  410.138,  409.836,  409.548,  409.274,  409.012,  408.763,  408.526,  408.299, 
      408.082,  407.875,  407.677,  407.486,  407.298,  407.113,  406.660,  406.214,  405.773,  405.334, 
      404.897,  404.462,  404.030,  403.598,  403.143,  402.694,  402.249,  401.807,  401.366,  400.931, 
      400.502,  400.081,  399.665,  399.258,  398.857,  398.460,  398.069,  397.680,  397.282,  396.886, 
      396.495,  396.104,  395.711,  395.321,  394.936,  394.555,  393.808,  393.060,  392.336,  391.624, 
      390.865,  390.130,  389.425,  388.749,  387.734,  386.673,  385.625,  384.564,  383.488,  382.422, 
      381.368,  380.329,  379.308,  378.301,  377.304,  376.312,  375.332,  374.365,  373.413,  372.478, 
      371.560,  370.671,  369.811,  368.985,  368.195,  367.439,  366.718,  366.030,  365.374,  364.743, 
      364.131,  363.544,  362.989,  362.465,  361.980,  361.522,  361.081,  360.660,  360.259,  359.880, 
      359.517,  359.175,  358.867,  358.590,  358.355,  358.128, ],
    [ 416.815,  416.295,  415.847,  415.437,  415.039,  414.650,  414.270,  413.901,  413.541,  413.193, 
      412.855,  412.530,  412.216,  411.914,  411.626,  411.351,  411.089,  410.839,  410.600,  410.372, 
      410.155,  409.947,  409.747,  409.554,  409.365,  409.180,  408.725,  408.278,  407.837,  407.397, 
      406.960,  406.524,  406.091,  405.630,  405.176,  404.726,  404.277,  403.830,  403.389,  402.952, 
      402.523,  402.101,  401.685,  401.277,  400.874,  400.477,  400.085,  399.682,  399.281,  398.883, 
      398.488,  398.088,  397.691,  397.298,  396.911,  396.528,  395.770,  395.019,  394.293,  393.551, 
      392.789,  392.053,  391.347,  390.542,  389.470,  388.404,  387.343,  386.254,  385.173,  384.104, 
      383.046,  382.003,  380.979,  379.970,  378.965,  377.967,  376.977,  376.000,  375.040,  374.094, 
      373.171,  372.278,  371.414,  370.585,  369.790,  369.033,  368.309,  367.616,  366.953,  366.314, 
      365.697,  365.106,  364.545,  364.018,  363.528,  363.065,  362.617,  362.190,  361.782,  361.397, 
      361.027,  360.682,  360.369,  360.089,  359.849,  359.614, ],
    [ 418.893,  418.373,  417.925,  417.515,  417.116,  416.728,  416.348,  415.979,  415.619,  415.271, 
      414.933,  414.608,  414.293,  413.991,  413.702,  413.426,  413.163,  412.912,  412.673,  412.443, 
      412.225,  412.016,  411.814,  411.620,  411.431,  411.245,  410.789,  410.342,  409.900,  409.460, 
      409.023,  408.586,  408.124,  407.663,  407.207,  406.754,  406.302,  405.854,  405.411,  404.974, 
      404.543,  404.121,  403.705,  403.296,  402.891,  402.493,  402.088,  401.681,  401.278,  400.877, 
      400.473,  400.070,  399.669,  399.275,  398.883,  398.501,  397.730,  396.976,  396.251,  395.477, 
      394.713,  393.975,  393.267,  392.281,  391.200,  390.130,  389.041,  387.944,  386.856,  385.783, 
      384.723,  383.676,  382.648,  381.634,  380.623,  379.615,  378.617,  377.631,  376.661,  375.707, 
      374.780,  373.882,  373.015,  372.182,  371.385,  370.625,  369.897,  369.199,  368.529,  367.883, 
      367.260,  366.666,  366.100,  365.570,  365.076,  364.604,  364.150,  363.717,  363.303,  362.912, 
      362.536,  362.189,  361.871,  361.588,  361.342,  361.098, ],
    [ 420.971,  420.451,  420.003,  419.593,  419.194,  418.805,  418.426,  418.057,  417.697,  417.347, 
      417.009,  416.682,  416.367,  416.064,  415.775,  415.498,  415.234,  414.982,  414.741,  414.511, 
      414.291,  414.081,  413.880,  413.685,  413.495,  413.309,  412.853,  412.406,  411.963,  411.523, 
      411.085,  410.623,  410.156,  409.695,  409.236,  408.779,  408.326,  407.876,  407.432,  406.994, 
      406.564,  406.141,  405.724,  405.312,  404.907,  404.500,  404.088,  403.679,  403.273,  402.865, 
      402.455,  402.047,  401.646,  401.247,  400.856,  400.468,  399.689,  398.934,  398.187,  397.401, 
      396.637,  395.897,  395.108,  394.015,  392.930,  391.844,  390.731,  389.628,  388.538,  387.461, 
      386.397,  385.347,  384.316,  383.295,  382.277,  381.260,  380.251,  379.257,  378.277,  377.318, 
      376.387,  375.485,  374.616,  373.779,  372.980,  372.217,  371.484,  370.778,  370.103,  369.450, 
      368.822,  368.224,  367.655,  367.121,  366.622,  366.142,  365.683,  365.243,  364.824,  364.427, 
      364.045,  363.694,  363.373,  363.087,  362.834,  362.580, ],
    [ 423.049,  422.529,  422.081,  421.671,  421.271,  420.882,  420.502,  420.131,  419.771,  419.421, 
      419.082,  418.755,  418.439,  418.135,  417.844,  417.567,  417.302,  417.049,  416.807,  416.577, 
      416.357,  416.146,  415.944,  415.749,  415.559,  415.373,  414.917,  414.469,  414.026,  413.585, 
      413.126,  412.655,  412.189,  411.724,  411.262,  410.803,  410.348,  409.897,  409.453,  409.014, 
      408.584,  408.160,  407.741,  407.329,  406.918,  406.500,  406.086,  405.675,  405.263,  404.848, 
      404.433,  404.025,  403.619,  403.220,  402.827,  402.429,  401.648,  400.891,  400.113,  399.325, 
      398.560,  397.818,  396.845,  395.745,  394.654,  393.537,  392.419,  391.311,  390.216,  389.137, 
      388.069,  387.015,  385.980,  384.953,  383.924,  382.899,  381.882,  380.878,  379.889,  378.927, 
      377.992,  377.087,  376.213,  375.374,  374.573,  373.806,  373.068,  372.357,  371.673,  371.016, 
      370.385,  369.781,  369.208,  368.671,  368.165,  367.679,  367.215,  366.768,  366.344,  365.940, 
      365.554,  365.200,  364.875,  364.585,  364.324,  364.062, ],
    [ 425.125,  424.605,  424.156,  423.745,  423.345,  422.955,  422.575,  422.204,  421.843,  421.492, 
      421.152,  420.824,  420.507,  420.203,  419.911,  419.633,  419.368,  419.115,  418.872,  418.641, 
      418.421,  418.210,  418.008,  417.813,  417.623,  417.437,  416.980,  416.532,  416.088,  415.631, 
      415.158,  414.688,  414.218,  413.750,  413.286,  412.825,  412.369,  411.918,  411.473,  411.034, 
      410.603,  410.177,  409.758,  409.343,  408.919,  408.499,  408.083,  407.667,  407.247,  406.827, 
      406.411,  405.999,  405.592,  405.193,  404.789,  404.389,  403.606,  402.849,  402.038,  401.249, 
      400.481,  399.694,  398.578,  397.474,  396.361,  395.227,  394.102,  392.991,  391.894,  390.810, 
      389.739,  388.683,  387.641,  386.605,  385.569,  384.533,  383.507,  382.493,  381.500,  380.534, 
      379.594,  378.687,  377.810,  376.968,  376.164,  375.393,  374.650,  373.933,  373.242,  372.579, 
      371.945,  371.336,  370.760,  370.219,  369.707,  369.215,  368.745,  368.291,  367.862,  367.452, 
      367.062,  366.704,  366.375,  366.082,  365.814,  365.544, ],
    [ 427.199,  426.678,  426.229,  425.818,  425.417,  425.026,  424.645,  424.273,  423.911,  423.560, 
      423.219,  422.891,  422.573,  422.269,  421.977,  421.698,  421.432,  421.178,  420.936,  420.705, 
      420.485,  420.274,  420.072,  419.877,  419.687,  419.500,  419.043,  418.594,  418.138,  417.663, 
      417.191,  416.718,  416.245,  415.774,  415.308,  414.847,  414.390,  413.938,  413.493,  413.054, 
      412.621,  412.194,  411.774,  411.345,  410.919,  410.496,  410.077,  409.652,  409.227,  408.805, 
      408.387,  407.972,  407.565,  407.156,  406.749,  406.348,  405.563,  404.776,  403.963,  403.173, 
      402.403,  401.430,  400.308,  399.196,  398.052,  396.913,  395.784,  394.669,  393.569,  392.482, 
      391.408,  390.346,  389.298,  388.252,  387.206,  386.163,  385.127,  384.105,  383.109,  382.138, 
      381.196,  380.285,  379.405,  378.562,  377.755,  376.979,  376.229,  375.505,  374.809,  374.142, 
      373.503,  372.891,  372.312,  371.766,  371.246,  370.749,  370.272,  369.813,  369.379,  368.962, 
      368.569,  368.207,  367.874,  367.579,  367.303,  367.024, ],
    [ 429.269,  428.748,  428.298,  427.886,  427.485,  427.093,  426.712,  426.339,  425.977,  425.625, 
      425.285,  424.956,  424.638,  424.333,  424.041,  423.762,  423.496,  423.242,  423.000,  422.769, 
      422.549,  422.338,  422.135,  421.940,  421.750,  421.563,  421.105,  420.647,  420.170,  419.696, 
      419.221,  418.745,  418.269,  417.797,  417.330,  416.868,  416.410,  415.958,  415.513,  415.072, 
      414.638,  414.211,  413.778,  413.345,  412.916,  412.491,  412.062,  411.633,  411.205,  410.781, 
      410.360,  409.945,  409.532,  409.117,  408.708,  408.306,  407.521,  406.702,  405.887,  405.095, 
      404.300,  403.161,  402.036,  400.894,  399.741,  398.596,  397.464,  396.346,  395.242,  394.152, 
      393.075,  392.006,  390.949,  389.896,  388.839,  387.787,  386.742,  385.716,  384.716,  383.741, 
      382.797,  381.882,  381.000,  380.154,  379.342,  378.562,  377.807,  377.076,  376.374,  375.704, 
      375.060,  374.444,  373.862,  373.311,  372.784,  372.282,  371.799,  371.335,  370.894,  370.472, 
      370.076,  369.709,  369.373,  369.076,  368.791,  368.505, ],
    [ 431.337,  430.815,  430.365,  429.952,  429.551,  429.159,  428.777,  428.405,  428.042,  427.690, 
      427.349,  427.020,  426.702,  426.397,  426.104,  425.826,  425.560,  425.306,  425.064,  424.832, 
      424.612,  424.401,  424.198,  424.003,  423.812,  423.625,  423.162,  422.680,  422.203,  421.726, 
      421.248,  420.769,  420.292,  419.818,  419.351,  418.888,  418.431,  417.978,  417.531,  417.089, 
      416.654,  416.218,  415.778,  415.343,  414.911,  414.478,  414.045,  413.611,  413.183,  412.754, 
      412.333,  411.914,  411.492,  411.076,  410.667,  410.264,  409.467,  408.627,  407.811,  407.017, 
      406.035,  404.891,  403.751,  402.584,  401.425,  400.277,  399.142,  398.021,  396.913,  395.820, 
      394.737,  393.662,  392.595,  391.532,  390.468,  389.405,  388.354,  387.325,  386.320,  385.343, 
      384.395,  383.477,  382.594,  381.745,  380.929,  380.143,  379.381,  378.644,  377.937,  377.264, 
      376.616,  375.997,  375.411,  374.854,  374.320,  373.813,  373.324,  372.855,  372.408,  371.980, 
      371.582,  371.211,  370.872,  370.570,  370.277,  369.984, ],
    [ 433.402,  432.881,  432.430,  432.017,  431.615,  431.223,  430.841,  430.468,  430.106,  429.754, 
      429.413,  429.083,  428.766,  428.460,  428.168,  427.889,  427.623,  427.369,  427.127,  426.896, 
      426.675,  426.464,  426.261,  426.065,  425.875,  425.685,  425.195,  424.713,  424.234,  423.754, 
      423.272,  422.792,  422.313,  421.840,  421.371,  420.908,  420.451,  419.997,  419.548,  419.106, 
      418.665,  418.218,  417.776,  417.339,  416.901,  416.461,  416.022,  415.589,  415.156,  414.727, 
      414.304,  413.875,  413.452,  413.035,  412.624,  412.221,  411.393,  410.551,  409.734,  408.925, 
      407.765,  406.616,  405.442,  404.271,  403.108,  401.956,  400.818,  399.693,  398.582,  397.486, 
      396.395,  395.311,  394.236,  393.164,  392.091,  391.019,  389.964,  388.931,  387.923,  386.944, 
      385.992,  385.072,  384.186,  383.333,  382.513,  381.722,  380.952,  380.210,  379.500,  378.823, 
      378.170,  377.549,  376.959,  376.395,  375.856,  375.343,  374.847,  374.374,  373.921,  373.489, 
      373.088,  372.713,  372.371,  372.063,  371.760,  371.463, ],
    [ 435.466,  434.944,  434.494,  434.081,  433.679,  433.287,  432.905,  432.532,  432.170,  431.817, 
      431.476,  431.147,  430.829,  430.524,  430.231,  429.953,  429.686,  429.433,  429.190,  428.958, 
      428.737,  428.526,  428.323,  428.121,  427.918,  427.718,  427.228,  426.744,  426.262,  425.778, 
      425.296,  424.813,  424.334,  423.860,  423.391,  422.928,  422.469,  422.014,  421.565,  421.118, 
      420.664,  420.216,  419.773,  419.329,  418.884,  418.440,  418.000,  417.563,  417.129,  416.700, 
      416.266,  415.835,  415.411,  414.993,  414.582,  414.179,  413.318,  412.475,  411.655,  410.657, 
      409.494,  408.320,  407.133,  405.955,  404.788,  403.633,  402.492,  401.364,  400.250,  399.147, 
      398.050,  396.957,  395.871,  394.791,  393.708,  392.631,  391.572,  390.535,  389.524,  388.541, 
      387.587,  386.666,  385.777,  384.920,  384.095,  383.297,  382.522,  381.774,  381.062,  380.379, 
      379.723,  379.099,  378.505,  377.933,  377.390,  376.870,  376.369,  375.891,  375.432,  374.997, 
      374.592,  374.214,  373.869,  373.554,  373.242,  372.942, ],
    [ 437.530,  437.008,  436.558,  436.145,  435.743,  435.351,  434.969,  434.596,  434.233,  433.881, 
      433.540,  433.210,  432.892,  432.587,  432.295,  432.016,  431.749,  431.495,  431.252,  431.021, 
      430.798,  430.576,  430.361,  430.153,  429.950,  429.750,  429.259,  428.772,  428.286,  427.802, 
      427.317,  426.835,  426.355,  425.880,  425.411,  424.948,  424.487,  424.031,  423.579,  423.119, 
      422.664,  422.213,  421.765,  421.313,  420.863,  420.418,  419.975,  419.536,  419.101,  418.663, 
      418.226,  417.794,  417.369,  416.950,  416.539,  416.110,  415.242,  414.399,  413.565,  412.388, 
      411.214,  410.010,  408.817,  407.636,  406.466,  405.308,  404.163,  403.032,  401.914,  400.804, 
      399.698,  398.596,  397.502,  396.412,  395.320,  394.240,  393.178,  392.137,  391.125,  390.138, 
      389.182,  388.258,  387.366,  386.505,  385.674,  384.870,  384.089,  383.337,  382.621,  381.934, 
      381.276,  380.648,  380.048,  379.470,  378.922,  378.396,  377.890,  377.406,  376.941,  376.504, 
      376.095,  375.714,  375.367,  375.044,  374.724,  374.419, ],
    [ 439.594,  439.072,  438.622,  438.208,  437.807,  437.414,  437.032,  436.659,  436.296,  435.944, 
      435.603,  435.273,  434.955,  434.650,  434.357,  434.078,  433.811,  433.557,  433.305,  433.063, 
      432.831,  432.608,  432.394,  432.186,  431.983,  431.782,  431.287,  430.796,  430.310,  429.823, 
      429.339,  428.855,  428.375,  427.900,  427.431,  426.965,  426.504,  426.047,  425.581,  425.119, 
      424.660,  424.207,  423.749,  423.294,  422.841,  422.393,  421.948,  421.508,  421.067,  420.624, 
      420.185,  419.753,  419.327,  418.908,  418.478,  418.036,  417.166,  416.321,  415.295,  414.114, 
      412.907,  411.698,  410.500,  409.315,  408.143,  406.981,  405.833,  404.699,  403.574,  402.456, 
      401.342,  400.229,  399.127,  398.027,  396.931,  395.848,  394.781,  393.739,  392.722,  391.733, 
      390.776,  389.849,  388.953,  388.088,  387.251,  386.440,  385.655,  384.900,  384.179,  383.489, 
      382.828,  382.196,  381.589,  381.006,  380.452,  379.921,  379.410,  378.919,  378.450,  378.011, 
      377.597,  377.213,  376.864,  376.534,  376.205,  375.896, ],
    [ 441.658,  441.135,  440.685,  440.271,  439.870,  439.478,  439.095,  438.722,  438.359,  438.007, 
      437.665,  437.336,  437.018,  436.712,  436.417,  436.129,  435.854,  435.590,  435.338,  435.096, 
      434.864,  434.641,  434.426,  434.217,  434.012,  433.810,  433.312,  432.820,  432.331,  431.844, 
      431.359,  430.875,  430.395,  429.920,  429.449,  428.982,  428.520,  428.049,  427.580,  427.115, 
      426.655,  426.192,  425.730,  425.271,  424.818,  424.366,  423.921,  423.476,  423.027,  422.583, 
      422.144,  421.710,  421.284,  420.852,  420.403,  419.960,  419.091,  418.220,  417.025,  415.821, 
      414.597,  413.381,  412.180,  410.993,  409.817,  408.652,  407.501,  406.362,  405.231,  404.103, 
      402.979,  401.858,  400.746,  399.639,  398.541,  397.453,  396.384,  395.339,  394.319,  393.328, 
      392.368,  391.439,  390.539,  389.667,  388.824,  388.008,  387.218,  386.461,  385.736,  385.042, 
      384.379,  383.742,  383.128,  382.540,  381.981,  381.444,  380.929,  380.433,  379.958,  379.516, 
      379.099,  378.712,  378.361,  378.021,  377.686,  377.372, ],
    [ 443.721,  443.199,  442.748,  442.334,  441.932,  441.540,  441.158,  440.785,  440.422,  440.069, 
      439.727,  439.390,  439.064,  438.751,  438.450,  438.162,  437.887,  437.623,  437.370,  437.128, 
      436.895,  436.671,  436.455,  436.245,  436.038,  435.835,  435.336,  434.842,  434.352,  433.865, 
      433.379,  432.896,  432.415,  431.938,  431.465,  430.998,  430.524,  430.049,  429.577,  429.110, 
      428.641,  428.173,  427.708,  427.248,  426.791,  426.339,  425.890,  425.436,  424.987,  424.542, 
      424.101,  423.668,  423.234,  422.778,  422.328,  421.884,  421.014,  419.950,  418.747,  417.511, 
      416.282,  415.063,  413.858,  412.669,  411.489,  410.321,  409.167,  408.020,  406.880,  405.744, 
      404.611,  403.482,  402.360,  401.250,  400.147,  399.056,  397.985,  396.936,  395.914,  394.922, 
      393.959,  393.026,  392.122,  391.244,  390.395,  389.574,  388.780,  388.019,  387.290,  386.595, 
      385.928,  385.286,  384.666,  384.073,  383.508,  382.965,  382.446,  381.943,  381.467,  381.022, 
      380.601,  380.211,  379.855,  379.508,  379.165,  378.847, ],
    [ 445.783,  445.261,  444.810,  444.397,  443.995,  443.602,  443.217,  442.837,  442.467,  442.108, 
      441.760,  441.423,  441.097,  440.783,  440.483,  440.195,  439.919,  439.655,  439.401,  439.157, 
      438.923,  438.698,  438.480,  438.269,  438.062,  437.859,  437.357,  436.863,  436.373,  435.885, 
      435.400,  434.915,  434.433,  433.955,  433.482,  433.003,  432.523,  432.046,  431.573,  431.097, 
      430.623,  430.151,  429.685,  429.221,  428.763,  428.310,  427.851,  427.396,  426.946,  426.499, 
      426.059,  425.622,  425.160,  424.703,  424.252,  423.808,  422.893,  421.678,  420.440,  419.200, 
      417.965,  416.742,  415.535,  414.342,  413.159,  411.989,  410.827,  409.674,  408.526,  407.379, 
      406.237,  405.098,  403.972,  402.858,  401.751,  400.658,  399.584,  398.532,  397.508,  396.514, 
      395.548,  394.611,  393.701,  392.819,  391.963,  391.138,  390.342,  389.577,  388.844,  388.146, 
      387.475,  386.827,  386.202,  385.604,  385.033,  384.486,  383.960,  383.453,  382.974,  382.525, 
      382.102,  381.710,  381.347,  380.990,  380.645,  380.323, ],
    [ 447.845,  447.320,  446.864,  446.445,  446.037,  445.638,  445.249,  444.870,  444.500,  444.141, 
      443.792,  443.455,  443.130,  442.815,  442.514,  442.225,  441.948,  441.683,  441.428,  441.183, 
      440.948,  440.722,  440.504,  440.292,  440.085,  439.880,  439.379,  438.884,  438.394,  437.906, 
      437.419,  436.933,  436.450,  435.971,  435.488,  435.003,  434.521,  434.042,  433.560,  433.080, 
      432.600,  432.128,  431.658,  431.194,  430.735,  430.271,  429.811,  429.355,  428.903,  428.457, 
      428.016,  427.549,  427.085,  426.627,  426.176,  425.732,  424.623,  423.385,  422.130,  420.883, 
      419.645,  418.420,  417.210,  416.013,  414.827,  413.652,  412.485,  411.322,  410.164,  409.010, 
      407.858,  406.711,  405.581,  404.464,  403.354,  402.258,  401.181,  400.127,  399.102,  398.105, 
      397.136,  396.194,  395.279,  394.389,  393.530,  392.700,  391.902,  391.132,  390.397,  389.696, 
      389.021,  388.366,  387.736,  387.134,  386.557,  386.006,  385.474,  384.962,  384.481,  384.027, 
      383.601,  383.207,  382.837,  382.472,  382.123,  381.799, ],
    [ 449.880,  449.353,  448.897,  448.477,  448.069,  447.671,  447.282,  446.903,  446.533,  446.173, 
      445.823,  445.486,  445.159,  444.844,  444.542,  444.251,  443.973,  443.707,  443.452,  443.207, 
      442.972,  442.745,  442.526,  442.314,  442.106,  441.902,  441.400,  440.904,  440.414,  439.925, 
      439.438,  438.950,  438.466,  437.979,  437.488,  437.001,  436.517,  436.030,  435.543,  435.057, 
      434.578,  434.102,  433.631,  433.167,  432.697,  432.231,  431.769,  431.313,  430.861,  430.414, 
      429.944,  429.474,  429.009,  428.551,  428.100,  427.583,  426.345,  425.075,  423.815,  422.565, 
      421.324,  420.096,  418.882,  417.682,  416.494,  415.311,  414.135,  412.964,  411.796,  410.634, 
      409.472,  408.322,  407.189,  406.067,  404.956,  403.857,  402.776,  401.722,  400.694,  399.694, 
      398.721,  397.774,  396.853,  395.959,  395.094,  394.263,  393.460,  392.687,  391.950,  391.245, 
      390.564,  389.904,  389.269,  388.661,  388.079,  387.523,  386.986,  386.470,  385.987,  385.529, 
      385.101,  384.704,  384.327,  383.953,  383.600,  383.273, ],
    [ 451.912,  451.385,  450.929,  450.510,  450.101,  449.702,  449.313,  448.932,  448.562,  448.201, 
      447.851,  447.513,  447.185,  446.869,  446.566,  446.275,  445.997,  445.731,  445.475,  445.229, 
      444.993,  444.766,  444.548,  444.335,  444.128,  443.923,  443.420,  442.924,  442.433,  441.944, 
      441.455,  440.967,  440.475,  439.979,  439.486,  438.997,  438.505,  438.013,  437.521,  437.035, 
      436.552,  436.075,  435.604,  435.129,  434.657,  434.190,  433.728,  433.270,  432.818,  432.346, 
      431.869,  431.398,  430.933,  430.475,  429.934,  429.311,  428.037,  426.763,  425.497,  424.244, 
      423.001,  421.769,  420.552,  419.350,  418.155,  416.966,  415.781,  414.600,  413.424,  412.251, 
      411.084,  409.931,  408.793,  407.670,  406.556,  405.454,  404.371,  403.315,  402.285,  401.281, 
      400.304,  399.351,  398.424,  397.525,  396.657,  395.823,  395.016,  394.240,  393.501,  392.793, 
      392.105,  391.439,  390.800,  390.186,  389.600,  389.039,  388.496,  387.978,  387.492,  387.031, 
      386.600,  386.201,  385.816,  385.434,  385.077,  384.747, ]])

_toff_tbl = np.array([
    [   0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000, 
        0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000, 
        0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000, 
        0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000, 
        0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000, 
        0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000, 
        0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000, 
        0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000, 
        0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000, 
        0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000,    0.000, 
        0.000,    0.000,    0.000,    0.000,    0.000,    0.000, ],
    [  93.500,   47.900,   28.800,   19.800,   14.900,   12.000,   10.100,    8.700,    7.600,    6.800, 
        6.200,    5.700,    5.200,    4.900,    4.600,    4.300,    4.100,    3.800,    3.600,    3.500, 
        3.300,    3.200,    3.000,    2.900,    2.800,    2.600,    2.400,    2.200,    2.000,    1.800, 
        1.700,    1.600,    1.500,    1.400,    1.300,    1.300,    1.200,    1.100,    1.100,    1.000, 
        1.000,    1.000,    0.900,    0.900,    0.900,    0.800,    0.800,    0.800,    0.700,    0.700, 
        0.700,    0.700,    0.700,    0.600,    0.600,    0.600,    0.600,    0.600,    0.500,    0.500, 
        0.500,    0.500,    0.500,    0.400,    0.400,    0.400,    0.400,    0.400,    0.400,    0.400, 
        0.400,    0.300,    0.300,    0.300,    0.300,    0.300,    0.300,    0.300,    0.300,    0.300, 
        0.300,    0.300,    0.300,    0.300,    0.300,    0.300,    0.300,    0.300,    0.300,    0.200, 
        0.200,    0.200,    0.200,    0.200,    0.200,    0.200,    0.200,    0.200,    0.200,    0.200, 
        0.200,    0.200,    0.200,    0.200,    0.200,    0.200, ],
    [  97.000,   69.100,   49.300,   36.300,   28.400,   23.200,   19.700,   17.100,   15.100,   13.500, 
      12.300,   11.300,   10.400,    9.700,    9.100,    8.600,    8.100,    7.700,    7.300,    6.900, 
        6.600,    6.300,    6.000,    5.800,    5.500,    5.300,    4.800,    4.300,    4.000,    3.700, 
        3.400,    3.200,    3.000,    2.800,    2.600,    2.500,    2.400,    2.300,    2.200,    2.100, 
        2.000,    1.900,    1.800,    1.800,    1.700,    1.600,    1.600,    1.500,    1.500,    1.400, 
        1.400,    1.400,    1.300,    1.300,    1.300,    1.200,    1.200,    1.100,    1.100,    1.000, 
        1.000,    0.900,    0.900,    0.900,    0.900,    0.800,    0.800,    0.800,    0.800,    0.700, 
        0.700,    0.700,    0.700,    0.700,    0.600,    0.600,    0.600,    0.600,    0.600,    0.600, 
        0.600,    0.600,    0.500,    0.500,    0.500,    0.500,    0.500,    0.500,    0.500,    0.500, 
        0.500,    0.500,    0.500,    0.500,    0.500,    0.500,    0.400,    0.400,    0.400,    0.400, 
        0.400,    0.400,    0.400,    0.400,    0.400,    0.400, ],
    [ 100.500,   80.000,   62.700,   48.900,   39.500,   33.100,   28.400,   24.900,   22.100,   19.900, 
      18.200,   16.700,   15.500,   14.400,   13.500,   12.800,   12.000,   11.400,   10.800,   10.300, 
        9.900,    9.400,    9.000,    8.600,    8.200,    7.900,    7.100,    6.500,    6.000,    5.500, 
        5.100,    4.800,    4.500,    4.200,    4.000,    3.700,    3.600,    3.400,    3.200,    3.100, 
        3.000,    2.900,    2.700,    2.600,    2.600,    2.500,    2.400,    2.300,    2.200,    2.200, 
        2.100,    2.000,    2.000,    1.900,    1.900,    1.800,    1.800,    1.700,    1.600,    1.500, 
        1.500,    1.400,    1.400,    1.300,    1.300,    1.200,    1.200,    1.200,    1.100,    1.100, 
        1.100,    1.000,    1.000,    1.000,    1.000,    0.900,    0.900,    0.900,    0.900,    0.900, 
        0.900,    0.800,    0.800,    0.800,    0.800,    0.800,    0.800,    0.800,    0.800,    0.700, 
        0.700,    0.700,    0.700,    0.700,    0.700,    0.700,    0.700,    0.700,    0.600,    0.600, 
        0.600,    0.600,    0.600,    0.600,    0.600,    0.600, ],
    [ 104.000,   86.800,   71.800,   58.100,   48.500,   41.500,   36.100,   31.900,   28.600,   25.900, 
      23.700,   21.900,   20.300,   19.000,   17.900,   16.900,   15.900,   15.100,   14.400,   13.700, 
      13.100,   12.500,   12.000,   11.400,   11.000,   10.500,    9.500,    8.600,    7.900,    7.300, 
        6.800,    6.300,    5.900,    5.600,    5.300,    5.000,    4.700,    4.500,    4.300,    4.100, 
        4.000,    3.800,    3.700,    3.500,    3.400,    3.300,    3.200,    3.100,    3.000,    2.900, 
        2.800,    2.700,    2.700,    2.600,    2.500,    2.500,    2.300,    2.200,    2.100,    2.100, 
        2.000,    1.900,    1.800,    1.800,    1.700,    1.600,    1.600,    1.500,    1.500,    1.500, 
        1.400,    1.400,    1.400,    1.300,    1.300,    1.300,    1.200,    1.200,    1.200,    1.200, 
        1.100,    1.100,    1.100,    1.100,    1.100,    1.000,    1.000,    1.000,    1.000,    1.000, 
        1.000,    1.000,    0.900,    0.900,    0.900,    0.900,    0.900,    0.900,    0.900,    0.900, 
        0.800,    0.800,    0.800,    0.800,    0.800,    0.800, ],
    [ 106.700,   91.600,   78.400,   65.000,   55.600,   48.500,   42.800,   38.300,   34.600,   31.500, 
      29.000,   26.800,   25.000,   23.400,   22.100,   20.800,   19.700,   18.800,   17.800,   17.000, 
      16.200,   15.500,   14.900,   14.200,   13.600,   13.100,   11.800,   10.800,    9.900,    9.100, 
        8.500,    7.900,    7.400,    7.000,    6.600,    6.200,    5.900,    5.600,    5.400,    5.200, 
        4.900,    4.800,    4.600,    4.400,    4.300,    4.100,    4.000,    3.800,    3.700,    3.600, 
        3.500,    3.400,    3.300,    3.200,    3.200,    3.100,    2.900,    2.800,    2.700,    2.600, 
        2.500,    2.400,    2.300,    2.200,    2.100,    2.100,    2.000,    1.900,    1.900,    1.800, 
        1.800,    1.700,    1.700,    1.600,    1.600,    1.600,    1.500,    1.500,    1.500,    1.400, 
        1.400,    1.400,    1.400,    1.400,    1.300,    1.300,    1.300,    1.300,    1.300,    1.200, 
        1.200,    1.200,    1.200,    1.200,    1.100,    1.100,    1.100,    1.100,    1.100,    1.100, 
        1.000,    1.000,    1.000,    1.000,    1.000,    1.000, ],
    [ 109.500,   95.700,   83.400,   70.300,   61.300,   54.300,   48.600,   43.800,   39.900,   36.600, 
      33.900,   31.500,   29.400,   27.600,   26.100,   24.700,   23.400,   22.300,   21.200,   20.200, 
      19.400,   18.500,   17.700,   17.000,   16.300,   15.600,   14.100,   12.900,   11.800,   10.900, 
      10.100,    9.500,    8.900,    8.300,    7.900,    7.500,    7.100,    6.800,    6.500,    6.200, 
        5.900,    5.700,    5.500,    5.300,    5.100,    4.900,    4.800,    4.600,    4.500,    4.300, 
        4.200,    4.100,    4.000,    3.900,    3.800,    3.700,    3.500,    3.400,    3.200,    3.100, 
        3.000,    2.800,    2.700,    2.600,    2.600,    2.500,    2.400,    2.300,    2.300,    2.200, 
        2.100,    2.100,    2.000,    2.000,    1.900,    1.900,    1.800,    1.800,    1.800,    1.700, 
        1.700,    1.700,    1.600,    1.600,    1.600,    1.600,    1.500,    1.500,    1.500,    1.500, 
        1.500,    1.400,    1.400,    1.400,    1.400,    1.400,    1.300,    1.300,    1.300,    1.300, 
        1.300,    1.200,    1.200,    1.200,    1.200,    1.200, ],
    [ 111.800,   99.100,   87.300,   74.400,   65.800,   59.100,   53.500,   48.700,   44.700,   41.300, 
      38.400,   35.800,   33.600,   31.600,   29.900,   28.400,   27.000,   25.700,   24.500,   23.400, 
      22.400,   21.400,   20.500,   19.700,   18.900,   18.100,   16.400,   15.000,   13.700,   12.700, 
      11.800,   11.000,   10.300,    9.700,    9.200,    8.700,    8.300,    7.900,    7.500,    7.200, 
        6.900,    6.600,    6.400,    6.200,    5.900,    5.700,    5.600,    5.400,    5.200,    5.100, 
        4.900,    4.800,    4.600,    4.500,    4.400,    4.300,    4.100,    3.900,    3.700,    3.600, 
        3.400,    3.300,    3.200,    3.100,    3.000,    2.900,    2.800,    2.700,    2.600,    2.600, 
        2.500,    2.400,    2.400,    2.300,    2.200,    2.200,    2.100,    2.100,    2.100,    2.000, 
        2.000,    2.000,    1.900,    1.900,    1.900,    1.800,    1.800,    1.800,    1.800,    1.700, 
        1.700,    1.700,    1.600,    1.600,    1.600,    1.600,    1.600,    1.500,    1.500,    1.500, 
        1.500,    1.400,    1.400,    1.400,    1.400,    1.400, ],
    [ 113.800,  101.900,   90.000,   77.700,   69.600,   63.200,   57.700,   53.100,   49.000,   45.500, 
      42.500,   39.800,   37.500,   35.400,   33.600,   31.900,   30.400,   29.000,   27.700,   26.500, 
      25.300,   24.300,   23.300,   22.300,   21.400,   20.600,   18.600,   17.000,   15.600,   14.500, 
      13.400,   12.500,   11.800,   11.100,   10.500,    9.900,    9.400,    9.000,    8.600,    8.200, 
        7.900,    7.600,    7.300,    7.000,    6.800,    6.600,    6.300,    6.100,    5.900,    5.800, 
        5.600,    5.500,    5.300,    5.200,    5.000,    4.900,    4.700,    4.500,    4.300,    4.100, 
        3.900,    3.800,    3.700,    3.500,    3.400,    3.300,    3.200,    3.100,    3.000,    2.900, 
        2.800,    2.800,    2.700,    2.600,    2.600,    2.500,    2.500,    2.400,    2.400,    2.300, 
        2.300,    2.200,    2.200,    2.200,    2.100,    2.100,    2.100,    2.000,    2.000,    2.000, 
        1.900,    1.900,    1.900,    1.900,    1.800,    1.800,    1.800,    1.800,    1.700,    1.700, 
        1.700,    1.700,    1.600,    1.600,    1.600,    1.600, ],
    [ 115.900,  104.100,   92.400,   80.300,   72.700,   66.600,   61.400,   56.800,   52.900,   49.400, 
      46.300,   43.500,   41.100,   39.000,   37.000,   35.300,   33.600,   32.200,   30.700,   29.400, 
      28.200,   27.100,   26.000,   24.900,   23.900,   23.000,   20.800,   19.000,   17.500,   16.200, 
      15.100,   14.100,   13.200,   12.400,   11.800,   11.100,   10.600,   10.100,    9.600,    9.200, 
        8.900,    8.500,    8.200,    7.900,    7.600,    7.400,    7.100,    6.900,    6.700,    6.500, 
        6.300,    6.100,    6.000,    5.800,    5.700,    5.500,    5.300,    5.000,    4.800,    4.600, 
        4.400,    4.300,    4.100,    4.000,    3.800,    3.700,    3.600,    3.500,    3.400,    3.300, 
        3.200,    3.100,    3.000,    3.000,    2.900,    2.800,    2.800,    2.700,    2.700,    2.600, 
        2.600,    2.500,    2.500,    2.400,    2.400,    2.400,    2.300,    2.300,    2.300,    2.200, 
        2.200,    2.200,    2.100,    2.100,    2.100,    2.000,    2.000,    2.000,    1.900,    1.900, 
        1.900,    1.900,    1.800,    1.800,    1.800,    1.800, ],
    [ 117.300,  106.100,   94.100,   82.400,   75.300,   69.600,   64.600,   60.200,   56.300,   52.800, 
      49.800,   47.000,   44.500,   42.300,   40.300,   38.500,   36.800,   35.200,   33.700,   32.300, 
      31.000,   29.700,   28.600,   27.400,   26.300,   25.300,   23.000,   21.000,   19.400,   17.900, 
      16.700,   15.600,   14.600,   13.800,   13.000,   12.400,   11.700,   11.200,   10.700,   10.200, 
        9.800,    9.500,    9.100,    8.800,    8.500,    8.200,    7.900,    7.700,    7.400,    7.200, 
        7.000,    6.800,    6.600,    6.500,    6.300,    6.100,    5.800,    5.600,    5.300,    5.100, 
        4.900,    4.700,    4.600,    4.400,    4.300,    4.100,    4.000,    3.900,    3.800,    3.700, 
        3.600,    3.500,    3.400,    3.300,    3.200,    3.100,    3.100,    3.000,    2.900,    2.900, 
        2.800,    2.800,    2.700,    2.700,    2.700,    2.600,    2.600,    2.500,    2.500,    2.500, 
        2.400,    2.400,    2.400,    2.300,    2.300,    2.300,    2.200,    2.200,    2.200,    2.100, 
        2.100,    2.100,    2.000,    2.000,    2.000,    2.000, ],
    [ 118.700,  107.800,   95.800,   84.100,   77.600,   72.200,   67.400,   63.200,   59.400,   56.000, 
      53.000,   50.200,   47.700,   45.500,   43.400,   41.500,   39.800,   38.100,   36.500,   35.100, 
      33.700,   32.400,   31.100,   29.900,   28.700,   27.600,   25.100,   23.000,   21.200,   19.600, 
      18.300,   17.100,   16.000,   15.100,   14.300,   13.600,   12.900,   12.300,   11.700,   11.300, 
      10.800,   10.400,   10.000,    9.600,    9.300,    9.000,    8.700,    8.400,    8.200,    7.900, 
        7.700,    7.500,    7.300,    7.100,    6.900,    6.700,    6.400,    6.100,    5.900,    5.600, 
        5.400,    5.200,    5.000,    4.800,    4.700,    4.500,    4.400,    4.300,    4.100,    4.000, 
        3.900,    3.800,    3.700,    3.600,    3.500,    3.400,    3.400,    3.300,    3.200,    3.200, 
        3.100,    3.100,    3.000,    3.000,    2.900,    2.900,    2.800,    2.800,    2.800,    2.700, 
        2.700,    2.600,    2.600,    2.600,    2.500,    2.500,    2.400,    2.400,    2.400,    2.300, 
        2.300,    2.300,    2.200,    2.200,    2.200,    2.200, ],
    [ 120.000,  109.100,   97.400,   85.600,   79.500,   74.400,   69.900,   65.800,   62.100,   58.900, 
      55.900,   53.100,   50.600,   48.400,   46.300,   44.400,   42.600,   40.900,   39.200,   37.700, 
      36.300,   34.900,   33.600,   32.300,   31.000,   29.800,   27.200,   24.900,   23.000,   21.300, 
      19.800,   18.500,   17.400,   16.400,   15.500,   14.700,   14.000,   13.400,   12.800,   12.300, 
      11.800,   11.300,   10.900,   10.500,   10.100,    9.800,    9.500,    9.200,    8.900,    8.600, 
        8.400,    8.200,    7.900,    7.700,    7.500,    7.300,    7.000,    6.700,    6.400,    6.100, 
        5.900,    5.700,    5.500,    5.300,    5.100,    4.900,    4.800,    4.600,    4.500,    4.400, 
        4.300,    4.200,    4.000,    3.900,    3.800,    3.800,    3.700,    3.600,    3.500,    3.500, 
        3.400,    3.400,    3.300,    3.200,    3.200,    3.100,    3.100,    3.000,    3.000,    3.000, 
        2.900,    2.900,    2.800,    2.800,    2.700,    2.700,    2.700,    2.600,    2.600,    2.600, 
        2.500,    2.500,    2.400,    2.400,    2.400,    2.300, ],
    [ 120.900,  110.400,   98.300,   86.800,   81.200,   76.400,   72.100,   68.200,   64.700,   61.500, 
      58.600,   55.800,   53.400,   51.200,   49.100,   47.200,   45.300,   43.500,   41.800,   40.300, 
      38.800,   37.300,   35.900,   34.600,   33.300,   32.000,   29.200,   26.800,   24.700,   22.900, 
      21.400,   20.000,   18.800,   17.700,   16.800,   15.900,   15.200,   14.500,   13.800,   13.200, 
      12.700,   12.200,   11.800,   11.400,   11.000,   10.600,   10.200,    9.900,    9.600,    9.300, 
        9.100,    8.800,    8.600,    8.400,    8.200,    8.000,    7.600,    7.200,    6.900,    6.600, 
        6.400,    6.100,    5.900,    5.700,    5.500,    5.300,    5.200,    5.000,    4.900,    4.700, 
        4.600,    4.500,    4.400,    4.300,    4.200,    4.100,    4.000,    3.900,    3.800,    3.800, 
        3.700,    3.600,    3.600,    3.500,    3.500,    3.400,    3.400,    3.300,    3.300,    3.200, 
        3.200,    3.100,    3.100,    3.000,    3.000,    2.900,    2.900,    2.800,    2.800,    2.800, 
        2.700,    2.700,    2.700,    2.600,    2.600,    2.500, ],
    [ 121.800,  111.100,   98.800,   87.900,   82.700,   78.200,   74.100,   70.300,   66.900,   63.900, 
      61.000,   58.400,   56.000,   53.700,   51.700,   49.700,   47.900,   46.100,   44.300,   42.700, 
      41.200,   39.700,   38.200,   36.800,   35.400,   34.100,   31.200,   28.600,   26.500,   24.600, 
      22.900,   21.400,   20.200,   19.000,   18.000,   17.100,   16.300,   15.500,   14.800,   14.200, 
      13.700,   13.100,   12.700,   12.200,   11.800,   11.400,   11.000,   10.700,   10.300,   10.000, 
        9.800,    9.500,    9.200,    9.000,    8.800,    8.600,    8.200,    7.800,    7.500,    7.200, 
        6.900,    6.600,    6.400,    6.200,    5.900,    5.800,    5.600,    5.400,    5.300,    5.100, 
        5.000,    4.800,    4.700,    4.600,    4.500,    4.400,    4.300,    4.200,    4.100,    4.000, 
        4.000,    3.900,    3.800,    3.800,    3.700,    3.700,    3.600,    3.600,    3.500,    3.500, 
        3.400,    3.300,    3.300,    3.300,    3.200,    3.200,    3.100,    3.100,    3.000,    3.000, 
        2.900,    2.900,    2.900,    2.800,    2.800,    2.700, ],
    [ 122.600,  111.600,   99.300,   88.800,   84.000,   79.700,   75.900,   72.300,   69.000,   66.100, 
      63.300,   60.700,   58.400,   56.200,   54.100,   52.200,   50.300,   48.500,   46.700,   45.100, 
      43.500,   41.900,   40.500,   39.000,   37.600,   36.200,   33.100,   30.400,   28.200,   26.100, 
      24.400,   22.800,   21.500,   20.300,   19.200,   18.300,   17.400,   16.600,   15.900,   15.200, 
      14.600,   14.100,   13.500,   13.100,   12.600,   12.200,   11.800,   11.400,   11.100,   10.800, 
      10.400,   10.200,    9.900,    9.600,    9.400,    9.200,    8.700,    8.300,    8.000,    7.700, 
        7.400,    7.100,    6.800,    6.600,    6.400,    6.200,    6.000,    5.800,    5.600,    5.500, 
        5.300,    5.200,    5.100,    4.900,    4.800,    4.700,    4.600,    4.500,    4.400,    4.300, 
        4.300,    4.200,    4.100,    4.100,    4.000,    3.900,    3.900,    3.800,    3.800,    3.700, 
        3.600,    3.600,    3.500,    3.500,    3.400,    3.400,    3.300,    3.300,    3.200,    3.200, 
        3.100,    3.100,    3.100,    3.000,    3.000,    2.900, ],
    [ 123.100,  112.400,  100.000,   90.000,   85.200,   81.200,   77.500,   74.100,   70.900,   68.100, 
      65.500,   62.900,   60.600,   58.500,   56.400,   54.500,   52.600,   50.800,   49.000,   47.300, 
      45.700,   44.100,   42.600,   41.100,   39.600,   38.200,   35.000,   32.200,   29.800,   27.700, 
      25.900,   24.200,   22.800,   21.600,   20.400,   19.400,   18.500,   17.600,   16.900,   16.200, 
      15.500,   15.000,   14.400,   13.900,   13.400,   13.000,   12.600,   12.200,   11.800,   11.500, 
      11.100,   10.800,   10.500,   10.300,   10.000,    9.800,    9.300,    8.900,    8.500,    8.200, 
        7.800,    7.500,    7.300,    7.000,    6.800,    6.600,    6.400,    6.200,    6.000,    5.800, 
        5.700,    5.500,    5.400,    5.300,    5.100,    5.000,    4.900,    4.800,    4.700,    4.600, 
        4.500,    4.500,    4.400,    4.300,    4.300,    4.200,    4.100,    4.100,    4.000,    3.900, 
        3.900,    3.800,    3.800,    3.700,    3.700,    3.600,    3.600,    3.500,    3.500,    3.400, 
        3.400,    3.300,    3.300,    3.200,    3.200,    3.100, ],
    [ 123.400,  112.600,  100.700,   90.000,   86.300,   82.500,   79.000,   75.700,   72.700,   70.000, 
      67.400,   64.900,   62.700,   60.600,   58.600,   56.700,   54.800,   53.000,   51.200,   49.500, 
      47.800,   46.200,   44.700,   43.100,   41.600,   40.100,   36.800,   33.900,   31.400,   29.200, 
      27.300,   25.600,   24.100,   22.800,   21.600,   20.500,   19.600,   18.700,   17.900,   17.100, 
      16.500,   15.900,   15.300,   14.700,   14.200,   13.800,   13.300,   12.900,   12.500,   12.200, 
      11.800,   11.500,   11.200,   10.900,   10.600,   10.400,    9.900,    9.400,    9.000,    8.700, 
        8.300,    8.000,    7.700,    7.500,    7.200,    7.000,    6.800,    6.600,    6.400,    6.200, 
        6.000,    5.900,    5.700,    5.600,    5.400,    5.300,    5.200,    5.100,    5.000,    4.900, 
        4.800,    4.700,    4.700,    4.600,    4.500,    4.500,    4.400,    4.300,    4.300,    4.200, 
        4.100,    4.100,    4.000,    3.900,    3.900,    3.800,    3.800,    3.700,    3.700,    3.600, 
        3.600,    3.500,    3.500,    3.400,    3.400,    3.300, ],
    [ 123.800,  112.800,  100.700,   91.200,   87.300,   83.700,   80.400,   77.200,   74.300,   71.700, 
      69.300,   66.800,   64.700,   62.600,   60.700,   58.800,   56.900,   55.100,   53.300,   51.600, 
      49.900,   48.300,   46.700,   45.100,   43.500,   42.000,   38.600,   35.600,   33.000,   30.700, 
      28.700,   27.000,   25.400,   24.000,   22.800,   21.700,   20.600,   19.700,   18.900,   18.100, 
      17.400,   16.700,   16.100,   15.600,   15.000,   14.500,   14.100,   13.600,   13.200,   12.800, 
      12.500,   12.100,   11.800,   11.500,   11.200,   11.000,   10.500,   10.000,    9.600,    9.200, 
        8.800,    8.500,    8.200,    7.900,    7.600,    7.400,    7.200,    6.900,    6.700,    6.600, 
        6.400,    6.200,    6.100,    5.900,    5.800,    5.600,    5.500,    5.400,    5.300,    5.200, 
        5.100,    5.000,    4.900,    4.900,    4.800,    4.700,    4.600,    4.600,    4.500,    4.400, 
        4.400,    4.300,    4.200,    4.200,    4.100,    4.100,    4.000,    3.900,    3.900,    3.800, 
        3.800,    3.700,    3.700,    3.600,    3.600,    3.500, ],
    [ 124.000,  113.100,  100.800,   91.900,   88.200,   84.800,   81.600,   78.600,   75.900,   73.300, 
      71.000,   68.600,   66.500,   64.600,   62.700,   60.800,   58.900,   57.100,   55.300,   53.600, 
      51.900,   50.200,   48.600,   47.000,   45.400,   43.800,   40.300,   37.200,   34.600,   32.200, 
      30.100,   28.300,   26.700,   25.200,   24.000,   22.800,   21.700,   20.700,   19.900,   19.100, 
      18.300,   17.600,   17.000,   16.400,   15.800,   15.300,   14.800,   14.400,   13.900,   13.500, 
      13.200,   12.800,   12.500,   12.200,   11.900,   11.600,   11.000,   10.500,   10.100,    9.700, 
        9.300,    8.900,    8.600,    8.300,    8.100,    7.800,    7.600,    7.300,    7.100,    6.900, 
        6.700,    6.600,    6.400,    6.200,    6.100,    5.900,    5.800,    5.700,    5.600,    5.500, 
        5.400,    5.300,    5.200,    5.100,    5.100,    5.000,    4.900,    4.800,    4.800,    4.700, 
        4.600,    4.500,    4.500,    4.400,    4.300,    4.300,    4.200,    4.200,    4.100,    4.000, 
        4.000,    3.900,    3.900,    3.800,    3.800,    3.700, ],
    [ 124.100,  113.400,  100.900,   92.500,   89.000,   85.800,   82.800,   79.900,   77.300,   74.900, 
      72.600,   70.300,   68.300,   66.400,   64.500,   62.700,   60.800,   59.100,   57.200,   55.500, 
      53.800,   52.100,   50.500,   48.800,   47.100,   45.600,   41.900,   38.800,   36.100,   33.600, 
      31.500,   29.600,   27.900,   26.400,   25.100,   23.900,   22.800,   21.800,   20.800,   20.000, 
      19.200,   18.500,   17.800,   17.200,   16.600,   16.100,   15.600,   15.100,   14.700,   14.200, 
      13.800,   13.500,   13.100,   12.800,   12.500,   12.200,   11.600,   11.100,   10.600,   10.200, 
        9.800,    9.400,    9.100,    8.800,    8.500,    8.200,    7.900,    7.700,    7.500,    7.300, 
        7.100,    6.900,    6.700,    6.600,    6.400,    6.200,    6.100,    6.000,    5.900,    5.800, 
        5.700,    5.600,    5.500,    5.400,    5.300,    5.200,    5.200,    5.100,    5.000,    4.900, 
        4.800,    4.800,    4.700,    4.600,    4.600,    4.500,    4.400,    4.400,    4.300,    4.300, 
        4.200,    4.100,    4.100,    4.000,    4.000,    3.900, ],
    [ 124.300,  113.500,  101.100,   93.200,   90.000,   86.800,   83.900,   81.100,   78.600,   76.300, 
      74.100,   71.900,   70.000,   68.100,   66.300,   64.500,   62.700,   60.900,   59.000,   57.300, 
      55.600,   53.900,   52.200,   50.600,   48.900,   47.300,   43.600,   40.300,   37.500,   35.000, 
      32.800,   30.900,   29.200,   27.600,   26.200,   25.000,   23.800,   22.800,   21.800,   20.900, 
      20.100,   19.400,   18.700,   18.000,   17.400,   16.900,   16.300,   15.800,   15.400,   14.900, 
      14.500,   14.100,   13.700,   13.400,   13.100,   12.700,   12.200,   11.600,   11.100,   10.700, 
      10.300,    9.900,    9.500,    9.200,    8.900,    8.600,    8.300,    8.100,    7.900,    7.600, 
        7.400,    7.200,    7.100,    6.900,    6.700,    6.600,    6.400,    6.300,    6.200,    6.100, 
        6.000,    5.900,    5.800,    5.700,    5.600,    5.500,    5.400,    5.300,    5.300,    5.200, 
        5.100,    5.000,    4.900,    4.900,    4.800,    4.700,    4.700,    4.600,    4.500,    4.500, 
        4.400,    4.300,    4.300,    4.200,    4.200,    4.100, ],
    [ 124.500,  113.500,  101.200,   93.800,   90.700,   87.700,   84.900,   82.300,   79.900,   77.700, 
      75.600,   73.400,   71.500,   69.700,   68.000,   66.200,   64.400,   62.700,   60.800,   59.100, 
      57.400,   55.600,   54.000,   52.300,   50.500,   48.900,   45.100,   41.800,   39.000,   36.400, 
      34.100,   32.100,   30.400,   28.800,   27.300,   26.000,   24.800,   23.700,   22.800,   21.900, 
      21.000,   20.200,   19.500,   18.800,   18.200,   17.600,   17.100,   16.600,   16.100,   15.600, 
      15.200,   14.800,   14.400,   14.000,   13.700,   13.300,   12.700,   12.200,   11.600,   11.200, 
      10.700,   10.300,   10.000,    9.600,    9.300,    9.000,    8.700,    8.500,    8.200,    8.000, 
        7.800,    7.600,    7.400,    7.200,    7.000,    6.900,    6.700,    6.600,    6.500,    6.300, 
        6.200,    6.100,    6.000,    5.900,    5.800,    5.800,    5.700,    5.600,    5.500,    5.400, 
        5.300,    5.300,    5.200,    5.100,    5.000,    5.000,    4.900,    4.800,    4.700,    4.700, 
        4.600,    4.500,    4.500,    4.400,    4.400,    4.300, ],
    [ 124.500,  113.600,  101.400,   94.400,   91.300,   88.500,   85.900,   83.300,   81.100,   79.000, 
      77.000,   74.800,   73.000,   71.300,   69.600,   67.900,   66.100,   64.400,   62.500,   60.800, 
      59.000,   57.300,   55.600,   53.900,   52.200,   50.500,   46.700,   43.300,   40.400,   37.700, 
      35.400,   33.400,   31.500,   29.900,   28.400,   27.100,   25.800,   24.700,   23.700,   22.800, 
      21.900,   21.100,   20.400,   19.700,   19.000,   18.400,   17.800,   17.300,   16.800,   16.300, 
      15.800,   15.400,   15.000,   14.600,   14.300,   13.900,   13.300,   12.700,   12.200,   11.700, 
      11.200,   10.800,   10.400,   10.100,    9.700,    9.400,    9.100,    8.900,    8.600,    8.400, 
        8.100,    7.900,    7.700,    7.500,    7.300,    7.200,    7.000,    6.900,    6.800,    6.600, 
        6.500,    6.400,    6.300,    6.200,    6.100,    6.000,    5.900,    5.800,    5.800,    5.700, 
        5.600,    5.500,    5.400,    5.300,    5.300,    5.200,    5.100,    5.000,    5.000,    4.900, 
        4.800,    4.800,    4.700,    4.600,    4.600,    4.500, ],
    [ 124.500,  113.600,  101.700,   94.900,   92.100,   89.300,   86.800,   84.400,   82.200,   80.200, 
      78.300,   76.200,   74.500,   72.800,   71.100,   69.500,   67.700,   66.000,   64.100,   62.400, 
      60.700,   58.900,   57.200,   55.500,   53.700,   52.000,   48.100,   44.700,   41.700,   39.100, 
      36.700,   34.600,   32.700,   31.000,   29.500,   28.100,   26.800,   25.700,   24.600,   23.700, 
      22.800,   22.000,   21.200,   20.500,   19.800,   19.100,   18.500,   18.000,   17.500,   17.000, 
      16.500,   16.100,   15.600,   15.200,   14.900,   14.500,   13.800,   13.200,   12.700,   12.200, 
      11.700,   11.300,   10.900,   10.500,   10.100,    9.800,    9.500,    9.200,    9.000,    8.700, 
        8.500,    8.300,    8.100,    7.900,    7.700,    7.500,    7.300,    7.200,    7.000,    6.900, 
        6.800,    6.700,    6.600,    6.500,    6.400,    6.300,    6.200,    6.100,    6.000,    5.900, 
        5.800,    5.700,    5.600,    5.600,    5.500,    5.400,    5.300,    5.300,    5.200,    5.100, 
        5.000,    5.000,    4.900,    4.800,    4.800,    4.700, ],
    [ 124.600,  113.700,  101.900,   95.400,   92.800,   90.000,   87.700,   85.300,   83.300,   81.300, 
      79.500,   77.500,   75.800,   74.200,   72.600,   71.000,   69.200,   67.500,   65.700,   64.000, 
      62.200,   60.500,   58.800,   57.000,   55.200,   53.500,   49.600,   46.100,   43.000,   40.300, 
      37.900,   35.700,   33.800,   32.100,   30.600,   29.100,   27.800,   26.600,   25.600,   24.600, 
      23.600,   22.800,   22.000,   21.200,   20.500,   19.900,   19.300,   18.700,   18.100,   17.600, 
      17.100,   16.700,   16.300,   15.900,   15.500,   15.100,   14.400,   13.800,   13.200,   12.700, 
      12.200,   11.700,   11.300,   10.900,   10.600,   10.200,    9.900,    9.600,    9.300,    9.100, 
        8.800,    8.600,    8.400,    8.200,    8.000,    7.800,    7.600,    7.500,    7.300,    7.200, 
        7.100,    7.000,    6.800,    6.700,    6.600,    6.500,    6.400,    6.300,    6.200,    6.200, 
        6.100,    6.000,    5.900,    5.800,    5.700,    5.600,    5.500,    5.500,    5.400,    5.300, 
        5.200,    5.200,    5.100,    5.000,    5.000,    4.900, ],
    [ 124.700,  114.000,  102.600,   96.800,   94.400,   92.000,   90.000,   87.600,   85.700,   84.000, 
      82.400,   80.500,   79.000,   77.600,   76.100,   74.600,   72.800,   71.200,   69.300,   67.600, 
      65.900,   64.200,   62.400,   60.600,   58.800,   57.000,   52.900,   49.400,   46.200,   43.400, 
      40.800,   38.600,   36.600,   34.800,   33.100,   31.600,   30.200,   29.000,   27.800,   26.800, 
      25.800,   24.900,   24.000,   23.200,   22.400,   21.700,   21.100,   20.400,   19.800,   19.300, 
      18.800,   18.300,   17.800,   17.400,   16.900,   16.500,   15.800,   15.100,   14.500,   13.900, 
      13.300,   12.900,   12.400,   12.000,   11.600,   11.200,   10.900,   10.600,   10.300,   10.000, 
        9.700,    9.500,    9.200,    9.000,    8.800,    8.600,    8.400,    8.200,    8.100,    7.900, 
        7.800,    7.700,    7.500,    7.400,    7.300,    7.200,    7.100,    7.000,    6.900,    6.800, 
        6.700,    6.600,    6.500,    6.400,    6.300,    6.200,    6.100,    6.000,    5.900,    5.800, 
        5.800,    5.700,    5.600,    5.500,    5.500,    5.400, ],
    [ 124.900,  114.300,  103.300,   98.000,   95.800,   93.700,   91.600,   90.000,   88.000,   86.500, 
      85.000,   83.100,   81.900,   80.600,   79.300,   77.800,   76.100,   74.500,   72.700,   71.000, 
      69.300,   67.500,   65.800,   63.900,   62.000,   60.200,   56.000,   52.400,   49.100,   46.200, 
      43.600,   41.300,   39.200,   37.300,   35.600,   34.000,   32.600,   31.200,   30.000,   28.900, 
      27.800,   26.900,   26.000,   25.100,   24.300,   23.500,   22.800,   22.200,   21.500,   20.900, 
      20.400,   19.800,   19.300,   18.900,   18.400,   18.000,   17.200,   16.400,   15.700,   15.100, 
      14.500,   14.000,   13.500,   13.100,   12.600,   12.200,   11.900,   11.500,   11.200,   10.900, 
      10.600,   10.300,   10.100,    9.800,    9.600,    9.300,    9.100,    9.000,    8.800,    8.600, 
        8.500,    8.300,    8.200,    8.100,    8.000,    7.800,    7.700,    7.600,    7.500,    7.400, 
        7.300,    7.200,    7.100,    7.000,    6.900,    6.800,    6.700,    6.600,    6.500,    6.400, 
        6.300,    6.200,    6.100,    6.000,    5.900,    5.900, ],
    [ 125.100,  114.600,  104.000,   99.200,   97.200,   95.300,   93.300,   91.300,   90.000,   88.700, 
      87.400,   85.500,   84.600,   83.400,   82.200,   80.800,   79.100,   77.600,   75.700,   74.100, 
      72.300,   70.600,   68.800,   66.900,   65.000,   63.200,   58.900,   55.200,   51.900,   48.900, 
      46.200,   43.800,   41.700,   39.700,   38.000,   36.300,   34.800,   33.400,   32.100,   31.000, 
      29.900,   28.800,   27.900,   27.000,   26.100,   25.300,   24.600,   23.800,   23.200,   22.500, 
      21.900,   21.400,   20.800,   20.300,   19.900,   19.400,   18.500,   17.700,   17.000,   16.300, 
      15.700,   15.100,   14.600,   14.100,   13.600,   13.200,   12.800,   12.400,   12.100,   11.800, 
      11.400,   11.200,   10.900,   10.600,   10.300,   10.100,    9.900,    9.700,    9.500,    9.300, 
        9.200,    9.000,    8.900,    8.700,    8.600,    8.500,    8.400,    8.200,    8.100,    8.000, 
        7.900,    7.700,    7.600,    7.500,    7.400,    7.300,    7.200,    7.100,    7.000,    6.900, 
        6.800,    6.700,    6.600,    6.500,    6.400,    6.300, ],
    [ 125.300,  115.000,  104.800,  100.400,   98.500,   96.700,   94.900,   93.200,   92.100,   91.000, 
      90.000,   87.500,   87.100,   86.100,   84.900,   83.600,   81.900,   80.400,   78.500,   76.900, 
      75.200,   73.400,   71.600,   69.700,   67.800,   65.900,   61.600,   57.800,   54.500,   51.400, 
      48.700,   46.200,   44.000,   42.000,   40.200,   38.500,   36.900,   35.500,   34.200,   32.900, 
      31.800,   30.700,   29.700,   28.800,   27.900,   27.100,   26.300,   25.500,   24.800,   24.100, 
      23.500,   22.900,   22.300,   21.800,   21.300,   20.800,   19.900,   19.000,   18.200,   17.500, 
      16.800,   16.200,   15.700,   15.200,   14.700,   14.200,   13.800,   13.400,   13.000,   12.600, 
      12.300,   12.000,   11.700,   11.400,   11.100,   10.900,   10.600,   10.400,   10.200,   10.100, 
        9.900,    9.700,    9.600,    9.400,    9.300,    9.100,    9.000,    8.900,    8.700,    8.600, 
        8.500,    8.300,    8.200,    8.100,    8.000,    7.900,    7.800,    7.600,    7.500,    7.400, 
        7.300,    7.200,    7.100,    7.000,    6.900,    6.800, ],
    [ 125.600,  115.500,  105.600,  101.500,   99.800,   98.100,   96.400,   94.800,   93.900,   93.100, 
      92.100,   89.100,   90.000,   88.500,   87.400,   86.200,   84.500,   83.000,   81.100,   79.500, 
      77.800,   76.000,   74.200,   72.300,   70.300,   68.400,   64.000,   60.200,   56.800,   53.800, 
      51.000,   48.500,   46.300,   44.200,   42.400,   40.600,   39.000,   37.500,   36.200,   34.900, 
      33.700,   32.600,   31.500,   30.500,   29.600,   28.700,   27.900,   27.100,   26.400,   25.700, 
      25.000,   24.400,   23.800,   23.200,   22.700,   22.200,   21.200,   20.300,   19.500,   18.700, 
      18.000,   17.300,   16.800,   16.200,   15.700,   15.200,   14.700,   14.300,   13.900,   13.500, 
      13.200,   12.800,   12.500,   12.200,   11.900,   11.600,   11.400,   11.200,   11.000,   10.800, 
      10.600,   10.400,   10.200,   10.100,    9.900,    9.800,    9.600,    9.500,    9.300,    9.200, 
        9.100,    8.900,    8.800,    8.700,    8.600,    8.400,    8.300,    8.200,    8.100,    8.000, 
        7.800,    7.700,    7.600,    7.500,    7.400,    7.300, ],
    [ 125.900,  116.000,  106.500,  102.600,  101.000,   99.400,   97.800,   96.300,   95.700,   95.100, 
      94.400,   91.900,   91.600,   90.700,   90.000,   88.600,   86.900,   85.500,   83.500,   81.900, 
      80.200,   78.400,   76.600,   74.600,   72.600,   70.600,   66.300,   62.400,   59.100,   56.000, 
      53.100,   50.600,   48.400,   46.300,   44.400,   42.600,   41.000,   39.500,   38.100,   36.700, 
      35.500,   34.400,   33.300,   32.300,   31.300,   30.400,   29.500,   28.700,   27.900,   27.200, 
      26.500,   25.800,   25.200,   24.600,   24.100,   23.500,   22.500,   21.500,   20.700,   19.900, 
      19.100,   18.400,   17.800,   17.200,   16.700,   16.200,   15.700,   15.200,   14.800,   14.400, 
      14.000,   13.700,   13.300,   13.000,   12.700,   12.400,   12.100,   11.900,   11.700,   11.500, 
      11.300,   11.100,   10.900,   10.700,   10.600,   10.400,   10.300,   10.100,   10.000,    9.800, 
        9.700,    9.500,    9.400,    9.300,    9.100,    9.000,    8.900,    8.700,    8.600,    8.500, 
        8.400,    8.200,    8.100,    8.000,    7.900,    7.800, ],
    [ 126.200,  116.500,  107.300,  103.700,  102.200,  100.700,   99.200,   98.400,   97.400,   97.000, 
      96.600,   94.700,   93.800,   93.300,   92.200,   90.700,   89.100,   87.800,   85.800,   84.200, 
      82.400,   80.600,   78.800,   76.800,   74.700,   72.700,   68.300,   64.500,   61.100,   58.000, 
      55.200,   52.700,   50.400,   48.300,   46.400,   44.500,   42.900,   41.300,   39.900,   38.500, 
      37.300,   36.100,   35.000,   33.900,   32.900,   32.000,   31.100,   30.200,   29.400,   28.700, 
      27.900,   27.300,   26.600,   26.000,   25.400,   24.800,   23.800,   22.800,   21.900,   21.000, 
      20.200,   19.500,   18.900,   18.300,   17.700,   17.100,   16.600,   16.100,   15.700,   15.300, 
      14.900,   14.500,   14.100,   13.800,   13.400,   13.100,   12.900,   12.600,   12.400,   12.200, 
      12.000,   11.800,   11.600,   11.400,   11.200,   11.100,   10.900,   10.700,   10.600,   10.400, 
      10.300,   10.100,   10.000,    9.800,    9.700,    9.500,    9.400,    9.300,    9.100,    9.000, 
        8.900,    8.800,    8.600,    8.500,    8.400,    8.300, ],
    [ 126.600,  117.000,  108.200,  104.800,  103.300,  101.900,  100.500,  100.000,   99.500,   98.800, 
      98.600,   96.900,   95.900,   95.600,   94.400,   92.800,   91.200,   90.000,   87.900,   86.300, 
      84.500,   82.700,   80.900,   78.800,   76.700,   74.700,   70.300,   66.400,   63.000,   59.900, 
      57.100,   54.600,   52.300,   50.200,   48.200,   46.400,   44.700,   43.100,   41.600,   40.300, 
      39.000,   37.800,   36.600,   35.600,   34.500,   33.600,   32.600,   31.800,   30.900,   30.100, 
      29.400,   28.700,   28.000,   27.400,   26.700,   26.100,   25.000,   24.000,   23.100,   22.200, 
      21.400,   20.600,   19.900,   19.300,   18.700,   18.100,   17.500,   17.100,   16.600,   16.100, 
      15.700,   15.300,   15.000,   14.600,   14.200,   13.900,   13.600,   13.300,   13.100,   12.900, 
      12.700,   12.400,   12.200,   12.100,   11.900,   11.700,   11.500,   11.400,   11.200,   11.000, 
      10.900,   10.700,   10.500,   10.400,   10.300,   10.100,   10.000,    9.800,    9.700,    9.500, 
        9.400,    9.300,    9.100,    9.000,    8.900,    8.800, ],
    [ 126.900,  117.600,  109.000,  105.800,  104.400,  103.100,  102.300,  101.500,  101.300,  101.000, 
      99.100,   98.900,   98.300,   97.600,   96.500,   94.700,   93.300,   91.000,   90.000,   88.200, 
      86.400,   84.600,   82.800,   80.600,   78.500,   76.500,   72.000,   68.200,   64.800,   61.700, 
      58.900,   56.300,   54.100,   51.900,   50.000,   48.100,   46.400,   44.800,   43.300,   41.900, 
      40.600,   39.400,   38.200,   37.100,   36.100,   35.100,   34.100,   33.200,   32.400,   31.500, 
      30.800,   30.000,   29.400,   28.700,   28.100,   27.400,   26.300,   25.200,   24.200,   23.300, 
      22.400,   21.700,   21.000,   20.300,   19.600,   19.000,   18.500,   18.000,   17.500,   17.000, 
      16.500,   16.100,   15.800,   15.400,   15.000,   14.600,   14.300,   14.100,   13.800,   13.600, 
      13.300,   13.100,   12.900,   12.700,   12.500,   12.300,   12.100,   12.000,   11.800,   11.600, 
      11.400,   11.300,   11.100,   11.000,   10.800,   10.700,   10.500,   10.400,   10.200,   10.100, 
        9.900,    9.800,    9.600,    9.500,    9.400,    9.300, ],
    [ 127.300,  118.100,  109.900,  106.800,  105.500,  104.200,  103.800,  103.500,  103.000,  103.000, 
      99.400,  100.800,  100.500,   99.700,   98.500,   96.500,   95.200,   91.800,   91.800,   90.000, 
      88.200,   86.400,   84.500,   82.300,   80.100,   78.100,   73.700,   69.900,   66.500,   63.400, 
      60.500,   58.000,   55.700,   53.600,   51.600,   49.800,   48.100,   46.400,   44.900,   43.500, 
      42.200,   41.000,   39.800,   38.700,   37.600,   36.600,   35.600,   34.700,   33.800,   32.900, 
      32.100,   31.400,   30.700,   30.000,   29.300,   28.700,   27.500,   26.400,   25.400,   24.400, 
      23.500,   22.700,   22.000,   21.300,   20.600,   20.000,   19.400,   18.800,   18.300,   17.800, 
      17.400,   16.900,   16.500,   16.100,   15.700,   15.400,   15.100,   14.800,   14.500,   14.300, 
      14.000,   13.800,   13.600,   13.400,   13.200,   13.000,   12.800,   12.600,   12.400,   12.200, 
      12.000,   11.900,   11.700,   11.500,   11.400,   11.200,   11.000,   10.900,   10.700,   10.600, 
      10.400,   10.300,   10.100,   10.000,    9.900,    9.700, ],
    [ 127.700,  118.700,  110.700,  107.800,  106.600,  106.000,  105.200,  105.200,  105.200,  103.500, 
      103.100,  102.800,  102.500,  101.500,  100.400,   98.300,   97.000,   92.800,   93.500,   91.700, 
      90.000,   88.000,   86.100,   83.900,   81.600,   79.600,   75.200,   71.400,   68.100,   65.000, 
      62.100,   59.600,   57.300,   55.200,   53.200,   51.400,   49.600,   48.000,   46.500,   45.100, 
      43.800,   42.500,   41.300,   40.100,   39.100,   38.000,   37.000,   36.100,   35.200,   34.300, 
      33.500,   32.700,   32.000,   31.300,   30.600,   29.900,   28.700,   27.600,   26.500,   25.500, 
      24.600,   23.800,   23.000,   22.300,   21.600,   20.900,   20.300,   19.700,   19.200,   18.700, 
      18.200,   17.800,   17.300,   16.900,   16.500,   16.100,   15.800,   15.500,   15.200,   15.000, 
      14.700,   14.500,   14.200,   14.000,   13.800,   13.600,   13.400,   13.200,   13.000,   12.800, 
      12.600,   12.400,   12.300,   12.100,   11.900,   11.800,   11.600,   11.400,   11.300,   11.100, 
      10.900,   10.800,   10.600,   10.500,   10.400,   10.200, ],
    [ 128.100,  119.300,  111.600,  108.800,  108.100,  107.400,  107.200,  106.700,  106.500,  105.300, 
      104.900,  105.000,  104.300,  103.300,  101.600,   99.900,   98.700,   96.200,   95.200,   93.300, 
      91.300,   89.400,   87.600,   85.300,   83.000,   81.000,   76.600,   72.800,   69.500,   66.400, 
      63.600,   61.100,   58.800,   56.700,   54.800,   52.900,   51.100,   49.500,   48.000,   46.600, 
      45.200,   44.000,   42.700,   41.600,   40.500,   39.400,   38.400,   37.400,   36.500,   35.600, 
      34.800,   34.000,   33.300,   32.500,   31.800,   31.200,   29.900,   28.700,   27.600,   26.600, 
      25.700,   24.800,   24.000,   23.200,   22.500,   21.800,   21.200,   20.600,   20.100,   19.500, 
      19.000,   18.600,   18.100,   17.700,   17.200,   16.900,   16.500,   16.200,   15.900,   15.600, 
      15.400,   15.100,   14.900,   14.700,   14.500,   14.200,   14.000,   13.800,   13.600,   13.400, 
      13.200,   13.000,   12.800,   12.700,   12.500,   12.300,   12.100,   12.000,   11.800,   11.600, 
      11.500,   11.300,   11.200,   11.000,   10.900,   10.700, ],
    [ 128.600,  119.900,  112.500,  110.100,  109.500,  109.200,  108.800,  109.000,  106.700,  107.200, 
      107.100,  106.800,  106.000,  104.800,  103.100,  101.500,   99.900,   97.700,   96.700,   94.800, 
      92.800,   90.800,   88.900,   86.600,   84.300,   82.200,   77.900,   74.200,   70.900,   67.800, 
      65.000,   62.500,   60.300,   58.200,   56.200,   54.300,   52.600,   51.000,   49.400,   48.000, 
      46.700,   45.400,   44.100,   43.000,   41.800,   40.800,   39.700,   38.800,   37.800,   36.900, 
      36.100,   35.300,   34.500,   33.800,   33.000,   32.400,   31.100,   29.800,   28.700,   27.700, 
      26.700,   25.800,   25.000,   24.200,   23.400,   22.800,   22.100,   21.500,   20.900,   20.400, 
      19.800,   19.400,   18.900,   18.400,   18.000,   17.600,   17.200,   16.900,   16.600,   16.300, 
      16.100,   15.800,   15.600,   15.300,   15.100,   14.900,   14.600,   14.400,   14.200,   14.000, 
      13.800,   13.600,   13.400,   13.200,   13.100,   12.900,   12.700,   12.500,   12.300,   12.100, 
      12.000,   11.800,   11.700,   11.500,   11.300,   11.200, ],
    [ 129.000,  120.500,  113.500,  111.500,  111.200,  110.800,  110.900,  109.100,  109.400,  109.300, 
      109.000,  108.500,  107.700,  106.000,  104.500,  103.100,  100.100,   99.200,   97.800,   96.300, 
      94.200,   92.100,   90.000,   87.800,   85.400,   83.400,   79.100,   75.400,   72.100,   69.100, 
      66.300,   63.800,   61.600,   59.500,   57.600,   55.700,   54.000,   52.300,   50.800,   49.400, 
      48.000,   46.700,   45.500,   44.300,   43.200,   42.100,   41.000,   40.000,   39.100,   38.200, 
      37.300,   36.500,   35.700,   35.000,   34.200,   33.500,   32.200,   31.000,   29.800,   28.700, 
      27.700,   26.800,   26.000,   25.200,   24.400,   23.700,   23.000,   22.400,   21.800,   21.200, 
      20.700,   20.200,   19.700,   19.200,   18.700,   18.300,   18.000,   17.600,   17.300,   17.000, 
      16.700,   16.500,   16.200,   16.000,   15.700,   15.500,   15.300,   15.000,   14.800,   14.600, 
      14.400,   14.200,   14.000,   13.800,   13.600,   13.400,   13.200,   13.000,   12.900,   12.700, 
      12.500,   12.300,   12.100,   12.000,   11.800,   11.700, ],
    [ 129.500,  121.400,  114.800,  113.200,  112.700,  112.900,  111.100,  111.400,  111.400,  111.300, 
      110.700,  110.100,  108.700,  107.300,  105.900,  104.000,  102.100,  100.700,   99.100,   97.100, 
      95.500,   93.400,   91.100,   88.800,   86.500,   84.400,   80.200,   76.500,   73.300,   70.300, 
      67.500,   65.100,   62.900,   60.800,   58.900,   57.000,   55.300,   53.700,   52.100,   50.700, 
      49.300,   48.000,   46.800,   45.600,   44.500,   43.400,   42.300,   41.300,   40.300,   39.400, 
      38.500,   37.700,   36.900,   36.100,   35.400,   34.700,   33.300,   32.000,   30.900,   29.800, 
      28.700,   27.800,   26.900,   26.100,   25.300,   24.600,   23.900,   23.200,   22.600,   22.000, 
      21.500,   20.900,   20.500,   20.000,   19.500,   19.100,   18.700,   18.300,   18.000,   17.700, 
      17.400,   17.100,   16.900,   16.600,   16.400,   16.100,   15.900,   15.700,   15.400,   15.200, 
      15.000,   14.800,   14.600,   14.400,   14.200,   14.000,   13.800,   13.600,   13.400,   13.200, 
      13.000,   12.800,   12.600,   12.500,   12.300,   12.100, ],
    [ 130.300,  122.300,  116.400,  114.600,  114.800,  113.100,  113.300,  113.400,  113.300,  112.900, 
      112.200,  111.100,  109.900,  108.600,  107.300,  104.100,  103.500,  101.700,  100.300,   98.200, 
      96.400,   94.600,   92.000,   90.000,   87.400,   85.400,   81.200,   77.600,   74.400,   71.500, 
      68.700,   66.300,   64.100,   62.000,   60.100,   58.300,   56.500,   54.900,   53.400,   52.000, 
      50.600,   49.300,   48.000,   46.800,   45.700,   44.600,   43.500,   42.500,   41.500,   40.600, 
      39.700,   38.900,   38.100,   37.300,   36.500,   35.800,   34.400,   33.100,   31.900,   30.800, 
      29.700,   28.800,   27.900,   27.000,   26.200,   25.500,   24.700,   24.100,   23.400,   22.800, 
      22.300,   21.700,   21.200,   20.700,   20.200,   19.800,   19.400,   19.000,   18.700,   18.400, 
      18.100,   17.800,   17.500,   17.300,   17.000,   16.800,   16.500,   16.300,   16.000,   15.800, 
      15.600,   15.300,   15.100,   14.900,   14.700,   14.500,   14.300,   14.100,   13.900,   13.700, 
      13.500,   13.300,   13.100,   13.000,   12.800,   12.600, ],
    [ 131.100,  123.600,  117.700,  116.400,  114.800,  115.200,  115.300,  115.300,  114.900,  114.400, 
      113.200,  112.200,  111.100,  109.900,  107.400,  106.100,  104.800,  102.800,  101.300,   99.300, 
      97.300,   95.700,   92.900,   90.400,   88.300,   86.300,   82.200,   78.600,   75.400,   72.500, 
      69.800,   67.400,   65.200,   63.200,   61.300,   59.500,   57.700,   56.100,   54.600,   53.200, 
      51.800,   50.500,   49.300,   48.100,   46.900,   45.800,   44.700,   43.700,   42.700,   41.800, 
      40.900,   40.000,   39.200,   38.400,   37.700,   36.900,   35.500,   34.200,   33.000,   31.800, 
      30.700,   29.700,   28.800,   28.000,   27.100,   26.300,   25.600,   24.900,   24.300,   23.600, 
      23.100,   22.500,   22.000,   21.500,   20.900,   20.500,   20.100,   19.700,   19.400,   19.000, 
      18.700,   18.400,   18.200,   17.900,   17.600,   17.400,   17.100,   16.900,   16.600,   16.400, 
      16.100,   15.900,   15.700,   15.500,   15.300,   15.100,   14.800,   14.600,   14.400,   14.200, 
      14.000,   13.800,   13.600,   13.500,   13.300,   13.100, ],
    [ 131.900,  125.100,  119.100,  116.500,  117.100,  117.300,  117.100,  116.800,  116.300,  115.300, 
      114.300,  113.400,  112.300,  110.200,  108.800,  107.300,  105.600,  103.900,  102.100,  100.400, 
      98.200,   96.300,   93.800,   91.200,   89.100,   87.100,   83.000,   79.500,   76.400,   73.500, 
      70.800,   68.400,   66.300,   64.300,   62.400,   60.600,   58.900,   57.300,   55.800,   54.300, 
      53.000,   51.700,   50.400,   49.200,   48.100,   47.000,   45.900,   44.800,   43.800,   42.900, 
      42.000,   41.100,   40.300,   39.500,   38.700,   38.000,   36.600,   35.200,   34.000,   32.800, 
      31.700,   30.700,   29.800,   28.900,   28.000,   27.200,   26.500,   25.800,   25.100,   24.400, 
      23.800,   23.300,   22.700,   22.200,   21.700,   21.200,   20.800,   20.400,   20.100,   19.700, 
      19.400,   19.100,   18.800,   18.500,   18.300,   18.000,   17.700,   17.500,   17.200,   17.000, 
      16.700,   16.500,   16.300,   16.000,   15.800,   15.600,   15.400,   15.200,   14.900,   14.700, 
      14.500,   14.300,   14.100,   13.900,   13.800,   13.600, ],
    [ 132.900,  125.400,  120.300,  119.000,  119.200,  118.900,  118.600,  118.000,  117.100,  116.300, 
      115.400,  114.500,  112.600,  111.300,  109.900,  108.200,  106.500,  104.600,  103.000,  101.000, 
      99.200,   97.100,   94.600,   91.800,   90.000,   87.800,   83.900,   80.400,   77.300,   74.500, 
      71.800,   69.400,   67.300,   65.400,   63.500,   61.700,   60.000,   58.400,   56.900,   55.500, 
      54.100,   52.800,   51.600,   50.400,   49.200,   48.100,   47.000,   46.000,   45.000,   44.000, 
      43.100,   42.200,   41.400,   40.600,   39.800,   39.000,   37.600,   36.200,   35.000,   33.800, 
      32.700,   31.600,   30.700,   29.800,   28.900,   28.100,   27.300,   26.600,   25.900,   25.200, 
      24.600,   24.000,   23.500,   22.900,   22.400,   21.900,   21.500,   21.100,   20.700,   20.400, 
      20.100,   19.700,   19.400,   19.200,   18.900,   18.600,   18.300,   18.100,   17.800,   17.600, 
      17.300,   17.100,   16.800,   16.600,   16.400,   16.100,   15.900,   15.700,   15.500,   15.300, 
      15.000,   14.800,   14.600,   14.400,   14.200,   14.000, ],
    [ 132.900,  126.800,  122.200,  121.100,  120.700,  120.300,  119.600,  118.900,  118.100,  117.300, 
      116.400,  114.700,  113.600,  112.300,  110.700,  109.000,  107.400,  105.400,  103.900,  101.700, 
      100.000,   97.800,   95.100,   92.400,   90.400,   88.500,   84.600,   81.200,   78.200,   75.400, 
      72.700,   70.400,   68.300,   66.400,   64.500,   62.700,   61.000,   59.500,   58.000,   56.600, 
      55.200,   53.900,   52.700,   51.500,   50.300,   49.200,   48.100,   47.100,   46.000,   45.100, 
      44.200,   43.300,   42.500,   41.600,   40.800,   40.100,   38.600,   37.200,   36.000,   34.700, 
      33.600,   32.600,   31.600,   30.600,   29.700,   28.900,   28.100,   27.400,   26.700,   26.000, 
      25.400,   24.800,   24.200,   23.700,   23.100,   22.600,   22.200,   21.800,   21.400,   21.000, 
      20.700,   20.400,   20.100,   19.800,   19.500,   19.200,   18.900,   18.700,   18.400,   18.200, 
      17.900,   17.600,   17.400,   17.100,   16.900,   16.700,   16.400,   16.200,   16.000,   15.800, 
      15.500,   15.300,   15.100,   14.900,   14.700,   14.500, ],
    [ 134.800,  128.500,  123.800,  122.500,  122.000,  121.200,  120.500,  119.800,  119.200,  118.200, 
      116.500,  115.700,  114.600,  113.100,  111.500,  109.900,  108.000,  106.200,  104.400,  102.600, 
      100.500,   98.300,   95.600,   93.000,   91.000,   89.100,   85.300,   82.000,   79.000,   76.200, 
      73.600,   71.300,   69.200,   67.300,   65.500,   63.700,   62.000,   60.500,   59.000,   57.600, 
      56.300,   55.000,   53.700,   52.500,   51.400,   50.200,   49.200,   48.100,   47.100,   46.100, 
      45.200,   44.300,   43.500,   42.700,   41.900,   41.100,   39.600,   38.200,   36.900,   35.700, 
      34.500,   33.500,   32.500,   31.500,   30.600,   29.800,   29.000,   28.200,   27.500,   26.800, 
      26.200,   25.500,   25.000,   24.400,   23.800,   23.300,   22.900,   22.500,   22.100,   21.700, 
      21.400,   21.000,   20.700,   20.400,   20.100,   19.800,   19.500,   19.300,   19.000,   18.700, 
      18.500,   18.200,   17.900,   17.700,   17.500,   17.200,   17.000,   16.700,   16.500,   16.300, 
      16.000,   15.800,   15.600,   15.400,   15.200,   15.000, ],
    [ 135.900,  129.700,  125.100,  123.500,  122.700,  122.100,  121.500,  120.800,  119.700,  118.200, 
      117.600,  116.700,  115.300,  113.800,  112.300,  110.400,  108.700,  106.800,  105.100,  103.300, 
      101.200,   98.800,   96.200,   93.600,   91.500,   90.000,   86.000,   82.700,   79.700,   77.000, 
      74.400,   72.100,   70.100,   68.200,   66.400,   64.700,   63.000,   61.500,   60.000,   58.600, 
      57.300,   56.000,   54.800,   53.600,   52.400,   51.300,   50.200,   49.100,   48.100,   47.100, 
      46.200,   45.300,   44.500,   43.700,   42.900,   42.100,   40.600,   39.200,   37.900,   36.600, 
      35.400,   34.400,   33.400,   32.400,   31.500,   30.600,   29.800,   29.000,   28.300,   27.600, 
      26.900,   26.300,   25.700,   25.100,   24.500,   24.000,   23.600,   23.100,   22.700,   22.400, 
      22.000,   21.700,   21.300,   21.000,   20.700,   20.400,   20.100,   19.900,   19.600,   19.300, 
      19.000,   18.800,   18.500,   18.300,   18.000,   17.800,   17.500,   17.300,   17.000,   16.800, 
      16.600,   16.300,   16.100,   15.900,   15.700,   15.500, ],
    [ 136.700,  130.500,  125.800,  124.300,  123.700,  123.000,  122.300,  121.000,  119.800,  119.400, 
      118.500,  117.300,  115.900,  114.600,  112.700,  111.000,  109.300,  107.300,  105.800,  103.700, 
      101.600,   99.300,   96.800,   94.200,   92.000,   90.000,   86.600,   83.400,   80.500,   77.700, 
      75.100,   72.900,   71.000,   69.100,   67.300,   65.600,   63.900,   62.400,   60.900,   59.600, 
      58.300,   57.000,   55.700,   54.500,   53.400,   52.300,   51.200,   50.100,   49.100,   48.100, 
      47.200,   46.300,   45.500,   44.600,   43.800,   43.000,   41.500,   40.100,   38.800,   37.500, 
      36.300,   35.200,   34.200,   33.200,   32.300,   31.400,   30.600,   29.800,   29.100,   28.300, 
      27.700,   27.000,   26.400,   25.800,   25.200,   24.700,   24.200,   23.800,   23.400,   23.000, 
      22.700,   22.300,   22.000,   21.700,   21.400,   21.100,   20.700,   20.500,   20.200,   19.900, 
      19.600,   19.300,   19.100,   18.800,   18.600,   18.300,   18.000,   17.800,   17.500,   17.300, 
      17.100,   16.800,   16.600,   16.400,   16.200,   15.900, ],
    [ 137.200,  131.200,  126.600,  125.200,  124.600,  123.500,  122.300,  121.900,  121.100,  120.200, 
      119.000,  117.900,  116.600,  115.000,  113.300,  111.600,  109.800,  108.000,  106.200,  104.300, 
      102.000,   99.600,   97.000,   94.500,   92.500,   90.700,   87.200,   84.000,   81.100,   78.400, 
      75.900,   73.700,   71.800,   69.900,   68.100,   66.400,   64.800,   63.300,   61.900,   60.500, 
      59.200,   57.900,   56.700,   55.500,   54.400,   53.200,   52.200,   51.100,   50.100,   49.100, 
      48.200,   47.300,   46.400,   45.600,   44.800,   44.000,   42.500,   41.000,   39.700,   38.400, 
      37.200,   36.100,   35.100,   34.100,   33.100,   32.200,   31.400,   30.600,   29.800,   29.100, 
      28.400,   27.800,   27.200,   26.500,   25.900,   25.400,   24.900,   24.500,   24.100,   23.700, 
      23.300,   22.900,   22.600,   22.300,   22.000,   21.700,   21.300,   21.000,   20.800,   20.500, 
      20.200,   19.900,   19.600,   19.400,   19.100,   18.800,   18.600,   18.300,   18.100,   17.800, 
      17.600,   17.300,   17.100,   16.900,   16.600,   16.400, ],
    [ 137.800,  131.900,  127.500,  125.700,  124.600,  123.500,  123.400,  122.700,  121.700,  120.700, 
      119.600,  118.400,  117.000,  115.500,  113.900,  112.000,  110.300,  108.500,  106.700,  104.500, 
      102.400,  100.000,   97.300,   94.900,   93.000,   91.100,   87.700,   84.600,   81.800,   79.100, 
      76.600,   74.400,   72.500,   70.700,   68.900,   67.300,   65.700,   64.200,   62.700,   61.400, 
      60.100,   58.800,   57.600,   56.400,   55.300,   54.200,   53.100,   52.000,   51.000,   50.000, 
      49.100,   48.200,   47.400,   46.500,   45.700,   44.900,   43.400,   41.900,   40.600,   39.300, 
      38.100,   37.000,   35.900,   34.900,   33.900,   33.000,   32.200,   31.400,   30.600,   29.900, 
      29.100,   28.500,   27.900,   27.200,   26.600,   26.100,   25.600,   25.100,   24.700,   24.300, 
      23.900,   23.600,   23.200,   22.900,   22.600,   22.300,   21.900,   21.600,   21.300,   21.000, 
      20.700,   20.400,   20.200,   19.900,   19.600,   19.400,   19.100,   18.800,   18.600,   18.300, 
      18.100,   17.800,   17.600,   17.300,   17.100,   16.900, ],
    [ 138.400,  132.300,  127.600,  125.800,  125.500,  124.800,  124.000,  123.100,  122.300,  121.300, 
      120.000,  118.800,  117.500,  116.000,  114.300,  112.500,  110.800,  108.900,  107.000,  104.900, 
      102.600,  100.400,   97.700,   95.400,   93.500,   91.500,   88.200,   85.200,   82.400,   79.700, 
      77.200,   75.100,   73.200,   71.400,   69.700,   68.100,   66.500,   65.000,   63.600,   62.200, 
      61.000,   59.700,   58.500,   57.300,   56.200,   55.100,   54.000,   52.900,   51.900,   50.900, 
      50.000,   49.100,   48.300,   47.400,   46.600,   45.800,   44.300,   42.800,   41.500,   40.200, 
      38.900,   37.800,   36.800,   35.700,   34.700,   33.800,   32.900,   32.100,   31.300,   30.600, 
      29.900,   29.200,   28.600,   27.900,   27.300,   26.700,   26.200,   25.800,   25.400,   24.900, 
      24.600,   24.200,   23.800,   23.500,   23.200,   22.900,   22.500,   22.200,   21.900,   21.600, 
      21.300,   21.000,   20.700,   20.400,   20.200,   19.900,   19.600,   19.300,   19.100,   18.800, 
      18.600,   18.300,   18.100,   17.800,   17.600,   17.300, ],
    [ 138.400,  132.400,  128.400,  126.900,  126.200,  125.300,  124.500,  123.700,  122.700,  121.600, 
      120.500,  119.400,  117.900,  116.400,  114.700,  113.000,  111.200,  109.200,  107.300,  105.200, 
      103.000,  100.500,   98.100,   95.800,   93.900,   91.800,   88.700,   85.700,   82.900,   80.300, 
      77.800,   75.800,   73.900,   72.200,   70.500,   68.800,   67.200,   65.800,   64.400,   63.100, 
      61.800,   60.600,   59.400,   58.200,   57.100,   56.000,   54.900,   53.800,   52.800,   51.800, 
      50.900,   50.000,   49.200,   48.300,   47.500,   46.700,   45.200,   43.700,   42.300,   41.000, 
      39.800,   38.600,   37.600,   36.500,   35.500,   34.600,   33.700,   32.900,   32.100,   31.300, 
      30.600,   29.900,   29.300,   28.600,   28.000,   27.400,   26.900,   26.400,   26.000,   25.600, 
      25.200,   24.800,   24.500,   24.100,   23.800,   23.400,   23.100,   22.800,   22.500,   22.200, 
      21.900,   21.600,   21.300,   21.000,   20.700,   20.400,   20.100,   19.900,   19.600,   19.300, 
      19.000,   18.800,   18.500,   18.300,   18.100,   17.800, ],
    [ 139.000,  133.300,  129.000,  127.400,  126.600,  125.800,  125.000,  124.100,  123.100,  122.100, 
      120.900,  119.700,  118.300,  116.800,  115.100,  113.300,  111.500,  109.500,  107.600,  105.400, 
      103.200,  100.800,   98.300,   96.000,   94.100,   92.200,   89.200,   86.200,   83.500,   80.900, 
      78.400,   76.400,   74.600,   72.800,   71.200,   69.600,   68.000,   66.600,   65.200,   63.900, 
      62.600,   61.400,   60.200,   59.000,   57.900,   56.800,   55.800,   54.700,   53.700,   52.700, 
      51.800,   50.900,   50.000,   49.200,   48.400,   47.600,   46.000,   44.500,   43.200,   41.900, 
      40.600,   39.400,   38.400,   37.300,   36.300,   35.400,   34.500,   33.600,   32.800,   32.100, 
      31.300,   30.600,   30.000,   29.300,   28.600,   28.100,   27.500,   27.100,   26.600,   26.200, 
      25.800,   25.400,   25.100,   24.700,   24.400,   24.000,   23.700,   23.400,   23.100,   22.700, 
      22.400,   22.100,   21.800,   21.500,   21.200,   21.000,   20.700,   20.400,   20.100,   19.800, 
      19.500,   19.300,   19.000,   18.800,   18.500,   18.300, ],
    [ 139.400,  133.700,  129.400,  127.900,  127.100,  126.200,  125.300,  124.500,  123.600,  122.400, 
      121.200,  120.000,  118.700,  117.100,  115.400,  113.600,  111.700,  109.800,  107.800,  105.700, 
      103.400,  101.100,   98.600,   96.300,   94.400,   92.600,   89.600,   86.700,   84.000,   81.500, 
      79.000,   77.000,   75.200,   73.500,   71.900,   70.300,   68.700,   67.300,   65.900,   64.600, 
      63.400,   62.200,   61.000,   59.900,   58.700,   57.700,   56.600,   55.500,   54.500,   53.500, 
      52.600,   51.700,   50.900,   50.000,   49.200,   48.400,   46.900,   45.400,   44.000,   42.700, 
      41.400,   40.200,   39.200,   38.100,   37.100,   36.100,   35.200,   34.400,   33.600,   32.800, 
      32.000,   31.300,   30.700,   30.000,   29.300,   28.700,   28.200,   27.700,   27.300,   26.800, 
      26.400,   26.000,   25.700,   25.300,   25.000,   24.600,   24.300,   24.000,   23.600,   23.300, 
      23.000,   22.700,   22.400,   22.100,   21.800,   21.500,   21.200,   20.900,   20.600,   20.300, 
      20.000,   19.800,   19.500,   19.300,   19.000,   18.700, ],
    [ 139.700,  134.100,  129.900,  128.300,  127.400,  126.600,  125.700,  124.800,  123.800,  122.700, 
      121.600,  120.500,  118.900,  117.400,  115.700,  113.800,  112.000,  109.900,  108.000,  105.900, 
      103.600,  101.300,   98.900,   96.700,   94.800,   92.900,   90.000,   87.100,   84.500,   82.000, 
      79.500,   77.600,   75.800,   74.100,   72.500,   70.900,   69.400,   68.000,   66.700,   65.400, 
      64.200,   63.000,   61.800,   60.700,   59.500,   58.500,   57.400,   56.300,   55.300,   54.400, 
      53.400,   52.500,   51.700,   50.900,   50.100,   49.200,   47.700,   46.200,   44.800,   43.500, 
      42.200,   41.000,   40.000,   38.900,   37.800,   36.900,   36.000,   35.100,   34.300,   33.500, 
      32.700,   32.000,   31.300,   30.600,   30.000,   29.400,   28.800,   28.300,   27.900,   27.500, 
      27.000,   26.700,   26.300,   25.900,   25.600,   25.200,   24.900,   24.500,   24.200,   23.900, 
      23.500,   23.200,   22.900,   22.600,   22.300,   22.000,   21.700,   21.400,   21.100,   20.800, 
      20.500,   20.300,   20.000,   19.700,   19.500,   19.200, ],
    [ 140.200,  134.700,  130.500,  129.000,  128.100,  127.200,  126.300,  125.400,  124.500,  123.300, 
      122.100,  120.900,  119.400,  117.800,  116.100,  114.200,  112.300,  110.300,  108.400,  106.200, 
      104.000,  101.800,   99.300,   97.100,   95.300,   93.400,   90.600,   88.000,   85.400,   82.900, 
      80.500,   78.600,   77.000,   75.300,   73.700,   72.200,   70.700,   69.400,   68.000,   66.800, 
      65.600,   64.400,   63.300,   62.200,   61.100,   60.000,   58.900,   57.900,   56.900,   55.900, 
      55.000,   54.100,   53.300,   52.500,   51.600,   50.800,   49.300,   47.800,   46.400,   45.000, 
      43.700,   42.600,   41.500,   40.400,   39.300,   38.300,   37.400,   36.500,   35.700,   34.900, 
      34.100,   33.400,   32.700,   32.000,   31.300,   30.600,   30.100,   29.600,   29.100,   28.700, 
      28.300,   27.900,   27.500,   27.100,   26.700,   26.400,   26.000,   25.700,   25.300,   25.000, 
      24.600,   24.300,   24.000,   23.700,   23.400,   23.000,   22.700,   22.400,   22.100,   21.800, 
      21.500,   21.200,   20.900,   20.700,   20.400,   20.100, ],
    [ 140.600,  135.100,  131.000,  129.500,  128.700,  127.700,  126.800,  125.900,  124.800,  123.700, 
      122.500,  121.200,  119.700,  118.100,  116.400,  114.500,  112.600,  110.600,  108.700,  106.500, 
      104.400,  102.100,   99.800,   97.600,   95.700,   93.800,   91.100,   88.800,   86.200,   83.800, 
      81.400,   79.600,   78.000,   76.400,   74.900,   73.400,   71.900,   70.600,   69.300,   68.100, 
      67.000,   65.800,   64.700,   63.600,   62.500,   61.400,   60.400,   59.400,   58.400,   57.400, 
      56.500,   55.600,   54.800,   54.000,   53.200,   52.400,   50.800,   49.300,   47.900,   46.500, 
      45.200,   44.100,   43.000,   41.800,   40.800,   39.800,   38.800,   37.900,   37.100,   36.300, 
      35.400,   34.700,   34.000,   33.300,   32.500,   31.900,   31.400,   30.800,   30.400,   29.900, 
      29.500,   29.100,   28.700,   28.300,   27.900,   27.500,   27.100,   26.800,   26.500,   26.100, 
      25.700,   25.400,   25.000,   24.700,   24.400,   24.100,   23.700,   23.400,   23.100,   22.800, 
      22.500,   22.200,   21.900,   21.600,   21.300,   21.000, ],
    [ 140.900,  135.500,  131.400,  129.900,  129.000,  128.000,  127.100,  126.200,  125.100,  124.000, 
      122.700,  121.400,  120.000,  118.300,  116.600,  114.700,  112.900,  110.800,  109.000,  106.900, 
      104.700,  102.500,  100.100,   98.000,   96.200,   95.500,   91.700,   89.500,   87.000,   84.600, 
      82.300,   80.500,   78.900,   77.400,   75.900,   74.500,   73.100,   71.800,   70.500,   69.400, 
      68.200,   67.100,   66.000,   64.900,   63.900,   62.800,   61.800,   60.800,   59.800,   58.900, 
      57.900,   57.100,   56.300,   55.400,   54.600,   53.800,   52.300,   50.800,   49.400,   48.000, 
      46.700,   45.500,   44.400,   43.300,   42.200,   41.200,   40.200,   39.300,   38.400,   37.600, 
      36.800,   36.000,   35.300,   34.500,   33.800,   33.200,   32.600,   32.100,   31.600,   31.100, 
      30.700,   30.200,   29.800,   29.400,   29.100,   28.700,   28.300,   27.900,   27.600,   27.200, 
      26.800,   26.400,   26.100,   25.800,   25.400,   25.100,   24.800,   24.400,   24.100,   23.800, 
      23.500,   23.100,   22.800,   22.500,   22.300,   22.000, ],
    [ 141.200,  135.800,  131.700,  130.100,  129.200,  128.300,  127.300,  126.400,  125.300,  124.200, 
      122.900,  121.600,  120.100,  118.500,  116.800,  114.900,  113.100,  111.100,  109.200,  107.100, 
      105.000,  102.700,  100.400,   98.300,   96.400,   95.700,   92.100,   90.000,   87.700,   85.300, 
      83.000,   81.300,   79.800,   78.300,   76.900,   75.500,   74.100,   72.900,   71.700,   70.500, 
      69.400,   68.300,   67.200,   66.200,   65.100,   64.100,   63.100,   62.100,   61.100,   60.200, 
      59.300,   58.400,   57.600,   56.800,   56.000,   55.200,   53.700,   52.200,   50.800,   49.400, 
      48.100,   46.900,   45.800,   44.600,   43.500,   42.500,   41.600,   40.600,   39.800,   38.900, 
      38.100,   37.300,   36.600,   35.800,   35.000,   34.400,   33.800,   33.300,   32.800,   32.300, 
      31.800,   31.400,   31.000,   30.600,   30.200,   29.800,   29.400,   29.000,   28.700,   28.300, 
      27.900,   27.500,   27.200,   26.800,   26.500,   26.100,   25.800,   25.400,   25.100,   24.700, 
      24.400,   24.100,   23.800,   23.500,   23.200,   22.900, ],
    [ 141.300,  136.000,  131.900,  130.300,  129.400,  128.500,  127.500,  126.500,  125.500,  124.300, 
      123.100,  121.800,  120.300,  118.700,  117.000,  115.100,  113.300,  111.300,  109.500,  107.300, 
      105.200,  103.000,  100.700,   98.600,   97.700,   96.000,   92.600,   90.600,   88.300,   86.000, 
      83.700,   82.100,   80.600,   79.200,   77.800,   76.400,   75.100,   73.900,   72.700,   71.600, 
      70.500,   69.500,   68.400,   67.400,   66.400,   65.300,   64.400,   63.400,   62.400,   61.500, 
      60.600,   59.800,   59.000,   58.200,   57.400,   56.500,   55.000,   53.500,   52.100,   50.800, 
      49.400,   48.200,   47.100,   46.000,   44.900,   43.800,   42.900,   41.900,   41.000,   40.200, 
      39.300,   38.500,   37.800,   37.000,   36.200,   35.600,   35.000,   34.400,   33.900,   33.400, 
      33.000,   32.500,   32.100,   31.700,   31.300,   30.900,   30.500,   30.100,   29.700,   29.300, 
      28.900,   28.600,   28.200,   27.800,   27.500,   27.100,   26.800,   26.400,   26.100,   25.700, 
      25.400,   25.000,   24.700,   24.400,   24.100,   23.800, ],
    [ 141.400,  136.100,  132.000,  130.500,  129.500,  128.600,  127.700,  126.700,  125.600,  124.500, 
      123.200,  122.000,  120.500,  118.900,  117.100,  115.300,  113.500,  111.500,  109.600,  107.600, 
      105.400,  103.200,  100.900,   99.600,   97.800,   96.400,   93.100,   91.000,   88.900,   86.600, 
      84.300,   82.800,   81.400,   80.000,   78.700,   77.300,   76.000,   74.800,   73.700,   72.600, 
      71.600,   70.600,   69.500,   68.500,   67.500,   66.500,   65.500,   64.600,   63.600,   62.700, 
      61.800,   61.000,   60.200,   59.400,   58.600,   57.800,   56.300,   54.800,   53.400,   52.100, 
      50.700,   49.500,   48.400,   47.300,   46.200,   45.100,   44.100,   43.200,   42.300,   41.400, 
      40.600,   39.800,   39.000,   38.200,   37.400,   36.700,   36.100,   35.600,   35.100,   34.600, 
      34.100,   33.700,   33.200,   32.800,   32.400,   32.000,   31.600,   31.200,   30.800,   30.400, 
      30.000,   29.600,   29.200,   28.900,   28.500,   28.100,   27.700,   27.400,   27.000,   26.700, 
      26.300,   26.000,   25.600,   25.300,   25.000,   24.700, ],
    [ 141.500,  136.200,  132.200,  130.600,  129.700,  128.700,  127.800,  126.800,  125.800,  124.700, 
      123.400,  122.100,  120.600,  119.000,  117.300,  115.400,  113.600,  111.700,  109.800,  107.700, 
      105.600,  103.400,  101.800,   99.700,   98.100,   96.800,   93.600,   91.400,   89.500,   87.200, 
      84.900,   83.400,   82.100,   80.800,   79.500,   78.100,   76.900,   75.700,   74.600,   73.600, 
      72.600,   71.600,   70.600,   69.600,   68.600,   67.600,   66.700,   65.700,   64.700,   63.900, 
      63.000,   62.200,   61.400,   60.600,   59.800,   59.100,   57.500,   56.100,   54.700,   53.300, 
      52.000,   50.800,   49.700,   48.500,   47.400,   46.400,   45.400,   44.400,   43.500,   42.600, 
      41.800,   41.000,   40.200,   39.400,   38.600,   37.900,   37.300,   36.700,   36.200,   35.700, 
      35.200,   34.800,   34.300,   33.900,   33.500,   33.100,   32.700,   32.300,   31.900,   31.500, 
      31.000,   30.600,   30.200,   29.900,   29.500,   29.100,   28.700,   28.300,   28.000,   27.600, 
      27.300,   26.900,   26.600,   26.200,   25.900,   25.600, ],
    [ 141.600,  136.300,  132.300,  130.700,  129.800,  128.900,  127.900,  126.900,  125.900,  124.800, 
      123.500,  122.200,  120.800,  119.200,  117.400,  115.600,  113.800,  111.800,  109.900,  107.900, 
      105.700,  104.100,  101.900,   99.900,   98.400,   97.200,   94.000,   91.800,   90.000,   87.700, 
      85.500,   84.000,   82.800,   81.500,   80.200,   78.900,   77.700,   76.600,   75.500,   74.500, 
      73.500,   72.500,   71.600,   70.600,   69.600,   68.700,   67.700,   66.800,   65.800,   65.000, 
      64.100,   63.300,   62.600,   61.800,   61.000,   60.200,   58.700,   57.300,   55.900,   54.500, 
      53.200,   52.000,   50.900,   49.700,   48.600,   47.600,   46.600,   45.600,   44.700,   43.800, 
      42.900,   42.100,   41.400,   40.500,   39.700,   39.000,   38.400,   37.800,   37.300,   36.800, 
      36.300,   35.900,   35.400,   35.000,   34.600,   34.200,   33.700,   33.300,   32.900,   32.500, 
      32.100,   31.600,   31.300,   30.900,   30.500,   30.100,   29.700,   29.300,   28.900,   28.600, 
      28.200,   27.800,   27.500,   27.100,   26.800,   26.400, ],
    [ 141.700,  136.400,  132.400,  130.800,  129.900,  129.000,  128.000,  127.100,  126.000,  124.900, 
      123.600,  122.300,  120.900,  119.300,  117.500,  115.700,  113.900,  111.900,  110.100,  108.000, 
      106.300,  104.200,  102.000,  100.200,   98.800,   97.400,   94.500,   92.100,   90.300,   88.200, 
      85.900,   84.600,   83.400,   82.100,   80.900,   79.700,   78.400,   77.400,   76.300,   75.300, 
      74.400,   73.500,   72.500,   71.500,   70.600,   69.700,   68.700,   67.800,   66.900,   66.000, 
      65.200,   64.400,   63.600,   62.900,   62.100,   61.300,   59.800,   58.400,   57.000,   55.700, 
      54.400,   53.200,   52.100,   50.900,   49.800,   48.700,   47.700,   46.800,   45.900,   45.000, 
      44.100,   43.300,   42.500,   41.700,   40.800,   40.100,   39.500,   38.900,   38.400,   37.900, 
      37.400,   36.900,   36.500,   36.100,   35.700,   35.200,   34.800,   34.300,   34.000,   33.500, 
      33.100,   32.700,   32.300,   31.900,   31.500,   31.100,   30.700,   30.300,   29.900,   29.500, 
      29.100,   28.700,   28.400,   28.000,   27.700,   27.300, ],
    [ 141.800,  136.500,  132.500,  130.900,  130.000,  129.100,  128.100,  127.100,  126.100,  125.000, 
      123.700,  122.400,  121.000,  119.400,  117.600,  115.800,  114.000,  112.000,  110.500,  108.500, 
      106.400,  104.300,  102.200,  100.500,   99.000,   97.600,   94.900,   92.500,   90.600,   88.600, 
      86.400,   85.100,   84.000,   82.800,   81.600,   80.400,   79.200,   78.100,   77.100,   76.200, 
      75.200,   74.300,   73.400,   72.400,   71.500,   70.600,   69.700,   68.800,   67.900,   67.000, 
      66.200,   65.400,   64.700,   63.900,   63.200,   62.400,   60.900,   59.500,   58.200,   56.800, 
      55.500,   54.300,   53.200,   52.100,   50.900,   49.900,   48.900,   47.900,   47.000,   46.100, 
      45.200,   44.400,   43.600,   42.800,   41.900,   41.200,   40.600,   40.000,   39.500,   39.000, 
      38.500,   38.000,   37.600,   37.100,   36.700,   36.300,   35.800,   35.400,   35.000,   34.500, 
      34.100,   33.700,   33.200,   32.800,   32.500,   32.000,   31.600,   31.200,   30.800,   30.400, 
      30.000,   29.600,   29.300,   28.900,   28.600,   28.200, ],
    [ 141.800,  136.500,  132.500,  131.000,  130.100,  129.200,  128.200,  127.200,  126.200,  125.000, 
      123.800,  122.500,  121.100,  119.500,  117.700,  115.900,  114.400,  112.400,  110.600,  108.600, 
      106.500,  104.500,  102.500,  100.700,   99.200,   97.900,   95.400,   92.900,   90.900,   89.100, 
      86.800,   85.600,   84.500,   83.300,   82.200,   81.000,   79.800,   78.800,   77.900,   76.900, 
      76.100,   75.100,   74.200,   73.300,   72.400,   71.500,   70.600,   69.700,   68.800,   68.000, 
      67.200,   66.400,   65.700,   65.000,   64.200,   63.500,   62.000,   60.600,   59.200,   57.900, 
      56.600,   55.400,   54.300,   53.200,   52.000,   51.000,   50.000,   49.000,   48.100,   47.200, 
      46.300,   45.500,   44.700,   43.900,   43.000,   42.300,   41.700,   41.100,   40.500,   40.000, 
      39.500,   39.100,   38.600,   38.200,   37.700,   37.300,   36.800,   36.400,   36.000,   35.500, 
      35.100,   34.600,   34.200,   33.800,   33.400,   33.000,   32.600,   32.200,   31.800,   31.300, 
      30.900,   30.500,   30.200,   29.800,   29.400,   29.000, ],
    [ 141.900,  136.600,  132.600,  131.100,  130.100,  129.200,  128.300,  127.300,  126.300,  125.100, 
      123.900,  122.600,  121.100,  119.800,  118.000,  116.200,  114.400,  112.500,  110.700,  108.700, 
      106.700,  104.700,  102.700,  100.800,   99.400,   98.100,   95.800,   93.300,   91.200,   89.500, 
      87.200,   86.100,   85.100,   83.900,   82.800,   81.600,   80.500,   79.500,   78.600,   77.700, 
      76.800,   75.900,   75.000,   74.100,   73.300,   72.400,   71.500,   70.600,   69.700,   68.900, 
      68.100,   67.400,   66.700,   65.900,   65.200,   64.500,   63.000,   61.600,   60.300,   58.900, 
      57.600,   56.500,   55.400,   54.200,   53.100,   52.100,   51.100,   50.100,   49.200,   48.300, 
      47.400,   46.500,   45.800,   44.900,   44.100,   43.300,   42.700,   42.100,   41.600,   41.100, 
      40.600,   40.100,   39.600,   39.200,   38.800,   38.300,   37.800,   37.400,   37.000,   36.500, 
      36.100,   35.600,   35.200,   34.800,   34.400,   34.000,   33.500,   33.100,   32.700,   32.300, 
      31.800,   31.400,   31.100,   30.700,   30.300,   29.900, ],
    [ 141.900,  136.700,  132.700,  131.100,  130.200,  129.300,  128.300,  127.400,  126.300,  125.200, 
      124.100,  122.800,  121.400,  119.800,  118.100,  116.300,  114.500,  112.600,  110.800,  108.900, 
      106.900,  104.900,  102.800,  101.000,   99.600,   98.400,   96.200,   93.600,   91.500,   90.000, 
      87.600,   86.600,   85.600,   84.400,   83.300,   82.200,   81.100,   80.100,   79.200,   78.400, 
      77.500,   76.700,   75.800,   74.900,   74.100,   73.200,   72.300,   71.500,   70.600,   69.800, 
      69.000,   68.300,   67.600,   66.900,   66.200,   65.400,   64.000,   62.600,   61.300,   60.000, 
      58.700,   57.500,   56.400,   55.300,   54.200,   53.100,   52.100,   51.200,   50.200,   49.300, 
      48.400,   47.600,   46.800,   45.900,   45.100,   44.300,   43.700,   43.100,   42.600,   42.100, 
      41.600,   41.100,   40.600,   40.200,   39.800,   39.300,   38.800,   38.400,   38.000,   37.500, 
      37.000,   36.600,   36.200,   35.700,   35.300,   34.900,   34.400,   34.000,   33.600,   33.200, 
      32.700,   32.300,   31.900,   31.500,   31.200,   30.800, ],
    [ 142.000,  136.700,  132.700,  131.200,  130.300,  129.300,  128.400,  127.600,  126.500,  125.400, 
      124.100,  122.900,  121.400,  119.900,  118.200,  116.400,  114.600,  112.700,  111.000,  109.000, 
      107.000,  105.000,  103.000,  101.200,   99.900,   98.700,   96.300,   94.000,   91.700,   90.000, 
      88.000,   87.000,   86.000,   84.900,   83.900,   82.800,   81.700,   80.800,   79.900,   79.000, 
      78.200,   77.400,   76.600,   75.700,   74.900,   74.000,   73.100,   72.300,   71.400,   70.600, 
      69.900,   69.100,   68.500,   67.800,   67.100,   66.300,   64.900,   63.500,   62.200,   60.900, 
      59.700,   58.500,   57.400,   56.300,   55.200,   54.100,   53.100,   52.200,   51.300,   50.300, 
      49.400,   48.600,   47.800,   47.000,   46.100,   45.400,   44.700,   44.100,   43.600,   43.100, 
      42.600,   42.100,   41.600,   41.200,   40.800,   40.300,   39.800,   39.400,   39.000,   38.500, 
      38.000,   37.500,   37.100,   36.700,   36.300,   35.800,   35.400,   34.900,   34.500,   34.100, 
      33.600,   33.200,   32.800,   32.400,   32.000,   31.600, ],
    [ 142.000,  136.700,  132.800,  131.400,  130.500,  129.500,  128.600,  127.600,  126.600,  125.400, 
      124.200,  122.900,  121.500,  120.000,  118.300,  116.500,  114.700,  112.900,  111.100,  109.100, 
      107.100,  105.100,  103.100,  101.400,  100.100,   99.100,   96.500,   94.300,   92.000,   90.300, 
      88.300,   87.400,   86.500,   85.400,   84.400,   83.300,   82.200,   81.300,   80.500,   79.700, 
      78.900,   78.100,   77.300,   76.400,   75.600,   74.800,   73.900,   73.100,   72.200,   71.400, 
      70.700,   70.000,   69.300,   68.600,   68.000,   67.200,   65.800,   64.500,   63.200,   61.900, 
      60.600,   59.500,   58.400,   57.300,   56.200,   55.100,   54.100,   53.200,   52.300,   51.300, 
      50.400,   49.600,   48.800,   47.900,   47.100,   46.300,   45.700,   45.100,   44.600,   44.100, 
      43.600,   43.100,   42.600,   42.200,   41.700,   41.300,   40.800,   40.300,   39.900,   39.400, 
      38.900,   38.500,   38.000,   37.600,   37.200,   36.700,   36.300,   35.800,   35.400,   34.900, 
      34.500,   34.100,   33.700,   33.300,   32.900,   32.400, ],
    [ 142.100,  136.900,  132.900,  131.400,  130.500,  129.600,  128.600,  127.700,  126.600,  125.500, 
      124.300,  123.000,  121.600,  120.100,  118.400,  116.600,  114.800,  112.900,  111.100,  109.200, 
      107.200,  105.300,  103.300,  101.700,  100.400,   99.400,   96.700,   94.700,   92.300,   90.400, 
      88.600,   87.800,   86.900,   85.900,   84.900,   83.800,   82.700,   81.900,   81.100,   80.300, 
      79.500,   78.800,   77.900,   77.100,   76.300,   75.500,   74.700,   73.800,   73.000,   72.200, 
      71.500,   70.800,   70.200,   69.500,   68.800,   68.100,   66.700,   65.400,   64.100,   62.800, 
      61.500,   60.400,   59.300,   58.200,   57.100,   56.100,   55.100,   54.100,   53.200,   52.300, 
      51.400,   50.600,   49.800,   48.900,   48.000,   47.300,   46.700,   46.100,   45.600,   45.000, 
      44.500,   44.000,   43.600,   43.100,   42.700,   42.200,   41.700,   41.300,   40.900,   40.400, 
      39.900,   39.400,   39.000,   38.500,   38.100,   37.600,   37.200,   36.700,   36.300,   35.800, 
      35.400,   34.900,   34.500,   34.100,   33.700,   33.300, ],
    [ 142.100,  136.900,  133.000,  131.400,  130.500,  129.600,  128.700,  127.700,  126.700,  125.600, 
      124.400,  123.100,  121.700,  120.100,  118.400,  116.600,  114.900,  113.000,  111.200,  109.300, 
      107.400,  105.400,  103.500,  101.900,  100.700,   99.600,   96.900,   95.000,   92.500,   90.600, 
      88.900,   88.200,   87.300,   86.300,   85.300,   84.300,   83.200,   82.400,   81.600,   80.900, 
      80.200,   79.400,   78.600,   77.800,   77.000,   76.200,   75.400,   74.500,   73.700,   73.000, 
      72.300,   71.600,   71.000,   70.300,   69.600,   68.900,   67.500,   66.200,   65.000,   63.700, 
      62.400,   61.300,   60.300,   59.200,   58.000,   57.000,   56.000,   55.100,   54.200,   53.300, 
      52.400,   51.500,   50.800,   49.900,   49.000,   48.200,   47.600,   47.000,   46.500,   46.000, 
      45.500,   45.000,   44.500,   44.100,   43.700,   43.200,   42.700,   42.200,   41.800,   41.300, 
      40.800,   40.300,   39.900,   39.500,   39.000,   38.600,   38.100,   37.600,   37.200,   36.700, 
      36.200,   35.800,   35.400,   34.900,   34.500,   34.100, ],
    [ 142.200,  137.000,  133.000,  131.500,  130.600,  129.700,  128.800,  127.800,  126.800,  125.700, 
      124.400,  123.200,  121.700,  120.200,  118.500,  116.700,  115.000,  113.100,  111.300,  109.400, 
      107.500,  105.600,  103.800,  102.200,  100.900,   99.700,   97.100,   95.400,   92.800,   90.800, 
      89.200,   88.600,   87.700,   86.700,   85.800,   84.700,   83.700,   82.900,   82.200,   81.400, 
      80.700,   80.000,   79.200,   78.400,   77.700,   76.900,   76.100,   75.200,   74.400,   73.700, 
      73.000,   72.300,   71.700,   71.100,   70.400,   69.700,   68.400,   67.000,   65.800,   64.500, 
      63.300,   62.200,   61.200,   60.100,   59.000,   57.900,   57.000,   56.000,   55.100,   54.200, 
      53.300,   52.500,   51.700,   50.800,   49.900,   49.200,   48.500,   48.000,   47.400,   46.900, 
      46.400,   45.900,   45.500,   45.000,   44.600,   44.100,   43.600,   43.200,   42.700,   42.200, 
      41.700,   41.300,   40.800,   40.400,   39.900,   39.400,   38.900,   38.500,   38.000,   37.600, 
      37.100,   36.600,   36.200,   35.800,   35.400,   34.900, ],
    [ 142.200,  137.000,  133.100,  131.600,  130.700,  129.800,  128.800,  127.900,  126.800,  125.700, 
      124.500,  123.200,  121.800,  120.200,  118.600,  116.800,  115.000,  113.200,  111.500,  109.600, 
      107.700,  105.800,  104.000,  102.400,  101.000,   99.800,   97.400,   95.500,   93.000,   90.900, 
      89.400,   89.000,   88.100,   87.100,   86.200,   85.200,   84.200,   83.400,   82.700,   82.000, 
      81.300,   80.600,   79.800,   79.000,   78.300,   77.500,   76.700,   75.900,   75.100,   74.400, 
      73.700,   73.000,   72.500,   71.800,   71.200,   70.500,   69.100,   67.900,   66.600,   65.400, 
      64.100,   63.000,   62.000,   60.900,   59.800,   58.800,   57.900,   56.900,   56.000,   55.100, 
      54.200,   53.400,   52.600,   51.700,   50.800,   50.100,   49.400,   48.900,   48.300,   47.800, 
      47.300,   46.800,   46.400,   45.900,   45.500,   45.000,   44.500,   44.100,   43.700,   43.200, 
      42.600,   42.200,   41.700,   41.200,   40.800,   40.300,   39.800,   39.300,   38.900,   38.400, 
      37.900,   37.500,   37.000,   36.600,   36.200,   35.700, ],
    [ 142.300,  137.100,  133.100,  131.600,  130.700,  129.800,  128.800,  127.900,  126.900,  125.700, 
      124.500,  123.300,  121.800,  120.300,  118.600,  116.900,  115.200,  113.300,  111.600,  109.700, 
      107.900,  106.000,  104.100,  102.400,  101.100,  100.000,   97.600,   95.600,   93.300,   91.100, 
      89.700,   89.400,   88.500,   87.500,   86.600,   85.600,   84.600,   83.900,   83.200,   82.500, 
      81.900,   81.100,   80.400,   79.600,   78.900,   78.100,   77.400,   76.600,   75.800,   75.100, 
      74.400,   73.700,   73.200,   72.500,   71.900,   71.200,   69.900,   68.600,   67.400,   66.200, 
      64.900,   63.900,   62.900,   61.800,   60.700,   59.700,   58.700,   57.800,   56.900,   56.000, 
      55.100,   54.200,   53.500,   52.600,   51.700,   51.000,   50.300,   49.800,   49.200,   48.700, 
      48.200,   47.800,   47.300,   46.900,   46.400,   45.900,   45.500,   45.000,   44.600,   44.100, 
      43.500,   43.100,   42.600,   42.100,   41.700,   41.200,   40.700,   40.200,   39.700,   39.300, 
      38.800,   38.300,   37.900,   37.400,   37.000,   36.500, ],
    [ 142.300,  137.100,  133.200,  131.600,  130.700,  129.800,  128.900,  127.900,  126.900,  125.800, 
      124.600,  123.300,  121.900,  120.400,  118.700,  117.000,  115.300,  113.400,  111.700,  109.900, 
      108.000,  106.100,  104.200,  102.500,  101.200,  100.100,   97.900,   95.700,   93.500,   91.300, 
      90.000,   89.700,   88.800,   87.900,   87.000,   86.000,   85.100,   84.300,   83.700,   83.000, 
      82.400,   81.700,   81.000,   80.200,   79.500,   78.700,   78.000,   77.200,   76.400,   75.700, 
      75.000,   74.400,   73.900,   73.200,   72.600,   71.900,   70.600,   69.400,   68.200,   67.000, 
      65.700,   64.700,   63.700,   62.600,   61.500,   60.500,   59.600,   58.600,   57.700,   56.800, 
      55.900,   55.100,   54.400,   53.500,   52.600,   51.800,   51.200,   50.600,   50.100,   49.600, 
      49.100,   48.600,   48.200,   47.700,   47.300,   46.800,   46.300,   45.900,   45.500,   45.000, 
      44.400,   43.900,   43.500,   43.000,   42.600,   42.100,   41.500,   41.100,   40.600,   40.100, 
      39.600,   39.100,   38.700,   38.200,   37.800,   37.300, ],
    [ 142.300,  137.100,  133.200,  131.700,  130.800,  129.900,  128.900,  128.000,  127.000,  125.900, 
      124.600,  123.400,  122.000,  120.500,  118.800,  117.100,  115.400,  113.600,  111.900,  110.000, 
      108.100,  106.200,  104.300,  102.700,  101.400,  100.300,   98.100,   95.800,   93.800,   91.400, 
      90.200,   90.000,   89.100,   88.200,   87.400,   86.400,   85.500,   84.800,   84.100,   83.500, 
      82.900,   82.200,   81.500,   80.800,   80.100,   79.300,   78.600,   77.800,   77.000,   76.300, 
      75.700,   75.100,   74.500,   73.900,   73.300,   72.600,   71.400,   70.100,   68.900,   67.700, 
      66.500,   65.400,   64.500,   63.400,   62.300,   61.300,   60.400,   59.500,   58.600,   57.700, 
      56.800,   56.000,   55.200,   54.300,   53.400,   52.700,   52.100,   51.500,   51.000,   50.500, 
      50.000,   49.500,   49.100,   48.600,   48.200,   47.700,   47.200,   46.800,   46.300,   45.800, 
      45.300,   44.800,   44.300,   43.900,   43.400,   42.900,   42.400,   41.900,   41.400,   40.900, 
      40.400,   40.000,   39.500,   39.000,   38.600,   38.100, ],
    [ 142.300,  137.200,  133.300,  131.700,  130.800,  129.900,  129.000,  128.100,  127.000,  125.900, 
      124.700,  123.500,  122.100,  120.600,  118.900,  117.200,  115.500,  113.700,  111.900,  110.000, 
      108.100,  106.300,  104.400,  102.800,  101.600,  100.500,   98.300,   95.900,   94.000,   91.600, 
      90.400,   90.500,   89.400,   88.600,   87.800,   86.800,   85.900,   85.200,   84.600,   83.900, 
      83.400,   82.700,   82.000,   81.300,   80.600,   79.900,   79.100,   78.400,   77.600,   76.900, 
      76.300,   75.700,   75.200,   74.600,   74.000,   73.300,   72.000,   70.800,   69.700,   68.400, 
      67.300,   66.200,   65.300,   64.200,   63.100,   62.100,   61.200,   60.300,   59.400,   58.500, 
      57.600,   56.800,   56.100,   55.200,   54.300,   53.500,   52.900,   52.300,   51.800,   51.300, 
      50.900,   50.400,   49.900,   49.500,   49.100,   48.600,   48.100,   47.600,   47.200,   46.700, 
      46.200,   45.700,   45.200,   44.700,   44.300,   43.800,   43.200,   42.700,   42.300,   41.800, 
      41.300,   40.800,   40.300,   39.800,   39.400,   38.900, ],
    [ 142.400,  137.200,  133.300,  131.800,  130.900,  130.000,  129.100,  128.100,  127.100,  126.000, 
      124.800,  123.600,  122.200,  120.700,  119.000,  117.300,  115.500,  113.700,  112.000,  110.100, 
      108.200,  106.400,  104.500,  103.000,  101.700,  100.700,   98.400,   96.100,   94.300,   91.700, 
      90.600,   90.900,   89.700,   88.900,   88.200,   87.200,   86.200,   85.600,   85.000,   84.400, 
      83.900,   83.200,   82.500,   81.800,   81.100,   80.400,   79.700,   78.900,   78.200,   77.500, 
      76.900,   76.300,   75.800,   75.200,   74.600,   74.000,   72.700,   71.500,   70.400,   69.200, 
      68.000,   66.900,   66.000,   65.000,   63.900,   62.900,   62.000,   61.100,   60.200,   59.300, 
      58.400,   57.600,   56.900,   56.000,   55.100,   54.300,   53.700,   53.200,   52.700,   52.200, 
      51.700,   51.200,   50.800,   50.400,   49.900,   49.500,   49.000,   48.500,   48.100,   47.600, 
      47.000,   46.500,   46.000,   45.600,   45.100,   44.600,   44.100,   43.600,   43.100,   42.600, 
      42.100,   41.600,   41.100,   40.600,   40.200,   39.700, ],
    [ 142.400,  137.300,  133.400,  131.800,  131.000,  130.100,  129.100,  128.200,  127.200,  126.100, 
      124.900,  123.700,  122.200,  120.700,  119.100,  117.300,  115.600,  113.700,  112.000,  110.200, 
      108.300,  106.500,  104.700,  103.100,  101.900,  100.900,   98.400,   96.300,   94.500,   91.900, 
      90.800,   91.300,   90.000,   89.200,   88.500,   87.500,   86.600,   86.000,   85.400,   84.800, 
      84.300,   83.700,   83.000,   82.300,   81.700,   80.900,   80.200,   79.500,   78.700,   78.100, 
      77.500,   76.900,   76.400,   75.800,   75.200,   74.600,   73.400,   72.200,   71.000,   69.900, 
      68.700,   67.700,   66.700,   65.700,   64.600,   63.700,   62.800,   61.800,   61.000,   60.100, 
      59.200,   58.400,   57.700,   56.800,   55.900,   55.100,   54.500,   54.000,   53.500,   53.000, 
      52.500,   52.100,   51.600,   51.200,   50.800,   50.300,   49.800,   49.400,   48.900,   48.400, 
      47.900,   47.400,   46.900,   46.400,   45.900,   45.400,   44.900,   44.400,   43.900,   43.400, 
      42.900,   42.400,   41.900,   41.400,   40.900,   40.500, ],
    [ 142.500,  137.300,  133.400,  131.900,  131.000,  130.100,  129.200,  128.300,  127.300,  126.100, 
      124.900,  123.700,  122.300,  120.700,  119.100,  117.300,  115.600,  113.800,  112.100,  110.300, 
      108.400,  106.600,  104.800,  103.300,  102.100,  101.000,   98.500,   96.400,   94.700,   92.000, 
      91.000,   91.600,   90.500,   89.500,   88.900,   87.800,   86.900,   86.400,   85.800,   85.300, 
      84.800,   84.200,   83.500,   82.800,   82.200,   81.400,   80.800,   80.000,   79.300,   78.600, 
      78.000,   77.500,   77.000,   76.400,   75.900,   75.200,   74.000,   72.800,   71.700,   70.500, 
      69.400,   68.400,   67.500,   66.400,   65.400,   64.400,   63.500,   62.600,   61.700,   60.900, 
      60.000,   59.200,   58.500,   57.600,   56.600,   55.900,   55.300,   54.800,   54.300,   53.800, 
      53.300,   52.900,   52.500,   52.000,   51.600,   51.100,   50.600,   50.200,   49.800,   49.200, 
      48.700,   48.200,   47.700,   47.200,   46.800,   46.200,   45.700,   45.200,   44.700,   44.200, 
      43.700,   43.200,   42.700,   42.200,   41.700,   41.200, ],
    [ 142.500,  137.400,  133.500,  132.000,  131.100,  130.200,  129.200,  128.300,  127.300,  126.200, 
      124.900,  123.700,  122.300,  120.800,  119.100,  117.400,  115.700,  113.900,  112.200,  110.400, 
      108.500,  106.700,  104.900,  103.500,  102.200,  101.100,   98.600,   96.600,   94.700,   92.200, 
      91.200,   92.000,   91.000,   89.700,   89.200,   88.200,   87.300,   86.800,   86.200,   85.700, 
      85.200,   84.600,   84.000,   83.300,   82.700,   81.900,   81.300,   80.500,   79.800,   79.200, 
      78.600,   78.000,   77.600,   77.000,   76.400,   75.800,   74.600,   73.400,   72.300,   71.200, 
      70.000,   69.000,   68.100,   67.100,   66.100,   65.100,   64.200,   63.300,   62.500,   61.600, 
      60.700,   59.900,   59.200,   58.300,   57.400,   56.700,   56.100,   55.600,   55.100,   54.600, 
      54.200,   53.700,   53.300,   52.900,   52.400,   52.000,   51.500,   51.000,   50.600,   50.100, 
      49.500,   49.000,   48.500,   48.000,   47.600,   47.100,   46.500,   46.000,   45.500,   45.000, 
      44.400,   43.900,   43.400,   43.000,   42.500,   42.000, ],
    [ 142.500,  137.400,  133.500,  132.000,  131.100,  130.200,  129.300,  128.300,  127.300,  126.200, 
      125.000,  123.800,  122.400,  120.800,  119.200,  117.500,  115.800,  114.000,  112.300,  110.500, 
      108.600,  106.900,  105.100,  103.500,  102.200,  101.100,   98.700,   96.800,   94.700,   92.300, 
      91.400,   92.300,   91.500,   90.000,   89.600,   88.500,   87.600,   87.100,   86.600,   86.100, 
      85.600,   85.100,   84.400,   83.700,   83.100,   82.400,   81.800,   81.000,   80.300,   79.700, 
      79.100,   78.600,   78.100,   77.600,   77.000,   76.400,   75.200,   74.100,   73.000,   71.800, 
      70.700,   69.700,   68.800,   67.800,   66.800,   65.800,   64.900,   64.100,   63.200,   62.300, 
      61.500,   60.700,   60.000,   59.100,   58.200,   57.400,   56.900,   56.300,   55.900,   55.400, 
      54.900,   54.500,   54.100,   53.700,   53.300,   52.800,   52.300,   51.800,   51.400,   50.900, 
      50.300,   49.800,   49.300,   48.900,   48.400,   47.900,   47.300,   46.800,   46.300,   45.700, 
      45.200,   44.700,   44.200,   43.700,   43.200,   42.700, ],
    [ 142.500,  137.400,  133.500,  132.000,  131.100,  130.200,  129.300,  128.300,  127.300,  126.200, 
      125.000,  123.800,  122.400,  120.900,  119.300,  117.500,  115.800,  114.100,  112.400,  110.600, 
      108.800,  107.000,  105.100,  103.600,  102.300,  101.200,   98.800,   96.900,   94.800,   92.500, 
      91.600,   92.600,   91.500,   90.600,   90.000,   88.800,   87.900,   87.500,   87.000,   86.500, 
      86.100,   85.500,   84.900,   84.200,   83.600,   82.900,   82.200,   81.500,   80.800,   80.200, 
      79.600,   79.100,   78.700,   78.100,   77.600,   77.000,   75.800,   74.700,   73.600,   72.400, 
      71.300,   70.400,   69.500,   68.500,   67.500,   66.500,   65.600,   64.800,   63.900,   63.100, 
      62.200,   61.400,   60.700,   59.800,   58.900,   58.200,   57.600,   57.100,   56.600,   56.200, 
      55.700,   55.300,   54.900,   54.500,   54.100,   53.600,   53.100,   52.600,   52.200,   51.700, 
      51.100,   50.600,   50.100,   49.700,   49.200,   48.700,   48.100,   47.600,   47.000,   46.500, 
      46.000,   45.500,   45.000,   44.500,   44.000,   43.500, ],
    [ 142.600,  137.400,  133.500,  132.000,  131.200,  130.200,  129.300,  128.400,  127.400,  126.300, 
      125.100,  123.900,  122.500,  121.000,  119.300,  117.600,  115.900,  114.100,  112.500,  110.700, 
      108.800,  107.000,  105.200,  103.600,  102.400,  101.300,   99.000,   97.100,   94.800,   92.600, 
      91.800,   92.600,   91.600,   91.100,   90.200,   89.100,   88.200,   87.900,   87.400,   86.900, 
      86.500,   85.900,   85.300,   84.600,   84.000,   83.300,   82.700,   82.000,   81.200,   80.700, 
      80.100,   79.600,   79.200,   78.700,   78.100,   77.500,   76.400,   75.200,   74.200,   73.100, 
      71.900,   71.000,   70.100,   69.100,   68.100,   67.200,   66.300,   65.500,   64.600,   63.800, 
      62.900,   62.100,   61.400,   60.500,   59.600,   58.900,   58.300,   57.800,   57.400,   56.900, 
      56.500,   56.100,   55.600,   55.200,   54.800,   54.400,   53.900,   53.400,   53.000,   52.500, 
      51.900,   51.400,   50.900,   50.400,   50.000,   49.400,   48.900,   48.300,   47.800,   47.300, 
      46.700,   46.200,   45.700,   45.200,   44.700,   44.200, ],
    [ 142.600,  137.500,  133.600,  132.100,  131.200,  130.300,  129.400,  128.400,  127.400,  126.300, 
      125.100,  123.900,  122.500,  121.000,  119.400,  117.700,  116.000,  114.200,  112.600,  110.700, 
      108.900,  107.000,  105.200,  103.700,  102.400,  101.400,   99.100,   97.200,   94.900,   92.800, 
      92.000,   92.600,   91.700,   91.500,   90.300,   89.300,   88.500,   88.200,   87.700,   87.200, 
      86.900,   86.300,   85.700,   85.000,   84.500,   83.800,   83.100,   82.400,   81.700,   81.100, 
      80.600,   80.100,   79.700,   79.200,   78.700,   78.100,   76.900,   75.800,   74.800,   73.600, 
      72.500,   71.600,   70.800,   69.800,   68.800,   67.800,   67.000,   66.100,   65.300,   64.500, 
      63.600,   62.800,   62.100,   61.200,   60.300,   59.600,   59.100,   58.600,   58.100,   57.700, 
      57.200,   56.800,   56.400,   56.000,   55.600,   55.200,   54.700,   54.200,   53.800,   53.300, 
      52.700,   52.200,   51.700,   51.200,   50.700,   50.200,   49.600,   49.100,   48.600,   48.000, 
      47.500,   47.000,   46.500,   46.000,   45.500,   44.900, ],
    [ 142.600,  137.500,  133.600,  132.100,  131.200,  130.300,  129.400,  128.500,  127.500,  126.400, 
      125.200,  124.000,  122.600,  121.100,  119.500,  117.800,  116.100,  114.300,  112.600,  110.700, 
      108.900,  107.100,  105.300,  103.700,  102.500,  101.500,   99.200,   97.200,   94.900,   92.900, 
      92.100,   93.300,   92.400,   91.500,   90.500,   89.600,   88.800,   88.500,   88.100,   87.600, 
      87.300,   86.700,   86.100,   85.500,   84.900,   84.200,   83.600,   82.900,   82.100,   81.600, 
      81.100,   80.600,   80.200,   79.700,   79.200,   78.600,   77.500,   76.400,   75.300,   74.200, 
      73.100,   72.200,   71.400,   70.400,   69.400,   68.500,   67.600,   66.800,   66.000,   65.100, 
      64.300,   63.500,   62.800,   61.900,   61.000,   60.300,   59.800,   59.300,   58.900,   58.400, 
      58.000,   57.600,   57.200,   56.800,   56.400,   55.900,   55.400,   55.000,   54.600,   54.100, 
      53.500,   53.000,   52.500,   52.000,   51.500,   51.000,   50.400,   49.900,   49.300,   48.800, 
      48.200,   47.700,   47.200,   46.700,   46.200,   45.700, ],
    [ 142.600,  137.500,  133.700,  132.200,  131.300,  130.400,  129.500,  128.500,  127.500,  126.400, 
      125.200,  124.000,  122.700,  121.100,  119.500,  117.800,  116.100,  114.300,  112.600,  110.800, 
      108.900,  107.100,  105.300,  103.800,  102.600,  101.600,   99.400,   97.300,   95.000,   93.100, 
      92.300,   93.500,   92.400,   91.600,   90.700,   89.800,   89.100,   88.900,   88.400,   88.000, 
      87.600,   87.100,   86.500,   85.900,   85.300,   84.600,   84.000,   83.300,   82.600,   82.000, 
      81.500,   81.100,   80.700,   80.200,   79.700,   79.100,   78.000,   76.900,   75.900,   74.800, 
      73.700,   72.800,   72.000,   71.000,   70.000,   69.100,   68.300,   67.400,   66.600,   65.800, 
      64.900,   64.200,   63.500,   62.600,   61.700,   61.000,   60.500,   60.000,   59.600,   59.100, 
      58.700,   58.300,   57.900,   57.500,   57.200,   56.700,   56.200,   55.800,   55.400,   54.800, 
      54.300,   53.700,   53.200,   52.800,   52.300,   51.700,   51.200,   50.600,   50.100,   49.500, 
      49.000,   48.400,   47.900,   47.400,   46.900,   46.400, ],
    [ 142.700,  137.600,  133.700,  132.200,  131.300,  130.400,  129.500,  128.600,  127.600,  126.500, 
      125.300,  124.100,  122.700,  121.200,  119.500,  117.800,  116.100,  114.300,  112.600,  110.800, 
      109.000,  107.200,  105.400,  103.900,  102.700,  101.700,   99.500,   97.300,   95.100,   93.200, 
      92.500,   93.500,   92.500,   91.700,   90.800,   90.200,   89.400,   89.200,   88.800,   88.300, 
      88.000,   87.500,   86.900,   86.200,   85.700,   85.000,   84.400,   83.700,   83.000,   82.500, 
      82.000,   81.500,   81.200,   80.700,   80.200,   79.600,   78.500,   77.400,   76.400,   75.300, 
      74.200,   73.400,   72.600,   71.600,   70.600,   69.700,   68.900,   68.100,   67.300,   66.400, 
      65.600,   64.800,   64.200,   63.300,   62.400,   61.700,   61.200,   60.700,   60.300,   59.800, 
      59.400,   59.000,   58.700,   58.300,   57.900,   57.500,   57.000,   56.500,   56.100,   55.600, 
      55.000,   54.500,   54.000,   53.500,   53.000,   52.500,   51.900,   51.300,   50.800,   50.300, 
      49.700,   49.200,   48.700,   48.100,   47.600,   47.100, ],
    [ 142.700,  137.600,  133.700,  132.300,  131.400,  130.500,  129.500,  128.600,  127.600,  126.500, 
      125.300,  124.100,  122.700,  121.200,  119.500,  117.800,  116.100,  114.300,  112.700,  110.800, 
      109.000,  107.200,  105.500,  104.000,  102.800,  101.800,   99.500,   97.300,   95.200,   93.400, 
      94.400,   93.600,   93.000,   92.400,   91.000,   90.900,   89.600,   89.600,   89.100,   88.600, 
      88.400,   87.900,   87.300,   86.600,   86.100,   85.400,   84.800,   84.100,   83.400,   82.900, 
      82.400,   82.000,   81.600,   81.200,   80.700,   80.100,   79.000,   77.900,   76.900,   75.900, 
      74.800,   73.900,   73.100,   72.200,   71.200,   70.300,   69.500,   68.700,   67.900,   67.100, 
      66.200,   65.500,   64.800,   63.900,   63.000,   62.400,   61.800,   61.400,   61.000,   60.500, 
      60.100,   59.800,   59.400,   59.000,   58.600,   58.200,   57.700,   57.300,   56.900,   56.300, 
      55.800,   55.300,   54.800,   54.300,   53.800,   53.200,   52.600,   52.100,   51.600,   51.000, 
      50.400,   49.900,   49.400,   48.800,   48.300,   47.800, ],
    [ 142.700,  137.600,  133.800,  132.300,  131.400,  130.500,  129.600,  128.600,  127.600,  126.500, 
      125.300,  124.100,  122.700,  121.200,  119.600,  117.800,  116.200,  114.400,  112.700,  110.900, 
      109.100,  107.300,  105.600,  104.100,  102.900,  101.900,   99.500,   97.400,   95.300,   93.500, 
      94.800,   94.200,   93.100,   92.400,   91.200,   91.400,   89.800,   90.000,   89.500,   89.000, 
      88.700,   88.200,   87.600,   87.000,   86.500,   85.800,   85.200,   84.500,   83.800,   83.300, 
      82.900,   82.400,   82.100,   81.600,   81.200,   80.600,   79.500,   78.400,   77.500,   76.400, 
      75.300,   74.500,   73.700,   72.800,   71.800,   70.900,   70.100,   69.300,   68.500,   67.700, 
      66.900,   66.100,   65.500,   64.600,   63.700,   63.000,   62.500,   62.000,   61.600,   61.200, 
      60.800,   60.500,   60.100,   59.700,   59.400,   58.900,   58.400,   58.000,   57.600,   57.100, 
      56.500,   56.000,   55.500,   55.000,   54.500,   54.000,   53.400,   52.800,   52.300,   51.700, 
      51.100,   50.600,   50.100,   49.500,   49.000,   48.500, ],
    [ 142.700,  137.600,  133.800,  132.300,  131.400,  130.500,  129.600,  128.600,  127.600,  126.500, 
      125.300,  124.100,  122.700,  121.200,  119.600,  117.900,  116.200,  114.400,  112.800,  111.000, 
      109.200,  107.400,  105.600,  104.200,  103.000,  102.000,   99.600,   97.400,   95.400,   93.700, 
      94.800,   94.300,   93.100,   92.500,   91.400,   91.500,   90.300,   90.200,   89.800,   89.300, 
      89.000,   88.600,   88.000,   87.400,   86.900,   86.200,   85.600,   84.900,   84.200,   83.700, 
      83.300,   82.900,   82.500,   82.100,   81.600,   81.000,   80.000,   78.900,   78.000,   76.900, 
      75.800,   75.000,   74.300,   73.300,   72.400,   71.500,   70.700,   69.900,   69.100,   68.300, 
      67.500,   66.800,   66.100,   65.200,   64.300,   63.600,   63.100,   62.700,   62.300,   61.900, 
      61.500,   61.200,   60.800,   60.400,   60.100,   59.700,   59.200,   58.700,   58.300,   57.800, 
      57.200,   56.700,   56.200,   55.700,   55.200,   54.700,   54.100,   53.500,   53.000,   52.400, 
      51.900,   51.300,   50.800,   50.200,   49.700,   49.200, ],
    [ 142.700,  137.600,  133.800,  132.300,  131.400,  130.500,  129.600,  128.700,  127.600,  126.500, 
      125.300,  124.100,  122.700,  121.200,  119.600,  117.900,  116.200,  114.500,  112.800,  111.000, 
      109.200,  107.500,  105.700,  104.300,  103.100,  102.000,   99.600,   97.500,   95.500,   93.800, 
      94.900,   94.300,   93.600,   92.600,   92.200,   91.600,   91.000,   90.300,   90.200,   89.500, 
      89.300,   88.900,   88.300,   87.700,   87.200,   86.600,   86.000,   85.300,   84.600,   84.100, 
      83.700,   83.300,   83.000,   82.600,   82.100,   81.500,   80.400,   79.400,   78.400,   77.400, 
      76.300,   75.500,   74.800,   73.900,   72.900,   72.000,   71.300,   70.500,   69.700,   68.900, 
      68.100,   67.400,   66.700,   65.800,   64.900,   64.300,   63.800,   63.300,   63.000,   62.600, 
      62.200,   61.800,   61.500,   61.100,   60.800,   60.400,   59.900,   59.500,   59.100,   58.500, 
      58.000,   57.400,   56.900,   56.500,   56.000,   55.400,   54.800,   54.200,   53.700,   53.100, 
      52.600,   52.000,   51.500,   50.900,   50.400,   49.800, ],
    [ 142.700,  137.700,  133.800,  132.300,  131.400,  130.500,  129.600,  128.700,  127.700,  126.600, 
      125.400,  124.200,  122.800,  121.300,  119.700,  118.000,  116.300,  114.500,  112.900,  111.100, 
      109.300,  107.500,  105.800,  104.300,  103.100,  102.000,   99.600,   97.500,   95.500,   94.000, 
      95.400,   94.700,   93.600,   92.700,   92.300,   91.700,   91.500,   90.500,   90.600,   89.800, 
      89.600,   89.200,   88.700,   88.000,   87.600,   86.900,   86.400,   85.700,   85.000,   84.500, 
      84.100,   83.700,   83.400,   83.000,   82.500,   81.900,   80.900,   79.900,   78.900,   77.900, 
      76.800,   76.000,   75.300,   74.400,   73.400,   72.600,   71.800,   71.000,   70.300,   69.500, 
      68.700,   68.000,   67.300,   66.500,   65.500,   64.900,   64.400,   64.000,   63.600,   63.200, 
      62.900,   62.500,   62.200,   61.800,   61.500,   61.100,   60.600,   60.200,   59.800,   59.300, 
      58.700,   58.200,   57.700,   57.200,   56.700,   56.100,   55.500,   54.900,   54.400,   53.800, 
      53.300,   52.700,   52.200,   51.600,   51.100,   50.500, ],
    [ 142.800,  137.700,  133.800,  132.300,  131.400,  130.500,  129.600,  128.700,  127.700,  126.600, 
      125.400,  124.200,  122.800,  121.300,  119.700,  118.000,  116.300,  114.600,  112.900,  111.100, 
      109.400,  107.600,  105.800,  104.300,  103.100,  102.000,   99.700,   97.600,   95.600,   94.100, 
      95.400,   94.700,   94.100,   93.300,   92.400,   91.800,   91.500,   90.700,   91.000,   90.000, 
      89.800,   89.500,   89.000,   88.400,   87.900,   87.300,   86.700,   86.000,   85.300,   84.900, 
      84.500,   84.100,   83.800,   83.400,   83.000,   82.400,   81.300,   80.300,   79.400,   78.400, 
      77.300,   76.500,   75.800,   74.900,   74.000,   73.100,   72.400,   71.600,   70.800,   70.100, 
      69.200,   68.500,   67.900,   67.100,   66.100,   65.500,   65.000,   64.600,   64.300,   63.900, 
      63.500,   63.200,   62.900,   62.500,   62.200,   61.800,   61.300,   60.900,   60.500,   60.000, 
      59.400,   58.900,   58.400,   57.900,   57.400,   56.800,   56.200,   55.600,   55.100,   54.500, 
      53.900,   53.400,   52.800,   52.300,   51.800,   51.200, ],
    [ 142.800,  137.700,  133.800,  132.300,  131.500,  130.600,  129.700,  128.700,  127.700,  126.600, 
      125.400,  124.200,  122.800,  121.400,  119.700,  118.100,  116.400,  114.600,  113.000,  111.200, 
      109.400,  107.600,  105.800,  104.300,  103.100,  102.000,   99.700,   97.600,   95.700,   94.200, 
      95.400,   95.000,   94.100,   93.300,   92.900,   91.900,   91.600,   90.800,   91.400,   90.700, 
      90.200,   89.700,   89.300,   88.700,   88.300,   87.600,   87.100,   86.400,   85.700,   85.300, 
      84.900,   84.500,   84.300,   83.800,   83.400,   82.800,   81.800,   80.800,   79.900,   78.800, 
      77.800,   77.000,   76.300,   75.400,   74.500,   73.700,   72.900,   72.100,   71.400,   70.600, 
      69.800,   69.100,   68.500,   67.600,   66.700,   66.100,   65.600,   65.200,   64.900,   64.500, 
      64.200,   63.800,   63.500,   63.200,   62.900,   62.500,   62.000,   61.600,   61.200,   60.700, 
      60.100,   59.600,   59.100,   58.600,   58.100,   57.500,   56.900,   56.300,   55.800,   55.200, 
      54.600,   54.100,   53.500,   53.000,   52.400,   51.900, ],
    [ 142.800,  137.700,  133.900,  132.400,  131.500,  130.600,  129.700,  128.800,  127.800,  126.700, 
      125.500,  124.300,  122.900,  121.400,  119.800,  118.100,  116.400,  114.700,  113.000,  111.200, 
      109.400,  107.600,  105.900,  104.300,  103.100,  102.100,   99.700,   97.700,   95.800,   96.100, 
      95.900,   95.100,   94.500,   93.800,   93.000,   92.100,   91.700,   91.800,   91.800,   91.200, 
      90.800,   90.000,   89.500,   89.000,   88.600,   88.000,   87.400,   86.700,   86.000,   85.600, 
      85.200,   84.900,   84.700,   84.300,   83.800,   83.200,   82.200,   81.200,   80.300,   79.300, 
      78.300,   77.500,   76.800,   76.000,   75.000,   74.200,   73.400,   72.700,   71.900,   71.200, 
      70.400,   69.700,   69.100,   68.200,   67.300,   66.700,   66.200,   65.800,   65.500,   65.200, 
      64.800,   64.500,   64.200,   63.900,   63.600,   63.100,   62.700,   62.300,   61.900,   61.400, 
      60.800,   60.300,   59.800,   59.300,   58.800,   58.200,   57.600,   57.000,   56.500,   55.900, 
      55.300,   54.700,   54.200,   53.600,   53.100,   52.500, ],
    [ 142.800,  137.700,  133.900,  132.400,  131.500,  130.600,  129.700,  128.800,  127.800,  126.700, 
      125.500,  124.300,  122.900,  121.400,  119.800,  118.100,  116.500,  114.700,  113.000,  111.200, 
      109.400,  107.600,  105.900,  104.400,  103.200,  102.100,   99.800,   97.800,   95.900,   96.100, 
      95.900,   95.400,   94.500,   93.800,   93.100,   92.200,   91.800,   92.300,   92.100,   91.600, 
      91.300,   90.600,   89.800,   89.300,   88.900,   88.300,   87.800,   87.100,   86.400,   86.000, 
      85.600,   85.300,   85.100,   84.700,   84.200,   83.600,   82.600,   81.600,   80.800,   79.700, 
      78.700,   78.000,   77.300,   76.400,   75.500,   74.700,   74.000,   73.200,   72.500,   71.700, 
      70.900,   70.300,   69.700,   68.800,   67.900,   67.200,   66.800,   66.400,   66.100,   65.800, 
      65.500,   65.100,   64.800,   64.500,   64.200,   63.800,   63.400,   62.900,   62.600,   62.000, 
      61.500,   60.900,   60.400,   59.900,   59.500,   58.900,   58.300,   57.700,   57.100,   56.600, 
      56.000,   55.400,   54.800,   54.300,   53.700,   53.200, ],
    [ 142.800,  137.700,  133.900,  132.400,  131.500,  130.600,  129.700,  128.800,  127.800,  126.700, 
      125.500,  124.300,  123.000,  121.500,  119.800,  118.100,  116.500,  114.700,  113.000,  111.200, 
      109.400,  107.600,  105.900,  104.400,  103.200,  102.100,   99.800,   97.900,   96.100,   96.500, 
      96.200,   95.700,   94.900,   94.200,   93.200,   92.300,   91.900,   92.400,   92.100,   92.000, 
      91.700,   91.100,   90.000,   89.500,   89.200,   88.600,   88.100,   87.400,   86.700,   86.300, 
      86.000,   85.600,   85.500,   85.100,   84.600,   84.100,   83.000,   82.100,   81.200,   80.200, 
      79.200,   78.400,   77.800,   76.900,   76.000,   75.200,   74.500,   73.700,   73.000,   72.300, 
      71.500,   70.800,   70.200,   69.300,   68.400,   67.800,   67.400,   67.000,   66.700,   66.400, 
      66.100,   65.800,   65.500,   65.200,   64.900,   64.500,   64.000,   63.600,   63.200,   62.700, 
      62.100,   61.600,   61.100,   60.600,   60.100,   59.600,   58.900,   58.400,   57.800,   57.200, 
      56.600,   56.000,   55.500,   54.900,   54.400,   53.800, ],
    [ 142.800,  137.800,  133.900,  132.400,  131.600,  130.700,  129.800,  128.800,  127.800,  126.700, 
      125.500,  124.300,  123.000,  121.500,  119.800,  118.100,  116.500,  114.700,  113.000,  111.200, 
      109.400,  107.700,  105.900,  104.400,  103.200,  102.200,   99.900,   97.900,   96.200,   96.500, 
      96.200,   95.700,   94.900,   94.300,   93.700,   92.500,   92.700,   92.400,   92.200,   92.400, 
      92.100,   91.500,   90.700,   89.800,   89.500,   88.900,   88.500,   87.700,   87.000,   86.700, 
      86.300,   86.000,   85.900,   85.500,   85.000,   84.400,   83.400,   82.500,   81.600,   80.600, 
      79.600,   78.900,   78.300,   77.400,   76.500,   75.700,   75.000,   74.200,   73.500,   72.800, 
      72.000,   71.300,   70.800,   69.900,   69.000,   68.400,   68.000,   67.600,   67.300,   67.000, 
      66.700,   66.400,   66.100,   65.800,   65.500,   65.100,   64.700,   64.300,   63.900,   63.400, 
      62.800,   62.300,   61.800,   61.300,   60.800,   60.200,   59.600,   59.000,   58.500,   57.900, 
      57.300,   56.700,   56.100,   55.600,   55.000,   54.400, ],
    [ 142.900,  137.800,  134.000,  132.500,  131.600,  130.700,  129.800,  128.800,  127.800,  126.800, 
      125.600,  124.300,  123.000,  121.500,  119.800,  118.100,  116.500,  114.700,  113.000,  111.200, 
      109.500,  107.700,  105.900,  104.400,  103.200,  102.200,   99.900,   98.000,   97.700,   97.000, 
      96.500,   96.000,   95.000,   94.300,   93.700,   92.700,   92.700,   92.500,   92.300,   92.800, 
      92.400,   91.900,   91.200,   90.000,   89.700,   89.200,   88.800,   88.000,   87.300,   87.000, 
      86.700,   86.400,   86.300,   85.900,   85.400,   84.800,   83.800,   82.900,   82.000,   81.000, 
      80.000,   79.300,   78.700,   77.900,   77.000,   76.200,   75.500,   74.700,   74.000,   73.300, 
      72.500,   71.900,   71.300,   70.400,   69.500,   68.900,   68.500,   68.200,   67.900,   67.600, 
      67.300,   67.000,   66.700,   66.500,   66.200,   65.800,   65.300,   64.900,   64.600,   64.100, 
      63.500,   63.000,   62.400,   62.000,   61.500,   60.900,   60.300,   59.700,   59.100,   58.500, 
      57.900,   57.300,   56.800,   56.200,   55.700,   55.100, ],
    [ 142.900,  137.800,  134.000,  132.500,  131.600,  130.700,  129.800,  128.800,  127.800,  126.800, 
      125.600,  124.300,  123.000,  121.500,  119.900,  118.200,  116.500,  114.700,  113.100,  111.300, 
      109.500,  107.700,  106.000,  104.500,  103.300,  102.200,  100.000,   98.100,   97.700,   97.000, 
      96.700,   96.300,   95.300,   94.800,   93.800,   92.800,   93.300,   93.100,   93.200,   93.200, 
      92.800,   92.300,   91.600,   90.700,   90.000,   89.500,   89.100,   88.300,   87.600,   87.400, 
      87.000,   86.700,   86.600,   86.200,   85.800,   85.200,   84.200,   83.300,   82.400,   81.400, 
      80.500,   79.800,   79.200,   78.300,   77.400,   76.600,   75.900,   75.200,   74.500,   73.800, 
      73.000,   72.400,   71.900,   71.000,   70.100,   69.500,   69.100,   68.700,   68.500,   68.200, 
      67.900,   67.600,   67.400,   67.100,   66.800,   66.400,   66.000,   65.600,   65.200,   64.700, 
      64.100,   63.600,   63.100,   62.600,   62.100,   61.600,   60.900,   60.300,   59.800,   59.200, 
      58.600,   58.000,   57.400,   56.800,   56.300,   55.700, ],
    [ 142.900,  137.800,  134.000,  132.500,  131.600,  130.700,  129.800,  128.900,  127.900,  126.800, 
      125.600,  124.400,  123.000,  121.500,  119.900,  118.200,  116.500,  114.700,  113.100,  111.300, 
      109.500,  107.700,  106.000,  104.500,  103.300,  102.300,  100.100,   98.200,   98.000,   97.200, 
      96.800,   96.300,   95.600,   95.100,   94.300,   93.500,   93.300,   93.100,   93.200,   93.500, 
      93.200,   92.500,   92.000,   91.200,   90.600,   89.700,   89.500,   88.600,   87.900,   87.700, 
      87.400,   87.100,   87.000,   86.600,   86.200,   85.600,   84.600,   83.700,   82.800,   81.900, 
      80.900,   80.200,   79.600,   78.800,   77.900,   77.100,   76.400,   75.700,   75.000,   74.300, 
      73.500,   72.900,   72.400,   71.500,   70.600,   70.000,   69.600,   69.300,   69.000,   68.800, 
      68.500,   68.200,   68.000,   67.700,   67.500,   67.100,   66.600,   66.200,   65.900,   65.400, 
      64.800,   64.300,   63.800,   63.300,   62.800,   62.200,   61.600,   61.000,   60.400,   59.800, 
      59.200,   58.600,   58.000,   57.500,   56.900,   56.300, ],
    [ 142.900,  137.800,  134.000,  132.500,  131.600,  130.700,  129.800,  128.900,  127.900,  126.800, 
      125.600,  124.400,  123.000,  121.500,  119.900,  118.200,  116.500,  114.700,  113.100,  111.300, 
      109.500,  107.800,  106.000,  104.500,  103.400,  102.300,  100.100,   98.300,   98.000,   97.200, 
      97.000,   96.500,   95.900,   95.100,   94.300,   93.600,   93.800,   93.900,   93.200,   93.900, 
      93.300,   92.600,   92.100,   91.700,   91.100,   90.000,   89.800,   88.900,   88.200,   88.000, 
      87.700,   87.400,   87.400,   87.000,   86.600,   86.000,   85.000,   84.000,   83.200,   82.300, 
      81.300,   80.600,   80.100,   79.200,   78.300,   77.600,   76.900,   76.200,   75.500,   74.800, 
      74.000,   73.400,   72.900,   72.000,   71.100,   70.500,   70.100,   69.800,   69.600,   69.300, 
      69.100,   68.800,   68.600,   68.300,   68.100,   67.700,   67.300,   66.900,   66.500,   66.000, 
      65.400,   64.900,   64.400,   63.900,   63.400,   62.800,   62.200,   61.600,   61.000,   60.400, 
      59.800,   59.200,   58.700,   58.100,   57.500,   56.900, ],
    [ 142.900,  137.800,  134.000,  132.500,  131.600,  130.700,  129.800,  128.900,  127.900,  126.800, 
      125.600,  124.400,  123.000,  121.500,  119.900,  118.200,  116.500,  114.800,  113.100,  111.300, 
      109.500,  107.800,  106.000,  104.600,  103.400,  102.400,  100.200,   99.400,   98.400,   97.400, 
      97.200,   96.600,   96.000,   95.200,   94.400,   93.700,   93.800,   93.900,   94.200,   94.200, 
      93.300,   92.600,   92.100,   92.100,   91.500,   90.600,   90.000,   89.100,   88.500,   88.300, 
      88.100,   87.800,   87.800,   87.400,   86.900,   86.300,   85.300,   84.400,   83.600,   82.600, 
      81.700,   81.000,   80.500,   79.700,   78.800,   78.000,   77.300,   76.700,   76.000,   75.300, 
      74.500,   73.900,   73.400,   72.500,   71.600,   71.000,   70.700,   70.400,   70.200,   69.900, 
      69.700,   69.400,   69.200,   69.000,   68.700,   68.300,   67.900,   67.500,   67.200,   66.600, 
      66.000,   65.500,   65.000,   64.500,   64.100,   63.500,   62.800,   62.200,   61.700,   61.100, 
      60.400,   59.900,   59.300,   58.700,   58.100,   57.500, ],
    [ 142.900,  137.800,  134.000,  132.500,  131.600,  130.700,  129.800,  128.900,  127.900,  126.800, 
      125.600,  124.400,  123.000,  121.500,  119.900,  118.200,  116.500,  114.800,  113.100,  111.300, 
      109.600,  107.800,  106.100,  104.600,  103.400,  102.400,  100.300,   99.400,   98.400,   97.700, 
      97.300,   96.600,   96.000,   95.600,   94.500,   94.200,   94.200,   94.700,   94.200,   94.400, 
      93.400,   93.300,   92.200,   92.500,   91.500,   91.100,   90.200,   89.400,   88.800,   88.700, 
      88.400,   88.100,   88.100,   87.700,   87.300,   86.700,   85.700,   84.800,   84.000,   83.000, 
      82.100,   81.400,   80.900,   80.100,   79.200,   78.500,   77.800,   77.100,   76.500,   75.800, 
      75.000,   74.400,   73.900,   73.000,   72.100,   71.500,   71.200,   70.900,   70.700,   70.500, 
      70.200,   70.000,   69.800,   69.600,   69.300,   69.000,   68.500,   68.100,   67.800,   67.300, 
      66.700,   66.200,   65.700,   65.200,   64.700,   64.100,   63.500,   62.900,   62.300,   61.700, 
      61.100,   60.500,   59.900,   59.300,   58.700,   58.100, ],
    [ 142.900,  137.800,  134.000,  132.500,  131.600,  130.700,  129.800,  128.900,  127.900,  126.800, 
      125.600,  124.400,  123.000,  121.500,  119.900,  118.200,  116.600,  114.800,  113.200,  111.400, 
      109.600,  107.800,  106.100,  104.700,  103.500,  102.500,  100.300,   99.700,   98.600,   97.700, 
      97.500,   96.900,   96.300,   95.600,   94.600,   94.200,   94.800,   94.800,   94.300,   94.400, 
      94.200,   93.500,   93.100,   92.600,   91.600,   91.500,   90.400,   89.600,   89.000,   89.000, 
      88.700,   88.400,   88.500,   88.100,   87.600,   87.000,   86.000,   85.100,   84.400,   83.400, 
      82.400,   81.900,   81.400,   80.500,   79.600,   78.900,   78.200,   77.600,   76.900,   76.200, 
      75.500,   74.900,   74.400,   73.500,   72.600,   72.000,   71.700,   71.400,   71.300,   71.000, 
      70.800,   70.600,   70.400,   70.200,   69.900,   69.600,   69.100,   68.800,   68.400,   67.900, 
      67.300,   66.800,   66.300,   65.800,   65.300,   64.700,   64.100,   63.500,   62.900,   62.300, 
      61.700,   61.100,   60.500,   59.900,   59.300,   58.700, ],
    [ 142.900,  137.800,  134.000,  132.500,  131.600,  130.700,  129.800,  128.900,  127.900,  126.800, 
      125.600,  124.400,  123.000,  121.500,  119.900,  118.200,  116.600,  114.800,  113.200,  111.400, 
      109.600,  107.900,  106.200,  104.700,  103.500,  102.500,  101.300,  100.000,   98.600,   97.900, 
      97.700,   97.200,   96.600,   95.700,   94.700,   94.700,   94.900,   94.800,   95.000,   94.500, 
      94.300,   93.500,   93.200,   92.600,   91.700,   91.500,   90.600,   89.900,   89.300,   89.300, 
      89.100,   88.700,   88.900,   88.500,   88.000,   87.400,   86.400,   85.500,   84.700,   83.800, 
      82.800,   82.200,   81.800,   80.900,   80.000,   79.300,   78.700,   78.000,   77.400,   76.700, 
      75.900,   75.400,   74.900,   74.000,   73.100,   72.500,   72.200,   72.000,   71.800,   71.600, 
      71.400,   71.200,   71.000,   70.700,   70.500,   70.200,   69.700,   69.400,   69.100,   68.500, 
      67.900,   67.400,   66.900,   66.400,   65.900,   65.400,   64.700,   64.100,   63.500,   62.900, 
      62.300,   61.700,   61.100,   60.500,   59.900,   59.300, ],
    [ 142.900,  137.800,  134.000,  132.500,  131.600,  130.700,  129.800,  128.900,  127.900,  126.800, 
      125.600,  124.400,  123.000,  121.600,  119.900,  118.300,  116.600,  114.800,  113.200,  111.400, 
      109.700,  107.900,  106.200,  104.700,  103.600,  102.600,  101.500,  100.000,   98.800,   98.100, 
      97.700,   97.400,   96.600,   96.000,   94.800,   95.100,   95.200,   95.300,   95.500,   95.200, 
      94.300,   93.600,   93.200,   92.700,   92.400,   91.600,   90.800,   90.400,   89.500,   89.600, 
      89.400,   89.000,   89.200,   88.800,   88.400,   87.700,   86.700,   85.900,   85.100,   84.100, 
      83.200,   82.600,   82.200,   81.300,   80.500,   79.700,   79.100,   78.500,   77.800,   77.100, 
      76.400,   75.800,   75.400,   74.500,   73.500,   73.000,   72.700,   72.500,   72.300,   72.100, 
      71.900,   71.700,   71.500,   71.300,   71.100,   70.800,   70.300,   70.000,   69.700,   69.100, 
      68.500,   68.000,   67.500,   67.000,   66.600,   66.000,   65.300,   64.700,   64.100,   63.500, 
      62.900,   62.300,   61.700,   61.100,   60.500,   59.900, ],
    [ 142.900,  137.800,  134.000,  132.500,  131.600,  130.800,  129.800,  128.900,  127.900,  126.800, 
      125.600,  124.400,  123.100,  121.600,  120.000,  118.300,  116.600,  114.900,  113.200,  111.400, 
      109.700,  107.900,  106.200,  104.800,  103.600,  103.400,  101.500,  100.200,   99.000,   98.300, 
      97.800,   97.400,   96.700,   96.100,   95.400,   95.400,   95.700,   95.300,   95.800,   95.600, 
      94.900,   94.200,   93.700,   93.500,   92.400,   91.700,   90.900,   91.000,   89.800,   90.000, 
      89.700,   89.300,   89.600,   89.200,   88.700,   88.000,   87.100,   86.200,   85.500,   84.500, 
      83.600,   83.000,   82.600,   81.700,   80.900,   80.200,   79.500,   78.900,   78.300,   77.600, 
      76.900,   76.300,   75.800,   74.900,   74.000,   73.500,   73.200,   73.000,   72.800,   72.600, 
      72.500,   72.300,   72.100,   71.900,   71.700,   71.400,   70.900,   70.600,   70.300,   69.700, 
      69.100,   68.600,   68.100,   67.600,   67.200,   66.600,   65.900,   65.300,   64.700,   64.100, 
      63.400,   62.900,   62.300,   61.700,   61.100,   60.500, ],
    [ 142.900,  137.800,  134.000,  132.500,  131.700,  130.800,  129.900,  128.900,  127.900,  126.800, 
      125.600,  124.400,  123.100,  121.600,  120.000,  118.300,  116.600,  114.900,  113.300,  111.500, 
      109.700,  108.000,  106.300,  104.800,  104.400,  103.400,  101.800,  100.200,   99.200,   98.400, 
      98.200,   97.500,   97.000,   96.100,   95.400,   95.400,   95.700,   96.000,   96.200,   95.600, 
      95.100,   94.200,   93.700,   93.500,   92.500,   92.400,   91.100,   91.500,   90.000,   90.200, 
      90.000,   89.600,   90.000,   89.500,   89.100,   88.300,   87.400,   86.500,   85.800,   84.900, 
      83.900,   83.400,   83.000,   82.100,   81.300,   80.600,   80.000,   79.300,   78.700,   78.000, 
      77.300,   76.700,   76.300,   75.400,   74.500,   73.900,   73.700,   73.500,   73.400,   73.200, 
      73.000,   72.800,   72.700,   72.500,   72.300,   72.000,   71.500,   71.200,   70.900,   70.300, 
      69.700,   69.200,   68.700,   68.200,   67.800,   67.200,   66.500,   65.900,   65.300,   64.700, 
      64.000,   63.400,   62.800,   62.300,   61.700,   61.000, ],
    [ 142.900,  137.900,  134.000,  132.500,  131.700,  130.800,  129.900,  128.900,  127.900,  126.900, 
      125.700,  124.500,  123.100,  121.600,  120.000,  118.300,  116.700,  114.900,  113.300,  111.500, 
      109.700,  108.000,  106.300,  105.500,  104.600,  103.600,  101.800,  100.400,   99.400,   98.600, 
      98.300,   97.900,   97.000,   96.200,   95.800,   95.700,   96.300,   96.000,   96.200,   95.600, 
      95.200,   94.800,   93.700,   93.500,   93.000,   92.400,   91.300,   91.500,   90.700,   90.300, 
      90.500,   89.800,   90.300,   90.000,   89.400,   88.600,   87.700,   86.900,   86.200,   85.200, 
      84.300,   83.800,   83.400,   82.500,   81.600,   81.000,   80.400,   79.700,   79.100,   78.500, 
      77.700,   77.200,   76.800,   75.900,   74.900,   74.400,   74.200,   74.000,   73.900,   73.700, 
      73.600,   73.400,   73.200,   73.100,   72.900,   72.600,   72.100,   71.800,   71.500,   70.900, 
      70.300,   69.800,   69.300,   68.800,   68.400,   67.800,   67.100,   66.500,   65.900,   65.300, 
      64.600,   64.000,   63.400,   62.800,   62.300,   61.600, ],
    [ 142.900,  137.900,  134.000,  132.600,  131.700,  130.800,  129.900,  128.900,  128.000,  126.900, 
      125.700,  124.500,  123.100,  121.600,  120.000,  118.300,  116.700,  114.900,  113.300,  111.500, 
      109.800,  108.600,  106.900,  105.700,  104.600,  103.800,  101.900,  100.500,   99.400,   98.600, 
      98.500,   98.000,   97.300,   96.300,   95.800,   96.200,   96.400,   96.600,   96.200,   96.300, 
      95.700,   94.900,   94.100,   94.200,   93.100,   92.500,   92.200,   91.600,   91.400,   90.500, 
      90.900,   91.100,   90.700,   90.000,   89.700,   88.900,   88.000,   87.200,   86.500,   85.500, 
      84.600,   84.100,   83.800,   82.900,   82.000,   81.400,   80.800,   80.200,   79.600,   78.900, 
      78.200,   77.600,   77.200,   76.300,   75.400,   74.900,   74.600,   74.500,   74.400,   74.200, 
      74.100,   73.900,   73.800,   73.600,   73.500,   73.100,   72.700,   72.400,   72.100,   71.500, 
      70.900,   70.400,   69.900,   69.400,   68.900,   68.300,   67.700,   67.000,   66.500,   65.800, 
      65.200,   64.600,   64.000,   63.400,   62.800,   62.200, ],
    [ 142.900,  137.900,  134.100,  132.600,  131.700,  130.800,  129.900,  129.000,  128.000,  126.900, 
      125.700,  124.500,  123.100,  121.600,  120.000,  118.400,  116.700,  115.000,  113.300,  112.000, 
      110.300,  108.600,  107.100,  105.700,  104.800,  103.800,  102.100,  100.700,   99.600,   98.800, 
      98.500,   98.000,   97.400,   96.400,   96.500,   96.500,   96.800,   97.100,   96.900,   96.300, 
      95.700,   94.900,   94.900,   94.200,   93.600,   92.600,   92.200,   91.700,   91.500,   90.700, 
      91.300,   91.700,   91.000,   90.300,   90.000,   89.200,   88.300,   87.500,   86.800,   85.900, 
      85.000,   84.500,   84.200,   83.300,   82.400,   81.700,   81.200,   80.600,   80.000,   79.300, 
      78.600,   78.100,   77.700,   76.700,   75.800,   75.300,   75.100,   74.900,   74.900,   74.700, 
      74.600,   74.500,   74.400,   74.200,   74.100,   73.700,   73.300,   73.000,   72.700,   72.100, 
      71.500,   71.000,   70.500,   70.000,   69.500,   68.900,   68.200,   67.600,   67.000,   66.400, 
      65.800,   65.100,   64.600,   64.000,   63.400,   62.700, ],
    [ 142.900,  137.900,  134.100,  132.600,  131.700,  130.800,  129.900,  129.000,  128.000,  126.900, 
      125.700,  124.500,  123.100,  121.700,  120.100,  118.400,  116.700,  115.400,  113.800,  112.000, 
      110.400,  108.700,  107.300,  105.900,  104.800,  104.000,  102.200,  100.900,   99.700,   99.000, 
      98.700,   98.300,   97.400,   96.800,   96.700,   96.900,   96.800,   97.400,   97.100,   96.300, 
      95.700,   95.400,   95.000,   94.300,   93.600,   92.700,   92.300,   91.800,   91.500,   90.800, 
      91.700,   91.900,   91.400,   90.400,   90.200,   89.500,   88.600,   87.800,   87.200,   86.200, 
      85.300,   84.900,   84.500,   83.700,   82.800,   82.100,   81.600,   81.000,   80.400,   79.700, 
      79.000,   78.500,   78.100,   77.200,   76.200,   75.800,   75.600,   75.400,   75.400,   75.200, 
      75.100,   75.000,   74.900,   74.800,   74.600,   74.300,   73.900,   73.500,   73.200,   72.700, 
      72.100,   71.600,   71.100,   70.600,   70.100,   69.500,   68.800,   68.200,   67.600,   67.000, 
      66.300,   65.700,   65.100,   64.500,   63.900,   63.300, ],
    [ 142.900,  137.900,  134.100,  132.600,  131.700,  130.800,  129.900,  129.000,  128.000,  126.900, 
      125.700,  124.500,  123.200,  121.700,  120.400,  118.700,  117.100,  115.400,  113.900,  112.100, 
      110.400,  108.900,  107.300,  105.900,  104.900,  104.100,  102.200,  101.000,   99.800,   99.200, 
      98.900,   98.300,   97.500,   96.900,   97.000,   97.400,   97.300,   97.500,   97.100,   96.900, 
      96.100,   95.400,   95.000,   94.600,   94.000,   93.300,   92.900,   91.900,   91.600,   91.900, 
      92.100,   92.200,   91.700,   90.600,   90.400,   89.700,   88.900,   88.100,   87.500,   86.500, 
      85.600,   85.200,   84.900,   84.000,   83.200,   82.500,   82.000,   81.400,   80.800,   80.100, 
      79.400,   78.900,   78.500,   77.600,   76.700,   76.200,   76.000,   75.900,   75.900,   75.800, 
      75.700,   75.600,   75.400,   75.300,   75.200,   74.900,   74.400,   74.100,   73.800,   73.300, 
      72.600,   72.100,   71.600,   71.200,   70.700,   70.100,   69.400,   68.800,   68.200,   67.500, 
      66.900,   66.300,   65.700,   65.100,   64.500,   63.800, ],
    [ 143.000,  137.900,  134.100,  132.600,  131.700,  130.800,  129.900,  129.000,  128.000,  126.900, 
      125.700,  124.800,  123.400,  122.000,  120.400,  118.800,  117.200,  115.500,  113.900,  112.300, 
      110.600,  108.900,  107.400,  106.000,  105.000,  104.100,  102.400,  101.200,  100.000,   99.200, 
      99.000,   98.500,   97.500,   97.200,   97.400,   97.500,   97.400,   97.600,   97.600,   96.900, 
      96.200,   95.700,   95.500,   94.700,   94.000,   93.700,   92.900,   92.100,   91.800,   92.400, 
      92.100,   92.300,   92.100,   90.800,   90.500,   90.000,   89.200,   88.400,   87.800,   86.800, 
      85.900,   85.600,   85.300,   84.400,   83.500,   82.900,   82.300,   81.800,   81.200,   80.500, 
      79.800,   79.300,   79.000,   78.000,   77.100,   76.600,   76.500,   76.400,   76.400,   76.300, 
      76.200,   76.100,   76.000,   75.900,   75.800,   75.400,   75.000,   74.700,   74.400,   73.800, 
      73.200,   72.700,   72.200,   71.700,   71.300,   70.600,   69.900,   69.300,   68.700,   68.100, 
      67.400,   66.800,   66.200,   65.600,   65.000,   64.400, ],
    [ 143.000,  137.900,  134.100,  132.600,  131.700,  130.900,  129.900,  129.200,  128.300,  127.200, 
      126.000,  124.800,  123.500,  122.100,  120.500,  118.800,  117.300,  115.600,  114.000,  112.300, 
      110.600,  109.000,  107.400,  106.100,  105.100,  104.200,  102.500,  101.200,  100.200,   99.500, 
      99.200,   98.600,   97.600,   97.500,   97.600,   97.800,   98.300,   98.200,   97.700,   97.300, 
      96.600,   95.700,   95.500,   95.000,   94.400,   93.800,   93.000,   92.200,   91.900,   92.400, 
      92.100,   92.300,   92.100,   90.900,   90.700,   90.500,   89.500,   88.700,   88.100,   87.100, 
      86.300,   85.900,   85.600,   84.800,   83.900,   83.300,   82.700,   82.200,   81.600,   80.900, 
      80.200,   79.700,   79.400,   78.500,   77.500,   77.000,   76.900,   76.800,   76.800,   76.800, 
      76.700,   76.600,   76.500,   76.400,   76.300,   76.000,   75.600,   75.200,   75.000,   74.400, 
      73.800,   73.300,   72.800,   72.300,   71.800,   71.200,   70.500,   69.900,   69.300,   68.600, 
      68.000,   67.400,   66.800,   66.100,   65.600,   64.900, ],
    [ 143.000,  137.900,  134.100,  132.800,  132.000,  131.100,  130.200,  129.200,  128.300,  127.200, 
      126.100,  124.900,  123.500,  122.100,  120.600,  118.900,  117.300,  115.600,  114.100,  112.300, 
      110.600,  109.100,  107.500,  106.200,  105.100,  104.300,  102.600,  101.200,  100.300,   99.700, 
      99.200,   98.600,   98.000,   97.900,   98.000,   97.800,   98.600,   98.300,   97.700,   97.300, 
      96.600,   96.000,   96.000,   95.300,   94.500,   94.200,   93.100,   92.400,   92.700,   92.400, 
      92.200,   92.400,   92.100,   92.000,   90.900,   91.100,   89.700,   89.000,   88.400,   87.500, 
      86.600,   86.200,   86.000,   85.100,   84.200,   83.600,   83.100,   82.500,   82.000,   81.300, 
      80.600,   80.200,   79.800,   78.900,   77.900,   77.500,   77.300,   77.300,   77.300,   77.200, 
      77.200,   77.100,   77.100,   77.000,   76.900,   76.500,   76.100,   75.800,   75.500,   75.000, 
      74.300,   73.800,   73.300,   72.900,   72.400,   71.800,   71.000,   70.400,   69.800,   69.200, 
      68.500,   67.900,   67.300,   66.700,   66.100,   65.400, ],
    [ 143.100,  138.100,  134.300,  132.800,  132.000,  131.100,  130.200,  129.300,  128.300,  127.200, 
      126.100,  124.900,  123.600,  122.100,  120.600,  118.900,  117.300,  115.600,  114.100,  112.400, 
      110.700,  109.200,  107.600,  106.300,  105.200,  104.500,  102.800,  101.400,  100.400,   99.700, 
      99.500,   98.700,   98.300,   98.100,   98.400,   98.300,   98.600,   98.300,   98.100,   97.600, 
      97.000,   96.600,   96.000,   95.300,   94.800,   94.200,   93.700,   92.500,   92.700,   93.000, 
      93.100,   93.100,   92.200,   92.400,   91.800,   91.500,   90.000,   89.300,   88.700,   87.800, 
      86.900,   86.600,   86.400,   85.500,   84.600,   84.000,   83.500,   82.900,   82.400,   81.700, 
      81.000,   80.600,   80.200,   79.300,   78.300,   77.900,   77.800,   77.700,   77.800,   77.700, 
      77.700,   77.600,   77.600,   77.500,   77.400,   77.100,   76.700,   76.400,   76.100,   75.500, 
      74.900,   74.400,   73.900,   73.400,   72.900,   72.300,   71.600,   71.000,   70.400,   69.700, 
      69.000,   68.400,   67.800,   67.200,   66.600,   66.000, ],
    [ 143.200,  138.100,  134.400,  132.900,  132.000,  131.100,  130.300,  129.400,  128.400,  127.300, 
      126.100,  125.000,  123.600,  122.200,  120.600,  119.000,  117.400,  115.700,  114.100,  112.500, 
      110.800,  109.200,  107.700,  106.400,  105.400,  104.600,  102.800,  101.700,  100.600,   99.900, 
      99.500,   98.700,   98.300,   98.500,   98.500,   99.000,   98.600,   98.800,   98.200,   97.700, 
      97.000,   96.600,   96.300,   95.600,   94.900,   94.300,   93.700,   92.700,   93.200,   93.000, 
      93.200,   93.500,   93.100,   92.800,   91.900,   91.500,   90.500,   89.500,   89.000,   88.000, 
      87.200,   86.900,   86.700,   85.800,   84.900,   84.300,   83.800,   83.300,   82.800,   82.100, 
      81.400,   81.000,   80.600,   79.700,   78.700,   78.300,   78.200,   78.200,   78.300,   78.200, 
      78.200,   78.200,   78.100,   78.000,   78.000,   77.600,   77.200,   76.900,   76.700,   76.100, 
      75.400,   74.900,   74.400,   74.000,   73.500,   72.900,   72.100,   71.500,   70.900,   70.200, 
      69.600,   68.900,   68.300,   67.700,   67.100,   66.500, ],
    [ 143.200,  138.100,  134.400,  132.900,  132.100,  131.200,  130.300,  129.400,  128.400,  127.400, 
      126.200,  125.000,  123.600,  122.200,  120.700,  119.000,  117.400,  115.800,  114.200,  112.500, 
      110.900,  109.300,  107.700,  106.500,  105.500,  104.700,  103.000,  101.700,  100.700,   99.900, 
      99.500,   98.800,   98.800,   98.700,   98.800,   99.400,   99.300,   98.800,   98.500,   98.000, 
      97.300,   97.000,   96.300,   95.900,   94.900,   94.700,   93.800,   92.800,   93.300,   93.100, 
      93.200,   93.900,   93.500,   92.800,   91.900,   91.600,   91.000,   89.800,   89.300,   88.300, 
      87.500,   87.200,   87.100,   86.100,   85.300,   84.700,   84.200,   83.700,   83.100,   82.500, 
      81.800,   81.400,   81.000,   80.100,   79.100,   78.700,   78.600,   78.600,   78.700,   78.700, 
      78.700,   78.700,   78.600,   78.600,   78.500,   78.200,   77.800,   77.500,   77.200,   76.600, 
      76.000,   75.500,   75.000,   74.500,   74.000,   73.400,   72.700,   72.000,   71.400,   70.800, 
      70.100,   69.500,   68.900,   68.300,   67.700,   67.000, ],
    [ 143.200,  138.200,  134.400,  132.900,  132.100,  131.200,  130.300,  129.400,  128.400,  127.400, 
      126.200,  125.000,  123.700,  122.200,  120.700,  119.100,  117.500,  115.800,  114.300,  112.600, 
      111.000,  109.300,  107.800,  106.600,  105.600,  104.700,  103.100,  101.700,  100.800,  100.200, 
      99.600,   99.100,   99.000,   99.400,   99.200,   99.500,   99.300,   99.200,   98.500,   98.300, 
      97.500,   97.100,   96.600,   96.000,   95.300,   94.800,   94.300,   93.500,   93.700,   93.800, 
      94.100,   94.300,   93.800,   92.800,   92.000,   91.700,   91.500,   90.000,   89.500,   88.600, 
      87.800,   87.600,   87.400,   86.500,   85.600,   85.000,   84.600,   84.000,   83.500,   82.900, 
      82.200,   81.700,   81.500,   80.500,   79.500,   79.100,   79.100,   79.100,   79.200,   79.200, 
      79.200,   79.200,   79.200,   79.100,   79.100,   78.700,   78.300,   78.000,   77.800,   77.200, 
      76.500,   76.000,   75.500,   75.000,   74.600,   73.900,   73.200,   72.600,   72.000,   71.300, 
      70.600,   70.000,   69.400,   68.800,   68.200,   67.500, ],
    [ 143.200,  138.200,  134.400,  133.000,  132.100,  131.200,  130.300,  129.400,  128.500,  127.400, 
      126.200,  125.100,  123.700,  122.300,  120.700,  119.100,  117.500,  115.800,  114.300,  112.600, 
      111.000,  109.400,  107.900,  106.600,  105.600,  104.800,  103.200,  102.000,  101.000,  100.200, 
      99.700,   99.400,   99.500,   99.400,   99.800,   99.500,   99.700,   99.200,   98.800,   98.400, 
      98.000,   97.400,   96.800,   96.200,   95.600,   95.100,   94.300,   93.600,   93.800,   93.900, 
      94.200,   94.600,   93.800,   93.500,   93.000,   92.400,   91.500,   90.700,   89.800,   88.900, 
      88.100,   87.900,   87.700,   86.800,   85.900,   85.400,   84.900,   84.400,   83.900,   83.300, 
      82.600,   82.100,   81.900,   80.900,   79.900,   79.500,   79.500,   79.500,   79.700,   79.700, 
      79.700,   79.700,   79.700,   79.600,   79.600,   79.300,   78.800,   78.500,   78.300,   77.700, 
      77.000,   76.500,   76.000,   75.600,   75.100,   74.500,   73.700,   73.100,   72.500,   71.800, 
      71.100,   70.500,   69.900,   69.300,   68.700,   68.000, ],
    [ 143.200,  138.200,  134.500,  133.000,  132.100,  131.300,  130.400,  129.500,  128.500,  127.400, 
      126.300,  125.100,  123.800,  122.300,  120.800,  119.200,  117.600,  115.900,  114.400,  112.700, 
      111.100,  109.500,  107.900,  106.600,  105.700,  105.000,  103.200,  102.100,  101.000,  100.200, 
      99.700,   99.800,   99.600,   99.700,  100.200,  100.100,   99.700,   99.500,   99.100,   98.600, 
      98.000,   97.600,   96.800,   96.500,   95.900,   95.100,   94.400,   94.100,   94.200,   94.700, 
      94.900,   95.000,   93.800,   93.600,   93.000,   92.400,   91.600,   91.200,   90.000,   89.100, 
      88.400,   88.200,   88.100,   87.100,   86.200,   85.700,   85.300,   84.700,   84.200,   83.600, 
      82.900,   82.500,   82.200,   81.200,   80.300,   79.900,   79.900,   80.000,   80.100,   80.100, 
      80.200,   80.200,   80.200,   80.200,   80.100,   79.800,   79.400,   79.100,   78.900,   78.300, 
      77.600,   77.100,   76.600,   76.100,   75.600,   75.000,   74.200,   73.600,   73.000,   72.300, 
      71.600,   71.000,   70.400,   69.800,   69.200,   68.500, ],
    [ 143.200,  138.200,  134.500,  133.000,  132.200,  131.300,  130.400,  129.500,  128.500,  127.500, 
      126.300,  125.100,  123.800,  122.400,  120.800,  119.200,  117.600,  116.000,  114.500,  112.800, 
      111.100,  109.500,  108.100,  106.900,  105.900,  105.100,  103.500,  102.200,  101.200,  100.300, 
      100.000,  100.000,  100.300,  100.100,  100.300,  100.100,  100.100,   99.500,   99.100,   98.600, 
      98.300,   97.700,   97.100,   96.500,   95.900,   95.500,   94.500,   94.200,   94.800,   94.800, 
      94.900,   95.100,   94.400,   93.700,   93.100,   92.500,   91.700,   91.600,   90.600,   89.400, 
      88.700,   88.500,   88.400,   87.400,   86.600,   86.000,   85.600,   85.100,   84.600,   84.000, 
      83.300,   82.900,   82.600,   81.600,   80.600,   80.300,   80.300,   80.400,   80.600,   80.600, 
      80.700,   80.700,   80.700,   80.700,   80.700,   80.300,   79.900,   79.600,   79.400,   78.800, 
      78.100,   77.600,   77.100,   76.600,   76.200,   75.500,   74.800,   74.100,   73.500,   72.800, 
      72.100,   71.500,   70.900,   70.300,   69.700,   69.000, ],
    [ 143.300,  138.300,  134.500,  133.100,  132.200,  131.300,  130.400,  129.500,  128.600,  127.500, 
      126.300,  125.200,  123.900,  122.400,  120.900,  119.300,  117.700,  116.000,  114.500,  112.800, 
      111.200,  109.600,  108.200,  107.000,  106.000,  105.100,  103.500,  102.300,  101.200,  100.300, 
      100.200,  100.400,  100.300,  100.800,  100.300,  100.400,  100.100,   99.800,   99.400,   99.200, 
      98.700,   97.900,   97.300,   96.600,   96.000,   95.600,   94.600,   94.300,   94.800,   95.300, 
      94.900,   95.100,   94.500,   94.300,   93.500,   93.000,   92.400,   92.100,   91.100,   89.600, 
      88.900,   88.800,   88.700,   87.700,   86.900,   86.400,   86.000,   85.400,   85.000,   84.300, 
      83.700,   83.300,   83.000,   82.000,   81.000,   80.700,   80.700,   80.800,   81.000,   81.100, 
      81.200,   81.200,   81.200,   81.200,   81.200,   80.900,   80.400,   80.200,   79.900,   79.300, 
      78.600,   78.100,   77.600,   77.200,   76.700,   76.000,   75.300,   74.600,   74.000,   73.300, 
      72.600,   72.000,   71.400,   70.800,   70.200,   69.500, ],
    [ 143.300,  138.300,  134.500,  133.100,  132.200,  131.400,  130.500,  129.600,  128.600,  127.600, 
      126.400,  125.200,  123.900,  122.500,  120.900,  119.300,  117.700,  116.000,  114.500,  112.900, 
      111.300,  109.800,  108.300,  107.000,  106.000,  105.300,  103.600,  102.400,  101.200,  100.400, 
      100.600,  100.600,  100.600,  101.100,  100.800,  100.500,  100.400,  100.100,   99.400,   99.200, 
      98.700,   98.100,   97.500,   96.900,   96.300,   95.600,   94.700,   94.700,   95.200,   95.300, 
      95.800,   95.700,   95.100,   94.400,   93.600,   93.100,   92.400,   92.500,   91.500,   89.800, 
      89.200,   89.100,   89.100,   88.000,   87.200,   86.700,   86.300,   85.800,   85.300,   84.700, 
      84.000,   83.600,   83.400,   82.400,   81.400,   81.000,   81.100,   81.300,   81.500,   81.600, 
      81.600,   81.700,   81.700,   81.700,   81.800,   81.400,   80.900,   80.700,   80.500,   79.800, 
      79.100,   78.600,   78.100,   77.700,   77.200,   76.600,   75.800,   75.100,   74.500,   73.800, 
      73.100,   72.500,   71.900,   71.300,   70.700,   70.000, ],
    [ 143.300,  138.300,  134.600,  133.100,  132.300,  131.400,  130.500,  129.600,  128.700,  127.600, 
      126.400,  125.300,  123.900,  122.500,  120.900,  119.300,  117.800,  116.200,  114.600,  113.000, 
      111.400,  109.800,  108.300,  107.100,  106.200,  105.400,  103.700,  102.500,  101.300,  100.900, 
      100.900,  101.200,  100.900,  101.100,  100.800,  100.800,  100.400,  100.100,   99.600,   99.500, 
      98.900,   98.300,   97.800,   97.100,   96.600,   96.000,   94.800,   95.100,   95.700,   96.000, 
      96.200,   95.900,   95.200,   94.400,   94.000,   93.500,   92.500,   92.500,   91.500,   90.000, 
      89.500,   89.400,   89.400,   88.300,   87.500,   87.000,   86.600,   86.100,   85.700,   85.000, 
      84.400,   84.000,   83.800,   82.700,   81.700,   81.400,   81.500,   81.700,   81.900,   82.000, 
      82.100,   82.200,   82.200,   82.300,   82.300,   81.900,   81.500,   81.200,   81.000,   80.400, 
      79.600,   79.100,   78.700,   78.200,   77.700,   77.100,   76.300,   75.600,   75.000,   74.300, 
      73.600,   73.000,   72.400,   71.700,   71.100,   70.400, ],
    [ 143.300,  138.300,  134.600,  133.100,  132.300,  131.400,  130.500,  129.600,  128.700,  127.600, 
      126.400,  125.300,  123.900,  122.500,  121.000,  119.400,  117.800,  116.200,  114.700,  113.000, 
      111.400,  109.800,  108.500,  107.200,  106.200,  105.400,  103.900,  102.500,  101.400,  100.900, 
      101.200,  101.400,  101.700,  101.600,  101.200,  100.800,  100.600,  100.300,  100.200,   99.800, 
      99.000,   98.300,   97.800,   97.400,   96.600,   96.000,   95.300,   95.400,   95.700,   96.500, 
      96.600,   96.000,   95.300,   94.700,   94.000,   93.600,   93.000,   92.600,   91.600,   90.300, 
      89.700,   89.600,   89.700,   88.600,   87.800,   87.300,   87.000,   86.500,   86.000,   85.400, 
      84.700,   84.400,   84.200,   83.100,   82.100,   81.800,   81.900,   82.100,   82.400,   82.500, 
      82.600,   82.700,   82.800,   82.800,   82.800,   82.500,   82.000,   81.700,   81.500,   80.900, 
      80.200,   79.600,   79.200,   78.700,   78.200,   77.600,   76.800,   76.100,   75.500,   74.800, 
      74.100,   73.500,   72.900,   72.200,   71.600,   70.900, ],
    [ 143.400,  138.400,  134.600,  133.200,  132.300,  131.400,  130.500,  129.600,  128.700,  127.600, 
      126.500,  125.400,  124.000,  122.600,  121.100,  119.500,  117.900,  116.200,  114.700,  113.000, 
      111.500,  110.000,  108.500,  107.200,  106.300,  105.500,  103.900,  102.500,  101.600,  101.400, 
      101.600,  101.800,  101.800,  101.600,  101.200,  101.100,  100.900,  100.500,  100.200,   99.800, 
      99.100,   98.500,   97.800,   97.400,   96.900,   96.100,   95.400,   95.700,   96.300,   96.500, 
      96.700,   96.000,   95.700,   94.800,   94.400,   94.000,   93.100,   92.600,   91.700,   90.500, 
      90.000,   89.800,   90.000,   88.900,   88.100,   87.600,   87.300,   86.800,   86.400,   85.700, 
      85.100,   84.700,   84.500,   83.400,   82.400,   82.100,   82.300,   82.500,   82.800,   82.900, 
      83.100,   83.200,   83.300,   83.300,   83.300,   83.000,   82.500,   82.200,   82.100,   81.400, 
      80.700,   80.100,   79.700,   79.200,   78.700,   78.100,   77.300,   76.600,   76.000,   75.300, 
      74.600,   73.900,   73.300,   72.700,   72.100,   71.400, ],
    [ 143.400,  138.400,  134.600,  133.200,  132.300,  131.400,  130.600,  129.700,  128.700,  127.700, 
      126.500,  125.400,  124.100,  122.600,  121.100,  119.500,  117.900,  116.300,  114.900,  113.200, 
      111.600,  110.000,  108.500,  107.400,  106.400,  105.700,  103.900,  102.600,  101.800,  101.800, 
      101.900,  102.400,  101.800,  101.900,  101.500,  101.300,  101.100,  100.500,  100.400,  100.000, 
      99.200,   98.700,   98.300,   97.700,   97.000,   96.200,   95.800,   96.200,   96.300,   96.500, 
      96.700,   96.400,   95.800,   95.300,   94.400,   94.000,   93.500,   93.400,   92.400,   90.700, 
      90.500,   91.200,   90.500,   89.100,   88.400,   87.900,   87.600,   87.100,   86.700,   86.100, 
      85.400,   85.100,   84.900,   83.800,   82.800,   82.500,   82.700,   82.900,   83.300,   83.400, 
      83.600,   83.700,   83.800,   83.800,   83.900,   83.500,   83.000,   82.800,   82.600,   81.900, 
      81.200,   80.600,   80.200,   79.700,   79.200,   78.600,   77.700,   77.100,   76.500,   75.800, 
      75.100,   74.400,   73.800,   73.200,   72.600,   71.900, ],
    [ 143.400,  138.400,  134.700,  133.200,  132.400,  131.500,  130.600,  129.800,  128.800,  127.700, 
      126.600,  125.400,  124.100,  122.700,  121.100,  119.600,  118.000,  116.400,  114.900,  113.200, 
      111.600,  110.100,  108.600,  107.400,  106.500,  105.700,  104.000,  102.800,  102.200,  101.900, 
      102.200,  102.600,  102.300,  101.900,  101.800,  101.300,  101.200,  101.000,  100.700,  100.100, 
      99.400,   98.900,   98.500,   97.900,   97.300,   96.200,   96.200,   96.400,   96.700,   97.400, 
      97.300,   96.400,   96.200,   95.300,   94.800,   94.400,   93.600,   93.500,   92.400,   90.900, 
      91.000,   91.700,   90.800,   89.400,   88.700,   88.200,   87.900,   87.500,   87.000,   86.400, 
      85.700,   85.400,   85.300,   84.100,   83.100,   82.900,   83.100,   83.400,   83.700,   83.900, 
      84.100,   84.200,   84.300,   84.300,   84.400,   84.000,   83.500,   83.300,   83.100,   82.400, 
      81.600,   81.100,   80.700,   80.200,   79.700,   79.000,   78.200,   77.500,   76.900,   76.200, 
      75.500,   74.900,   74.300,   73.600,   73.000,   72.300, ],
    [ 143.400,  138.400,  134.700,  133.300,  132.400,  131.500,  130.700,  129.800,  128.800,  127.700, 
      126.600,  125.400,  124.100,  122.800,  121.200,  119.600,  118.000,  116.400,  114.900,  113.300, 
      111.700,  110.100,  108.700,  107.500,  106.500,  105.700,  104.000,  103.000,  102.700,  102.500, 
      102.500,  102.600,  102.300,  102.300,  101.800,  101.600,  101.300,  101.000,  100.700,  100.100, 
      99.600,   98.900,   98.500,   97.900,   97.300,   96.300,   96.400,   96.900,   97.300,   97.800, 
      97.400,   97.000,   96.200,   95.600,   95.100,   94.400,   94.000,   93.500,   92.500,   91.100, 
      91.500,   91.900,   91.200,   89.600,   88.900,   88.500,   88.200,   87.800,   87.400,   86.700, 
      86.100,   85.800,   85.600,   84.400,   83.400,   83.200,   83.500,   83.800,   84.200,   84.300, 
      84.500,   84.700,   84.800,   84.800,   84.900,   84.500,   84.000,   83.800,   83.600,   82.900, 
      82.100,   81.600,   81.200,   80.700,   80.200,   79.500,   78.700,   78.000,   77.400,   76.700, 
      76.000,   75.300,   74.700,   74.100,   73.500,   72.800, ],
    [ 143.400,  138.500,  134.700,  133.300,  132.400,  131.600,  130.700,  129.800,  128.800,  127.800, 
      126.700,  125.500,  124.200,  122.800,  121.200,  119.600,  118.100,  116.500,  115.000,  113.300, 
      111.800,  110.200,  108.700,  107.500,  106.500,  105.700,  104.100,  103.300,  102.800,  102.700, 
      103.300,  103.100,  102.600,  102.300,  102.000,  101.800,  101.500,  101.300,  100.900,  100.300, 
      99.800,   99.200,   98.500,   98.200,   97.300,   96.800,   96.700,   97.400,   97.700,   97.900, 
      97.400,   97.000,   96.500,   95.900,   95.100,   94.800,   94.000,   94.200,   92.600,   91.300, 
      92.000,   92.200,   91.500,   89.800,   89.200,   88.800,   88.500,   88.100,   87.700,   87.100, 
      86.400,   86.100,   86.000,   84.800,   83.700,   83.600,   83.900,   84.200,   84.600,   84.800, 
      85.000,   85.200,   85.300,   85.400,   85.400,   85.000,   84.500,   84.300,   84.200,   83.400, 
      82.600,   82.100,   81.600,   81.200,   80.700,   80.000,   79.200,   78.500,   77.900,   77.200, 
      76.500,   75.800,   75.200,   74.600,   73.900,   73.200, ],
    [ 143.400,  138.500,  134.700,  133.300,  132.500,  131.600,  130.700,  129.800,  128.900,  127.800, 
      126.700,  125.500,  124.200,  122.800,  121.300,  119.700,  118.100,  116.500,  115.000,  113.400, 
      111.800,  110.300,  108.800,  107.500,  106.600,  105.800,  104.400,  103.500,  103.300,  103.000, 
      103.300,  103.100,  102.900,  102.500,  102.200,  101.800,  101.800,  101.500,  101.000,  100.500, 
      100.000,   99.400,   99.000,   98.200,   97.400,   97.100,   97.300,   97.400,   97.700,   97.900, 
      97.700,   97.400,   96.500,   95.900,   95.500,   95.100,   94.400,   94.200,   93.200,   92.200, 
      92.100,   92.500,   91.900,   90.200,   89.400,   89.100,   88.800,   88.400,   88.000,   87.400, 
      86.700,   86.500,   86.300,   85.100,   84.100,   83.900,   84.300,   84.600,   85.100,   85.300, 
      85.500,   85.600,   85.800,   85.900,   86.000,   85.500,   85.000,   84.800,   84.700,   83.900, 
      83.100,   82.600,   82.100,   81.700,   81.200,   80.500,   79.600,   79.000,   78.400,   77.600, 
      76.900,   76.300,   75.600,   75.000,   74.400,   73.700, ],
    [ 143.500,  138.500,  134.800,  133.400,  132.500,  131.600,  130.700,  129.800,  128.900,  127.800, 
      126.700,  125.600,  124.200,  122.800,  121.300,  119.800,  118.200,  116.600,  115.100,  113.400, 
      111.800,  110.300,  108.800,  107.600,  106.600,  106.000,  104.700,  103.800,  103.500,  103.700, 
      103.700,  103.400,  102.900,  102.700,  102.400,  102.000,  102.000,  101.500,  101.200,  100.600, 
      100.000,   99.600,   99.000,   98.500,   97.500,   97.100,   97.600,   97.800,   98.600,   98.500, 
      97.700,   97.400,   96.900,   96.200,   95.800,   95.400,   94.800,   94.600,   93.200,   92.200, 
      92.100,   92.800,   92.000,   90.800,   89.700,   89.400,   89.100,   88.700,   88.400,   87.700, 
      87.100,   86.800,   86.700,   85.400,   84.400,   84.300,   84.700,   85.000,   85.500,   85.700, 
      86.000,   86.100,   86.300,   86.400,   86.500,   86.000,   85.500,   85.300,   85.200,   84.400, 
      83.600,   83.100,   82.600,   82.200,   81.700,   81.000,   80.100,   79.400,   78.800,   78.100, 
      77.400,   76.700,   76.100,   75.500,   74.800,   74.100, ],
    [ 143.500,  138.500,  134.800,  133.400,  132.500,  131.600,  130.800,  129.900,  128.900,  127.900, 
      126.700,  125.600,  124.300,  122.900,  121.400,  119.800,  118.200,  116.600,  115.100,  113.400, 
      111.900,  110.300,  108.800,  107.600,  106.800,  106.100,  104.900,  104.400,  103.800,  103.800, 
      103.700,  103.400,  103.200,  102.900,  102.500,  102.400,  102.200,  101.700,  101.300,  100.800, 
      100.200,   99.600,   99.000,   98.500,   97.600,   97.700,   97.900,   98.600,   98.900,   98.500, 
      98.200,   97.700,   96.900,   96.500,   95.800,   95.500,   94.800,   94.600,   93.700,   92.300, 
      92.200,   93.100,   92.100,   91.400,   90.000,   89.600,   89.400,   89.000,   88.700,   88.000, 
      87.400,   87.100,   87.000,   85.700,   84.700,   84.600,   85.000,   85.500,   86.000,   86.200, 
      86.500,   86.600,   86.800,   86.900,   87.000,   86.500,   86.000,   85.800,   85.700,   84.800, 
      84.000,   83.600,   83.100,   82.600,   82.200,   81.400,   80.600,   79.900,   79.300,   78.500, 
      77.800,   77.200,   76.500,   75.900,   75.300,   74.600, ],
    [ 143.500,  138.600,  134.800,  133.400,  132.500,  131.700,  130.800,  129.900,  129.000,  127.900, 
      126.800,  125.600,  124.300,  122.900,  121.400,  119.800,  118.200,  116.600,  115.100,  113.500, 
      111.900,  110.300,  109.000,  107.800,  106.900,  106.400,  105.100,  104.600,  104.500,  104.200, 
      104.000,  103.700,  103.400,  102.900,  102.600,  102.600,  102.300,  101.800,  101.500,  101.000, 
      100.400,  100.000,   99.200,   98.600,   98.000,   97.900,   98.400,   98.600,   98.900,   98.800, 
      98.200,   97.800,   97.200,   96.500,   96.100,   95.800,   95.100,   94.900,   93.700,   92.400, 
      92.800,   93.500,   92.200,   91.500,   90.400,   89.800,   89.600,   89.300,   89.000,   88.300, 
      87.700,   87.500,   87.400,   86.000,   85.000,   85.000,   85.400,   85.900,   86.400,   86.600, 
      87.000,   87.100,   87.300,   87.400,   87.500,   87.000,   86.500,   86.300,   86.200,   85.300, 
      84.500,   84.000,   83.600,   83.100,   82.600,   81.900,   81.000,   80.300,   79.700,   79.000, 
      78.200,   77.600,   77.000,   76.300,   75.700,   75.000, ],
    [ 143.500,  138.600,  134.900,  133.400,  132.600,  131.700,  130.800,  129.900,  129.000,  127.900, 
      126.800,  125.600,  124.300,  122.900,  121.400,  119.800,  118.200,  116.600,  115.100,  113.500, 
      111.900,  110.500,  109.100,  107.900,  107.300,  106.700,  105.600,  104.800,  104.500,  104.200, 
      104.000,  103.900,  103.400,  103.100,  103.000,  102.600,  102.400,  102.000,  101.600,  101.000, 
      100.500,  100.000,   99.500,   98.600,   98.200,   98.400,   98.700,   99.400,   99.500,   98.800, 
      98.500,   98.000,   97.400,   96.700,   96.300,   96.100,   95.400,   95.000,   94.200,   92.600, 
      92.800,   93.800,   92.700,   91.500,   91.000,   90.200,   89.800,   89.500,   89.400,   88.600, 
      88.000,   87.800,   87.700,   86.300,   85.300,   85.300,   85.800,   86.300,   86.900,   87.100, 
      87.400,   87.600,   87.800,   87.900,   88.100,   87.500,   87.000,   86.800,   86.700,   85.800, 
      85.000,   84.500,   84.000,   83.600,   83.100,   82.400,   81.500,   80.800,   80.200,   79.400, 
      78.700,   78.000,   77.400,   76.800,   76.200,   75.400, ],
    [ 143.500,  138.600,  134.900,  133.400,  132.600,  131.700,  130.900,  130.000,  129.000,  127.900, 
      126.800,  125.700,  124.300,  122.900,  121.400,  119.800,  118.300,  116.600,  115.200,  113.600, 
      112.100,  110.600,  109.300,  108.200,  107.500,  106.800,  105.800,  105.500,  104.900,  104.400, 
      104.300,  104.100,  103.600,  103.300,  103.000,  102.800,  102.500,  102.100,  101.800,  101.300, 
      100.600,  100.000,   99.500,   98.700,   98.300,   98.600,   99.500,   99.700,   99.500,   99.200, 
      98.900,   98.300,   97.600,   97.000,   96.600,   96.300,   95.500,   95.300,   94.200,   92.700, 
      93.600,   94.100,   92.800,   91.600,   91.400,   90.800,   90.200,   89.800,   89.700,   88.900, 
      88.400,   88.100,   88.100,   86.600,   85.600,   85.600,   86.200,   86.700,   87.300,   87.600, 
      87.900,   88.100,   88.300,   88.400,   88.600,   88.000,   87.400,   87.300,   87.200,   86.200, 
      85.400,   85.000,   84.500,   84.000,   83.600,   82.800,   81.900,   81.200,   80.600,   79.900, 
      79.100,   78.500,   77.800,   77.200,   76.600,   75.800, ],
    [ 143.600,  138.600,  134.900,  133.500,  132.600,  131.700,  130.900,  130.000,  129.000,  128.000, 
      126.800,  125.700,  124.400,  122.900,  121.400,  119.800,  118.400,  116.700,  115.300,  113.700, 
      112.300,  110.900,  109.600,  108.500,  107.800,  107.200,  106.100,  105.500,  104.900,  104.500, 
      104.400,  104.100,  103.800,  103.400,  103.200,  103.000,  102.700,  102.300,  101.800,  101.500, 
      100.900,  100.200,   99.500,   98.800,   98.900,   99.400,   99.500,   99.700,   99.500,   99.300, 
      98.900,   98.500,   97.800,   97.200,   96.800,   96.300,   96.400,   95.600,   94.200,   92.800, 
      93.600,   94.500,   92.800,   91.700,   91.500,   91.300,   90.800,   90.000,   90.000,   89.200, 
      88.700,   88.400,   88.400,   86.900,   85.900,   86.000,   86.600,   87.100,   87.800,   88.000, 
      88.400,   88.600,   88.800,   88.900,   89.100,   88.500,   87.900,   87.800,   87.700,   86.700, 
      85.900,   85.400,   85.000,   84.500,   84.000,   83.300,   82.300,   81.600,   81.000,   80.300, 
      79.500,   78.900,   78.300,   77.600,   77.000,   76.300, ],
    [ 143.600,  138.600,  134.900,  133.500,  132.600,  131.800,  130.900,  130.000,  129.000,  128.000, 
      126.800,  125.700,  124.400,  123.000,  121.500,  120.000,  118.500,  116.800,  115.500,  113.900, 
      112.600,  111.100,  109.900,  108.900,  108.000,  107.400,  106.700,  105.900,  105.200,  104.700, 
      104.600,  104.300,  103.900,  103.700,  103.400,  103.100,  102.800,  102.400,  101.800,  101.500, 
      100.900,  100.400,   99.600,   99.100,   99.400,   99.400,  100.200,  100.200,   99.800,   99.500, 
      99.100,   98.500,   97.800,   97.400,   96.900,   96.600,   96.400,   95.600,   94.700,   93.000, 
      93.700,   94.800,   93.600,   91.800,   91.500,   91.700,   91.200,   90.600,   90.400,   89.400, 
      89.000,   88.700,   88.700,   87.200,   86.200,   86.300,   87.000,   87.600,   88.300,   88.500, 
      88.900,   89.000,   89.400,   89.300,   89.600,   88.900,   88.400,   88.300,   88.200,   87.200, 
      86.300,   85.900,   85.400,   85.000,   84.500,   83.700,   82.800,   82.100,   81.500,   80.700, 
      80.000,   79.300,   78.700,   78.100,   77.400,   76.700, ],
    [ 143.600,  138.600,  134.900,  133.500,  132.600,  131.800,  130.900,  130.000,  129.000,  128.100, 
      126.900,  125.800,  124.500,  123.100,  121.600,  120.100,  118.600,  117.000,  115.700,  114.100, 
      112.600,  111.500,  110.100,  109.000,  108.300,  108.100,  106.700,  105.900,  105.400,  104.900, 
      104.600,  104.400,  104.000,  103.900,  103.600,  103.200,  102.900,  102.500,  102.100,  101.700, 
      101.100,  100.500,   99.700,   99.300,   99.600,  100.300,  100.500,  100.200,  100.100,   99.800, 
      99.400,   98.700,   98.100,   97.700,   97.100,   96.900,   96.800,   95.900,   95.000,   93.700, 
      94.600,   95.100,   93.600,   92.600,   91.600,   92.100,   91.600,   91.000,   90.700,   89.700, 
      89.300,   89.000,   89.000,   87.500,   86.500,   86.600,   87.400,   88.000,   88.800,   88.900, 
      89.400,   89.500,   90.000,   89.800,   90.000,   89.400,   88.800,   88.700,   88.700,   87.600, 
      86.700,   86.300,   85.900,   85.400,   85.000,   84.100,   83.200,   82.500,   81.900,   81.100, 
      80.400,   79.700,   79.100,   78.500,   77.900,   77.100, ],
    [ 143.600,  138.600,  134.900,  133.500,  132.600,  131.800,  131.000,  130.100,  129.100,  128.100, 
      127.000,  125.800,  124.600,  123.300,  121.700,  120.300,  118.800,  117.200,  115.900,  114.400, 
      112.900,  111.600,  110.400,  109.200,  108.900,  108.200,  107.000,  106.100,  105.400,  105.100, 
      104.800,  104.600,  104.300,  104.100,  103.700,  103.400,  103.100,  102.600,  102.300,  101.800, 
      101.200,  100.500,   99.700,   99.900,  100.200,  100.400,  100.500,  100.200,  100.100,   99.800, 
      99.400,   98.900,   98.300,   97.900,   97.300,   97.100,   96.800,   96.200,   95.100,   93.700, 
      94.600,   95.500,   93.700,   92.600,   92.300,   92.500,   92.000,   91.400,   91.100,   90.000, 
      89.600,   89.300,   89.400,   87.800,   86.800,   87.000,   87.800,   88.400,   89.300,   89.400, 
      90.000,   90.000,   90.400,   90.600,   90.500,   89.800,   89.300,   89.200,   89.200,   88.000, 
      87.200,   86.800,   86.300,   85.900,   85.400,   84.600,   83.600,   82.900,   82.300,   81.600, 
      80.800,   80.100,   79.500,   78.900,   78.300,   77.500, ],
    [ 143.600,  138.700,  135.000,  133.600,  132.700,  131.900,  131.000,  130.100,  129.200,  128.200, 
      127.100,  126.000,  124.800,  123.400,  121.900,  120.400,  119.100,  117.500,  116.000,  114.600, 
      113.200,  111.800,  110.900,  109.700,  108.900,  108.500,  107.000,  106.300,  105.600,  105.300, 
      105.000,  104.700,  104.500,  104.200,  103.800,  103.500,  103.200,  102.800,  102.300,  101.800, 
      101.300,  100.500,  100.000,  100.400,  100.500,  101.100,  100.900,  100.500,  100.400,  100.000, 
      99.500,   99.100,   98.500,   98.100,   97.600,   97.300,   97.200,   96.400,   95.100,   93.800, 
      95.300,   95.500,   94.100,   92.700,   92.400,   92.500,   92.400,   91.800,   91.400,   90.400, 
      90.000,   89.500,   89.700,   88.000,   87.100,   87.300,   88.200,   88.800,   89.700,   89.800, 
      90.300,   90.600,   91.000,   91.700,   90.800,   90.400,   89.700,   89.600,   89.600,   88.500, 
      87.600,   87.200,   86.800,   86.300,   85.800,   85.000,   84.000,   83.300,   82.700,   82.000, 
      81.200,   80.600,   79.900,   79.300,   78.700,   77.900, ],
    [ 143.600,  138.700,  135.000,  133.600,  132.800,  132.000,  131.100,  130.200,  129.400,  128.300, 
      127.200,  126.100,  124.900,  123.600,  122.100,  120.600,  119.100,  117.600,  116.300,  114.700, 
      113.600,  112.200,  110.900,  109.800,  109.200,  108.500,  107.300,  106.400,  105.800,  105.300, 
      105.100,  104.900,  104.700,  104.200,  103.900,  103.600,  103.200,  103.000,  102.600,  102.000, 
      101.400,  100.600,  100.300,  100.500,  101.200,  101.300,  101.000,  100.800,  100.700,  100.300, 
      99.700,   99.300,   98.700,   98.300,   97.800,   97.600,   97.400,   96.500,   95.500,   94.300, 
      95.300,   95.500,   94.100,   92.800,   92.400,   92.500,   92.500,   92.200,   91.500,   90.900, 
      90.300,   89.800,   90.000,   88.300,   87.400,   87.600,   88.600,   89.300,   90.200,   90.500, 
      90.700,   91.300,   91.600,   92.200,   91.200,   91.100,   90.000,   90.000,   90.000,   88.900, 
      88.000,   87.700,   87.200,   86.800,   86.300,   85.400,   84.400,   83.700,   83.200,   82.400, 
      81.600,   81.000,   80.300,   79.700,   79.100,   78.300, ],
    [ 143.700,  138.800,  135.100,  133.800,  132.900,  132.100,  131.200,  130.400,  129.400,  128.400, 
      127.400,  126.300,  125.000,  123.700,  122.200,  120.800,  119.300,  118.000,  116.600,  115.100, 
      113.600,  112.200,  111.100,  110.000,  109.200,  108.700,  107.500,  106.500,  105.900,  105.400, 
      105.200,  105.100,  104.800,  104.300,  104.100,  103.700,  103.500,  103.000,  102.600,  102.200, 
      101.400,  100.700,  100.800,  101.100,  101.200,  101.300,  101.200,  100.800,  100.700,  100.300, 
      99.900,   99.500,   98.900,   98.500,   98.000,   97.800,   97.400,   96.500,   95.500,   94.400, 
      95.400,   96.100,   94.500,   93.400,   93.300,   92.600,   92.500,   92.600,   91.500,   91.300, 
      90.600,   90.000,   90.300,   88.500,   87.600,   88.000,   89.000,   89.700,   90.600,   91.200, 
      91.100,   91.900,   92.200,   92.600,   91.500,   91.700,   90.900,   91.200,   90.500,   89.300, 
      88.500,   88.100,   87.700,   87.200,   86.700,   85.900,   84.800,   84.200,   83.600,   82.800, 
      82.000,   81.400,   80.700,   80.100,   79.500,   78.700, ],
    [ 143.800,  138.900,  135.200,  133.800,  133.000,  132.200,  131.400,  130.500,  129.600,  128.500, 
      127.500,  126.300,  125.100,  123.800,  122.600,  121.100,  119.600,  118.000,  116.600,  115.300, 
      113.900,  112.500,  111.100,  110.200,  109.400,  108.900,  107.600,  106.700,  106.100,  105.600, 
      105.500,  105.300,  104.900,  104.500,  104.200,  103.800,  103.600,  103.300,  102.800,  102.200, 
      101.500,  101.100,  101.200,  101.400,  102.000,  101.700,  101.200,  101.100,  100.800,  100.400, 
      100.100,   99.600,   99.100,   98.700,   98.200,   98.000,   97.700,   96.800,   95.600,   95.100, 
      96.500,   96.100,   95.000,   93.400,   93.300,   93.600,   92.600,   92.900,   91.600,   91.400, 
      91.000,   91.700,   90.700,   88.800,   87.900,   88.300,   89.400,   90.300,   91.000,   91.800, 
      91.500,   92.500,   92.600,   93.100,   92.600,   91.900,   91.500,   92.000,   90.800,   89.700, 
      88.900,   88.600,   88.100,   87.600,   87.200,   86.300,   85.200,   84.600,   84.000,   83.200, 
      82.400,   81.800,   81.100,   80.500,   79.900,   79.100, ],
    [ 143.900,  139.000,  135.400,  133.900,  133.100,  132.300,  131.400,  130.600,  129.700,  128.700, 
      127.600,  126.700,  125.400,  124.000,  122.600,  121.100,  119.600,  118.200,  116.800,  115.300, 
      113.900,  112.600,  111.300,  110.200,  109.600,  108.900,  107.600,  106.800,  106.100,  105.900, 
      105.700,  105.400,  105.000,  104.700,  104.300,  103.900,  103.600,  103.300,  102.800,  102.300, 
      101.500,  101.500,  101.600,  102.000,  102.000,  101.700,  101.500,  101.300,  101.100,  100.600, 
      100.200,   99.800,   99.200,   98.900,   98.400,   98.700,   97.900,   97.100,   95.700,   95.200, 
      96.900,   96.200,   95.000,   93.500,   93.400,   93.700,   93.500,   92.900,   92.400,   91.500, 
      91.400,   92.500,   91.000,   89.000,   88.200,   88.600,   89.800,   91.100,   91.400,   92.000, 
      92.500,   92.700,   92.800,   93.600,   93.000,   92.200,   91.900,   92.300,   91.200,   90.000, 
      89.300,   89.100,   88.500,   88.100,   87.600,   86.700,   85.600,   85.000,   84.400,   83.600, 
      82.800,   82.200,   81.500,   80.900,   80.300,   79.500, ],
    [ 143.900,  139.100,  135.400,  134.000,  133.200,  132.400,  131.500,  130.800,  129.900,  128.900, 
      127.800,  126.700,  125.400,  124.000,  122.800,  121.200,  119.800,  118.200,  116.800,  115.500, 
      114.000,  112.600,  111.500,  110.400,  109.700,  109.000,  107.800,  107.000,  106.200,  106.100, 
      105.800,  105.500,  105.100,  104.800,  104.300,  104.200,  103.900,  103.500,  103.000,  102.300, 
      101.800,  102.000,  101.900,  102.700,  102.400,  101.900,  101.800,  101.300,  101.200,  100.800, 
      100.400,  100.000,   99.400,   99.100,   98.600,   98.700,   98.100,   97.300,   95.800,   95.900, 
      97.200,   96.700,   95.100,   93.600,   93.800,   93.700,   93.700,   92.900,   92.800,   91.600, 
      91.700,   93.200,   91.400,   89.300,   88.500,   89.000,   90.200,   91.900,   91.800,   92.300, 
      92.700,   92.900,   93.600,   94.100,   93.200,   92.500,   92.100,   92.800,   91.500,   90.600, 
      89.700,   89.500,   89.000,   88.500,   88.000,   87.100,   86.000,   85.300,   84.800,   84.000, 
      83.200,   82.500,   81.900,   81.300,   80.700,   79.900, ],
    [ 144.000,  139.100,  135.600,  134.200,  133.400,  132.600,  131.700,  130.800,  129.900,  128.900, 
      127.900,  126.800,  125.600,  124.200,  122.800,  121.200,  119.900,  118.400,  117.000,  115.500, 
      114.200,  112.800,  111.500,  110.500,  109.700,  109.200,  107.900,  107.100,  106.600,  106.200, 
      105.900,  105.600,  105.200,  104.900,  104.500,  104.200,  103.900,  103.500,  103.000,  102.400, 
      102.000,  102.100,  102.700,  102.800,  102.400,  102.300,  102.000,  101.700,  101.400,  100.900, 
      100.600,  100.100,   99.600,   99.300,   98.800,   99.000,   98.300,   97.300,   96.200,   96.000, 
      97.600,   96.700,   95.400,   94.100,   94.400,   94.500,   93.700,   93.600,   93.100,   91.700, 
      92.100,   93.400,   91.400,   89.500,   88.800,   89.300,   90.700,   92.000,   92.200,   92.700, 
      93.500,   93.600,   94.300,   94.700,   94.000,   92.800,   92.900,   93.300,   92.400,   90.900, 
      90.000,   90.000,   89.400,   88.900,   88.400,   87.500,   86.400,   85.700,   85.200,   84.400, 
      83.600,   82.900,   82.300,   81.700,   81.000,   80.200, ],
    [ 144.100,  139.300,  135.600,  134.200,  133.400,  132.600,  131.800,  131.000,  130.000,  129.000, 
      127.900,  126.800,  125.600,  124.300,  122.900,  121.400,  119.900,  118.500,  117.100,  115.600, 
      114.200,  112.900,  111.600,  110.700,  109.900,  109.300,  108.100,  107.100,  106.700,  106.200, 
      106.000,  105.700,  105.400,  104.900,  104.700,  104.300,  104.100,  103.700,  103.000,  102.400, 
      102.400,  102.700,  103.300,  102.800,  102.600,  102.300,  102.000,  101.700,  101.500,  101.100, 
      100.700,  100.300,   99.900,   99.500,   99.000,   99.300,   98.500,   97.600,   96.200,   96.600, 
      97.900,   97.000,   95.700,   94.100,   94.500,   94.600,   93.800,   94.000,   93.100,   91.800, 
      92.500,   93.500,   91.500,   89.700,   89.000,   89.600,   91.200,   92.400,   92.700,   93.600, 
      93.700,   94.400,   94.900,   95.200,   94.300,   93.100,   93.500,   93.800,   92.600,   91.300, 
      90.500,   90.300,   89.800,   89.300,   88.800,   87.900,   86.800,   86.100,   85.600,   84.700, 
      83.900,   83.300,   82.700,   82.000,   81.400,   80.600, ],
    [ 144.100,  139.400,  135.700,  134.300,  133.500,  132.700,  131.800,  131.000,  130.000,  129.100, 
      128.000,  126.900,  125.600,  124.300,  123.000,  121.500,  120.000,  118.500,  117.200,  115.700, 
      114.300,  113.000,  111.700,  110.800,  110.000,  109.500,  108.200,  107.400,  106.900,  106.400, 
      106.200,  105.900,  105.500,  105.200,  104.700,  104.500,  104.100,  103.700,  103.100,  102.800, 
      102.800,  103.400,  103.500,  103.200,  102.900,  102.500,  102.100,  101.900,  101.700,  101.200, 
      100.900,  100.500,  100.100,   99.600,   99.600,   99.400,   98.700,   97.800,   96.300,   97.200, 
      98.200,   97.000,   95.700,   94.600,   94.500,   94.600,   94.600,   94.200,   93.200,   91.900, 
      92.800,   94.000,   91.600,   90.000,   89.300,   90.000,   91.700,   92.800,   93.500,   93.800, 
      94.300,   94.800,   95.100,   95.700,   94.500,   93.500,   93.600,   94.300,   92.900,   91.700, 
      90.900,   90.700,   90.200,   89.700,   89.200,   88.300,   87.200,   86.500,   85.900,   85.100, 
      84.300,   83.700,   83.100,   82.400,   81.800,   81.000, ],
    [ 144.200,  139.400,  135.700,  134.300,  133.500,  132.800,  131.900,  131.000,  130.100,  129.100, 
      128.100,  127.000,  125.700,  124.400,  123.000,  121.500,  120.100,  118.600,  117.300,  115.800, 
      114.400,  113.100,  111.900,  110.900,  110.100,  109.500,  108.500,  107.600,  106.900,  106.500, 
      106.300,  106.000,  105.500,  105.300,  105.000,  104.600,  104.300,  103.700,  103.100,  103.200, 
      103.400,  103.400,  103.500,  103.400,  102.900,  102.700,  102.400,  102.000,  101.800,  101.400, 
      101.000,  100.800,  100.300,   99.800,   99.900,   99.500,   98.700,   97.900,   96.400,   97.200, 
      98.200,   97.400,   95.700,   94.900,   94.800,   95.400,   94.600,   94.200,   93.900,   92.600, 
      92.900,   94.200,   91.700,   90.200,   89.600,   90.600,   92.200,   93.200,   93.800,   94.600, 
      95.300,   95.600,   95.800,   95.900,   95.000,   93.800,   94.500,   94.800,   93.200,   92.100, 
      91.300,   91.100,   90.600,   90.000,   89.600,   88.700,   87.500,   86.900,   86.300,   85.500, 
      84.700,   84.100,   83.400,   82.800,   82.200,   81.400, ],
    [ 144.200,  139.400,  135.800,  134.400,  133.600,  132.800,  132.000,  131.100,  130.200,  129.200, 
      128.100,  127.000,  125.800,  124.500,  123.000,  121.500,  120.200,  118.700,  117.300,  115.900, 
      114.500,  113.300,  112.000,  110.900,  110.200,  109.500,  108.600,  107.700,  107.000,  106.600, 
      106.400,  106.100,  105.700,  105.300,  105.000,  104.600,  104.300,  103.700,  103.500,  103.600, 
      103.600,  104.100,  103.800,  103.400,  103.100,  102.800,  102.500,  102.200,  102.000,  101.700, 
      101.300,  100.900,  100.400,  100.000,  100.200,   99.600,   98.700,   98.100,   96.800,   97.700, 
      98.600,   97.400,   96.100,   95.300,   95.300,   95.400,   94.600,   94.300,   93.900,   92.700, 
      92.900,   94.300,   92.300,   90.400,   90.000,   91.300,   92.400,   94.200,   94.200,   95.200, 
      95.400,   95.700,   96.500,   96.600,   95.800,   94.500,   94.700,   95.400,   93.800,   92.500, 
      91.700,   91.400,   91.000,   90.900,   90.000,   89.100,   87.900,   87.200,   86.700,   85.900, 
      85.100,   84.400,   83.800,   83.200,   82.600,   81.700, ],
    [ 144.200,  139.500,  135.900,  134.500,  133.700,  132.800,  132.000,  131.100,  130.300,  129.300, 
      128.200,  127.100,  125.900,  124.500,  123.100,  121.600,  120.300,  118.800,  117.400,  116.000, 
      114.600,  113.300,  112.100,  111.000,  110.500,  109.800,  108.800,  107.800,  107.200,  106.800, 
      106.600,  106.100,  105.800,  105.500,  105.100,  104.800,  104.300,  103.800,  103.800,  104.200, 
      104.100,  104.100,  103.800,  103.600,  103.400,  103.000,  102.600,  102.300,  102.100,  101.800, 
      101.500,  101.100,  100.600,  100.500,  100.300,   99.800,   99.200,   98.200,   97.100,   98.200, 
      98.600,   97.600,   96.100,   95.300,   95.400,   95.400,   95.400,   95.000,   93.900,   93.200, 
      93.000,   95.000,   92.400,   90.600,   90.200,   91.700,   92.500,   94.500,   95.000,   95.400, 
      95.900,   96.700,   96.500,   97.200,   95.900,   94.800,   95.500,   95.900,   94.100,   92.800, 
      92.100,   91.800,   91.300,   91.500,   90.400,   89.500,   88.300,   87.600,   87.100,   86.200, 
      85.400,   84.800,   84.200,   83.500,   82.900,   82.100, ],
    [ 144.300,  139.500,  135.900,  134.500,  133.700,  132.900,  132.000,  131.200,  130.300,  129.300, 
      128.200,  127.100,  125.900,  124.600,  123.200,  121.700,  120.300,  118.800,  117.500,  116.000, 
      114.700,  113.300,  112.300,  111.300,  110.600,  110.000,  108.800,  107.900,  107.300,  106.900, 
      106.600,  106.300,  105.900,  105.600,  105.200,  104.800,  104.400,  104.200,  104.300,  104.400, 
      104.800,  104.500,  104.000,  103.800,  103.400,  103.200,  102.800,  102.400,  102.200,  102.000, 
      101.600,  101.300,  100.800,  100.800,  100.300,  100.000,   99.300,   98.200,   97.200,   99.000, 
      99.100,   97.800,   96.200,   95.600,   95.800,   96.000,   95.400,   95.000,   94.700,   93.200, 
      93.900,   95.100,   92.400,   90.800,   90.300,   91.800,   93.400,   94.800,   95.700,   96.000, 
      96.600,   96.700,   97.500,   97.300,   96.400,   95.400,   95.600,   96.100,   94.300,   92.800, 
      92.500,   92.100,   92.400,   92.100,   90.800,   90.000,   88.600,   88.000,   87.400,   86.600, 
      85.800,   85.100,   84.500,   83.900,   83.300,   82.400, ],
    [ 144.300,  139.500,  135.900,  134.500,  133.800,  132.900,  132.100,  131.200,  130.300,  129.400, 
      128.300,  127.200,  126.000,  124.700,  123.300,  121.800,  120.400,  118.900,  117.500,  116.100, 
      114.900,  113.600,  112.400,  111.400,  110.700,  110.100,  108.900,  108.000,  107.400,  107.000, 
      106.600,  106.400,  106.100,  105.700,  105.300,  104.800,  104.400,  104.500,  104.800,  104.800, 
      104.800,  104.500,  104.300,  104.000,  103.700,  103.300,  102.900,  102.700,  102.400,  102.100, 
      101.800,  101.400,  101.400,  101.000,  100.500,  100.300,   99.400,   98.300,   97.700,   99.400, 
      99.100,   98.100,   96.500,   95.900,   96.400,   96.000,   96.000,   95.100,   94.700,   93.700, 
      94.300,   95.300,   93.000,   90.900,   90.500,   92.700,   94.100,   95.700,   95.900,   96.500, 
      97.300,   97.600,   97.800,   98.100,   96.800,   95.600,   96.300,   96.600,   94.600,   93.000, 
      92.900,   92.500,   92.500,   92.500,   91.200,   90.000,   88.900,   88.300,   87.800,   86.900, 
      86.100,   85.500,   84.900,   84.200,   83.600,   82.800, ],
    [ 144.400,  139.600,  136.000,  134.600,  133.800,  133.000,  132.100,  131.300,  130.400,  129.400, 
      128.300,  127.300,  126.000,  124.700,  123.300,  121.800,  120.400,  119.000,  117.700,  116.300, 
      115.000,  113.700,  112.500,  111.500,  110.700,  110.200,  108.900,  108.100,  107.500,  107.000, 
      106.900,  106.400,  106.100,  105.700,  105.300,  104.900,  104.800,  104.900,  105.000,  105.500, 
      105.100,  104.700,  104.500,  104.000,  103.700,  103.400,  103.000,  102.800,  102.600,  102.200, 
      102.100,  101.700,  101.600,  101.200,  100.800,  100.500,   99.800,   98.600,   98.200,   99.800, 
      99.100,   98.300,   96.500,   96.200,   96.500,   96.600,   96.000,   95.700,   94.800,   93.700, 
      94.600,   95.500,   93.000,   91.100,   90.600,   92.800,   94.700,   95.900,   96.600,   97.000, 
      97.700,   98.300,   98.700,   98.200,   97.200,   96.500,   96.400,   96.800,   95.300,   93.700, 
      93.200,   92.900,   92.800,   92.700,   91.600,   90.400,   89.300,   88.700,   88.200,   87.300, 
      86.500,   85.900,   85.300,   84.600,   84.000,   83.100, ],
    [ 144.400,  139.600,  136.000,  134.600,  133.900,  133.000,  132.200,  131.300,  130.400,  129.400, 
      128.400,  127.300,  126.100,  124.700,  123.400,  122.000,  120.600,  119.200,  117.800,  116.400, 
      115.100,  113.800,  112.500,  111.600,  110.800,  110.300,  109.100,  108.300,  107.600,  107.200, 
      106.900,  106.700,  106.200,  105.900,  105.400,  104.900,  105.000,  105.400,  106.000,  105.500, 
      105.300,  104.900,  104.700,  104.300,  103.800,  103.600,  103.200,  103.000,  102.800,  102.500, 
      102.200,  101.800,  101.900,  101.300,  101.000,  100.500,   99.800,   98.600,   98.800,   99.900, 
      99.300,   98.400,   96.800,   96.600,   97.000,   96.600,   96.000,   95.700,   95.500,   94.100, 
      95.000,   95.700,   93.100,   91.300,   90.800,   93.500,   95.200,   96.300,   97.200,   97.800, 
      98.300,   98.400,   99.300,   98.700,   97.600,   97.000,   97.000,   97.200,   95.500,   94.100, 
      93.600,   93.200,   93.100,   93.500,   92.000,   90.600,   89.600,   89.000,   88.500,   87.600, 
      86.800,   86.200,   85.600,   85.000,   84.300,   83.500, ],
    [ 144.400,  139.600,  136.100,  134.700,  133.900,  133.000,  132.200,  131.400,  130.500,  129.500, 
      128.400,  127.400,  126.200,  124.900,  123.500,  122.100,  120.700,  119.300,  117.900,  116.500, 
      115.100,  113.800,  112.600,  111.700,  110.900,  110.300,  109.200,  108.400,  107.600,  107.300, 
      107.000,  106.700,  106.400,  105.900,  105.400,  105.300,  105.400,  106.000,  106.100,  105.800, 
      105.600,  105.100,  104.700,  104.400,  103.900,  103.700,  103.400,  103.100,  102.900,  102.600, 
      102.300,  102.300,  101.900,  101.500,  101.100,  100.500,   99.800,   98.700,   99.200,   99.900, 
      99.500,   98.500,   96.900,   96.800,   97.000,   97.200,   96.800,   96.300,   95.500,   94.700, 
      95.300,   95.700,   93.100,   91.500,   90.900,   93.600,   95.200,   97.000,   97.400,   98.200, 
      98.600,   99.400,   99.900,   99.200,   98.000,   97.400,   97.600,   97.600,   95.800,   94.200, 
      94.000,   93.600,   93.400,   93.600,   92.300,   90.800,   90.000,   89.300,   88.800,   88.000, 
      87.100,   86.600,   86.000,   85.300,   84.700,   83.800, ],
    [ 144.400,  139.700,  136.100,  134.700,  133.900,  133.100,  132.300,  131.500,  130.600,  129.600, 
      128.500,  127.500,  126.300,  124.900,  123.600,  122.100,  120.700,  119.300,  118.000,  116.500, 
      115.200,  113.900,  112.700,  111.800,  111.100,  110.400,  109.300,  108.400,  107.800,  107.300, 
      107.200,  106.800,  106.400,  105.900,  105.400,  105.600,  106.000,  106.600,  106.400,  106.000, 
      105.600,  105.300,  105.000,  104.500,  104.200,  103.800,  103.500,  103.300,  103.100,  102.800, 
      102.500,  102.500,  102.000,  101.600,  101.300,  100.900,  100.000,   98.700,   99.600,  100.300, 
      99.700,   98.600,   97.200,   97.200,   97.500,   97.200,   96.800,   96.300,   95.900,   94.700, 
      95.700,   95.800,   93.700,   91.600,   91.800,   94.200,   96.000,   97.700,   98.000,   98.900, 
      99.200,   99.500,   99.900,   99.500,   98.100,   97.800,   98.600,   97.900,   96.000,   94.300, 
      94.400,   94.400,   94.100,   93.800,   92.700,   91.100,   90.300,   89.600,   89.200,   88.300, 
      87.500,   86.900,   86.300,   85.600,   85.000,   84.100, ],
    [ 144.400,  139.700,  136.200,  134.800,  134.000,  133.200,  132.400,  131.500,  130.600,  129.700, 
      128.600,  127.600,  126.300,  125.000,  123.600,  122.200,  120.800,  119.300,  118.000,  116.600, 
      115.300,  114.000,  112.800,  111.800,  111.100,  110.600,  109.400,  108.500,  107.900,  107.500, 
      107.300,  106.900,  106.400,  105.900,  105.800,  105.900,  106.500,  106.600,  106.400,  106.200, 
      105.700,  105.300,  105.000,  104.600,  104.300,  103.900,  103.700,  103.500,  103.300,  103.000, 
      103.000,  102.700,  102.200,  101.800,  101.300,  101.000,  100.000,   99.100,  100.700,  100.300, 
      99.700,   98.800,   97.200,   97.200,   97.500,   97.700,   97.400,   96.900,   95.900,   95.600, 
      96.000,   96.200,   93.700,   91.600,   91.900,   94.700,   96.600,   98.400,   98.500,   99.400, 
      99.700,  100.300,  100.800,  100.000,   98.400,   98.300,   99.200,   98.300,   96.300,   95.000, 
      95.200,   94.600,   94.400,   94.500,   93.100,   91.300,   90.900,   90.000,   89.500,   88.600, 
      87.800,   87.200,   86.600,   86.000,   85.400,   84.500, ],
    [ 144.500,  139.800,  136.200,  134.900,  134.100,  133.200,  132.500,  131.600,  130.700,  129.700, 
      128.600,  127.600,  126.400,  125.100,  123.700,  122.200,  120.800,  119.400,  118.100,  116.700, 
      115.400,  114.100,  112.900,  112.000,  111.200,  110.700,  109.400,  108.700,  107.900,  107.500, 
      107.300,  107.000,  106.400,  106.200,  106.100,  106.400,  107.100,  106.900,  106.500,  106.200, 
      105.900,  105.600,  105.100,  104.800,  104.400,  104.200,  103.800,  103.600,  103.400,  103.200, 
      103.200,  102.900,  102.300,  101.900,  101.500,  101.000,  100.100,   99.300,  101.100,  100.600, 
      100.100,   98.800,   97.700,   97.800,   98.000,   97.700,   97.400,   96.900,   96.200,   95.600, 
      97.500,   96.400,   93.800,   91.600,   92.000,   95.200,   97.600,   98.500,   99.200,  100.200, 
      100.300,  101.100,  100.800,  100.300,   98.900,   98.900,   99.800,   98.400,   96.700,   95.500, 
      95.400,   94.900,   95.000,   94.600,   93.400,   91.600,   91.300,   90.400,   89.800,   88.900, 
      88.100,   87.600,   87.000,   86.300,   85.700,   84.800, ],
    [ 144.600,  139.800,  136.300,  134.900,  134.100,  133.300,  132.500,  131.600,  130.700,  129.700, 
      128.700,  127.600,  126.400,  125.100,  123.700,  122.300,  120.900,  119.500,  118.100,  116.800, 
      115.400,  114.200,  113.000,  112.100,  111.300,  110.700,  109.600,  108.700,  108.100,  107.700, 
      107.400,  107.000,  106.500,  106.300,  106.700,  107.000,  107.100,  106.900,  106.800,  106.400, 
      106.000,  105.700,  105.300,  104.900,  104.500,  104.300,  104.000,  103.800,  103.600,  103.700, 
      103.400,  103.000,  102.600,  102.100,  101.800,  101.400,  100.400,  100.200,  101.400,  100.600, 
      100.200,   99.100,   97.900,   98.100,   98.400,   98.100,   97.900,   97.300,   96.200,   95.600, 
      97.600,   96.400,   94.200,   91.600,   93.200,   95.900,   98.500,   99.500,   99.700,  100.600, 
      100.900,  101.800,  101.400,  100.500,   99.400,   99.400,   99.900,   98.800,   97.000,   95.800, 
      95.700,   95.500,   95.200,   95.300,   93.800,   91.800,   91.600,   90.900,   90.000,   89.200, 
      88.400,   87.900,   87.300,   86.600,   86.000,   85.100, ],
    [ 144.600,  139.800,  136.300,  134.900,  134.100,  133.300,  132.500,  131.600,  130.800,  129.800, 
      128.700,  127.700,  126.500,  125.100,  123.800,  122.300,  120.900,  119.500,  118.200,  116.800, 
      115.500,  114.300,  113.000,  112.100,  111.300,  110.900,  109.700,  108.900,  108.100,  107.800, 
      107.400,  107.000,  106.700,  106.900,  107.000,  107.600,  107.400,  107.100,  106.900,  106.500, 
      106.200,  105.800,  105.400,  105.000,  104.800,  104.400,  104.200,  104.000,  103.800,  103.900, 
      103.600,  103.100,  102.700,  102.100,  101.800,  101.400,  100.400,  100.300,  101.400,  100.800, 
      100.300,   99.100,   98.300,   98.600,   98.500,   98.100,   97.900,   97.300,   96.300,   96.600, 
      98.100,   96.800,   94.300,   91.600,   93.300,   96.500,   98.700,   99.700,  100.600,  101.300, 
      101.400,  102.400,  101.800,  100.800,   99.500,   99.800,  100.300,   98.900,   97.300,   96.200, 
      96.000,   95.800,   95.500,   95.400,   94.200,   92.000,   91.700,   91.400,   90.700,   89.500, 
      88.800,   88.200,   87.600,   87.000,   86.400,   85.400, ],
    [ 144.600,  139.900,  136.300,  135.000,  134.200,  133.400,  132.500,  131.700,  130.800,  129.800, 
      128.800,  127.700,  126.500,  125.200,  123.800,  122.400,  121.000,  119.600,  118.300,  116.900, 
      115.600,  114.300,  113.100,  112.200,  111.600,  110.900,  109.700,  108.900,  108.300,  107.800, 
      107.500,  107.100,  107.100,  107.400,  108.000,  107.800,  107.600,  107.300,  107.100,  106.600, 
      106.300,  105.900,  105.500,  105.200,  104.900,  104.600,  104.400,  104.100,  103.900,  104.100, 
      103.700,  103.300,  102.800,  102.400,  102.000,  101.600,  100.400,  100.900,  101.700,  101.000, 
      100.500,   99.300,   98.500,   98.600,   98.900,   98.600,   98.400,   97.800,   96.300,   96.600, 
      98.200,   96.800,   94.300,   91.700,   93.300,   97.400,   99.400,  100.500,  101.100,  101.600, 
      102.200,  102.700,  102.200,  101.300,  100.100,  100.300,  100.600,   99.100,   97.600,   96.600, 
      96.600,   96.400,   96.000,   95.700,   94.200,   92.800,   91.900,   91.800,   91.200,   89.800, 
      89.100,   88.500,   88.000,   87.300,   86.700,   85.800, ],
    [ 144.600,  139.900,  136.400,  135.000,  134.200,  133.400,  132.600,  131.700,  130.800,  129.900, 
      128.800,  127.800,  126.500,  125.300,  123.900,  122.500,  121.100,  119.600,  118.300,  116.900, 
      115.600,  114.400,  113.300,  112.300,  111.600,  110.900,  109.900,  109.000,  108.400,  107.800, 
      107.500,  107.400,  107.400,  107.900,  108.000,  107.800,  107.800,  107.400,  107.100,  106.800, 
      106.400,  106.000,  105.600,  105.400,  105.000,  104.700,  104.500,  104.300,  104.500,  104.200, 
      103.800,  103.500,  102.900,  102.500,  102.100,  101.600,  100.500,  102.200,  101.700,  101.200, 
      100.500,   99.400,   98.800,   99.000,   98.900,   98.600,   98.400,   97.800,   96.600,   96.600, 
      98.400,   96.800,   94.400,   92.400,   94.200,   98.100,  100.100,  100.900,  101.700,  102.300, 
      102.900,  103.400,  102.400,  101.400,  100.700,  101.400,  100.800,   99.600,   97.800,   96.900, 
      96.800,   96.600,   96.600,   95.900,   94.300,   92.900,   92.600,   92.300,   91.600,   90.200, 
      89.400,   88.800,   88.300,   87.600,   87.000,   86.100, ],
    [ 144.700,  139.900,  136.400,  135.000,  134.200,  133.400,  132.600,  131.800,  130.900,  129.900, 
      128.900,  127.800,  126.600,  125.300,  123.900,  122.500,  121.100,  119.600,  118.400,  117.000, 
      115.800,  114.500,  113.300,  112.300,  111.700,  111.100,  110.000,  109.200,  108.400,  107.900, 
      107.700,  107.700,  108.000,  108.500,  108.300,  108.000,  107.800,  107.600,  107.300,  106.900, 
      106.500,  106.100,  105.800,  105.500,  105.200,  105.000,  104.600,  104.400,  104.700,  104.300, 
      103.900,  103.600,  103.100,  102.500,  102.100,  101.600,  100.800,  102.600,  102.000,  101.300, 
      100.500,   99.600,   99.300,   99.400,   99.300,   99.100,   98.800,   98.200,   96.800,   97.600, 
      98.500,   97.000,   94.500,   92.600,   94.700,   98.700,  100.700,  101.400,  102.300,  102.900, 
      103.600,  103.400,  102.800,  101.600,  101.300,  102.000,  101.100,   99.800,   97.800,   97.300, 
      97.400,   97.200,   96.800,   96.200,   95.000,   93.500,   92.700,   92.700,   91.600,   90.800, 
      89.700,   89.100,   88.600,   87.900,   87.300,   86.400, ],
    [ 144.700,  140.000,  136.400,  135.100,  134.300,  133.500,  132.700,  131.800,  130.900,  130.000, 
      128.900,  127.800,  126.600,  125.300,  123.900,  122.600,  121.200,  119.800,  118.500,  117.100, 
      115.800,  114.500,  113.400,  112.500,  111.700,  111.100,  110.000,  109.200,  108.400,  107.900, 
      107.800,  108.400,  108.900,  108.700,  108.300,  108.200,  107.900,  107.700,  107.300,  107.000, 
      106.600,  106.300,  105.900,  105.700,  105.300,  105.100,  104.800,  105.000,  104.800,  104.400, 
      104.200,  103.600,  103.300,  102.800,  102.300,  101.700,  101.000,  102.900,  102.100,  101.600, 
      100.700,   99.600,   99.300,   99.400,   99.700,   99.600,   98.800,   98.200,   97.100,   97.900, 
      98.500,   97.000,   94.900,   92.600,   95.700,   98.700,  101.200,  102.000,  103.000,  103.300, 
      104.300,  103.900,  103.200,  102.000,  101.700,  102.400,  101.400,  100.000,   98.400,   97.600, 
      97.600,   97.700,   97.000,   96.500,   95.200,   93.600,   92.800,   93.100,   91.800,   91.300, 
      90.000,   89.400,   88.900,   88.200,   87.700,   86.700, ],
    [ 144.700,  140.000,  136.500,  135.100,  134.300,  133.500,  132.700,  131.800,  130.900,  130.000, 
      128.900,  127.800,  126.700,  125.400,  124.100,  122.600,  121.300,  119.800,  118.500,  117.100, 
      115.900,  114.600,  113.500,  112.500,  111.800,  111.200,  110.100,  109.200,  108.500,  108.200, 
      108.400,  108.900,  108.900,  108.700,  108.600,  108.300,  108.100,  107.800,  107.400,  107.100, 
      106.800,  106.400,  106.000,  105.800,  105.500,  105.200,  105.000,  105.200,  104.900,  104.600, 
      104.300,  103.900,  103.300,  102.900,  102.300,  101.900,  101.700,  102.900,  102.200,  101.700, 
      100.900,  100.100,   99.800,   99.800,   99.700,   99.600,   99.200,   98.600,   97.300,   99.500, 
      98.500,   97.200,   95.000,   92.600,   95.900,  100.400,  102.000,  102.800,  103.600,  104.500, 
      104.800,  104.200,  103.400,  102.400,  102.100,  102.700,  101.700,  100.200,   98.400,   98.000, 
      97.900,   97.800,   97.800,   96.800,   95.200,   93.700,   93.400,   93.200,   92.500,   91.600, 
      90.400,   89.700,   89.200,   88.500,   88.000,   87.000, ],
    [ 144.700,  140.000,  136.500,  135.100,  134.300,  133.500,  132.700,  131.900,  131.000,  130.100, 
      129.000,  128.000,  126.800,  125.500,  124.100,  122.700,  121.300,  119.900,  118.600,  117.200, 
      115.900,  114.700,  113.500,  112.600,  111.900,  111.300,  110.100,  109.200,  108.500,  108.500, 
      108.800,  109.400,  109.200,  108.900,  108.600,  108.500,  108.200,  107.900,  107.600,  107.300, 
      106.900,  106.500,  106.300,  106.000,  105.600,  105.400,  105.500,  105.300,  105.000,  104.800, 
      104.300,  104.000,  103.600,  103.000,  102.400,  102.000,  102.400,  103.100,  102.400,  101.900, 
      100.900,  100.300,  100.000,  100.200,  100.100,  100.000,   99.200,   98.600,   98.200,   99.600, 
      98.900,   97.300,   95.000,   92.700,   97.500,  101.100,  102.500,  103.500,  104.100,  105.000, 
      105.300,  104.700,  103.800,  102.900,  102.800,  102.900,  101.800,  100.500,   98.900,   98.700, 
      98.500,   98.400,   98.200,   97.100,   95.800,   93.800,   93.500,   93.300,   92.600,   91.700, 
      91.000,   90.000,   89.500,   88.800,   88.300,   87.300, ],
    [ 144.700,  140.000,  136.500,  135.200,  134.400,  133.600,  132.800,  132.000,  131.100,  130.100, 
      129.000,  128.000,  126.800,  125.500,  124.200,  122.800,  121.400,  119.900,  118.600,  117.300, 
      116.000,  114.700,  113.600,  112.700,  112.000,  111.400,  110.200,  109.300,  108.800,  109.200, 
      109.800,  109.600,  109.200,  109.100,  108.800,  108.600,  108.300,  108.000,  107.700,  107.400, 
      107.000,  106.700,  106.400,  106.100,  105.800,  105.900,  105.700,  105.400,  105.100,  104.900, 
      104.500,  104.100,  103.600,  103.100,  102.600,  102.000,  103.700,  103.200,  102.500,  101.900, 
      101.100,  100.500,  100.400,  100.500,  100.400,  100.000,   99.500,   98.800,   98.200,  100.000, 
      98.900,   97.300,   95.100,   92.700,   98.100,  101.800,  103.100,  104.200,  104.900,  105.600, 
      105.500,  105.000,  103.900,  103.400,  103.800,  103.200,  102.300,  100.800,   99.300,   98.900, 
      99.100,   98.800,   98.600,   97.400,   95.800,   94.400,   94.200,   94.000,   92.700,   91.900, 
      91.500,   90.600,   89.800,   89.100,   88.600,   87.600, ],
    [ 144.800,  140.100,  136.600,  135.200,  134.400,  133.600,  132.800,  132.000,  131.100,  130.100, 
      129.100,  128.100,  126.900,  125.600,  124.200,  122.800,  121.400,  120.000,  118.700,  117.300, 
      116.000,  114.800,  113.600,  112.700,  112.000,  111.400,  110.200,  109.500,  109.300,  109.600, 
      109.800,  109.600,  109.500,  109.200,  108.900,  108.700,  108.400,  108.100,  107.800,  107.500, 
      107.200,  106.800,  106.600,  106.300,  106.000,  106.000,  105.800,  105.500,  105.300,  104.900, 
      104.700,  104.300,  103.700,  103.100,  102.600,  102.300,  104.100,  103.400,  102.700,  101.900, 
      101.200,  100.900,  100.700,  100.500,  100.400,  100.400,   99.900,   98.800,   98.900,  100.200, 
      99.200,   97.600,   95.200,   93.800,   99.700,  102.700,  103.700,  104.800,  105.300,  106.300, 
      105.900,  105.200,  104.100,  103.900,  104.400,  103.400,  102.400,  101.000,   99.600,   99.200, 
      99.200,   99.000,   99.000,   97.700,   96.400,   94.800,   94.300,   94.100,   92.900,   92.100, 
      91.600,   91.100,   90.000,   89.400,   88.900,   87.800, ],
    [ 144.800,  140.100,  136.600,  135.200,  134.400,  133.600,  132.900,  132.000,  131.100,  130.200, 
      129.100,  128.100,  126.900,  125.600,  124.300,  122.800,  121.500,  120.000,  118.800,  117.400, 
      116.100,  114.800,  113.700,  112.700,  112.000,  111.400,  110.400,  110.000,  109.800,  110.100, 
      110.100,  109.800,  109.500,  109.300,  109.000,  108.800,  108.500,  108.200,  108.000,  107.700, 
      107.300,  107.000,  106.700,  106.400,  106.400,  106.200,  105.900,  105.600,  105.500,  105.200, 
      104.700,  104.300,  103.800,  103.300,  102.700,  102.700,  104.100,  103.600,  102.900,  102.100, 
      101.400,  101.300,  101.100,  100.900,  100.900,  100.700,  100.200,   98.800,   98.900,  100.300, 
      99.200,   97.600,   95.300,   93.900,  100.700,  103.200,  104.500,  105.300,  106.300,  106.700, 
      106.400,  105.600,  104.500,  104.500,  104.600,  103.600,  102.600,  101.100,   99.900,   99.700, 
      99.700,   99.400,   99.200,   98.000,   96.400,   94.900,   94.700,   94.200,   93.100,   92.300, 
      91.800,   91.600,   90.700,   89.700,   89.200,   88.100, ],
    [ 144.800,  140.200,  136.700,  135.300,  134.500,  133.700,  132.900,  132.000,  131.200,  130.200, 
      129.200,  128.100,  126.900,  125.600,  124.300,  122.900,  121.500,  120.100,  118.800,  117.400, 
      116.100,  114.900,  113.700,  112.700,  112.000,  111.400,  110.500,  110.400,  110.700,  110.400, 
      110.200,  110.000,  109.600,  109.400,  109.100,  108.900,  108.700,  108.400,  108.100,  107.800, 
      107.500,  107.100,  106.900,  106.600,  106.700,  106.300,  106.000,  105.800,  105.500,  105.300, 
      105.000,  104.500,  103.800,  103.300,  102.900,  103.700,  104.400,  103.700,  103.000,  102.100, 
      101.700,  101.400,  101.400,  101.200,  101.300,  100.700,  100.200,   99.200,   99.700,  100.300, 
      99.400,   97.900,   95.400,   94.000,  100.700,  103.900,  105.100,  106.000,  106.800,  107.300, 
      106.600,  105.900,  104.800,  104.800,  104.900,  103.900,  102.800,  101.100,  100.300,  100.200, 
      100.200,   99.900,   99.400,   98.300,   96.900,   95.300,   94.800,   94.800,   93.600,   92.500, 
      91.900,   91.600,   91.200,   90.000,   89.500,   88.400, ],
    [ 144.900,  140.200,  136.700,  135.300,  134.500,  133.700,  132.900,  132.100,  131.200,  130.200, 
      129.200,  128.200,  127.000,  125.700,  124.300,  122.900,  121.500,  120.100,  118.800,  117.400, 
      116.100,  114.900,  113.700,  112.800,  112.200,  111.700,  111.300,  111.300,  110.700,  110.400, 
      110.400,  110.100,  109.800,  109.600,  109.200,  109.000,  108.800,  108.500,  108.300,  107.900, 
      107.600,  107.300,  107.100,  107.100,  106.700,  106.400,  106.200,  105.900,  105.700,  105.300, 
      105.000,  104.500,  104.000,  103.400,  103.100,  104.900,  104.400,  103.800,  103.100,  102.300, 
      102.000,  101.700,  101.700,  101.500,  101.300,  101.000,  100.500,   99.400,  101.300,  100.300, 
      99.400,   97.900,   95.500,   94.900,  102.700,  104.600,  105.900,  106.700,  107.600,  107.600, 
      106.900,  106.000,  105.400,  105.900,  105.100,  104.000,  103.000,  101.500,  100.600,  100.400, 
      100.600,  100.000,   99.600,   98.400,   96.900,   95.600,   95.200,   94.900,   94.100,   92.700, 
      92.200,   91.800,   91.600,   90.500,   89.700,   88.600, ],
    [ 144.900,  140.200,  136.700,  135.300,  134.600,  133.700,  132.900,  132.100,  131.200,  130.300, 
      129.200,  128.200,  127.000,  125.700,  124.300,  122.900,  121.500,  120.100,  118.800,  117.500, 
      116.200,  114.900,  113.900,  113.000,  112.500,  112.100,  111.800,  111.300,  110.900,  110.500, 
      110.500,  110.200,  110.000,  109.600,  109.400,  109.100,  108.900,  108.700,  108.400,  108.000, 
      107.800,  107.400,  107.200,  107.300,  106.800,  106.600,  106.300,  105.900,  105.800,  105.500, 
      105.100,  104.500,  104.000,  103.400,  103.800,  105.300,  104.600,  103.900,  103.200,  102.400, 
      102.100,  102.200,  101.700,  101.500,  101.600,  101.100,  100.500,   99.500,  101.600,  100.700, 
      99.600,   97.900,   95.600,   95.200,  103.700,  105.200,  106.700,  107.400,  108.300,  107.900, 
      107.200,  106.100,  106.000,  106.600,  105.400,  104.400,  103.200,  101.900,  100.900,  101.000, 
      100.700,  100.900,   99.800,   98.500,   97.000,   96.000,   95.800,   95.500,   94.200,   92.900, 
      92.900,   92.000,   91.700,   91.100,   90.000,   88.900, ],
    [ 144.900,  140.200,  136.700,  135.400,  134.600,  133.800,  133.000,  132.100,  131.300,  130.300, 
      129.200,  128.200,  127.000,  125.700,  124.300,  122.900,  121.600,  120.100,  118.900,  117.600, 
      116.400,  115.100,  114.200,  113.400,  113.100,  112.900,  112.200,  111.600,  111.000,  110.700, 
      110.600,  110.300,  110.000,  109.700,  109.500,  109.200,  109.000,  108.800,  108.500,  108.200, 
      107.900,  107.600,  107.600,  107.400,  107.000,  106.700,  106.400,  106.200,  105.800,  105.500, 
      105.100,  104.700,  104.100,  103.800,  105.500,  105.300,  104.600,  104.200,  103.300,  102.500, 
      102.500,  102.500,  102.100,  101.900,  101.900,  101.300,  100.700,  100.300,  101.700,  100.700, 
      99.600,   98.200,   95.700,   96.200,  104.500,  106.100,  107.100,  107.900,  108.300,  108.100, 
      107.500,  106.500,  106.500,  106.600,  105.600,  104.500,  103.500,  102.200,  101.500,  101.100, 
      101.100,  101.000,  100.100,   98.900,   97.100,   96.100,   95.800,   95.500,   94.300,   93.200, 
      93.000,   92.200,   91.800,   91.600,   90.200,   89.100, ],
    [ 144.900,  140.200,  136.700,  135.400,  134.600,  133.800,  133.000,  132.100,  131.300,  130.300, 
      129.200,  128.200,  127.000,  125.700,  124.400,  122.900,  121.600,  120.200,  119.000,  117.700, 
      116.700,  115.500,  114.700,  114.100,  113.900,  113.300,  112.400,  111.700,  111.200,  110.800, 
      110.600,  110.400,  110.200,  109.800,  109.600,  109.300,  109.200,  108.900,  108.600,  108.400, 
      108.000,  108.000,  107.800,  107.500,  107.100,  106.900,  106.400,  106.300,  106.100,  105.700, 
      105.200,  104.800,  104.100,  104.500,  105.800,  105.500,  104.900,  104.300,  103.300,  102.600, 
      102.800,  102.800,  102.400,  102.200,  101.900,  101.600,  100.700,  100.800,  101.900,  100.900, 
      99.600,   98.200,   95.700,  101.000,  105.200,  106.800,  107.800,  108.900,  108.800,  108.500, 
      107.700,  106.900,  107.100,  106.900,  105.900,  104.700,  103.500,  102.500,  102.000,  101.600, 
      101.500,  101.300,  100.400,   99.000,   97.400,   96.400,   96.200,   95.900,   94.800,   93.800, 
      93.100,   92.900,   92.000,   91.600,   90.300,   89.400, ],
    [ 144.900,  140.200,  136.800,  135.400,  134.600,  133.800,  133.000,  132.200,  131.300,  130.300, 
      129.300,  128.200,  127.000,  125.800,  124.400,  123.100,  121.700,  120.500,  119.400,  118.300, 
      117.000,  116.100,  115.400,  114.500,  113.900,  113.500,  112.400,  111.900,  111.300,  110.900, 
      110.800,  110.500,  110.200,  110.000,  109.700,  109.500,  109.300,  109.000,  108.800,  108.500, 
      108.200,  108.300,  107.900,  107.500,  107.300,  106.900,  106.700,  106.300,  106.100,  105.700, 
      105.400,  104.800,  104.500,  106.100,  105.800,  105.500,  104.900,  104.400,  103.500,  102.900, 
      102.900,  103.100,  102.600,  102.600,  102.200,  101.900,  100.700,  100.800,  102.000,  100.900, 
      99.800,   98.300,   95.700,  101.900,  105.800,  107.500,  108.500,  109.700,  109.400,  108.700, 
      107.800,  107.400,  108.100,  107.200,  106.000,  104.900,  103.800,  102.800,  102.500,  102.000, 
      101.900,  101.700,  100.700,   99.300,   97.700,   96.700,   96.500,   95.900,   94.900,   93.900, 
      93.300,   93.000,   92.200,   91.800,   90.400,   89.600, ],
    [ 144.900,  140.300,  136.800,  135.400,  134.600,  133.800,  133.000,  132.200,  131.300,  130.400, 
      129.300,  128.300,  127.100,  126.000,  124.800,  123.400,  122.300,  120.900,  119.900,  118.900, 
      117.700,  116.500,  115.400,  114.700,  114.100,  113.500,  112.700,  112.000,  111.400,  111.100, 
      110.900,  110.600,  110.300,  110.100,  109.800,  109.600,  109.500,  109.200,  108.900,  108.700, 
      108.700,  108.400,  108.000,  107.700,  107.400,  107.000,  106.700,  106.500,  106.200,  105.700, 
      105.400,  104.800,  105.600,  106.400,  106.000,  105.700,  105.000,  104.400,  103.500,  103.200, 
      103.300,  103.300,  102.900,  102.600,  102.500,  102.100,  101.000,  101.600,  102.000,  101.100, 
      99.800,   98.300,   95.700,  104.300,  106.800,  108.200,  109.300,  109.800,  109.600,  109.000, 
      108.100,  108.100,  108.400,  107.500,  106.300,  105.100,  103.900,  103.100,  102.600,  102.400, 
      102.300,  101.900,  101.000,   99.700,   97.800,   97.000,   97.200,   96.300,   95.000,   94.100, 
      93.500,   93.200,   92.500,   92.600,   90.600,   89.800, ],
    [ 144.900,  140.300,  136.800,  135.400,  134.600,  133.900,  133.100,  132.300,  131.400,  130.500, 
      129.500,  128.600,  127.400,  126.300,  125.000,  123.800,  122.800,  121.400,  120.200,  118.900, 
      117.800,  116.700,  115.600,  114.700,  114.200,  113.600,  112.800,  112.100,  111.500,  111.200, 
      111.000,  110.800,  110.500,  110.200,  110.000,  109.700,  109.500,  109.400,  109.100,  109.000, 
      108.800,  108.500,  108.100,  107.800,  107.400,  107.200,  107.000,  106.700,  106.200,  105.900, 
      105.400,  105.200,  106.800,  106.400,  106.300,  105.800,  105.300,  104.400,  103.700,  103.600, 
      103.600,  103.600,  103.200,  102.900,  102.500,  102.100,  101.200,  103.200,  102.300,  101.100, 
      100.000,   98.400,   95.700,  105.400,  107.500,  108.900,  109.700,  110.300,  110.000,  109.300, 
      108.400,  108.500,  108.600,  107.600,  106.400,  105.300,  104.300,  103.400,  102.800,  102.800, 
      103.000,  102.100,  101.100,   99.700,   98.100,   97.300,   97.300,   96.600,   95.500,   94.300, 
      94.100,   93.800,   93.200,   92.600,   90.700,   90.000, ],
    [ 145.000,  140.300,  136.900,  135.500,  134.700,  134.000,  133.300,  132.500,  131.600,  130.800, 
      129.800,  129.000,  128.000,  126.800,  125.500,  124.100,  122.800,  121.400,  120.400,  119.100, 
      117.800,  116.700,  115.700,  115.000,  114.300,  113.800,  112.900,  112.100,  111.600,  111.200, 
      111.100,  110.800,  110.600,  110.300,  110.100,  109.900,  109.700,  109.500,  109.200,  109.300, 
      108.900,  108.500,  108.200,  107.900,  107.600,  107.200,  107.000,  106.700,  106.300,  105.900, 
      105.500,  106.200,  106.900,  106.600,  106.300,  106.000,  105.400,  104.600,  103.800,  103.700, 
      104.100,  103.900,  103.600,  103.200,  102.700,  102.300,  101.600,  103.300,  102.300,  101.200, 
      100.100,   98.500,   96.100,  106.100,  108.400,  109.700,  111.000,  110.700,  110.100,  109.400, 
      108.900,  109.600,  108.800,  107.800,  106.600,  105.400,  104.600,  103.900,  103.300,  103.100, 
      103.100,  102.300,  101.100,  100.000,   98.400,   97.800,   97.800,   96.900,   95.600,   94.900, 
      94.200,   93.900,   93.300,   92.800,   90.800,   90.400, ],
    [ 145.100,  140.500,  137.000,  135.700,  135.100,  134.300,  133.500,  132.800,  132.200,  131.200, 
      130.200,  129.200,  128.000,  126.800,  125.600,  124.200,  122.900,  121.600,  120.400,  119.200, 
      117.900,  116.900,  115.800,  115.000,  114.400,  113.900,  112.900,  112.300,  111.700,  111.400, 
      111.200,  111.000,  110.700,  110.500,  110.200,  110.000,  109.900,  109.600,  109.700,  109.400, 
      109.000,  108.600,  108.400,  107.900,  107.700,  107.500,  107.100,  106.700,  106.400,  106.000, 
      105.800,  107.500,  107.000,  106.800,  106.400,  106.100,  105.500,  104.700,  104.100,  103.900, 
      104.400,  104.100,  103.600,  103.500,  103.000,  102.300,  102.000,  103.400,  102.500,  101.300, 
      100.100,   98.500,   96.100,  107.100,  109.200,  110.500,  111.500,  111.000,  110.400,  109.500, 
      109.600,  110.000,  109.100,  107.900,  106.800,  105.500,  104.900,  104.400,  103.700,  103.500, 
      103.700,  102.600,  101.500,  100.100,   98.600,   98.000,   97.800,   97.200,   96.000,   95.000, 
      94.800,   94.000,   93.400,   92.900,   91.000,   90.800, ],
    [ 145.200,  140.600,  137.300,  136.200,  135.400,  134.600,  133.800,  133.000,  132.200,  131.200, 
      130.300,  129.300,  128.100,  126.900,  125.600,  124.200,  123.000,  121.700,  120.600,  119.300, 
      118.100,  117.000,  115.900,  115.200,  114.500,  114.000,  113.100,  112.400,  111.800,  111.500, 
      111.300,  111.100,  110.800,  110.600,  110.400,  110.200,  110.000,  110.000,  109.800,  109.500, 
      109.100,  108.800,  108.400,  108.200,  107.700,  107.500,  107.100,  106.700,  106.500,  106.200, 
      107.800,  107.500,  107.200,  106.800,  106.500,  106.200,  105.500,  104.800,  104.300,  104.200, 
      104.600,  104.200,  103.900,  103.500,  103.300,  102.300,  102.400,  103.400,  102.500,  101.400, 
      100.300,   98.600,  105.600,  108.100,  109.800,  111.300,  111.700,  111.300,  110.700,  109.900, 
      110.100,  110.200,  109.200,  108.200,  107.000,  105.700,  105.200,  104.800,  104.000,  104.100, 
      103.800,  102.800,  101.600,  100.100,   98.900,   98.300,   98.200,   97.200,   96.100,   95.100, 
      94.900,   94.600,   93.600,   93.100,   91.800,   91.100, ],
    [ 145.500,  140.900,  137.500,  136.200,  135.400,  134.600,  133.900,  133.100,  132.200,  131.300, 
      130.300,  129.300,  128.200,  127.000,  125.700,  124.400,  123.100,  121.800,  120.600,  119.400, 
      118.200,  117.100,  116.000,  115.200,  114.600,  114.200,  113.200,  112.400,  111.900,  111.600, 
      111.500,  111.200,  111.000,  110.700,  110.500,  110.300,  110.100,  110.200,  109.900,  109.600, 
      109.200,  108.900,  108.600,  108.200,  107.900,  107.600,  107.200,  106.900,  106.500,  106.800, 
      108.100,  107.800,  107.400,  106.900,  106.600,  106.400,  105.700,  104.900,  104.500,  104.700, 
      104.900,  104.400,  104.200,  103.700,  103.500,  102.600,  104.500,  103.400,  102.600,  101.400, 
      100.400,   98.700,  106.700,  109.100,  110.700,  112.100,  111.900,  111.600,  110.700,  110.400, 
      111.100,  110.300,  109.500,  108.300,  107.200,  106.100,  105.400,  105.000,  104.600,  104.200, 
      103.900,  103.100,  101.800,  100.400,   99.100,   98.700,   98.600,   97.600,   96.200,   95.300, 
      95.400,   94.700,   94.300,   93.800,   92.200,   91.300, ],
    [ 145.500,  141.000,  137.600,  136.300,  135.500,  134.700,  133.900,  133.100,  132.300,  131.400, 
      130.400,  129.400,  128.300,  127.100,  125.800,  124.500,  123.200,  121.900,  120.700,  119.500, 
      118.200,  117.100,  116.100,  115.300,  114.700,  114.200,  113.200,  112.500,  112.000,  111.700, 
      111.500,  111.300,  111.100,  110.900,  110.600,  110.400,  110.500,  110.300,  110.000,  109.700, 
      109.400,  108.900,  108.700,  108.400,  108.100,  107.600,  107.300,  106.900,  106.800,  108.500, 
      108.100,  108.000,  107.400,  107.100,  106.900,  106.500,  105.700,  105.000,  104.900,  105.200, 
      105.100,  104.700,  104.400,  103.900,  103.500,  102.700,  104.600,  103.700,  102.600,  101.600, 
      100.400,   98.700,  107.500,  109.900,  111.700,  112.500,  112.200,  111.800,  111.000,  111.100, 
      111.400,  110.600,  109.600,  108.500,  107.200,  106.400,  105.900,  105.400,  105.000,  104.800, 
      104.100,  103.300,  102.100,  100.600,   99.400,   98.900,   98.600,   97.900,   96.700,   95.800, 
      95.400,   94.800,   94.400,   93.900,   92.400,   91.400, ],
    [ 145.500,  141.000,  137.600,  136.300,  135.500,  134.800,  134.000,  133.200,  132.400,  131.400, 
      130.400,  129.400,  128.300,  127.100,  125.800,  124.500,  123.300,  121.900,  120.700,  119.500, 
      118.400,  117.300,  116.200,  115.400,  114.800,  114.300,  113.300,  112.700,  112.200,  111.800, 
      111.700,  111.500,  111.200,  110.900,  110.800,  110.900,  110.700,  110.400,  110.100,  109.800, 
      109.400,  109.200,  108.700,  108.500,  108.100,  107.600,  107.300,  107.000,  107.800,  108.500, 
      108.300,  108.000,  107.600,  107.200,  106.900,  106.600,  105.800,  105.300,  105.000,  105.400, 
      105.400,  105.000,  104.400,  104.200,  103.600,  103.400,  104.700,  103.700,  102.700,  101.600, 
      100.400,   98.700,  108.400,  110.700,  112.700,  112.800,  112.500,  112.000,  111.300,  111.500, 
      111.600,  110.800,  109.800,  108.700,  107.500,  106.700,  106.300,  105.700,  105.300,  105.100, 
      104.400,  103.300,  102.400,  100.700,   99.800,   99.200,   98.900,   97.900,   96.800,   95.900, 
      95.900,   95.400,   94.500,   94.000,   92.700,   91.500, ],
    [ 145.600,  141.100,  137.700,  136.400,  135.600,  134.800,  134.000,  133.300,  132.400,  131.500, 
      130.500,  129.500,  128.400,  127.200,  125.900,  124.600,  123.300,  122.100,  120.900,  119.600, 
      118.400,  117.300,  116.300,  115.500,  114.800,  114.400,  113.500,  112.700,  112.200,  111.900, 
      111.800,  111.600,  111.300,  111.100,  111.100,  111.000,  110.700,  110.500,  110.300,  109.900, 
      109.600,  109.200,  108.900,  108.600,  108.100,  107.800,  107.400,  107.300,  109.000,  108.700, 
      108.500,  108.000,  107.700,  107.400,  107.000,  106.600,  106.000,  105.500,  105.500,  105.600, 
      105.400,  105.300,  104.700,  104.400,  103.600,  103.800,  104.700,  103.900,  102.800,  101.700, 
      100.500,   98.700,  109.900,  111.600,  112.900,  113.200,  112.700,  112.100,  111.800,  112.600, 
      111.800,  111.000,  109.900,  108.900,  107.500,  106.900,  106.700,  106.100,  105.600,  105.400, 
      104.600,  103.600,  102.400,  101.100,  100.000,   99.800,   99.100,   98.300,   96.900,   96.500, 
      96.000,   95.400,   94.700,   94.200,   93.000,   91.500, ],
    [ 145.600,  141.100,  137.700,  136.400,  135.600,  134.900,  134.100,  133.300,  132.500,  131.600, 
      130.500,  129.500,  128.400,  127.200,  126.000,  124.700,  123.400,  122.100,  120.900,  119.600, 
      118.500,  117.400,  116.400,  115.500,  115.000,  114.400,  113.500,  112.900,  112.400,  112.100, 
      111.900,  111.700,  111.500,  111.300,  111.400,  111.100,  110.900,  110.600,  110.400,  109.900, 
      109.700,  109.400,  109.000,  108.600,  108.100,  107.800,  107.600,  109.300,  109.000,  108.700, 
      108.500,  108.200,  107.800,  107.500,  107.100,  106.700,  106.000,  105.900,  106.000,  105.900, 
      105.600,  105.500,  104.900,  104.600,  103.800,  105.700,  104.700,  103.900,  102.900,  101.900, 
      100.600,  108.400,  110.600,  112.300,  113.300,  113.500,  113.100,  112.300,  112.500,  112.700, 
      112.100,  111.100,  110.100,  109.000,  107.900,  107.200,  106.800,  106.600,  105.900,  105.500, 
      104.800,  103.600,  102.400,  101.300,  100.300,   99.800,   99.300,   98.300,   97.100,   96.500, 
      96.100,   95.900,   94.800,   94.400,   93.300,   92.200, ],
    [ 145.700,  141.100,  137.800,  136.500,  135.700,  134.900,  134.100,  133.300,  132.500,  131.600, 
      130.600,  129.600,  128.500,  127.300,  126.000,  124.700,  123.400,  122.200,  121.000,  119.700, 
      118.600,  117.400,  116.400,  115.700,  115.000,  114.600,  113.700,  113.000,  112.500,  112.200, 
      112.100,  111.800,  111.600,  111.700,  111.400,  111.200,  111.000,  110.800,  110.400,  110.200, 
      109.700,  109.400,  109.100,  108.600,  108.300,  107.900,  108.100,  109.400,  109.200,  108.900, 
      108.700,  108.300,  108.000,  107.600,  107.100,  106.700,  106.100,  106.200,  106.200,  106.100, 
      106.100,  105.700,  105.100,  104.600,  104.000,  105.800,  105.000,  104.000,  102.900,  101.900, 
      100.600,  109.500,  111.700,  113.500,  113.800,  113.600,  113.100,  112.800,  112.900,  113.000, 
      112.100,  111.300,  110.300,  109.000,  108.200,  107.600,  107.400,  106.900,  106.500,  105.700, 
      105.100,  103.900,  102.500,  101.500,  100.600,  100.100,   99.600,   98.600,   97.500,   96.900, 
      96.600,   96.000,   95.400,   94.600,   93.500,   92.200, ],
    [ 145.700,  141.100,  137.800,  136.500,  135.700,  135.000,  134.200,  133.400,  132.600,  131.600, 
      130.700,  129.700,  128.500,  127.300,  126.100,  124.800,  123.500,  122.200,  121.000,  119.800, 
      118.600,  117.500,  116.600,  115.700,  115.200,  114.700,  113.700,  113.100,  112.600,  112.400, 
      112.200,  112.000,  112.000,  111.700,  111.500,  111.300,  111.100,  110.800,  110.600,  110.200, 
      109.900,  109.500,  109.100,  108.800,  108.300,  108.200,  109.700,  109.500,  109.400,  109.000, 
      108.700,  108.500,  108.000,  107.600,  107.200,  106.900,  106.500,  106.300,  106.400,  106.300, 
      106.100,  106.000,  105.300,  104.700,  104.300,  105.900,  105.000,  104.100,  103.100,  101.900, 
      100.700,  110.900,  112.600,  113.900,  114.100,  113.900,  113.200,  113.300,  113.900,  113.100, 
      112.300,  111.400,  110.500,  109.200,  108.500,  108.200,  107.800,  107.200,  106.800,  105.900, 
      105.100,  104.100,  102.900,  101.700,  100.800,  100.400,   99.800,   99.000,   97.600,   97.000, 
      96.700,   96.100,   95.500,   94.800,   93.800,   92.300, ],
    [ 145.700,  141.200,  137.800,  136.500,  135.800,  135.000,  134.200,  133.400,  132.600,  131.700, 
      130.700,  129.700,  128.600,  127.400,  126.100,  124.800,  123.600,  122.200,  121.100,  119.900, 
      118.700,  117.600,  116.600,  115.900,  115.200,  114.800,  113.900,  113.200,  112.700,  112.400, 
      112.400,  112.400,  112.100,  111.900,  111.600,  111.500,  111.200,  111.000,  110.600,  110.400, 
      110.000,  109.500,  109.100,  108.800,  108.300,  110.200,  109.700,  109.700,  109.400,  109.200, 
      108.800,  108.600,  108.100,  107.700,  107.300,  107.000,  106.700,  106.800,  106.600,  106.600, 
      106.400,  106.200,  105.600,  104.800,  105.000,  105.900,  105.200,  104.100,  103.100,  102.000, 
      100.700,  111.800,  113.900,  114.100,  114.400,  114.100,  113.600,  113.900,  114.000,  113.400, 
      112.600,  111.600,  110.500,  109.600,  108.700,  108.300,  108.100,  107.600,  107.000,  106.200, 
      105.400,  104.300,  102.900,  102.000,  101.100,  100.800,   99.800,   99.000,   98.100,   97.500, 
      97.000,   96.600,   95.700,   95.300,   94.000,   92.300, ],
    [ 145.800,  141.200,  137.900,  136.600,  135.800,  135.100,  134.300,  133.500,  132.600,  131.700, 
      130.700,  129.700,  128.600,  127.400,  126.200,  124.900,  123.600,  122.300,  121.200,  120.000, 
      118.800,  117.700,  116.700,  115.900,  115.300,  114.900,  114.000,  113.400,  112.900,  112.600, 
      112.700,  112.500,  112.200,  112.000,  111.700,  111.500,  111.200,  111.100,  110.800,  110.400, 
      110.000,  109.600,  109.200,  108.800,  109.100,  110.200,  109.900,  109.700,  109.600,  109.200, 
      109.000,  108.700,  108.200,  107.700,  107.400,  107.100,  106.900,  107.000,  106.900,  106.800, 
      106.600,  106.200,  105.700,  104.900,  106.800,  105.900,  105.200,  104.200,  103.100,  102.000, 
      110.400,  112.900,  114.500,  114.500,  114.600,  114.200,  114.100,  115.000,  114.300,  113.500, 
      112.700,  111.700,  110.600,  109.800,  109.200,  108.700,  108.300,  108.200,  107.200,  106.400, 
      105.400,  104.400,  103.100,  102.200,  101.300,  101.000,  100.300,   99.100,   98.100,   97.600, 
      97.400,   96.700,   95.800,   95.500,   94.000,   92.800, ],
    [ 145.800,  141.300,  137.900,  136.600,  135.900,  135.100,  134.300,  133.500,  132.700,  131.800, 
      130.800,  129.800,  128.700,  127.500,  126.200,  124.900,  123.700,  122.400,  121.300,  120.100, 
      118.900,  117.800,  116.800,  116.000,  115.500,  115.000,  114.100,  113.500,  113.000,  112.700, 
      112.900,  112.600,  112.300,  112.100,  111.900,  111.500,  111.500,  111.100,  110.800,  110.500, 
      110.000,  109.700,  109.300,  109.100,  110.600,  110.300,  110.100,  109.800,  109.600,  109.500, 
      109.100,  108.700,  108.300,  107.900,  107.500,  107.300,  107.200,  107.400,  107.100,  107.000, 
      106.900,  106.400,  105.700,  105.000,  106.900,  106.200,  105.300,  104.300,  103.300,  102.100, 
      112.300,  113.900,  114.700,  114.800,  114.800,  114.400,  114.700,  115.100,  114.500,  113.600, 
      112.800,  111.900,  110.800,  110.100,  109.500,  109.000,  108.800,  108.500,  107.400,  106.600, 
      105.600,  104.400,  103.400,  102.600,  101.900,  101.100,  100.300,   99.400,   98.300,   97.900, 
      97.500,   97.100,   96.000,   95.600,   94.000,   92.800, ],
    [ 145.800,  141.300,  137.900,  136.600,  135.900,  135.100,  134.400,  133.600,  132.700,  131.800, 
      130.800,  129.900,  128.800,  127.600,  126.300,  125.000,  123.800,  122.500,  121.400,  120.100, 
      119.000,  117.900,  116.900,  116.200,  115.600,  115.100,  114.200,  113.600,  113.200,  113.200, 
      113.000,  112.700,  112.400,  112.200,  111.900,  111.800,  111.500,  111.200,  110.900,  110.500, 
      110.200,  109.700,  109.300,  111.000,  110.600,  110.500,  110.100,  109.900,  109.700,  109.500, 
      109.200,  108.800,  108.400,  108.000,  107.600,  107.500,  107.600,  107.600,  107.300,  107.300, 
      107.100,  106.600,  105.800,  105.700,  107.000,  106.200,  105.400,  104.300,  103.300,  102.100, 
      113.200,  114.900,  115.000,  115.000,  114.900,  114.900,  115.800,  115.300,  114.500,  113.800, 
      112.900,  112.000,  111.100,  110.300,  109.800,  109.500,  109.200,  108.700,  107.600,  106.600, 
      105.800,  104.600,  103.600,  102.800,  102.200,  101.300,  100.600,   99.500,   98.700,   98.300, 
      97.800,   97.200,   96.800,   95.800,   94.500,   92.900, ],
    [ 145.800,  141.300,  138.000,  136.700,  135.900,  135.200,  134.400,  133.600,  132.800,  131.900, 
      130.900,  129.900,  128.800,  127.600,  126.400,  125.100,  123.800,  122.600,  121.400,  120.200, 
      119.100,  118.000,  117.000,  116.300,  115.700,  115.200,  114.300,  113.700,  113.500,  113.200, 
      113.000,  112.800,  112.600,  112.300,  112.100,  111.800,  111.600,  111.400,  111.000,  110.500, 
      110.200,  109.800,  111.400,  111.000,  110.800,  110.500,  110.300,  110.000,  109.900,  109.600, 
      109.200,  108.800,  108.400,  108.000,  107.900,  107.900,  108.000,  107.900,  107.700,  107.500, 
      107.300,  106.800,  105.900,  106.000,  107.000,  106.300,  105.400,  104.500,  103.400,  102.200, 
      114.300,  115.300,  115.300,  115.300,  115.100,  115.500,  116.000,  115.500,  114.700,  113.900, 
      113.100,  112.200,  111.400,  110.700,  110.200,  109.700,  109.300,  108.800,  107.800,  106.900, 
      106.000,  104.800,  103.900,  103.000,  102.500,  101.500,  100.900,   99.600,   98.800,   98.600, 
      97.900,   97.300,   96.900,   96.000,   94.800,   93.300, ],
    [ 145.900,  141.400,  138.000,  136.700,  136.000,  135.200,  134.500,  133.700,  132.900,  131.900, 
      131.000,  130.000,  128.900,  127.700,  126.400,  125.200,  123.900,  122.700,  121.500,  120.300, 
      119.200,  118.100,  117.200,  116.400,  115.800,  115.300,  114.500,  114.100,  113.600,  113.300, 
      113.200,  113.000,  112.600,  112.300,  112.100,  112.000,  111.600,  111.400,  111.000,  110.700, 
      110.200,  110.000,  111.400,  111.200,  110.900,  110.600,  110.400,  110.200,  110.000,  109.600, 
      109.300,  108.900,  108.500,  108.100,  108.100,  108.100,  108.200,  108.100,  107.900,  107.700, 
      107.400,  107.000,  106.000,  107.800,  107.200,  106.400,  105.400,  104.500,  103.400,  113.400, 
      115.700,  115.700,  115.500,  115.400,  115.400,  116.000,  116.200,  115.600,  114.900,  114.100, 
      113.200,  112.200,  111.600,  111.300,  110.500,  110.200,  110.000,  109.000,  108.000,  107.100, 
      106.100,  104.800,  104.000,  103.200,  102.700,  101.700,  100.900,  100.000,   99.100,   98.700, 
      98.300,   97.700,   97.300,   96.200,   95.100,   93.300, ],
    [ 145.900,  141.400,  138.100,  136.800,  136.000,  135.300,  134.500,  133.700,  132.900,  132.000, 
      131.000,  130.000,  128.900,  127.800,  126.500,  125.200,  124.000,  122.700,  121.600,  120.400, 
      119.300,  118.200,  117.300,  116.500,  115.900,  115.500,  114.800,  114.200,  113.700,  113.400, 
      113.200,  113.000,  112.700,  112.500,  112.200,  112.000,  111.700,  111.400,  111.100,  110.700, 
      110.400,  111.800,  111.600,  111.400,  111.000,  110.700,  110.500,  110.300,  110.000,  109.700, 
      109.300,  109.100,  108.600,  108.500,  108.300,  108.300,  108.600,  108.300,  108.200,  107.900, 
      107.600,  107.100,  106.200,  107.900,  107.200,  106.400,  105.500,  104.600,  103.400,  115.000, 
      116.000,  116.000,  115.800,  115.500,  116.000,  116.800,  116.400,  115.800,  115.000,  114.200, 
      113.300,  112.600,  111.800,  111.400,  110.900,  110.400,  110.200,  109.200,  108.100,  107.100, 
      106.100,  105.100,  104.200,  103.400,  102.900,  102.000,  101.200,  100.100,   99.400,   99.000, 
      98.700,   97.800,   97.400,   96.200,   95.100,   93.700, ],
    [ 145.900,  141.400,  138.100,  136.800,  136.100,  135.300,  134.600,  133.800,  133.000,  132.000, 
      131.100,  130.100,  129.000,  127.800,  126.600,  125.300,  124.100,  122.800,  121.700,  120.500, 
      119.400,  118.300,  117.300,  116.600,  116.200,  115.800,  115.000,  114.300,  113.800,  113.500, 
      113.400,  113.000,  112.900,  112.500,  112.300,  112.100,  111.800,  111.500,  111.100,  110.700, 
      112.300,  112.000,  111.800,  111.400,  111.100,  110.800,  110.600,  110.400,  110.100,  109.800, 
      109.400,  109.100,  108.800,  108.800,  108.600,  108.900,  108.800,  108.500,  108.400,  108.100, 
      107.800,  107.100,  106.700,  107.900,  107.300,  106.500,  105.500,  104.700,  103.500,  116.300, 
      116.400,  116.200,  115.900,  115.800,  116.500,  117.000,  116.500,  116.000,  115.100,  114.400, 
      113.400,  112.800,  112.200,  111.700,  111.200,  110.700,  110.300,  109.400,  108.400,  107.300, 
      106.300,  105.400,  104.600,  104.000,  103.100,  102.100,  101.200,  100.500,   99.500,   99.100, 
      98.700,   98.300,   97.500,   96.200,   95.100,   93.800, ],
    [ 146.000,  141.500,  138.200,  136.900,  136.100,  135.400,  134.600,  133.800,  133.000,  132.100, 
      131.200,  130.200,  129.100,  127.900,  126.700,  125.400,  124.200,  122.900,  121.800,  120.600, 
      119.500,  118.400,  117.600,  116.900,  116.400,  115.900,  115.000,  114.400,  114.000,  113.700, 
      113.400,  113.300,  112.900,  112.700,  112.400,  112.100,  111.800,  111.500,  111.200,  112.700, 
      112.300,  112.000,  111.800,  111.600,  111.200,  111.000,  110.800,  110.400,  110.200,  109.900, 
      109.500,  109.200,  109.100,  109.000,  109.100,  109.100,  108.900,  108.900,  108.600,  108.300, 
      108.000,  107.100,  108.700,  107.900,  107.300,  106.500,  105.700,  104.700,  115.100,  116.600, 
      116.600,  116.500,  116.000,  116.300,  117.500,  117.200,  116.700,  116.000,  115.300,  114.400, 
      113.700,  113.000,  112.700,  112.100,  111.600,  111.300,  110.400,  109.600,  108.400,  107.500, 
      106.400,  105.500,  104.700,  104.200,  103.200,  102.400,  101.500,  100.500,   99.900,   99.400, 
      99.100,   98.300,   97.700,   96.600,   95.600,   94.300, ],
    [ 146.000,  141.500,  138.200,  136.900,  136.200,  135.400,  134.700,  133.900,  133.100,  132.200, 
      131.200,  130.200,  129.100,  128.000,  126.700,  125.500,  124.200,  123.000,  121.900,  120.800, 
      119.800,  118.800,  117.800,  117.000,  116.500,  116.000,  115.100,  114.500,  114.000,  113.700, 
      113.500,  113.300,  113.000,  112.800,  112.400,  112.100,  111.900,  111.600,  111.400,  112.700, 
      112.400,  112.200,  112.000,  111.600,  111.400,  111.100,  110.800,  110.500,  110.300,  110.000, 
      109.600,  109.500,  109.300,  109.300,  109.300,  109.300,  109.100,  109.100,  108.800,  108.500, 
      108.000,  107.300,  108.800,  108.100,  107.400,  106.700,  105.700,  104.700,  116.900,  117.000, 
      116.900,  116.700,  116.300,  117.000,  117.600,  117.400,  116.800,  116.200,  115.400,  114.600, 
      113.900,  113.400,  112.800,  112.500,  111.900,  111.500,  110.600,  109.700,  108.500,  107.500, 
      106.600,  105.800,  105.100,  104.400,  103.400,  102.400,  101.600,  100.700,  100.200,   99.700, 
      99.200,   98.500,   97.800,   96.900,   95.700,   94.300, ],
    [ 146.000,  141.600,  138.200,  137.000,  136.200,  135.500,  134.700,  134.000,  133.100,  132.200, 
      131.300,  130.300,  129.200,  128.100,  126.800,  125.600,  124.500,  123.300,  122.200,  121.000, 
      119.900,  118.900,  117.900,  117.100,  116.500,  116.100,  115.200,  114.600,  114.100,  113.800, 
      113.600,  113.400,  113.100,  112.800,  112.400,  112.200,  111.900,  111.700,  113.100,  112.900, 
      112.600,  112.200,  112.000,  111.700,  111.500,  111.200,  110.800,  110.500,  110.300,  110.000, 
      109.900,  109.800,  109.500,  109.800,  109.700,  109.600,  109.300,  109.300,  109.000,  108.700, 
      108.100,  107.500,  108.800,  108.200,  107.500,  106.700,  105.800,  104.800,  117.200,  117.200, 
      117.100,  116.700,  116.800,  117.900,  117.800,  117.500,  117.000,  116.300,  115.500,  114.600, 
      114.200,  113.700,  113.300,  112.800,  112.400,  111.600,  110.800,  109.700,  108.700,  107.500, 
      106.800,  105.900,  105.200,  104.700,  103.600,  102.600,  101.700,  101.000,  100.400,  100.000, 
      99.500,   98.600,   98.200,   97.100,   95.700,   95.000, ],
    [ 146.100,  141.600,  138.300,  137.000,  136.300,  135.500,  134.800,  134.000,  133.200,  132.300, 
      131.300,  130.400,  129.400,  128.300,  127.000,  125.800,  124.600,  123.300,  122.200,  121.100, 
      119.900,  118.900,  117.900,  117.200,  116.700,  116.200,  115.300,  114.700,  114.200,  113.900, 
      113.800,  113.500,  113.200,  112.800,  112.600,  112.300,  112.000,  113.500,  113.300,  113.000, 
      112.700,  112.400,  112.000,  111.800,  111.600,  111.200,  110.900,  110.600,  110.400,  110.100, 
      110.100,  110.100,  110.100,  110.000,  109.900,  109.800,  109.700,  109.500,  109.000,  108.800, 
      108.100,  109.700,  108.800,  108.200,  107.500,  106.800,  105.900,  117.500,  117.500,  117.500, 
      117.300,  117.100,  117.500,  118.100,  118.000,  117.600,  117.100,  116.400,  115.500,  115.000, 
      114.400,  114.000,  113.500,  113.000,  112.600,  111.800,  111.000,  109.900,  108.900,  107.800, 
      107.000,  106.300,  105.700,  104.800,  103.800,  102.900,  102.000,  101.300,  100.700,  100.100, 
      99.600,   99.300,   98.400,   97.300,   95.700,   95.000, ],
    [ 146.100,  141.700,  138.300,  137.100,  136.400,  135.600,  134.900,  134.200,  133.300,  132.500, 
      131.500,  130.600,  129.500,  128.400,  127.100,  125.900,  124.600,  123.400,  122.300,  121.100, 
      120.000,  119.000,  118.000,  117.300,  116.700,  116.300,  115.400,  114.700,  114.300,  114.000, 
      113.800,  113.500,  113.200,  112.900,  112.600,  112.300,  113.800,  113.600,  113.400,  113.000, 
      112.800,  112.400,  112.200,  112.000,  111.600,  111.300,  111.000,  110.700,  110.500,  110.400, 
      110.400,  110.300,  110.400,  110.300,  110.000,  110.000,  110.000,  109.700,  109.300,  108.800, 
      108.300,  109.800,  109.000,  108.300,  107.500,  106.800,  105.900,  117.900,  117.800,  117.700, 
      117.400,  117.500,  118.400,  118.300,  118.200,  117.800,  117.200,  116.600,  115.700,  115.200, 
      114.700,  114.300,  113.900,  113.300,  112.800,  111.900,  111.100,  110.100,  108.900,  107.800, 
      107.200,  106.400,  105.900,  104.900,  103.900,  102.900,  102.100,  101.600,  100.800,  100.400, 
      100.000,   99.300,   98.500,   97.400,   96.000,   95.000, ],
    [ 146.100,  141.700,  138.500,  137.200,  136.500,  135.800,  135.000,  134.300,  133.400,  132.600, 
      131.600,  130.600,  129.600,  128.400,  127.200,  125.900,  124.700,  123.500,  122.300,  121.200, 
      120.100,  119.100,  118.100,  117.400,  116.800,  116.400,  115.400,  114.900,  114.300,  114.100, 
      113.900,  113.600,  113.200,  113.000,  112.600,  114.100,  114.000,  113.800,  113.400,  113.200, 
      112.900,  112.600,  112.300,  112.000,  111.700,  111.400,  111.100,  110.800,  110.800,  110.700, 
      110.600,  110.700,  110.600,  110.500,  110.400,  110.200,  110.200,  109.900,  109.500,  108.900, 
      108.400,  109.800,  109.000,  108.400,  107.700,  106.800,  105.900,  118.100,  118.100,  117.900, 
      117.700,  118.200,  118.600,  118.500,  118.200,  117.900,  117.300,  116.600,  116.000,  115.400, 
      115.100,  114.700,  114.200,  113.800,  112.900,  112.100,  111.300,  110.300,  108.900,  108.100, 
      107.300,  106.700,  106.100,  105.200,  104.100,  103.200,  102.400,  101.600,  101.100,  100.700, 
      100.000,   99.700,   98.700,   97.400,   96.000,   95.700, ],
    [ 146.300,  141.900,  138.600,  137.300,  136.600,  135.800,  135.000,  134.300,  133.500,  132.600, 
      131.600,  130.700,  129.600,  128.500,  127.200,  126.000,  124.800,  123.500,  122.400,  121.300, 
      120.200,  119.200,  118.200,  117.400,  116.800,  116.400,  115.600,  114.900,  114.500,  114.100, 
      113.900,  113.600,  113.300,  113.000,  114.400,  114.100,  114.100,  113.800,  113.600,  113.300, 
      113.000,  112.700,  112.400,  112.000,  111.700,  111.400,  111.100,  111.000,  111.100,  110.900, 
      111.000,  111.100,  110.900,  110.700,  110.600,  110.600,  110.300,  110.000,  109.600,  108.900, 
      110.600,  109.800,  109.100,  108.400,  107.700,  107.000,  118.500,  118.400,  118.300,  118.000, 
      118.200,  119.100,  118.800,  118.600,  118.400,  118.000,  117.500,  116.700,  116.200,  115.800, 
      115.300,  115.100,  114.600,  114.100,  113.100,  112.300,  111.300,  110.400,  109.100,  108.200, 
      107.600,  106.800,  106.300,  105.400,  104.300,  103.300,  102.500,  102.100,  101.400,  100.800, 
      100.400,   99.800,   98.800,   97.800,   96.200,   95.800, ],
    [ 146.300,  141.900,  138.600,  137.300,  136.600,  135.800,  135.100,  134.300,  133.500,  132.700, 
      131.700,  130.700,  129.700,  128.500,  127.300,  126.000,  124.900,  123.600,  122.500,  121.400, 
      120.200,  119.200,  118.200,  117.600,  117.000,  116.600,  115.600,  115.000,  114.600,  114.200, 
      113.900,  113.700,  113.300,  114.800,  114.400,  114.300,  114.100,  114.000,  113.700,  113.500, 
      113.100,  112.800,  112.400,  112.100,  111.800,  111.500,  111.200,  111.200,  111.300,  111.400, 
      111.300,  111.300,  111.100,  111.000,  110.700,  110.800,  110.500,  110.200,  109.600,  109.100, 
      110.700,  110.000,  109.200,  108.400,  107.800,  107.000,  118.800,  118.700,  118.500,  118.300, 
      118.800,  119.300,  119.000,  118.700,  118.500,  118.200,  117.500,  117.000,  116.400,  116.200, 
      115.700,  115.400,  115.000,  114.200,  113.300,  112.400,  111.500,  110.400,  109.200,  108.400, 
      107.900,  107.300,  106.500,  105.400,  104.400,  103.300,  102.800,  102.400,  101.600,  101.100, 
      100.500,   99.900,   98.800,   98.000,   96.400,   95.800, ],
    [ 146.300,  141.900,  138.600,  137.400,  136.600,  135.900,  135.100,  134.400,  133.600,  132.700, 
      131.700,  130.800,  129.700,  128.600,  127.400,  126.100,  124.900,  123.700,  122.500,  121.400, 
      120.200,  119.300,  118.400,  117.600,  117.000,  116.600,  115.800,  115.100,  114.600,  114.200, 
      114.000,  113.700,  113.400,  114.800,  114.600,  114.400,  114.200,  114.000,  113.700,  113.500, 
      113.200,  112.800,  112.500,  112.200,  111.900,  111.600,  111.500,  111.500,  111.500,  111.700, 
      111.700,  111.400,  111.300,  111.200,  111.200,  111.000,  110.700,  110.400,  109.700,  109.200, 
      110.700,  110.100,  109.300,  108.500,  107.800,  119.100,  119.000,  119.000,  118.600,  118.700, 
      119.700,  119.500,  119.100,  118.900,  118.600,  118.300,  117.600,  117.200,  116.800,  116.300, 
      116.000,  115.600,  115.300,  114.300,  113.500,  112.600,  111.600,  110.500,  109.400,  108.700, 
      108.000,  107.500,  106.700,  105.700,  104.600,  103.600,  103.100,  102.600,  101.900,  101.400, 
      100.600,  100.000,   98.800,   98.000,   96.500,   96.400, ],
    [ 146.400,  141.900,  138.700,  137.400,  136.700,  136.000,  135.200,  134.400,  133.600,  132.700, 
      131.800,  130.900,  129.800,  128.600,  127.400,  126.100,  124.900,  123.700,  122.600,  121.500, 
      120.400,  119.400,  118.400,  117.600,  117.200,  116.700,  115.800,  115.100,  114.600,  114.300, 
      114.000,  113.700,  115.100,  114.900,  114.700,  114.500,  114.300,  114.000,  113.800,  113.600, 
      113.200,  112.900,  112.600,  112.300,  112.000,  111.800,  111.800,  111.700,  112.000,  111.900, 
      111.800,  111.700,  111.400,  111.400,  111.300,  111.100,  110.800,  110.500,  109.700,  111.300, 
      110.900,  110.100,  109.300,  108.500,  107.900,  119.400,  119.300,  119.100,  118.900,  119.400, 
      120.000,  119.600,  119.200,  119.000,  118.800,  118.300,  117.900,  117.400,  117.200,  116.700, 
      116.400,  115.800,  115.400,  114.500,  113.600,  112.600,  111.800,  110.600,  109.600,  108.800, 
      108.300,  107.800,  106.800,  105.900,  104.800,  103.700,  103.300,  102.700,  102.000,  101.400, 
      101.200,  100.400,   99.300,   98.000,   96.700,   96.700, ],
    [ 146.400,  142.000,  138.700,  137.500,  136.700,  136.000,  135.300,  134.500,  133.700,  132.800, 
      131.800,  130.900,  129.800,  128.600,  127.500,  126.300,  125.100,  123.800,  122.700,  121.500, 
      120.400,  119.500,  118.500,  117.800,  117.200,  116.800,  115.900,  115.200,  114.700,  114.300, 
      114.100,  115.500,  115.200,  115.100,  114.800,  114.600,  114.400,  114.200,  114.000,  113.600, 
      113.300,  113.000,  112.600,  112.400,  112.200,  112.200,  112.000,  112.300,  112.300,  112.200, 
      112.000,  111.900,  111.800,  111.600,  111.500,  111.300,  111.000,  110.600,  109.900,  111.400, 
      110.900,  110.200,  109.300,  108.600,  108.000,  119.700,  119.600,  119.200,  119.500,  120.400, 
      120.100,  119.800,  119.300,  119.100,  118.900,  118.400,  118.100,  117.800,  117.300,  117.100, 
      116.600,  116.300,  115.500,  114.700,  113.600,  112.700,  111.800,  110.700,  109.800,  109.100, 
      108.700,  108.000,  107.000,  106.100,  104.900,  104.000,  103.400,  102.900,  102.400,  101.700, 
      101.300,  100.500,   99.500,   98.400,   96.900,   96.700, ],
    [ 146.400,  142.000,  138.800,  137.500,  136.800,  136.000,  135.300,  134.500,  133.700,  132.800, 
      131.900,  131.000,  129.900,  128.700,  127.500,  126.300,  125.100,  123.800,  122.800,  121.600, 
      120.500,  119.500,  118.500,  117.800,  117.300,  116.800,  115.900,  115.300,  114.700,  114.300, 
      115.800,  115.600,  115.400,  115.100,  114.900,  114.700,  114.500,  114.300,  114.000,  113.700, 
      113.300,  113.000,  112.700,  112.500,  112.400,  112.400,  112.500,  112.400,  112.400,  112.400, 
      112.300,  112.100,  112.000,  111.800,  111.700,  111.500,  111.100,  110.600,  110.000,  111.400, 
      111.000,  110.300,  109.400,  108.700,  120.000,  119.900,  119.700,  119.500,  120.800,  120.500, 
      120.200,  120.000,  119.400,  119.200,  118.900,  118.700,  118.300,  118.200,  117.700,  117.400, 
      116.900,  116.500,  115.600,  114.900,  113.800,  112.900,  111.800,  110.900,  109.900,  109.300, 
      108.900,  108.100,  107.100,  106.100,  105.000,  104.100,  103.800,  103.200,  102.500,  101.800, 
      101.600,  100.600,   99.600,   98.400,   96.900,   96.800, ],
    [ 146.400,  142.000,  138.800,  137.500,  136.800,  136.100,  135.300,  134.600,  133.800,  132.900, 
      131.900,  131.000,  129.900,  128.700,  127.600,  126.400,  125.200,  123.900,  122.800,  121.600, 
      120.600,  119.600,  118.600,  117.900,  117.300,  116.800,  116.000,  115.300,  114.700,  116.000, 
      115.900,  115.700,  115.400,  115.200,  115.000,  114.800,  114.600,  114.300,  114.000,  113.700, 
      113.400,  113.100,  112.800,  112.800,  112.700,  112.800,  112.800,  112.700,  112.700,  112.500, 
      112.500,  112.500,  112.200,  111.900,  111.800,  111.600,  111.300,  110.600,  112.100,  111.600, 
      111.000,  110.300,  109.400,  108.800,  120.300,  120.200,  119.800,  120.300,  120.900,  120.700, 
      120.400,  120.000,  119.600,  119.300,  119.100,  118.900,  118.600,  118.300,  117.900,  117.600, 
      117.400,  116.600,  115.800,  114.900,  113.900,  113.000,  112.100,  111.100,  110.100,  109.500, 
      109.100,  108.200,  107.300,  106.300,  105.200,  104.300,  104.000,  103.400,  102.700,  102.100, 
      101.600,  100.800,   99.600,   98.400,   97.200,   97.400, ],
    [ 146.500,  142.100,  138.800,  137.600,  136.900,  136.100,  135.400,  134.600,  133.800,  132.900, 
      132.000,  131.100,  130.000,  128.800,  127.600,  126.400,  125.200,  124.000,  122.900,  121.700, 
      120.600,  119.600,  118.600,  117.900,  117.300,  116.900,  116.000,  115.300,  116.400,  116.200, 
      116.000,  115.700,  115.600,  115.300,  115.100,  114.900,  114.700,  114.300,  114.100,  113.800, 
      113.500,  113.200,  113.100,  113.000,  112.900,  113.100,  113.100,  112.900,  112.900,  112.900, 
      112.700,  112.600,  112.300,  112.200,  112.000,  111.900,  111.300,  110.700,  112.100,  111.600, 
      111.100,  110.300,  109.500,  120.600,  120.500,  120.300,  120.100,  121.400,  121.100,  120.800, 
      120.600,  120.100,  119.700,  119.300,  119.300,  119.100,  119.000,  118.700,  118.200,  117.900, 
      117.500,  116.700,  116.000,  115.100,  114.100,  113.000,  112.100,  111.300,  110.400,  109.900, 
      109.200,  108.400,  107.400,  106.500,  105.300,  104.700,  104.200,  103.600,  103.000,  102.200, 
      101.700,  100.800,   99.600,   98.600,   97.700,   97.700, ],
    [ 146.500,  142.100,  138.800,  137.600,  136.900,  136.200,  135.400,  134.700,  133.900,  133.000, 
      132.000,  131.100,  130.000,  128.900,  127.700,  126.400,  125.200,  124.000,  122.900,  121.700, 
      120.600,  119.600,  118.700,  118.000,  117.400,  116.900,  116.000,  116.900,  116.500,  116.300, 
      116.000,  115.900,  115.600,  115.500,  115.200,  115.000,  114.700,  114.400,  114.200,  113.900, 
      113.600,  113.400,  113.300,  113.300,  113.300,  113.300,  113.200,  113.200,  113.100,  113.100, 
      112.900,  112.800,  112.500,  112.200,  112.100,  112.000,  111.300,  112.800,  112.100,  111.700, 
      111.100,  110.400,  109.500,  120.800,  120.800,  120.400,  120.900,  121.500,  121.300,  120.900, 
      120.600,  120.300,  119.800,  119.500,  119.500,  119.400,  119.100,  119.000,  118.500,  118.300, 
      117.600,  116.900,  116.100,  115.200,  114.200,  113.100,  112.300,  111.400,  110.600,  110.100, 
      109.400,  108.500,  107.600,  106.500,  105.500,  104.800,  104.300,  103.900,  103.100,  102.400, 
      102.100,  100.800,  100.100,   98.800,   97.700,   98.000, ],
    [ 146.500,  142.200,  138.900,  137.700,  136.900,  136.200,  135.400,  134.700,  133.900,  133.000, 
      132.100,  131.100,  130.000,  128.900,  127.700,  126.400,  125.200,  124.000,  122.900,  121.700, 
      120.700,  119.700,  118.700,  118.000,  117.400,  117.000,  117.600,  117.100,  116.700,  116.300, 
      116.200,  116.000,  115.800,  115.500,  115.300,  115.000,  114.800,  114.500,  114.300,  114.000, 
      113.800,  113.700,  113.500,  113.700,  113.600,  113.600,  113.500,  113.400,  113.400,  113.200, 
      113.200,  112.900,  112.700,  112.500,  112.300,  112.000,  111.400,  112.900,  112.300,  111.700, 
      111.100,  110.400,  121.100,  121.100,  120.800,  120.800,  121.900,  121.700,  121.400,  121.100, 
      120.700,  120.400,  119.800,  119.800,  119.700,  119.800,  119.500,  119.300,  118.800,  118.500, 
      117.800,  117.100,  116.200,  115.200,  114.200,  113.300,  112.500,  111.500,  110.700,  110.300, 
      109.500,  108.700,  107.800,  106.600,  105.600,  105.000,  104.700,  104.100,  103.400,  102.800, 
      102.200,  101.200,  100.100,   98.800,   98.200,   98.000, ],
    [ 146.600,  142.200,  138.900,  137.700,  137.000,  136.200,  135.500,  134.700,  133.900,  133.000, 
      132.100,  131.100,  130.000,  128.900,  127.700,  126.400,  125.300,  124.100,  123.000,  121.800, 
      120.700,  119.700,  118.800,  118.000,  118.900,  118.500,  117.700,  117.200,  116.700,  116.400, 
      116.300,  116.100,  115.800,  115.600,  115.300,  115.000,  114.800,  114.600,  114.300,  114.200, 
      114.100,  113.900,  114.100,  114.000,  113.900,  113.700,  113.700,  113.700,  113.600,  113.400, 
      113.300,  113.100,  112.900,  112.700,  112.400,  112.100,  111.600,  112.900,  112.400,  111.800, 
      111.200,  110.500,  121.400,  121.300,  121.000,  121.600,  122.100,  121.800,  121.500,  121.200, 
      120.900,  120.500,  119.900,  120.000,  120.000,  119.900,  119.800,  119.400,  119.300,  118.600, 
      117.900,  117.200,  116.200,  115.300,  114.200,  113.300,  112.600,  111.800,  111.200,  110.400, 
      109.800,  108.900,  107.800,  106.800,  105.900,  105.400,  104.900,  104.200,  103.400,  103.100, 
      102.300,  101.400,  100.200,   98.900,   98.200,   98.800, ],
    [ 146.600,  142.200,  138.900,  137.700,  137.000,  136.200,  135.500,  134.700,  133.900,  133.000, 
      132.100,  131.100,  130.100,  129.000,  127.800,  126.500,  125.300,  124.100,  123.000,  121.800, 
      120.700,  121.100,  120.200,  119.500,  119.000,  118.600,  117.800,  117.200,  116.800,  116.500, 
      116.400,  116.200,  115.900,  115.600,  115.300,  115.100,  114.900,  114.700,  114.500,  114.300, 
      114.300,  114.300,  114.400,  114.200,  114.100,  114.000,  113.800,  113.900,  113.700,  113.700, 
      113.500,  113.200,  113.100,  112.800,  112.500,  112.100,  113.600,  113.000,  112.400,  111.800, 
      111.200,  121.800,  121.700,  121.300,  121.500,  122.500,  122.300,  122.000,  121.600,  121.300, 
      121.000,  120.600,  120.200,  120.100,  120.400,  120.200,  120.100,  119.700,  119.400,  118.700, 
      118.100,  117.200,  116.300,  115.400,  114.400,  113.600,  112.700,  112.000,  111.300,  110.600, 
      109.800,  109.000,  108.000,  106.900,  106.200,  105.600,  105.100,  104.400,  103.700,  103.200, 
      102.400,  101.500,  100.500,   99.100,   98.800,   99.000, ],
    [ 146.600,  142.200,  139.000,  137.700,  137.000,  136.200,  135.500,  134.700,  134.000,  133.100, 
      132.100,  131.200,  130.100,  129.000,  127.800,  126.500,  125.300,  124.100,  124.200,  123.100, 
      122.000,  121.200,  120.300,  119.600,  119.100,  118.700,  117.800,  117.300,  116.900,  116.700, 
      116.500,  116.200,  115.900,  115.700,  115.400,  115.100,  114.900,  114.800,  114.700,  114.700, 
      114.700,  114.600,  114.500,  114.400,  114.200,  114.200,  114.200,  114.000,  113.900,  113.800, 
      113.600,  113.500,  113.200,  112.900,  112.500,  112.300,  113.600,  113.000,  112.500,  111.800, 
      111.300,  122.000,  121.800,  121.600,  122.900,  122.600,  122.400,  122.000,  121.700,  121.400, 
      121.000,  120.600,  120.400,  120.500,  120.500,  120.700,  120.400,  120.200,  119.500,  118.900, 
      118.200,  117.300,  116.500,  115.500,  114.500,  113.600,  113.100,  112.200,  111.500,  110.700, 
      110.000,  109.200,  108.100,  107.100,  106.300,  105.600,  105.200,  104.600,  103.800,  103.300, 
      102.500,  101.500,  100.500,   99.100,   98.900,   99.200, ],
    [ 146.600,  142.200,  139.000,  137.800,  137.000,  136.300,  135.500,  134.800,  134.000,  133.100, 
      132.100,  131.200,  130.100,  129.000,  127.800,  127.600,  126.400,  125.200,  124.300,  123.200, 
      122.100,  121.300,  120.400,  119.700,  119.100,  118.700,  118.000,  117.400,  117.000,  116.700, 
      116.500,  116.200,  116.000,  115.700,  115.400,  115.200,  115.000,  115.000,  115.000,  115.100, 
      115.000,  114.900,  114.800,  114.600,  114.500,  114.500,  114.300,  114.200,  114.200,  114.000, 
      113.900,  113.600,  113.300,  112.900,  112.600,  114.200,  113.600,  113.100,  112.500,  111.900, 
      111.400,  122.300,  121.800,  122.300,  123.000,  122.900,  122.500,  122.200,  121.800,  121.500, 
      121.200,  120.900,  120.600,  120.800,  121.000,  120.900,  120.800,  120.300,  119.600,  119.000, 
      118.200,  117.400,  116.600,  115.600,  114.600,  113.800,  113.200,  112.600,  111.600,  110.900, 
      110.200,  109.200,  108.300,  107.400,  106.700,  106.000,  105.500,  104.900,  104.300,  103.600, 
      102.500,  101.800,  100.600,   99.500,   98.900,   99.400, ],
    [ 146.600,  142.200,  139.000,  137.800,  137.000,  136.300,  135.500,  134.800,  134.000,  133.100, 
      133.000,  132.100,  131.100,  130.000,  128.900,  127.700,  126.500,  125.300,  124.400,  123.300, 
      122.300,  121.300,  120.400,  119.700,  119.300,  118.900,  118.000,  117.500,  117.100,  116.800, 
      116.500,  116.300,  116.100,  115.800,  115.500,  115.300,  115.300,  115.400,  115.400,  115.300, 
      115.200,  115.100,  114.900,  114.900,  114.700,  114.700,  114.500,  114.300,  114.300,  114.100, 
      114.000,  113.600,  113.300,  112.900,  112.700,  114.200,  113.700,  113.100,  112.500,  111.900, 
      122.700,  122.400,  122.100,  123.300,  123.200,  123.000,  122.700,  122.300,  122.000,  121.500, 
      121.200,  121.100,  121.100,  121.100,  121.200,  121.200,  121.100,  120.400,  119.800,  119.100, 
      118.300,  117.500,  116.600,  115.700,  114.800,  114.000,  113.400,  112.700,  111.800,  110.900, 
      110.200,  109.400,  108.400,  107.600,  106.800,  106.200,  105.600,  104.900,  104.300,  103.700, 
      102.800,  101.900,  100.600,   99.900,   99.500,   99.400, ],
    [ 146.600,  142.300,  139.000,  137.800,  137.000,  136.300,  136.300,  135.600,  134.800,  133.900, 
      133.100,  132.200,  131.100,  130.000,  128.900,  127.700,  126.600,  125.400,  124.400,  123.300, 
      122.300,  121.400,  120.500,  119.800,  119.300,  118.900,  118.200,  117.600,  117.200,  116.800, 
      116.600,  116.300,  116.100,  115.800,  115.600,  115.600,  115.700,  115.600,  115.700,  115.600, 
      115.400,  115.300,  115.200,  115.000,  115.000,  114.800,  114.600,  114.500,  114.400,  114.200, 
      114.000,  113.700,  113.300,  113.100,  114.600,  114.200,  113.800,  113.200,  112.500,  112.000, 
      122.800,  122.500,  123.800,  123.500,  123.300,  123.000,  122.700,  122.400,  122.100,  121.700, 
      121.500,  121.500,  121.300,  121.400,  121.500,  121.500,  121.100,  120.500,  120.000,  119.200, 
      118.400,  117.700,  116.600,  115.800,  115.000,  114.200,  113.500,  113.000,  111.900,  111.200, 
      110.300,  109.400,  108.600,  107.800,  107.000,  106.400,  105.800,  105.200,  104.600,  103.800, 
      103.000,  101.900,  100.700,   99.900,   99.800,   99.400, ],
    [ 147.200,  142.900,  139.700,  138.500,  137.800,  137.100,  136.400,  135.600,  134.900,  134.000, 
      133.200,  132.200,  131.200,  130.100,  128.900,  127.800,  126.600,  125.500,  124.500,  123.400, 
      122.400,  121.400,  120.600,  119.900,  119.400,  119.000,  118.200,  117.700,  117.200,  116.900, 
      116.600,  116.400,  116.200,  115.900,  115.900,  116.000,  115.800,  116.000,  116.000,  115.800, 
      115.700,  115.400,  115.400,  115.200,  115.100,  115.000,  114.800,  114.700,  114.600,  114.400, 
      114.100,  113.700,  113.400,  114.900,  114.600,  114.400,  113.800,  113.300,  112.600,  123.200, 
      122.900,  123.000,  123.900,  123.700,  123.500,  123.200,  122.800,  122.500,  122.100,  121.900, 
      121.700,  121.800,  121.600,  121.600,  121.800,  121.800,  121.200,  120.700,  120.000,  119.300, 
      118.600,  117.700,  116.800,  115.900,  115.000,  114.400,  113.900,  113.100,  112.100,  111.300, 
      110.500,  109.500,  108.600,  108.000,  107.200,  106.600,  106.000,  105.300,  104.700,  103.900, 
      103.100,  102.200,  100.900,  100.300,   99.900,   99.800, ],
    [ 147.200,  142.900,  139.800,  138.500,  137.800,  137.200,  136.400,  135.700,  134.900,  134.100, 
      133.200,  132.200,  131.200,  130.100,  129.000,  127.800,  126.700,  125.500,  124.500,  123.500, 
      122.400,  121.500,  120.700,  120.000,  119.500,  119.100,  118.300,  117.700,  117.200,  116.900, 
      116.700,  116.500,  116.200,  116.200,  116.100,  116.100,  116.300,  116.300,  116.100,  116.000, 
      115.800,  115.700,  115.600,  115.500,  115.300,  115.100,  115.000,  114.900,  114.700,  114.400, 
      114.100,  113.800,  115.300,  114.900,  114.700,  114.400,  113.900,  113.300,  112.600,  123.300, 
      123.200,  124.300,  124.100,  123.900,  123.600,  123.300,  123.000,  122.600,  122.200,  122.100, 
      122.100,  121.900,  121.900,  121.900,  122.200,  121.800,  121.300,  120.800,  120.200,  119.400, 
      118.700,  117.700,  116.900,  116.100,  115.300,  114.600,  114.100,  113.200,  112.100,  111.400, 
      110.600,  109.700,  108.800,  108.200,  107.400,  106.800,  106.300,  105.400,  104.800,  104.000, 
      103.100,  102.200,  100.900,  100.300,  100.600,   99.900, ],
    [ 147.200,  143.000,  139.800,  138.600,  137.900,  137.200,  136.400,  135.700,  135.000,  134.100, 
      133.200,  132.300,  131.300,  130.200,  129.000,  127.900,  126.800,  125.600,  124.600,  123.600, 
      122.500,  121.600,  120.800,  120.100,  119.600,  119.200,  118.300,  117.700,  117.200,  116.900, 
      116.800,  116.600,  116.500,  116.400,  116.400,  116.600,  116.600,  116.400,  116.400,  116.200, 
      116.100,  115.900,  115.800,  115.600,  115.400,  115.300,  115.200,  115.000,  114.700,  114.400, 
      114.200,  115.700,  115.300,  115.100,  114.700,  114.500,  113.900,  113.400,  123.700,  123.300, 
      123.900,  124.500,  124.200,  123.900,  123.700,  123.400,  123.100,  122.700,  122.500,  122.300, 
      122.400,  122.300,  122.200,  122.200,  122.300,  121.900,  121.500,  120.900,  120.200,  119.500, 
      118.700,  117.900,  117.100,  116.200,  115.400,  114.900,  114.200,  113.300,  112.300,  111.600, 
      110.700,  109.900,  109.200,  108.300,  107.600,  106.900,  106.300,  105.800,  105.100,  104.000, 
      103.100,  102.200,  101.000,  100.800,  100.800,   99.900, ],
    [ 147.200,  143.000,  139.800,  138.600,  137.900,  137.200,  136.500,  135.800,  135.000,  134.200, 
      133.300,  132.400,  131.300,  130.200,  129.100,  128.000,  126.900,  125.700,  124.700,  123.600, 
      122.600,  121.700,  120.800,  120.100,  119.600,  119.200,  118.400,  117.800,  117.300,  117.000, 
      116.900,  116.800,  116.700,  116.700,  116.900,  116.900,  116.700,  116.700,  116.500,  116.500, 
      116.300,  116.000,  116.000,  115.800,  115.700,  115.500,  115.300,  115.000,  114.800,  114.500, 
      116.000,  115.700,  115.400,  115.100,  114.800,  114.500,  114.000,  124.000,  123.700,  123.800, 
      124.900,  124.600,  124.300,  124.100,  123.800,  123.500,  123.100,  122.700,  122.700,  122.600, 
      122.600,  122.700,  122.500,  122.600,  122.400,  122.100,  121.700,  121.000,  120.300,  119.600, 
      118.700,  117.900,  117.200,  116.400,  115.600,  115.100,  114.300,  113.400,  112.500,  111.600, 
      110.700,  109.900,  109.200,  108.500,  107.800,  107.100,  106.600,  106.000,  105.200,  104.300, 
      103.500,  102.300,  101.400,  100.800,  100.900,  100.200, ],
    [ 147.300,  143.000,  139.900,  138.700,  138.000,  137.300,  136.500,  135.800,  135.000,  134.200, 
      133.300,  132.400,  131.400,  130.300,  129.200,  128.000,  126.900,  125.700,  124.700,  123.700, 
      122.600,  121.700,  120.800,  120.100,  119.700,  119.200,  118.500,  117.800,  117.400,  117.100, 
      117.100,  117.000,  117.000,  117.200,  117.100,  117.000,  117.000,  116.800,  116.800,  116.600, 
      116.400,  116.300,  116.100,  115.900,  115.800,  115.600,  115.300,  115.000,  114.800,  116.300, 
      116.000,  115.800,  115.400,  115.200,  114.900,  114.600,  114.000,  124.200,  124.000,  125.200, 
      125.000,  124.700,  124.500,  124.100,  123.900,  123.600,  123.200,  123.100,  123.000,  122.900, 
      122.900,  122.900,  122.900,  122.700,  122.500,  122.200,  121.700,  121.000,  120.400,  119.600, 
      118.900,  118.100,  117.300,  116.600,  116.000,  115.300,  114.400,  113.600,  112.600,  111.800, 
      110.900,  110.100,  109.500,  108.700,  108.000,  107.300,  106.600,  106.100,  105.300,  104.500, 
      103.500,  102.400,  101.600,  101.300,  101.000,  100.200, ],
    [ 147.300,  143.000,  139.900,  138.700,  138.000,  137.300,  136.600,  135.900,  135.100,  134.300, 
      133.400,  132.500,  131.400,  130.400,  129.200,  128.100,  126.900,  125.800,  124.700,  123.700, 
      122.600,  121.700,  120.900,  120.200,  119.700,  119.300,  118.500,  117.900,  117.500,  117.300, 
      117.300,  117.300,  117.500,  117.400,  117.300,  117.300,  117.100,  117.100,  117.000,  116.800, 
      116.700,  116.400,  116.200,  116.100,  115.900,  115.600,  115.300,  115.000,  116.600,  116.300, 
      116.200,  115.800,  115.500,  115.200,  114.900,  114.600,  124.600,  124.200,  124.700,  125.400, 
      125.200,  124.900,  124.500,  124.200,  124.000,  123.600,  123.400,  123.200,  123.400,  123.300, 
      123.200,  123.200,  123.100,  122.800,  122.700,  122.300,  121.800,  121.100,  120.500,  119.700, 
      118.900,  118.300,  117.500,  116.800,  116.100,  115.300,  114.600,  113.600,  112.700,  111.800, 
      111.100,  110.500,  109.700,  109.000,  108.100,  107.500,  107.100,  106.200,  105.300,  104.500, 
      103.700,  102.500,  101.600,  101.600,  101.000,  100.400, ],
    [ 147.400,  143.100,  140.000,  138.800,  138.100,  137.400,  136.600,  135.900,  135.100,  134.300, 
      133.400,  132.500,  131.500,  130.400,  129.200,  128.100,  126.900,  125.800,  124.800,  123.700, 
      122.700,  121.800,  120.900,  120.300,  119.700,  119.300,  118.500,  118.000,  117.700,  117.600, 
      117.600,  117.800,  117.700,  117.600,  117.500,  117.400,  117.400,  117.300,  117.100,  117.000, 
      116.800,  116.500,  116.500,  116.200,  116.000,  115.600,  115.300,  116.900,  116.700,  116.500, 
      116.200,  115.900,  115.500,  115.200,  114.900,  114.600,  124.600,  124.700,  125.700,  125.500, 
      125.300,  125.000,  124.600,  124.400,  124.000,  123.700,  123.600,  123.500,  123.400,  123.500, 
      123.500,  123.600,  123.200,  122.900,  122.800,  122.400,  121.900,  121.300,  120.500,  119.800, 
      119.100,  118.400,  117.700,  117.100,  116.300,  115.500,  114.600,  113.800,  112.800,  112.000, 
      111.200,  110.500,  109.800,  109.100,  108.300,  107.600,  107.100,  106.500,  105.300,  104.500, 
      103.700,  102.600,  102.000,  102.200,  101.400,  100.400, ],
    [ 147.400,  143.100,  140.000,  138.800,  138.100,  137.400,  136.700,  135.900,  135.200,  134.300, 
      133.400,  132.500,  131.500,  130.400,  129.300,  128.100,  127.000,  125.800,  124.800,  123.800, 
      122.800,  121.800,  120.900,  120.300,  119.800,  119.400,  118.600,  118.200,  118.100,  117.800, 
      118.100,  118.100,  117.900,  117.800,  117.600,  117.700,  117.600,  117.400,  117.300,  117.100, 
      116.900,  116.800,  116.600,  116.300,  116.000,  115.700,  117.200,  116.900,  116.700,  116.500, 
      116.300,  115.900,  115.600,  115.300,  115.000,  125.000,  124.800,  126.000,  125.800,  125.700, 
      125.500,  125.100,  124.700,  124.500,  124.200,  124.000,  123.800,  123.900,  123.800,  123.800, 
      123.700,  123.700,  123.300,  123.100,  122.900,  122.500,  122.000,  121.300,  120.500,  119.900, 
      119.200,  118.500,  117.900,  117.200,  116.400,  115.700,  114.800,  113.900,  112.900,  112.000, 
      111.400,  110.800,  110.000,  109.300,  108.500,  107.800,  107.300,  106.500,  105.600,  104.800, 
      103.700,  102.700,  102.400,  102.300,  101.500,  100.500, ],
    [ 147.400,  143.200,  140.000,  138.800,  138.100,  137.400,  136.700,  136.000,  135.200,  134.400, 
      133.500,  132.600,  131.500,  130.400,  129.300,  128.200,  127.000,  125.900,  124.900,  123.800, 
      122.800,  121.900,  121.000,  120.400,  119.900,  119.500,  118.800,  118.600,  118.300,  118.300, 
      118.300,  118.200,  118.100,  117.900,  117.900,  117.800,  117.700,  117.600,  117.500,  117.200, 
      117.100,  116.900,  116.600,  116.300,  116.000,  117.500,  117.200,  116.900,  116.800,  116.600, 
      116.300,  116.000,  115.700,  115.300,  125.200,  125.100,  126.400,  126.200,  126.000,  125.700, 
      125.500,  125.200,  124.900,  124.500,  124.400,  124.200,  124.200,  124.100,  124.100,  124.100, 
      124.200,  123.800,  123.400,  123.200,  123.000,  122.600,  122.100,  121.400,  120.700,  120.000, 
      119.400,  118.700,  117.900,  117.400,  116.500,  115.700,  114.900,  114.000,  113.000,  112.200, 
      111.700,  110.900,  110.300,  109.500,  108.700,  107.900,  107.400,  106.700,  105.800,  104.800, 
      103.800,  102.800,  102.500,  102.500,  101.500,  100.800, ],
    [ 147.400,  143.200,  140.000,  138.800,  138.100,  137.400,  136.700,  136.000,  135.200,  134.400, 
      133.500,  132.600,  131.600,  130.500,  129.300,  128.200,  127.100,  125.900,  124.900,  123.800, 
      122.900,  122.000,  121.100,  120.600,  120.100,  119.700,  119.200,  118.900,  118.700,  118.500, 
      118.400,  118.400,  118.200,  118.200,  118.100,  118.100,  117.900,  117.800,  117.600,  117.500, 
      117.200,  116.900,  116.600,  116.300,  117.800,  117.500,  117.200,  117.000,  116.900,  116.600, 
      116.300,  116.000,  115.700,  125.500,  125.400,  125.600,  126.500,  126.400,  126.100,  125.900, 
      125.600,  125.300,  124.900,  124.600,  124.500,  124.500,  124.400,  124.400,  124.400,  124.400, 
      124.200,  123.900,  123.500,  123.200,  123.000,  122.700,  122.100,  121.400,  120.800,  120.200, 
      119.500,  118.900,  118.300,  117.500,  116.600,  115.800,  115.000,  114.200,  113.100,  112.300, 
      111.700,  111.100,  110.400,  109.700,  108.800,  108.200,  107.700,  106.700,  105.800,  104.800, 
      103.900,  103.200,  102.500,  102.600,  101.700,  100.800, ],
    [ 147.400,  143.200,  140.100,  138.900,  138.200,  137.400,  136.800,  136.000,  135.300,  134.400, 
      133.500,  132.600,  131.600,  130.500,  129.400,  128.200,  127.200,  126.000,  125.000,  123.900, 
      123.000,  122.200,  121.300,  120.800,  120.300,  120.000,  119.500,  119.100,  118.900,  118.800, 
      118.700,  118.700,  118.500,  118.400,  118.300,  118.200,  118.100,  117.900,  117.800,  117.600, 
      117.300,  116.900,  116.600,  118.100,  117.800,  117.500,  117.300,  117.100,  116.900,  116.600, 
      116.400,  116.000,  125.800,  125.800,  126.200,  126.900,  126.700,  126.500,  126.200,  125.900, 
      125.700,  125.400,  125.000,  124.900,  124.700,  124.800,  124.700,  124.700,  124.600,  124.600, 
      124.300,  124.000,  123.600,  123.400,  123.100,  122.800,  122.100,  121.500,  120.900,  120.300, 
      119.700,  119.000,  118.400,  117.600,  116.700,  116.000,  115.200,  114.200,  113.200,  112.600, 
      112.000,  111.300,  110.600,  109.900,  109.000,  108.400,  107.700,  106.700,  105.800,  105.100, 
      104.000,  103.200,  103.100,  102.600,  101.700,  100.900, ]])

_dep_grd = np.array([
      0.000,    2.000,    4.000,    6.000,    8.000,   10.000,   12.000,   14.000,   16.000,   18.000, 
      20.000,   22.000,   24.000,   26.000,   28.000,   30.000,   32.000,   34.000,   36.000,   38.000, 
      40.000,   42.000,   44.000,   46.000,   48.000,   50.000,   55.000,   60.000,   65.000,   70.000, 
      75.000,   80.000,   85.000,   90.000,   95.000,  100.000,  105.000,  110.000,  115.000,  120.000, 
    125.000,  130.000,  135.000,  140.000,  145.000,  150.000,  155.000,  160.000,  165.000,  170.000, 
    175.000,  180.000,  185.000,  190.000,  195.000,  200.000,  210.000,  220.000,  230.000,  240.000, 
    250.000,  260.000,  270.000,  280.000,  290.000,  300.000,  310.000,  320.000,  330.000,  340.000, 
    350.000,  360.000,  370.000,  380.000,  390.000,  400.000,  410.000,  420.000,  430.000,  440.000, 
    450.000,  460.000,  470.000,  480.000,  490.000,  500.000,  510.000,  520.000,  530.000,  540.000, 
    550.000,  560.000,  570.000,  580.000,  590.000,  600.000,  610.000,  620.000,  630.000,  640.000, 
    650.000,  660.000,  670.000,  680.000,  690.000,  700.000, ])

_dist_grd = np.array([
      0.000,    2.000,    4.000,    6.000,    8.000,   10.000,   12.000,   14.000,   16.000,   18.000, 
      20.000,   22.000,   24.000,   26.000,   28.000,   30.000,   32.000,   34.000,   36.000,   38.000, 
      40.000,   42.000,   44.000,   46.000,   48.000,   50.000,   55.000,   60.000,   65.000,   70.000, 
      75.000,   80.000,   85.000,   90.000,   95.000,  100.000,  105.000,  110.000,  115.000,  120.000, 
    125.000,  130.000,  135.000,  140.000,  145.000,  150.000,  155.000,  160.000,  165.000,  170.000, 
    175.000,  180.000,  185.000,  190.000,  195.000,  200.000,  210.000,  220.000,  230.000,  240.000, 
    250.000,  260.000,  270.000,  280.000,  290.000,  300.000,  310.000,  320.000,  330.000,  340.000, 
    350.000,  360.000,  370.000,  380.000,  390.000,  400.000,  410.000,  420.000,  430.000,  440.000, 
    450.000,  460.000,  470.000,  480.000,  490.000,  500.000,  510.000,  520.000,  530.000,  540.000, 
    550.000,  560.000,  570.000,  580.000,  590.000,  600.000,  610.000,  620.000,  630.000,  640.000, 
    650.000,  660.000,  670.000,  680.000,  690.000,  700.000,  710.000,  720.000,  730.000,  740.000, 
    750.000,  760.000,  770.000,  780.000,  790.000,  800.000,  810.000,  820.000,  830.000,  840.000, 
    850.000,  860.000,  870.000,  880.000,  890.000,  900.000,  910.000,  920.000,  930.000,  940.000, 
    950.000,  960.000,  970.000,  980.000,  990.000, 1000.000, 1010.000, 1020.000, 1030.000, 1040.000, 
    1050.000, 1060.000, 1070.000, 1080.000, 1090.000, 1100.000, 1110.000, 1120.000, 1130.000, 1140.000, 
    1150.000, 1160.000, 1170.000, 1180.000, 1190.000, 1200.000, 1210.000, 1220.000, 1230.000, 1240.000, 
    1250.000, 1260.000, 1270.000, 1280.000, 1290.000, 1300.000, 1310.000, 1320.000, 1330.000, 1340.000, 
    1350.000, 1360.000, 1370.000, 1380.000, 1390.000, 1400.000, 1410.000, 1420.000, 1430.000, 1440.000, 
    1450.000, 1460.000, 1470.000, 1480.000, 1490.000, 1500.000, 1510.000, 1520.000, 1530.000, 1540.000, 
    1550.000, 1560.000, 1570.000, 1580.000, 1590.000, 1600.000, 1610.000, 1620.000, 1630.000, 1640.000, 
    1650.000, 1660.000, 1670.000, 1680.000, 1690.000, 1700.000, 1710.000, 1720.000, 1730.000, 1740.000, 
    1750.000, 1760.000, 1770.000, 1780.000, 1790.000, 1800.000, 1810.000, 1820.000, 1830.000, 1840.000, 
    1850.000, 1860.000, 1870.000, 1880.000, 1890.000, 1900.000, 1910.000, 1920.000, 1930.000, 1940.000, 
    1950.000, 1960.000, 1970.000, 1980.000, 1990.000, 2000.000, ])

