import numpy as np
import scipy.interpolate

def brocher_density(vp):
    """
    Brocher (2005)'s empirical relationship between vp and density
    
    Parameters
    ----------
    vp : float
        P-wave velocity in km/s
    
    Returns
    -------
    rho : float
        density in g/cm^3
    """

    rho = 1.6612*vp - 0.4721 * vp**2 + 0.0671*vp**3 - 0.0043*vp**4 + 0.000106*vp**5

    return rho

def vmodel_hinet(z):

    """
    Hi-net速度構造
    
    Parameters
    ----------
    z : float
        depth in km
    
    Returns
    -------
    vp, vs : floats
        wavespeeds of P and S waves
    """

    R0 = 6369.5 # Earth's Radius
    R1 = 6337.5 # Moho's Radius
    zmoho = R0 - R1
    aP0 = 5.5
    aP1 = 7.8
    aS0 = 3.25
    aS1 = 4.41
    bP0 = -53.5
    bP1 = -2.20
    bS0 = -53.0
    bS1 = -1.9

    if z <= zmoho:
        vp = aP0 * ((R0 - z)/R0)**bP0
        vs = aS0 * ((R0 - z)/R0)**bS0
    else:
        vp = aP1 * ((R0 - z)/R1)**bP1
        vs = aS1 * ((R0 - z)/R1)**bS1
    
    return vp, vs


def vmodel_tohoku(z):

    """
    東北大ルーチン速度構造
    
    Parameters
    ----------
    z : float
        depth in km
    
    Returns
    -------
    vp, vs : floats
        wavespeeds of P and S waves
    """

    R0 = 6369.5 # Earth's Radius
    R1 = 6338.5 # Moho's Radius
    zmoho = R0 - R1
    aP0 = 5.4
    aP1 = 7.75
    aS0 = 3.15
    aS1 = 4.35
    bP0 = -59.0
    bP1 = -2.3
    bS0 = -59.0
    bS1 = -2.3

    if z <= zmoho:
        vp = aP0 * ((R0 - z)/R0)**bP0
        vs = aS0 * ((R0 - z)/R0)**bS0
    else:
        vp = aP1 * ((R0 - z)/R1)**bP1
        vs = aS1 * ((R0 - z)/R1)**bS1
    
    return vp, vs


def vmodel_iasp91(z):

    """
    iasp91 velcity model
    
    Parameters
    ----------
    z : float
        depth in km

    Returns
    -------
    vp, vs: floats
        wavespeeds in km/s

    Notes
    -----
    Kennet and Engdahl (1991) GJI 105, 429-465, https://doi.org/10.1111/j.1365-246X.1991.tb06724.x
    """

    x = (6371. - z) / 6371.

    if z > 6371.0: 
        vp = 0
        vs = 0
    elif z >= 5153.9:
        vp = 11.24094 - 4.09689 * x**2
        vs =  3.56454 - 3.45241 * x**2
    elif z >= 2889:
        vp = 10.03904 + 3.75665 * x - 13.67046 * x**2
        vs = 0.0
    elif z >= 2740:
        vp = 14.49470 - 1.47089 * x
        vs =  8.16616 - 1.58206 * x
    elif z >= 760: 
        vp = 25.1486 - 41.1538 * x + 51.9932 * x**2 - 26.6083 * x**3
        vs = 12.9303 - 21.2590 * x + 27.8988 * x**2 - 14.1080 * x**3
    elif z >= 660:
        vp = 25.96984 - 16.93412 * x
        vs = 20.76890 - 16.53147 * x
    elif z >= 410:
        vp = 29.38896 - 21.40656 * x
        vs = 17.70732 - 13.50652 * x
    elif z >= 210:
        vp = 30.78765 - 23.25415 * x
        vs = 15.24213 - 11.08552 * x
    elif z >= 120:
        vp = 25.41389 - 17.69722 * x
        vs = 5.75020  - 1.27420 * x 
    elif z >= 35:
        vp = 8.78541 - 0.74953 * x
        vs = 6.706231 - 2.248585 * x
    elif z >= 20:
        vp = 6.50
        vs = 3.75
    elif z >= 0.0:
        vp = 5.80
        vs = 3.36
    else:
        vp = 0.0
        vs = 0.0

    return vp, vs


def vmodel_prem(z):
 
    """
    Return the modified PREM 1D velocity structure
     
    Compared to the original model of the Diewonski and Anderson (1981), 
    the seawater layer is substituted to a crustal structure if type='modified' is given:
     rho=2.6 g/cm^3, vp=5.8 km/s, vs=3.2 km/s qmu=600, qkappa = 57823
    Or, '
    
    This routine returns Q_P and Q_S rather than Q_kappa and Q_mu, by converting using 
    a formula of (9.59) & (9.60) of Dahlen and Tromp (1998)

    Parameters
    ----------
    z : float
        depth in km
    
    Returns
    -------
    vp, vs : floats
        wavespeeds of P and S waves
    rho : float
        density in g/cm^3
    qp, qs: float
        Quality factor of P and S waves
    """

    r = 6371 - z
    x = r / 6371.0

    if r <= 1221.5: # inner core
        rho = 13.0885              -  8.8381*x*x
        vp  = 11.2622              -  6.3640*x*x
        vs  =  3.6678              -  4.4475*x*x
        qm  = 84.6
        qk  = 1327.7        
    elif r <=3480: # outer core
        rho = 12.5815 -  1.2638*x -  3.6426*x*x -  5.5281*x*x*x
        vp  = 11.0487 -  4.0362*x +  4.8023*x*x - 13.5732*x*x*x
        vs  = 0.0
        qm  = 1e100
        qk  = 57823
    elif r <= 3630.0:   # Lower Mantle 1
        rho =  7.9565 -  6.4761*x +  5.5283*x*x -  3.0807*x*x*x
        vp  = 15.3891 -  5.3181*x +  5.5242*x*x -  2.5514*x*x*x
        vs  =  6.9254 +  1.4672*x -  2.0834*x*x +  0.9783*x*x*x
        qm  = 312
        qk  = 57823
    elif r <= 5600.0:   # Lower Mantle 2
        rho =  7.9565 -  6.4761*x +  5.5283*x*x -  3.0807*x*x*x
        vp  = 24.9520 - 40.4673*x + 51.4832*x*x - 26.6419*x*x*x
        vs  = 11.1671 - 13.7818*x + 17.4575*x*x -  9.2777*x*x*x
        qm  = 312
        qk  = 57823
    elif r <= 5701.0:   # Lower Mantle 3
        rho =  7.9565 -  6.4761*x +  5.5283*x*x -  3.0807*x*x*x
        vp  = 29.2766 - 23.6027*x +  5.5242*x*x -  2.5514*x*x*x
        vs  = 22.3459 - 17.2473*x -  2.0834*x*x +  0.9783*x*x*x
        qm  = 312
        qk  = 57823
    elif r <= 5771.0:  # Transition zone 1
        rho =  5.3197 -  1.4836*x
        vp  = 19.0957 -  9.8672*x
        vs  =  9.9839 -  4.9324*x
        qm  = 143
        qk  = 57823
    elif r <= 5971.0:  # Transition zone 2
        rho = 11.2494 -  8.0298*x
        vp  = 39.7027 - 32.6166*x
        vs  = 22.3512 - 18.5856*x
        qm  = 143
        qk  = 57823
    elif r <= 6151.0:  # Transition zone 3
        rho =  7.1089 -  3.8045*x
        vp  = 20.3926 - 12.2569*x
        vs  =  8.9496 -  4.4597*x
        qm  = 143
        qk  = 57823
    elif r <= 6346.6:  # LVZ
        rho =  2.6910 +  0.6924*x
        vp  =  4.1875 +  3.9382*x
        vs  =  2.1519 +  2.3481*x
        qm  = 80
        qk  = 57823
    elif r <= 6346.6:  # LID
        rho =  2.6910 +  0.6924*x
        vp  =  4.1875 +  3.9382*x
        vs  =  2.1519 +  2.3481*x
        qm  =  600
        qk  =  57823
    elif r <= 6356.0:  # Crust 1
        rho =  2.900
        vp  =  6.800
        vs  =  3.900
        qm  =  600
        qk  =  57823
    elif r <= 6368.0:  # Crust 2
        rho =  2.600
        vp  =  5.800
        vs  =  3.200
        qm  =  600
        qk  =  57823
    elif r <= 6371.0:  #  Crust 3
        rho = 1.020
        vp  = 1.450
        vs  = 0.0
        qm  = 1e100
        qk  = 57823

    # atetnuation
    qp, qs = _qkqm_to_qpqs(qk, qm, vp, vs)

    return vp, vs, rho, qp, qs


def vmodel_jma(z, model = '2001'):
    """
    JMA2001 velocity model

    Parameters
    ----------
    z : float
        depth in km
    model : str
        velocity model version ('2001', '2020a', '2020b', '2020c')
    
    Returns
    -------
    vp, vs : floats
        wavespeeds of P and S waves in km/s
    """

    if model == '2020a':
        vp = float(_f_vp_jma_2020a(z))
        vs = float(_f_vs_jma_2020a(z))
    elif model == '2020b':
        vp = float(_f_vp_jma_2020b(z))
        vs = float(_f_vs_jma_2020b(z))
    elif model == '2020c':
        vp = float(_f_vp_jma_2020c(z))
        vs = float(_f_vs_jma_2020c(z))
    else:
        vp = float(_f_vp_jma_2001(z))
        vs = float(_f_vs_jma_2001(z))

    return vp, vs

def vmodel_ak135(z):

    """
    AK135 velocity model

    Parameters
    ----------
    z : float
        depth in km
    
    Returns
    -------
    vp, vs : floats
        wavespeeds of P and S waves in km/s
    rho : float
        density in g/cm^3
    qp, qs: float
        Quality factors of P and S waves
    """

    vp  = float(_f_vp_ak135(z))
    vs  = float(_f_vs_ak135(z))
    rho = float(_f_rho_ak135(z))
    qk  = float(_f_qk_ak135(z))
    qm  = float(_f_qm_ak135(z))
    qp, qs = _qkqm_to_qpqs(qk ,qm, vp, vs)

    return vp, vs, rho, qp, qs


def _qkqm_to_qpqs(qk, qm, vp, vs):

    """ Convert Q_kappa and Q_mu to Q_P and Q_S 
    
    Reference
    ---------
    a formula of (9.59) & (9.60) of Dahlen and Tromp (1998)
    """

    qmi = 1.0 / (qm + 1e-100)
    qki = 1.0 / (qk + 1e-100)
    qpi = ( 1.0 - (4.0 / 3.0) * vs * vs / vp / vp ) * qki + 4.0 / 3.0 * vs * vs / vp / vp * qmi
    qsi = qmi
    qp = 1.0 / qpi
    qs = 1.0 / qsi

    return qp, qs



#### DATASET


###
### JMA
###
_vp_jma_2001 = np.array([
 4.800,  4.950,  5.090,  5.220,  5.340,  5.450,  5.550,  5.640,  5.720,  5.780, 
 5.820,  5.840,  5.860,  5.880,  5.900,  5.920,  5.940,  5.960,  5.980,  6.000, 
 6.020,  6.040,  6.060,  6.080,  6.100,  6.120,  6.140,  6.160,  6.180,  6.200, 
 6.220,  6.242,  6.264,  6.286,  6.308,  6.330,  6.354,  6.378,  6.402,  6.426, 
 6.450,  6.476,  6.502,  6.528,  6.544,  6.570,  6.596,  6.622,  6.648,  6.674, 
 6.700,  6.728,  6.756,  6.784,  6.812,  6.840,  6.868,  6.896,  6.924,  6.952, 
 6.980,  7.006,  7.032,  7.058,  7.084,  7.110,  7.136,  7.162,  7.188,  7.204, 
 7.230,  7.254,  7.278,  7.302,  7.326,  7.348,  7.370,  7.392,  7.414,  7.434, 
 7.454,  7.474,  7.494,  7.512,  7.530,  7.548,  7.566,  7.582,  7.598,  7.612, 
 7.626,  7.638,  7.650,  7.660,  7.670,  7.679,  7.688,  7.696,  7.704,  7.711, 
 7.718,  7.724,  7.730,  7.735,  7.740,  7.750,  7.755,  7.760,  7.765,  7.770, 
 7.775,  7.779,  7.783,  7.787,  7.791,  7.795,  7.799,  7.803,  7.807,  7.811, 
 7.815,  7.818,  7.821,  7.824,  7.827,  7.830,  7.833,  7.836,  7.839,  7.842, 
 7.845,  7.847,  7.849,  7.851,  7.853,  7.855,  7.857,  7.859,  7.861,  7.863, 
 7.865,  7.866,  7.867,  7.868,  7.869,  7.870,  7.871,  7.872,  7.873,  7.874, 
 7.875,  7.876,  7.877,  7.878,  7.879,  7.880,  7.881,  7.883,  7.885,  7.887, 
 7.889,  7.891,  7.893,  7.895,  7.896,  7.898,  7.900,  7.901,  7.903,  7.905, 
 7.906,  7.908,  7.909,  7.911,  7.912,  7.914,  7.915,  7.917,  7.918,  7.920, 
 7.921,  7.923,  7.924,  7.926,  7.927,  7.928,  7.930,  7.931,  7.933,  7.934, 
 7.936,  7.937,  7.938,  7.940,  7.941,  7.943,  7.944,  7.945,  7.947,  7.948, 
 7.949,  7.951,  7.952,  7.953,  7.954,  7.955,  7.956,  7.957,  7.959,  7.960, 
 7.961,  7.963,  7.964,  7.965,  7.967,  7.968,  7.970,  7.971,  7.973,  7.974, 
 7.976,  7.977,  7.979,  7.981,  7.982,  7.984,  7.985,  7.987,  7.989,  7.990, 
 7.992,  7.994,  7.996,  7.997,  7.999,  8.001,  8.002,  8.004,  8.006,  8.008, 
 8.009,  8.011,  8.013,  8.015,  8.017,  8.019,  8.021,  8.023,  8.025,  8.027, 
 8.028,  8.030,  8.032,  8.034,  8.036,  8.038,  8.039,  8.041,  8.043,  8.045, 
 8.046,  8.048,  8.050,  8.052,  8.053,  8.055,  8.057,  8.058,  8.060,  8.062, 
 8.063,  8.065,  8.067,  8.068,  8.070,  8.072,  8.073,  8.075,  8.076,  8.078, 
 8.079,  8.081,  8.083,  8.085,  8.086,  8.088,  8.090,  8.091,  8.093,  8.095, 
 8.096,  8.098,  8.099,  8.101,  8.102,  8.104,  8.105,  8.107,  8.108,  8.110, 
 8.111,  8.113,  8.114,  8.116,  8.117,  8.118,  8.120,  8.121,  8.123,  8.124, 
 8.126,  8.127,  8.128,  8.130,  8.131,  8.133,  8.134,  8.135,  8.137,  8.138, 
 8.139,  8.141,  8.142,  8.143,  8.144,  8.145,  8.146,  8.147,  8.149,  8.150, 
 8.151,  8.153,  8.154,  8.155,  8.157,  8.158,  8.160,  8.161,  8.163,  8.164, 
 8.166,  8.167,  8.169,  8.171,  8.172,  8.174,  8.175,  8.177,  8.179,  8.180, 
 8.182,  8.184,  8.186,  8.187,  8.189,  8.191,  8.192,  8.194,  8.196,  8.198, 
 8.199,  8.202,  8.204,  8.205,  8.207,  8.209,  8.211,  8.213,  8.215,  8.217, 
 8.219,  8.221,  8.223,  8.224,  8.226,  8.228,  8.230,  8.232,  8.233,  8.235, 
 8.237,  8.238,  8.240,  8.242,  8.244,  8.245,  8.247,  8.249,  8.250,  8.252, 
 8.254,  8.255,  8.257,  8.258,  8.260,  8.262,  8.263,  8.265,  8.266,  8.268, 
 8.269,  8.271,  8.272,  8.274,  8.275,  8.277,  8.278,  8.280,  8.281,  8.283, 
 8.284,  8.286,  8.287,  8.289,  8.290,  8.292,  8.293,  8.295,  8.296,  8.298, 
 8.299,  8.301,  8.302,  8.304,  8.305,  8.307,  8.308,  8.310,  8.311,  8.313, 
 8.314,  8.316,  8.317,  8.319,  8.320,  8.322,  8.323,  8.325,  8.326,  8.328, 
 8.329,  8.331,  8.333,  8.335,  8.336,  8.338,  8.340,  8.341,  8.343,  8.345, 
 8.346,  8.348,  8.349,  8.351,  8.352,  8.354,  8.355,  8.357,  8.358,  8.360, 
 8.361,  8.363,  8.364,  8.366,  8.367,  8.368,  8.370,  8.371,  8.373,  8.374, 
 8.376,  8.377,  8.378,  8.380,  8.381,  8.383,  8.384,  8.385,  8.387,  8.388, 
 8.389,  8.390,  8.391,  8.393,  8.394,  8.395,  8.396,  8.397,  8.398,  8.400, 
 8.401,  8.402,  8.404,  8.405,  8.406,  8.408,  8.409,  8.411,  8.412,  8.414, 
 8.415,  8.417,  8.419,  8.420,  8.422,  8.423,  8.425,  8.427,  8.428,  8.430, 
 8.432,  8.434,  8.435,  8.437,  8.439,  8.441,  8.442,  8.444,  8.446,  8.448, 
 8.449,  8.452,  8.454,  8.456,  8.458,  8.460,  8.462,  8.464,  8.466,  8.468, 
 8.470,  8.472,  8.474,  8.476,  8.478,  8.479,  8.481,  8.483,  8.485,  8.486, 
 8.488,  8.490,  8.492,  8.493,  8.495,  8.496,  8.498,  8.500,  8.501,  8.503, 
 8.504,  8.506,  8.507,  8.509,  8.511,  8.512,  8.514,  8.515,  8.517,  8.518, 
 8.519,  8.521,  8.522,  8.523,  8.525,  8.526,  8.527,  8.529,  8.530,  8.531, 
 8.533,  8.534,  8.535,  8.537,  8.538,  8.540,  8.541,  8.543,  8.544,  8.546, 
 8.547,  8.549,  8.550,  8.552,  8.553,  8.555,  8.557,  8.558,  8.560,  8.561, 
 8.563,  8.565,  8.566,  8.568,  8.569,  8.571,  8.573,  8.574,  8.576,  8.578, 
 8.579,  8.581,  8.583,  8.585,  8.586,  8.588,  8.590,  8.592,  8.593,  8.595, 
 8.597,  8.599,  8.600,  8.602,  8.604,  8.606,  8.607,  8.609,  8.611,  8.613, 
 8.614,  8.616,  8.618,  8.620,  8.621,  8.623,  8.625,  8.627,  8.628,  8.630, 
 8.632,  8.634,  8.635,  8.637,  8.639,  8.641,  8.642,  8.644,  8.646,  8.648, 
 8.649,  8.651,  8.653,  8.655,  8.657,  8.658,  8.660,  8.662,  8.664,  8.666, 
 8.667,  8.669,  8.671,  8.673,  8.674,  8.676,  8.678,  8.680,  8.681,  8.683, 
 8.685,  8.687,  8.688,  8.690,  8.692,  8.694,  8.695,  8.697,  8.699,  8.700, 
 8.702,  8.704,  8.706,  8.707,  8.709,  8.711,  8.713,  8.714,  8.716,  8.718, 
 8.719,  8.721,  8.722,  8.723,  8.725,  8.726,  8.728,  8.729,  8.731,  8.732, 
 8.734,  8.735,  8.737,  8.739,  8.740,  8.742,  8.744,  8.746,  8.748,  8.749, 
 8.751,  8.753,  8.755,  8.757,  8.759,  8.760,  8.762,  8.764,  8.766,  8.768, 
 8.770,  8.772,  8.774,  8.776,  8.778,  8.780,  8.782,  8.784,  8.786,  8.788, 
 8.789,  8.792,  8.795,  8.798,  8.801,  8.803,  8.806,  8.808,  8.810,  8.812, 
 8.815,  8.817,  8.819,  8.821,  8.823,  8.824,  8.826,  8.828,  8.830,  8.831, 
 8.833,  8.835,  8.836,  8.838,  8.839,  8.841,  8.842,  8.843,  8.845,  8.846, 
 8.847,  8.849,  8.850,  8.851,  8.852,  8.854,  8.855,  8.856,  8.857,  8.858, 
 8.859,  8.860,  8.861,  8.861,  8.862,  8.863,  8.864,  8.864,  8.865,  8.866, 
 8.867,  8.869,  8.870,  8.871,  8.872,  8.873,  8.875,  8.876,  8.878,  8.879, 
 8.881,  8.882,  8.884,  8.885,  8.887,  8.889,  8.891,  8.892,  8.894,  8.896, 
 8.898,  8.900,  8.902,  8.904,  8.906,  8.908,  8.911,  8.913,  8.915,  8.917, 
 8.919,  8.922,  8.924,  8.927,  8.929,  8.932,  8.934,  8.937,  8.939,  8.942, 
 8.944,  8.947,  8.949,  8.952,  8.955,  8.957,  8.960,  8.963,  8.965,  8.968, 
 8.971,  8.974,  8.977,  8.979,  8.982,  8.985,  8.988,  8.991,  8.994,  8.997, 
 9.000,  9.002,  9.005,  9.008,  9.011,  9.014,  9.017,  9.020,  9.023,  9.026, 
 9.029,  9.033,  9.036,  9.039,  9.042,  9.045,  9.049,  9.052,  9.055,  9.058, 
 9.061,  9.065,  9.068,  9.071,  9.074,  9.077,  9.081,  9.084,  9.087,  9.090, 
 9.094,  9.097,  9.100,  9.103,  9.107,  9.110,  9.113,  9.116,  9.120,  9.123, 
 9.126,  9.130,  9.133,  9.136,  9.139,  9.143,  9.146,  9.149,  9.153,  9.156, 
 9.159,  9.163,  9.166,  9.169,  9.173,  9.176,  9.179,  9.183,  9.186,  9.190, 
 9.193,  9.196,  9.200,  9.203,  9.207,  9.210,  9.214,  9.217,  9.221,  9.224, 
 9.228,  9.231,  9.235,  9.238,  9.242,  9.245,  9.249,  9.252,  9.256,  9.260, 
 9.263,  9.267,  9.270,  9.274,  9.278,  9.281,  9.285,  9.289,  9.292,  9.296, 
 9.299,  9.303,  9.307,  9.310,  9.314,  9.317,  9.321,  9.325,  9.328,  9.332, 
 9.336,  9.339,  9.343,  9.347,  9.351,  9.354,  9.358,  9.362,  9.366,  9.369, 
 9.373,  9.377,  9.381,  9.384,  9.388,  9.392,  9.396,  9.400,  9.404,  9.407, 
 9.411,  9.415,  9.419,  9.423,  9.426,  9.430,  9.434,  9.438,  9.442,  9.446, 
 9.449,  9.454,  9.458,  9.462,  9.466,  9.470,  9.474,  9.478,  9.482,  9.486, 
 9.490,  9.493,  9.497,  9.501,  9.505,  9.509,  9.513,  9.516,  9.520,  9.524, 
 9.528,  9.531,  9.535,  9.539,  9.542,  9.546,  9.550,  9.553,  9.557,  9.560, 
 9.564,  9.568,  9.571,  9.575,  9.578,  9.582,  9.585,  9.589,  9.592,  9.596, 
 9.599,  9.603,  9.606,  9.609,  9.612,  9.615,  9.618,  9.622,  9.625,  9.628, 
 9.632,  9.635,  9.638,  9.642,  9.645,  9.649,  9.652,  9.656,  9.659,  9.663, 
 9.666,  9.670,  9.673,  9.677,  9.681,  9.684,  9.688,  9.692,  9.695,  9.699, 
 9.703,  9.706,  9.710,  9.714,  9.717,  9.721,  9.725,  9.728,  9.732,  9.736, 
 9.739,  9.744,  9.748,  9.752,  9.756,  9.760,  9.764,  9.768,  9.772,  9.776, 
 9.780,  9.784,  9.787,  9.791,  9.795,  9.798,  9.802,  9.805,  9.809,  9.812, 
 9.816,  9.819,  9.823,  9.826,  9.829,  9.833,  9.836,  9.839,  9.842,  9.845, 
 9.849,  9.852,  9.855,  9.858,  9.861,  9.864,  9.867,  9.870,  9.873,  9.876, 
 9.879,  9.882,  9.885,  9.888,  9.890,  9.893,  9.896,  9.899,  9.901,  9.904, 
 9.907,  9.910,  9.913,  9.916,  9.919,  9.922,  9.925,  9.928,  9.931,  9.934, 
 9.937,  9.940,  9.943,  9.946,  9.949,  9.952,  9.955,  9.958,  9.961,  9.964, 
 9.968,  9.971,  9.974,  9.977,  9.980,  9.983,  9.987,  9.990,  9.993,  9.996, 
10.000, 10.003, 10.006, 10.009, 10.012, 10.015, 10.018, 10.022, 10.025, 10.028, 
10.031, 10.035, 10.038, 10.041, 10.044, 10.048, 10.051, 10.054, 10.057, 10.061, 
10.064, 10.067, 10.071, 10.074, 10.077, 10.080, 10.084, 10.087, 10.090, 10.094, 
10.097, 10.100, 10.103, 10.107, 10.110, 10.113, 10.117, 10.120, 10.123, 10.126, 
10.129, 10.133, 10.137, 10.140, 10.144, 10.147, 10.150, 10.154, 10.157, 10.160, 
10.164, 10.167, 10.170, 10.173, 10.176, 10.179, 10.182, 10.185, 10.188, 10.191, 
10.194, 10.197, 10.200, 10.203, 10.206, 10.209, 10.212, 10.215, 10.217, 10.220, 
10.223, 10.226, 10.228, 10.231, 10.234, 10.236, 10.239, 10.242, 10.244, 10.247, 
10.250, 10.252, 10.254, 10.256, 10.259, 10.261, 10.264, 10.266, 10.268, 10.271, 
10.273, 10.276, 10.278, 10.281, 10.283, 10.286, 10.288, 10.291, 10.293, 10.296, 
10.298, 10.301, 10.303, 10.306, 10.308, 10.311, 10.313, 10.316, 10.319, 10.321, 
10.324, 10.326, 10.329, 10.331, 10.334, 10.337, 10.339, 10.342, 10.344, 10.347, 
10.349, 10.352, 10.355, 10.358, 10.361, 10.363, 10.366, 10.369, 10.371, 10.374, 
10.377, 10.379, 10.382, 10.385, 10.387, 10.390, 10.392, 10.395, 10.397, 10.400, 
10.402, 10.405, 10.407, 10.410, 10.412, 10.414, 10.417, 10.419, 10.422, 10.424, 
10.426, 10.429, 10.431, 10.433, 10.436, 10.438, 10.440, 10.443, 10.445, 10.447, 
10.449, 10.452, 10.454, 10.456, 10.458, 10.460, 10.462, 10.465, 10.467, 10.469, 
10.471, 10.473, 10.476, 10.478, 10.480, 10.482, 10.484, 10.487, 10.489, 10.491, 
10.493, 10.496, 10.498, 10.500, 10.503, 10.505, 10.507, 10.509, 10.512, 10.514, 
10.516, 10.519, 10.521, 10.523, 10.526, 10.528, 10.530, 10.533, 10.535, 10.537, 
10.539, 10.542, 10.544, 10.546, 10.549, 10.551, 10.553, 10.556, 10.558, 10.560, 
10.563, 10.565, 10.567, 10.569, 10.572, 10.574, 10.576, 10.579, 10.581, 10.583, 
10.585, 10.588, 10.590, 10.592, 10.594, 10.597, 10.599, 10.601, 10.603, 10.606, 
10.608, 10.610, 10.612, 10.614, 10.617, 10.619, 10.621, 10.623, 10.625, 10.627, 
10.629, 10.632, 10.634, 10.637, 10.639, 10.641, 10.643, 10.646, 10.648, 10.650, 
10.652, 10.654, 10.656, 10.658, 10.661, 10.663, 10.665, 10.667, 10.669, 10.671, 
10.673, 10.674, 10.676, 10.678, 10.680, 10.682, 10.684, 10.686, 10.688, 10.690, 
10.691, 10.693, 10.695, 10.697, 10.699, 10.701, 10.702, 10.704, 10.706, 10.708, 
10.709, 10.711, 10.712, 10.714, 10.715, 10.717, 10.718, 10.720, 10.721, 10.723, 
10.724, 10.726, 10.728, 10.729, 10.731, 10.733, 10.735, 10.736, 10.738, 10.740, 
10.742, 10.744, 10.745, 10.747, 10.749, 10.751, 10.753, 10.755, 10.757, 10.758, 
10.760, 10.762, 10.764, 10.766, 10.768, 10.770, 10.772, 10.774, 10.776, 10.778, 
10.779, 10.782, 10.784, 10.786, 10.788, 10.791, 10.793, 10.795, 10.797, 10.799, 
10.801, 10.803, 10.805, 10.806, 10.808, 10.810, 10.812, 10.814, 10.816, 10.817, 
10.819, 10.821, 10.822, 10.824, 10.826, 10.827, 10.829, 10.830, 10.832, 10.834, 
10.835, 10.837, 10.838, 10.840, 10.841, 10.842, 10.844, 10.845, 10.847, 10.848, 
10.849, 10.851, 10.852, 10.853, 10.854, 10.856, 10.857, 10.858, 10.859, 10.861, 
10.862, 10.863, 10.864, 10.866, 10.867, 10.868, 10.869, 10.871, 10.872, 10.873, 
10.874, 10.876, 10.877, 10.878, 10.879, 10.881, 10.882, 10.883, 10.884, 10.886, 
10.887, 10.888, 10.889, 10.891, 10.892, 10.893, 10.894, 10.896, 10.897, 10.898, 
10.899, 10.901, 10.902, 10.904, 10.905, 10.906, 10.908, 10.909, 10.910, 10.911, 
10.913, 10.914, 10.915, 10.917, 10.918, 10.919, 10.920, 10.922, 10.923, 10.924, 
10.925, 10.927, 10.928, 10.929, 10.930, 10.931, 10.933, 10.934, 10.935, 10.936, 
10.937, 10.939, 10.940, 10.941, 10.942, 10.943, 10.945, 10.946, 10.947, 10.948, 
10.949, 10.950, 10.951, 10.952, 10.953, 10.954, 10.955, 10.956, 10.957, 10.958, 
10.959, 10.961, 10.962, 10.963, 10.964, 10.965, 10.967, 10.968, 10.969, 10.970, 
10.972, 10.973, 10.974, 10.976, 10.977, 10.978, 10.980, 10.981, 10.983, 10.984, 
10.985, 10.987, 10.988, 10.990, 10.991, 10.992, 10.994, 10.995, 10.997, 10.998, 
11.000, 11.001, 11.003, 11.005, 11.006, 11.008, 11.010, 11.011, 11.013, 11.014, 
11.016, 11.017, 11.019, 11.020, 11.021, 11.023, 11.024, 11.025, 11.027, 11.028, 
11.029, 11.030, 11.031, 11.033, 11.034, 11.035, 11.036, 11.037, 11.038, 11.039, 
11.040, 11.041, 11.042, 11.043, 11.044, 11.045, 11.046, 11.047, 11.048, 11.049, 
11.049, 11.050, 11.051, 11.052, 11.053, 11.053, 11.054, 11.055, 11.056, 11.057, 
11.057, 11.058, 11.059, 11.060, 11.060, 11.061, 11.062, 11.063, 11.063, 11.064, 
11.065, 11.066, 11.066, 11.067, 11.068, 11.069, 11.069, 11.070, 11.071, 11.071, 
11.072, 11.073, 11.074, 11.074, 11.075, 11.076, 11.077, 11.077, 11.078, 11.079, 
11.079, 11.080, 11.081, 11.081, 11.082, 11.082, 11.083, 11.084, 11.084, 11.085, 
11.086, 11.086, 11.087, 11.088, 11.089, 11.089, 11.090, 11.091, 11.092, 11.092, 
11.093, 11.094, 11.095, 11.095, 11.096, 11.097, 11.098, 11.099, 11.100, 11.100, 
11.101, 11.102, 11.103, 11.104, 11.104, 11.105, 11.106, 11.107, 11.108, 11.109, 
11.109, 11.111, 11.112, 11.112, 11.113, 11.114, 11.115, 11.116, 11.117, 11.118, 
11.119, 11.120, 11.121, 11.121, 11.122, 11.123, 11.124, 11.125, 11.125, 11.126, 
11.127, 11.127, 11.128, 11.129, 11.130, 11.130, 11.131, 11.132, 11.132, 11.133, 
11.134, 11.134, 11.135, 11.135, 11.136, 11.137, 11.137, 11.138, 11.138, 11.139, 
11.139, 11.140, 11.140, 11.141, 11.141, 11.142, 11.142, 11.143, 11.143, 11.144, 
11.144, 11.145, 11.145, 11.146, 11.146, 11.147, 11.147, 11.148, 11.148, 11.149, 
11.149, 11.150, 11.150, 11.151, 11.151, 11.152, 11.152, 11.153, 11.153, 11.154, 
11.154, 11.155, 11.155, 11.156, 11.156, 11.157, 11.157, 11.158, 11.158, 11.159, 
11.159, 11.160, 11.161, 11.161, 11.162, 11.162, 11.163, 11.163, 11.164, 11.164, 
11.165, 11.165, 11.166, 11.166, 11.167, 11.167, 11.168, 11.168, 11.169, 11.169, 
11.170, 11.170, 11.171, 11.171, 11.172, 11.172, 11.173, 11.173, 11.174, 11.174, 
11.175, 11.175, 11.176, 11.176, 11.177, 11.177, 11.178, 11.178, 11.179, 11.179, 
11.179, 11.180, 11.180, 11.180, 11.181, 11.181, 11.182, 11.182, 11.182, 11.183, 
11.183, 11.184, 11.184, 11.185, 11.185, 11.186, 11.186, 11.187, 11.187, 11.188, 
11.188, 11.189, 11.189, 11.190, 11.190, 11.191, 11.191, 11.192, 11.193, 11.193, 
11.194, 11.194, 11.195, 11.195, 11.196, 11.197, 11.197, 11.198, 11.198, 11.199, 
])
_vs_jma_2001 = np.array([
 2.844,  2.931,  3.012,  3.088,  3.157,  3.221,  3.278,  3.329,  3.375,  3.409, 
 3.431,  3.441,  3.451,  3.461,  3.471,  3.481,  3.491,  3.501,  3.511,  3.521, 
 3.531,  3.541,  3.552,  3.562,  3.572,  3.583,  3.593,  3.603,  3.614,  3.624, 
 3.634,  3.646,  3.657,  3.669,  3.680,  3.692,  3.704,  3.717,  3.729,  3.742, 
 3.754,  3.768,  3.782,  3.795,  3.803,  3.817,  3.831,  3.844,  3.858,  3.872, 
 3.885,  3.900,  3.915,  3.929,  3.944,  3.959,  3.974,  3.988,  4.003,  4.018, 
 4.032,  4.046,  4.059,  4.073,  4.086,  4.100,  4.113,  4.127,  4.140,  4.148, 
 4.161,  4.173,  4.186,  4.198,  4.210,  4.221,  4.232,  4.243,  4.254,  4.264, 
 4.274,  4.284,  4.294,  4.303,  4.311,  4.320,  4.329,  4.336,  4.344,  4.350, 
 4.356,  4.362,  4.367,  4.371,  4.375,  4.379,  4.382,  4.385,  4.388,  4.390, 
 4.393,  4.395,  4.396,  4.398,  4.399,  4.403,  4.404,  4.405,  4.406,  4.408, 
 4.409,  4.409,  4.410,  4.411,  4.411,  4.412,  4.413,  4.413,  4.414,  4.415, 
 4.415,  4.416,  4.416,  4.417,  4.417,  4.417,  4.418,  4.418,  4.419,  4.419, 
 4.419,  4.419,  4.419,  4.419,  4.419,  4.419,  4.419,  4.419,  4.419,  4.419, 
 4.419,  4.419,  4.419,  4.419,  4.419,  4.420,  4.420,  4.420,  4.420,  4.420, 
 4.420,  4.420,  4.420,  4.420,  4.420,  4.420,  4.420,  4.420,  4.420,  4.420, 
 4.420,  4.420,  4.421,  4.422,  4.423,  4.423,  4.424,  4.425,  4.425,  4.426, 
 4.427,  4.428,  4.428,  4.429,  4.429,  4.430,  4.430,  4.431,  4.431,  4.432, 
 4.433,  4.434,  4.434,  4.435,  4.435,  4.436,  4.437,  4.437,  4.438,  4.439, 
 4.440,  4.440,  4.441,  4.442,  4.442,  4.443,  4.443,  4.444,  4.445,  4.445, 
 4.446,  4.447,  4.447,  4.448,  4.448,  4.448,  4.449,  4.449,  4.450,  4.451, 
 4.451,  4.452,  4.453,  4.453,  4.454,  4.455,  4.456,  4.456,  4.457,  4.457, 
 4.458,  4.459,  4.460,  4.461,  4.461,  4.462,  4.463,  4.464,  4.465,  4.465, 
 4.466,  4.467,  4.468,  4.468,  4.469,  4.470,  4.471,  4.472,  4.473,  4.474, 
 4.474,  4.475,  4.476,  4.477,  4.478,  4.479,  4.480,  4.481,  4.482,  4.483, 
 4.484,  4.485,  4.486,  4.487,  4.488,  4.489,  4.489,  4.490,  4.491,  4.492, 
 4.492,  4.493,  4.494,  4.495,  4.496,  4.497,  4.498,  4.498,  4.499,  4.500, 
 4.501,  4.502,  4.503,  4.503,  4.504,  4.505,  4.506,  4.507,  4.507,  4.508, 
 4.508,  4.509,  4.510,  4.511,  4.512,  4.513,  4.514,  4.514,  4.515,  4.516, 
 4.517,  4.518,  4.518,  4.519,  4.519,  4.520,  4.521,  4.522,  4.522,  4.523, 
 4.524,  4.525,  4.525,  4.526,  4.527,  4.527,  4.528,  4.528,  4.529,  4.530, 
 4.531,  4.531,  4.532,  4.533,  4.533,  4.534,  4.535,  4.535,  4.536,  4.536, 
 4.537,  4.538,  4.538,  4.539,  4.539,  4.540,  4.540,  4.540,  4.541,  4.542, 
 4.542,  4.543,  4.544,  4.544,  4.545,  4.545,  4.546,  4.547,  4.548,  4.548, 
 4.549,  4.550,  4.551,  4.552,  4.552,  4.553,  4.554,  4.555,  4.556,  4.556, 
 4.557,  4.558,  4.559,  4.559,  4.560,  4.561,  4.562,  4.563,  4.564,  4.565, 
 4.565,  4.567,  4.568,  4.568,  4.569,  4.570,  4.571,  4.572,  4.573,  4.574, 
 4.575,  4.576,  4.577,  4.577,  4.578,  4.579,  4.580,  4.581,  4.582,  4.583, 
 4.584,  4.584,  4.585,  4.587,  4.588,  4.588,  4.589,  4.590,  4.591,  4.592, 
 4.593,  4.594,  4.595,  4.595,  4.597,  4.598,  4.598,  4.599,  4.600,  4.601, 
 4.602,  4.603,  4.603,  4.604,  4.604,  4.605,  4.606,  4.607,  4.607,  4.608, 
 4.609,  4.610,  4.610,  4.611,  4.611,  4.612,  4.613,  4.614,  4.614,  4.615, 
 4.616,  4.617,  4.617,  4.618,  4.619,  4.620,  4.621,  4.622,  4.622,  4.623, 
 4.624,  4.625,  4.626,  4.627,  4.627,  4.628,  4.629,  4.630,  4.631,  4.632, 
 4.632,  4.633,  4.634,  4.635,  4.636,  4.637,  4.638,  4.638,  4.639,  4.640, 
 4.641,  4.642,  4.642,  4.643,  4.643,  4.644,  4.645,  4.646,  4.646,  4.647, 
 4.648,  4.649,  4.649,  4.650,  4.651,  4.651,  4.653,  4.653,  4.654,  4.655, 
 4.656,  4.656,  4.657,  4.658,  4.659,  4.660,  4.660,  4.661,  4.662,  4.663, 
 4.663,  4.664,  4.664,  4.665,  4.665,  4.666,  4.666,  4.667,  4.667,  4.668, 
 4.669,  4.669,  4.670,  4.670,  4.671,  4.672,  4.672,  4.673,  4.674,  4.675, 
 4.675,  4.676,  4.677,  4.678,  4.679,  4.679,  4.681,  4.682,  4.682,  4.683, 
 4.684,  4.686,  4.686,  4.687,  4.688,  4.689,  4.690,  4.691,  4.692,  4.693, 
 4.694,  4.695,  4.696,  4.697,  4.698,  4.699,  4.700,  4.701,  4.702,  4.703, 
 4.704,  4.705,  4.706,  4.707,  4.708,  4.709,  4.710,  4.711,  4.712,  4.712, 
 4.713,  4.714,  4.715,  4.716,  4.717,  4.717,  4.718,  4.720,  4.720,  4.721, 
 4.722,  4.723,  4.723,  4.725,  4.726,  4.726,  4.727,  4.728,  4.729,  4.730, 
 4.730,  4.731,  4.732,  4.732,  4.733,  4.733,  4.734,  4.735,  4.735,  4.736, 
 4.737,  4.737,  4.737,  4.738,  4.739,  4.740,  4.740,  4.741,  4.742,  4.743, 
 4.743,  4.744,  4.745,  4.746,  4.746,  4.748,  4.749,  4.749,  4.750,  4.751, 
 4.752,  4.753,  4.754,  4.755,  4.755,  4.756,  4.757,  4.758,  4.759,  4.760, 
 4.761,  4.762,  4.763,  4.764,  4.764,  4.765,  4.766,  4.767,  4.768,  4.769, 
 4.769,  4.770,  4.771,  4.772,  4.773,  4.774,  4.774,  4.775,  4.776,  4.777, 
 4.778,  4.779,  4.780,  4.781,  4.781,  4.783,  4.784,  4.785,  4.785,  4.786, 
 4.788,  4.789,  4.789,  4.790,  4.791,  4.793,  4.793,  4.794,  4.795,  4.796, 
 4.797,  4.798,  4.799,  4.800,  4.801,  4.801,  4.802,  4.803,  4.804,  4.805, 
 4.806,  4.807,  4.808,  4.809,  4.809,  4.810,  4.811,  4.812,  4.812,  4.813, 
 4.814,  4.815,  4.816,  4.817,  4.818,  4.819,  4.820,  4.821,  4.822,  4.823, 
 4.824,  4.825,  4.826,  4.826,  4.828,  4.829,  4.830,  4.830,  4.831,  4.833, 
 4.833,  4.834,  4.835,  4.835,  4.836,  4.836,  4.837,  4.838,  4.839,  4.839, 
 4.840,  4.841,  4.842,  4.842,  4.843,  4.844,  4.845,  4.846,  4.847,  4.847, 
 4.848,  4.849,  4.850,  4.852,  4.853,  4.853,  4.854,  4.855,  4.857,  4.858, 
 4.859,  4.860,  4.861,  4.862,  4.863,  4.864,  4.865,  4.866,  4.868,  4.869, 
 4.869,  4.871,  4.873,  4.874,  4.876,  4.877,  4.879,  4.880,  4.881,  4.882, 
 4.884,  4.885,  4.886,  4.887,  4.888,  4.889,  4.890,  4.891,  4.892,  4.893, 
 4.894,  4.895,  4.895,  4.896,  4.897,  4.898,  4.899,  4.899,  4.900,  4.901, 
 4.901,  4.902,  4.903,  4.904,  4.904,  4.905,  4.906,  4.906,  4.907,  4.907, 
 4.908,  4.909,  4.909,  4.909,  4.910,  4.910,  4.911,  4.911,  4.911,  4.912, 
 4.912,  4.914,  4.914,  4.915,  4.915,  4.916,  4.917,  4.917,  4.919,  4.919, 
 4.920,  4.921,  4.922,  4.922,  4.924,  4.925,  4.926,  4.926,  4.927,  4.929, 
 4.930,  4.931,  4.932,  4.933,  4.934,  4.935,  4.937,  4.938,  4.939,  4.940, 
 4.941,  4.943,  4.944,  4.946,  4.947,  4.948,  4.950,  4.951,  4.952,  4.954, 
 4.955,  4.957,  4.958,  4.960,  4.961,  4.962,  4.964,  4.966,  4.967,  4.968, 
 4.970,  4.972,  4.973,  4.975,  4.976,  4.978,  4.980,  4.981,  4.983,  4.984, 
 4.986,  4.987,  4.989,  4.991,  4.992,  4.994,  4.996,  4.997,  4.999,  5.001, 
 5.002,  5.004,  5.006,  5.008,  5.009,  5.011,  5.013,  5.015,  5.017,  5.018, 
 5.020,  5.022,  5.024,  5.025,  5.027,  5.029,  5.031,  5.033,  5.034,  5.036, 
 5.038,  5.040,  5.042,  5.043,  5.045,  5.047,  5.049,  5.050,  5.053,  5.054, 
 5.056,  5.058,  5.060,  5.061,  5.063,  5.065,  5.067,  5.069,  5.071,  5.073, 
 5.074,  5.076,  5.078,  5.080,  5.082,  5.084,  5.085,  5.088,  5.089,  5.091, 
 5.093,  5.095,  5.097,  5.099,  5.101,  5.102,  5.105,  5.106,  5.109,  5.110, 
 5.112,  5.114,  5.116,  5.118,  5.120,  5.122,  5.124,  5.126,  5.128,  5.130, 
 5.132,  5.134,  5.136,  5.138,  5.140,  5.142,  5.144,  5.146,  5.148,  5.150, 
 5.152,  5.154,  5.156,  5.158,  5.160,  5.162,  5.164,  5.166,  5.168,  5.170, 
 5.172,  5.174,  5.176,  5.178,  5.181,  5.182,  5.184,  5.187,  5.189,  5.191, 
 5.193,  5.195,  5.197,  5.199,  5.201,  5.203,  5.206,  5.208,  5.210,  5.212, 
 5.214,  5.216,  5.218,  5.220,  5.222,  5.224,  5.227,  5.229,  5.231,  5.233, 
 5.235,  5.238,  5.240,  5.242,  5.244,  5.247,  5.249,  5.251,  5.253,  5.255, 
 5.258,  5.259,  5.261,  5.264,  5.266,  5.268,  5.270,  5.272,  5.274,  5.276, 
 5.279,  5.280,  5.283,  5.285,  5.286,  5.289,  5.291,  5.293,  5.295,  5.296, 
 5.299,  5.301,  5.302,  5.305,  5.306,  5.309,  5.310,  5.312,  5.314,  5.316, 
 5.318,  5.320,  5.322,  5.324,  5.325,  5.327,  5.329,  5.331,  5.332,  5.334, 
 5.336,  5.338,  5.340,  5.342,  5.343,  5.346,  5.347,  5.350,  5.351,  5.353, 
 5.355,  5.357,  5.359,  5.361,  5.363,  5.365,  5.367,  5.370,  5.371,  5.373, 
 5.376,  5.377,  5.380,  5.382,  5.383,  5.386,  5.388,  5.389,  5.392,  5.394, 
 5.396,  5.398,  5.401,  5.403,  5.405,  5.407,  5.409,  5.412,  5.414,  5.416, 
 5.418,  5.420,  5.422,  5.424,  5.427,  5.428,  5.430,  5.432,  5.434,  5.436, 
 5.438,  5.440,  5.442,  5.444,  5.445,  5.448,  5.449,  5.451,  5.453,  5.454, 
 5.457,  5.458,  5.460,  5.461,  5.463,  5.465,  5.466,  5.468,  5.470,  5.471, 
 5.473,  5.475,  5.476,  5.478,  5.479,  5.481,  5.483,  5.484,  5.485,  5.487, 
 5.489,  5.490,  5.492,  5.494,  5.495,  5.497,  5.499,  5.500,  5.502,  5.504, 
 5.505,  5.507,  5.509,  5.510,  5.512,  5.514,  5.515,  5.517,  5.519,  5.520, 
 5.522,  5.524,  5.526,  5.527,  5.529,  5.531,  5.533,  5.535,  5.536,  5.538, 
 5.540,  5.542,  5.543,  5.545,  5.547,  5.548,  5.550,  5.552,  5.554,  5.556, 
 5.557,  5.560,  5.561,  5.563,  5.565,  5.567,  5.568,  5.570,  5.572,  5.574, 
 5.576,  5.577,  5.580,  5.581,  5.583,  5.584,  5.587,  5.588,  5.590,  5.592, 
 5.594,  5.596,  5.597,  5.599,  5.601,  5.603,  5.605,  5.607,  5.608,  5.610, 
 5.612,  5.614,  5.616,  5.618,  5.620,  5.622,  5.623,  5.625,  5.627,  5.629, 
 5.631,  5.633,  5.634,  5.636,  5.638,  5.639,  5.641,  5.643,  5.644,  5.646, 
 5.648,  5.649,  5.651,  5.653,  5.654,  5.656,  5.658,  5.659,  5.660,  5.662, 
 5.664,  5.665,  5.666,  5.668,  5.670,  5.671,  5.673,  5.674,  5.675,  5.677, 
 5.679,  5.680,  5.681,  5.682,  5.684,  5.685,  5.686,  5.688,  5.689,  5.690, 
 5.691,  5.693,  5.694,  5.696,  5.697,  5.699,  5.700,  5.701,  5.702,  5.704, 
 5.705,  5.707,  5.708,  5.710,  5.711,  5.712,  5.714,  5.715,  5.717,  5.718, 
 5.720,  5.721,  5.722,  5.724,  5.725,  5.727,  5.728,  5.730,  5.731,  5.732, 
 5.734,  5.735,  5.737,  5.739,  5.740,  5.741,  5.743,  5.745,  5.746,  5.747, 
 5.749,  5.750,  5.752,  5.753,  5.755,  5.756,  5.757,  5.759,  5.760,  5.762, 
 5.763,  5.765,  5.766,  5.767,  5.768,  5.770,  5.771,  5.772,  5.774,  5.775, 
 5.776,  5.778,  5.779,  5.780,  5.782,  5.783,  5.784,  5.786,  5.787,  5.788, 
 5.789,  5.791,  5.792,  5.793,  5.794,  5.795,  5.796,  5.798,  5.799,  5.800, 
 5.801,  5.802,  5.804,  5.805,  5.806,  5.807,  5.808,  5.810,  5.811,  5.812, 
 5.813,  5.815,  5.816,  5.817,  5.819,  5.820,  5.821,  5.822,  5.824,  5.825, 
 5.826,  5.828,  5.829,  5.830,  5.832,  5.833,  5.834,  5.835,  5.837,  5.838, 
 5.839,  5.840,  5.842,  5.843,  5.844,  5.845,  5.847,  5.848,  5.849,  5.850, 
 5.852,  5.853,  5.854,  5.855,  5.857,  5.858,  5.859,  5.861,  5.862,  5.863, 
 5.864,  5.866,  5.867,  5.868,  5.869,  5.871,  5.872,  5.873,  5.874,  5.876, 
 5.877,  5.878,  5.879,  5.880,  5.882,  5.883,  5.884,  5.885,  5.886,  5.888, 
 5.889,  5.890,  5.891,  5.893,  5.894,  5.895,  5.896,  5.898,  5.899,  5.900, 
 5.901,  5.902,  5.904,  5.905,  5.906,  5.907,  5.909,  5.910,  5.911,  5.912, 
 5.913,  5.914,  5.915,  5.916,  5.917,  5.918,  5.919,  5.920,  5.921,  5.922, 
 5.923,  5.924,  5.925,  5.926,  5.927,  5.929,  5.929,  5.930,  5.931,  5.932, 
 5.933,  5.934,  5.935,  5.936,  5.936,  5.937,  5.938,  5.939,  5.940,  5.941, 
 5.941,  5.942,  5.943,  5.944,  5.945,  5.946,  5.947,  5.948,  5.949,  5.950, 
 5.951,  5.952,  5.953,  5.954,  5.955,  5.956,  5.957,  5.958,  5.960,  5.960, 
 5.961,  5.962,  5.963,  5.965,  5.966,  5.967,  5.968,  5.969,  5.970,  5.971, 
 5.972,  5.973,  5.975,  5.976,  5.977,  5.978,  5.980,  5.981,  5.982,  5.983, 
 5.984,  5.985,  5.986,  5.987,  5.988,  5.989,  5.990,  5.991,  5.992,  5.993, 
 5.994,  5.995,  5.996,  5.997,  5.998,  5.998,  5.999,  6.000,  6.001,  6.002, 
 6.003,  6.004,  6.004,  6.006,  6.006,  6.007,  6.008,  6.008,  6.009,  6.010, 
 6.011,  6.012,  6.012,  6.013,  6.013,  6.014,  6.015,  6.016,  6.016,  6.017, 
 6.018,  6.018,  6.019,  6.020,  6.020,  6.021,  6.022,  6.023,  6.023,  6.024, 
 6.024,  6.025,  6.026,  6.027,  6.027,  6.028,  6.029,  6.029,  6.030,  6.031, 
 6.032,  6.032,  6.033,  6.034,  6.034,  6.035,  6.035,  6.037,  6.037,  6.038, 
 6.038,  6.039,  6.040,  6.041,  6.042,  6.042,  6.043,  6.044,  6.044,  6.045, 
 6.046,  6.047,  6.047,  6.048,  6.049,  6.049,  6.050,  6.051,  6.052,  6.052, 
 6.053,  6.054,  6.054,  6.055,  6.055,  6.056,  6.057,  6.058,  6.058,  6.059, 
 6.059,  6.060,  6.061,  6.061,  6.062,  6.063,  6.064,  6.064,  6.065,  6.065, 
 6.066,  6.066,  6.067,  6.068,  6.068,  6.069,  6.069,  6.070,  6.070,  6.071, 
 6.071,  6.073,  6.073,  6.074,  6.074,  6.075,  6.076,  6.076,  6.077,  6.078, 
 6.079,  6.079,  6.080,  6.081,  6.081,  6.082,  6.083,  6.084,  6.085,  6.085, 
 6.086,  6.087,  6.088,  6.089,  6.089,  6.090,  6.091,  6.091,  6.093,  6.093, 
 6.094,  6.095,  6.096,  6.097,  6.098,  6.099,  6.100,  6.100,  6.101,  6.102, 
 6.103,  6.104,  6.105,  6.105,  6.106,  6.107,  6.107,  6.108,  6.109,  6.110, 
 6.110,  6.111,  6.111,  6.112,  6.113,  6.114,  6.114,  6.115,  6.115,  6.116, 
 6.116,  6.117,  6.117,  6.118,  6.119,  6.119,  6.120,  6.120,  6.121,  6.121, 
 6.121,  6.122,  6.122,  6.123,  6.124,  6.124,  6.124,  6.125,  6.125,  6.126, 
 6.126,  6.126,  6.127,  6.127,  6.127,  6.128,  6.129,  6.129,  6.129,  6.130, 
 6.130,  6.131,  6.131,  6.131,  6.132,  6.132,  6.132,  6.133,  6.134,  6.134, 
 6.134,  6.135,  6.135,  6.135,  6.136,  6.136,  6.137,  6.137,  6.137,  6.138, 
 6.138,  6.139,  6.139,  6.139,  6.140,  6.140,  6.140,  6.141,  6.141,  6.141, 
 6.142,  6.142,  6.142,  6.143,  6.143,  6.143,  6.144,  6.145,  6.145,  6.145, 
 6.146,  6.146,  6.147,  6.147,  6.147,  6.148,  6.148,  6.149,  6.150,  6.150, 
 6.150,  6.151,  6.151,  6.152,  6.152,  6.152,  6.153,  6.153,  6.154,  6.155, 
 6.155,  6.156,  6.156,  6.156,  6.157,  6.157,  6.158,  6.158,  6.159,  6.160, 
 6.160,  6.161,  6.161,  6.161,  6.162,  6.162,  6.163,  6.163,  6.163,  6.164, 
 6.165,  6.165,  6.165,  6.166,  6.166,  6.166,  6.167,  6.167,  6.167,  6.168, 
 6.168,  6.168,  6.169,  6.169,  6.170,  6.170,  6.170,  6.171,  6.171,  6.171, 
 6.171,  6.172,  6.172,  6.172,  6.172,  6.173,  6.173,  6.173,  6.173,  6.174, 
 6.174,  6.175,  6.175,  6.175,  6.175,  6.176,  6.176,  6.176,  6.176,  6.177, 
 6.177,  6.177,  6.177,  6.178,  6.178,  6.178,  6.178,  6.179,  6.179,  6.180, 
 6.180,  6.180,  6.180,  6.181,  6.181,  6.181,  6.181,  6.182,  6.182,  6.182, 
 6.182,  6.183,  6.183,  6.183,  6.184,  6.184,  6.184,  6.184,  6.185,  6.185, 
 6.186,  6.186,  6.186,  6.186,  6.187,  6.187,  6.187,  6.187,  6.188,  6.188, 
 6.188,  6.188,  6.189,  6.189,  6.189,  6.189,  6.190,  6.190,  6.191,  6.191, 
 6.191,  6.191,  6.192,  6.192,  6.192,  6.192,  6.193,  6.193,  6.193,  6.193, 
 6.193,  6.194,  6.194,  6.194,  6.194,  6.194,  6.195,  6.195,  6.195,  6.196, 
 6.196,  6.196,  6.196,  6.197,  6.197,  6.197,  6.197,  6.198,  6.198,  6.198, 
 6.198,  6.199,  6.199,  6.199,  6.199,  6.200,  6.200,  6.201,  6.201,  6.201, 
 6.202,  6.202,  6.202,  6.202,  6.203,  6.203,  6.203,  6.204,  6.204,  6.204, 
])
_dep_jma_2001 = np.array([
  0.0,   0.5,   1.0,   1.5,   2.0,   2.5,   3.0,   3.5,   4.0,   4.5, 
  5.0,   5.5,   6.0,   6.5,   7.0,   7.5,   8.0,   8.5,   9.0,   9.5, 
 10.0,  10.5,  11.0,  11.5,  12.0,  12.5,  13.0,  13.5,  14.0,  14.5, 
 15.0,  15.5,  16.0,  16.5,  17.0,  17.5,  18.0,  18.5,  19.0,  19.5, 
 20.0,  20.5,  21.0,  21.5,  22.0,  22.5,  23.0,  23.5,  24.0,  24.5, 
 25.0,  25.5,  26.0,  26.5,  27.0,  27.5,  28.0,  28.5,  29.0,  29.5, 
 30.0,  30.5,  31.0,  31.5,  32.0,  32.5,  33.0,  33.5,  34.0,  34.5, 
 35.0,  35.5,  36.0,  36.5,  37.0,  37.5,  38.0,  38.5,  39.0,  39.5, 
 40.0,  40.5,  41.0,  41.5,  42.0,  42.5,  43.0,  43.5,  44.0,  44.5, 
 45.0,  45.5,  46.0,  46.5,  47.0,  47.5,  48.0,  48.5,  49.0,  49.5, 
 50.0,  50.5,  51.0,  51.5,  52.0,  52.5,  53.0,  53.5,  54.0,  54.5, 
 55.0,  55.5,  56.0,  56.5,  57.0,  57.5,  58.0,  58.5,  59.0,  59.5, 
 60.0,  60.5,  61.0,  61.5,  62.0,  62.5,  63.0,  63.5,  64.0,  64.5, 
 65.0,  65.5,  66.0,  66.5,  67.0,  67.5,  68.0,  68.5,  69.0,  69.5, 
 70.0,  70.5,  71.0,  71.5,  72.0,  72.5,  73.0,  73.5,  74.0,  74.5, 
 75.0,  75.5,  76.0,  76.5,  77.0,  77.5,  78.0,  78.5,  79.0,  79.5, 
 80.0,  80.5,  81.0,  81.5,  82.0,  82.5,  83.0,  83.5,  84.0,  84.5, 
 85.0,  85.5,  86.0,  86.5,  87.0,  87.5,  88.0,  88.5,  89.0,  89.5, 
 90.0,  90.5,  91.0,  91.5,  92.0,  92.5,  93.0,  93.5,  94.0,  94.5, 
 95.0,  95.5,  96.0,  96.5,  97.0,  97.5,  98.0,  98.5,  99.0,  99.5, 
100.0, 100.5, 101.0, 101.5, 102.0, 102.5, 103.0, 103.5, 104.0, 104.5, 
105.0, 105.5, 106.0, 106.5, 107.0, 107.5, 108.0, 108.5, 109.0, 109.5, 
110.0, 110.5, 111.0, 111.5, 112.0, 112.5, 113.0, 113.5, 114.0, 114.5, 
115.0, 115.5, 116.0, 116.5, 117.0, 117.5, 118.0, 118.5, 119.0, 119.5, 
120.0, 120.5, 121.0, 121.5, 122.0, 122.5, 123.0, 123.5, 124.0, 124.5, 
125.0, 125.5, 126.0, 126.5, 127.0, 127.5, 128.0, 128.5, 129.0, 129.5, 
130.0, 130.5, 131.0, 131.5, 132.0, 132.5, 133.0, 133.5, 134.0, 134.5, 
135.0, 135.5, 136.0, 136.5, 137.0, 137.5, 138.0, 138.5, 139.0, 139.5, 
140.0, 140.5, 141.0, 141.5, 142.0, 142.5, 143.0, 143.5, 144.0, 144.5, 
145.0, 145.5, 146.0, 146.5, 147.0, 147.5, 148.0, 148.5, 149.0, 149.5, 
150.0, 150.5, 151.0, 151.5, 152.0, 152.5, 153.0, 153.5, 154.0, 154.5, 
155.0, 155.5, 156.0, 156.5, 157.0, 157.5, 158.0, 158.5, 159.0, 159.5, 
160.0, 160.5, 161.0, 161.5, 162.0, 162.5, 163.0, 163.5, 164.0, 164.5, 
165.0, 165.5, 166.0, 166.5, 167.0, 167.5, 168.0, 168.5, 169.0, 169.5, 
170.0, 170.5, 171.0, 171.5, 172.0, 172.5, 173.0, 173.5, 174.0, 174.5, 
175.0, 175.5, 176.0, 176.5, 177.0, 177.5, 178.0, 178.5, 179.0, 179.5, 
180.0, 180.5, 181.0, 181.5, 182.0, 182.5, 183.0, 183.5, 184.0, 184.5, 
185.0, 185.5, 186.0, 186.5, 187.0, 187.5, 188.0, 188.5, 189.0, 189.5, 
190.0, 190.5, 191.0, 191.5, 192.0, 192.5, 193.0, 193.5, 194.0, 194.5, 
195.0, 195.5, 196.0, 196.5, 197.0, 197.5, 198.0, 198.5, 199.0, 199.5, 
200.0, 200.5, 201.0, 201.5, 202.0, 202.5, 203.0, 203.5, 204.0, 204.5, 
205.0, 205.5, 206.0, 206.5, 207.0, 207.5, 208.0, 208.5, 209.0, 209.5, 
210.0, 210.5, 211.0, 211.5, 212.0, 212.5, 213.0, 213.5, 214.0, 214.5, 
215.0, 215.5, 216.0, 216.5, 217.0, 217.5, 218.0, 218.5, 219.0, 219.5, 
220.0, 220.5, 221.0, 221.5, 222.0, 222.5, 223.0, 223.5, 224.0, 224.5, 
225.0, 225.5, 226.0, 226.5, 227.0, 227.5, 228.0, 228.5, 229.0, 229.5, 
230.0, 230.5, 231.0, 231.5, 232.0, 232.5, 233.0, 233.5, 234.0, 234.5, 
235.0, 235.5, 236.0, 236.5, 237.0, 237.5, 238.0, 238.5, 239.0, 239.5, 
240.0, 240.5, 241.0, 241.5, 242.0, 242.5, 243.0, 243.5, 244.0, 244.5, 
245.0, 245.5, 246.0, 246.5, 247.0, 247.5, 248.0, 248.5, 249.0, 249.5, 
250.0, 250.5, 251.0, 251.5, 252.0, 252.5, 253.0, 253.5, 254.0, 254.5, 
255.0, 255.5, 256.0, 256.5, 257.0, 257.5, 258.0, 258.5, 259.0, 259.5, 
260.0, 260.5, 261.0, 261.5, 262.0, 262.5, 263.0, 263.5, 264.0, 264.5, 
265.0, 265.5, 266.0, 266.5, 267.0, 267.5, 268.0, 268.5, 269.0, 269.5, 
270.0, 270.5, 271.0, 271.5, 272.0, 272.5, 273.0, 273.5, 274.0, 274.5, 
275.0, 275.5, 276.0, 276.5, 277.0, 277.5, 278.0, 278.5, 279.0, 279.5, 
280.0, 280.5, 281.0, 281.5, 282.0, 282.5, 283.0, 283.5, 284.0, 284.5, 
285.0, 285.5, 286.0, 286.5, 287.0, 287.5, 288.0, 288.5, 289.0, 289.5, 
290.0, 290.5, 291.0, 291.5, 292.0, 292.5, 293.0, 293.5, 294.0, 294.5, 
295.0, 295.5, 296.0, 296.5, 297.0, 297.5, 298.0, 298.5, 299.0, 299.5, 
300.0, 300.5, 301.0, 301.5, 302.0, 302.5, 303.0, 303.5, 304.0, 304.5, 
305.0, 305.5, 306.0, 306.5, 307.0, 307.5, 308.0, 308.5, 309.0, 309.5, 
310.0, 310.5, 311.0, 311.5, 312.0, 312.5, 313.0, 313.5, 314.0, 314.5, 
315.0, 315.5, 316.0, 316.5, 317.0, 317.5, 318.0, 318.5, 319.0, 319.5, 
320.0, 320.5, 321.0, 321.5, 322.0, 322.5, 323.0, 323.5, 324.0, 324.5, 
325.0, 325.5, 326.0, 326.5, 327.0, 327.5, 328.0, 328.5, 329.0, 329.5, 
330.0, 330.5, 331.0, 331.5, 332.0, 332.5, 333.0, 333.5, 334.0, 334.5, 
335.0, 335.5, 336.0, 336.5, 337.0, 337.5, 338.0, 338.5, 339.0, 339.5, 
340.0, 340.5, 341.0, 341.5, 342.0, 342.5, 343.0, 343.5, 344.0, 344.5, 
345.0, 345.5, 346.0, 346.5, 347.0, 347.5, 348.0, 348.5, 349.0, 349.5, 
350.0, 350.5, 351.0, 351.5, 352.0, 352.5, 353.0, 353.5, 354.0, 354.5, 
355.0, 355.5, 356.0, 356.5, 357.0, 357.5, 358.0, 358.5, 359.0, 359.5, 
360.0, 360.5, 361.0, 361.5, 362.0, 362.5, 363.0, 363.5, 364.0, 364.5, 
365.0, 365.5, 366.0, 366.5, 367.0, 367.5, 368.0, 368.5, 369.0, 369.5, 
370.0, 370.5, 371.0, 371.5, 372.0, 372.5, 373.0, 373.5, 374.0, 374.5, 
375.0, 375.5, 376.0, 376.5, 377.0, 377.5, 378.0, 378.5, 379.0, 379.5, 
380.0, 380.5, 381.0, 381.5, 382.0, 382.5, 383.0, 383.5, 384.0, 384.5, 
385.0, 385.5, 386.0, 386.5, 387.0, 387.5, 388.0, 388.5, 389.0, 389.5, 
390.0, 390.5, 391.0, 391.5, 392.0, 392.5, 393.0, 393.5, 394.0, 394.5, 
395.0, 395.5, 396.0, 396.5, 397.0, 397.5, 398.0, 398.5, 399.0, 399.5, 
400.0, 400.5, 401.0, 401.5, 402.0, 402.5, 403.0, 403.5, 404.0, 404.5, 
405.0, 405.5, 406.0, 406.5, 407.0, 407.5, 408.0, 408.5, 409.0, 409.5, 
410.0, 410.5, 411.0, 411.5, 412.0, 412.5, 413.0, 413.5, 414.0, 414.5, 
415.0, 415.5, 416.0, 416.5, 417.0, 417.5, 418.0, 418.5, 419.0, 419.5, 
420.0, 420.5, 421.0, 421.5, 422.0, 422.5, 423.0, 423.5, 424.0, 424.5, 
425.0, 425.5, 426.0, 426.5, 427.0, 427.5, 428.0, 428.5, 429.0, 429.5, 
430.0, 430.5, 431.0, 431.5, 432.0, 432.5, 433.0, 433.5, 434.0, 434.5, 
435.0, 435.5, 436.0, 436.5, 437.0, 437.5, 438.0, 438.5, 439.0, 439.5, 
440.0, 440.5, 441.0, 441.5, 442.0, 442.5, 443.0, 443.5, 444.0, 444.5, 
445.0, 445.5, 446.0, 446.5, 447.0, 447.5, 448.0, 448.5, 449.0, 449.5, 
450.0, 450.5, 451.0, 451.5, 452.0, 452.5, 453.0, 453.5, 454.0, 454.5, 
455.0, 455.5, 456.0, 456.5, 457.0, 457.5, 458.0, 458.5, 459.0, 459.5, 
460.0, 460.5, 461.0, 461.5, 462.0, 462.5, 463.0, 463.5, 464.0, 464.5, 
465.0, 465.5, 466.0, 466.5, 467.0, 467.5, 468.0, 468.5, 469.0, 469.5, 
470.0, 470.5, 471.0, 471.5, 472.0, 472.5, 473.0, 473.5, 474.0, 474.5, 
475.0, 475.5, 476.0, 476.5, 477.0, 477.5, 478.0, 478.5, 479.0, 479.5, 
480.0, 480.5, 481.0, 481.5, 482.0, 482.5, 483.0, 483.5, 484.0, 484.5, 
485.0, 485.5, 486.0, 486.5, 487.0, 487.5, 488.0, 488.5, 489.0, 489.5, 
490.0, 490.5, 491.0, 491.5, 492.0, 492.5, 493.0, 493.5, 494.0, 494.5, 
495.0, 495.5, 496.0, 496.5, 497.0, 497.5, 498.0, 498.5, 499.0, 499.5, 
500.0, 500.5, 501.0, 501.5, 502.0, 502.5, 503.0, 503.5, 504.0, 504.5, 
505.0, 505.5, 506.0, 506.5, 507.0, 507.5, 508.0, 508.5, 509.0, 509.5, 
510.0, 510.5, 511.0, 511.5, 512.0, 512.5, 513.0, 513.5, 514.0, 514.5, 
515.0, 515.5, 516.0, 516.5, 517.0, 517.5, 518.0, 518.5, 519.0, 519.5, 
520.0, 520.5, 521.0, 521.5, 522.0, 522.5, 523.0, 523.5, 524.0, 524.5, 
525.0, 525.5, 526.0, 526.5, 527.0, 527.5, 528.0, 528.5, 529.0, 529.5, 
530.0, 530.5, 531.0, 531.5, 532.0, 532.5, 533.0, 533.5, 534.0, 534.5, 
535.0, 535.5, 536.0, 536.5, 537.0, 537.5, 538.0, 538.5, 539.0, 539.5, 
540.0, 540.5, 541.0, 541.5, 542.0, 542.5, 543.0, 543.5, 544.0, 544.5, 
545.0, 545.5, 546.0, 546.5, 547.0, 547.5, 548.0, 548.5, 549.0, 549.5, 
550.0, 550.5, 551.0, 551.5, 552.0, 552.5, 553.0, 553.5, 554.0, 554.5, 
555.0, 555.5, 556.0, 556.5, 557.0, 557.5, 558.0, 558.5, 559.0, 559.5, 
560.0, 560.5, 561.0, 561.5, 562.0, 562.5, 563.0, 563.5, 564.0, 564.5, 
565.0, 565.5, 566.0, 566.5, 567.0, 567.5, 568.0, 568.5, 569.0, 569.5, 
570.0, 570.5, 571.0, 571.5, 572.0, 572.5, 573.0, 573.5, 574.0, 574.5, 
575.0, 575.5, 576.0, 576.5, 577.0, 577.5, 578.0, 578.5, 579.0, 579.5, 
580.0, 580.5, 581.0, 581.5, 582.0, 582.5, 583.0, 583.5, 584.0, 584.5, 
585.0, 585.5, 586.0, 586.5, 587.0, 587.5, 588.0, 588.5, 589.0, 589.5, 
590.0, 590.5, 591.0, 591.5, 592.0, 592.5, 593.0, 593.5, 594.0, 594.5, 
595.0, 595.5, 596.0, 596.5, 597.0, 597.5, 598.0, 598.5, 599.0, 599.5, 
600.0, 600.5, 601.0, 601.5, 602.0, 602.5, 603.0, 603.5, 604.0, 604.5, 
605.0, 605.5, 606.0, 606.5, 607.0, 607.5, 608.0, 608.5, 609.0, 609.5, 
610.0, 610.5, 611.0, 611.5, 612.0, 612.5, 613.0, 613.5, 614.0, 614.5, 
615.0, 615.5, 616.0, 616.5, 617.0, 617.5, 618.0, 618.5, 619.0, 619.5, 
620.0, 620.5, 621.0, 621.5, 622.0, 622.5, 623.0, 623.5, 624.0, 624.5, 
625.0, 625.5, 626.0, 626.5, 627.0, 627.5, 628.0, 628.5, 629.0, 629.5, 
630.0, 630.5, 631.0, 631.5, 632.0, 632.5, 633.0, 633.5, 634.0, 634.5, 
635.0, 635.5, 636.0, 636.5, 637.0, 637.5, 638.0, 638.5, 639.0, 639.5, 
640.0, 640.5, 641.0, 641.5, 642.0, 642.5, 643.0, 643.5, 644.0, 644.5, 
645.0, 645.5, 646.0, 646.5, 647.0, 647.5, 648.0, 648.5, 649.0, 649.5, 
650.0, 650.5, 651.0, 651.5, 652.0, 652.5, 653.0, 653.5, 654.0, 654.5, 
655.0, 655.5, 656.0, 656.5, 657.0, 657.5, 658.0, 658.5, 659.0, 659.5, 
660.0, 660.5, 661.0, 661.5, 662.0, 662.5, 663.0, 663.5, 664.0, 664.5, 
665.0, 665.5, 666.0, 666.5, 667.0, 667.5, 668.0, 668.5, 669.0, 669.5, 
670.0, 670.5, 671.0, 671.5, 672.0, 672.5, 673.0, 673.5, 674.0, 674.5, 
675.0, 675.5, 676.0, 676.5, 677.0, 677.5, 678.0, 678.5, 679.0, 679.5, 
680.0, 680.5, 681.0, 681.5, 682.0, 682.5, 683.0, 683.5, 684.0, 684.5, 
685.0, 685.5, 686.0, 686.5, 687.0, 687.5, 688.0, 688.5, 689.0, 689.5, 
690.0, 690.5, 691.0, 691.5, 692.0, 692.5, 693.0, 693.5, 694.0, 694.5, 
695.0, 695.5, 696.0, 696.5, 697.0, 697.5, 698.0, 698.5, 699.0, 699.5, 
700.0, 700.5, 701.0, 701.5, 702.0, 702.5, 703.0, 703.5, 704.0, 704.5, 
705.0, 705.5, 706.0, 706.5, 707.0, 707.5, 708.0, 708.5, 709.0, 709.5, 
710.0, 710.5, 711.0, 711.5, 712.0, 712.5, 713.0, 713.5, 714.0, 714.5, 
715.0, 715.5, 716.0, 716.5, 717.0, 717.5, 718.0, 718.5, 719.0, 719.5, 
720.0, 720.5, 721.0, 721.5, 722.0, 722.5, 723.0, 723.5, 724.0, 724.5, 
725.0, 725.5, 726.0, 726.5, 727.0, 727.5, 728.0, 728.5, 729.0, 729.5, 
730.0, 730.5, 731.0, 731.5, 732.0, 732.5, 733.0, 733.5, 734.0, 734.5, 
735.0, 735.5, 736.0, 736.5, 737.0, 737.5, 738.0, 738.5, 739.0, 739.5, 
740.0, 740.5, 741.0, 741.5, 742.0, 742.5, 743.0, 743.5, 744.0, 744.5, 
745.0, 745.5, 746.0, 746.5, 747.0, 747.5, 748.0, 748.5, 749.0, 749.5, 
750.0, 750.5, 751.0, 751.5, 752.0, 752.5, 753.0, 753.5, 754.0, 754.5, 
755.0, 755.5, 756.0, 756.5, 757.0, 757.5, 758.0, 758.5, 759.0, 759.5, 
760.0, 760.5, 761.0, 761.5, 762.0, 762.5, 763.0, 763.5, 764.0, 764.5, 
765.0, 765.5, 766.0, 766.5, 767.0, 767.5, 768.0, 768.5, 769.0, 769.5, 
770.0, 770.5, 771.0, 771.5, 772.0, 772.5, 773.0, 773.5, 774.0, 774.5, 
775.0, 775.5, 776.0, 776.5, 777.0, 777.5, 778.0, 778.5, 779.0, 779.5, 
780.0, 780.5, 781.0, 781.5, 782.0, 782.5, 783.0, 783.5, 784.0, 784.5, 
785.0, 785.5, 786.0, 786.5, 787.0, 787.5, 788.0, 788.5, 789.0, 789.5, 
790.0, 790.5, 791.0, 791.5, 792.0, 792.5, 793.0, 793.5, 794.0, 794.5, 
795.0, 795.5, 796.0, 796.5, 797.0, 797.5, 798.0, 798.5, 799.0, 799.5, 
800.0, 800.5, 801.0, 801.5, 802.0, 802.5, 803.0, 803.5, 804.0, 804.5, 
805.0, 805.5, 806.0, 806.5, 807.0, 807.5, 808.0, 808.5, 809.0, 809.5, 
810.0, 810.5, 811.0, 811.5, 812.0, 812.5, 813.0, 813.5, 814.0, 814.5, 
815.0, 815.5, 816.0, 816.5, 817.0, 817.5, 818.0, 818.5, 819.0, 819.5, 
820.0, 820.5, 821.0, 821.5, 822.0, 822.5, 823.0, 823.5, 824.0, 824.5, 
825.0, 825.5, 826.0, 826.5, 827.0, 827.5, 828.0, 828.5, 829.0, 829.5, 
830.0, 830.5, 831.0, 831.5, 832.0, 832.5, 833.0, 833.5, 834.0, 834.5, 
835.0, 835.5, 836.0, 836.5, 837.0, 837.5, 838.0, 838.5, 839.0, 839.5, 
840.0, 840.5, 841.0, 841.5, 842.0, 842.5, 843.0, 843.5, 844.0, 844.5, 
845.0, 845.5, 846.0, 846.5, 847.0, 847.5, 848.0, 848.5, 849.0, 849.5, 
850.0, 850.5, 851.0, 851.5, 852.0, 852.5, 853.0, 853.5, 854.0, 854.5, 
855.0, 855.5, 856.0, 856.5, 857.0, 857.5, 858.0, 858.5, 859.0, 859.5, 
860.0, 860.5, 861.0, 861.5, 862.0, 862.5, 863.0, 863.5, 864.0, 864.5, 
865.0, 865.5, 866.0, 866.5, 867.0, 867.5, 868.0, 868.5, 869.0, 869.5, 
870.0, 870.5, 871.0, 871.5, 872.0, 872.5, 873.0, 873.5, 874.0, 874.5, 
875.0, 875.5, 876.0, 876.5, 877.0, 877.5, 878.0, 878.5, 879.0, 879.5, 
880.0, 880.5, 881.0, 881.5, 882.0, 882.5, 883.0, 883.5, 884.0, 884.5, 
885.0, 885.5, 886.0, 886.5, 887.0, 887.5, 888.0, 888.5, 889.0, 889.5, 
890.0, 890.5, 891.0, 891.5, 892.0, 892.5, 893.0, 893.5, 894.0, 894.5, 
895.0, 895.5, 896.0, 896.5, 897.0, 897.5, 898.0, 898.5, 899.0, 899.5, 
900.0, 900.5, 901.0, 901.5, 902.0, 902.5, 903.0, 903.5, 904.0, 904.5, 
905.0, 905.5, 906.0, 906.5, 907.0, 907.5, 908.0, 908.5, 909.0, 909.5, 
910.0, 910.5, 911.0, 911.5, 912.0, 912.5, 913.0, 913.5, 914.0, 914.5, 
915.0, 915.5, 916.0, 916.5, 917.0, 917.5, 918.0, 918.5, 919.0, 919.5, 
920.0, 920.5, 921.0, 921.5, 922.0, 922.5, 923.0, 923.5, 924.0, 924.5, 
925.0, 925.5, 926.0, 926.5, 927.0, 927.5, 928.0, 928.5, 929.0, 929.5, 
930.0, 930.5, 931.0, 931.5, 932.0, 932.5, 933.0, 933.5, 934.0, 934.5, 
935.0, 935.5, 936.0, 936.5, 937.0, 937.5, 938.0, 938.5, 939.0, 939.5, 
])
_f_vp_jma_2001 = scipy.interpolate.interp1d(_dep_jma_2001, _vp_jma_2001, kind='linear')
_f_vs_jma_2001 = scipy.interpolate.interp1d(_dep_jma_2001, _vs_jma_2001, kind='linear')
_vp_jma_2020a = np.array([
 4.800,  4.808,  4.816,  4.824,  4.832,  4.840,  4.848,  4.856,  4.864,  4.872, 
 5.027,  5.170,  5.297,  5.412,  5.519,  5.622,  5.725,  5.827,  5.927,  6.024, 
 6.117,  6.205,  6.287,  6.362,  6.429,  6.489,  6.543,  6.592,  6.637,  6.679, 
 6.719,  6.758,  6.796,  6.833,  6.869,  6.904,  6.938,  6.971,  7.003,  7.034, 
 7.064,  7.093,  7.121,  7.149,  7.177,  7.205,  7.233,  7.261,  7.289,  7.317, 
 7.345,  7.373,  7.401,  7.428,  7.453,  7.475,  7.493,  7.508,  7.522,  7.535, 
 7.547,  7.558,  7.568,  7.577,  7.585,  7.593,  7.600,  7.607,  7.613,  7.619, 
 7.625,  7.630,  7.635,  7.640,  7.645,  7.650,  7.655,  7.660,  7.665,  7.670, 
 7.675,  7.679,  7.683,  7.687,  7.691,  7.695,  7.699,  7.703,  7.707,  7.711, 
 7.715,  7.719,  7.723,  7.727,  7.731,  7.735,  7.739,  7.743,  7.747,  7.751, 
 7.755,  7.758,  7.761,  7.764,  7.767,  7.770,  7.773,  7.776,  7.779,  7.782, 
 7.785,  7.788,  7.791,  7.794,  7.797,  7.800,  7.803,  7.806,  7.809,  7.812, 
 7.815,  7.818,  7.821,  7.824,  7.827,  7.830,  7.833,  7.836,  7.839,  7.842, 
 7.845,  7.847,  7.849,  7.851,  7.853,  7.855,  7.857,  7.859,  7.861,  7.863, 
 7.865,  7.866,  7.867,  7.868,  7.869,  7.870,  7.871,  7.872,  7.873,  7.874, 
 7.875,  7.876,  7.877,  7.878,  7.879,  7.880,  7.881,  7.883,  7.885,  7.887, 
 7.889,  7.891,  7.893,  7.895,  7.896,  7.898,  7.900,  7.901,  7.903,  7.905, 
 7.906,  7.908,  7.909,  7.911,  7.912,  7.914,  7.915,  7.917,  7.918,  7.920, 
 7.921,  7.923,  7.924,  7.926,  7.927,  7.928,  7.930,  7.931,  7.933,  7.934, 
 7.936,  7.937,  7.938,  7.940,  7.941,  7.943,  7.944,  7.945,  7.947,  7.948, 
 7.949,  7.951,  7.952,  7.953,  7.954,  7.955,  7.956,  7.957,  7.959,  7.960, 
 7.961,  7.963,  7.964,  7.965,  7.967,  7.968,  7.970,  7.971,  7.973,  7.974, 
 7.976,  7.977,  7.979,  7.981,  7.982,  7.984,  7.985,  7.987,  7.989,  7.990, 
 7.992,  7.994,  7.996,  7.997,  7.999,  8.001,  8.002,  8.004,  8.006,  8.008, 
 8.009,  8.011,  8.013,  8.015,  8.017,  8.019,  8.021,  8.023,  8.025,  8.027, 
 8.028,  8.030,  8.032,  8.034,  8.036,  8.038,  8.039,  8.041,  8.043,  8.045, 
 8.046,  8.048,  8.050,  8.052,  8.053,  8.055,  8.057,  8.058,  8.060,  8.062, 
 8.063,  8.065,  8.067,  8.068,  8.070,  8.072,  8.073,  8.075,  8.076,  8.078, 
 8.079,  8.081,  8.083,  8.085,  8.086,  8.088,  8.090,  8.091,  8.093,  8.095, 
 8.096,  8.098,  8.099,  8.101,  8.102,  8.104,  8.105,  8.107,  8.108,  8.110, 
 8.111,  8.113,  8.114,  8.116,  8.117,  8.118,  8.120,  8.121,  8.123,  8.124, 
 8.126,  8.127,  8.128,  8.130,  8.131,  8.133,  8.134,  8.135,  8.137,  8.138, 
 8.139,  8.141,  8.142,  8.143,  8.144,  8.145,  8.146,  8.147,  8.149,  8.150, 
 8.151,  8.153,  8.154,  8.155,  8.157,  8.158,  8.160,  8.161,  8.163,  8.164, 
 8.166,  8.167,  8.169,  8.171,  8.172,  8.174,  8.175,  8.177,  8.179,  8.180, 
 8.182,  8.184,  8.186,  8.187,  8.189,  8.191,  8.192,  8.194,  8.196,  8.198, 
 8.199,  8.202,  8.204,  8.205,  8.207,  8.209,  8.211,  8.213,  8.215,  8.217, 
 8.219,  8.221,  8.223,  8.224,  8.226,  8.228,  8.230,  8.232,  8.233,  8.235, 
 8.237,  8.238,  8.240,  8.242,  8.244,  8.245,  8.247,  8.249,  8.250,  8.252, 
 8.254,  8.255,  8.257,  8.258,  8.260,  8.262,  8.263,  8.265,  8.266,  8.268, 
 8.269,  8.271,  8.272,  8.274,  8.275,  8.277,  8.278,  8.280,  8.281,  8.283, 
 8.284,  8.286,  8.287,  8.289,  8.290,  8.292,  8.293,  8.295,  8.296,  8.298, 
 8.299,  8.301,  8.302,  8.304,  8.305,  8.307,  8.308,  8.310,  8.311,  8.313, 
 8.314,  8.316,  8.317,  8.319,  8.320,  8.322,  8.323,  8.325,  8.326,  8.328, 
 8.329,  8.331,  8.333,  8.335,  8.336,  8.338,  8.340,  8.341,  8.343,  8.345, 
 8.346,  8.348,  8.349,  8.351,  8.352,  8.354,  8.355,  8.357,  8.358,  8.360, 
 8.361,  8.363,  8.364,  8.366,  8.367,  8.368,  8.370,  8.371,  8.373,  8.374, 
 8.376,  8.377,  8.378,  8.380,  8.381,  8.383,  8.384,  8.385,  8.387,  8.388, 
 8.389,  8.390,  8.391,  8.393,  8.394,  8.395,  8.396,  8.397,  8.398,  8.400, 
 8.401,  8.402,  8.404,  8.405,  8.406,  8.408,  8.409,  8.411,  8.412,  8.414, 
 8.415,  8.417,  8.419,  8.420,  8.422,  8.423,  8.425,  8.427,  8.428,  8.430, 
 8.432,  8.434,  8.435,  8.437,  8.439,  8.441,  8.442,  8.444,  8.446,  8.448, 
 8.449,  8.452,  8.454,  8.456,  8.458,  8.460,  8.462,  8.464,  8.466,  8.468, 
 8.470,  8.472,  8.474,  8.476,  8.478,  8.479,  8.481,  8.483,  8.485,  8.486, 
 8.488,  8.490,  8.492,  8.493,  8.495,  8.496,  8.498,  8.500,  8.501,  8.503, 
 8.504,  8.506,  8.507,  8.509,  8.511,  8.512,  8.514,  8.515,  8.517,  8.518, 
 8.519,  8.521,  8.522,  8.523,  8.525,  8.526,  8.527,  8.529,  8.530,  8.531, 
 8.533,  8.534,  8.535,  8.537,  8.538,  8.540,  8.541,  8.543,  8.544,  8.546, 
 8.547,  8.549,  8.550,  8.552,  8.553,  8.555,  8.557,  8.558,  8.560,  8.561, 
 8.563,  8.565,  8.566,  8.568,  8.569,  8.571,  8.573,  8.574,  8.576,  8.578, 
 8.579,  8.581,  8.583,  8.585,  8.586,  8.588,  8.590,  8.592,  8.593,  8.595, 
 8.597,  8.599,  8.600,  8.602,  8.604,  8.606,  8.607,  8.609,  8.611,  8.613, 
 8.614,  8.616,  8.618,  8.620,  8.621,  8.623,  8.625,  8.627,  8.628,  8.630, 
 8.632,  8.634,  8.635,  8.637,  8.639,  8.641,  8.642,  8.644,  8.646,  8.648, 
 8.649,  8.651,  8.653,  8.655,  8.657,  8.658,  8.660,  8.662,  8.664,  8.666, 
 8.667,  8.669,  8.671,  8.673,  8.674,  8.676,  8.678,  8.680,  8.681,  8.683, 
 8.685,  8.687,  8.688,  8.690,  8.692,  8.694,  8.695,  8.697,  8.699,  8.700, 
 8.702,  8.704,  8.706,  8.707,  8.709,  8.711,  8.713,  8.714,  8.716,  8.718, 
 8.719,  8.721,  8.722,  8.723,  8.725,  8.726,  8.728,  8.729,  8.731,  8.732, 
 8.734,  8.735,  8.737,  8.739,  8.740,  8.742,  8.744,  8.746,  8.748,  8.749, 
 8.751,  8.753,  8.755,  8.757,  8.759,  8.760,  8.762,  8.764,  8.766,  8.768, 
 8.770,  8.772,  8.774,  8.776,  8.778,  8.780,  8.782,  8.784,  8.786,  8.788, 
 8.789,  8.792,  8.795,  8.798,  8.801,  8.803,  8.806,  8.808,  8.810,  8.812, 
 8.815,  8.817,  8.819,  8.821,  8.823,  8.824,  8.826,  8.828,  8.830,  8.831, 
 8.833,  8.835,  8.836,  8.838,  8.839,  8.841,  8.842,  8.843,  8.845,  8.846, 
 8.847,  8.849,  8.850,  8.851,  8.852,  8.854,  8.855,  8.856,  8.857,  8.858, 
 8.859,  8.860,  8.861,  8.861,  8.862,  8.863,  8.864,  8.864,  8.865,  8.866, 
 8.867,  8.869,  8.870,  8.871,  8.872,  8.873,  8.875,  8.876,  8.878,  8.879, 
 8.881,  8.882,  8.884,  8.885,  8.887,  8.889,  8.891,  8.892,  8.894,  8.896, 
 8.898,  8.900,  8.902,  8.904,  8.906,  8.908,  8.911,  8.913,  8.915,  8.917, 
 8.919,  8.922,  8.924,  8.927,  8.929,  8.932,  8.934,  8.937,  8.939,  8.942, 
 8.944,  8.947,  8.949,  8.952,  8.955,  8.957,  8.960,  8.963,  8.965,  8.968, 
 8.971,  8.974,  8.977,  8.979,  8.982,  8.985,  8.988,  8.991,  8.994,  8.997, 
 9.000,  9.002,  9.005,  9.008,  9.011,  9.014,  9.017,  9.020,  9.023,  9.026, 
 9.029,  9.033,  9.036,  9.039,  9.042,  9.045,  9.049,  9.052,  9.055,  9.058, 
 9.061,  9.065,  9.068,  9.071,  9.074,  9.077,  9.081,  9.084,  9.087,  9.090, 
 9.094,  9.097,  9.100,  9.103,  9.107,  9.110,  9.113,  9.116,  9.120,  9.123, 
 9.126,  9.130,  9.133,  9.136,  9.139,  9.143,  9.146,  9.149,  9.153,  9.156, 
 9.159,  9.163,  9.166,  9.169,  9.173,  9.176,  9.179,  9.183,  9.186,  9.190, 
 9.193,  9.196,  9.200,  9.203,  9.207,  9.210,  9.214,  9.217,  9.221,  9.224, 
 9.228,  9.231,  9.235,  9.238,  9.242,  9.245,  9.249,  9.252,  9.256,  9.260, 
 9.263,  9.267,  9.270,  9.274,  9.278,  9.281,  9.285,  9.289,  9.292,  9.296, 
 9.299,  9.303,  9.307,  9.310,  9.314,  9.317,  9.321,  9.325,  9.328,  9.332, 
 9.336,  9.339,  9.343,  9.347,  9.351,  9.354,  9.358,  9.362,  9.366,  9.369, 
 9.373,  9.377,  9.381,  9.384,  9.388,  9.392,  9.396,  9.400,  9.404,  9.407, 
 9.411,  9.415,  9.419,  9.423,  9.426,  9.430,  9.434,  9.438,  9.442,  9.446, 
 9.449,  9.454,  9.458,  9.462,  9.466,  9.470,  9.474,  9.478,  9.482,  9.486, 
 9.490,  9.493,  9.497,  9.501,  9.505,  9.509,  9.513,  9.516,  9.520,  9.524, 
 9.528,  9.531,  9.535,  9.539,  9.542,  9.546,  9.550,  9.553,  9.557,  9.560, 
 9.564,  9.568,  9.571,  9.575,  9.578,  9.582,  9.585,  9.589,  9.592,  9.596, 
 9.599,  9.603,  9.606,  9.609,  9.612,  9.615,  9.618,  9.622,  9.625,  9.628, 
 9.632,  9.635,  9.638,  9.642,  9.645,  9.649,  9.652,  9.656,  9.659,  9.663, 
 9.666,  9.670,  9.673,  9.677,  9.681,  9.684,  9.688,  9.692,  9.695,  9.699, 
 9.703,  9.706,  9.710,  9.714,  9.717,  9.721,  9.725,  9.728,  9.732,  9.736, 
 9.739,  9.744,  9.748,  9.752,  9.756,  9.760,  9.764,  9.768,  9.772,  9.776, 
 9.780,  9.784,  9.787,  9.791,  9.795,  9.798,  9.802,  9.805,  9.809,  9.812, 
 9.816,  9.819,  9.823,  9.826,  9.829,  9.833,  9.836,  9.839,  9.842,  9.845, 
 9.849,  9.852,  9.855,  9.858,  9.861,  9.864,  9.867,  9.870,  9.873,  9.876, 
 9.879,  9.882,  9.885,  9.888,  9.890,  9.893,  9.896,  9.899,  9.901,  9.904, 
 9.907,  9.910,  9.913,  9.916,  9.919,  9.922,  9.925,  9.928,  9.931,  9.934, 
 9.937,  9.940,  9.943,  9.946,  9.949,  9.952,  9.955,  9.958,  9.961,  9.964, 
 9.968,  9.971,  9.974,  9.977,  9.980,  9.983,  9.987,  9.990,  9.993,  9.996, 
10.000, 10.003, 10.006, 10.009, 10.012, 10.015, 10.018, 10.022, 10.025, 10.028, 
10.031, 10.035, 10.038, 10.041, 10.044, 10.048, 10.051, 10.054, 10.057, 10.061, 
10.064, 10.067, 10.071, 10.074, 10.077, 10.080, 10.084, 10.087, 10.090, 10.094, 
10.097, 10.100, 10.103, 10.107, 10.110, 10.113, 10.117, 10.120, 10.123, 10.126, 
10.129, 10.133, 10.137, 10.140, 10.144, 10.147, 10.150, 10.154, 10.157, 10.160, 
10.164, 10.167, 10.170, 10.173, 10.176, 10.179, 10.182, 10.185, 10.188, 10.191, 
10.194, 10.197, 10.200, 10.203, 10.206, 10.209, 10.212, 10.215, 10.217, 10.220, 
10.223, 10.226, 10.228, 10.231, 10.234, 10.236, 10.239, 10.242, 10.244, 10.247, 
10.250, 10.252, 10.254, 10.256, 10.259, 10.261, 10.264, 10.266, 10.268, 10.271, 
10.273, 10.276, 10.278, 10.281, 10.283, 10.286, 10.288, 10.291, 10.293, 10.296, 
10.298, 10.301, 10.303, 10.306, 10.308, 10.311, 10.313, 10.316, 10.319, 10.321, 
10.324, 10.326, 10.329, 10.331, 10.334, 10.337, 10.339, 10.342, 10.344, 10.347, 
10.349, 10.352, 10.355, 10.358, 10.361, 10.363, 10.366, 10.369, 10.371, 10.374, 
10.377, 10.379, 10.382, 10.385, 10.387, 10.390, 10.392, 10.395, 10.397, 10.400, 
10.402, 10.405, 10.407, 10.410, 10.412, 10.414, 10.417, 10.419, 10.422, 10.424, 
10.426, 10.429, 10.431, 10.433, 10.436, 10.438, 10.440, 10.443, 10.445, 10.447, 
10.449, 10.452, 10.454, 10.456, 10.458, 10.460, 10.462, 10.465, 10.467, 10.469, 
10.471, 10.473, 10.476, 10.478, 10.480, 10.482, 10.484, 10.487, 10.489, 10.491, 
10.493, 10.496, 10.498, 10.500, 10.503, 10.505, 10.507, 10.509, 10.512, 10.514, 
10.516, 10.519, 10.521, 10.523, 10.526, 10.528, 10.530, 10.533, 10.535, 10.537, 
10.539, 10.542, 10.544, 10.546, 10.549, 10.551, 10.553, 10.556, 10.558, 10.560, 
10.563, 10.565, 10.567, 10.569, 10.572, 10.574, 10.576, 10.579, 10.581, 10.583, 
10.585, 10.588, 10.590, 10.592, 10.594, 10.597, 10.599, 10.601, 10.603, 10.606, 
10.608, 10.610, 10.612, 10.614, 10.617, 10.619, 10.621, 10.623, 10.625, 10.627, 
10.629, 10.632, 10.634, 10.637, 10.639, 10.641, 10.643, 10.646, 10.648, 10.650, 
10.652, 10.654, 10.656, 10.658, 10.661, 10.663, 10.665, 10.667, 10.669, 10.671, 
10.673, 10.674, 10.676, 10.678, 10.680, 10.682, 10.684, 10.686, 10.688, 10.690, 
10.691, 10.693, 10.695, 10.697, 10.699, 10.701, 10.702, 10.704, 10.706, 10.708, 
10.709, 10.711, 10.712, 10.714, 10.715, 10.717, 10.718, 10.720, 10.721, 10.723, 
10.724, 10.726, 10.728, 10.729, 10.731, 10.733, 10.735, 10.736, 10.738, 10.740, 
10.742, 10.744, 10.745, 10.747, 10.749, 10.751, 10.753, 10.755, 10.757, 10.758, 
10.760, 10.762, 10.764, 10.766, 10.768, 10.770, 10.772, 10.774, 10.776, 10.778, 
10.779, 10.782, 10.784, 10.786, 10.788, 10.791, 10.793, 10.795, 10.797, 10.799, 
10.801, 10.803, 10.805, 10.806, 10.808, 10.810, 10.812, 10.814, 10.816, 10.817, 
10.819, 10.821, 10.822, 10.824, 10.826, 10.827, 10.829, 10.830, 10.832, 10.834, 
10.835, 10.837, 10.838, 10.840, 10.841, 10.842, 10.844, 10.845, 10.847, 10.848, 
10.849, 10.851, 10.852, 10.853, 10.854, 10.856, 10.857, 10.858, 10.859, 10.861, 
10.862, 10.863, 10.864, 10.866, 10.867, 10.868, 10.869, 10.871, 10.872, 10.873, 
10.874, 10.876, 10.877, 10.878, 10.879, 10.881, 10.882, 10.883, 10.884, 10.886, 
10.887, 10.888, 10.889, 10.891, 10.892, 10.893, 10.894, 10.896, 10.897, 10.898, 
10.899, 10.901, 10.902, 10.904, 10.905, 10.906, 10.908, 10.909, 10.910, 10.911, 
10.913, 10.914, 10.915, 10.917, 10.918, 10.919, 10.920, 10.922, 10.923, 10.924, 
10.925, 10.927, 10.928, 10.929, 10.930, 10.931, 10.933, 10.934, 10.935, 10.936, 
10.937, 10.939, 10.940, 10.941, 10.942, 10.943, 10.945, 10.946, 10.947, 10.948, 
10.949, 10.950, 10.951, 10.952, 10.953, 10.954, 10.955, 10.956, 10.957, 10.958, 
10.959, 10.961, 10.962, 10.963, 10.964, 10.965, 10.967, 10.968, 10.969, 10.970, 
10.972, 10.973, 10.974, 10.976, 10.977, 10.978, 10.980, 10.981, 10.983, 10.984, 
10.985, 10.987, 10.988, 10.990, 10.991, 10.992, 10.994, 10.995, 10.997, 10.998, 
11.000, 11.001, 11.003, 11.005, 11.006, 11.008, 11.010, 11.011, 11.013, 11.014, 
11.016, 11.017, 11.019, 11.020, 11.021, 11.023, 11.024, 11.025, 11.027, 11.028, 
11.029, 11.030, 11.031, 11.033, 11.034, 11.035, 11.036, 11.037, 11.038, 11.039, 
11.040, 11.041, 11.042, 11.043, 11.044, 11.045, 11.046, 11.047, 11.048, 11.049, 
11.049, 11.050, 11.051, 11.052, 11.053, 11.053, 11.054, 11.055, 11.056, 11.057, 
11.057, 11.058, 11.059, 11.060, 11.060, 11.061, 11.062, 11.063, 11.063, 11.064, 
11.065, 11.066, 11.066, 11.067, 11.068, 11.069, 11.069, 11.070, 11.071, 11.071, 
11.072, 11.073, 11.074, 11.074, 11.075, 11.076, 11.077, 11.077, 11.078, 11.079, 
11.079, 11.080, 11.081, 11.081, 11.082, 11.082, 11.083, 11.084, 11.084, 11.085, 
11.086, 11.086, 11.087, 11.088, 11.089, 11.089, 11.090, 11.091, 11.092, 11.092, 
11.093, 11.094, 11.095, 11.095, 11.096, 11.097, 11.098, 11.099, 11.100, 11.100, 
11.101, 11.102, 11.103, 11.104, 11.104, 11.105, 11.106, 11.107, 11.108, 11.109, 
11.109, 11.111, 11.112, 11.112, 11.113, 11.114, 11.115, 11.116, 11.117, 11.118, 
11.119, 11.120, 11.121, 11.121, 11.122, 11.123, 11.124, 11.125, 11.125, 11.126, 
11.127, 11.127, 11.128, 11.129, 11.130, 11.130, 11.131, 11.132, 11.132, 11.133, 
11.134, 11.134, 11.135, 11.135, 11.136, 11.137, 11.137, 11.138, 11.138, 11.139, 
11.139, 11.140, 11.140, 11.141, 11.141, 11.142, 11.142, 11.143, 11.143, 11.144, 
11.144, 11.145, 11.145, 11.146, 11.146, 11.147, 11.147, 11.148, 11.148, 11.149, 
11.149, 11.150, 11.150, 11.151, 11.151, 11.152, 11.152, 11.153, 11.153, 11.154, 
11.154, 11.155, 11.155, 11.156, 11.156, 11.157, 11.157, 11.158, 11.158, 11.159, 
11.159, 11.160, 11.161, 11.161, 11.162, 11.162, 11.163, 11.163, 11.164, 11.164, 
11.165, 11.165, 11.166, 11.166, 11.167, 11.167, 11.168, 11.168, 11.169, 11.169, 
11.170, 11.170, 11.171, 11.171, 11.172, 11.172, 11.173, 11.173, 11.174, 11.174, 
11.175, 11.175, 11.176, 11.176, 11.177, 11.177, 11.178, 11.178, 11.179, 11.179, 
11.179, 11.180, 11.180, 11.180, 11.181, 11.181, 11.182, 11.182, 11.182, 11.183, 
11.183, 11.184, 11.184, 11.185, 11.185, 11.186, 11.186, 11.187, 11.187, 11.188, 
11.188, 11.189, 11.189, 11.190, 11.190, 11.191, 11.191, 11.192, 11.193, 11.193, 
11.194, 11.194, 11.195, 11.195, 11.196, 11.197, 11.197, 11.198, 11.198, 11.199, 
])
_vs_jma_2020a = np.array([
 2.796,  2.800,  2.804,  2.808,  2.812,  2.816,  2.820,  2.824,  2.828,  2.832, 
 2.921,  3.003,  3.076,  3.142,  3.203,  3.262,  3.321,  3.380,  3.437,  3.492, 
 3.545,  3.595,  3.642,  3.684,  3.722,  3.756,  3.786,  3.813,  3.838,  3.862, 
 3.884,  3.905,  3.926,  3.946,  3.966,  3.985,  4.004,  4.022,  4.039,  4.056, 
 4.072,  4.088,  4.103,  4.118,  4.133,  4.148,  4.163,  4.178,  4.193,  4.208, 
 4.223,  4.238,  4.253,  4.268,  4.281,  4.293,  4.302,  4.309,  4.316,  4.323, 
 4.328,  4.334,  4.338,  4.342,  4.346,  4.349,  4.352,  4.355,  4.357,  4.359, 
 4.362,  4.363,  4.365,  4.367,  4.369,  4.370,  4.372,  4.374,  4.376,  4.377, 
 4.379,  4.380,  4.381,  4.382,  4.384,  4.385,  4.386,  4.387,  4.388,  4.389, 
 4.391,  4.392,  4.393,  4.394,  4.395,  4.396,  4.397,  4.399,  4.400,  4.401, 
 4.402,  4.403,  4.403,  4.404,  4.404,  4.405,  4.405,  4.406,  4.407,  4.407, 
 4.408,  4.408,  4.409,  4.409,  4.410,  4.411,  4.411,  4.412,  4.412,  4.413, 
 4.413,  4.414,  4.415,  4.415,  4.416,  4.416,  4.417,  4.417,  4.418,  4.418, 
 4.419,  4.419,  4.419,  4.419,  4.419,  4.419,  4.419,  4.419,  4.419,  4.419, 
 4.419,  4.419,  4.419,  4.419,  4.419,  4.420,  4.420,  4.420,  4.420,  4.420, 
 4.420,  4.420,  4.420,  4.420,  4.420,  4.420,  4.420,  4.420,  4.420,  4.420, 
 4.420,  4.420,  4.421,  4.422,  4.423,  4.423,  4.424,  4.425,  4.425,  4.426, 
 4.427,  4.428,  4.428,  4.429,  4.429,  4.430,  4.430,  4.431,  4.431,  4.432, 
 4.433,  4.434,  4.434,  4.435,  4.435,  4.436,  4.437,  4.437,  4.438,  4.439, 
 4.440,  4.440,  4.441,  4.442,  4.442,  4.443,  4.443,  4.444,  4.445,  4.445, 
 4.446,  4.447,  4.447,  4.448,  4.448,  4.448,  4.449,  4.449,  4.450,  4.451, 
 4.451,  4.452,  4.453,  4.453,  4.454,  4.455,  4.456,  4.456,  4.457,  4.457, 
 4.458,  4.459,  4.460,  4.461,  4.461,  4.462,  4.463,  4.464,  4.465,  4.465, 
 4.466,  4.467,  4.468,  4.468,  4.469,  4.470,  4.471,  4.472,  4.473,  4.474, 
 4.474,  4.475,  4.476,  4.477,  4.478,  4.479,  4.480,  4.481,  4.482,  4.483, 
 4.484,  4.485,  4.486,  4.487,  4.488,  4.489,  4.489,  4.490,  4.491,  4.492, 
 4.492,  4.493,  4.494,  4.495,  4.496,  4.497,  4.498,  4.498,  4.499,  4.500, 
 4.501,  4.502,  4.503,  4.503,  4.504,  4.505,  4.506,  4.507,  4.507,  4.508, 
 4.508,  4.509,  4.510,  4.511,  4.512,  4.513,  4.514,  4.514,  4.515,  4.516, 
 4.517,  4.518,  4.518,  4.519,  4.519,  4.520,  4.521,  4.522,  4.522,  4.523, 
 4.524,  4.525,  4.525,  4.526,  4.527,  4.527,  4.528,  4.528,  4.529,  4.530, 
 4.531,  4.531,  4.532,  4.533,  4.533,  4.534,  4.535,  4.535,  4.536,  4.536, 
 4.537,  4.538,  4.538,  4.539,  4.539,  4.540,  4.540,  4.540,  4.541,  4.542, 
 4.542,  4.543,  4.544,  4.544,  4.545,  4.545,  4.546,  4.547,  4.548,  4.548, 
 4.549,  4.550,  4.551,  4.552,  4.552,  4.553,  4.554,  4.555,  4.556,  4.556, 
 4.557,  4.558,  4.559,  4.559,  4.560,  4.561,  4.562,  4.563,  4.564,  4.565, 
 4.565,  4.567,  4.568,  4.568,  4.569,  4.570,  4.571,  4.572,  4.573,  4.574, 
 4.575,  4.576,  4.577,  4.577,  4.578,  4.579,  4.580,  4.581,  4.582,  4.583, 
 4.584,  4.584,  4.585,  4.587,  4.588,  4.588,  4.589,  4.590,  4.591,  4.592, 
 4.593,  4.594,  4.595,  4.595,  4.597,  4.598,  4.598,  4.599,  4.600,  4.601, 
 4.602,  4.603,  4.603,  4.604,  4.604,  4.605,  4.606,  4.607,  4.607,  4.608, 
 4.609,  4.610,  4.610,  4.611,  4.611,  4.612,  4.613,  4.614,  4.614,  4.615, 
 4.616,  4.617,  4.617,  4.618,  4.619,  4.620,  4.621,  4.622,  4.622,  4.623, 
 4.624,  4.625,  4.626,  4.627,  4.627,  4.628,  4.629,  4.630,  4.631,  4.632, 
 4.632,  4.633,  4.634,  4.635,  4.636,  4.637,  4.638,  4.638,  4.639,  4.640, 
 4.641,  4.642,  4.642,  4.643,  4.643,  4.644,  4.645,  4.646,  4.646,  4.647, 
 4.648,  4.649,  4.649,  4.650,  4.651,  4.651,  4.653,  4.653,  4.654,  4.655, 
 4.656,  4.656,  4.657,  4.658,  4.659,  4.660,  4.660,  4.661,  4.662,  4.663, 
 4.663,  4.664,  4.664,  4.665,  4.665,  4.666,  4.666,  4.667,  4.667,  4.668, 
 4.669,  4.669,  4.670,  4.670,  4.671,  4.672,  4.672,  4.673,  4.674,  4.675, 
 4.675,  4.676,  4.677,  4.678,  4.679,  4.679,  4.681,  4.682,  4.682,  4.683, 
 4.684,  4.686,  4.686,  4.687,  4.688,  4.689,  4.690,  4.691,  4.692,  4.693, 
 4.694,  4.695,  4.696,  4.697,  4.698,  4.699,  4.700,  4.701,  4.702,  4.703, 
 4.704,  4.705,  4.706,  4.707,  4.708,  4.709,  4.710,  4.711,  4.712,  4.712, 
 4.713,  4.714,  4.715,  4.716,  4.717,  4.717,  4.718,  4.720,  4.720,  4.721, 
 4.722,  4.723,  4.723,  4.725,  4.726,  4.726,  4.727,  4.728,  4.729,  4.730, 
 4.730,  4.731,  4.732,  4.732,  4.733,  4.733,  4.734,  4.735,  4.735,  4.736, 
 4.737,  4.737,  4.737,  4.738,  4.739,  4.740,  4.740,  4.741,  4.742,  4.743, 
 4.743,  4.744,  4.745,  4.746,  4.746,  4.748,  4.749,  4.749,  4.750,  4.751, 
 4.752,  4.753,  4.754,  4.755,  4.755,  4.756,  4.757,  4.758,  4.759,  4.760, 
 4.761,  4.762,  4.763,  4.764,  4.764,  4.765,  4.766,  4.767,  4.768,  4.769, 
 4.769,  4.770,  4.771,  4.772,  4.773,  4.774,  4.774,  4.775,  4.776,  4.777, 
 4.778,  4.779,  4.780,  4.781,  4.781,  4.783,  4.784,  4.785,  4.785,  4.786, 
 4.788,  4.789,  4.789,  4.790,  4.791,  4.793,  4.793,  4.794,  4.795,  4.796, 
 4.797,  4.798,  4.799,  4.800,  4.801,  4.801,  4.802,  4.803,  4.804,  4.805, 
 4.806,  4.807,  4.808,  4.809,  4.809,  4.810,  4.811,  4.812,  4.812,  4.813, 
 4.814,  4.815,  4.816,  4.817,  4.818,  4.819,  4.820,  4.821,  4.822,  4.823, 
 4.824,  4.825,  4.826,  4.826,  4.828,  4.829,  4.830,  4.830,  4.831,  4.833, 
 4.833,  4.834,  4.835,  4.835,  4.836,  4.836,  4.837,  4.838,  4.839,  4.839, 
 4.840,  4.841,  4.842,  4.842,  4.843,  4.844,  4.845,  4.846,  4.847,  4.847, 
 4.848,  4.849,  4.850,  4.852,  4.853,  4.853,  4.854,  4.855,  4.857,  4.858, 
 4.859,  4.860,  4.861,  4.862,  4.863,  4.864,  4.865,  4.866,  4.868,  4.869, 
 4.869,  4.871,  4.873,  4.874,  4.876,  4.877,  4.879,  4.880,  4.881,  4.882, 
 4.884,  4.885,  4.886,  4.887,  4.888,  4.889,  4.890,  4.891,  4.892,  4.893, 
 4.894,  4.895,  4.895,  4.896,  4.897,  4.898,  4.899,  4.899,  4.900,  4.901, 
 4.901,  4.902,  4.903,  4.904,  4.904,  4.905,  4.906,  4.906,  4.907,  4.907, 
 4.908,  4.909,  4.909,  4.909,  4.910,  4.910,  4.911,  4.911,  4.911,  4.912, 
 4.912,  4.914,  4.914,  4.915,  4.915,  4.916,  4.917,  4.917,  4.919,  4.919, 
 4.920,  4.921,  4.922,  4.922,  4.924,  4.925,  4.926,  4.926,  4.927,  4.929, 
 4.930,  4.931,  4.932,  4.933,  4.934,  4.935,  4.937,  4.938,  4.939,  4.940, 
 4.941,  4.943,  4.944,  4.946,  4.947,  4.948,  4.950,  4.951,  4.952,  4.954, 
 4.955,  4.957,  4.958,  4.960,  4.961,  4.962,  4.964,  4.966,  4.967,  4.968, 
 4.970,  4.972,  4.973,  4.975,  4.976,  4.978,  4.980,  4.981,  4.983,  4.984, 
 4.986,  4.987,  4.989,  4.991,  4.992,  4.994,  4.996,  4.997,  4.999,  5.001, 
 5.002,  5.004,  5.006,  5.008,  5.009,  5.011,  5.013,  5.015,  5.017,  5.018, 
 5.020,  5.022,  5.024,  5.025,  5.027,  5.029,  5.031,  5.033,  5.034,  5.036, 
 5.038,  5.040,  5.042,  5.043,  5.045,  5.047,  5.049,  5.050,  5.053,  5.054, 
 5.056,  5.058,  5.060,  5.061,  5.063,  5.065,  5.067,  5.069,  5.071,  5.073, 
 5.074,  5.076,  5.078,  5.080,  5.082,  5.084,  5.085,  5.088,  5.089,  5.091, 
 5.093,  5.095,  5.097,  5.099,  5.101,  5.102,  5.105,  5.106,  5.109,  5.110, 
 5.112,  5.114,  5.116,  5.118,  5.120,  5.122,  5.124,  5.126,  5.128,  5.130, 
 5.132,  5.134,  5.136,  5.138,  5.140,  5.142,  5.144,  5.146,  5.148,  5.150, 
 5.152,  5.154,  5.156,  5.158,  5.160,  5.162,  5.164,  5.166,  5.168,  5.170, 
 5.172,  5.174,  5.176,  5.178,  5.181,  5.182,  5.184,  5.187,  5.189,  5.191, 
 5.193,  5.195,  5.197,  5.199,  5.201,  5.203,  5.206,  5.208,  5.210,  5.212, 
 5.214,  5.216,  5.218,  5.220,  5.222,  5.224,  5.227,  5.229,  5.231,  5.233, 
 5.235,  5.238,  5.240,  5.242,  5.244,  5.247,  5.249,  5.251,  5.253,  5.255, 
 5.258,  5.259,  5.261,  5.264,  5.266,  5.268,  5.270,  5.272,  5.274,  5.276, 
 5.279,  5.280,  5.283,  5.285,  5.286,  5.289,  5.291,  5.293,  5.295,  5.296, 
 5.299,  5.301,  5.302,  5.305,  5.306,  5.309,  5.310,  5.312,  5.314,  5.316, 
 5.318,  5.320,  5.322,  5.324,  5.325,  5.327,  5.329,  5.331,  5.332,  5.334, 
 5.336,  5.338,  5.340,  5.342,  5.343,  5.346,  5.347,  5.350,  5.351,  5.353, 
 5.355,  5.357,  5.359,  5.361,  5.363,  5.365,  5.367,  5.370,  5.371,  5.373, 
 5.376,  5.377,  5.380,  5.382,  5.383,  5.386,  5.388,  5.389,  5.392,  5.394, 
 5.396,  5.398,  5.401,  5.403,  5.405,  5.407,  5.409,  5.412,  5.414,  5.416, 
 5.418,  5.420,  5.422,  5.424,  5.427,  5.428,  5.430,  5.432,  5.434,  5.436, 
 5.438,  5.440,  5.442,  5.444,  5.445,  5.448,  5.449,  5.451,  5.453,  5.454, 
 5.457,  5.458,  5.460,  5.461,  5.463,  5.465,  5.466,  5.468,  5.470,  5.471, 
 5.473,  5.475,  5.476,  5.478,  5.479,  5.481,  5.483,  5.484,  5.485,  5.487, 
 5.489,  5.490,  5.492,  5.494,  5.495,  5.497,  5.499,  5.500,  5.502,  5.504, 
 5.505,  5.507,  5.509,  5.510,  5.512,  5.514,  5.515,  5.517,  5.519,  5.520, 
 5.522,  5.524,  5.526,  5.527,  5.529,  5.531,  5.533,  5.535,  5.536,  5.538, 
 5.540,  5.542,  5.543,  5.545,  5.547,  5.548,  5.550,  5.552,  5.554,  5.556, 
 5.557,  5.560,  5.561,  5.563,  5.565,  5.567,  5.568,  5.570,  5.572,  5.574, 
 5.576,  5.577,  5.580,  5.581,  5.583,  5.584,  5.587,  5.588,  5.590,  5.592, 
 5.594,  5.596,  5.597,  5.599,  5.601,  5.603,  5.605,  5.607,  5.608,  5.610, 
 5.612,  5.614,  5.616,  5.618,  5.620,  5.622,  5.623,  5.625,  5.627,  5.629, 
 5.631,  5.633,  5.634,  5.636,  5.638,  5.639,  5.641,  5.643,  5.644,  5.646, 
 5.648,  5.649,  5.651,  5.653,  5.654,  5.656,  5.658,  5.659,  5.660,  5.662, 
 5.664,  5.665,  5.666,  5.668,  5.670,  5.671,  5.673,  5.674,  5.675,  5.677, 
 5.679,  5.680,  5.681,  5.682,  5.684,  5.685,  5.686,  5.688,  5.689,  5.690, 
 5.691,  5.693,  5.694,  5.696,  5.697,  5.699,  5.700,  5.701,  5.702,  5.704, 
 5.705,  5.707,  5.708,  5.710,  5.711,  5.712,  5.714,  5.715,  5.717,  5.718, 
 5.720,  5.721,  5.722,  5.724,  5.725,  5.727,  5.728,  5.730,  5.731,  5.732, 
 5.734,  5.735,  5.737,  5.739,  5.740,  5.741,  5.743,  5.745,  5.746,  5.747, 
 5.749,  5.750,  5.752,  5.753,  5.755,  5.756,  5.757,  5.759,  5.760,  5.762, 
 5.763,  5.765,  5.766,  5.767,  5.768,  5.770,  5.771,  5.772,  5.774,  5.775, 
 5.776,  5.778,  5.779,  5.780,  5.782,  5.783,  5.784,  5.786,  5.787,  5.788, 
 5.789,  5.791,  5.792,  5.793,  5.794,  5.795,  5.796,  5.798,  5.799,  5.800, 
 5.801,  5.802,  5.804,  5.805,  5.806,  5.807,  5.808,  5.810,  5.811,  5.812, 
 5.813,  5.815,  5.816,  5.817,  5.819,  5.820,  5.821,  5.822,  5.824,  5.825, 
 5.826,  5.828,  5.829,  5.830,  5.832,  5.833,  5.834,  5.835,  5.837,  5.838, 
 5.839,  5.840,  5.842,  5.843,  5.844,  5.845,  5.847,  5.848,  5.849,  5.850, 
 5.852,  5.853,  5.854,  5.855,  5.857,  5.858,  5.859,  5.861,  5.862,  5.863, 
 5.864,  5.866,  5.867,  5.868,  5.869,  5.871,  5.872,  5.873,  5.874,  5.876, 
 5.877,  5.878,  5.879,  5.880,  5.882,  5.883,  5.884,  5.885,  5.886,  5.888, 
 5.889,  5.890,  5.891,  5.893,  5.894,  5.895,  5.896,  5.898,  5.899,  5.900, 
 5.901,  5.902,  5.904,  5.905,  5.906,  5.907,  5.909,  5.910,  5.911,  5.912, 
 5.913,  5.914,  5.915,  5.916,  5.917,  5.918,  5.919,  5.920,  5.921,  5.922, 
 5.923,  5.924,  5.925,  5.926,  5.927,  5.929,  5.929,  5.930,  5.931,  5.932, 
 5.933,  5.934,  5.935,  5.936,  5.936,  5.937,  5.938,  5.939,  5.940,  5.941, 
 5.941,  5.942,  5.943,  5.944,  5.945,  5.946,  5.947,  5.948,  5.949,  5.950, 
 5.951,  5.952,  5.953,  5.954,  5.955,  5.956,  5.957,  5.958,  5.960,  5.960, 
 5.961,  5.962,  5.963,  5.965,  5.966,  5.967,  5.968,  5.969,  5.970,  5.971, 
 5.972,  5.973,  5.975,  5.976,  5.977,  5.978,  5.980,  5.981,  5.982,  5.983, 
 5.984,  5.985,  5.986,  5.987,  5.988,  5.989,  5.990,  5.991,  5.992,  5.993, 
 5.994,  5.995,  5.996,  5.997,  5.998,  5.998,  5.999,  6.000,  6.001,  6.002, 
 6.003,  6.004,  6.004,  6.006,  6.006,  6.007,  6.008,  6.008,  6.009,  6.010, 
 6.011,  6.012,  6.012,  6.013,  6.013,  6.014,  6.015,  6.016,  6.016,  6.017, 
 6.018,  6.018,  6.019,  6.020,  6.020,  6.021,  6.022,  6.023,  6.023,  6.024, 
 6.024,  6.025,  6.026,  6.027,  6.027,  6.028,  6.029,  6.029,  6.030,  6.031, 
 6.032,  6.032,  6.033,  6.034,  6.034,  6.035,  6.035,  6.037,  6.037,  6.038, 
 6.038,  6.039,  6.040,  6.041,  6.042,  6.042,  6.043,  6.044,  6.044,  6.045, 
 6.046,  6.047,  6.047,  6.048,  6.049,  6.049,  6.050,  6.051,  6.052,  6.052, 
 6.053,  6.054,  6.054,  6.055,  6.055,  6.056,  6.057,  6.058,  6.058,  6.059, 
 6.059,  6.060,  6.061,  6.061,  6.062,  6.063,  6.064,  6.064,  6.065,  6.065, 
 6.066,  6.066,  6.067,  6.068,  6.068,  6.069,  6.069,  6.070,  6.070,  6.071, 
 6.071,  6.073,  6.073,  6.074,  6.074,  6.075,  6.076,  6.076,  6.077,  6.078, 
 6.079,  6.079,  6.080,  6.081,  6.081,  6.082,  6.083,  6.084,  6.085,  6.085, 
 6.086,  6.087,  6.088,  6.089,  6.089,  6.090,  6.091,  6.091,  6.093,  6.093, 
 6.094,  6.095,  6.096,  6.097,  6.098,  6.099,  6.100,  6.100,  6.101,  6.102, 
 6.103,  6.104,  6.105,  6.105,  6.106,  6.107,  6.107,  6.108,  6.109,  6.110, 
 6.110,  6.111,  6.111,  6.112,  6.113,  6.114,  6.114,  6.115,  6.115,  6.116, 
 6.116,  6.117,  6.117,  6.118,  6.119,  6.119,  6.120,  6.120,  6.121,  6.121, 
 6.121,  6.122,  6.122,  6.123,  6.124,  6.124,  6.124,  6.125,  6.125,  6.126, 
 6.126,  6.126,  6.127,  6.127,  6.127,  6.128,  6.129,  6.129,  6.129,  6.130, 
 6.130,  6.131,  6.131,  6.131,  6.132,  6.132,  6.132,  6.133,  6.134,  6.134, 
 6.134,  6.135,  6.135,  6.135,  6.136,  6.136,  6.137,  6.137,  6.137,  6.138, 
 6.138,  6.139,  6.139,  6.139,  6.140,  6.140,  6.140,  6.141,  6.141,  6.141, 
 6.142,  6.142,  6.142,  6.143,  6.143,  6.143,  6.144,  6.145,  6.145,  6.145, 
 6.146,  6.146,  6.147,  6.147,  6.147,  6.148,  6.148,  6.149,  6.150,  6.150, 
 6.150,  6.151,  6.151,  6.152,  6.152,  6.152,  6.153,  6.153,  6.154,  6.155, 
 6.155,  6.156,  6.156,  6.156,  6.157,  6.157,  6.158,  6.158,  6.159,  6.160, 
 6.160,  6.161,  6.161,  6.161,  6.162,  6.162,  6.163,  6.163,  6.163,  6.164, 
 6.165,  6.165,  6.165,  6.166,  6.166,  6.166,  6.167,  6.167,  6.167,  6.168, 
 6.168,  6.168,  6.169,  6.169,  6.170,  6.170,  6.170,  6.171,  6.171,  6.171, 
 6.171,  6.172,  6.172,  6.172,  6.172,  6.173,  6.173,  6.173,  6.173,  6.174, 
 6.174,  6.175,  6.175,  6.175,  6.175,  6.176,  6.176,  6.176,  6.176,  6.177, 
 6.177,  6.177,  6.177,  6.178,  6.178,  6.178,  6.178,  6.179,  6.179,  6.180, 
 6.180,  6.180,  6.180,  6.181,  6.181,  6.181,  6.181,  6.182,  6.182,  6.182, 
 6.182,  6.183,  6.183,  6.183,  6.184,  6.184,  6.184,  6.184,  6.185,  6.185, 
 6.186,  6.186,  6.186,  6.186,  6.187,  6.187,  6.187,  6.187,  6.188,  6.188, 
 6.188,  6.188,  6.189,  6.189,  6.189,  6.189,  6.190,  6.190,  6.191,  6.191, 
 6.191,  6.191,  6.192,  6.192,  6.192,  6.192,  6.193,  6.193,  6.193,  6.193, 
 6.193,  6.194,  6.194,  6.194,  6.194,  6.194,  6.195,  6.195,  6.195,  6.196, 
 6.196,  6.196,  6.196,  6.197,  6.197,  6.197,  6.197,  6.198,  6.198,  6.198, 
 6.198,  6.199,  6.199,  6.199,  6.199,  6.200,  6.200,  6.201,  6.201,  6.201, 
 6.202,  6.202,  6.202,  6.202,  6.203,  6.203,  6.203,  6.204,  6.204,  6.204, 
])
_dep_jma_2020a = np.array([
  0.0,   0.5,   1.0,   1.5,   2.0,   2.5,   3.0,   3.5,   4.0,   4.5, 
  5.0,   5.5,   6.0,   6.5,   7.0,   7.5,   8.0,   8.5,   9.0,   9.5, 
 10.0,  10.5,  11.0,  11.5,  12.0,  12.5,  13.0,  13.5,  14.0,  14.5, 
 15.0,  15.5,  16.0,  16.5,  17.0,  17.5,  18.0,  18.5,  19.0,  19.5, 
 20.0,  20.5,  21.0,  21.5,  22.0,  22.5,  23.0,  23.5,  24.0,  24.5, 
 25.0,  25.5,  26.0,  26.5,  27.0,  27.5,  28.0,  28.5,  29.0,  29.5, 
 30.0,  30.5,  31.0,  31.5,  32.0,  32.5,  33.0,  33.5,  34.0,  34.5, 
 35.0,  35.5,  36.0,  36.5,  37.0,  37.5,  38.0,  38.5,  39.0,  39.5, 
 40.0,  40.5,  41.0,  41.5,  42.0,  42.5,  43.0,  43.5,  44.0,  44.5, 
 45.0,  45.5,  46.0,  46.5,  47.0,  47.5,  48.0,  48.5,  49.0,  49.5, 
 50.0,  50.5,  51.0,  51.5,  52.0,  52.5,  53.0,  53.5,  54.0,  54.5, 
 55.0,  55.5,  56.0,  56.5,  57.0,  57.5,  58.0,  58.5,  59.0,  59.5, 
 60.0,  60.5,  61.0,  61.5,  62.0,  62.5,  63.0,  63.5,  64.0,  64.5, 
 65.0,  65.5,  66.0,  66.5,  67.0,  67.5,  68.0,  68.5,  69.0,  69.5, 
 70.0,  70.5,  71.0,  71.5,  72.0,  72.5,  73.0,  73.5,  74.0,  74.5, 
 75.0,  75.5,  76.0,  76.5,  77.0,  77.5,  78.0,  78.5,  79.0,  79.5, 
 80.0,  80.5,  81.0,  81.5,  82.0,  82.5,  83.0,  83.5,  84.0,  84.5, 
 85.0,  85.5,  86.0,  86.5,  87.0,  87.5,  88.0,  88.5,  89.0,  89.5, 
 90.0,  90.5,  91.0,  91.5,  92.0,  92.5,  93.0,  93.5,  94.0,  94.5, 
 95.0,  95.5,  96.0,  96.5,  97.0,  97.5,  98.0,  98.5,  99.0,  99.5, 
100.0, 100.5, 101.0, 101.5, 102.0, 102.5, 103.0, 103.5, 104.0, 104.5, 
105.0, 105.5, 106.0, 106.5, 107.0, 107.5, 108.0, 108.5, 109.0, 109.5, 
110.0, 110.5, 111.0, 111.5, 112.0, 112.5, 113.0, 113.5, 114.0, 114.5, 
115.0, 115.5, 116.0, 116.5, 117.0, 117.5, 118.0, 118.5, 119.0, 119.5, 
120.0, 120.5, 121.0, 121.5, 122.0, 122.5, 123.0, 123.5, 124.0, 124.5, 
125.0, 125.5, 126.0, 126.5, 127.0, 127.5, 128.0, 128.5, 129.0, 129.5, 
130.0, 130.5, 131.0, 131.5, 132.0, 132.5, 133.0, 133.5, 134.0, 134.5, 
135.0, 135.5, 136.0, 136.5, 137.0, 137.5, 138.0, 138.5, 139.0, 139.5, 
140.0, 140.5, 141.0, 141.5, 142.0, 142.5, 143.0, 143.5, 144.0, 144.5, 
145.0, 145.5, 146.0, 146.5, 147.0, 147.5, 148.0, 148.5, 149.0, 149.5, 
150.0, 150.5, 151.0, 151.5, 152.0, 152.5, 153.0, 153.5, 154.0, 154.5, 
155.0, 155.5, 156.0, 156.5, 157.0, 157.5, 158.0, 158.5, 159.0, 159.5, 
160.0, 160.5, 161.0, 161.5, 162.0, 162.5, 163.0, 163.5, 164.0, 164.5, 
165.0, 165.5, 166.0, 166.5, 167.0, 167.5, 168.0, 168.5, 169.0, 169.5, 
170.0, 170.5, 171.0, 171.5, 172.0, 172.5, 173.0, 173.5, 174.0, 174.5, 
175.0, 175.5, 176.0, 176.5, 177.0, 177.5, 178.0, 178.5, 179.0, 179.5, 
180.0, 180.5, 181.0, 181.5, 182.0, 182.5, 183.0, 183.5, 184.0, 184.5, 
185.0, 185.5, 186.0, 186.5, 187.0, 187.5, 188.0, 188.5, 189.0, 189.5, 
190.0, 190.5, 191.0, 191.5, 192.0, 192.5, 193.0, 193.5, 194.0, 194.5, 
195.0, 195.5, 196.0, 196.5, 197.0, 197.5, 198.0, 198.5, 199.0, 199.5, 
200.0, 200.5, 201.0, 201.5, 202.0, 202.5, 203.0, 203.5, 204.0, 204.5, 
205.0, 205.5, 206.0, 206.5, 207.0, 207.5, 208.0, 208.5, 209.0, 209.5, 
210.0, 210.5, 211.0, 211.5, 212.0, 212.5, 213.0, 213.5, 214.0, 214.5, 
215.0, 215.5, 216.0, 216.5, 217.0, 217.5, 218.0, 218.5, 219.0, 219.5, 
220.0, 220.5, 221.0, 221.5, 222.0, 222.5, 223.0, 223.5, 224.0, 224.5, 
225.0, 225.5, 226.0, 226.5, 227.0, 227.5, 228.0, 228.5, 229.0, 229.5, 
230.0, 230.5, 231.0, 231.5, 232.0, 232.5, 233.0, 233.5, 234.0, 234.5, 
235.0, 235.5, 236.0, 236.5, 237.0, 237.5, 238.0, 238.5, 239.0, 239.5, 
240.0, 240.5, 241.0, 241.5, 242.0, 242.5, 243.0, 243.5, 244.0, 244.5, 
245.0, 245.5, 246.0, 246.5, 247.0, 247.5, 248.0, 248.5, 249.0, 249.5, 
250.0, 250.5, 251.0, 251.5, 252.0, 252.5, 253.0, 253.5, 254.0, 254.5, 
255.0, 255.5, 256.0, 256.5, 257.0, 257.5, 258.0, 258.5, 259.0, 259.5, 
260.0, 260.5, 261.0, 261.5, 262.0, 262.5, 263.0, 263.5, 264.0, 264.5, 
265.0, 265.5, 266.0, 266.5, 267.0, 267.5, 268.0, 268.5, 269.0, 269.5, 
270.0, 270.5, 271.0, 271.5, 272.0, 272.5, 273.0, 273.5, 274.0, 274.5, 
275.0, 275.5, 276.0, 276.5, 277.0, 277.5, 278.0, 278.5, 279.0, 279.5, 
280.0, 280.5, 281.0, 281.5, 282.0, 282.5, 283.0, 283.5, 284.0, 284.5, 
285.0, 285.5, 286.0, 286.5, 287.0, 287.5, 288.0, 288.5, 289.0, 289.5, 
290.0, 290.5, 291.0, 291.5, 292.0, 292.5, 293.0, 293.5, 294.0, 294.5, 
295.0, 295.5, 296.0, 296.5, 297.0, 297.5, 298.0, 298.5, 299.0, 299.5, 
300.0, 300.5, 301.0, 301.5, 302.0, 302.5, 303.0, 303.5, 304.0, 304.5, 
305.0, 305.5, 306.0, 306.5, 307.0, 307.5, 308.0, 308.5, 309.0, 309.5, 
310.0, 310.5, 311.0, 311.5, 312.0, 312.5, 313.0, 313.5, 314.0, 314.5, 
315.0, 315.5, 316.0, 316.5, 317.0, 317.5, 318.0, 318.5, 319.0, 319.5, 
320.0, 320.5, 321.0, 321.5, 322.0, 322.5, 323.0, 323.5, 324.0, 324.5, 
325.0, 325.5, 326.0, 326.5, 327.0, 327.5, 328.0, 328.5, 329.0, 329.5, 
330.0, 330.5, 331.0, 331.5, 332.0, 332.5, 333.0, 333.5, 334.0, 334.5, 
335.0, 335.5, 336.0, 336.5, 337.0, 337.5, 338.0, 338.5, 339.0, 339.5, 
340.0, 340.5, 341.0, 341.5, 342.0, 342.5, 343.0, 343.5, 344.0, 344.5, 
345.0, 345.5, 346.0, 346.5, 347.0, 347.5, 348.0, 348.5, 349.0, 349.5, 
350.0, 350.5, 351.0, 351.5, 352.0, 352.5, 353.0, 353.5, 354.0, 354.5, 
355.0, 355.5, 356.0, 356.5, 357.0, 357.5, 358.0, 358.5, 359.0, 359.5, 
360.0, 360.5, 361.0, 361.5, 362.0, 362.5, 363.0, 363.5, 364.0, 364.5, 
365.0, 365.5, 366.0, 366.5, 367.0, 367.5, 368.0, 368.5, 369.0, 369.5, 
370.0, 370.5, 371.0, 371.5, 372.0, 372.5, 373.0, 373.5, 374.0, 374.5, 
375.0, 375.5, 376.0, 376.5, 377.0, 377.5, 378.0, 378.5, 379.0, 379.5, 
380.0, 380.5, 381.0, 381.5, 382.0, 382.5, 383.0, 383.5, 384.0, 384.5, 
385.0, 385.5, 386.0, 386.5, 387.0, 387.5, 388.0, 388.5, 389.0, 389.5, 
390.0, 390.5, 391.0, 391.5, 392.0, 392.5, 393.0, 393.5, 394.0, 394.5, 
395.0, 395.5, 396.0, 396.5, 397.0, 397.5, 398.0, 398.5, 399.0, 399.5, 
400.0, 400.5, 401.0, 401.5, 402.0, 402.5, 403.0, 403.5, 404.0, 404.5, 
405.0, 405.5, 406.0, 406.5, 407.0, 407.5, 408.0, 408.5, 409.0, 409.5, 
410.0, 410.5, 411.0, 411.5, 412.0, 412.5, 413.0, 413.5, 414.0, 414.5, 
415.0, 415.5, 416.0, 416.5, 417.0, 417.5, 418.0, 418.5, 419.0, 419.5, 
420.0, 420.5, 421.0, 421.5, 422.0, 422.5, 423.0, 423.5, 424.0, 424.5, 
425.0, 425.5, 426.0, 426.5, 427.0, 427.5, 428.0, 428.5, 429.0, 429.5, 
430.0, 430.5, 431.0, 431.5, 432.0, 432.5, 433.0, 433.5, 434.0, 434.5, 
435.0, 435.5, 436.0, 436.5, 437.0, 437.5, 438.0, 438.5, 439.0, 439.5, 
440.0, 440.5, 441.0, 441.5, 442.0, 442.5, 443.0, 443.5, 444.0, 444.5, 
445.0, 445.5, 446.0, 446.5, 447.0, 447.5, 448.0, 448.5, 449.0, 449.5, 
450.0, 450.5, 451.0, 451.5, 452.0, 452.5, 453.0, 453.5, 454.0, 454.5, 
455.0, 455.5, 456.0, 456.5, 457.0, 457.5, 458.0, 458.5, 459.0, 459.5, 
460.0, 460.5, 461.0, 461.5, 462.0, 462.5, 463.0, 463.5, 464.0, 464.5, 
465.0, 465.5, 466.0, 466.5, 467.0, 467.5, 468.0, 468.5, 469.0, 469.5, 
470.0, 470.5, 471.0, 471.5, 472.0, 472.5, 473.0, 473.5, 474.0, 474.5, 
475.0, 475.5, 476.0, 476.5, 477.0, 477.5, 478.0, 478.5, 479.0, 479.5, 
480.0, 480.5, 481.0, 481.5, 482.0, 482.5, 483.0, 483.5, 484.0, 484.5, 
485.0, 485.5, 486.0, 486.5, 487.0, 487.5, 488.0, 488.5, 489.0, 489.5, 
490.0, 490.5, 491.0, 491.5, 492.0, 492.5, 493.0, 493.5, 494.0, 494.5, 
495.0, 495.5, 496.0, 496.5, 497.0, 497.5, 498.0, 498.5, 499.0, 499.5, 
500.0, 500.5, 501.0, 501.5, 502.0, 502.5, 503.0, 503.5, 504.0, 504.5, 
505.0, 505.5, 506.0, 506.5, 507.0, 507.5, 508.0, 508.5, 509.0, 509.5, 
510.0, 510.5, 511.0, 511.5, 512.0, 512.5, 513.0, 513.5, 514.0, 514.5, 
515.0, 515.5, 516.0, 516.5, 517.0, 517.5, 518.0, 518.5, 519.0, 519.5, 
520.0, 520.5, 521.0, 521.5, 522.0, 522.5, 523.0, 523.5, 524.0, 524.5, 
525.0, 525.5, 526.0, 526.5, 527.0, 527.5, 528.0, 528.5, 529.0, 529.5, 
530.0, 530.5, 531.0, 531.5, 532.0, 532.5, 533.0, 533.5, 534.0, 534.5, 
535.0, 535.5, 536.0, 536.5, 537.0, 537.5, 538.0, 538.5, 539.0, 539.5, 
540.0, 540.5, 541.0, 541.5, 542.0, 542.5, 543.0, 543.5, 544.0, 544.5, 
545.0, 545.5, 546.0, 546.5, 547.0, 547.5, 548.0, 548.5, 549.0, 549.5, 
550.0, 550.5, 551.0, 551.5, 552.0, 552.5, 553.0, 553.5, 554.0, 554.5, 
555.0, 555.5, 556.0, 556.5, 557.0, 557.5, 558.0, 558.5, 559.0, 559.5, 
560.0, 560.5, 561.0, 561.5, 562.0, 562.5, 563.0, 563.5, 564.0, 564.5, 
565.0, 565.5, 566.0, 566.5, 567.0, 567.5, 568.0, 568.5, 569.0, 569.5, 
570.0, 570.5, 571.0, 571.5, 572.0, 572.5, 573.0, 573.5, 574.0, 574.5, 
575.0, 575.5, 576.0, 576.5, 577.0, 577.5, 578.0, 578.5, 579.0, 579.5, 
580.0, 580.5, 581.0, 581.5, 582.0, 582.5, 583.0, 583.5, 584.0, 584.5, 
585.0, 585.5, 586.0, 586.5, 587.0, 587.5, 588.0, 588.5, 589.0, 589.5, 
590.0, 590.5, 591.0, 591.5, 592.0, 592.5, 593.0, 593.5, 594.0, 594.5, 
595.0, 595.5, 596.0, 596.5, 597.0, 597.5, 598.0, 598.5, 599.0, 599.5, 
600.0, 600.5, 601.0, 601.5, 602.0, 602.5, 603.0, 603.5, 604.0, 604.5, 
605.0, 605.5, 606.0, 606.5, 607.0, 607.5, 608.0, 608.5, 609.0, 609.5, 
610.0, 610.5, 611.0, 611.5, 612.0, 612.5, 613.0, 613.5, 614.0, 614.5, 
615.0, 615.5, 616.0, 616.5, 617.0, 617.5, 618.0, 618.5, 619.0, 619.5, 
620.0, 620.5, 621.0, 621.5, 622.0, 622.5, 623.0, 623.5, 624.0, 624.5, 
625.0, 625.5, 626.0, 626.5, 627.0, 627.5, 628.0, 628.5, 629.0, 629.5, 
630.0, 630.5, 631.0, 631.5, 632.0, 632.5, 633.0, 633.5, 634.0, 634.5, 
635.0, 635.5, 636.0, 636.5, 637.0, 637.5, 638.0, 638.5, 639.0, 639.5, 
640.0, 640.5, 641.0, 641.5, 642.0, 642.5, 643.0, 643.5, 644.0, 644.5, 
645.0, 645.5, 646.0, 646.5, 647.0, 647.5, 648.0, 648.5, 649.0, 649.5, 
650.0, 650.5, 651.0, 651.5, 652.0, 652.5, 653.0, 653.5, 654.0, 654.5, 
655.0, 655.5, 656.0, 656.5, 657.0, 657.5, 658.0, 658.5, 659.0, 659.5, 
660.0, 660.5, 661.0, 661.5, 662.0, 662.5, 663.0, 663.5, 664.0, 664.5, 
665.0, 665.5, 666.0, 666.5, 667.0, 667.5, 668.0, 668.5, 669.0, 669.5, 
670.0, 670.5, 671.0, 671.5, 672.0, 672.5, 673.0, 673.5, 674.0, 674.5, 
675.0, 675.5, 676.0, 676.5, 677.0, 677.5, 678.0, 678.5, 679.0, 679.5, 
680.0, 680.5, 681.0, 681.5, 682.0, 682.5, 683.0, 683.5, 684.0, 684.5, 
685.0, 685.5, 686.0, 686.5, 687.0, 687.5, 688.0, 688.5, 689.0, 689.5, 
690.0, 690.5, 691.0, 691.5, 692.0, 692.5, 693.0, 693.5, 694.0, 694.5, 
695.0, 695.5, 696.0, 696.5, 697.0, 697.5, 698.0, 698.5, 699.0, 699.5, 
700.0, 700.5, 701.0, 701.5, 702.0, 702.5, 703.0, 703.5, 704.0, 704.5, 
705.0, 705.5, 706.0, 706.5, 707.0, 707.5, 708.0, 708.5, 709.0, 709.5, 
710.0, 710.5, 711.0, 711.5, 712.0, 712.5, 713.0, 713.5, 714.0, 714.5, 
715.0, 715.5, 716.0, 716.5, 717.0, 717.5, 718.0, 718.5, 719.0, 719.5, 
720.0, 720.5, 721.0, 721.5, 722.0, 722.5, 723.0, 723.5, 724.0, 724.5, 
725.0, 725.5, 726.0, 726.5, 727.0, 727.5, 728.0, 728.5, 729.0, 729.5, 
730.0, 730.5, 731.0, 731.5, 732.0, 732.5, 733.0, 733.5, 734.0, 734.5, 
735.0, 735.5, 736.0, 736.5, 737.0, 737.5, 738.0, 738.5, 739.0, 739.5, 
740.0, 740.5, 741.0, 741.5, 742.0, 742.5, 743.0, 743.5, 744.0, 744.5, 
745.0, 745.5, 746.0, 746.5, 747.0, 747.5, 748.0, 748.5, 749.0, 749.5, 
750.0, 750.5, 751.0, 751.5, 752.0, 752.5, 753.0, 753.5, 754.0, 754.5, 
755.0, 755.5, 756.0, 756.5, 757.0, 757.5, 758.0, 758.5, 759.0, 759.5, 
760.0, 760.5, 761.0, 761.5, 762.0, 762.5, 763.0, 763.5, 764.0, 764.5, 
765.0, 765.5, 766.0, 766.5, 767.0, 767.5, 768.0, 768.5, 769.0, 769.5, 
770.0, 770.5, 771.0, 771.5, 772.0, 772.5, 773.0, 773.5, 774.0, 774.5, 
775.0, 775.5, 776.0, 776.5, 777.0, 777.5, 778.0, 778.5, 779.0, 779.5, 
780.0, 780.5, 781.0, 781.5, 782.0, 782.5, 783.0, 783.5, 784.0, 784.5, 
785.0, 785.5, 786.0, 786.5, 787.0, 787.5, 788.0, 788.5, 789.0, 789.5, 
790.0, 790.5, 791.0, 791.5, 792.0, 792.5, 793.0, 793.5, 794.0, 794.5, 
795.0, 795.5, 796.0, 796.5, 797.0, 797.5, 798.0, 798.5, 799.0, 799.5, 
800.0, 800.5, 801.0, 801.5, 802.0, 802.5, 803.0, 803.5, 804.0, 804.5, 
805.0, 805.5, 806.0, 806.5, 807.0, 807.5, 808.0, 808.5, 809.0, 809.5, 
810.0, 810.5, 811.0, 811.5, 812.0, 812.5, 813.0, 813.5, 814.0, 814.5, 
815.0, 815.5, 816.0, 816.5, 817.0, 817.5, 818.0, 818.5, 819.0, 819.5, 
820.0, 820.5, 821.0, 821.5, 822.0, 822.5, 823.0, 823.5, 824.0, 824.5, 
825.0, 825.5, 826.0, 826.5, 827.0, 827.5, 828.0, 828.5, 829.0, 829.5, 
830.0, 830.5, 831.0, 831.5, 832.0, 832.5, 833.0, 833.5, 834.0, 834.5, 
835.0, 835.5, 836.0, 836.5, 837.0, 837.5, 838.0, 838.5, 839.0, 839.5, 
840.0, 840.5, 841.0, 841.5, 842.0, 842.5, 843.0, 843.5, 844.0, 844.5, 
845.0, 845.5, 846.0, 846.5, 847.0, 847.5, 848.0, 848.5, 849.0, 849.5, 
850.0, 850.5, 851.0, 851.5, 852.0, 852.5, 853.0, 853.5, 854.0, 854.5, 
855.0, 855.5, 856.0, 856.5, 857.0, 857.5, 858.0, 858.5, 859.0, 859.5, 
860.0, 860.5, 861.0, 861.5, 862.0, 862.5, 863.0, 863.5, 864.0, 864.5, 
865.0, 865.5, 866.0, 866.5, 867.0, 867.5, 868.0, 868.5, 869.0, 869.5, 
870.0, 870.5, 871.0, 871.5, 872.0, 872.5, 873.0, 873.5, 874.0, 874.5, 
875.0, 875.5, 876.0, 876.5, 877.0, 877.5, 878.0, 878.5, 879.0, 879.5, 
880.0, 880.5, 881.0, 881.5, 882.0, 882.5, 883.0, 883.5, 884.0, 884.5, 
885.0, 885.5, 886.0, 886.5, 887.0, 887.5, 888.0, 888.5, 889.0, 889.5, 
890.0, 890.5, 891.0, 891.5, 892.0, 892.5, 893.0, 893.5, 894.0, 894.5, 
895.0, 895.5, 896.0, 896.5, 897.0, 897.5, 898.0, 898.5, 899.0, 899.5, 
900.0, 900.5, 901.0, 901.5, 902.0, 902.5, 903.0, 903.5, 904.0, 904.5, 
905.0, 905.5, 906.0, 906.5, 907.0, 907.5, 908.0, 908.5, 909.0, 909.5, 
910.0, 910.5, 911.0, 911.5, 912.0, 912.5, 913.0, 913.5, 914.0, 914.5, 
915.0, 915.5, 916.0, 916.5, 917.0, 917.5, 918.0, 918.5, 919.0, 919.5, 
920.0, 920.5, 921.0, 921.5, 922.0, 922.5, 923.0, 923.5, 924.0, 924.5, 
925.0, 925.5, 926.0, 926.5, 927.0, 927.5, 928.0, 928.5, 929.0, 929.5, 
930.0, 930.5, 931.0, 931.5, 932.0, 932.5, 933.0, 933.5, 934.0, 934.5, 
935.0, 935.5, 936.0, 936.5, 937.0, 937.5, 938.0, 938.5, 939.0, 939.5, 
])
_f_vp_jma_2020a = scipy.interpolate.interp1d(_dep_jma_2020a, _vp_jma_2020a, kind='linear')
_f_vs_jma_2020a = scipy.interpolate.interp1d(_dep_jma_2020a, _vs_jma_2020a, kind='linear')
_vp_jma_2020b = np.array([
 4.800,  4.807,  4.814,  4.821,  4.828,  4.835,  4.842,  4.849,  4.856,  4.863, 
 4.870,  4.877,  4.884,  4.891,  5.127,  5.491,  5.727,  5.899,  6.039,  6.163, 
 6.279,  6.391,  6.503,  6.615,  6.727,  6.839,  6.951,  7.063,  7.175,  7.287, 
 7.399,  7.511,  7.603,  7.677,  7.735,  7.779,  7.811,  7.833,  7.847,  7.855, 
 7.859,  7.861,  7.863,  7.865,  7.867,  7.869,  7.871,  7.873,  7.875,  7.877, 
 7.879,  7.881,  7.883,  7.885,  7.887,  7.889,  7.891,  7.893,  7.895,  7.897, 
 7.899,  7.901,  7.903,  7.905,  7.907,  7.909,  7.911,  7.913,  7.915,  7.917, 
 7.919,  7.921,  7.923,  7.925,  7.927,  7.929,  7.931,  7.933,  7.935,  7.937, 
 7.939,  7.941,  7.943,  7.945,  7.947,  7.949,  7.951,  7.953,  7.955,  7.957, 
 7.959,  7.961,  7.963,  7.965,  7.967,  7.969,  7.971,  7.973,  7.975,  7.977, 
 7.979,  7.981,  7.983,  7.985,  7.987,  7.989,  7.991,  7.993,  7.995,  7.997, 
 7.999,  8.001,  8.003,  8.005,  8.007,  8.009,  8.012,  8.014,  8.016,  8.018, 
 8.020,  8.022,  8.023,  8.025,  8.027,  8.029,  8.030,  8.032,  8.034,  8.035, 
 8.037,  8.039,  8.041,  8.042,  8.044,  8.046,  8.048,  8.050,  8.051,  8.053, 
 8.055,  8.057,  8.059,  8.061,  8.063,  8.066,  8.068,  8.070,  8.072,  8.074, 
 8.076,  8.078,  8.081,  8.083,  8.086,  8.088,  8.090,  8.093,  8.095,  8.098, 
 8.100,  8.103,  8.106,  8.108,  8.111,  8.114,  8.117,  8.120,  8.122,  8.125, 
 8.128,  8.131,  8.134,  8.136,  8.139,  8.142,  8.145,  8.148,  8.150,  8.153, 
 8.156,  8.159,  8.161,  8.164,  8.166,  8.168,  8.171,  8.173,  8.176,  8.178, 
 8.181,  8.183,  8.185,  8.187,  8.189,  8.191,  8.192,  8.194,  8.196,  8.198, 
 8.200,  8.201,  8.201,  8.202,  8.203,  8.204,  8.204,  8.205,  8.206,  8.207, 
 8.207,  8.207,  8.207,  8.207,  8.207,  8.207,  8.207,  8.207,  8.207,  8.208, 
 8.208,  8.208,  8.208,  8.208,  8.208,  8.208,  8.208,  8.208,  8.208,  8.209, 
 8.209,  8.209,  8.209,  8.209,  8.209,  8.209,  8.209,  8.209,  8.209,  8.210, 
 8.210,  8.210,  8.210,  8.210,  8.210,  8.210,  8.210,  8.210,  8.210,  8.210, 
 8.210,  8.211,  8.211,  8.211,  8.211,  8.211,  8.211,  8.211,  8.211,  8.211, 
 8.211,  8.211,  8.212,  8.212,  8.213,  8.213,  8.214,  8.214,  8.215,  8.215, 
 8.216,  8.216,  8.216,  8.217,  8.217,  8.218,  8.218,  8.219,  8.219,  8.220, 
 8.220,  8.220,  8.221,  8.221,  8.221,  8.222,  8.222,  8.222,  8.222,  8.223, 
 8.223,  8.223,  8.224,  8.224,  8.224,  8.225,  8.225,  8.225,  8.225,  8.226, 
 8.226,  8.226,  8.226,  8.226,  8.226,  8.226,  8.226,  8.227,  8.227,  8.227, 
 8.227,  8.227,  8.227,  8.227,  8.227,  8.227,  8.227,  8.228,  8.228,  8.228, 
 8.228,  8.228,  8.228,  8.228,  8.228,  8.228,  8.228,  8.229,  8.229,  8.229, 
 8.229,  8.229,  8.229,  8.229,  8.229,  8.229,  8.229,  8.230,  8.230,  8.230, 
 8.230,  8.230,  8.230,  8.230,  8.230,  8.230,  8.231,  8.231,  8.231,  8.231, 
 8.231,  8.231,  8.231,  8.231,  8.232,  8.232,  8.232,  8.232,  8.232,  8.232, 
 8.232,  8.233,  8.233,  8.233,  8.233,  8.233,  8.233,  8.234,  8.234,  8.234, 
 8.234,  8.234,  8.235,  8.235,  8.235,  8.235,  8.236,  8.236,  8.236,  8.237, 
 8.237,  8.238,  8.240,  8.242,  8.244,  8.245,  8.247,  8.249,  8.250,  8.252, 
 8.254,  8.255,  8.257,  8.258,  8.260,  8.262,  8.263,  8.265,  8.266,  8.268, 
 8.269,  8.271,  8.272,  8.274,  8.275,  8.277,  8.278,  8.280,  8.281,  8.283, 
 8.284,  8.286,  8.287,  8.289,  8.290,  8.292,  8.293,  8.295,  8.296,  8.298, 
 8.299,  8.301,  8.302,  8.304,  8.305,  8.307,  8.308,  8.310,  8.311,  8.313, 
 8.314,  8.316,  8.317,  8.319,  8.320,  8.322,  8.323,  8.325,  8.326,  8.328, 
 8.329,  8.331,  8.333,  8.335,  8.336,  8.338,  8.340,  8.341,  8.343,  8.345, 
 8.346,  8.348,  8.349,  8.351,  8.352,  8.354,  8.355,  8.357,  8.358,  8.360, 
 8.361,  8.363,  8.364,  8.366,  8.367,  8.368,  8.370,  8.371,  8.373,  8.374, 
 8.376,  8.377,  8.378,  8.380,  8.381,  8.383,  8.384,  8.385,  8.387,  8.388, 
 8.389,  8.390,  8.391,  8.393,  8.394,  8.395,  8.396,  8.397,  8.398,  8.400, 
 8.401,  8.402,  8.404,  8.405,  8.406,  8.408,  8.409,  8.411,  8.412,  8.414, 
 8.415,  8.417,  8.419,  8.420,  8.422,  8.423,  8.425,  8.427,  8.428,  8.430, 
 8.432,  8.434,  8.435,  8.437,  8.439,  8.441,  8.442,  8.444,  8.446,  8.448, 
 8.449,  8.452,  8.454,  8.456,  8.458,  8.460,  8.462,  8.464,  8.466,  8.468, 
 8.470,  8.472,  8.474,  8.476,  8.478,  8.479,  8.481,  8.483,  8.485,  8.486, 
 8.488,  8.490,  8.492,  8.493,  8.495,  8.496,  8.498,  8.500,  8.501,  8.503, 
 8.504,  8.506,  8.507,  8.509,  8.511,  8.512,  8.514,  8.515,  8.517,  8.518, 
 8.519,  8.521,  8.522,  8.523,  8.525,  8.526,  8.527,  8.529,  8.530,  8.531, 
 8.533,  8.534,  8.535,  8.537,  8.538,  8.540,  8.541,  8.543,  8.544,  8.546, 
 8.547,  8.549,  8.550,  8.552,  8.553,  8.555,  8.557,  8.558,  8.560,  8.561, 
 8.563,  8.565,  8.566,  8.568,  8.569,  8.571,  8.573,  8.574,  8.576,  8.578, 
 8.579,  8.581,  8.583,  8.585,  8.586,  8.588,  8.590,  8.592,  8.593,  8.595, 
 8.597,  8.599,  8.600,  8.602,  8.604,  8.606,  8.607,  8.609,  8.611,  8.613, 
 8.614,  8.616,  8.618,  8.620,  8.621,  8.623,  8.625,  8.627,  8.628,  8.630, 
 8.632,  8.634,  8.635,  8.637,  8.639,  8.641,  8.642,  8.644,  8.646,  8.648, 
 8.649,  8.651,  8.653,  8.655,  8.657,  8.658,  8.660,  8.662,  8.664,  8.666, 
 8.667,  8.669,  8.671,  8.673,  8.674,  8.676,  8.678,  8.680,  8.681,  8.683, 
 8.685,  8.687,  8.688,  8.690,  8.692,  8.694,  8.695,  8.697,  8.699,  8.700, 
 8.702,  8.704,  8.706,  8.707,  8.709,  8.711,  8.713,  8.714,  8.716,  8.718, 
 8.719,  8.721,  8.722,  8.723,  8.725,  8.726,  8.728,  8.729,  8.731,  8.732, 
 8.734,  8.735,  8.737,  8.739,  8.740,  8.742,  8.744,  8.746,  8.748,  8.749, 
 8.751,  8.753,  8.755,  8.757,  8.759,  8.760,  8.762,  8.764,  8.766,  8.768, 
 8.770,  8.772,  8.774,  8.776,  8.778,  8.780,  8.782,  8.784,  8.786,  8.788, 
 8.789,  8.792,  8.795,  8.798,  8.801,  8.803,  8.806,  8.808,  8.810,  8.812, 
 8.815,  8.817,  8.819,  8.821,  8.823,  8.824,  8.826,  8.828,  8.830,  8.831, 
 8.833,  8.835,  8.836,  8.838,  8.839,  8.841,  8.842,  8.843,  8.845,  8.846, 
 8.847,  8.849,  8.850,  8.851,  8.852,  8.854,  8.855,  8.856,  8.857,  8.858, 
 8.859,  8.860,  8.861,  8.861,  8.862,  8.863,  8.864,  8.864,  8.865,  8.866, 
 8.867,  8.869,  8.870,  8.871,  8.872,  8.873,  8.875,  8.876,  8.878,  8.879, 
 8.881,  8.882,  8.884,  8.885,  8.887,  8.889,  8.891,  8.892,  8.894,  8.896, 
 8.898,  8.900,  8.902,  8.904,  8.906,  8.908,  8.911,  8.913,  8.915,  8.917, 
 8.919,  8.922,  8.924,  8.927,  8.929,  8.932,  8.934,  8.937,  8.939,  8.942, 
 8.944,  8.947,  8.949,  8.952,  8.955,  8.957,  8.960,  8.963,  8.965,  8.968, 
 8.971,  8.974,  8.977,  8.979,  8.982,  8.985,  8.988,  8.991,  8.994,  8.997, 
 9.000,  9.002,  9.005,  9.008,  9.011,  9.014,  9.017,  9.020,  9.023,  9.026, 
 9.029,  9.033,  9.036,  9.039,  9.042,  9.045,  9.049,  9.052,  9.055,  9.058, 
 9.061,  9.065,  9.068,  9.071,  9.074,  9.077,  9.081,  9.084,  9.087,  9.090, 
 9.094,  9.097,  9.100,  9.103,  9.107,  9.110,  9.113,  9.116,  9.120,  9.123, 
 9.126,  9.130,  9.133,  9.136,  9.139,  9.143,  9.146,  9.149,  9.153,  9.156, 
 9.159,  9.163,  9.166,  9.169,  9.173,  9.176,  9.179,  9.183,  9.186,  9.190, 
 9.193,  9.196,  9.200,  9.203,  9.207,  9.210,  9.214,  9.217,  9.221,  9.224, 
 9.228,  9.231,  9.235,  9.238,  9.242,  9.245,  9.249,  9.252,  9.256,  9.260, 
 9.263,  9.267,  9.270,  9.274,  9.278,  9.281,  9.285,  9.289,  9.292,  9.296, 
 9.299,  9.303,  9.307,  9.310,  9.314,  9.317,  9.321,  9.325,  9.328,  9.332, 
 9.336,  9.339,  9.343,  9.347,  9.351,  9.354,  9.358,  9.362,  9.366,  9.369, 
 9.373,  9.377,  9.381,  9.384,  9.388,  9.392,  9.396,  9.400,  9.404,  9.407, 
 9.411,  9.415,  9.419,  9.423,  9.426,  9.430,  9.434,  9.438,  9.442,  9.446, 
 9.449,  9.454,  9.458,  9.462,  9.466,  9.470,  9.474,  9.478,  9.482,  9.486, 
 9.490,  9.493,  9.497,  9.501,  9.505,  9.509,  9.513,  9.516,  9.520,  9.524, 
 9.528,  9.531,  9.535,  9.539,  9.542,  9.546,  9.550,  9.553,  9.557,  9.560, 
 9.564,  9.568,  9.571,  9.575,  9.578,  9.582,  9.585,  9.589,  9.592,  9.596, 
 9.599,  9.603,  9.606,  9.609,  9.612,  9.615,  9.618,  9.622,  9.625,  9.628, 
 9.632,  9.635,  9.638,  9.642,  9.645,  9.649,  9.652,  9.656,  9.659,  9.663, 
 9.666,  9.670,  9.673,  9.677,  9.681,  9.684,  9.688,  9.692,  9.695,  9.699, 
 9.703,  9.706,  9.710,  9.714,  9.717,  9.721,  9.725,  9.728,  9.732,  9.736, 
 9.739,  9.744,  9.748,  9.752,  9.756,  9.760,  9.764,  9.768,  9.772,  9.776, 
 9.780,  9.784,  9.787,  9.791,  9.795,  9.798,  9.802,  9.805,  9.809,  9.812, 
 9.816,  9.819,  9.823,  9.826,  9.829,  9.833,  9.836,  9.839,  9.842,  9.845, 
 9.849,  9.852,  9.855,  9.858,  9.861,  9.864,  9.867,  9.870,  9.873,  9.876, 
 9.879,  9.882,  9.885,  9.888,  9.890,  9.893,  9.896,  9.899,  9.901,  9.904, 
 9.907,  9.910,  9.913,  9.916,  9.919,  9.922,  9.925,  9.928,  9.931,  9.934, 
 9.937,  9.940,  9.943,  9.946,  9.949,  9.952,  9.955,  9.958,  9.961,  9.964, 
 9.968,  9.971,  9.974,  9.977,  9.980,  9.983,  9.987,  9.990,  9.993,  9.996, 
10.000, 10.003, 10.006, 10.009, 10.012, 10.015, 10.018, 10.022, 10.025, 10.028, 
10.031, 10.035, 10.038, 10.041, 10.044, 10.048, 10.051, 10.054, 10.057, 10.061, 
10.064, 10.067, 10.071, 10.074, 10.077, 10.080, 10.084, 10.087, 10.090, 10.094, 
10.097, 10.100, 10.103, 10.107, 10.110, 10.113, 10.117, 10.120, 10.123, 10.126, 
10.129, 10.133, 10.137, 10.140, 10.144, 10.147, 10.150, 10.154, 10.157, 10.160, 
10.164, 10.167, 10.170, 10.173, 10.176, 10.179, 10.182, 10.185, 10.188, 10.191, 
10.194, 10.197, 10.200, 10.203, 10.206, 10.209, 10.212, 10.215, 10.217, 10.220, 
10.223, 10.226, 10.228, 10.231, 10.234, 10.236, 10.239, 10.242, 10.244, 10.247, 
10.250, 10.252, 10.254, 10.256, 10.259, 10.261, 10.264, 10.266, 10.268, 10.271, 
10.273, 10.276, 10.278, 10.281, 10.283, 10.286, 10.288, 10.291, 10.293, 10.296, 
10.298, 10.301, 10.303, 10.306, 10.308, 10.311, 10.313, 10.316, 10.319, 10.321, 
10.324, 10.326, 10.329, 10.331, 10.334, 10.337, 10.339, 10.342, 10.344, 10.347, 
10.349, 10.352, 10.355, 10.358, 10.361, 10.363, 10.366, 10.369, 10.371, 10.374, 
10.377, 10.379, 10.382, 10.385, 10.387, 10.390, 10.392, 10.395, 10.397, 10.400, 
10.402, 10.405, 10.407, 10.410, 10.412, 10.414, 10.417, 10.419, 10.422, 10.424, 
10.426, 10.429, 10.431, 10.433, 10.436, 10.438, 10.440, 10.443, 10.445, 10.447, 
10.449, 10.452, 10.454, 10.456, 10.458, 10.460, 10.462, 10.465, 10.467, 10.469, 
10.471, 10.473, 10.476, 10.478, 10.480, 10.482, 10.484, 10.487, 10.489, 10.491, 
10.493, 10.496, 10.498, 10.500, 10.503, 10.505, 10.507, 10.509, 10.512, 10.514, 
10.516, 10.519, 10.521, 10.523, 10.526, 10.528, 10.530, 10.533, 10.535, 10.537, 
10.539, 10.542, 10.544, 10.546, 10.549, 10.551, 10.553, 10.556, 10.558, 10.560, 
10.563, 10.565, 10.567, 10.569, 10.572, 10.574, 10.576, 10.579, 10.581, 10.583, 
10.585, 10.588, 10.590, 10.592, 10.594, 10.597, 10.599, 10.601, 10.603, 10.606, 
10.608, 10.610, 10.612, 10.614, 10.617, 10.619, 10.621, 10.623, 10.625, 10.627, 
10.629, 10.632, 10.634, 10.637, 10.639, 10.641, 10.643, 10.646, 10.648, 10.650, 
10.652, 10.654, 10.656, 10.658, 10.661, 10.663, 10.665, 10.667, 10.669, 10.671, 
10.673, 10.674, 10.676, 10.678, 10.680, 10.682, 10.684, 10.686, 10.688, 10.690, 
10.691, 10.693, 10.695, 10.697, 10.699, 10.701, 10.702, 10.704, 10.706, 10.708, 
10.709, 10.711, 10.712, 10.714, 10.715, 10.717, 10.718, 10.720, 10.721, 10.723, 
10.724, 10.726, 10.728, 10.729, 10.731, 10.733, 10.735, 10.736, 10.738, 10.740, 
10.742, 10.744, 10.745, 10.747, 10.749, 10.751, 10.753, 10.755, 10.757, 10.758, 
10.760, 10.762, 10.764, 10.766, 10.768, 10.770, 10.772, 10.774, 10.776, 10.778, 
10.779, 10.782, 10.784, 10.786, 10.788, 10.791, 10.793, 10.795, 10.797, 10.799, 
10.801, 10.803, 10.805, 10.806, 10.808, 10.810, 10.812, 10.814, 10.816, 10.817, 
10.819, 10.821, 10.822, 10.824, 10.826, 10.827, 10.829, 10.830, 10.832, 10.834, 
10.835, 10.837, 10.838, 10.840, 10.841, 10.842, 10.844, 10.845, 10.847, 10.848, 
10.849, 10.851, 10.852, 10.853, 10.854, 10.856, 10.857, 10.858, 10.859, 10.861, 
10.862, 10.863, 10.864, 10.866, 10.867, 10.868, 10.869, 10.871, 10.872, 10.873, 
10.874, 10.876, 10.877, 10.878, 10.879, 10.881, 10.882, 10.883, 10.884, 10.886, 
10.887, 10.888, 10.889, 10.891, 10.892, 10.893, 10.894, 10.896, 10.897, 10.898, 
10.899, 10.901, 10.902, 10.904, 10.905, 10.906, 10.908, 10.909, 10.910, 10.911, 
10.913, 10.914, 10.915, 10.917, 10.918, 10.919, 10.920, 10.922, 10.923, 10.924, 
10.925, 10.927, 10.928, 10.929, 10.930, 10.931, 10.933, 10.934, 10.935, 10.936, 
10.937, 10.939, 10.940, 10.941, 10.942, 10.943, 10.945, 10.946, 10.947, 10.948, 
10.949, 10.950, 10.951, 10.952, 10.953, 10.954, 10.955, 10.956, 10.957, 10.958, 
10.959, 10.961, 10.962, 10.963, 10.964, 10.965, 10.967, 10.968, 10.969, 10.970, 
10.972, 10.973, 10.974, 10.976, 10.977, 10.978, 10.980, 10.981, 10.983, 10.984, 
10.985, 10.987, 10.988, 10.990, 10.991, 10.992, 10.994, 10.995, 10.997, 10.998, 
11.000, 11.001, 11.003, 11.005, 11.006, 11.008, 11.010, 11.011, 11.013, 11.014, 
11.016, 11.017, 11.019, 11.020, 11.021, 11.023, 11.024, 11.025, 11.027, 11.028, 
11.029, 11.030, 11.031, 11.033, 11.034, 11.035, 11.036, 11.037, 11.038, 11.039, 
11.040, 11.041, 11.042, 11.043, 11.044, 11.045, 11.046, 11.047, 11.048, 11.049, 
11.049, 11.050, 11.051, 11.052, 11.053, 11.053, 11.054, 11.055, 11.056, 11.057, 
11.057, 11.058, 11.059, 11.060, 11.060, 11.061, 11.062, 11.063, 11.063, 11.064, 
11.065, 11.066, 11.066, 11.067, 11.068, 11.069, 11.069, 11.070, 11.071, 11.071, 
11.072, 11.073, 11.074, 11.074, 11.075, 11.076, 11.077, 11.077, 11.078, 11.079, 
11.079, 11.080, 11.081, 11.081, 11.082, 11.082, 11.083, 11.084, 11.084, 11.085, 
11.086, 11.086, 11.087, 11.088, 11.089, 11.089, 11.090, 11.091, 11.092, 11.092, 
11.093, 11.094, 11.095, 11.095, 11.096, 11.097, 11.098, 11.099, 11.100, 11.100, 
11.101, 11.102, 11.103, 11.104, 11.104, 11.105, 11.106, 11.107, 11.108, 11.109, 
11.109, 11.111, 11.112, 11.112, 11.113, 11.114, 11.115, 11.116, 11.117, 11.118, 
11.119, 11.120, 11.121, 11.121, 11.122, 11.123, 11.124, 11.125, 11.125, 11.126, 
11.127, 11.127, 11.128, 11.129, 11.130, 11.130, 11.131, 11.132, 11.132, 11.133, 
11.134, 11.134, 11.135, 11.135, 11.136, 11.137, 11.137, 11.138, 11.138, 11.139, 
11.139, 11.140, 11.140, 11.141, 11.141, 11.142, 11.142, 11.143, 11.143, 11.144, 
11.144, 11.145, 11.145, 11.146, 11.146, 11.147, 11.147, 11.148, 11.148, 11.149, 
11.149, 11.150, 11.150, 11.151, 11.151, 11.152, 11.152, 11.153, 11.153, 11.154, 
11.154, 11.155, 11.155, 11.156, 11.156, 11.157, 11.157, 11.158, 11.158, 11.159, 
11.159, 11.160, 11.161, 11.161, 11.162, 11.162, 11.163, 11.163, 11.164, 11.164, 
11.165, 11.165, 11.166, 11.166, 11.167, 11.167, 11.168, 11.168, 11.169, 11.169, 
11.170, 11.170, 11.171, 11.171, 11.172, 11.172, 11.173, 11.173, 11.174, 11.174, 
11.175, 11.175, 11.176, 11.176, 11.177, 11.177, 11.178, 11.178, 11.179, 11.179, 
11.179, 11.180, 11.180, 11.180, 11.181, 11.181, 11.182, 11.182, 11.182, 11.183, 
11.183, 11.184, 11.184, 11.185, 11.185, 11.186, 11.186, 11.187, 11.187, 11.188, 
11.188, 11.189, 11.189, 11.190, 11.190, 11.191, 11.191, 11.192, 11.193, 11.193, 
11.194, 11.194, 11.195, 11.195, 11.196, 11.197, 11.197, 11.198, 11.198, 11.199, 
])
_vs_jma_2020b = np.array([
 2.759,  2.763,  2.767,  2.771,  2.775,  2.779,  2.783,  2.787,  2.791,  2.795, 
 2.799,  2.803,  2.807,  2.811,  2.947,  3.156,  3.291,  3.390,  3.471,  3.542, 
 3.609,  3.673,  3.737,  3.802,  3.866,  3.930,  3.995,  4.059,  4.124,  4.188, 
 4.252,  4.317,  4.370,  4.412,  4.445,  4.471,  4.489,  4.502,  4.510,  4.514, 
 4.517,  4.518,  4.519,  4.520,  4.521,  4.522,  4.524,  4.525,  4.526,  4.527, 
 4.528,  4.529,  4.530,  4.532,  4.533,  4.534,  4.535,  4.536,  4.537,  4.539, 
 4.540,  4.541,  4.542,  4.543,  4.544,  4.545,  4.547,  4.548,  4.549,  4.550, 
 4.551,  4.552,  4.553,  4.555,  4.556,  4.557,  4.558,  4.559,  4.560,  4.561, 
 4.563,  4.564,  4.565,  4.566,  4.567,  4.568,  4.570,  4.571,  4.572,  4.573, 
 4.574,  4.575,  4.576,  4.578,  4.579,  4.580,  4.581,  4.582,  4.583,  4.584, 
 4.586,  4.587,  4.588,  4.589,  4.590,  4.591,  4.593,  4.594,  4.595,  4.596, 
 4.597,  4.598,  4.599,  4.601,  4.602,  4.603,  4.605,  4.606,  4.607,  4.608, 
 4.609,  4.610,  4.611,  4.612,  4.613,  4.614,  4.615,  4.616,  4.617,  4.618, 
 4.619,  4.620,  4.621,  4.622,  4.623,  4.624,  4.625,  4.626,  4.627,  4.628, 
 4.629,  4.630,  4.632,  4.633,  4.634,  4.636,  4.637,  4.638,  4.639,  4.640, 
 4.641,  4.643,  4.644,  4.645,  4.647,  4.648,  4.649,  4.651,  4.652,  4.654, 
 4.655,  4.657,  4.659,  4.660,  4.661,  4.663,  4.665,  4.667,  4.668,  4.670, 
 4.671,  4.673,  4.675,  4.676,  4.678,  4.679,  4.681,  4.683,  4.684,  4.686, 
 4.687,  4.689,  4.690,  4.692,  4.693,  4.694,  4.696,  4.697,  4.699,  4.700, 
 4.702,  4.703,  4.704,  4.705,  4.706,  4.707,  4.708,  4.709,  4.710,  4.711, 
 4.713,  4.713,  4.713,  4.714,  4.714,  4.715,  4.715,  4.716,  4.716,  4.717, 
 4.717,  4.717,  4.717,  4.717,  4.717,  4.717,  4.717,  4.717,  4.717,  4.717, 
 4.717,  4.717,  4.717,  4.717,  4.717,  4.717,  4.717,  4.717,  4.717,  4.718, 
 4.718,  4.718,  4.718,  4.718,  4.718,  4.718,  4.718,  4.718,  4.718,  4.718, 
 4.718,  4.718,  4.718,  4.718,  4.718,  4.718,  4.718,  4.718,  4.718,  4.718, 
 4.718,  4.719,  4.719,  4.719,  4.719,  4.719,  4.719,  4.719,  4.719,  4.719, 
 4.719,  4.719,  4.720,  4.720,  4.720,  4.720,  4.721,  4.721,  4.721,  4.721, 
 4.722,  4.722,  4.722,  4.722,  4.722,  4.723,  4.723,  4.724,  4.724,  4.724, 
 4.724,  4.724,  4.725,  4.725,  4.725,  4.725,  4.725,  4.725,  4.725,  4.726, 
 4.726,  4.726,  4.726,  4.726,  4.726,  4.727,  4.727,  4.727,  4.727,  4.728, 
 4.728,  4.728,  4.728,  4.728,  4.728,  4.728,  4.728,  4.728,  4.728,  4.728, 
 4.728,  4.728,  4.728,  4.728,  4.728,  4.728,  4.728,  4.729,  4.729,  4.729, 
 4.729,  4.729,  4.729,  4.729,  4.729,  4.729,  4.729,  4.729,  4.729,  4.729, 
 4.729,  4.729,  4.729,  4.729,  4.729,  4.729,  4.729,  4.730,  4.730,  4.730, 
 4.730,  4.730,  4.730,  4.730,  4.730,  4.730,  4.730,  4.730,  4.730,  4.730, 
 4.730,  4.730,  4.730,  4.730,  4.731,  4.731,  4.731,  4.731,  4.731,  4.731, 
 4.731,  4.732,  4.732,  4.732,  4.732,  4.732,  4.732,  4.732,  4.732,  4.732, 
 4.732,  4.732,  4.733,  4.733,  4.733,  4.733,  4.733,  4.733,  4.733,  4.734, 
 4.734,  4.734,  4.736,  4.737,  4.738,  4.739,  4.740,  4.741,  4.741,  4.743, 
 4.744,  4.744,  4.745,  4.746,  4.747,  4.748,  4.749,  4.750,  4.751,  4.752, 
 4.752,  4.753,  4.754,  4.755,  4.756,  4.757,  4.757,  4.759,  4.759,  4.760, 
 4.761,  4.762,  4.763,  4.764,  4.764,  4.766,  4.766,  4.767,  4.768,  4.769, 
 4.770,  4.771,  4.771,  4.772,  4.773,  4.774,  4.775,  4.776,  4.776,  4.778, 
 4.778,  4.779,  4.780,  4.781,  4.782,  4.783,  4.783,  4.784,  4.785,  4.786, 
 4.787,  4.788,  4.789,  4.790,  4.791,  4.792,  4.793,  4.794,  4.795,  4.796, 
 4.797,  4.798,  4.798,  4.799,  4.800,  4.801,  4.802,  4.803,  4.803,  4.805, 
 4.805,  4.806,  4.807,  4.808,  4.809,  4.809,  4.810,  4.811,  4.812,  4.813, 
 4.814,  4.814,  4.815,  4.816,  4.817,  4.818,  4.818,  4.819,  4.820,  4.821, 
 4.821,  4.822,  4.822,  4.824,  4.824,  4.825,  4.825,  4.826,  4.826,  4.828, 
 4.828,  4.829,  4.830,  4.830,  4.831,  4.832,  4.833,  4.834,  4.834,  4.836, 
 4.836,  4.837,  4.839,  4.839,  4.840,  4.841,  4.842,  4.843,  4.844,  4.845, 
 4.846,  4.847,  4.848,  4.849,  4.850,  4.851,  4.852,  4.853,  4.854,  4.855, 
 4.856,  4.857,  4.859,  4.860,  4.861,  4.862,  4.863,  4.864,  4.866,  4.867, 
 4.868,  4.869,  4.870,  4.871,  4.872,  4.873,  4.874,  4.875,  4.876,  4.877, 
 4.878,  4.879,  4.880,  4.881,  4.882,  4.883,  4.884,  4.885,  4.886,  4.887, 
 4.887,  4.889,  4.889,  4.890,  4.891,  4.892,  4.893,  4.894,  4.895,  4.895, 
 4.896,  4.897,  4.898,  4.898,  4.899,  4.900,  4.901,  4.902,  4.902,  4.903, 
 4.904,  4.905,  4.905,  4.906,  4.907,  4.908,  4.909,  4.910,  4.910,  4.911, 
 4.912,  4.913,  4.914,  4.915,  4.916,  4.917,  4.918,  4.918,  4.920,  4.920, 
 4.921,  4.922,  4.923,  4.924,  4.925,  4.926,  4.927,  4.928,  4.929,  4.930, 
 4.930,  4.932,  4.933,  4.934,  4.934,  4.936,  4.937,  4.938,  4.939,  4.940, 
 4.941,  4.942,  4.943,  4.944,  4.945,  4.946,  4.947,  4.948,  4.949,  4.950, 
 4.951,  4.952,  4.953,  4.954,  4.955,  4.956,  4.957,  4.958,  4.959,  4.960, 
 4.961,  4.962,  4.963,  4.964,  4.965,  4.966,  4.967,  4.968,  4.969,  4.970, 
 4.971,  4.972,  4.973,  4.974,  4.975,  4.976,  4.977,  4.978,  4.979,  4.980, 
 4.981,  4.982,  4.983,  4.984,  4.985,  4.986,  4.987,  4.989,  4.989,  4.990, 
 4.991,  4.993,  4.993,  4.994,  4.995,  4.997,  4.997,  4.998,  4.999,  5.000, 
 5.001,  5.002,  5.003,  5.004,  5.005,  5.006,  5.007,  5.008,  5.009,  5.010, 
 5.011,  5.012,  5.013,  5.013,  5.014,  5.015,  5.016,  5.017,  5.018,  5.018, 
 5.020,  5.020,  5.021,  5.022,  5.023,  5.024,  5.025,  5.026,  5.028,  5.028, 
 5.029,  5.030,  5.032,  5.033,  5.034,  5.034,  5.036,  5.037,  5.038,  5.039, 
 5.040,  5.041,  5.043,  5.044,  5.045,  5.046,  5.047,  5.048,  5.049,  5.051, 
 5.051,  5.053,  5.055,  5.056,  5.058,  5.059,  5.061,  5.062,  5.063,  5.064, 
 5.066,  5.067,  5.068,  5.070,  5.071,  5.071,  5.072,  5.074,  5.075,  5.075, 
 5.076,  5.078,  5.078,  5.079,  5.080,  5.081,  5.082,  5.082,  5.083,  5.084, 
 5.084,  5.086,  5.086,  5.087,  5.087,  5.089,  5.089,  5.090,  5.090,  5.091, 
 5.091,  5.092,  5.093,  5.093,  5.093,  5.094,  5.094,  5.094,  5.095,  5.095, 
 5.096,  5.097,  5.098,  5.098,  5.099,  5.099,  5.101,  5.101,  5.102,  5.103, 
 5.104,  5.105,  5.106,  5.106,  5.107,  5.109,  5.110,  5.110,  5.111,  5.113, 
 5.114,  5.115,  5.116,  5.117,  5.118,  5.120,  5.121,  5.122,  5.124,  5.125, 
 5.126,  5.128,  5.129,  5.130,  5.132,  5.133,  5.134,  5.136,  5.137,  5.139, 
 5.140,  5.142,  5.143,  5.145,  5.147,  5.148,  5.149,  5.151,  5.152,  5.154, 
 5.156,  5.157,  5.159,  5.160,  5.162,  5.164,  5.166,  5.167,  5.169,  5.171, 
 5.172,  5.174,  5.175,  5.177,  5.179,  5.180,  5.182,  5.184,  5.186,  5.187, 
 5.189,  5.191,  5.193,  5.195,  5.197,  5.198,  5.201,  5.202,  5.204,  5.206, 
 5.207,  5.210,  5.211,  5.213,  5.215,  5.217,  5.219,  5.221,  5.222,  5.224, 
 5.226,  5.228,  5.230,  5.232,  5.234,  5.236,  5.237,  5.239,  5.241,  5.243, 
 5.245,  5.247,  5.249,  5.251,  5.252,  5.255,  5.256,  5.258,  5.260,  5.262, 
 5.264,  5.266,  5.268,  5.270,  5.272,  5.274,  5.275,  5.278,  5.279,  5.282, 
 5.283,  5.285,  5.287,  5.289,  5.291,  5.293,  5.295,  5.297,  5.299,  5.301, 
 5.303,  5.305,  5.307,  5.309,  5.311,  5.313,  5.316,  5.317,  5.320,  5.322, 
 5.324,  5.326,  5.328,  5.330,  5.332,  5.334,  5.336,  5.339,  5.340,  5.343, 
 5.344,  5.347,  5.349,  5.351,  5.353,  5.355,  5.357,  5.359,  5.361,  5.363, 
 5.366,  5.367,  5.370,  5.372,  5.374,  5.376,  5.378,  5.380,  5.383,  5.384, 
 5.387,  5.389,  5.391,  5.393,  5.395,  5.398,  5.400,  5.402,  5.405,  5.406, 
 5.409,  5.411,  5.413,  5.416,  5.417,  5.420,  5.422,  5.424,  5.426,  5.429, 
 5.430,  5.433,  5.436,  5.438,  5.440,  5.443,  5.445,  5.447,  5.449,  5.452, 
 5.454,  5.456,  5.458,  5.460,  5.463,  5.465,  5.467,  5.469,  5.471,  5.474, 
 5.476,  5.478,  5.480,  5.482,  5.484,  5.486,  5.489,  5.490,  5.493,  5.494, 
 5.497,  5.499,  5.501,  5.503,  5.505,  5.507,  5.509,  5.511,  5.513,  5.515, 
 5.517,  5.519,  5.521,  5.522,  5.524,  5.526,  5.528,  5.530,  5.532,  5.533, 
 5.536,  5.537,  5.539,  5.541,  5.543,  5.545,  5.547,  5.549,  5.551,  5.553, 
 5.555,  5.557,  5.559,  5.561,  5.564,  5.566,  5.568,  5.570,  5.572,  5.574, 
 5.576,  5.578,  5.580,  5.583,  5.584,  5.587,  5.589,  5.591,  5.593,  5.595, 
 5.597,  5.600,  5.602,  5.605,  5.607,  5.609,  5.611,  5.614,  5.616,  5.618, 
 5.621,  5.623,  5.625,  5.627,  5.629,  5.631,  5.633,  5.635,  5.637,  5.639, 
 5.641,  5.643,  5.645,  5.647,  5.649,  5.651,  5.653,  5.655,  5.656,  5.658, 
 5.660,  5.662,  5.664,  5.666,  5.667,  5.669,  5.671,  5.672,  5.674,  5.676, 
 5.678,  5.679,  5.681,  5.683,  5.684,  5.686,  5.687,  5.689,  5.690,  5.692, 
 5.694,  5.695,  5.697,  5.699,  5.701,  5.702,  5.704,  5.706,  5.707,  5.709, 
 5.711,  5.713,  5.714,  5.716,  5.718,  5.720,  5.721,  5.723,  5.725,  5.726, 
 5.729,  5.730,  5.732,  5.734,  5.736,  5.737,  5.740,  5.741,  5.743,  5.745, 
 5.747,  5.749,  5.751,  5.752,  5.754,  5.756,  5.757,  5.760,  5.761,  5.763, 
 5.765,  5.767,  5.769,  5.771,  5.772,  5.775,  5.776,  5.778,  5.780,  5.782, 
 5.784,  5.786,  5.788,  5.790,  5.791,  5.793,  5.795,  5.797,  5.799,  5.801, 
 5.803,  5.805,  5.806,  5.809,  5.810,  5.812,  5.814,  5.816,  5.818,  5.820, 
 5.821,  5.824,  5.826,  5.828,  5.830,  5.832,  5.833,  5.836,  5.837,  5.839, 
 5.841,  5.843,  5.845,  5.847,  5.848,  5.850,  5.852,  5.853,  5.855,  5.857, 
 5.859,  5.860,  5.862,  5.864,  5.866,  5.867,  5.869,  5.871,  5.872,  5.874, 
 5.875,  5.877,  5.878,  5.880,  5.882,  5.883,  5.884,  5.886,  5.887,  5.889, 
 5.891,  5.892,  5.893,  5.894,  5.896,  5.897,  5.899,  5.900,  5.901,  5.903, 
 5.904,  5.906,  5.907,  5.909,  5.910,  5.911,  5.913,  5.914,  5.916,  5.917, 
 5.918,  5.920,  5.921,  5.923,  5.924,  5.926,  5.927,  5.929,  5.930,  5.932, 
 5.933,  5.934,  5.936,  5.937,  5.939,  5.941,  5.942,  5.944,  5.945,  5.947, 
 5.948,  5.949,  5.951,  5.953,  5.955,  5.956,  5.957,  5.959,  5.960,  5.962, 
 5.964,  5.965,  5.967,  5.968,  5.970,  5.971,  5.972,  5.974,  5.975,  5.977, 
 5.978,  5.980,  5.981,  5.983,  5.984,  5.985,  5.987,  5.988,  5.990,  5.991, 
 5.992,  5.994,  5.995,  5.996,  5.998,  5.999,  6.000,  6.002,  6.003,  6.004, 
 6.005,  6.007,  6.008,  6.009,  6.010,  6.011,  6.013,  6.014,  6.016,  6.017, 
 6.018,  6.019,  6.021,  6.022,  6.023,  6.024,  6.025,  6.027,  6.028,  6.029, 
 6.030,  6.032,  6.033,  6.034,  6.036,  6.037,  6.039,  6.040,  6.041,  6.043, 
 6.044,  6.045,  6.047,  6.048,  6.049,  6.051,  6.052,  6.053,  6.055,  6.056, 
 6.057,  6.059,  6.060,  6.061,  6.063,  6.064,  6.065,  6.067,  6.068,  6.069, 
 6.071,  6.072,  6.073,  6.074,  6.076,  6.077,  6.078,  6.080,  6.081,  6.082, 
 6.083,  6.085,  6.086,  6.087,  6.089,  6.090,  6.091,  6.093,  6.094,  6.095, 
 6.097,  6.098,  6.099,  6.100,  6.102,  6.103,  6.104,  6.105,  6.106,  6.107, 
 6.109,  6.110,  6.111,  6.113,  6.114,  6.116,  6.117,  6.118,  6.120,  6.121, 
 6.122,  6.123,  6.124,  6.125,  6.127,  6.128,  6.129,  6.130,  6.132,  6.133, 
 6.134,  6.134,  6.136,  6.137,  6.138,  6.139,  6.140,  6.141,  6.143,  6.144, 
 6.144,  6.145,  6.147,  6.148,  6.149,  6.150,  6.151,  6.152,  6.153,  6.154, 
 6.155,  6.156,  6.156,  6.157,  6.158,  6.159,  6.160,  6.161,  6.161,  6.163, 
 6.163,  6.164,  6.166,  6.166,  6.167,  6.168,  6.170,  6.170,  6.171,  6.172, 
 6.174,  6.175,  6.175,  6.176,  6.178,  6.179,  6.180,  6.181,  6.182,  6.183, 
 6.184,  6.185,  6.186,  6.187,  6.189,  6.190,  6.191,  6.192,  6.193,  6.194, 
 6.195,  6.197,  6.198,  6.199,  6.200,  6.202,  6.203,  6.204,  6.205,  6.206, 
 6.207,  6.209,  6.210,  6.210,  6.211,  6.213,  6.214,  6.215,  6.216,  6.217, 
 6.218,  6.219,  6.220,  6.221,  6.222,  6.222,  6.224,  6.224,  6.225,  6.226, 
 6.227,  6.228,  6.229,  6.230,  6.230,  6.231,  6.232,  6.233,  6.234,  6.234, 
 6.235,  6.236,  6.237,  6.237,  6.238,  6.239,  6.240,  6.240,  6.241,  6.242, 
 6.243,  6.243,  6.244,  6.245,  6.245,  6.246,  6.247,  6.248,  6.248,  6.249, 
 6.249,  6.251,  6.251,  6.252,  6.252,  6.253,  6.254,  6.255,  6.255,  6.256, 
 6.257,  6.257,  6.258,  6.259,  6.260,  6.260,  6.261,  6.262,  6.263,  6.263, 
 6.264,  6.265,  6.266,  6.267,  6.267,  6.268,  6.269,  6.270,  6.270,  6.271, 
 6.272,  6.272,  6.273,  6.274,  6.275,  6.275,  6.276,  6.277,  6.278,  6.278, 
 6.279,  6.280,  6.280,  6.281,  6.282,  6.282,  6.283,  6.284,  6.284,  6.285, 
 6.286,  6.287,  6.287,  6.288,  6.289,  6.289,  6.290,  6.291,  6.291,  6.292, 
 6.293,  6.293,  6.294,  6.294,  6.295,  6.295,  6.296,  6.297,  6.297,  6.298, 
 6.298,  6.299,  6.300,  6.301,  6.301,  6.302,  6.303,  6.303,  6.304,  6.305, 
 6.306,  6.306,  6.307,  6.308,  6.309,  6.309,  6.310,  6.311,  6.312,  6.313, 
 6.313,  6.314,  6.315,  6.316,  6.317,  6.317,  6.318,  6.319,  6.320,  6.321, 
 6.322,  6.322,  6.324,  6.325,  6.325,  6.326,  6.328,  6.328,  6.329,  6.330, 
 6.331,  6.332,  6.333,  6.333,  6.334,  6.335,  6.336,  6.336,  6.337,  6.338, 
 6.339,  6.339,  6.340,  6.341,  6.341,  6.342,  6.343,  6.343,  6.344,  6.344, 
 6.345,  6.345,  6.346,  6.347,  6.347,  6.348,  6.348,  6.349,  6.349,  6.350, 
 6.350,  6.351,  6.351,  6.352,  6.352,  6.352,  6.353,  6.353,  6.354,  6.355, 
 6.355,  6.355,  6.356,  6.356,  6.356,  6.357,  6.357,  6.358,  6.358,  6.359, 
 6.359,  6.360,  6.360,  6.360,  6.361,  6.361,  6.361,  6.362,  6.363,  6.363, 
 6.363,  6.364,  6.364,  6.364,  6.365,  6.366,  6.366,  6.366,  6.367,  6.367, 
 6.367,  6.368,  6.368,  6.368,  6.369,  6.369,  6.370,  6.370,  6.370,  6.371, 
 6.371,  6.371,  6.372,  6.372,  6.373,  6.373,  6.374,  6.374,  6.375,  6.375, 
 6.375,  6.376,  6.376,  6.376,  6.377,  6.378,  6.378,  6.379,  6.379,  6.379, 
 6.380,  6.380,  6.381,  6.382,  6.382,  6.382,  6.383,  6.383,  6.384,  6.384, 
 6.384,  6.386,  6.386,  6.386,  6.387,  6.387,  6.388,  6.389,  6.389,  6.390, 
 6.390,  6.391,  6.391,  6.391,  6.392,  6.393,  6.393,  6.394,  6.394,  6.394, 
 6.395,  6.395,  6.395,  6.396,  6.397,  6.397,  6.397,  6.398,  6.398,  6.398, 
 6.399,  6.399,  6.399,  6.399,  6.400,  6.401,  6.401,  6.401,  6.401,  6.402, 
 6.402,  6.402,  6.402,  6.403,  6.403,  6.403,  6.403,  6.404,  6.404,  6.405, 
 6.405,  6.405,  6.405,  6.406,  6.406,  6.406,  6.406,  6.407,  6.407,  6.407, 
 6.407,  6.408,  6.408,  6.409,  6.409,  6.409,  6.409,  6.410,  6.410,  6.410, 
 6.410,  6.411,  6.411,  6.411,  6.411,  6.412,  6.412,  6.413,  6.413,  6.413, 
 6.413,  6.414,  6.414,  6.414,  6.415,  6.415,  6.416,  6.416,  6.416,  6.416, 
 6.417,  6.417,  6.417,  6.417,  6.418,  6.418,  6.418,  6.418,  6.419,  6.419, 
 6.420,  6.420,  6.420,  6.420,  6.421,  6.421,  6.421,  6.421,  6.422,  6.422, 
 6.422,  6.422,  6.423,  6.423,  6.424,  6.424,  6.424,  6.424,  6.425,  6.425, 
 6.425,  6.425,  6.425,  6.425,  6.426,  6.426,  6.426,  6.426,  6.426,  6.427, 
 6.427,  6.428,  6.428,  6.428,  6.428,  6.429,  6.429,  6.429,  6.429,  6.430, 
 6.430,  6.430,  6.430,  6.431,  6.431,  6.432,  6.432,  6.432,  6.433,  6.433, 
 6.433,  6.433,  6.434,  6.434,  6.434,  6.435,  6.435,  6.436,  6.436,  6.436, 
])
_dep_jma_2020b = np.array([
  0.0,   0.5,   1.0,   1.5,   2.0,   2.5,   3.0,   3.5,   4.0,   4.5, 
  5.0,   5.5,   6.0,   6.5,   7.0,   7.5,   8.0,   8.5,   9.0,   9.5, 
 10.0,  10.5,  11.0,  11.5,  12.0,  12.5,  13.0,  13.5,  14.0,  14.5, 
 15.0,  15.5,  16.0,  16.5,  17.0,  17.5,  18.0,  18.5,  19.0,  19.5, 
 20.0,  20.5,  21.0,  21.5,  22.0,  22.5,  23.0,  23.5,  24.0,  24.5, 
 25.0,  25.5,  26.0,  26.5,  27.0,  27.5,  28.0,  28.5,  29.0,  29.5, 
 30.0,  30.5,  31.0,  31.5,  32.0,  32.5,  33.0,  33.5,  34.0,  34.5, 
 35.0,  35.5,  36.0,  36.5,  37.0,  37.5,  38.0,  38.5,  39.0,  39.5, 
 40.0,  40.5,  41.0,  41.5,  42.0,  42.5,  43.0,  43.5,  44.0,  44.5, 
 45.0,  45.5,  46.0,  46.5,  47.0,  47.5,  48.0,  48.5,  49.0,  49.5, 
 50.0,  50.5,  51.0,  51.5,  52.0,  52.5,  53.0,  53.5,  54.0,  54.5, 
 55.0,  55.5,  56.0,  56.5,  57.0,  57.5,  58.0,  58.5,  59.0,  59.5, 
 60.0,  60.5,  61.0,  61.5,  62.0,  62.5,  63.0,  63.5,  64.0,  64.5, 
 65.0,  65.5,  66.0,  66.5,  67.0,  67.5,  68.0,  68.5,  69.0,  69.5, 
 70.0,  70.5,  71.0,  71.5,  72.0,  72.5,  73.0,  73.5,  74.0,  74.5, 
 75.0,  75.5,  76.0,  76.5,  77.0,  77.5,  78.0,  78.5,  79.0,  79.5, 
 80.0,  80.5,  81.0,  81.5,  82.0,  82.5,  83.0,  83.5,  84.0,  84.5, 
 85.0,  85.5,  86.0,  86.5,  87.0,  87.5,  88.0,  88.5,  89.0,  89.5, 
 90.0,  90.5,  91.0,  91.5,  92.0,  92.5,  93.0,  93.5,  94.0,  94.5, 
 95.0,  95.5,  96.0,  96.5,  97.0,  97.5,  98.0,  98.5,  99.0,  99.5, 
100.0, 100.5, 101.0, 101.5, 102.0, 102.5, 103.0, 103.5, 104.0, 104.5, 
105.0, 105.5, 106.0, 106.5, 107.0, 107.5, 108.0, 108.5, 109.0, 109.5, 
110.0, 110.5, 111.0, 111.5, 112.0, 112.5, 113.0, 113.5, 114.0, 114.5, 
115.0, 115.5, 116.0, 116.5, 117.0, 117.5, 118.0, 118.5, 119.0, 119.5, 
120.0, 120.5, 121.0, 121.5, 122.0, 122.5, 123.0, 123.5, 124.0, 124.5, 
125.0, 125.5, 126.0, 126.5, 127.0, 127.5, 128.0, 128.5, 129.0, 129.5, 
130.0, 130.5, 131.0, 131.5, 132.0, 132.5, 133.0, 133.5, 134.0, 134.5, 
135.0, 135.5, 136.0, 136.5, 137.0, 137.5, 138.0, 138.5, 139.0, 139.5, 
140.0, 140.5, 141.0, 141.5, 142.0, 142.5, 143.0, 143.5, 144.0, 144.5, 
145.0, 145.5, 146.0, 146.5, 147.0, 147.5, 148.0, 148.5, 149.0, 149.5, 
150.0, 150.5, 151.0, 151.5, 152.0, 152.5, 153.0, 153.5, 154.0, 154.5, 
155.0, 155.5, 156.0, 156.5, 157.0, 157.5, 158.0, 158.5, 159.0, 159.5, 
160.0, 160.5, 161.0, 161.5, 162.0, 162.5, 163.0, 163.5, 164.0, 164.5, 
165.0, 165.5, 166.0, 166.5, 167.0, 167.5, 168.0, 168.5, 169.0, 169.5, 
170.0, 170.5, 171.0, 171.5, 172.0, 172.5, 173.0, 173.5, 174.0, 174.5, 
175.0, 175.5, 176.0, 176.5, 177.0, 177.5, 178.0, 178.5, 179.0, 179.5, 
180.0, 180.5, 181.0, 181.5, 182.0, 182.5, 183.0, 183.5, 184.0, 184.5, 
185.0, 185.5, 186.0, 186.5, 187.0, 187.5, 188.0, 188.5, 189.0, 189.5, 
190.0, 190.5, 191.0, 191.5, 192.0, 192.5, 193.0, 193.5, 194.0, 194.5, 
195.0, 195.5, 196.0, 196.5, 197.0, 197.5, 198.0, 198.5, 199.0, 199.5, 
200.0, 200.5, 201.0, 201.5, 202.0, 202.5, 203.0, 203.5, 204.0, 204.5, 
205.0, 205.5, 206.0, 206.5, 207.0, 207.5, 208.0, 208.5, 209.0, 209.5, 
210.0, 210.5, 211.0, 211.5, 212.0, 212.5, 213.0, 213.5, 214.0, 214.5, 
215.0, 215.5, 216.0, 216.5, 217.0, 217.5, 218.0, 218.5, 219.0, 219.5, 
220.0, 220.5, 221.0, 221.5, 222.0, 222.5, 223.0, 223.5, 224.0, 224.5, 
225.0, 225.5, 226.0, 226.5, 227.0, 227.5, 228.0, 228.5, 229.0, 229.5, 
230.0, 230.5, 231.0, 231.5, 232.0, 232.5, 233.0, 233.5, 234.0, 234.5, 
235.0, 235.5, 236.0, 236.5, 237.0, 237.5, 238.0, 238.5, 239.0, 239.5, 
240.0, 240.5, 241.0, 241.5, 242.0, 242.5, 243.0, 243.5, 244.0, 244.5, 
245.0, 245.5, 246.0, 246.5, 247.0, 247.5, 248.0, 248.5, 249.0, 249.5, 
250.0, 250.5, 251.0, 251.5, 252.0, 252.5, 253.0, 253.5, 254.0, 254.5, 
255.0, 255.5, 256.0, 256.5, 257.0, 257.5, 258.0, 258.5, 259.0, 259.5, 
260.0, 260.5, 261.0, 261.5, 262.0, 262.5, 263.0, 263.5, 264.0, 264.5, 
265.0, 265.5, 266.0, 266.5, 267.0, 267.5, 268.0, 268.5, 269.0, 269.5, 
270.0, 270.5, 271.0, 271.5, 272.0, 272.5, 273.0, 273.5, 274.0, 274.5, 
275.0, 275.5, 276.0, 276.5, 277.0, 277.5, 278.0, 278.5, 279.0, 279.5, 
280.0, 280.5, 281.0, 281.5, 282.0, 282.5, 283.0, 283.5, 284.0, 284.5, 
285.0, 285.5, 286.0, 286.5, 287.0, 287.5, 288.0, 288.5, 289.0, 289.5, 
290.0, 290.5, 291.0, 291.5, 292.0, 292.5, 293.0, 293.5, 294.0, 294.5, 
295.0, 295.5, 296.0, 296.5, 297.0, 297.5, 298.0, 298.5, 299.0, 299.5, 
300.0, 300.5, 301.0, 301.5, 302.0, 302.5, 303.0, 303.5, 304.0, 304.5, 
305.0, 305.5, 306.0, 306.5, 307.0, 307.5, 308.0, 308.5, 309.0, 309.5, 
310.0, 310.5, 311.0, 311.5, 312.0, 312.5, 313.0, 313.5, 314.0, 314.5, 
315.0, 315.5, 316.0, 316.5, 317.0, 317.5, 318.0, 318.5, 319.0, 319.5, 
320.0, 320.5, 321.0, 321.5, 322.0, 322.5, 323.0, 323.5, 324.0, 324.5, 
325.0, 325.5, 326.0, 326.5, 327.0, 327.5, 328.0, 328.5, 329.0, 329.5, 
330.0, 330.5, 331.0, 331.5, 332.0, 332.5, 333.0, 333.5, 334.0, 334.5, 
335.0, 335.5, 336.0, 336.5, 337.0, 337.5, 338.0, 338.5, 339.0, 339.5, 
340.0, 340.5, 341.0, 341.5, 342.0, 342.5, 343.0, 343.5, 344.0, 344.5, 
345.0, 345.5, 346.0, 346.5, 347.0, 347.5, 348.0, 348.5, 349.0, 349.5, 
350.0, 350.5, 351.0, 351.5, 352.0, 352.5, 353.0, 353.5, 354.0, 354.5, 
355.0, 355.5, 356.0, 356.5, 357.0, 357.5, 358.0, 358.5, 359.0, 359.5, 
360.0, 360.5, 361.0, 361.5, 362.0, 362.5, 363.0, 363.5, 364.0, 364.5, 
365.0, 365.5, 366.0, 366.5, 367.0, 367.5, 368.0, 368.5, 369.0, 369.5, 
370.0, 370.5, 371.0, 371.5, 372.0, 372.5, 373.0, 373.5, 374.0, 374.5, 
375.0, 375.5, 376.0, 376.5, 377.0, 377.5, 378.0, 378.5, 379.0, 379.5, 
380.0, 380.5, 381.0, 381.5, 382.0, 382.5, 383.0, 383.5, 384.0, 384.5, 
385.0, 385.5, 386.0, 386.5, 387.0, 387.5, 388.0, 388.5, 389.0, 389.5, 
390.0, 390.5, 391.0, 391.5, 392.0, 392.5, 393.0, 393.5, 394.0, 394.5, 
395.0, 395.5, 396.0, 396.5, 397.0, 397.5, 398.0, 398.5, 399.0, 399.5, 
400.0, 400.5, 401.0, 401.5, 402.0, 402.5, 403.0, 403.5, 404.0, 404.5, 
405.0, 405.5, 406.0, 406.5, 407.0, 407.5, 408.0, 408.5, 409.0, 409.5, 
410.0, 410.5, 411.0, 411.5, 412.0, 412.5, 413.0, 413.5, 414.0, 414.5, 
415.0, 415.5, 416.0, 416.5, 417.0, 417.5, 418.0, 418.5, 419.0, 419.5, 
420.0, 420.5, 421.0, 421.5, 422.0, 422.5, 423.0, 423.5, 424.0, 424.5, 
425.0, 425.5, 426.0, 426.5, 427.0, 427.5, 428.0, 428.5, 429.0, 429.5, 
430.0, 430.5, 431.0, 431.5, 432.0, 432.5, 433.0, 433.5, 434.0, 434.5, 
435.0, 435.5, 436.0, 436.5, 437.0, 437.5, 438.0, 438.5, 439.0, 439.5, 
440.0, 440.5, 441.0, 441.5, 442.0, 442.5, 443.0, 443.5, 444.0, 444.5, 
445.0, 445.5, 446.0, 446.5, 447.0, 447.5, 448.0, 448.5, 449.0, 449.5, 
450.0, 450.5, 451.0, 451.5, 452.0, 452.5, 453.0, 453.5, 454.0, 454.5, 
455.0, 455.5, 456.0, 456.5, 457.0, 457.5, 458.0, 458.5, 459.0, 459.5, 
460.0, 460.5, 461.0, 461.5, 462.0, 462.5, 463.0, 463.5, 464.0, 464.5, 
465.0, 465.5, 466.0, 466.5, 467.0, 467.5, 468.0, 468.5, 469.0, 469.5, 
470.0, 470.5, 471.0, 471.5, 472.0, 472.5, 473.0, 473.5, 474.0, 474.5, 
475.0, 475.5, 476.0, 476.5, 477.0, 477.5, 478.0, 478.5, 479.0, 479.5, 
480.0, 480.5, 481.0, 481.5, 482.0, 482.5, 483.0, 483.5, 484.0, 484.5, 
485.0, 485.5, 486.0, 486.5, 487.0, 487.5, 488.0, 488.5, 489.0, 489.5, 
490.0, 490.5, 491.0, 491.5, 492.0, 492.5, 493.0, 493.5, 494.0, 494.5, 
495.0, 495.5, 496.0, 496.5, 497.0, 497.5, 498.0, 498.5, 499.0, 499.5, 
500.0, 500.5, 501.0, 501.5, 502.0, 502.5, 503.0, 503.5, 504.0, 504.5, 
505.0, 505.5, 506.0, 506.5, 507.0, 507.5, 508.0, 508.5, 509.0, 509.5, 
510.0, 510.5, 511.0, 511.5, 512.0, 512.5, 513.0, 513.5, 514.0, 514.5, 
515.0, 515.5, 516.0, 516.5, 517.0, 517.5, 518.0, 518.5, 519.0, 519.5, 
520.0, 520.5, 521.0, 521.5, 522.0, 522.5, 523.0, 523.5, 524.0, 524.5, 
525.0, 525.5, 526.0, 526.5, 527.0, 527.5, 528.0, 528.5, 529.0, 529.5, 
530.0, 530.5, 531.0, 531.5, 532.0, 532.5, 533.0, 533.5, 534.0, 534.5, 
535.0, 535.5, 536.0, 536.5, 537.0, 537.5, 538.0, 538.5, 539.0, 539.5, 
540.0, 540.5, 541.0, 541.5, 542.0, 542.5, 543.0, 543.5, 544.0, 544.5, 
545.0, 545.5, 546.0, 546.5, 547.0, 547.5, 548.0, 548.5, 549.0, 549.5, 
550.0, 550.5, 551.0, 551.5, 552.0, 552.5, 553.0, 553.5, 554.0, 554.5, 
555.0, 555.5, 556.0, 556.5, 557.0, 557.5, 558.0, 558.5, 559.0, 559.5, 
560.0, 560.5, 561.0, 561.5, 562.0, 562.5, 563.0, 563.5, 564.0, 564.5, 
565.0, 565.5, 566.0, 566.5, 567.0, 567.5, 568.0, 568.5, 569.0, 569.5, 
570.0, 570.5, 571.0, 571.5, 572.0, 572.5, 573.0, 573.5, 574.0, 574.5, 
575.0, 575.5, 576.0, 576.5, 577.0, 577.5, 578.0, 578.5, 579.0, 579.5, 
580.0, 580.5, 581.0, 581.5, 582.0, 582.5, 583.0, 583.5, 584.0, 584.5, 
585.0, 585.5, 586.0, 586.5, 587.0, 587.5, 588.0, 588.5, 589.0, 589.5, 
590.0, 590.5, 591.0, 591.5, 592.0, 592.5, 593.0, 593.5, 594.0, 594.5, 
595.0, 595.5, 596.0, 596.5, 597.0, 597.5, 598.0, 598.5, 599.0, 599.5, 
600.0, 600.5, 601.0, 601.5, 602.0, 602.5, 603.0, 603.5, 604.0, 604.5, 
605.0, 605.5, 606.0, 606.5, 607.0, 607.5, 608.0, 608.5, 609.0, 609.5, 
610.0, 610.5, 611.0, 611.5, 612.0, 612.5, 613.0, 613.5, 614.0, 614.5, 
615.0, 615.5, 616.0, 616.5, 617.0, 617.5, 618.0, 618.5, 619.0, 619.5, 
620.0, 620.5, 621.0, 621.5, 622.0, 622.5, 623.0, 623.5, 624.0, 624.5, 
625.0, 625.5, 626.0, 626.5, 627.0, 627.5, 628.0, 628.5, 629.0, 629.5, 
630.0, 630.5, 631.0, 631.5, 632.0, 632.5, 633.0, 633.5, 634.0, 634.5, 
635.0, 635.5, 636.0, 636.5, 637.0, 637.5, 638.0, 638.5, 639.0, 639.5, 
640.0, 640.5, 641.0, 641.5, 642.0, 642.5, 643.0, 643.5, 644.0, 644.5, 
645.0, 645.5, 646.0, 646.5, 647.0, 647.5, 648.0, 648.5, 649.0, 649.5, 
650.0, 650.5, 651.0, 651.5, 652.0, 652.5, 653.0, 653.5, 654.0, 654.5, 
655.0, 655.5, 656.0, 656.5, 657.0, 657.5, 658.0, 658.5, 659.0, 659.5, 
660.0, 660.5, 661.0, 661.5, 662.0, 662.5, 663.0, 663.5, 664.0, 664.5, 
665.0, 665.5, 666.0, 666.5, 667.0, 667.5, 668.0, 668.5, 669.0, 669.5, 
670.0, 670.5, 671.0, 671.5, 672.0, 672.5, 673.0, 673.5, 674.0, 674.5, 
675.0, 675.5, 676.0, 676.5, 677.0, 677.5, 678.0, 678.5, 679.0, 679.5, 
680.0, 680.5, 681.0, 681.5, 682.0, 682.5, 683.0, 683.5, 684.0, 684.5, 
685.0, 685.5, 686.0, 686.5, 687.0, 687.5, 688.0, 688.5, 689.0, 689.5, 
690.0, 690.5, 691.0, 691.5, 692.0, 692.5, 693.0, 693.5, 694.0, 694.5, 
695.0, 695.5, 696.0, 696.5, 697.0, 697.5, 698.0, 698.5, 699.0, 699.5, 
700.0, 700.5, 701.0, 701.5, 702.0, 702.5, 703.0, 703.5, 704.0, 704.5, 
705.0, 705.5, 706.0, 706.5, 707.0, 707.5, 708.0, 708.5, 709.0, 709.5, 
710.0, 710.5, 711.0, 711.5, 712.0, 712.5, 713.0, 713.5, 714.0, 714.5, 
715.0, 715.5, 716.0, 716.5, 717.0, 717.5, 718.0, 718.5, 719.0, 719.5, 
720.0, 720.5, 721.0, 721.5, 722.0, 722.5, 723.0, 723.5, 724.0, 724.5, 
725.0, 725.5, 726.0, 726.5, 727.0, 727.5, 728.0, 728.5, 729.0, 729.5, 
730.0, 730.5, 731.0, 731.5, 732.0, 732.5, 733.0, 733.5, 734.0, 734.5, 
735.0, 735.5, 736.0, 736.5, 737.0, 737.5, 738.0, 738.5, 739.0, 739.5, 
740.0, 740.5, 741.0, 741.5, 742.0, 742.5, 743.0, 743.5, 744.0, 744.5, 
745.0, 745.5, 746.0, 746.5, 747.0, 747.5, 748.0, 748.5, 749.0, 749.5, 
750.0, 750.5, 751.0, 751.5, 752.0, 752.5, 753.0, 753.5, 754.0, 754.5, 
755.0, 755.5, 756.0, 756.5, 757.0, 757.5, 758.0, 758.5, 759.0, 759.5, 
760.0, 760.5, 761.0, 761.5, 762.0, 762.5, 763.0, 763.5, 764.0, 764.5, 
765.0, 765.5, 766.0, 766.5, 767.0, 767.5, 768.0, 768.5, 769.0, 769.5, 
770.0, 770.5, 771.0, 771.5, 772.0, 772.5, 773.0, 773.5, 774.0, 774.5, 
775.0, 775.5, 776.0, 776.5, 777.0, 777.5, 778.0, 778.5, 779.0, 779.5, 
780.0, 780.5, 781.0, 781.5, 782.0, 782.5, 783.0, 783.5, 784.0, 784.5, 
785.0, 785.5, 786.0, 786.5, 787.0, 787.5, 788.0, 788.5, 789.0, 789.5, 
790.0, 790.5, 791.0, 791.5, 792.0, 792.5, 793.0, 793.5, 794.0, 794.5, 
795.0, 795.5, 796.0, 796.5, 797.0, 797.5, 798.0, 798.5, 799.0, 799.5, 
800.0, 800.5, 801.0, 801.5, 802.0, 802.5, 803.0, 803.5, 804.0, 804.5, 
805.0, 805.5, 806.0, 806.5, 807.0, 807.5, 808.0, 808.5, 809.0, 809.5, 
810.0, 810.5, 811.0, 811.5, 812.0, 812.5, 813.0, 813.5, 814.0, 814.5, 
815.0, 815.5, 816.0, 816.5, 817.0, 817.5, 818.0, 818.5, 819.0, 819.5, 
820.0, 820.5, 821.0, 821.5, 822.0, 822.5, 823.0, 823.5, 824.0, 824.5, 
825.0, 825.5, 826.0, 826.5, 827.0, 827.5, 828.0, 828.5, 829.0, 829.5, 
830.0, 830.5, 831.0, 831.5, 832.0, 832.5, 833.0, 833.5, 834.0, 834.5, 
835.0, 835.5, 836.0, 836.5, 837.0, 837.5, 838.0, 838.5, 839.0, 839.5, 
840.0, 840.5, 841.0, 841.5, 842.0, 842.5, 843.0, 843.5, 844.0, 844.5, 
845.0, 845.5, 846.0, 846.5, 847.0, 847.5, 848.0, 848.5, 849.0, 849.5, 
850.0, 850.5, 851.0, 851.5, 852.0, 852.5, 853.0, 853.5, 854.0, 854.5, 
855.0, 855.5, 856.0, 856.5, 857.0, 857.5, 858.0, 858.5, 859.0, 859.5, 
860.0, 860.5, 861.0, 861.5, 862.0, 862.5, 863.0, 863.5, 864.0, 864.5, 
865.0, 865.5, 866.0, 866.5, 867.0, 867.5, 868.0, 868.5, 869.0, 869.5, 
870.0, 870.5, 871.0, 871.5, 872.0, 872.5, 873.0, 873.5, 874.0, 874.5, 
875.0, 875.5, 876.0, 876.5, 877.0, 877.5, 878.0, 878.5, 879.0, 879.5, 
880.0, 880.5, 881.0, 881.5, 882.0, 882.5, 883.0, 883.5, 884.0, 884.5, 
885.0, 885.5, 886.0, 886.5, 887.0, 887.5, 888.0, 888.5, 889.0, 889.5, 
890.0, 890.5, 891.0, 891.5, 892.0, 892.5, 893.0, 893.5, 894.0, 894.5, 
895.0, 895.5, 896.0, 896.5, 897.0, 897.5, 898.0, 898.5, 899.0, 899.5, 
900.0, 900.5, 901.0, 901.5, 902.0, 902.5, 903.0, 903.5, 904.0, 904.5, 
905.0, 905.5, 906.0, 906.5, 907.0, 907.5, 908.0, 908.5, 909.0, 909.5, 
910.0, 910.5, 911.0, 911.5, 912.0, 912.5, 913.0, 913.5, 914.0, 914.5, 
915.0, 915.5, 916.0, 916.5, 917.0, 917.5, 918.0, 918.5, 919.0, 919.5, 
920.0, 920.5, 921.0, 921.5, 922.0, 922.5, 923.0, 923.5, 924.0, 924.5, 
925.0, 925.5, 926.0, 926.5, 927.0, 927.5, 928.0, 928.5, 929.0, 929.5, 
930.0, 930.5, 931.0, 931.5, 932.0, 932.5, 933.0, 933.5, 934.0, 934.5, 
935.0, 935.5, 936.0, 936.5, 937.0, 937.5, 938.0, 938.5, 939.0, 939.5, 
])
_f_vp_jma_2020b = scipy.interpolate.interp1d(_dep_jma_2020b, _vp_jma_2020b, kind='linear')
_f_vs_jma_2020b = scipy.interpolate.interp1d(_dep_jma_2020b, _vs_jma_2020b, kind='linear')
_vp_jma_2020c = np.array([
 4.800,  4.805,  4.810,  4.815,  4.820,  4.825,  4.830,  4.835,  4.840,  4.845, 
 4.850,  4.855,  4.860,  4.865,  4.870,  4.875,  4.880,  5.149,  5.441,  5.683, 
 5.885,  6.055,  6.200,  6.326,  6.438,  6.555,  6.665,  6.770,  6.871,  6.967, 
 7.057,  7.140,  7.215,  7.281,  7.337,  7.384,  7.423,  7.455,  7.481,  7.502, 
 7.519,  7.533,  7.545,  7.556,  7.566,  7.575,  7.583,  7.590,  7.596,  7.601, 
 7.605,  7.608,  7.611,  7.614,  7.617,  7.620,  7.623,  7.626,  7.629,  7.632, 
 7.635,  7.638,  7.641,  7.644,  7.647,  7.650,  7.653,  7.656,  7.659,  7.662, 
 7.665,  7.668,  7.671,  7.674,  7.677,  7.680,  7.683,  7.686,  7.689,  7.692, 
 7.695,  7.698,  7.701,  7.704,  7.707,  7.710,  7.713,  7.716,  7.719,  7.722, 
 7.725,  7.728,  7.731,  7.734,  7.737,  7.740,  7.743,  7.746,  7.749,  7.752, 
 7.755,  7.758,  7.761,  7.764,  7.767,  7.770,  7.773,  7.776,  7.779,  7.782, 
 7.785,  7.788,  7.791,  7.794,  7.797,  7.800,  7.803,  7.806,  7.809,  7.812, 
 7.815,  7.818,  7.821,  7.824,  7.827,  7.830,  7.833,  7.836,  7.839,  7.842, 
 7.845,  7.847,  7.849,  7.851,  7.853,  7.855,  7.857,  7.859,  7.861,  7.863, 
 7.865,  7.866,  7.867,  7.868,  7.869,  7.870,  7.871,  7.872,  7.873,  7.874, 
 7.875,  7.876,  7.877,  7.878,  7.879,  7.880,  7.881,  7.883,  7.885,  7.887, 
 7.889,  7.891,  7.893,  7.895,  7.896,  7.898,  7.900,  7.901,  7.903,  7.905, 
 7.906,  7.908,  7.909,  7.911,  7.912,  7.914,  7.915,  7.917,  7.918,  7.920, 
 7.921,  7.923,  7.924,  7.926,  7.927,  7.928,  7.930,  7.931,  7.933,  7.934, 
 7.936,  7.937,  7.938,  7.940,  7.941,  7.943,  7.944,  7.945,  7.947,  7.948, 
 7.949,  7.951,  7.952,  7.953,  7.954,  7.955,  7.956,  7.957,  7.959,  7.960, 
 7.961,  7.963,  7.964,  7.965,  7.967,  7.968,  7.970,  7.971,  7.973,  7.974, 
 7.976,  7.977,  7.979,  7.981,  7.982,  7.984,  7.985,  7.987,  7.989,  7.990, 
 7.992,  7.994,  7.996,  7.997,  7.999,  8.001,  8.002,  8.004,  8.006,  8.008, 
 8.009,  8.011,  8.013,  8.015,  8.017,  8.019,  8.021,  8.023,  8.025,  8.027, 
 8.028,  8.030,  8.032,  8.034,  8.036,  8.038,  8.039,  8.041,  8.043,  8.045, 
 8.046,  8.048,  8.050,  8.052,  8.053,  8.055,  8.057,  8.058,  8.060,  8.062, 
 8.063,  8.065,  8.067,  8.068,  8.070,  8.072,  8.073,  8.075,  8.076,  8.078, 
 8.079,  8.081,  8.083,  8.085,  8.086,  8.088,  8.090,  8.091,  8.093,  8.095, 
 8.096,  8.098,  8.099,  8.101,  8.102,  8.104,  8.105,  8.107,  8.108,  8.110, 
 8.111,  8.113,  8.114,  8.116,  8.117,  8.118,  8.120,  8.121,  8.123,  8.124, 
 8.126,  8.127,  8.128,  8.130,  8.131,  8.133,  8.134,  8.135,  8.137,  8.138, 
 8.139,  8.141,  8.142,  8.143,  8.144,  8.145,  8.146,  8.147,  8.149,  8.150, 
 8.151,  8.153,  8.154,  8.155,  8.157,  8.158,  8.160,  8.161,  8.163,  8.164, 
 8.166,  8.167,  8.169,  8.171,  8.172,  8.174,  8.175,  8.177,  8.179,  8.180, 
 8.182,  8.184,  8.186,  8.187,  8.189,  8.191,  8.192,  8.194,  8.196,  8.198, 
 8.199,  8.202,  8.204,  8.205,  8.207,  8.209,  8.211,  8.213,  8.215,  8.217, 
 8.219,  8.221,  8.223,  8.224,  8.226,  8.228,  8.230,  8.232,  8.233,  8.235, 
 8.237,  8.238,  8.240,  8.242,  8.244,  8.245,  8.247,  8.249,  8.250,  8.252, 
 8.254,  8.255,  8.257,  8.258,  8.260,  8.262,  8.263,  8.265,  8.266,  8.268, 
 8.269,  8.271,  8.272,  8.274,  8.275,  8.277,  8.278,  8.280,  8.281,  8.283, 
 8.284,  8.286,  8.287,  8.289,  8.290,  8.292,  8.293,  8.295,  8.296,  8.298, 
 8.299,  8.301,  8.302,  8.304,  8.305,  8.307,  8.308,  8.310,  8.311,  8.313, 
 8.314,  8.316,  8.317,  8.319,  8.320,  8.322,  8.323,  8.325,  8.326,  8.328, 
 8.329,  8.331,  8.333,  8.335,  8.336,  8.338,  8.340,  8.341,  8.343,  8.345, 
 8.346,  8.348,  8.349,  8.351,  8.352,  8.354,  8.355,  8.357,  8.358,  8.360, 
 8.361,  8.363,  8.364,  8.366,  8.367,  8.368,  8.370,  8.371,  8.373,  8.374, 
 8.376,  8.377,  8.378,  8.380,  8.381,  8.383,  8.384,  8.385,  8.387,  8.388, 
 8.389,  8.390,  8.391,  8.393,  8.394,  8.395,  8.396,  8.397,  8.398,  8.400, 
 8.401,  8.402,  8.404,  8.405,  8.406,  8.408,  8.409,  8.411,  8.412,  8.414, 
 8.415,  8.417,  8.419,  8.420,  8.422,  8.423,  8.425,  8.427,  8.428,  8.430, 
 8.432,  8.434,  8.435,  8.437,  8.439,  8.441,  8.442,  8.444,  8.446,  8.448, 
 8.449,  8.452,  8.454,  8.456,  8.458,  8.460,  8.462,  8.464,  8.466,  8.468, 
 8.470,  8.472,  8.474,  8.476,  8.478,  8.479,  8.481,  8.483,  8.485,  8.486, 
 8.488,  8.490,  8.492,  8.493,  8.495,  8.496,  8.498,  8.500,  8.501,  8.503, 
 8.504,  8.506,  8.507,  8.509,  8.511,  8.512,  8.514,  8.515,  8.517,  8.518, 
 8.519,  8.521,  8.522,  8.523,  8.525,  8.526,  8.527,  8.529,  8.530,  8.531, 
 8.533,  8.534,  8.535,  8.537,  8.538,  8.540,  8.541,  8.543,  8.544,  8.546, 
 8.547,  8.549,  8.550,  8.552,  8.553,  8.555,  8.557,  8.558,  8.560,  8.561, 
 8.563,  8.565,  8.566,  8.568,  8.569,  8.571,  8.573,  8.574,  8.576,  8.578, 
 8.579,  8.581,  8.583,  8.585,  8.586,  8.588,  8.590,  8.592,  8.593,  8.595, 
 8.597,  8.599,  8.600,  8.602,  8.604,  8.606,  8.607,  8.609,  8.611,  8.613, 
 8.614,  8.616,  8.618,  8.620,  8.621,  8.623,  8.625,  8.627,  8.628,  8.630, 
 8.632,  8.634,  8.635,  8.637,  8.639,  8.641,  8.642,  8.644,  8.646,  8.648, 
 8.649,  8.651,  8.653,  8.655,  8.657,  8.658,  8.660,  8.662,  8.664,  8.666, 
 8.667,  8.669,  8.671,  8.673,  8.674,  8.676,  8.678,  8.680,  8.681,  8.683, 
 8.685,  8.687,  8.688,  8.690,  8.692,  8.694,  8.695,  8.697,  8.699,  8.700, 
 8.702,  8.704,  8.706,  8.707,  8.709,  8.711,  8.713,  8.714,  8.716,  8.718, 
 8.719,  8.721,  8.722,  8.723,  8.725,  8.726,  8.728,  8.729,  8.731,  8.732, 
 8.734,  8.735,  8.737,  8.739,  8.740,  8.742,  8.744,  8.746,  8.748,  8.749, 
 8.751,  8.753,  8.755,  8.757,  8.759,  8.760,  8.762,  8.764,  8.766,  8.768, 
 8.770,  8.772,  8.774,  8.776,  8.778,  8.780,  8.782,  8.784,  8.786,  8.788, 
 8.789,  8.792,  8.795,  8.798,  8.801,  8.803,  8.806,  8.808,  8.810,  8.812, 
 8.815,  8.817,  8.819,  8.821,  8.823,  8.824,  8.826,  8.828,  8.830,  8.831, 
 8.833,  8.835,  8.836,  8.838,  8.839,  8.841,  8.842,  8.843,  8.845,  8.846, 
 8.847,  8.849,  8.850,  8.851,  8.852,  8.854,  8.855,  8.856,  8.857,  8.858, 
 8.859,  8.860,  8.861,  8.861,  8.862,  8.863,  8.864,  8.864,  8.865,  8.866, 
 8.867,  8.869,  8.870,  8.871,  8.872,  8.873,  8.875,  8.876,  8.878,  8.879, 
 8.881,  8.882,  8.884,  8.885,  8.887,  8.889,  8.891,  8.892,  8.894,  8.896, 
 8.898,  8.900,  8.902,  8.904,  8.906,  8.908,  8.911,  8.913,  8.915,  8.917, 
 8.919,  8.922,  8.924,  8.927,  8.929,  8.932,  8.934,  8.937,  8.939,  8.942, 
 8.944,  8.947,  8.949,  8.952,  8.955,  8.957,  8.960,  8.963,  8.965,  8.968, 
 8.971,  8.974,  8.977,  8.979,  8.982,  8.985,  8.988,  8.991,  8.994,  8.997, 
 9.000,  9.002,  9.005,  9.008,  9.011,  9.014,  9.017,  9.020,  9.023,  9.026, 
 9.029,  9.033,  9.036,  9.039,  9.042,  9.045,  9.049,  9.052,  9.055,  9.058, 
 9.061,  9.065,  9.068,  9.071,  9.074,  9.077,  9.081,  9.084,  9.087,  9.090, 
 9.094,  9.097,  9.100,  9.103,  9.107,  9.110,  9.113,  9.116,  9.120,  9.123, 
 9.126,  9.130,  9.133,  9.136,  9.139,  9.143,  9.146,  9.149,  9.153,  9.156, 
 9.159,  9.163,  9.166,  9.169,  9.173,  9.176,  9.179,  9.183,  9.186,  9.190, 
 9.193,  9.196,  9.200,  9.203,  9.207,  9.210,  9.214,  9.217,  9.221,  9.224, 
 9.228,  9.231,  9.235,  9.238,  9.242,  9.245,  9.249,  9.252,  9.256,  9.260, 
 9.263,  9.267,  9.270,  9.274,  9.278,  9.281,  9.285,  9.289,  9.292,  9.296, 
 9.299,  9.303,  9.307,  9.310,  9.314,  9.317,  9.321,  9.325,  9.328,  9.332, 
 9.336,  9.339,  9.343,  9.347,  9.351,  9.354,  9.358,  9.362,  9.366,  9.369, 
 9.373,  9.377,  9.381,  9.384,  9.388,  9.392,  9.396,  9.400,  9.404,  9.407, 
 9.411,  9.415,  9.419,  9.423,  9.426,  9.430,  9.434,  9.438,  9.442,  9.446, 
 9.449,  9.454,  9.458,  9.462,  9.466,  9.470,  9.474,  9.478,  9.482,  9.486, 
 9.490,  9.493,  9.497,  9.501,  9.505,  9.509,  9.513,  9.516,  9.520,  9.524, 
 9.528,  9.531,  9.535,  9.539,  9.542,  9.546,  9.550,  9.553,  9.557,  9.560, 
 9.564,  9.568,  9.571,  9.575,  9.578,  9.582,  9.585,  9.589,  9.592,  9.596, 
 9.599,  9.603,  9.606,  9.609,  9.612,  9.615,  9.618,  9.622,  9.625,  9.628, 
 9.632,  9.635,  9.638,  9.642,  9.645,  9.649,  9.652,  9.656,  9.659,  9.663, 
 9.666,  9.670,  9.673,  9.677,  9.681,  9.684,  9.688,  9.692,  9.695,  9.699, 
 9.703,  9.706,  9.710,  9.714,  9.717,  9.721,  9.725,  9.728,  9.732,  9.736, 
 9.739,  9.744,  9.748,  9.752,  9.756,  9.760,  9.764,  9.768,  9.772,  9.776, 
 9.780,  9.784,  9.787,  9.791,  9.795,  9.798,  9.802,  9.805,  9.809,  9.812, 
 9.816,  9.819,  9.823,  9.826,  9.829,  9.833,  9.836,  9.839,  9.842,  9.845, 
 9.849,  9.852,  9.855,  9.858,  9.861,  9.864,  9.867,  9.870,  9.873,  9.876, 
 9.879,  9.882,  9.885,  9.888,  9.890,  9.893,  9.896,  9.899,  9.901,  9.904, 
 9.907,  9.910,  9.913,  9.916,  9.919,  9.922,  9.925,  9.928,  9.931,  9.934, 
 9.937,  9.940,  9.943,  9.946,  9.949,  9.952,  9.955,  9.958,  9.961,  9.964, 
 9.968,  9.971,  9.974,  9.977,  9.980,  9.983,  9.987,  9.990,  9.993,  9.996, 
10.000, 10.003, 10.006, 10.009, 10.012, 10.015, 10.018, 10.022, 10.025, 10.028, 
10.031, 10.035, 10.038, 10.041, 10.044, 10.048, 10.051, 10.054, 10.057, 10.061, 
10.064, 10.067, 10.071, 10.074, 10.077, 10.080, 10.084, 10.087, 10.090, 10.094, 
10.097, 10.100, 10.103, 10.107, 10.110, 10.113, 10.117, 10.120, 10.123, 10.126, 
10.129, 10.133, 10.137, 10.140, 10.144, 10.147, 10.150, 10.154, 10.157, 10.160, 
10.164, 10.167, 10.170, 10.173, 10.176, 10.179, 10.182, 10.185, 10.188, 10.191, 
10.194, 10.197, 10.200, 10.203, 10.206, 10.209, 10.212, 10.215, 10.217, 10.220, 
10.223, 10.226, 10.228, 10.231, 10.234, 10.236, 10.239, 10.242, 10.244, 10.247, 
10.250, 10.252, 10.254, 10.256, 10.259, 10.261, 10.264, 10.266, 10.268, 10.271, 
10.273, 10.276, 10.278, 10.281, 10.283, 10.286, 10.288, 10.291, 10.293, 10.296, 
10.298, 10.301, 10.303, 10.306, 10.308, 10.311, 10.313, 10.316, 10.319, 10.321, 
10.324, 10.326, 10.329, 10.331, 10.334, 10.337, 10.339, 10.342, 10.344, 10.347, 
10.349, 10.352, 10.355, 10.358, 10.361, 10.363, 10.366, 10.369, 10.371, 10.374, 
10.377, 10.379, 10.382, 10.385, 10.387, 10.390, 10.392, 10.395, 10.397, 10.400, 
10.402, 10.405, 10.407, 10.410, 10.412, 10.414, 10.417, 10.419, 10.422, 10.424, 
10.426, 10.429, 10.431, 10.433, 10.436, 10.438, 10.440, 10.443, 10.445, 10.447, 
10.449, 10.452, 10.454, 10.456, 10.458, 10.460, 10.462, 10.465, 10.467, 10.469, 
10.471, 10.473, 10.476, 10.478, 10.480, 10.482, 10.484, 10.487, 10.489, 10.491, 
10.493, 10.496, 10.498, 10.500, 10.503, 10.505, 10.507, 10.509, 10.512, 10.514, 
10.516, 10.519, 10.521, 10.523, 10.526, 10.528, 10.530, 10.533, 10.535, 10.537, 
10.539, 10.542, 10.544, 10.546, 10.549, 10.551, 10.553, 10.556, 10.558, 10.560, 
10.563, 10.565, 10.567, 10.569, 10.572, 10.574, 10.576, 10.579, 10.581, 10.583, 
10.585, 10.588, 10.590, 10.592, 10.594, 10.597, 10.599, 10.601, 10.603, 10.606, 
10.608, 10.610, 10.612, 10.614, 10.617, 10.619, 10.621, 10.623, 10.625, 10.627, 
10.629, 10.632, 10.634, 10.637, 10.639, 10.641, 10.643, 10.646, 10.648, 10.650, 
10.652, 10.654, 10.656, 10.658, 10.661, 10.663, 10.665, 10.667, 10.669, 10.671, 
10.673, 10.674, 10.676, 10.678, 10.680, 10.682, 10.684, 10.686, 10.688, 10.690, 
10.691, 10.693, 10.695, 10.697, 10.699, 10.701, 10.702, 10.704, 10.706, 10.708, 
10.709, 10.711, 10.712, 10.714, 10.715, 10.717, 10.718, 10.720, 10.721, 10.723, 
10.724, 10.726, 10.728, 10.729, 10.731, 10.733, 10.735, 10.736, 10.738, 10.740, 
10.742, 10.744, 10.745, 10.747, 10.749, 10.751, 10.753, 10.755, 10.757, 10.758, 
10.760, 10.762, 10.764, 10.766, 10.768, 10.770, 10.772, 10.774, 10.776, 10.778, 
10.779, 10.782, 10.784, 10.786, 10.788, 10.791, 10.793, 10.795, 10.797, 10.799, 
10.801, 10.803, 10.805, 10.806, 10.808, 10.810, 10.812, 10.814, 10.816, 10.817, 
10.819, 10.821, 10.822, 10.824, 10.826, 10.827, 10.829, 10.830, 10.832, 10.834, 
10.835, 10.837, 10.838, 10.840, 10.841, 10.842, 10.844, 10.845, 10.847, 10.848, 
10.849, 10.851, 10.852, 10.853, 10.854, 10.856, 10.857, 10.858, 10.859, 10.861, 
10.862, 10.863, 10.864, 10.866, 10.867, 10.868, 10.869, 10.871, 10.872, 10.873, 
10.874, 10.876, 10.877, 10.878, 10.879, 10.881, 10.882, 10.883, 10.884, 10.886, 
10.887, 10.888, 10.889, 10.891, 10.892, 10.893, 10.894, 10.896, 10.897, 10.898, 
10.899, 10.901, 10.902, 10.904, 10.905, 10.906, 10.908, 10.909, 10.910, 10.911, 
10.913, 10.914, 10.915, 10.917, 10.918, 10.919, 10.920, 10.922, 10.923, 10.924, 
10.925, 10.927, 10.928, 10.929, 10.930, 10.931, 10.933, 10.934, 10.935, 10.936, 
10.937, 10.939, 10.940, 10.941, 10.942, 10.943, 10.945, 10.946, 10.947, 10.948, 
10.949, 10.950, 10.951, 10.952, 10.953, 10.954, 10.955, 10.956, 10.957, 10.958, 
10.959, 10.961, 10.962, 10.963, 10.964, 10.965, 10.967, 10.968, 10.969, 10.970, 
10.972, 10.973, 10.974, 10.976, 10.977, 10.978, 10.980, 10.981, 10.983, 10.984, 
10.985, 10.987, 10.988, 10.990, 10.991, 10.992, 10.994, 10.995, 10.997, 10.998, 
11.000, 11.001, 11.003, 11.005, 11.006, 11.008, 11.010, 11.011, 11.013, 11.014, 
11.016, 11.017, 11.019, 11.020, 11.021, 11.023, 11.024, 11.025, 11.027, 11.028, 
11.029, 11.030, 11.031, 11.033, 11.034, 11.035, 11.036, 11.037, 11.038, 11.039, 
11.040, 11.041, 11.042, 11.043, 11.044, 11.045, 11.046, 11.047, 11.048, 11.049, 
11.049, 11.050, 11.051, 11.052, 11.053, 11.053, 11.054, 11.055, 11.056, 11.057, 
11.057, 11.058, 11.059, 11.060, 11.060, 11.061, 11.062, 11.063, 11.063, 11.064, 
11.065, 11.066, 11.066, 11.067, 11.068, 11.069, 11.069, 11.070, 11.071, 11.071, 
11.072, 11.073, 11.074, 11.074, 11.075, 11.076, 11.077, 11.077, 11.078, 11.079, 
11.079, 11.080, 11.081, 11.081, 11.082, 11.082, 11.083, 11.084, 11.084, 11.085, 
11.086, 11.086, 11.087, 11.088, 11.089, 11.089, 11.090, 11.091, 11.092, 11.092, 
11.093, 11.094, 11.095, 11.095, 11.096, 11.097, 11.098, 11.099, 11.100, 11.100, 
11.101, 11.102, 11.103, 11.104, 11.104, 11.105, 11.106, 11.107, 11.108, 11.109, 
11.109, 11.111, 11.112, 11.112, 11.113, 11.114, 11.115, 11.116, 11.117, 11.118, 
11.119, 11.120, 11.121, 11.121, 11.122, 11.123, 11.124, 11.125, 11.125, 11.126, 
11.127, 11.127, 11.128, 11.129, 11.130, 11.130, 11.131, 11.132, 11.132, 11.133, 
11.134, 11.134, 11.135, 11.135, 11.136, 11.137, 11.137, 11.138, 11.138, 11.139, 
11.139, 11.140, 11.140, 11.141, 11.141, 11.142, 11.142, 11.143, 11.143, 11.144, 
11.144, 11.145, 11.145, 11.146, 11.146, 11.147, 11.147, 11.148, 11.148, 11.149, 
11.149, 11.150, 11.150, 11.151, 11.151, 11.152, 11.152, 11.153, 11.153, 11.154, 
11.154, 11.155, 11.155, 11.156, 11.156, 11.157, 11.157, 11.158, 11.158, 11.159, 
11.159, 11.160, 11.161, 11.161, 11.162, 11.162, 11.163, 11.163, 11.164, 11.164, 
11.165, 11.165, 11.166, 11.166, 11.167, 11.167, 11.168, 11.168, 11.169, 11.169, 
11.170, 11.170, 11.171, 11.171, 11.172, 11.172, 11.173, 11.173, 11.174, 11.174, 
11.175, 11.175, 11.176, 11.176, 11.177, 11.177, 11.178, 11.178, 11.179, 11.179, 
11.179, 11.180, 11.180, 11.180, 11.181, 11.181, 11.182, 11.182, 11.182, 11.183, 
11.183, 11.184, 11.184, 11.185, 11.185, 11.186, 11.186, 11.187, 11.187, 11.188, 
11.188, 11.189, 11.189, 11.190, 11.190, 11.191, 11.191, 11.192, 11.193, 11.193, 
11.194, 11.194, 11.195, 11.195, 11.196, 11.197, 11.197, 11.198, 11.198, 11.199, 
])
_vs_jma_2020c = np.array([
 2.796,  2.799,  2.801,  2.803,  2.805,  2.807,  2.809,  2.812,  2.814,  2.816, 
 2.818,  2.820,  2.822,  2.825,  2.827,  2.829,  2.831,  2.986,  3.155,  3.294, 
 3.411,  3.508,  3.591,  3.663,  3.727,  3.794,  3.856,  3.916,  3.974,  4.028, 
 4.079,  4.126,  4.168,  4.205,  4.236,  4.262,  4.284,  4.301,  4.315,  4.326, 
 4.335,  4.342,  4.347,  4.353,  4.357,  4.361,  4.365,  4.368,  4.370,  4.372, 
 4.373,  4.374,  4.374,  4.375,  4.375,  4.376,  4.376,  4.377,  4.378,  4.378, 
 4.379,  4.379,  4.380,  4.381,  4.381,  4.382,  4.382,  4.383,  4.383,  4.384, 
 4.385,  4.385,  4.386,  4.386,  4.387,  4.388,  4.388,  4.389,  4.389,  4.390, 
 4.390,  4.391,  4.392,  4.392,  4.393,  4.393,  4.394,  4.394,  4.395,  4.396, 
 4.396,  4.397,  4.397,  4.398,  4.399,  4.399,  4.400,  4.400,  4.401,  4.401, 
 4.402,  4.403,  4.403,  4.404,  4.404,  4.405,  4.405,  4.406,  4.407,  4.407, 
 4.408,  4.408,  4.409,  4.409,  4.410,  4.411,  4.411,  4.412,  4.412,  4.413, 
 4.413,  4.414,  4.415,  4.415,  4.416,  4.416,  4.417,  4.417,  4.418,  4.418, 
 4.419,  4.419,  4.419,  4.419,  4.419,  4.419,  4.419,  4.419,  4.419,  4.419, 
 4.419,  4.419,  4.419,  4.419,  4.419,  4.420,  4.420,  4.420,  4.420,  4.420, 
 4.420,  4.420,  4.420,  4.420,  4.420,  4.420,  4.420,  4.420,  4.420,  4.420, 
 4.420,  4.420,  4.421,  4.422,  4.423,  4.423,  4.424,  4.425,  4.425,  4.426, 
 4.427,  4.428,  4.428,  4.429,  4.429,  4.430,  4.430,  4.431,  4.431,  4.432, 
 4.433,  4.434,  4.434,  4.435,  4.435,  4.436,  4.437,  4.437,  4.438,  4.439, 
 4.440,  4.440,  4.441,  4.442,  4.442,  4.443,  4.443,  4.444,  4.445,  4.445, 
 4.446,  4.447,  4.447,  4.448,  4.448,  4.448,  4.449,  4.449,  4.450,  4.451, 
 4.451,  4.452,  4.453,  4.453,  4.454,  4.455,  4.456,  4.456,  4.457,  4.457, 
 4.458,  4.459,  4.460,  4.461,  4.461,  4.462,  4.463,  4.464,  4.465,  4.465, 
 4.466,  4.467,  4.468,  4.468,  4.469,  4.470,  4.471,  4.472,  4.473,  4.474, 
 4.474,  4.475,  4.476,  4.477,  4.478,  4.479,  4.480,  4.481,  4.482,  4.483, 
 4.484,  4.485,  4.486,  4.487,  4.488,  4.489,  4.489,  4.490,  4.491,  4.492, 
 4.492,  4.493,  4.494,  4.495,  4.496,  4.497,  4.498,  4.498,  4.499,  4.500, 
 4.501,  4.502,  4.503,  4.503,  4.504,  4.505,  4.506,  4.507,  4.507,  4.508, 
 4.508,  4.509,  4.510,  4.511,  4.512,  4.513,  4.514,  4.514,  4.515,  4.516, 
 4.517,  4.518,  4.518,  4.519,  4.519,  4.520,  4.521,  4.522,  4.522,  4.523, 
 4.524,  4.525,  4.525,  4.526,  4.527,  4.527,  4.528,  4.528,  4.529,  4.530, 
 4.531,  4.531,  4.532,  4.533,  4.533,  4.534,  4.535,  4.535,  4.536,  4.536, 
 4.537,  4.538,  4.538,  4.539,  4.539,  4.540,  4.540,  4.540,  4.541,  4.542, 
 4.542,  4.543,  4.544,  4.544,  4.545,  4.545,  4.546,  4.547,  4.548,  4.548, 
 4.549,  4.550,  4.551,  4.552,  4.552,  4.553,  4.554,  4.555,  4.556,  4.556, 
 4.557,  4.558,  4.559,  4.559,  4.560,  4.561,  4.562,  4.563,  4.564,  4.565, 
 4.565,  4.567,  4.568,  4.568,  4.569,  4.570,  4.571,  4.572,  4.573,  4.574, 
 4.575,  4.576,  4.577,  4.577,  4.578,  4.579,  4.580,  4.581,  4.582,  4.583, 
 4.584,  4.584,  4.585,  4.587,  4.588,  4.588,  4.589,  4.590,  4.591,  4.592, 
 4.593,  4.594,  4.595,  4.595,  4.597,  4.598,  4.598,  4.599,  4.600,  4.601, 
 4.602,  4.603,  4.603,  4.604,  4.604,  4.605,  4.606,  4.607,  4.607,  4.608, 
 4.609,  4.610,  4.610,  4.611,  4.611,  4.612,  4.613,  4.614,  4.614,  4.615, 
 4.616,  4.617,  4.617,  4.618,  4.619,  4.620,  4.621,  4.622,  4.622,  4.623, 
 4.624,  4.625,  4.626,  4.627,  4.627,  4.628,  4.629,  4.630,  4.631,  4.632, 
 4.632,  4.633,  4.634,  4.635,  4.636,  4.637,  4.638,  4.638,  4.639,  4.640, 
 4.641,  4.642,  4.642,  4.643,  4.643,  4.644,  4.645,  4.646,  4.646,  4.647, 
 4.648,  4.649,  4.649,  4.650,  4.651,  4.651,  4.653,  4.653,  4.654,  4.655, 
 4.656,  4.656,  4.657,  4.658,  4.659,  4.660,  4.660,  4.661,  4.662,  4.663, 
 4.663,  4.664,  4.664,  4.665,  4.665,  4.666,  4.666,  4.667,  4.667,  4.668, 
 4.669,  4.669,  4.670,  4.670,  4.671,  4.672,  4.672,  4.673,  4.674,  4.675, 
 4.675,  4.676,  4.677,  4.678,  4.679,  4.679,  4.681,  4.682,  4.682,  4.683, 
 4.684,  4.686,  4.686,  4.687,  4.688,  4.689,  4.690,  4.691,  4.692,  4.693, 
 4.694,  4.695,  4.696,  4.697,  4.698,  4.699,  4.700,  4.701,  4.702,  4.703, 
 4.704,  4.705,  4.706,  4.707,  4.708,  4.709,  4.710,  4.711,  4.712,  4.712, 
 4.713,  4.714,  4.715,  4.716,  4.717,  4.717,  4.718,  4.720,  4.720,  4.721, 
 4.722,  4.723,  4.723,  4.725,  4.726,  4.726,  4.727,  4.728,  4.729,  4.730, 
 4.730,  4.731,  4.732,  4.732,  4.733,  4.733,  4.734,  4.735,  4.735,  4.736, 
 4.737,  4.737,  4.737,  4.738,  4.739,  4.740,  4.740,  4.741,  4.742,  4.743, 
 4.743,  4.744,  4.745,  4.746,  4.746,  4.748,  4.749,  4.749,  4.750,  4.751, 
 4.752,  4.753,  4.754,  4.755,  4.755,  4.756,  4.757,  4.758,  4.759,  4.760, 
 4.761,  4.762,  4.763,  4.764,  4.764,  4.765,  4.766,  4.767,  4.768,  4.769, 
 4.769,  4.770,  4.771,  4.772,  4.773,  4.774,  4.774,  4.775,  4.776,  4.777, 
 4.778,  4.779,  4.780,  4.781,  4.781,  4.783,  4.784,  4.785,  4.785,  4.786, 
 4.788,  4.789,  4.789,  4.790,  4.791,  4.793,  4.793,  4.794,  4.795,  4.796, 
 4.797,  4.798,  4.799,  4.800,  4.801,  4.801,  4.802,  4.803,  4.804,  4.805, 
 4.806,  4.807,  4.808,  4.809,  4.809,  4.810,  4.811,  4.812,  4.812,  4.813, 
 4.814,  4.815,  4.816,  4.817,  4.818,  4.819,  4.820,  4.821,  4.822,  4.823, 
 4.824,  4.825,  4.826,  4.826,  4.828,  4.829,  4.830,  4.830,  4.831,  4.833, 
 4.833,  4.834,  4.835,  4.835,  4.836,  4.836,  4.837,  4.838,  4.839,  4.839, 
 4.840,  4.841,  4.842,  4.842,  4.843,  4.844,  4.845,  4.846,  4.847,  4.847, 
 4.848,  4.849,  4.850,  4.852,  4.853,  4.853,  4.854,  4.855,  4.857,  4.858, 
 4.859,  4.860,  4.861,  4.862,  4.863,  4.864,  4.865,  4.866,  4.868,  4.869, 
 4.869,  4.871,  4.873,  4.874,  4.876,  4.877,  4.879,  4.880,  4.881,  4.882, 
 4.884,  4.885,  4.886,  4.887,  4.888,  4.889,  4.890,  4.891,  4.892,  4.893, 
 4.894,  4.895,  4.895,  4.896,  4.897,  4.898,  4.899,  4.899,  4.900,  4.901, 
 4.901,  4.902,  4.903,  4.904,  4.904,  4.905,  4.906,  4.906,  4.907,  4.907, 
 4.908,  4.909,  4.909,  4.909,  4.910,  4.910,  4.911,  4.911,  4.911,  4.912, 
 4.912,  4.914,  4.914,  4.915,  4.915,  4.916,  4.917,  4.917,  4.919,  4.919, 
 4.920,  4.921,  4.922,  4.922,  4.924,  4.925,  4.926,  4.926,  4.927,  4.929, 
 4.930,  4.931,  4.932,  4.933,  4.934,  4.935,  4.937,  4.938,  4.939,  4.940, 
 4.941,  4.943,  4.944,  4.946,  4.947,  4.948,  4.950,  4.951,  4.952,  4.954, 
 4.955,  4.957,  4.958,  4.960,  4.961,  4.962,  4.964,  4.966,  4.967,  4.968, 
 4.970,  4.972,  4.973,  4.975,  4.976,  4.978,  4.980,  4.981,  4.983,  4.984, 
 4.986,  4.987,  4.989,  4.991,  4.992,  4.994,  4.996,  4.997,  4.999,  5.001, 
 5.002,  5.004,  5.006,  5.008,  5.009,  5.011,  5.013,  5.015,  5.017,  5.018, 
 5.020,  5.022,  5.024,  5.025,  5.027,  5.029,  5.031,  5.033,  5.034,  5.036, 
 5.038,  5.040,  5.042,  5.043,  5.045,  5.047,  5.049,  5.050,  5.053,  5.054, 
 5.056,  5.058,  5.060,  5.061,  5.063,  5.065,  5.067,  5.069,  5.071,  5.073, 
 5.074,  5.076,  5.078,  5.080,  5.082,  5.084,  5.085,  5.088,  5.089,  5.091, 
 5.093,  5.095,  5.097,  5.099,  5.101,  5.102,  5.105,  5.106,  5.109,  5.110, 
 5.112,  5.114,  5.116,  5.118,  5.120,  5.122,  5.124,  5.126,  5.128,  5.130, 
 5.132,  5.134,  5.136,  5.138,  5.140,  5.142,  5.144,  5.146,  5.148,  5.150, 
 5.152,  5.154,  5.156,  5.158,  5.160,  5.162,  5.164,  5.166,  5.168,  5.170, 
 5.172,  5.174,  5.176,  5.178,  5.181,  5.182,  5.184,  5.187,  5.189,  5.191, 
 5.193,  5.195,  5.197,  5.199,  5.201,  5.203,  5.206,  5.208,  5.210,  5.212, 
 5.214,  5.216,  5.218,  5.220,  5.222,  5.224,  5.227,  5.229,  5.231,  5.233, 
 5.235,  5.238,  5.240,  5.242,  5.244,  5.247,  5.249,  5.251,  5.253,  5.255, 
 5.258,  5.259,  5.261,  5.264,  5.266,  5.268,  5.270,  5.272,  5.274,  5.276, 
 5.279,  5.280,  5.283,  5.285,  5.286,  5.289,  5.291,  5.293,  5.295,  5.296, 
 5.299,  5.301,  5.302,  5.305,  5.306,  5.309,  5.310,  5.312,  5.314,  5.316, 
 5.318,  5.320,  5.322,  5.324,  5.325,  5.327,  5.329,  5.331,  5.332,  5.334, 
 5.336,  5.338,  5.340,  5.342,  5.343,  5.346,  5.347,  5.350,  5.351,  5.353, 
 5.355,  5.357,  5.359,  5.361,  5.363,  5.365,  5.367,  5.370,  5.371,  5.373, 
 5.376,  5.377,  5.380,  5.382,  5.383,  5.386,  5.388,  5.389,  5.392,  5.394, 
 5.396,  5.398,  5.401,  5.403,  5.405,  5.407,  5.409,  5.412,  5.414,  5.416, 
 5.418,  5.420,  5.422,  5.424,  5.427,  5.428,  5.430,  5.432,  5.434,  5.436, 
 5.438,  5.440,  5.442,  5.444,  5.445,  5.448,  5.449,  5.451,  5.453,  5.454, 
 5.457,  5.458,  5.460,  5.461,  5.463,  5.465,  5.466,  5.468,  5.470,  5.471, 
 5.473,  5.475,  5.476,  5.478,  5.479,  5.481,  5.483,  5.484,  5.485,  5.487, 
 5.489,  5.490,  5.492,  5.494,  5.495,  5.497,  5.499,  5.500,  5.502,  5.504, 
 5.505,  5.507,  5.509,  5.510,  5.512,  5.514,  5.515,  5.517,  5.519,  5.520, 
 5.522,  5.524,  5.526,  5.527,  5.529,  5.531,  5.533,  5.535,  5.536,  5.538, 
 5.540,  5.542,  5.543,  5.545,  5.547,  5.548,  5.550,  5.552,  5.554,  5.556, 
 5.557,  5.560,  5.561,  5.563,  5.565,  5.567,  5.568,  5.570,  5.572,  5.574, 
 5.576,  5.577,  5.580,  5.581,  5.583,  5.584,  5.587,  5.588,  5.590,  5.592, 
 5.594,  5.596,  5.597,  5.599,  5.601,  5.603,  5.605,  5.607,  5.608,  5.610, 
 5.612,  5.614,  5.616,  5.618,  5.620,  5.622,  5.623,  5.625,  5.627,  5.629, 
 5.631,  5.633,  5.634,  5.636,  5.638,  5.639,  5.641,  5.643,  5.644,  5.646, 
 5.648,  5.649,  5.651,  5.653,  5.654,  5.656,  5.658,  5.659,  5.660,  5.662, 
 5.664,  5.665,  5.666,  5.668,  5.670,  5.671,  5.673,  5.674,  5.675,  5.677, 
 5.679,  5.680,  5.681,  5.682,  5.684,  5.685,  5.686,  5.688,  5.689,  5.690, 
 5.691,  5.693,  5.694,  5.696,  5.697,  5.699,  5.700,  5.701,  5.702,  5.704, 
 5.705,  5.707,  5.708,  5.710,  5.711,  5.712,  5.714,  5.715,  5.717,  5.718, 
 5.720,  5.721,  5.722,  5.724,  5.725,  5.727,  5.728,  5.730,  5.731,  5.732, 
 5.734,  5.735,  5.737,  5.739,  5.740,  5.741,  5.743,  5.745,  5.746,  5.747, 
 5.749,  5.750,  5.752,  5.753,  5.755,  5.756,  5.757,  5.759,  5.760,  5.762, 
 5.763,  5.765,  5.766,  5.767,  5.768,  5.770,  5.771,  5.772,  5.774,  5.775, 
 5.776,  5.778,  5.779,  5.780,  5.782,  5.783,  5.784,  5.786,  5.787,  5.788, 
 5.789,  5.791,  5.792,  5.793,  5.794,  5.795,  5.796,  5.798,  5.799,  5.800, 
 5.801,  5.802,  5.804,  5.805,  5.806,  5.807,  5.808,  5.810,  5.811,  5.812, 
 5.813,  5.815,  5.816,  5.817,  5.819,  5.820,  5.821,  5.822,  5.824,  5.825, 
 5.826,  5.828,  5.829,  5.830,  5.832,  5.833,  5.834,  5.835,  5.837,  5.838, 
 5.839,  5.840,  5.842,  5.843,  5.844,  5.845,  5.847,  5.848,  5.849,  5.850, 
 5.852,  5.853,  5.854,  5.855,  5.857,  5.858,  5.859,  5.861,  5.862,  5.863, 
 5.864,  5.866,  5.867,  5.868,  5.869,  5.871,  5.872,  5.873,  5.874,  5.876, 
 5.877,  5.878,  5.879,  5.880,  5.882,  5.883,  5.884,  5.885,  5.886,  5.888, 
 5.889,  5.890,  5.891,  5.893,  5.894,  5.895,  5.896,  5.898,  5.899,  5.900, 
 5.901,  5.902,  5.904,  5.905,  5.906,  5.907,  5.909,  5.910,  5.911,  5.912, 
 5.913,  5.914,  5.915,  5.916,  5.917,  5.918,  5.919,  5.920,  5.921,  5.922, 
 5.923,  5.924,  5.925,  5.926,  5.927,  5.929,  5.929,  5.930,  5.931,  5.932, 
 5.933,  5.934,  5.935,  5.936,  5.936,  5.937,  5.938,  5.939,  5.940,  5.941, 
 5.941,  5.942,  5.943,  5.944,  5.945,  5.946,  5.947,  5.948,  5.949,  5.950, 
 5.951,  5.952,  5.953,  5.954,  5.955,  5.956,  5.957,  5.958,  5.960,  5.960, 
 5.961,  5.962,  5.963,  5.965,  5.966,  5.967,  5.968,  5.969,  5.970,  5.971, 
 5.972,  5.973,  5.975,  5.976,  5.977,  5.978,  5.980,  5.981,  5.982,  5.983, 
 5.984,  5.985,  5.986,  5.987,  5.988,  5.989,  5.990,  5.991,  5.992,  5.993, 
 5.994,  5.995,  5.996,  5.997,  5.998,  5.998,  5.999,  6.000,  6.001,  6.002, 
 6.003,  6.004,  6.004,  6.006,  6.006,  6.007,  6.008,  6.008,  6.009,  6.010, 
 6.011,  6.012,  6.012,  6.013,  6.013,  6.014,  6.015,  6.016,  6.016,  6.017, 
 6.018,  6.018,  6.019,  6.020,  6.020,  6.021,  6.022,  6.023,  6.023,  6.024, 
 6.024,  6.025,  6.026,  6.027,  6.027,  6.028,  6.029,  6.029,  6.030,  6.031, 
 6.032,  6.032,  6.033,  6.034,  6.034,  6.035,  6.035,  6.037,  6.037,  6.038, 
 6.038,  6.039,  6.040,  6.041,  6.042,  6.042,  6.043,  6.044,  6.044,  6.045, 
 6.046,  6.047,  6.047,  6.048,  6.049,  6.049,  6.050,  6.051,  6.052,  6.052, 
 6.053,  6.054,  6.054,  6.055,  6.055,  6.056,  6.057,  6.058,  6.058,  6.059, 
 6.059,  6.060,  6.061,  6.061,  6.062,  6.063,  6.064,  6.064,  6.065,  6.065, 
 6.066,  6.066,  6.067,  6.068,  6.068,  6.069,  6.069,  6.070,  6.070,  6.071, 
 6.071,  6.073,  6.073,  6.074,  6.074,  6.075,  6.076,  6.076,  6.077,  6.078, 
 6.079,  6.079,  6.080,  6.081,  6.081,  6.082,  6.083,  6.084,  6.085,  6.085, 
 6.086,  6.087,  6.088,  6.089,  6.089,  6.090,  6.091,  6.091,  6.093,  6.093, 
 6.094,  6.095,  6.096,  6.097,  6.098,  6.099,  6.100,  6.100,  6.101,  6.102, 
 6.103,  6.104,  6.105,  6.105,  6.106,  6.107,  6.107,  6.108,  6.109,  6.110, 
 6.110,  6.111,  6.111,  6.112,  6.113,  6.114,  6.114,  6.115,  6.115,  6.116, 
 6.116,  6.117,  6.117,  6.118,  6.119,  6.119,  6.120,  6.120,  6.121,  6.121, 
 6.121,  6.122,  6.122,  6.123,  6.124,  6.124,  6.124,  6.125,  6.125,  6.126, 
 6.126,  6.126,  6.127,  6.127,  6.127,  6.128,  6.129,  6.129,  6.129,  6.130, 
 6.130,  6.131,  6.131,  6.131,  6.132,  6.132,  6.132,  6.133,  6.134,  6.134, 
 6.134,  6.135,  6.135,  6.135,  6.136,  6.136,  6.137,  6.137,  6.137,  6.138, 
 6.138,  6.139,  6.139,  6.139,  6.140,  6.140,  6.140,  6.141,  6.141,  6.141, 
 6.142,  6.142,  6.142,  6.143,  6.143,  6.143,  6.144,  6.145,  6.145,  6.145, 
 6.146,  6.146,  6.147,  6.147,  6.147,  6.148,  6.148,  6.149,  6.150,  6.150, 
 6.150,  6.151,  6.151,  6.152,  6.152,  6.152,  6.153,  6.153,  6.154,  6.155, 
 6.155,  6.156,  6.156,  6.156,  6.157,  6.157,  6.158,  6.158,  6.159,  6.160, 
 6.160,  6.161,  6.161,  6.161,  6.162,  6.162,  6.163,  6.163,  6.163,  6.164, 
 6.165,  6.165,  6.165,  6.166,  6.166,  6.166,  6.167,  6.167,  6.167,  6.168, 
 6.168,  6.168,  6.169,  6.169,  6.170,  6.170,  6.170,  6.171,  6.171,  6.171, 
 6.171,  6.172,  6.172,  6.172,  6.172,  6.173,  6.173,  6.173,  6.173,  6.174, 
 6.174,  6.175,  6.175,  6.175,  6.175,  6.176,  6.176,  6.176,  6.176,  6.177, 
 6.177,  6.177,  6.177,  6.178,  6.178,  6.178,  6.178,  6.179,  6.179,  6.180, 
 6.180,  6.180,  6.180,  6.181,  6.181,  6.181,  6.181,  6.182,  6.182,  6.182, 
 6.182,  6.183,  6.183,  6.183,  6.184,  6.184,  6.184,  6.184,  6.185,  6.185, 
 6.186,  6.186,  6.186,  6.186,  6.187,  6.187,  6.187,  6.187,  6.188,  6.188, 
 6.188,  6.188,  6.189,  6.189,  6.189,  6.189,  6.190,  6.190,  6.191,  6.191, 
 6.191,  6.191,  6.192,  6.192,  6.192,  6.192,  6.193,  6.193,  6.193,  6.193, 
 6.193,  6.194,  6.194,  6.194,  6.194,  6.194,  6.195,  6.195,  6.195,  6.196, 
 6.196,  6.196,  6.196,  6.197,  6.197,  6.197,  6.197,  6.198,  6.198,  6.198, 
 6.198,  6.199,  6.199,  6.199,  6.199,  6.200,  6.200,  6.201,  6.201,  6.201, 
 6.202,  6.202,  6.202,  6.202,  6.203,  6.203,  6.203,  6.204,  6.204,  6.204, 
])
_dep_jma_2020c = np.array([
  0.0,   0.5,   1.0,   1.5,   2.0,   2.5,   3.0,   3.5,   4.0,   4.5, 
  5.0,   5.5,   6.0,   6.5,   7.0,   7.5,   8.0,   8.5,   9.0,   9.5, 
 10.0,  10.5,  11.0,  11.5,  12.0,  12.5,  13.0,  13.5,  14.0,  14.5, 
 15.0,  15.5,  16.0,  16.5,  17.0,  17.5,  18.0,  18.5,  19.0,  19.5, 
 20.0,  20.5,  21.0,  21.5,  22.0,  22.5,  23.0,  23.5,  24.0,  24.5, 
 25.0,  25.5,  26.0,  26.5,  27.0,  27.5,  28.0,  28.5,  29.0,  29.5, 
 30.0,  30.5,  31.0,  31.5,  32.0,  32.5,  33.0,  33.5,  34.0,  34.5, 
 35.0,  35.5,  36.0,  36.5,  37.0,  37.5,  38.0,  38.5,  39.0,  39.5, 
 40.0,  40.5,  41.0,  41.5,  42.0,  42.5,  43.0,  43.5,  44.0,  44.5, 
 45.0,  45.5,  46.0,  46.5,  47.0,  47.5,  48.0,  48.5,  49.0,  49.5, 
 50.0,  50.5,  51.0,  51.5,  52.0,  52.5,  53.0,  53.5,  54.0,  54.5, 
 55.0,  55.5,  56.0,  56.5,  57.0,  57.5,  58.0,  58.5,  59.0,  59.5, 
 60.0,  60.5,  61.0,  61.5,  62.0,  62.5,  63.0,  63.5,  64.0,  64.5, 
 65.0,  65.5,  66.0,  66.5,  67.0,  67.5,  68.0,  68.5,  69.0,  69.5, 
 70.0,  70.5,  71.0,  71.5,  72.0,  72.5,  73.0,  73.5,  74.0,  74.5, 
 75.0,  75.5,  76.0,  76.5,  77.0,  77.5,  78.0,  78.5,  79.0,  79.5, 
 80.0,  80.5,  81.0,  81.5,  82.0,  82.5,  83.0,  83.5,  84.0,  84.5, 
 85.0,  85.5,  86.0,  86.5,  87.0,  87.5,  88.0,  88.5,  89.0,  89.5, 
 90.0,  90.5,  91.0,  91.5,  92.0,  92.5,  93.0,  93.5,  94.0,  94.5, 
 95.0,  95.5,  96.0,  96.5,  97.0,  97.5,  98.0,  98.5,  99.0,  99.5, 
100.0, 100.5, 101.0, 101.5, 102.0, 102.5, 103.0, 103.5, 104.0, 104.5, 
105.0, 105.5, 106.0, 106.5, 107.0, 107.5, 108.0, 108.5, 109.0, 109.5, 
110.0, 110.5, 111.0, 111.5, 112.0, 112.5, 113.0, 113.5, 114.0, 114.5, 
115.0, 115.5, 116.0, 116.5, 117.0, 117.5, 118.0, 118.5, 119.0, 119.5, 
120.0, 120.5, 121.0, 121.5, 122.0, 122.5, 123.0, 123.5, 124.0, 124.5, 
125.0, 125.5, 126.0, 126.5, 127.0, 127.5, 128.0, 128.5, 129.0, 129.5, 
130.0, 130.5, 131.0, 131.5, 132.0, 132.5, 133.0, 133.5, 134.0, 134.5, 
135.0, 135.5, 136.0, 136.5, 137.0, 137.5, 138.0, 138.5, 139.0, 139.5, 
140.0, 140.5, 141.0, 141.5, 142.0, 142.5, 143.0, 143.5, 144.0, 144.5, 
145.0, 145.5, 146.0, 146.5, 147.0, 147.5, 148.0, 148.5, 149.0, 149.5, 
150.0, 150.5, 151.0, 151.5, 152.0, 152.5, 153.0, 153.5, 154.0, 154.5, 
155.0, 155.5, 156.0, 156.5, 157.0, 157.5, 158.0, 158.5, 159.0, 159.5, 
160.0, 160.5, 161.0, 161.5, 162.0, 162.5, 163.0, 163.5, 164.0, 164.5, 
165.0, 165.5, 166.0, 166.5, 167.0, 167.5, 168.0, 168.5, 169.0, 169.5, 
170.0, 170.5, 171.0, 171.5, 172.0, 172.5, 173.0, 173.5, 174.0, 174.5, 
175.0, 175.5, 176.0, 176.5, 177.0, 177.5, 178.0, 178.5, 179.0, 179.5, 
180.0, 180.5, 181.0, 181.5, 182.0, 182.5, 183.0, 183.5, 184.0, 184.5, 
185.0, 185.5, 186.0, 186.5, 187.0, 187.5, 188.0, 188.5, 189.0, 189.5, 
190.0, 190.5, 191.0, 191.5, 192.0, 192.5, 193.0, 193.5, 194.0, 194.5, 
195.0, 195.5, 196.0, 196.5, 197.0, 197.5, 198.0, 198.5, 199.0, 199.5, 
200.0, 200.5, 201.0, 201.5, 202.0, 202.5, 203.0, 203.5, 204.0, 204.5, 
205.0, 205.5, 206.0, 206.5, 207.0, 207.5, 208.0, 208.5, 209.0, 209.5, 
210.0, 210.5, 211.0, 211.5, 212.0, 212.5, 213.0, 213.5, 214.0, 214.5, 
215.0, 215.5, 216.0, 216.5, 217.0, 217.5, 218.0, 218.5, 219.0, 219.5, 
220.0, 220.5, 221.0, 221.5, 222.0, 222.5, 223.0, 223.5, 224.0, 224.5, 
225.0, 225.5, 226.0, 226.5, 227.0, 227.5, 228.0, 228.5, 229.0, 229.5, 
230.0, 230.5, 231.0, 231.5, 232.0, 232.5, 233.0, 233.5, 234.0, 234.5, 
235.0, 235.5, 236.0, 236.5, 237.0, 237.5, 238.0, 238.5, 239.0, 239.5, 
240.0, 240.5, 241.0, 241.5, 242.0, 242.5, 243.0, 243.5, 244.0, 244.5, 
245.0, 245.5, 246.0, 246.5, 247.0, 247.5, 248.0, 248.5, 249.0, 249.5, 
250.0, 250.5, 251.0, 251.5, 252.0, 252.5, 253.0, 253.5, 254.0, 254.5, 
255.0, 255.5, 256.0, 256.5, 257.0, 257.5, 258.0, 258.5, 259.0, 259.5, 
260.0, 260.5, 261.0, 261.5, 262.0, 262.5, 263.0, 263.5, 264.0, 264.5, 
265.0, 265.5, 266.0, 266.5, 267.0, 267.5, 268.0, 268.5, 269.0, 269.5, 
270.0, 270.5, 271.0, 271.5, 272.0, 272.5, 273.0, 273.5, 274.0, 274.5, 
275.0, 275.5, 276.0, 276.5, 277.0, 277.5, 278.0, 278.5, 279.0, 279.5, 
280.0, 280.5, 281.0, 281.5, 282.0, 282.5, 283.0, 283.5, 284.0, 284.5, 
285.0, 285.5, 286.0, 286.5, 287.0, 287.5, 288.0, 288.5, 289.0, 289.5, 
290.0, 290.5, 291.0, 291.5, 292.0, 292.5, 293.0, 293.5, 294.0, 294.5, 
295.0, 295.5, 296.0, 296.5, 297.0, 297.5, 298.0, 298.5, 299.0, 299.5, 
300.0, 300.5, 301.0, 301.5, 302.0, 302.5, 303.0, 303.5, 304.0, 304.5, 
305.0, 305.5, 306.0, 306.5, 307.0, 307.5, 308.0, 308.5, 309.0, 309.5, 
310.0, 310.5, 311.0, 311.5, 312.0, 312.5, 313.0, 313.5, 314.0, 314.5, 
315.0, 315.5, 316.0, 316.5, 317.0, 317.5, 318.0, 318.5, 319.0, 319.5, 
320.0, 320.5, 321.0, 321.5, 322.0, 322.5, 323.0, 323.5, 324.0, 324.5, 
325.0, 325.5, 326.0, 326.5, 327.0, 327.5, 328.0, 328.5, 329.0, 329.5, 
330.0, 330.5, 331.0, 331.5, 332.0, 332.5, 333.0, 333.5, 334.0, 334.5, 
335.0, 335.5, 336.0, 336.5, 337.0, 337.5, 338.0, 338.5, 339.0, 339.5, 
340.0, 340.5, 341.0, 341.5, 342.0, 342.5, 343.0, 343.5, 344.0, 344.5, 
345.0, 345.5, 346.0, 346.5, 347.0, 347.5, 348.0, 348.5, 349.0, 349.5, 
350.0, 350.5, 351.0, 351.5, 352.0, 352.5, 353.0, 353.5, 354.0, 354.5, 
355.0, 355.5, 356.0, 356.5, 357.0, 357.5, 358.0, 358.5, 359.0, 359.5, 
360.0, 360.5, 361.0, 361.5, 362.0, 362.5, 363.0, 363.5, 364.0, 364.5, 
365.0, 365.5, 366.0, 366.5, 367.0, 367.5, 368.0, 368.5, 369.0, 369.5, 
370.0, 370.5, 371.0, 371.5, 372.0, 372.5, 373.0, 373.5, 374.0, 374.5, 
375.0, 375.5, 376.0, 376.5, 377.0, 377.5, 378.0, 378.5, 379.0, 379.5, 
380.0, 380.5, 381.0, 381.5, 382.0, 382.5, 383.0, 383.5, 384.0, 384.5, 
385.0, 385.5, 386.0, 386.5, 387.0, 387.5, 388.0, 388.5, 389.0, 389.5, 
390.0, 390.5, 391.0, 391.5, 392.0, 392.5, 393.0, 393.5, 394.0, 394.5, 
395.0, 395.5, 396.0, 396.5, 397.0, 397.5, 398.0, 398.5, 399.0, 399.5, 
400.0, 400.5, 401.0, 401.5, 402.0, 402.5, 403.0, 403.5, 404.0, 404.5, 
405.0, 405.5, 406.0, 406.5, 407.0, 407.5, 408.0, 408.5, 409.0, 409.5, 
410.0, 410.5, 411.0, 411.5, 412.0, 412.5, 413.0, 413.5, 414.0, 414.5, 
415.0, 415.5, 416.0, 416.5, 417.0, 417.5, 418.0, 418.5, 419.0, 419.5, 
420.0, 420.5, 421.0, 421.5, 422.0, 422.5, 423.0, 423.5, 424.0, 424.5, 
425.0, 425.5, 426.0, 426.5, 427.0, 427.5, 428.0, 428.5, 429.0, 429.5, 
430.0, 430.5, 431.0, 431.5, 432.0, 432.5, 433.0, 433.5, 434.0, 434.5, 
435.0, 435.5, 436.0, 436.5, 437.0, 437.5, 438.0, 438.5, 439.0, 439.5, 
440.0, 440.5, 441.0, 441.5, 442.0, 442.5, 443.0, 443.5, 444.0, 444.5, 
445.0, 445.5, 446.0, 446.5, 447.0, 447.5, 448.0, 448.5, 449.0, 449.5, 
450.0, 450.5, 451.0, 451.5, 452.0, 452.5, 453.0, 453.5, 454.0, 454.5, 
455.0, 455.5, 456.0, 456.5, 457.0, 457.5, 458.0, 458.5, 459.0, 459.5, 
460.0, 460.5, 461.0, 461.5, 462.0, 462.5, 463.0, 463.5, 464.0, 464.5, 
465.0, 465.5, 466.0, 466.5, 467.0, 467.5, 468.0, 468.5, 469.0, 469.5, 
470.0, 470.5, 471.0, 471.5, 472.0, 472.5, 473.0, 473.5, 474.0, 474.5, 
475.0, 475.5, 476.0, 476.5, 477.0, 477.5, 478.0, 478.5, 479.0, 479.5, 
480.0, 480.5, 481.0, 481.5, 482.0, 482.5, 483.0, 483.5, 484.0, 484.5, 
485.0, 485.5, 486.0, 486.5, 487.0, 487.5, 488.0, 488.5, 489.0, 489.5, 
490.0, 490.5, 491.0, 491.5, 492.0, 492.5, 493.0, 493.5, 494.0, 494.5, 
495.0, 495.5, 496.0, 496.5, 497.0, 497.5, 498.0, 498.5, 499.0, 499.5, 
500.0, 500.5, 501.0, 501.5, 502.0, 502.5, 503.0, 503.5, 504.0, 504.5, 
505.0, 505.5, 506.0, 506.5, 507.0, 507.5, 508.0, 508.5, 509.0, 509.5, 
510.0, 510.5, 511.0, 511.5, 512.0, 512.5, 513.0, 513.5, 514.0, 514.5, 
515.0, 515.5, 516.0, 516.5, 517.0, 517.5, 518.0, 518.5, 519.0, 519.5, 
520.0, 520.5, 521.0, 521.5, 522.0, 522.5, 523.0, 523.5, 524.0, 524.5, 
525.0, 525.5, 526.0, 526.5, 527.0, 527.5, 528.0, 528.5, 529.0, 529.5, 
530.0, 530.5, 531.0, 531.5, 532.0, 532.5, 533.0, 533.5, 534.0, 534.5, 
535.0, 535.5, 536.0, 536.5, 537.0, 537.5, 538.0, 538.5, 539.0, 539.5, 
540.0, 540.5, 541.0, 541.5, 542.0, 542.5, 543.0, 543.5, 544.0, 544.5, 
545.0, 545.5, 546.0, 546.5, 547.0, 547.5, 548.0, 548.5, 549.0, 549.5, 
550.0, 550.5, 551.0, 551.5, 552.0, 552.5, 553.0, 553.5, 554.0, 554.5, 
555.0, 555.5, 556.0, 556.5, 557.0, 557.5, 558.0, 558.5, 559.0, 559.5, 
560.0, 560.5, 561.0, 561.5, 562.0, 562.5, 563.0, 563.5, 564.0, 564.5, 
565.0, 565.5, 566.0, 566.5, 567.0, 567.5, 568.0, 568.5, 569.0, 569.5, 
570.0, 570.5, 571.0, 571.5, 572.0, 572.5, 573.0, 573.5, 574.0, 574.5, 
575.0, 575.5, 576.0, 576.5, 577.0, 577.5, 578.0, 578.5, 579.0, 579.5, 
580.0, 580.5, 581.0, 581.5, 582.0, 582.5, 583.0, 583.5, 584.0, 584.5, 
585.0, 585.5, 586.0, 586.5, 587.0, 587.5, 588.0, 588.5, 589.0, 589.5, 
590.0, 590.5, 591.0, 591.5, 592.0, 592.5, 593.0, 593.5, 594.0, 594.5, 
595.0, 595.5, 596.0, 596.5, 597.0, 597.5, 598.0, 598.5, 599.0, 599.5, 
600.0, 600.5, 601.0, 601.5, 602.0, 602.5, 603.0, 603.5, 604.0, 604.5, 
605.0, 605.5, 606.0, 606.5, 607.0, 607.5, 608.0, 608.5, 609.0, 609.5, 
610.0, 610.5, 611.0, 611.5, 612.0, 612.5, 613.0, 613.5, 614.0, 614.5, 
615.0, 615.5, 616.0, 616.5, 617.0, 617.5, 618.0, 618.5, 619.0, 619.5, 
620.0, 620.5, 621.0, 621.5, 622.0, 622.5, 623.0, 623.5, 624.0, 624.5, 
625.0, 625.5, 626.0, 626.5, 627.0, 627.5, 628.0, 628.5, 629.0, 629.5, 
630.0, 630.5, 631.0, 631.5, 632.0, 632.5, 633.0, 633.5, 634.0, 634.5, 
635.0, 635.5, 636.0, 636.5, 637.0, 637.5, 638.0, 638.5, 639.0, 639.5, 
640.0, 640.5, 641.0, 641.5, 642.0, 642.5, 643.0, 643.5, 644.0, 644.5, 
645.0, 645.5, 646.0, 646.5, 647.0, 647.5, 648.0, 648.5, 649.0, 649.5, 
650.0, 650.5, 651.0, 651.5, 652.0, 652.5, 653.0, 653.5, 654.0, 654.5, 
655.0, 655.5, 656.0, 656.5, 657.0, 657.5, 658.0, 658.5, 659.0, 659.5, 
660.0, 660.5, 661.0, 661.5, 662.0, 662.5, 663.0, 663.5, 664.0, 664.5, 
665.0, 665.5, 666.0, 666.5, 667.0, 667.5, 668.0, 668.5, 669.0, 669.5, 
670.0, 670.5, 671.0, 671.5, 672.0, 672.5, 673.0, 673.5, 674.0, 674.5, 
675.0, 675.5, 676.0, 676.5, 677.0, 677.5, 678.0, 678.5, 679.0, 679.5, 
680.0, 680.5, 681.0, 681.5, 682.0, 682.5, 683.0, 683.5, 684.0, 684.5, 
685.0, 685.5, 686.0, 686.5, 687.0, 687.5, 688.0, 688.5, 689.0, 689.5, 
690.0, 690.5, 691.0, 691.5, 692.0, 692.5, 693.0, 693.5, 694.0, 694.5, 
695.0, 695.5, 696.0, 696.5, 697.0, 697.5, 698.0, 698.5, 699.0, 699.5, 
700.0, 700.5, 701.0, 701.5, 702.0, 702.5, 703.0, 703.5, 704.0, 704.5, 
705.0, 705.5, 706.0, 706.5, 707.0, 707.5, 708.0, 708.5, 709.0, 709.5, 
710.0, 710.5, 711.0, 711.5, 712.0, 712.5, 713.0, 713.5, 714.0, 714.5, 
715.0, 715.5, 716.0, 716.5, 717.0, 717.5, 718.0, 718.5, 719.0, 719.5, 
720.0, 720.5, 721.0, 721.5, 722.0, 722.5, 723.0, 723.5, 724.0, 724.5, 
725.0, 725.5, 726.0, 726.5, 727.0, 727.5, 728.0, 728.5, 729.0, 729.5, 
730.0, 730.5, 731.0, 731.5, 732.0, 732.5, 733.0, 733.5, 734.0, 734.5, 
735.0, 735.5, 736.0, 736.5, 737.0, 737.5, 738.0, 738.5, 739.0, 739.5, 
740.0, 740.5, 741.0, 741.5, 742.0, 742.5, 743.0, 743.5, 744.0, 744.5, 
745.0, 745.5, 746.0, 746.5, 747.0, 747.5, 748.0, 748.5, 749.0, 749.5, 
750.0, 750.5, 751.0, 751.5, 752.0, 752.5, 753.0, 753.5, 754.0, 754.5, 
755.0, 755.5, 756.0, 756.5, 757.0, 757.5, 758.0, 758.5, 759.0, 759.5, 
760.0, 760.5, 761.0, 761.5, 762.0, 762.5, 763.0, 763.5, 764.0, 764.5, 
765.0, 765.5, 766.0, 766.5, 767.0, 767.5, 768.0, 768.5, 769.0, 769.5, 
770.0, 770.5, 771.0, 771.5, 772.0, 772.5, 773.0, 773.5, 774.0, 774.5, 
775.0, 775.5, 776.0, 776.5, 777.0, 777.5, 778.0, 778.5, 779.0, 779.5, 
780.0, 780.5, 781.0, 781.5, 782.0, 782.5, 783.0, 783.5, 784.0, 784.5, 
785.0, 785.5, 786.0, 786.5, 787.0, 787.5, 788.0, 788.5, 789.0, 789.5, 
790.0, 790.5, 791.0, 791.5, 792.0, 792.5, 793.0, 793.5, 794.0, 794.5, 
795.0, 795.5, 796.0, 796.5, 797.0, 797.5, 798.0, 798.5, 799.0, 799.5, 
800.0, 800.5, 801.0, 801.5, 802.0, 802.5, 803.0, 803.5, 804.0, 804.5, 
805.0, 805.5, 806.0, 806.5, 807.0, 807.5, 808.0, 808.5, 809.0, 809.5, 
810.0, 810.5, 811.0, 811.5, 812.0, 812.5, 813.0, 813.5, 814.0, 814.5, 
815.0, 815.5, 816.0, 816.5, 817.0, 817.5, 818.0, 818.5, 819.0, 819.5, 
820.0, 820.5, 821.0, 821.5, 822.0, 822.5, 823.0, 823.5, 824.0, 824.5, 
825.0, 825.5, 826.0, 826.5, 827.0, 827.5, 828.0, 828.5, 829.0, 829.5, 
830.0, 830.5, 831.0, 831.5, 832.0, 832.5, 833.0, 833.5, 834.0, 834.5, 
835.0, 835.5, 836.0, 836.5, 837.0, 837.5, 838.0, 838.5, 839.0, 839.5, 
840.0, 840.5, 841.0, 841.5, 842.0, 842.5, 843.0, 843.5, 844.0, 844.5, 
845.0, 845.5, 846.0, 846.5, 847.0, 847.5, 848.0, 848.5, 849.0, 849.5, 
850.0, 850.5, 851.0, 851.5, 852.0, 852.5, 853.0, 853.5, 854.0, 854.5, 
855.0, 855.5, 856.0, 856.5, 857.0, 857.5, 858.0, 858.5, 859.0, 859.5, 
860.0, 860.5, 861.0, 861.5, 862.0, 862.5, 863.0, 863.5, 864.0, 864.5, 
865.0, 865.5, 866.0, 866.5, 867.0, 867.5, 868.0, 868.5, 869.0, 869.5, 
870.0, 870.5, 871.0, 871.5, 872.0, 872.5, 873.0, 873.5, 874.0, 874.5, 
875.0, 875.5, 876.0, 876.5, 877.0, 877.5, 878.0, 878.5, 879.0, 879.5, 
880.0, 880.5, 881.0, 881.5, 882.0, 882.5, 883.0, 883.5, 884.0, 884.5, 
885.0, 885.5, 886.0, 886.5, 887.0, 887.5, 888.0, 888.5, 889.0, 889.5, 
890.0, 890.5, 891.0, 891.5, 892.0, 892.5, 893.0, 893.5, 894.0, 894.5, 
895.0, 895.5, 896.0, 896.5, 897.0, 897.5, 898.0, 898.5, 899.0, 899.5, 
900.0, 900.5, 901.0, 901.5, 902.0, 902.5, 903.0, 903.5, 904.0, 904.5, 
905.0, 905.5, 906.0, 906.5, 907.0, 907.5, 908.0, 908.5, 909.0, 909.5, 
910.0, 910.5, 911.0, 911.5, 912.0, 912.5, 913.0, 913.5, 914.0, 914.5, 
915.0, 915.5, 916.0, 916.5, 917.0, 917.5, 918.0, 918.5, 919.0, 919.5, 
920.0, 920.5, 921.0, 921.5, 922.0, 922.5, 923.0, 923.5, 924.0, 924.5, 
925.0, 925.5, 926.0, 926.5, 927.0, 927.5, 928.0, 928.5, 929.0, 929.5, 
930.0, 930.5, 931.0, 931.5, 932.0, 932.5, 933.0, 933.5, 934.0, 934.5, 
935.0, 935.5, 936.0, 936.5, 937.0, 937.5, 938.0, 938.5, 939.0, 939.5, 
])
_f_vp_jma_2020c = scipy.interpolate.interp1d(_dep_jma_2020c, _vp_jma_2020c, kind='linear')
_f_vs_jma_2020c = scipy.interpolate.interp1d(_dep_jma_2020c, _vs_jma_2020c, kind='linear')


### AK135
_vp_ak135 = np.array([
      1.4500,   1.4500,   1.6500,   1.6500,   5.8000,   5.8000,   6.8000,   6.8000,   8.0355,   8.0379, 
      8.0400,   8.0450,   8.0505,   8.0505,   8.1750,   8.3007,   8.3007,   8.4822,   8.6650,   8.8476, 
      9.0302,   9.3601,   9.5280,   9.6962,   9.8640,  10.0320,  10.2000,  10.7909,  10.9222,  11.0553, 
     11.1355,  11.2228,  11.3068,  11.3897,  11.4704,  11.5493,  11.6265,  11.7020,  11.7768,  11.8491, 
     11.9208,  11.9891,  12.0571,  12.1247,  12.1912,  12.2558,  12.3181,  12.3813,  12.4427,  12.5030, 
     12.5638,  12.6226,  12.6807,  12.7384,  12.7956,  12.8524,  12.9093,  12.9663,  13.0226,  13.0786, 
     13.1337,  13.1895,  13.2465,  13.3017,  13.3584,  13.4156,  13.4741,  13.5311,  13.5899,  13.6498, 
     13.6498,  13.6533,  13.6570,  13.6601,   8.0000,   8.0382,   8.1283,   8.2213,   8.3122,   8.4001, 
      8.4861,   8.5692,   8.6496,   8.7283,   8.8036,   8.8761,   8.9461,   9.0138,   9.0792,   9.1426, 
      9.2042,   9.2634,   9.3205,   9.3760,   9.4297,   9.4814,   9.5306,   9.5777,   9.6232,   9.6673, 
      9.7100,   9.7513,   9.7914,   9.8304,   9.8682,   9.9051,   9.9410,   9.9761,  10.0103,  10.0439, 
     10.0768,  10.1095,  10.1415,  10.1739,  10.2049,  10.2329,  10.2565,  10.2745,  10.2854,  10.2890, 
     11.0427,  11.0585,  11.0718,  11.0850,  11.0983,  11.1166,  11.1316,  11.1457,  11.1590,  11.1715, 
     11.1832,  11.1941,  11.2041,  11.2134,  11.2219,  11.2295,  11.2364,  11.2424,  11.2477,  11.2521, 
     11.2586,  11.2606,  11.2618,  11.2622, ])
_vs_ak135 = np.array([
      0.0000,   0.0000,   1.0000,   1.0000,   3.2000,   3.2000,   3.9000,   3.9000,   4.4839,   4.4856, 
      4.4800,   4.4900,   4.5000,   4.5000,   4.5090,   4.5184,   4.5184,   4.6094,   4.6964,   4.7832, 
      4.8702,   5.0806,   5.1864,   5.2922,   5.3989,   5.5047,   5.6104,   5.9607,   6.0898,   6.2100, 
      6.2424,   6.2799,   6.3164,   6.3519,   6.3860,   6.4182,   6.4514,   6.4822,   6.5131,   6.5431, 
      6.5728,   6.6009,   6.6285,   6.6554,   6.6813,   6.7070,   6.7323,   6.7579,   6.7820,   6.8056, 
      6.8289,   6.8517,   6.8743,   6.8972,   6.9194,   6.9416,   6.9625,   6.9852,   7.0069,   7.0286, 
      7.0504,   7.0722,   7.0932,   7.1144,   7.1368,   7.1584,   7.1804,   7.2031,   7.2253,   7.2485, 
      7.2485,   7.2593,   7.2700,   7.2817,   0.0000,   0.0000,   0.0000,   0.0000,   0.0000,   0.0000, 
      0.0000,   0.0000,   0.0000,   0.0000,   0.0000,   0.0000,   0.0000,   0.0000,   0.0000,   0.0000, 
      0.0000,   0.0000,   0.0000,   0.0000,   0.0000,   0.0000,   0.0000,   0.0000,   0.0000,   0.0000, 
      0.0000,   0.0000,   0.0000,   0.0000,   0.0000,   0.0000,   0.0000,   0.0000,   0.0000,   0.0000, 
      0.0000,   0.0000,   0.0000,   0.0000,   0.0000,   0.0000,   0.0000,   0.0000,   0.0000,   0.0000, 
      3.5043,   3.5187,   3.5314,   3.5435,   3.5551,   3.5661,   3.5765,   3.5864,   3.5957,   3.6044, 
      3.6126,   3.6202,   3.6272,   3.6337,   3.6396,   3.6450,   3.6498,   3.6540,   3.6577,   3.6608, 
      3.6653,   3.6667,   3.6675,   3.6678, ])
_rho_ak135 = np.array([
        1.02,     1.02,     2.00,     2.00,     2.60,     2.60,     2.92,     2.92,     3.64,     3.58, 
        3.50,     3.50,     3.43,     3.43,     3.37,     3.32,     3.32,     3.37,     3.41,     3.46, 
        3.51,     3.93,     3.93,     3.92,     3.92,     3.92,     3.92,     4.24,     4.30,     4.36, 
        4.41,     4.46,     4.52,     4.57,     4.59,     4.62,     4.65,     4.67,     4.70,     4.73, 
        4.75,     4.78,     4.80,     4.83,     4.86,     4.88,     4.91,     4.93,     4.96,     4.98, 
        5.01,     5.03,     5.05,     5.08,     5.10,     5.13,     5.15,     5.17,     5.20,     5.22, 
        5.24,     5.26,     5.29,     5.31,     5.33,     5.35,     5.37,     5.40,     5.42,     5.44, 
        5.69,     5.72,     5.75,     5.77,     9.91,     9.99,    10.07,    10.15,    10.22,    10.30, 
       10.37,    10.44,    10.51,    10.57,    10.64,    10.70,    10.76,    10.83,    10.89,    10.94, 
       11.00,    11.06,    11.11,    11.16,    11.21,    11.26,    11.31,    11.36,    11.41,    11.45, 
       11.50,    11.54,    11.58,    11.62,    11.66,    11.70,    11.74,    11.77,    11.81,    11.84, 
       11.88,    11.91,    11.94,    11.97,    12.00,    12.03,    12.06,    12.09,    12.11,    12.14, 
       12.70,    12.73,    12.75,    12.78,    12.80,    12.82,    12.84,    12.86,    12.88,    12.89, 
       12.91,    12.92,    12.94,    12.95,    12.96,    12.97,    12.98,    12.99,    12.99,    13.00, 
       13.01,    13.01,    13.01,    13.01, ])
_qk_ak135 = np.array([
    57822.00, 57822.00,   163.35,   163.35,  1478.30,  1478.30,  1368.02,  1368.02,   950.50,   972.77, 
     1008.71,   182.03,   182.57,   182.57,   188.72,   200.97,   338.47,   346.37,   355.85,   366.34, 
      377.93,   413.66,   417.32,   419.94,   422.55,   425.51,   428.69,  1350.54,  1311.17,  1277.93, 
     1269.44,  1260.68,  1251.69,  1243.02,  1234.54,  1226.52,  1217.91,  1210.02,  1202.04,  1193.99, 
     1186.06,  1178.19,  1170.53,  1163.16,  1156.04,  1148.76,  1141.32,  1134.01,  1127.02,  1120.09, 
     1108.58,  1097.16,  1085.97,  1070.38,  1064.23,  1058.03,  1048.09,  1042.07,  1032.14,  1018.38, 
     1008.79,   999.44,   990.77,   985.63,   976.81,   968.46,   960.36,   952.00,   940.88,   933.21, 
      722.73,   726.87,   725.11,   723.12, 57822.00, 57822.00, 57822.00, 57822.00, 57822.00, 57822.00, 
    57822.00, 57822.00, 57822.00, 57822.00, 57822.00, 57822.00, 57822.00, 57822.00, 57822.00, 57822.00, 
    57822.00, 57822.00, 57822.00, 57822.00, 57822.00, 57822.00, 57822.00, 57822.00, 57822.00, 57822.00, 
    57822.00, 57822.00, 57822.00, 57822.00, 57822.00, 57822.00, 57822.00, 57822.00, 57822.00, 57822.00, 
    57822.00, 57822.00, 57822.00, 57822.00, 57822.00, 57822.00, 57822.00, 57822.00, 57822.00, 57822.00, 
      633.26,   629.89,   626.87,   624.08,   621.50,   619.71,   617.78,   615.93,   614.21,   612.62, 
      611.12,   609.74,   608.48,   607.31,   606.26,   605.28,   604.44,   603.69,   603.04,   602.49, 
      601.70,   601.46,   601.32,   601.27, ])
_qm_ak135 = np.array([
        0.00,     0.00,    80.00,    80.00,   599.99,   599.99,   599.99,   599.99,   394.62,   403.93, 
      417.59,    75.60,    76.06,    76.06,    76.55,    79.40,   133.72,   136.38,   139.38,   142.76, 
      146.57,   162.50,   164.87,   166.80,   168.78,   170.82,   172.93,   549.45,   543.48,   537.63, 
      531.91,   526.32,   520.83,   515.46,   510.20,   505.05,   500.00,   495.05,   490.20,   485.44, 
      480.77,   476.19,   471.70,   467.29,   462.96,   458.72,   454.55,   450.45,   446.43,   442.48, 
      436.68,   431.03,   425.53,   418.41,   414.94,   411.52,   406.50,   403.23,   398.41,   392.16, 
      387.60,   383.14,   378.79,   375.94,   371.75,   367.65,   363.64,   359.71,   354.61,   350.88, 
      271.74,   273.97,   273.97,   273.97,     0.00,     0.00,     0.00,     0.00,     0.00,     0.00, 
        0.00,     0.00,     0.00,     0.00,     0.00,     0.00,     0.00,     0.00,     0.00,     0.00, 
        0.00,     0.00,     0.00,     0.00,     0.00,     0.00,     0.00,     0.00,     0.00,     0.00, 
        0.00,     0.00,     0.00,     0.00,     0.00,     0.00,     0.00,     0.00,     0.00,     0.00, 
        0.00,     0.00,     0.00,     0.00,     0.00,     0.00,     0.00,     0.00,     0.00,     0.00, 
       85.03,    85.03,    85.03,    85.03,    85.03,    85.03,    85.03,    85.03,    85.03,    85.03, 
       85.03,    85.03,    85.03,    85.03,    85.03,    85.03,    85.03,    85.03,    85.03,    85.03, 
       85.03,    85.03,    85.03,    85.03, ])
_dep_ak135 = np.array([
        0.00,     3.00,     3.00,     3.30,     3.30,    10.00,    10.00,    18.00,    18.00,    43.00, 
       80.00,    80.00,   120.00,   120.00,   165.00,   210.00,   210.00,   260.00,   310.00,   360.00, 
      410.00,   410.00,   460.00,   510.00,   560.00,   610.00,   660.00,   660.00,   710.00,   760.00, 
      809.50,   859.00,   908.50,   958.00,  1007.50,  1057.00,  1106.50,  1156.00,  1205.50,  1255.00, 
     1304.50,  1354.00,  1403.50,  1453.00,  1502.50,  1552.00,  1601.50,  1651.00,  1700.50,  1750.00, 
     1799.50,  1849.00,  1898.50,  1948.00,  1997.50,  2047.00,  2096.50,  2146.00,  2195.50,  2245.00, 
     2294.50,  2344.00,  2393.50,  2443.00,  2492.50,  2542.00,  2591.50,  2640.00,  2690.00,  2740.00, 
     2740.00,  2789.67,  2839.33,  2891.50,  2891.50,  2939.33,  2989.66,  3039.99,  3090.32,  3140.66, 
     3190.99,  3241.32,  3291.65,  3341.98,  3392.31,  3442.64,  3492.97,  3543.30,  3593.64,  3643.97, 
     3694.30,  3744.63,  3794.96,  3845.29,  3895.62,  3945.95,  3996.28,  4046.62,  4096.95,  4147.28, 
     4197.61,  4247.94,  4298.27,  4348.60,  4398.93,  4449.26,  4499.60,  4549.93,  4600.26,  4650.59, 
     4700.92,  4751.25,  4801.58,  4851.91,  4902.24,  4952.58,  5002.91,  5053.24,  5103.57,  5153.50, 
     5153.50,  5204.61,  5255.32,  5306.04,  5356.75,  5407.46,  5458.17,  5508.89,  5559.60,  5610.31, 
     5661.02,  5711.74,  5762.45,  5813.16,  5863.87,  5914.59,  5965.30,  6016.01,  6066.72,  6117.44, 
     6218.86,  6269.57,  6320.29,  6371.00, ])

_f_vp_ak135  = scipy.interpolate.interp1d(_dep_ak135, _vp_ak135, kind='linear')
_f_vs_ak135  = scipy.interpolate.interp1d(_dep_ak135, _vs_ak135, kind='linear')
_f_rho_ak135 = scipy.interpolate.interp1d(_dep_ak135, _rho_ak135, kind='linear')
_f_qk_ak135  = scipy.interpolate.interp1d(_dep_ak135, _qk_ak135, kind='linear')
_f_qm_ak135  = scipy.interpolate.interp1d(_dep_ak135, _qm_ak135, kind='linear')

