"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const s3 = require("../lib");
module.exports = {
    'default properties'(test) {
        const stack = new core_1.Stack();
        const myBucket = new s3.Bucket(stack, 'MyBucket');
        const myBucketPolicy = new s3.BucketPolicy(stack, 'MyBucketPolicy', {
            bucket: myBucket,
        });
        myBucketPolicy.document.addStatements(new aws_iam_1.PolicyStatement({
            resources: [myBucket.bucketArn],
            actions: ['s3:GetObject*'],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::BucketPolicy', {
            Bucket: {
                'Ref': 'MyBucketF68F3FF0',
            },
            PolicyDocument: {
                'Version': '2012-10-17',
                'Statement': [
                    {
                        'Action': 's3:GetObject*',
                        'Effect': 'Allow',
                        'Resource': { 'Fn::GetAtt': ['MyBucketF68F3FF0', 'Arn'] },
                    },
                ],
            },
        }));
        test.done();
    },
    'when specifying a removalPolicy at creation'(test) {
        const stack = new core_1.Stack();
        const myBucket = new s3.Bucket(stack, 'MyBucket');
        const myBucketPolicy = new s3.BucketPolicy(stack, 'MyBucketPolicy', {
            bucket: myBucket,
            removalPolicy: core_1.RemovalPolicy.RETAIN,
        });
        myBucketPolicy.document.addStatements(new aws_iam_1.PolicyStatement({
            resources: [myBucket.bucketArn],
            actions: ['s3:GetObject*'],
        }));
        assert_1.expect(stack).toMatch({
            'Resources': {
                'MyBucketF68F3FF0': {
                    'Type': 'AWS::S3::Bucket',
                    'DeletionPolicy': 'Retain',
                    'UpdateReplacePolicy': 'Retain',
                },
                'MyBucketPolicy0AFEFDBE': {
                    'Type': 'AWS::S3::BucketPolicy',
                    'Properties': {
                        'Bucket': {
                            'Ref': 'MyBucketF68F3FF0',
                        },
                        'PolicyDocument': {
                            'Statement': [
                                {
                                    'Action': 's3:GetObject*',
                                    'Effect': 'Allow',
                                    'Resource': { 'Fn::GetAtt': ['MyBucketF68F3FF0', 'Arn'] },
                                },
                            ],
                            'Version': '2012-10-17',
                        },
                    },
                    'DeletionPolicy': 'Retain',
                    'UpdateReplacePolicy': 'Retain',
                },
            },
        });
        test.done();
    },
    'when specifying a removalPolicy after creation'(test) {
        var _a;
        const stack = new core_1.Stack();
        const myBucket = new s3.Bucket(stack, 'MyBucket');
        myBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            resources: [myBucket.bucketArn],
            actions: ['s3:GetObject*'],
        }));
        (_a = myBucket.policy) === null || _a === void 0 ? void 0 : _a.applyRemovalPolicy(core_1.RemovalPolicy.RETAIN);
        assert_1.expect(stack).toMatch({
            'Resources': {
                'MyBucketF68F3FF0': {
                    'Type': 'AWS::S3::Bucket',
                    'DeletionPolicy': 'Retain',
                    'UpdateReplacePolicy': 'Retain',
                },
                'MyBucketPolicyE7FBAC7B': {
                    'Type': 'AWS::S3::BucketPolicy',
                    'Properties': {
                        'Bucket': {
                            'Ref': 'MyBucketF68F3FF0',
                        },
                        'PolicyDocument': {
                            'Statement': [
                                {
                                    'Action': 's3:GetObject*',
                                    'Effect': 'Allow',
                                    'Resource': { 'Fn::GetAtt': ['MyBucketF68F3FF0', 'Arn'] },
                                },
                            ],
                            'Version': '2012-10-17',
                        },
                    },
                    'DeletionPolicy': 'Retain',
                    'UpdateReplacePolicy': 'Retain',
                },
            },
        });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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