#!/usr/bin/env python
# coding: utf-8

import os
import numpy as np
import matplotlib
matplotlib.use('Agg')
import matplotlib.pyplot as plt
plt.style.use('ggplot')

from geometric.molecule import Molecule
from torsiondrive.tools import read_scan_xyz


def plot_1d_curve(grid_data, filename):
    """
    Plot torsiondrive 1D scan results as a potential energy curve

    Parameters
    ----------
    grid_data: dict
        Dictionary with data {grid_id: energy, ..}, generated by read_scan_xyz()

    filename: str
        Output pdf filename for saving the plot
    """
    if not grid_data:
        print('grid_data is empty, returning')
        return
    sorted_grid_id = sorted(grid_data.keys())
    assert len(sorted_grid_id[0]) == 1, "only 1d grid plot is supported"
    dihedrals = [grid_id[0] for grid_id in sorted_grid_id]
    energies = [grid_data[grid_id] for grid_id in sorted_grid_id]
    # convert energy to kcal/mol
    energies = np.array(energies) * 627.509
    # convert absolute energy to relative energy
    energies -= energies.min()
    # plot
    plt.Figure()
    plt.plot(dihedrals, energies, '-o')
    #plt.xticks(dihedrals[1::2])
    plt.xlabel('Dihedral Angle [Degree]')
    plt.ylabel('Relative Energy [kcal/mol]')
    plt.title("Torsion Energy Profile")
    plt.savefig(filename)
    plt.close()

def main():
    import argparse
    parser = argparse.ArgumentParser(formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument('xyz', nargs='*', default=['scan.xyz'], help='Input scan xyz file')
    args = parser.parse_args()

    for f in args.xyz:
        data = read_scan_xyz(f)
        pdf_filename = os.path.splitext(f)[0] + '.pdf'
        plot_1d_curve(data, pdf_filename)
        print("Generated %s" % pdf_filename)

if __name__ == '__main__':
    main()
