///////////////////////////////////////////////////////////////////////////////
///
/// \brief AbstractIODevice is an interface to handle all possibles I/O that can
///        act as a stream (file / TCP socket / pipe and so on).
///
///////////////////////////////////////////////////////////////////////////////

#ifndef JIMINY_ABSTRACT_IO_DEVICE_H
#define JIMINY_ABSTRACT_IO_DEVICE_H

#include <memory>

#include "jiminy/core/Types.h"


namespace jiminy
{
    /// Possible modes for a device (their availability depend of the concrete device).
    enum OpenMode
    {
        NOT_OPEN       = 0x000,  ///< Device is not opened.
        READ_ONLY      = 0x001,  ///< Read only mode.
        WRITE_ONLY     = 0x002,  ///< Write only mode.
        READ_WRITE     = 0x004,  ///< Read/Write mode (WARNING: (READ_ONLY | WRITE_ONLY) != READ_WRITE)
        APPEND         = 0x008,  ///< Open the device in append mode.
        TRUNCATE       = 0x010,  ///< Truncate the device at opening.
        UNBUFFERED     = 0x020,  ///< Do not use intermediate buffer if possible.
        NEW_ONLY       = 0x040,  ///< Create the device at opening, fail if the device already exists.
        EXISTING_ONLY  = 0x080,  ///< Do not create the device if it does not exists.
        NON_BLOCKING   = 0x100,  ///< Open the device in non blocking mode.
        SYNC           = 0x200,  ///< Open the device in sync mode (ensure that write are finished at return).
    };

    /// Facility operators to avoid cast.
    enum OpenMode operator | (enum OpenMode const & modeA, enum OpenMode const & modeB);
    enum OpenMode operator & (enum OpenMode const & modeA, enum OpenMode const & modeB);
    enum OpenMode operator |= (enum OpenMode & modeA, enum OpenMode const & modeB);
    enum OpenMode operator &= (enum OpenMode & modeA, enum OpenMode const & modeB);
    enum OpenMode operator ~(enum OpenMode mode);

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////
    /// \brief Base interface class for all I/O devices.
    /////////////////////////////////////////////////////////////////////////////////////////////////////////////
    class AbstractIODevice
    {
    public:
        AbstractIODevice(void) = default;
        virtual ~AbstractIODevice(void) = default;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// \brief Open the device.
        ///
        /// \param mode Mode to apply for opening the device.
        ///
        /// \return hresult_t::SUCCESS if successful, another hresult_t value otherwise.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        hresult_t open(enum OpenMode mode);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// \brief Write data in the device.
        /// \details The default implementation manage only POD type. For specific type, the template shall be extended
        ///          with specific implementation.
        ///
        /// \param Value to write into the device.
        ///
        /// \return hresult_t::SUCCESS if successful, another hresult_t value otherwise.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        hresult_t close(void);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// \return The current opening modes.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        enum OpenMode openModes(void) const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// \return The supported opening modes.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        enum OpenMode supportedModes(void) const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// \return true if the device is writable, false otherwise.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        bool_t isWritable(void) const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// \return true if the device is readable, false otherwise.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        bool_t isReadable(void) const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// \return true if the device is opened, false otherwise.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        bool_t isOpen(void) const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// \return true if the device is sequential (i.e socket), false if the device support random-access (i.e regular file).
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        virtual bool_t isSequential(void) const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// \brief For random-access devices, this function returns the size of the device.
        ///        For sequential devices, bytesAvailable() is returned.
        ///
        /// \return The size of the device.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        virtual int64_t size(void);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// \brief Move the current position cursor to pos if possible.
        ///
        /// \param pos  Desired new position of the cursor.
        ///
        /// \return hresult_t::SUCCESS if successful, another hresult_t value otherwise.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        virtual hresult_t seek(int64_t pos);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// \return The current cursor position (0 if there is not concept of position cursor).
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        virtual int64_t pos(void);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// \brief resize the device to provided size.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        virtual hresult_t resize(int64_t size);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// \brief Returns the number of bytes that are available for reading. Commonly used with sequential device.
        ///
        /// \return The available bytesfor reading.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        virtual int64_t bytesAvailable(void);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// \brief Write data in the device.
        /// \details The default implementation manage only POD type. For specific type, the template shall be extended
        ///          with specific implementation.
        ///
        /// \param Value to write into the device.
        ///
        /// \return hresult_t::SUCCESS if successful, another hresult_t value otherwise.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        template<typename T>
        hresult_t write(T const & valueIn);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// \brief Write data in the device.
        ///
        /// \param data      Buffer of data to write.
        /// \param dataSize  Number of bytes to write.
        ///
        /// \return hresult_t::SUCCESS if successful, another hresult_t value otherwise.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        virtual hresult_t write(void    const * data,
                                int64_t         dataSize);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// \brief Write data in the device.
        ///
        /// \param data      Buffer of data to write.
        /// \param dataSize  Number of bytes to write.
        ///
        /// \return the number of bytes written, -1 in case of error
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        virtual int64_t writeData(void    const * data,
                                  int64_t         dataSize) = 0;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// \brief Read data in the device.
        /// \details The default implementation manage only POD type. For specific type, the template shall be extended
        ///          with specific implementation.
        ///
        /// \param Value to store read data.
        ///
        /// \return hresult_t::SUCCESS if successful, another hresult_t value otherwise.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        template<typename T>
        hresult_t read(T & valueIn);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// \brief Read data from the device.
        ///
        /// \param data      Buffer to store read data.
        /// \param dataSize  Number of bytes to read.
        ///
        /// \return hresult_t::SUCCESS if successful, another hresult_t value otherwise.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        virtual hresult_t read(void    * data,
                               int64_t   dataSize);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// \brief Read data in the device.
        ///
        /// \param data      Buffer of data to read.
        /// \param dataSize  Number of bytes to read.
        ///
        /// \return the number of bytes read, -1 in case of error
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        virtual int64_t readData(void    * data,
                                 int64_t   dataSize) = 0;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// \brief Retrieve the latest error. Usefull for call that do not return an error code directly.
        ///
        /// \return The latest generated error.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        hresult_t getLastError(void) const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// \brief Set the device blocking fashion.
        ///
        /// \return The latest generated error.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        virtual hresult_t setBlockingMode(bool_t shouldBlock);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// \brief Set the device backend (reset the old one if any).
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        bool_t isBackendValid();

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// \brief Set the device backend (reset the old one if any).
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        virtual void setBackend(std::unique_ptr<AbstractIODevice> io);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// \brief Reset the device backend.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////
        virtual void removeBackend(void);

    protected:
        virtual hresult_t doOpen(enum OpenMode mode) = 0;
        virtual hresult_t doClose(void) = 0;

        enum OpenMode modes_{OpenMode::NOT_OPEN};           ///< Current opening mode.
        enum OpenMode supportedModes_{OpenMode::NOT_OPEN};  ///< Supported modes of the device.
        hresult_t lastError_{hresult_t::SUCCESS};           ///< Latest generated error.
        std::unique_ptr<AbstractIODevice> io_{nullptr};     ///< Backend to use if any
    };
}

#include "AbstractIODevice.tpp"

#endif // JIMINY_ABSTRACT_IO_DEVICE_H
