"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationLoadBalancedCodeDeployedFargateService = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_synthetics_alpha_1 = require("@aws-cdk/aws-synthetics-alpha");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const aws_codedeploy_1 = require("aws-cdk-lib/aws-codedeploy");
const aws_ecs_1 = require("aws-cdk-lib/aws-ecs");
const aws_ecs_patterns_1 = require("aws-cdk-lib/aws-ecs-patterns");
const aws_elasticloadbalancingv2_1 = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const api_canary_1 = require("../api-canary");
const ecs_deployment_1 = require("../ecs-deployment");
/**
 * A Fargate service running on an ECS cluster fronted by an application load balancer and deployed by CodeDeploy.
 */
class ApplicationLoadBalancedCodeDeployedFargateService extends aws_ecs_patterns_1.ApplicationLoadBalancedFargateService {
    /**
     * Constructs a new instance of the ApplicationLoadBalancedCodeDeployedFargateService class.
     */
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            deploymentController: {
                type: aws_ecs_1.DeploymentControllerType.CODE_DEPLOY,
            },
        });
        if (props.deregistrationDelay) {
            this.targetGroup.setAttribute('deregistration_delay.timeout_seconds', props.deregistrationDelay.toSeconds().toString());
        }
        if (props.healthCheck) {
            this.targetGroup.configureHealthCheck(props.healthCheck);
        }
        this.accessLogBucket = props.accessLogBucket ??
            new aws_s3_1.Bucket(this, 'AccessLogBucket', {
                encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
                blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
                enforceSSL: true,
            });
        this.loadBalancer.logAccessLogs(this.accessLogBucket, props.accessLogPrefix);
        const alarms = [];
        if (props.responseTimeAlarmThreshold) {
            const responseTimeAlarm = new aws_cloudwatch_1.Alarm(this, 'ResponseTimeAlarm', {
                metric: this.loadBalancer.metrics.targetResponseTime({
                    period: aws_cdk_lib_1.Duration.minutes(1),
                    statistic: 'p95',
                }),
                evaluationPeriods: 2,
                threshold: props.responseTimeAlarmThreshold.toSeconds(),
                comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            });
            alarms.push(responseTimeAlarm);
        }
        const protocol = props.protocol ?? (props.certificate ? aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS : aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP);
        const testHostName = props.domainName ? props.domainName : this.loadBalancer.loadBalancerDnsName;
        if (props.apiTestSteps?.length) {
            this.apiCanary = new api_canary_1.ApiCanary(this, 'Canary', {
                baseUrl: `${protocol.toLowerCase()}://${testHostName}`,
                durationAlarmThreshold: props.apiCanaryTimeout,
                schedule: aws_synthetics_alpha_1.Schedule.rate(props.apiCanarySchedule ?? aws_cdk_lib_1.Duration.minutes(5)),
                threadCount: props.apiCanaryThreadCount,
                steps: props.apiTestSteps,
            });
            this.apiCanary.node.addDependency(this.service);
            alarms.push(this.apiCanary.successAlarm);
            if (this.apiCanary.durationAlarm) {
                alarms.push(this.apiCanary.durationAlarm);
            }
        }
        if (alarms.length > 0) {
            this.healthAlarm = new aws_cloudwatch_1.CompositeAlarm(this, 'HealthAlarm', {
                alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(...alarms),
            });
        }
        else {
            this.healthAlarm = undefined;
        }
        let testPort;
        if (props.testPort) {
            testPort = props.testPort;
        }
        else if (props.listenerPort) {
            testPort = props.listenerPort + 1;
        }
        else if (protocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP) {
            testPort = 8080;
        }
        else if (protocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
            testPort = 8443;
        }
        else {
            throw new Error('Unable to determine port for test listener');
        }
        let certificates;
        if (props.certificate) {
            certificates = [props.certificate];
        }
        this.testListener = this.loadBalancer.addListener('TestListener', {
            protocol,
            port: testPort,
            open: props.openListener ?? true,
            sslPolicy: props.sslPolicy,
            certificates: certificates,
        });
        this.greenTargetGroup = new aws_elasticloadbalancingv2_1.ApplicationTargetGroup(this, 'GreenTargetGroup', {
            vpc: this.cluster.vpc,
            port: testPort,
            protocol: props.targetProtocol ?? aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
            protocolVersion: props.protocolVersion,
            deregistrationDelay: props.deregistrationDelay,
            healthCheck: props.healthCheck,
            targetType: aws_elasticloadbalancingv2_1.TargetType.IP,
        });
        this.listener.node.addDependency(this.greenTargetGroup);
        this.testListener.addTargetGroups('ECS', {
            targetGroups: [this.greenTargetGroup],
        });
        this.application = new aws_codedeploy_1.EcsApplication(this, 'Application');
        this.deploymentGroup = new aws_codedeploy_1.EcsDeploymentGroup(this, 'DeploymentGroup', {
            application: this.application,
            alarms: this.healthAlarm ? [this.healthAlarm] : undefined,
            service: this.service,
            blueGreenDeploymentConfig: {
                blueTargetGroup: this.targetGroup,
                greenTargetGroup: this.greenTargetGroup,
                listener: this.listener,
                testListener: this.testListener,
                terminationWaitTime: props.terminationWaitTime ?? aws_cdk_lib_1.Duration.minutes(10),
            },
            deploymentConfig: props.deploymentConfig ?? aws_codedeploy_1.EcsDeploymentConfig.ALL_AT_ONCE,
            autoRollback: {
                stoppedDeployment: true,
            },
        });
        this.deployment = new ecs_deployment_1.EcsDeployment({
            deploymentGroup: this.deploymentGroup,
            timeout: props.deploymentTimeout ?? aws_cdk_lib_1.Duration.minutes(60),
            targetService: {
                taskDefinition: this.taskDefinition,
                containerName: this.taskDefinition.defaultContainer.containerName,
                containerPort: this.taskDefinition.defaultContainer.containerPort,
            },
            hooks: props.hooks,
        });
    }
    addServiceAsTarget(service) {
        super.addServiceAsTarget(service);
    }
}
exports.ApplicationLoadBalancedCodeDeployedFargateService = ApplicationLoadBalancedCodeDeployedFargateService;
_a = JSII_RTTI_SYMBOL_1;
ApplicationLoadBalancedCodeDeployedFargateService[_a] = { fqn: "@cdklabs/cdk-ecs-codedeploy.ApplicationLoadBalancedCodeDeployedFargateService", version: "0.0.184" };
//# sourceMappingURL=data:application/json;base64,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