import json
from alfasim_sdk import convert_description_to_alfacase
from pathlib import Path

from alfasim_score.common import OperationType
from alfasim_score.converter.alfacase.base_operation import BaseOperationBuilder
from alfasim_score.converter.alfacase.injection_operation import InjectionOperationBuilder
from alfasim_score.converter.alfacase.production_operation import ProductionOperationBuilder
from alfasim_score.converter.alfacase.score_input_data import ScoreInputData
from alfasim_score.converter.alfacase.score_input_reader import ScoreInputReader
from alfasim_score.converter.alfacase.score_output_generator import ScoreOutputBuilder


class AlfasimScoreConverter:
    """
    This class handles the process of convertions between ALFAsim and SCORE file formats:
    - it can convert the SCORE input file format into an alfacase to be used by ALFAsim simulator
    - it can use a ALFAsim result into a SCORE output file.
    """

    def __init__(self, score_input_file: Path, score_output_file: Path):
        score_reader = ScoreInputReader(score_input_file)
        self.score_data = ScoreInputData(score_reader)
        self.alfacase_builder = self._get_score_to_alfacase_builder()
        self.output_builder = ScoreOutputBuilder(self.score_data, score_output_file)

    def _get_score_to_alfacase_builder(self) -> BaseOperationBuilder:
        """Convert SCORE input file to an alfacase description."""
        operation_type = self.score_data.operation_data["type"]
        if operation_type == OperationType.PRODUCTION:
            return ProductionOperationBuilder(self.score_data)
        else:
            return InjectionOperationBuilder(self.score_data)

    def generate_alfasim_input_file(self, alfacase_filepath: Path) -> None:
        """Create the ALFAsim input file (AKA alfacase) from an SCORE input file."""
        alfacase_description = self.alfacase_builder.generate_operation_alfacase_description()
        alfacase_content = convert_description_to_alfacase(alfacase_description)
        alfacase_filepath.write_text(data=alfacase_content, encoding="utf-8")

    def generate_score_output_file(self, alfasim_results_folder: Path) -> None:
        """Create the output file for SCORE based on the results generated by ALFAsim."""
        output = self.output_builder.generate_output_results(alfasim_results_folder)
        json_data = json.dumps(output, indent=2)
        self.output_builder.score_output_filepath.write_text(json_data, encoding="utf-8")
