"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Workload = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const WORKLOAD_SYMBOL = Symbol.for("@shady-island/core.Workload");
const IMPORTED_STACKS = new Map();
/**
 * A collection of Stacks in an Environment representing a deployment Tier.
 *
 * Consider deriving a subclass of `Workload` and creating your `Stack` objects
 * within its constructor.
 *
 * The difference between this class and a `Stage` is that a `Stage` is meant to
 * be deployed with CDK Pipelines. This class can be used with `cdk deploy`.
 * This class also provides context loading capabilities.
 *
 * It is an anti-pattern to provide a `Workload` instance as the parent scope to
 * the `aws-cdk-lib.Stack` constructor. You should either use the
 * `createStack()` method, create your own sub-class of `Stack` and provide a
 * `Workload` instance as the parent scope, or use the `import()` method to
 * essentially _import_ a `Stack` and its constructs into a `Workload` without
 * changing its scope.
 */
class Workload extends constructs_1.Construct {
    /**
     * Creates a new Workload.
     *
     * @param scope - The construct scope.
     * @param id - The construct ID.
     * @param props - The constructor options.
     */
    constructor(scope, id, props) {
        super(scope, id);
        Object.defineProperty(this, WORKLOAD_SYMBOL, { value: true });
        const { env: { region, account } = {}, tier, workloadName, contextFile, baseDomainName, } = props;
        this.workloadName = workloadName ?? `${id}`.toLowerCase();
        this.tier = tier;
        this.publicDomainName = baseDomainName
            ? `${tier.id}.${this.workloadName}.${baseDomainName}`.toLowerCase()
            : undefined;
        this._stacks = new Map();
        const { region: stageRegion, account: stageAccount } = aws_cdk_lib_1.Stage.of(this);
        this.region = region || stageRegion;
        this.account = account || stageAccount;
        if (contextFile) {
            this.loadContext(contextFile);
        }
    }
    /**
     * Return the Workload the construct is contained within, fails if there is no
     * workload up the tree.
     *
     * @param construct - The construct whose parent nodes will be searched
     * @returns The Workload containing the construct
     * @throws Error - if none of the construct's parents are a workload
     */
    static of(construct) {
        const constructStack = aws_cdk_lib_1.Stack.of(construct);
        if (IMPORTED_STACKS.has(constructStack)) {
            return IMPORTED_STACKS.get(constructStack);
        }
        const workload = construct.node.scopes
            .reverse()
            .slice(1)
            .find(Workload.isWorkload);
        if (!workload) {
            throw new Error("No workload exists in the parentage of the provided construct");
        }
        return workload;
    }
    /**
     * Test whether the given construct is a Workload.
     *
     * @param x - The value to test
     * @returns Whether the value is a Workload object.
     */
    static isWorkload(x) {
        return x !== null && typeof x === "object" && WORKLOAD_SYMBOL in x;
    }
    loadContext(filename) {
        try {
            fs.accessSync(filename, fs.constants.F_OK);
        }
        catch (err) {
            throw new Error(`Context file does not exist: ${filename}`);
        }
        try {
            fs.accessSync(filename, fs.constants.R_OK);
        }
        catch (err) {
            throw new Error(`Context file is not readable: ${filename}`);
        }
        const data = fs.readFileSync(filename, { encoding: "utf8" });
        let defaults = {};
        try {
            defaults = JSON.parse(data);
        }
        catch (e) {
            throw new Error(`Context file contains invalid JSON syntax: ${filename}`);
        }
        for (const [k, v] of Object.entries(defaults)) {
            this.node.setContext(k, v);
        }
    }
    /**
     * @returns The stacks created by invoking `createStack`
     */
    get stacks() {
        return [...this._stacks.values()];
    }
    /**
     * Register the provided `Stack` as being part of this `Workload`.
     *
     * @param stack - The stack to register.
     * @returns The provided Stack
     */
    registerStack(stack) {
        this.tier.applyTags(stack);
        this._stacks.set(stack.stackName, stack);
        return stack;
    }
    /**
     * Forces a return value for `Workload.of` for one or more `Stack` objects.
     *
     * Normally, a construct must be within the scope of the `Workload` instance,
     * such as a construct that is a descendant of a `Stack` returned from
     * `createStack()`.
     *
     * That means that any `Stack` instances you created in your CDK application
     * _before_ installing the `shady-island` library would not be able to be part
     * of a `Workload` unless you changed the `scope` argument of the `Stack`
     * constructor from the `App` or `Stage` to the desired `Workload` instance.
     * However, that's bad news for a `Stack` that has already been deployed to
     * CloudFormation because the resource identifier of persistent child
     * constructs (e.g. RDS databases, S3 buckets) would change.
     *
     * A successful call to this method will register the provided `Stack` objects
     * and all their construct descendants as members of that `Workload` instance.
     * Calling `Workload.of()` with any of the provided `Stack` objects or their
     * descendant constructs will return that `Workload` instance.
     *
     * If any of the `Stack` objects provided to this method already belong to a
     * different `Workload` object, or whose parent scope is not identical to the
     * parent scope of this `Workload` (i.e. the `Stage` or the `App`), an error
     * will be thrown.
     *
     * @param stacks - The `Stack` instances to import to this `Workload`
     * @throws {Error} if any of the stacks are already part of another workload
     * @throws {Error} if any of the stacks have a different parent scope
     */
    import(...stacks) {
        for (const stack of stacks) {
            if (stack.node.scope === this) {
                continue;
            }
            if (Workload.isWorkload(stack.node.scope) && stack.node.scope !== this) {
                throw new Error("The Stack is already contained within a different Workload");
            }
            if (stack.node.scope !== this.node.scope) {
                throw new Error("The Stack must be contained within the same scope as this Workload");
            }
            IMPORTED_STACKS.set(this.registerStack(stack), this);
        }
    }
    /**
     * Adds a stack to the Workload.
     *
     * This method will return a `Stack` with this Workload as its scope. By
     * default, the `stackName` property provided to the `Stack` will be this
     * Workload's `workloadName`, its `tier`, and the value of the `id`
     * parameter separated by hyphens, all in lowercase.
     *
     * @example
     *
     * const exampleDev = new Workload(app, 'Example', {
     *   tier: Tier.DEVELOPMENT,
     *   env: { account: '123456789012', region: 'us-east-1' },
     * });
     * const networkStack = exampleDev.createStack('Network', {});
     * assert.strictEqual(networkStack.stackName, 'example-dev-network').
     *
     * You can override the `env` and `stackName` properties in the `props`
     * argument if desired.
     *
     * The stack will have a `DeploymentTier` tag added, set to the tier label.
     *
     * @param id - The Stack construct id (e.g. "Network")
     * @param props - The new Stack properties
     */
    createStack(id, props) {
        const { stackName, ...options } = props ?? {};
        const newStackName = stackName ?? `${this.workloadName}-${this.tier.id}-${id}`.toLowerCase();
        const stack = new aws_cdk_lib_1.Stack(this, id, {
            stackName: newStackName,
            env: this.account && this.region
                ? { account: this.account, region: this.region }
                : undefined,
            ...options,
        });
        return this.registerStack(stack);
    }
}
exports.Workload = Workload;
_a = JSII_RTTI_SYMBOL_1;
Workload[_a] = { fqn: "shady-island.Workload", version: "0.1.1" };
//# sourceMappingURL=data:application/json;base64,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