"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const codedeploy_generated_1 = require("../codedeploy.generated");
const utils_1 = require("../utils");
const application_1 = require("./application");
const deployment_config_1 = require("./deployment-config");
/**
 * @resource AWS::CodeDeploy::DeploymentGroup
 */
class LambdaDeploymentGroup extends cdk.Resource {
    /**
     * Import an Lambda Deployment Group defined either outside the CDK,
     * or in a different CDK Stack and exported using the {@link #export} method.
     *
     * @param scope the parent Construct for this new Construct
     * @param id the logical ID of this new Construct
     * @param attrs the properties of the referenced Deployment Group
     * @returns a Construct representing a reference to an existing Deployment Group
     */
    static fromLambdaDeploymentGroupAttributes(scope, id, attrs) {
        return new ImportedLambdaDeploymentGroup(scope, id, attrs);
    }
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.deploymentGroupName,
        });
        this.application = props.application || new application_1.LambdaApplication(this, 'Application');
        this.alarms = props.alarms || [];
        this.role = props.role || new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('codedeploy.amazonaws.com')
        });
        this.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSCodeDeployRoleForLambda'));
        this.deploymentConfig = props.deploymentConfig || deployment_config_1.LambdaDeploymentConfig.CANARY_10PERCENT_5MINUTES;
        const resource = new codedeploy_generated_1.CfnDeploymentGroup(this, 'Resource', {
            applicationName: this.application.applicationName,
            serviceRoleArn: this.role.roleArn,
            deploymentGroupName: this.physicalName,
            deploymentConfigName: this.deploymentConfig.deploymentConfigName,
            deploymentStyle: {
                deploymentType: 'BLUE_GREEN',
                deploymentOption: 'WITH_TRAFFIC_CONTROL'
            },
            alarmConfiguration: cdk.Lazy.anyValue({ produce: () => utils_1.renderAlarmConfiguration(this.alarms, props.ignorePollAlarmsFailure) }),
            autoRollbackConfiguration: cdk.Lazy.anyValue({ produce: () => utils_1.renderAutoRollbackConfiguration(this.alarms, props.autoRollback) }),
        });
        this.deploymentGroupName = this.getResourceNameAttribute(resource.ref);
        this.deploymentGroupArn = this.getResourceArnAttribute(utils_1.arnForDeploymentGroup(this.application.applicationName, resource.ref), {
            service: 'codedeploy',
            resource: 'deploymentgroup',
            resourceName: `${this.application.applicationName}/${this.physicalName}`,
            sep: ':',
        });
        if (props.preHook) {
            this.addPreHook(props.preHook);
        }
        if (props.postHook) {
            this.addPostHook(props.postHook);
        }
        props.alias.node.defaultChild.cfnOptions.updatePolicy = {
            codeDeployLambdaAliasUpdate: {
                applicationName: this.application.applicationName,
                deploymentGroupName: resource.ref,
                beforeAllowTrafficHook: cdk.Lazy.stringValue({ produce: () => this.preHook && this.preHook.functionName }),
                afterAllowTrafficHook: cdk.Lazy.stringValue({ produce: () => this.postHook && this.postHook.functionName }),
            }
        };
    }
    /**
     * Associates an additional alarm with this Deployment Group.
     *
     * @param alarm the alarm to associate with this Deployment Group
     */
    addAlarm(alarm) {
        this.alarms.push(alarm);
    }
    /**
     * Associate a function to run before deployment begins.
     * @param preHook function to run before deployment beings
     * @throws an error if a pre-hook function is already configured
     */
    addPreHook(preHook) {
        if (this.preHook !== undefined) {
            throw new Error('A pre-hook function is already defined for this deployment group');
        }
        this.preHook = preHook;
        this.grantPutLifecycleEventHookExecutionStatus(this.preHook);
        this.preHook.grantInvoke(this.role);
    }
    /**
     * Associate a function to run after deployment completes.
     * @param postHook function to run after deployment completes
     * @throws an error if a post-hook function is already configured
     */
    addPostHook(postHook) {
        if (this.postHook !== undefined) {
            throw new Error('A post-hook function is already defined for this deployment group');
        }
        this.postHook = postHook;
        this.grantPutLifecycleEventHookExecutionStatus(this.postHook);
        this.postHook.grantInvoke(this.role);
    }
    /**
     * Grant a principal permission to codedeploy:PutLifecycleEventHookExecutionStatus
     * on this deployment group resource.
     * @param grantee to grant permission to
     */
    grantPutLifecycleEventHookExecutionStatus(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            resourceArns: [this.deploymentGroupArn],
            actions: ['codedeploy:PutLifecycleEventHookExecutionStatus'],
        });
    }
}
exports.LambdaDeploymentGroup = LambdaDeploymentGroup;
class ImportedLambdaDeploymentGroup extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.application = props.application;
        this.deploymentGroupName = props.deploymentGroupName;
        this.deploymentGroupArn = utils_1.arnForDeploymentGroup(props.application.applicationName, props.deploymentGroupName);
        this.deploymentConfig = props.deploymentConfig || deployment_config_1.LambdaDeploymentConfig.CANARY_10PERCENT_5MINUTES;
    }
}
//# sourceMappingURL=data:application/json;base64,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