"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const codedeploy_generated_1 = require("../codedeploy.generated");
const utils_1 = require("../utils");
/**
 * Minimum number of healthy hosts for a server deployment.
 */
class MinimumHealthyHosts {
    constructor(json) {
        this.json = json;
    }
    /**
     * The minimum healhty hosts threshold expressed as an absolute number.
     */
    static count(value) {
        return new MinimumHealthyHosts({
            type: 'HOST_COUNT',
            value
        });
    }
    /**
     * The minmum healhty hosts threshold expressed as a percentage of the fleet.
     */
    static percentage(value) {
        return new MinimumHealthyHosts({
            type: 'FLEET_PERCENT',
            value
        });
    }
    /**
     * @internal
     */
    get _json() {
        return this.json;
    }
}
exports.MinimumHealthyHosts = MinimumHealthyHosts;
/**
 * A custom Deployment Configuration for an EC2/on-premise Deployment Group.
 *
 * @resource AWS::CodeDeploy::DeploymentConfig
 */
class ServerDeploymentConfig extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.deploymentConfigName,
        });
        const resource = new codedeploy_generated_1.CfnDeploymentConfig(this, 'Resource', {
            deploymentConfigName: this.physicalName,
            minimumHealthyHosts: props.minimumHealthyHosts._json,
        });
        this.deploymentConfigName = resource.ref;
        this.deploymentConfigArn = utils_1.arnForDeploymentConfig(this.deploymentConfigName);
    }
    /**
     * Import a custom Deployment Configuration for an EC2/on-premise Deployment Group defined either outside the CDK,
     * or in a different CDK Stack and exported using the {@link #export} method.
     *
     * @param scope the parent Construct for this new Construct
     * @param id the logical ID of this new Construct
     * @param serverDeploymentConfigName the properties of the referenced custom Deployment Configuration
     * @returns a Construct representing a reference to an existing custom Deployment Configuration
     */
    static fromServerDeploymentConfigName(scope, id, serverDeploymentConfigName) {
        ignore(scope);
        ignore(id);
        return deploymentConfig(serverDeploymentConfigName);
    }
}
ServerDeploymentConfig.ONE_AT_A_TIME = deploymentConfig('CodeDeployDefault.OneAtATime');
ServerDeploymentConfig.HALF_AT_A_TIME = deploymentConfig('CodeDeployDefault.HalfAtATime');
ServerDeploymentConfig.ALL_AT_ONCE = deploymentConfig('CodeDeployDefault.AllAtOnce');
exports.ServerDeploymentConfig = ServerDeploymentConfig;
function deploymentConfig(name) {
    return {
        deploymentConfigName: name,
        deploymentConfigArn: utils_1.arnForDeploymentConfig(name),
    };
}
function ignore(_x) { return; }
//# sourceMappingURL=data:application/json;base64,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