"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const core_1 = require("@aws-cdk/core");
const codedeploy_generated_1 = require("../codedeploy.generated");
const utils_1 = require("../utils");
const application_1 = require("./application");
const deployment_config_1 = require("./deployment-config");
const load_balancer_1 = require("./load-balancer");
/**
 * Represents a reference to a CodeDeploy EC2/on-premise Deployment Group.
 *
 * If you're managing the Deployment Group alongside the rest of your CDK resources,
 * use the {@link ServerDeploymentGroup} class.
 *
 * If you want to reference an already existing Deployment Group,
 * or one defined in a different CDK Stack,
 * use the {@link #import} method.
 */
class ServerDeploymentGroupBase extends cdk.Resource {
    constructor(scope, id, deploymentConfig, props) {
        super(scope, id, props);
        this.deploymentConfig = deploymentConfig || deployment_config_1.ServerDeploymentConfig.ONE_AT_A_TIME;
    }
}
class ImportedServerDeploymentGroup extends ServerDeploymentGroupBase {
    constructor(scope, id, props) {
        super(scope, id, props.deploymentConfig);
        this.role = undefined;
        this.autoScalingGroups = undefined;
        this.application = props.application;
        this.deploymentGroupName = props.deploymentGroupName;
        this.deploymentGroupArn = utils_1.arnForDeploymentGroup(props.application.applicationName, props.deploymentGroupName);
    }
}
/**
 * Represents a set of instance tag groups.
 * An instance will match a set if it matches all of the groups in the set -
 * in other words, sets follow 'and' semantics.
 * You can have a maximum of 3 tag groups inside a set.
 */
class InstanceTagSet {
    constructor(...instanceTagGroups) {
        if (instanceTagGroups.length > 3) {
            throw new Error('An instance tag set can have a maximum of 3 instance tag groups, ' +
                `but ${instanceTagGroups.length} were provided`);
        }
        this._instanceTagGroups = instanceTagGroups;
    }
    get instanceTagGroups() {
        return this._instanceTagGroups.slice();
    }
}
exports.InstanceTagSet = InstanceTagSet;
/**
 * A CodeDeploy Deployment Group that deploys to EC2/on-premise instances.
 * @resource AWS::CodeDeploy::DeploymentGroup
 */
class ServerDeploymentGroup extends ServerDeploymentGroupBase {
    /**
     * Import an EC2/on-premise Deployment Group defined either outside the CDK,
     * or in a different CDK Stack and exported using the {@link #export} method.
     *
     * @param scope the parent Construct for this new Construct
     * @param id the logical ID of this new Construct
     * @param attrs the properties of the referenced Deployment Group
     * @returns a Construct representing a reference to an existing Deployment Group
     */
    static fromServerDeploymentGroupAttributes(scope, id, attrs) {
        return new ImportedServerDeploymentGroup(scope, id, attrs);
    }
    constructor(scope, id, props = {}) {
        super(scope, id, props.deploymentConfig, {
            physicalName: props.deploymentGroupName,
        });
        this.application = props.application || new application_1.ServerApplication(this, 'Application');
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('codedeploy.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSCodeDeployRole')],
        });
        this._autoScalingGroups = props.autoScalingGroups || [];
        this.installAgent = props.installAgent === undefined ? true : props.installAgent;
        this.codeDeployBucket = s3.Bucket.fromBucketName(this, 'Bucket', `aws-codedeploy-${core_1.Stack.of(this).region}`);
        for (const asg of this._autoScalingGroups) {
            this.addCodeDeployAgentInstallUserData(asg);
        }
        this.alarms = props.alarms || [];
        const resource = new codedeploy_generated_1.CfnDeploymentGroup(this, 'Resource', {
            applicationName: this.application.applicationName,
            deploymentGroupName: this.physicalName,
            serviceRoleArn: this.role.roleArn,
            deploymentConfigName: props.deploymentConfig &&
                props.deploymentConfig.deploymentConfigName,
            autoScalingGroups: cdk.Lazy.listValue({ produce: () => this._autoScalingGroups.map(asg => asg.autoScalingGroupName) }, { omitEmpty: true }),
            loadBalancerInfo: this.loadBalancerInfo(props.loadBalancer),
            deploymentStyle: props.loadBalancer === undefined
                ? undefined
                : {
                    deploymentOption: 'WITH_TRAFFIC_CONTROL',
                },
            ec2TagSet: this.ec2TagSet(props.ec2InstanceTags),
            onPremisesTagSet: this.onPremiseTagSet(props.onPremiseInstanceTags),
            alarmConfiguration: cdk.Lazy.anyValue({ produce: () => utils_1.renderAlarmConfiguration(this.alarms, props.ignorePollAlarmsFailure) }),
            autoRollbackConfiguration: cdk.Lazy.anyValue({ produce: () => utils_1.renderAutoRollbackConfiguration(this.alarms, props.autoRollback) }),
        });
        this.deploymentGroupName = this.getResourceNameAttribute(resource.ref);
        this.deploymentGroupArn = this.getResourceArnAttribute(utils_1.arnForDeploymentGroup(this.application.applicationName, resource.ref), {
            service: 'codedeploy',
            resource: 'deploymentgroup',
            resourceName: `${this.application.applicationName}/${this.physicalName}`,
            sep: ':',
        });
    }
    /**
     * Adds an additional auto-scaling group to this Deployment Group.
     *
     * @param asg the auto-scaling group to add to this Deployment Group.
     * [disable-awslint:ref-via-interface] is needed in order to install the code
     * deploy agent by updating the ASGs user data.
     */
    addAutoScalingGroup(asg) {
        this._autoScalingGroups.push(asg);
        this.addCodeDeployAgentInstallUserData(asg);
    }
    /**
     * Associates an additional alarm with this Deployment Group.
     *
     * @param alarm the alarm to associate with this Deployment Group
     */
    addAlarm(alarm) {
        this.alarms.push(alarm);
    }
    get autoScalingGroups() {
        return this._autoScalingGroups.slice();
    }
    addCodeDeployAgentInstallUserData(asg) {
        if (!this.installAgent) {
            return;
        }
        this.codeDeployBucket.grantRead(asg.role, 'latest/*');
        switch (asg.osType) {
            case ec2.OperatingSystemType.LINUX:
                asg.addUserData('PKG_CMD=`which yum 2>/dev/null`', 'if [ -z "$PKG_CMD" ]; then', 'PKG_CMD=apt-get', 'else', 'PKG=CMD=yum', 'fi', '$PKG_CMD update -y', '$PKG_CMD install -y ruby2.0', 'if [ $? -ne 0 ]; then', '$PKG_CMD install -y ruby', 'fi', '$PKG_CMD install -y awscli', 'TMP_DIR=`mktemp -d`', 'cd $TMP_DIR', `aws s3 cp s3://aws-codedeploy-${core_1.Stack.of(this).region}/latest/install . --region ${core_1.Stack.of(this).region}`, 'chmod +x ./install', './install auto', 'rm -fr $TMP_DIR');
                break;
            case ec2.OperatingSystemType.WINDOWS:
                asg.addUserData('Set-Variable -Name TEMPDIR -Value (New-TemporaryFile).DirectoryName', `aws s3 cp s3://aws-codedeploy-${core_1.Stack.of(this).region}/latest/codedeploy-agent.msi $TEMPDIR\\codedeploy-agent.msi`, '$TEMPDIR\\codedeploy-agent.msi /quiet /l c:\\temp\\host-agent-install-log.txt');
                break;
        }
    }
    loadBalancerInfo(loadBalancer) {
        if (!loadBalancer) {
            return undefined;
        }
        switch (loadBalancer.generation) {
            case load_balancer_1.LoadBalancerGeneration.FIRST:
                return {
                    elbInfoList: [
                        { name: loadBalancer.name },
                    ],
                };
            case load_balancer_1.LoadBalancerGeneration.SECOND:
                return {
                    targetGroupInfoList: [
                        { name: loadBalancer.name },
                    ]
                };
        }
    }
    ec2TagSet(tagSet) {
        if (!tagSet || tagSet.instanceTagGroups.length === 0) {
            return undefined;
        }
        return {
            ec2TagSetList: tagSet.instanceTagGroups.map(tagGroup => {
                return {
                    ec2TagGroup: this.tagGroup2TagsArray(tagGroup),
                };
            }),
        };
    }
    onPremiseTagSet(tagSet) {
        if (!tagSet || tagSet.instanceTagGroups.length === 0) {
            return undefined;
        }
        return {
            onPremisesTagSetList: tagSet.instanceTagGroups.map(tagGroup => {
                return {
                    onPremisesTagGroup: this.tagGroup2TagsArray(tagGroup),
                };
            }),
        };
    }
    tagGroup2TagsArray(tagGroup) {
        const tagsInGroup = [];
        for (const tagKey in tagGroup) {
            if (tagGroup.hasOwnProperty(tagKey)) {
                const tagValues = tagGroup[tagKey];
                if (tagKey.length > 0) {
                    if (tagValues.length > 0) {
                        for (const tagValue of tagValues) {
                            tagsInGroup.push({
                                key: tagKey,
                                value: tagValue,
                                type: 'KEY_AND_VALUE',
                            });
                        }
                    }
                    else {
                        tagsInGroup.push({
                            key: tagKey,
                            type: 'KEY_ONLY',
                        });
                    }
                }
                else {
                    if (tagValues.length > 0) {
                        for (const tagValue of tagValues) {
                            tagsInGroup.push({
                                value: tagValue,
                                type: 'VALUE_ONLY',
                            });
                        }
                    }
                    else {
                        throw new Error('Cannot specify both an empty key and no values for an instance tag filter');
                    }
                }
            }
        }
        return tagsInGroup;
    }
}
exports.ServerDeploymentGroup = ServerDeploymentGroup;
//# sourceMappingURL=data:application/json;base64,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