"use strict";
var _a, _b, _c, _d, _e, _f;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Env = exports.extractContainerPorts = exports.EnvFrom = exports.Cpu = exports.MountPropagation = exports.Container = exports.ImagePullPolicy = exports.EnvValue = exports.ResourceFieldPaths = exports.EnvFieldPaths = exports.ContainerSecurityContext = exports.Protocol = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const k8s = require("./imports/k8s");
const probe = require("./probe");
const utils_1 = require("./utils");
/**
 * Network protocols.
 */
var Protocol;
(function (Protocol) {
    /**
     * TCP.
     */
    Protocol["TCP"] = "TCP";
    /**
     * UDP.
     */
    Protocol["UDP"] = "UDP";
    /**
     * SCTP.
     */
    Protocol["SCTP"] = "SCTP";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
/**
 * Container security attributes and settings.
 */
class ContainerSecurityContext {
    constructor(props = {}) {
        this.ensureNonRoot = props.ensureNonRoot ?? true;
        this.privileged = props.privileged ?? false;
        this.readOnlyRootFilesystem = props.readOnlyRootFilesystem ?? true;
        this.user = props.user ?? 25000;
        this.group = props.group ?? 26000;
        this.allowPrivilegeEscalation = props.allowPrivilegeEscalation ?? false;
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            runAsGroup: this.group,
            runAsUser: this.user,
            runAsNonRoot: this.ensureNonRoot,
            privileged: this.privileged,
            readOnlyRootFilesystem: this.readOnlyRootFilesystem,
            allowPrivilegeEscalation: this.allowPrivilegeEscalation,
        };
    }
}
exports.ContainerSecurityContext = ContainerSecurityContext;
_a = JSII_RTTI_SYMBOL_1;
ContainerSecurityContext[_a] = { fqn: "cdk8s-plus-23.ContainerSecurityContext", version: "2.0.5" };
var EnvFieldPaths;
(function (EnvFieldPaths) {
    /**
     * The name of the pod.
     */
    EnvFieldPaths["POD_NAME"] = "metadata.name";
    /**
     * The namespace of the pod.
     */
    EnvFieldPaths["POD_NAMESPACE"] = "metadata.namespace";
    /**
     * The uid of the pod.
     */
    EnvFieldPaths["POD_UID"] = "metadata.uid";
    /**
     * The labels of the pod.
     */
    EnvFieldPaths["POD_LABEL"] = "metadata.labels";
    /**
     * The annotations of the pod.
     */
    EnvFieldPaths["POD_ANNOTATION"] = "metadata.annotations";
    /**
     * The ipAddress of the pod.
     */
    EnvFieldPaths["POD_IP"] = "status.podIP";
    /**
     * The service account name of the pod.
     */
    EnvFieldPaths["SERVICE_ACCOUNT_NAME"] = "spec.serviceAccountName";
    /**
     * The name of the node.
     */
    EnvFieldPaths["NODE_NAME"] = "spec.nodeName";
    /**
     * The ipAddress of the node.
     */
    EnvFieldPaths["NODE_IP"] = "status.hostIP";
    /**
     * The ipAddresess of the pod.
     */
    EnvFieldPaths["POD_IPS"] = "status.podIPs";
})(EnvFieldPaths = exports.EnvFieldPaths || (exports.EnvFieldPaths = {}));
var ResourceFieldPaths;
(function (ResourceFieldPaths) {
    /**
     * CPU limit of the container.
     */
    ResourceFieldPaths["CPU_LIMIT"] = "limits.cpu";
    /**
     * Memory limit of the container.
     */
    ResourceFieldPaths["MEMORY_LIMIT"] = "limits.memory";
    /**
     * CPU request of the container.
     */
    ResourceFieldPaths["CPU_REQUEST"] = "requests.cpu";
    /**
     * Memory request of the container.
     */
    ResourceFieldPaths["MEMORY_REQUEST"] = "requests.memory";
    /**
     * Ephemeral storage limit of the container.
     */
    ResourceFieldPaths["STORAGE_LIMIT"] = "limits.ephemeral-storage";
    /**
     * Ephemeral storage request of the container.
     */
    ResourceFieldPaths["STORAGE_REQUEST"] = "requests.ephemeral-storage";
})(ResourceFieldPaths = exports.ResourceFieldPaths || (exports.ResourceFieldPaths = {}));
/**
 * Utility class for creating reading env values from various sources.
 */
class EnvValue {
    constructor(value, valueFrom) {
        this.value = value;
        this.valueFrom = valueFrom;
    }
    /**
     * Create a value by reading a specific key inside a config map.
     *
     * @param configMap - The config map.
     * @param key - The key to extract the value from.
     * @param options - Additional options.
     */
    static fromConfigMap(configMap, key, options = {}) {
        const source = {
            configMapKeyRef: {
                name: configMap.name,
                key,
                optional: options.optional,
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Defines an environment value from a secret JSON value.
     *
     * @param secretValue The secret value (secrent + key)
     * @param options Additional options
     */
    static fromSecretValue(secretValue, options = {}) {
        const source = {
            secretKeyRef: {
                name: secretValue.secret.name,
                key: secretValue.key,
                optional: options.optional,
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from the given argument.
     *
     * @param value - The value.
     */
    static fromValue(value) {
        return new EnvValue(value);
    }
    /**
     *
     * Create a value from a field reference.
     *
     * @param fieldPath: The field reference.
     * @param options: Additional options.
     */
    static fromFieldRef(fieldPath, options = {}) {
        let needsKey = false;
        if (fieldPath === EnvFieldPaths.POD_LABEL || fieldPath === EnvFieldPaths.POD_ANNOTATION) {
            needsKey = true;
        }
        if (needsKey && options.key === undefined) {
            throw new Error(`${fieldPath} requires a key`);
        }
        const source = {
            fieldRef: {
                fieldPath: (`${fieldPath}` + (needsKey ? `['${options.key}']` : '')),
                ...(options.apiVersion ? { apiVersion: options.apiVersion } : {}),
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from a resource.
     *
     * @param resource: Resource to select the value from.
     * @param options: Additional options.
     */
    static fromResource(resource, options = {}) {
        const source = {
            resourceFieldRef: {
                resource: `${resource}`,
                ...(options.divisor ? { divisor: k8s.IntOrString.fromString(options.divisor) } : {}),
                ...(options.container ? { containerName: options.container.name } : {}),
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from a key in the current process environment.
     *
     * @param key - The key to read.
     * @param options - Additional options.
     */
    static fromProcess(key, options = {}) {
        const value = process.env[key];
        if (options.required && !value) {
            throw new Error(`Missing ${key} env variable`);
        }
        return EnvValue.fromValue(value);
    }
}
exports.EnvValue = EnvValue;
_b = JSII_RTTI_SYMBOL_1;
EnvValue[_b] = { fqn: "cdk8s-plus-23.EnvValue", version: "2.0.5" };
var ImagePullPolicy;
(function (ImagePullPolicy) {
    /**
     * Every time the kubelet launches a container, the kubelet queries the container image registry
     * to resolve the name to an image digest. If the kubelet has a container image with that exact
     * digest cached locally, the kubelet uses its cached image; otherwise, the kubelet downloads
     * (pulls) the image with the resolved digest, and uses that image to launch the container.
     *
     * Default is Always if ImagePullPolicy is omitted and either the image tag is :latest or
     * the image tag is omitted.
     */
    ImagePullPolicy["ALWAYS"] = "Always";
    /**
     * The image is pulled only if it is not already present locally.
     *
     * Default is IfNotPresent if ImagePullPolicy is omitted and the image tag is present but
     * not :latest
     */
    ImagePullPolicy["IF_NOT_PRESENT"] = "IfNotPresent";
    /**
     * The image is assumed to exist locally. No attempt is made to pull the image.
     */
    ImagePullPolicy["NEVER"] = "Never";
})(ImagePullPolicy = exports.ImagePullPolicy || (exports.ImagePullPolicy = {}));
/**
 * A single application container that you want to run within a pod.
 */
class Container {
    constructor(props) {
        this._ports = [];
        if (props instanceof Container) {
            throw new Error('Attempted to construct a container from a Container object.');
        }
        const defaultResourceSpec = {
            cpu: {
                request: Cpu.millis(1000),
                limit: Cpu.millis(1500),
            },
            memory: {
                request: cdk8s_1.Size.mebibytes(512),
                limit: cdk8s_1.Size.mebibytes(2048),
            },
        };
        if (props.port && props.portNumber) {
            throw new Error('Either \'port\' or \'portNumber\' is allowed. Use \'portNumber\' since \'port\' is deprecated');
        }
        const portNumber = props.portNumber ?? props.port;
        const defaultProbeConfiguration = probe.Probe.fromTcpSocket({ port: portNumber });
        this.name = props.name ?? 'main';
        this.image = props.image;
        this.portNumber = portNumber;
        this._command = props.command;
        this._args = props.args;
        this._readiness = props.readiness;
        this._liveness = props.liveness;
        this._startup = props.startup ?? (this.portNumber ? defaultProbeConfiguration : undefined);
        this._lifecycle = props.lifecycle;
        this.resources = props.resources ?? defaultResourceSpec;
        this.workingDir = props.workingDir;
        this.mounts = props.volumeMounts ?? [];
        this.imagePullPolicy = props.imagePullPolicy ?? ImagePullPolicy.ALWAYS;
        this.securityContext = new ContainerSecurityContext(props.securityContext);
        this.env = new Env(props.envFrom ?? [], props.envVariables ?? {});
        if (this.portNumber) {
            this.addPort({
                number: this.portNumber,
            });
        }
        for (const port of props.ports ?? []) {
            this.addPort(port);
        }
    }
    /**
     * @deprecated - use `portNumber`.
     */
    get port() {
        return this.portNumber;
    }
    /**
     * Ports exposed by this containers.
     * Returns a copy, use `addPort` to modify.
     */
    get ports() {
        return [...this._ports];
    }
    /**
     * Entrypoint array (the command to execute when the container starts).
     * @returns a copy of the entrypoint array, cannot be modified
     */
    get command() {
        return this._command ? [...this._command] : undefined;
    }
    /**
     * Arguments to the entrypoint.
     *
     * @returns a copy of the arguments array, cannot be modified.
     */
    get args() {
        return this._args ? [...this._args] : undefined;
    }
    /**
     * Mount a volume to a specific path so that it is accessible by the container.
     * Every pod that is configured to use this container will autmoatically have access to the volume.
     *
     * @param path - The desired path in the container.
     * @param storage - The storage to mount.
     */
    mount(path, storage, options = {}) {
        this.mounts.push({ path, volume: storage.asVolume(), ...options });
    }
    /**
     * Add a port to expose from this container.
     */
    addPort(port) {
        const names = this._ports.map(p => p.name).filter(x => x);
        const numbers = this._ports.map(p => p.number);
        if (port.name && names.includes(port.name)) {
            throw new Error(`Port with name ${port.name} already exists`);
        }
        if (numbers.includes(port.number)) {
            throw new Error(`Port with number ${port.number} already exists`);
        }
        this._ports.push(port);
    }
    /**
     * @internal
     */
    _toKube() {
        const volumeMounts = [];
        for (const mount of this.mounts) {
            volumeMounts.push({
                name: mount.volume.name,
                mountPath: mount.path,
                readOnly: mount.readOnly,
                mountPropagation: mount.propagation,
                subPath: mount.subPath,
                subPathExpr: mount.subPathExpr,
            });
        }
        const ports = new Array();
        for (const port of this.ports) {
            ports.push({
                containerPort: port.number,
                protocol: port.protocol ? this._portProtocolToKube(port.protocol) : undefined,
                name: port.name,
                hostPort: port.hostPort,
                hostIp: port.hostIp,
            });
        }
        // Resource requests and limits
        const cpuLimit = this.resources?.cpu?.limit?.amount;
        const cpuRequest = this.resources?.cpu?.request?.amount;
        const memoryLimit = this.resources?.memory?.limit;
        const memoryRequest = this.resources?.memory?.request;
        const ephemeralStorageLimit = this.resources?.ephemeralStorage?.limit;
        const ephemeralStorageRequest = this.resources?.ephemeralStorage?.request;
        const limits = {};
        const requests = {};
        if (cpuLimit) {
            limits.cpu = k8s.Quantity.fromString(cpuLimit);
        }
        if (memoryLimit) {
            limits.memory = k8s.Quantity.fromString(memoryLimit.toMebibytes().toString() + 'Mi');
        }
        if (ephemeralStorageLimit) {
            limits['ephemeral-storage'] = k8s.Quantity.fromString(ephemeralStorageLimit.toGibibytes().toString() + 'Gi');
        }
        if (cpuRequest) {
            requests.cpu = k8s.Quantity.fromString(cpuRequest);
        }
        if (memoryRequest) {
            requests.memory = k8s.Quantity.fromString(memoryRequest.toMebibytes().toString() + 'Mi');
        }
        if (ephemeralStorageRequest) {
            requests['ephemeral-storage'] = k8s.Quantity.fromString(ephemeralStorageRequest.toGibibytes().toString() + 'Gi');
        }
        let resourceRequirements = undefined;
        if (Object.keys(limits).length > 0 || Object.keys(requests).length > 0) {
            resourceRequirements = {
                limits: utils_1.undefinedIfEmpty(limits),
                requests: utils_1.undefinedIfEmpty(requests),
            };
        }
        const env = this.env._toKube();
        return {
            name: this.name,
            image: this.image,
            imagePullPolicy: this._imagePullPolicyToKube(this.imagePullPolicy),
            ports: utils_1.undefinedIfEmpty(ports),
            volumeMounts: utils_1.undefinedIfEmpty(volumeMounts),
            command: this.command,
            args: this.args,
            workingDir: this.workingDir,
            env: env.variables,
            envFrom: env.from,
            readinessProbe: this._readiness?._toKube(this),
            livenessProbe: this._liveness?._toKube(this),
            startupProbe: this._startup?._toKube(this),
            lifecycle: this._lifecycle ? {
                postStart: this._lifecycle.postStart?._toKube(this),
                preStop: this._lifecycle.preStop?._toKube(this),
            } : undefined,
            resources: resourceRequirements,
            securityContext: this.securityContext._toKube(),
        };
    }
    _imagePullPolicyToKube(imagePullPolicy) {
        switch (imagePullPolicy) {
            case ImagePullPolicy.ALWAYS:
                return k8s.IoK8SApiCoreV1ContainerImagePullPolicy.ALWAYS;
            case ImagePullPolicy.IF_NOT_PRESENT:
                return k8s.IoK8SApiCoreV1ContainerImagePullPolicy.IF_NOT_PRESENT;
            case ImagePullPolicy.NEVER:
                return k8s.IoK8SApiCoreV1ContainerImagePullPolicy.NEVER;
            default:
                throw new Error(`Unsupported image pull policy: ${imagePullPolicy}`);
        }
    }
    _portProtocolToKube(protocol) {
        switch (protocol) {
            case Protocol.SCTP:
                return k8s.IoK8SApiCoreV1ContainerPortProtocol.SCTP;
            case Protocol.TCP:
                return k8s.IoK8SApiCoreV1ContainerPortProtocol.TCP;
            case Protocol.UDP:
                return k8s.IoK8SApiCoreV1ContainerPortProtocol.UDP;
            default:
                throw new Error(`Unsupported port protocol: ${protocol}`);
        }
    }
}
exports.Container = Container;
_c = JSII_RTTI_SYMBOL_1;
Container[_c] = { fqn: "cdk8s-plus-23.Container", version: "2.0.5" };
var MountPropagation;
(function (MountPropagation) {
    /**
     * This volume mount will not receive any subsequent mounts that are mounted
     * to this volume or any of its subdirectories by the host. In similar
     * fashion, no mounts created by the Container will be visible on the host.
     *
     * This is the default mode.
     *
     * This mode is equal to `private` mount propagation as described in the Linux
     * kernel documentation
     */
    MountPropagation["NONE"] = "None";
    /**
     * This volume mount will receive all subsequent mounts that are mounted to
     * this volume or any of its subdirectories.
     *
     * In other words, if the host mounts anything inside the volume mount, the
     * Container will see it mounted there.
     *
     * Similarly, if any Pod with Bidirectional mount propagation to the same
     * volume mounts anything there, the Container with HostToContainer mount
     * propagation will see it.
     *
     * This mode is equal to `rslave` mount propagation as described in the Linux
     * kernel documentation
     */
    MountPropagation["HOST_TO_CONTAINER"] = "HostToContainer";
    /**
     * This volume mount behaves the same the HostToContainer mount. In addition,
     * all volume mounts created by the Container will be propagated back to the
     * host and to all Containers of all Pods that use the same volume
     *
     * A typical use case for this mode is a Pod with a FlexVolume or CSI driver
     * or a Pod that needs to mount something on the host using a hostPath volume.
     *
     * This mode is equal to `rshared` mount propagation as described in the Linux
     * kernel documentation
     *
     * Caution: Bidirectional mount propagation can be dangerous. It can damage
     * the host operating system and therefore it is allowed only in privileged
     * Containers. Familiarity with Linux kernel behavior is strongly recommended.
     * In addition, any volume mounts created by Containers in Pods must be
     * destroyed (unmounted) by the Containers on termination.
     *
     */
    MountPropagation["BIDIRECTIONAL"] = "Bidirectional";
})(MountPropagation = exports.MountPropagation || (exports.MountPropagation = {}));
/**
 * Represents the amount of CPU.
 * The amount can be passed as millis or units.
 */
class Cpu {
    constructor(amount) {
        this.amount = amount;
    }
    static millis(amount) {
        return new Cpu(amount + 'm');
    }
    static units(amount) {
        return new Cpu(amount.toString());
    }
}
exports.Cpu = Cpu;
_d = JSII_RTTI_SYMBOL_1;
Cpu[_d] = { fqn: "cdk8s-plus-23.Cpu", version: "2.0.5" };
/**
 * A collection of env variables defined in other resources.
 */
class EnvFrom {
    constructor(configMap, prefix, sec) {
        this.configMap = configMap;
        this.prefix = prefix;
        this.sec = sec;
    }
    ;
    /**
     * @internal
     */
    _toKube() {
        return {
            configMapRef: this.configMap ? {
                name: this.configMap.name,
            } : undefined,
            secretRef: this.sec ? {
                name: this.sec.name,
            } : undefined,
            prefix: this.prefix,
        };
    }
}
exports.EnvFrom = EnvFrom;
_e = JSII_RTTI_SYMBOL_1;
EnvFrom[_e] = { fqn: "cdk8s-plus-23.EnvFrom", version: "2.0.5" };
function extractContainerPorts(selector) {
    if (!selector) {
        return [];
    }
    // we don't use instanceof intentionally since it can create
    // cyclic import problems.
    const containers = selector.containers ?? [];
    return containers.flatMap(c => c.ports);
}
exports.extractContainerPorts = extractContainerPorts;
/**
 * Container environment variables.
 */
class Env {
    constructor(sources, variables) {
        this._sources = sources;
        this._variables = variables;
    }
    /**
     * Selects a ConfigMap to populate the environment variables with.
     * The contents of the target ConfigMap's Data field will represent
     * the key-value pairs as environment variables.
     */
    static fromConfigMap(configMap, prefix) {
        return new EnvFrom(configMap, prefix, undefined);
    }
    /**
     * Selects a Secret to populate the environment variables with.
     * The contents of the target Secret's Data field will represent
     * the key-value pairs as environment variables.
     */
    static fromSecret(secr) {
        return new EnvFrom(undefined, undefined, secr);
    }
    /**
     * Add a single variable by name and value.
     * The variable value can come from various dynamic sources such a secrets of config maps.
     * Use `EnvValue.fromXXX` to select sources.
     */
    addVariable(name, value) {
        this._variables[name] = value;
    }
    /**
     * The environment variables for this container.
     * Returns a copy. To add environment variables use `container.env.addVariable()`.
     */
    get variables() {
        return { ...this._variables };
    }
    /**
     * Add a collection of variables by copying from another source.
     * Use `Env.fromXXX` functions to select sources.
     */
    copyFrom(from) {
        this._sources.push(from);
    }
    /**
     * The list of sources used to populate the container environment,
     * in addition to the `variables`.
     *
     * Returns a copy. To add a source use `container.env.copyFrom()`.
     */
    get sources() {
        return [...this._sources];
    }
    renderEnv(env) {
        const result = new Array();
        for (const [name, v] of Object.entries(env)) {
            result.push({
                name,
                value: v.value,
                valueFrom: v.valueFrom,
            });
        }
        return result;
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            from: utils_1.undefinedIfEmpty(this._sources.map(s => s._toKube())),
            variables: utils_1.undefinedIfEmpty(this.renderEnv(this._variables)),
        };
    }
}
exports.Env = Env;
_f = JSII_RTTI_SYMBOL_1;
Env[_f] = { fqn: "cdk8s-plus-23.Env", version: "2.0.5" };
//# sourceMappingURL=data:application/json;base64,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