import contextlib
import warnings
from contextlib import asynccontextmanager
from typing import AsyncIterator

from fastapi.routing import Mount

with warnings.catch_warnings():
    warnings.simplefilter('ignore')

from fastapi import (
    FastAPI,
)

import ide.agenthub  # noqa F401 (we import this to get the agents registered)
from ide import __version__
from ide.server.routes.conversation import app as conversation_api_router
from ide.server.routes.feedback import app as feedback_api_router
from ide.server.routes.files import app as files_api_router
from ide.server.routes.git import app as git_api_router
from ide.server.routes.health import add_health_endpoints
from ide.server.routes.manage_conversations import (
    app as manage_conversation_api_router,
)
from ide.server.routes.mcp import mcp_server
from ide.server.routes.public import app as public_api_router
from ide.server.routes.secrets import app as secrets_router
from ide.server.routes.security import app as security_api_router
from ide.server.routes.settings import app as settings_router
from ide.server.routes.trajectory import app as trajectory_router
from ide.server.shared import conversation_manager

mcp_app = mcp_server.http_app(path='/mcp')


def combine_lifespans(*lifespans):
    # Create a combined lifespan to manage multiple session managers
    @contextlib.asynccontextmanager
    async def combined_lifespan(app):
        async with contextlib.AsyncExitStack() as stack:
            for lifespan in lifespans:
                await stack.enter_async_context(lifespan(app))
            yield

    return combined_lifespan


@asynccontextmanager
async def _lifespan(app: FastAPI) -> AsyncIterator[None]:
    async with conversation_manager:
        yield


app = FastAPI(
    title='IDE',
    description='IDE: Code Less, Make More',
    version=__version__,
    lifespan=combine_lifespans(_lifespan, mcp_app.lifespan),
    routes=[Mount(path='/mcp', app=mcp_app)],
)


app.include_router(public_api_router)
app.include_router(files_api_router)
app.include_router(security_api_router)
app.include_router(feedback_api_router)
app.include_router(conversation_api_router)
app.include_router(manage_conversation_api_router)
app.include_router(settings_router)
app.include_router(secrets_router)
app.include_router(git_api_router)
app.include_router(trajectory_router)
add_health_endpoints(app)
