"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationLoadBalancedCodeDeployedFargateService = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_synthetics_alpha_1 = require("@aws-cdk/aws-synthetics-alpha");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const aws_codedeploy_1 = require("aws-cdk-lib/aws-codedeploy");
const aws_ecs_1 = require("aws-cdk-lib/aws-ecs");
const aws_ecs_patterns_1 = require("aws-cdk-lib/aws-ecs-patterns");
const aws_elasticloadbalancingv2_1 = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const api_canary_1 = require("../api-canary");
const ecs_deployment_1 = require("../ecs-deployment");
/**
 * A Fargate service running on an ECS cluster fronted by an application load balancer and deployed by CodeDeploy.
 */
class ApplicationLoadBalancedCodeDeployedFargateService extends aws_ecs_patterns_1.ApplicationLoadBalancedFargateService {
    /**
     * Constructs a new instance of the ApplicationLoadBalancedCodeDeployedFargateService class.
     */
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            deploymentController: {
                type: aws_ecs_1.DeploymentControllerType.CODE_DEPLOY,
            },
        });
        if (props.deregistrationDelay) {
            this.targetGroup.setAttribute('deregistration_delay.timeout_seconds', props.deregistrationDelay.toSeconds().toString());
        }
        if (props.healthCheck) {
            this.targetGroup.configureHealthCheck(props.healthCheck);
        }
        this.accessLogBucket = props.accessLogBucket ??
            new aws_s3_1.Bucket(this, 'AccessLogBucket', {
                encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
                blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
                enforceSSL: true,
            });
        this.loadBalancer.logAccessLogs(this.accessLogBucket, props.accessLogPrefix);
        const alarms = [];
        if (props.responseTimeAlarmThreshold) {
            const responseTimeAlarm = new aws_cloudwatch_1.Alarm(this, 'ResponseTimeAlarm', {
                metric: this.loadBalancer.metrics.targetResponseTime({
                    period: aws_cdk_lib_1.Duration.minutes(1),
                    statistic: 'p95',
                }),
                evaluationPeriods: 2,
                threshold: props.responseTimeAlarmThreshold.toSeconds(),
                comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            });
            alarms.push(responseTimeAlarm);
        }
        const protocol = props.protocol ?? (props.certificate ? aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS : aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP);
        const testHostName = props.domainName ? props.domainName : this.loadBalancer.loadBalancerDnsName;
        if (props.apiTestSteps?.length) {
            this.apiCanary = new api_canary_1.ApiCanary(this, 'Canary', {
                baseUrl: `${protocol.toLowerCase()}://${testHostName}`,
                durationAlarmThreshold: props.apiCanaryTimeout,
                schedule: aws_synthetics_alpha_1.Schedule.rate(props.apiCanarySchedule ?? aws_cdk_lib_1.Duration.minutes(5)),
                threadCount: props.apiCanaryThreadCount,
                steps: props.apiTestSteps,
            });
            this.apiCanary.node.addDependency(this.service);
            alarms.push(this.apiCanary.successAlarm);
            if (this.apiCanary.durationAlarm) {
                alarms.push(this.apiCanary.durationAlarm);
            }
        }
        this.healthAlarm = new aws_cloudwatch_1.CompositeAlarm(this, 'HealthAlarm', {
            alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(...alarms),
        });
        let testPort;
        if (props.listenerPort) {
            testPort = props.listenerPort + 1;
        }
        else if (protocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP) {
            testPort = 8080;
        }
        else if (protocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
            testPort = 8443;
        }
        else {
            throw new Error('Unable to determine port for test listener');
        }
        let certificates;
        if (props.certificate) {
            certificates = [props.certificate];
        }
        this.testListener = this.loadBalancer.addListener('TestListener', {
            protocol,
            port: testPort,
            open: props.openListener ?? true,
            sslPolicy: props.sslPolicy,
            certificates: certificates,
        });
        this.greenTargetGroup = new aws_elasticloadbalancingv2_1.ApplicationTargetGroup(this, 'GreenTargetGroup', {
            vpc: this.cluster.vpc,
            port: testPort,
            protocol: props.targetProtocol ?? aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
            protocolVersion: props.protocolVersion,
            deregistrationDelay: props.deregistrationDelay,
            healthCheck: props.healthCheck,
            targetType: aws_elasticloadbalancingv2_1.TargetType.IP,
        });
        this.testListener.addTargetGroups('ECS', {
            targetGroups: [this.greenTargetGroup],
        });
        this.application = new aws_codedeploy_1.EcsApplication(this, 'Application');
        this.deploymentGroup = new aws_codedeploy_1.EcsDeploymentGroup(this, 'DeploymentGroup', {
            application: this.application,
            alarms: [this.healthAlarm],
            service: this.service,
            blueGreenDeploymentConfig: {
                blueTargetGroup: this.targetGroup,
                greenTargetGroup: this.greenTargetGroup,
                listener: this.listener,
                testListener: this.testListener,
                terminationWaitTime: props.terminationWaitTime ?? aws_cdk_lib_1.Duration.minutes(10),
            },
            deploymentConfig: props.deploymentConfig ?? aws_codedeploy_1.EcsDeploymentConfig.ALL_AT_ONCE,
            autoRollback: {
                stoppedDeployment: true,
            },
        });
        this.deployment = new ecs_deployment_1.EcsDeployment({
            deploymentGroup: this.deploymentGroup,
            timeout: props.deploymentTimeout ?? aws_cdk_lib_1.Duration.minutes(60),
            targetService: {
                taskDefinition: this.taskDefinition,
                containerName: this.taskDefinition.defaultContainer.containerName,
                containerPort: this.taskDefinition.defaultContainer.containerPort,
            },
        });
    }
    addServiceAsTarget(service) {
        super.addServiceAsTarget(service);
    }
}
exports.ApplicationLoadBalancedCodeDeployedFargateService = ApplicationLoadBalancedCodeDeployedFargateService;
_a = JSII_RTTI_SYMBOL_1;
ApplicationLoadBalancedCodeDeployedFargateService[_a] = { fqn: "@cdklabs/cdk-ecs-codedeploy.ApplicationLoadBalancedCodeDeployedFargateService", version: "0.0.89" };
//# sourceMappingURL=data:application/json;base64,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