"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CognitoToApiGatewayToLambda = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const api = require("aws-cdk-lib/aws-apigateway");
const defaults = require("@aws-solutions-constructs/core");
// Note: To ensure CDKv2 compatibility, keep the import statement for Construct separate
const constructs_1 = require("constructs");
class CognitoToApiGatewayToLambda extends constructs_1.Construct {
    /**
     * @summary Constructs a new instance of the CognitoToApiGatewayToLambda class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {CognitoToApiGatewayToLambdaProps} props - user provided props for the construct
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        defaults.CheckLambdaProps(props);
        // This Construct requires that the auth type be COGNITO regardless of what is specified in the props
        if (props.apiGatewayProps) {
            if (props.apiGatewayProps.defaultMethodOptions === undefined) {
                props.apiGatewayProps.defaultMethodOptions = {
                    authorizationType: api.AuthorizationType.COGNITO,
                };
            }
            else if (props.apiGatewayProps?.defaultMethodOptions.authorizationType === undefined) {
                props.apiGatewayProps.defaultMethodOptions.authorizationType = api.AuthorizationType.COGNITO;
            }
            else if (props.apiGatewayProps?.defaultMethodOptions.authorizationType !== 'COGNITO_USER_POOLS') {
                defaults.printWarning('Overriding Authorization type to be AuthorizationType.COGNITO');
                props.apiGatewayProps.defaultMethodOptions.authorizationType = api.AuthorizationType.COGNITO;
            }
        }
        if (props.apiGatewayProps && (typeof props.apiGatewayProps.proxy !== 'undefined') && (props.apiGatewayProps.proxy === false)) {
            defaults.printWarning('For non-proxy API, addAuthorizers() method must be called after all the resources and methods for API are fully defined. Not calling addAuthorizers() will result in API methods NOT protected by Cognito.');
        }
        this.lambdaFunction = defaults.buildLambdaFunction(this, {
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps
        });
        const globalLambdaRestApiResponse = defaults.GlobalLambdaRestApi(this, this.lambdaFunction, props.apiGatewayProps, props.logGroupProps);
        this.apiGateway = globalLambdaRestApiResponse.api;
        this.apiGatewayCloudWatchRole = globalLambdaRestApiResponse.role;
        this.apiGatewayLogGroup = globalLambdaRestApiResponse.group;
        this.userPool = defaults.buildUserPool(this, props.cognitoUserPoolProps);
        this.userPoolClient = defaults.buildUserPoolClient(this, this.userPool, props.cognitoUserPoolClientProps);
        this.apiGatewayAuthorizer = new api.CfnAuthorizer(this, 'CognitoAuthorizer', {
            restApiId: this.apiGateway.restApiId,
            type: 'COGNITO_USER_POOLS',
            providerArns: [this.userPool.userPoolArn],
            identitySource: "method.request.header.Authorization",
            name: "authorizer"
        });
        this.addAuthorizers();
    }
    addAuthorizers() {
        this.apiGateway.methods.forEach((apiMethod) => {
            // Leave the authorizer NONE for HTTP OPTIONS method to support CORS, for the rest set it to COGNITO
            const child = apiMethod.node.findChild('Resource');
            if (apiMethod.httpMethod === 'OPTIONS') {
                child.addPropertyOverride('AuthorizationType', 'NONE');
            }
            else {
                child.addPropertyOverride('AuthorizationType', 'COGNITO_USER_POOLS');
                child.addPropertyOverride('AuthorizerId', { Ref: this.apiGatewayAuthorizer.logicalId });
            }
        });
    }
}
exports.CognitoToApiGatewayToLambda = CognitoToApiGatewayToLambda;
_a = JSII_RTTI_SYMBOL_1;
CognitoToApiGatewayToLambda[_a] = { fqn: "@aws-solutions-constructs/aws-cognito-apigateway-lambda.CognitoToApiGatewayToLambda", version: "2.53.0" };
//# sourceMappingURL=data:application/json;base64,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