import math 
def footing_size_first_principles(load_kN: float, bearing_capacity_kPa: float, safety_factor: float = 1.5) -> None:
    """
    Calculates the minimum square footing size based on first principles.
    Formula: A = P / q_allow
    
    :param load_kN: Applied vertical load in kN
    :param bearing_capacity_kPa: Allowable soil bearing capacity in kPa
    :param safety_factor: Factor of safety to apply to bearing capacity (default = 1.5)
    """
    if bearing_capacity_kPa <= 0:
        raise ValueError("Bearing capacity must be greater than zero.")
    
    # Adjust allowable bearing capacity by safety factor
    q_safe = bearing_capacity_kPa / safety_factor
    
    # Compute required footing area (A = P / q)
    area_m2 = load_kN / q_safe
    
    # Assume square footing, so side length = sqrt(area)
    side_length_m = area_m2 ** 0.5
    
    # Print results
    print(f"Required footing area: {area_m2:.3f} m²")
    print(f"Approximate footing size: {side_length_m:.3f} m × {side_length_m:.3f} m")

# Example usage
#footing_size_first_principles(40, 100)
if __name__ == "__main__":
    footing_size_first_principles(40, 100)



def get_val_stress_point_load(p=None, z=None, r=None):
    """
    p: Point load in kN
    z: Depth of calculation in meters
    r: Horizontal distance from load application in meters
    """
    if p is None or z is None or r is None:
        raise ValueError("All parameters (p, z, r) must be provided.")

    if (r**2 + z**2) == 0:
        raise ValueError("r and z cannot both be zero.")

    sigma_z = (3 * p * z**3) / (2 * math.pi * (r**2 + z**2)**(5/2))
    
    print(f'Calculated vertical stress at depth {z:.2f}m, horizontal distance {r:.2f}m: {sigma_z:.3f} kN/m²')
    
    return sigma_z

if __name__ == "__main__":
    get_val_stress_point_load(10, z=3, r=3)


#footing under x,y,z ,Mx,My,Mz#
def design_footing(
    Rz, Rx, Ry, Mx, My, Mz,
    fck=30, fyk=500,
    q_allow=200,
    B_init=2.5, L_init=2.5, h=0.6,
    gamma_c=1.5, gamma_s=1.15,
    gamma_G=1.35, gamma_Q=1.5
):
    """
    Designs a square footing under axial force and biaxial moments according to Eurocode 2 and 7.
    
    Inputs:
        Rz, Rx, Ry (kN): Vertical and horizontal reactions
        Mx, My, Mz (kNm): Moments about X, Y, Z axes
        fck (MPa): Concrete strength
        fyk (MPa): Steel yield strength
        q_allow (kN/m²): Allowable bearing pressure
        B_init, L_init, h (m): Initial footing dimensions
        gamma_*: Partial safety factors
    
    Returns:
        dict: Detailed design output
    """
    fcd = fck / gamma_c
    fyd = fyk / gamma_s

    # Design actions
    N_Ed = Rz * gamma_G
    H_Edx = Rx * gamma_Q
    H_Edy = Ry * gamma_Q
    M_Edx = Mx * gamma_Q
    M_Edy = My * gamma_Q
    T_Ed = Mz * gamma_Q

    # Eccentricity
    ex = M_Edy / N_Ed
    ey = M_Edx / N_Ed

    B = B_init
    L = L_init
    d = h - 0.05

    # Check bearing capacity with eccentricity
    B_eff = B - 2 * abs(ey)
    L_eff = L - 2 * abs(ex)

    if B_eff <= 0 or L_eff <= 0:
        return {"Status": "Unstable", "Message": "Eccentricity too high for footing stability."}

    A_eff = B_eff * L_eff
    q_d = N_Ed / A_eff

    if q_d > q_allow:
        return {"Status": "Not OK", "Message": "Bearing capacity exceeded.", "q_d": round(q_d, 2)}

    # Design bending moment
    M_ux = q_d * B * L**2 / 8
    M_uy = q_d * L * B**2 / 8

    # Required steel
    Asx = M_ux * 1e6 / (0.9 * d * 1e3 * fyd)
    Asy = M_uy * 1e6 / (0.9 * d * 1e3 * fyd)

    # Min steel per Eurocode 2 9.2.1.1
    fctm = 2.9 * (fck)**(2/3)
    As_min = max(0.0013 * B * 1e3 * d, 0.26 * fctm / fyk * B * 1e3 * d)

    Asx = max(Asx, As_min)
    Asy = max(Asy, As_min)

    return {
        "Status": "OK",
        "Footing Dimensions (m)": {"Width B": B, "Length L": L, "Depth h": h},
        "Design Loads (kN)": {"Vertical N_Ed": round(N_Ed, 2), "Horz X": round(H_Edx, 2), "Horz Y": round(H_Edy, 2)},
        "Eccentricity (m)": {"ex": round(ex, 3), "ey": round(ey, 3)},
        "Effective Area (m²)": round(A_eff, 3),
        "Bearing Check": f"{round(q_d, 2)} <= {q_allow} kN/m²",
        "Bending Moments (kNm)": {"Mx": round(M_ux, 2), "My": round(M_uy, 2)},
        "Required RFT (mm²/m)": {
            "Longitudinal Asx": round(Asx, 2),
            "Transverse Asy": round(Asy, 2),
            "Minimum As (Eurocode)": round(As_min, 2)
        },
        "Code References": [
            "EN 1992-1-1: Eurocode 2 – Concrete Design",
            "EN 1997-1: Eurocode 7 – Geotechnical Design",
            "EN 1990: Basis of Structural Design"
        ]
    }

# Example usage
if __name__ == "__main__":
    result = design_footing(
        Rz=1000, Rx=50, Ry=30,
        Mx=20, My=40, Mz=10
    )
    import pprint
    pprint.pprint(result, width=100)
