# Copyright (c) 2015, Ecole Polytechnique Federale de Lausanne, Blue Brain Project
# All rights reserved.
#
# This file is part of NeuroM <https://github.com/BlueBrain/NeuroM>
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#     1. Redistributions of source code must retain the above copyright
#        notice, this list of conditions and the following disclaimer.
#     2. Redistributions in binary form must reproduce the above copyright
#        notice, this list of conditions and the following disclaimer in the
#        documentation and/or other materials provided with the distribution.
#     3. Neither the name of the copyright holder nor the names of
#        its contributors may be used to endorse or promote products
#        derived from this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY
# DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
# (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
# LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
# ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
# SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

"""Helper code for neurom applications."""
import logging
import sys
from pathlib import Path

import yaml

from neurom.exceptions import ConfigError

if sys.version_info >= (3, 9):  # pragma: no cover
    import importlib.resources as importlib_resources
else:
    import importlib_resources  # pragma: no cover

EXAMPLE_CHECK_CONFIG = Path(importlib_resources.files("neurom.apps"), "config", "morph_check.yaml")
EXAMPLE_STATS_CONFIG = Path(importlib_resources.files("neurom.apps"), "config", "morph_stats.yaml")

L = logging.getLogger(__name__)


def get_config(config, default_config):
    """Load configuration from file if in config, else use default.

    Args:
        config (str|Path): path to a config file
        default_config (str|Path): path to a default config file

    Returns:
        dict: config dictionary
    """
    if not config:
        L.warning('Using default config %s', default_config)
        config = default_config
    try:
        with open(config, 'r') as f:
            return yaml.load(f, Loader=yaml.SafeLoader)
    except yaml.YAMLError as e:
        raise ConfigError(f'Invalid yaml file: \n {e}') from e
