from __future__ import annotations

import logging
from typing import Any, cast

from sift.runs.v2.runs_pb2 import (
    CreateAutomaticRunAssociationForAssetsRequest,
    CreateRunRequest,
    CreateRunResponse,
    DeleteRunRequest,
    GetRunRequest,
    GetRunResponse,
    ListRunsRequest,
    ListRunsResponse,
    StopRunRequest,
    UpdateRunRequest,
    UpdateRunResponse,
)
from sift.runs.v2.runs_pb2_grpc import RunServiceStub

from sift_client._internal.low_level_wrappers.base import LowLevelClientBase
from sift_client.transport import WithGrpcClient
from sift_client.transport.grpc_transport import GrpcClient
from sift_client.types.run import Run, RunUpdate
from sift_client.util.metadata import metadata_dict_to_proto

# Configure logging
logger = logging.getLogger(__name__)


class RunsLowLevelClient(LowLevelClientBase, WithGrpcClient):
    """
    Low-level client for the RunsAPI.

    This class provides a thin wrapper around the autogenerated bindings for the RunsAPI.
    """

    def __init__(self, grpc_client: GrpcClient):
        """
        Initialize the RunsLowLevelClient.

        Args:
            grpc_client: The gRPC client to use for making API calls.
        """
        super().__init__(grpc_client)

    async def get_run(self, run_id: str) -> Run:
        """
        Get a run by run_id.

        Args:
            run_id: The run ID to get.

        Returns:
            The Run.

        Raises:
            ValueError: If run_id is not provided.
        """
        if not run_id:
            raise ValueError("run_id must be provided")

        request = GetRunRequest(run_id=run_id)
        response = await self._grpc_client.get_stub(RunServiceStub).GetRun(request)
        grpc_run = cast(GetRunResponse, response).run
        return Run._from_proto(grpc_run)

    async def list_runs(
        self,
        *,
        page_size: int | None = None,
        page_token: str | None = None,
        query_filter: str | None = None,
        order_by: str | None = None,
    ) -> tuple[list[Run], str]:
        """
        List runs with optional filtering and pagination.

        Args:
            page_size: The maximum number of runs to return.
            page_token: A page token for pagination.
            query_filter: A CEL filter string.
            order_by: How to order the retrieved runs.

        Returns:
            A tuple of (runs, next_page_token).
        """
        request_kwargs: dict[str, Any] = {}
        if page_size is not None:
            request_kwargs["page_size"] = page_size
        if page_token is not None:
            request_kwargs["page_token"] = page_token
        if query_filter is not None:
            request_kwargs["filter"] = query_filter
        if order_by is not None:
            request_kwargs["order_by"] = order_by

        request = ListRunsRequest(**request_kwargs)
        response = await self._grpc_client.get_stub(RunServiceStub).ListRuns(request)
        response = cast(ListRunsResponse, response)

        runs = [Run._from_proto(run) for run in response.runs]
        return runs, response.next_page_token

    async def list_all_runs(
        self,
        *,
        query_filter: str | None = None,
        order_by: str | None = None,
        max_results: int | None = None,
    ) -> list[Run]:
        """
        List all runs with optional filtering.

        Args:
            query_filter: A CEL filter string.
            order_by: How to order the retrieved runs.
            max_results: Maximum number of results to return.

        Returns:
            A list of all matching runs.
        """
        return await self._handle_pagination(
            self.list_runs,
            kwargs={"query_filter": query_filter},
            order_by=order_by,
            max_results=max_results,
        )

    async def create_run(
        self,
        *,
        name: str,
        description: str,
        tags: list[str] | None = None,
        start_time: Any | None = None,
        stop_time: Any | None = None,
        organization_id: str | None = None,
        client_key: str | None = None,
        metadata: dict[str, str | float | bool] | None = None,
    ) -> Run:
        """
        Create a new run.

        Args:
            name: The name of the run.
            description: The description of the run.
            tags: Tags to associate with the run.
            start_time: The start time of the run.
            stop_time: The stop time of the run.
            organization_id: The organization ID.
            client_key: A unique client key for the run.
            metadata: Metadata values for the run.

        Returns:
            The created Run.
        """
        request_kwargs: dict[str, Any] = {
            "name": name,
            "description": description,
        }

        if tags is not None:
            request_kwargs["tags"] = tags
        if start_time is not None:
            request_kwargs["start_time"] = start_time
        if stop_time is not None:
            request_kwargs["stop_time"] = stop_time
        if organization_id is not None:
            request_kwargs["organization_id"] = organization_id
        if client_key is not None:
            request_kwargs["client_key"] = client_key
        if metadata is not None:
            metadata_proto = metadata_dict_to_proto(metadata)
            request_kwargs["metadata"] = metadata_proto

        request = CreateRunRequest(**request_kwargs)
        response = await self._grpc_client.get_stub(RunServiceStub).CreateRun(request)
        grpc_run = cast(CreateRunResponse, response).run
        return Run._from_proto(grpc_run)

    async def update_run(self, run: Run, update: RunUpdate) -> Run:
        """
        Update an existing run.

        Args:
            run: The run to update.
            update: The updates to apply.

        Returns:
            The updated Run.
        """
        run_proto, field_mask = update.to_proto_with_mask()

        request = UpdateRunRequest(run=run_proto, update_mask=field_mask)
        response = await self._grpc_client.get_stub(RunServiceStub).UpdateRun(request)
        grpc_run = cast(UpdateRunResponse, response).run
        return Run._from_proto(grpc_run)

    async def delete_run(self, run_id: str) -> None:
        """
        Delete a run.

        Args:
            run_id: The ID of the run to delete.

        Raises:
            ValueError: If run_id is not provided.
        """
        if not run_id:
            raise ValueError("run_id must be provided")

        request = DeleteRunRequest(run_id=run_id)
        await self._grpc_client.get_stub(RunServiceStub).DeleteRun(request)

    async def stop_run(self, run_id: str) -> None:
        """
        Stop a run by setting its stop time to the current time.

        Args:
            run_id: The ID of the run to stop.

        Raises:
            ValueError: If run_id is not provided.
        """
        if not run_id:
            raise ValueError("run_id must be provided")

        request = StopRunRequest(run_id=run_id)
        await self._grpc_client.get_stub(RunServiceStub).StopRun(request)

    async def create_automatic_run_association_for_assets(
        self, run_id: str, asset_names: list[str]
    ) -> None:
        """
        Associate assets with a run for automatic data ingestion.

        Args:
            run_id: The ID of the run.
            asset_names: List of asset names to associate.

        Raises:
            ValueError: If run_id or asset_names are not provided.
        """
        if not run_id:
            raise ValueError("run_id must be provided")
        if not asset_names:
            raise ValueError("asset_names must be provided")

        request = CreateAutomaticRunAssociationForAssetsRequest(
            run_id=run_id, asset_names=asset_names
        )
        await self._grpc_client.get_stub(RunServiceStub).CreateAutomaticRunAssociationForAssets(
            request
        )
