"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const events = require("aws-cdk-lib/aws-events");
const ec2 = require("aws-cdk-lib/aws-ec2");
const lib_1 = require("../lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const defaults = require("@aws-solutions-constructs/core");
const xrayPolicyStatement = {
    Action: [
        "xray:PutTraceSegments",
        "xray:PutTelemetryRecords"
    ],
    Effect: "Allow",
    Resource: "*"
};
const defaultEventBusPolicyStatement = {
    Action: "events:PutEvents",
    Effect: "Allow",
    Resource: {
        "Fn::Join": [
            "",
            [
                "arn:",
                {
                    Ref: "AWS::Partition"
                },
                ":events:",
                {
                    Ref: "AWS::Region"
                },
                ":",
                {
                    Ref: "AWS::AccountId"
                },
                ":event-bus/default"
            ]
        ]
    }
};
const vpcConfig = {
    VpcConfig: {
        SecurityGroupIds: [
            {
                "Fn::GetAtt": [
                    "lambdatoeventbridgestackReplaceDefaultSecurityGroupsecuritygroup59EF0706",
                    "GroupId",
                ],
            },
        ],
        SubnetIds: [
            {
                Ref: "VpcisolatedSubnet1SubnetE62B1B9B",
            },
            {
                Ref: "VpcisolatedSubnet2Subnet39217055",
            },
        ],
    },
};
test('Test minimal deployment with new Lambda function', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const construct = new lib_1.LambdaToEventbridge(stack, 'lambda-to-eventbridge-stack', {
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        }
    });
    // Check Props
    const lambdaFunction = construct.lambdaFunction;
    const vpc = construct.vpc;
    const eventBus = construct.eventBus;
    expect(lambdaFunction).toBeDefined();
    expect(vpc).toBeUndefined();
    expect(eventBus).toBeUndefined();
    // Check EVENTBUS_NAME Env variable
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: "1",
                EVENTBUS_NAME: "default"
            }
        }
    });
    // Since using default event bus, there shouldn't be any eventbus
    template.resourceCountIs('AWS::Events::EventBus', 0);
    // Check Lambda Function permissions to access default Event Bridge
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                xrayPolicyStatement,
                defaultEventBusPolicyStatement
            ]
        }
    });
});
test("Confirm CheckVpcProps is being called", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const testVpc = new ec2.Vpc(stack, "test-vpc", {});
    const app = () => {
        // Helper declaration
        new lib_1.LambdaToEventbridge(stack, "lambda-to-eventbridge-stack", {
            lambdaFunctionProps: {
                runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                handler: "index.handler",
                code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            },
            existingVpc: testVpc,
            deployVpc: true,
        });
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide an existingVpc or some combination of deployVpc and vpcProps, but not both.\n');
});
test("Confirm CheckEventBridgeProps is called", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => {
        // Helper declaration
        new lib_1.LambdaToEventbridge(stack, 'lambda-to-eventbridge-stack', {
            lambdaFunctionProps: {
                runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                handler: 'index.handler',
                code: lambda.Code.fromAsset(`${__dirname}/lambda`)
            },
            eventBusProps: { eventBusName: 'test' },
            existingEventBusInterface: new events.EventBus(stack, `new-event-bus`, { eventBusName: 'test' })
        });
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide existingEventBusInterface or eventBusProps, but not both.\n');
});
test('Test deployment w/ existing eventbus', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const existingEventBus = new events.EventBus(stack, 'existing-eventbus', {
        eventBusName: 'customeventbus'
    });
    new lib_1.LambdaToEventbridge(stack, 'lambda-to-eventbridge-stack', {
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        },
        existingEventBusInterface: existingEventBus
    });
    // Check Lambda Permissions
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                xrayPolicyStatement,
                {
                    Action: "events:PutEvents",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "existingeventbus2A99FA49",
                            "Arn"
                        ]
                    }
                }
            ]
        }
    });
    template.hasResourceProperties('AWS::Events::EventBus', {
        Name: `customeventbus`
    });
});
test("Test minimal deployment that deploys a VPC without vpcProps", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    new lib_1.LambdaToEventbridge(stack, "lambda-to-eventbridge-stack", {
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: "index.handler",
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        deployVpc: true,
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", vpcConfig);
    template.hasResourceProperties("AWS::EC2::VPC", {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
    template.hasResourceProperties("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Interface",
    });
    template.resourceCountIs("AWS::EC2::Subnet", 2);
    template.resourceCountIs("AWS::EC2::InternetGateway", 0);
});
test("Test minimal deployment that deploys a VPC w/vpcProps", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    new lib_1.LambdaToEventbridge(stack, "lambda-to-eventbridge-stack", {
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: "index.handler",
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        vpcProps: {
            enableDnsHostnames: false,
            enableDnsSupport: false,
            ipAddresses: ec2.IpAddresses.cidr("192.68.0.0/16"),
        },
        deployVpc: true,
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", vpcConfig);
    template.hasResourceProperties("AWS::EC2::VPC", {
        CidrBlock: "192.68.0.0/16",
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
    template.hasResourceProperties("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Interface",
    });
    template.resourceCountIs("AWS::EC2::Subnet", 2);
    template.resourceCountIs("AWS::EC2::InternetGateway", 0);
});
test("Test minimal deployment with an existing VPC", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const testVpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    new lib_1.LambdaToEventbridge(stack, "lambda-to-eventbridge-stack", {
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: "index.handler",
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        existingVpc: testVpc,
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdatoeventbridgestackReplaceDefaultSecurityGroupsecuritygroup59EF0706",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "testvpcPrivateSubnet1Subnet865FB50A",
                },
                {
                    Ref: "testvpcPrivateSubnet2Subnet23D3396F",
                },
            ],
        },
    });
    template.hasResourceProperties("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Interface",
    });
    // Check Lambda Function permissions to access default Event Bridge
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "ec2:CreateNetworkInterface",
                        "ec2:DescribeNetworkInterfaces",
                        "ec2:DeleteNetworkInterface",
                        "ec2:AssignPrivateIpAddresses",
                        "ec2:UnassignPrivateIpAddresses"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                xrayPolicyStatement,
                defaultEventBusPolicyStatement
            ]
        }
    });
});
test("Test minimal deployment with an existing VPC and existing Lambda function not in a VPC", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const testLambdaFunction = new lambda.Function(stack, 'test-lamba', {
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: "index.handler",
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    });
    const testVpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    const app = () => {
        // buildLambdaFunction should throw an error if the Lambda function is not
        // attached to a VPC
        new lib_1.LambdaToEventbridge(stack, "lambda-to-eventbridge-stack", {
            existingLambdaObj: testLambdaFunction,
            existingVpc: testVpc,
        });
    };
    // Assertion
    expect(app).toThrowError('A Lambda function must be bound to a VPC upon creation, it cannot be added to a VPC in a subsequent construct');
});
test('Test lambda function custom environment variable', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    new lib_1.LambdaToEventbridge(stack, 'lambda-to-eventbridge-stack', {
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        eventBusProps: { eventBusName: 'customeventbus' },
        eventBusEnvironmentVariableName: 'CUSTOM_EVENTBUS_NAME'
    });
    // Check environment variables
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::Function', {
        Handler: 'index.handler',
        Runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_STRING,
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
                CUSTOM_EVENTBUS_NAME: {
                    Ref: "lambdatoeventbridgestackcustomeventbus25825BEF"
                }
            }
        }
    });
    template.hasResourceProperties('AWS::Events::EventBus', {
        Name: `customeventbus`
    });
    // Check lambda permissions to custom event bus
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                xrayPolicyStatement,
                {
                    Action: "events:PutEvents",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "lambdatoeventbridgestackcustomeventbus25825BEF",
                            "Arn"
                        ]
                    }
                }
            ]
        }
    });
});
test('check multiple constructs in a single stack', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        eventBusProps: { eventBusName: 'test' }
    };
    new lib_1.LambdaToEventbridge(stack, 'test-new-lambda-eventbridge1', props);
    new lib_1.LambdaToEventbridge(stack, 'test-new-lambda-eventbridge2', props);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::Events::EventBus', 2);
});
test('check multiple lambda functions publishing to single event bus', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props1 = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        eventBusProps: { eventBusName: 'test' }
    };
    const construct = new lib_1.LambdaToEventbridge(stack, 'test-new-lambda-eventbridge1', props1);
    const props2 = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        existingEventBusInterface: construct.eventBus
    };
    new lib_1.LambdaToEventbridge(stack, 'test-new-lambda-eventbridge2', props2);
    // Make sure only single event bus exists
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::Events::EventBus', 1);
    // Make sure 2 lambda functions exist
    template.resourceCountIs('AWS::Lambda::Function', 2);
    // Check whether lambdas have permission to publish to the event bus
    template.hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                xrayPolicyStatement,
                {
                    Action: "events:PutEvents",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testnewlambdaeventbridge1test13E0B622",
                            "Arn"
                        ]
                    }
                }
            ]
        }
    });
    // Check environment variables
    template.hasResourceProperties('AWS::Lambda::Function', {
        Handler: 'index.handler',
        Runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_STRING,
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
                EVENTBUS_NAME: {
                    Ref: "testnewlambdaeventbridge1test13E0B622"
                }
            }
        }
    });
});
test('Confirm call to CheckLambdaProps', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const lambdaFunction = new lambda.Function(stack, 'a-function', {
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    });
    const props = {
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        existingLambdaObj: lambdaFunction,
    };
    const app = () => {
        new lib_1.LambdaToEventbridge(stack, 'test-construct', props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide lambdaFunctionProps or existingLambdaObj, but not both.\n');
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXdzLWxhbWJkYS1ldmVudGJyaWRnZS50ZXN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiYXdzLWxhbWJkYS1ldmVudGJyaWRnZS50ZXN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQTs7Ozs7Ozs7Ozs7R0FXRzs7QUFFSCxVQUFVO0FBQ1YsNkNBQW9DO0FBQ3BDLGlEQUFpRDtBQUNqRCxpREFBaUQ7QUFDakQsMkNBQTJDO0FBQzNDLGdDQUF1RTtBQUN2RSx1REFBa0Q7QUFDbEQsMkRBQTJEO0FBRTNELE1BQU0sbUJBQW1CLEdBQUc7SUFDMUIsTUFBTSxFQUFFO1FBQ04sdUJBQXVCO1FBQ3ZCLDBCQUEwQjtLQUMzQjtJQUNELE1BQU0sRUFBRSxPQUFPO0lBQ2YsUUFBUSxFQUFFLEdBQUc7Q0FDZCxDQUFDO0FBQ0YsTUFBTSw4QkFBOEIsR0FBRztJQUNyQyxNQUFNLEVBQUUsa0JBQWtCO0lBQzFCLE1BQU0sRUFBRSxPQUFPO0lBQ2YsUUFBUSxFQUFFO1FBQ1IsVUFBVSxFQUFFO1lBQ1YsRUFBRTtZQUNGO2dCQUNFLE1BQU07Z0JBQ047b0JBQ0UsR0FBRyxFQUFFLGdCQUFnQjtpQkFDdEI7Z0JBQ0QsVUFBVTtnQkFDVjtvQkFDRSxHQUFHLEVBQUUsYUFBYTtpQkFDbkI7Z0JBQ0QsR0FBRztnQkFDSDtvQkFDRSxHQUFHLEVBQUUsZ0JBQWdCO2lCQUN0QjtnQkFDRCxvQkFBb0I7YUFDckI7U0FDRjtLQUNGO0NBQ0YsQ0FBQztBQUVGLE1BQU0sU0FBUyxHQUFHO0lBQ2hCLFNBQVMsRUFBRTtRQUNULGdCQUFnQixFQUFFO1lBQ2hCO2dCQUNFLFlBQVksRUFBRTtvQkFDWiwwRUFBMEU7b0JBQzFFLFNBQVM7aUJBQ1Y7YUFDRjtTQUNGO1FBQ0QsU0FBUyxFQUFFO1lBQ1Q7Z0JBQ0UsR0FBRyxFQUFFLGtDQUFrQzthQUN4QztZQUNEO2dCQUNFLEdBQUcsRUFBRSxrQ0FBa0M7YUFDeEM7U0FDRjtLQUNGO0NBQ0YsQ0FBQztBQUVGLElBQUksQ0FBQyxrREFBa0QsRUFBRSxHQUFHLEVBQUU7SUFDNUQsUUFBUTtJQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksbUJBQUssRUFBRSxDQUFDO0lBQzFCLHFCQUFxQjtJQUNyQixNQUFNLFNBQVMsR0FBRyxJQUFJLHlCQUFtQixDQUFDLEtBQUssRUFBRSw2QkFBNkIsRUFBRTtRQUM5RSxtQkFBbUIsRUFBRTtZQUNuQixPQUFPLEVBQUUsUUFBUSxDQUFDLHFDQUFxQztZQUN2RCxPQUFPLEVBQUUsZUFBZTtZQUN4QixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztTQUNuRDtLQUNGLENBQUMsQ0FBQztJQUVILGNBQWM7SUFDZCxNQUFNLGNBQWMsR0FBRyxTQUFTLENBQUMsY0FBYyxDQUFDO0lBQ2hELE1BQU0sR0FBRyxHQUFHLFNBQVMsQ0FBQyxHQUFHLENBQUM7SUFDMUIsTUFBTSxRQUFRLEdBQUcsU0FBUyxDQUFDLFFBQVEsQ0FBQztJQUNwQyxNQUFNLENBQUMsY0FBYyxDQUFDLENBQUMsV0FBVyxFQUFFLENBQUM7SUFDckMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLGFBQWEsRUFBRSxDQUFDO0lBQzVCLE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQyxhQUFhLEVBQUUsQ0FBQztJQUVqQyxtQ0FBbUM7SUFDbkMsTUFBTSxRQUFRLEdBQUcscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDM0MsUUFBUSxDQUFDLHFCQUFxQixDQUFDLHVCQUF1QixFQUFFO1FBQ3RELFdBQVcsRUFBRTtZQUNYLFNBQVMsRUFBRTtnQkFDVCxtQ0FBbUMsRUFBRSxHQUFHO2dCQUN4QyxhQUFhLEVBQUUsU0FBUzthQUN6QjtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsaUVBQWlFO0lBQ2pFLFFBQVEsQ0FBQyxlQUFlLENBQUMsdUJBQXVCLEVBQUUsQ0FBQyxDQUFDLENBQUM7SUFFckQsbUVBQW1FO0lBQ25FLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyxrQkFBa0IsRUFBRTtRQUNqRCxjQUFjLEVBQUU7WUFDZCxTQUFTLEVBQUU7Z0JBQ1QsbUJBQW1CO2dCQUNuQiw4QkFBOEI7YUFDL0I7U0FDRjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHVDQUF1QyxFQUFFLEdBQUcsRUFBRTtJQUNqRCxRQUFRO0lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFFMUIsTUFBTSxPQUFPLEdBQUcsSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFFbkQsTUFBTSxHQUFHLEdBQUcsR0FBRyxFQUFFO1FBQ2YscUJBQXFCO1FBQ3JCLElBQUkseUJBQW1CLENBQUMsS0FBSyxFQUFFLDZCQUE2QixFQUFFO1lBQzVELG1CQUFtQixFQUFFO2dCQUNuQixPQUFPLEVBQUUsUUFBUSxDQUFDLHFDQUFxQztnQkFDdkQsT0FBTyxFQUFFLGVBQWU7Z0JBQ3hCLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLFNBQVMsU0FBUyxDQUFDO2FBQ25EO1lBQ0QsV0FBVyxFQUFFLE9BQU87WUFDcEIsU0FBUyxFQUFFLElBQUk7U0FDaEIsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDO0lBQ0YsWUFBWTtJQUNaLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxZQUFZLENBQUMsc0dBQXNHLENBQUMsQ0FBQztBQUNuSSxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyx5Q0FBeUMsRUFBRSxHQUFHLEVBQUU7SUFDbkQsUUFBUTtJQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksbUJBQUssRUFBRSxDQUFDO0lBRTFCLE1BQU0sR0FBRyxHQUFHLEdBQUcsRUFBRTtRQUNmLHFCQUFxQjtRQUNyQixJQUFJLHlCQUFtQixDQUFDLEtBQUssRUFBRSw2QkFBNkIsRUFBRTtZQUM1RCxtQkFBbUIsRUFBRTtnQkFDbkIsT0FBTyxFQUFFLFFBQVEsQ0FBQyxxQ0FBcUM7Z0JBQ3ZELE9BQU8sRUFBRSxlQUFlO2dCQUN4QixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQzthQUNuRDtZQUNELGFBQWEsRUFBRSxFQUFFLFlBQVksRUFBRSxNQUFNLEVBQUU7WUFDdkMseUJBQXlCLEVBQUUsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxlQUFlLEVBQUUsRUFBRyxZQUFZLEVBQUUsTUFBTSxFQUFHLENBQUM7U0FDbkcsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDO0lBQ0YsWUFBWTtJQUNaLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxZQUFZLENBQUMsb0ZBQW9GLENBQUMsQ0FBQztBQUNqSCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxzQ0FBc0MsRUFBRSxHQUFHLEVBQUU7SUFDaEQsUUFBUTtJQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksbUJBQUssRUFBRSxDQUFDO0lBQzFCLHFCQUFxQjtJQUNyQixNQUFNLGdCQUFnQixHQUFHLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsbUJBQW1CLEVBQUU7UUFDdkUsWUFBWSxFQUFFLGdCQUFnQjtLQUMvQixDQUFDLENBQUM7SUFFSCxJQUFJLHlCQUFtQixDQUFDLEtBQUssRUFBRSw2QkFBNkIsRUFBRTtRQUM1RCxtQkFBbUIsRUFBRTtZQUNuQixPQUFPLEVBQUUsUUFBUSxDQUFDLHFDQUFxQztZQUN2RCxPQUFPLEVBQUUsZUFBZTtZQUN4QixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztTQUNuRDtRQUNELHlCQUF5QixFQUFFLGdCQUFnQjtLQUM1QyxDQUFDLENBQUM7SUFFSCwyQkFBMkI7SUFDM0IsTUFBTSxRQUFRLEdBQUcscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDM0MsUUFBUSxDQUFDLHFCQUFxQixDQUFDLGtCQUFrQixFQUFFO1FBQ2pELGNBQWMsRUFBRTtZQUNkLFNBQVMsRUFBRTtnQkFDVCxtQkFBbUI7Z0JBQ25CO29CQUNFLE1BQU0sRUFBRSxrQkFBa0I7b0JBQzFCLE1BQU0sRUFBRSxPQUFPO29CQUNmLFFBQVEsRUFBRTt3QkFDUixZQUFZLEVBQUU7NEJBQ1osMEJBQTBCOzRCQUMxQixLQUFLO3lCQUNOO3FCQUNGO2lCQUNGO2FBQ0Y7U0FDRjtLQUNGLENBQUMsQ0FBQztJQUNILFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyx1QkFBdUIsRUFBRTtRQUN0RCxJQUFJLEVBQUUsZ0JBQWdCO0tBQ3ZCLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDZEQUE2RCxFQUFFLEdBQUcsRUFBRTtJQUN2RSxRQUFRO0lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFDMUIscUJBQXFCO0lBQ3JCLElBQUkseUJBQW1CLENBQUMsS0FBSyxFQUFFLDZCQUE2QixFQUFFO1FBQzVELG1CQUFtQixFQUFFO1lBQ25CLE9BQU8sRUFBRSxRQUFRLENBQUMscUNBQXFDO1lBQ3ZELE9BQU8sRUFBRSxlQUFlO1lBQ3hCLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLFNBQVMsU0FBUyxDQUFDO1NBQ25EO1FBQ0QsU0FBUyxFQUFFLElBQUk7S0FDaEIsQ0FBQyxDQUFDO0lBRUgsTUFBTSxRQUFRLEdBQUcscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDM0MsUUFBUSxDQUFDLHFCQUFxQixDQUFDLHVCQUF1QixFQUFFLFNBQVMsQ0FBQyxDQUFDO0lBRW5FLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyxlQUFlLEVBQUU7UUFDOUMsa0JBQWtCLEVBQUUsSUFBSTtRQUN4QixnQkFBZ0IsRUFBRSxJQUFJO0tBQ3ZCLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyx1QkFBdUIsRUFBRTtRQUN0RCxlQUFlLEVBQUUsV0FBVztLQUM3QixDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMsZUFBZSxDQUFDLGtCQUFrQixFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQ2hELFFBQVEsQ0FBQyxlQUFlLENBQUMsMkJBQTJCLEVBQUUsQ0FBQyxDQUFDLENBQUM7QUFDM0QsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsdURBQXVELEVBQUUsR0FBRyxFQUFFO0lBQ2pFLFFBQVE7SUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLG1CQUFLLEVBQUUsQ0FBQztJQUMxQixxQkFBcUI7SUFDckIsSUFBSSx5QkFBbUIsQ0FBQyxLQUFLLEVBQUUsNkJBQTZCLEVBQUU7UUFDNUQsbUJBQW1CLEVBQUU7WUFDbkIsT0FBTyxFQUFFLFFBQVEsQ0FBQyxxQ0FBcUM7WUFDdkQsT0FBTyxFQUFFLGVBQWU7WUFDeEIsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsU0FBUyxTQUFTLENBQUM7U0FDbkQ7UUFDRCxRQUFRLEVBQUU7WUFDUixrQkFBa0IsRUFBRSxLQUFLO1lBQ3pCLGdCQUFnQixFQUFFLEtBQUs7WUFDdkIsV0FBVyxFQUFFLEdBQUcsQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQztTQUNuRDtRQUNELFNBQVMsRUFBRSxJQUFJO0tBQ2hCLENBQUMsQ0FBQztJQUVILE1BQU0sUUFBUSxHQUFHLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzNDLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyx1QkFBdUIsRUFBRSxTQUFTLENBQUMsQ0FBQztJQUVuRSxRQUFRLENBQUMscUJBQXFCLENBQUMsZUFBZSxFQUFFO1FBQzlDLFNBQVMsRUFBRSxlQUFlO1FBQzFCLGtCQUFrQixFQUFFLElBQUk7UUFDeEIsZ0JBQWdCLEVBQUUsSUFBSTtLQUN2QixDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMscUJBQXFCLENBQUMsdUJBQXVCLEVBQUU7UUFDdEQsZUFBZSxFQUFFLFdBQVc7S0FDN0IsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLGVBQWUsQ0FBQyxrQkFBa0IsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUNoRCxRQUFRLENBQUMsZUFBZSxDQUFDLDJCQUEyQixFQUFFLENBQUMsQ0FBQyxDQUFDO0FBQzNELENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDhDQUE4QyxFQUFFLEdBQUcsRUFBRTtJQUN4RCxRQUFRO0lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFFMUIsTUFBTSxPQUFPLEdBQUcsSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFFbkQscUJBQXFCO0lBQ3JCLElBQUkseUJBQW1CLENBQUMsS0FBSyxFQUFFLDZCQUE2QixFQUFFO1FBQzVELG1CQUFtQixFQUFFO1lBQ25CLE9BQU8sRUFBRSxRQUFRLENBQUMscUNBQXFDO1lBQ3ZELE9BQU8sRUFBRSxlQUFlO1lBQ3hCLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLFNBQVMsU0FBUyxDQUFDO1NBQ25EO1FBQ0QsV0FBVyxFQUFFLE9BQU87S0FDckIsQ0FBQyxDQUFDO0lBRUgsTUFBTSxRQUFRLEdBQUcscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDM0MsUUFBUSxDQUFDLHFCQUFxQixDQUFDLHVCQUF1QixFQUFFO1FBQ3RELFNBQVMsRUFBRTtZQUNULGdCQUFnQixFQUFFO2dCQUNoQjtvQkFDRSxZQUFZLEVBQUU7d0JBQ1osMEVBQTBFO3dCQUMxRSxTQUFTO3FCQUNWO2lCQUNGO2FBQ0Y7WUFDRCxTQUFTLEVBQUU7Z0JBQ1Q7b0JBQ0UsR0FBRyxFQUFFLHFDQUFxQztpQkFDM0M7Z0JBQ0Q7b0JBQ0UsR0FBRyxFQUFFLHFDQUFxQztpQkFDM0M7YUFDRjtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLHFCQUFxQixDQUFDLHVCQUF1QixFQUFFO1FBQ3RELGVBQWUsRUFBRSxXQUFXO0tBQzdCLENBQUMsQ0FBQztJQUVILG1FQUFtRTtJQUNuRSxRQUFRLENBQUMscUJBQXFCLENBQUMsa0JBQWtCLEVBQUU7UUFDakQsY0FBYyxFQUFFO1lBQ2QsU0FBUyxFQUFFO2dCQUNUO29CQUNFLE1BQU0sRUFBRTt3QkFDTiw0QkFBNEI7d0JBQzVCLCtCQUErQjt3QkFDL0IsNEJBQTRCO3dCQUM1Qiw4QkFBOEI7d0JBQzlCLGdDQUFnQztxQkFDakM7b0JBQ0QsTUFBTSxFQUFFLE9BQU87b0JBQ2YsUUFBUSxFQUFFLEdBQUc7aUJBQ2Q7Z0JBQ0QsbUJBQW1CO2dCQUNuQiw4QkFBOEI7YUFDL0I7U0FDRjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHdGQUF3RixFQUFFLEdBQUcsRUFBRTtJQUNsRyxRQUFRO0lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFFMUIsTUFBTSxrQkFBa0IsR0FBRyxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLFlBQVksRUFBRTtRQUNsRSxPQUFPLEVBQUUsUUFBUSxDQUFDLHFDQUFxQztRQUN2RCxPQUFPLEVBQUUsZUFBZTtRQUN4QixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztLQUNuRCxDQUFDLENBQUM7SUFFSCxNQUFNLE9BQU8sR0FBRyxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRSxFQUFFLENBQUMsQ0FBQztJQUVuRCxxQkFBcUI7SUFDckIsTUFBTSxHQUFHLEdBQUcsR0FBRyxFQUFFO1FBQ2YsMEVBQTBFO1FBQzFFLG9CQUFvQjtRQUNwQixJQUFJLHlCQUFtQixDQUFDLEtBQUssRUFBRSw2QkFBNkIsRUFBRTtZQUM1RCxpQkFBaUIsRUFBRSxrQkFBa0I7WUFDckMsV0FBVyxFQUFFLE9BQU87U0FDckIsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDO0lBRUYsWUFBWTtJQUNaLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxZQUFZLENBQUMsK0dBQStHLENBQUMsQ0FBQztBQUU1SSxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxrREFBa0QsRUFBRSxHQUFHLEVBQUU7SUFDNUQsUUFBUTtJQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksbUJBQUssRUFBRSxDQUFDO0lBRTFCLHFCQUFxQjtJQUNyQixJQUFJLHlCQUFtQixDQUFDLEtBQUssRUFBRSw2QkFBNkIsRUFBRTtRQUM1RCxtQkFBbUIsRUFBRTtZQUNuQixPQUFPLEVBQUUsUUFBUSxDQUFDLHFDQUFxQztZQUN2RCxPQUFPLEVBQUUsZUFBZTtZQUN4QixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztTQUNuRDtRQUNELGFBQWEsRUFBRSxFQUFDLFlBQVksRUFBRSxnQkFBZ0IsRUFBQztRQUMvQywrQkFBK0IsRUFBRSxzQkFBc0I7S0FDeEQsQ0FBQyxDQUFDO0lBRUgsOEJBQThCO0lBQzlCLE1BQU0sUUFBUSxHQUFHLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzNDLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyx1QkFBdUIsRUFBRTtRQUN0RCxPQUFPLEVBQUUsZUFBZTtRQUN4QixPQUFPLEVBQUcsUUFBUSxDQUFDLG9DQUFvQztRQUN2RCxXQUFXLEVBQUU7WUFDWCxTQUFTLEVBQUU7Z0JBQ1QsbUNBQW1DLEVBQUUsR0FBRztnQkFDeEMsb0JBQW9CLEVBQUU7b0JBQ3BCLEdBQUcsRUFBRSxnREFBZ0Q7aUJBQ3REO2FBQ0Y7U0FDRjtLQUNGLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyx1QkFBdUIsRUFBRTtRQUN0RCxJQUFJLEVBQUUsZ0JBQWdCO0tBQ3ZCLENBQUMsQ0FBQztJQUVILCtDQUErQztJQUMvQyxRQUFRLENBQUMscUJBQXFCLENBQUMsa0JBQWtCLEVBQUU7UUFDakQsY0FBYyxFQUFFO1lBQ2QsU0FBUyxFQUFFO2dCQUNULG1CQUFtQjtnQkFDbkI7b0JBQ0UsTUFBTSxFQUFFLGtCQUFrQjtvQkFDMUIsTUFBTSxFQUFFLE9BQU87b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLFlBQVksRUFBRTs0QkFDWixnREFBZ0Q7NEJBQ2hELEtBQUs7eUJBQ047cUJBQ0Y7aUJBQ0Y7YUFDRjtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsNkNBQTZDLEVBQUUsR0FBRyxFQUFFO0lBQ3ZELE1BQU0sS0FBSyxHQUFHLElBQUksbUJBQUssRUFBRSxDQUFDO0lBRTFCLE1BQU0sS0FBSyxHQUE2QjtRQUN0QyxtQkFBbUIsRUFBRTtZQUNuQixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztZQUNsRCxPQUFPLEVBQUUsUUFBUSxDQUFDLHFDQUFxQztZQUN2RCxPQUFPLEVBQUUsZUFBZTtTQUN6QjtRQUNELGFBQWEsRUFBRSxFQUFFLFlBQVksRUFBRSxNQUFNLEVBQUU7S0FDeEMsQ0FBQztJQUNGLElBQUkseUJBQW1CLENBQUMsS0FBSyxFQUFFLDhCQUE4QixFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ3RFLElBQUkseUJBQW1CLENBQUMsS0FBSyxFQUFFLDhCQUE4QixFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ3RFLE1BQU0sUUFBUSxHQUFHLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzNDLFFBQVEsQ0FBQyxlQUFlLENBQUMsdUJBQXVCLEVBQUUsQ0FBQyxDQUFDLENBQUM7QUFDdkQsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsZ0VBQWdFLEVBQUUsR0FBRyxFQUFFO0lBQzFFLE1BQU0sS0FBSyxHQUFHLElBQUksbUJBQUssRUFBRSxDQUFDO0lBRTFCLE1BQU0sTUFBTSxHQUE2QjtRQUN2QyxtQkFBbUIsRUFBRTtZQUNuQixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztZQUNsRCxPQUFPLEVBQUUsUUFBUSxDQUFDLHFDQUFxQztZQUN2RCxPQUFPLEVBQUUsZUFBZTtTQUN6QjtRQUNELGFBQWEsRUFBRSxFQUFFLFlBQVksRUFBRSxNQUFNLEVBQUU7S0FDeEMsQ0FBQztJQUNGLE1BQU0sU0FBUyxHQUFHLElBQUkseUJBQW1CLENBQUMsS0FBSyxFQUFFLDhCQUE4QixFQUFFLE1BQU0sQ0FBQyxDQUFDO0lBRXpGLE1BQU0sTUFBTSxHQUE2QjtRQUN2QyxtQkFBbUIsRUFBRTtZQUNuQixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztZQUNsRCxPQUFPLEVBQUUsUUFBUSxDQUFDLHFDQUFxQztZQUN2RCxPQUFPLEVBQUUsZUFBZTtTQUN6QjtRQUNELHlCQUF5QixFQUFFLFNBQVMsQ0FBQyxRQUFRO0tBQzlDLENBQUM7SUFDRixJQUFJLHlCQUFtQixDQUFDLEtBQUssRUFBRSw4QkFBOEIsRUFBRSxNQUFNLENBQUMsQ0FBQztJQUV2RSx5Q0FBeUM7SUFDekMsTUFBTSxRQUFRLEdBQUcscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDM0MsUUFBUSxDQUFDLGVBQWUsQ0FBQyx1QkFBdUIsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUVyRCxxQ0FBcUM7SUFDckMsUUFBUSxDQUFDLGVBQWUsQ0FBQyx1QkFBdUIsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUVyRCxvRUFBb0U7SUFDcEUsUUFBUSxDQUFDLHFCQUFxQixDQUFDLGtCQUFrQixFQUFFO1FBQ2pELGNBQWMsRUFBRTtZQUNkLFNBQVMsRUFBRTtnQkFDVCxtQkFBbUI7Z0JBQ25CO29CQUNFLE1BQU0sRUFBRSxrQkFBa0I7b0JBQzFCLE1BQU0sRUFBRSxPQUFPO29CQUNmLFFBQVEsRUFBRTt3QkFDUixZQUFZLEVBQUU7NEJBQ1osdUNBQXVDOzRCQUN2QyxLQUFLO3lCQUNOO3FCQUNGO2lCQUNGO2FBQ0Y7U0FDRjtLQUNGLENBQUMsQ0FBQztJQUVILDhCQUE4QjtJQUM5QixRQUFRLENBQUMscUJBQXFCLENBQUMsdUJBQXVCLEVBQUU7UUFDdEQsT0FBTyxFQUFFLGVBQWU7UUFDeEIsT0FBTyxFQUFHLFFBQVEsQ0FBQyxvQ0FBb0M7UUFDdkQsV0FBVyxFQUFFO1lBQ1gsU0FBUyxFQUFFO2dCQUNULG1DQUFtQyxFQUFFLEdBQUc7Z0JBQ3hDLGFBQWEsRUFBRTtvQkFDYixHQUFHLEVBQUUsdUNBQXVDO2lCQUM3QzthQUNGO1NBQ0Y7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxrQ0FBa0MsRUFBRSxHQUFHLEVBQUU7SUFDNUMsZ0JBQWdCO0lBQ2hCLE1BQU0sS0FBSyxHQUFHLElBQUksbUJBQUssRUFBRSxDQUFDO0lBQzFCLE1BQU0sY0FBYyxHQUFHLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFO1FBQzlELE9BQU8sRUFBRSxRQUFRLENBQUMscUNBQXFDO1FBQ3ZELE9BQU8sRUFBRSxlQUFlO1FBQ3hCLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLFNBQVMsU0FBUyxDQUFDO0tBQ25ELENBQUMsQ0FBQztJQUVILE1BQU0sS0FBSyxHQUE2QjtRQUN0QyxtQkFBbUIsRUFBRTtZQUNuQixPQUFPLEVBQUUsUUFBUSxDQUFDLHFDQUFxQztZQUN2RCxPQUFPLEVBQUUsZUFBZTtZQUN4QixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztTQUNuRDtRQUNELGlCQUFpQixFQUFFLGNBQWM7S0FDbEMsQ0FBQztJQUNGLE1BQU0sR0FBRyxHQUFHLEdBQUcsRUFBRTtRQUNmLElBQUkseUJBQW1CLENBQUMsS0FBSyxFQUFFLGdCQUFnQixFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQzFELENBQUMsQ0FBQztJQUNGLFlBQVk7SUFDWixNQUFNLENBQUMsR0FBRyxDQUFDLENBQUMsWUFBWSxDQUFDLGtGQUFrRixDQUFDLENBQUM7QUFDL0csQ0FBQyxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqICBDb3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqXG4gKiAgTGljZW5zZWQgdW5kZXIgdGhlIEFwYWNoZSBMaWNlbnNlLCBWZXJzaW9uIDIuMCAodGhlIFwiTGljZW5zZVwiKS4gWW91IG1heSBub3QgdXNlIHRoaXMgZmlsZSBleGNlcHQgaW4gY29tcGxpYW5jZVxuICogIHdpdGggdGhlIExpY2Vuc2UuIEEgY29weSBvZiB0aGUgTGljZW5zZSBpcyBsb2NhdGVkIGF0XG4gKlxuICogICAgICBodHRwOi8vd3d3LmFwYWNoZS5vcmcvbGljZW5zZXMvTElDRU5TRS0yLjBcbiAqXG4gKiAgb3IgaW4gdGhlICdsaWNlbnNlJyBmaWxlIGFjY29tcGFueWluZyB0aGlzIGZpbGUuIFRoaXMgZmlsZSBpcyBkaXN0cmlidXRlZCBvbiBhbiAnQVMgSVMnIEJBU0lTLCBXSVRIT1VUIFdBUlJBTlRJRVNcbiAqICBPUiBDT05ESVRJT05TIE9GIEFOWSBLSU5ELCBleHByZXNzIG9yIGltcGxpZWQuIFNlZSB0aGUgTGljZW5zZSBmb3IgdGhlIHNwZWNpZmljIGxhbmd1YWdlIGdvdmVybmluZyBwZXJtaXNzaW9uc1xuICogIGFuZCBsaW1pdGF0aW9ucyB1bmRlciB0aGUgTGljZW5zZS5cbiAqL1xuXG4vLyBJbXBvcnRzXG5pbXBvcnQgeyBTdGFjayB9IGZyb20gXCJhd3MtY2RrLWxpYlwiO1xuaW1wb3J0ICogYXMgbGFtYmRhIGZyb20gXCJhd3MtY2RrLWxpYi9hd3MtbGFtYmRhXCI7XG5pbXBvcnQgKiBhcyBldmVudHMgZnJvbSBcImF3cy1jZGstbGliL2F3cy1ldmVudHNcIjtcbmltcG9ydCAqIGFzIGVjMiBmcm9tIFwiYXdzLWNkay1saWIvYXdzLWVjMlwiO1xuaW1wb3J0IHsgTGFtYmRhVG9FdmVudGJyaWRnZSwgTGFtYmRhVG9FdmVudGJyaWRnZVByb3BzIH0gZnJvbSAnLi4vbGliJztcbmltcG9ydCB7IFRlbXBsYXRlIH0gZnJvbSAnYXdzLWNkay1saWIvYXNzZXJ0aW9ucyc7XG5pbXBvcnQgKiBhcyBkZWZhdWx0cyBmcm9tIFwiQGF3cy1zb2x1dGlvbnMtY29uc3RydWN0cy9jb3JlXCI7XG5cbmNvbnN0IHhyYXlQb2xpY3lTdGF0ZW1lbnQgPSB7XG4gIEFjdGlvbjogW1xuICAgIFwieHJheTpQdXRUcmFjZVNlZ21lbnRzXCIsXG4gICAgXCJ4cmF5OlB1dFRlbGVtZXRyeVJlY29yZHNcIlxuICBdLFxuICBFZmZlY3Q6IFwiQWxsb3dcIixcbiAgUmVzb3VyY2U6IFwiKlwiXG59O1xuY29uc3QgZGVmYXVsdEV2ZW50QnVzUG9saWN5U3RhdGVtZW50ID0ge1xuICBBY3Rpb246IFwiZXZlbnRzOlB1dEV2ZW50c1wiLFxuICBFZmZlY3Q6IFwiQWxsb3dcIixcbiAgUmVzb3VyY2U6IHtcbiAgICBcIkZuOjpKb2luXCI6IFtcbiAgICAgIFwiXCIsXG4gICAgICBbXG4gICAgICAgIFwiYXJuOlwiLFxuICAgICAgICB7XG4gICAgICAgICAgUmVmOiBcIkFXUzo6UGFydGl0aW9uXCJcbiAgICAgICAgfSxcbiAgICAgICAgXCI6ZXZlbnRzOlwiLFxuICAgICAgICB7XG4gICAgICAgICAgUmVmOiBcIkFXUzo6UmVnaW9uXCJcbiAgICAgICAgfSxcbiAgICAgICAgXCI6XCIsXG4gICAgICAgIHtcbiAgICAgICAgICBSZWY6IFwiQVdTOjpBY2NvdW50SWRcIlxuICAgICAgICB9LFxuICAgICAgICBcIjpldmVudC1idXMvZGVmYXVsdFwiXG4gICAgICBdXG4gICAgXVxuICB9XG59O1xuXG5jb25zdCB2cGNDb25maWcgPSB7XG4gIFZwY0NvbmZpZzoge1xuICAgIFNlY3VyaXR5R3JvdXBJZHM6IFtcbiAgICAgIHtcbiAgICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgICBcImxhbWJkYXRvZXZlbnRicmlkZ2VzdGFja1JlcGxhY2VEZWZhdWx0U2VjdXJpdHlHcm91cHNlY3VyaXR5Z3JvdXA1OUVGMDcwNlwiLFxuICAgICAgICAgIFwiR3JvdXBJZFwiLFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICBdLFxuICAgIFN1Ym5ldElkczogW1xuICAgICAge1xuICAgICAgICBSZWY6IFwiVnBjaXNvbGF0ZWRTdWJuZXQxU3VibmV0RTYyQjFCOUJcIixcbiAgICAgIH0sXG4gICAgICB7XG4gICAgICAgIFJlZjogXCJWcGNpc29sYXRlZFN1Ym5ldDJTdWJuZXQzOTIxNzA1NVwiLFxuICAgICAgfSxcbiAgICBdLFxuICB9LFxufTtcblxudGVzdCgnVGVzdCBtaW5pbWFsIGRlcGxveW1lbnQgd2l0aCBuZXcgTGFtYmRhIGZ1bmN0aW9uJywgKCkgPT4ge1xuICAvLyBTdGFja1xuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAvLyBIZWxwZXIgZGVjbGFyYXRpb25cbiAgY29uc3QgY29uc3RydWN0ID0gbmV3IExhbWJkYVRvRXZlbnRicmlkZ2Uoc3RhY2ssICdsYW1iZGEtdG8tZXZlbnRicmlkZ2Utc3RhY2snLCB7XG4gICAgbGFtYmRhRnVuY3Rpb25Qcm9wczoge1xuICAgICAgcnVudGltZTogZGVmYXVsdHMuQ09NTUVSQ0lBTF9SRUdJT05fTEFNQkRBX05PREVfUlVOVElNRSxcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJyxcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApXG4gICAgfVxuICB9KTtcblxuICAvLyBDaGVjayBQcm9wc1xuICBjb25zdCBsYW1iZGFGdW5jdGlvbiA9IGNvbnN0cnVjdC5sYW1iZGFGdW5jdGlvbjtcbiAgY29uc3QgdnBjID0gY29uc3RydWN0LnZwYztcbiAgY29uc3QgZXZlbnRCdXMgPSBjb25zdHJ1Y3QuZXZlbnRCdXM7XG4gIGV4cGVjdChsYW1iZGFGdW5jdGlvbikudG9CZURlZmluZWQoKTtcbiAgZXhwZWN0KHZwYykudG9CZVVuZGVmaW5lZCgpO1xuICBleHBlY3QoZXZlbnRCdXMpLnRvQmVVbmRlZmluZWQoKTtcblxuICAvLyBDaGVjayBFVkVOVEJVU19OQU1FIEVudiB2YXJpYWJsZVxuICBjb25zdCB0ZW1wbGF0ZSA9IFRlbXBsYXRlLmZyb21TdGFjayhzdGFjayk7XG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcyhcIkFXUzo6TGFtYmRhOjpGdW5jdGlvblwiLCB7XG4gICAgRW52aXJvbm1lbnQ6IHtcbiAgICAgIFZhcmlhYmxlczoge1xuICAgICAgICBBV1NfTk9ERUpTX0NPTk5FQ1RJT05fUkVVU0VfRU5BQkxFRDogXCIxXCIsXG4gICAgICAgIEVWRU5UQlVTX05BTUU6IFwiZGVmYXVsdFwiXG4gICAgICB9XG4gICAgfVxuICB9KTtcblxuICAvLyBTaW5jZSB1c2luZyBkZWZhdWx0IGV2ZW50IGJ1cywgdGhlcmUgc2hvdWxkbid0IGJlIGFueSBldmVudGJ1c1xuICB0ZW1wbGF0ZS5yZXNvdXJjZUNvdW50SXMoJ0FXUzo6RXZlbnRzOjpFdmVudEJ1cycsIDApO1xuXG4gIC8vIENoZWNrIExhbWJkYSBGdW5jdGlvbiBwZXJtaXNzaW9ucyB0byBhY2Nlc3MgZGVmYXVsdCBFdmVudCBCcmlkZ2VcbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKFwiQVdTOjpJQU06OlBvbGljeVwiLCB7XG4gICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICB4cmF5UG9saWN5U3RhdGVtZW50LFxuICAgICAgICBkZWZhdWx0RXZlbnRCdXNQb2xpY3lTdGF0ZW1lbnRcbiAgICAgIF1cbiAgICB9XG4gIH0pO1xufSk7XG5cbnRlc3QoXCJDb25maXJtIENoZWNrVnBjUHJvcHMgaXMgYmVpbmcgY2FsbGVkXCIsICgpID0+IHtcbiAgLy8gU3RhY2tcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICBjb25zdCB0ZXN0VnBjID0gbmV3IGVjMi5WcGMoc3RhY2ssIFwidGVzdC12cGNcIiwge30pO1xuXG4gIGNvbnN0IGFwcCA9ICgpID0+IHtcbiAgICAvLyBIZWxwZXIgZGVjbGFyYXRpb25cbiAgICBuZXcgTGFtYmRhVG9FdmVudGJyaWRnZShzdGFjaywgXCJsYW1iZGEtdG8tZXZlbnRicmlkZ2Utc3RhY2tcIiwge1xuICAgICAgbGFtYmRhRnVuY3Rpb25Qcm9wczoge1xuICAgICAgICBydW50aW1lOiBkZWZhdWx0cy5DT01NRVJDSUFMX1JFR0lPTl9MQU1CREFfTk9ERV9SVU5USU1FLFxuICAgICAgICBoYW5kbGVyOiBcImluZGV4LmhhbmRsZXJcIixcbiAgICAgICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYCksXG4gICAgICB9LFxuICAgICAgZXhpc3RpbmdWcGM6IHRlc3RWcGMsXG4gICAgICBkZXBsb3lWcGM6IHRydWUsXG4gICAgfSk7XG4gIH07XG4gIC8vIEFzc2VydGlvblxuICBleHBlY3QoYXBwKS50b1Rocm93RXJyb3IoJ0Vycm9yIC0gRWl0aGVyIHByb3ZpZGUgYW4gZXhpc3RpbmdWcGMgb3Igc29tZSBjb21iaW5hdGlvbiBvZiBkZXBsb3lWcGMgYW5kIHZwY1Byb3BzLCBidXQgbm90IGJvdGguXFxuJyk7XG59KTtcblxudGVzdChcIkNvbmZpcm0gQ2hlY2tFdmVudEJyaWRnZVByb3BzIGlzIGNhbGxlZFwiLCAoKSA9PiB7XG4gIC8vIFN0YWNrXG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgY29uc3QgYXBwID0gKCkgPT4ge1xuICAgIC8vIEhlbHBlciBkZWNsYXJhdGlvblxuICAgIG5ldyBMYW1iZGFUb0V2ZW50YnJpZGdlKHN0YWNrLCAnbGFtYmRhLXRvLWV2ZW50YnJpZGdlLXN0YWNrJywge1xuICAgICAgbGFtYmRhRnVuY3Rpb25Qcm9wczoge1xuICAgICAgICBydW50aW1lOiBkZWZhdWx0cy5DT01NRVJDSUFMX1JFR0lPTl9MQU1CREFfTk9ERV9SVU5USU1FLFxuICAgICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApXG4gICAgICB9LFxuICAgICAgZXZlbnRCdXNQcm9wczogeyBldmVudEJ1c05hbWU6ICd0ZXN0JyB9LFxuICAgICAgZXhpc3RpbmdFdmVudEJ1c0ludGVyZmFjZTogbmV3IGV2ZW50cy5FdmVudEJ1cyhzdGFjaywgYG5ldy1ldmVudC1idXNgLCB7ICBldmVudEJ1c05hbWU6ICd0ZXN0JyAgfSlcbiAgICB9KTtcbiAgfTtcbiAgLy8gQXNzZXJ0aW9uXG4gIGV4cGVjdChhcHApLnRvVGhyb3dFcnJvcignRXJyb3IgLSBFaXRoZXIgcHJvdmlkZSBleGlzdGluZ0V2ZW50QnVzSW50ZXJmYWNlIG9yIGV2ZW50QnVzUHJvcHMsIGJ1dCBub3QgYm90aC5cXG4nKTtcbn0pO1xuXG50ZXN0KCdUZXN0IGRlcGxveW1lbnQgdy8gZXhpc3RpbmcgZXZlbnRidXMnLCAoKSA9PiB7XG4gIC8vIFN0YWNrXG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gIC8vIEhlbHBlciBkZWNsYXJhdGlvblxuICBjb25zdCBleGlzdGluZ0V2ZW50QnVzID0gbmV3IGV2ZW50cy5FdmVudEJ1cyhzdGFjaywgJ2V4aXN0aW5nLWV2ZW50YnVzJywge1xuICAgIGV2ZW50QnVzTmFtZTogJ2N1c3RvbWV2ZW50YnVzJ1xuICB9KTtcblxuICBuZXcgTGFtYmRhVG9FdmVudGJyaWRnZShzdGFjaywgJ2xhbWJkYS10by1ldmVudGJyaWRnZS1zdGFjaycsIHtcbiAgICBsYW1iZGFGdW5jdGlvblByb3BzOiB7XG4gICAgICBydW50aW1lOiBkZWZhdWx0cy5DT01NRVJDSUFMX1JFR0lPTl9MQU1CREFfTk9ERV9SVU5USU1FLFxuICAgICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInLFxuICAgICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYClcbiAgICB9LFxuICAgIGV4aXN0aW5nRXZlbnRCdXNJbnRlcmZhY2U6IGV4aXN0aW5nRXZlbnRCdXNcbiAgfSk7XG5cbiAgLy8gQ2hlY2sgTGFtYmRhIFBlcm1pc3Npb25zXG4gIGNvbnN0IHRlbXBsYXRlID0gVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKTtcbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKFwiQVdTOjpJQU06OlBvbGljeVwiLCB7XG4gICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICB4cmF5UG9saWN5U3RhdGVtZW50LFxuICAgICAgICB7XG4gICAgICAgICAgQWN0aW9uOiBcImV2ZW50czpQdXRFdmVudHNcIixcbiAgICAgICAgICBFZmZlY3Q6IFwiQWxsb3dcIixcbiAgICAgICAgICBSZXNvdXJjZToge1xuICAgICAgICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgICAgICAgXCJleGlzdGluZ2V2ZW50YnVzMkE5OUZBNDlcIixcbiAgICAgICAgICAgICAgXCJBcm5cIlxuICAgICAgICAgICAgXVxuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgXVxuICAgIH1cbiAgfSk7XG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpFdmVudHM6OkV2ZW50QnVzJywge1xuICAgIE5hbWU6IGBjdXN0b21ldmVudGJ1c2BcbiAgfSk7XG59KTtcblxudGVzdChcIlRlc3QgbWluaW1hbCBkZXBsb3ltZW50IHRoYXQgZGVwbG95cyBhIFZQQyB3aXRob3V0IHZwY1Byb3BzXCIsICgpID0+IHtcbiAgLy8gU3RhY2tcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgLy8gSGVscGVyIGRlY2xhcmF0aW9uXG4gIG5ldyBMYW1iZGFUb0V2ZW50YnJpZGdlKHN0YWNrLCBcImxhbWJkYS10by1ldmVudGJyaWRnZS1zdGFja1wiLCB7XG4gICAgbGFtYmRhRnVuY3Rpb25Qcm9wczoge1xuICAgICAgcnVudGltZTogZGVmYXVsdHMuQ09NTUVSQ0lBTF9SRUdJT05fTEFNQkRBX05PREVfUlVOVElNRSxcbiAgICAgIGhhbmRsZXI6IFwiaW5kZXguaGFuZGxlclwiLFxuICAgICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYCksXG4gICAgfSxcbiAgICBkZXBsb3lWcGM6IHRydWUsXG4gIH0pO1xuXG4gIGNvbnN0IHRlbXBsYXRlID0gVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKTtcbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKFwiQVdTOjpMYW1iZGE6OkZ1bmN0aW9uXCIsIHZwY0NvbmZpZyk7XG5cbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKFwiQVdTOjpFQzI6OlZQQ1wiLCB7XG4gICAgRW5hYmxlRG5zSG9zdG5hbWVzOiB0cnVlLFxuICAgIEVuYWJsZURuc1N1cHBvcnQ6IHRydWUsXG4gIH0pO1xuXG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcyhcIkFXUzo6RUMyOjpWUENFbmRwb2ludFwiLCB7XG4gICAgVnBjRW5kcG9pbnRUeXBlOiBcIkludGVyZmFjZVwiLFxuICB9KTtcblxuICB0ZW1wbGF0ZS5yZXNvdXJjZUNvdW50SXMoXCJBV1M6OkVDMjo6U3VibmV0XCIsIDIpO1xuICB0ZW1wbGF0ZS5yZXNvdXJjZUNvdW50SXMoXCJBV1M6OkVDMjo6SW50ZXJuZXRHYXRld2F5XCIsIDApO1xufSk7XG5cbnRlc3QoXCJUZXN0IG1pbmltYWwgZGVwbG95bWVudCB0aGF0IGRlcGxveXMgYSBWUEMgdy92cGNQcm9wc1wiLCAoKSA9PiB7XG4gIC8vIFN0YWNrXG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gIC8vIEhlbHBlciBkZWNsYXJhdGlvblxuICBuZXcgTGFtYmRhVG9FdmVudGJyaWRnZShzdGFjaywgXCJsYW1iZGEtdG8tZXZlbnRicmlkZ2Utc3RhY2tcIiwge1xuICAgIGxhbWJkYUZ1bmN0aW9uUHJvcHM6IHtcbiAgICAgIHJ1bnRpbWU6IGRlZmF1bHRzLkNPTU1FUkNJQUxfUkVHSU9OX0xBTUJEQV9OT0RFX1JVTlRJTUUsXG4gICAgICBoYW5kbGVyOiBcImluZGV4LmhhbmRsZXJcIixcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApLFxuICAgIH0sXG4gICAgdnBjUHJvcHM6IHtcbiAgICAgIGVuYWJsZURuc0hvc3RuYW1lczogZmFsc2UsXG4gICAgICBlbmFibGVEbnNTdXBwb3J0OiBmYWxzZSxcbiAgICAgIGlwQWRkcmVzc2VzOiBlYzIuSXBBZGRyZXNzZXMuY2lkcihcIjE5Mi42OC4wLjAvMTZcIiksXG4gICAgfSxcbiAgICBkZXBsb3lWcGM6IHRydWUsXG4gIH0pO1xuXG4gIGNvbnN0IHRlbXBsYXRlID0gVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKTtcbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKFwiQVdTOjpMYW1iZGE6OkZ1bmN0aW9uXCIsIHZwY0NvbmZpZyk7XG5cbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKFwiQVdTOjpFQzI6OlZQQ1wiLCB7XG4gICAgQ2lkckJsb2NrOiBcIjE5Mi42OC4wLjAvMTZcIixcbiAgICBFbmFibGVEbnNIb3N0bmFtZXM6IHRydWUsXG4gICAgRW5hYmxlRG5zU3VwcG9ydDogdHJ1ZSxcbiAgfSk7XG5cbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKFwiQVdTOjpFQzI6OlZQQ0VuZHBvaW50XCIsIHtcbiAgICBWcGNFbmRwb2ludFR5cGU6IFwiSW50ZXJmYWNlXCIsXG4gIH0pO1xuXG4gIHRlbXBsYXRlLnJlc291cmNlQ291bnRJcyhcIkFXUzo6RUMyOjpTdWJuZXRcIiwgMik7XG4gIHRlbXBsYXRlLnJlc291cmNlQ291bnRJcyhcIkFXUzo6RUMyOjpJbnRlcm5ldEdhdGV3YXlcIiwgMCk7XG59KTtcblxudGVzdChcIlRlc3QgbWluaW1hbCBkZXBsb3ltZW50IHdpdGggYW4gZXhpc3RpbmcgVlBDXCIsICgpID0+IHtcbiAgLy8gU3RhY2tcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICBjb25zdCB0ZXN0VnBjID0gbmV3IGVjMi5WcGMoc3RhY2ssIFwidGVzdC12cGNcIiwge30pO1xuXG4gIC8vIEhlbHBlciBkZWNsYXJhdGlvblxuICBuZXcgTGFtYmRhVG9FdmVudGJyaWRnZShzdGFjaywgXCJsYW1iZGEtdG8tZXZlbnRicmlkZ2Utc3RhY2tcIiwge1xuICAgIGxhbWJkYUZ1bmN0aW9uUHJvcHM6IHtcbiAgICAgIHJ1bnRpbWU6IGRlZmF1bHRzLkNPTU1FUkNJQUxfUkVHSU9OX0xBTUJEQV9OT0RFX1JVTlRJTUUsXG4gICAgICBoYW5kbGVyOiBcImluZGV4LmhhbmRsZXJcIixcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApLFxuICAgIH0sXG4gICAgZXhpc3RpbmdWcGM6IHRlc3RWcGMsXG4gIH0pO1xuXG4gIGNvbnN0IHRlbXBsYXRlID0gVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKTtcbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKFwiQVdTOjpMYW1iZGE6OkZ1bmN0aW9uXCIsIHtcbiAgICBWcGNDb25maWc6IHtcbiAgICAgIFNlY3VyaXR5R3JvdXBJZHM6IFtcbiAgICAgICAge1xuICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICBcImxhbWJkYXRvZXZlbnRicmlkZ2VzdGFja1JlcGxhY2VEZWZhdWx0U2VjdXJpdHlHcm91cHNlY3VyaXR5Z3JvdXA1OUVGMDcwNlwiLFxuICAgICAgICAgICAgXCJHcm91cElkXCIsXG4gICAgICAgICAgXSxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgICBTdWJuZXRJZHM6IFtcbiAgICAgICAge1xuICAgICAgICAgIFJlZjogXCJ0ZXN0dnBjUHJpdmF0ZVN1Ym5ldDFTdWJuZXQ4NjVGQjUwQVwiLFxuICAgICAgICB9LFxuICAgICAgICB7XG4gICAgICAgICAgUmVmOiBcInRlc3R2cGNQcml2YXRlU3VibmV0MlN1Ym5ldDIzRDMzOTZGXCIsXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgIH0sXG4gIH0pO1xuXG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcyhcIkFXUzo6RUMyOjpWUENFbmRwb2ludFwiLCB7XG4gICAgVnBjRW5kcG9pbnRUeXBlOiBcIkludGVyZmFjZVwiLFxuICB9KTtcblxuICAvLyBDaGVjayBMYW1iZGEgRnVuY3Rpb24gcGVybWlzc2lvbnMgdG8gYWNjZXNzIGRlZmF1bHQgRXZlbnQgQnJpZGdlXG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcyhcIkFXUzo6SUFNOjpQb2xpY3lcIiwge1xuICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICBTdGF0ZW1lbnQ6IFtcbiAgICAgICAge1xuICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgXCJlYzI6Q3JlYXRlTmV0d29ya0ludGVyZmFjZVwiLFxuICAgICAgICAgICAgXCJlYzI6RGVzY3JpYmVOZXR3b3JrSW50ZXJmYWNlc1wiLFxuICAgICAgICAgICAgXCJlYzI6RGVsZXRlTmV0d29ya0ludGVyZmFjZVwiLFxuICAgICAgICAgICAgXCJlYzI6QXNzaWduUHJpdmF0ZUlwQWRkcmVzc2VzXCIsXG4gICAgICAgICAgICBcImVjMjpVbmFzc2lnblByaXZhdGVJcEFkZHJlc3Nlc1wiXG4gICAgICAgICAgXSxcbiAgICAgICAgICBFZmZlY3Q6IFwiQWxsb3dcIixcbiAgICAgICAgICBSZXNvdXJjZTogXCIqXCJcbiAgICAgICAgfSxcbiAgICAgICAgeHJheVBvbGljeVN0YXRlbWVudCxcbiAgICAgICAgZGVmYXVsdEV2ZW50QnVzUG9saWN5U3RhdGVtZW50XG4gICAgICBdXG4gICAgfVxuICB9KTtcbn0pO1xuXG50ZXN0KFwiVGVzdCBtaW5pbWFsIGRlcGxveW1lbnQgd2l0aCBhbiBleGlzdGluZyBWUEMgYW5kIGV4aXN0aW5nIExhbWJkYSBmdW5jdGlvbiBub3QgaW4gYSBWUENcIiwgKCkgPT4ge1xuICAvLyBTdGFja1xuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gIGNvbnN0IHRlc3RMYW1iZGFGdW5jdGlvbiA9IG5ldyBsYW1iZGEuRnVuY3Rpb24oc3RhY2ssICd0ZXN0LWxhbWJhJywge1xuICAgIHJ1bnRpbWU6IGRlZmF1bHRzLkNPTU1FUkNJQUxfUkVHSU9OX0xBTUJEQV9OT0RFX1JVTlRJTUUsXG4gICAgaGFuZGxlcjogXCJpbmRleC5oYW5kbGVyXCIsXG4gICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYCksXG4gIH0pO1xuXG4gIGNvbnN0IHRlc3RWcGMgPSBuZXcgZWMyLlZwYyhzdGFjaywgXCJ0ZXN0LXZwY1wiLCB7fSk7XG5cbiAgLy8gSGVscGVyIGRlY2xhcmF0aW9uXG4gIGNvbnN0IGFwcCA9ICgpID0+IHtcbiAgICAvLyBidWlsZExhbWJkYUZ1bmN0aW9uIHNob3VsZCB0aHJvdyBhbiBlcnJvciBpZiB0aGUgTGFtYmRhIGZ1bmN0aW9uIGlzIG5vdFxuICAgIC8vIGF0dGFjaGVkIHRvIGEgVlBDXG4gICAgbmV3IExhbWJkYVRvRXZlbnRicmlkZ2Uoc3RhY2ssIFwibGFtYmRhLXRvLWV2ZW50YnJpZGdlLXN0YWNrXCIsIHtcbiAgICAgIGV4aXN0aW5nTGFtYmRhT2JqOiB0ZXN0TGFtYmRhRnVuY3Rpb24sXG4gICAgICBleGlzdGluZ1ZwYzogdGVzdFZwYyxcbiAgICB9KTtcbiAgfTtcblxuICAvLyBBc3NlcnRpb25cbiAgZXhwZWN0KGFwcCkudG9UaHJvd0Vycm9yKCdBIExhbWJkYSBmdW5jdGlvbiBtdXN0IGJlIGJvdW5kIHRvIGEgVlBDIHVwb24gY3JlYXRpb24sIGl0IGNhbm5vdCBiZSBhZGRlZCB0byBhIFZQQyBpbiBhIHN1YnNlcXVlbnQgY29uc3RydWN0Jyk7XG5cbn0pO1xuXG50ZXN0KCdUZXN0IGxhbWJkYSBmdW5jdGlvbiBjdXN0b20gZW52aXJvbm1lbnQgdmFyaWFibGUnLCAoKSA9PiB7XG4gIC8vIFN0YWNrXG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgLy8gSGVscGVyIGRlY2xhcmF0aW9uXG4gIG5ldyBMYW1iZGFUb0V2ZW50YnJpZGdlKHN0YWNrLCAnbGFtYmRhLXRvLWV2ZW50YnJpZGdlLXN0YWNrJywge1xuICAgIGxhbWJkYUZ1bmN0aW9uUHJvcHM6IHtcbiAgICAgIHJ1bnRpbWU6IGRlZmF1bHRzLkNPTU1FUkNJQUxfUkVHSU9OX0xBTUJEQV9OT0RFX1JVTlRJTUUsXG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQoYCR7X19kaXJuYW1lfS9sYW1iZGFgKSxcbiAgICB9LFxuICAgIGV2ZW50QnVzUHJvcHM6IHtldmVudEJ1c05hbWU6ICdjdXN0b21ldmVudGJ1cyd9LFxuICAgIGV2ZW50QnVzRW52aXJvbm1lbnRWYXJpYWJsZU5hbWU6ICdDVVNUT01fRVZFTlRCVVNfTkFNRSdcbiAgfSk7XG5cbiAgLy8gQ2hlY2sgZW52aXJvbm1lbnQgdmFyaWFibGVzXG4gIGNvbnN0IHRlbXBsYXRlID0gVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKTtcbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkxhbWJkYTo6RnVuY3Rpb24nLCB7XG4gICAgSGFuZGxlcjogJ2luZGV4LmhhbmRsZXInLFxuICAgIFJ1bnRpbWU6ICBkZWZhdWx0cy5DT01NRVJDSUFMX1JFR0lPTl9MQU1CREFfTk9ERV9TVFJJTkcsXG4gICAgRW52aXJvbm1lbnQ6IHtcbiAgICAgIFZhcmlhYmxlczoge1xuICAgICAgICBBV1NfTk9ERUpTX0NPTk5FQ1RJT05fUkVVU0VfRU5BQkxFRDogJzEnLFxuICAgICAgICBDVVNUT01fRVZFTlRCVVNfTkFNRToge1xuICAgICAgICAgIFJlZjogXCJsYW1iZGF0b2V2ZW50YnJpZGdlc3RhY2tjdXN0b21ldmVudGJ1czI1ODI1QkVGXCJcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cbiAgfSk7XG5cbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkV2ZW50czo6RXZlbnRCdXMnLCB7XG4gICAgTmFtZTogYGN1c3RvbWV2ZW50YnVzYFxuICB9KTtcblxuICAvLyBDaGVjayBsYW1iZGEgcGVybWlzc2lvbnMgdG8gY3VzdG9tIGV2ZW50IGJ1c1xuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoXCJBV1M6OklBTTo6UG9saWN5XCIsIHtcbiAgICBQb2xpY3lEb2N1bWVudDoge1xuICAgICAgU3RhdGVtZW50OiBbXG4gICAgICAgIHhyYXlQb2xpY3lTdGF0ZW1lbnQsXG4gICAgICAgIHtcbiAgICAgICAgICBBY3Rpb246IFwiZXZlbnRzOlB1dEV2ZW50c1wiLFxuICAgICAgICAgIEVmZmVjdDogXCJBbGxvd1wiLFxuICAgICAgICAgIFJlc291cmNlOiB7XG4gICAgICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgICAgICBcImxhbWJkYXRvZXZlbnRicmlkZ2VzdGFja2N1c3RvbWV2ZW50YnVzMjU4MjVCRUZcIixcbiAgICAgICAgICAgICAgXCJBcm5cIlxuICAgICAgICAgICAgXVxuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgXVxuICAgIH1cbiAgfSk7XG59KTtcblxudGVzdCgnY2hlY2sgbXVsdGlwbGUgY29uc3RydWN0cyBpbiBhIHNpbmdsZSBzdGFjaycsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICBjb25zdCBwcm9wczogTGFtYmRhVG9FdmVudGJyaWRnZVByb3BzID0ge1xuICAgIGxhbWJkYUZ1bmN0aW9uUHJvcHM6IHtcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApLFxuICAgICAgcnVudGltZTogZGVmYXVsdHMuQ09NTUVSQ0lBTF9SRUdJT05fTEFNQkRBX05PREVfUlVOVElNRSxcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJ1xuICAgIH0sXG4gICAgZXZlbnRCdXNQcm9wczogeyBldmVudEJ1c05hbWU6ICd0ZXN0JyB9XG4gIH07XG4gIG5ldyBMYW1iZGFUb0V2ZW50YnJpZGdlKHN0YWNrLCAndGVzdC1uZXctbGFtYmRhLWV2ZW50YnJpZGdlMScsIHByb3BzKTtcbiAgbmV3IExhbWJkYVRvRXZlbnRicmlkZ2Uoc3RhY2ssICd0ZXN0LW5ldy1sYW1iZGEtZXZlbnRicmlkZ2UyJywgcHJvcHMpO1xuICBjb25zdCB0ZW1wbGF0ZSA9IFRlbXBsYXRlLmZyb21TdGFjayhzdGFjayk7XG4gIHRlbXBsYXRlLnJlc291cmNlQ291bnRJcygnQVdTOjpFdmVudHM6OkV2ZW50QnVzJywgMik7XG59KTtcblxudGVzdCgnY2hlY2sgbXVsdGlwbGUgbGFtYmRhIGZ1bmN0aW9ucyBwdWJsaXNoaW5nIHRvIHNpbmdsZSBldmVudCBidXMnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgY29uc3QgcHJvcHMxOiBMYW1iZGFUb0V2ZW50YnJpZGdlUHJvcHMgPSB7XG4gICAgbGFtYmRhRnVuY3Rpb25Qcm9wczoge1xuICAgICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYCksXG4gICAgICBydW50aW1lOiBkZWZhdWx0cy5DT01NRVJDSUFMX1JFR0lPTl9MQU1CREFfTk9ERV9SVU5USU1FLFxuICAgICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInXG4gICAgfSxcbiAgICBldmVudEJ1c1Byb3BzOiB7IGV2ZW50QnVzTmFtZTogJ3Rlc3QnIH1cbiAgfTtcbiAgY29uc3QgY29uc3RydWN0ID0gbmV3IExhbWJkYVRvRXZlbnRicmlkZ2Uoc3RhY2ssICd0ZXN0LW5ldy1sYW1iZGEtZXZlbnRicmlkZ2UxJywgcHJvcHMxKTtcblxuICBjb25zdCBwcm9wczI6IExhbWJkYVRvRXZlbnRicmlkZ2VQcm9wcyA9IHtcbiAgICBsYW1iZGFGdW5jdGlvblByb3BzOiB7XG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQoYCR7X19kaXJuYW1lfS9sYW1iZGFgKSxcbiAgICAgIHJ1bnRpbWU6IGRlZmF1bHRzLkNPTU1FUkNJQUxfUkVHSU9OX0xBTUJEQV9OT0RFX1JVTlRJTUUsXG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcidcbiAgICB9LFxuICAgIGV4aXN0aW5nRXZlbnRCdXNJbnRlcmZhY2U6IGNvbnN0cnVjdC5ldmVudEJ1c1xuICB9O1xuICBuZXcgTGFtYmRhVG9FdmVudGJyaWRnZShzdGFjaywgJ3Rlc3QtbmV3LWxhbWJkYS1ldmVudGJyaWRnZTInLCBwcm9wczIpO1xuXG4gIC8vIE1ha2Ugc3VyZSBvbmx5IHNpbmdsZSBldmVudCBidXMgZXhpc3RzXG4gIGNvbnN0IHRlbXBsYXRlID0gVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKTtcbiAgdGVtcGxhdGUucmVzb3VyY2VDb3VudElzKCdBV1M6OkV2ZW50czo6RXZlbnRCdXMnLCAxKTtcblxuICAvLyBNYWtlIHN1cmUgMiBsYW1iZGEgZnVuY3Rpb25zIGV4aXN0XG4gIHRlbXBsYXRlLnJlc291cmNlQ291bnRJcygnQVdTOjpMYW1iZGE6OkZ1bmN0aW9uJywgMik7XG5cbiAgLy8gQ2hlY2sgd2hldGhlciBsYW1iZGFzIGhhdmUgcGVybWlzc2lvbiB0byBwdWJsaXNoIHRvIHRoZSBldmVudCBidXNcbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICBTdGF0ZW1lbnQ6IFtcbiAgICAgICAgeHJheVBvbGljeVN0YXRlbWVudCxcbiAgICAgICAge1xuICAgICAgICAgIEFjdGlvbjogXCJldmVudHM6UHV0RXZlbnRzXCIsXG4gICAgICAgICAgRWZmZWN0OiBcIkFsbG93XCIsXG4gICAgICAgICAgUmVzb3VyY2U6IHtcbiAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICAgIFwidGVzdG5ld2xhbWJkYWV2ZW50YnJpZGdlMXRlc3QxM0UwQjYyMlwiLFxuICAgICAgICAgICAgICBcIkFyblwiXG4gICAgICAgICAgICBdXG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICBdXG4gICAgfVxuICB9KTtcblxuICAvLyBDaGVjayBlbnZpcm9ubWVudCB2YXJpYWJsZXNcbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkxhbWJkYTo6RnVuY3Rpb24nLCB7XG4gICAgSGFuZGxlcjogJ2luZGV4LmhhbmRsZXInLFxuICAgIFJ1bnRpbWU6ICBkZWZhdWx0cy5DT01NRVJDSUFMX1JFR0lPTl9MQU1CREFfTk9ERV9TVFJJTkcsXG4gICAgRW52aXJvbm1lbnQ6IHtcbiAgICAgIFZhcmlhYmxlczoge1xuICAgICAgICBBV1NfTk9ERUpTX0NPTk5FQ1RJT05fUkVVU0VfRU5BQkxFRDogJzEnLFxuICAgICAgICBFVkVOVEJVU19OQU1FOiB7XG4gICAgICAgICAgUmVmOiBcInRlc3RuZXdsYW1iZGFldmVudGJyaWRnZTF0ZXN0MTNFMEI2MjJcIlxuICAgICAgICB9XG4gICAgICB9XG4gICAgfVxuICB9KTtcbn0pO1xuXG50ZXN0KCdDb25maXJtIGNhbGwgdG8gQ2hlY2tMYW1iZGFQcm9wcycsICgpID0+IHtcbiAgLy8gSW5pdGlhbCBTZXR1cFxuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICBjb25zdCBsYW1iZGFGdW5jdGlvbiA9IG5ldyBsYW1iZGEuRnVuY3Rpb24oc3RhY2ssICdhLWZ1bmN0aW9uJywge1xuICAgIHJ1bnRpbWU6IGRlZmF1bHRzLkNPTU1FUkNJQUxfUkVHSU9OX0xBTUJEQV9OT0RFX1JVTlRJTUUsXG4gICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInLFxuICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApLFxuICB9KTtcblxuICBjb25zdCBwcm9wczogTGFtYmRhVG9FdmVudGJyaWRnZVByb3BzID0ge1xuICAgIGxhbWJkYUZ1bmN0aW9uUHJvcHM6IHtcbiAgICAgIHJ1bnRpbWU6IGRlZmF1bHRzLkNPTU1FUkNJQUxfUkVHSU9OX0xBTUJEQV9OT0RFX1JVTlRJTUUsXG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQoYCR7X19kaXJuYW1lfS9sYW1iZGFgKSxcbiAgICB9LFxuICAgIGV4aXN0aW5nTGFtYmRhT2JqOiBsYW1iZGFGdW5jdGlvbixcbiAgfTtcbiAgY29uc3QgYXBwID0gKCkgPT4ge1xuICAgIG5ldyBMYW1iZGFUb0V2ZW50YnJpZGdlKHN0YWNrLCAndGVzdC1jb25zdHJ1Y3QnLCBwcm9wcyk7XG4gIH07XG4gIC8vIEFzc2VydGlvblxuICBleHBlY3QoYXBwKS50b1Rocm93RXJyb3IoJ0Vycm9yIC0gRWl0aGVyIHByb3ZpZGUgbGFtYmRhRnVuY3Rpb25Qcm9wcyBvciBleGlzdGluZ0xhbWJkYU9iaiwgYnV0IG5vdCBib3RoLlxcbicpO1xufSk7XG4iXX0=