import pytest
import os
from tubefetcherlib import get_video_stream_url, get_video_info, get_available_qualities, StreamQuality, Config
from tubefetcherlib.exceptions import VideoNotFoundError, ProviderAuthenticationError, StreamUnavailableError, InvalidURLError

# URL de um vídeo de teste do Instagram (substitua por uma URL real de um vídeo público)
TEST_INSTAGRAM_URL = "https://www.instagram.com/p/C71_y_1o_1o/" # Exemplo de URL de post com vídeo

# Credenciais para o Instaloader (obtidas de variáveis de ambiente)
INSTAGRAM_USERNAME = os.environ.get("INSTAGRAM_USERNAME")
INSTAGRAM_PASSWORD = os.environ.get("INSTAGRAM_PASSWORD")

@pytest.fixture(scope="module")
def instagram_config():
    return Config(instagram_session_dir=".", timeout=60, retries=5)

@pytest.mark.integration
def test_get_real_instagram_stream_url_with_optional_credentials(instagram_config):
    """Testa a obtenção de uma URL de stream real do Instagram com credenciais opcionais."""
    try:
        stream_url = get_video_stream_url(TEST_INSTAGRAM_URL, 
                                          instagram_username=INSTAGRAM_USERNAME, 
                                          instagram_password=INSTAGRAM_PASSWORD,
                                          config=instagram_config)
        print(f"\nReal Instagram Stream URL: {stream_url}") # Para depuração
        assert stream_url.startswith("http")
        assert "cdninstagram.com" in stream_url or "instagram.com" in stream_url
    except (VideoNotFoundError, ProviderAuthenticationError, StreamUnavailableError, InvalidURLError) as e:
        pytest.fail(f"Falha ao obter a URL do stream do Instagram: {e}")

@pytest.mark.integration
def test_get_real_instagram_video_info_with_optional_credentials(instagram_config):
    """Testa a obtenção de informações reais do vídeo do Instagram com credenciais opcionais."""
    try:
        video_info = get_video_info(TEST_INSTAGRAM_URL, 
                                    instagram_username=INSTAGRAM_USERNAME, 
                                    instagram_password=INSTAGRAM_PASSWORD,
                                    config=instagram_config)
        print(f"\nReal Instagram Video Info: {video_info}") # Para depuração
        assert isinstance(video_info, dict)
        assert "title" in video_info
        assert "author" in video_info
        assert "is_video" in video_info
        assert video_info["is_video"] is True
    except (VideoNotFoundError, ProviderAuthenticationError, StreamUnavailableError, InvalidURLError) as e:
        pytest.fail(f"Falha ao obter informações do vídeo do Instagram: {e}")

@pytest.mark.integration
def test_get_real_instagram_available_qualities(instagram_config):
    """Testa a obtenção de qualidades disponíveis para vídeo do Instagram."""
    try:
        qualities = get_available_qualities(TEST_INSTAGRAM_URL,
                                            instagram_username=INSTAGRAM_USERNAME,
                                            instagram_password=INSTAGRAM_PASSWORD,
                                            config=instagram_config)
        print(f"\nReal Instagram Available Qualities: {qualities}")
        assert StreamQuality.HIGHEST in qualities
    except (VideoNotFoundError, ProviderAuthenticationError, StreamUnavailableError, InvalidURLError) as e:
        pytest.fail(f"Falha ao obter qualidades disponíveis do Instagram: {e}")
