import pytest
from tubefetcherlib import get_video_stream_url, get_video_info, get_available_qualities, StreamQuality, Config
from tubefetcherlib.exceptions import VideoNotFoundError, StreamUnavailableError, UnsupportedQualityError

# URL de um vídeo de teste (ex: um vídeo sem direitos autorais ou de domínio público)
TEST_YOUTUBE_URL = "https://www.youtube.com/watch?v=LXb3EKWsInQ" # Exemplo: Vídeo de domínio público

@pytest.fixture(scope="module")
def youtube_config():
    return Config(timeout=60, retries=5)

@pytest.mark.integration
def test_get_real_youtube_stream_url(youtube_config):
    """Testa a obtenção de uma URL de stream real do YouTube."""
    try:
        stream_url = get_video_stream_url(TEST_YOUTUBE_URL, quality=StreamQuality.HIGHEST, config=youtube_config)
        print(f"\nReal Stream URL: {stream_url}") # Para depuração
        assert stream_url.startswith("http")
        assert "googlevideo.com" in stream_url or "youtube.com" in stream_url # Verifica se é uma URL de stream válida
    except (VideoNotFoundError, StreamUnavailableError, UnsupportedQualityError) as e:
        pytest.fail(f"Falha ao obter a URL do stream: {e}")

@pytest.mark.integration
def test_get_real_youtube_video_info(youtube_config):
    """Testa a obtenção de informações reais do vídeo do YouTube."""
    try:
        video_info = get_video_info(TEST_YOUTUBE_URL, config=youtube_config)
        print(f"\nReal Video Info: {video_info}") # Para depuração
        assert isinstance(video_info, dict)
        assert "title" in video_info
        assert "author" in video_info
        assert "length" in video_info
        assert "views" in video_info
        assert "thumbnail_url" in video_info
    except VideoNotFoundError as e:
        pytest.fail(f"Falha ao obter informações do vídeo: {e}")

@pytest.mark.integration
def test_get_real_youtube_available_qualities(youtube_config):
    """Testa a obtenção de qualidades disponíveis para vídeo do YouTube."""
    try:
        qualities = get_available_qualities(TEST_YOUTUBE_URL, config=youtube_config)
        print(f"\nReal YouTube Available Qualities: {qualities}")
        assert StreamQuality.HIGHEST in qualities
        assert StreamQuality.LOWEST in qualities
    except VideoNotFoundError as e:
        pytest.fail(f"Falha ao obter qualidades disponíveis: {e}")
