import pytest
from unittest.mock import MagicMock, patch
from tubefetcherlib.providers.youtube import YouTubeProvider
from tubefetcherlib.providers.instagram import InstagramProvider
from tubefetcherlib.provider_factory import VideoProviderFactory
from tubefetcherlib.main import get_video_stream_url, get_video_info, get_available_qualities
from tubefetcherlib.providers.base import StreamQuality
from tubefetcherlib.exceptions import VideoNotFoundError, StreamUnavailableError, UnsupportedQualityError, InvalidURLError, ProviderAuthenticationError
from tubefetcherlib.config import Config
import os
from datetime import datetime

# Mock para pytubefix.YouTube
class MockStream:
    def __init__(self, url, resolution=None):
        self._url = url
        self._resolution = resolution

    @property
    def url(self):
        return self._url

    @property
    def resolution(self):
        return self._resolution

class MockStreamQuery:
    def __init__(self, streams):
        self._streams = list(streams) # Work on a copy

    def get_highest_resolution(self):
        return next((s for s in self._streams if s.resolution == "1080p"), self._streams[0])

    def get_lowest_resolution(self):
        return next((s for s in self._streams if s.resolution == "144p"), self._streams[-1])

    def filter(self, type=None, progressive=None, res=None):
        if type:
            # Mocking type filter, assuming all are video for simplicity
            pass
        if progressive:
            # Mocking progressive filter
            pass
        if res:
            self._streams = [s for s in self._streams if s.resolution == res]
        return self # Return self for chaining

    def order_by(self, key):
        if key == "resolution":
            self._streams = sorted(self._streams, key=lambda s: int(s.resolution[:-1]) if s.resolution else 0, reverse=True)
        return self # Return self for chaining

    def desc(self):
        return self # Return self for chaining

    def first(self):
        return self._streams[0] if self._streams else None

    def __iter__(self):
        return iter(self._streams)

class MockYouTube:
    def __init__(self, url):
        self.title = "Test Video Title"
        self.author = "Test Author"
        self.length = 300
        self.views = 100000
        self.thumbnail_url = "http://example.com/thumbnail.jpg"
        self.description = "This is a test video description."
        self.streams = MockStreamQuery([
            MockStream("http://example.com/stream_1080p", "1080p"),
            MockStream("http://example.com/stream_720p", "720p"),
            MockStream("http://example.com/stream_480p", "480p"),
            MockStream("http://example.com/stream_144p", "144p"),
        ])

# Mock para instaloader
class MockPost:
    def __init__(self, shortcode, is_video=True):
        self.shortcode = shortcode
        self.is_video = is_video
        self.video_url = f"http://example.com/instagram_video_{shortcode}.mp4"
        self.title = f"Instagram Post {shortcode}"
        self.owner_username = "test_instagram_user"
        self.likes = 100
        self.comments = 10
        self.date_utc = datetime(2023, 1, 1, 12, 0, 0) # Changed to datetime object
        self.url = f"http://example.com/instagram_thumbnail_{shortcode}.jpg"

class MockInstaloader:
    def __init__(self, compress_json=False, dirname_pattern=None, filename_pattern=None, download_videos=False, download_geotags=False, download_comments=False, download_pictures=False, download_video_thumbnails=False, save_metadata=False, post_metadata_txt_pattern="", quiet=True):
        self.context = MagicMock()

    def login(self, username, password):
        if username == "test_user" and password == "test_pass":
            return True
        raise Exception("Login failed")

    def load_session_from_file(self, username, filename):
        if username == "test_user" and filename == os.path.join(".", "test_user.session"):
            return True
        raise FileNotFoundError

    def save_session_to_file(self, filename):
        pass


@pytest.fixture
def mock_instaloader_instance():
    with patch('instaloader.Instaloader', MockInstaloader) as mock_L:
        yield mock_L

@pytest.fixture
def mock_instaloader_post_from_shortcode():
    with patch('instaloader.Post.from_shortcode', side_effect=lambda ctx, sc: MockPost(sc)) as mock_from_shortcode:
        yield mock_from_shortcode


class TestInstagramProvider:
    def test_get_stream_url(self, mock_instaloader_instance, mock_instaloader_post_from_shortcode):
        provider = InstagramProvider()
        url = provider.get_stream_url("https://www.instagram.com/p/test_shortcode/", quality=StreamQuality.HIGHEST)
        assert url == "http://example.com/instagram_video_test_shortcode.mp4"

    def test_get_stream_url_unsupported_quality(self, mock_instaloader_instance, mock_instaloader_post_from_shortcode):
        provider = InstagramProvider()
        with pytest.raises(UnsupportedQualityError):
            provider.get_stream_url("https://www.instagram.com/p/test_shortcode/", quality=StreamQuality.P720)

    def test_get_video_info(self, mock_instaloader_instance, mock_instaloader_post_from_shortcode):
        provider = InstagramProvider()
        info = provider.get_video_info("https://www.instagram.com/p/test_shortcode/")
        assert info["title"] == "Instagram Post test_shortcode"
        assert info["author"] == "test_instagram_user"

    def test_get_available_qualities(self, mock_instaloader_instance, mock_instaloader_post_from_shortcode):
        provider = InstagramProvider()
        qualities = provider.get_available_qualities("https://www.instagram.com/p/test_shortcode/")
        assert qualities == [StreamQuality.HIGHEST]

    def test_instagram_provider_login_success(self, mock_instaloader_instance):
        provider = InstagramProvider(username="test_user", password="test_pass")
        # No exception means success

    def test_instagram_provider_login_fail(self, mock_instaloader_instance):
        with pytest.raises(ProviderAuthenticationError):
            InstagramProvider(username="bad_user", password="bad_pass")

    def test_instagram_provider_load_session(self, mock_instaloader_instance):
        with patch('os.path.exists', return_value=True):
            with patch('instaloader.Instaloader.load_session_from_file', return_value=None) as mock_load_session:
                provider = InstagramProvider(username="test_user", password="test_pass")
                mock_load_session.assert_called_once_with("test_user", filename=os.path.join(".", "test_user.session"))


# Testes para YouTubeProvider
@patch('tubefetcherlib.providers.youtube.YouTube', MockYouTube)
class TestYouTubeProvider:
    def test_get_highest_stream_url(self):
        provider = YouTubeProvider()
        url = provider.get_stream_url("https://www.youtube.com/watch?v=test", quality=StreamQuality.HIGHEST)
        assert url == "http://example.com/stream_1080p"

    def test_get_lowest_stream_url(self):
        provider = YouTubeProvider()
        url = provider.get_stream_url("https://www.youtube.com/watch?v=test", quality=StreamQuality.LOWEST)
        assert url == "http://example.com/stream_144p"

    def test_get_specific_quality_stream_url(self):
        provider = YouTubeProvider()
        url = provider.get_stream_url("https://www.youtube.com/watch?v=test", quality=StreamQuality.P720)
        assert url == "http://example.com/stream_720p"

    def test_get_video_info(self):
        provider = YouTubeProvider()
        info = provider.get_video_info("https://www.youtube.com/watch?v=test")
        assert info["title"] == "Test Video Title"
        assert info["author"] == "Test Author"
        assert info["length"] == 300

    def test_get_available_qualities(self):
        provider = YouTubeProvider()
        qualities = provider.get_available_qualities("https://www.youtube.com/watch?v=test")
        expected_qualities = [
            StreamQuality.HIGHEST,
            StreamQuality.P1080,
            StreamQuality.P720,
            StreamQuality.P480,
            StreamQuality.P144,
            StreamQuality.LOWEST,
        ]
        assert sorted(qualities, key=lambda x: x.value, reverse=True) == sorted(expected_qualities, key=lambda x: x.value, reverse=True)


# Testes para VideoProviderFactory
class TestVideoProviderFactory:
    def test_video_provider_factory_youtube(self):
        provider = VideoProviderFactory.get_provider("https://www.youtube.com/watch?v=test")
        assert isinstance(provider, YouTubeProvider)

    def test_video_provider_factory_instagram(self):
        provider = VideoProviderFactory.get_provider("https://www.instagram.com/p/test/")
        assert isinstance(provider, InstagramProvider)

    def test_video_provider_factory_unsupported(self):
        with pytest.raises(InvalidURLError, match="Unsupported or invalid video URL"):
            VideoProviderFactory.get_provider("https://www.vimeo.com/test")


# Testes para as funções principais da biblioteca
@patch('tubefetcherlib.provider_factory.VideoProviderFactory.get_provider')
def test_get_video_stream_url(mock_get_provider):
    mock_provider = MagicMock()
    mock_provider.get_stream_url.return_value = "mock_stream_url"
    mock_get_provider.return_value = mock_provider

    url = get_video_stream_url("https://www.youtube.com/watch?v=test", quality=StreamQuality.HIGHEST)
    assert url == "mock_stream_url"
    mock_get_provider.assert_called_once_with("https://www.youtube.com/watch?v=test", instagram_username=None, instagram_password=None, config=Config())
    mock_provider.get_stream_url.assert_called_once_with("https://www.youtube.com/watch?v=test", StreamQuality.HIGHEST)

@patch('tubefetcherlib.provider_factory.VideoProviderFactory.get_provider')
def test_get_video_info(mock_get_provider):
    mock_provider = MagicMock()
    mock_provider.get_video_info.return_value = {"title": "Mock Title"}
    mock_get_provider.return_value = mock_provider

    info = get_video_info("https://www.youtube.com/watch?v=test")
    assert info["title"] == "Mock Title"
    mock_get_provider.assert_called_once_with("https://www.youtube.com/watch?v=test", instagram_username=None, instagram_password=None, config=Config())
    mock_provider.get_video_info.assert_called_once_with("https://www.youtube.com/watch?v=test")

@patch('tubefetcherlib.provider_factory.VideoProviderFactory.get_provider')
def test_get_available_qualities(mock_get_provider):
    mock_provider = MagicMock()
    mock_provider.get_available_qualities.return_value = [StreamQuality.HIGHEST]
    mock_get_provider.return_value = mock_provider

    qualities = get_available_qualities("https://www.youtube.com/watch?v=test")
    assert qualities == [StreamQuality.HIGHEST]
    mock_get_provider.assert_called_once_with("https://www.youtube.com/watch?v=test", instagram_username=None, instagram_password=None, config=Config())
    mock_provider.get_available_qualities.assert_called_once_with("https://www.youtube.com/watch?v=test")
