from pytubefix import YouTube, exceptions
from typing import Dict, Any, List
from .base import VideoProvider, StreamQuality
from ..exceptions import VideoNotFoundError, StreamUnavailableError, UnsupportedQualityError
from tubefetcherlib.config import Config
import logging

logger = logging.getLogger(__name__)

class YouTubeProvider(VideoProvider):
    def __init__(self, config: Config = Config()):
        self.config = config

    def _get_youtube_instance(self, video_url: str) -> YouTube:
        # pytubefix doesn't directly support proxies or retries in its constructor
        # We'll rely on the underlying requests library for proxies if configured globally
        # For retries, we'd need to wrap pytubefix calls in a retry mechanism
        return YouTube(video_url)

    def get_stream_url(self, video_url: str, quality: StreamQuality = StreamQuality.HIGHEST) -> str:
        try:
            yt = self._get_youtube_instance(video_url)
            stream = None
            if quality == StreamQuality.HIGHEST:
                stream = yt.streams.get_highest_resolution()
            elif quality == StreamQuality.LOWEST:
                stream = yt.streams.get_lowest_resolution()
            elif quality.value.endswith("p"):
                stream = yt.streams.filter(res=quality.value).first()
            else:
                raise UnsupportedQualityError(f"Unsupported quality format: {quality.value}")
            
            if not stream:
                raise StreamUnavailableError(f"No stream found for quality: {quality.value}")
                
            return stream.url
        except exceptions.VideoUnavailable:
            raise VideoNotFoundError(f"YouTube video not found or unavailable: {video_url}")
        except Exception as e:
            logger.error(f"Error in YouTubeProvider.get_stream_url for {video_url}: {e}")
            raise StreamUnavailableError(f"Error getting YouTube stream URL: {e}")

    def get_video_info(self, video_url: str) -> Dict[str, Any]:
        try:
            yt = self._get_youtube_instance(video_url)
            return {
                "title": yt.title,
                "author": yt.author,
                "length": yt.length,
                "views": yt.views,
                "thumbnail_url": yt.thumbnail_url,
                "description": yt.description,
            }
        except exceptions.VideoUnavailable:
            raise VideoNotFoundError(f"YouTube video not found or unavailable: {video_url}")
        except Exception as e:
            logger.error(f"Error in YouTubeProvider.get_video_info for {video_url}: {e}")
            raise VideoNotFoundError(f"Error getting YouTube video info: {e}")

    def get_available_qualities(self, video_url: str) -> List[StreamQuality]:
        try:
            yt = self._get_youtube_instance(video_url)
            available_res = set()
            for stream in yt.streams.filter(type="video", progressive=True).order_by("resolution").desc():
                if stream.resolution:
                    try:
                        available_res.add(StreamQuality(stream.resolution))
                    except ValueError: # Handle resolutions not in our Enum
                        pass
            
            # Add highest and lowest if they are not already there
            if yt.streams.get_highest_resolution():
                available_res.add(StreamQuality.HIGHEST)
            if yt.streams.get_lowest_resolution():
                available_res.add(StreamQuality.LOWEST)

            return sorted(list(available_res), key=lambda x: x.value, reverse=True)
        except exceptions.VideoUnavailable:
            raise VideoNotFoundError(f"YouTube video not found or unavailable: {video_url}")
        except Exception as e:
            logger.error(f"Error in YouTubeProvider.get_available_qualities for {video_url}: {e}")
            raise StreamUnavailableError(f"Error getting available qualities for YouTube video: {e}")
