"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ImagePipeline = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("node:path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const ec2 = require("aws-cdk-lib/aws-ec2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_kms_1 = require("aws-cdk-lib/aws-kms");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const cdk_iam_floyd_1 = require("cdk-iam-floyd");
const constructs_1 = require("constructs");
const CryptoJS = require("crypto-js");
const yaml = require("js-yaml");
const CheckStateMachineStatus_function_1 = require("../Lambdas/CheckStateMachineStatus/CheckStateMachineStatus-function");
const StartStateMachine_function_1 = require("../Lambdas/StartStateMachine/StartStateMachine-function");
const StateMachine_1 = require("../StateMachine");
class ImagePipeline extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        let infrastructureConfig;
        let imageRecipe;
        this.imageRecipeComponents = [];
        this.imageId = '';
        this.imagePipelineArn = '';
        const account = aws_cdk_lib_1.Stack.of(this).account;
        const region = aws_cdk_lib_1.Stack.of(this).region;
        const stackName = aws_cdk_lib_1.Stack.of(this).stackName;
        const topicKmsKey = new aws_kms_1.Key(this, 'TopicKmsKey', {
            description: 'KMS Key used to encrypt the SNS topic',
            alias: `/ami-builder/${stackName}-TopicKmsKey`,
        });
        // Construct code below
        const topic = new aws_cdk_lib_1.aws_sns.Topic(this, 'ImageBuilderTopic', {
            displayName: 'Image Builder Notify',
            masterKey: topicKmsKey,
        });
        if (props.email != null) {
            topic.addSubscription(new aws_cdk_lib_1.aws_sns_subscriptions.EmailSubscription(props.email));
        }
        const role = new aws_cdk_lib_1.aws_iam.Role(this, 'Role', {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('ec2.amazonaws.com'),
            description: 'IAM role used as part of an Image Builder pipeline',
        });
        role.addManagedPolicy(aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilder'));
        role.addManagedPolicy(aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilderECRContainerBuilds'));
        role.addManagedPolicy(aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        if (typeof props.additionalPolicies !== 'undefined' && props.additionalPolicies.length >= 1) {
            for (const policy of props.additionalPolicies) {
                role.addManagedPolicy(policy);
            }
        }
        const instanceProfileName = props.profileName ?? `${stackName}-InstanceProfile`;
        const infraConfigName = `${stackName}-InfrastructureConfiguration`;
        const imageRecipeName = `${stackName}-ImageRecipe`;
        const imageRecipeVersion = props.imageRecipeVersion ?? '0.0.1';
        const pipelineBaseName = `${stackName}-ImagePipeline`;
        const pipelineName = `${pipelineBaseName}-${imageRecipeVersion.replace(/\./g, '-')}`;
        // Use the first volume configuration's KMS key ID as the effective KMS key ID
        // If the first volume configuration does not have a KMS key ID, use the default AMI encryption key
        // Keep in mind though that default AMI encryption key is not compatible with cross-account AMI sharing
        const distributionKeyAlias = props.distributionKmsKeyAlias ?? 'alias/aws/ebs';
        const ebsVolumeEffectiveKmsAlias = props.ebsVolumeConfigurations?.[0]?.ebs?.kmsKeyId
            ? props.ebsVolumeConfigurations[0].ebs.kmsKeyId
            : (props.distributionKmsKeyAlias ?? 'alias/aws/ebs');
        const distributionAccountIds = props.distributionAccountIds ?? [];
        const distributionRegions = props.distributionRegions ?? [];
        if ((props.distributionRegions?.length === 0 && props.distributionAccountIds?.length) ||
            (props.distributionRegions?.length && props.distributionAccountIds?.length === 0)) {
            throw new Error('distributionRegions must be specified if distributionAccountIds is specified');
        }
        if ((props.distributionRegions?.length && props.distributionAccountIds?.length) && props.distributionKmsKeyAlias === 'alias/aws/ebs') {
            throw new Error('If distributionRegions and distributionAccountIds are specified, distributionKmsKeyAlias must be specified');
        }
        const profile = new aws_cdk_lib_1.aws_iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [role.roleName],
            instanceProfileName: instanceProfileName,
        });
        const defaultGroup = new ec2.SecurityGroup(this, 'SecurityGroup', {
            vpc: props.vpc,
            allowAllOutbound: true,
        });
        if (props.securityGroups == null || props.subnetId == null) {
            infrastructureConfig = new aws_cdk_lib_1.aws_imagebuilder.CfnInfrastructureConfiguration(this, 'InfrastructureConfiguration', {
                instanceProfileName: instanceProfileName,
                name: infraConfigName,
                description: 'Infrastructure Configuration for Image Builder',
                instanceTypes: props.instanceTypes ?? ['t3.medium', 'm5.large', 'm5.xlarge'],
                snsTopicArn: topic.topicArn,
                securityGroupIds: [defaultGroup.securityGroupId],
                subnetId: props.vpc.publicSubnets[0].subnetId,
            });
        }
        else {
            infrastructureConfig = new aws_cdk_lib_1.aws_imagebuilder.CfnInfrastructureConfiguration(this, 'InfrastructureConfiguration', {
                instanceProfileName: instanceProfileName,
                name: infraConfigName,
                description: 'Infrastructure Configuration for Image Builder',
                instanceTypes: props.instanceTypes ?? ['t3.medium', 'm5.large', 'm5.xlarge'],
                snsTopicArn: topic.topicArn,
                securityGroupIds: props.securityGroups,
                subnetId: props.subnetId,
            });
        }
        infrastructureConfig.addDependency(profile);
        /**
             * Image recipe configuration
             */
        let imageRecipeProps;
        imageRecipeProps = {
            components: [],
            name: imageRecipeName,
            parentImage: props.parentImage,
            version: imageRecipeVersion ?? '0.0.1',
        };
        if (props.userDataScript) {
            imageRecipeProps = {
                ...imageRecipeProps,
                additionalInstanceConfiguration: {
                    userDataOverride: props.userDataScript,
                },
            };
        }
        if (props.ebsVolumeConfigurations?.[0].ebs) {
            // Go over each configuration and set the default
            for (const config of props.ebsVolumeConfigurations) {
                // If encryption is not set, enable it
                if (config.ebs.encrypted === undefined) {
                    config.ebs.encrypted = true;
                }
                // If KMS Key ID is not set, use the default
                if (!config.ebs.kmsKeyId) {
                    config.ebs.kmsKeyId = ebsVolumeEffectiveKmsAlias;
                }
                // If deleteOnTermination is not set, enable it
                if (!config.ebs.deleteOnTermination) {
                    config.ebs.deleteOnTermination = true;
                }
            }
            imageRecipeProps = {
                ...imageRecipeProps,
                blockDeviceMappings: props.ebsVolumeConfigurations,
            };
        }
        else {
            imageRecipeProps = {
                ...imageRecipeProps,
                blockDeviceMappings: [
                    {
                        deviceName: '/dev/sda1',
                        ebs: {
                            kmsKeyId: 'alias/aws/ebs',
                            encrypted: true,
                            volumeSize: 30,
                            volumeType: 'gp3',
                            deleteOnTermination: true,
                        },
                    },
                ],
            };
        }
        imageRecipe = new aws_cdk_lib_1.aws_imagebuilder.CfnImageRecipe(this, 'ImageRecipe', imageRecipeProps);
        props.components.forEach((component, index) => {
            const componentName = component.name?.replace(/ /g, '-');
            component.name = componentName;
            const componentHash = this.stringToMD5(JSON.stringify(component));
            const componentDocumentHash = this.stringToMD5(JSON.stringify(component.componentDocument));
            const componentDocument = {
                name: componentName ?? `${stackName}-${componentDocumentHash}-${index}`,
                description: component.description,
                schemaVersion: component.schemaVersion ?? '1.0',
                phases: component.componentDocument.phases,
            };
            const data = yaml.dump(componentDocument);
            let newComponent = new aws_cdk_lib_1.aws_imagebuilder.CfnComponent(this, component.name ?? `Component-${index}`, {
                name: component.name ?? `${stackName.toLowerCase()}-${componentHash}-${index}`,
                description: component.description,
                platform: component.platform ?? 'Linux',
                version: component.componentVersion ?? props.imageRecipeVersion ?? '0.0.1',
                data: data,
            });
            // add the component to the Image Recipe
            this.imageRecipeComponents.push({ componentArn: newComponent.attrArn });
            imageRecipe.components = this.imageRecipeComponents;
        });
        const imageRecipeHash = this.stringToMD5(JSON.stringify(this.imageRecipeComponents));
        imageRecipe.name = `${imageRecipeName}-${imageRecipeHash}-${imageRecipeVersion.replace(/\./g, '-')}`;
        let imagePipelineProps;
        imagePipelineProps = {
            infrastructureConfigurationArn: infrastructureConfig.attrArn,
            name: pipelineName,
            description: 'Image pipeline',
            imageRecipeArn: imageRecipe.attrArn,
        };
        if (props.enableVulnScans) {
            imagePipelineProps = {
                ...imagePipelineProps,
                imageScanningConfiguration: {
                    imageScanningEnabled: props.enableVulnScans,
                    ecrConfiguration: {
                        repositoryName: props.vulnScansRepoName,
                        containerTags: props.vulnScansRepoTags,
                    },
                },
            };
        }
        if (distributionAccountIds.length > 0 && distributionRegions.length > 0) {
            const distributionsList = [];
            distributionRegions.forEach(distributionRegion => {
                const distributionConfig = {
                    region: distributionRegion,
                    amiDistributionConfiguration: {
                        //Capital case here because it's an object of type any, but capital case is what is expected in CloudFormation
                        //https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-imagebuilder-distributionconfiguration-amidistributionconfiguration.html
                        Name: `${imageRecipeName}-${distributionRegion}-{{imagebuilder:buildDate}}`,
                        Description: `copy AMI ${imageRecipeName} to ${distributionRegion}`,
                        TargetAccountIds: distributionAccountIds,
                        LaunchPermissionConfiguration: {
                            UserIds: distributionAccountIds,
                        },
                        KmsKeyId: distributionKeyAlias, //using default AWS-managed key if one isn't given is prohibited
                    },
                };
                distributionsList.push(distributionConfig);
            });
            const amiDistributionConfiguration = new aws_cdk_lib_1.aws_imagebuilder.CfnDistributionConfiguration(this, 'amiDistributionConfiguration', {
                name: `${imageRecipeName}-distribution-config-${imageRecipeVersion.replace(/\./g, '-')}`,
                description: `Cross account distribution settings for ${imageRecipeName}`,
                distributions: distributionsList,
            });
            imagePipelineProps = {
                ...imagePipelineProps,
                distributionConfigurationArn: amiDistributionConfiguration.attrArn,
            };
        }
        const imageBuildPipeline = new aws_cdk_lib_1.aws_imagebuilder.CfnImagePipeline(this, 'ImagePipeline', imagePipelineProps);
        this.imagePipelineArn = imageBuildPipeline.attrArn;
        const startStateMachineFunction = new StartStateMachine_function_1.StartStateMachineFunction(this, 'StartStateMachineFunction', {
            description: 'StartStateMachineFunction',
            memorySize: 128,
            timeout: aws_cdk_lib_1.Duration.minutes(12),
            initialPolicy: [
                new cdk_iam_floyd_1.Statement.States().allow()
                    .toStartExecution()
                    .toGetExecutionHistory()
                    .toListExecutions()
                    .toDescribeExecution()
                    .onAllResources(),
                new cdk_iam_floyd_1.Statement.Imagebuilder().allow()
                    .toListImagePipelines()
                    .toStartImagePipelineExecution()
                    .toGetImage()
                    .onAllResources(),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
                    actions: [
                        'kms:*',
                    ],
                    // arn:{partition}:{service}:{region}:{account}:{resource}{sep}{resource-name}/
                    resources: [aws_cdk_lib_1.Arn.format({
                            service: 'kms',
                            resource: 'alias/aws/ebs',
                            account: '*',
                            region: '*',
                            partition: 'aws',
                        })],
                }),
            ],
        });
        const checkStateMachineStatusFunction = new CheckStateMachineStatus_function_1.CheckStateMachineStatusFunction(this, 'CheckStateMachineStatusFunction', {
            description: 'CheckStateMachineStatusFunction',
            memorySize: 128,
            timeout: aws_cdk_lib_1.Duration.minutes(12),
            initialPolicy: [
                new cdk_iam_floyd_1.Statement.States().allow()
                    .toDescribeExecution()
                    .toListExecutions()
                    .toGetExecutionHistory()
                    .onAllResources(),
                new cdk_iam_floyd_1.Statement.Imagebuilder().allow()
                    .toListImagePipelines()
                    .toStartImagePipelineExecution()
                    .toGetImage()
                    .onAllResources(),
                new cdk_iam_floyd_1.Statement.Ec2().allow()
                    .toModifyImageAttribute()
                    .onAllResources(),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
                    actions: [
                        'kms:*',
                    ],
                    resources: [aws_cdk_lib_1.Arn.format({
                            service: 'kms',
                            resource: 'alias/aws/ebs',
                            account: '*',
                            region: '*',
                            partition: 'aws',
                        })],
                }),
            ],
        });
        const provider = new custom_resources_1.Provider(this, 'ResourceProvider', {
            onEventHandler: startStateMachineFunction,
            isCompleteHandler: checkStateMachineStatusFunction,
            totalTimeout: aws_cdk_lib_1.Duration.minutes(60),
            queryInterval: aws_cdk_lib_1.Duration.minutes(2),
        });
        const stateMachineRole = new aws_iam_1.Role(this, 'StateMachineRole', {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('states.amazonaws.com'),
            managedPolicies: [
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaRole'),
            ],
            inlinePolicies: {
                StateMachinePolicy: new aws_cdk_lib_1.aws_iam.PolicyDocument({
                    statements: [
                        new cdk_iam_floyd_1.Statement.Xray().allow()
                            .toPutTraceSegments()
                            .toPutTelemetryRecords()
                            .toGetSamplingRules()
                            .toGetSamplingTargets()
                            .onAllResources(),
                        new aws_cdk_lib_1.aws_iam.PolicyStatement({
                            effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
                            actions: [
                                'imagebuilder:*',
                            ],
                            resources: [`arn:aws:imagebuilder:${region ?? '*'}:${account ?? '*'}:image-pipeline/${pipelineBaseName.toLowerCase()}-*`],
                        }),
                        new cdk_iam_floyd_1.Statement.Imagebuilder().allow()
                            .toGetImage()
                            .onAllResources(),
                    ],
                }),
            },
        });
        const stateMachine = new StateMachine_1.StateMachine(this, 'StateMachine', {
            definition: JSON.parse(fs.readFileSync(path.join(__dirname, '..', 'StateMachineFiles', 'ImageBuildingStateMachine.json'), 'utf8').toString()),
            role: stateMachineRole,
        });
        const effectiveRemovalPolicy = props.debugImagePipeline === true ? aws_cdk_lib_1.RemovalPolicy.RETAIN_ON_UPDATE_OR_DELETE : aws_cdk_lib_1.RemovalPolicy.DESTROY;
        const componentsHash = this.stringToMD5(JSON.stringify(props.components));
        const imagePipeline = new aws_cdk_lib_1.CustomResource(this, 'CustomResource', {
            serviceToken: provider.serviceToken,
            properties: {
                ImagePipelineArn: imageBuildPipeline.attrArn,
                StateMachineArn: stateMachine.stateMachineArn,
                Version: props.imageRecipeVersion ?? '0.0.1',
                ComponentsHash: componentsHash,
                PhysicalResourceId: `${stackName}-ImagePipeline-${imageRecipeVersion.replace(/\./g, '-')}`,
                AmiKmsId: distributionKeyAlias,
                SharedAccountIds: distributionAccountIds.join(','),
                DebugImagePipeline: props.debugImagePipeline === true ? 'true' : 'false',
            },
            removalPolicy: effectiveRemovalPolicy,
        });
        this.imageId = imagePipeline.getAttString('AmiId');
    }
    stringToMD5(inputString) {
        return CryptoJS.MD5(inputString).toString();
    }
}
exports.ImagePipeline = ImagePipeline;
_a = JSII_RTTI_SYMBOL_1;
ImagePipeline[_a] = { fqn: "@jjrawlins/cdk-ami-builder.ImagePipeline", version: "0.0.12" };
//# sourceMappingURL=data:application/json;base64,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