"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
exports.onCreate = onCreate;
exports.onDelete = onDelete;
exports.onUpdate = onUpdate;
// eslint-disable-next-line import/no-extraneous-dependencies
const client_sfn_1 = require("@aws-sdk/client-sfn");
const region = process.env.AWS_REGION;
const sfnClient = new client_sfn_1.SFNClient({ region: region }); // Replace with your region
const handler = async (event, context) => {
    console.log('Lambda is invoked with:' + JSON.stringify(event));
    console.log('Context is invoked with:' + JSON.stringify(context));
    console.log('Lambda is invoked with:' + JSON.stringify(event));
    const physicalResourceId = event.ResourceProperties.PhysicalResourceId;
    let response = {
        Status: 'SUCCESS',
        Reason: 'See the details in CloudWatch Log Stream: ' + context.logStreamName,
        PhysicalResourceId: physicalResourceId,
        StackId: event.StackId,
        RequestId: event.RequestId,
        LogicalResourceId: event.LogicalResourceId,
    };
    switch (event.RequestType) {
        case 'Create':
            response = await onCreate(response, event, context);
            break;
        case 'Delete':
            response = await onDelete(response, event, context);
            break;
        case 'Update':
            response = await onUpdate(response, event, context);
            break;
        default:
            throw new Error('Unknown Request Type of CloudFormation');
    }
    console.log('Return value:', JSON.stringify(response));
    return response;
};
exports.handler = handler;
/**
 * Executes the create event for a CloudFormation custom resource.
 * @param {CdkCustomResourceResponse} response - The custom resource response object.
 * @param {CloudFormationCustomResourceCreateEvent} event - The create event object.
 * @param {Context} context - The AWS Lambda context object.
 * @return {Promise<CdkCustomResourceResponse>} - A promise that resolves to the custom resource response object.
 */
async function onCreate(response, event, context) {
    try {
        console.log('We are in the Create Event');
        console.log('Context is invoked with:' + JSON.stringify(context));
        const stateMachineArn = event.ResourceProperties.StateMachineArn;
        const imagePipelineArn = event.ResourceProperties.ImagePipelineArn;
        const executionName = event.RequestId;
        await startExecution(stateMachineArn, executionName, imagePipelineArn);
    }
    catch (error) {
        if (error instanceof Error) {
            response.Reason = error.message;
        }
        response.Status = 'FAILED';
        response.Data = { Result: error };
        return response;
    }
    return response;
}
/**
 * Handles delete event for a cloud formation custom resource.
 *
 * @param {CdkCustomResourceResponse} response - The custom resource response object.
 * @param {CloudFormationCustomResourceDeleteEvent} event - The delete event object.
 * @param {Context} context - The AWS lambda context object.
 * @returns {Promise<CdkCustomResourceResponse>} - The updated custom resource response object.
 */
async function onDelete(response, event, context) {
    try {
        console.log('We are in the Delete Event');
        console.log('Context is invoked with:' + JSON.stringify(context));
        console.log('Event is invoked with:' + JSON.stringify(event));
        response.Status = 'SUCCESS';
    }
    catch (error) {
        if (error instanceof Error) {
            response.Reason = error.message;
        }
        response.Status = 'FAILED';
        response.Data = { Result: error };
        return response;
    }
    return response;
}
/**
 * Executes the onUpdate logic for a CloudFormation custom resource.
 *
 * @param {CdkCustomResourceResponse} response - The response object for the custom resource.
 * @param {CloudFormationCustomResourceUpdateEvent} event - The update event object from CloudFormation.
 * @param {Context} context - The execution context object.
 * @returns {Promise<CdkCustomResourceResponse>} A promise that resolves to the updated response object.
 */
async function onUpdate(response, event, context) {
    try {
        const executionName = event.RequestId;
        const imagePipelineArn = event.ResourceProperties.ImagePipelineArn;
        const stateMachineArn = event.ResourceProperties.StateMachineArn;
        const amiKmsId = event.ResourceProperties.AmiKmsId;
        const accounts = event.ResourceProperties.SharedAccountIds.split(',');
        console.log('Resource properties: ' + JSON.stringify(event.ResourceProperties));
        console.log('Context is invoked with:' + JSON.stringify(context));
        console.log('State Machine ARN: ' + stateMachineArn);
        console.log('Image Pipeline ARN: ' + imagePipelineArn);
        console.log('Execution Name: ' + executionName);
        console.log('Key Id: ' + amiKmsId);
        console.log('Accounts: ' + accounts);
        await startExecution(stateMachineArn, executionName, imagePipelineArn);
    }
    catch (error) {
        if (error instanceof Error) {
            response.Reason = error.message;
        }
        response.Status = 'FAILED';
        response.Data = { Result: error };
        return response;
    }
    return response;
}
/**
 * Starts the execution of a state machine.
 *
 * @param {string} stateMachineArn - The ARN (Amazon Resource Name) of the state machine.
 * @param {string} executionName - The name for the execution.
 * @param {string} imagePipelineArn - The ARN of the image pipeline.
 *
 * @return {Promise<void>} - A Promise that resolves when the execution has been started.
 */
async function startExecution(stateMachineArn, executionName, imagePipelineArn) {
    const startExecutionCommand = new client_sfn_1.StartExecutionCommand({
        stateMachineArn: stateMachineArn,
        name: executionName,
        input: JSON.stringify({
            clientToken: executionName,
            imagePipelineArn: imagePipelineArn,
        }),
    });
    await sfnClient.send(startExecutionCommand);
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiU3RhcnRTdGF0ZU1hY2hpbmUubGFtYmRhLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vc3JjL0xhbWJkYXMvU3RhcnRTdGF0ZU1hY2hpbmUvU3RhcnRTdGF0ZU1hY2hpbmUubGFtYmRhLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQXdEQSw0QkFvQkM7QUFVRCw0QkFrQkM7QUFVRCw0QkEyQkM7QUE3SUQsNkRBQTZEO0FBQzdELG9EQUF1RTtBQVV2RSxNQUFNLE1BQU0sR0FBRyxPQUFPLENBQUMsR0FBRyxDQUFDLFVBQVUsQ0FBQztBQUN0QyxNQUFNLFNBQVMsR0FBRyxJQUFJLHNCQUFTLENBQUMsRUFBRSxNQUFNLEVBQUUsTUFBTSxFQUFFLENBQUMsQ0FBQyxDQUFDLDJCQUEyQjtBQUV6RSxNQUFNLE9BQU8sR0FBRyxLQUFLLEVBQzFCLEtBQXdDLEVBQ3hDLE9BQWdCLEVBQ29CLEVBQUU7SUFDdEMsT0FBTyxDQUFDLEdBQUcsQ0FBQyx5QkFBeUIsR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7SUFDL0QsT0FBTyxDQUFDLEdBQUcsQ0FBQywwQkFBMEIsR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7SUFDbEUsT0FBTyxDQUFDLEdBQUcsQ0FBQyx5QkFBeUIsR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7SUFDL0QsTUFBTSxrQkFBa0IsR0FBRyxLQUFLLENBQUMsa0JBQWtCLENBQUMsa0JBQWtCLENBQUM7SUFFdkUsSUFBSSxRQUFRLEdBQThCO1FBQ3hDLE1BQU0sRUFBRSxTQUFTO1FBQ2pCLE1BQU0sRUFBRSw0Q0FBNEMsR0FBRyxPQUFPLENBQUMsYUFBYTtRQUM1RSxrQkFBa0IsRUFBRSxrQkFBa0I7UUFDdEMsT0FBTyxFQUFFLEtBQUssQ0FBQyxPQUFPO1FBQ3RCLFNBQVMsRUFBRSxLQUFLLENBQUMsU0FBUztRQUMxQixpQkFBaUIsRUFBRSxLQUFLLENBQUMsaUJBQWlCO0tBQzNDLENBQUM7SUFFRixRQUFRLEtBQUssQ0FBQyxXQUFXLEVBQUUsQ0FBQztRQUMxQixLQUFLLFFBQVE7WUFDWCxRQUFRLEdBQUcsTUFBTSxRQUFRLENBQUMsUUFBUSxFQUFFLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztZQUNwRCxNQUFNO1FBQ1IsS0FBSyxRQUFRO1lBQ1gsUUFBUSxHQUFHLE1BQU0sUUFBUSxDQUFDLFFBQVEsRUFBRSxLQUFLLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDcEQsTUFBTTtRQUNSLEtBQUssUUFBUTtZQUNYLFFBQVEsR0FBRyxNQUFNLFFBQVEsQ0FBQyxRQUFRLEVBQUUsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ3BELE1BQU07UUFDUjtZQUNFLE1BQU0sSUFBSSxLQUFLLENBQUMsd0NBQXdDLENBQUMsQ0FBQztJQUM5RCxDQUFDO0lBQ0QsT0FBTyxDQUFDLEdBQUcsQ0FBQyxlQUFlLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDO0lBQ3ZELE9BQU8sUUFBUSxDQUFDO0FBQ2xCLENBQUMsQ0FBQztBQWpDVyxRQUFBLE9BQU8sV0FpQ2xCO0FBRUY7Ozs7OztHQU1HO0FBQ0ksS0FBSyxVQUFVLFFBQVEsQ0FDNUIsUUFBbUMsRUFDbkMsS0FBOEMsRUFDOUMsT0FBZ0I7SUFDaEIsSUFBSSxDQUFDO1FBQ0gsT0FBTyxDQUFDLEdBQUcsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1FBQzFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsMEJBQTBCLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO1FBQ2xFLE1BQU0sZUFBZSxHQUFHLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxlQUFlLENBQUM7UUFDakUsTUFBTSxnQkFBZ0IsR0FBRyxLQUFLLENBQUMsa0JBQWtCLENBQUMsZ0JBQWdCLENBQUM7UUFDbkUsTUFBTSxhQUFhLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQztRQUN0QyxNQUFNLGNBQWMsQ0FBQyxlQUFlLEVBQUUsYUFBYSxFQUFFLGdCQUFnQixDQUFDLENBQUM7SUFDekUsQ0FBQztJQUFDLE9BQU8sS0FBSyxFQUFFLENBQUM7UUFDZixJQUFJLEtBQUssWUFBWSxLQUFLLEVBQUUsQ0FBQztZQUMzQixRQUFRLENBQUMsTUFBTSxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUM7UUFDbEMsQ0FBQztRQUNELFFBQVEsQ0FBQyxNQUFNLEdBQUcsUUFBUSxDQUFDO1FBQzNCLFFBQVEsQ0FBQyxJQUFJLEdBQUcsRUFBRSxNQUFNLEVBQUUsS0FBSyxFQUFFLENBQUM7UUFDbEMsT0FBTyxRQUFRLENBQUM7SUFDbEIsQ0FBQztJQUNELE9BQU8sUUFBUSxDQUFDO0FBQ2xCLENBQUM7QUFFRDs7Ozs7OztHQU9HO0FBQ0ksS0FBSyxVQUFVLFFBQVEsQ0FDNUIsUUFBbUMsRUFDbkMsS0FBOEMsRUFDOUMsT0FBZ0I7SUFDaEIsSUFBSSxDQUFDO1FBQ0gsT0FBTyxDQUFDLEdBQUcsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1FBQzFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsMEJBQTBCLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO1FBQ2xFLE9BQU8sQ0FBQyxHQUFHLENBQUMsd0JBQXdCLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO1FBQzlELFFBQVEsQ0FBQyxNQUFNLEdBQUcsU0FBUyxDQUFDO0lBQzlCLENBQUM7SUFBQyxPQUFPLEtBQUssRUFBRSxDQUFDO1FBQ2YsSUFBSSxLQUFLLFlBQVksS0FBSyxFQUFFLENBQUM7WUFDM0IsUUFBUSxDQUFDLE1BQU0sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBQ2xDLENBQUM7UUFDRCxRQUFRLENBQUMsTUFBTSxHQUFHLFFBQVEsQ0FBQztRQUMzQixRQUFRLENBQUMsSUFBSSxHQUFHLEVBQUUsTUFBTSxFQUFFLEtBQUssRUFBRSxDQUFDO1FBQ2xDLE9BQU8sUUFBUSxDQUFDO0lBQ2xCLENBQUM7SUFDRCxPQUFPLFFBQVEsQ0FBQztBQUNsQixDQUFDO0FBRUQ7Ozs7Ozs7R0FPRztBQUNJLEtBQUssVUFBVSxRQUFRLENBQzVCLFFBQW1DLEVBQ25DLEtBQThDLEVBQzlDLE9BQWdCO0lBQ2hCLElBQUksQ0FBQztRQUNILE1BQU0sYUFBYSxHQUFHLEtBQUssQ0FBQyxTQUFTLENBQUM7UUFDdEMsTUFBTSxnQkFBZ0IsR0FBRyxLQUFLLENBQUMsa0JBQWtCLENBQUMsZ0JBQWdCLENBQUM7UUFDbkUsTUFBTSxlQUFlLEdBQUcsS0FBSyxDQUFDLGtCQUFrQixDQUFDLGVBQWUsQ0FBQztRQUNqRSxNQUFNLFFBQVEsR0FBRyxLQUFLLENBQUMsa0JBQWtCLENBQUMsUUFBUSxDQUFDO1FBQ25ELE1BQU0sUUFBUSxHQUFHLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDdEUsT0FBTyxDQUFDLEdBQUcsQ0FBQyx1QkFBdUIsR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUM7UUFDaEYsT0FBTyxDQUFDLEdBQUcsQ0FBQywwQkFBMEIsR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7UUFDbEUsT0FBTyxDQUFDLEdBQUcsQ0FBQyxxQkFBcUIsR0FBRyxlQUFlLENBQUMsQ0FBQztRQUNyRCxPQUFPLENBQUMsR0FBRyxDQUFDLHNCQUFzQixHQUFHLGdCQUFnQixDQUFDLENBQUM7UUFDdkQsT0FBTyxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsR0FBRyxhQUFhLENBQUMsQ0FBQztRQUNoRCxPQUFPLENBQUMsR0FBRyxDQUFDLFVBQVUsR0FBRyxRQUFRLENBQUMsQ0FBQztRQUNuQyxPQUFPLENBQUMsR0FBRyxDQUFDLFlBQVksR0FBRyxRQUFRLENBQUMsQ0FBQztRQUNyQyxNQUFNLGNBQWMsQ0FBQyxlQUFlLEVBQUUsYUFBYSxFQUFFLGdCQUFnQixDQUFDLENBQUM7SUFDekUsQ0FBQztJQUFDLE9BQU8sS0FBSyxFQUFFLENBQUM7UUFDZixJQUFJLEtBQUssWUFBWSxLQUFLLEVBQUUsQ0FBQztZQUMzQixRQUFRLENBQUMsTUFBTSxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUM7UUFDbEMsQ0FBQztRQUNELFFBQVEsQ0FBQyxNQUFNLEdBQUcsUUFBUSxDQUFDO1FBQzNCLFFBQVEsQ0FBQyxJQUFJLEdBQUcsRUFBRSxNQUFNLEVBQUUsS0FBSyxFQUFFLENBQUM7UUFDbEMsT0FBTyxRQUFRLENBQUM7SUFDbEIsQ0FBQztJQUNELE9BQU8sUUFBUSxDQUFDO0FBQ2xCLENBQUM7QUFFRDs7Ozs7Ozs7R0FRRztBQUNILEtBQUssVUFBVSxjQUFjLENBQUMsZUFBdUIsRUFBRSxhQUFxQixFQUFFLGdCQUF3QjtJQUNwRyxNQUFNLHFCQUFxQixHQUFHLElBQUksa0NBQXFCLENBQUM7UUFDdEQsZUFBZSxFQUFFLGVBQWU7UUFDaEMsSUFBSSxFQUFFLGFBQWE7UUFDbkIsS0FBSyxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUM7WUFDcEIsV0FBVyxFQUFFLGFBQWE7WUFDMUIsZ0JBQWdCLEVBQUUsZ0JBQWdCO1NBQ25DLENBQUM7S0FDSCxDQUFDLENBQUM7SUFDSCxNQUFNLFNBQVMsQ0FBQyxJQUFJLENBQUMscUJBQXFCLENBQUMsQ0FBQztBQUM5QyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIGltcG9ydC9uby1leHRyYW5lb3VzLWRlcGVuZGVuY2llc1xuaW1wb3J0IHsgU0ZOQ2xpZW50LCBTdGFydEV4ZWN1dGlvbkNvbW1hbmQgfSBmcm9tICdAYXdzLXNkay9jbGllbnQtc2ZuJztcbmltcG9ydCB7XG4gIENka0N1c3RvbVJlc291cmNlUmVzcG9uc2UsXG4gIENsb3VkRm9ybWF0aW9uQ3VzdG9tUmVzb3VyY2VDcmVhdGVFdmVudCxcbiAgQ2xvdWRGb3JtYXRpb25DdXN0b21SZXNvdXJjZURlbGV0ZUV2ZW50LFxuICBDbG91ZEZvcm1hdGlvbkN1c3RvbVJlc291cmNlRXZlbnQsXG4gIENsb3VkRm9ybWF0aW9uQ3VzdG9tUmVzb3VyY2VVcGRhdGVFdmVudCxcbiAgQ29udGV4dCxcbn0gZnJvbSAnYXdzLWxhbWJkYSc7XG5cbmNvbnN0IHJlZ2lvbiA9IHByb2Nlc3MuZW52LkFXU19SRUdJT047XG5jb25zdCBzZm5DbGllbnQgPSBuZXcgU0ZOQ2xpZW50KHsgcmVnaW9uOiByZWdpb24gfSk7IC8vIFJlcGxhY2Ugd2l0aCB5b3VyIHJlZ2lvblxuXG5leHBvcnQgY29uc3QgaGFuZGxlciA9IGFzeW5jIChcbiAgZXZlbnQ6IENsb3VkRm9ybWF0aW9uQ3VzdG9tUmVzb3VyY2VFdmVudCxcbiAgY29udGV4dDogQ29udGV4dCxcbik6IFByb21pc2U8Q2RrQ3VzdG9tUmVzb3VyY2VSZXNwb25zZT4gPT4ge1xuICBjb25zb2xlLmxvZygnTGFtYmRhIGlzIGludm9rZWQgd2l0aDonICsgSlNPTi5zdHJpbmdpZnkoZXZlbnQpKTtcbiAgY29uc29sZS5sb2coJ0NvbnRleHQgaXMgaW52b2tlZCB3aXRoOicgKyBKU09OLnN0cmluZ2lmeShjb250ZXh0KSk7XG4gIGNvbnNvbGUubG9nKCdMYW1iZGEgaXMgaW52b2tlZCB3aXRoOicgKyBKU09OLnN0cmluZ2lmeShldmVudCkpO1xuICBjb25zdCBwaHlzaWNhbFJlc291cmNlSWQgPSBldmVudC5SZXNvdXJjZVByb3BlcnRpZXMuUGh5c2ljYWxSZXNvdXJjZUlkO1xuXG4gIGxldCByZXNwb25zZTogQ2RrQ3VzdG9tUmVzb3VyY2VSZXNwb25zZSA9IHtcbiAgICBTdGF0dXM6ICdTVUNDRVNTJyxcbiAgICBSZWFzb246ICdTZWUgdGhlIGRldGFpbHMgaW4gQ2xvdWRXYXRjaCBMb2cgU3RyZWFtOiAnICsgY29udGV4dC5sb2dTdHJlYW1OYW1lLFxuICAgIFBoeXNpY2FsUmVzb3VyY2VJZDogcGh5c2ljYWxSZXNvdXJjZUlkLFxuICAgIFN0YWNrSWQ6IGV2ZW50LlN0YWNrSWQsXG4gICAgUmVxdWVzdElkOiBldmVudC5SZXF1ZXN0SWQsXG4gICAgTG9naWNhbFJlc291cmNlSWQ6IGV2ZW50LkxvZ2ljYWxSZXNvdXJjZUlkLFxuICB9O1xuXG4gIHN3aXRjaCAoZXZlbnQuUmVxdWVzdFR5cGUpIHtcbiAgICBjYXNlICdDcmVhdGUnOlxuICAgICAgcmVzcG9uc2UgPSBhd2FpdCBvbkNyZWF0ZShyZXNwb25zZSwgZXZlbnQsIGNvbnRleHQpO1xuICAgICAgYnJlYWs7XG4gICAgY2FzZSAnRGVsZXRlJzpcbiAgICAgIHJlc3BvbnNlID0gYXdhaXQgb25EZWxldGUocmVzcG9uc2UsIGV2ZW50LCBjb250ZXh0KTtcbiAgICAgIGJyZWFrO1xuICAgIGNhc2UgJ1VwZGF0ZSc6XG4gICAgICByZXNwb25zZSA9IGF3YWl0IG9uVXBkYXRlKHJlc3BvbnNlLCBldmVudCwgY29udGV4dCk7XG4gICAgICBicmVhaztcbiAgICBkZWZhdWx0OlxuICAgICAgdGhyb3cgbmV3IEVycm9yKCdVbmtub3duIFJlcXVlc3QgVHlwZSBvZiBDbG91ZEZvcm1hdGlvbicpO1xuICB9XG4gIGNvbnNvbGUubG9nKCdSZXR1cm4gdmFsdWU6JywgSlNPTi5zdHJpbmdpZnkocmVzcG9uc2UpKTtcbiAgcmV0dXJuIHJlc3BvbnNlO1xufTtcblxuLyoqXG4gKiBFeGVjdXRlcyB0aGUgY3JlYXRlIGV2ZW50IGZvciBhIENsb3VkRm9ybWF0aW9uIGN1c3RvbSByZXNvdXJjZS5cbiAqIEBwYXJhbSB7Q2RrQ3VzdG9tUmVzb3VyY2VSZXNwb25zZX0gcmVzcG9uc2UgLSBUaGUgY3VzdG9tIHJlc291cmNlIHJlc3BvbnNlIG9iamVjdC5cbiAqIEBwYXJhbSB7Q2xvdWRGb3JtYXRpb25DdXN0b21SZXNvdXJjZUNyZWF0ZUV2ZW50fSBldmVudCAtIFRoZSBjcmVhdGUgZXZlbnQgb2JqZWN0LlxuICogQHBhcmFtIHtDb250ZXh0fSBjb250ZXh0IC0gVGhlIEFXUyBMYW1iZGEgY29udGV4dCBvYmplY3QuXG4gKiBAcmV0dXJuIHtQcm9taXNlPENka0N1c3RvbVJlc291cmNlUmVzcG9uc2U+fSAtIEEgcHJvbWlzZSB0aGF0IHJlc29sdmVzIHRvIHRoZSBjdXN0b20gcmVzb3VyY2UgcmVzcG9uc2Ugb2JqZWN0LlxuICovXG5leHBvcnQgYXN5bmMgZnVuY3Rpb24gb25DcmVhdGUoXG4gIHJlc3BvbnNlOiBDZGtDdXN0b21SZXNvdXJjZVJlc3BvbnNlLFxuICBldmVudDogQ2xvdWRGb3JtYXRpb25DdXN0b21SZXNvdXJjZUNyZWF0ZUV2ZW50LFxuICBjb250ZXh0OiBDb250ZXh0KTogUHJvbWlzZTxDZGtDdXN0b21SZXNvdXJjZVJlc3BvbnNlPiB7XG4gIHRyeSB7XG4gICAgY29uc29sZS5sb2coJ1dlIGFyZSBpbiB0aGUgQ3JlYXRlIEV2ZW50Jyk7XG4gICAgY29uc29sZS5sb2coJ0NvbnRleHQgaXMgaW52b2tlZCB3aXRoOicgKyBKU09OLnN0cmluZ2lmeShjb250ZXh0KSk7XG4gICAgY29uc3Qgc3RhdGVNYWNoaW5lQXJuID0gZXZlbnQuUmVzb3VyY2VQcm9wZXJ0aWVzLlN0YXRlTWFjaGluZUFybjtcbiAgICBjb25zdCBpbWFnZVBpcGVsaW5lQXJuID0gZXZlbnQuUmVzb3VyY2VQcm9wZXJ0aWVzLkltYWdlUGlwZWxpbmVBcm47XG4gICAgY29uc3QgZXhlY3V0aW9uTmFtZSA9IGV2ZW50LlJlcXVlc3RJZDtcbiAgICBhd2FpdCBzdGFydEV4ZWN1dGlvbihzdGF0ZU1hY2hpbmVBcm4sIGV4ZWN1dGlvbk5hbWUsIGltYWdlUGlwZWxpbmVBcm4pO1xuICB9IGNhdGNoIChlcnJvcikge1xuICAgIGlmIChlcnJvciBpbnN0YW5jZW9mIEVycm9yKSB7XG4gICAgICByZXNwb25zZS5SZWFzb24gPSBlcnJvci5tZXNzYWdlO1xuICAgIH1cbiAgICByZXNwb25zZS5TdGF0dXMgPSAnRkFJTEVEJztcbiAgICByZXNwb25zZS5EYXRhID0geyBSZXN1bHQ6IGVycm9yIH07XG4gICAgcmV0dXJuIHJlc3BvbnNlO1xuICB9XG4gIHJldHVybiByZXNwb25zZTtcbn1cblxuLyoqXG4gKiBIYW5kbGVzIGRlbGV0ZSBldmVudCBmb3IgYSBjbG91ZCBmb3JtYXRpb24gY3VzdG9tIHJlc291cmNlLlxuICpcbiAqIEBwYXJhbSB7Q2RrQ3VzdG9tUmVzb3VyY2VSZXNwb25zZX0gcmVzcG9uc2UgLSBUaGUgY3VzdG9tIHJlc291cmNlIHJlc3BvbnNlIG9iamVjdC5cbiAqIEBwYXJhbSB7Q2xvdWRGb3JtYXRpb25DdXN0b21SZXNvdXJjZURlbGV0ZUV2ZW50fSBldmVudCAtIFRoZSBkZWxldGUgZXZlbnQgb2JqZWN0LlxuICogQHBhcmFtIHtDb250ZXh0fSBjb250ZXh0IC0gVGhlIEFXUyBsYW1iZGEgY29udGV4dCBvYmplY3QuXG4gKiBAcmV0dXJucyB7UHJvbWlzZTxDZGtDdXN0b21SZXNvdXJjZVJlc3BvbnNlPn0gLSBUaGUgdXBkYXRlZCBjdXN0b20gcmVzb3VyY2UgcmVzcG9uc2Ugb2JqZWN0LlxuICovXG5leHBvcnQgYXN5bmMgZnVuY3Rpb24gb25EZWxldGUoXG4gIHJlc3BvbnNlOiBDZGtDdXN0b21SZXNvdXJjZVJlc3BvbnNlLFxuICBldmVudDogQ2xvdWRGb3JtYXRpb25DdXN0b21SZXNvdXJjZURlbGV0ZUV2ZW50LFxuICBjb250ZXh0OiBDb250ZXh0KTogUHJvbWlzZTxDZGtDdXN0b21SZXNvdXJjZVJlc3BvbnNlPiB7XG4gIHRyeSB7XG4gICAgY29uc29sZS5sb2coJ1dlIGFyZSBpbiB0aGUgRGVsZXRlIEV2ZW50Jyk7XG4gICAgY29uc29sZS5sb2coJ0NvbnRleHQgaXMgaW52b2tlZCB3aXRoOicgKyBKU09OLnN0cmluZ2lmeShjb250ZXh0KSk7XG4gICAgY29uc29sZS5sb2coJ0V2ZW50IGlzIGludm9rZWQgd2l0aDonICsgSlNPTi5zdHJpbmdpZnkoZXZlbnQpKTtcbiAgICByZXNwb25zZS5TdGF0dXMgPSAnU1VDQ0VTUyc7XG4gIH0gY2F0Y2ggKGVycm9yKSB7XG4gICAgaWYgKGVycm9yIGluc3RhbmNlb2YgRXJyb3IpIHtcbiAgICAgIHJlc3BvbnNlLlJlYXNvbiA9IGVycm9yLm1lc3NhZ2U7XG4gICAgfVxuICAgIHJlc3BvbnNlLlN0YXR1cyA9ICdGQUlMRUQnO1xuICAgIHJlc3BvbnNlLkRhdGEgPSB7IFJlc3VsdDogZXJyb3IgfTtcbiAgICByZXR1cm4gcmVzcG9uc2U7XG4gIH1cbiAgcmV0dXJuIHJlc3BvbnNlO1xufVxuXG4vKipcbiAqIEV4ZWN1dGVzIHRoZSBvblVwZGF0ZSBsb2dpYyBmb3IgYSBDbG91ZEZvcm1hdGlvbiBjdXN0b20gcmVzb3VyY2UuXG4gKlxuICogQHBhcmFtIHtDZGtDdXN0b21SZXNvdXJjZVJlc3BvbnNlfSByZXNwb25zZSAtIFRoZSByZXNwb25zZSBvYmplY3QgZm9yIHRoZSBjdXN0b20gcmVzb3VyY2UuXG4gKiBAcGFyYW0ge0Nsb3VkRm9ybWF0aW9uQ3VzdG9tUmVzb3VyY2VVcGRhdGVFdmVudH0gZXZlbnQgLSBUaGUgdXBkYXRlIGV2ZW50IG9iamVjdCBmcm9tIENsb3VkRm9ybWF0aW9uLlxuICogQHBhcmFtIHtDb250ZXh0fSBjb250ZXh0IC0gVGhlIGV4ZWN1dGlvbiBjb250ZXh0IG9iamVjdC5cbiAqIEByZXR1cm5zIHtQcm9taXNlPENka0N1c3RvbVJlc291cmNlUmVzcG9uc2U+fSBBIHByb21pc2UgdGhhdCByZXNvbHZlcyB0byB0aGUgdXBkYXRlZCByZXNwb25zZSBvYmplY3QuXG4gKi9cbmV4cG9ydCBhc3luYyBmdW5jdGlvbiBvblVwZGF0ZShcbiAgcmVzcG9uc2U6IENka0N1c3RvbVJlc291cmNlUmVzcG9uc2UsXG4gIGV2ZW50OiBDbG91ZEZvcm1hdGlvbkN1c3RvbVJlc291cmNlVXBkYXRlRXZlbnQsXG4gIGNvbnRleHQ6IENvbnRleHQpOiBQcm9taXNlPENka0N1c3RvbVJlc291cmNlUmVzcG9uc2U+IHtcbiAgdHJ5IHtcbiAgICBjb25zdCBleGVjdXRpb25OYW1lID0gZXZlbnQuUmVxdWVzdElkO1xuICAgIGNvbnN0IGltYWdlUGlwZWxpbmVBcm4gPSBldmVudC5SZXNvdXJjZVByb3BlcnRpZXMuSW1hZ2VQaXBlbGluZUFybjtcbiAgICBjb25zdCBzdGF0ZU1hY2hpbmVBcm4gPSBldmVudC5SZXNvdXJjZVByb3BlcnRpZXMuU3RhdGVNYWNoaW5lQXJuO1xuICAgIGNvbnN0IGFtaUttc0lkID0gZXZlbnQuUmVzb3VyY2VQcm9wZXJ0aWVzLkFtaUttc0lkO1xuICAgIGNvbnN0IGFjY291bnRzID0gZXZlbnQuUmVzb3VyY2VQcm9wZXJ0aWVzLlNoYXJlZEFjY291bnRJZHMuc3BsaXQoJywnKTtcbiAgICBjb25zb2xlLmxvZygnUmVzb3VyY2UgcHJvcGVydGllczogJyArIEpTT04uc3RyaW5naWZ5KGV2ZW50LlJlc291cmNlUHJvcGVydGllcykpO1xuICAgIGNvbnNvbGUubG9nKCdDb250ZXh0IGlzIGludm9rZWQgd2l0aDonICsgSlNPTi5zdHJpbmdpZnkoY29udGV4dCkpO1xuICAgIGNvbnNvbGUubG9nKCdTdGF0ZSBNYWNoaW5lIEFSTjogJyArIHN0YXRlTWFjaGluZUFybik7XG4gICAgY29uc29sZS5sb2coJ0ltYWdlIFBpcGVsaW5lIEFSTjogJyArIGltYWdlUGlwZWxpbmVBcm4pO1xuICAgIGNvbnNvbGUubG9nKCdFeGVjdXRpb24gTmFtZTogJyArIGV4ZWN1dGlvbk5hbWUpO1xuICAgIGNvbnNvbGUubG9nKCdLZXkgSWQ6ICcgKyBhbWlLbXNJZCk7XG4gICAgY29uc29sZS5sb2coJ0FjY291bnRzOiAnICsgYWNjb3VudHMpO1xuICAgIGF3YWl0IHN0YXJ0RXhlY3V0aW9uKHN0YXRlTWFjaGluZUFybiwgZXhlY3V0aW9uTmFtZSwgaW1hZ2VQaXBlbGluZUFybik7XG4gIH0gY2F0Y2ggKGVycm9yKSB7XG4gICAgaWYgKGVycm9yIGluc3RhbmNlb2YgRXJyb3IpIHtcbiAgICAgIHJlc3BvbnNlLlJlYXNvbiA9IGVycm9yLm1lc3NhZ2U7XG4gICAgfVxuICAgIHJlc3BvbnNlLlN0YXR1cyA9ICdGQUlMRUQnO1xuICAgIHJlc3BvbnNlLkRhdGEgPSB7IFJlc3VsdDogZXJyb3IgfTtcbiAgICByZXR1cm4gcmVzcG9uc2U7XG4gIH1cbiAgcmV0dXJuIHJlc3BvbnNlO1xufVxuXG4vKipcbiAqIFN0YXJ0cyB0aGUgZXhlY3V0aW9uIG9mIGEgc3RhdGUgbWFjaGluZS5cbiAqXG4gKiBAcGFyYW0ge3N0cmluZ30gc3RhdGVNYWNoaW5lQXJuIC0gVGhlIEFSTiAoQW1hem9uIFJlc291cmNlIE5hbWUpIG9mIHRoZSBzdGF0ZSBtYWNoaW5lLlxuICogQHBhcmFtIHtzdHJpbmd9IGV4ZWN1dGlvbk5hbWUgLSBUaGUgbmFtZSBmb3IgdGhlIGV4ZWN1dGlvbi5cbiAqIEBwYXJhbSB7c3RyaW5nfSBpbWFnZVBpcGVsaW5lQXJuIC0gVGhlIEFSTiBvZiB0aGUgaW1hZ2UgcGlwZWxpbmUuXG4gKlxuICogQHJldHVybiB7UHJvbWlzZTx2b2lkPn0gLSBBIFByb21pc2UgdGhhdCByZXNvbHZlcyB3aGVuIHRoZSBleGVjdXRpb24gaGFzIGJlZW4gc3RhcnRlZC5cbiAqL1xuYXN5bmMgZnVuY3Rpb24gc3RhcnRFeGVjdXRpb24oc3RhdGVNYWNoaW5lQXJuOiBzdHJpbmcsIGV4ZWN1dGlvbk5hbWU6IHN0cmluZywgaW1hZ2VQaXBlbGluZUFybjogc3RyaW5nKTogUHJvbWlzZTx2b2lkPiB7XG4gIGNvbnN0IHN0YXJ0RXhlY3V0aW9uQ29tbWFuZCA9IG5ldyBTdGFydEV4ZWN1dGlvbkNvbW1hbmQoe1xuICAgIHN0YXRlTWFjaGluZUFybjogc3RhdGVNYWNoaW5lQXJuLFxuICAgIG5hbWU6IGV4ZWN1dGlvbk5hbWUsXG4gICAgaW5wdXQ6IEpTT04uc3RyaW5naWZ5KHtcbiAgICAgIGNsaWVudFRva2VuOiBleGVjdXRpb25OYW1lLFxuICAgICAgaW1hZ2VQaXBlbGluZUFybjogaW1hZ2VQaXBlbGluZUFybixcbiAgICB9KSxcbiAgfSk7XG4gIGF3YWl0IHNmbkNsaWVudC5zZW5kKHN0YXJ0RXhlY3V0aW9uQ29tbWFuZCk7XG59XG4iXX0=