"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Aiops = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [aiops](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonaioperations.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Aiops extends shared_1.PolicyStatement {
    /**
     * Grants permission to create a new investigation in the specified investigation group
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     * - kms:GenerateDataKey
     * - sts:SetContext
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/API_CreateInvestigation.html
     */
    toCreateInvestigation() {
        return this.to('CreateInvestigation');
    }
    /**
     * Grants permission to create a new investigation event in the specified investigation group
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     * - kms:GenerateDataKey
     * - sts:SetContext
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/API_CreateInvestigationEvent.html
     */
    toCreateInvestigationEvent() {
        return this.to('CreateInvestigationEvent');
    }
    /**
     * Grants permission to create a new investigation group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - aiops:TagResource
     * - cloudtrail:DescribeTrails
     * - iam:PassRole
     * - kms:Decrypt
     * - kms:DescribeKey
     * - kms:GenerateDataKey
     * - sso:CreateApplication
     * - sso:DeleteApplication
     * - sso:PutApplicationAccessScope
     * - sso:PutApplicationAssignmentConfiguration
     * - sso:PutApplicationAuthenticationMethod
     * - sso:PutApplicationGrant
     * - sso:TagResource
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/API_CreateInvestigationGroup.html
     */
    toCreateInvestigationGroup() {
        return this.to('CreateInvestigationGroup');
    }
    /**
     * Grants permission to create an investigation resource in the specified investigation group
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - cloudwatch:DescribeAlarmHistory
     * - cloudwatch:DescribeAlarms
     * - cloudwatch:GetInsightRuleReport
     * - cloudwatch:GetMetricData
     * - kms:GenerateDataKey
     * - logs:GetQueryResults
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/API_CreateInvestigationResource.html
     */
    toCreateInvestigationResource() {
        return this.to('CreateInvestigationResource');
    }
    /**
     * Grants permission to delete an investigation in the specified investigation group
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - sts:SetContext
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/API_DeleteInvestigation.html
     */
    toDeleteInvestigation() {
        return this.to('DeleteInvestigation');
    }
    /**
     * Grants permission to delete the specified investigation group
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - sso:DeleteApplication
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/API_DeleteInvestigationGroup.html
     */
    toDeleteInvestigationGroup() {
        return this.to('DeleteInvestigationGroup');
    }
    /**
     * Grants permission to delete the investigation group policy attached to an investigation group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/API_DeleteInvestigationGroupPolicy.html
     */
    toDeleteInvestigationGroupPolicy() {
        return this.to('DeleteInvestigationGroupPolicy');
    }
    /**
     * Grants permission to retrieve an investigation in the specified investigation group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/API_GetInvestigation.html
     */
    toGetInvestigation() {
        return this.to('GetInvestigation');
    }
    /**
     * Grants permission to retrieve an investigation event in the specified investigation group
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/API_GetInvestigationEvent.html
     */
    toGetInvestigationEvent() {
        return this.to('GetInvestigationEvent');
    }
    /**
     * Grants permission to retrieve the specified investigation group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/API_GetInvestigationGroup.html
     */
    toGetInvestigationGroup() {
        return this.to('GetInvestigationGroup');
    }
    /**
     * Grants permission to retrieve the investigation group policy attached to an investigation group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/API_GetInvestigationGroupPolicy.html
     */
    toGetInvestigationGroupPolicy() {
        return this.to('GetInvestigationGroupPolicy');
    }
    /**
     * Grants permission to retrieve an investigation resource in the specified investigation group
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/API_GetInvestigationResource.html
     */
    toGetInvestigationResource() {
        return this.to('GetInvestigationResource');
    }
    /**
     * Grants permission to list all investigation events in the specified investigation group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/API_ListInvestigationEvents.html
     */
    toListInvestigationEvents() {
        return this.to('ListInvestigationEvents');
    }
    /**
     * Grants permission to list all investigation groups in the AWS account making the request
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/API_ListInvestigationGroups.html
     */
    toListInvestigationGroups() {
        return this.to('ListInvestigationGroups');
    }
    /**
     * Grants permission to list all investigations that are in the specified investigation group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/API_ListInvestigations.html
     */
    toListInvestigations() {
        return this.to('ListInvestigations');
    }
    /**
     * Grants permission to list the tags for the specified resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to create/update the investigation group policy attached to an investigation group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/API_PutInvestigationGroupPolicy.html
     */
    toPutInvestigationGroupPolicy() {
        return this.to('PutInvestigationGroupPolicy');
    }
    /**
     * Grants permission to add or update the specified tags for the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove the specified tags from the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an investigation in the specified investigation group
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     * - kms:GenerateDataKey
     * - sts:SetContext
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/API_UpdateInvestigation.html
     */
    toUpdateInvestigation() {
        return this.to('UpdateInvestigation');
    }
    /**
     * Grants permission to update an investigation event in the specified investigation group
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     * - kms:GenerateDataKey
     * - sts:SetContext
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/API_UpdateInvestigationEvent.html
     */
    toUpdateInvestigationEvent() {
        return this.to('UpdateInvestigationEvent');
    }
    /**
     * Grants permission to update the specified investigation group
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - cloudtrail:DescribeTrails
     * - iam:PassRole
     * - kms:Decrypt
     * - kms:DescribeKey
     * - kms:GenerateDataKey
     * - sso:CreateApplication
     * - sso:DeleteApplication
     * - sso:PutApplicationAccessScope
     * - sso:PutApplicationAssignmentConfiguration
     * - sso:PutApplicationAuthenticationMethod
     * - sso:PutApplicationGrant
     * - sso:TagResource
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/API_UpdateInvestigationGroup.html
     */
    toUpdateInvestigationGroup() {
        return this.to('UpdateInvestigationGroup');
    }
    /**
     * Adds a resource of type investigation-group to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/API_InvestigationGroup.html
     *
     * @param investigationGroupId - Identifier for the investigationGroupId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onInvestigationGroup(investigationGroupId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:aiops:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:investigation-group/${investigationGroupId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateInvestigationGroup()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - investigation-group
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateInvestigationGroup()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [aiops](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonaioperations.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'aiops';
        this.accessLevelList = {
            Write: [
                'CreateInvestigation',
                'CreateInvestigationEvent',
                'CreateInvestigationGroup',
                'CreateInvestigationResource',
                'DeleteInvestigation',
                'DeleteInvestigationGroup',
                'DeleteInvestigationGroupPolicy',
                'PutInvestigationGroupPolicy',
                'UpdateInvestigation',
                'UpdateInvestigationEvent',
                'UpdateInvestigationGroup'
            ],
            Read: [
                'GetInvestigation',
                'GetInvestigationEvent',
                'GetInvestigationGroup',
                'GetInvestigationGroupPolicy',
                'GetInvestigationResource'
            ],
            List: [
                'ListInvestigationEvents',
                'ListInvestigationGroups',
                'ListInvestigations',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Aiops = Aiops;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYWlvcGVyYXRpb25zLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiYWlvcGVyYXRpb25zLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUNBLHlDQUF5RDtBQUd6RDs7OztHQUlHO0FBQ0gsTUFBYSxLQUFNLFNBQVEsd0JBQWU7SUFHeEM7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksMEJBQTBCO1FBQy9CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO0lBQzdDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQXlCRztJQUNJLDBCQUEwQjtRQUMvQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTBCLENBQUMsQ0FBQztJQUM3QyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7O09BY0c7SUFDSSw2QkFBNkI7UUFDbEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixDQUFDLENBQUM7SUFDaEQsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksMEJBQTBCO1FBQy9CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO0lBQzdDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQ0FBZ0M7UUFDckMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdDQUFnQyxDQUFDLENBQUM7SUFDbkQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQWtCLENBQUMsQ0FBQztJQUNyQyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksdUJBQXVCO1FBQzVCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO0lBQzFDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDZCQUE2QjtRQUNsQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLENBQUMsQ0FBQztJQUNoRCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksMEJBQTBCO1FBQy9CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO0lBQzdDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx5QkFBeUI7UUFDOUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7SUFDNUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlCQUF5QjtRQUM5QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMseUJBQXlCLENBQUMsQ0FBQztJQUM1QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDZCQUE2QjtRQUNsQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLENBQUMsQ0FBQztJQUNoRCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGFBQWE7UUFDbEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ2hDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksMEJBQTBCO1FBQy9CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO0lBQzdDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FvQkc7SUFDSSwwQkFBMEI7UUFDL0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7SUFDN0MsQ0FBQztJQW1DRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxvQkFBb0IsQ0FBQyxvQkFBNEIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUM3RyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixVQUFXLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSx3QkFBeUIsb0JBQXFCLEVBQUUsQ0FBQyxDQUFDO0lBQzFMLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxlQUFlLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBNEI7UUFDM0YsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFtQixNQUFPLEVBQUUsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2hGLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLGdCQUFnQixDQUFDLE1BQWMsRUFBRSxLQUF3QixFQUFFLFFBQTRCO1FBQzVGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBb0IsTUFBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksWUFBWSxDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDeEUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2pFLENBQUM7SUFFRDs7O09BR0c7SUFDSCxZQUFZLEtBQWdDO1FBQzFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztRQTdhUixrQkFBYSxHQUFHLE9BQU8sQ0FBQztRQW9VckIsb0JBQWUsR0FBb0I7WUFDM0MsS0FBSyxFQUFFO2dCQUNMLHFCQUFxQjtnQkFDckIsMEJBQTBCO2dCQUMxQiwwQkFBMEI7Z0JBQzFCLDZCQUE2QjtnQkFDN0IscUJBQXFCO2dCQUNyQiwwQkFBMEI7Z0JBQzFCLGdDQUFnQztnQkFDaEMsNkJBQTZCO2dCQUM3QixxQkFBcUI7Z0JBQ3JCLDBCQUEwQjtnQkFDMUIsMEJBQTBCO2FBQzNCO1lBQ0QsSUFBSSxFQUFFO2dCQUNKLGtCQUFrQjtnQkFDbEIsdUJBQXVCO2dCQUN2Qix1QkFBdUI7Z0JBQ3ZCLDZCQUE2QjtnQkFDN0IsMEJBQTBCO2FBQzNCO1lBQ0QsSUFBSSxFQUFFO2dCQUNKLHlCQUF5QjtnQkFDekIseUJBQXlCO2dCQUN6QixvQkFBb0I7Z0JBQ3BCLHFCQUFxQjthQUN0QjtZQUNELE9BQU8sRUFBRTtnQkFDUCxhQUFhO2dCQUNiLGVBQWU7YUFDaEI7U0FDRixDQUFDO0lBMkVGLENBQUM7Q0FDRjtBQWhiRCxzQkFnYkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBBY2Nlc3NMZXZlbExpc3QgfSBmcm9tICcuLi8uLi9zaGFyZWQvYWNjZXNzLWxldmVsJztcbmltcG9ydCB7IFBvbGljeVN0YXRlbWVudCwgT3BlcmF0b3IgfSBmcm9tICcuLi8uLi9zaGFyZWQnO1xuaW1wb3J0IHsgYXdzX2lhbSBhcyBpYW0gfSBmcm9tIFwiYXdzLWNkay1saWJcIjtcblxuLyoqXG4gKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW2Fpb3BzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hbWF6b25haW9wZXJhdGlvbnMuaHRtbCkuXG4gKlxuICogQHBhcmFtIHNpZCBbU0lEXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX3NpZC5odG1sKSBvZiB0aGUgc3RhdGVtZW50XG4gKi9cbmV4cG9ydCBjbGFzcyBBaW9wcyBleHRlbmRzIFBvbGljeVN0YXRlbWVudCB7XG4gIHB1YmxpYyBzZXJ2aWNlUHJlZml4ID0gJ2Fpb3BzJztcblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgbmV3IGludmVzdGlnYXRpb24gaW4gdGhlIHNwZWNpZmllZCBpbnZlc3RpZ2F0aW9uIGdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0ga21zOkRlY3J5cHRcbiAgICogLSBrbXM6R2VuZXJhdGVEYXRhS2V5XG4gICAqIC0gc3RzOlNldENvbnRleHRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2gvbGF0ZXN0L21vbml0b3JpbmcvQVBJX0NyZWF0ZUludmVzdGlnYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlSW52ZXN0aWdhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlSW52ZXN0aWdhdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIG5ldyBpbnZlc3RpZ2F0aW9uIGV2ZW50IGluIHRoZSBzcGVjaWZpZWQgaW52ZXN0aWdhdGlvbiBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIGttczpEZWNyeXB0XG4gICAqIC0ga21zOkdlbmVyYXRlRGF0YUtleVxuICAgKiAtIHN0czpTZXRDb250ZXh0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoL2xhdGVzdC9tb25pdG9yaW5nL0FQSV9DcmVhdGVJbnZlc3RpZ2F0aW9uRXZlbnQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlSW52ZXN0aWdhdGlvbkV2ZW50KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVJbnZlc3RpZ2F0aW9uRXZlbnQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBuZXcgaW52ZXN0aWdhdGlvbiBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIGFpb3BzOlRhZ1Jlc291cmNlXG4gICAqIC0gY2xvdWR0cmFpbDpEZXNjcmliZVRyYWlsc1xuICAgKiAtIGlhbTpQYXNzUm9sZVxuICAgKiAtIGttczpEZWNyeXB0XG4gICAqIC0ga21zOkRlc2NyaWJlS2V5XG4gICAqIC0ga21zOkdlbmVyYXRlRGF0YUtleVxuICAgKiAtIHNzbzpDcmVhdGVBcHBsaWNhdGlvblxuICAgKiAtIHNzbzpEZWxldGVBcHBsaWNhdGlvblxuICAgKiAtIHNzbzpQdXRBcHBsaWNhdGlvbkFjY2Vzc1Njb3BlXG4gICAqIC0gc3NvOlB1dEFwcGxpY2F0aW9uQXNzaWdubWVudENvbmZpZ3VyYXRpb25cbiAgICogLSBzc286UHV0QXBwbGljYXRpb25BdXRoZW50aWNhdGlvbk1ldGhvZFxuICAgKiAtIHNzbzpQdXRBcHBsaWNhdGlvbkdyYW50XG4gICAqIC0gc3NvOlRhZ1Jlc291cmNlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoL2xhdGVzdC9tb25pdG9yaW5nL0FQSV9DcmVhdGVJbnZlc3RpZ2F0aW9uR3JvdXAuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlSW52ZXN0aWdhdGlvbkdyb3VwKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVJbnZlc3RpZ2F0aW9uR3JvdXAnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYW4gaW52ZXN0aWdhdGlvbiByZXNvdXJjZSBpbiB0aGUgc3BlY2lmaWVkIGludmVzdGlnYXRpb24gZ3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBjbG91ZHdhdGNoOkRlc2NyaWJlQWxhcm1IaXN0b3J5XG4gICAqIC0gY2xvdWR3YXRjaDpEZXNjcmliZUFsYXJtc1xuICAgKiAtIGNsb3Vkd2F0Y2g6R2V0SW5zaWdodFJ1bGVSZXBvcnRcbiAgICogLSBjbG91ZHdhdGNoOkdldE1ldHJpY0RhdGFcbiAgICogLSBrbXM6R2VuZXJhdGVEYXRhS2V5XG4gICAqIC0gbG9nczpHZXRRdWVyeVJlc3VsdHNcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2gvbGF0ZXN0L21vbml0b3JpbmcvQVBJX0NyZWF0ZUludmVzdGlnYXRpb25SZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVJbnZlc3RpZ2F0aW9uUmVzb3VyY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZUludmVzdGlnYXRpb25SZXNvdXJjZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhbiBpbnZlc3RpZ2F0aW9uIGluIHRoZSBzcGVjaWZpZWQgaW52ZXN0aWdhdGlvbiBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIHN0czpTZXRDb250ZXh0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoL2xhdGVzdC9tb25pdG9yaW5nL0FQSV9EZWxldGVJbnZlc3RpZ2F0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUludmVzdGlnYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZUludmVzdGlnYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgdGhlIHNwZWNpZmllZCBpbnZlc3RpZ2F0aW9uIGdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gc3NvOkRlbGV0ZUFwcGxpY2F0aW9uXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoL2xhdGVzdC9tb25pdG9yaW5nL0FQSV9EZWxldGVJbnZlc3RpZ2F0aW9uR3JvdXAuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlSW52ZXN0aWdhdGlvbkdyb3VwKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVJbnZlc3RpZ2F0aW9uR3JvdXAnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgdGhlIGludmVzdGlnYXRpb24gZ3JvdXAgcG9saWN5IGF0dGFjaGVkIHRvIGFuIGludmVzdGlnYXRpb24gZ3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaC9sYXRlc3QvbW9uaXRvcmluZy9BUElfRGVsZXRlSW52ZXN0aWdhdGlvbkdyb3VwUG9saWN5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUludmVzdGlnYXRpb25Hcm91cFBvbGljeSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlSW52ZXN0aWdhdGlvbkdyb3VwUG9saWN5Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYW4gaW52ZXN0aWdhdGlvbiBpbiB0aGUgc3BlY2lmaWVkIGludmVzdGlnYXRpb24gZ3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoL2xhdGVzdC9tb25pdG9yaW5nL0FQSV9HZXRJbnZlc3RpZ2F0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEludmVzdGlnYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldEludmVzdGlnYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBhbiBpbnZlc3RpZ2F0aW9uIGV2ZW50IGluIHRoZSBzcGVjaWZpZWQgaW52ZXN0aWdhdGlvbiBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0ga21zOkRlY3J5cHRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2gvbGF0ZXN0L21vbml0b3JpbmcvQVBJX0dldEludmVzdGlnYXRpb25FdmVudC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRJbnZlc3RpZ2F0aW9uRXZlbnQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldEludmVzdGlnYXRpb25FdmVudCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIHRoZSBzcGVjaWZpZWQgaW52ZXN0aWdhdGlvbiBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2gvbGF0ZXN0L21vbml0b3JpbmcvQVBJX0dldEludmVzdGlnYXRpb25Hcm91cC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRJbnZlc3RpZ2F0aW9uR3JvdXAoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldEludmVzdGlnYXRpb25Hcm91cCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIHRoZSBpbnZlc3RpZ2F0aW9uIGdyb3VwIHBvbGljeSBhdHRhY2hlZCB0byBhbiBpbnZlc3RpZ2F0aW9uIGdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaC9sYXRlc3QvbW9uaXRvcmluZy9BUElfR2V0SW52ZXN0aWdhdGlvbkdyb3VwUG9saWN5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEludmVzdGlnYXRpb25Hcm91cFBvbGljeSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0SW52ZXN0aWdhdGlvbkdyb3VwUG9saWN5Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYW4gaW52ZXN0aWdhdGlvbiByZXNvdXJjZSBpbiB0aGUgc3BlY2lmaWVkIGludmVzdGlnYXRpb24gZ3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIGttczpEZWNyeXB0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoL2xhdGVzdC9tb25pdG9yaW5nL0FQSV9HZXRJbnZlc3RpZ2F0aW9uUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0SW52ZXN0aWdhdGlvblJlc291cmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRJbnZlc3RpZ2F0aW9uUmVzb3VyY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGFsbCBpbnZlc3RpZ2F0aW9uIGV2ZW50cyBpbiB0aGUgc3BlY2lmaWVkIGludmVzdGlnYXRpb24gZ3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoL2xhdGVzdC9tb25pdG9yaW5nL0FQSV9MaXN0SW52ZXN0aWdhdGlvbkV2ZW50cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0SW52ZXN0aWdhdGlvbkV2ZW50cygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdEludmVzdGlnYXRpb25FdmVudHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGFsbCBpbnZlc3RpZ2F0aW9uIGdyb3VwcyBpbiB0aGUgQVdTIGFjY291bnQgbWFraW5nIHRoZSByZXF1ZXN0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaC9sYXRlc3QvbW9uaXRvcmluZy9BUElfTGlzdEludmVzdGlnYXRpb25Hcm91cHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdEludmVzdGlnYXRpb25Hcm91cHMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RJbnZlc3RpZ2F0aW9uR3JvdXBzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBhbGwgaW52ZXN0aWdhdGlvbnMgdGhhdCBhcmUgaW4gdGhlIHNwZWNpZmllZCBpbnZlc3RpZ2F0aW9uIGdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaC9sYXRlc3QvbW9uaXRvcmluZy9BUElfTGlzdEludmVzdGlnYXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RJbnZlc3RpZ2F0aW9ucygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdEludmVzdGlnYXRpb25zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCB0aGUgdGFncyBmb3IgdGhlIHNwZWNpZmllZCByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2gvbGF0ZXN0L21vbml0b3JpbmcvQVBJX0xpc3RUYWdzRm9yUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFRhZ3NGb3JSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFRhZ3NGb3JSZXNvdXJjZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZS91cGRhdGUgdGhlIGludmVzdGlnYXRpb24gZ3JvdXAgcG9saWN5IGF0dGFjaGVkIHRvIGFuIGludmVzdGlnYXRpb24gZ3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaC9sYXRlc3QvbW9uaXRvcmluZy9BUElfUHV0SW52ZXN0aWdhdGlvbkdyb3VwUG9saWN5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1B1dEludmVzdGlnYXRpb25Hcm91cFBvbGljeSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnUHV0SW52ZXN0aWdhdGlvbkdyb3VwUG9saWN5Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYWRkIG9yIHVwZGF0ZSB0aGUgc3BlY2lmaWVkIHRhZ3MgZm9yIHRoZSBzcGVjaWZpZWQgcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoL2xhdGVzdC9tb25pdG9yaW5nL0FQSV9UYWdSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9UYWdSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVGFnUmVzb3VyY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZW1vdmUgdGhlIHNwZWNpZmllZCB0YWdzIGZyb20gdGhlIHNwZWNpZmllZCByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2gvbGF0ZXN0L21vbml0b3JpbmcvQVBJX1VudGFnUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVW50YWdSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVW50YWdSZXNvdXJjZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhbiBpbnZlc3RpZ2F0aW9uIGluIHRoZSBzcGVjaWZpZWQgaW52ZXN0aWdhdGlvbiBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIGttczpEZWNyeXB0XG4gICAqIC0ga21zOkdlbmVyYXRlRGF0YUtleVxuICAgKiAtIHN0czpTZXRDb250ZXh0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoL2xhdGVzdC9tb25pdG9yaW5nL0FQSV9VcGRhdGVJbnZlc3RpZ2F0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUludmVzdGlnYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZUludmVzdGlnYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYW4gaW52ZXN0aWdhdGlvbiBldmVudCBpbiB0aGUgc3BlY2lmaWVkIGludmVzdGlnYXRpb24gZ3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBrbXM6RGVjcnlwdFxuICAgKiAtIGttczpHZW5lcmF0ZURhdGFLZXlcbiAgICogLSBzdHM6U2V0Q29udGV4dFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaC9sYXRlc3QvbW9uaXRvcmluZy9BUElfVXBkYXRlSW52ZXN0aWdhdGlvbkV2ZW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUludmVzdGlnYXRpb25FdmVudCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlSW52ZXN0aWdhdGlvbkV2ZW50Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIHRoZSBzcGVjaWZpZWQgaW52ZXN0aWdhdGlvbiBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIGNsb3VkdHJhaWw6RGVzY3JpYmVUcmFpbHNcbiAgICogLSBpYW06UGFzc1JvbGVcbiAgICogLSBrbXM6RGVjcnlwdFxuICAgKiAtIGttczpEZXNjcmliZUtleVxuICAgKiAtIGttczpHZW5lcmF0ZURhdGFLZXlcbiAgICogLSBzc286Q3JlYXRlQXBwbGljYXRpb25cbiAgICogLSBzc286RGVsZXRlQXBwbGljYXRpb25cbiAgICogLSBzc286UHV0QXBwbGljYXRpb25BY2Nlc3NTY29wZVxuICAgKiAtIHNzbzpQdXRBcHBsaWNhdGlvbkFzc2lnbm1lbnRDb25maWd1cmF0aW9uXG4gICAqIC0gc3NvOlB1dEFwcGxpY2F0aW9uQXV0aGVudGljYXRpb25NZXRob2RcbiAgICogLSBzc286UHV0QXBwbGljYXRpb25HcmFudFxuICAgKiAtIHNzbzpUYWdSZXNvdXJjZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaC9sYXRlc3QvbW9uaXRvcmluZy9BUElfVXBkYXRlSW52ZXN0aWdhdGlvbkdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUludmVzdGlnYXRpb25Hcm91cCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlSW52ZXN0aWdhdGlvbkdyb3VwJyk7XG4gIH1cblxuICBwcm90ZWN0ZWQgYWNjZXNzTGV2ZWxMaXN0OiBBY2Nlc3NMZXZlbExpc3QgPSB7XG4gICAgV3JpdGU6IFtcbiAgICAgICdDcmVhdGVJbnZlc3RpZ2F0aW9uJyxcbiAgICAgICdDcmVhdGVJbnZlc3RpZ2F0aW9uRXZlbnQnLFxuICAgICAgJ0NyZWF0ZUludmVzdGlnYXRpb25Hcm91cCcsXG4gICAgICAnQ3JlYXRlSW52ZXN0aWdhdGlvblJlc291cmNlJyxcbiAgICAgICdEZWxldGVJbnZlc3RpZ2F0aW9uJyxcbiAgICAgICdEZWxldGVJbnZlc3RpZ2F0aW9uR3JvdXAnLFxuICAgICAgJ0RlbGV0ZUludmVzdGlnYXRpb25Hcm91cFBvbGljeScsXG4gICAgICAnUHV0SW52ZXN0aWdhdGlvbkdyb3VwUG9saWN5JyxcbiAgICAgICdVcGRhdGVJbnZlc3RpZ2F0aW9uJyxcbiAgICAgICdVcGRhdGVJbnZlc3RpZ2F0aW9uRXZlbnQnLFxuICAgICAgJ1VwZGF0ZUludmVzdGlnYXRpb25Hcm91cCdcbiAgICBdLFxuICAgIFJlYWQ6IFtcbiAgICAgICdHZXRJbnZlc3RpZ2F0aW9uJyxcbiAgICAgICdHZXRJbnZlc3RpZ2F0aW9uRXZlbnQnLFxuICAgICAgJ0dldEludmVzdGlnYXRpb25Hcm91cCcsXG4gICAgICAnR2V0SW52ZXN0aWdhdGlvbkdyb3VwUG9saWN5JyxcbiAgICAgICdHZXRJbnZlc3RpZ2F0aW9uUmVzb3VyY2UnXG4gICAgXSxcbiAgICBMaXN0OiBbXG4gICAgICAnTGlzdEludmVzdGlnYXRpb25FdmVudHMnLFxuICAgICAgJ0xpc3RJbnZlc3RpZ2F0aW9uR3JvdXBzJyxcbiAgICAgICdMaXN0SW52ZXN0aWdhdGlvbnMnLFxuICAgICAgJ0xpc3RUYWdzRm9yUmVzb3VyY2UnXG4gICAgXSxcbiAgICBUYWdnaW5nOiBbXG4gICAgICAnVGFnUmVzb3VyY2UnLFxuICAgICAgJ1VudGFnUmVzb3VyY2UnXG4gICAgXVxuICB9O1xuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBpbnZlc3RpZ2F0aW9uLWdyb3VwIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2gvbGF0ZXN0L21vbml0b3JpbmcvQVBJX0ludmVzdGlnYXRpb25Hcm91cC5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBpbnZlc3RpZ2F0aW9uR3JvdXBJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBpbnZlc3RpZ2F0aW9uR3JvdXBJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uSW52ZXN0aWdhdGlvbkdyb3VwKGludmVzdGlnYXRpb25Hcm91cElkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YWlvcHM6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OmludmVzdGlnYXRpb24tZ3JvdXAvJHsgaW52ZXN0aWdhdGlvbkdyb3VwSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSB0YWdzIHRoYXQgYXJlIHBhc3NlZCBpbiB0aGUgcmVxdWVzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtcmVxdWVzdHRhZ1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQ3JlYXRlSW52ZXN0aWdhdGlvbkdyb3VwKClcbiAgICogLSAudG9UYWdSZXNvdXJjZSgpXG4gICAqXG4gICAqIEBwYXJhbSB0YWdLZXkgVGhlIHRhZyBrZXkgdG8gY2hlY2tcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NSZXF1ZXN0VGFnKHRhZ0tleTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYXdzOlJlcXVlc3RUYWcvJHsgdGFnS2V5IH1gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgdGFncyBhc3NvY2lhdGVkIHdpdGggdGhlIHJlc291cmNlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy1yZXNvdXJjZXRhZ1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIGludmVzdGlnYXRpb24tZ3JvdXBcbiAgICpcbiAgICogQHBhcmFtIHRhZ0tleSBUaGUgdGFnIGtleSB0byBjaGVja1xuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1Jlc291cmNlVGFnKHRhZ0tleTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYXdzOlJlc291cmNlVGFnLyR7IHRhZ0tleSB9YCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgdGhlIHRhZyBrZXlzIHRoYXQgYXJlIHBhc3NlZCBpbiB0aGUgcmVxdWVzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtdGFna2V5c1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQ3JlYXRlSW52ZXN0aWdhdGlvbkdyb3VwKClcbiAgICogLSAudG9UYWdSZXNvdXJjZSgpXG4gICAqIC0gLnRvVW50YWdSZXNvdXJjZSgpXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzVGFnS2V5cyh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYXdzOlRhZ0tleXNgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW2Fpb3BzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hbWF6b25haW9wZXJhdGlvbnMuaHRtbCkuXG4gICAqXG4gICAqL1xuICBjb25zdHJ1Y3Rvcihwcm9wcz86IGlhbS5Qb2xpY3lTdGF0ZW1lbnRQcm9wcykge1xuICAgIHN1cGVyKHByb3BzKTtcbiAgfVxufVxuIl19