"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ArcZonalShift = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [arc-zonal-shift](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonapplicationrecoverycontroller-zonalshift.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class ArcZonalShift extends shared_1.PolicyStatement {
    /**
     * Grants permission to cancel an active zonal shift
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifResourceIdentifier()
     * - .ifAwsResourceTag()
     * - .ifElasticloadbalancingResourceTag()
     *
     * https://docs.aws.amazon.com/arc-zonal-shift/latest/api/API_CancelZonalShift.html
     */
    toCancelZonalShift() {
        return this.to('CancelZonalShift');
    }
    /**
     * Grants permission to create a practice run configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifResourceIdentifier()
     * - .ifAwsResourceTag()
     * - .ifElasticloadbalancingResourceTag()
     *
     * Dependent actions:
     * - cloudwatch:DescribeAlarms
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/arc-zonal-shift/latest/api/API_CreatePracticeRunConfiguration.html
     */
    toCreatePracticeRunConfiguration() {
        return this.to('CreatePracticeRunConfiguration');
    }
    /**
     * Grants permission to delete a practice run configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifResourceIdentifier()
     * - .ifAwsResourceTag()
     * - .ifElasticloadbalancingResourceTag()
     *
     * https://docs.aws.amazon.com/arc-zonal-shift/latest/api/API_DeletePracticeRunConfiguration.html
     */
    toDeletePracticeRunConfiguration() {
        return this.to('DeletePracticeRunConfiguration');
    }
    /**
     * Grants permission to get autoshift observer notification status
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/arc-zonal-shift/latest/api/API_GetAutoshiftObserverNotificationStatus.html
     */
    toGetAutoshiftObserverNotificationStatus() {
        return this.to('GetAutoshiftObserverNotificationStatus');
    }
    /**
     * Grants permission to get information about a managed resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifResourceIdentifier()
     * - .ifAwsResourceTag()
     * - .ifElasticloadbalancingResourceTag()
     *
     * https://docs.aws.amazon.com/arc-zonal-shift/latest/api/API_GetManagedResource.html
     */
    toGetManagedResource() {
        return this.to('GetManagedResource');
    }
    /**
     * Grants permission to list active and completed autoshifts
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/arc-zonal-shift/latest/api/API_ListAutoshifts.html
     */
    toListAutoshifts() {
        return this.to('ListAutoshifts');
    }
    /**
     * Grants permission to list managed resources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/arc-zonal-shift/latest/api/API_ListManagedResources.html
     */
    toListManagedResources() {
        return this.to('ListManagedResources');
    }
    /**
     * Grants permission to list zonal shifts
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/arc-zonal-shift/latest/api/API_ListZonalShifts.html
     */
    toListZonalShifts() {
        return this.to('ListZonalShifts');
    }
    /**
     * Grants permission to start a zonal shift
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifResourceIdentifier()
     * - .ifAwsResourceTag()
     * - .ifElasticloadbalancingResourceTag()
     *
     * https://docs.aws.amazon.com/arc-zonal-shift/latest/api/API_StartZonalShift.html
     */
    toStartZonalShift() {
        return this.to('StartZonalShift');
    }
    /**
     * Grants permission to update autoshift observer notification status
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/arc-zonal-shift/latest/api/API_UpdateAutoshiftObserverNotificationStatus.html
     */
    toUpdateAutoshiftObserverNotificationStatus() {
        return this.to('UpdateAutoshiftObserverNotificationStatus');
    }
    /**
     * Grants permission to update a practice run configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifResourceIdentifier()
     * - .ifAwsResourceTag()
     * - .ifElasticloadbalancingResourceTag()
     *
     * Dependent actions:
     * - cloudwatch:DescribeAlarms
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/arc-zonal-shift/latest/api/API_UpdatePracticeRunConfiguration.html
     */
    toUpdatePracticeRunConfiguration() {
        return this.to('UpdatePracticeRunConfiguration');
    }
    /**
     * Grants permission to update a zonal autoshift status
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifResourceIdentifier()
     * - .ifAwsResourceTag()
     * - .ifElasticloadbalancingResourceTag()
     *
     * https://docs.aws.amazon.com/arc-zonal-shift/latest/api/API_UpdateZonalAutoshiftConfiguration.html
     */
    toUpdateZonalAutoshiftConfiguration() {
        return this.to('UpdateZonalAutoshiftConfiguration');
    }
    /**
     * Grants permission to update an existing zonal shift
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifResourceIdentifier()
     * - .ifAwsResourceTag()
     * - .ifElasticloadbalancingResourceTag()
     *
     * https://docs.aws.amazon.com/arc-zonal-shift/latest/api/API_UpdateZonalShift.html
     */
    toUpdateZonalShift() {
        return this.to('UpdateZonalShift');
    }
    /**
     * Adds a resource of type ALB to the statement
     *
     * https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-shift.resource-types.html
     *
     * @param loadBalancerName - Identifier for the loadBalancerName.
     * @param loadBalancerId - Identifier for the loadBalancerId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifResourceIdentifier()
     * - .ifAwsResourceTag()
     * - .ifElasticloadbalancingResourceTag()
     */
    onALB(loadBalancerName, loadBalancerId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elasticloadbalancing:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:loadbalancer/app/${loadBalancerName}/${loadBalancerId}`);
    }
    /**
     * Adds a resource of type NLB to the statement
     *
     * https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-shift.resource-types.html
     *
     * @param loadBalancerName - Identifier for the loadBalancerName.
     * @param loadBalancerId - Identifier for the loadBalancerId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifResourceIdentifier()
     * - .ifAwsResourceTag()
     * - .ifElasticloadbalancingResourceTag()
     */
    onNLB(loadBalancerName, loadBalancerId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elasticloadbalancing:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:loadbalancer/net/${loadBalancerName}/${loadBalancerId}`);
    }
    /**
     * Filters access by the resource identifier of the managed resource
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonroute53applicationrecoverycontroller-zonalshift.html#amazonroute53applicationrecoverycontroller-zonalshift-policy-keys
     *
     * Applies to actions:
     * - .toCancelZonalShift()
     * - .toCreatePracticeRunConfiguration()
     * - .toDeletePracticeRunConfiguration()
     * - .toGetManagedResource()
     * - .toStartZonalShift()
     * - .toUpdatePracticeRunConfiguration()
     * - .toUpdateZonalAutoshiftConfiguration()
     * - .toUpdateZonalShift()
     *
     * Applies to resource types:
     * - ALB
     * - NLB
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceIdentifier(value, operator) {
        return this.if(`ResourceIdentifier`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the managed resource
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/userguide/load-balancer-authentication-access-control.html#elb-condition-keys
     *
     * Applies to actions:
     * - .toCancelZonalShift()
     * - .toCreatePracticeRunConfiguration()
     * - .toDeletePracticeRunConfiguration()
     * - .toGetManagedResource()
     * - .toStartZonalShift()
     * - .toUpdatePracticeRunConfiguration()
     * - .toUpdateZonalAutoshiftConfiguration()
     * - .toUpdateZonalShift()
     *
     * Applies to resource types:
     * - ALB
     * - NLB
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the managed resource
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/userguide/load-balancer-authentication-access-control.html#elb-condition-keys
     *
     * Applies to actions:
     * - .toCancelZonalShift()
     * - .toCreatePracticeRunConfiguration()
     * - .toDeletePracticeRunConfiguration()
     * - .toGetManagedResource()
     * - .toStartZonalShift()
     * - .toUpdatePracticeRunConfiguration()
     * - .toUpdateZonalAutoshiftConfiguration()
     * - .toUpdateZonalShift()
     *
     * Applies to resource types:
     * - ALB
     * - NLB
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifElasticloadbalancingResourceTag(tagKey, value, operator) {
        return this.if(`elasticloadbalancing:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [arc-zonal-shift](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonapplicationrecoverycontroller-zonalshift.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'arc-zonal-shift';
        this.accessLevelList = {
            Write: [
                'CancelZonalShift',
                'CreatePracticeRunConfiguration',
                'DeletePracticeRunConfiguration',
                'StartZonalShift',
                'UpdateAutoshiftObserverNotificationStatus',
                'UpdatePracticeRunConfiguration',
                'UpdateZonalAutoshiftConfiguration',
                'UpdateZonalShift'
            ],
            Read: [
                'GetAutoshiftObserverNotificationStatus',
                'GetManagedResource'
            ],
            List: [
                'ListAutoshifts',
                'ListManagedResources',
                'ListZonalShifts'
            ]
        };
    }
}
exports.ArcZonalShift = ArcZonalShift;
//# sourceMappingURL=data:application/json;base64,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