"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AppmeshPreview = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [appmesh-preview](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsappmeshpreview.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class AppmeshPreview extends shared_1.PolicyStatement {
    /**
     * Grants permission to create a gateway route that is associated with a virtual gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateGatewayRoute.html
     */
    toCreateGatewayRoute() {
        return this.to('CreateGatewayRoute');
    }
    /**
     * Grants permission to create a service mesh
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateMesh.html
     */
    toCreateMesh() {
        return this.to('CreateMesh');
    }
    /**
     * Grants permission to create a route that is associated with a virtual router
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateRoute.html
     */
    toCreateRoute() {
        return this.to('CreateRoute');
    }
    /**
     * Grants permission to create a virtual gateway within a service mesh
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateVirtualGateway.html
     */
    toCreateVirtualGateway() {
        return this.to('CreateVirtualGateway');
    }
    /**
     * Grants permission to create a virtual node within a service mesh
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateVirtualNode.html
     */
    toCreateVirtualNode() {
        return this.to('CreateVirtualNode');
    }
    /**
     * Grants permission to create a virtual router within a service mesh
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateVirtualRouter.html
     */
    toCreateVirtualRouter() {
        return this.to('CreateVirtualRouter');
    }
    /**
     * Grants permission to create a virtual service within a service mesh
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateVirtualService.html
     */
    toCreateVirtualService() {
        return this.to('CreateVirtualService');
    }
    /**
     * Grants permission to delete an existing gateway route
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteGatewayRoute.html
     */
    toDeleteGatewayRoute() {
        return this.to('DeleteGatewayRoute');
    }
    /**
     * Grants permission to delete an existing service mesh
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteMesh.html
     */
    toDeleteMesh() {
        return this.to('DeleteMesh');
    }
    /**
     * Grants permission to delete the RAM access control policy for a mesh
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html
     */
    toDeleteMeshPolicy() {
        return this.to('DeleteMeshPolicy');
    }
    /**
     * Grants permission to delete an existing route
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteRoute.html
     */
    toDeleteRoute() {
        return this.to('DeleteRoute');
    }
    /**
     * Grants permission to delete an existing virtual gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteVirtualGateway.html
     */
    toDeleteVirtualGateway() {
        return this.to('DeleteVirtualGateway');
    }
    /**
     * Grants permission to delete an existing virtual node
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteVirtualNode.html
     */
    toDeleteVirtualNode() {
        return this.to('DeleteVirtualNode');
    }
    /**
     * Grants permission to delete an existing virtual router
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteVirtualRouter.html
     */
    toDeleteVirtualRouter() {
        return this.to('DeleteVirtualRouter');
    }
    /**
     * Grants permission to delete an existing virtual service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteVirtualService.html
     */
    toDeleteVirtualService() {
        return this.to('DeleteVirtualService');
    }
    /**
     * Grants permission to describe an existing gateway route
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeGatewayRoute.html
     */
    toDescribeGatewayRoute() {
        return this.to('DescribeGatewayRoute');
    }
    /**
     * Grants permission to describe an existing service mesh
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeMesh.html
     */
    toDescribeMesh() {
        return this.to('DescribeMesh');
    }
    /**
     * Grants permission to describe an existing route
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeRoute.html
     */
    toDescribeRoute() {
        return this.to('DescribeRoute');
    }
    /**
     * Grants permission to describe an existing virtual gateway
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeVirtualGateway.html
     */
    toDescribeVirtualGateway() {
        return this.to('DescribeVirtualGateway');
    }
    /**
     * Grants permission to describe an existing virtual node
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeVirtualNode.html
     */
    toDescribeVirtualNode() {
        return this.to('DescribeVirtualNode');
    }
    /**
     * Grants permission to describe an existing virtual router
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeVirtualRouter.html
     */
    toDescribeVirtualRouter() {
        return this.to('DescribeVirtualRouter');
    }
    /**
     * Grants permission to describe an existing virtual service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeVirtualService.html
     */
    toDescribeVirtualService() {
        return this.to('DescribeVirtualService');
    }
    /**
     * Grants permission to read the RAM access control policy for a mesh
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html
     */
    toGetMeshPolicy() {
        return this.to('GetMeshPolicy');
    }
    /**
     * Grants permission to list existing gateway routes in a service mesh
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListGatewayRoutes.html
     */
    toListGatewayRoutes() {
        return this.to('ListGatewayRoutes');
    }
    /**
     * Grants permission to list existing service meshes
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListMeshes.html
     */
    toListMeshes() {
        return this.to('ListMeshes');
    }
    /**
     * Grants permission to list existing routes in a service mesh
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListRoutes.html
     */
    toListRoutes() {
        return this.to('ListRoutes');
    }
    /**
     * Grants permission to list existing virtual gateways in a service mesh
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListVirtualGateways.html
     */
    toListVirtualGateways() {
        return this.to('ListVirtualGateways');
    }
    /**
     * Grants permission to list existing virtual nodes
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListVirtualNodes.html
     */
    toListVirtualNodes() {
        return this.to('ListVirtualNodes');
    }
    /**
     * Grants permission to list existing virtual routers in a service mesh
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListVirtualRouters.html
     */
    toListVirtualRouters() {
        return this.to('ListVirtualRouters');
    }
    /**
     * Grants permission to list existing virtual services in a service mesh
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListVirtualServices.html
     */
    toListVirtualServices() {
        return this.to('ListVirtualServices');
    }
    /**
     * Grants permission to define the RAM access control policy for a mesh
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html
     */
    toPutMeshPolicy() {
        return this.to('PutMeshPolicy');
    }
    /**
     * Grants permission to receive streamed resources for an App Mesh endpoint (VirtualNode/VirtualGateway)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/envoy.html
     */
    toStreamAggregatedResources() {
        return this.to('StreamAggregatedResources');
    }
    /**
     * Grants permission to update an existing gateway route for a specified service mesh and virtual gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateGatewayRoute.html
     */
    toUpdateGatewayRoute() {
        return this.to('UpdateGatewayRoute');
    }
    /**
     * Grants permission to update an existing service mesh
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateMesh.html
     */
    toUpdateMesh() {
        return this.to('UpdateMesh');
    }
    /**
     * Grants permission to update an existing route for a specified service mesh and virtual router
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateRoute.html
     */
    toUpdateRoute() {
        return this.to('UpdateRoute');
    }
    /**
     * Grants permission to update an existing virtual gateway in a specified service mesh
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateVirtualGateway.html
     */
    toUpdateVirtualGateway() {
        return this.to('UpdateVirtualGateway');
    }
    /**
     * Grants permission to update an existing virtual node in a specified service mesh
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateVirtualNode.html
     */
    toUpdateVirtualNode() {
        return this.to('UpdateVirtualNode');
    }
    /**
     * Grants permission to update an existing virtual router in a specified service mesh
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateVirtualRouter.html
     */
    toUpdateVirtualRouter() {
        return this.to('UpdateVirtualRouter');
    }
    /**
     * Grants permission to update an existing virtual service in a specified service mesh
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateVirtualService.html
     */
    toUpdateVirtualService() {
        return this.to('UpdateVirtualService');
    }
    /**
     * Adds a resource of type mesh to the statement
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/meshes.html
     *
     * @param meshName - Identifier for the meshName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onMesh(meshName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:appmesh-preview:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:mesh/${meshName}`);
    }
    /**
     * Adds a resource of type virtualService to the statement
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_services.html
     *
     * @param meshName - Identifier for the meshName.
     * @param virtualServiceName - Identifier for the virtualServiceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onVirtualService(meshName, virtualServiceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:appmesh-preview:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:mesh/${meshName}/virtualService/${virtualServiceName}`);
    }
    /**
     * Adds a resource of type virtualNode to the statement
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_nodes.html
     *
     * @param meshName - Identifier for the meshName.
     * @param virtualNodeName - Identifier for the virtualNodeName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onVirtualNode(meshName, virtualNodeName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:appmesh-preview:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:mesh/${meshName}/virtualNode/${virtualNodeName}`);
    }
    /**
     * Adds a resource of type virtualRouter to the statement
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_routers.html
     *
     * @param meshName - Identifier for the meshName.
     * @param virtualRouterName - Identifier for the virtualRouterName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onVirtualRouter(meshName, virtualRouterName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:appmesh-preview:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:mesh/${meshName}/virtualRouter/${virtualRouterName}`);
    }
    /**
     * Adds a resource of type route to the statement
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/routes.html
     *
     * @param meshName - Identifier for the meshName.
     * @param virtualRouterName - Identifier for the virtualRouterName.
     * @param routeName - Identifier for the routeName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onRoute(meshName, virtualRouterName, routeName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:appmesh-preview:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:mesh/${meshName}/virtualRouter/${virtualRouterName}/route/${routeName}`);
    }
    /**
     * Adds a resource of type virtualGateway to the statement
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_gateways.html
     *
     * @param meshName - Identifier for the meshName.
     * @param virtualGatewayName - Identifier for the virtualGatewayName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onVirtualGateway(meshName, virtualGatewayName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:appmesh-preview:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:mesh/${meshName}/virtualGateway/${virtualGatewayName}`);
    }
    /**
     * Adds a resource of type gatewayRoute to the statement
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_gateways.html
     *
     * @param meshName - Identifier for the meshName.
     * @param virtualGatewayName - Identifier for the virtualGatewayName.
     * @param gatewayRouteName - Identifier for the gatewayRouteName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onGatewayRoute(meshName, virtualGatewayName, gatewayRouteName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:appmesh-preview:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:mesh/${meshName}/virtualGateway/${virtualGatewayName}/gatewayRoute/${gatewayRouteName}`);
    }
    /**
     * Statement provider for service [appmesh-preview](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsappmeshpreview.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'appmesh-preview';
        this.accessLevelList = {
            Write: [
                'CreateGatewayRoute',
                'CreateMesh',
                'CreateRoute',
                'CreateVirtualGateway',
                'CreateVirtualNode',
                'CreateVirtualRouter',
                'CreateVirtualService',
                'DeleteGatewayRoute',
                'DeleteMesh',
                'DeleteMeshPolicy',
                'DeleteRoute',
                'DeleteVirtualGateway',
                'DeleteVirtualNode',
                'DeleteVirtualRouter',
                'DeleteVirtualService',
                'PutMeshPolicy',
                'UpdateGatewayRoute',
                'UpdateMesh',
                'UpdateRoute',
                'UpdateVirtualGateway',
                'UpdateVirtualNode',
                'UpdateVirtualRouter',
                'UpdateVirtualService'
            ],
            Read: [
                'DescribeGatewayRoute',
                'DescribeMesh',
                'DescribeRoute',
                'DescribeVirtualGateway',
                'DescribeVirtualNode',
                'DescribeVirtualRouter',
                'DescribeVirtualService',
                'GetMeshPolicy',
                'StreamAggregatedResources'
            ],
            List: [
                'ListGatewayRoutes',
                'ListMeshes',
                'ListRoutes',
                'ListVirtualGateways',
                'ListVirtualNodes',
                'ListVirtualRouters',
                'ListVirtualServices'
            ]
        };
    }
}
exports.AppmeshPreview = AppmeshPreview;
//# sourceMappingURL=data:application/json;base64,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