"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Appsync = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [appsync](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsappsync.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Appsync extends shared_1.PolicyStatement {
    /**
     * Grants permission to attach a GraphQL API to a custom domain name in AppSync
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_AssociateApi.html
     */
    toAssociateApi() {
        return this.to('AssociateApi');
    }
    /**
     * Grants permission to associate a merged API to a source API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_AssociateMergedGraphqlApi.html
     */
    toAssociateMergedGraphqlApi() {
        return this.to('AssociateMergedGraphqlApi');
    }
    /**
     * Grants permission to associate a source API to a merged API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_AssociateSourceGraphqlApi.html
     */
    toAssociateSourceGraphqlApi() {
        return this.to('AssociateSourceGraphqlApi');
    }
    /**
     * Grants permission to create an API
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateApi.html
     */
    toCreateApi() {
        return this.to('CreateApi');
    }
    /**
     * Grants permission to create an API cache in AppSync
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateApiCache.html
     */
    toCreateApiCache() {
        return this.to('CreateApiCache');
    }
    /**
     * Grants permission to create a unique key that you can distribute to clients who are executing your API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateApiKey.html
     */
    toCreateApiKey() {
        return this.to('CreateApiKey');
    }
    /**
     * Grants permission to create a channel namespace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateChannelNamespace.html
     */
    toCreateChannelNamespace() {
        return this.to('CreateChannelNamespace');
    }
    /**
     * Grants permission to create a data source
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateDataSource.html
     */
    toCreateDataSource() {
        return this.to('CreateDataSource');
    }
    /**
     * Grants permission to create a custom domain name in AppSync
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateDomainName.html
     */
    toCreateDomainName() {
        return this.to('CreateDomainName');
    }
    /**
     * Grants permission to create a new function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateFunction.html
     */
    toCreateFunction() {
        return this.to('CreateFunction');
    }
    /**
     * Grants permission to create a GraphQL API, which is the top level AppSync resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifVisibility()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateGraphqlApi.html
     */
    toCreateGraphqlApi() {
        return this.to('CreateGraphqlApi');
    }
    /**
     * Grants permission to create a resolver. A resolver converts incoming requests into a format that a data source can understand, and converts the data source's responses into GraphQL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateResolver.html
     */
    toCreateResolver() {
        return this.to('CreateResolver');
    }
    /**
     * Grants permission to create a type
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateType.html
     */
    toCreateType() {
        return this.to('CreateType');
    }
    /**
     * Grants permission to delete a API. This will also clean up every AppSync resource below that API
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteApi.html
     */
    toDeleteApi() {
        return this.to('DeleteApi');
    }
    /**
     * Grants permission to delete an API cache in AppSync
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteApiCache.html
     */
    toDeleteApiCache() {
        return this.to('DeleteApiCache');
    }
    /**
     * Grants permission to delete an API key
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteApiKey.html
     */
    toDeleteApiKey() {
        return this.to('DeleteApiKey');
    }
    /**
     * Grants permission to delete a channel namespace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteChannelNamespace.html
     */
    toDeleteChannelNamespace() {
        return this.to('DeleteChannelNamespace');
    }
    /**
     * Grants permission to delete a data source
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteDataSource.html
     */
    toDeleteDataSource() {
        return this.to('DeleteDataSource');
    }
    /**
     * Grants permission to delete a custom domain name in AppSync
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteDomainName.html
     */
    toDeleteDomainName() {
        return this.to('DeleteDomainName');
    }
    /**
     * Grants permission to delete a function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteFunction.html
     */
    toDeleteFunction() {
        return this.to('DeleteFunction');
    }
    /**
     * Grants permission to delete a GraphQL Api. This will also clean up every AppSync resource below that API
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteGraphqlApi.html
     */
    toDeleteGraphqlApi() {
        return this.to('DeleteGraphqlApi');
    }
    /**
     * Grants permission to delete a resolver
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteResolver.html
     */
    toDeleteResolver() {
        return this.to('DeleteResolver');
    }
    /**
     * Grants permission to remove a resource policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/merge-api.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to delete a type
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteType.html
     */
    toDeleteType() {
        return this.to('DeleteType');
    }
    /**
     * Grants permission to detach a GraphQL API to a custom domain name in AppSync
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DisassociateApi.html
     */
    toDisassociateApi() {
        return this.to('DisassociateApi');
    }
    /**
     * Grants permission to remove an associated source API from a merged API identified by the source API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DisassociateMergedGraphqlApi.html
     */
    toDisassociateMergedGraphqlApi() {
        return this.to('DisassociateMergedGraphqlApi');
    }
    /**
     * Grants permission to remove an associated source API from a merged API identified by the merged API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DisassociateSourceGraphqlApi.html
     */
    toDisassociateSourceGraphqlApi() {
        return this.to('DisassociateSourceGraphqlApi');
    }
    /**
     * Grants permission to evaluate code with a runtime and context
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_EvaluateCode.html
     */
    toEvaluateCode() {
        return this.to('EvaluateCode');
    }
    /**
     * Grants permission to evaluate template mapping
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_EvaluateMappingTemplate.html
     */
    toEvaluateMappingTemplate() {
        return this.to('EvaluateMappingTemplate');
    }
    /**
     * Grants permission to connect to an Event API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/using-your-event-api.html
     */
    toEventConnect() {
        return this.to('EventConnect');
    }
    /**
     * Grants permission to publish events to a channel namespace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/using-your-event-api.html
     */
    toEventPublish() {
        return this.to('EventPublish');
    }
    /**
     * Grants permission to subscribe to a channel namespace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/using-your-event-api.html
     */
    toEventSubscribe() {
        return this.to('EventSubscribe');
    }
    /**
     * Grants permission to flush an API cache in AppSync
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_FlushApiCache.html
     */
    toFlushApiCache() {
        return this.to('FlushApiCache');
    }
    /**
     * Grants permission to retrieve an API
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetApi.html
     */
    toGetApi() {
        return this.to('GetApi');
    }
    /**
     * Grants permission to read custom domain name - GraphQL API association details in AppSync
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetApiAssociation.html
     */
    toGetApiAssociation() {
        return this.to('GetApiAssociation');
    }
    /**
     * Grants permission to read information about an API cache in AppSync
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetApiCache.html
     */
    toGetApiCache() {
        return this.to('GetApiCache');
    }
    /**
     * Grants permission to retrieve a channel namespace
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetChannelNamespace.html
     */
    toGetChannelNamespace() {
        return this.to('GetChannelNamespace');
    }
    /**
     * Grants permission to retrieve a data source
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetDataSource.html
     */
    toGetDataSource() {
        return this.to('GetDataSource');
    }
    /**
     * Grants permission to retrieve a data source introspection
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetDataSourceIntrospection.html
     */
    toGetDataSourceIntrospection() {
        return this.to('GetDataSourceIntrospection');
    }
    /**
     * Grants permission to read information about a custom domain name in AppSync
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetDomainName.html
     */
    toGetDomainName() {
        return this.to('GetDomainName');
    }
    /**
     * Grants permission to retrieve a function
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetFunction.html
     */
    toGetFunction() {
        return this.to('GetFunction');
    }
    /**
     * Grants permission to retrieve a GraphQL API
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetGraphqlApi.html
     */
    toGetGraphqlApi() {
        return this.to('GetGraphqlApi');
    }
    /**
     * Grants permission to retrieve the environment variables for a GraphQL API
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetGraphqlApiEnvironmentVariables.html
     */
    toGetGraphqlApiEnvironmentVariables() {
        return this.to('GetGraphqlApiEnvironmentVariables');
    }
    /**
     * Grants permission to retrieve the introspection schema for a GraphQL API
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetIntrospectionSchema.html
     */
    toGetIntrospectionSchema() {
        return this.to('GetIntrospectionSchema');
    }
    /**
     * Grants permission to retrieve a resolver
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetResolver.html
     */
    toGetResolver() {
        return this.to('GetResolver');
    }
    /**
     * Grants permission to read a resource policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/merge-api.html
     */
    toGetResourcePolicy() {
        return this.to('GetResourcePolicy');
    }
    /**
     * Grants permission to retrieve the current status of a schema creation operation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetSchemaCreationStatus.html
     */
    toGetSchemaCreationStatus() {
        return this.to('GetSchemaCreationStatus');
    }
    /**
     * Grants permission to read information about a merged API associated source API
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetSourceApiAssociation.html
     */
    toGetSourceApiAssociation() {
        return this.to('GetSourceApiAssociation');
    }
    /**
     * Grants permission to retrieve a type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetType.html
     */
    toGetType() {
        return this.to('GetType');
    }
    /**
     * Grants permission to send a GraphQL query to a GraphQL API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/security-authz.html#aws-iam-authorization
     */
    toGraphQL() {
        return this.to('GraphQL');
    }
    /**
     * Grants permission to list the API keys for a given API
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListApiKeys.html
     */
    toListApiKeys() {
        return this.to('ListApiKeys');
    }
    /**
     * Grants permission to list APIs
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListApis.html
     */
    toListApis() {
        return this.to('ListApis');
    }
    /**
     * Grants permission to list channel namespace
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListChannelNamespaces.html
     */
    toListChannelNamespaces() {
        return this.to('ListChannelNamespaces');
    }
    /**
     * Grants permission to list the data sources for a given API
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListDataSources.html
     */
    toListDataSources() {
        return this.to('ListDataSources');
    }
    /**
     * Grants permission to enumerate custom domain names in AppSync
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListDomainNames.html
     */
    toListDomainNames() {
        return this.to('ListDomainNames');
    }
    /**
     * Grants permission to list the functions for a given API
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListFunctions.html
     */
    toListFunctions() {
        return this.to('ListFunctions');
    }
    /**
     * Grants permission to list GraphQL APIs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListGraphqlApis.html
     */
    toListGraphqlApis() {
        return this.to('ListGraphqlApis');
    }
    /**
     * Grants permission to list the resolvers for a given API and type
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListResolvers.html
     */
    toListResolvers() {
        return this.to('ListResolvers');
    }
    /**
     * Grants permission to list the resolvers that are associated with a specific function
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListResolversByFunction.html
     */
    toListResolversByFunction() {
        return this.to('ListResolversByFunction');
    }
    /**
     * Grants permission to list source APIs associated to a given merged API
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListSourceApiAssociations.html
     */
    toListSourceApiAssociations() {
        return this.to('ListSourceApiAssociations');
    }
    /**
     * Grants permission to list the tags for a resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list the types for a given API
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListTypes.html
     */
    toListTypes() {
        return this.to('ListTypes');
    }
    /**
     * Grants permission to list the types for a given merged API and source API association
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListTypesByAssociation.html
     */
    toListTypesByAssociation() {
        return this.to('ListTypesByAssociation');
    }
    /**
     * Grants permission to update the environment variables for a GraphQL API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_PutGraphqlApiEnvironmentVariables.html
     */
    toPutGraphqlApiEnvironmentVariables() {
        return this.to('PutGraphqlApiEnvironmentVariables');
    }
    /**
     * Grants permission to set a resource policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/merge-api.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to set a web ACL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/WAF-Integration.html
     */
    toSetWebACL() {
        return this.to('SetWebACL');
    }
    /**
     * Grants permission to send a GraphQL query to a source API of a merged API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/using-your-api.html
     */
    toSourceGraphQL() {
        return this.to('SourceGraphQL');
    }
    /**
     * Grants permission to introspect a data source
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_StartDataSourceIntrospection.html
     */
    toStartDataSourceIntrospection() {
        return this.to('StartDataSourceIntrospection');
    }
    /**
     * Grants permission to add a new schema to your GraphQL API. This operation is asynchronous - GetSchemaCreationStatus can show when it has completed
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_StartSchemaCreation.html
     */
    toStartSchemaCreation() {
        return this.to('StartSchemaCreation');
    }
    /**
     * Grants permission to initiate a schema merge for a given merged API and associated source API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_StartSchemaMerge.html
     */
    toStartSchemaMerge() {
        return this.to('StartSchemaMerge');
    }
    /**
     * Grants permission to tag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an API
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateApi.html
     */
    toUpdateApi() {
        return this.to('UpdateApi');
    }
    /**
     * Grants permission to update an API cache in AppSync
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateApiCache.html
     */
    toUpdateApiCache() {
        return this.to('UpdateApiCache');
    }
    /**
     * Grants permission to update an API key for a given API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateApiKey.html
     */
    toUpdateApiKey() {
        return this.to('UpdateApiKey');
    }
    /**
     * Grants permission to update a channel namespace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateChannelNamespace.html
     */
    toUpdateChannelNamespace() {
        return this.to('UpdateChannelNamespace');
    }
    /**
     * Grants permission to update a data source
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateDataSource.html
     */
    toUpdateDataSource() {
        return this.to('UpdateDataSource');
    }
    /**
     * Grants permission to update a custom domain name in AppSync
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateDomainName.html
     */
    toUpdateDomainName() {
        return this.to('UpdateDomainName');
    }
    /**
     * Grants permission to update an existing function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateFunction.html
     */
    toUpdateFunction() {
        return this.to('UpdateFunction');
    }
    /**
     * Grants permission to update a GraphQL API
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateGraphqlApi.html
     */
    toUpdateGraphqlApi() {
        return this.to('UpdateGraphqlApi');
    }
    /**
     * Grants permission to update a resolver
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateResolver.html
     */
    toUpdateResolver() {
        return this.to('UpdateResolver');
    }
    /**
     * Grants permission to update a merged API source API association
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateSourceApiAssociation.html
     */
    toUpdateSourceApiAssociation() {
        return this.to('UpdateSourceApiAssociation');
    }
    /**
     * Grants permission to update a type
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateType.html
     */
    toUpdateType() {
        return this.to('UpdateType');
    }
    /**
     * Adds a resource of type datasource to the statement
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/attaching-a-data-source.html
     *
     * @param graphQLAPIId - Identifier for the graphQLAPIId.
     * @param datasourceName - Identifier for the datasourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onDatasource(graphQLAPIId, datasourceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:appsync:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apis/${graphQLAPIId}/datasources/${datasourceName}`);
    }
    /**
     * Adds a resource of type domain to the statement
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/custom-domain-name.html
     *
     * @param domainName - Identifier for the domainName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDomain(domainName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:appsync:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:domainnames/${domainName}`);
    }
    /**
     * Adds a resource of type graphqlapi to the statement
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/designing-a-graphql-api.html
     *
     * @param graphQLAPIId - Identifier for the graphQLAPIId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGraphqlapi(graphQLAPIId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:appsync:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apis/${graphQLAPIId}`);
    }
    /**
     * Adds a resource of type field to the statement
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/configuring-resolvers.html
     *
     * @param graphQLAPIId - Identifier for the graphQLAPIId.
     * @param typeName - Identifier for the typeName.
     * @param fieldName - Identifier for the fieldName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onField(graphQLAPIId, typeName, fieldName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:appsync:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apis/${graphQLAPIId}/types/${typeName}/fields/${fieldName}`);
    }
    /**
     * Adds a resource of type type to the statement
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/designing-your-schema.html#adding-a-root-query-type
     *
     * @param graphQLAPIId - Identifier for the graphQLAPIId.
     * @param typeName - Identifier for the typeName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onType(graphQLAPIId, typeName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:appsync:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apis/${graphQLAPIId}/types/${typeName}`);
    }
    /**
     * Adds a resource of type function to the statement
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/pipeline-resolvers.html
     *
     * @param graphQLAPIId - Identifier for the graphQLAPIId.
     * @param functionId - Identifier for the functionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onFunction(graphQLAPIId, functionId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:appsync:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apis/${graphQLAPIId}/functions/${functionId}`);
    }
    /**
     * Adds a resource of type sourceApiAssociation to the statement
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/merged-api.html
     *
     * @param mergedGraphQLAPIId - Identifier for the mergedGraphQLAPIId.
     * @param associationid - Identifier for the associationid.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onSourceApiAssociation(mergedGraphQLAPIId, associationid, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:appsync:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apis/${mergedGraphQLAPIId}/sourceApiAssociations/${associationid}`);
    }
    /**
     * Adds a resource of type mergedApiAssociation to the statement
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/merged-api.html
     *
     * @param sourceGraphQLAPIId - Identifier for the sourceGraphQLAPIId.
     * @param associationid - Identifier for the associationid.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onMergedApiAssociation(sourceGraphQLAPIId, associationid, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:appsync:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apis/${sourceGraphQLAPIId}/mergedApiAssociations/${associationid}`);
    }
    /**
     * Adds a resource of type api to the statement
     *
     * https://docs.aws.amazon.com/appsync/latest/eventapi/event-api-welcome.html
     *
     * @param apiId - Identifier for the apiId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onApi(apiId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:appsync:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apis/${apiId}`);
    }
    /**
     * Adds a resource of type channelNamespace to the statement
     *
     * https://docs.aws.amazon.com/appsync/latest/eventapi/channel-namespaces.html
     *
     * @param apiId - Identifier for the apiId.
     * @param channelNamespaceName - Identifier for the channelNamespaceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onChannelNamespace(apiId, channelNamespaceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:appsync:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apis/${apiId}/channelNamespace/${channelNamespaceName}`);
    }
    /**
     * Filters access by the visibility of an API
     *
     * Applies to actions:
     * - .toCreateGraphqlApi()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVisibility(value, operator) {
        return this.if(`Visibility`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateApi()
     * - .toCreateChannelNamespace()
     * - .toCreateDomainName()
     * - .toCreateGraphqlApi()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toCreateApi()
     * - .toCreateChannelNamespace()
     * - .toCreateDomainName()
     * - .toDeleteApi()
     * - .toDeleteChannelNamespace()
     * - .toDeleteDomainName()
     * - .toDeleteGraphqlApi()
     * - .toGetApi()
     * - .toGetChannelNamespace()
     * - .toGetDomainName()
     * - .toGetGraphqlApi()
     * - .toListApis()
     * - .toListChannelNamespaces()
     * - .toListDomainNames()
     * - .toListTagsForResource()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateApi()
     * - .toUpdateChannelNamespace()
     * - .toUpdateDomainName()
     * - .toUpdateGraphqlApi()
     *
     * Applies to resource types:
     * - domain
     * - graphqlapi
     * - api
     * - channelNamespace
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateApi()
     * - .toCreateChannelNamespace()
     * - .toCreateDomainName()
     * - .toCreateGraphqlApi()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [appsync](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsappsync.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'appsync';
        this.accessLevelList = {
            Write: [
                'AssociateApi',
                'AssociateMergedGraphqlApi',
                'AssociateSourceGraphqlApi',
                'CreateApi',
                'CreateApiCache',
                'CreateApiKey',
                'CreateChannelNamespace',
                'CreateDataSource',
                'CreateDomainName',
                'CreateFunction',
                'CreateGraphqlApi',
                'CreateResolver',
                'CreateType',
                'DeleteApi',
                'DeleteApiCache',
                'DeleteApiKey',
                'DeleteChannelNamespace',
                'DeleteDataSource',
                'DeleteDomainName',
                'DeleteFunction',
                'DeleteGraphqlApi',
                'DeleteResolver',
                'DeleteResourcePolicy',
                'DeleteType',
                'DisassociateApi',
                'DisassociateMergedGraphqlApi',
                'DisassociateSourceGraphqlApi',
                'EventConnect',
                'EventPublish',
                'EventSubscribe',
                'FlushApiCache',
                'GraphQL',
                'PutGraphqlApiEnvironmentVariables',
                'PutResourcePolicy',
                'SetWebACL',
                'SourceGraphQL',
                'StartDataSourceIntrospection',
                'StartSchemaCreation',
                'StartSchemaMerge',
                'UpdateApi',
                'UpdateApiCache',
                'UpdateApiKey',
                'UpdateChannelNamespace',
                'UpdateDataSource',
                'UpdateDomainName',
                'UpdateFunction',
                'UpdateGraphqlApi',
                'UpdateResolver',
                'UpdateSourceApiAssociation',
                'UpdateType'
            ],
            Read: [
                'EvaluateCode',
                'EvaluateMappingTemplate',
                'GetApi',
                'GetApiAssociation',
                'GetApiCache',
                'GetChannelNamespace',
                'GetDataSource',
                'GetDataSourceIntrospection',
                'GetDomainName',
                'GetFunction',
                'GetGraphqlApi',
                'GetGraphqlApiEnvironmentVariables',
                'GetIntrospectionSchema',
                'GetResolver',
                'GetResourcePolicy',
                'GetSchemaCreationStatus',
                'GetSourceApiAssociation',
                'GetType',
                'ListTagsForResource'
            ],
            List: [
                'ListApiKeys',
                'ListApis',
                'ListChannelNamespaces',
                'ListDataSources',
                'ListDomainNames',
                'ListFunctions',
                'ListGraphqlApis',
                'ListResolvers',
                'ListResolversByFunction',
                'ListSourceApiAssociations',
                'ListTypes',
                'ListTypesByAssociation'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Appsync = Appsync;
//# sourceMappingURL=data:application/json;base64,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