"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Batch = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [batch](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsbatch.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Batch extends shared_1.PolicyStatement {
    /**
     * Grants permission to cancel a job in an AWS Batch job queue in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_CancelJob.html
     */
    toCancelJob() {
        return this.to('CancelJob');
    }
    /**
     * Grants permission to create an AWS Batch compute environment in your account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_CreateComputeEnvironment.html
     */
    toCreateComputeEnvironment() {
        return this.to('CreateComputeEnvironment');
    }
    /**
     * Grants permission to create an AWS Batch consumable resource in your account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_CreateConsumableResource.html
     */
    toCreateConsumableResource() {
        return this.to('CreateConsumableResource');
    }
    /**
     * Grants permission to create an AWS Batch job queue in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_CreateJobQueue.html
     */
    toCreateJobQueue() {
        return this.to('CreateJobQueue');
    }
    /**
     * Grants permission to create an AWS Batch scheduling policy in your account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_CreateSchedulingPolicy.html
     */
    toCreateSchedulingPolicy() {
        return this.to('CreateSchedulingPolicy');
    }
    /**
     * Grants permission to delete an AWS Batch compute environment in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_DeleteComputeEnvironment.html
     */
    toDeleteComputeEnvironment() {
        return this.to('DeleteComputeEnvironment');
    }
    /**
     * Grants permission to delete an AWS Batch consumable resource in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_DeleteConsumableResource.html
     */
    toDeleteConsumableResource() {
        return this.to('DeleteConsumableResource');
    }
    /**
     * Grants permission to delete an AWS Batch job queue in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_DeleteJobQueue.html
     */
    toDeleteJobQueue() {
        return this.to('DeleteJobQueue');
    }
    /**
     * Grants permission to delete an AWS Batch scheduling policy in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_DeleteSchedulingPolicy.html
     */
    toDeleteSchedulingPolicy() {
        return this.to('DeleteSchedulingPolicy');
    }
    /**
     * Grants permission to deregister an AWS Batch job definition in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_DeregisterJobDefinition.html
     */
    toDeregisterJobDefinition() {
        return this.to('DeregisterJobDefinition');
    }
    /**
     * Grants permission to describe one or more AWS Batch compute environments in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_DescribeComputeEnvironments.html
     */
    toDescribeComputeEnvironments() {
        return this.to('DescribeComputeEnvironments');
    }
    /**
     * Grants permission to describe one or more AWS Batch consumable resource in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_DescribeConsumableResource.html
     */
    toDescribeConsumableResource() {
        return this.to('DescribeConsumableResource');
    }
    /**
     * Grants permission to describe one or more AWS Batch job definitions in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_DescribeJobDefinitions.html
     */
    toDescribeJobDefinitions() {
        return this.to('DescribeJobDefinitions');
    }
    /**
     * Grants permission to describe one or more AWS Batch job queues in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_DescribeJobQueues.html
     */
    toDescribeJobQueues() {
        return this.to('DescribeJobQueues');
    }
    /**
     * Grants permission to describe a list of AWS Batch jobs in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_DescribeJobs.html
     */
    toDescribeJobs() {
        return this.to('DescribeJobs');
    }
    /**
     * Grants permission to describe one or more AWS Batch scheduling policies in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_DescribeSchedulingPolicies.html
     */
    toDescribeSchedulingPolicies() {
        return this.to('DescribeSchedulingPolicies');
    }
    /**
     * Grants permission to get a snapshot of an AWS Batch job queue in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_GetJobQueueSnapshot.html
     */
    toGetJobQueueSnapshot() {
        return this.to('GetJobQueueSnapshot');
    }
    /**
     * Grants permission to list AWS Batch consumable resources in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_ListConsumableResources.html
     */
    toListConsumableResources() {
        return this.to('ListConsumableResources');
    }
    /**
     * Grants permission to list jobs for a specified AWS Batch job queue in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_ListJobs.html
     */
    toListJobs() {
        return this.to('ListJobs');
    }
    /**
     * Grants permission to list AWS Batch jobs that require a specific consumable resource in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_ListJobsByConsumableResource.html
     */
    toListJobsByConsumableResource() {
        return this.to('ListJobsByConsumableResource');
    }
    /**
     * Grants permission to list AWS Batch scheduling policies in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_ListSchedulingPolicies.html
     */
    toListSchedulingPolicies() {
        return this.to('ListSchedulingPolicies');
    }
    /**
     * Grants permission to list tags for an AWS Batch resource in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to register an AWS Batch job definition in your account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUser()
     * - .ifPrivileged()
     * - .ifImage()
     * - .ifLogDriver()
     * - .ifAWSLogsGroup()
     * - .ifAWSLogsRegion()
     * - .ifAWSLogsStreamPrefix()
     * - .ifAWSLogsCreateGroup()
     * - .ifEKSServiceAccountName()
     * - .ifEKSImage()
     * - .ifEKSRunAsUser()
     * - .ifEKSRunAsGroup()
     * - .ifEKSPrivileged()
     * - .ifEKSNamespace()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_RegisterJobDefinition.html
     */
    toRegisterJobDefinition() {
        return this.to('RegisterJobDefinition');
    }
    /**
     * Grants permission to submit an AWS Batch job from a job definition in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_SubmitJob.html
     */
    toSubmitJob() {
        return this.to('SubmitJob');
    }
    /**
     * Grants permission to tag an AWS Batch resource in your account
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to terminate a job in an AWS Batch job queue in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_TerminateJob.html
     */
    toTerminateJob() {
        return this.to('TerminateJob');
    }
    /**
     * Grants permission to untag an AWS Batch resource in your account
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an AWS Batch compute environment in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_UpdateComputeEnvironment.html
     */
    toUpdateComputeEnvironment() {
        return this.to('UpdateComputeEnvironment');
    }
    /**
     * Grants permission to update an AWS Batch consumable resource in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_UpdateConsumableResource.html
     */
    toUpdateConsumableResource() {
        return this.to('UpdateConsumableResource');
    }
    /**
     * Grants permission to update an AWS Batch job queue in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_UpdateJobQueue.html
     */
    toUpdateJobQueue() {
        return this.to('UpdateJobQueue');
    }
    /**
     * Grants permission to update an AWS Batch scheduling policy in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_UpdateSchedulingPolicy.html
     */
    toUpdateSchedulingPolicy() {
        return this.to('UpdateSchedulingPolicy');
    }
    /**
     * Adds a resource of type compute-environment to the statement
     *
     * https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html
     *
     * @param computeEnvironmentName - Identifier for the computeEnvironmentName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onComputeEnvironment(computeEnvironmentName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:batch:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:compute-environment/${computeEnvironmentName}`);
    }
    /**
     * Adds a resource of type job-queue to the statement
     *
     * https://docs.aws.amazon.com/batch/latest/userguide/job_queues.html
     *
     * @param jobQueueName - Identifier for the jobQueueName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onJobQueue(jobQueueName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:batch:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:job-queue/${jobQueueName}`);
    }
    /**
     * Adds a resource of type job-definition to the statement
     *
     * https://docs.aws.amazon.com/batch/latest/userguide/job_definitions.html
     *
     * @param jobDefinitionName - Identifier for the jobDefinitionName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onJobDefinition(jobDefinitionName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:batch:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:job-definition/${jobDefinitionName}`);
    }
    /**
     * Adds a resource of type job-definition-revision to the statement
     *
     * https://docs.aws.amazon.com/batch/latest/userguide/job_definitions.html
     *
     * @param jobDefinitionName - Identifier for the jobDefinitionName.
     * @param revision - Identifier for the revision.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onJobDefinitionRevision(jobDefinitionName, revision, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:batch:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:job-definition/${jobDefinitionName}:${revision}`);
    }
    /**
     * Adds a resource of type job to the statement
     *
     * https://docs.aws.amazon.com/batch/latest/userguide/jobs.html
     *
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onJob(jobId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:batch:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:job/${jobId}`);
    }
    /**
     * Adds a resource of type scheduling-policy to the statement
     *
     * https://docs.aws.amazon.com/batch/latest/userguide/scheduling-policies.html
     *
     * @param schedulingPolicyName - Identifier for the schedulingPolicyName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSchedulingPolicy(schedulingPolicyName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:batch:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:scheduling-policy/${schedulingPolicyName}`);
    }
    /**
     * Adds a resource of type consumable-resource to the statement
     *
     * https://docs.aws.amazon.com/batch/latest/userguide/resource-aware-scheduling.html
     *
     * @param consumableResourceName - Identifier for the consumableResourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConsumableResource(consumableResourceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:batch:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:consumable-resource/${consumableResourceName}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateComputeEnvironment()
     * - .toCreateConsumableResource()
     * - .toCreateJobQueue()
     * - .toCreateSchedulingPolicy()
     * - .toRegisterJobDefinition()
     * - .toSubmitJob()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - compute-environment
     * - job-queue
     * - job-definition-revision
     * - job
     * - scheduling-policy
     * - consumable-resource
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateComputeEnvironment()
     * - .toCreateConsumableResource()
     * - .toCreateJobQueue()
     * - .toCreateSchedulingPolicy()
     * - .toRegisterJobDefinition()
     * - .toSubmitJob()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the specified logging driver to determine whether awslogs group will be created for the logs
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsbatch.html#awsbatch-policy-keys
     *
     * Applies to actions:
     * - .toRegisterJobDefinition()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifAWSLogsCreateGroup(value) {
        return this.if(`AWSLogsCreateGroup`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the awslogs group where the logs are located
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsbatch.html#awsbatch-policy-keys
     *
     * Applies to actions:
     * - .toRegisterJobDefinition()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAWSLogsGroup(value, operator) {
        return this.if(`AWSLogsGroup`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the region where the logs are sent to
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsbatch.html#awsbatch-policy-keys
     *
     * Applies to actions:
     * - .toRegisterJobDefinition()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAWSLogsRegion(value, operator) {
        return this.if(`AWSLogsRegion`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the awslogs log stream prefix
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsbatch.html#awsbatch-policy-keys
     *
     * Applies to actions:
     * - .toRegisterJobDefinition()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAWSLogsStreamPrefix(value, operator) {
        return this.if(`AWSLogsStreamPrefix`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the image used to start a container for an Amazon EKS job
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsbatch.html#awsbatch-policy-keys
     *
     * Applies to actions:
     * - .toRegisterJobDefinition()
     * - .toSubmitJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEKSImage(value, operator) {
        return this.if(`EKSImage`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the namespace of a cluster used to run the pod for an Amazon EKS job
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsbatch.html#awsbatch-policy-keys
     *
     * Applies to actions:
     * - .toRegisterJobDefinition()
     * - .toSubmitJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEKSNamespace(value, operator) {
        return this.if(`EKSNamespace`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the specified privileged parameter value that determines whether the container is given elevated privileges on the host container instance (similar to the root user) for an Amazon EKS job
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsbatch.html#awsbatch-policy-keys
     *
     * Applies to actions:
     * - .toRegisterJobDefinition()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifEKSPrivileged(value) {
        return this.if(`EKSPrivileged`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the specified group numeric ID (gid) used to start a container in an Amazon EKS job
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsbatch.html#awsbatch-policy-keys
     *
     * Applies to actions:
     * - .toRegisterJobDefinition()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifEKSRunAsGroup(value, operator) {
        return this.if(`EKSRunAsGroup`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by the specified user numeric ID (uid) used to start a a container in an Amazon EKS job
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsbatch.html#awsbatch-policy-keys
     *
     * Applies to actions:
     * - .toRegisterJobDefinition()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifEKSRunAsUser(value, operator) {
        return this.if(`EKSRunAsUser`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by the name of the service account used to run the pod for an Amazon EKS job
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsbatch.html#awsbatch-policy-keys
     *
     * Applies to actions:
     * - .toRegisterJobDefinition()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEKSServiceAccountName(value, operator) {
        return this.if(`EKSServiceAccountName`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the image used to start a container
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsbatch.html#awsbatch-policy-keys
     *
     * Applies to actions:
     * - .toRegisterJobDefinition()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifImage(value, operator) {
        return this.if(`Image`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the log driver used for the container
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsbatch.html#awsbatch-policy-keys
     *
     * Applies to actions:
     * - .toRegisterJobDefinition()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLogDriver(value, operator) {
        return this.if(`LogDriver`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the specified privileged parameter value that determines whether the container is given elevated privileges on the host container instance (similar to the root user)
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsbatch.html#awsbatch-policy-keys
     *
     * Applies to actions:
     * - .toRegisterJobDefinition()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifPrivileged(value) {
        return this.if(`Privileged`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the shareIdentifier used inside submit job
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsbatch.html#awsbatch-policy-keys
     *
     * Applies to actions:
     * - .toSubmitJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifShareIdentifier(value, operator) {
        return this.if(`ShareIdentifier`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by user name or numeric uid used inside the container
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsbatch.html#awsbatch-policy-keys
     *
     * Applies to actions:
     * - .toRegisterJobDefinition()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifUser(value, operator) {
        return this.if(`User`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [batch](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsbatch.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'batch';
        this.accessLevelList = {
            Write: [
                'CancelJob',
                'CreateComputeEnvironment',
                'CreateConsumableResource',
                'CreateJobQueue',
                'CreateSchedulingPolicy',
                'DeleteComputeEnvironment',
                'DeleteConsumableResource',
                'DeleteJobQueue',
                'DeleteSchedulingPolicy',
                'DeregisterJobDefinition',
                'RegisterJobDefinition',
                'SubmitJob',
                'TerminateJob',
                'UpdateComputeEnvironment',
                'UpdateConsumableResource',
                'UpdateJobQueue',
                'UpdateSchedulingPolicy'
            ],
            Read: [
                'DescribeComputeEnvironments',
                'DescribeConsumableResource',
                'DescribeJobDefinitions',
                'DescribeJobQueues',
                'DescribeJobs',
                'DescribeSchedulingPolicies',
                'GetJobQueueSnapshot',
                'ListSchedulingPolicies',
                'ListTagsForResource'
            ],
            List: [
                'ListConsumableResources',
                'ListJobs',
                'ListJobsByConsumableResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Batch = Batch;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYmF0Y2guanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJiYXRjaC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFDQSx5Q0FBeUQ7QUFHekQ7Ozs7R0FJRztBQUNILE1BQWEsS0FBTSxTQUFRLHdCQUFlO0lBR3hDOzs7Ozs7T0FNRztJQUNJLFdBQVc7UUFDaEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFdBQVcsQ0FBQyxDQUFDO0lBQzlCLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksMEJBQTBCO1FBQy9CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO0lBQzdDLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksMEJBQTBCO1FBQy9CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO0lBQzdDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSx3QkFBd0I7UUFDN0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7SUFDM0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDBCQUEwQjtRQUMvQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTBCLENBQUMsQ0FBQztJQUM3QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMEJBQTBCO1FBQy9CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO0lBQzdDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUJBQXlCO1FBQzlCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO0lBQzVDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw2QkFBNkI7UUFDbEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixDQUFDLENBQUM7SUFDaEQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDRCQUE0QjtRQUNqQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNEJBQTRCLENBQUMsQ0FBQztJQUMvQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0JBQXdCO1FBQzdCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO0lBQzNDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGNBQWM7UUFDbkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGNBQWMsQ0FBQyxDQUFDO0lBQ2pDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw0QkFBNEI7UUFDakMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDRCQUE0QixDQUFDLENBQUM7SUFDL0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUJBQXlCO1FBQzlCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO0lBQzVDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxVQUFVO1FBQ2YsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFVBQVUsQ0FBQyxDQUFDO0lBQzdCLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw4QkFBOEI7UUFDbkMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDhCQUE4QixDQUFDLENBQUM7SUFDakQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09Bd0JHO0lBQ0ksdUJBQXVCO1FBQzVCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO0lBQzFDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxXQUFXO1FBQ2hCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxXQUFXLENBQUMsQ0FBQztJQUM5QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGFBQWE7UUFDbEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ2hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxjQUFjLENBQUMsQ0FBQztJQUNqQyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksZUFBZTtRQUNwQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDBCQUEwQjtRQUMvQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTBCLENBQUMsQ0FBQztJQUM3QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMEJBQTBCO1FBQy9CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO0lBQzdDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBNENEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLG9CQUFvQixDQUFDLHNCQUE4QixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQy9HLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLFVBQVcsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLHdCQUF5QixzQkFBdUIsRUFBRSxDQUFDLENBQUM7SUFDNUwsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLFVBQVUsQ0FBQyxZQUFvQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQzNGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLFVBQVcsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLGNBQWUsWUFBYSxFQUFFLENBQUMsQ0FBQztJQUN4SyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksZUFBZSxDQUFDLGlCQUF5QixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3JHLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLFVBQVcsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLG1CQUFvQixpQkFBa0IsRUFBRSxDQUFDLENBQUM7SUFDbEwsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSx1QkFBdUIsQ0FBQyxpQkFBeUIsRUFBRSxRQUFnQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQy9ILE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLFVBQVcsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLG1CQUFvQixpQkFBa0IsSUFBSyxRQUFTLEVBQUUsQ0FBQyxDQUFDO0lBQ2hNLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxLQUFLLENBQUMsS0FBYSxFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQy9FLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLFVBQVcsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLFFBQVMsS0FBTSxFQUFFLENBQUMsQ0FBQztJQUMzSixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksa0JBQWtCLENBQUMsb0JBQTRCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDM0csT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsVUFBVyxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsc0JBQXVCLG9CQUFxQixFQUFFLENBQUMsQ0FBQztJQUN4TCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksb0JBQW9CLENBQUMsc0JBQThCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDL0csT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsVUFBVyxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsd0JBQXlCLHNCQUF1QixFQUFFLENBQUMsQ0FBQztJQUM1TCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7O09BaUJHO0lBQ0ksZUFBZSxDQUFDLE1BQWMsRUFBRSxLQUF3QixFQUFFLFFBQTRCO1FBQzNGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBbUIsTUFBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNoRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7T0FnQkc7SUFDSSxnQkFBZ0IsQ0FBQyxNQUFjLEVBQUUsS0FBd0IsRUFBRSxRQUE0QjtRQUM1RixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW9CLE1BQU8sRUFBRSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDakYsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7OztPQWlCRztJQUNJLFlBQVksQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ3hFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRSxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksb0JBQW9CLENBQUMsS0FBZTtRQUN6QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLEVBQUUsQ0FBQyxPQUFPLEtBQUssS0FBSyxXQUFXLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEVBQUUsTUFBTSxDQUFDLENBQUM7SUFDOUYsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxjQUFjLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUMxRSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsY0FBYyxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDbEUsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxlQUFlLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUMzRSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDbkUsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxxQkFBcUIsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ2pGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ3pFLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLFVBQVUsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ3RFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxVQUFVLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUM5RCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxjQUFjLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUMxRSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsY0FBYyxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDbEUsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLGVBQWUsQ0FBQyxLQUFlO1FBQ3BDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLEVBQUUsQ0FBQyxPQUFPLEtBQUssS0FBSyxXQUFXLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEVBQUUsTUFBTSxDQUFDLENBQUM7SUFDekYsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxlQUFlLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUMzRSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksZUFBZSxDQUFDLENBQUM7SUFDdEUsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxjQUFjLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUMxRSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsY0FBYyxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksZUFBZSxDQUFDLENBQUM7SUFDckUsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSx1QkFBdUIsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ25GLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx1QkFBdUIsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQzNFLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksT0FBTyxDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDbkUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQU8sRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQzNELENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksV0FBVyxDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDdkUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFdBQVcsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQy9ELENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxZQUFZLENBQUMsS0FBZTtRQUNqQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsWUFBWSxFQUFFLENBQUMsT0FBTyxLQUFLLEtBQUssV0FBVyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxFQUFFLE1BQU0sQ0FBQyxDQUFDO0lBQ3RGLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksaUJBQWlCLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUM3RSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUJBQWlCLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNyRSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLE1BQU0sQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ2xFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUMxRCxDQUFDO0lBRUQ7OztPQUdHO0lBQ0gsWUFBWSxLQUFnQztRQUMxQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7UUFqMEJSLGtCQUFhLEdBQUcsT0FBTyxDQUFDO1FBNFhyQixvQkFBZSxHQUFvQjtZQUMzQyxLQUFLLEVBQUU7Z0JBQ0wsV0FBVztnQkFDWCwwQkFBMEI7Z0JBQzFCLDBCQUEwQjtnQkFDMUIsZ0JBQWdCO2dCQUNoQix3QkFBd0I7Z0JBQ3hCLDBCQUEwQjtnQkFDMUIsMEJBQTBCO2dCQUMxQixnQkFBZ0I7Z0JBQ2hCLHdCQUF3QjtnQkFDeEIseUJBQXlCO2dCQUN6Qix1QkFBdUI7Z0JBQ3ZCLFdBQVc7Z0JBQ1gsY0FBYztnQkFDZCwwQkFBMEI7Z0JBQzFCLDBCQUEwQjtnQkFDMUIsZ0JBQWdCO2dCQUNoQix3QkFBd0I7YUFDekI7WUFDRCxJQUFJLEVBQUU7Z0JBQ0osNkJBQTZCO2dCQUM3Qiw0QkFBNEI7Z0JBQzVCLHdCQUF3QjtnQkFDeEIsbUJBQW1CO2dCQUNuQixjQUFjO2dCQUNkLDRCQUE0QjtnQkFDNUIscUJBQXFCO2dCQUNyQix3QkFBd0I7Z0JBQ3hCLHFCQUFxQjthQUN0QjtZQUNELElBQUksRUFBRTtnQkFDSix5QkFBeUI7Z0JBQ3pCLFVBQVU7Z0JBQ1YsOEJBQThCO2FBQy9CO1lBQ0QsT0FBTyxFQUFFO2dCQUNQLGFBQWE7Z0JBQ2IsZUFBZTthQUNoQjtTQUNGLENBQUM7SUE4WkYsQ0FBQztDQUNGO0FBcDBCRCxzQkFvMEJDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQWNjZXNzTGV2ZWxMaXN0IH0gZnJvbSAnLi4vLi4vc2hhcmVkL2FjY2Vzcy1sZXZlbCc7XG5pbXBvcnQgeyBQb2xpY3lTdGF0ZW1lbnQsIE9wZXJhdG9yIH0gZnJvbSAnLi4vLi4vc2hhcmVkJztcbmltcG9ydCB7IGF3c19pYW0gYXMgaWFtIH0gZnJvbSBcImF3cy1jZGstbGliXCI7XG5cbi8qKlxuICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFtiYXRjaF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYXdzYmF0Y2guaHRtbCkuXG4gKlxuICogQHBhcmFtIHNpZCBbU0lEXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX3NpZC5odG1sKSBvZiB0aGUgc3RhdGVtZW50XG4gKi9cbmV4cG9ydCBjbGFzcyBCYXRjaCBleHRlbmRzIFBvbGljeVN0YXRlbWVudCB7XG4gIHB1YmxpYyBzZXJ2aWNlUHJlZml4ID0gJ2JhdGNoJztcblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY2FuY2VsIGEgam9iIGluIGFuIEFXUyBCYXRjaCBqb2IgcXVldWUgaW4geW91ciBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2JhdGNoL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NhbmNlbEpvYi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DYW5jZWxKb2IoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NhbmNlbEpvYicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhbiBBV1MgQmF0Y2ggY29tcHV0ZSBlbnZpcm9ubWVudCBpbiB5b3VyIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYmF0Y2gvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlQ29tcHV0ZUVudmlyb25tZW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUNvbXB1dGVFbnZpcm9ubWVudCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlQ29tcHV0ZUVudmlyb25tZW50Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGFuIEFXUyBCYXRjaCBjb25zdW1hYmxlIHJlc291cmNlIGluIHlvdXIgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9iYXRjaC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVDb25zdW1hYmxlUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlQ29uc3VtYWJsZVJlc291cmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVDb25zdW1hYmxlUmVzb3VyY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYW4gQVdTIEJhdGNoIGpvYiBxdWV1ZSBpbiB5b3VyIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYmF0Y2gvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlSm9iUXVldWUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlSm9iUXVldWUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZUpvYlF1ZXVlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGFuIEFXUyBCYXRjaCBzY2hlZHVsaW5nIHBvbGljeSBpbiB5b3VyIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYmF0Y2gvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlU2NoZWR1bGluZ1BvbGljeS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVTY2hlZHVsaW5nUG9saWN5KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVTY2hlZHVsaW5nUG9saWN5Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGFuIEFXUyBCYXRjaCBjb21wdXRlIGVudmlyb25tZW50IGluIHlvdXIgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9iYXRjaC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVDb21wdXRlRW52aXJvbm1lbnQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlQ29tcHV0ZUVudmlyb25tZW50KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVDb21wdXRlRW52aXJvbm1lbnQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYW4gQVdTIEJhdGNoIGNvbnN1bWFibGUgcmVzb3VyY2UgaW4geW91ciBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2JhdGNoL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUNvbnN1bWFibGVSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVDb25zdW1hYmxlUmVzb3VyY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZUNvbnN1bWFibGVSZXNvdXJjZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhbiBBV1MgQmF0Y2ggam9iIHF1ZXVlIGluIHlvdXIgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9iYXRjaC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVKb2JRdWV1ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVKb2JRdWV1ZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlSm9iUXVldWUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYW4gQVdTIEJhdGNoIHNjaGVkdWxpbmcgcG9saWN5IGluIHlvdXIgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9iYXRjaC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVTY2hlZHVsaW5nUG9saWN5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVNjaGVkdWxpbmdQb2xpY3koKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZVNjaGVkdWxpbmdQb2xpY3knKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZXJlZ2lzdGVyIGFuIEFXUyBCYXRjaCBqb2IgZGVmaW5pdGlvbiBpbiB5b3VyIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYmF0Y2gvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVyZWdpc3RlckpvYkRlZmluaXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVyZWdpc3RlckpvYkRlZmluaXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlcmVnaXN0ZXJKb2JEZWZpbml0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVzY3JpYmUgb25lIG9yIG1vcmUgQVdTIEJhdGNoIGNvbXB1dGUgZW52aXJvbm1lbnRzIGluIHlvdXIgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2JhdGNoL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlQ29tcHV0ZUVudmlyb25tZW50cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZUNvbXB1dGVFbnZpcm9ubWVudHMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlQ29tcHV0ZUVudmlyb25tZW50cycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlc2NyaWJlIG9uZSBvciBtb3JlIEFXUyBCYXRjaCBjb25zdW1hYmxlIHJlc291cmNlIGluIHlvdXIgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2JhdGNoL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlQ29uc3VtYWJsZVJlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlQ29uc3VtYWJsZVJlc291cmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZUNvbnN1bWFibGVSZXNvdXJjZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlc2NyaWJlIG9uZSBvciBtb3JlIEFXUyBCYXRjaCBqb2IgZGVmaW5pdGlvbnMgaW4geW91ciBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYmF0Y2gvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVKb2JEZWZpbml0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZUpvYkRlZmluaXRpb25zKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZUpvYkRlZmluaXRpb25zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVzY3JpYmUgb25lIG9yIG1vcmUgQVdTIEJhdGNoIGpvYiBxdWV1ZXMgaW4geW91ciBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYmF0Y2gvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVKb2JRdWV1ZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVKb2JRdWV1ZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlSm9iUXVldWVzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVzY3JpYmUgYSBsaXN0IG9mIEFXUyBCYXRjaCBqb2JzIGluIHlvdXIgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2JhdGNoL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlSm9icy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZUpvYnMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlSm9icycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlc2NyaWJlIG9uZSBvciBtb3JlIEFXUyBCYXRjaCBzY2hlZHVsaW5nIHBvbGljaWVzIGluIHlvdXIgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2JhdGNoL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlU2NoZWR1bGluZ1BvbGljaWVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlU2NoZWR1bGluZ1BvbGljaWVzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZVNjaGVkdWxpbmdQb2xpY2llcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBhIHNuYXBzaG90IG9mIGFuIEFXUyBCYXRjaCBqb2IgcXVldWUgaW4geW91ciBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYmF0Y2gvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0Sm9iUXVldWVTbmFwc2hvdC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRKb2JRdWV1ZVNuYXBzaG90KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRKb2JRdWV1ZVNuYXBzaG90Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBBV1MgQmF0Y2ggY29uc3VtYWJsZSByZXNvdXJjZXMgaW4geW91ciBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYmF0Y2gvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdENvbnN1bWFibGVSZXNvdXJjZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdENvbnN1bWFibGVSZXNvdXJjZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RDb25zdW1hYmxlUmVzb3VyY2VzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBqb2JzIGZvciBhIHNwZWNpZmllZCBBV1MgQmF0Y2ggam9iIHF1ZXVlIGluIHlvdXIgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2JhdGNoL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RKb2JzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RKb2JzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0Sm9icycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgQVdTIEJhdGNoIGpvYnMgdGhhdCByZXF1aXJlIGEgc3BlY2lmaWMgY29uc3VtYWJsZSByZXNvdXJjZSBpbiB5b3VyIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9iYXRjaC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0Sm9ic0J5Q29uc3VtYWJsZVJlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RKb2JzQnlDb25zdW1hYmxlUmVzb3VyY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RKb2JzQnlDb25zdW1hYmxlUmVzb3VyY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IEFXUyBCYXRjaCBzY2hlZHVsaW5nIHBvbGljaWVzIGluIHlvdXIgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2JhdGNoL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RTY2hlZHVsaW5nUG9saWNpZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFNjaGVkdWxpbmdQb2xpY2llcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFNjaGVkdWxpbmdQb2xpY2llcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgdGFncyBmb3IgYW4gQVdTIEJhdGNoIHJlc291cmNlIGluIHlvdXIgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2JhdGNoL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RUYWdzRm9yUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFRhZ3NGb3JSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFRhZ3NGb3JSZXNvdXJjZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJlZ2lzdGVyIGFuIEFXUyBCYXRjaCBqb2IgZGVmaW5pdGlvbiBpbiB5b3VyIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZlVzZXIoKVxuICAgKiAtIC5pZlByaXZpbGVnZWQoKVxuICAgKiAtIC5pZkltYWdlKClcbiAgICogLSAuaWZMb2dEcml2ZXIoKVxuICAgKiAtIC5pZkFXU0xvZ3NHcm91cCgpXG4gICAqIC0gLmlmQVdTTG9nc1JlZ2lvbigpXG4gICAqIC0gLmlmQVdTTG9nc1N0cmVhbVByZWZpeCgpXG4gICAqIC0gLmlmQVdTTG9nc0NyZWF0ZUdyb3VwKClcbiAgICogLSAuaWZFS1NTZXJ2aWNlQWNjb3VudE5hbWUoKVxuICAgKiAtIC5pZkVLU0ltYWdlKClcbiAgICogLSAuaWZFS1NSdW5Bc1VzZXIoKVxuICAgKiAtIC5pZkVLU1J1bkFzR3JvdXAoKVxuICAgKiAtIC5pZkVLU1ByaXZpbGVnZWQoKVxuICAgKiAtIC5pZkVLU05hbWVzcGFjZSgpXG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9iYXRjaC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9SZWdpc3RlckpvYkRlZmluaXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUmVnaXN0ZXJKb2JEZWZpbml0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdSZWdpc3RlckpvYkRlZmluaXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBzdWJtaXQgYW4gQVdTIEJhdGNoIGpvYiBmcm9tIGEgam9iIGRlZmluaXRpb24gaW4geW91ciBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2JhdGNoL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1N1Ym1pdEpvYi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TdWJtaXRKb2IoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1N1Ym1pdEpvYicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHRhZyBhbiBBV1MgQmF0Y2ggcmVzb3VyY2UgaW4geW91ciBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogVGFnZ2luZ1xuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYmF0Y2gvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVGFnUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVGFnUmVzb3VyY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1RhZ1Jlc291cmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdGVybWluYXRlIGEgam9iIGluIGFuIEFXUyBCYXRjaCBqb2IgcXVldWUgaW4geW91ciBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2JhdGNoL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1Rlcm1pbmF0ZUpvYi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9UZXJtaW5hdGVKb2IoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1Rlcm1pbmF0ZUpvYicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVudGFnIGFuIEFXUyBCYXRjaCByZXNvdXJjZSBpbiB5b3VyIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9iYXRjaC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VbnRhZ1Jlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VudGFnUmVzb3VyY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VudGFnUmVzb3VyY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYW4gQVdTIEJhdGNoIGNvbXB1dGUgZW52aXJvbm1lbnQgaW4geW91ciBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2JhdGNoL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZUNvbXB1dGVFbnZpcm9ubWVudC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVDb21wdXRlRW52aXJvbm1lbnQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZUNvbXB1dGVFbnZpcm9ubWVudCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhbiBBV1MgQmF0Y2ggY29uc3VtYWJsZSByZXNvdXJjZSBpbiB5b3VyIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYmF0Y2gvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlQ29uc3VtYWJsZVJlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUNvbnN1bWFibGVSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlQ29uc3VtYWJsZVJlc291cmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIGFuIEFXUyBCYXRjaCBqb2IgcXVldWUgaW4geW91ciBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2JhdGNoL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZUpvYlF1ZXVlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUpvYlF1ZXVlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVKb2JRdWV1ZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhbiBBV1MgQmF0Y2ggc2NoZWR1bGluZyBwb2xpY3kgaW4geW91ciBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2JhdGNoL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZVNjaGVkdWxpbmdQb2xpY3kuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlU2NoZWR1bGluZ1BvbGljeSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlU2NoZWR1bGluZ1BvbGljeScpO1xuICB9XG5cbiAgcHJvdGVjdGVkIGFjY2Vzc0xldmVsTGlzdDogQWNjZXNzTGV2ZWxMaXN0ID0ge1xuICAgIFdyaXRlOiBbXG4gICAgICAnQ2FuY2VsSm9iJyxcbiAgICAgICdDcmVhdGVDb21wdXRlRW52aXJvbm1lbnQnLFxuICAgICAgJ0NyZWF0ZUNvbnN1bWFibGVSZXNvdXJjZScsXG4gICAgICAnQ3JlYXRlSm9iUXVldWUnLFxuICAgICAgJ0NyZWF0ZVNjaGVkdWxpbmdQb2xpY3knLFxuICAgICAgJ0RlbGV0ZUNvbXB1dGVFbnZpcm9ubWVudCcsXG4gICAgICAnRGVsZXRlQ29uc3VtYWJsZVJlc291cmNlJyxcbiAgICAgICdEZWxldGVKb2JRdWV1ZScsXG4gICAgICAnRGVsZXRlU2NoZWR1bGluZ1BvbGljeScsXG4gICAgICAnRGVyZWdpc3RlckpvYkRlZmluaXRpb24nLFxuICAgICAgJ1JlZ2lzdGVySm9iRGVmaW5pdGlvbicsXG4gICAgICAnU3VibWl0Sm9iJyxcbiAgICAgICdUZXJtaW5hdGVKb2InLFxuICAgICAgJ1VwZGF0ZUNvbXB1dGVFbnZpcm9ubWVudCcsXG4gICAgICAnVXBkYXRlQ29uc3VtYWJsZVJlc291cmNlJyxcbiAgICAgICdVcGRhdGVKb2JRdWV1ZScsXG4gICAgICAnVXBkYXRlU2NoZWR1bGluZ1BvbGljeSdcbiAgICBdLFxuICAgIFJlYWQ6IFtcbiAgICAgICdEZXNjcmliZUNvbXB1dGVFbnZpcm9ubWVudHMnLFxuICAgICAgJ0Rlc2NyaWJlQ29uc3VtYWJsZVJlc291cmNlJyxcbiAgICAgICdEZXNjcmliZUpvYkRlZmluaXRpb25zJyxcbiAgICAgICdEZXNjcmliZUpvYlF1ZXVlcycsXG4gICAgICAnRGVzY3JpYmVKb2JzJyxcbiAgICAgICdEZXNjcmliZVNjaGVkdWxpbmdQb2xpY2llcycsXG4gICAgICAnR2V0Sm9iUXVldWVTbmFwc2hvdCcsXG4gICAgICAnTGlzdFNjaGVkdWxpbmdQb2xpY2llcycsXG4gICAgICAnTGlzdFRhZ3NGb3JSZXNvdXJjZSdcbiAgICBdLFxuICAgIExpc3Q6IFtcbiAgICAgICdMaXN0Q29uc3VtYWJsZVJlc291cmNlcycsXG4gICAgICAnTGlzdEpvYnMnLFxuICAgICAgJ0xpc3RKb2JzQnlDb25zdW1hYmxlUmVzb3VyY2UnXG4gICAgXSxcbiAgICBUYWdnaW5nOiBbXG4gICAgICAnVGFnUmVzb3VyY2UnLFxuICAgICAgJ1VudGFnUmVzb3VyY2UnXG4gICAgXVxuICB9O1xuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBjb21wdXRlLWVudmlyb25tZW50IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2JhdGNoL2xhdGVzdC91c2VyZ3VpZGUvY29tcHV0ZV9lbnZpcm9ubWVudHMuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gY29tcHV0ZUVudmlyb25tZW50TmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBjb21wdXRlRW52aXJvbm1lbnROYW1lLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25Db21wdXRlRW52aXJvbm1lbnQoY29tcHV0ZUVudmlyb25tZW50TmFtZTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmJhdGNoOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpjb21wdXRlLWVudmlyb25tZW50LyR7IGNvbXB1dGVFbnZpcm9ubWVudE5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGpvYi1xdWV1ZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9iYXRjaC9sYXRlc3QvdXNlcmd1aWRlL2pvYl9xdWV1ZXMuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gam9iUXVldWVOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGpvYlF1ZXVlTmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uSm9iUXVldWUoam9iUXVldWVOYW1lOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YmF0Y2g6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OmpvYi1xdWV1ZS8keyBqb2JRdWV1ZU5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGpvYi1kZWZpbml0aW9uIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2JhdGNoL2xhdGVzdC91c2VyZ3VpZGUvam9iX2RlZmluaXRpb25zLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGpvYkRlZmluaXRpb25OYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGpvYkRlZmluaXRpb25OYW1lLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICovXG4gIHB1YmxpYyBvbkpvYkRlZmluaXRpb24oam9iRGVmaW5pdGlvbk5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpiYXRjaDokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06am9iLWRlZmluaXRpb24vJHsgam9iRGVmaW5pdGlvbk5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGpvYi1kZWZpbml0aW9uLXJldmlzaW9uIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2JhdGNoL2xhdGVzdC91c2VyZ3VpZGUvam9iX2RlZmluaXRpb25zLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGpvYkRlZmluaXRpb25OYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGpvYkRlZmluaXRpb25OYW1lLlxuICAgKiBAcGFyYW0gcmV2aXNpb24gLSBJZGVudGlmaWVyIGZvciB0aGUgcmV2aXNpb24uXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkpvYkRlZmluaXRpb25SZXZpc2lvbihqb2JEZWZpbml0aW9uTmFtZTogc3RyaW5nLCByZXZpc2lvbjogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmJhdGNoOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpqb2ItZGVmaW5pdGlvbi8keyBqb2JEZWZpbml0aW9uTmFtZSB9OiR7IHJldmlzaW9uIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBqb2IgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYmF0Y2gvbGF0ZXN0L3VzZXJndWlkZS9qb2JzLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGpvYklkIC0gSWRlbnRpZmllciBmb3IgdGhlIGpvYklkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25Kb2Ioam9iSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpiYXRjaDokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06am9iLyR7IGpvYklkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBzY2hlZHVsaW5nLXBvbGljeSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9iYXRjaC9sYXRlc3QvdXNlcmd1aWRlL3NjaGVkdWxpbmctcG9saWNpZXMuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gc2NoZWR1bGluZ1BvbGljeU5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgc2NoZWR1bGluZ1BvbGljeU5hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvblNjaGVkdWxpbmdQb2xpY3koc2NoZWR1bGluZ1BvbGljeU5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpiYXRjaDokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06c2NoZWR1bGluZy1wb2xpY3kvJHsgc2NoZWR1bGluZ1BvbGljeU5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGNvbnN1bWFibGUtcmVzb3VyY2UgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYmF0Y2gvbGF0ZXN0L3VzZXJndWlkZS9yZXNvdXJjZS1hd2FyZS1zY2hlZHVsaW5nLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGNvbnN1bWFibGVSZXNvdXJjZU5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgY29uc3VtYWJsZVJlc291cmNlTmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uQ29uc3VtYWJsZVJlc291cmNlKGNvbnN1bWFibGVSZXNvdXJjZU5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpiYXRjaDokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06Y29uc3VtYWJsZS1yZXNvdXJjZS8keyBjb25zdW1hYmxlUmVzb3VyY2VOYW1lIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgdGFncyB0aGF0IGFyZSBwYXNzZWQgaW4gdGhlIHJlcXVlc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXJlcXVlc3R0YWdcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0NyZWF0ZUNvbXB1dGVFbnZpcm9ubWVudCgpXG4gICAqIC0gLnRvQ3JlYXRlQ29uc3VtYWJsZVJlc291cmNlKClcbiAgICogLSAudG9DcmVhdGVKb2JRdWV1ZSgpXG4gICAqIC0gLnRvQ3JlYXRlU2NoZWR1bGluZ1BvbGljeSgpXG4gICAqIC0gLnRvUmVnaXN0ZXJKb2JEZWZpbml0aW9uKClcbiAgICogLSAudG9TdWJtaXRKb2IoKVxuICAgKiAtIC50b1RhZ1Jlc291cmNlKClcbiAgICpcbiAgICogQHBhcmFtIHRhZ0tleSBUaGUgdGFnIGtleSB0byBjaGVja1xuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1JlcXVlc3RUYWcodGFnS2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6UmVxdWVzdFRhZy8keyB0YWdLZXkgfWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSB0YWdzIGFzc29jaWF0ZWQgd2l0aCB0aGUgcmVzb3VyY2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXJlc291cmNldGFnXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gY29tcHV0ZS1lbnZpcm9ubWVudFxuICAgKiAtIGpvYi1xdWV1ZVxuICAgKiAtIGpvYi1kZWZpbml0aW9uLXJldmlzaW9uXG4gICAqIC0gam9iXG4gICAqIC0gc2NoZWR1bGluZy1wb2xpY3lcbiAgICogLSBjb25zdW1hYmxlLXJlc291cmNlXG4gICAqXG4gICAqIEBwYXJhbSB0YWdLZXkgVGhlIHRhZyBrZXkgdG8gY2hlY2tcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NSZXNvdXJjZVRhZyh0YWdLZXk6IHN0cmluZywgdmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpSZXNvdXJjZVRhZy8keyB0YWdLZXkgfWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSB0YWcga2V5cyB0aGF0IGFyZSBwYXNzZWQgaW4gdGhlIHJlcXVlc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXRhZ2tleXNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0NyZWF0ZUNvbXB1dGVFbnZpcm9ubWVudCgpXG4gICAqIC0gLnRvQ3JlYXRlQ29uc3VtYWJsZVJlc291cmNlKClcbiAgICogLSAudG9DcmVhdGVKb2JRdWV1ZSgpXG4gICAqIC0gLnRvQ3JlYXRlU2NoZWR1bGluZ1BvbGljeSgpXG4gICAqIC0gLnRvUmVnaXN0ZXJKb2JEZWZpbml0aW9uKClcbiAgICogLSAudG9TdWJtaXRKb2IoKVxuICAgKiAtIC50b1RhZ1Jlc291cmNlKClcbiAgICogLSAudG9VbnRhZ1Jlc291cmNlKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NUYWdLZXlzKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6VGFnS2V5c2AsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSBzcGVjaWZpZWQgbG9nZ2luZyBkcml2ZXIgdG8gZGV0ZXJtaW5lIHdoZXRoZXIgYXdzbG9ncyBncm91cCB3aWxsIGJlIGNyZWF0ZWQgZm9yIHRoZSBsb2dzXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXJ2aWNlLWF1dGhvcml6YXRpb24vbGF0ZXN0L3JlZmVyZW5jZS9saXN0X2F3c2JhdGNoLmh0bWwjYXdzYmF0Y2gtcG9saWN5LWtleXNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b1JlZ2lzdGVySm9iRGVmaW5pdGlvbigpXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBgdHJ1ZWAgb3IgYGZhbHNlYC4gKipEZWZhdWx0OioqIGB0cnVlYFxuICAgKi9cbiAgcHVibGljIGlmQVdTTG9nc0NyZWF0ZUdyb3VwKHZhbHVlPzogYm9vbGVhbikge1xuICAgIHJldHVybiB0aGlzLmlmKGBBV1NMb2dzQ3JlYXRlR3JvdXBgLCAodHlwZW9mIHZhbHVlICE9PSAndW5kZWZpbmVkJyA/IHZhbHVlIDogdHJ1ZSksICdCb29sJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgdGhlIGF3c2xvZ3MgZ3JvdXAgd2hlcmUgdGhlIGxvZ3MgYXJlIGxvY2F0ZWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYXdzYmF0Y2guaHRtbCNhd3NiYXRjaC1wb2xpY3kta2V5c1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvUmVnaXN0ZXJKb2JEZWZpbml0aW9uKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBV1NMb2dzR3JvdXAodmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYEFXU0xvZ3NHcm91cGAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSByZWdpb24gd2hlcmUgdGhlIGxvZ3MgYXJlIHNlbnQgdG9cbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYXdzYmF0Y2guaHRtbCNhd3NiYXRjaC1wb2xpY3kta2V5c1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvUmVnaXN0ZXJKb2JEZWZpbml0aW9uKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBV1NMb2dzUmVnaW9uKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBBV1NMb2dzUmVnaW9uYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgdGhlIGF3c2xvZ3MgbG9nIHN0cmVhbSBwcmVmaXhcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYXdzYmF0Y2guaHRtbCNhd3NiYXRjaC1wb2xpY3kta2V5c1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvUmVnaXN0ZXJKb2JEZWZpbml0aW9uKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBV1NMb2dzU3RyZWFtUHJlZml4KHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBBV1NMb2dzU3RyZWFtUHJlZml4YCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgdGhlIGltYWdlIHVzZWQgdG8gc3RhcnQgYSBjb250YWluZXIgZm9yIGFuIEFtYXpvbiBFS1Mgam9iXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXJ2aWNlLWF1dGhvcml6YXRpb24vbGF0ZXN0L3JlZmVyZW5jZS9saXN0X2F3c2JhdGNoLmh0bWwjYXdzYmF0Y2gtcG9saWN5LWtleXNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b1JlZ2lzdGVySm9iRGVmaW5pdGlvbigpXG4gICAqIC0gLnRvU3VibWl0Sm9iKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZFS1NJbWFnZSh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgRUtTSW1hZ2VgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgbmFtZXNwYWNlIG9mIGEgY2x1c3RlciB1c2VkIHRvIHJ1biB0aGUgcG9kIGZvciBhbiBBbWF6b24gRUtTIGpvYlxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hd3NiYXRjaC5odG1sI2F3c2JhdGNoLXBvbGljeS1rZXlzXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9SZWdpc3RlckpvYkRlZmluaXRpb24oKVxuICAgKiAtIC50b1N1Ym1pdEpvYigpXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmRUtTTmFtZXNwYWNlKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBFS1NOYW1lc3BhY2VgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgc3BlY2lmaWVkIHByaXZpbGVnZWQgcGFyYW1ldGVyIHZhbHVlIHRoYXQgZGV0ZXJtaW5lcyB3aGV0aGVyIHRoZSBjb250YWluZXIgaXMgZ2l2ZW4gZWxldmF0ZWQgcHJpdmlsZWdlcyBvbiB0aGUgaG9zdCBjb250YWluZXIgaW5zdGFuY2UgKHNpbWlsYXIgdG8gdGhlIHJvb3QgdXNlcikgZm9yIGFuIEFtYXpvbiBFS1Mgam9iXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXJ2aWNlLWF1dGhvcml6YXRpb24vbGF0ZXN0L3JlZmVyZW5jZS9saXN0X2F3c2JhdGNoLmh0bWwjYXdzYmF0Y2gtcG9saWN5LWtleXNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b1JlZ2lzdGVySm9iRGVmaW5pdGlvbigpXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBgdHJ1ZWAgb3IgYGZhbHNlYC4gKipEZWZhdWx0OioqIGB0cnVlYFxuICAgKi9cbiAgcHVibGljIGlmRUtTUHJpdmlsZWdlZCh2YWx1ZT86IGJvb2xlYW4pIHtcbiAgICByZXR1cm4gdGhpcy5pZihgRUtTUHJpdmlsZWdlZGAsICh0eXBlb2YgdmFsdWUgIT09ICd1bmRlZmluZWQnID8gdmFsdWUgOiB0cnVlKSwgJ0Jvb2wnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgc3BlY2lmaWVkIGdyb3VwIG51bWVyaWMgSUQgKGdpZCkgdXNlZCB0byBzdGFydCBhIGNvbnRhaW5lciBpbiBhbiBBbWF6b24gRUtTIGpvYlxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hd3NiYXRjaC5odG1sI2F3c2JhdGNoLXBvbGljeS1rZXlzXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9SZWdpc3RlckpvYkRlZmluaXRpb24oKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtudW1lcmljIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19OdW1lcmljKS4gKipEZWZhdWx0OioqIGBOdW1lcmljRXF1YWxzYFxuICAgKi9cbiAgcHVibGljIGlmRUtTUnVuQXNHcm91cCh2YWx1ZTogbnVtYmVyIHwgbnVtYmVyW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgRUtTUnVuQXNHcm91cGAsIHZhbHVlLCBvcGVyYXRvciA/PyAnTnVtZXJpY0VxdWFscycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSBzcGVjaWZpZWQgdXNlciBudW1lcmljIElEICh1aWQpIHVzZWQgdG8gc3RhcnQgYSBhIGNvbnRhaW5lciBpbiBhbiBBbWF6b24gRUtTIGpvYlxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hd3NiYXRjaC5odG1sI2F3c2JhdGNoLXBvbGljeS1rZXlzXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9SZWdpc3RlckpvYkRlZmluaXRpb24oKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtudW1lcmljIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19OdW1lcmljKS4gKipEZWZhdWx0OioqIGBOdW1lcmljRXF1YWxzYFxuICAgKi9cbiAgcHVibGljIGlmRUtTUnVuQXNVc2VyKHZhbHVlOiBudW1iZXIgfCBudW1iZXJbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBFS1NSdW5Bc1VzZXJgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ051bWVyaWNFcXVhbHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgbmFtZSBvZiB0aGUgc2VydmljZSBhY2NvdW50IHVzZWQgdG8gcnVuIHRoZSBwb2QgZm9yIGFuIEFtYXpvbiBFS1Mgam9iXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXJ2aWNlLWF1dGhvcml6YXRpb24vbGF0ZXN0L3JlZmVyZW5jZS9saXN0X2F3c2JhdGNoLmh0bWwjYXdzYmF0Y2gtcG9saWN5LWtleXNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b1JlZ2lzdGVySm9iRGVmaW5pdGlvbigpXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmRUtTU2VydmljZUFjY291bnROYW1lKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBFS1NTZXJ2aWNlQWNjb3VudE5hbWVgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgaW1hZ2UgdXNlZCB0byBzdGFydCBhIGNvbnRhaW5lclxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hd3NiYXRjaC5odG1sI2F3c2JhdGNoLXBvbGljeS1rZXlzXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9SZWdpc3RlckpvYkRlZmluaXRpb24oKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkltYWdlKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBJbWFnZWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSBsb2cgZHJpdmVyIHVzZWQgZm9yIHRoZSBjb250YWluZXJcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYXdzYmF0Y2guaHRtbCNhd3NiYXRjaC1wb2xpY3kta2V5c1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvUmVnaXN0ZXJKb2JEZWZpbml0aW9uKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZMb2dEcml2ZXIodmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYExvZ0RyaXZlcmAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSBzcGVjaWZpZWQgcHJpdmlsZWdlZCBwYXJhbWV0ZXIgdmFsdWUgdGhhdCBkZXRlcm1pbmVzIHdoZXRoZXIgdGhlIGNvbnRhaW5lciBpcyBnaXZlbiBlbGV2YXRlZCBwcml2aWxlZ2VzIG9uIHRoZSBob3N0IGNvbnRhaW5lciBpbnN0YW5jZSAoc2ltaWxhciB0byB0aGUgcm9vdCB1c2VyKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hd3NiYXRjaC5odG1sI2F3c2JhdGNoLXBvbGljeS1rZXlzXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9SZWdpc3RlckpvYkRlZmluaXRpb24oKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgYHRydWVgIG9yIGBmYWxzZWAuICoqRGVmYXVsdDoqKiBgdHJ1ZWBcbiAgICovXG4gIHB1YmxpYyBpZlByaXZpbGVnZWQodmFsdWU/OiBib29sZWFuKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYFByaXZpbGVnZWRgLCAodHlwZW9mIHZhbHVlICE9PSAndW5kZWZpbmVkJyA/IHZhbHVlIDogdHJ1ZSksICdCb29sJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgdGhlIHNoYXJlSWRlbnRpZmllciB1c2VkIGluc2lkZSBzdWJtaXQgam9iXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXJ2aWNlLWF1dGhvcml6YXRpb24vbGF0ZXN0L3JlZmVyZW5jZS9saXN0X2F3c2JhdGNoLmh0bWwjYXdzYmF0Y2gtcG9saWN5LWtleXNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b1N1Ym1pdEpvYigpXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmU2hhcmVJZGVudGlmaWVyKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBTaGFyZUlkZW50aWZpZXJgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB1c2VyIG5hbWUgb3IgbnVtZXJpYyB1aWQgdXNlZCBpbnNpZGUgdGhlIGNvbnRhaW5lclxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hd3NiYXRjaC5odG1sI2F3c2JhdGNoLXBvbGljeS1rZXlzXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9SZWdpc3RlckpvYkRlZmluaXRpb24oKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlVzZXIodmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYFVzZXJgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW2JhdGNoXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hd3NiYXRjaC5odG1sKS5cbiAgICpcbiAgICovXG4gIGNvbnN0cnVjdG9yKHByb3BzPzogaWFtLlBvbGljeVN0YXRlbWVudFByb3BzKSB7XG4gICAgc3VwZXIocHJvcHMpO1xuICB9XG59XG4iXX0=